/*
 * Quantis PCI driver
 *
 * Copyright (c) 2004-2010 id Quantique SA, Carouge/Geneva, Switzerland
 * All rights reserved.
 *
 * ----------------------------------------------------------------------------
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY.
 *
 * ----------------------------------------------------------------------------
 *
 * Alternatively, this software may be distributed under the terms of the
 * terms of the GNU General Public License version 2 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * ----------------------------------------------------------------------------
 *
 * For history of changes, ChangeLog.txt
 *
 * This header file describes the register of the Quantis PCI card. The code has been adapted from
 * a microsoft provided sample file.
 *
 Module Name:

    Driver.c - Main driver file (contains the entry point) for the Quantis number generator

Environment:

    Kernel mode
 */

#include "CommonWindows.h"

QUANTIS_EXTERN_C_START

#include "Quantis.h"
#include "driver.tmh" // auto-generated by WPP

//
// Make sure the initialization code is removed from memory after use.
//

#ifdef ALLOC_PRAGMA
#pragma alloc_text(INIT, DriverEntry)
#pragma alloc_text(PAGE, QuantisEvtDriverContextCleanup)
#endif


NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT  DriverObject,
    IN PUNICODE_STRING RegistryPath
    )
/*++

Routine Description:

    Installable driver initialization entry point.
    This entry point is called directly by the I/O system.

Arguments:

    DriverObject - Pointer to the driver object created by the I/O manager.

    RegistryPath - Pointer to the driver specific key
                   \Registry
                      \Machine
                         \System
                            \CurrentControlSet
                               \Services
                                  \<DriverName>

Return Value:

    NTSTATUS

--*/
{

    NTSTATUS               status = STATUS_SUCCESS;
    WDF_DRIVER_CONFIG      config;
    WDFDRIVER              driver;
    WDF_OBJECT_ATTRIBUTES  attributes,LockAttributes;
    PQUANTIS_DRIVER_EXTENSION drvExt;

    KdBreakPoint();

    //
    // Initialize WPP Tracing
    //
    WPP_INIT_TRACING( DriverObject, RegistryPath );

    KdPrint(("Quantis Number Generator - Driver Framework Edition.\n"));
    KdPrint(("Built %s %s\n", __DATE__, __TIME__));
	
    DoTraceMessage(QUANTIS_DEBUG, "Entering function 'DriverEntry'.");

	KdPrint(("DriverObject 0x%p\n", DriverObject));
    DoTraceMessage(QUANTIS_INFO, "DriverObject 0x%p", DriverObject);

    DoTraceMessage(QUANTIS_INFO,"Built %s %s", __DATE__, __TIME__);
    //
    // Initialize the Driver Config structure..
    //
    WDF_DRIVER_CONFIG_INIT( &config, QuantisEvtDeviceAdd );

    //
    // Register a cleanup callback so that we can call WPP_CLEANUP when
    // the framework driver object is deleted during driver unload.
    //
    WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&attributes,QUANTIS_DRIVER_EXTENSION);

    attributes.EvtCleanupCallback = QuantisEvtDriverContextCleanup;

    status = WdfDriverCreate( DriverObject,
                              RegistryPath,
                              &attributes,
                              &config,
                              &driver );

    if (!NT_SUCCESS(status)) {
	    KdPrint(("WdfDriverCreate failed with status %!STATUS!\n", status));
        DoTraceMessage(QUANTIS_ERROR,
                    "WdfDriverCreate failed with status %!STATUS!\n", status);
        //
        // Cleanup tracing here because DriverContextCleanup will not be called
        // as we have failed to create WDFDRIVER object itself.
        // Please note that if your return failure from DriverEntry after the
        // WDFDRIVER object is created successfully, you don't have to
        // call WPP cleanup because in those cases DriverContextCleanup
        // will be executed when the framework deletes the DriverObject.
        //
        WPP_CLEANUP(DriverObject);
    }

    drvExt=QuantisGetDrvExt(driver);
    /* Initialisation of driver's global variables */
    drvExt->NumberOfCards=0;
   
    WDF_OBJECT_ATTRIBUTES_INIT(&LockAttributes);
    /* By setting the parent to be the driver, the spin lock will be deleted
       when the driver will be deleted. */
    LockAttributes.ParentObject=driver;    
    status = WdfSpinLockCreate(&LockAttributes,&drvExt->NbCardsLock);
    if(!NT_SUCCESS(status)){
        KdPrint(("Failed to create lock for counting Quantis cards %!STATUS!\n", status));
        DoTraceMessage(QUANTIS_ERROR,
                    "Failed to create lock for counting Quantis cards %!STATUS!\n", status);
        return status;
    }
    
	KdPrint(("Exiting normally function 'DriverEntry'.\n"));
    DoTraceMessage(QUANTIS_DEBUG,"Exiting normally function 'DriverEntry'.");
    
    return status;
}


VOID
QuantisEvtDriverContextCleanup(
    IN WDFDRIVER Driver
    )
/*++
Routine Description:

    Free all the resources allocated in DriverEntry.

Arguments:

    Driver - handle to a WDF Driver object.

Return Value:

    VOID.

--*/
{   
    PAGED_CODE ();
    
    KdPrint(("CommonEvtDriverContextCleanup Entered\n"));
    DoTraceMessage(QUANTIS_DEBUG,"CommonEvtDriverContextCleanup Entered");

    //
    // Stop WPP Tracing
    //
    WPP_CLEANUP( WdfDriverWdmGetDriverObject( Driver ) );

    KdPrint(("CommonEvtDriverContextCleanup Exited normally\n"));
    DoTraceMessage(QUANTIS_DEBUG,"CommonEvtDriverContextCleanup Exited normally");
}

QUANTIS_EXTERN_C_END

