package net.sf.saxon.instruct;
import net.sf.saxon.Configuration;
import net.sf.saxon.Err;
import net.sf.saxon.event.ReceiverOptions;
import net.sf.saxon.event.SequenceReceiver;
import net.sf.saxon.expr.*;
import net.sf.saxon.functions.StringFn;
import net.sf.saxon.functions.SystemFunction;
import net.sf.saxon.om.Item;
import net.sf.saxon.om.Orphan;
import net.sf.saxon.om.StandardNames;
import net.sf.saxon.pattern.NodeKindTest;
import net.sf.saxon.trans.StaticError;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.type.*;
import net.sf.saxon.value.Value;
import net.sf.saxon.value.Whitespace;

import java.io.PrintStream;

/**
* An xsl:value-of element in the stylesheet. <br>
* The xsl:value-of element takes attributes:<ul>
* <li>a mandatory attribute select="expression".
* This must be a valid String expression</li>
* <li>an optional disable-output-escaping attribute, value "yes" or "no"</li>
* <li>an optional separator attribute. This is handled at compile-time: if the separator attribute
* is present, the select expression passed in here will be a call to the string-join() function.</li>
* </ul>
*/

public final class ValueOf extends SimpleNodeConstructor {

    private int options;
    private boolean isNumberingInstruction = false;  // set to true if generated by xsl:number
    private boolean noNodeIfEmpty;

    public ValueOf(Expression select, boolean disable, boolean noNodeIfEmpty) {
        this.select = select;
        this.options = (disable ? ReceiverOptions.DISABLE_ESCAPING : 0);
        this.noNodeIfEmpty = noNodeIfEmpty;
        adoptChildExpression(select);

        // If value is fixed, test whether there are any special characters that might need to be
        // escaped when the time comes for serialization
        if (select instanceof StringLiteral) {
            boolean special = false;
            CharSequence val = ((StringLiteral)select).getStringValue();
            for (int k=0; k<val.length(); k++) {
                char c = val.charAt(k);
                if ((int)c<33 || (int)c>126 ||
                         c=='<' || c=='>' || c=='&') {
                    special = true;
                    break;
                 }
            }
            if (!special) {
                this.options |= ReceiverOptions.NO_SPECIAL_CHARS;
            }
        }
    }

    /**
     * Indicate that this is really an xsl:nunber instruction
     */

    public void setIsNumberingInstruction() {
        isNumberingInstruction = true;
    }

    /**
     * Determine whether this is really an xsl:number instruction
     */

    public boolean isNumberingInstruction() {
        return isNumberingInstruction;
    }

    /**
    * Get the name of this instruction for diagnostic and tracing purposes
    */

    public int getInstructionNameCode() {
        if (isNumberingInstruction) {
            return StandardNames.XSL_NUMBER;
        } else if (select instanceof StringLiteral) {
            return StandardNames.XSL_TEXT;
        } else {
            return StandardNames.XSL_VALUE_OF;
        }
    }

    /**
    * Offer promotion for subexpressions. The offer will be accepted if the subexpression
    * is not dependent on the factors (e.g. the context item) identified in the PromotionOffer.
    * By default the offer is not accepted - this is appropriate in the case of simple expressions
    * such as constant values and variable references where promotion would give no performance
    * advantage. This method is always called at compile time.
    *
    * @param offer details of the offer, for example the offer to move
    *     expressions that don't depend on the context to an outer level in
    *     the containing expression
    * @exception XPathException if any error is detected
    */

    protected void promoteInst(PromotionOffer offer) throws XPathException {
        super.promoteInst(offer);
    }

    public ItemType getItemType(TypeHierarchy th) {
        return NodeKindTest.TEXT;
    }

    public int computeCardinality() {
        if (noNodeIfEmpty) {
            return StaticProperty.ALLOWS_ZERO_OR_ONE;
        } else {
            return StaticProperty.EXACTLY_ONE;
        }
    }

    public void localTypeCheck(StaticContext env, ItemType contextItemType) {

    }

    /**
      * Check statically that the results of the expression are capable of constructing the content
      * of a given schema type.
      *
      * @param parentType The schema type
      * @param env        the static context
      * @param whole
      * @throws net.sf.saxon.trans.XPathException
      *          if the expression doesn't match the required content type
      */

     public void checkPermittedContents(SchemaType parentType, StaticContext env, boolean whole) throws XPathException {
         // if the expression is a constant value, check that it is valid for the type
         if (select instanceof Literal) {
             Value selectValue = ((Literal)select).getValue();
             SimpleType stype = null;
             if (parentType instanceof SimpleType && whole) {
                 stype = (SimpleType)parentType;
             } else if (parentType instanceof ComplexType && ((ComplexType)parentType).isSimpleContent()) {
                 stype = ((ComplexType)parentType).getSimpleContentType();
             }
             if (whole && stype != null && !stype.isNamespaceSensitive()) {
                        // Can't validate namespace-sensitive content statically
                 XPathException err = stype.validateContent(
                         selectValue.getStringValue(), null, env.getConfiguration().getNameChecker());
                 if (err != null) {
                     err.setLocator(this);
                     throw err;
                 }
                 return;
             }
             if (parentType instanceof ComplexType &&
                     !((ComplexType)parentType).isSimpleContent() &&
                     !((ComplexType)parentType).isMixedContent() &&
                     !Whitespace.isWhite(selectValue.getStringValue())) {
                 StaticError err = new StaticError("Complex type " + parentType.getDescription() +
                         " does not allow text content " +
                         Err.wrap(selectValue.getStringValue()));
                 err.setLocator(this);
                 err.setIsTypeError(true);
                 throw err;
             }
         }
    }

    /**
     * Convert this value-of instruction to an expression that delivers the string-value of the resulting
     * text node. This will often be a call on the string-join function.
     */

    public Expression convertToStringJoin(StaticContext env) {
        if (select.getItemType(env.getConfiguration().getTypeHierarchy()).equals(BuiltInAtomicType.UNTYPED_ATOMIC)) {
            return select;
        } else {
            StringFn fn = (StringFn) SystemFunction.makeSystemFunction("string", 2, env.getNamePool());
            Expression[] args = new Expression[1];
            args[0] = select;
            fn.setArguments(args);
            CastExpression cast = new CastExpression(fn, BuiltInAtomicType.UNTYPED_ATOMIC, false);
            return cast;
        }
    }


    /**
     * Process this instruction, sending the resulting text node to the current output destination
     * @param context
     * @return Always returns null
     * @throws XPathException
     */
    public TailCall processLeavingTail(XPathContext context) throws XPathException {
        SequenceReceiver out = context.getReceiver();
        Item item = select.evaluateItem(context);
        if (item != null) {
            out.characters(item.getStringValueCS(), locationId, options);
        }
        return null;
    }

    /**
     * Evaluate this expression, returning the resulting text node to the caller
     * @param context
     * @return the parentless text node that results from evaluating this instruction, or null to
     * represent an empty sequence
     * @throws XPathException
     */

    public Item evaluateItem(XPathContext context) throws XPathException {
        try {
            CharSequence val;
            Item item = select.evaluateItem(context);
            if (item == null) {
                if (noNodeIfEmpty) {
                    return null;
                } else {
                    val = "";
                }
            } else {
                val = item.getStringValueCS();
            }
            Orphan o = new Orphan(context.getController().getConfiguration());
            o.setNodeKind(Type.TEXT);
            o.setStringValue(val);
            return o;
        } catch (XPathException err) {
            if (err.getLocator() == null) {
                err.setLocator(this);
            }
            throw err;
        }
    }

    /**
     * Display this instruction as an expression, for diagnostics
     */

    public void display(int level, PrintStream out, Configuration config) {
        out.println(ExpressionTool.indent(level) + "value-of");
        select.display(level+1, out, config);
    }
}

//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay.
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved.
//
// Contributor(s): none.
//
