/*
 * PlayStation CPU emulator by smf
 *
 * GTE divider reverse engineering by pSXAuthor.
 *
 * Licensed to the MAME Team for distribution under the MAME license.
 *
 * Known chip id's
 *   CXD8530AQ
 *   CXD8530BQ
 *   CXD8530CQ
 *   CXD8661R
 *   CXD8606BQ
 *   CXD8606CQ
 *
 * The PlayStation CPU is based on the LSI LR33300.
 *
 * Differences from the LR33300:
 *
 *  There is only 1k of data cache ram ( the LR33300 has 2k )
 *
 *  There is no data cache tag ram, so the data cache ram can only be used as a fast area
 *  of ram ( which is a standard LR33300 feature ).
 *
 *  If COP0 is disabled in user mode you get a coprocessor unusable exception, while
 *  the LR33300 is documented to generate a reserved instruction exception.
 *
 * Known limitations of the emulation:
 *
 *  Only read & write break points are emulated, trace and program counter breakpoints are not.
 *
 *  Load/Store timings are based on load scheduling turned off & no write cache. This affects when
 *  bus error exceptions occur and also when the read & write handlers are called. A scheduled
 *  load will complete if a load breakpoint fires, but an unscheduled load will not.
 *
 *  Reading from the data and instruction cache at the same time causes a bus conflict that
 *  corrupts the data in a reliable but strange way, which is not emulated.
 *
 *  Values written to COP1 & COP3 can be read back by the next instruction, which is not emulated.
 *  Because of loadscheduling the value loaded with LWC1/LWC3 can be read by more than the next
 *  instruction.
 *
 *  SWC0 writes stale data from a previous operation, this is only partially emulated as the timing
 *  is complicated. Left over instruction fetches are currently emulated as they are the most
 *  'interesting' and have no impact on the rest of the emulation.
 *
 *  MTC0 timing is not emulated, switching to user mode while in kernel space continues
 *  execution for another two instructions before taking an exception. Using RFE to do the same
 *  thing causes the exception straight away, unless the RFE is the first instructio that follows
 *  an MTC0 instruction.
 *
 *  The PRId register should be 1 on some revisions of the CPU ( there might be other values too ).
 *
 *  Moving to the HI/LO register after a multiply or divide, but before reading the results will
 *  always abort the operation as if you did it immediately. In reality it should complete on it's
 *  own, and aborting before it completes would result in returning the working results.
 *
 *  Running code in cached address space does not use or update the instruction cache.
 *
 *  Wait states are not emulated.
 *
 *  Bus errors caused by instruction fetches are not supported.
 *
 */

#include "debugger.h"
#include "psx.h"

#define LOG_BIOSCALL ( 0 )

#define EXC_INT ( 0 )
#define EXC_ADEL ( 4 )
#define EXC_ADES ( 5 )
#define EXC_DBE ( 7 )
#define EXC_SYS ( 8 )
#define EXC_BP ( 9 )
#define EXC_RI ( 10 )
#define EXC_CPU ( 11 )
#define EXC_OVF ( 12 )

#define CP0_INDEX ( 0 )
#define CP0_RANDOM ( 1 )
#define CP0_ENTRYLO ( 2 )
#define CP0_CONTEXT ( 4 )
#define CP0_ENTRYHI ( 10 )

#define CP0_BPC ( 3 )
#define CP0_BDA ( 5 )
#define CP0_TAR ( 6 )
#define CP0_DCIC ( 7 )
#define CP0_BADA ( 8 )
#define CP0_BDAM ( 9 )
#define CP0_BPCM ( 11 )
#define CP0_SR ( 12 )
#define CP0_CAUSE ( 13 )
#define CP0_EPC ( 14 )
#define CP0_PRID ( 15 )

#define DCIC_STATUS ( 0x3f )
#define DCIC_DB ( 1L << 0 )
#define DCIC_DA ( 1L << 2 )
#define DCIC_R ( 1L << 3 )
#define DCIC_W ( 1L << 4 )
#define DCIC_DE ( 1L << 23 )
#define DCIC_DAE ( 1L << 25 )
#define DCIC_DR ( 1L << 26 )
#define DCIC_DW ( 1L << 27 )
#define DCIC_KD ( 1L << 29 )
#define DCIC_UD ( 1L << 30 )
#define DCIC_TR ( 1L << 31 )

#define SR_IEC ( 1L << 0 )
#define SR_KUC ( 1L << 1 )
#define SR_ISC ( 1L << 16 )
#define SR_SWC ( 1L << 17 )
#define SR_BEV ( 1L << 22 )
#define SR_CU0 ( 1L << 28 )
#define SR_CU1 ( 1L << 29 )
#define SR_CU2 ( 1L << 30 )
#define SR_CU3 ( 1L << 31 )

#define CAUSE_EXC ( 31L << 2 )
#define CAUSE_IP ( 255L << 8 )
#define CAUSE_IP2 ( 1L << 10 )
#define CAUSE_IP3 ( 1L << 11 )
#define CAUSE_IP4 ( 1L << 12 )
#define CAUSE_IP5 ( 1L << 13 )
#define CAUSE_IP6 ( 1L << 14 )
#define CAUSE_IP7 ( 1L << 15 )
#define CAUSE_CE ( 3L << 28 )
#define CAUSE_BT ( 1L << 30 )
#define CAUSE_BD ( 1L << 31 )

#define BIU_LOCK ( 0x00000001 )
#define BIU_INV  ( 0x00000002 )
#define BIU_TAG  ( 0x00000004 )
#define BIU_RAM  ( 0x00000008 )
#define BIU_DS   ( 0x00000080 )
#define BIU_IS1  ( 0x00000800 )

#define ICACHE_ENTRIES ( 0x400 )
#define DCACHE_ENTRIES ( 0x100 )

#define TAG_MATCH_MASK ( 0 - ( ICACHE_ENTRIES * 4 ) )
#define TAG_MATCH ( 0x10 )
#define TAG_VALID ( 0x0f )

#define MULTIPLIER_OPERATION_IDLE ( 0 )
#define MULTIPLIER_OPERATION_MULT ( 1 )
#define MULTIPLIER_OPERATION_MULTU ( 2 )
#define MULTIPLIER_OPERATION_DIV ( 3 )
#define MULTIPLIER_OPERATION_DIVU ( 4 )

static const char *const delayn[] =
{
	"",   "at", "v0", "v1", "a0", "a1", "a2", "a3",
	"t0", "t1", "t2", "t3", "t4", "t5", "t6", "t7",
	"s0", "s1", "s2", "s3", "s4", "s5", "s6", "s7",
	"t8", "t9", "k0", "k1", "gp", "sp", "fp", "ra",
	"pc", "!pc"
};

typedef struct _psxcpu_state psxcpu_state;

struct _psxcpu_state
{
	int icount;
	UINT32 op;
	UINT32 pc;
	UINT32 delayv;
	UINT32 delayr;
	UINT32 hi;
	UINT32 lo;
	UINT32 biu;
	UINT32 berr;
	UINT32 r[ 32 ];
	UINT32 cp0r[ 16 ];
	PAIR cp2cr[ 32 ];
	PAIR cp2dr[ 32 ];
	UINT32 icacheTag[ ICACHE_ENTRIES / 4 ];
	UINT32 icache[ ICACHE_ENTRIES ];
	UINT32 dcache[ DCACHE_ENTRIES ];
	int multiplier_operation;
	UINT32 multiplier_operand1;
	UINT32 multiplier_operand2;
	cpu_irq_callback irq_callback;
	const device_config *device;
	const address_space *program;
	int bus_attached;
	UINT32 bad_byte_address_mask;
	UINT32 bad_half_address_mask;
	UINT32 bad_word_address_mask;
};

INLINE psxcpu_state *get_safe_token(const device_config *device)
{
	assert(device != NULL);
	assert(device->token != NULL);
	assert(device->type == CPU);
	assert(cpu_get_type(device) == CPU_PSXCPU ||
		   cpu_get_type(device) == CPU_CXD8661R);
	return (psxcpu_state *)device->token;
}

static const UINT32 mips_mtc0_writemask[]=
{
	0x00000000, /* !INDEX */
	0x00000000, /* !RANDOM */
	0x00000000, /* !ENTRYLO */
	0xffffffff, /* BPC */
	0x00000000, /* !CONTEXT */
	0xffffffff, /* BDA */
	0x00000000, /* TAR */
	0xff80f03f, /* DCIC */
	0x00000000, /* BADA */
	0xffffffff, /* BDAM */
	0x00000000, /* !ENTRYHI */
	0xffffffff, /* BPCM */
	0xf04fff3f, /* SR */
	0x00000300, /* CAUSE */
	0x00000000, /* EPC */
	0x00000000  /* PRID */
};

#if 0
void ATTR_PRINTF(1,2) GTELOG(const char *a,...)
{
	va_list va;
	char s_text[ 1024 ];
	va_start( va, a );
	vsprintf( s_text, a, va );
	va_end( va );
	logerror( "%08x: GTE: %08x %s\n", psxcpu->pc, INS_COFUN( psxcpu->op ), s_text );
}
#else
INLINE void ATTR_PRINTF(1,2) GTELOG(const char *a, ...) {}
#endif

static UINT32 getcp1dr( psxcpu_state *psxcpu, int reg );
static void setcp1dr( psxcpu_state *psxcpu, int reg, UINT32 value );
static UINT32 getcp1cr( psxcpu_state *psxcpu, int reg );
static void setcp1cr( psxcpu_state *psxcpu, int reg, UINT32 value );
//static void docop1( int op );


static UINT32 getcp2dr( psxcpu_state *psxcpu, int reg );
static void setcp2dr( psxcpu_state *psxcpu, int reg, UINT32 value );
static UINT32 getcp2cr( psxcpu_state *psxcpu, int reg );
static void setcp2cr( psxcpu_state *psxcpu, int reg, UINT32 value );
static void docop2( psxcpu_state *psxcpu, int op );


static UINT32 getcp3dr( psxcpu_state *psxcpu, int reg );
static void setcp3dr( psxcpu_state *psxcpu, int reg, UINT32 value );
static UINT32 getcp3cr( psxcpu_state *psxcpu, int reg );
static void setcp3cr( psxcpu_state *psxcpu, int reg, UINT32 value );
//static void docop3( int op );


static void mips_exception( psxcpu_state *psxcpu, int exception );
static void mips_load_bad_address( psxcpu_state *psxcpu, UINT32 address );

static void mips_stop( psxcpu_state *psxcpu )
{
	debugger_break( psxcpu->program->machine );
	debugger_instruction_hook( psxcpu->program->cpu,  psxcpu->pc );
}

static UINT32 mips_cache_readword( psxcpu_state *psxcpu, UINT32 offset )
{
	UINT32 data = 0;

	if( ( psxcpu->biu & BIU_TAG ) != 0 )
	{
		if( ( psxcpu->biu & BIU_IS1 ) != 0 )
		{
			UINT32 tag = psxcpu->icacheTag[ ( offset / 16 ) % ( ICACHE_ENTRIES / 4 ) ];
			data |= tag & TAG_VALID;

			if( ( ( tag ^ offset ) & TAG_MATCH_MASK ) == 0 )
			{
				data |= TAG_MATCH;
			}
		}
	}
	else if( ( psxcpu->biu & ( BIU_LOCK | BIU_INV ) ) != 0 )
	{
	}
	else
	{
		if( ( psxcpu->biu & BIU_IS1 ) == BIU_IS1 )
		{
			data |= psxcpu->icache[ ( offset / 4 ) % ICACHE_ENTRIES ];
		}

		if( ( psxcpu->biu & BIU_DS ) == BIU_DS )
		{
			data |= psxcpu->dcache[ ( offset / 4 ) % DCACHE_ENTRIES ];
		}
	}

	return data;
}

static void mips_cache_writeword( psxcpu_state *psxcpu, UINT32 offset, UINT32 data )
{
	if( ( psxcpu->biu & BIU_TAG ) != 0 )
	{
		if( ( psxcpu->biu & BIU_IS1 ) != 0 )
		{
			psxcpu->icacheTag[ ( offset / 16 ) % ( ICACHE_ENTRIES / 4 ) ] = ( data & TAG_VALID ) | ( offset & TAG_MATCH_MASK );
		}
	}
	else if( ( psxcpu->biu & ( BIU_LOCK | BIU_INV ) ) != 0 )
	{
		if( ( psxcpu->biu & BIU_IS1 ) != 0 )
		{
			psxcpu->icacheTag[ ( offset / 16 ) % ( ICACHE_ENTRIES / 4 ) ] = ( offset & TAG_MATCH_MASK );
		}
	}
	else
	{
		if( ( psxcpu->biu & BIU_IS1 ) != 0 )
		{
			psxcpu->icache[ ( offset / 4 ) % ICACHE_ENTRIES ] = data;
		}

		if( ( psxcpu->biu & BIU_DS ) != 0 )
		{
			psxcpu->dcache[ ( offset / 4 ) % DCACHE_ENTRIES ] = data;
		}
	}
}

INLINE UINT8 psx_readbyte( psxcpu_state *psxcpu, UINT32 address )
{
	if( psxcpu->bus_attached )
	{
		return memory_read_byte_32le( psxcpu->program, address );
	}
	else
	{
		return mips_cache_readword( psxcpu, address ) >> ( ( address & 3 ) * 8 );
	}
}

INLINE UINT16 psx_readhalf( psxcpu_state *psxcpu, UINT32 address )
{
	if( psxcpu->bus_attached )
	{
		return memory_read_word_32le( psxcpu->program, address );
	}
	else
	{
		return mips_cache_readword( psxcpu, address ) >> ( ( address & 2 ) * 8 );
	}
}

INLINE UINT32 psx_readword( psxcpu_state *psxcpu, UINT32 address )
{
	if( psxcpu->bus_attached )
	{
		return memory_read_dword_32le( psxcpu->program, address );
	}
	else
	{
		return mips_cache_readword( psxcpu, address );
	}
}

INLINE UINT32 psx_readword_masked( psxcpu_state *psxcpu, UINT32 address, UINT32 mask )
{
	if( psxcpu->bus_attached )
	{
		return memory_read_dword_masked_32le( psxcpu->program, address, mask );
	}
	else
	{
		return mips_cache_readword( psxcpu, address );
	}
}

INLINE void psx_writeword( psxcpu_state *psxcpu, UINT32 address, UINT32 data )
{
	if( psxcpu->bus_attached )
	{
		memory_write_dword_32le( psxcpu->program, address, data );
	}
	else
	{
		mips_cache_writeword( psxcpu, address, data );
	}
}

INLINE void psx_writeword_masked( psxcpu_state *psxcpu, UINT32 address, UINT32 data, UINT32 mask )
{
	if( psxcpu->bus_attached )
	{
		memory_write_dword_masked_32le( psxcpu->program, address, data, mask );
	}
	else
	{
		mips_cache_writeword( psxcpu, address, data );
	}
}


static const struct
{
	int address;
	int operation;
	const char *prototype;
} bioscalls[] =
{
	{ 0xa0, 0x00, "int open(const char *name, int mode)" },
	{ 0xa0, 0x01, "int lseek(int fd, int offset, int whence)" },
	{ 0xa0, 0x02, "int read(int fd, void *buf, int nbytes)" },
	{ 0xa0, 0x03, "int write(int fd, void *buf, int nbytes)" },
	{ 0xa0, 0x04, "int close(int fd)" },
	{ 0xa0, 0x05, "int ioctl(int fd, int cmd, int arg)" },
	{ 0xa0, 0x06, "void exit(int code)" },
	{ 0xa0, 0x07, "sys_a0_07()" },
	{ 0xa0, 0x08, "char getc(int fd)" },
	{ 0xa0, 0x09, "void putc(char c, int fd)" },
	{ 0xa0, 0x0a, "todigit()" },
	{ 0xa0, 0x0b, "double atof(const char *s)" },
	{ 0xa0, 0x0c, "long strtoul(const char *s, char **ptr, int base)" },
	{ 0xa0, 0x0d, "unsigned long strtol(const char *s, char **ptr, int base)" },
	{ 0xa0, 0x0e, "int abs(int val)" },
	{ 0xa0, 0x0f, "long labs(long lval)" },
	{ 0xa0, 0x10, "long atoi(const char *s)" },
	{ 0xa0, 0x11, "int atol(const char *s)" },
	{ 0xa0, 0x12, "atob()" },
	{ 0xa0, 0x13, "int setjmp(jmp_buf *ctx)" },
	{ 0xa0, 0x14, "void longjmp(jmp_buf *ctx, int value)" },
	{ 0xa0, 0x15, "char *strcat(char *dst, const char *src)" },
	{ 0xa0, 0x16, "char *strncat(char *dst, const char *src, size_t n)" },
	{ 0xa0, 0x17, "int strcmp(const char *dst, const char *src)" },
	{ 0xa0, 0x18, "int strncmp(const char *dst, const char *src, size_t n)" },
	{ 0xa0, 0x19, "char *strcpy(char *dst, const char *src)" },
	{ 0xa0, 0x1a, "char *strncpy(char *dst, const char *src, size_t n)" },
	{ 0xa0, 0x1b, "size_t strlen(const char *s)" },
	{ 0xa0, 0x1c, "int index(const char *s, int c)" },
	{ 0xa0, 0x1d, "int rindex(const char *s, int c)" },
	{ 0xa0, 0x1e, "char *strchr(const char *s, int c)" },
	{ 0xa0, 0x1f, "char *strrchr(const char *s, int c)" },
	{ 0xa0, 0x20, "char *strpbrk(const char *dst, const char *src)" },
	{ 0xa0, 0x21, "size_t strspn(const char *s, const char *set)" },
	{ 0xa0, 0x22, "size_t strcspn(const char *s, const char *set)" },
	{ 0xa0, 0x23, "char *strtok(char *s, const char *set)" },
	{ 0xa0, 0x24, "char *strstr(const char *s, const char *set)" },
	{ 0xa0, 0x25, "int toupper(int c)" },
	{ 0xa0, 0x26, "int tolower(int c)" },
	{ 0xa0, 0x27, "void bcopy(const void *src, void *dst, size_t len)" },
	{ 0xa0, 0x28, "void bzero(void *ptr, size_t len)" },
	{ 0xa0, 0x29, "int bcmp(const void *ptr1, const void *ptr2, int len)" },
	{ 0xa0, 0x2a, "void *memcpy(void *dst, const void *src, size_t n)" },
	{ 0xa0, 0x2b, "void *memset(void *dst, char c, size_t n)" },
	{ 0xa0, 0x2c, "void *memmove(void *dst, const void *src, size_t n)" },
	{ 0xa0, 0x2d, "int memcmp(const void *dst, const void *src, size_t n)" },
	{ 0xa0, 0x2e, "void *memchr(const void *s, int c, size_t n)" },
	{ 0xa0, 0x2f, "int rand()" },
	{ 0xa0, 0x30, "void srand(unsigned int seed)" },
	{ 0xa0, 0x31, "void qsort(void *base, int nel, int width, int (*cmp)(void *, void *))" },
	{ 0xa0, 0x32, "double strtod(const char *s, char **endptr)" },
	{ 0xa0, 0x33, "void *malloc(int size)" },
	{ 0xa0, 0x34, "void free(void *buf)" },
	{ 0xa0, 0x35, "void *lsearch(void *key, void *base, int belp, int width, int (*cmp)(void *, void *))" },
	{ 0xa0, 0x36, "void *bsearch(void *key, void *base, int nel, int size, int (*cmp)(void *, void *))" },
	{ 0xa0, 0x37, "void *calloc(int size, int n)" },
	{ 0xa0, 0x38, "void *realloc(void *buf, int n)" },
	{ 0xa0, 0x39, "InitHeap(void *block, int size)" },
	{ 0xa0, 0x3a, "void _exit(int code)" },
	{ 0xa0, 0x3b, "char getchar(void)" },
	{ 0xa0, 0x3c, "void putchar(char c)" },
	{ 0xa0, 0x3d, "char *gets(char *s)" },
	{ 0xa0, 0x3e, "void puts(const char *s)" },
	{ 0xa0, 0x3f, "int printf(const char *fmt, ...)" },
	{ 0xa0, 0x40, "sys_a0_40()" },
	{ 0xa0, 0x41, "int LoadTest(const char *name, struct EXEC *header)" },
	{ 0xa0, 0x42, "int Load(const char *name, struct EXEC *header)" },
	{ 0xa0, 0x43, "int Exec(struct EXEC *header, int argc, char **argv)" },
	{ 0xa0, 0x44, "void FlushCache()" },
	{ 0xa0, 0x45, "void InstallInterruptHandler()" },
	{ 0xa0, 0x46, "GPU_dw(int x, int y, int w, int h, long *data)" },
	{ 0xa0, 0x47, "mem2vram(int x, int y, int w, int h, long *data)" },
	{ 0xa0, 0x48, "SendGPU(int status)" },
	{ 0xa0, 0x49, "GPU_cw(long cw)" },
	{ 0xa0, 0x4a, "GPU_cwb(long *pkt, int len)" },
	{ 0xa0, 0x4b, "SendPackets(void *ptr)" },
	{ 0xa0, 0x4c, "sys_a0_4c()" },
	{ 0xa0, 0x4d, "int GetGPUStatus()" },
	{ 0xa0, 0x4e, "GPU_sync()" },
	{ 0xa0, 0x4f, "sys_a0_4f()" },
	{ 0xa0, 0x50, "sys_a0_50()" },
	{ 0xa0, 0x51, "int LoadExec(const char *name, int, int)" },
	{ 0xa0, 0x52, "GetSysSp()" },
	{ 0xa0, 0x53, "sys_a0_53()" },
	{ 0xa0, 0x54, "_96_init()" },
	{ 0xa0, 0x55, "_bu_init()" },
	{ 0xa0, 0x56, "_96_remove()" },
	{ 0xa0, 0x57, "sys_a0_57()" },
	{ 0xa0, 0x58, "sys_a0_58()" },
	{ 0xa0, 0x59, "sys_a0_59()" },
	{ 0xa0, 0x5a, "sys_a0_5a()" },
	{ 0xa0, 0x5b, "dev_tty_init()" },
	{ 0xa0, 0x5c, "dev_tty_open()" },
	{ 0xa0, 0x5d, "dev_tty_5d()" },
	{ 0xa0, 0x5e, "dev_tty_ioctl()" },
	{ 0xa0, 0x5f, "dev_cd_open()" },
	{ 0xa0, 0x60, "dev_cd_read()" },
	{ 0xa0, 0x61, "dev_cd_close()" },
	{ 0xa0, 0x62, "dev_cd_firstfile()" },
	{ 0xa0, 0x63, "dev_cd_nextfile()" },
	{ 0xa0, 0x64, "dev_cd_chdir()" },
	{ 0xa0, 0x65, "dev_card_open()" },
	{ 0xa0, 0x66, "dev_card_read()" },
	{ 0xa0, 0x67, "dev_card_write()" },
	{ 0xa0, 0x68, "dev_card_close()" },
	{ 0xa0, 0x69, "dev_card_firstfile()" },
	{ 0xa0, 0x6a, "dev_card_nextfile()" },
	{ 0xa0, 0x6b, "dev_card_erase()" },
	{ 0xa0, 0x6c, "dev_card_undelete()" },
	{ 0xa0, 0x6d, "dev_card_format()" },
	{ 0xa0, 0x6e, "dev_card_rename()" },
	{ 0xa0, 0x6f, "dev_card_6f()" },
	{ 0xa0, 0x70, "_bu_init()" },
	{ 0xa0, 0x71, "_96_init()" },
	{ 0xa0, 0x72, "_96_remove()" },
	{ 0xa0, 0x73, "sys_a0_73()" },
	{ 0xa0, 0x74, "sys_a0_74()" },
	{ 0xa0, 0x75, "sys_a0_75()" },
	{ 0xa0, 0x76, "sys_a0_76()" },
	{ 0xa0, 0x77, "sys_a0_77()" },
	{ 0xa0, 0x78, "_96_CdSeekL()" },
	{ 0xa0, 0x79, "sys_a0_79()" },
	{ 0xa0, 0x7a, "sys_a0_7a()" },
	{ 0xa0, 0x7b, "sys_a0_7b()" },
	{ 0xa0, 0x7c, "_96_CdGetStatus()" },
	{ 0xa0, 0x7d, "sys_a0_7d()" },
	{ 0xa0, 0x7e, "_96_CdRead()" },
	{ 0xa0, 0x7f, "sys_a0_7f()" },
	{ 0xa0, 0x80, "sys_a0_80()" },
	{ 0xa0, 0x81, "sys_a0_81()" },
	{ 0xa0, 0x82, "sys_a0_82()" },
	{ 0xa0, 0x83, "sys_a0_83()" },
	{ 0xa0, 0x84, "sys_a0_84()" },
	{ 0xa0, 0x85, "_96_CdStop()" },
	{ 0xa0, 0x84, "sys_a0_84()" },
	{ 0xa0, 0x85, "sys_a0_85()" },
	{ 0xa0, 0x86, "sys_a0_86()" },
	{ 0xa0, 0x87, "sys_a0_87()" },
	{ 0xa0, 0x88, "sys_a0_88()" },
	{ 0xa0, 0x89, "sys_a0_89()" },
	{ 0xa0, 0x8a, "sys_a0_8a()" },
	{ 0xa0, 0x8b, "sys_a0_8b()" },
	{ 0xa0, 0x8c, "sys_a0_8c()" },
	{ 0xa0, 0x8d, "sys_a0_8d()" },
	{ 0xa0, 0x8e, "sys_a0_8e()" },
	{ 0xa0, 0x8f, "sys_a0_8f()" },
	{ 0xa0, 0x90, "sys_a0_90()" },
	{ 0xa0, 0x91, "sys_a0_91()" },
	{ 0xa0, 0x92, "sys_a0_92()" },
	{ 0xa0, 0x93, "sys_a0_93()" },
	{ 0xa0, 0x94, "sys_a0_94()" },
	{ 0xa0, 0x95, "sys_a0_95()" },
	{ 0xa0, 0x96, "AddCDROMDevice()" },
	{ 0xa0, 0x97, "AddMemCardDevice()" },
	{ 0xa0, 0x98, "DisableKernelIORedirection()" },
	{ 0xa0, 0x99, "EnableKernelIORedirection()" },
	{ 0xa0, 0x9a, "sys_a0_9a()" },
	{ 0xa0, 0x9b, "sys_a0_9b()" },
	{ 0xa0, 0x9c, "void SetConf(int Event, int TCB, int Stack)" },
	{ 0xa0, 0x9d, "void GetConf(int *Event, int *TCB, int *Stack)" },
	{ 0xa0, 0x9e, "sys_a0_9e()" },
	{ 0xa0, 0x9f, "void SetMem(int size)" },
	{ 0xa0, 0xa0, "_boot()" },
	{ 0xa0, 0xa1, "SystemError()" },
	{ 0xa0, 0xa2, "EnqueueCdIntr()" },
	{ 0xa0, 0xa3, "DequeueCdIntr()" },
	{ 0xa0, 0xa4, "sys_a0_a4()" },
	{ 0xa0, 0xa5, "ReadSector(int count, int sector, void *buffer)" },
	{ 0xa0, 0xa6, "get_cd_status()" },
	{ 0xa0, 0xa7, "bufs_cb_0()" },
	{ 0xa0, 0xa8, "bufs_cb_1()" },
	{ 0xa0, 0xa9, "bufs_cb_2()" },
	{ 0xa0, 0xaa, "bufs_cb_3()" },
	{ 0xa0, 0xab, "_card_info()" },
	{ 0xa0, 0xac, "_card_load()" },
	{ 0xa0, 0xad, "_card_auto()" },
	{ 0xa0, 0xae, "bufs_cb_4()" },
	{ 0xa0, 0xaf, "sys_a0_af()" },
	{ 0xa0, 0xb0, "sys_a0_b0()" },
	{ 0xa0, 0xb1, "sys_a0_b1()" },
	{ 0xa0, 0xb2, "do_a_long_jmp()" },
	{ 0xa0, 0xb3, "sys_a0_b3()" },
	{ 0xa0, 0xb4, "GetKernelInfo(int sub_function)" },
	{ 0xb0, 0x00, "SysMalloc()" },
	{ 0xb0, 0x01, "sys_b0_01()" },
	{ 0xb0, 0x02, "sys_b0_02()" },
	{ 0xb0, 0x03, "sys_b0_03()" },
	{ 0xb0, 0x04, "sys_b0_04()" },
	{ 0xb0, 0x05, "sys_b0_05()" },
	{ 0xb0, 0x06, "sys_b0_06()" },
	{ 0xb0, 0x07, "void DeliverEvent(u_long class, u_long event)" },
	{ 0xb0, 0x08, "long OpenEvent(u_long class, long spec, long mode, long (*func)())" },
	{ 0xb0, 0x09, "long CloseEvent(long event)" },
	{ 0xb0, 0x0a, "long WaitEvent(long event)" },
	{ 0xb0, 0x0b, "long TestEvent(long event)" },
	{ 0xb0, 0x0c, "long EnableEvent(long event)" },
	{ 0xb0, 0x0d, "long DisableEvent(long event)" },
	{ 0xb0, 0x0e, "OpenTh()" },
	{ 0xb0, 0x0f, "CloseTh()" },
	{ 0xb0, 0x10, "ChangeTh()" },
	{ 0xb0, 0x11, "sys_b0_11()" },
	{ 0xb0, 0x12, "int InitPAD(char *buf1, int len1, char *buf2, int len2)" },
	{ 0xb0, 0x13, "int StartPAD(void)" },
	{ 0xb0, 0x14, "int StopPAD(void)" },
	{ 0xb0, 0x15, "PAD_init(u_long nazo, u_long *pad_buf)" },
	{ 0xb0, 0x16, "u_long PAD_dr()" },
	{ 0xb0, 0x17, "void ReturnFromException(void)" },
	{ 0xb0, 0x18, "ResetEntryInt()" },
	{ 0xb0, 0x19, "HookEntryInt()" },
	{ 0xb0, 0x1a, "sys_b0_1a()" },
	{ 0xb0, 0x1b, "sys_b0_1b()" },
	{ 0xb0, 0x1c, "sys_b0_1c()" },
	{ 0xb0, 0x1d, "sys_b0_1d()" },
	{ 0xb0, 0x1e, "sys_b0_1e()" },
	{ 0xb0, 0x1f, "sys_b0_1f()" },
	{ 0xb0, 0x20, "UnDeliverEvent(int class, int event)" },
	{ 0xb0, 0x21, "sys_b0_21()" },
	{ 0xb0, 0x22, "sys_b0_22()" },
	{ 0xb0, 0x23, "sys_b0_23()" },
	{ 0xb0, 0x24, "sys_b0_24()" },
	{ 0xb0, 0x25, "sys_b0_25()" },
	{ 0xb0, 0x26, "sys_b0_26()" },
	{ 0xb0, 0x27, "sys_b0_27()" },
	{ 0xb0, 0x28, "sys_b0_28()" },
	{ 0xb0, 0x29, "sys_b0_29()" },
	{ 0xb0, 0x2a, "sys_b0_2a()" },
	{ 0xb0, 0x2b, "sys_b0_2b()" },
	{ 0xb0, 0x2c, "sys_b0_2c()" },
	{ 0xb0, 0x2d, "sys_b0_2d()" },
	{ 0xb0, 0x2e, "sys_b0_2e()" },
	{ 0xb0, 0x2f, "sys_b0_2f()" },
	{ 0xb0, 0x2f, "sys_b0_30()" },
	{ 0xb0, 0x31, "sys_b0_31()" },
	{ 0xb0, 0x32, "int open(const char *name, int access)" },
	{ 0xb0, 0x33, "int lseek(int fd, long pos, int seektype)" },
	{ 0xb0, 0x34, "int read(int fd, void *buf, int nbytes)" },
	{ 0xb0, 0x35, "int write(int fd, void *buf, int nbytes)" },
	{ 0xb0, 0x36, "close(int fd)" },
	{ 0xb0, 0x37, "int ioctl(int fd, int cmd, int arg)" },
	{ 0xb0, 0x38, "exit(int exitcode)" },
	{ 0xb0, 0x39, "sys_b0_39()" },
	{ 0xb0, 0x3a, "char getc(int fd)" },
	{ 0xb0, 0x3b, "putc(int fd, char ch)" },
	{ 0xb0, 0x3c, "char getchar(void)" },
	{ 0xb0, 0x3d, "putchar(char ch)" },
	{ 0xb0, 0x3e, "char *gets(char *s)" },
	{ 0xb0, 0x3f, "puts(const char *s)" },
	{ 0xb0, 0x40, "int cd(const char *path)" },
	{ 0xb0, 0x41, "int format(const char *fs)" },
	{ 0xb0, 0x42, "struct DIRENTRY* firstfile(const char *name, struct DIRENTRY *dir)" },
	{ 0xb0, 0x43, "struct DIRENTRY* nextfile(struct DIRENTRY *dir)" },
	{ 0xb0, 0x44, "int rename(const char *oldname, const char *newname)" },
	{ 0xb0, 0x45, "int delete(const char *name)" },
	{ 0xb0, 0x46, "undelete()" },
	{ 0xb0, 0x47, "AddDevice()" },
	{ 0xb0, 0x48, "RemoveDevice()" },
	{ 0xb0, 0x49, "PrintInstalledDevices()" },
	{ 0xb0, 0x4a, "InitCARD()" },
	{ 0xb0, 0x4b, "StartCARD()" },
	{ 0xb0, 0x4c, "StopCARD()" },
	{ 0xb0, 0x4d, "sys_b0_4d()" },
	{ 0xb0, 0x4e, "_card_write()" },
	{ 0xb0, 0x4f, "_card_read()" },
	{ 0xb0, 0x50, "_new_card()" },
	{ 0xb0, 0x51, "void *Krom2RawAdd(int code)" },
	{ 0xb0, 0x52, "sys_b0_52()" },
	{ 0xb0, 0x53, "sys_b0_53()" },
	{ 0xb0, 0x54, "long _get_errno(void)" },
	{ 0xb0, 0x55, "long _get_error(long fd)" },
	{ 0xb0, 0x56, "GetC0Table()" },
	{ 0xb0, 0x57, "GetB0Table()" },
	{ 0xb0, 0x58, "_card_chan()" },
	{ 0xb0, 0x59, "sys_b0_59()" },
	{ 0xb0, 0x5a, "sys_b0_5a()" },
	{ 0xb0, 0x5b, "ChangeClearPAD(int, int)" },
	{ 0xb0, 0x5c, "_card_status()" },
	{ 0xb0, 0x5d, "_card_wait()" },
	{ 0xc0, 0x00, "InitRCnt()" },
	{ 0xc0, 0x01, "InitException()" },
	{ 0xc0, 0x02, "SysEnqIntRP(int index, long *queue)" },
	{ 0xc0, 0x03, "SysDeqIntRP(int index, long *queue)" },
	{ 0xc0, 0x04, "int get_free_EvCB_slot(void)" },
	{ 0xc0, 0x05, "get_free_TCB_slot()" },
	{ 0xc0, 0x06, "ExceptionHandler()" },
	{ 0xc0, 0x07, "InstallExceptionHandlers()" },
	{ 0xc0, 0x08, "SysInitMemory()" },
	{ 0xc0, 0x09, "SysInitKMem()" },
	{ 0xc0, 0x0a, "ChangeClearRCnt()" },
	{ 0xc0, 0x0b, "SystemError()" },
	{ 0xc0, 0x0c, "InitDefInt()" },
	{ 0xc0, 0x0d, "sys_c0_0d()" },
	{ 0xc0, 0x0e, "sys_c0_0e()" },
	{ 0xc0, 0x0f, "sys_c0_0f()" },
	{ 0xc0, 0x10, "sys_c0_10()" },
	{ 0xc0, 0x11, "sys_c0_11()" },
	{ 0xc0, 0x12, "InstallDevices()" },
	{ 0xc0, 0x13, "FlushStdInOutPut()" },
	{ 0xc0, 0x14, "sys_c0_14()" },
	{ 0xc0, 0x15, "_cdevinput()" },
	{ 0xc0, 0x16, "_cdevscan()" },
	{ 0xc0, 0x17, "char _circgetc(struct device_buf *circ)" },
	{ 0xc0, 0x18, "_circputc(char c, struct device_buf *circ)" },
	{ 0xc0, 0x19, "ioabort(const char *str)" },
	{ 0xc0, 0x1a, "sys_c0_1a()" },
	{ 0xc0, 0x1b, "KernelRedirect(int flag)" },
	{ 0xc0, 0x1c, "PatchA0Table()" },
	{ 0x00, 0x00, NULL }
};

static UINT32 log_bioscall_parameter( psxcpu_state *psxcpu, int parm )
{
	if( parm < 4 )
	{
		return psxcpu->r[ 4 + parm ];
	}
	else
	{
		return psx_readword( psxcpu, psxcpu->r[ 29 ] + ( parm * 4 ) );
	}
}

static const char *log_bioscall_string( psxcpu_state *psxcpu, int parm )
{
	int pos;
	UINT32 address;
	static char string[ 1024 ];

	address = log_bioscall_parameter( psxcpu, parm );
	if( address == 0 )
	{
		return "NULL";
	}

	pos = 0;
	string[ pos++ ] = '\"';

	for( ;; )
	{
		UINT8 c = psx_readbyte( psxcpu, address );
		if( c == 0 )
		{
			break;
		}
		else if( c == '\t' )
		{
			string[ pos++ ] = '\\';
			string[ pos++ ] = 't';
		}
		else if( c == '\r' )
		{
			string[ pos++ ] = '\\';
			string[ pos++ ] = 'r';
		}
		else if( c == '\n' )
		{
			string[ pos++ ] = '\\';
			string[ pos++ ] = 'n';
		}
		else if( c < 32 || c > 127 )
		{
			string[ pos++ ] = '\\';
			string[ pos++ ] = ( ( c / 64 ) % 8 ) + '0';
			string[ pos++ ] = ( ( c / 8 ) % 8 ) + '0';
			string[ pos++ ] = ( ( c / 1 ) % 8 ) + '0';
		}
		else
		{
			string[ pos++ ] = c;
		}
		address++;
	}

	string[ pos++ ] = '\"';
	string[ pos++ ] = 0;

	return string;
}

static const char *log_bioscall_hex( psxcpu_state *psxcpu, int parm )
{
	static char string[ 1024 ];

	sprintf( string, "0x%08x", log_bioscall_parameter( psxcpu, parm ) );

	return string;
}

static const char *log_bioscall_char( psxcpu_state *psxcpu, int parm )
{
	int c;
	static char string[ 1024 ];

	c = log_bioscall_parameter( psxcpu, parm );
	if( c < 32 || c > 127 )
	{
		sprintf( string, "0x%02x", c );
	}
	else
	{
		sprintf( string, "'%c'", c );
	}

	return string;
}

static void log_bioscall( psxcpu_state *psxcpu )
{
	int address = psxcpu->pc - 0x04;
	if( address == 0xa0 ||
		address == 0xb0 ||
		address == 0xc0 )
	{
		char buf[ 1024 ];
		int operation = psxcpu->r[ 9 ] & 0xff;
		int bioscall = 0;

		if( ( address == 0xa0 && operation == 0x3c ) ||
			( address == 0xb0 && operation == 0x3d ) )
		{
			putchar( log_bioscall_parameter( psxcpu, 0 ) );
		}

		if( ( address == 0xa0 && operation == 0x03 ) ||
			( address == 0xb0 && operation == 0x35 ) )
		{
			int fd = log_bioscall_parameter( psxcpu, 0 );
			int buf = log_bioscall_parameter( psxcpu, 1 );
			int nbytes = log_bioscall_parameter( psxcpu, 2 );

			if( fd == 1 )
			{
				while( nbytes > 0 )
				{
					UINT8 c = psx_readbyte( psxcpu, buf );
					putchar( c );
					nbytes--;
					buf++;
				}
			}
		}

		while( bioscalls[ bioscall ].prototype != NULL &&
			( bioscalls[ bioscall ].address != address ||
			bioscalls[ bioscall ].operation != operation ) )
		{
			bioscall++;
		}

		if( bioscalls[ bioscall ].prototype != NULL )
		{
			const char *prototype = bioscalls[ bioscall ].prototype;
			const char *parmstart = NULL;
			int parm = 0;
			int parmlen = -1;
			int brackets = 0;
			int pos = 0;

			while( *( prototype ) != 0 )
			{
				int ch = *( prototype );

				switch( ch )
				{
				case '(':
					brackets++;
					prototype++;
					if( brackets == 1 )
					{
						buf[ pos++ ] = ch;
						parmstart = prototype;
					}
					break;

				case ')':
					if( brackets == 1 )
					{
						parmlen = prototype - parmstart;
					}
					prototype++;
					brackets--;
					break;

				case ',':
					if( brackets == 1 )
					{
						parmlen = prototype - parmstart;
					}
					prototype++;
					break;

				default:
					if( brackets == 0 )
					{
						buf[ pos++ ] = ch;
					}
					prototype++;
					break;
				}

				if( parmlen >= 0 )
				{
					while( parmlen > 0 && parmstart[ 0 ] == ' ' )
					{
						parmstart++;
						parmlen--;
					}
					while( parmlen > 0 && parmstart[ parmlen - 1 ] == ' ' )
					{
						parmlen--;
					}

					if( parmlen == 0 ||
						( parmlen == 4 && memcmp( parmstart, "void", 4 ) == 0 ) )
					{
						parm = -1;
					}
					else if( parmlen == 3 && memcmp( parmstart, "...", 3 ) == 0 )
					{
						if( parm > 0 )
						{
							UINT32 format = log_bioscall_parameter( psxcpu, parm - 1 );
							const char *parmstr = NULL;
							int percent = 0;

							for( ;; )
							{
								UINT8 c = psx_readbyte( psxcpu, format );
								if( c == 0 )
								{
									break;
								}
								if( percent == 0 )
								{
									if( c == '%' )
									{
										percent = 1;
									}
								}
								else
								{
									if( c == '%' )
									{
										percent = 0;
									}
									else if( c == '*' )
									{
										parmstr = log_bioscall_hex( psxcpu, parm );
									}
									else if( c == 's' )
									{
										parmstr = log_bioscall_string( psxcpu, parm );
										percent = 0;
									}
									else if( c == 'c' )
									{
										parmstr = log_bioscall_char( psxcpu, parm );
										percent = 0;
									}
									else if( c != '-' && c != '.' && c != 'l' && ( c < '0' || c > '9' ) )
									{
										parmstr = log_bioscall_hex( psxcpu, parm );
										percent = 0;
									}
								}

								if( parmstr != NULL )
								{
									if( parm > 0 )
									{
										buf[ pos++ ] = ',';
									}
									buf[ pos++ ] = ' ';

									strcpy( &buf[ pos ], parmstr );
									pos += strlen( parmstr );
									parmstr = NULL;

									parm++;
								}
								format++;
							}
						}
					}
					else if( parmlen > 0 )
					{
						const char *parmstr;

						int typelen = parmlen;
						while( typelen > 0 && parmstart[ typelen - 1 ] != ' ' && parmstart[ typelen - 1 ] != '*' )
						{
							typelen--;
						}

						if( typelen == 5 && memcmp( parmstart, "char ", 5 ) == 0 )
						{
							parmstr = log_bioscall_char( psxcpu, parm );
						}
						else if( typelen == 12 && memcmp( parmstart, "const char *", 12 ) == 0 )
						{
							parmstr = log_bioscall_string( psxcpu, parm );
						}
						else
						{
							parmstr = log_bioscall_hex( psxcpu, parm );
						}

						if( parm > 0 )
						{
							buf[ pos++ ] = ',';
						}
						buf[ pos++ ] = ' ';

						strcpy( &buf[ pos ], parmstr );
						pos += strlen( parmstr );
					}

					parmlen = -1;
					parm++;

					if( ch == ',' )
					{
						parmstart = prototype;
					}
					else
					{
						if( parm > 0 )
						{
							buf[ pos++ ] = ' ';
						}
						buf[ pos++ ] = ch;
					}
				}
			}
			buf[ pos ] = 0;
		}
		else
		{
			sprintf( buf, "unknown_%02x_%02x", address, operation );
		}
		logerror( "%08x: bioscall %s\n", (unsigned int)psxcpu->r[ 31 ] - 8, buf );
	}
}

static void log_syscall( psxcpu_state *psxcpu )
{
	char buf[ 1024 ];
	int operation = psxcpu->r[ 4 ];

	switch( operation )
	{
	case 0:
		strcpy( buf, "void Exception()" );
		break;

	case 1:
		strcpy( buf, "void EnterCriticalSection()" );
		break;

	case 2:
		strcpy( buf, "void ExitCriticalSection()" );
		break;

	default:
		sprintf( buf, "unknown_%02x", operation );
		break;
	}
	logerror( "%08x: syscall %s\n", (unsigned int)psxcpu->r[ 31 ] - 8, buf );
}

static void mips_update_memory_handlers( psxcpu_state *psxcpu )
{
	if( ( psxcpu->cp0r[ CP0_SR ] & SR_ISC ) != 0 )
	{
		psxcpu->bus_attached = 0;
	}
	else
	{
		psxcpu->bus_attached = 1;
	}
}

INLINE void funct_mthi( psxcpu_state *psxcpu )
{
	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_IDLE;
	psxcpu->hi = psxcpu->r[ INS_RS( psxcpu->op ) ];
}

INLINE void funct_mtlo( psxcpu_state *psxcpu )
{
	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_IDLE;
	psxcpu->lo = psxcpu->r[ INS_RS( psxcpu->op ) ];
}

INLINE void funct_mult( psxcpu_state *psxcpu )
{
	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_MULT;
	psxcpu->multiplier_operand1 = psxcpu->r[ INS_RS( psxcpu->op ) ];
	psxcpu->multiplier_operand2 = psxcpu->r[ INS_RT( psxcpu->op ) ];
	psxcpu->lo = psxcpu->multiplier_operand1;
}

INLINE void funct_multu( psxcpu_state *psxcpu )
{
	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_MULTU;
	psxcpu->multiplier_operand1 = psxcpu->r[ INS_RS( psxcpu->op ) ];
	psxcpu->multiplier_operand2 = psxcpu->r[ INS_RT( psxcpu->op ) ];
	psxcpu->lo = psxcpu->multiplier_operand1;
}

INLINE void funct_div( psxcpu_state *psxcpu )
{
	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_DIV;
	psxcpu->multiplier_operand1 = psxcpu->r[ INS_RS( psxcpu->op ) ];
	psxcpu->multiplier_operand2 = psxcpu->r[ INS_RT( psxcpu->op ) ];
	psxcpu->lo = psxcpu->multiplier_operand1;
	psxcpu->hi = 0;
}

INLINE void funct_divu( psxcpu_state *psxcpu )
{
	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_DIVU;
	psxcpu->multiplier_operand1 = psxcpu->r[ INS_RS( psxcpu->op ) ];
	psxcpu->multiplier_operand2 = psxcpu->r[ INS_RT( psxcpu->op ) ];
	psxcpu->lo = psxcpu->multiplier_operand1;
	psxcpu->hi = 0;
}

static void multiplier_update( psxcpu_state *psxcpu )
{
	switch( psxcpu->multiplier_operation )
	{
	case MULTIPLIER_OPERATION_MULT:
		{
			INT64 result = mul_32x32( (INT32)psxcpu->multiplier_operand1, (INT32)psxcpu->multiplier_operand2 );
			psxcpu->lo = EXTRACT_64LO( result );
			psxcpu->hi = EXTRACT_64HI( result );
		}
		break;

	case MULTIPLIER_OPERATION_MULTU:
		{
			UINT64 result = mulu_32x32( psxcpu->multiplier_operand1, psxcpu->multiplier_operand2 );
			psxcpu->lo = EXTRACT_64LO( result );
			psxcpu->hi = EXTRACT_64HI( result );
		}
		break;

	case MULTIPLIER_OPERATION_DIV:
		if( psxcpu->multiplier_operand2 != 0 )
		{
			psxcpu->lo = (INT32)psxcpu->multiplier_operand1 / (INT32)psxcpu->multiplier_operand2;
			psxcpu->hi = (INT32)psxcpu->multiplier_operand1 % (INT32)psxcpu->multiplier_operand2;
		}
		else
		{
			if( (INT32)psxcpu->multiplier_operand1 < 0 )
			{
				psxcpu->lo = 1;
			}
			else
			{
				psxcpu->lo = 0xffffffff;
			}

			psxcpu->hi = psxcpu->multiplier_operand1;
		}
		break;

	case MULTIPLIER_OPERATION_DIVU:
		if( psxcpu->multiplier_operand2 != 0 )
		{
			psxcpu->lo = psxcpu->multiplier_operand1 / psxcpu->multiplier_operand2;
			psxcpu->hi = psxcpu->multiplier_operand1 % psxcpu->multiplier_operand2;
		}
		else
		{
			psxcpu->lo = 0xffffffff;
			psxcpu->hi = psxcpu->multiplier_operand1;
		}
		break;
	}

	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_IDLE;
}

static UINT32 mips_get_hi( psxcpu_state *psxcpu )
{
	if( psxcpu->multiplier_operation != MULTIPLIER_OPERATION_IDLE )
	{
		multiplier_update( psxcpu );
	}

	return psxcpu->hi;
}

static UINT32 mips_get_lo( psxcpu_state *psxcpu )
{
	if( psxcpu->multiplier_operation != MULTIPLIER_OPERATION_IDLE )
	{
		multiplier_update( psxcpu );
	}
	return psxcpu->lo;
}

static int mips_execute_unstoppable_instructions( psxcpu_state *psxcpu, int executeCop2 )
{
	switch( INS_OP( psxcpu->op ) )
	{
	case OP_SPECIAL:
		switch( INS_FUNCT( psxcpu->op ) )
		{
		case FUNCT_MTHI:
			funct_mthi( psxcpu );
			break;

		case FUNCT_MTLO:
			funct_mtlo( psxcpu );
			break;

		case FUNCT_MULT:
			funct_mult( psxcpu );
			break;

		case FUNCT_MULTU:
			funct_multu( psxcpu );
			break;

		case FUNCT_DIV:
			funct_div( psxcpu );
			break;

		case FUNCT_DIVU:
			funct_divu( psxcpu );
			break;
		}
		break;

	case OP_COP2:
		if( executeCop2 )
		{
			switch( INS_CO( psxcpu->op ) )
			{
			case 1:
				if( ( psxcpu->cp0r[ CP0_SR ] & SR_CU2 ) == 0 )
				{
					return 0;
				}

				docop2( psxcpu, INS_COFUN( psxcpu->op ) );
				break;
			}
		}
	}

	return 1;
}

static void mips_update_address_masks( psxcpu_state *psxcpu )
{
	if( ( psxcpu->cp0r[ CP0_SR ] & SR_KUC ) != 0 )
	{
		psxcpu->bad_byte_address_mask = 0x80000000;
		psxcpu->bad_half_address_mask = 0x80000001;
		psxcpu->bad_word_address_mask = 0x80000003;
	}
	else
	{
		psxcpu->bad_byte_address_mask = 0;
		psxcpu->bad_half_address_mask = 1;
		psxcpu->bad_word_address_mask = 3;
	}
}

static READ32_HANDLER( psx_berr_r )
{
	psxcpu_state *psxcpu = get_safe_token(space->cpu);

	psxcpu->berr = 1;

	return 0;
}

static WRITE32_HANDLER( psx_berr_w )
{
	psxcpu_state *psxcpu = get_safe_token(space->cpu);

	psxcpu->berr = 1;
}

static void mips_update_scratchpad( const address_space *space )
{
	psxcpu_state *psxcpu = get_safe_token(space->cpu);

	if( ( psxcpu->biu & BIU_RAM ) == 0 )
	{
		memory_install_readwrite32_handler( space, 0x1f800000, 0x1f8003ff, 0, 0, psx_berr_r, psx_berr_w );
	}
	else if( ( psxcpu->biu & BIU_DS ) == 0 )
	{
		memory_install_readwrite32_handler( space, 0x1f800000, 0x1f8003ff, 0, 0, psx_berr_r, (write32_space_func)SMH_NOP );
	}
	else
	{
		memory_install_readwrite32_handler( space, 0x1f800000, 0x1f8003ff, 0, 0, (read32_space_func)SMH_BANK(32), (write32_space_func)SMH_BANK(32) );

		memory_set_bankptr(space->machine,  32, psxcpu->dcache );
	}
}

INLINE void mips_set_cp0r( psxcpu_state *psxcpu, int reg, UINT32 value )
{
	UINT32 old = psxcpu->cp0r[ reg ];

	psxcpu->cp0r[ reg ] = value;

	if( reg == CP0_SR )
	{
		if( ( old & SR_ISC ) != ( value & SR_ISC ) )
		{
			mips_update_memory_handlers( psxcpu );
		}

		if( ( old & SR_KUC ) != ( value & SR_KUC ) )
		{
			mips_update_address_masks( psxcpu );
		}
	}

	if( ( reg == CP0_SR || reg == CP0_CAUSE ) &&
		( psxcpu->cp0r[ CP0_SR ] & SR_IEC ) != 0 &&
		( psxcpu->cp0r[ CP0_SR ] & psxcpu->cp0r[ CP0_CAUSE ] & CAUSE_IP ) != 0 )
	{
		psxcpu->op = memory_decrypted_read_dword( psxcpu->program, psxcpu->pc );
		mips_execute_unstoppable_instructions( psxcpu, 1 );
		mips_exception( psxcpu, EXC_INT );
	}
	else if( reg == CP0_SR &&
		psxcpu->delayr != PSXCPU_DELAYR_PC &&
		( psxcpu->pc & psxcpu->bad_word_address_mask ) != 0 )
	{
		mips_load_bad_address( psxcpu, psxcpu->pc );
	}
}

INLINE void mips_commit_delayed_load( psxcpu_state *psxcpu )
{
	if( psxcpu->delayr != 0 )
	{
		psxcpu->r[ psxcpu->delayr ] = psxcpu->delayv;
		psxcpu->delayr = 0;
		psxcpu->delayv = 0;
	}
}

static void mips_set_pc( psxcpu_state *psxcpu, unsigned pc )
{
	psxcpu->pc = pc;
}

static void mips_fetch_next_op( psxcpu_state *psxcpu )
{
	if( psxcpu->delayr == PSXCPU_DELAYR_PC )
	{
		UINT32 safepc = psxcpu->delayv & ~psxcpu->bad_word_address_mask;

		psxcpu->op = memory_decrypted_read_dword( psxcpu->program, safepc );
	}
	else
	{
		psxcpu->op = memory_decrypted_read_dword( psxcpu->program, psxcpu->pc + 4 );
	}
}

INLINE int mips_advance_pc( psxcpu_state *psxcpu )
{
	if( psxcpu->delayr == PSXCPU_DELAYR_PC )
	{
		psxcpu->pc = psxcpu->delayv;
		psxcpu->delayr = 0;
		psxcpu->delayv = 0;

		if( ( psxcpu->pc & psxcpu->bad_word_address_mask ) != 0 )
		{
			mips_load_bad_address( psxcpu, psxcpu->pc );
			return 0;
		}
	}
	else if( psxcpu->delayr == PSXCPU_DELAYR_NOTPC )
	{
		psxcpu->delayr = 0;
		psxcpu->delayv = 0;
		psxcpu->pc += 4;
	}
	else
	{
		mips_commit_delayed_load( psxcpu );
		psxcpu->pc += 4;
	}

	return 1;
}

INLINE void mips_load( psxcpu_state *psxcpu, UINT32 reg, UINT32 value )
{
	mips_advance_pc( psxcpu );

	if( reg != 0 )
	{
		psxcpu->r[ reg ] = value;
	}
}

INLINE void mips_delayed_load( psxcpu_state *psxcpu, UINT32 reg, UINT32 value )
{
	mips_advance_pc( psxcpu );

	psxcpu->delayr = reg;
	psxcpu->delayv = value;
}

INLINE void mips_branch( psxcpu_state *psxcpu, UINT32 address )
{
	mips_advance_pc( psxcpu );

	psxcpu->delayr = PSXCPU_DELAYR_PC;
	psxcpu->delayv = address;
}

INLINE void mips_conditional_branch( psxcpu_state *psxcpu, int takeBranch )
{
	mips_advance_pc( psxcpu );

	if( takeBranch )
	{
		psxcpu->delayr = PSXCPU_DELAYR_PC;
		psxcpu->delayv = psxcpu->pc + ( PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) ) << 2 );
	}
	else
	{
		psxcpu->delayr = PSXCPU_DELAYR_NOTPC;
		psxcpu->delayv = 0;
	}
}

INLINE void mips_unconditional_branch( psxcpu_state *psxcpu )
{
	mips_advance_pc( psxcpu );

	psxcpu->delayr = PSXCPU_DELAYR_PC;
	psxcpu->delayv = ( psxcpu->pc & 0xf0000000 ) + ( INS_TARGET( psxcpu->op ) << 2 );
}

static void mips_common_exception( psxcpu_state *psxcpu, int exception, UINT32 romOffset, UINT32 ramOffset )
{
	int cause = ( exception << 2 ) | ( ( ( psxcpu->op >> 26 ) & 3 ) << 28 );

	if( psxcpu->delayr == PSXCPU_DELAYR_PC )
	{
		cause |= CAUSE_BT;
		mips_set_cp0r( psxcpu, CP0_TAR, psxcpu->delayv );
	}
	else if( psxcpu->delayr == PSXCPU_DELAYR_NOTPC )
	{
		mips_set_cp0r( psxcpu, CP0_TAR, psxcpu->pc + 4 );
	}
	else
	{
		mips_commit_delayed_load( psxcpu );
	}

	if( psxcpu->delayr == PSXCPU_DELAYR_PC || psxcpu->delayr == PSXCPU_DELAYR_NOTPC )
	{
		cause |= CAUSE_BD;
		mips_set_cp0r( psxcpu, CP0_EPC, psxcpu->pc - 4 );
	}
	else
	{
		mips_set_cp0r( psxcpu, CP0_EPC, psxcpu->pc );
	}

	if( LOG_BIOSCALL && exception != EXC_INT )
	{
		logerror( "%08x: Exception %d\n", psxcpu->pc, exception );
	}

	psxcpu->delayr = 0;
	psxcpu->delayv = 0;
	psxcpu->berr = 0;

	if( psxcpu->cp0r[ CP0_SR ] & SR_BEV )
	{
		mips_set_pc( psxcpu, romOffset );
	}
	else
	{
		mips_set_pc( psxcpu, ramOffset );
	}

	mips_set_cp0r( psxcpu, CP0_SR, ( psxcpu->cp0r[ CP0_SR ] & ~0x3f ) | ( ( psxcpu->cp0r[ CP0_SR ] << 2 ) & 0x3f ) );
	mips_set_cp0r( psxcpu, CP0_CAUSE, ( psxcpu->cp0r[ CP0_CAUSE ] & ~( CAUSE_EXC | CAUSE_BD | CAUSE_BT | CAUSE_CE ) ) | cause );
}

static void mips_exception( psxcpu_state *psxcpu, int exception )
{
	mips_common_exception( psxcpu, exception, 0xbfc00180, 0x80000080 );
}

static void mips_breakpoint_exception( psxcpu_state *psxcpu )
{
	mips_fetch_next_op( psxcpu );
	mips_execute_unstoppable_instructions( psxcpu, 1 );
	mips_common_exception( psxcpu, EXC_BP, 0xbfc00140, 0x80000040 );
}

static void mips_load_bus_error_exception( psxcpu_state *psxcpu )
{
	mips_fetch_next_op( psxcpu );
	mips_execute_unstoppable_instructions( psxcpu, 0 );
	mips_common_exception( psxcpu, EXC_DBE, 0xbfc00180, 0x80000080 );
}

static void mips_store_bus_error_exception( psxcpu_state *psxcpu )
{
	mips_fetch_next_op( psxcpu );

	if( mips_execute_unstoppable_instructions( psxcpu, 1 ) )
	{
		if( !mips_advance_pc( psxcpu ) )
		{
			return;
		}

		mips_fetch_next_op( psxcpu );
		mips_execute_unstoppable_instructions( psxcpu, 0 );
	}

	mips_common_exception( psxcpu, EXC_DBE, 0xbfc00180, 0x80000080 );
}

static void mips_load_bad_address( psxcpu_state *psxcpu, UINT32 address )
{
	mips_set_cp0r( psxcpu, CP0_BADA, address );
	mips_exception( psxcpu, EXC_ADEL );
}

static void mips_store_bad_address( psxcpu_state *psxcpu, UINT32 address )
{
	mips_set_cp0r( psxcpu, CP0_BADA, address );
	mips_exception( psxcpu, EXC_ADES );
}

INLINE int mips_data_address_breakpoint( psxcpu_state *psxcpu, int dcic_rw, int dcic_status, UINT32 address )
{
	if( address < 0x1f000000 || address > 0x1fffffff )
	{
		if( ( psxcpu->cp0r[ CP0_DCIC ] & DCIC_DE ) != 0 &&
			( ( ( psxcpu->cp0r[ CP0_DCIC ] & DCIC_KD ) != 0 && ( psxcpu->cp0r[ CP0_SR ] & SR_KUC ) == 0 ) ||
			( ( psxcpu->cp0r[ CP0_DCIC ] & DCIC_UD ) != 0 && ( psxcpu->cp0r[ CP0_SR ] & SR_KUC ) != 0 ) ) )
		{
			if( ( psxcpu->cp0r[ CP0_DCIC ] & dcic_rw ) == dcic_rw &&
				( address & psxcpu->cp0r[ CP0_BDAM ] ) == ( psxcpu->cp0r[ CP0_BDA ] & psxcpu->cp0r[ CP0_BDAM ] ) )
			{
				psxcpu->cp0r[ CP0_DCIC ] = ( psxcpu->cp0r[ CP0_DCIC ] & ~DCIC_STATUS ) | dcic_status;

				if( ( psxcpu->cp0r[ CP0_DCIC ] & DCIC_TR ) != 0 )
				{
					return 1;
				}
			}
		}
	}

	return 0;
}

INLINE int mips_load_data_address_breakpoint( psxcpu_state *psxcpu, UINT32 address )
{
	return mips_data_address_breakpoint( psxcpu, DCIC_DR | DCIC_DAE, DCIC_DB | DCIC_DA | DCIC_R, address );
}

INLINE int mips_store_data_address_breakpoint( psxcpu_state *psxcpu, UINT32 address )
{
	return mips_data_address_breakpoint( psxcpu, DCIC_DW | DCIC_DAE, DCIC_DB | DCIC_DA | DCIC_W, address );
}

static STATE_POSTLOAD( mips_postload )
{
	psxcpu_state *psxcpu = (psxcpu_state *)param;

	mips_update_memory_handlers( psxcpu );
	mips_update_address_masks( psxcpu );
	mips_update_scratchpad( psxcpu->program );
}

static void mips_state_register( const char *type, const device_config *device )
{
	psxcpu_state *psxcpu = get_safe_token(device);

	state_save_register_device_item( device, 0, psxcpu->op );
	state_save_register_device_item( device, 0, psxcpu->pc );
	state_save_register_device_item( device, 0, psxcpu->delayv );
	state_save_register_device_item( device, 0, psxcpu->delayr );
	state_save_register_device_item( device, 0, psxcpu->hi );
	state_save_register_device_item( device, 0, psxcpu->lo );
	state_save_register_device_item( device, 0, psxcpu->biu );
	state_save_register_device_item_array( device, 0, psxcpu->r );
	state_save_register_device_item_array( device, 0, psxcpu->cp0r );
	state_save_register_device_item_array( device, 0, psxcpu->cp2cr );
	state_save_register_device_item_array( device, 0, psxcpu->cp2dr );
	state_save_register_device_item_array( device, 0, psxcpu->icacheTag );
	state_save_register_device_item_array( device, 0, psxcpu->icache );
	state_save_register_device_item_array( device, 0, psxcpu->dcache );
	state_save_register_device_item( device, 0, psxcpu->multiplier_operation );
	state_save_register_device_item( device, 0, psxcpu->multiplier_operand1 );
	state_save_register_device_item( device, 0, psxcpu->multiplier_operand2 );
	state_save_register_postload( device->machine, mips_postload, psxcpu );
}

static CPU_INIT( psxcpu )
{
	psxcpu_state *psxcpu = get_safe_token(device);
//  psxcpu->intf = (psxcpu_interface *) device->static_config;

	psxcpu->irq_callback = irqcallback;
	psxcpu->device = device;
	psxcpu->program = memory_find_address_space(device, ADDRESS_SPACE_PROGRAM);

	mips_state_register( "psxcpu", device );
}

static CPU_RESET( psxcpu )
{
	psxcpu_state *psxcpu = get_safe_token(device);

	psxcpu->delayr = 0;
	psxcpu->delayv = 0;

	psxcpu->multiplier_operation = MULTIPLIER_OPERATION_IDLE;

	mips_update_memory_handlers( psxcpu );
	mips_update_address_masks( psxcpu );
	mips_update_scratchpad( psxcpu->program );

	mips_set_cp0r( psxcpu, CP0_SR, SR_BEV );
	mips_set_cp0r( psxcpu, CP0_CAUSE, 0x00000000 );
	mips_set_cp0r( psxcpu, CP0_PRID, 0x00000002 );
	mips_set_cp0r( psxcpu, CP0_DCIC, 0x00000000 );
	mips_set_cp0r( psxcpu, CP0_BPCM, 0xffffffff );
	mips_set_cp0r( psxcpu, CP0_BDAM, 0xffffffff );
	mips_set_pc( psxcpu, 0xbfc00000 );
}

static CPU_EXIT( psxcpu )
{
}

static UINT32 mips_get_register_from_pipeline( psxcpu_state *psxcpu, int reg )
{
	if( psxcpu->delayr == reg )
	{
		UINT32 data = psxcpu->delayv;

		psxcpu->delayr = 0;
		psxcpu->delayv = 0;

		return data;
	}

	return psxcpu->r[ reg ];
}

static int mips_cop0_usable( psxcpu_state *psxcpu )
{
	if( ( psxcpu->cp0r[ CP0_SR ] & SR_KUC ) != 0 && ( psxcpu->cp0r[ CP0_SR ] & SR_CU0 ) == 0 )
	{
		mips_exception( psxcpu, EXC_CPU );

		return 0;
	}

	return 1;
}

static void mips_lwc( psxcpu_state *psxcpu, int cop, int sr_cu )
{
	UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
	int breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

	if( ( psxcpu->cp0r[ CP0_SR ] & sr_cu ) == 0 )
	{
		mips_exception( psxcpu, EXC_CPU );
	}
	else if( ( address & psxcpu->bad_word_address_mask ) != 0 )
	{
		mips_load_bad_address( psxcpu, address );
	}
	else if( breakpoint )
	{
		mips_breakpoint_exception( psxcpu );
	}
	else
	{
		UINT32 data = psx_readword( psxcpu, address );

		if( psxcpu->berr )
		{
			mips_load_bus_error_exception( psxcpu );
		}
		else
		{
			int reg = INS_RT( psxcpu->op );

			mips_advance_pc( psxcpu );

			switch( cop )
			{
			case 0:
				/* lwc0 doesn't update any cop0 registers */
				break;

			case 1:
				setcp1dr( psxcpu, reg, data );
				break;

			case 2:
				setcp2dr( psxcpu, reg, data );
				break;

			case 3:
				setcp3dr( psxcpu, reg, data );
				break;
			}
		}
	}
}

static void mips_swc( psxcpu_state *psxcpu, int cop, int sr_cu )
{
	UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
	int breakpoint = mips_store_data_address_breakpoint( psxcpu, address );

	if( ( psxcpu->cp0r[ CP0_SR ] & sr_cu ) == 0 )
	{
		mips_exception( psxcpu, EXC_CPU );
	}
	else if( ( address & psxcpu->bad_word_address_mask ) != 0 )
	{
		mips_store_bad_address( psxcpu, address );
	}
	else
	{
		UINT32 data = 0;

		switch( cop )
		{
		case 0:
			{
				int address;

				if( psxcpu->delayr == PSXCPU_DELAYR_PC )
				{
					switch( psxcpu->delayv & 0x0c )
					{
					case 0x0c:
						address = psxcpu->delayv;
						break;

					default:
						address = psxcpu->delayv + 4;
						break;
					}
				}
				else
				{
					switch( psxcpu->pc & 0x0c )
					{
					case 0x0:
					case 0xc:
						address = psxcpu->pc + 0x08;
						break;

					default:
						address = psxcpu->pc | 0x0c;
						break;
					}
				}

				data = memory_read_dword_32le( psxcpu->program, address );
			}
			break;

		case 1:
			data = getcp1dr( psxcpu, INS_RT( psxcpu->op ) );
			break;

		case 2:
			data = getcp2dr( psxcpu, INS_RT( psxcpu->op ) );
			break;

		case 3:
			data = getcp3dr( psxcpu, INS_RT( psxcpu->op ) );
			break;
		}

		psx_writeword( psxcpu, address, data );

		if( breakpoint )
		{
			mips_breakpoint_exception( psxcpu );
		}
		else if( psxcpu->berr )
		{
			mips_store_bus_error_exception( psxcpu );
		}
		else
		{
			mips_advance_pc( psxcpu );
		}
	}
}

static void mips_bc( psxcpu_state *psxcpu, int cop, int sr_cu, int condition )
{
	if( ( psxcpu->cp0r[ CP0_SR ] & sr_cu ) == 0 )
	{
		mips_exception( psxcpu, EXC_CPU );
	}
	else
	{
		mips_conditional_branch( psxcpu, !condition );
	}
}

static CPU_EXECUTE( psxcpu )
{
	psxcpu_state *psxcpu = get_safe_token(device);

	psxcpu->icount = cycles;

	do
	{
		if (LOG_BIOSCALL) log_bioscall( psxcpu );
		debugger_instruction_hook(device,  psxcpu->pc );

		psxcpu->op = memory_decrypted_read_dword( psxcpu->program, psxcpu->pc );
		switch( INS_OP( psxcpu->op ) )
		{
		case OP_SPECIAL:
			switch( INS_FUNCT( psxcpu->op ) )
			{
			case FUNCT_SLL:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] << INS_SHAMT( psxcpu->op ) );
				break;

			case FUNCT_SRL:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] >> INS_SHAMT( psxcpu->op ) );
				break;

			case FUNCT_SRA:
				mips_load( psxcpu, INS_RD( psxcpu->op ), (INT32)psxcpu->r[ INS_RT( psxcpu->op ) ] >> INS_SHAMT( psxcpu->op ) );
				break;

			case FUNCT_SLLV:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] << ( psxcpu->r[ INS_RS( psxcpu->op ) ] & 31 ) );
				break;

			case FUNCT_SRLV:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] >> ( psxcpu->r[ INS_RS( psxcpu->op ) ] & 31 ) );
				break;

			case FUNCT_SRAV:
				mips_load( psxcpu, INS_RD( psxcpu->op ), (INT32)psxcpu->r[ INS_RT( psxcpu->op ) ] >> ( psxcpu->r[ INS_RS( psxcpu->op ) ] & 31 ) );
				break;

			case FUNCT_JR:
				mips_branch( psxcpu, psxcpu->r[ INS_RS( psxcpu->op ) ] );
				break;

			case FUNCT_JALR:
				mips_branch( psxcpu, psxcpu->r[ INS_RS( psxcpu->op ) ] );
				if( INS_RD( psxcpu->op ) != 0 )
				{
					psxcpu->r[ INS_RD( psxcpu->op ) ] = psxcpu->pc + 4;
				}
				break;

			case FUNCT_SYSCALL:
				if (LOG_BIOSCALL) log_syscall( psxcpu );
				mips_exception( psxcpu, EXC_SYS );
				break;

			case FUNCT_BREAK:
				mips_exception( psxcpu, EXC_BP );
				break;

			case FUNCT_MFHI:
				mips_load( psxcpu, INS_RD( psxcpu->op ), mips_get_hi( psxcpu ) );
				break;

			case FUNCT_MTHI:
				funct_mthi( psxcpu );
				mips_advance_pc( psxcpu );
				break;

			case FUNCT_MFLO:
				mips_load( psxcpu, INS_RD( psxcpu->op ), mips_get_lo( psxcpu ) );
				break;

			case FUNCT_MTLO:
				funct_mtlo( psxcpu );
				mips_advance_pc( psxcpu );
				break;

			case FUNCT_MULT:
				funct_mult( psxcpu );
				mips_advance_pc( psxcpu );
				break;

			case FUNCT_MULTU:
				funct_multu( psxcpu );
				mips_advance_pc( psxcpu );
				break;

			case FUNCT_DIV:
				funct_div( psxcpu );
				mips_advance_pc( psxcpu );
				break;

			case FUNCT_DIVU:
				funct_divu( psxcpu );
				mips_advance_pc( psxcpu );
				break;

			case FUNCT_ADD:
				{
					UINT32 result = psxcpu->r[ INS_RS( psxcpu->op ) ] + psxcpu->r[ INS_RT( psxcpu->op ) ];
					if( (INT32)( ~( psxcpu->r[ INS_RS( psxcpu->op ) ] ^ psxcpu->r[ INS_RT( psxcpu->op ) ] ) & ( psxcpu->r[ INS_RS( psxcpu->op ) ] ^ result ) ) < 0 )
					{
						mips_exception( psxcpu, EXC_OVF );
					}
					else
					{
						mips_load( psxcpu, INS_RD( psxcpu->op ), result );
					}
				}
				break;

			case FUNCT_ADDU:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] + psxcpu->r[ INS_RT( psxcpu->op ) ] );
				break;

			case FUNCT_SUB:
				{
					UINT32 result = psxcpu->r[ INS_RS( psxcpu->op ) ] - psxcpu->r[ INS_RT( psxcpu->op ) ];
					if( (INT32)( ( psxcpu->r[ INS_RS( psxcpu->op ) ] ^ psxcpu->r[ INS_RT( psxcpu->op ) ] ) & ( psxcpu->r[ INS_RS( psxcpu->op ) ] ^ result ) ) < 0 )
					{
						mips_exception( psxcpu, EXC_OVF );
					}
					else
					{
						mips_load( psxcpu, INS_RD( psxcpu->op ), result );
					}
				}
				break;

			case FUNCT_SUBU:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] - psxcpu->r[ INS_RT( psxcpu->op ) ] );
				break;

			case FUNCT_AND:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] & psxcpu->r[ INS_RT( psxcpu->op ) ] );
				break;

			case FUNCT_OR:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] | psxcpu->r[ INS_RT( psxcpu->op ) ] );
				break;

			case FUNCT_XOR:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] ^ psxcpu->r[ INS_RT( psxcpu->op ) ] );
				break;

			case FUNCT_NOR:
				mips_load( psxcpu, INS_RD( psxcpu->op ), ~( psxcpu->r[ INS_RS( psxcpu->op ) ] | psxcpu->r[ INS_RT( psxcpu->op ) ] ) );
				break;

			case FUNCT_SLT:
				mips_load( psxcpu, INS_RD( psxcpu->op ), (INT32)psxcpu->r[ INS_RS( psxcpu->op ) ] < (INT32)psxcpu->r[ INS_RT( psxcpu->op ) ] );
				break;

			case FUNCT_SLTU:
				mips_load( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] < psxcpu->r[ INS_RT( psxcpu->op ) ] );
				break;

			default:
				mips_exception( psxcpu, EXC_RI );
				break;
			}
			break;

		case OP_REGIMM:
			switch( INS_RT_REGIMM( psxcpu->op ) )
			{
			case RT_BLTZ:
				mips_conditional_branch( psxcpu, (INT32)psxcpu->r[ INS_RS( psxcpu->op ) ] < 0 );

				if( INS_RT( psxcpu->op ) == RT_BLTZAL )
				{
					psxcpu->r[ 31 ] = psxcpu->pc + 4;
				}
				break;

			case RT_BGEZ:
				mips_conditional_branch( psxcpu, (INT32)psxcpu->r[ INS_RS( psxcpu->op ) ] >= 0 );

				if( INS_RT( psxcpu->op ) == RT_BGEZAL )
				{
					psxcpu->r[ 31 ] = psxcpu->pc + 4;
				}
				break;
			}
			break;

		case OP_J:
			mips_unconditional_branch( psxcpu );
			break;

		case OP_JAL:
			mips_unconditional_branch( psxcpu );
			psxcpu->r[ 31 ] = psxcpu->pc + 4;
			break;

		case OP_BEQ:
			mips_conditional_branch( psxcpu, psxcpu->r[ INS_RS( psxcpu->op ) ] == psxcpu->r[ INS_RT( psxcpu->op ) ] );
			break;

		case OP_BNE:
			mips_conditional_branch( psxcpu, psxcpu->r[ INS_RS( psxcpu->op ) ] != psxcpu->r[ INS_RT( psxcpu->op ) ] );
			break;

		case OP_BLEZ:
			mips_conditional_branch( psxcpu, (INT32)psxcpu->r[ INS_RS( psxcpu->op ) ] < 0 || psxcpu->r[ INS_RS( psxcpu->op ) ] == psxcpu->r[ INS_RT( psxcpu->op ) ] );
			break;

		case OP_BGTZ:
			mips_conditional_branch( psxcpu, (INT32)psxcpu->r[ INS_RS( psxcpu->op ) ] >= 0 && psxcpu->r[ INS_RS( psxcpu->op ) ] != psxcpu->r[ INS_RT( psxcpu->op ) ] );
			break;

		case OP_ADDI:
			{
				UINT32 immediate = PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				UINT32 result = psxcpu->r[ INS_RS( psxcpu->op ) ] + immediate;
				if( (INT32)( ~( psxcpu->r[ INS_RS( psxcpu->op ) ] ^ immediate ) & ( psxcpu->r[ INS_RS( psxcpu->op ) ] ^ result ) ) < 0 )
				{
					mips_exception( psxcpu, EXC_OVF );
				}
				else
				{
					mips_load( psxcpu, INS_RT( psxcpu->op ), result );
				}
			}
			break;

		case OP_ADDIU:
			mips_load( psxcpu, INS_RT( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) ) );
			break;

		case OP_SLTI:
			mips_load( psxcpu, INS_RT( psxcpu->op ), (INT32)psxcpu->r[ INS_RS( psxcpu->op ) ] < PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) ) );
			break;

		case OP_SLTIU:
			mips_load( psxcpu, INS_RT( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] < (UINT32)PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) ) );
			break;

		case OP_ANDI:
			mips_load( psxcpu, INS_RT( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] & INS_IMMEDIATE( psxcpu->op ) );
			break;

		case OP_ORI:
			mips_load( psxcpu, INS_RT( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] | INS_IMMEDIATE( psxcpu->op ) );
			break;

		case OP_XORI:
			mips_load( psxcpu, INS_RT( psxcpu->op ), psxcpu->r[ INS_RS( psxcpu->op ) ] ^ INS_IMMEDIATE( psxcpu->op ) );
			break;

		case OP_LUI:
			mips_load( psxcpu, INS_RT( psxcpu->op ), INS_IMMEDIATE( psxcpu->op ) << 16 );
			break;

		case OP_COP0:
			switch( INS_RS( psxcpu->op ) )
			{
			case RS_MFC:
				{
					int reg = INS_RD( psxcpu->op );

					if( reg == CP0_INDEX ||
						reg == CP0_RANDOM ||
						reg == CP0_ENTRYLO ||
						reg == CP0_CONTEXT ||
						reg == CP0_ENTRYHI )
					{
						mips_exception( psxcpu, EXC_RI );
					}
					else if( reg < 16 )
					{
						if( mips_cop0_usable( psxcpu ) )
						{
							mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), psxcpu->cp0r[ reg ] );
						}
					}
					else
					{
						mips_advance_pc( psxcpu );
					}
				}
				break;

			case RS_CFC:
				mips_exception( psxcpu, EXC_RI );
				break;

			case RS_MTC:
				{
					int reg = INS_RD( psxcpu->op );

					if( reg == CP0_INDEX ||
						reg == CP0_RANDOM ||
						reg == CP0_ENTRYLO ||
						reg == CP0_CONTEXT ||
						reg == CP0_ENTRYHI )
					{
						mips_exception( psxcpu, EXC_RI );
					}
					else if( reg < 16 )
					{
						if( mips_cop0_usable( psxcpu ) )
						{
							UINT32 data = ( psxcpu->cp0r[ reg ] & ~mips_mtc0_writemask[ reg ] ) |
								( psxcpu->r[ INS_RT( psxcpu->op ) ] & mips_mtc0_writemask[ reg ] );
							mips_advance_pc( psxcpu );

							mips_set_cp0r( psxcpu, reg, data );
						}
					}
					else
					{
						mips_advance_pc( psxcpu );
					}
				}
				break;

			case RS_CTC:
				mips_exception( psxcpu, EXC_RI );
				break;

			case RS_BC:
			case RS_BC_ALT:
				switch( INS_BC( psxcpu->op ) )
				{
				case BC_BCF:
					mips_bc( psxcpu, 0, SR_CU0, 0 );
					break;

				case BC_BCT:
					mips_bc( psxcpu, 0, SR_CU0, 1 );
					break;
				}
				break;

			default:
				switch( INS_CO( psxcpu->op ) )
				{
				case 1:
					switch( INS_CF( psxcpu->op ) )
					{
					case CF_TLBR:
					case CF_TLBWI:
					case CF_TLBWR:
					case CF_TLBP:
						mips_exception( psxcpu, EXC_RI );
						break;

					case CF_RFE:
						if( mips_cop0_usable( psxcpu ) )
						{
							mips_advance_pc( psxcpu );
							mips_set_cp0r( psxcpu, CP0_SR, ( psxcpu->cp0r[ CP0_SR ] & ~0xf ) | ( ( psxcpu->cp0r[ CP0_SR ] >> 2 ) & 0xf ) );
						}
						break;

					default:
						mips_advance_pc( psxcpu );
						break;
					}
					break;

				default:
					mips_advance_pc( psxcpu );
					break;
				}
				break;
			}
			break;

		case OP_COP1:
			if( ( psxcpu->cp0r[ CP0_SR ] & SR_CU1 ) == 0 )
			{
				mips_exception( psxcpu, EXC_CPU );
			}
			else
			{
				switch( INS_RS( psxcpu->op ) )
				{
				case RS_MFC:
					mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), getcp1dr( psxcpu, INS_RD( psxcpu->op ) ) );
					break;

				case RS_CFC:
					mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), getcp1cr( psxcpu, INS_RD( psxcpu->op ) ) );
					break;

				case RS_MTC:
					setcp1dr( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] );
					mips_advance_pc( psxcpu );
					break;

				case RS_CTC:
					setcp1cr( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] );
					mips_advance_pc( psxcpu );
					break;

				case RS_BC:
				case RS_BC_ALT:
					switch( INS_BC( psxcpu->op ) )
					{
					case BC_BCF:
						mips_bc( psxcpu, 1, SR_CU1, 0 );
						break;

					case BC_BCT:
						mips_bc( psxcpu, 1, SR_CU1, 1 );
						break;
					}
					break;

				default:
					mips_advance_pc( psxcpu );
					break;
				}
			}
			break;

		case OP_COP2:
			if( ( psxcpu->cp0r[ CP0_SR ] & SR_CU2 ) == 0 )
			{
				mips_exception( psxcpu, EXC_CPU );
			}
			else
			{
				switch( INS_RS( psxcpu->op ) )
				{
				case RS_MFC:
					mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), getcp2dr( psxcpu, INS_RD( psxcpu->op ) ) );
					break;

				case RS_CFC:
					mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), getcp2cr( psxcpu, INS_RD( psxcpu->op ) ) );
					break;

				case RS_MTC:
					setcp2dr( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] );
					mips_advance_pc( psxcpu );
					break;

				case RS_CTC:
					setcp2cr( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] );
					mips_advance_pc( psxcpu );
					break;

				case RS_BC:
				case RS_BC_ALT:
					switch( INS_BC( psxcpu->op ) )
					{
					case BC_BCF:
						mips_bc( psxcpu, 2, SR_CU2, 0 );
						break;

					case BC_BCT:
						mips_bc( psxcpu, 2, SR_CU2, 1 );
						break;
					}
					break;

				default:
					switch( INS_CO( psxcpu->op ) )
					{
					case 1:
						docop2( psxcpu, INS_COFUN( psxcpu->op ) );
						mips_advance_pc( psxcpu );
						break;

					default:
						mips_advance_pc( psxcpu );
						break;
					}
					break;
				}
			}
			break;

		case OP_COP3:
			if( ( psxcpu->cp0r[ CP0_SR ] & SR_CU3 ) == 0 )
			{
				mips_exception( psxcpu, EXC_CPU );
			}
			else
			{
				switch( INS_RS( psxcpu->op ) )
				{
				case RS_MFC:
					mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), getcp3dr( psxcpu, INS_RD( psxcpu->op ) ) );
					break;

				case RS_CFC:
					mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), getcp3cr( psxcpu, INS_RD( psxcpu->op ) ) );
					break;

				case RS_MTC:
					setcp3dr( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] );
					mips_advance_pc( psxcpu );
					break;

				case RS_CTC:
					setcp3cr( psxcpu, INS_RD( psxcpu->op ), psxcpu->r[ INS_RT( psxcpu->op ) ] );
					mips_advance_pc( psxcpu );
					break;

				case RS_BC:
				case RS_BC_ALT:
					switch( INS_BC( psxcpu->op ) )
					{
					case BC_BCF:
						mips_bc( psxcpu, 3, SR_CU3, 0 );
						break;

					case BC_BCT:
						mips_bc( psxcpu, 3, SR_CU3, 1 );
						break;
					}
					break;

				default:
					mips_advance_pc( psxcpu );
					break;
				}
			}
			break;

		case OP_LB:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_byte_address_mask ) != 0 )
				{
					mips_load_bad_address( psxcpu, address );
				}
				else if( breakpoint )
				{
					mips_breakpoint_exception( psxcpu );
				}
				else
				{
					UINT32 data = PSXCPU_BYTE_EXTEND( psx_readbyte( psxcpu, address ) );

					if( psxcpu->berr )
					{
						mips_load_bus_error_exception( psxcpu );
					}
					else
					{
						mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), data );
					}
				}
			}
			break;

		case OP_LH:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_half_address_mask ) != 0 )
				{
					mips_load_bad_address( psxcpu, address );
				}
				else if( breakpoint )
				{
					mips_breakpoint_exception( psxcpu );
				}
				else
				{
					UINT32 data = PSXCPU_WORD_EXTEND( psx_readhalf( psxcpu, address ) );

					if( psxcpu->berr )
					{
						mips_load_bus_error_exception( psxcpu );
					}
					else
					{
						mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), data );
					}
				}
			}
			break;

		case OP_LWL:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int load_type = address & 3;
				int breakpoint;

				address &= ~3;
				breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_byte_address_mask ) != 0 )
				{
					mips_load_bad_address( psxcpu, address );
				}
				else if( breakpoint )
				{
					mips_breakpoint_exception( psxcpu );
				}
				else
				{
					UINT32 data = mips_get_register_from_pipeline( psxcpu, INS_RT( psxcpu->op ) );

					switch( load_type )
					{
					case 0:
						data = ( data & 0x00ffffff ) | ( psx_readword_masked( psxcpu, address, 0x000000ff ) << 24 );
						break;

					case 1:
						data = ( data & 0x0000ffff ) | ( psx_readword_masked( psxcpu, address, 0x0000ffff ) << 16 );
						break;

					case 2:
						data = ( data & 0x000000ff ) | ( psx_readword_masked( psxcpu, address, 0x00ffffff ) << 8 );
						break;

					case 3:
						data = psx_readword( psxcpu, address );
						break;
					}

					if( psxcpu->berr )
					{
						mips_load_bus_error_exception( psxcpu );
					}
					else
					{
						mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), data );
					}
				}
			}
			break;

		case OP_LW:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_word_address_mask ) != 0 )
				{
					mips_load_bad_address( psxcpu, address );
				}
				else if( breakpoint )
				{
					mips_breakpoint_exception( psxcpu );
				}
				else
				{
					UINT32 data = psx_readword( psxcpu, address );

					if( psxcpu->berr )
					{
						mips_load_bus_error_exception( psxcpu );
					}
					else
					{
						mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), data );
					}
				}
			}
			break;

		case OP_LBU:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_byte_address_mask ) != 0 )
				{
					mips_load_bad_address( psxcpu, address );
				}
				else if( breakpoint )
				{
					mips_breakpoint_exception( psxcpu );
				}
				else
				{
					UINT32 data = psx_readbyte( psxcpu, address );

					if( psxcpu->berr )
					{
						mips_load_bus_error_exception( psxcpu );
					}
					else
					{
						mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), data );
					}
				}
			}
			break;

		case OP_LHU:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_half_address_mask ) != 0 )
				{
					mips_load_bad_address( psxcpu, address );
				}
				else if( breakpoint )
				{
					mips_breakpoint_exception( psxcpu );
				}
				else
				{
					UINT32 data = psx_readhalf( psxcpu, address );

					if( psxcpu->berr )
					{
						mips_load_bus_error_exception( psxcpu );
					}
					else
					{
						mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), data );
					}
				}
			}
			break;

		case OP_LWR:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_load_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_byte_address_mask ) != 0 )
				{
					mips_load_bad_address( psxcpu, address );
				}
				else if( breakpoint )
				{
					mips_breakpoint_exception( psxcpu );
				}
				else
				{
					UINT32 data = mips_get_register_from_pipeline( psxcpu, INS_RT( psxcpu->op ) );

					switch( address & 3 )
					{
					case 0:
						data = psx_readword( psxcpu, address );
						break;

					case 1:
						data = ( data & 0xff000000 ) | ( psx_readword_masked( psxcpu, address, 0x00ffffff ) >> 8 );
						break;

					case 2:
						data = ( data & 0xffff0000 ) | ( psx_readword_masked( psxcpu, address, 0xffff0000 ) >> 16 );
						break;

					case 3:
						data = ( data & 0xffffff00 ) | ( psx_readword_masked( psxcpu, address, 0xff000000 ) >> 24 );
						break;
					}

					if( psxcpu->berr )
					{
						mips_load_bus_error_exception( psxcpu );
					}
					else
					{
						mips_delayed_load( psxcpu, INS_RT( psxcpu->op ), data );
					}
				}
			}
			break;

		case OP_SB:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_store_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_byte_address_mask ) != 0 )
				{
					mips_store_bad_address( psxcpu, address );
				}
				else
				{
					int shift = 8 * ( address & 3 );
					psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] << shift, 0xff << shift );

					if( breakpoint )
					{
						mips_breakpoint_exception( psxcpu );
					}
					else if( psxcpu->berr )
					{
						mips_store_bus_error_exception( psxcpu );
					}
					else
					{
						mips_advance_pc( psxcpu );
					}
				}
			}
			break;

		case OP_SH:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_store_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_half_address_mask ) != 0 )
				{
					mips_store_bad_address( psxcpu, address );
				}
				else
				{
					int shift = 8 * ( address & 2 );
					psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] << shift, 0xffff << shift );

					if( breakpoint )
					{
						mips_breakpoint_exception( psxcpu );
					}
					else if( psxcpu->berr )
					{
						mips_store_bus_error_exception( psxcpu );
					}
					else
					{
						mips_advance_pc( psxcpu );
					}
				}
			}
			break;

		case OP_SWL:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int save_type = address & 3;
				int breakpoint;

				address &= ~3;
				breakpoint = mips_store_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_byte_address_mask ) != 0 )
				{
					mips_store_bad_address( psxcpu, address );
				}
				else
				{
					switch( save_type )
					{
					case 0:
						psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] >> 24, 0x000000ff );
						break;

					case 1:
						psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] >> 16, 0x0000ffff );
						break;

					case 2:
						psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] >> 8, 0x00ffffff );
						break;

					case 3:
						psx_writeword( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] );
						break;
					}

					if( breakpoint )
					{
						mips_breakpoint_exception( psxcpu );
					}
					else if( psxcpu->berr )
					{
						mips_store_bus_error_exception( psxcpu );
					}
					else
					{
						mips_advance_pc( psxcpu );
					}
				}
			}
			break;

		case OP_SW:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_store_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_word_address_mask ) != 0 )
				{
					mips_store_bad_address( psxcpu, address );
				}
				else
				{
					psx_writeword( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] );

					if( breakpoint )
					{
						mips_breakpoint_exception( psxcpu );
					}
					else if( psxcpu->berr )
					{
						mips_store_bus_error_exception( psxcpu );
					}
					else
					{
						mips_advance_pc( psxcpu );
					}
				}
			}
			break;

		case OP_SWR:
			{
				UINT32 address = psxcpu->r[ INS_RS( psxcpu->op ) ] + PSXCPU_WORD_EXTEND( INS_IMMEDIATE( psxcpu->op ) );
				int breakpoint = mips_store_data_address_breakpoint( psxcpu, address );

				if( ( address & psxcpu->bad_byte_address_mask ) != 0 )
				{
					mips_store_bad_address( psxcpu, address );
				}
				else
				{
					switch( address & 3 )
					{
					case 0:
						psx_writeword( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] );
						break;

					case 1:
						psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] << 8, 0xffffff00 );
						break;

					case 2:
						psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] << 16, 0xffff0000 );
						break;

					case 3:
						psx_writeword_masked( psxcpu, address, psxcpu->r[ INS_RT( psxcpu->op ) ] << 24, 0xff000000 );
						break;
					}

					if( breakpoint )
					{
						mips_breakpoint_exception( psxcpu );
					}
					else if( psxcpu->berr )
					{
						mips_store_bus_error_exception( psxcpu );
					}
					else
					{
						mips_advance_pc( psxcpu );
					}
				}
			}
			break;

		case OP_LWC0:
			mips_lwc( psxcpu, 0, SR_CU0 );
			break;

		case OP_LWC1:
			mips_lwc( psxcpu, 1, SR_CU1 );
			break;

		case OP_LWC2:
			mips_lwc( psxcpu, 2, SR_CU2 );
			break;

		case OP_LWC3:
			mips_lwc( psxcpu, 3, SR_CU3 );
			break;

		case OP_SWC0:
			mips_swc( psxcpu, 0, SR_CU0 );
			break;

		case OP_SWC1:
			mips_swc( psxcpu, 1, SR_CU1 );
			break;

		case OP_SWC2:
			mips_swc( psxcpu, 2, SR_CU2 );
			break;

		case OP_SWC3:
			mips_swc( psxcpu, 3, SR_CU3 );
			break;

		default:
			logerror( "%08x: unknown opcode %08x\n", psxcpu->pc, psxcpu->op );
			mips_stop( psxcpu );
			mips_exception( psxcpu, EXC_RI );
			break;
		}
		psxcpu->icount--;
	} while( psxcpu->icount > 0 );

	return cycles - psxcpu->icount;
}

static void set_irq_line( psxcpu_state *psxcpu, int irqline, int state )
{
	UINT32 ip;

	switch( irqline )
	{
	case PSXCPU_IRQ0:
		ip = CAUSE_IP2;
		break;

	case PSXCPU_IRQ1:
		ip = CAUSE_IP3;
		break;

	case PSXCPU_IRQ2:
		ip = CAUSE_IP4;
		break;

	case PSXCPU_IRQ3:
		ip = CAUSE_IP5;
		break;

	case PSXCPU_IRQ4:
		ip = CAUSE_IP6;
		break;

	case PSXCPU_IRQ5:
		ip = CAUSE_IP7;
		break;

	default:
		return;
	}

	switch( state )
	{
	case CLEAR_LINE:
		mips_set_cp0r( psxcpu, CP0_CAUSE, psxcpu->cp0r[ CP0_CAUSE ] & ~ip );
		break;

	case ASSERT_LINE:
		mips_set_cp0r( psxcpu, CP0_CAUSE, psxcpu->cp0r[ CP0_CAUSE ] |= ip );

		if( psxcpu->irq_callback )
		{
			/* HOLD_LINE interrupts are not supported by the architecture.
            By acknowledging the interupt here they are treated like PULSE_LINE
            interrupts, so if the interrupt isn't enabled it will be ignored.
            There is also a problem with PULSE_LINE interrupts as the interrupt
            pending bits aren't latched the emulated code won't know what caused
            the interrupt. */
			(*psxcpu->irq_callback)( psxcpu->device, irqline );
		}
		break;
	}
}

static READ32_HANDLER( psx_biu_r )
{
	psxcpu_state *psxcpu = get_safe_token(space->cpu);

	return psxcpu->biu;
}

static WRITE32_HANDLER( psx_biu_w )
{
	psxcpu_state *psxcpu = get_safe_token(space->cpu);
	UINT32 old = psxcpu->biu;

	COMBINE_DATA( &psxcpu->biu );

	if( ( old & ( BIU_RAM | BIU_DS ) ) != ( psxcpu->biu & ( BIU_RAM | BIU_DS ) ) )
	{
		mips_update_scratchpad( space );
	}
}

// On-board RAM and peripherals
static ADDRESS_MAP_START( psxcpu_internal_map, ADDRESS_SPACE_PROGRAM, 32 )
	AM_RANGE(0x00800000, 0x1effffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0x1f800400, 0x1f800fff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0x20000000, 0x7fffffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0x80800000, 0x9effffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0xa0800000, 0xbeffffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0xc0000000, 0xfffdffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0xfffe0130, 0xfffe0133) AM_READWRITE(psx_biu_r, psx_biu_w)
ADDRESS_MAP_END

static ADDRESS_MAP_START( cxd8661r_internal_map, ADDRESS_SPACE_PROGRAM, 32 )
	AM_RANGE(0x01000000, 0x1effffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0x1f800400, 0x1f800fff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0x20000000, 0x7fffffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0x81000000, 0x9effffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0xa1000000, 0xbeffffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0xc0000000, 0xfffdffff) AM_READWRITE(psx_berr_r, psx_berr_w)
	AM_RANGE(0xfffe0130, 0xfffe0133) AM_READWRITE(psx_biu_r, psx_biu_w)
ADDRESS_MAP_END

/****************************************************************************
 * Return a formatted string for a register
 ****************************************************************************/

static CPU_DISASSEMBLE( psxcpu )
{
	psxcpu_state *psxcpu = get_safe_token(device);
	DasmPSXCPU_state state;

	state.pc = psxcpu->pc;
	state.delayr = psxcpu->delayr;
	state.delayv = psxcpu->delayv;
	memcpy( state.r, psxcpu->r, sizeof( state.r ) );
	return DasmPSXCPU( &state, buffer, pc, opram );
}


static UINT32 getcp1dr( psxcpu_state *psxcpu, int reg )
{
	/* if a mtc/ctc precedes then this will get the value moved (which cop1 register is irrelevant). */
	/* if a mfc/cfc follows then it will get the same value as this one. */
	return memory_read_dword_32le( psxcpu->program, psxcpu->pc + 4 );
}

static void setcp1dr( psxcpu_state *psxcpu, int reg, UINT32 value )
{
}

static UINT32 getcp1cr( psxcpu_state *psxcpu, int reg )
{
	/* if a mtc/ctc precedes then this will get the value moved (which cop1 register is irrelevant). */
	/* if a mfc/cfc follows then it will get the same value as this one. */
	return memory_read_dword_32le( psxcpu->program, psxcpu->pc + 4 );
}

static void setcp1cr( psxcpu_state *psxcpu, int reg, UINT32 value )
{
}


static UINT32 getcp3dr( psxcpu_state *psxcpu, int reg )
{
	/* if you have mtc/ctc with an mfc/cfc directly afterwards then you get the value that was moved. */
	/* if you have an lwc with an mfc/cfc somewhere after it then you get the value that is loaded */
	/* otherwise you get the next opcode. which register you transfer to or from is irrelevant. */
	return memory_read_dword_32le( psxcpu->program, psxcpu->pc + 4 );
}

static void setcp3dr( psxcpu_state *psxcpu, int reg, UINT32 value )
{
}

static UINT32 getcp3cr( psxcpu_state *psxcpu, int reg )
{
	/* if you have mtc/ctc with an mfc/cfc directly afterwards then you get the value that was moved. */
	/* if you have an lwc with an mfc/cfc somewhere after it then you get the value that is loaded */
	/* otherwise you get the next opcode. which register you transfer to or from is irrelevant. */
	return memory_read_dword_32le( psxcpu->program, psxcpu->pc + 4 );
}

static void setcp3cr( psxcpu_state *psxcpu, int reg, UINT32 value )
{
}


/* preliminary gte code */

#define VX( n ) ( n < 3 ? psxcpu->cp2dr[ n << 1 ].sw.l : psxcpu->cp2dr[ 9 ].sw.l )
#define VY( n ) ( n < 3 ? psxcpu->cp2dr[ n << 1 ].sw.h : psxcpu->cp2dr[ 10 ].sw.l )
#define VZ( n ) ( n < 3 ? psxcpu->cp2dr[ ( n << 1 ) + 1 ].sw.l : psxcpu->cp2dr[ 11 ].sw.l )
#define MX11( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) ].sw.l : 0 )
#define MX12( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) ].sw.h : 0 )
#define MX13( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 1 ].sw.l : 0 )
#define MX21( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 1 ].sw.h : 0 )
#define MX22( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 2 ].sw.l : 0 )
#define MX23( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 2 ].sw.h : 0 )
#define MX31( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 3 ].sw.l : 0 )
#define MX32( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 3 ].sw.h : 0 )
#define MX33( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 4 ].sw.l : 0 )
#define CV1( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 5 ].sd : 0 )
#define CV2( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 6 ].sd : 0 )
#define CV3( n ) ( n < 3 ? psxcpu->cp2cr[ ( n << 3 ) + 7 ].sd : 0 )

#define VXY0 ( psxcpu->cp2dr[ 0 ].d )
#define VX0  ( psxcpu->cp2dr[ 0 ].sw.l )
#define VY0  ( psxcpu->cp2dr[ 0 ].sw.h )
#define VZ0  ( psxcpu->cp2dr[ 1 ].sw.l )
#define VXY1 ( psxcpu->cp2dr[ 2 ].d )
#define VX1  ( psxcpu->cp2dr[ 2 ].w.l )
#define VY1  ( psxcpu->cp2dr[ 2 ].w.h )
#define VZ1  ( psxcpu->cp2dr[ 3 ].w.l )
#define VXY2 ( psxcpu->cp2dr[ 4 ].d )
#define VX2  ( psxcpu->cp2dr[ 4 ].w.l )
#define VY2  ( psxcpu->cp2dr[ 4 ].w.h )
#define VZ2  ( psxcpu->cp2dr[ 5 ].w.l )
#define RGB  ( psxcpu->cp2dr[ 6 ].d )
#define R    ( psxcpu->cp2dr[ 6 ].b.l )
#define G    ( psxcpu->cp2dr[ 6 ].b.h )
#define B    ( psxcpu->cp2dr[ 6 ].b.h2 )
#define CODE ( psxcpu->cp2dr[ 6 ].b.h3 )
#define OTZ  ( psxcpu->cp2dr[ 7 ].w.l )
#define IR0  ( psxcpu->cp2dr[ 8 ].sw.l )
#define IR1  ( psxcpu->cp2dr[ 9 ].sw.l )
#define IR2  ( psxcpu->cp2dr[ 10 ].sw.l )
#define IR3  ( psxcpu->cp2dr[ 11 ].sw.l )
#define SXY0 ( psxcpu->cp2dr[ 12 ].d )
#define SX0  ( psxcpu->cp2dr[ 12 ].sw.l )
#define SY0  ( psxcpu->cp2dr[ 12 ].sw.h )
#define SXY1 ( psxcpu->cp2dr[ 13 ].d )
#define SX1  ( psxcpu->cp2dr[ 13 ].sw.l )
#define SY1  ( psxcpu->cp2dr[ 13 ].sw.h )
#define SXY2 ( psxcpu->cp2dr[ 14 ].d )
#define SX2  ( psxcpu->cp2dr[ 14 ].sw.l )
#define SY2  ( psxcpu->cp2dr[ 14 ].sw.h )
#define SXYP ( psxcpu->cp2dr[ 15 ].d )
#define SXP  ( psxcpu->cp2dr[ 15 ].sw.l )
#define SYP  ( psxcpu->cp2dr[ 15 ].sw.h )
#define SZ0  ( psxcpu->cp2dr[ 16 ].w.l )
#define SZ1  ( psxcpu->cp2dr[ 17 ].w.l )
#define SZ2  ( psxcpu->cp2dr[ 18 ].w.l )
#define SZ3  ( psxcpu->cp2dr[ 19 ].w.l )
#define RGB0 ( psxcpu->cp2dr[ 20 ].d )
#define R0   ( psxcpu->cp2dr[ 20 ].b.l )
#define G0   ( psxcpu->cp2dr[ 20 ].b.h )
#define B0   ( psxcpu->cp2dr[ 20 ].b.h2 )
#define CD0  ( psxcpu->cp2dr[ 20 ].b.h3 )
#define RGB1 ( psxcpu->cp2dr[ 21 ].d )
#define R1   ( psxcpu->cp2dr[ 21 ].b.l )
#define G1   ( psxcpu->cp2dr[ 21 ].b.h )
#define B1   ( psxcpu->cp2dr[ 21 ].b.h2 )
#define CD1  ( psxcpu->cp2dr[ 21 ].b.h3 )
#define RGB2 ( psxcpu->cp2dr[ 22 ].d )
#define R2   ( psxcpu->cp2dr[ 22 ].b.l )
#define G2   ( psxcpu->cp2dr[ 22 ].b.h )
#define B2   ( psxcpu->cp2dr[ 22 ].b.h2 )
#define CD2  ( psxcpu->cp2dr[ 22 ].b.h3 )
#define RES1 ( psxcpu->cp2dr[ 23 ].d )
#define MAC0 ( psxcpu->cp2dr[ 24 ].sd )
#define MAC1 ( psxcpu->cp2dr[ 25 ].sd )
#define MAC2 ( psxcpu->cp2dr[ 26 ].sd )
#define MAC3 ( psxcpu->cp2dr[ 27 ].sd )
#define IRGB ( psxcpu->cp2dr[ 28 ].d )
#define ORGB ( psxcpu->cp2dr[ 29 ].d )
#define LZCS ( psxcpu->cp2dr[ 30 ].d )
#define LZCR ( psxcpu->cp2dr[ 31 ].d )

#define R11 ( psxcpu->cp2cr[ 0 ].sw.l )
#define R12 ( psxcpu->cp2cr[ 0 ].sw.h )
#define R13 ( psxcpu->cp2cr[ 1 ].sw.l )
#define R21 ( psxcpu->cp2cr[ 1 ].sw.h )
#define R22 ( psxcpu->cp2cr[ 2 ].sw.l )
#define R23 ( psxcpu->cp2cr[ 2 ].sw.h )
#define R31 ( psxcpu->cp2cr[ 3 ].sw.l )
#define R32 ( psxcpu->cp2cr[ 3 ].sw.h )
#define R33 ( psxcpu->cp2cr[ 4 ].sw.l )
#define TRX ( psxcpu->cp2cr[ 5 ].sd )
#define TRY ( psxcpu->cp2cr[ 6 ].sd )
#define TRZ ( psxcpu->cp2cr[ 7 ].sd )
#define L11 ( psxcpu->cp2cr[ 8 ].sw.l )
#define L12 ( psxcpu->cp2cr[ 8 ].sw.h )
#define L13 ( psxcpu->cp2cr[ 9 ].sw.l )
#define L21 ( psxcpu->cp2cr[ 9 ].sw.h )
#define L22 ( psxcpu->cp2cr[ 10 ].sw.l )
#define L23 ( psxcpu->cp2cr[ 10 ].sw.h )
#define L31 ( psxcpu->cp2cr[ 11 ].sw.l )
#define L32 ( psxcpu->cp2cr[ 11 ].sw.h )
#define L33 ( psxcpu->cp2cr[ 12 ].sw.l )
#define RBK ( psxcpu->cp2cr[ 13 ].sd )
#define GBK ( psxcpu->cp2cr[ 14 ].sd )
#define BBK ( psxcpu->cp2cr[ 15 ].sd )
#define LR1 ( psxcpu->cp2cr[ 16 ].sw.l )
#define LR2 ( psxcpu->cp2cr[ 16 ].sw.h )
#define LR3 ( psxcpu->cp2cr[ 17 ].sw.l )
#define LG1 ( psxcpu->cp2cr[ 17 ].sw.h )
#define LG2 ( psxcpu->cp2cr[ 18 ].sw.l )
#define LG3 ( psxcpu->cp2cr[ 18 ].sw.h )
#define LB1 ( psxcpu->cp2cr[ 19 ].sw.l )
#define LB2 ( psxcpu->cp2cr[ 19 ].sw.h )
#define LB3 ( psxcpu->cp2cr[ 20 ].sw.l )
#define RFC ( psxcpu->cp2cr[ 21 ].sd )
#define GFC ( psxcpu->cp2cr[ 22 ].sd )
#define BFC ( psxcpu->cp2cr[ 23 ].sd )
#define OFX ( psxcpu->cp2cr[ 24 ].sd )
#define OFY ( psxcpu->cp2cr[ 25 ].sd )
#define H   ( psxcpu->cp2cr[ 26 ].sw.l )
#define DQA ( psxcpu->cp2cr[ 27 ].sw.l )
#define DQB ( psxcpu->cp2cr[ 28 ].sd )
#define ZSF3 ( psxcpu->cp2cr[ 29 ].sw.l )
#define ZSF4 ( psxcpu->cp2cr[ 30 ].sw.l )
#define FLAG ( psxcpu->cp2cr[ 31 ].d )

INLINE INT32 LIM( psxcpu_state *psxcpu, INT32 value, INT32 max, INT32 min, UINT32 flag )
{
	if( value > max )
	{
		FLAG |= flag;
		return max;
	}
	else if( value < min )
	{
		FLAG |= flag;
		return min;
	}
	return value;
}

static UINT32 getcp2dr( psxcpu_state *psxcpu, int reg )
{
	switch( reg )
	{
	case 1:
	case 3:
	case 5:
	case 8:
	case 9:
	case 10:
	case 11:
		psxcpu->cp2dr[ reg ].d = (INT32)psxcpu->cp2dr[ reg ].sw.l;
		break;

	case 7:
	case 16:
	case 17:
	case 18:
	case 19:
		psxcpu->cp2dr[ reg ].d = (UINT32)psxcpu->cp2dr[ reg ].w.l;
		break;

	case 15:
		psxcpu->cp2dr[ reg ].d = SXY2;
		break;

	case 28:
	case 29:
		psxcpu->cp2dr[ reg ].d = LIM( psxcpu, IR1 >> 7, 0x1f, 0, 0 ) | ( LIM( psxcpu, IR2 >> 7, 0x1f, 0, 0 ) << 5 ) | ( LIM( psxcpu, IR3 >> 7, 0x1f, 0, 0 ) << 10 );
		break;
	}

	GTELOG( "get CP2DR%u=%08x", reg, psxcpu->cp2dr[ reg ].d );
	return psxcpu->cp2dr[ reg ].d;
}

static void setcp2dr( psxcpu_state *psxcpu, int reg, UINT32 value )
{
	GTELOG( "set CP2DR%u=%08x", reg, value );

	switch( reg )
	{
	case 15:
		SXY0 = SXY1;
		SXY1 = SXY2;
		SXY2 = value;
		break;

	case 28:
		IR1 = ( value & 0x1f ) << 7;
		IR2 = ( value & 0x3e0 ) << 2;
		IR3 = ( value & 0x7c00 ) >> 3;
		break;

	case 30:
	{
		UINT32 lzcs = value;
		UINT32 lzcr = 0;

		if( ( lzcs & 0x80000000 ) == 0 )
		{
			lzcs = ~lzcs;
		}
		while( ( lzcs & 0x80000000 ) != 0 )
		{
			lzcr++;
			lzcs <<= 1;
		}
		LZCR = lzcr;
		break;
	}

	case 31:
		value = psxcpu->cp2dr[ reg ].d;
		break;
	}

	psxcpu->cp2dr[ reg ].d = value;
}

static UINT32 getcp2cr( psxcpu_state *psxcpu, int reg )
{
	GTELOG( "get CP2CR%u=%08x", reg, psxcpu->cp2cr[ reg ].d );

	return psxcpu->cp2cr[ reg ].d;
}

static void setcp2cr( psxcpu_state *psxcpu, int reg, UINT32 value )
{
	GTELOG( "set CP2CR%u=%08x", reg, value );

	switch( reg )
	{
	case 4:
	case 12:
	case 20:
	case 26:
	case 27:
	case 29:
	case 30:
		value = (INT32)(INT16) value;
		break;

	case 31:
		value = value & 0x7ffff000;
		if( ( value & 0x7f87e000 ) != 0 )
		{
			value |= 0x80000000;
		}
		break;
	}

	psxcpu->cp2cr[ reg ].d = value;
}

INLINE INT64 BOUNDS( psxcpu_state *psxcpu, INT64 n_value, INT64 n_max, int n_maxflag, INT64 n_min, int n_minflag )
{
	if( n_value > n_max )
	{
		FLAG |= n_maxflag;
	}
	else if( n_value < n_min )
	{
		FLAG |= n_minflag;
	}
	return n_value;
}

static const UINT16 reciprocals[ 32768 ]=
{
	0x0,0xfffc,0xfff8,0xfff4,0xfff0,0xffec,0xffe8,0xffe4,0xffe0,0xffdc,0xffd8,0xffd4,0xffd0,0xffcc,0xffc8,0xffc4,
	0xffc0,0xffbc,0xffb8,0xffb4,0xffb0,0xffac,0xffa8,0xffa4,0xffa0,0xff9c,0xff98,0xff94,0xff90,0xff8c,0xff88,0xff84,
	0xff80,0xff7c,0xff78,0xff74,0xff70,0xff6c,0xff68,0xff64,0xff60,0xff5c,0xff58,0xff54,0xff50,0xff4c,0xff48,0xff44,
	0xff40,0xff3c,0xff38,0xff34,0xff30,0xff2c,0xff28,0xff24,0xff20,0xff1c,0xff18,0xff14,0xff10,0xff0c,0xff08,0xff04,
	0xff01,0xfefd,0xfef9,0xfef5,0xfef1,0xfeed,0xfee9,0xfee5,0xfee1,0xfedd,0xfed9,0xfed5,0xfed1,0xfecd,0xfec9,0xfec5,
	0xfec1,0xfebd,0xfeb9,0xfeb5,0xfeb1,0xfead,0xfea9,0xfea5,0xfea1,0xfe9d,0xfe99,0xfe95,0xfe91,0xfe8d,0xfe89,0xfe85,
	0xfe81,0xfe7e,0xfe7a,0xfe76,0xfe72,0xfe6e,0xfe6a,0xfe66,0xfe62,0xfe5e,0xfe5a,0xfe56,0xfe52,0xfe4e,0xfe4a,0xfe46,
	0xfe42,0xfe3e,0xfe3a,0xfe36,0xfe32,0xfe2e,0xfe2a,0xfe26,0xfe22,0xfe1e,0xfe1a,0xfe16,0xfe12,0xfe0e,0xfe0a,0xfe06,
	0xfe02,0xfdfe,0xfdfa,0xfdf6,0xfdf2,0xfdee,0xfdea,0xfde6,0xfde2,0xfdde,0xfdda,0xfdd6,0xfdd2,0xfdce,0xfdca,0xfdc6,
	0xfdc2,0xfdbe,0xfdba,0xfdb6,0xfdb2,0xfdae,0xfdaa,0xfda6,0xfda2,0xfd9e,0xfd9a,0xfd96,0xfd92,0xfd8e,0xfd8a,0xfd86,
	0xfd82,0xfd7f,0xfd7b,0xfd77,0xfd73,0xfd6f,0xfd6b,0xfd67,0xfd63,0xfd5f,0xfd5b,0xfd57,0xfd53,0xfd4f,0xfd4b,0xfd47,
	0xfd43,0xfd3f,0xfd3b,0xfd37,0xfd33,0xfd2f,0xfd2b,0xfd27,0xfd23,0xfd1f,0xfd1b,0xfd17,0xfd13,0xfd0f,0xfd0b,0xfd07,
	0xfd04,0xfd00,0xfcfc,0xfcf8,0xfcf4,0xfcf0,0xfcec,0xfce8,0xfce4,0xfce0,0xfcdc,0xfcd8,0xfcd4,0xfcd0,0xfccc,0xfcc8,
	0xfcc4,0xfcc0,0xfcbd,0xfcb9,0xfcb5,0xfcb1,0xfcad,0xfca9,0xfca5,0xfca1,0xfc9d,0xfc99,0xfc95,0xfc91,0xfc8d,0xfc89,
	0xfc87,0xfc83,0xfc7f,0xfc7b,0xfc77,0xfc73,0xfc6f,0xfc6b,0xfc67,0xfc63,0xfc5f,0xfc5b,0xfc57,0xfc53,0xfc4f,0xfc4b,
	0xfc47,0xfc43,0xfc40,0xfc3c,0xfc38,0xfc34,0xfc30,0xfc2c,0xfc28,0xfc24,0xfc20,0xfc1c,0xfc18,0xfc14,0xfc10,0xfc0c,
	0xfc08,0xfc04,0xfc00,0xfbfc,0xfbf8,0xfbf4,0xfbf0,0xfbec,0xfbe8,0xfbe4,0xfbe0,0xfbdc,0xfbd8,0xfbd4,0xfbd0,0xfbcc,
	0xfbc8,0xfbc4,0xfbc1,0xfbbd,0xfbb9,0xfbb5,0xfbb1,0xfbad,0xfba9,0xfba5,0xfba1,0xfb9d,0xfb99,0xfb95,0xfb91,0xfb8d,
	0xfb8b,0xfb87,0xfb83,0xfb7f,0xfb7b,0xfb77,0xfb73,0xfb6f,0xfb6b,0xfb67,0xfb63,0xfb5f,0xfb5b,0xfb57,0xfb53,0xfb4f,
	0xfb4b,0xfb47,0xfb43,0xfb40,0xfb3c,0xfb38,0xfb34,0xfb30,0xfb2c,0xfb28,0xfb24,0xfb20,0xfb1c,0xfb18,0xfb14,0xfb10,
	0xfb0d,0xfb09,0xfb05,0xfb01,0xfafd,0xfaf9,0xfaf5,0xfaf1,0xfaed,0xfae9,0xfae5,0xfae1,0xfadd,0xfad9,0xfad5,0xfad2,
	0xface,0xfaca,0xfac6,0xfac2,0xfabe,0xfaba,0xfab6,0xfab2,0xfaae,0xfaaa,0xfaa6,0xfaa2,0xfa9e,0xfa9a,0xfa96,0xfa92,
	0xfa8e,0xfa8a,0xfa86,0xfa82,0xfa7f,0xfa7b,0xfa77,0xfa73,0xfa6f,0xfa6b,0xfa67,0xfa65,0xfa61,0xfa5d,0xfa59,0xfa55,
	0xfa51,0xfa4d,0xfa49,0xfa45,0xfa41,0xfa3d,0xfa39,0xfa35,0xfa31,0xfa2d,0xfa2a,0xfa26,0xfa22,0xfa1e,0xfa1a,0xfa16,
	0xfa12,0xfa0e,0xfa0a,0xfa06,0xfa02,0xf9fe,0xf9fa,0xf9f6,0xf9f2,0xf9ee,0xf9ea,0xf9e6,0xf9e2,0xf9de,0xf9da,0xf9d6,
	0xf9d3,0xf9cf,0xf9cb,0xf9c7,0xf9c3,0xf9bf,0xf9bd,0xf9b9,0xf9b5,0xf9b1,0xf9ad,0xf9a9,0xf9a5,0xf9a1,0xf99d,0xf999,
	0xf995,0xf991,0xf98d,0xf989,0xf985,0xf982,0xf97e,0xf97a,0xf976,0xf972,0xf96e,0xf96a,0xf966,0xf962,0xf95e,0xf95a,
	0xf956,0xf952,0xf94e,0xf94a,0xf946,0xf942,0xf93e,0xf93a,0xf936,0xf932,0xf92e,0xf92b,0xf927,0xf923,0xf91f,0xf91b,
	0xf918,0xf914,0xf910,0xf90c,0xf908,0xf904,0xf900,0xf8fc,0xf8f8,0xf8f4,0xf8f0,0xf8ec,0xf8e8,0xf8e4,0xf8e0,0xf8dd,
	0xf8db,0xf8d7,0xf8d3,0xf8cf,0xf8cb,0xf8c7,0xf8c3,0xf8bf,0xf8bb,0xf8b7,0xf8b3,0xf8af,0xf8ab,0xf8a7,0xf8a3,0xf89f,
	0xf89c,0xf898,0xf894,0xf890,0xf88c,0xf888,0xf884,0xf880,0xf87c,0xf878,0xf874,0xf870,0xf86c,0xf868,0xf864,0xf860,
	0xf85f,0xf85b,0xf857,0xf853,0xf84f,0xf84b,0xf847,0xf843,0xf83f,0xf83b,0xf837,0xf833,0xf82f,0xf82b,0xf827,0xf823,
	0xf820,0xf81c,0xf818,0xf814,0xf810,0xf80c,0xf808,0xf804,0xf800,0xf7fc,0xf7f8,0xf7f4,0xf7f0,0xf7ec,0xf7e8,0xf7e4,
	0xf7e2,0xf7df,0xf7db,0xf7d7,0xf7d3,0xf7cf,0xf7cb,0xf7c7,0xf7c3,0xf7bf,0xf7bb,0xf7b7,0xf7b3,0xf7af,0xf7ab,0xf7a7,
	0xf7a3,0xf7a0,0xf79c,0xf798,0xf794,0xf790,0xf78c,0xf788,0xf784,0xf780,0xf77c,0xf778,0xf774,0xf770,0xf76c,0xf768,
	0xf766,0xf763,0xf75f,0xf75b,0xf757,0xf753,0xf74f,0xf74b,0xf747,0xf743,0xf73f,0xf73b,0xf737,0xf733,0xf72f,0xf72b,
	0xf728,0xf724,0xf720,0xf71c,0xf718,0xf714,0xf711,0xf70d,0xf709,0xf705,0xf701,0xf6fd,0xf6f9,0xf6f5,0xf6f1,0xf6ed,
	0xf6e9,0xf6e5,0xf6e2,0xf6de,0xf6da,0xf6d6,0xf6d2,0xf6ce,0xf6ca,0xf6c6,0xf6c4,0xf6c0,0xf6bc,0xf6b8,0xf6b4,0xf6b0,
	0xf6ad,0xf6a9,0xf6a5,0xf6a1,0xf69d,0xf699,0xf695,0xf691,0xf68d,0xf689,0xf685,0xf681,0xf67e,0xf67a,0xf676,0xf672,
	0xf66e,0xf66a,0xf666,0xf662,0xf660,0xf65c,0xf658,0xf654,0xf650,0xf64c,0xf649,0xf645,0xf641,0xf63d,0xf639,0xf635,
	0xf631,0xf62d,0xf629,0xf625,0xf621,0xf61d,0xf619,0xf616,0xf612,0xf60e,0xf60a,0xf606,0xf602,0xf600,0xf5fc,0xf5f8,
	0xf5f4,0xf5f0,0xf5ec,0xf5e8,0xf5e5,0xf5e1,0xf5dd,0xf5d9,0xf5d5,0xf5d1,0xf5cd,0xf5c9,0xf5c5,0xf5c1,0xf5bd,0xf5b9,
	0xf5b5,0xf5b2,0xf5ae,0xf5aa,0xf5a6,0xf5a2,0xf59e,0xf59c,0xf598,0xf594,0xf590,0xf58c,0xf588,0xf584,0xf581,0xf57d,
	0xf579,0xf575,0xf571,0xf56d,0xf569,0xf565,0xf561,0xf55d,0xf559,0xf555,0xf551,0xf54e,0xf54a,0xf546,0xf542,0xf53e,
	0xf53a,0xf537,0xf533,0xf52f,0xf52b,0xf527,0xf523,0xf51f,0xf51b,0xf517,0xf513,0xf511,0xf50e,0xf50a,0xf506,0xf502,
	0xf4fe,0xf4fa,0xf4f6,0xf4f2,0xf4ee,0xf4ea,0xf4e6,0xf4e3,0xf4df,0xf4db,0xf4d7,0xf4d3,0xf4cf,0xf4cb,0xf4c7,0xf4c3,
	0xf4c1,0xf4bd,0xf4ba,0xf4b6,0xf4b2,0xf4ae,0xf4aa,0xf4a6,0xf4a2,0xf49e,0xf49a,0xf496,0xf492,0xf48f,0xf48b,0xf487,
	0xf483,0xf47f,0xf47b,0xf477,0xf473,0xf46f,0xf46d,0xf469,0xf466,0xf462,0xf45e,0xf45a,0xf456,0xf452,0xf44e,0xf44a,
	0xf446,0xf442,0xf43f,0xf43b,0xf437,0xf433,0xf42f,0xf42b,0xf427,0xf423,0xf41f,0xf41d,0xf419,0xf415,0xf412,0xf40e,
	0xf40a,0xf406,0xf402,0xf3fe,0xf3fa,0xf3f6,0xf3f2,0xf3ee,0xf3eb,0xf3e7,0xf3e3,0xf3df,0xf3db,0xf3d7,0xf3d3,0xf3cf,
	0xf3cd,0xf3c9,0xf3c5,0xf3c2,0xf3be,0xf3ba,0xf3b6,0xf3b2,0xf3ae,0xf3aa,0xf3a6,0xf3a2,0xf39e,0xf39a,0xf397,0xf393,
	0xf38f,0xf38b,0xf387,0xf383,0xf37f,0xf37b,0xf379,0xf375,0xf371,0xf36e,0xf36a,0xf366,0xf362,0xf35e,0xf35a,0xf356,
	0xf352,0xf34f,0xf34b,0xf347,0xf343,0xf33f,0xf33b,0xf337,0xf333,0xf32f,0xf32c,0xf328,0xf324,0xf320,0xf31c,0xf318,
	0xf314,0xf310,0xf30c,0xf30b,0xf307,0xf303,0xf2ff,0xf2fb,0xf2f7,0xf2f3,0xf2ef,0xf2eb,0xf2e7,0xf2e4,0xf2e0,0xf2dc,
	0xf2d8,0xf2d4,0xf2d0,0xf2cc,0xf2c8,0xf2c6,0xf2c3,0xf2bf,0xf2bb,0xf2b7,0xf2b3,0xf2af,0xf2ab,0xf2a7,0xf2a3,0xf2a0,
	0xf29c,0xf298,0xf294,0xf290,0xf28c,0xf288,0xf284,0xf282,0xf27e,0xf27b,0xf277,0xf273,0xf26f,0xf26b,0xf267,0xf263,
	0xf25f,0xf25b,0xf258,0xf254,0xf250,0xf24c,0xf248,0xf244,0xf240,0xf23c,0xf23a,0xf236,0xf233,0xf22f,0xf22b,0xf227,
	0xf223,0xf21f,0xf21b,0xf217,0xf213,0xf210,0xf20c,0xf208,0xf204,0xf200,0xf1fc,0xf1f8,0xf1f6,0xf1f2,0xf1ee,0xf1eb,
	0xf1e7,0xf1e3,0xf1df,0xf1db,0xf1d7,0xf1d3,0xf1cf,0xf1cb,0xf1c8,0xf1c4,0xf1c0,0xf1bc,0xf1b8,0xf1b4,0xf1b2,0xf1ae,
	0xf1aa,0xf1a7,0xf1a3,0xf19f,0xf19b,0xf197,0xf193,0xf18f,0xf18b,0xf187,0xf184,0xf180,0xf17c,0xf178,0xf174,0xf170,
	0xf16c,0xf168,0xf165,0xf161,0xf15d,0xf159,0xf155,0xf151,0xf14f,0xf14b,0xf147,0xf144,0xf140,0xf13c,0xf138,0xf134,
	0xf130,0xf12c,0xf128,0xf125,0xf121,0xf11d,0xf119,0xf115,0xf113,0xf10f,0xf10b,0xf108,0xf104,0xf100,0xf0fc,0xf0f8,
	0xf0f4,0xf0f0,0xf0ec,0xf0e9,0xf0e5,0xf0e1,0xf0dd,0xf0d9,0xf0d7,0xf0d3,0xf0cf,0xf0cb,0xf0c8,0xf0c4,0xf0c0,0xf0bc,
	0xf0b8,0xf0b4,0xf0b0,0xf0ac,0xf0a9,0xf0a5,0xf0a1,0xf09d,0xf09b,0xf097,0xf093,0xf08f,0xf08c,0xf088,0xf084,0xf080,
	0xf07c,0xf078,0xf074,0xf070,0xf06d,0xf069,0xf065,0xf061,0xf05f,0xf05b,0xf057,0xf053,0xf04f,0xf04c,0xf048,0xf044,
	0xf040,0xf03c,0xf038,0xf034,0xf030,0xf02d,0xf029,0xf025,0xf023,0xf01f,0xf01b,0xf017,0xf013,0xf010,0xf00c,0xf008,
	0xf004,0xf000,0xeffc,0xeff8,0xeff4,0xeff1,0xefed,0xefe9,0xefe7,0xefe3,0xefdf,0xefdb,0xefd7,0xefd3,0xefd0,0xefcc,
	0xefc8,0xefc4,0xefc0,0xefbc,0xefb8,0xefb4,0xefb1,0xefad,0xefab,0xefa7,0xefa3,0xef9f,0xef9b,0xef97,0xef94,0xef90,
	0xef8b,0xef87,0xef83,0xef80,0xef7e,0xef7a,0xef76,0xef72,0xef6e,0xef6a,0xef66,0xef63,0xef5f,0xef5b,0xef57,0xef53,
	0xef4f,0xef4b,0xef48,0xef46,0xef42,0xef3e,0xef3a,0xef36,0xef32,0xef2e,0xef2b,0xef27,0xef23,0xef1f,0xef1b,0xef17,
	0xef13,0xef0f,0xef0e,0xef0a,0xef06,0xef02,0xeefe,0xeefa,0xeef6,0xeef2,0xeeef,0xeeeb,0xeee7,0xeee3,0xeedf,0xeedb,
	0xeed7,0xeed5,0xeed2,0xeece,0xeeca,0xeec6,0xeec2,0xeebe,0xeeba,0xeeb7,0xeeb3,0xeeaf,0xeeab,0xeea7,0xeea3,0xee9f,
	0xee9d,0xee9a,0xee96,0xee92,0xee8e,0xee8a,0xee86,0xee82,0xee7e,0xee7b,0xee77,0xee73,0xee6f,0xee6b,0xee67,0xee63,
	0xee61,0xee5e,0xee5a,0xee56,0xee52,0xee4e,0xee4a,0xee46,0xee43,0xee3f,0xee3b,0xee37,0xee33,0xee2f,0xee2b,0xee29,
	0xee26,0xee22,0xee1e,0xee1a,0xee16,0xee12,0xee0e,0xee0a,0xee07,0xee03,0xedff,0xedfb,0xedf7,0xedf3,0xedf1,0xeded,
	0xedea,0xede6,0xede2,0xedde,0xedda,0xedd6,0xedd2,0xedce,0xedcb,0xedc7,0xedc3,0xedbf,0xedbb,0xedb9,0xedb5,0xedb1,
	0xedad,0xeda9,0xeda6,0xeda2,0xeda0,0xed9c,0xed98,0xed94,0xed90,0xed8d,0xed89,0xed85,0xed81,0xed7d,0xed79,0xed75,
	0xed72,0xed70,0xed6c,0xed68,0xed64,0xed60,0xed5c,0xed59,0xed55,0xed51,0xed4d,0xed49,0xed45,0xed41,0xed3e,0xed3c,
	0xed38,0xed34,0xed30,0xed2c,0xed28,0xed25,0xed21,0xed1d,0xed19,0xed15,0xed11,0xed0d,0xed0c,0xed08,0xed04,0xed00,
	0xecfc,0xecf8,0xecf4,0xecf1,0xeced,0xece9,0xece5,0xece1,0xecdd,0xecd9,0xecd8,0xecd4,0xecd0,0xeccc,0xecc8,0xecc4,
	0xecc0,0xecbd,0xecb9,0xecb5,0xecb1,0xecad,0xeca9,0xeca7,0xeca4,0xeca0,0xec9c,0xec98,0xec94,0xec90,0xec8c,0xec89,
	0xec85,0xec81,0xec7d,0xec79,0xec75,0xec73,0xec70,0xec6c,0xec68,0xec64,0xec60,0xec5c,0xec58,0xec55,0xec51,0xec4d,
	0xec49,0xec45,0xec43,0xec3f,0xec3c,0xec38,0xec34,0xec30,0xec2c,0xec28,0xec24,0xec21,0xec1d,0xec19,0xec15,0xec11,
	0xec0f,0xec0b,0xec08,0xec04,0xec00,0xebfc,0xebf8,0xebf4,0xebf0,0xebed,0xebe9,0xebe5,0xebe1,0xebdf,0xebdb,0xebd7,
	0xebd3,0xebcf,0xebcb,0xebc7,0xebc6,0xebc2,0xebbe,0xebba,0xebb6,0xebb2,0xebaf,0xebab,0xeba7,0xeba3,0xeb9f,0xeb9b,
	0xeb99,0xeb96,0xeb92,0xeb8e,0xeb8a,0xeb86,0xeb82,0xeb7f,0xeb7b,0xeb77,0xeb73,0xeb6f,0xeb6d,0xeb69,0xeb66,0xeb62,
	0xeb5e,0xeb5a,0xeb56,0xeb52,0xeb4f,0xeb4b,0xeb47,0xeb43,0xeb41,0xeb3d,0xeb3a,0xeb36,0xeb32,0xeb2e,0xeb2a,0xeb26,
	0xeb23,0xeb1f,0xeb1b,0xeb17,0xeb15,0xeb11,0xeb0d,0xeb0a,0xeb06,0xeb02,0xeafe,0xeafa,0xeaf6,0xeaf3,0xeaef,0xeaeb,
	0xeae9,0xeae5,0xeae1,0xeadd,0xeada,0xead6,0xead2,0xeace,0xeaca,0xeac6,0xeac3,0xeabf,0xeabb,0xeab9,0xeab5,0xeab1,
	0xeaae,0xeaaa,0xeaa6,0xeaa2,0xea9e,0xea9a,0xea97,0xea93,0xea8f,0xea8d,0xea89,0xea85,0xea81,0xea7e,0xea7a,0xea76,
	0xea72,0xea6e,0xea6a,0xea67,0xea63,0xea61,0xea5d,0xea59,0xea55,0xea51,0xea4e,0xea4a,0xea46,0xea42,0xea3e,0xea3a,
	0xea37,0xea35,0xea31,0xea2d,0xea29,0xea25,0xea22,0xea1e,0xea1a,0xea16,0xea12,0xea0e,0xea0b,0xea09,0xea05,0xea01,
	0xe9fc,0xe9f8,0xe9f5,0xe9f3,0xe9ef,0xe9eb,0xe9e7,0xe9e3,0xe9df,0xe9dc,0xe9d8,0xe9d4,0xe9d0,0xe9cc,0xe9ca,0xe9c7,
	0xe9c3,0xe9bf,0xe9bb,0xe9b7,0xe9b4,0xe9b0,0xe9ac,0xe9a8,0xe9a4,0xe9a0,0xe99f,0xe99b,0xe997,0xe993,0xe98f,0xe98b,
	0xe988,0xe984,0xe980,0xe97c,0xe978,0xe976,0xe973,0xe96f,0xe96b,0xe967,0xe963,0xe960,0xe95c,0xe958,0xe954,0xe950,
	0xe94e,0xe94a,0xe947,0xe943,0xe93f,0xe93b,0xe937,0xe934,0xe930,0xe92c,0xe928,0xe926,0xe922,0xe91f,0xe91b,0xe917,
	0xe913,0xe90f,0xe90b,0xe908,0xe904,0xe900,0xe8fe,0xe8fa,0xe8f6,0xe8f3,0xe8ef,0xe8eb,0xe8e7,0xe8e3,0xe8e0,0xe8dc,
	0xe8d8,0xe8d6,0xe8d2,0xe8ce,0xe8cb,0xe8c7,0xe8c3,0xe8bf,0xe8bb,0xe8b7,0xe8b4,0xe8b0,0xe8ae,0xe8aa,0xe8a6,0xe8a2,
	0xe89f,0xe89b,0xe897,0xe893,0xe88f,0xe88b,0xe888,0xe886,0xe882,0xe87e,0xe87a,0xe876,0xe873,0xe86f,0xe86b,0xe867,
	0xe863,0xe860,0xe85c,0xe85a,0xe856,0xe852,0xe84e,0xe84b,0xe847,0xe843,0xe83f,0xe83b,0xe837,0xe834,0xe832,0xe82e,
	0xe829,0xe825,0xe821,0xe81f,0xe81b,0xe818,0xe814,0xe810,0xe80c,0xe808,0xe805,0xe801,0xe7fd,0xe7fb,0xe7f7,0xe7f4,
	0xe7f0,0xe7ec,0xe7e8,0xe7e4,0xe7e0,0xe7dd,0xe7d9,0xe7d5,0xe7d3,0xe7cf,0xe7cc,0xe7c8,0xe7c4,0xe7c0,0xe7bc,0xe7b9,
	0xe7b5,0xe7b1,0xe7af,0xe7ab,0xe7a7,0xe7a4,0xe7a0,0xe79c,0xe798,0xe794,0xe791,0xe78d,0xe78b,0xe787,0xe783,0xe77f,
	0xe77c,0xe778,0xe774,0xe770,0xe76c,0xe769,0xe767,0xe763,0xe75f,0xe75b,0xe758,0xe754,0xe750,0xe74c,0xe748,0xe744,
	0xe743,0xe73f,0xe73b,0xe737,0xe733,0xe730,0xe72c,0xe728,0xe724,0xe720,0xe71d,0xe71b,0xe717,0xe713,0xe70f,0xe70b,
	0xe708,0xe704,0xe700,0xe6fc,0xe6f8,0xe6f6,0xe6f3,0xe6ef,0xe6eb,0xe6e7,0xe6e3,0xe6e0,0xe6dc,0xe6d8,0xe6d4,0xe6d2,
	0xe6cf,0xe6cb,0xe6c7,0xe6c3,0xe6bf,0xe6bc,0xe6b8,0xe6b4,0xe6b0,0xe6ae,0xe6aa,0xe6a7,0xe6a3,0xe69f,0xe69b,0xe697,
	0xe694,0xe690,0xe68c,0xe688,0xe686,0xe682,0xe67f,0xe67b,0xe677,0xe673,0xe66f,0xe66c,0xe668,0xe664,0xe662,0xe65e,
	0xe659,0xe655,0xe651,0xe64f,0xe64c,0xe648,0xe644,0xe640,0xe63c,0xe639,0xe635,0xe631,0xe62f,0xe62b,0xe628,0xe624,
	0xe620,0xe61c,0xe618,0xe615,0xe611,0xe60d,0xe60b,0xe607,0xe604,0xe600,0xe5fc,0xe5f8,0xe5f4,0xe5f1,0xe5ed,0xe5eb,
	0xe5e7,0xe5e3,0xe5e0,0xe5dc,0xe5d8,0xe5d4,0xe5d0,0xe5cd,0xe5c9,0xe5c7,0xe5c3,0xe5bf,0xe5bc,0xe5b8,0xe5b4,0xe5b0,
	0xe5ac,0xe5a9,0xe5a7,0xe5a3,0xe59f,0xe59b,0xe598,0xe594,0xe590,0xe58c,0xe588,0xe585,0xe583,0xe57f,0xe57b,0xe577,
	0xe574,0xe570,0xe56c,0xe568,0xe564,0xe563,0xe55f,0xe55b,0xe557,0xe553,0xe550,0xe54c,0xe548,0xe544,0xe540,0xe53f,
	0xe53b,0xe537,0xe533,0xe52f,0xe52c,0xe528,0xe524,0xe520,0xe51e,0xe51b,0xe517,0xe513,0xe50f,0xe50b,0xe508,0xe504,
	0xe500,0xe4fc,0xe4fa,0xe4f7,0xe4f3,0xe4ef,0xe4eb,0xe4e7,0xe4e4,0xe4e0,0xe4dc,0xe4da,0xe4d6,0xe4d3,0xe4cf,0xe4cb,
	0xe4c7,0xe4c3,0xe4c0,0xe4bc,0xe4b8,0xe4b6,0xe4b2,0xe4af,0xe4ab,0xe4a7,0xe4a3,0xe49f,0xe49c,0xe498,0xe496,0xe492,
	0xe48e,0xe48a,0xe486,0xe482,0xe47f,0xe47d,0xe479,0xe475,0xe471,0xe46e,0xe46a,0xe466,0xe462,0xe45f,0xe45d,0xe459,
	0xe455,0xe451,0xe44e,0xe44a,0xe446,0xe442,0xe43e,0xe43d,0xe439,0xe435,0xe431,0xe42d,0xe42a,0xe426,0xe422,0xe41e,
	0xe41c,0xe419,0xe415,0xe411,0xe40d,0xe409,0xe406,0xe402,0xe3fe,0xe3fa,0xe3f8,0xe3f5,0xe3f1,0xe3ed,0xe3e9,0xe3e6,
	0xe3e2,0xe3de,0xe3da,0xe3d8,0xe3d5,0xe3d1,0xe3cd,0xe3c9,0xe3c5,0xe3c2,0xe3be,0xe3ba,0xe3b8,0xe3b4,0xe3b1,0xe3ad,
	0xe3a9,0xe3a5,0xe3a1,0xe39e,0xe39a,0xe398,0xe394,0xe390,0xe38d,0xe389,0xe385,0xe381,0xe37e,0xe37a,0xe378,0xe374,
	0xe370,0xe36d,0xe369,0xe365,0xe361,0xe35d,0xe35a,0xe358,0xe354,0xe350,0xe34c,0xe349,0xe345,0xe341,0xe33d,0xe339,
	0xe338,0xe334,0xe330,0xe32c,0xe328,0xe325,0xe321,0xe31d,0xe319,0xe316,0xe314,0xe310,0xe30c,0xe308,0xe305,0xe301,
	0xe2fd,0xe2f9,0xe2f5,0xe2f4,0xe2f0,0xe2ec,0xe2e8,0xe2e4,0xe2e1,0xe2dd,0xe2d9,0xe2d5,0xe2d3,0xe2d0,0xe2cc,0xe2c8,
	0xe2c6,0xe2c2,0xe2be,0xe2ba,0xe2b7,0xe2b3,0xe2af,0xe2ab,0xe2a8,0xe2a6,0xe2a2,0xe29e,0xe29a,0xe297,0xe293,0xe28f,
	0xe28b,0xe288,0xe286,0xe282,0xe27e,0xe27a,0xe277,0xe273,0xe26f,0xe26b,0xe269,0xe266,0xe262,0xe25e,0xe25a,0xe257,
	0xe253,0xe24f,0xe24b,0xe249,0xe246,0xe242,0xe23e,0xe23a,0xe237,0xe233,0xe22f,0xe22d,0xe229,0xe226,0xe222,0xe21e,
	0xe21a,0xe217,0xe213,0xe20f,0xe20d,0xe209,0xe206,0xe202,0xe1fe,0xe1fa,0xe1f7,0xe1f3,0xe1f1,0xe1ed,0xe1e9,0xe1e6,
	0xe1e2,0xe1de,0xe1da,0xe1d7,0xe1d3,0xe1d1,0xe1cd,0xe1c9,0xe1c6,0xe1c2,0xe1be,0xe1ba,0xe1b7,0xe1b5,0xe1b1,0xe1ad,
	0xe1a9,0xe1a6,0xe1a2,0xe19e,0xe19a,0xe197,0xe195,0xe191,0xe18d,0xe189,0xe186,0xe182,0xe17e,0xe17a,0xe178,0xe175,
	0xe171,0xe16d,0xe169,0xe166,0xe162,0xe15e,0xe15a,0xe158,0xe155,0xe151,0xe14d,0xe149,0xe146,0xe142,0xe13e,0xe13c,
	0xe138,0xe135,0xe131,0xe12d,0xe129,0xe126,0xe122,0xe11e,0xe11c,0xe118,0xe115,0xe111,0xe10d,0xe109,0xe106,0xe102,
	0xe0ff,0xe0fb,0xe0f8,0xe0f4,0xe0f2,0xe0ee,0xe0ea,0xe0e7,0xe0e3,0xe0df,0xe0db,0xe0d8,0xe0d6,0xe0d2,0xe0ce,0xe0cb,
	0xe0c7,0xe0c3,0xe0bf,0xe0bc,0xe0ba,0xe0b6,0xe0b2,0xe0ae,0xe0ab,0xe0a7,0xe0a3,0xe09f,0xe09e,0xe09a,0xe096,0xe092,
	0xe08f,0xe08b,0xe087,0xe083,0xe081,0xe07e,0xe07a,0xe076,0xe072,0xe06f,0xe06b,0xe067,0xe065,0xe062,0xe05e,0xe05a,
	0xe056,0xe053,0xe04f,0xe04b,0xe049,0xe045,0xe042,0xe03e,0xe03a,0xe036,0xe033,0xe02f,0xe02d,0xe029,0xe026,0xe022,
	0xe01e,0xe01a,0xe017,0xe013,0xe011,0xe00d,0xe009,0xe006,0xe002,0xdffe,0xdffa,0xdff7,0xdff5,0xdff1,0xdfed,0xdfea,
	0xdfe6,0xdfe2,0xdfde,0xdfdb,0xdfd9,0xdfd5,0xdfd1,0xdfcd,0xdfca,0xdfc6,0xdfc2,0xdfbe,0xdfbd,0xdfb9,0xdfb5,0xdfb1,
	0xdfae,0xdfaa,0xdfa6,0xdfa2,0xdfa0,0xdf9d,0xdf99,0xdf95,0xdf91,0xdf8e,0xdf8a,0xdf86,0xdf84,0xdf81,0xdf7d,0xdf79,
	0xdf75,0xdf72,0xdf6e,0xdf6a,0xdf68,0xdf64,0xdf61,0xdf5d,0xdf59,0xdf55,0xdf52,0xdf4e,0xdf4c,0xdf48,0xdf45,0xdf41,
	0xdf3d,0xdf3a,0xdf36,0xdf32,0xdf2e,0xdf2b,0xdf27,0xdf23,0xdf21,0xdf1e,0xdf1a,0xdf16,0xdf12,0xdf0f,0xdf0b,0xdf07,
	0xdf05,0xdf02,0xdefe,0xdefa,0xdef6,0xdef3,0xdeef,0xdeed,0xdee9,0xdee6,0xdee2,0xdede,0xdeda,0xded7,0xded3,0xded1,
	0xdecd,0xdeca,0xdec6,0xdec2,0xdebe,0xdebb,0xdeb9,0xdeb5,0xdeb1,0xdeae,0xdeaa,0xdea6,0xdea2,0xde9f,0xde9d,0xde99,
	0xde95,0xde92,0xde8e,0xde8a,0xde86,0xde85,0xde81,0xde7d,0xde79,0xde76,0xde72,0xde6e,0xde6a,0xde69,0xde65,0xde61,
	0xde5d,0xde5a,0xde56,0xde52,0xde50,0xde4d,0xde49,0xde45,0xde41,0xde3e,0xde3a,0xde36,0xde34,0xde31,0xde2d,0xde29,
	0xde25,0xde22,0xde1e,0xde1c,0xde18,0xde15,0xde11,0xde0d,0xde09,0xde06,0xde02,0xde00,0xddfc,0xddf9,0xddf5,0xddf1,
	0xdded,0xddea,0xdde8,0xdde4,0xdde0,0xdddd,0xddd9,0xddd5,0xddd1,0xddce,0xddcc,0xddc8,0xddc4,0xddc1,0xddbd,0xddb9,
	0xddb5,0xddb3,0xddb0,0xddac,0xdda8,0xdda5,0xdda1,0xdd9d,0xdd99,0xdd97,0xdd94,0xdd90,0xdd8c,0xdd89,0xdd85,0xdd81,
	0xdd7f,0xdd7b,0xdd77,0xdd74,0xdd70,0xdd6c,0xdd6a,0xdd66,0xdd63,0xdd5f,0xdd5b,0xdd58,0xdd54,0xdd52,0xdd4e,0xdd4b,
	0xdd47,0xdd43,0xdd3f,0xdd3c,0xdd38,0xdd36,0xdd32,0xdd2f,0xdd2b,0xdd27,0xdd23,0xdd20,0xdd1e,0xdd1a,0xdd16,0xdd13,
	0xdd0f,0xdd0b,0xdd07,0xdd06,0xdd02,0xdcfe,0xdcfa,0xdcf7,0xdcf3,0xdcef,0xdcec,0xdcea,0xdce6,0xdce2,0xdcde,0xdcdb,
	0xdcd7,0xdcd3,0xdcd1,0xdcce,0xdcca,0xdcc6,0xdcc3,0xdcbf,0xdcbb,0xdcb9,0xdcb5,0xdcb2,0xdcae,0xdcaa,0xdca7,0xdca3,
	0xdca1,0xdc9d,0xdc9a,0xdc96,0xdc92,0xdc8e,0xdc8b,0xdc87,0xdc85,0xdc81,0xdc7e,0xdc7a,0xdc76,0xdc72,0xdc6f,0xdc6d,
	0xdc69,0xdc65,0xdc62,0xdc5e,0xdc5a,0xdc56,0xdc55,0xdc51,0xdc4d,0xdc49,0xdc46,0xdc42,0xdc3e,0xdc3a,0xdc39,0xdc35,
	0xdc31,0xdc2d,0xdc2a,0xdc26,0xdc22,0xdc20,0xdc1d,0xdc19,0xdc15,0xdc12,0xdc0e,0xdc0a,0xdc08,0xdc04,0xdc01,0xdbfd,
	0xdbf9,0xdbf6,0xdbf2,0xdbee,0xdbec,0xdbe9,0xdbe5,0xdbe1,0xdbdd,0xdbda,0xdbd6,0xdbd4,0xdbd0,0xdbcd,0xdbc9,0xdbc5,
	0xdbc3,0xdbbf,0xdbbb,0xdbb8,0xdbb4,0xdbb0,0xdbae,0xdbab,0xdba7,0xdba3,0xdba0,0xdb9c,0xdb98,0xdb96,0xdb93,0xdb8f,
	0xdb8b,0xdb87,0xdb84,0xdb80,0xdb7e,0xdb7a,0xdb77,0xdb73,0xdb6f,0xdb6c,0xdb6a,0xdb66,0xdb62,0xdb5f,0xdb5b,0xdb57,
	0xdb53,0xdb52,0xdb4e,0xdb4a,0xdb47,0xdb43,0xdb3f,0xdb3b,0xdb3a,0xdb36,0xdb32,0xdb2e,0xdb2b,0xdb27,0xdb23,0xdb21,
	0xdb1e,0xdb1a,0xdb16,0xdb13,0xdb0f,0xdb0b,0xdb09,0xdb06,0xdb02,0xdafe,0xdafa,0xdaf7,0xdaf3,0xdaf1,0xdaed,0xdaea,
	0xdae6,0xdae2,0xdadf,0xdadb,0xdad9,0xdad5,0xdad2,0xdace,0xdaca,0xdac6,0xdac3,0xdac1,0xdabd,0xdab9,0xdab6,0xdab2,
	0xdaae,0xdaab,0xdaa9,0xdaa5,0xdaa1,0xda9e,0xda9a,0xda96,0xda93,0xda91,0xda8d,0xda89,0xda86,0xda82,0xda7e,0xda7a,
	0xda79,0xda75,0xda71,0xda6d,0xda6a,0xda66,0xda62,0xda60,0xda5d,0xda59,0xda55,0xda52,0xda4e,0xda4c,0xda48,0xda45,
	0xda41,0xda3d,0xda39,0xda36,0xda34,0xda30,0xda2c,0xda29,0xda25,0xda21,0xda1e,0xda1c,0xda18,0xda14,0xda11,0xda0d,
	0xda0a,0xda06,0xda03,0xd9ff,0xd9fb,0xd9f9,0xd9f6,0xd9f2,0xd9ee,0xd9eb,0xd9e7,0xd9e3,0xd9e1,0xd9de,0xd9da,0xd9d6,
	0xd9d3,0xd9cf,0xd9cb,0xd9c9,0xd9c6,0xd9c2,0xd9be,0xd9bb,0xd9b7,0xd9b5,0xd9b1,0xd9ae,0xd9aa,0xd9a6,0xd9a3,0xd99f,
	0xd99d,0xd999,0xd996,0xd992,0xd98e,0xd98b,0xd989,0xd985,0xd981,0xd97e,0xd97a,0xd976,0xd973,0xd971,0xd96d,0xd969,
	0xd966,0xd962,0xd95e,0xd95c,0xd959,0xd955,0xd951,0xd94e,0xd94a,0xd946,0xd944,0xd941,0xd93d,0xd939,0xd936,0xd932,
	0xd930,0xd92c,0xd929,0xd925,0xd921,0xd91e,0xd91a,0xd918,0xd914,0xd911,0xd90d,0xd909,0xd906,0xd902,0xd900,0xd8fc,
	0xd8f9,0xd8f5,0xd8f1,0xd8ee,0xd8ec,0xd8e8,0xd8e4,0xd8e1,0xd8dd,0xd8d9,0xd8d6,0xd8d4,0xd8d0,0xd8cc,0xd8c9,0xd8c5,
	0xd8c1,0xd8bf,0xd8bc,0xd8b8,0xd8b4,0xd8b1,0xd8ad,0xd8a9,0xd8a7,0xd8a4,0xd8a0,0xd89c,0xd899,0xd895,0xd893,0xd88f,
	0xd88c,0xd888,0xd884,0xd881,0xd87d,0xd87b,0xd877,0xd874,0xd870,0xd86c,0xd868,0xd865,0xd863,0xd85f,0xd85c,0xd858,
	0xd854,0xd851,0xd84d,0xd849,0xd846,0xd844,0xd840,0xd83c,0xd839,0xd835,0xd831,0xd830,0xd82c,0xd828,0xd825,0xd821,
	0xd81d,0xd819,0xd818,0xd814,0xd810,0xd80d,0xd809,0xd805,0xd803,0xd800,0xd7fc,0xd7f8,0xd7f5,0xd7f1,0xd7ef,0xd7ec,
	0xd7e8,0xd7e4,0xd7e0,0xd7dd,0xd7db,0xd7d7,0xd7d4,0xd7d0,0xd7cc,0xd7c9,0xd7c5,0xd7c3,0xd7bf,0xd7bc,0xd7b8,0xd7b4,
	0xd7b1,0xd7af,0xd7ab,0xd7a7,0xd7a4,0xd7a0,0xd79c,0xd79b,0xd797,0xd793,0xd790,0xd78c,0xd788,0xd786,0xd783,0xd77f,
	0xd77b,0xd778,0xd774,0xd770,0xd76e,0xd76b,0xd767,0xd763,0xd760,0xd75c,0xd75a,0xd756,0xd753,0xd74f,0xd74b,0xd748,
	0xd746,0xd742,0xd73f,0xd73b,0xd737,0xd734,0xd732,0xd72e,0xd72a,0xd727,0xd723,0xd71f,0xd71c,0xd71a,0xd716,0xd712,
	0xd70f,0xd70b,0xd707,0xd706,0xd702,0xd6fe,0xd6fa,0xd6f7,0xd6f3,0xd6f1,0xd6ee,0xd6ea,0xd6e6,0xd6e3,0xd6df,0xd6dd,
	0xd6d9,0xd6d6,0xd6d2,0xd6ce,0xd6cb,0xd6c7,0xd6c5,0xd6c1,0xd6be,0xd6ba,0xd6b6,0xd6b3,0xd6b1,0xd6ad,0xd6aa,0xd6a6,
	0xd6a3,0xd6a0,0xd69c,0xd698,0xd695,0xd691,0xd68d,0xd68c,0xd688,0xd684,0xd681,0xd67d,0xd679,0xd677,0xd674,0xd670,
	0xd66c,0xd669,0xd665,0xd663,0xd65f,0xd65c,0xd658,0xd654,0xd651,0xd64f,0xd64b,0xd648,0xd644,0xd640,0xd63d,0xd63b,
	0xd637,0xd633,0xd630,0xd62c,0xd628,0xd627,0xd623,0xd61f,0xd61c,0xd618,0xd614,0xd612,0xd60f,0xd60b,0xd607,0xd604,
	0xd600,0xd5fe,0xd5fa,0xd5f7,0xd5f3,0xd5ef,0xd5ec,0xd5ea,0xd5e6,0xd5e3,0xd5df,0xd5db,0xd5d8,0xd5d6,0xd5d2,0xd5ce,
	0xd5cb,0xd5c7,0xd5c3,0xd5c0,0xd5be,0xd5ba,0xd5b7,0xd5b3,0xd5af,0xd5ac,0xd5aa,0xd5a6,0xd5a2,0xd59f,0xd59b,0xd597,
	0xd596,0xd592,0xd58e,0xd58b,0xd587,0xd583,0xd581,0xd57e,0xd57a,0xd576,0xd573,0xd56f,0xd56d,0xd569,0xd566,0xd562,
	0xd55e,0xd55b,0xd559,0xd555,0xd552,0xd54e,0xd54a,0xd547,0xd545,0xd541,0xd53d,0xd53a,0xd536,0xd532,0xd531,0xd52d,
	0xd529,0xd526,0xd522,0xd51e,0xd51c,0xd519,0xd515,0xd511,0xd50e,0xd50a,0xd508,0xd504,0xd501,0xd4fd,0xd4f9,0xd4f6,
	0xd4f3,0xd4ef,0xd4ec,0xd4ea,0xd4e6,0xd4e3,0xd4df,0xd4db,0xd4d8,0xd4d6,0xd4d2,0xd4cf,0xd4cb,0xd4c7,0xd4c4,0xd4c2,
	0xd4be,0xd4ba,0xd4b7,0xd4b3,0xd4b0,0xd4ae,0xd4aa,0xd4a6,0xd4a3,0xd49f,0xd49b,0xd49a,0xd496,0xd492,0xd48f,0xd48b,
	0xd489,0xd485,0xd482,0xd47e,0xd47a,0xd477,0xd475,0xd471,0xd46e,0xd46a,0xd466,0xd463,0xd461,0xd45d,0xd45a,0xd456,
	0xd452,0xd44f,0xd44d,0xd449,0xd445,0xd442,0xd43e,0xd43b,0xd439,0xd435,0xd431,0xd42e,0xd42a,0xd426,0xd425,0xd421,
	0xd41d,0xd41a,0xd416,0xd414,0xd410,0xd40d,0xd409,0xd405,0xd402,0xd400,0xd3fc,0xd3f9,0xd3f5,0xd3f1,0xd3ee,0xd3ec,
	0xd3e8,0xd3e5,0xd3e1,0xd3dd,0xd3da,0xd3d8,0xd3d4,0xd3d0,0xd3cd,0xd3c9,0xd3c6,0xd3c4,0xd3c0,0xd3bc,0xd3b9,0xd3b5,
	0xd3b3,0xd3b0,0xd3ac,0xd3a8,0xd3a5,0xd3a1,0xd39f,0xd39b,0xd398,0xd394,0xd390,0xd38d,0xd38b,0xd387,0xd384,0xd380,
	0xd37c,0xd379,0xd377,0xd373,0xd370,0xd36c,0xd368,0xd365,0xd363,0xd35f,0xd35b,0xd358,0xd354,0xd351,0xd34f,0xd34b,
	0xd348,0xd344,0xd340,0xd33d,0xd339,0xd335,0xd334,0xd330,0xd32c,0xd329,0xd325,0xd321,0xd320,0xd31c,0xd318,0xd315,
	0xd311,0xd30f,0xd30c,0xd308,0xd304,0xd301,0xd2fd,0xd2fb,0xd2f8,0xd2f4,0xd2f0,0xd2ed,0xd2eb,0xd2e7,0xd2e4,0xd2e0,
	0xd2dc,0xd2d9,0xd2d7,0xd2d3,0xd2d0,0xd2cc,0xd2c8,0xd2c6,0xd2c3,0xd2bf,0xd2bb,0xd2b8,0xd2b4,0xd2b2,0xd2af,0xd2ab,
	0xd2a7,0xd2a4,0xd2a0,0xd29e,0xd29b,0xd297,0xd293,0xd290,0xd28e,0xd28a,0xd287,0xd283,0xd27f,0xd27c,0xd27a,0xd276,
	0xd273,0xd26f,0xd26b,0xd26a,0xd266,0xd262,0xd25f,0xd25b,0xd257,0xd256,0xd252,0xd24e,0xd24b,0xd247,0xd245,0xd242,
	0xd23e,0xd23a,0xd237,0xd233,0xd231,0xd22e,0xd22a,0xd226,0xd223,0xd21f,0xd21d,0xd219,0xd216,0xd212,0xd20f,0xd20d,
	0xd209,0xd205,0xd202,0xd1fe,0xd1fb,0xd1f9,0xd1f5,0xd1f1,0xd1ee,0xd1ea,0xd1e8,0xd1e5,0xd1e1,0xd1dd,0xd1da,0xd1d6,
	0xd1d4,0xd1d1,0xd1cd,0xd1c9,0xd1c6,0xd1c4,0xd1c0,0xd1bd,0xd1b9,0xd1b5,0xd1b2,0xd1b0,0xd1ac,0xd1a9,0xd1a5,0xd1a1,
	0xd19e,0xd19a,0xd199,0xd195,0xd191,0xd18e,0xd18a,0xd188,0xd185,0xd181,0xd17d,0xd17a,0xd176,0xd174,0xd171,0xd16d,
	0xd169,0xd166,0xd164,0xd160,0xd15d,0xd159,0xd155,0xd152,0xd150,0xd14c,0xd149,0xd145,0xd141,0xd140,0xd13c,0xd138,
	0xd135,0xd131,0xd12d,0xd12c,0xd128,0xd124,0xd121,0xd11d,0xd11b,0xd118,0xd114,0xd110,0xd10d,0xd10b,0xd107,0xd104,
	0xd100,0xd0fc,0xd0f9,0xd0f7,0xd0f3,0xd0f0,0xd0ec,0xd0e8,0xd0e7,0xd0e3,0xd0df,0xd0dc,0xd0d8,0xd0d4,0xd0d3,0xd0cf,
	0xd0cb,0xd0c8,0xd0c4,0xd0c2,0xd0bf,0xd0bb,0xd0b7,0xd0b4,0xd0b0,0xd0ae,0xd0ab,0xd0a7,0xd0a3,0xd0a0,0xd09e,0xd09a,
	0xd097,0xd093,0xd08f,0xd08c,0xd08a,0xd086,0xd083,0xd07f,0xd07b,0xd07a,0xd076,0xd072,0xd06f,0xd06b,0xd069,0xd066,
	0xd062,0xd05e,0xd05b,0xd057,0xd055,0xd052,0xd04e,0xd04a,0xd047,0xd045,0xd041,0xd03e,0xd03a,0xd036,0xd033,0xd031,
	0xd02d,0xd02a,0xd026,0xd022,0xd021,0xd01d,0xd019,0xd016,0xd012,0xd00e,0xd00d,0xd009,0xd005,0xd002,0xcffe,0xcffc,
	0xcff8,0xcff4,0xcff2,0xcfef,0xcfeb,0xcfe8,0xcfe4,0xcfe2,0xcfde,0xcfdb,0xcfd7,0xcfd4,0xcfd2,0xcfce,0xcfcb,0xcfc7,
	0xcfc3,0xcfc2,0xcfbe,0xcfba,0xcfb7,0xcfb3,0xcfaf,0xcfae,0xcfaa,0xcfa6,0xcfa3,0xcf9f,0xcf9d,0xcf9a,0xcf96,0xcf92,
	0xcf8f,0xcf8d,0xcf89,0xcf86,0xcf82,0xcf7f,0xcf7d,0xcf79,0xcf76,0xcf72,0xcf6e,0xcf6b,0xcf69,0xcf65,0xcf62,0xcf5e,
	0xcf5a,0xcf59,0xcf55,0xcf51,0xcf4e,0xcf4a,0xcf48,0xcf45,0xcf41,0xcf3d,0xcf3a,0xcf38,0xcf34,0xcf31,0xcf2d,0xcf2a,
	0xcf28,0xcf24,0xcf21,0xcf1d,0xcf19,0xcf16,0xcf14,0xcf10,0xcf0d,0xcf09,0xcf05,0xcf04,0xcf00,0xcefc,0xcef9,0xcef5,
	0xcef3,0xcef0,0xceec,0xcee8,0xcee5,0xcee3,0xcedf,0xcedc,0xced8,0xced5,0xced1,0xcecf,0xcecc,0xcec8,0xcec4,0xcec1,
	0xcebf,0xcebb,0xceb8,0xceb4,0xceb0,0xceaf,0xceab,0xcea7,0xcea4,0xcea0,0xce9e,0xce9b,0xce97,0xce93,0xce90,0xce8c,
	0xce8a,0xce87,0xce83,0xce80,0xce7c,0xce7a,0xce77,0xce73,0xce6f,0xce6c,0xce6a,0xce66,0xce63,0xce5f,0xce5b,0xce5a,
	0xce57,0xce53,0xce4f,0xce4c,0xce48,0xce45,0xce43,0xce3f,0xce3c,0xce38,0xce34,0xce33,0xce2f,0xce2b,0xce28,0xce24,
	0xce22,0xce1f,0xce1b,0xce17,0xce14,0xce12,0xce0e,0xce0b,0xce07,0xce04,0xce02,0xcdfe,0xcdfb,0xcdf7,0xcdf3,0xcdf2,
	0xcdee,0xcdea,0xcde7,0xcde3,0xcde1,0xcdde,0xcdda,0xcdd6,0xcdd3,0xcdd1,0xcdcd,0xcdca,0xcdc6,0xcdc3,0xcdbf,0xcdbd,
	0xcdba,0xcdb6,0xcdb2,0xcdaf,0xcdad,0xcda9,0xcda6,0xcda2,0xcd9f,0xcd9d,0xcd99,0xcd96,0xcd92,0xcd8e,0xcd8d,0xcd89,
	0xcd85,0xcd82,0xcd7e,0xcd7c,0xcd79,0xcd75,0xcd71,0xcd6e,0xcd6c,0xcd68,0xcd65,0xcd61,0xcd5e,0xcd5c,0xcd58,0xcd55,
	0xcd51,0xcd4d,0xcd4c,0xcd48,0xcd44,0xcd41,0xcd3d,0xcd39,0xcd38,0xcd34,0xcd30,0xcd2d,0xcd29,0xcd27,0xcd24,0xcd20,
	0xcd1d,0xcd19,0xcd17,0xcd14,0xcd10,0xcd0c,0xcd09,0xcd07,0xcd03,0xcd00,0xccfc,0xccf9,0xccf7,0xccf3,0xccef,0xccec,
	0xcce8,0xcce6,0xcce3,0xccdf,0xccdc,0xccd8,0xccd6,0xccd3,0xcccf,0xcccb,0xccc8,0xccc6,0xccc2,0xccbf,0xccbb,0xccb8,
	0xccb5,0xccb2,0xccae,0xccad,0xcca9,0xcca5,0xcca2,0xcc9e,0xcc9c,0xcc99,0xcc95,0xcc92,0xcc8e,0xcc8c,0xcc89,0xcc85,
	0xcc81,0xcc7e,0xcc7c,0xcc78,0xcc75,0xcc71,0xcc6e,0xcc6c,0xcc68,0xcc65,0xcc61,0xcc5d,0xcc5c,0xcc58,0xcc54,0xcc51,
	0xcc4f,0xcc4b,0xcc48,0xcc44,0xcc41,0xcc3f,0xcc3b,0xcc38,0xcc34,0xcc31,0xcc2f,0xcc2b,0xcc28,0xcc24,0xcc20,0xcc1f,
	0xcc1b,0xcc17,0xcc14,0xcc10,0xcc0e,0xcc0b,0xcc07,0xcc04,0xcc00,0xcbfe,0xcbfb,0xcbf7,0xcbf3,0xcbf0,0xcbee,0xcbea,
	0xcbe7,0xcbe3,0xcbe0,0xcbde,0xcbda,0xcbd7,0xcbd3,0xcbcf,0xcbce,0xcbca,0xcbc7,0xcbc3,0xcbbf,0xcbbe,0xcbba,0xcbb6,
	0xcbb3,0xcbaf,0xcbad,0xcbaa,0xcba6,0xcba3,0xcb9f,0xcb9d,0xcb9a,0xcb96,0xcb92,0xcb8f,0xcb8d,0xcb89,0xcb86,0xcb82,
	0xcb80,0xcb7d,0xcb79,0xcb76,0xcb72,0xcb70,0xcb6d,0xcb69,0xcb65,0xcb62,0xcb60,0xcb5c,0xcb59,0xcb55,0xcb52,0xcb50,
	0xcb4c,0xcb49,0xcb45,0xcb42,0xcb40,0xcb3c,0xcb39,0xcb35,0xcb31,0xcb30,0xcb2c,0xcb28,0xcb25,0xcb21,0xcb1f,0xcb1c,
	0xcb19,0xcb15,0xcb12,0xcb0e,0xcb0b,0xcb09,0xcb05,0xcb02,0xcafe,0xcafa,0xcaf9,0xcaf5,0xcaf2,0xcaee,0xcaea,0xcae9,
	0xcae5,0xcae1,0xcade,0xcadc,0xcad8,0xcad5,0xcad1,0xcace,0xcacc,0xcac8,0xcac5,0xcac1,0xcabe,0xcabc,0xcab8,0xcab5,
	0xcab1,0xcaae,0xcaac,0xcaa8,0xcaa5,0xcaa1,0xca9f,0xca9c,0xca98,0xca94,0xca91,0xca8f,0xca8c,0xca88,0xca84,0xca81,
	0xca7f,0xca7b,0xca78,0xca74,0xca71,0xca6f,0xca6b,0xca68,0xca64,0xca62,0xca5f,0xca5b,0xca58,0xca54,0xca52,0xca4f,
	0xca4b,0xca48,0xca44,0xca42,0xca3f,0xca3b,0xca37,0xca34,0xca32,0xca2f,0xca2b,0xca27,0xca26,0xca22,0xca1e,0xca1b,
	0xca17,0xca15,0xca12,0xca0e,0xca0b,0xca07,0xca05,0xca02,0xc9fe,0xc9fb,0xc9f7,0xc9f5,0xc9f2,0xc9ee,0xc9eb,0xc9e9,
	0xc9e5,0xc9e2,0xc9de,0xc9da,0xc9d9,0xc9d5,0xc9d1,0xc9ce,0xc9ca,0xc9c9,0xc9c5,0xc9c1,0xc9be,0xc9ba,0xc9b8,0xc9b5,
	0xc9b1,0xc9ae,0xc9ac,0xc9a8,0xc9a5,0xc9a1,0xc99e,0xc99c,0xc998,0xc995,0xc991,0xc98e,0xc98c,0xc988,0xc985,0xc981,
	0xc97f,0xc97b,0xc978,0xc974,0xc972,0xc96f,0xc96b,0xc968,0xc964,0xc962,0xc95f,0xc95b,0xc957,0xc956,0xc952,0xc94f,
	0xc94b,0xc947,0xc946,0xc942,0xc93e,0xc93b,0xc937,0xc936,0xc932,0xc92e,0xc92b,0xc929,0xc925,0xc922,0xc91e,0xc91b,
	0xc919,0xc915,0xc912,0xc90e,0xc90b,0xc909,0xc905,0xc902,0xc8fe,0xc8fc,0xc8f9,0xc8f5,0xc8f2,0xc8ee,0xc8ec,0xc8e9,
	0xc8e5,0xc8e2,0xc8de,0xc8dc,0xc8d9,0xc8d5,0xc8d2,0xc8d0,0xc8cc,0xc8c9,0xc8c5,0xc8c2,0xc8c0,0xc8bc,0xc8b9,0xc8b5,
	0xc8b3,0xc8b0,0xc8ac,0xc8a9,0xc8a5,0xc8a3,0xc8a0,0xc89c,0xc898,0xc895,0xc893,0xc890,0xc88c,0xc888,0xc887,0xc883,
	0xc87f,0xc87c,0xc878,0xc877,0xc873,0xc86f,0xc86c,0xc868,0xc866,0xc863,0xc85f,0xc85c,0xc85a,0xc856,0xc853,0xc84f,
	0xc84c,0xc84a,0xc846,0xc843,0xc83f,0xc83c,0xc83a,0xc836,0xc833,0xc82f,0xc82d,0xc82a,0xc826,0xc823,0xc81f,0xc81d,
	0xc81a,0xc816,0xc813,0xc80f,0xc80d,0xc80a,0xc806,0xc803,0xc801,0xc7fd,0xc7fa,0xc7f6,0xc7f2,0xc7f1,0xc7ed,0xc7ea,
	0xc7e7,0xc7e4,0xc7e0,0xc7dc,0xc7db,0xc7d7,0xc7d4,0xc7d0,0xc7ce,0xc7cb,0xc7c7,0xc7c4,0xc7c0,0xc7be,0xc7bb,0xc7b7,
	0xc7b4,0xc7b2,0xc7ae,0xc7ab,0xc7a7,0xc7a4,0xc7a2,0xc79e,0xc79b,0xc797,0xc795,0xc792,0xc78e,0xc78b,0xc787,0xc785,
	0xc782,0xc77e,0xc77b,0xc779,0xc775,0xc772,0xc76e,0xc76b,0xc769,0xc765,0xc762,0xc75e,0xc75c,0xc759,0xc755,0xc752,
	0xc74e,0xc74c,0xc749,0xc745,0xc742,0xc740,0xc73c,0xc739,0xc735,0xc732,0xc730,0xc72c,0xc729,0xc725,0xc724,0xc720,
	0xc71c,0xc719,0xc715,0xc714,0xc710,0xc70c,0xc709,0xc707,0xc704,0xc700,0xc6fc,0xc6f9,0xc6f7,0xc6f4,0xc6f0,0xc6ec,
	0xc6eb,0xc6e7,0xc6e4,0xc6e0,0xc6dc,0xc6db,0xc6d7,0xc6d4,0xc6d0,0xc6ce,0xc6cb,0xc6c7,0xc6c4,0xc6c0,0xc6be,0xc6bb,
	0xc6b7,0xc6b4,0xc6b2,0xc6ae,0xc6ab,0xc6a7,0xc6a4,0xc6a2,0xc69e,0xc69b,0xc697,0xc695,0xc692,0xc68e,0xc68b,0xc687,
	0xc685,0xc682,0xc67e,0xc67b,0xc679,0xc675,0xc672,0xc66e,0xc66b,0xc669,0xc665,0xc662,0xc65e,0xc65c,0xc659,0xc655,
	0xc653,0xc650,0xc64c,0xc649,0xc645,0xc643,0xc640,0xc63c,0xc639,0xc637,0xc633,0xc630,0xc62c,0xc629,0xc627,0xc623,
	0xc620,0xc61c,0xc61b,0xc617,0xc614,0xc610,0xc60c,0xc60b,0xc607,0xc604,0xc600,0xc5fe,0xc5fb,0xc5f7,0xc5f4,0xc5f2,
	0xc5ee,0xc5eb,0xc5e7,0xc5e4,0xc5e2,0xc5de,0xc5db,0xc5d7,0xc5d5,0xc5d2,0xc5ce,0xc5cb,0xc5c7,0xc5c5,0xc5c2,0xc5be,
	0xc5bb,0xc5b9,0xc5b6,0xc5b2,0xc5ae,0xc5ad,0xc5a9,0xc5a6,0xc5a2,0xc59e,0xc59d,0xc599,0xc596,0xc592,0xc590,0xc58d,
	0xc589,0xc586,0xc582,0xc580,0xc57d,0xc579,0xc576,0xc574,0xc570,0xc56d,0xc569,0xc566,0xc564,0xc560,0xc55d,0xc559,
	0xc558,0xc554,0xc550,0xc54d,0xc54b,0xc548,0xc544,0xc540,0xc53d,0xc53b,0xc538,0xc534,0xc531,0xc52f,0xc52b,0xc528,
	0xc524,0xc521,0xc51f,0xc51b,0xc518,0xc514,0xc512,0xc50f,0xc50b,0xc508,0xc506,0xc502,0xc4ff,0xc4fb,0xc4f8,0xc4f6,
	0xc4f2,0xc4ef,0xc4eb,0xc4ea,0xc4e6,0xc4e2,0xc4df,0xc4db,0xc4da,0xc4d6,0xc4d3,0xc4cf,0xc4cd,0xc4ca,0xc4c6,0xc4c3,
	0xc4c0,0xc4bd,0xc4b9,0xc4b8,0xc4b4,0xc4b1,0xc4ad,0xc4ab,0xc4a8,0xc4a4,0xc4a1,0xc49f,0xc49b,0xc498,0xc494,0xc493,
	0xc48f,0xc48b,0xc488,0xc484,0xc483,0xc47f,0xc47c,0xc478,0xc476,0xc473,0xc46f,0xc46c,0xc46a,0xc466,0xc463,0xc45f,
	0xc45e,0xc45a,0xc457,0xc453,0xc44f,0xc44e,0xc44a,0xc447,0xc443,0xc441,0xc43e,0xc43a,0xc437,0xc435,0xc431,0xc42e,
	0xc42a,0xc427,0xc425,0xc422,0xc41e,0xc41a,0xc419,0xc415,0xc412,0xc40e,0xc40c,0xc409,0xc405,0xc402,0xc400,0xc3fc,
	0xc3f9,0xc3f5,0xc3f2,0xc3f0,0xc3ed,0xc3e9,0xc3e6,0xc3e4,0xc3e0,0xc3dd,0xc3d9,0xc3d7,0xc3d4,0xc3d0,0xc3cd,0xc3cb,
	0xc3c8,0xc3c4,0xc3c0,0xc3bd,0xc3bb,0xc3b8,0xc3b4,0xc3b1,0xc3af,0xc3ab,0xc3a8,0xc3a4,0xc3a2,0xc39f,0xc39b,0xc398,
	0xc396,0xc393,0xc38f,0xc38b,0xc388,0xc386,0xc383,0xc37f,0xc37c,0xc37a,0xc376,0xc373,0xc36f,0xc36d,0xc36a,0xc366,
	0xc363,0xc35f,0xc35e,0xc35a,0xc357,0xc353,0xc351,0xc34e,0xc34a,0xc347,0xc345,0xc341,0xc33e,0xc33a,0xc338,0xc335,
	0xc332,0xc32e,0xc32b,0xc327,0xc324,0xc322,0xc31f,0xc31b,0xc317,0xc316,0xc312,0xc30f,0xc30b,0xc309,0xc306,0xc302,
	0xc2ff,0xc2fd,0xc2fa,0xc2f6,0xc2f3,0xc2f1,0xc2ed,0xc2ea,0xc2e6,0xc2e5,0xc2e1,0xc2dd,0xc2da,0xc2d8,0xc2d5,0xc2d1,
	0xc2ce,0xc2ca,0xc2c8,0xc2c5,0xc2c1,0xc2be,0xc2bc,0xc2b9,0xc2b5,0xc2b2,0xc2b0,0xc2ac,0xc2a9,0xc2a5,0xc2a3,0xc2a0,
	0xc29c,0xc299,0xc297,0xc294,0xc290,0xc28d,0xc28b,0xc287,0xc284,0xc280,0xc27f,0xc27b,0xc278,0xc274,0xc272,0xc26f,
	0xc26b,0xc268,0xc264,0xc262,0xc25f,0xc25b,0xc258,0xc256,0xc253,0xc24f,0xc24c,0xc24a,0xc246,0xc243,0xc23f,0xc23e,
	0xc23a,0xc236,0xc233,0xc231,0xc22e,0xc22a,0xc227,0xc225,0xc221,0xc21e,0xc21a,0xc219,0xc215,0xc212,0xc20e,0xc20c,
	0xc209,0xc205,0xc202,0xc1fe,0xc1fc,0xc1f9,0xc1f5,0xc1f2,0xc1f0,0xc1ed,0xc1e9,0xc1e6,0xc1e4,0xc1e0,0xc1dd,0xc1d9,
	0xc1d8,0xc1d4,0xc1d1,0xc1cd,0xc1cb,0xc1c8,0xc1c4,0xc1c1,0xc1bf,0xc1bb,0xc1b8,0xc1b4,0xc1b3,0xc1af,0xc1ac,0xc1a8,
	0xc1a5,0xc1a1,0xc1a0,0xc19c,0xc199,0xc195,0xc193,0xc190,0xc18c,0xc189,0xc187,0xc184,0xc180,0xc17d,0xc17b,0xc177,
	0xc174,0xc170,0xc16e,0xc16b,0xc167,0xc164,0xc162,0xc15f,0xc15b,0xc158,0xc156,0xc152,0xc14f,0xc14b,0xc14a,0xc146,
	0xc143,0xc13f,0xc13d,0xc13a,0xc136,0xc133,0xc131,0xc12e,0xc12a,0xc127,0xc125,0xc121,0xc11e,0xc11a,0xc119,0xc115,
	0xc112,0xc10e,0xc10c,0xc109,0xc105,0xc102,0xc100,0xc0fc,0xc0f9,0xc0f5,0xc0f4,0xc0f0,0xc0ed,0xc0e9,0xc0e7,0xc0e4,
	0xc0e0,0xc0dd,0xc0d9,0xc0d8,0xc0d4,0xc0d1,0xc0cd,0xc0cb,0xc0c8,0xc0c4,0xc0c1,0xc0bf,0xc0bc,0xc0b8,0xc0b5,0xc0b3,
	0xc0af,0xc0ac,0xc0a8,0xc0a7,0xc0a3,0xc0a0,0xc09c,0xc09a,0xc097,0xc093,0xc090,0xc08e,0xc08a,0xc087,0xc083,0xc082,
	0xc07e,0xc07b,0xc077,0xc075,0xc072,0xc06e,0xc06b,0xc069,0xc066,0xc062,0xc05f,0xc05d,0xc059,0xc056,0xc052,0xc051,
	0xc04d,0xc04a,0xc046,0xc044,0xc041,0xc03d,0xc03a,0xc038,0xc035,0xc031,0xc02e,0xc02c,0xc028,0xc025,0xc021,0xc020,
	0xc01b,0xc019,0xc016,0xc012,0xc00f,0xc00d,0xc009,0xc006,0xc002,0xc001,0xbffd,0xbffa,0xbff6,0xbff4,0xbff1,0xbfed,
	0xbfea,0xbfe8,0xbfe5,0xbfe1,0xbfde,0xbfdc,0xbfd9,0xbfd5,0xbfd2,0xbfd0,0xbfcc,0xbfc9,0xbfc5,0xbfc4,0xbfc0,0xbfbd,
	0xbfb9,0xbfb7,0xbfb4,0xbfb0,0xbfad,0xbfab,0xbfa8,0xbfa4,0xbfa1,0xbf9f,0xbf9b,0xbf98,0xbf94,0xbf93,0xbf8f,0xbf8c,
	0xbf88,0xbf86,0xbf83,0xbf7f,0xbf7c,0xbf7a,0xbf77,0xbf73,0xbf70,0xbf6e,0xbf6b,0xbf67,0xbf64,0xbf62,0xbf5e,0xbf5b,
	0xbf59,0xbf56,0xbf52,0xbf4f,0xbf4d,0xbf49,0xbf46,0xbf42,0xbf41,0xbf3d,0xbf3a,0xbf36,0xbf34,0xbf31,0xbf2d,0xbf2a,
	0xbf28,0xbf25,0xbf21,0xbf1e,0xbf1c,0xbf18,0xbf15,0xbf11,0xbf10,0xbf0c,0xbf09,0xbf05,0xbf03,0xbf00,0xbefd,0xbef9,
	0xbef7,0xbef4,0xbef0,0xbeed,0xbeeb,0xbee8,0xbee4,0xbee1,0xbedf,0xbedb,0xbed8,0xbed4,0xbed3,0xbecf,0xbecc,0xbec8,
	0xbec6,0xbec3,0xbebf,0xbebc,0xbeba,0xbeb7,0xbeb3,0xbeb0,0xbeae,0xbeaa,0xbea7,0xbea3,0xbea2,0xbe9e,0xbe9b,0xbe97,
	0xbe96,0xbe92,0xbe8f,0xbe8b,0xbe8a,0xbe86,0xbe83,0xbe7f,0xbe7d,0xbe7a,0xbe76,0xbe73,0xbe71,0xbe6e,0xbe6a,0xbe67,
	0xbe65,0xbe62,0xbe5e,0xbe5b,0xbe59,0xbe55,0xbe52,0xbe4e,0xbe4d,0xbe49,0xbe46,0xbe42,0xbe40,0xbe3d,0xbe39,0xbe36,
	0xbe34,0xbe31,0xbe2d,0xbe2c,0xbe28,0xbe25,0xbe21,0xbe1f,0xbe1c,0xbe18,0xbe15,0xbe13,0xbe10,0xbe0c,0xbe09,0xbe07,
	0xbe03,0xbe00,0xbdfd,0xbdfb,0xbdf7,0xbdf4,0xbdf0,0xbdef,0xbdeb,0xbde8,0xbde4,0xbde2,0xbddf,0xbddb,0xbdd8,0xbdd6,
	0xbdd3,0xbdcf,0xbdcd,0xbdca,0xbdc7,0xbdc3,0xbdc1,0xbdbe,0xbdba,0xbdb7,0xbdb5,0xbdb2,0xbdae,0xbdab,0xbda9,0xbda5,
	0xbda2,0xbd9e,0xbd9d,0xbd99,0xbd96,0xbd92,0xbd91,0xbd8d,0xbd8a,0xbd86,0xbd84,0xbd81,0xbd7d,0xbd7a,0xbd78,0xbd75,
	0xbd71,0xbd6f,0xbd6c,0xbd68,0xbd65,0xbd63,0xbd60,0xbd5c,0xbd59,0xbd57,0xbd54,0xbd50,0xbd4d,0xbd4b,0xbd47,0xbd44,
	0xbd40,0xbd3f,0xbd3b,0xbd38,0xbd34,0xbd32,0xbd2f,0xbd2b,0xbd28,0xbd26,0xbd23,0xbd1f,0xbd1c,0xbd1a,0xbd17,0xbd13,
	0xbd10,0xbd0d,0xbd0b,0xbd08,0xbd04,0xbd01,0xbcff,0xbcfb,0xbcf8,0xbcf5,0xbcf3,0xbcef,0xbcec,0xbce8,0xbce7,0xbce3,
	0xbce0,0xbcde,0xbcdb,0xbcd7,0xbcd4,0xbcd2,0xbcce,0xbccb,0xbcc7,0xbcc6,0xbcc2,0xbcbf,0xbcbb,0xbcba,0xbcb6,0xbcb3,
	0xbcb1,0xbcad,0xbcaa,0xbca7,0xbca5,0xbca1,0xbc9e,0xbc9a,0xbc99,0xbc95,0xbc92,0xbc8e,0xbc8c,0xbc89,0xbc86,0xbc82,
	0xbc80,0xbc7d,0xbc79,0xbc78,0xbc74,0xbc71,0xbc6d,0xbc6c,0xbc68,0xbc65,0xbc61,0xbc5f,0xbc5c,0xbc58,0xbc55,0xbc53,
	0xbc50,0xbc4c,0xbc4b,0xbc47,0xbc44,0xbc40,0xbc3e,0xbc3b,0xbc38,0xbc34,0xbc32,0xbc2f,0xbc2b,0xbc28,0xbc26,0xbc23,
	0xbc1f,0xbc1d,0xbc1a,0xbc17,0xbc13,0xbc11,0xbc0e,0xbc0a,0xbc07,0xbc05,0xbc02,0xbbfe,0xbbfb,0xbbf9,0xbbf6,0xbbf2,
	0xbbf0,0xbbed,0xbbe9,0xbbe6,0xbbe4,0xbbe1,0xbbdd,0xbbda,0xbbd8,0xbbd5,0xbbd1,0xbbce,0xbbcc,0xbbc9,0xbbc5,0xbbc2,
	0xbbc0,0xbbbc,0xbbb9,0xbbb7,0xbbb4,0xbbb0,0xbbad,0xbbab,0xbba8,0xbba4,0xbba1,0xbb9f,0xbb9b,0xbb98,0xbb94,0xbb93,
	0xbb90,0xbb8c,0xbb89,0xbb87,0xbb84,0xbb80,0xbb7d,0xbb7b,0xbb77,0xbb74,0xbb70,0xbb6f,0xbb6b,0xbb68,0xbb66,0xbb63,
	0xbb5f,0xbb5c,0xbb5a,0xbb57,0xbb53,0xbb50,0xbb4e,0xbb4a,0xbb47,0xbb43,0xbb42,0xbb3e,0xbb3b,0xbb39,0xbb36,0xbb32,
	0xbb2f,0xbb2d,0xbb2a,0xbb26,0xbb23,0xbb21,0xbb1d,0xbb1a,0xbb18,0xbb15,0xbb11,0xbb0e,0xbb0c,0xbb09,0xbb05,0xbb02,
	0xbb00,0xbafd,0xbaf9,0xbaf6,0xbaf4,0xbaf0,0xbaed,0xbaeb,0xbae8,0xbae4,0xbae1,0xbadf,0xbadc,0xbad8,0xbad5,0xbad3,
	0xbad0,0xbacc,0xbaca,0xbac7,0xbac3,0xbac0,0xbabe,0xbabb,0xbab7,0xbab4,0xbab2,0xbaaf,0xbaab,0xbaa9,0xbaa6,0xbaa3,
	0xba9f,0xba9d,0xba9a,0xba96,0xba93,0xba91,0xba8e,0xba8a,0xba87,0xba85,0xba82,0xba7e,0xba7c,0xba79,0xba76,0xba72,
	0xba70,0xba6d,0xba69,0xba66,0xba64,0xba61,0xba5d,0xba5c,0xba58,0xba55,0xba51,0xba4f,0xba4c,0xba49,0xba45,0xba43,
	0xba40,0xba3c,0xba39,0xba37,0xba34,0xba30,0xba2f,0xba2b,0xba28,0xba24,0xba23,0xba1f,0xba1c,0xba18,0xba16,0xba13,
	0xba10,0xba0d,0xba09,0xba08,0xba04,0xba01,0xb9fd,0xb9fc,0xb9f8,0xb9f5,0xb9f3,0xb9ef,0xb9ec,0xb9e9,0xb9e7,0xb9e3,
	0xb9e0,0xb9dd,0xb9db,0xb9d7,0xb9d4,0xb9d2,0xb9cf,0xb9cb,0xb9c8,0xb9c6,0xb9c3,0xb9bf,0xb9bd,0xb9ba,0xb9b7,0xb9b3,
	0xb9b1,0xb9ae,0xb9ab,0xb9a7,0xb9a5,0xb9a2,0xb99e,0xb99d,0xb999,0xb996,0xb992,0xb991,0xb98d,0xb98a,0xb988,0xb985,
	0xb981,0xb97e,0xb97c,0xb979,0xb975,0xb972,0xb970,0xb96d,0xb969,0xb967,0xb964,0xb960,0xb95d,0xb95b,0xb958,0xb954,
	0xb953,0xb94f,0xb94c,0xb948,0xb947,0xb943,0xb940,0xb93c,0xb93b,0xb937,0xb934,0xb932,0xb92f,0xb92b,0xb928,0xb926,
	0xb922,0xb91f,0xb91c,0xb91a,0xb916,0xb913,0xb911,0xb90e,0xb90a,0xb907,0xb905,0xb902,0xb8fe,0xb8fd,0xb8f9,0xb8f6,
	0xb8f2,0xb8f0,0xb8ed,0xb8ea,0xb8e6,0xb8e4,0xb8e1,0xb8de,0xb8dc,0xb8d8,0xb8d5,0xb8d1,0xb8d0,0xb8cc,0xb8c9,0xb8c7,
	0xb8c4,0xb8c0,0xb8bd,0xb8bb,0xb8b8,0xb8b4,0xb8b1,0xb8af,0xb8ac,0xb8a8,0xb8a6,0xb8a3,0xb8a0,0xb89c,0xb89a,0xb897,
	0xb894,0xb890,0xb88f,0xb88b,0xb888,0xb884,0xb883,0xb87f,0xb87c,0xb87a,0xb877,0xb873,0xb870,0xb86e,0xb86b,0xb867,
	0xb865,0xb862,0xb85f,0xb85b,0xb859,0xb856,0xb853,0xb84f,0xb84d,0xb84a,0xb846,0xb845,0xb841,0xb83e,0xb83a,0xb839,
	0xb835,0xb832,0xb830,0xb82d,0xb829,0xb826,0xb824,0xb821,0xb81d,0xb81c,0xb818,0xb815,0xb811,0xb80f,0xb80c,0xb809,
	0xb807,0xb803,0xb800,0xb7fd,0xb7fb,0xb7f7,0xb7f4,0xb7f1,0xb7ef,0xb7eb,0xb7e8,0xb7e6,0xb7e3,0xb7df,0xb7dc,0xb7da,
	0xb7d7,0xb7d3,0xb7d2,0xb7ce,0xb7cb,0xb7c7,0xb7c6,0xb7c2,0xb7bf,0xb7bd,0xb7ba,0xb7b6,0xb7b3,0xb7b1,0xb7ae,0xb7aa,
	0xb7a8,0xb7a5,0xb7a1,0xb79e,0xb79c,0xb799,0xb795,0xb792,0xb790,0xb78d,0xb789,0xb788,0xb784,0xb781,0xb77d,0xb77c,
	0xb778,0xb775,0xb773,0xb770,0xb76c,0xb769,0xb767,0xb764,0xb760,0xb75e,0xb75b,0xb758,0xb754,0xb752,0xb74f,0xb74c,
	0xb74a,0xb746,0xb743,0xb740,0xb73e,0xb73a,0xb737,0xb733,0xb732,0xb72e,0xb72b,0xb729,0xb726,0xb722,0xb71f,0xb71d,
	0xb71a,0xb717,0xb713,0xb710,0xb70e,0xb70b,0xb707,0xb706,0xb702,0xb6ff,0xb6fc,0xb6fa,0xb6f6,0xb6f3,0xb6f1,0xb6ee,
	0xb6ea,0xb6e7,0xb6e5,0xb6e2,0xb6de,0xb6dd,0xb6d9,0xb6d6,0xb6d2,0xb6d1,0xb6cd,0xb6ca,0xb6c8,0xb6c5,0xb6c1,0xb6be,
	0xb6bc,0xb6b9,0xb6b5,0xb6b4,0xb6b0,0xb6ad,0xb6a9,0xb6a8,0xb6a4,0xb6a1,0xb69f,0xb69c,0xb698,0xb697,0xb693,0xb690,
	0xb68c,0xb68b,0xb687,0xb684,0xb682,0xb67f,0xb67b,0xb678,0xb676,0xb673,0xb66f,0xb66e,0xb66a,0xb667,0xb663,0xb662,
	0xb65e,0xb65b,0xb659,0xb656,0xb652,0xb64f,0xb64d,0xb64a,0xb646,0xb644,0xb641,0xb63e,0xb63a,0xb638,0xb635,0xb632,
	0xb630,0xb62c,0xb629,0xb626,0xb624,0xb621,0xb61d,0xb61b,0xb618,0xb615,0xb613,0xb60f,0xb60c,0xb609,0xb607,0xb603,
	0xb600,0xb5fe,0xb5fb,0xb5f7,0xb5f4,0xb5f2,0xb5ef,0xb5eb,0xb5ea,0xb5e6,0xb5e3,0xb5df,0xb5de,0xb5da,0xb5d7,0xb5d5,
	0xb5d2,0xb5ce,0xb5cb,0xb5c9,0xb5c6,0xb5c2,0xb5c1,0xb5bd,0xb5ba,0xb5b6,0xb5b5,0xb5b1,0xb5ae,0xb5ac,0xb5a9,0xb5a5,
	0xb5a2,0xb5a0,0xb59d,0xb59a,0xb596,0xb595,0xb591,0xb58e,0xb58c,0xb589,0xb585,0xb582,0xb580,0xb57d,0xb579,0xb577,
	0xb574,0xb571,0xb56f,0xb56c,0xb568,0xb565,0xb563,0xb560,0xb55c,0xb55a,0xb557,0xb554,0xb550,0xb54f,0xb54b,0xb548,
	0xb546,0xb543,0xb53f,0xb53d,0xb53a,0xb537,0xb533,0xb532,0xb52e,0xb52b,0xb529,0xb526,0xb522,0xb51f,0xb51d,0xb51a,
	0xb516,0xb514,0xb511,0xb50e,0xb50a,0xb509,0xb505,0xb502,0xb500,0xb4fd,0xb4f9,0xb4f7,0xb4f4,0xb4f1,0xb4ed,0xb4ec,
	0xb4e8,0xb4e5,0xb4e3,0xb4e0,0xb4dc,0xb4d9,0xb4d7,0xb4d4,0xb4d0,0xb4ce,0xb4cb,0xb4c8,0xb4c6,0xb4c3,0xb4bf,0xb4bc,
	0xb4ba,0xb4b7,0xb4b3,0xb4b1,0xb4ae,0xb4ab,0xb4a7,0xb4a6,0xb4a2,0xb49f,0xb49d,0xb49a,0xb496,0xb493,0xb491,0xb48e,
	0xb48a,0xb489,0xb485,0xb482,0xb480,0xb47d,0xb479,0xb476,0xb474,0xb471,0xb46d,0xb46b,0xb468,0xb465,0xb461,0xb460,
	0xb45c,0xb459,0xb457,0xb454,0xb450,0xb44e,0xb44b,0xb448,0xb444,0xb443,0xb43f,0xb43c,0xb43a,0xb437,0xb433,0xb430,
	0xb42d,0xb42b,0xb428,0xb424,0xb421,0xb41f,0xb41c,0xb418,0xb417,0xb413,0xb410,0xb40e,0xb40b,0xb407,0xb404,0xb402,
	0xb3ff,0xb3fb,0xb3fa,0xb3f6,0xb3f3,0xb3f1,0xb3ee,0xb3ea,0xb3e7,0xb3e5,0xb3e2,0xb3df,0xb3dd,0xb3da,0xb3d6,0xb3d4,
	0xb3d1,0xb3ce,0xb3ca,0xb3c9,0xb3c5,0xb3c2,0xb3c0,0xb3bd,0xb3b9,0xb3b8,0xb3b4,0xb3b1,0xb3ad,0xb3ac,0xb3a8,0xb3a5,
	0xb3a3,0xb3a0,0xb39c,0xb39b,0xb397,0xb394,0xb390,0xb38f,0xb38b,0xb388,0xb386,0xb383,0xb37f,0xb37e,0xb37a,0xb377,
	0xb375,0xb372,0xb36e,0xb36b,0xb369,0xb366,0xb363,0xb361,0xb35d,0xb35a,0xb358,0xb355,0xb352,0xb34e,0xb34c,0xb349,
	0xb346,0xb344,0xb341,0xb33d,0xb33b,0xb338,0xb335,0xb331,0xb330,0xb32c,0xb329,0xb327,0xb324,0xb320,0xb31f,0xb31b,
	0xb318,0xb314,0xb313,0xb30f,0xb30c,0xb30a,0xb307,0xb303,0xb302,0xb2fe,0xb2fb,0xb2f8,0xb2f6,0xb2f2,0xb2ef,0xb2ed,
	0xb2ea,0xb2e7,0xb2e5,0xb2e1,0xb2de,0xb2db,0xb2d9,0xb2d6,0xb2d2,0xb2d0,0xb2cd,0xb2ca,0xb2c8,0xb2c5,0xb2c1,0xb2be,
	0xb2bc,0xb2b9,0xb2b5,0xb2b2,0xb2b0,0xb2ad,0xb2aa,0xb2a8,0xb2a4,0xb2a1,0xb29f,0xb29c,0xb299,0xb295,0xb293,0xb290,
	0xb28d,0xb28b,0xb288,0xb284,0xb283,0xb27f,0xb27c,0xb27a,0xb277,0xb273,0xb270,0xb26e,0xb26b,0xb267,0xb266,0xb262,
	0xb25f,0xb25d,0xb25a,0xb256,0xb253,0xb251,0xb24e,0xb24b,0xb249,0xb246,0xb242,0xb240,0xb23d,0xb23a,0xb238,0xb235,
	0xb231,0xb22e,0xb22c,0xb229,0xb225,0xb224,0xb220,0xb21d,0xb21b,0xb218,0xb214,0xb211,0xb20f,0xb20c,0xb208,0xb207,
	0xb203,0xb200,0xb1fe,0xb1fb,0xb1f8,0xb1f6,0xb1f2,0xb1ef,0xb1ec,0xb1ea,0xb1e7,0xb1e3,0xb1e1,0xb1de,0xb1db,0xb1d9,
	0xb1d6,0xb1d2,0xb1cf,0xb1cd,0xb1ca,0xb1c6,0xb1c5,0xb1c1,0xb1be,0xb1bc,0xb1b9,0xb1b5,0xb1b4,0xb1b0,0xb1ad,0xb1aa,
	0xb1a8,0xb1a4,0xb1a1,0xb19f,0xb19c,0xb199,0xb197,0xb194,0xb190,0xb18d,0xb18b,0xb188,0xb184,0xb183,0xb17f,0xb17c,
	0xb17a,0xb177,0xb173,0xb172,0xb16e,0xb16b,0xb167,0xb166,0xb162,0xb15f,0xb15d,0xb15a,0xb156,0xb155,0xb151,0xb14e,
	0xb14c,0xb149,0xb145,0xb144,0xb140,0xb13d,0xb13b,0xb138,0xb134,0xb133,0xb12f,0xb12c,0xb129,0xb127,0xb124,0xb120,
	0xb11e,0xb11b,0xb118,0xb116,0xb113,0xb10f,0xb10e,0xb10a,0xb107,0xb103,0xb102,0xb0fe,0xb0fb,0xb0f9,0xb0f6,0xb0f2,
	0xb0f1,0xb0ed,0xb0ea,0xb0e8,0xb0e5,0xb0e2,0xb0de,0xb0dc,0xb0d9,0xb0d6,0xb0d4,0xb0d1,0xb0cd,0xb0cc,0xb0c8,0xb0c5,
	0xb0c3,0xb0c0,0xb0bc,0xb0b9,0xb0b7,0xb0b4,0xb0b1,0xb0af,0xb0ab,0xb0a8,0xb0a6,0xb0a3,0xb0a0,0xb09e,0xb09b,0xb097,
	0xb095,0xb092,0xb08f,0xb08b,0xb08a,0xb086,0xb083,0xb081,0xb07e,0xb07a,0xb079,0xb075,0xb072,0xb070,0xb06d,0xb069,
	0xb066,0xb064,0xb061,0xb05e,0xb05c,0xb059,0xb055,0xb053,0xb050,0xb04d,0xb04b,0xb048,0xb044,0xb041,0xb03f,0xb03c,
	0xb038,0xb037,0xb033,0xb030,0xb02e,0xb02b,0xb028,0xb026,0xb022,0xb01f,0xb01c,0xb01a,0xb017,0xb013,0xb012,0xb00e,
	0xb00b,0xb009,0xb006,0xb002,0xb001,0xaffd,0xaffa,0xaff6,0xaff5,0xaff1,0xafee,0xafec,0xafe9,0xafe6,0xafe4,0xafe0,
	0xafde,0xafdb,0xafd8,0xafd6,0xafd2,0xafcf,0xafcd,0xafca,0xafc7,0xafc5,0xafc2,0xafbe,0xafbd,0xafb9,0xafb6,0xafb2,
	0xafb1,0xafad,0xafaa,0xafa8,0xafa5,0xafa2,0xafa0,0xaf9d,0xaf99,0xaf98,0xaf94,0xaf91,0xaf8f,0xaf8c,0xaf88,0xaf87,
	0xaf83,0xaf80,0xaf7e,0xaf7b,0xaf78,0xaf74,0xaf72,0xaf6f,0xaf6c,0xaf6a,0xaf67,0xaf63,0xaf62,0xaf5e,0xaf5b,0xaf59,
	0xaf56,0xaf52,0xaf51,0xaf4d,0xaf4a,0xaf48,0xaf45,0xaf42,0xaf3e,0xaf3d,0xaf39,0xaf36,0xaf34,0xaf31,0xaf2d,0xaf2c,
	0xaf28,0xaf25,0xaf23,0xaf20,0xaf1d,0xaf1b,0xaf18,0xaf14,0xaf13,0xaf0f,0xaf0c,0xaf08,0xaf07,0xaf03,0xaf00,0xaefe,
	0xaefb,0xaef8,0xaef6,0xaef3,0xaeef,0xaeed,0xaeea,0xaee7,0xaee5,0xaee2,0xaede,0xaedd,0xaed9,0xaed6,0xaed3,0xaed1,
	0xaecd,0xaeca,0xaec8,0xaec5,0xaec2,0xaec0,0xaebd,0xaeb9,0xaeb8,0xaeb4,0xaeb1,0xaeaf,0xaeac,0xaea8,0xaea7,0xaea3,
	0xaea0,0xae9e,0xae9b,0xae98,0xae94,0xae93,0xae8f,0xae8c,0xae8a,0xae87,0xae83,0xae82,0xae7e,0xae7b,0xae79,0xae76,
	0xae74,0xae71,0xae6d,0xae6a,0xae68,0xae65,0xae61,0xae60,0xae5c,0xae59,0xae57,0xae54,0xae51,0xae4f,0xae4c,0xae48,
	0xae47,0xae43,0xae40,0xae3e,0xae3b,0xae37,0xae36,0xae32,0xae2f,0xae2d,0xae2a,0xae27,0xae23,0xae22,0xae1e,0xae1b,
	0xae19,0xae16,0xae12,0xae11,0xae0d,0xae0a,0xae08,0xae05,0xae02,0xae00,0xadfd,0xadf9,0xadf8,0xadf4,0xadf1,0xadef,
	0xadec,0xade8,0xade7,0xade3,0xade0,0xaddd,0xaddb,0xadd8,0xadd4,0xadd3,0xadcf,0xadcc,0xadca,0xadc7,0xadc4,0xadc2,
	0xadbe,0xadbb,0xadb9,0xadb6,0xadb3,0xadb1,0xadae,0xadaa,0xada9,0xada5,0xada2,0xada0,0xad9d,0xad9a,0xad96,0xad95,
	0xad91,0xad8e,0xad8c,0xad89,0xad85,0xad84,0xad80,0xad7d,0xad7b,0xad78,0xad75,0xad73,0xad70,0xad6c,0xad6b,0xad67,
	0xad64,0xad62,0xad5f,0xad5b,0xad5a,0xad56,0xad53,0xad50,0xad4e,0xad4b,0xad47,0xad46,0xad42,0xad3f,0xad3d,0xad3a,
	0xad36,0xad35,0xad31,0xad2e,0xad2c,0xad29,0xad26,0xad24,0xad21,0xad1d,0xad1c,0xad18,0xad15,0xad13,0xad10,0xad0c,
	0xad0a,0xad06,0xad05,0xad01,0xacfe,0xacfc,0xacf9,0xacf6,0xacf4,0xacf1,0xaced,0xacec,0xace8,0xace5,0xace3,0xace0,
	0xacdd,0xacdb,0xacd8,0xacd4,0xacd3,0xaccf,0xaccc,0xacca,0xacc7,0xacc4,0xacc2,0xacbf,0xacbb,0xacba,0xacb6,0xacb3,
	0xacb1,0xacae,0xacab,0xaca7,0xaca6,0xaca2,0xac9f,0xac9d,0xac9a,0xac96,0xac95,0xac91,0xac8e,0xac8c,0xac89,0xac86,
	0xac84,0xac81,0xac7d,0xac7c,0xac78,0xac75,0xac73,0xac70,0xac6d,0xac6b,0xac68,0xac64,0xac63,0xac5f,0xac5c,0xac5a,
	0xac57,0xac54,0xac52,0xac4f,0xac4b,0xac4a,0xac46,0xac43,0xac41,0xac3e,0xac3b,0xac39,0xac36,0xac32,0xac30,0xac2d,
	0xac2a,0xac28,0xac25,0xac21,0xac20,0xac1c,0xac19,0xac17,0xac14,0xac11,0xac0f,0xac0c,0xac08,0xac07,0xac03,0xac00,
	0xabfe,0xabfb,0xabf8,0xabf4,0xabf3,0xabef,0xabec,0xabea,0xabe7,0xabe4,0xabe2,0xabdf,0xabdb,0xabda,0xabd6,0xabd3,
	0xabd1,0xabce,0xabcb,0xabc9,0xabc5,0xabc2,0xabc0,0xabbd,0xabba,0xabb8,0xabb5,0xabb1,0xabb0,0xabac,0xaba9,0xaba7,
	0xaba3,0xaba2,0xab9e,0xab9b,0xab99,0xab96,0xab93,0xab91,0xab8e,0xab8a,0xab89,0xab85,0xab82,0xab80,0xab7d,0xab7a,
	0xab78,0xab75,0xab71,0xab70,0xab6c,0xab69,0xab67,0xab64,0xab61,0xab5f,0xab5c,0xab58,0xab57,0xab53,0xab50,0xab4e,
	0xab4b,0xab48,0xab46,0xab43,0xab3f,0xab3e,0xab3a,0xab37,0xab35,0xab32,0xab2f,0xab2d,0xab2a,0xab26,0xab25,0xab21,
	0xab1e,0xab1c,0xab19,0xab16,0xab14,0xab11,0xab0d,0xab0c,0xab08,0xab05,0xab03,0xab00,0xaafd,0xaafb,0xaaf8,0xaaf4,
	0xaaf3,0xaaef,0xaaec,0xaae9,0xaae7,0xaae4,0xaae0,0xaadf,0xaadb,0xaad8,0xaad6,0xaad3,0xaad0,0xaace,0xaacb,0xaac7,
	0xaac6,0xaac2,0xaabf,0xaabd,0xaaba,0xaab7,0xaab5,0xaab2,0xaaae,0xaaad,0xaaa9,0xaaa6,0xaaa4,0xaaa1,0xaa9e,0xaa9c,
	0xaa99,0xaa95,0xaa94,0xaa90,0xaa8d,0xaa8b,0xaa88,0xaa85,0xaa83,0xaa80,0xaa7c,0xaa7b,0xaa77,0xaa74,0xaa72,0xaa6f,
	0xaa6c,0xaa6a,0xaa67,0xaa63,0xaa62,0xaa5e,0xaa5b,0xaa59,0xaa56,0xaa53,0xaa51,0xaa4e,0xaa4a,0xaa49,0xaa45,0xaa42,
	0xaa41,0xaa3e,0xaa3a,0xaa39,0xaa35,0xaa32,0xaa30,0xaa2d,0xaa2a,0xaa28,0xaa25,0xaa21,0xaa20,0xaa1c,0xaa19,0xaa17,
	0xaa14,0xaa11,0xaa0f,0xaa0c,0xaa08,0xaa07,0xaa03,0xaa00,0xa9fe,0xa9fb,0xa9f8,0xa9f6,0xa9f3,0xa9ef,0xa9ee,0xa9ea,
	0xa9e7,0xa9e5,0xa9e2,0xa9df,0xa9dd,0xa9da,0xa9d6,0xa9d5,0xa9d1,0xa9ce,0xa9cc,0xa9c9,0xa9c6,0xa9c4,0xa9c1,0xa9bd,
	0xa9bc,0xa9b8,0xa9b5,0xa9b3,0xa9b0,0xa9ad,0xa9ab,0xa9a8,0xa9a4,0xa9a3,0xa99f,0xa99c,0xa99a,0xa997,0xa994,0xa992,
	0xa98f,0xa98c,0xa98a,0xa987,0xa983,0xa982,0xa97e,0xa97b,0xa979,0xa976,0xa973,0xa971,0xa96e,0xa96a,0xa969,0xa965,
	0xa962,0xa960,0xa95d,0xa95a,0xa958,0xa955,0xa951,0xa950,0xa94c,0xa949,0xa947,0xa944,0xa941,0xa93f,0xa93c,0xa938,
	0xa937,0xa933,0xa930,0xa92e,0xa92b,0xa928,0xa926,0xa923,0xa91f,0xa91e,0xa91a,0xa917,0xa915,0xa912,0xa90f,0xa90d,
	0xa90a,0xa906,0xa905,0xa901,0xa8fe,0xa8fc,0xa8f9,0xa8f6,0xa8f4,0xa8f1,0xa8ed,0xa8ec,0xa8e8,0xa8e5,0xa8e3,0xa8e0,
	0xa8de,0xa8db,0xa8d9,0xa8d6,0xa8d2,0xa8d1,0xa8cd,0xa8ca,0xa8c8,0xa8c5,0xa8c2,0xa8c0,0xa8bd,0xa8ba,0xa8b8,0xa8b5,
	0xa8b3,0xa8b0,0xa8ac,0xa8ab,0xa8a7,0xa8a4,0xa8a2,0xa89f,0xa89c,0xa89a,0xa897,0xa893,0xa892,0xa88e,0xa88b,0xa889,
	0xa886,0xa883,0xa881,0xa87e,0xa87b,0xa879,0xa876,0xa872,0xa871,0xa86d,0xa86a,0xa868,0xa865,0xa862,0xa860,0xa85d,
	0xa85b,0xa858,0xa854,0xa853,0xa850,0xa84c,0xa84b,0xa847,0xa844,0xa842,0xa83f,0xa83c,0xa83a,0xa837,0xa833,0xa832,
	0xa82e,0xa82b,0xa829,0xa826,0xa823,0xa821,0xa81e,0xa81b,0xa819,0xa816,0xa812,0xa811,0xa80d,0xa80a,0xa808,0xa805,
	0xa803,0xa800,0xa7fd,0xa7fb,0xa7f8,0xa7f4,0xa7f3,0xa7ef,0xa7ec,0xa7ea,0xa7e7,0xa7e4,0xa7e2,0xa7df,0xa7dc,0xa7da,
	0xa7d7,0xa7d3,0xa7d2,0xa7ce,0xa7cb,0xa7c9,0xa7c6,0xa7c3,0xa7c1,0xa7be,0xa7ba,0xa7b9,0xa7b5,0xa7b2,0xa7b1,0xa7ad,
	0xa7ac,0xa7a8,0xa7a5,0xa7a3,0xa7a0,0xa79d,0xa79b,0xa798,0xa794,0xa793,0xa78f,0xa78c,0xa78a,0xa787,0xa784,0xa782,
	0xa77f,0xa77c,0xa779,0xa777,0xa774,0xa770,0xa76f,0xa76b,0xa76a,0xa766,0xa763,0xa761,0xa75e,0xa75b,0xa759,0xa756,
	0xa753,0xa751,0xa74e,0xa74a,0xa749,0xa745,0xa742,0xa740,0xa73d,0xa73a,0xa738,0xa735,0xa732,0xa730,0xa72d,0xa72b,
	0xa728,0xa724,0xa723,0xa71f,0xa71c,0xa71a,0xa717,0xa714,0xa712,0xa70f,0xa70c,0xa70a,0xa707,0xa703,0xa702,0xa6fe,
	0xa6fb,0xa6f9,0xa6f6,0xa6f3,0xa6f1,0xa6ee,0xa6ec,0xa6e9,0xa6e6,0xa6e4,0xa6e1,0xa6dd,0xa6dc,0xa6d8,0xa6d5,0xa6d3,
	0xa6d0,0xa6cd,0xa6cb,0xa6c8,0xa6c5,0xa6c3,0xa6c0,0xa6bc,0xa6bb,0xa6b7,0xa6b4,0xa6b2,0xa6af,0xa6ad,0xa6aa,0xa6a7,
	0xa6a5,0xa6a2,0xa69f,0xa69d,0xa69a,0xa696,0xa695,0xa691,0xa68e,0xa68c,0xa689,0xa686,0xa684,0xa681,0xa67d,0xa67c,
	0xa679,0xa675,0xa674,0xa670,0xa66f,0xa66b,0xa668,0xa666,0xa663,0xa660,0xa65e,0xa65b,0xa657,0xa656,0xa653,0xa64f,
	0xa64e,0xa64a,0xa647,0xa645,0xa642,0xa63f,0xa63d,0xa63a,0xa636,0xa635,0xa631,0xa630,0xa62d,0xa629,0xa628,0xa624,
	0xa621,0xa61e,0xa61b,0xa619,0xa616,0xa614,0xa611,0xa60e,0xa60c,0xa609,0xa605,0xa604,0xa601,0xa5fd,0xa5fc,0xa5f8,
	0xa5f5,0xa5f3,0xa5f0,0xa5ee,0xa5eb,0xa5e8,0xa5e6,0xa5e3,0xa5e0,0xa5de,0xa5db,0xa5d7,0xa5d6,0xa5d3,0xa5cf,0xa5ce,
	0xa5ca,0xa5c7,0xa5c5,0xa5c2,0xa5c0,0xa5bd,0xa5ba,0xa5b8,0xa5b5,0xa5b2,0xa5b0,0xa5ad,0xa5a9,0xa5a8,0xa5a4,0xa5a1,
	0xa5a0,0xa59c,0xa59b,0xa597,0xa594,0xa592,0xa58f,0xa58c,0xa58a,0xa587,0xa584,0xa582,0xa57f,0xa57b,0xa57a,0xa576,
	0xa575,0xa571,0xa56e,0xa56d,0xa569,0xa566,0xa564,0xa561,0xa55e,0xa55c,0xa559,0xa556,0xa554,0xa551,0xa54d,0xa54c,
	0xa548,0xa547,0xa543,0xa540,0xa53e,0xa53b,0xa538,0xa536,0xa533,0xa530,0xa52e,0xa52b,0xa527,0xa526,0xa523,0xa521,
	0xa51e,0xa51a,0xa519,0xa515,0xa512,0xa510,0xa50d,0xa50a,0xa508,0xa505,0xa502,0xa500,0xa4fd,0xa4f9,0xa4f8,0xa4f4,
	0xa4f3,0xa4f0,0xa4ec,0xa4eb,0xa4e7,0xa4e4,0xa4e2,0xa4df,0xa4dc,0xa4da,0xa4d7,0xa4d4,0xa4d2,0xa4cf,0xa4cd,0xa4ca,
	0xa4c7,0xa4c3,0xa4c2,0xa4be,0xa4bb,0xa4b9,0xa4b6,0xa4b4,0xa4b1,0xa4ae,0xa4ac,0xa4a9,0xa4a6,0xa4a4,0xa4a1,0xa49e,
	0xa49c,0xa499,0xa495,0xa494,0xa490,0xa48f,0xa48b,0xa488,0xa487,0xa483,0xa480,0xa47e,0xa47b,0xa478,0xa476,0xa473,
	0xa471,0xa46e,0xa46b,0xa469,0xa466,0xa462,0xa461,0xa45e,0xa45a,0xa459,0xa455,0xa452,0xa450,0xa44d,0xa44b,0xa448,
	0xa445,0xa443,0xa440,0xa43d,0xa43b,0xa438,0xa435,0xa433,0xa430,0xa42c,0xa42b,0xa427,0xa426,0xa422,0xa41f,0xa41e,
	0xa41a,0xa417,0xa415,0xa412,0xa40f,0xa40d,0xa40a,0xa408,0xa405,0xa402,0xa400,0xa3fd,0xa3f9,0xa3f8,0xa3f5,0xa3f1,
	0xa3f0,0xa3ec,0xa3e9,0xa3e7,0xa3e4,0xa3e2,0xa3df,0xa3dc,0xa3da,0xa3d7,0xa3d4,0xa3d2,0xa3cf,0xa3cc,0xa3ca,0xa3c7,
	0xa3c5,0xa3c2,0xa3be,0xa3bd,0xa3b9,0xa3b6,0xa3b5,0xa3b1,0xa3ae,0xa3ac,0xa3a9,0xa3a6,0xa3a4,0xa3a1,0xa39f,0xa39c,
	0xa399,0xa397,0xa394,0xa390,0xa38f,0xa38c,0xa388,0xa387,0xa383,0xa380,0xa37e,0xa37b,0xa379,0xa376,0xa373,0xa371,
	0xa36e,0xa36c,0xa369,0xa365,0xa364,0xa361,0xa35d,0xa35c,0xa358,0xa357,0xa353,0xa350,0xa34f,0xa34b,0xa348,0xa346,
	0xa343,0xa340,0xa33e,0xa33b,0xa339,0xa336,0xa333,0xa331,0xa32e,0xa32b,0xa329,0xa326,0xa322,0xa321,0xa31d,0xa31c,
	0xa319,0xa315,0xa314,0xa310,0xa30d,0xa30b,0xa308,0xa305,0xa303,0xa300,0xa2fe,0xa2fb,0xa2f8,0xa2f6,0xa2f3,0xa2f0,
	0xa2ee,0xa2eb,0xa2e7,0xa2e6,0xa2e3,0xa2e1,0xa2de,0xa2da,0xa2d9,0xa2d5,0xa2d2,0xa2d1,0xa2cd,0xa2ca,0xa2c8,0xa2c5,
	0xa2c3,0xa2c0,0xa2bd,0xa2bb,0xa2b8,0xa2b5,0xa2b3,0xa2b0,0xa2ad,0xa2ab,0xa2a8,0xa2a4,0xa2a3,0xa29f,0xa29e,0xa29b,
	0xa297,0xa296,0xa292,0xa28f,0xa28d,0xa28a,0xa287,0xa285,0xa282,0xa280,0xa27d,0xa27a,0xa278,0xa275,0xa272,0xa270,
	0xa26d,0xa269,0xa268,0xa265,0xa263,0xa260,0xa25c,0xa25b,0xa257,0xa254,0xa253,0xa24f,0xa24c,0xa24a,0xa247,0xa245,
	0xa242,0xa23f,0xa23d,0xa23a,0xa237,0xa235,0xa232,0xa22f,0xa22d,0xa22a,0xa228,0xa225,0xa221,0xa220,0xa21c,0xa219,
	0xa217,0xa215,0xa212,0xa20e,0xa20d,0xa20a,0xa206,0xa205,0xa201,0xa200,0xa1fc,0xa1f9,0xa1f8,0xa1f4,0xa1f1,0xa1ef,
	0xa1ec,0xa1eb,0xa1e7,0xa1e4,0xa1e2,0xa1df,0xa1dc,0xa1da,0xa1d7,0xa1d5,0xa1d2,0xa1cf,0xa1cd,0xa1ca,0xa1c7,0xa1c5,
	0xa1c2,0xa1bf,0xa1bd,0xa1ba,0xa1b8,0xa1b5,0xa1b2,0xa1b0,0xa1ad,0xa1a9,0xa1a8,0xa1a5,0xa1a3,0xa1a0,0xa19c,0xa19b,
	0xa197,0xa194,0xa193,0xa18f,0xa18e,0xa18a,0xa187,0xa186,0xa182,0xa17f,0xa17d,0xa17a,0xa179,0xa175,0xa172,0xa170,
	0xa16d,0xa16a,0xa168,0xa165,0xa162,0xa160,0xa15d,0xa15b,0xa158,0xa155,0xa153,0xa150,0xa14d,0xa14b,0xa148,0xa146,
	0xa143,0xa140,0xa13e,0xa13b,0xa137,0xa136,0xa132,0xa131,0xa12e,0xa12a,0xa129,0xa125,0xa122,0xa121,0xa11d,0xa11c,
	0xa118,0xa115,0xa114,0xa110,0xa10d,0xa10b,0xa108,0xa105,0xa103,0xa100,0xa0fe,0xa0fb,0xa0f8,0xa0f6,0xa0f3,0xa0f0,
	0xa0ee,0xa0eb,0xa0e9,0xa0e6,0xa0e3,0xa0e1,0xa0de,0xa0db,0xa0d9,0xa0d6,0xa0d4,0xa0d1,0xa0ce,0xa0cc,0xa0c9,0xa0c5,
	0xa0c3,0xa0c0,0xa0be,0xa0bb,0xa0b9,0xa0b6,0xa0b3,0xa0b1,0xa0ae,0xa0ab,0xa0a9,0xa0a6,0xa0a4,0xa0a1,0xa09e,0xa09c,
	0xa099,0xa096,0xa094,0xa091,0xa08f,0xa08c,0xa089,0xa087,0xa084,0xa080,0xa07f,0xa07c,0xa07a,0xa077,0xa073,0xa072,
	0xa06f,0xa06b,0xa06a,0xa066,0xa065,0xa062,0xa05e,0xa05d,0xa059,0xa056,0xa055,0xa051,0xa050,0xa04c,0xa049,0xa048,
	0xa044,0xa041,0xa03f,0xa03c,0xa03b,0xa037,0xa034,0xa032,0xa02f,0xa02c,0xa02a,0xa027,0xa025,0xa022,0xa01f,0xa01d,
	0xa01a,0xa017,0xa015,0xa012,0xa010,0xa00d,0xa00a,0xa008,0xa005,0xa002,0xa000,0x9ffd,0x9ffb,0x9ff8,0x9ff5,0x9ff3,
	0x9ff0,0x9fed,0x9feb,0x9fe8,0x9fe6,0x9fe3,0x9fe0,0x9fde,0x9fdb,0x9fd7,0x9fd6,0x9fd3,0x9fd1,0x9fce,0x9fca,0x9fc9,
	0x9fc6,0x9fc2,0x9fc1,0x9fbd,0x9fbc,0x9fb9,0x9fb5,0x9fb4,0x9fb0,0x9fad,0x9fac,0x9fa8,0x9fa7,0x9fa3,0x9fa0,0x9f9f,
	0x9f9b,0x9f98,0x9f96,0x9f93,0x9f92,0x9f8e,0x9f8b,0x9f89,0x9f86,0x9f83,0x9f81,0x9f7e,0x9f7c,0x9f79,0x9f76,0x9f74,
	0x9f71,0x9f6e,0x9f6d,0x9f69,0x9f66,0x9f65,0x9f61,0x9f5e,0x9f5c,0x9f59,0x9f58,0x9f54,0x9f51,0x9f4f,0x9f4c,0x9f49,
	0x9f47,0x9f44,0x9f42,0x9f3f,0x9f3c,0x9f3a,0x9f37,0x9f34,0x9f32,0x9f2f,0x9f2d,0x9f2a,0x9f27,0x9f25,0x9f22,0x9f1f,
	0x9f1d,0x9f1a,0x9f18,0x9f15,0x9f12,0x9f10,0x9f0d,0x9f0b,0x9f08,0x9f05,0x9f03,0x9f00,0x9efd,0x9efb,0x9ef8,0x9ef6,
	0x9ef3,0x9ef0,0x9eee,0x9eeb,0x9ee8,0x9ee6,0x9ee3,0x9ee1,0x9ede,0x9edb,0x9ed9,0x9ed6,0x9ed3,0x9ed1,0x9ece,0x9ecc,
	0x9ec9,0x9ec6,0x9ec4,0x9ec1,0x9ebf,0x9ebc,0x9eb9,0x9eb7,0x9eb4,0x9eb1,0x9eaf,0x9eac,0x9eaa,0x9ea7,0x9ea4,0x9ea2,
	0x9e9f,0x9e9b,0x9e9a,0x9e97,0x9e95,0x9e92,0x9e8f,0x9e8d,0x9e8a,0x9e86,0x9e85,0x9e82,0x9e80,0x9e7d,0x9e79,0x9e78,
	0x9e75,0x9e73,0x9e70,0x9e6c,0x9e6b,0x9e68,0x9e64,0x9e63,0x9e60,0x9e5e,0x9e5b,0x9e57,0x9e56,0x9e53,0x9e4f,0x9e4e,
	0x9e4a,0x9e49,0x9e46,0x9e42,0x9e41,0x9e3d,0x9e3a,0x9e39,0x9e35,0x9e34,0x9e31,0x9e2d,0x9e2c,0x9e28,0x9e25,0x9e24,
	0x9e21,0x9e1e,0x9e1c,0x9e19,0x9e15,0x9e14,0x9e11,0x9e0d,0x9e0c,0x9e09,0x9e07,0x9e04,0x9e01,0x9dff,0x9dfc,0x9dfa,
	0x9df7,0x9df4,0x9df2,0x9def,0x9dec,0x9dea,0x9de7,0x9de5,0x9de2,0x9ddf,0x9ddd,0x9dda,0x9dd8,0x9dd5,0x9dd2,0x9dd0,
	0x9dcd,0x9dcb,0x9dc8,0x9dc5,0x9dc3,0x9dc0,0x9dbd,0x9dbb,0x9db8,0x9db6,0x9db3,0x9db0,0x9dae,0x9dab,0x9da9,0x9da6,
	0x9da3,0x9da1,0x9d9e,0x9d9b,0x9d99,0x9d96,0x9d94,0x9d91,0x9d8e,0x9d8c,0x9d89,0x9d88,0x9d84,0x9d81,0x9d7f,0x9d7c,
	0x9d7b,0x9d77,0x9d74,0x9d73,0x9d6f,0x9d6c,0x9d6a,0x9d67,0x9d66,0x9d62,0x9d5f,0x9d5e,0x9d5a,0x9d59,0x9d56,0x9d52,
	0x9d51,0x9d4d,0x9d4a,0x9d49,0x9d45,0x9d44,0x9d41,0x9d3d,0x9d3c,0x9d38,0x9d37,0x9d34,0x9d30,0x9d2f,0x9d2c,0x9d28,
	0x9d27,0x9d23,0x9d22,0x9d1f,0x9d1b,0x9d1a,0x9d17,0x9d15,0x9d12,0x9d0f,0x9d0d,0x9d0a,0x9d08,0x9d05,0x9d02,0x9d00,
	0x9cfd,0x9cfa,0x9cf8,0x9cf5,0x9cf3,0x9cf0,0x9ced,0x9ceb,0x9ce8,0x9ce6,0x9ce3,0x9ce0,0x9cde,0x9cdb,0x9cd8,0x9cd6,
	0x9cd3,0x9cd0,0x9cce,0x9ccb,0x9cc9,0x9cc6,0x9cc3,0x9cc1,0x9cbe,0x9cbc,0x9cb9,0x9cb6,0x9cb4,0x9cb1,0x9cae,0x9cac,
	0x9ca9,0x9ca7,0x9ca4,0x9ca1,0x9c9f,0x9c9c,0x9c9b,0x9c97,0x9c94,0x9c92,0x9c8f,0x9c8e,0x9c8a,0x9c87,0x9c86,0x9c82,
	0x9c81,0x9c7e,0x9c7a,0x9c79,0x9c75,0x9c72,0x9c71,0x9c6d,0x9c6c,0x9c69,0x9c65,0x9c64,0x9c61,0x9c5f,0x9c5c,0x9c59,
	0x9c57,0x9c54,0x9c52,0x9c4f,0x9c4c,0x9c4a,0x9c47,0x9c44,0x9c42,0x9c3f,0x9c3d,0x9c3a,0x9c37,0x9c35,0x9c32,0x9c30,
	0x9c2d,0x9c2a,0x9c28,0x9c25,0x9c23,0x9c20,0x9c1d,0x9c1b,0x9c18,0x9c17,0x9c13,0x9c10,0x9c0e,0x9c0b,0x9c08,0x9c06,
	0x9c03,0x9c02,0x9bfe,0x9bfb,0x9bfa,0x9bf6,0x9bf5,0x9bf2,0x9bee,0x9bed,0x9be9,0x9be8,0x9be5,0x9be1,0x9be0,0x9bdd,
	0x9bdb,0x9bd8,0x9bd5,0x9bd3,0x9bd0,0x9bcd,0x9bcb,0x9bc8,0x9bc6,0x9bc3,0x9bc0,0x9bbe,0x9bbb,0x9bb9,0x9bb6,0x9bb3,
	0x9bb1,0x9bae,0x9bac,0x9ba9,0x9ba6,0x9ba4,0x9ba1,0x9b9e,0x9b9c,0x9b99,0x9b97,0x9b94,0x9b91,0x9b8f,0x9b8c,0x9b8b,
	0x9b87,0x9b85,0x9b82,0x9b7f,0x9b7d,0x9b7a,0x9b78,0x9b75,0x9b72,0x9b70,0x9b6d,0x9b6c,0x9b68,0x9b65,0x9b64,0x9b60,
	0x9b5f,0x9b5c,0x9b58,0x9b57,0x9b53,0x9b52,0x9b4f,0x9b4b,0x9b4a,0x9b47,0x9b45,0x9b42,0x9b3f,0x9b3d,0x9b3a,0x9b37,
	0x9b35,0x9b32,0x9b30,0x9b2d,0x9b2a,0x9b28,0x9b25,0x9b23,0x9b20,0x9b1d,0x9b1b,0x9b18,0x9b16,0x9b13,0x9b10,0x9b0e,
	0x9b0b,0x9b0a,0x9b06,0x9b03,0x9b02,0x9afe,0x9afd,0x9afa,0x9af6,0x9af5,0x9af2,0x9af0,0x9aed,0x9aea,0x9ae8,0x9ae5,
	0x9ae3,0x9ae0,0x9add,0x9adb,0x9ad8,0x9ad5,0x9ad3,0x9ad0,0x9ace,0x9acb,0x9ac8,0x9ac6,0x9ac3,0x9ac1,0x9abe,0x9abb,
	0x9ab9,0x9ab6,0x9ab5,0x9ab1,0x9aae,0x9aad,0x9aa9,0x9aa8,0x9aa4,0x9aa1,0x9aa0,0x9a9c,0x9a9b,0x9a98,0x9a94,0x9a93,
	0x9a90,0x9a8e,0x9a8b,0x9a88,0x9a86,0x9a83,0x9a80,0x9a7e,0x9a7b,0x9a79,0x9a76,0x9a73,0x9a71,0x9a6e,0x9a6c,0x9a69,
	0x9a66,0x9a64,0x9a61,0x9a5f,0x9a5c,0x9a59,0x9a57,0x9a54,0x9a53,0x9a4f,0x9a4c,0x9a4b,0x9a47,0x9a46,0x9a43,0x9a3f,
	0x9a3e,0x9a3b,0x9a38,0x9a36,0x9a33,0x9a30,0x9a2e,0x9a2b,0x9a2a,0x9a26,0x9a23,0x9a22,0x9a1e,0x9a1d,0x9a1a,0x9a16,
	0x9a15,0x9a12,0x9a10,0x9a0d,0x9a0a,0x9a08,0x9a05,0x9a03,0x9a00,0x99fd,0x99fb,0x99f8,0x99f6,0x99f3,0x99f0,0x99ee,
	0x99eb,0x99ea,0x99e6,0x99e3,0x99e2,0x99de,0x99dd,0x99da,0x99d6,0x99d5,0x99d2,0x99d0,0x99cd,0x99ca,0x99c8,0x99c5,
	0x99c3,0x99c0,0x99bd,0x99bb,0x99b8,0x99b6,0x99b3,0x99b0,0x99ae,0x99ab,0x99aa,0x99a6,0x99a3,0x99a2,0x999e,0x999d,
	0x999a,0x9996,0x9995,0x9991,0x9990,0x998d,0x9989,0x9988,0x9985,0x9983,0x9980,0x997d,0x997b,0x9978,0x9976,0x9973,
	0x9970,0x996e,0x996b,0x9969,0x9966,0x9963,0x9961,0x995e,0x995d,0x9959,0x9956,0x9955,0x9951,0x9950,0x994d,0x9949,
	0x9948,0x9945,0x9943,0x9940,0x993d,0x993b,0x9938,0x9936,0x9933,0x9930,0x992e,0x992b,0x9929,0x9926,0x9923,0x9921,
	0x991e,0x991d,0x9919,0x9916,0x9915,0x9911,0x9910,0x990d,0x9909,0x9908,0x9905,0x9903,0x9900,0x98fd,0x98fb,0x98f8,
	0x98f5,0x98f2,0x98f1,0x98ed,0x98ec,0x98e9,0x98e6,0x98e4,0x98e1,0x98df,0x98dc,0x98d9,0x98d7,0x98d4,0x98d2,0x98cf,
	0x98ce,0x98ca,0x98c7,0x98c6,0x98c2,0x98c1,0x98be,0x98ba,0x98b9,0x98b6,0x98b4,0x98b1,0x98ae,0x98ac,0x98a9,0x98a7,
	0x98a4,0x98a1,0x989f,0x989c,0x989b,0x9897,0x9894,0x9893,0x988f,0x988e,0x988b,0x9887,0x9886,0x9883,0x9881,0x987e,
	0x987c,0x9879,0x9876,0x9874,0x9871,0x9870,0x986c,0x9869,0x9868,0x9864,0x9863,0x9860,0x985c,0x985b,0x9858,0x9856,
	0x9853,0x9850,0x984e,0x984b,0x9849,0x9846,0x9843,0x9841,0x983e,0x983d,0x9839,0x9836,0x9835,0x9831,0x9830,0x982d,
	0x982b,0x9828,0x9825,0x9823,0x9820,0x981e,0x981b,0x9818,0x9816,0x9813,0x9812,0x980e,0x980b,0x980a,0x9806,0x9805,
	0x9802,0x97fe,0x97fd,0x97fa,0x97f8,0x97f5,0x97f2,0x97f0,0x97ed,0x97eb,0x97e8,0x97e5,0x97e3,0x97e0,0x97df,0x97db,
	0x97da,0x97d7,0x97d3,0x97d2,0x97cf,0x97cd,0x97ca,0x97c7,0x97c5,0x97c2,0x97c0,0x97bd,0x97ba,0x97b8,0x97b5,0x97b4,
	0x97b0,0x97ad,0x97ac,0x97a9,0x97a5,0x97a4,0x97a1,0x979f,0x979c,0x9799,0x9797,0x9794,0x9792,0x978f,0x978c,0x978a,
	0x9787,0x9786,0x9782,0x9781,0x977e,0x977a,0x9779,0x9776,0x9774,0x9771,0x976e,0x976c,0x9769,0x9767,0x9764,0x9761,
	0x975f,0x975c,0x975b,0x9757,0x9756,0x9753,0x974f,0x974e,0x974b,0x9749,0x9746,0x9743,0x9741,0x973e,0x973c,0x9739,
	0x9736,0x9734,0x9731,0x9730,0x972d,0x9729,0x9728,0x9725,0x9723,0x9720,0x971e,0x971b,0x9718,0x9716,0x9713,0x9711,
	0x970e,0x970b,0x970a,0x9706,0x9705,0x9702,0x96fe,0x96fd,0x96fa,0x96f8,0x96f5,0x96f3,0x96f0,0x96ed,0x96eb,0x96e8,
	0x96e7,0x96e3,0x96e0,0x96df,0x96db,0x96da,0x96d7,0x96d3,0x96d2,0x96cf,0x96cd,0x96ca,0x96c7,0x96c5,0x96c2,0x96c0,
	0x96bd,0x96bc,0x96b8,0x96b5,0x96b4,0x96b1,0x96af,0x96ac,0x96a9,0x96a7,0x96a4,0x96a2,0x969f,0x969c,0x969a,0x9697,
	0x9695,0x9692,0x9691,0x968e,0x968a,0x9689,0x9686,0x9684,0x9681,0x967e,0x967c,0x9679,0x9677,0x9674,0x9671,0x966f,
	0x966d,0x966a,0x9667,0x9666,0x9662,0x9661,0x965e,0x965a,0x9659,0x9656,0x9654,0x9651,0x964e,0x964c,0x9649,0x9647,
	0x9644,0x9643,0x963f,0x963c,0x963b,0x9638,0x9636,0x9633,0x9630,0x962e,0x962b,0x9629,0x9626,0x9624,0x9621,0x961e,
	0x961d,0x9619,0x9618,0x9615,0x9611,0x9610,0x960d,0x960b,0x9608,0x9605,0x9603,0x9600,0x95fe,0x95fb,0x95fa,0x95f6,
	0x95f3,0x95f2,0x95ef,0x95ed,0x95ea,0x95e7,0x95e5,0x95e2,0x95e0,0x95dd,0x95dc,0x95d8,0x95d5,0x95d4,0x95d0,0x95cf,
	0x95cc,0x95c8,0x95c7,0x95c4,0x95c2,0x95bf,0x95bc,0x95ba,0x95b7,0x95b5,0x95b2,0x95b1,0x95ae,0x95aa,0x95a9,0x95a6,
	0x95a4,0x95a1,0x959e,0x959c,0x9599,0x9597,0x9594,0x9593,0x958f,0x958c,0x958b,0x9587,0x9586,0x9583,0x9580,0x957e,
	0x957b,0x9579,0x9576,0x9573,0x9571,0x956e,0x956d,0x9569,0x9568,0x9565,0x9561,0x9560,0x955d,0x955b,0x9558,0x9555,
	0x9553,0x9550,0x954e,0x954b,0x954a,0x9546,0x9543,0x9542,0x953f,0x953d,0x953a,0x9537,0x9535,0x9532,0x9530,0x952d,
	0x952b,0x9529,0x9526,0x9524,0x9521,0x951e,0x951c,0x9519,0x9518,0x9515,0x9511,0x9510,0x950d,0x950b,0x9508,0x9506,
	0x9503,0x9500,0x94fe,0x94fb,0x94fa,0x94f7,0x94f3,0x94f2,0x94ef,0x94ed,0x94ea,0x94e8,0x94e5,0x94e2,0x94e0,0x94dd,
	0x94dc,0x94d8,0x94d5,0x94d4,0x94d1,0x94cf,0x94cc,0x94ca,0x94c7,0x94c4,0x94c2,0x94bf,0x94be,0x94ba,0x94b7,0x94b6,
	0x94b2,0x94b1,0x94ae,0x94ac,0x94a9,0x94a6,0x94a4,0x94a1,0x949f,0x949c,0x9499,0x9498,0x9494,0x9493,0x9490,0x948e,
	0x948b,0x9488,0x9486,0x9483,0x9481,0x947e,0x947d,0x947a,0x9476,0x9475,0x9472,0x9470,0x946d,0x946a,0x9468,0x9465,
	0x9463,0x9460,0x945f,0x945b,0x9458,0x9457,0x9454,0x9452,0x944f,0x944c,0x944a,0x9447,0x9445,0x9442,0x9441,0x943d,
	0x943a,0x9439,0x9436,0x9434,0x9431,0x942e,0x942c,0x9429,0x9427,0x9424,0x9423,0x941f,0x941c,0x941b,0x9417,0x9416,
	0x9413,0x9410,0x940e,0x940b,0x9409,0x9406,0x9404,0x9401,0x93fe,0x93fd,0x93f9,0x93f8,0x93f5,0x93f1,0x93f0,0x93ed,
	0x93eb,0x93e9,0x93e6,0x93e4,0x93e1,0x93de,0x93dc,0x93d9,0x93d8,0x93d5,0x93d3,0x93d0,0x93cd,0x93cb,0x93c8,0x93c6,
	0x93c3,0x93c0,0x93be,0x93bb,0x93ba,0x93b7,0x93b5,0x93b2,0x93af,0x93ad,0x93aa,0x93a8,0x93a5,0x93a4,0x93a1,0x939d,
	0x939c,0x9399,0x9397,0x9394,0x9392,0x938f,0x938c,0x938b,0x9387,0x9386,0x9383,0x9381,0x937e,0x937b,0x9379,0x9376,
	0x9374,0x9371,0x9370,0x936d,0x9369,0x9368,0x9365,0x9363,0x9360,0x935e,0x935b,0x9358,0x9357,0x9353,0x9352,0x934f,
	0x934d,0x934a,0x9347,0x9345,0x9342,0x9341,0x933d,0x933a,0x9339,0x9336,0x9334,0x9331,0x932f,0x932c,0x9329,0x9327,
	0x9324,0x9323,0x931f,0x931e,0x931b,0x9318,0x9316,0x9313,0x9311,0x930e,0x930d,0x9309,0x9306,0x9305,0x9302,0x9300,
	0x92fd,0x92fb,0x92f8,0x92f5,0x92f3,0x92f0,0x92ef,0x92ec,0x92ea,0x92e7,0x92e4,0x92e2,0x92df,0x92dd,0x92da,0x92d9,
	0x92d5,0x92d2,0x92d1,0x92ce,0x92cc,0x92c9,0x92c6,0x92c4,0x92c1,0x92bf,0x92bc,0x92bb,0x92b8,0x92b4,0x92b3,0x92b0,
	0x92ae,0x92ab,0x92a8,0x92a6,0x92a3,0x92a2,0x929f,0x929d,0x929a,0x9297,0x9295,0x9292,0x9290,0x928d,0x928c,0x9289,
	0x9285,0x9284,0x9281,0x927f,0x927c,0x927a,0x9277,0x9274,0x9273,0x926f,0x926e,0x926b,0x9269,0x9266,0x9263,0x9261,
	0x925e,0x925d,0x925a,0x9258,0x9255,0x9252,0x9250,0x924d,0x924b,0x9248,0x9247,0x9244,0x9240,0x923f,0x923c,0x923a,
	0x9237,0x9235,0x9232,0x922f,0x922e,0x922a,0x9229,0x9226,0x9224,0x9221,0x921e,0x921c,0x9219,0x9218,0x9214,0x9213,
	0x9210,0x920d,0x920b,0x9208,0x9206,0x9203,0x9202,0x91fe,0x91fb,0x91fa,0x91f7,0x91f5,0x91f2,0x91f0,0x91ed,0x91ea,
	0x91e8,0x91e5,0x91e4,0x91e1,0x91df,0x91dc,0x91d9,0x91d7,0x91d4,0x91d2,0x91cf,0x91ce,0x91cb,0x91c7,0x91c6,0x91c3,
	0x91c1,0x91be,0x91bc,0x91b9,0x91b6,0x91b5,0x91b1,0x91b0,0x91ad,0x91ab,0x91a8,0x91a5,0x91a3,0x91a0,0x919f,0x919c,
	0x919a,0x9197,0x9194,0x9192,0x918f,0x918d,0x918a,0x9189,0x9186,0x9182,0x9181,0x917e,0x917c,0x9179,0x9177,0x9174,
	0x9172,0x916f,0x916e,0x916b,0x9167,0x9166,0x9163,0x9161,0x915e,0x915c,0x9159,0x9158,0x9155,0x9151,0x9150,0x914d,
	0x914b,0x9148,0x9146,0x9143,0x9140,0x913f,0x913c,0x913a,0x9137,0x9135,0x9132,0x912f,0x912d,0x912a,0x9129,0x9126,
	0x9124,0x9121,0x911e,0x911c,0x9119,0x9117,0x9114,0x9113,0x9110,0x910e,0x910b,0x9108,0x9106,0x9103,0x9102,0x90fe,
	0x90fd,0x90fa,0x90f7,0x90f5,0x90f2,0x90f0,0x90ed,0x90ec,0x90e9,0x90e5,0x90e4,0x90e1,0x90df,0x90dc,0x90da,0x90d7,
	0x90d6,0x90d3,0x90cf,0x90ce,0x90cb,0x90c9,0x90c6,0x90c4,0x90c1,0x90be,0x90bd,0x90ba,0x90b8,0x90b5,0x90b3,0x90b0,
	0x90ad,0x90ab,0x90a8,0x90a7,0x90a4,0x90a2,0x909f,0x909c,0x909a,0x9097,0x9095,0x9092,0x9091,0x908e,0x908c,0x9089,
	0x9086,0x9084,0x9081,0x9080,0x907c,0x907b,0x9078,0x9075,0x9073,0x9070,0x906e,0x906b,0x906a,0x9066,0x9063,0x9062,
	0x905f,0x905d,0x905a,0x9058,0x9055,0x9052,0x9051,0x904d,0x904c,0x9049,0x9047,0x9044,0x9042,0x903f,0x903c,0x903b,
	0x9038,0x9035,0x9034,0x9030,0x902f,0x902c,0x902a,0x9027,0x9026,0x9022,0x901f,0x901e,0x901b,0x9019,0x9016,0x9014,
	0x9011,0x900e,0x900d,0x9009,0x9008,0x9005,0x9003,0x9000,0x8ffe,0x8ffb,0x8ff8,0x8ff7,0x8ff4,0x8ff2,0x8fef,0x8fed,
	0x8fea,0x8fe7,0x8fe5,0x8fe2,0x8fe1,0x8fde,0x8fdc,0x8fd9,0x8fd7,0x8fd4,0x8fd1,0x8fd0,0x8fcc,0x8fcb,0x8fc8,0x8fc6,
	0x8fc3,0x8fc0,0x8fbe,0x8fbb,0x8fba,0x8fb7,0x8fb5,0x8fb2,0x8fb0,0x8fad,0x8faa,0x8fa9,0x8fa5,0x8fa4,0x8fa1,0x8f9f,
	0x8f9c,0x8f99,0x8f97,0x8f94,0x8f93,0x8f90,0x8f8e,0x8f8b,0x8f89,0x8f86,0x8f83,0x8f81,0x8f7e,0x8f7d,0x8f7a,0x8f78,
	0x8f75,0x8f72,0x8f70,0x8f6d,0x8f6c,0x8f68,0x8f67,0x8f64,0x8f62,0x8f5f,0x8f5c,0x8f5a,0x8f57,0x8f56,0x8f53,0x8f51,
	0x8f4e,0x8f4b,0x8f49,0x8f46,0x8f45,0x8f41,0x8f40,0x8f3d,0x8f3b,0x8f38,0x8f35,0x8f33,0x8f30,0x8f2f,0x8f2c,0x8f2a,
	0x8f27,0x8f24,0x8f22,0x8f1f,0x8f1d,0x8f1a,0x8f19,0x8f16,0x8f14,0x8f11,0x8f0e,0x8f0c,0x8f09,0x8f08,0x8f04,0x8f03,
	0x8f01,0x8efd,0x8efa,0x8ef9,0x8ef6,0x8ef4,0x8ef1,0x8eef,0x8eec,0x8eeb,0x8ee8,0x8ee5,0x8ee3,0x8ee0,0x8ede,0x8edb,
	0x8eda,0x8ed7,0x8ed5,0x8ed2,0x8ecf,0x8ecd,0x8eca,0x8ec9,0x8ec5,0x8ec4,0x8ec1,0x8ebf,0x8ebc,0x8eb9,0x8eb7,0x8eb4,
	0x8eb3,0x8eb0,0x8eae,0x8eab,0x8ea9,0x8ea6,0x8ea3,0x8ea2,0x8e9f,0x8e9d,0x8e9a,0x8e98,0x8e95,0x8e94,0x8e91,0x8e8d,
	0x8e8c,0x8e89,0x8e87,0x8e84,0x8e83,0x8e7f,0x8e7e,0x8e7b,0x8e78,0x8e76,0x8e73,0x8e71,0x8e6e,0x8e6d,0x8e6a,0x8e68,
	0x8e65,0x8e62,0x8e60,0x8e5d,0x8e5c,0x8e59,0x8e57,0x8e54,0x8e52,0x8e4f,0x8e4c,0x8e4b,0x8e48,0x8e46,0x8e43,0x8e41,
	0x8e3e,0x8e3d,0x8e3a,0x8e36,0x8e35,0x8e32,0x8e30,0x8e2d,0x8e2c,0x8e28,0x8e27,0x8e24,0x8e21,0x8e1f,0x8e1c,0x8e1a,
	0x8e17,0x8e16,0x8e13,0x8e11,0x8e0e,0x8e0b,0x8e09,0x8e06,0x8e05,0x8e02,0x8e00,0x8dfd,0x8dfb,0x8df8,0x8df5,0x8df4,
	0x8df0,0x8def,0x8dec,0x8dea,0x8de7,0x8de6,0x8de2,0x8ddf,0x8dde,0x8ddb,0x8dd9,0x8dd6,0x8dd4,0x8dd1,0x8dd0,0x8dcd,
	0x8dca,0x8dc8,0x8dc5,0x8dc3,0x8dc0,0x8dbd,0x8dbc,0x8db9,0x8db7,0x8db4,0x8db2,0x8daf,0x8dae,0x8dab,0x8da9,0x8da6,
	0x8da3,0x8da1,0x8d9e,0x8d9d,0x8d9a,0x8d98,0x8d95,0x8d93,0x8d90,0x8d8d,0x8d8c,0x8d88,0x8d87,0x8d84,0x8d82,0x8d7f,
	0x8d7e,0x8d7b,0x8d79,0x8d76,0x8d73,0x8d71,0x8d6e,0x8d6d,0x8d69,0x8d68,0x8d65,0x8d63,0x8d60,0x8d5d,0x8d5b,0x8d58,
	0x8d57,0x8d54,0x8d52,0x8d4f,0x8d4e,0x8d4a,0x8d49,0x8d46,0x8d43,0x8d41,0x8d3e,0x8d3c,0x8d39,0x8d38,0x8d35,0x8d33,
	0x8d30,0x8d2d,0x8d2b,0x8d28,0x8d27,0x8d24,0x8d22,0x8d1f,0x8d1d,0x8d1a,0x8d17,0x8d16,0x8d13,0x8d11,0x8d0e,0x8d0c,
	0x8d09,0x8d08,0x8d05,0x8d03,0x8d00,0x8cfd,0x8cfb,0x8cf8,0x8cf7,0x8cf4,0x8cf2,0x8cef,0x8ced,0x8cea,0x8ce7,0x8ce6,
	0x8ce3,0x8ce1,0x8cde,0x8cdc,0x8cd9,0x8cd8,0x8cd5,0x8cd3,0x8cd0,0x8ccd,0x8ccb,0x8cc8,0x8cc7,0x8cc4,0x8cc2,0x8cbf,
	0x8cbd,0x8cba,0x8cb7,0x8cb6,0x8cb2,0x8cb1,0x8cae,0x8cac,0x8ca9,0x8ca8,0x8ca5,0x8ca3,0x8ca0,0x8c9d,0x8c9b,0x8c98,
	0x8c96,0x8c93,0x8c91,0x8c8e,0x8c8d,0x8c8a,0x8c88,0x8c85,0x8c83,0x8c80,0x8c7f,0x8c7c,0x8c79,0x8c77,0x8c74,0x8c72,
	0x8c6f,0x8c6e,0x8c6b,0x8c69,0x8c66,0x8c65,0x8c61,0x8c5e,0x8c5d,0x8c5a,0x8c58,0x8c55,0x8c54,0x8c50,0x8c4f,0x8c4c,
	0x8c4a,0x8c47,0x8c44,0x8c43,0x8c3f,0x8c3e,0x8c3b,0x8c39,0x8c36,0x8c35,0x8c32,0x8c2e,0x8c2d,0x8c2a,0x8c28,0x8c25,
	0x8c24,0x8c20,0x8c1f,0x8c1c,0x8c1a,0x8c17,0x8c14,0x8c13,0x8c0f,0x8c0e,0x8c0b,0x8c09,0x8c06,0x8c05,0x8c02,0x8c00,
	0x8bfd,0x8bfa,0x8bf8,0x8bf5,0x8bf4,0x8bf1,0x8bef,0x8bec,0x8bea,0x8be7,0x8be6,0x8be3,0x8be0,0x8bde,0x8bdb,0x8bd9,
	0x8bd6,0x8bd5,0x8bd2,0x8bd0,0x8bcd,0x8bcb,0x8bc8,0x8bc5,0x8bc4,0x8bc1,0x8bbf,0x8bbc,0x8bba,0x8bb7,0x8bb6,0x8bb3,
	0x8bb1,0x8bae,0x8bab,0x8ba9,0x8ba6,0x8ba5,0x8ba2,0x8ba0,0x8b9d,0x8b9b,0x8b98,0x8b95,0x8b94,0x8b91,0x8b8f,0x8b8c,
	0x8b8a,0x8b87,0x8b86,0x8b83,0x8b81,0x8b7e,0x8b7b,0x8b79,0x8b76,0x8b75,0x8b72,0x8b70,0x8b6d,0x8b6c,0x8b68,0x8b67,
	0x8b64,0x8b61,0x8b60,0x8b5d,0x8b5b,0x8b58,0x8b55,0x8b53,0x8b50,0x8b4f,0x8b4c,0x8b4a,0x8b47,0x8b45,0x8b42,0x8b41,
	0x8b3e,0x8b3c,0x8b39,0x8b36,0x8b34,0x8b31,0x8b30,0x8b2d,0x8b2b,0x8b28,0x8b27,0x8b23,0x8b22,0x8b1f,0x8b1c,0x8b1a,
	0x8b17,0x8b16,0x8b13,0x8b11,0x8b0e,0x8b0c,0x8b09,0x8b08,0x8b05,0x8b02,0x8b00,0x8afd,0x8afb,0x8af8,0x8af7,0x8af4,
	0x8af2,0x8aef,0x8aed,0x8aea,0x8ae9,0x8ae6,0x8ae3,0x8ae1,0x8ade,0x8add,0x8ad9,0x8ad8,0x8ad5,0x8ad3,0x8ad0,0x8acf,
	0x8acc,0x8ac8,0x8ac7,0x8ac4,0x8ac2,0x8abf,0x8abe,0x8abb,0x8ab9,0x8ab6,0x8ab4,0x8ab1,0x8aae,0x8aad,0x8aaa,0x8aa8,
	0x8aa5,0x8aa3,0x8aa0,0x8a9f,0x8a9c,0x8a9a,0x8a97,0x8a96,0x8a92,0x8a8f,0x8a8e,0x8a8b,0x8a89,0x8a86,0x8a85,0x8a81,
	0x8a80,0x8a7d,0x8a7b,0x8a78,0x8a75,0x8a74,0x8a71,0x8a6f,0x8a6c,0x8a6a,0x8a67,0x8a66,0x8a63,0x8a61,0x8a5e,0x8a5b,
	0x8a59,0x8a56,0x8a55,0x8a52,0x8a50,0x8a4d,0x8a4b,0x8a48,0x8a47,0x8a44,0x8a42,0x8a3f,0x8a3c,0x8a3b,0x8a37,0x8a36,
	0x8a34,0x8a31,0x8a2e,0x8a2d,0x8a2a,0x8a28,0x8a25,0x8a23,0x8a20,0x8a1f,0x8a1c,0x8a1a,0x8a17,0x8a14,0x8a12,0x8a0f,
	0x8a0e,0x8a0b,0x8a09,0x8a06,0x8a05,0x8a02,0x8a00,0x89fd,0x89fb,0x89f8,0x89f5,0x89f4,0x89f1,0x89ef,0x89ec,0x89ea,
	0x89e7,0x89e6,0x89e3,0x89e1,0x89de,0x89dd,0x89da,0x89d6,0x89d5,0x89d2,0x89d0,0x89cd,0x89cc,0x89c9,0x89c7,0x89c4,
	0x89c2,0x89bf,0x89be,0x89bb,0x89b8,0x89b6,0x89b3,0x89b2,0x89ae,0x89ad,0x89aa,0x89a8,0x89a5,0x89a4,0x89a1,0x899f,
	0x899c,0x8999,0x8997,0x8994,0x8993,0x8990,0x898e,0x898b,0x8989,0x8986,0x8985,0x8982,0x8980,0x897d,0x897a,0x8979,
	0x8975,0x8974,0x8971,0x896f,0x896c,0x896b,0x8968,0x8966,0x8963,0x8961,0x895e,0x895b,0x895a,0x8957,0x8955,0x8952,
	0x8951,0x894d,0x894c,0x8949,0x8947,0x8944,0x8943,0x8940,0x893d,0x893b,0x8938,0x8936,0x8933,0x8932,0x892f,0x892d,
	0x892a,0x8929,0x8925,0x8924,0x8921,0x891e,0x891c,0x8919,0x8918,0x8915,0x8913,0x8910,0x890e,0x890b,0x890a,0x8907,
	0x8904,0x8901,0x8900,0x88fd,0x88fb,0x88f8,0x88f7,0x88f3,0x88f2,0x88ef,0x88ed,0x88ea,0x88e9,0x88e6,0x88e4,0x88e1,
	0x88e0,0x88dd,0x88d9,0x88d8,0x88d5,0x88d3,0x88d0,0x88cf,0x88cc,0x88ca,0x88c7,0x88c6,0x88c2,0x88c1,0x88be,0x88bc,
	0x88b9,0x88b6,0x88b5,0x88b2,0x88b0,0x88ad,0x88ac,0x88a8,0x88a7,0x88a4,0x88a2,0x889f,0x889e,0x889b,0x8899,0x8896,
	0x8895,0x8891,0x888e,0x888d,0x888a,0x8888,0x8885,0x8884,0x8881,0x887f,0x887c,0x887b,0x8877,0x8876,0x8873,0x8871,
	0x886e,0x886b,0x886a,0x8867,0x8865,0x8862,0x8860,0x885d,0x885c,0x8859,0x8857,0x8854,0x8853,0x8850,0x884e,0x884b,
	0x884a,0x8846,0x8843,0x8842,0x883f,0x883d,0x883a,0x8839,0x8836,0x8834,0x8831,0x882f,0x882c,0x882b,0x8828,0x8826,
	0x8823,0x8820,0x881f,0x881c,0x881a,0x8817,0x8815,0x8812,0x8811,0x880e,0x880c,0x8809,0x8808,0x8805,0x8803,0x8800,
	0x87fe,0x87fb,0x87f8,0x87f7,0x87f4,0x87f2,0x87ef,0x87ee,0x87eb,0x87e9,0x87e6,0x87e4,0x87e1,0x87e0,0x87dd,0x87db,
	0x87d7,0x87d6,0x87d3,0x87d1,0x87ce,0x87cd,0x87ca,0x87c8,0x87c5,0x87c4,0x87c0,0x87bd,0x87bc,0x87b9,0x87b7,0x87b4,
	0x87b3,0x87b0,0x87ae,0x87ab,0x87aa,0x87a6,0x87a5,0x87a2,0x87a0,0x879d,0x879c,0x8799,0x8796,0x8794,0x8791,0x8790,
	0x878d,0x878b,0x8788,0x8786,0x8783,0x8782,0x877f,0x877d,0x877a,0x8779,0x8776,0x8774,0x8771,0x876f,0x876c,0x8769,
	0x8768,0x8765,0x8763,0x8760,0x875f,0x875c,0x875a,0x8757,0x8756,0x8752,0x8751,0x874e,0x874c,0x8749,0x8748,0x8745,
	0x8743,0x8740,0x873d,0x873c,0x8739,0x8737,0x8734,0x8732,0x872f,0x872e,0x872b,0x8729,0x8726,0x8725,0x8722,0x8720,
	0x871d,0x871b,0x8718,0x8715,0x8714,0x8711,0x870f,0x870c,0x870b,0x8708,0x8706,0x8703,0x8702,0x86fe,0x86fd,0x86fa,
	0x86f8,0x86f5,0x86f4,0x86f1,0x86ef,0x86ec,0x86e9,0x86e8,0x86e5,0x86e3,0x86e0,0x86de,0x86db,0x86da,0x86d7,0x86d5,
	0x86d2,0x86d1,0x86ce,0x86cc,0x86c9,0x86c7,0x86c4,0x86c1,0x86c0,0x86bd,0x86bb,0x86b8,0x86b7,0x86b4,0x86b2,0x86af,
	0x86ae,0x86ab,0x86a8,0x86a6,0x86a3,0x86a1,0x869e,0x869d,0x869a,0x8698,0x8695,0x8694,0x8691,0x868f,0x868c,0x868b,
	0x8688,0x8686,0x8683,0x8681,0x867e,0x867d,0x867a,0x8677,0x8675,0x8672,0x8671,0x866e,0x866c,0x8669,0x8668,0x8665,
	0x8663,0x8660,0x865e,0x865b,0x865a,0x8657,0x8655,0x8652,0x8651,0x864e,0x864c,0x8649,0x8646,0x8645,0x8642,0x8640,
	0x863d,0x863b,0x8638,0x8637,0x8634,0x8632,0x862f,0x862e,0x862b,0x8629,0x8626,0x8625,0x8622,0x8620,0x861d,0x861b,
	0x8618,0x8615,0x8614,0x8611,0x860f,0x860c,0x860b,0x8608,0x8606,0x8603,0x8602,0x85fe,0x85fd,0x85fa,0x85f8,0x85f5,
	0x85f4,0x85f1,0x85ef,0x85ec,0x85eb,0x85e8,0x85e5,0x85e3,0x85e0,0x85de,0x85db,0x85da,0x85d7,0x85d5,0x85d2,0x85d1,
	0x85ce,0x85cc,0x85c9,0x85c8,0x85c5,0x85c3,0x85c0,0x85be,0x85bb,0x85ba,0x85b7,0x85b4,0x85b2,0x85af,0x85ae,0x85ab,
	0x85a9,0x85a6,0x85a5,0x85a2,0x85a0,0x859d,0x859b,0x8598,0x8597,0x8594,0x8592,0x858f,0x858e,0x858b,0x8589,0x8586,
	0x8584,0x8581,0x8580,0x857d,0x857b,0x8578,0x8577,0x8573,0x8572,0x856f,0x856d,0x856a,0x8569,0x8566,0x8563,0x8561,
	0x855e,0x855d,0x855a,0x8558,0x8555,0x8554,0x8551,0x854f,0x854c,0x854a,0x8547,0x8546,0x8543,0x8541,0x853e,0x853d,
	0x853a,0x8538,0x8535,0x8534,0x8531,0x852f,0x852c,0x8529,0x8528,0x8524,0x8523,0x8520,0x851e,0x851b,0x851a,0x8517,
	0x8515,0x8512,0x8511,0x850e,0x850c,0x8509,0x8508,0x8505,0x8503,0x8500,0x84fe,0x84fb,0x84fa,0x84f7,0x84f5,0x84f2,
	0x84f1,0x84ee,0x84eb,0x84e9,0x84e6,0x84e5,0x84e2,0x84e0,0x84dd,0x84dc,0x84d8,0x84d7,0x84d4,0x84d2,0x84cf,0x84ce,
	0x84cb,0x84c9,0x84c6,0x84c5,0x84c2,0x84c0,0x84bd,0x84bc,0x84b9,0x84b7,0x84b4,0x84b1,0x84af,0x84ac,0x84ab,0x84a8,
	0x84a6,0x84a3,0x84a2,0x849f,0x849d,0x849a,0x8499,0x8496,0x8494,0x8491,0x8490,0x848d,0x848b,0x8488,0x8486,0x8483,
	0x8482,0x847f,0x847d,0x847a,0x8477,0x8476,0x8473,0x8471,0x846e,0x846d,0x846a,0x8468,0x8465,0x8463,0x8460,0x845f,
	0x845c,0x8459,0x8458,0x8455,0x8453,0x8450,0x844f,0x844c,0x844a,0x8447,0x8446,0x8443,0x8441,0x843e,0x843d,0x843a,
	0x8438,0x8435,0x8434,0x8431,0x842f,0x842c,0x842a,0x8427,0x8426,0x8423,0x8421,0x841e,0x841d,0x841a,0x8418,0x8415,
	0x8414,0x8411,0x840e,0x840c,0x8409,0x8408,0x8405,0x8403,0x8400,0x83fe,0x83fb,0x83fa,0x83f7,0x83f5,0x83f2,0x83f1,
	0x83ee,0x83ec,0x83e9,0x83e8,0x83e5,0x83e3,0x83e0,0x83df,0x83dc,0x83da,0x83d7,0x83d6,0x83d3,0x83d1,0x83ce,0x83cc,
	0x83c9,0x83c6,0x83c5,0x83c2,0x83c0,0x83bd,0x83bc,0x83b9,0x83b7,0x83b4,0x83b3,0x83b0,0x83ae,0x83ab,0x83aa,0x83a7,
	0x83a5,0x83a2,0x83a1,0x839d,0x839c,0x8399,0x8397,0x8394,0x8393,0x8390,0x838e,0x838b,0x838a,0x8387,0x8385,0x8382,
	0x8381,0x837e,0x837b,0x8379,0x8376,0x8375,0x8372,0x8370,0x836d,0x836b,0x8368,0x8367,0x8364,0x8362,0x835f,0x835e,
	0x835b,0x8359,0x8356,0x8355,0x8352,0x8350,0x834d,0x834c,0x8349,0x8347,0x8344,0x8343,0x8340,0x833e,0x833b,0x8339,
	0x8336,0x8335,0x8332,0x8330,0x832d,0x832c,0x8329,0x8327,0x8324,0x8323,0x8320,0x831e,0x831b,0x831a,0x8317,0x8315,
	0x8312,0x8311,0x830e,0x830c,0x8309,0x8308,0x8305,0x8302,0x8300,0x82fd,0x82fb,0x82f8,0x82f7,0x82f4,0x82f2,0x82ef,
	0x82ee,0x82eb,0x82e9,0x82e6,0x82e5,0x82e2,0x82e0,0x82dd,0x82dc,0x82d9,0x82d7,0x82d4,0x82d3,0x82d0,0x82ce,0x82cb,
	0x82ca,0x82c7,0x82c5,0x82c2,0x82c1,0x82bd,0x82bc,0x82b9,0x82b7,0x82b4,0x82b3,0x82b0,0x82ae,0x82ab,0x82aa,0x82a7,
	0x82a5,0x82a2,0x829f,0x829e,0x829b,0x8299,0x8296,0x8295,0x8292,0x8290,0x828d,0x828c,0x8289,0x8287,0x8284,0x8283,
	0x8280,0x827e,0x827b,0x8279,0x8276,0x8275,0x8272,0x8270,0x826d,0x826c,0x8269,0x8267,0x8264,0x8263,0x8260,0x825e,
	0x825b,0x825a,0x8257,0x8255,0x8252,0x8251,0x824e,0x824c,0x8249,0x8248,0x8245,0x8242,0x8240,0x823d,0x823b,0x8238,
	0x8237,0x8234,0x8232,0x822f,0x822e,0x822b,0x8229,0x8226,0x8225,0x8222,0x8220,0x821d,0x821c,0x8219,0x8217,0x8214,
	0x8214,0x8211,0x820e,0x820c,0x8209,0x8208,0x8205,0x8203,0x8200,0x81fe,0x81fb,0x81fa,0x81f7,0x81f5,0x81f2,0x81f1,
	0x81ee,0x81ec,0x81e9,0x81e8,0x81e5,0x81e3,0x81e0,0x81df,0x81dc,0x81da,0x81d7,0x81d6,0x81d3,0x81d1,0x81ce,0x81cd,
	0x81ca,0x81c8,0x81c5,0x81c4,0x81c1,0x81bf,0x81bc,0x81bb,0x81b8,0x81b6,0x81b3,0x81b2,0x81af,0x81ad,0x81aa,0x81a9,
	0x81a6,0x81a4,0x81a1,0x81a0,0x819c,0x819b,0x8198,0x8196,0x8193,0x8192,0x818f,0x818d,0x818a,0x8189,0x8186,0x8184,
	0x8181,0x817e,0x817d,0x817a,0x8178,0x8175,0x8174,0x8171,0x816f,0x816c,0x816b,0x8168,0x8166,0x8163,0x8162,0x815f,
	0x815d,0x815a,0x8159,0x8156,0x8154,0x8151,0x8150,0x814d,0x814b,0x8148,0x8147,0x8144,0x8142,0x813f,0x813d,0x813a,
	0x8139,0x8136,0x8134,0x8131,0x8130,0x812d,0x812b,0x8128,0x8127,0x8124,0x8122,0x811f,0x811e,0x811b,0x8119,0x8116,
	0x8115,0x8112,0x8110,0x810d,0x810c,0x8109,0x8107,0x8104,0x8103,0x8100,0x80fe,0x80fb,0x80fa,0x80f7,0x80f5,0x80f2,
	0x80f0,0x80ef,0x80ec,0x80ea,0x80e7,0x80e6,0x80e3,0x80e1,0x80de,0x80dd,0x80da,0x80d8,0x80d5,0x80d4,0x80d1,0x80cf,
	0x80cc,0x80cb,0x80c8,0x80c6,0x80c3,0x80c2,0x80bf,0x80bd,0x80ba,0x80b9,0x80b6,0x80b4,0x80b1,0x80b0,0x80ad,0x80ab,
	0x80a8,0x80a7,0x80a4,0x80a2,0x809f,0x809e,0x809b,0x8099,0x8096,0x8095,0x8092,0x8090,0x808d,0x808c,0x8089,0x8087,
	0x8084,0x8083,0x8080,0x807e,0x807b,0x807a,0x8077,0x8075,0x8072,0x8071,0x806e,0x806c,0x8069,0x8068,0x8065,0x8063,
	0x8060,0x805f,0x805c,0x805a,0x8057,0x8056,0x8053,0x8051,0x804e,0x804d,0x804a,0x8048,0x8045,0x8044,0x8041,0x803f,
	0x803c,0x803b,0x8038,0x8036,0x8033,0x8032,0x802f,0x802d,0x802a,0x8029,0x8026,0x8024,0x8021,0x8020,0x801d,0x801b,
	0x8018,0x8017,0x8014,0x8012,0x800f,0x800e,0x800b,0x8009,0x8006,0x8005,0x8002,0x8000,0x7ffd,0x7ffc,0x7ff9,0x7ff7,
	0x7ff4,0x7ff3,0x7ff0,0x7fee,0x7feb,0x7fea,0x7fe7,0x7fe5,0x7fe2,0x7fe1,0x7fde,0x7fdc,0x7fd9,0x7fd8,0x7fd5,0x7fd3,
	0x7fd0,0x7fcd,0x7fcb,0x7fc8,0x7fc7,0x7fc4,0x7fc2,0x7fc0,0x7fbe,0x7fbb,0x7fba,0x7fb7,0x7fb5,0x7fb2,0x7fb1,0x7fae,
	0x7fac,0x7fa9,0x7fa8,0x7fa5,0x7fa3,0x7fa0,0x7f9f,0x7f9c,0x7f9a,0x7f97,0x7f96,0x7f93,0x7f91,0x7f8e,0x7f8d,0x7f8a,
	0x7f88,0x7f85,0x7f84,0x7f81,0x7f7f,0x7f7c,0x7f7b,0x7f78,0x7f76,0x7f73,0x7f72,0x7f6f,0x7f6d,0x7f6a,0x7f69,0x7f66,
	0x7f64,0x7f61,0x7f60,0x7f5d,0x7f5b,0x7f58,0x7f57,0x7f54,0x7f52,0x7f4f,0x7f4e,0x7f4b,0x7f49,0x7f46,0x7f45,0x7f42,
	0x7f40,0x7f3f,0x7f3c,0x7f3a,0x7f37,0x7f36,0x7f33,0x7f31,0x7f2e,0x7f2d,0x7f2a,0x7f28,0x7f25,0x7f24,0x7f21,0x7f1f,
	0x7f1c,0x7f1b,0x7f18,0x7f16,0x7f13,0x7f12,0x7f0f,0x7f0d,0x7f0a,0x7f09,0x7f06,0x7f04,0x7f01,0x7f00,0x7efd,0x7efc,
	0x7ef9,0x7ef7,0x7ef4,0x7ef3,0x7ef0,0x7eee,0x7eeb,0x7eea,0x7ee7,0x7ee5,0x7ee2,0x7ee1,0x7ede,0x7edc,0x7ed9,0x7ed8,
	0x7ed5,0x7ed3,0x7ed0,0x7ecf,0x7ecc,0x7eca,0x7ec7,0x7ec6,0x7ec3,0x7ec1,0x7ebe,0x7ebd,0x7eba,0x7eb8,0x7eb5,0x7eb4,
	0x7eb2,0x7eaf,0x7ead,0x7eaa,0x7ea9,0x7ea6,0x7ea4,0x7ea1,0x7ea0,0x7e9d,0x7e9b,0x7e98,0x7e97,0x7e94,0x7e92,0x7e8f,
	0x7e8e,0x7e8b,0x7e89,0x7e86,0x7e85,0x7e82,0x7e80,0x7e7d,0x7e7c,0x7e79,0x7e77,0x7e74,0x7e73,0x7e70,0x7e6e,0x7e6b,
	0x7e6a,0x7e67,0x7e65,0x7e62,0x7e61,0x7e5e,0x7e5d,0x7e5a,0x7e58,0x7e55,0x7e54,0x7e51,0x7e4f,0x7e4c,0x7e4b,0x7e48,
	0x7e46,0x7e43,0x7e42,0x7e3f,0x7e3d,0x7e3a,0x7e39,0x7e36,0x7e34,0x7e31,0x7e30,0x7e2d,0x7e2b,0x7e28,0x7e27,0x7e24,
	0x7e22,0x7e21,0x7e1e,0x7e1c,0x7e19,0x7e18,0x7e15,0x7e13,0x7e10,0x7e0f,0x7e0c,0x7e0a,0x7e07,0x7e06,0x7e03,0x7e01,
	0x7dff,0x7dfd,0x7dfa,0x7df9,0x7df6,0x7df4,0x7df1,0x7df0,0x7ded,0x7deb,0x7de8,0x7de7,0x7de4,0x7de2,0x7ddf,0x7dde,
	0x7ddb,0x7dd9,0x7dd6,0x7dd5,0x7dd2,0x7dd0,0x7dcd,0x7dcc,0x7dc9,0x7dc7,0x7dc4,0x7dc3,0x7dc0,0x7dbe,0x7dbb,0x7dba,
	0x7db7,0x7db5,0x7db2,0x7db1,0x7dae,0x7dac,0x7da9,0x7da8,0x7da5,0x7da3,0x7da1,0x7d9f,0x7d9c,0x7d9b,0x7d98,0x7d96,
	0x7d93,0x7d92,0x7d8f,0x7d8d,0x7d8a,0x7d89,0x7d86,0x7d84,0x7d81,0x7d80,0x7d7d,0x7d7c,0x7d79,0x7d77,0x7d74,0x7d73,
	0x7d70,0x7d6e,0x7d6b,0x7d6a,0x7d67,0x7d65,0x7d64,0x7d61,0x7d5f,0x7d5c,0x7d5b,0x7d58,0x7d56,0x7d53,0x7d52,0x7d4f,
	0x7d4d,0x7d4a,0x7d49,0x7d46,0x7d44,0x7d41,0x7d40,0x7d3d,0x7d3c,0x7d39,0x7d37,0x7d34,0x7d33,0x7d30,0x7d2e,0x7d2b,
	0x7d2a,0x7d27,0x7d25,0x7d22,0x7d21,0x7d1e,0x7d1c,0x7d19,0x7d18,0x7d15,0x7d13,0x7d10,0x7d0f,0x7d0c,0x7d0a,0x7d07,
	0x7d06,0x7d04,0x7d01,0x7d00,0x7cfd,0x7cfc,0x7cf9,0x7cf7,0x7cf4,0x7cf3,0x7cf0,0x7cee,0x7ceb,0x7cea,0x7ce7,0x7ce5,
	0x7ce2,0x7ce1,0x7cde,0x7cdc,0x7cd9,0x7cd8,0x7cd5,0x7cd3,0x7cd0,0x7ccf,0x7ccc,0x7cca,0x7cc7,0x7cc6,0x7cc3,0x7cc1,
	0x7cbf,0x7cbd,0x7cba,0x7cb9,0x7cb6,0x7cb4,0x7cb1,0x7cb0,0x7cad,0x7cab,0x7ca8,0x7ca7,0x7ca5,0x7ca2,0x7ca1,0x7c9e,
	0x7c9c,0x7c99,0x7c98,0x7c95,0x7c93,0x7c90,0x7c8f,0x7c8c,0x7c8a,0x7c87,0x7c86,0x7c83,0x7c81,0x7c7f,0x7c7d,0x7c7a,
	0x7c78,0x7c77,0x7c74,0x7c72,0x7c6f,0x7c6e,0x7c6b,0x7c69,0x7c66,0x7c65,0x7c62,0x7c60,0x7c5e,0x7c5c,0x7c59,0x7c58,
	0x7c55,0x7c53,0x7c50,0x7c4f,0x7c4c,0x7c4a,0x7c47,0x7c46,0x7c43,0x7c41,0x7c3e,0x7c3d,0x7c3a,0x7c38,0x7c35,0x7c34,
	0x7c32,0x7c30,0x7c2e,0x7c2b,0x7c2a,0x7c27,0x7c25,0x7c22,0x7c21,0x7c1e,0x7c1c,0x7c19,0x7c18,0x7c15,0x7c13,0x7c10,
	0x7c0f,0x7c0c,0x7c0a,0x7c07,0x7c06,0x7c03,0x7c01,0x7bff,0x7bfd,0x7bfa,0x7bf9,0x7bf6,0x7bf4,0x7bf1,0x7bf0,0x7bed,
	0x7beb,0x7bea,0x7be7,0x7be5,0x7be2,0x7be1,0x7bde,0x7bdc,0x7bd9,0x7bd8,0x7bd5,0x7bd3,0x7bd1,0x7bcf,0x7bcc,0x7bcb,
	0x7bc8,0x7bc6,0x7bc3,0x7bc2,0x7bbf,0x7bbd,0x7bba,0x7bb9,0x7bb6,0x7bb4,0x7bb1,0x7bb0,0x7bad,0x7bab,0x7ba8,0x7ba7,
	0x7ba5,0x7ba2,0x7ba1,0x7b9e,0x7b9d,0x7b9a,0x7b98,0x7b95,0x7b94,0x7b91,0x7b8f,0x7b8c,0x7b8b,0x7b88,0x7b86,0x7b83,
	0x7b82,0x7b7f,0x7b7d,0x7b7a,0x7b79,0x7b76,0x7b74,0x7b72,0x7b70,0x7b6d,0x7b6c,0x7b69,0x7b67,0x7b64,0x7b63,0x7b60,
	0x7b5f,0x7b5c,0x7b5a,0x7b57,0x7b56,0x7b53,0x7b51,0x7b4e,0x7b4d,0x7b4a,0x7b49,0x7b46,0x7b44,0x7b43,0x7b40,0x7b3e,
	0x7b3b,0x7b3a,0x7b37,0x7b35,0x7b32,0x7b31,0x7b2e,0x7b2c,0x7b29,0x7b28,0x7b25,0x7b24,0x7b21,0x7b1f,0x7b1c,0x7b1b,
	0x7b18,0x7b16,0x7b13,0x7b12,0x7b0f,0x7b0d,0x7b0a,0x7b09,0x7b07,0x7b04,0x7b03,0x7b00,0x7aff,0x7afc,0x7afa,0x7af7,
	0x7af6,0x7af3,0x7af1,0x7aee,0x7aed,0x7aea,0x7ae8,0x7ae5,0x7ae4,0x7ae1,0x7adf,0x7adc,0x7adb,0x7ad8,0x7ad7,0x7ad4,
	0x7ad2,0x7ad1,0x7ace,0x7acc,0x7ac9,0x7ac8,0x7ac5,0x7ac3,0x7ac0,0x7abf,0x7abc,0x7aba,0x7ab7,0x7ab6,0x7ab3,0x7ab2,
	0x7aaf,0x7aad,0x7aaa,0x7aa9,0x7aa6,0x7aa4,0x7aa1,0x7aa0,0x7a9d,0x7a9b,0x7a9a,0x7a97,0x7a95,0x7a92,0x7a91,0x7a8e,
	0x7a8d,0x7a8a,0x7a88,0x7a85,0x7a84,0x7a81,0x7a7f,0x7a7c,0x7a7b,0x7a78,0x7a76,0x7a73,0x7a72,0x7a6f,0x7a6d,0x7a6a,
	0x7a69,0x7a66,0x7a65,0x7a62,0x7a60,0x7a5f,0x7a5c,0x7a5a,0x7a57,0x7a56,0x7a53,0x7a51,0x7a4e,0x7a4d,0x7a4a,0x7a48,
	0x7a47,0x7a44,0x7a42,0x7a3f,0x7a3e,0x7a3b,0x7a3a,0x7a37,0x7a35,0x7a32,0x7a31,0x7a2e,0x7a2c,0x7a29,0x7a28,0x7a25,
	0x7a23,0x7a20,0x7a1f,0x7a1c,0x7a1b,0x7a18,0x7a16,0x7a15,0x7a12,0x7a10,0x7a0d,0x7a0c,0x7a09,0x7a07,0x7a04,0x7a03,
	0x7a00,0x79ff,0x79fc,0x79fa,0x79f7,0x79f6,0x79f3,0x79f1,0x79ee,0x79ed,0x79ea,0x79e8,0x79e7,0x79e4,0x79e2,0x79e0,
	0x79de,0x79db,0x79da,0x79d7,0x79d5,0x79d2,0x79d1,0x79ce,0x79cc,0x79c9,0x79c8,0x79c5,0x79c3,0x79c1,0x79bf,0x79bc,
	0x79bb,0x79b9,0x79b6,0x79b5,0x79b2,0x79b0,0x79ad,0x79ac,0x79a9,0x79a7,0x79a4,0x79a3,0x79a0,0x799f,0x799c,0x799a,
	0x7997,0x7996,0x7993,0x7991,0x798e,0x798d,0x798b,0x7988,0x7987,0x7984,0x7982,0x7980,0x797e,0x797b,0x797a,0x7977,
	0x7975,0x7972,0x7971,0x796e,0x796c,0x7969,0x7968,0x7965,0x7963,0x7961,0x795f,0x795e,0x795b,0x7959,0x7956,0x7955,
	0x7952,0x7950,0x794d,0x794c,0x7949,0x7947,0x7944,0x7943,0x7940,0x793f,0x793c,0x793a,0x7937,0x7936,0x7933,0x7931,
	0x792f,0x792e,0x792b,0x7929,0x7926,0x7925,0x7922,0x7920,0x791d,0x791c,0x791b,0x7918,0x7916,0x7913,0x7912,0x790f,
	0x790d,0x790a,0x7909,0x7906,0x7904,0x7901,0x7900,0x78fd,0x78fc,0x78f9,0x78f7,0x78f4,0x78f3,0x78f1,0x78ee,0x78ed,
	0x78ea,0x78e8,0x78e5,0x78e4,0x78e1,0x78e0,0x78dd,0x78db,0x78d8,0x78d7,0x78d4,0x78d2,0x78cf,0x78ce,0x78cc,0x78ca,
	0x78c8,0x78c5,0x78c4,0x78c1,0x78bf,0x78bc,0x78bb,0x78b8,0x78b6,0x78b3,0x78b2,0x78af,0x78ae,0x78ab,0x78a9,0x78a6,
	0x78a5,0x78a3,0x78a0,0x789f,0x789c,0x789a,0x7897,0x7896,0x7893,0x7892,0x788f,0x788d,0x788a,0x7889,0x7886,0x7884,
	0x7881,0x7880,0x787d,0x787b,0x787a,0x7877,0x7876,0x7873,0x7871,0x786e,0x786d,0x786a,0x7868,0x7865,0x7864,0x7861,
	0x785f,0x785d,0x785b,0x7858,0x7857,0x7855,0x7852,0x7851,0x784e,0x784c,0x7849,0x7848,0x7845,0x7844,0x7841,0x783f,
	0x783c,0x783b,0x7838,0x7836,0x7833,0x7832,0x782f,0x782d,0x782c,0x7829,0x7828,0x7825,0x7823,0x7820,0x781f,0x781c,
	0x781a,0x7819,0x7816,0x7815,0x7812,0x7810,0x780d,0x780c,0x7809,0x7807,0x7804,0x7803,0x7800,0x77ff,0x77fc,0x77fa,
	0x77f9,0x77f6,0x77f4,0x77f1,0x77f0,0x77ed,0x77eb,0x77e9,0x77e7,0x77e4,0x77e3,0x77e0,0x77de,0x77db,0x77da,0x77d7,
	0x77d5,0x77d4,0x77d1,0x77d0,0x77cd,0x77cb,0x77c8,0x77c7,0x77c4,0x77c2,0x77bf,0x77be,0x77bb,0x77ba,0x77b7,0x77b5,
	0x77b4,0x77b1,0x77af,0x77ac,0x77ab,0x77a8,0x77a6,0x77a3,0x77a2,0x779f,0x779e,0x779b,0x7799,0x7796,0x7795,0x7792,
	0x7790,0x778f,0x778c,0x778b,0x7788,0x7786,0x7783,0x7782,0x777f,0x777d,0x777a,0x7779,0x7776,0x7774,0x7772,0x7770,
	0x776f,0x776c,0x776a,0x7767,0x7766,0x7763,0x7761,0x775e,0x775d,0x775a,0x7759,0x7756,0x7754,0x7751,0x7750,0x774d,
	0x774b,0x774a,0x7747,0x7745,0x7743,0x7741,0x773e,0x773d,0x773a,0x7738,0x7735,0x7734,0x7731,0x772f,0x772d,0x772b,
	0x772a,0x7727,0x7725,0x7722,0x7721,0x771e,0x771c,0x7719,0x7718,0x7715,0x7714,0x7711,0x770f,0x770c,0x770b,0x7708,
	0x7707,0x7704,0x7703,0x7700,0x76fe,0x76fb,0x76fa,0x76f7,0x76f5,0x76f3,0x76f1,0x76ee,0x76ed,0x76eb,0x76e8,0x76e7,
	0x76e4,0x76e2,0x76e0,0x76de,0x76db,0x76da,0x76d7,0x76d5,0x76d2,0x76d1,0x76cf,0x76cd,0x76cb,0x76c8,0x76c7,0x76c4,
	0x76c2,0x76bf,0x76be,0x76bb,0x76ba,0x76b7,0x76b5,0x76b4,0x76b1,0x76af,0x76ac,0x76ab,0x76a8,0x76a7,0x76a4,0x76a2,
	0x769f,0x769e,0x769b,0x7699,0x7698,0x7695,0x7694,0x7691,0x768f,0x768c,0x768b,0x7688,0x7686,0x7683,0x7682,0x767f,
	0x767e,0x767c,0x7679,0x7678,0x7675,0x7673,0x7670,0x766f,0x766c,0x766b,0x7668,0x7666,0x7663,0x7662,0x7660,0x765e,
	0x765c,0x7659,0x7658,0x7655,0x7653,0x7650,0x764f,0x764c,0x764b,0x7648,0x7646,0x7645,0x7642,0x7640,0x763d,0x763c,
	0x7639,0x7638,0x7635,0x7633,0x7630,0x762f,0x762c,0x762a,0x7629,0x7626,0x7625,0x7622,0x7620,0x761d,0x761c,0x7619,
	0x7617,0x7614,0x7613,0x7610,0x760f,0x760d,0x760a,0x7609,0x7606,0x7604,0x7601,0x7600,0x75fd,0x75fc,0x75f9,0x75f7,
	0x75f5,0x75f2,0x75f0,0x75ed,0x75ec,0x75e9,0x75e8,0x75e6,0x75e3,0x75e2,0x75df,0x75dd,0x75db,0x75d9,0x75d6,0x75d5,
	0x75d2,0x75d0,0x75cf,0x75cc,0x75cb,0x75c8,0x75c6,0x75c3,0x75c2,0x75bf,0x75bd,0x75bb,0x75b9,0x75b6,0x75b5,0x75b3,
	0x75b0,0x75af,0x75ac,0x75aa,0x75a8,0x75a6,0x75a3,0x75a2,0x759f,0x759d,0x759c,0x7599,0x7598,0x7595,0x7593,0x7590,
	0x758f,0x758c,0x758a,0x7588,0x7586,0x7585,0x7582,0x7580,0x757d,0x757c,0x7579,0x7577,0x7575,0x7573,0x7570,0x756f,
	0x756d,0x756a,0x7569,0x7566,0x7565,0x7562,0x7560,0x755d,0x755c,0x7559,0x7557,0x7555,0x7553,0x7552,0x754f,0x754d,
	0x754a,0x7549,0x7546,0x7544,0x7542,0x7540,0x753d,0x753c,0x753a,0x7537,0x7536,0x7533,0x7532,0x752f,0x752d,0x752a,
	0x7529,0x7526,0x7524,0x7523,0x7520,0x751f,0x751c,0x751a,0x7517,0x7516,0x7513,0x7511,0x750f,0x750d,0x750a,0x7509,
	0x7507,0x7504,0x7503,0x7500,0x74ff,0x74fc,0x74fa,0x74f7,0x74f6,0x74f3,0x74f1,0x74f0,0x74ed,0x74ec,0x74e9,0x74e7,
	0x74e4,0x74e3,0x74e0,0x74de,0x74db,0x74da,0x74d7,0x74d6,0x74d3,0x74d1,0x74ce,0x74cd,0x74cb,0x74c9,0x74c7,0x74c4,
	0x74c3,0x74c0,0x74be,0x74bb,0x74ba,0x74b7,0x74b6,0x74b4,0x74b1,0x74b0,0x74ad,0x74ab,0x74a9,0x74a7,0x74a4,0x74a3,
	0x74a1,0x749e,0x749d,0x749a,0x7499,0x7496,0x7494,0x7491,0x7490,0x748d,0x748c,0x748a,0x7487,0x7486,0x7483,0x7481,
	0x747e,0x747d,0x747a,0x7479,0x7476,0x7474,0x7473,0x7470,0x746e,0x746c,0x746a,0x7467,0x7466,0x7463,0x7461,0x745e,
	0x745d,0x745c,0x7459,0x7457,0x7454,0x7453,0x7450,0x744e,0x744c,0x744a,0x7447,0x7446,0x7444,0x7441,0x7440,0x743d,
	0x743c,0x7439,0x7437,0x7434,0x7433,0x7430,0x742f,0x742d,0x742a,0x7429,0x7426,0x7424,0x7421,0x7420,0x741d,0x741c,
	0x741a,0x7417,0x7416,0x7413,0x7411,0x740f,0x740d,0x740a,0x7409,0x7406,0x7404,0x7403,0x7400,0x73ff,0x73fc,0x73fa,
	0x73f7,0x73f6,0x73f3,0x73f1,0x73ef,0x73ed,0x73ec,0x73e9,0x73e7,0x73e4,0x73e3,0x73e0,0x73df,0x73dc,0x73da,0x73d7,
	0x73d5,0x73d4,0x73d1,0x73cf,0x73cc,0x73cb,0x73c9,0x73c7,0x73c5,0x73c2,0x73c1,0x73be,0x73bc,0x73ba,0x73b8,0x73b7,
	0x73b4,0x73b2,0x73af,0x73ae,0x73ab,0x73aa,0x73a7,0x73a5,0x73a2,0x73a1,0x739f,0x739d,0x739b,0x7398,0x7397,0x7394,
	0x7392,0x738f,0x738e,0x738d,0x738a,0x7388,0x7385,0x7384,0x7381,0x7380,0x737d,0x737b,0x7378,0x7377,0x7375,0x7372,
	0x7371,0x736e,0x736d,0x736a,0x7368,0x7365,0x7364,0x7363,0x7360,0x735e,0x735b,0x735a,0x7357,0x7356,0x7353,0x7351,
	0x7350,0x734d,0x734b,0x7348,0x7347,0x7344,0x7343,0x7340,0x733e,0x733b,0x733a,0x7339,0x7336,0x7334,0x7331,0x7330,
	0x732d,0x732b,0x7329,0x7327,0x7326,0x7323,0x7321,0x731e,0x731d,0x731a,0x7319,0x7316,0x7314,0x7311,0x7310,0x730e,
	0x730c,0x730a,0x7307,0x7306,0x7303,0x7301,0x72ff,0x72fd,0x72fc,0x72f9,0x72f7,0x72f4,0x72f3,0x72f0,0x72ef,0x72ec,
	0x72ea,0x72e7,0x72e6,0x72e4,0x72e2,0x72e0,0x72dd,0x72dc,0x72d9,0x72d7,0x72d5,0x72d3,0x72d2,0x72cf,0x72cd,0x72ca,
	0x72c9,0x72c6,0x72c5,0x72c2,0x72c0,0x72bd,0x72bc,0x72b9,0x72b8,0x72b5,0x72b3,0x72b2,0x72af,0x72ad,0x72ab,0x72a9,
	0x72a6,0x72a5,0x72a2,0x72a0,0x729f,0x729c,0x729b,0x7298,0x7296,0x7293,0x7292,0x728f,0x728e,0x728c,0x7289,0x7288,
	0x7285,0x7284,0x7281,0x727f,0x727c,0x727b,0x7279,0x7277,0x7275,0x7272,0x7271,0x726e,0x726c,0x726a,0x7268,0x7267,
	0x7264,0x7262,0x725f,0x725e,0x725b,0x725a,0x7257,0x7255,0x7254,0x7251,0x724f,0x724d,0x724b,0x7248,0x7247,0x7244,
	0x7242,0x7241,0x723e,0x723d,0x723a,0x7238,0x7235,0x7234,0x7231,0x7230,0x722e,0x722b,0x722a,0x7227,0x7226,0x7223,
	0x7221,0x721e,0x721d,0x721b,0x7219,0x7217,0x7214,0x7213,0x7210,0x720e,0x720c,0x720a,0x7209,0x7206,0x7204,0x7201,
	0x7200,0x71fd,0x71fc,0x71f9,0x71f7,0x71f6,0x71f3,0x71f2,0x71ef,0x71ed,0x71ea,0x71e9,0x71e6,0x71e5,0x71e3,0x71e0,
	0x71df,0x71dc,0x71da,0x71d8,0x71d6,0x71d3,0x71d2,0x71d0,0x71cd,0x71cc,0x71c9,0x71c8,0x71c5,0x71c3,0x71c0,0x71bf,
	0x71bd,0x71ba,0x71b9,0x71b6,0x71b4,0x71b3,0x71b0,0x71ae,0x71ac,0x71aa,0x71a7,0x71a6,0x71a3,0x71a1,0x71a0,0x719d,
	0x719c,0x7199,0x7197,0x7194,0x7193,0x7190,0x718f,0x718d,0x718a,0x7189,0x7186,0x7185,0x7182,0x7180,0x717f,0x717c,
	0x717b,0x7178,0x7176,0x7173,0x7172,0x716f,0x716e,0x716c,0x7169,0x7168,0x7165,0x7163,0x7161,0x715f,0x715c,0x715b,
	0x7159,0x7156,0x7155,0x7152,0x7151,0x714e,0x714c,0x714a,0x7148,0x7147,0x7144,0x7142,0x713f,0x713e,0x713b,0x713a,
	0x7138,0x7135,0x7134,0x7131,0x7130,0x712d,0x712b,0x7128,0x7127,0x7125,0x7123,0x7121,0x711e,0x711d,0x711a,0x7119,
	0x7116,0x7114,0x7113,0x7110,0x710e,0x710c,0x710a,0x7107,0x7106,0x7103,0x7101,0x7100,0x70fd,0x70fc,0x70f9,0x70f7,
	0x70f4,0x70f3,0x70f0,0x70ef,0x70ed,0x70ea,0x70e9,0x70e6,0x70e5,0x70e2,0x70e0,0x70df,0x70dc,0x70db,0x70d8,0x70d6,
	0x70d3,0x70d2,0x70cf,0x70ce,0x70cc,0x70c9,0x70c8,0x70c5,0x70c3,0x70c1,0x70bf,0x70bc,0x70bb,0x70b9,0x70b6,0x70b5,
	0x70b2,0x70b1,0x70ae,0x70ad,0x70aa,0x70a8,0x70a5,0x70a4,0x70a2,0x70a0,0x709e,0x709b,0x709a,0x7097,0x7096,0x7093,
	0x7091,0x7090,0x708d,0x708b,0x7089,0x7087,0x7084,0x7083,0x7081,0x707f,0x707d,0x707a,0x7079,0x7076,0x7074,0x7072,
	0x7070,0x706f,0x706c,0x706a,0x7068,0x7066,0x7063,0x7062,0x7060,0x705d,0x705c,0x7059,0x7058,0x7055,0x7053,0x7051,
	0x704f,0x704e,0x704b,0x7049,0x7046,0x7045,0x7042,0x7041,0x703f,0x703c,0x703b,0x7038,0x7037,0x7034,0x7032,0x702f,
	0x702e,0x702d,0x702a,0x7028,0x7025,0x7024,0x7021,0x7020,0x701e,0x701b,0x701a,0x7017,0x7016,0x7013,0x7011,0x700e,
	0x700d,0x700c,0x7009,0x7007,0x7004,0x7003,0x7000,0x6fff,0x6ffd,0x6ffa,0x6ff9,0x6ff6,0x6ff5,0x6ff2,0x6ff0,0x6fed,
	0x6fec,0x6fea,0x6fe8,0x6fe6,0x6fe3,0x6fe2,0x6fdf,0x6fde,0x6fdc,0x6fd9,0x6fd8,0x6fd5,0x6fd3,0x6fd1,0x6fcf,0x6fcc,
	0x6fcb,0x6fc9,0x6fc7,0x6fc5,0x6fc2,0x6fc1,0x6fbe,0x6fbc,0x6fbb,0x6fb8,0x6fb7,0x6fb4,0x6fb2,0x6fb0,0x6fae,0x6fab,
	0x6fa9,0x6fa8,0x6fa5,0x6fa3,0x6fa2,0x6f9f,0x6f9e,0x6f9b,0x6f99,0x6f97,0x6f95,0x6f92,0x6f91,0x6f8f,0x6f8c,0x6f8b,
	0x6f88,0x6f87,0x6f84,0x6f82,0x6f81,0x6f7e,0x6f7d,0x6f7a,0x6f78,0x6f76,0x6f74,0x6f73,0x6f70,0x6f6e,0x6f6c,0x6f6a,
	0x6f67,0x6f66,0x6f64,0x6f61,0x6f60,0x6f5d,0x6f5c,0x6f59,0x6f57,0x6f55,0x6f53,0x6f52,0x6f4f,0x6f4d,0x6f4b,0x6f49,
	0x6f46,0x6f45,0x6f43,0x6f41,0x6f3f,0x6f3c,0x6f3b,0x6f38,0x6f36,0x6f35,0x6f32,0x6f31,0x6f2e,0x6f2c,0x6f2a,0x6f28,
	0x6f27,0x6f24,0x6f22,0x6f20,0x6f1e,0x6f1b,0x6f1a,0x6f17,0x6f16,0x6f14,0x6f11,0x6f10,0x6f0d,0x6f0b,0x6f09,0x6f07,
	0x6f06,0x6f03,0x6f01,0x6eff,0x6efd,0x6efa,0x6ef9,0x6ef7,0x6ef5,0x6ef3,0x6ef0,0x6eef,0x6eec,0x6eea,0x6ee8,0x6ee6,
	0x6ee5,0x6ee2,0x6ee0,0x6ede,0x6edc,0x6ed9,0x6ed8,0x6ed6,0x6ed4,0x6ed2,0x6ecf,0x6ece,0x6ecb,0x6eca,0x6ec8,0x6ec5,
	0x6ec4,0x6ec1,0x6ebf,0x6ebd,0x6ebb,0x6eba,0x6eb7,0x6eb5,0x6eb3,0x6eb1,0x6eae,0x6ead,0x6eaa,0x6ea9,0x6ea7,0x6ea4,
	0x6ea3,0x6ea0,0x6e9f,0x6e9c,0x6e9a,0x6e98,0x6e96,0x6e93,0x6e92,0x6e90,0x6e8e,0x6e8c,0x6e89,0x6e88,0x6e85,0x6e84,
	0x6e82,0x6e7f,0x6e7e,0x6e7b,0x6e7a,0x6e77,0x6e75,0x6e74,0x6e71,0x6e70,0x6e6d,0x6e6b,0x6e68,0x6e67,0x6e66,0x6e63,
	0x6e61,0x6e5e,0x6e5d,0x6e5a,0x6e59,0x6e57,0x6e54,0x6e53,0x6e50,0x6e4f,0x6e4c,0x6e4a,0x6e49,0x6e46,0x6e45,0x6e42,
	0x6e40,0x6e3d,0x6e3c,0x6e3b,0x6e38,0x6e36,0x6e33,0x6e32,0x6e2f,0x6e2e,0x6e2c,0x6e29,0x6e28,0x6e25,0x6e24,0x6e21,
	0x6e1f,0x6e1e,0x6e1b,0x6e1a,0x6e17,0x6e15,0x6e13,0x6e11,0x6e10,0x6e0d,0x6e0b,0x6e09,0x6e07,0x6e04,0x6e03,0x6e01,
	0x6dff,0x6dfd,0x6dfa,0x6df9,0x6df6,0x6df5,0x6df3,0x6df0,0x6def,0x6dec,0x6deb,0x6de8,0x6de6,0x6de5,0x6de2,0x6de1,
	0x6dde,0x6ddc,0x6dd9,0x6dd8,0x6dd7,0x6dd4,0x6dd2,0x6dcf,0x6dce,0x6dcb,0x6dca,0x6dc8,0x6dc5,0x6dc4,0x6dc1,0x6dc0,
	0x6dbd,0x6dbb,0x6dba,0x6db7,0x6db6,0x6db3,0x6db1,0x6daf,0x6dad,0x6dac,0x6da9,0x6da7,0x6da5,0x6da3,0x6da0,0x6d9f,
	0x6d9d,0x6d9a,0x6d99,0x6d96,0x6d94,0x6d93,0x6d90,0x6d8f,0x6d8c,0x6d8a,0x6d87,0x6d86,0x6d85,0x6d82,0x6d80,0x6d7d,
	0x6d7c,0x6d7b,0x6d78,0x6d76,0x6d73,0x6d72,0x6d6f,0x6d6e,0x6d6c,0x6d6a,0x6d68,0x6d65,0x6d64,0x6d61,0x6d60,0x6d5e,
	0x6d5b,0x6d5a,0x6d57,0x6d56,0x6d53,0x6d51,0x6d50,0x6d4d,0x6d4c,0x6d49,0x6d47,0x6d44,0x6d43,0x6d42,0x6d3f,0x6d3d,
	0x6d3a,0x6d39,0x6d36,0x6d35,0x6d33,0x6d30,0x6d2f,0x6d2c,0x6d2b,0x6d28,0x6d26,0x6d25,0x6d22,0x6d21,0x6d1e,0x6d1d,
	0x6d1b,0x6d18,0x6d17,0x6d14,0x6d13,0x6d10,0x6d0e,0x6d0d,0x6d0a,0x6d09,0x6d06,0x6d04,0x6d01,0x6d00,0x6cff,0x6cfc,
	0x6cfa,0x6cf7,0x6cf6,0x6cf3,0x6cf2,0x6cf0,0x6ced,0x6cec,0x6ce9,0x6ce8,0x6ce5,0x6ce3,0x6ce2,0x6cdf,0x6cde,0x6cdb,
	0x6cda,0x6cd7,0x6cd5,0x6cd4,0x6cd1,0x6cd0,0x6ccd,0x6ccb,0x6cc8,0x6cc7,0x6cc6,0x6cc3,0x6cc1,0x6cbe,0x6cbd,0x6cba,
	0x6cb9,0x6cb7,0x6cb4,0x6cb3,0x6cb0,0x6caf,0x6cad,0x6caa,0x6ca9,0x6ca6,0x6ca5,0x6ca2,0x6ca0,0x6c9f,0x6c9c,0x6c9b,
	0x6c98,0x6c97,0x6c94,0x6c92,0x6c90,0x6c8e,0x6c8b,0x6c8a,0x6c89,0x6c86,0x6c84,0x6c81,0x6c80,0x6c7f,0x6c7c,0x6c7a,
	0x6c77,0x6c76,0x6c73,0x6c72,0x6c70,0x6c6d,0x6c6c,0x6c69,0x6c68,0x6c65,0x6c64,0x6c62,0x6c5f,0x6c5e,0x6c5b,0x6c5a,
	0x6c58,0x6c55,0x6c54,0x6c51,0x6c50,0x6c4d,0x6c4b,0x6c4a,0x6c47,0x6c46,0x6c43,0x6c41,0x6c3f,0x6c3d,0x6c3c,0x6c39,
	0x6c37,0x6c35,0x6c33,0x6c30,0x6c2f,0x6c2e,0x6c2b,0x6c29,0x6c26,0x6c25,0x6c24,0x6c21,0x6c1f,0x6c1c,0x6c1b,0x6c18,
	0x6c17,0x6c15,0x6c12,0x6c11,0x6c0e,0x6c0d,0x6c0a,0x6c09,0x6c07,0x6c04,0x6c03,0x6c00,0x6bff,0x6bfd,0x6bfa,0x6bf9,
	0x6bf6,0x6bf5,0x6bf2,0x6bf0,0x6bef,0x6bec,0x6beb,0x6be8,0x6be6,0x6be4,0x6be2,0x6be1,0x6bde,0x6bdc,0x6bda,0x6bd8,
	0x6bd7,0x6bd4,0x6bd3,0x6bd0,0x6bce,0x6bcb,0x6bca,0x6bc9,0x6bc6,0x6bc4,0x6bc1,0x6bc0,0x6bbd,0x6bbc,0x6bba,0x6bb7,
	0x6bb6,0x6bb3,0x6bb2,0x6baf,0x6bae,0x6bac,0x6ba9,0x6ba8,0x6ba5,0x6ba4,0x6ba2,0x6b9f,0x6b9e,0x6b9b,0x6b9a,0x6b97,
	0x6b95,0x6b93,0x6b91,0x6b8f,0x6b8e,0x6b8b,0x6b8a,0x6b87,0x6b85,0x6b82,0x6b81,0x6b80,0x6b7d,0x6b7b,0x6b79,0x6b77,
	0x6b76,0x6b73,0x6b71,0x6b6f,0x6b6d,0x6b6a,0x6b69,0x6b68,0x6b65,0x6b63,0x6b60,0x6b5f,0x6b5e,0x6b5b,0x6b59,0x6b56,
	0x6b55,0x6b52,0x6b51,0x6b4f,0x6b4d,0x6b4b,0x6b48,0x6b47,0x6b45,0x6b43,0x6b41,0x6b3e,0x6b3d,0x6b3a,0x6b39,0x6b37,
	0x6b34,0x6b33,0x6b30,0x6b2f,0x6b2d,0x6b2b,0x6b29,0x6b26,0x6b25,0x6b22,0x6b21,0x6b1f,0x6b1c,0x6b1b,0x6b18,0x6b17,
	0x6b15,0x6b12,0x6b11,0x6b0e,0x6b0d,0x6b0a,0x6b09,0x6b07,0x6b04,0x6b03,0x6b00,0x6aff,0x6afc,0x6afa,0x6af9,0x6af6,
	0x6af5,0x6af2,0x6af0,0x6aef,0x6aec,0x6aeb,0x6ae8,0x6ae6,0x6ae4,0x6ae2,0x6ae1,0x6ade,0x6add,0x6ada,0x6ad8,0x6ad7,
	0x6ad4,0x6ad3,0x6ad0,0x6ace,0x6acc,0x6aca,0x6ac9,0x6ac6,0x6ac4,0x6ac2,0x6ac0,0x6abf,0x6abc,0x6abb,0x6ab8,0x6ab6,
	0x6ab3,0x6ab2,0x6ab1,0x6aae,0x6aac,0x6aaa,0x6aa8,0x6aa7,0x6aa4,0x6aa2,0x6aa0,0x6a9e,0x6a9b,0x6a9a,0x6a98,0x6a96,
	0x6a94,0x6a92,0x6a90,0x6a8d,0x6a8c,0x6a89,0x6a88,0x6a86,0x6a84,0x6a82,0x6a7f,0x6a7e,0x6a7c,0x6a7a,0x6a78,0x6a75,
	0x6a74,0x6a71,0x6a70,0x6a6e,0x6a6b,0x6a6a,0x6a67,0x6a66,0x6a64,0x6a62,0x6a60,0x6a5d,0x6a5c,0x6a59,0x6a58,0x6a56,
	0x6a53,0x6a52,0x6a4f,0x6a4e,0x6a4c,0x6a4a,0x6a48,0x6a45,0x6a44,0x6a42,0x6a40,0x6a3e,0x6a3b,0x6a3a,0x6a37,0x6a36,
	0x6a34,0x6a31,0x6a30,0x6a2d,0x6a2c,0x6a2a,0x6a28,0x6a26,0x6a23,0x6a22,0x6a1f,0x6a1e,0x6a1c,0x6a19,0x6a18,0x6a15,
	0x6a14,0x6a12,0x6a10,0x6a0e,0x6a0b,0x6a0a,0x6a08,0x6a06,0x6a04,0x6a01,0x6a00,0x69fd,0x69fc,0x69fa,0x69f8,0x69f6,
	0x69f3,0x69f2,0x69f0,0x69ee,0x69ec,0x69e9,0x69e8,0x69e5,0x69e4,0x69e2,0x69df,0x69de,0x69db,0x69da,0x69d8,0x69d6,
	0x69d4,0x69d1,0x69d0,0x69cf,0x69cc,0x69ca,0x69c7,0x69c6,0x69c3,0x69c2,0x69c0,0x69be,0x69bc,0x69b9,0x69b8,0x69b6,
	0x69b4,0x69b2,0x69af,0x69ae,0x69ab,0x69aa,0x69a8,0x69a6,0x69a4,0x69a1,0x69a0,0x699e,0x699c,0x699a,0x6997,0x6996,
	0x6994,0x6991,0x6990,0x698e,0x698c,0x698a,0x6987,0x6986,0x6983,0x6982,0x6980,0x697e,0x697c,0x6979,0x6978,0x6976,
	0x6974,0x6972,0x696f,0x696e,0x696d,0x696a,0x6968,0x6966,0x6964,0x6961,0x6960,0x695e,0x695c,0x695a,0x6957,0x6956,
	0x6955,0x6952,0x6950,0x694e,0x694c,0x694b,0x6948,0x6947,0x6944,0x6942,0x6941,0x693e,0x693d,0x693a,0x6938,0x6936,
	0x6934,0x6933,0x6930,0x692f,0x692c,0x692a,0x6929,0x6926,0x6925,0x6922,0x6920,0x691f,0x691c,0x691b,0x6918,0x6917,
	0x6915,0x6912,0x6911,0x690e,0x690d,0x690a,0x6908,0x6907,0x6904,0x6903,0x6900,0x68ff,0x68fd,0x68fa,0x68f9,0x68f6,
	0x68f5,0x68f3,0x68f0,0x68ef,0x68ec,0x68eb,0x68e8,0x68e7,0x68e5,0x68e2,0x68e1,0x68de,0x68dd,0x68db,0x68d9,0x68d7,
	0x68d4,0x68d3,0x68d1,0x68cf,0x68cd,0x68ca,0x68c9,0x68c8,0x68c5,0x68c3,0x68c1,0x68bf,0x68bc,0x68bb,0x68b9,0x68b7,
	0x68b5,0x68b2,0x68b1,0x68b0,0x68ad,0x68ab,0x68a9,0x68a7,0x68a6,0x68a3,0x68a2,0x689f,0x689d,0x689a,0x6899,0x6898,
	0x6895,0x6894,0x6891,0x688f,0x688d,0x688b,0x688a,0x6887,0x6886,0x6883,0x6881,0x6880,0x687d,0x687c,0x6879,0x6878,
	0x6876,0x6873,0x6872,0x686f,0x686e,0x686b,0x6869,0x6868,0x6865,0x6864,0x6861,0x6860,0x685e,0x685b,0x685a,0x6857,
	0x6856,0x6854,0x6852,0x6850,0x684d,0x684c,0x684b,0x6848,0x6846,0x6844,0x6842,0x6841,0x683e,0x683c,0x683a,0x6838,
	0x6837,0x6834,0x6833,0x6830,0x682e,0x682c,0x682a,0x6829,0x6826,0x6825,0x6822,0x6820,0x681f,0x681c,0x681b,0x6818,
	0x6817,0x6815,0x6812,0x6811,0x680e,0x680d,0x680b,0x6808,0x6807,0x6804,0x6803,0x6801,0x67ff,0x67fd,0x67fa,0x67f9,
	0x67f8,0x67f5,0x67f3,0x67f1,0x67ef,0x67ec,0x67eb,0x67ea,0x67e7,0x67e5,0x67e2,0x67e1,0x67e0,0x67dd,0x67db,0x67d9,
	0x67d7,0x67d6,0x67d3,0x67d2,0x67cf,0x67cd,0x67cc,0x67c9,0x67c8,0x67c5,0x67c4,0x67c2,0x67bf,0x67be,0x67bb,0x67ba,
	0x67b8,0x67b5,0x67b4,0x67b1,0x67b0,0x67ad,0x67ac,0x67aa,0x67a7,0x67a6,0x67a3,0x67a2,0x67a0,0x679e,0x679c,0x6799,
	0x6797,0x6796,0x6793,0x6792,0x6790,0x678e,0x678c,0x6789,0x6788,0x6787,0x6784,0x6782,0x6780,0x677e,0x677d,0x677a,
	0x6779,0x6776,0x6774,0x6773,0x6770,0x676f,0x676c,0x676b,0x6769,0x6766,0x6765,0x6762,0x6761,0x675f,0x675d,0x675b,
	0x6758,0x6757,0x6756,0x6753,0x6751,0x674f,0x674d,0x674c,0x6749,0x6748,0x6745,0x6743,0x6742,0x673f,0x673e,0x673b,
	0x6739,0x6738,0x6735,0x6734,0x6731,0x6730,0x672e,0x672b,0x672a,0x6727,0x6726,0x6724,0x6722,0x6720,0x671d,0x671c,
	0x671b,0x6718,0x6716,0x6714,0x6712,0x670f,0x670e,0x670d,0x670a,0x6708,0x6706,0x6704,0x6703,0x6700,0x66ff,0x66fc,
	0x66fa,0x66f9,0x66f6,0x66f5,0x66f2,0x66f1,0x66ef,0x66ec,0x66eb,0x66e8,0x66e7,0x66e5,0x66e3,0x66e1,0x66de,0x66dd,
	0x66dc,0x66d9,0x66d7,0x66d5,0x66d3,0x66d2,0x66cf,0x66ce,0x66cb,0x66c9,0x66c8,0x66c5,0x66c4,0x66c1,0x66bf,0x66be,
	0x66bb,0x66ba,0x66b7,0x66b6,0x66b4,0x66b1,0x66b0,0x66ad,0x66ac,0x66aa,0x66a8,0x66a6,0x66a3,0x66a2,0x66a1,0x669e,
	0x669d,0x669a,0x6698,0x6696,0x6694,0x6693,0x6690,0x668f,0x668c,0x668a,0x6689,0x6686,0x6685,0x6682,0x6681,0x667f,
	0x667c,0x667b,0x6678,0x6677,0x6675,0x6673,0x6671,0x666e,0x666d,0x666c,0x6669,0x6667,0x6665,0x6663,0x6662,0x665f,
	0x665e,0x665b,0x665a,0x6658,0x6655,0x6654,0x6651,0x6650,0x664e,0x664c,0x664a,0x6647,0x6646,0x6645,0x6642,0x6640,
	0x663e,0x663c,0x663b,0x6638,0x6637,0x6634,0x6632,0x6631,0x662e,0x662d,0x662a,0x6629,0x6627,0x6624,0x6623,0x6620,
	0x661f,0x661d,0x661b,0x6619,0x6616,0x6615,0x6614,0x6611,0x660f,0x660d,0x660b,0x660a,0x6607,0x6606,0x6603,0x6601,
	0x6600,0x65fd,0x65fc,0x65f9,0x65f8,0x65f6,0x65f3,0x65f2,0x65ef,0x65ee,0x65ec,0x65ea,0x65e8,0x65e5,0x65e4,0x65e3,
	0x65e0,0x65de,0x65dc,0x65da,0x65d9,0x65d6,0x65d5,0x65d2,0x65d0,0x65cf,0x65cc,0x65cb,0x65c8,0x65c7,0x65c5,0x65c2,
	0x65c1,0x65be,0x65bd,0x65bb,0x65b9,0x65b7,0x65b4,0x65b3,0x65b2,0x65af,0x65ad,0x65ab,0x65a9,0x65a8,0x65a5,0x65a4,
	0x65a2,0x659f,0x659e,0x659c,0x6599,0x6598,0x6595,0x6594,0x6592,0x6590,0x658e,0x658b,0x658a,0x6589,0x6586,0x6584,
	0x6582,0x6580,0x657f,0x657c,0x657b,0x6578,0x6577,0x6575,0x6572,0x6571,0x656e,0x656d,0x656b,0x6569,0x6567,0x6566,
	0x6563,0x6562,0x655f,0x655d,0x655c,0x6559,0x6558,0x6555,0x6554,0x6552,0x654f,0x654e,0x654b,0x654a,0x6549,0x6546,
	0x6544,0x6542,0x6540,0x653f,0x653c,0x653b,0x6538,0x6536,0x6535,0x6532,0x6531,0x652e,0x652d,0x652b,0x6528,0x6527,
	0x6526,0x6523,0x6521,0x651f,0x651d,0x651c,0x6519,0x6518,0x6515,0x6514,0x6512,0x650f,0x650e,0x650b,0x650a,0x6508,
	0x6506,0x6504,0x6501,0x6500,0x64ff,0x64fc,0x64fa,0x64f8,0x64f6,0x64f5,0x64f2,0x64f1,0x64ee,0x64ec,0x64eb,0x64e8,
	0x64e7,0x64e4,0x64e3,0x64e1,0x64df,0x64dd,0x64dc,0x64d9,0x64d8,0x64d5,0x64d3,0x64d2,0x64cf,0x64ce,0x64cb,0x64ca,
	0x64c8,0x64c5,0x64c4,0x64c1,0x64c0,0x64be,0x64bc,0x64ba,0x64b7,0x64b6,0x64b5,0x64b2,0x64b1,0x64ae,0x64ac,0x64ab,
	0x64a8,0x64a7,0x64a4,0x64a3,0x64a0,0x649f,0x649d,0x649a,0x6499,0x6496,0x6495,0x6493,0x6491,0x648f,0x648e,0x648b,
	0x648a,0x6487,0x6486,0x6484,0x6481,0x6480,0x647d,0x647c,0x647a,0x6478,0x6476,0x6473,0x6472,0x6471,0x646e,0x646c,
	0x646b,0x6468,0x6467,0x6464,0x6463,0x6461,0x645f,0x645d,0x645a,0x6459,0x6458,0x6455,0x6453,0x6451,0x644f,0x644e,
	0x644b,0x644a,0x6447,0x6446,0x6444,0x6441,0x6440,0x643f,0x643c,0x643a,0x6438,0x6436,0x6435,0x6432,0x6431,0x642e,
	0x642d,0x642b,0x6428,0x6427,0x6424,0x6423,0x6421,0x641f,0x641d,0x641a,0x6419,0x6418,0x6415,0x6413,0x6412,0x640f,
	0x640e,0x640b,0x640a,0x6408,0x6406,0x6404,0x6401,0x6400,0x63ff,0x63fc,0x63fa,0x63f8,0x63f6,0x63f5,0x63f2,0x63f1,
	0x63ef,0x63ed,0x63eb,0x63e8,0x63e7,0x63e6,0x63e3,0x63e1,0x63df,0x63dd,0x63dc,0x63d9,0x63d8,0x63d5,0x63d4,0x63d2,
	0x63cf,0x63ce,0x63cb,0x63ca,0x63c8,0x63c6,0x63c4,0x63c3,0x63c0,0x63bf,0x63bc,0x63ba,0x63b9,0x63b6,0x63b5,0x63b2,
	0x63b0,0x63af,0x63ac,0x63ab,0x63a9,0x63a6,0x63a5,0x63a4,0x63a1,0x639f,0x639d,0x639b,0x639a,0x6397,0x6396,0x6393,
	0x6392,0x6390,0x638d,0x638c,0x638b,0x6388,0x6387,0x6384,0x6382,0x6381,0x637e,0x637d,0x637a,0x6379,0x6377,0x6374,
	0x6373,0x6370,0x636f,0x636e,0x636b,0x6369,0x6368,0x6365,0x6364,0x6361,0x6360,0x635e,0x635c,0x635a,0x6357,0x6356,
	0x6355,0x6352,0x6350,0x634f,0x634c,0x634b,0x6348,0x6347,0x6345,0x6343,0x6341,0x633e,0x633d,0x633c,0x6339,0x6337,
	0x6336,0x6333,0x6332,0x632f,0x632e,0x632c,0x632a,0x6328,0x6325,0x6324,0x6323,0x6320,0x631f,0x631c,0x631a,0x6319,
	0x6316,0x6315,0x6313,0x6311,0x630f,0x630c,0x630b,0x630a,0x6307,0x6306,0x6303,0x6301,0x6300,0x62fd,0x62fc,0x62fa,
	0x62f8,0x62f6,0x62f4,0x62f2,0x62f1,0x62ee,0x62ed,0x62ea,0x62e8,0x62e7,0x62e4,0x62e3,0x62e0,0x62df,0x62dd,0x62db,
	0x62d9,0x62d8,0x62d5,0x62d4,0x62d1,0x62cf,0x62ce,0x62cb,0x62ca,0x62c7,0x62c6,0x62c4,0x62c2,0x62c0,0x62bf,0x62bc,
	0x62bb,0x62b8,0x62b7,0x62b4,0x62b2,0x62b1,0x62ae,0x62ad,0x62aa,0x62a9,0x62a7,0x62a5,0x62a3,0x62a2,0x629f,0x629e,
	0x629b,0x6299,0x6298,0x6295,0x6294,0x6291,0x6290,0x628e,0x628c,0x628a,0x6289,0x6286,0x6285,0x6282,0x6281,0x627f,
	0x627c,0x627b,0x6278,0x6277,0x6276,0x6273,0x6271,0x6270,0x626d,0x626c,0x6269,0x6268,0x6266,0x6264,0x6262,0x6261,
	0x625e,0x625d,0x625a,0x6259,0x6257,0x6254,0x6253,0x6250,0x624f,0x624d,0x624b,0x6249,0x6248,0x6245,0x6244,0x6241,
	0x6240,0x623e,0x623b,0x623a,0x6237,0x6236,0x6235,0x6232,0x6230,0x622f,0x622c,0x622b,0x6228,0x6227,0x6225,0x6223,
	0x6221,0x621e,0x621d,0x621c,0x6219,0x6218,0x6216,0x6213,0x6212,0x620f,0x620e,0x620c,0x620a,0x6208,0x6207,0x6204,
	0x6203,0x6200,0x61ff,0x61fd,0x61fa,0x61f9,0x61f6,0x61f5,0x61f4,0x61f1,0x61ef,0x61ee,0x61eb,0x61ea,0x61e7,0x61e6,
	0x61e4,0x61e2,0x61e0,0x61dd,0x61dc,0x61db,0x61d8,0x61d7,0x61d5,0x61d2,0x61d1,0x61ce,0x61cd,0x61cb,0x61c9,0x61c7,
	0x61c5,0x61c2,0x61c1,0x61c0,0x61bd,0x61bc,0x61b9,0x61b7,0x61b6,0x61b3,0x61b2,0x61b1,0x61ae,0x61ac,0x61aa,0x61a8,
	0x61a7,0x61a4,0x61a3,0x61a1,0x619f,0x619d,0x619a,0x6199,0x6198,0x6195,0x6194,0x6192,0x618f,0x618e,0x618b,0x618a,
	0x6189,0x6186,0x6184,0x6182,0x6180,0x617f,0x617c,0x617b,0x6179,0x6177,0x6175,0x6172,0x6171,0x6170,0x616d,0x616c,
	0x616a,0x6167,0x6166,0x6163,0x6162,0x6161,0x615e,0x615c,0x615b,0x6158,0x6157,0x6154,0x6153,0x6151,0x614f,0x614d,
	0x614c,0x6149,0x6148,0x6145,0x6144,0x6142,0x613f,0x613e,0x613b,0x613a,0x6139,0x6136,0x6134,0x6133,0x6130,0x612f,
	0x612c,0x612b,0x6129,0x6127,0x6125,0x6124,0x6121,0x6120,0x611d,0x611c,0x611a,0x6117,0x6116,0x6115,0x6112,0x6111,
	0x610e,0x610c,0x610b,0x6108,0x6107,0x6104,0x6103,0x6101,0x60ff,0x60fd,0x60fc,0x60f9,0x60f8,0x60f5,0x60f4,0x60f2,
	0x60ef,0x60ee,0x60ed,0x60ea,0x60e9,0x60e6,0x60e4,0x60e3,0x60e0,0x60df,0x60de,0x60db,0x60d9,0x60d7,0x60d5,0x60d4,
	0x60d1,0x60d0,0x60cd,0x60cc,0x60ca,0x60c7,0x60c6,0x60c3,0x60c2,0x60c1,0x60be,0x60bc,0x60bb,0x60b8,0x60b7,0x60b4,
	0x60b3,0x60b1,0x60af,0x60ad,0x60ac,0x60a9,0x60a8,0x60a5,0x60a4,0x60a2,0x60a0,0x609e,0x609d,0x609a,0x6099,0x6096,
	0x6095,0x6093,0x6090,0x608f,0x608e,0x608b,0x608a,0x6087,0x6085,0x6084,0x6081,0x6080,0x607f,0x607c,0x607a,0x6078,
	0x6076,0x6075,0x6072,0x6071,0x606f,0x606d,0x606b,0x6069,0x6067,0x6066,0x6063,0x6062,0x6060,0x605e,0x605c,0x6059,
	0x6058,0x6057,0x6054,0x6053,0x6051,0x604e,0x604d,0x604a,0x6049,0x6048,0x6045,0x6043,0x6042,0x603f,0x603e,0x603b,
	0x603a,0x6038,0x6036,0x6034,0x6033,0x6030,0x602f,0x602c,0x602b,0x6029,0x6027,0x6025,0x6024,0x6021,0x6020,0x601d,
	0x601c,0x601a,0x6017,0x6016,0x6015,0x6012,0x6011,0x600e,0x600c,0x600b,0x6008,0x6007,0x6006,0x6003,0x6001,0x5fff,
	0x5ffd,0x5ffc,0x5ff9,0x5ff8,0x5ff6,0x5ff4,0x5ff2,0x5ff0,0x5fee,0x5fed,0x5fea,0x5fe9,0x5fe7,0x5fe5,0x5fe3,0x5fe0,
	0x5fde,0x5fdd,0x5fdb,0x5fd9,0x5fd7,0x5fd5,0x5fd3,0x5fd2,0x5fcf,0x5fce,0x5fcc,0x5fca,0x5fc8,0x5fc5,0x5fc4,0x5fc3,
	0x5fc0,0x5fbf,0x5fbd,0x5fba,0x5fb9,0x5fb6,0x5fb5,0x5fb4,0x5fb1,0x5fb0,0x5fae,0x5fab,0x5faa,0x5fa7,0x5fa6,0x5fa5,
	0x5fa2,0x5fa0,0x5f9f,0x5f9c,0x5f9b,0x5f9a,0x5f97,0x5f95,0x5f93,0x5f91,0x5f90,0x5f8d,0x5f8c,0x5f8a,0x5f88,0x5f86,
	0x5f84,0x5f82,0x5f81,0x5f7e,0x5f7d,0x5f7b,0x5f79,0x5f77,0x5f75,0x5f73,0x5f72,0x5f6f,0x5f6e,0x5f6c,0x5f6a,0x5f68,
	0x5f67,0x5f64,0x5f63,0x5f60,0x5f5f,0x5f5d,0x5f5a,0x5f59,0x5f58,0x5f55,0x5f54,0x5f51,0x5f50,0x5f4e,0x5f4b,0x5f4a,
	0x5f49,0x5f46,0x5f45,0x5f42,0x5f40,0x5f3f,0x5f3c,0x5f3b,0x5f3a,0x5f37,0x5f35,0x5f33,0x5f31,0x5f30,0x5f2d,0x5f2c,
	0x5f2b,0x5f28,0x5f26,0x5f25,0x5f22,0x5f21,0x5f1e,0x5f1d,0x5f1b,0x5f19,0x5f17,0x5f16,0x5f13,0x5f12,0x5f0f,0x5f0e,
	0x5f0c,0x5f0a,0x5f08,0x5f07,0x5f04,0x5f03,0x5f00,0x5eff,0x5efd,0x5efb,0x5ef9,0x5ef8,0x5ef5,0x5ef4,0x5ef1,0x5ef0,
	0x5eee,0x5eeb,0x5eea,0x5ee7,0x5ee6,0x5ee4,0x5ee2,0x5ee0,0x5edf,0x5edc,0x5edb,0x5ed8,0x5ed7,0x5ed5,0x5ed3,0x5ed1,
	0x5ed0,0x5ecd,0x5ecc,0x5eca,0x5ec8,0x5ec6,0x5ec4,0x5ec2,0x5ec1,0x5ebe,0x5ebd,0x5ebb,0x5eb9,0x5eb7,0x5eb4,0x5eb3,
	0x5eb2,0x5eaf,0x5eae,0x5eac,0x5eaa,0x5ea8,0x5ea7,0x5ea4,0x5ea3,0x5ea0,0x5e9f,0x5e9d,0x5e9a,0x5e99,0x5e98,0x5e95,
	0x5e94,0x5e92,0x5e90,0x5e8e,0x5e8b,0x5e8a,0x5e89,0x5e86,0x5e85,0x5e83,0x5e81,0x5e7f,0x5e7c,0x5e7b,0x5e7a,0x5e77,
	0x5e76,0x5e74,0x5e71,0x5e70,0x5e6f,0x5e6c,0x5e6b,0x5e68,0x5e67,0x5e65,0x5e62,0x5e61,0x5e60,0x5e5d,0x5e5c,0x5e59,
	0x5e58,0x5e56,0x5e53,0x5e52,0x5e51,0x5e4e,0x5e4d,0x5e4b,0x5e48,0x5e47,0x5e44,0x5e43,0x5e42,0x5e3f,0x5e3e,0x5e3c,
	0x5e39,0x5e38,0x5e37,0x5e34,0x5e33,0x5e30,0x5e2e,0x5e2d,0x5e2a,0x5e29,0x5e28,0x5e25,0x5e24,0x5e21,0x5e1f,0x5e1e,
	0x5e1b,0x5e1a,0x5e19,0x5e16,0x5e15,0x5e13,0x5e10,0x5e0f,0x5e0c,0x5e0b,0x5e0a,0x5e07,0x5e05,0x5e04,0x5e01,0x5e00,
	0x5dff,0x5dfc,0x5dfb,0x5df8,0x5df6,0x5df5,0x5df2,0x5df1,0x5df0,0x5ded,0x5deb,0x5de9,0x5de7,0x5de6,0x5de3,0x5de2,
	0x5de1,0x5dde,0x5ddc,0x5ddb,0x5dd8,0x5dd7,0x5dd4,0x5dd3,0x5dd2,0x5dcf,0x5dcd,0x5dcc,0x5dc9,0x5dc8,0x5dc5,0x5dc4,
	0x5dc2,0x5dc0,0x5dbe,0x5dbd,0x5dba,0x5db9,0x5db8,0x5db5,0x5db3,0x5db1,0x5daf,0x5dae,0x5dab,0x5daa,0x5da9,0x5da6,
	0x5da4,0x5da3,0x5da0,0x5d9f,0x5d9c,0x5d9b,0x5d99,0x5d97,0x5d95,0x5d94,0x5d91,0x5d90,0x5d8d,0x5d8c,0x5d8a,0x5d88,
	0x5d86,0x5d85,0x5d82,0x5d81,0x5d7f,0x5d7d,0x5d7b,0x5d79,0x5d77,0x5d76,0x5d73,0x5d72,0x5d70,0x5d6e,0x5d6c,0x5d6a,
	0x5d68,0x5d67,0x5d64,0x5d63,0x5d61,0x5d5f,0x5d5d,0x5d5c,0x5d59,0x5d58,0x5d55,0x5d54,0x5d52,0x5d50,0x5d4e,0x5d4d,
	0x5d4a,0x5d49,0x5d47,0x5d45,0x5d43,0x5d41,0x5d3f,0x5d3e,0x5d3b,0x5d3a,0x5d38,0x5d36,0x5d34,0x5d32,0x5d30,0x5d2f,
	0x5d2c,0x5d2b,0x5d29,0x5d27,0x5d25,0x5d24,0x5d21,0x5d20,0x5d1d,0x5d1c,0x5d1a,0x5d18,0x5d16,0x5d15,0x5d12,0x5d11,
	0x5d0f,0x5d0e,0x5d0c,0x5d0a,0x5d08,0x5d05,0x5d04,0x5d03,0x5d00,0x5cff,0x5cfd,0x5cfb,0x5cf9,0x5cf8,0x5cf5,0x5cf4,
	0x5cf1,0x5cf0,0x5cee,0x5cec,0x5cea,0x5ce9,0x5ce6,0x5ce5,0x5ce3,0x5ce1,0x5cdf,0x5cdd,0x5cdb,0x5cda,0x5cd7,0x5cd6,
	0x5cd4,0x5cd2,0x5cd0,0x5ccf,0x5ccc,0x5ccb,0x5cc8,0x5cc7,0x5cc5,0x5cc3,0x5cc1,0x5cc0,0x5cbd,0x5cbc,0x5cba,0x5cb8,
	0x5cb6,0x5cb4,0x5cb2,0x5cb1,0x5cae,0x5cad,0x5cab,0x5ca9,0x5ca7,0x5ca6,0x5ca3,0x5ca2,0x5c9f,0x5c9e,0x5c9c,0x5c9a,
	0x5c98,0x5c97,0x5c94,0x5c93,0x5c92,0x5c8f,0x5c8d,0x5c8b,0x5c89,0x5c88,0x5c85,0x5c84,0x5c83,0x5c80,0x5c7e,0x5c7d,
	0x5c7a,0x5c79,0x5c76,0x5c75,0x5c74,0x5c71,0x5c6f,0x5c6e,0x5c6b,0x5c6a,0x5c69,0x5c66,0x5c65,0x5c62,0x5c60,0x5c5f,
	0x5c5c,0x5c5b,0x5c5a,0x5c57,0x5c56,0x5c54,0x5c52,0x5c50,0x5c4d,0x5c4c,0x5c4b,0x5c48,0x5c47,0x5c45,0x5c43,0x5c41,
	0x5c40,0x5c3d,0x5c3c,0x5c39,0x5c38,0x5c36,0x5c34,0x5c32,0x5c31,0x5c2e,0x5c2d,0x5c2b,0x5c29,0x5c27,0x5c25,0x5c23,
	0x5c22,0x5c21,0x5c1e,0x5c1d,0x5c1b,0x5c18,0x5c17,0x5c14,0x5c13,0x5c12,0x5c0f,0x5c0e,0x5c0c,0x5c0a,0x5c08,0x5c07,
	0x5c04,0x5c03,0x5c01,0x5bff,0x5bfd,0x5bfb,0x5bf9,0x5bf8,0x5bf5,0x5bf4,0x5bf2,0x5bf0,0x5bee,0x5bed,0x5bea,0x5be9,
	0x5be8,0x5be5,0x5be3,0x5be1,0x5bdf,0x5bde,0x5bdb,0x5bda,0x5bd9,0x5bd6,0x5bd5,0x5bd3,0x5bd0,0x5bcf,0x5bcc,0x5bcb,
	0x5bca,0x5bc7,0x5bc6,0x5bc4,0x5bc1,0x5bc0,0x5bbf,0x5bbc,0x5bbb,0x5bb9,0x5bb7,0x5bb5,0x5bb3,0x5bb1,0x5bb0,0x5bad,
	0x5bac,0x5baa,0x5ba8,0x5ba6,0x5ba5,0x5ba2,0x5ba1,0x5b9e,0x5b9d,0x5b9b,0x5b99,0x5b97,0x5b96,0x5b93,0x5b92,0x5b91,
	0x5b8e,0x5b8c,0x5b8b,0x5b88,0x5b87,0x5b84,0x5b83,0x5b82,0x5b7f,0x5b7e,0x5b7c,0x5b79,0x5b78,0x5b77,0x5b74,0x5b73,
	0x5b71,0x5b6f,0x5b6d,0x5b6a,0x5b69,0x5b68,0x5b65,0x5b64,0x5b62,0x5b60,0x5b5e,0x5b5d,0x5b5a,0x5b59,0x5b56,0x5b55,
	0x5b53,0x5b51,0x5b4f,0x5b4e,0x5b4b,0x5b4a,0x5b48,0x5b46,0x5b44,0x5b43,0x5b40,0x5b3f,0x5b3c,0x5b3b,0x5b3a,0x5b37,
	0x5b36,0x5b34,0x5b32,0x5b31,0x5b2e,0x5b2d,0x5b2b,0x5b29,0x5b27,0x5b26,0x5b23,0x5b22,0x5b1f,0x5b1e,0x5b1c,0x5b1a,
	0x5b18,0x5b17,0x5b14,0x5b13,0x5b12,0x5b0f,0x5b0e,0x5b0c,0x5b09,0x5b08,0x5b07,0x5b04,0x5b03,0x5b00,0x5aff,0x5afd,
	0x5afb,0x5af9,0x5af8,0x5af5,0x5af4,0x5af2,0x5af0,0x5aee,0x5aed,0x5aea,0x5ae9,0x5ae6,0x5ae5,0x5ae4,0x5ae1,0x5adf,
	0x5ade,0x5adb,0x5ada,0x5ad9,0x5ad6,0x5ad5,0x5ad3,0x5ad1,0x5acf,0x5acc,0x5acb,0x5aca,0x5ac7,0x5ac6,0x5ac4,0x5ac2,
	0x5ac0,0x5abf,0x5abc,0x5abb,0x5aba,0x5ab7,0x5ab5,0x5ab4,0x5ab1,0x5ab0,0x5aad,0x5aac,0x5aab,0x5aa8,0x5aa7,0x5aa5,
	0x5aa2,0x5aa1,0x5aa0,0x5a9d,0x5a9c,0x5a9a,0x5a98,0x5a96,0x5a94,0x5a92,0x5a91,0x5a8e,0x5a8d,0x5a8b,0x5a89,0x5a87,
	0x5a86,0x5a83,0x5a82,0x5a81,0x5a7e,0x5a7d,0x5a7a,0x5a78,0x5a77,0x5a74,0x5a73,0x5a72,0x5a6f,0x5a6e,0x5a6c,0x5a6a,
	0x5a68,0x5a67,0x5a64,0x5a63,0x5a61,0x5a5f,0x5a5d,0x5a5b,0x5a59,0x5a58,0x5a55,0x5a54,0x5a53,0x5a50,0x5a4e,0x5a4d,
	0x5a4c,0x5a49,0x5a48,0x5a45,0x5a44,0x5a42,0x5a40,0x5a3e,0x5a3d,0x5a3a,0x5a39,0x5a37,0x5a35,0x5a33,0x5a32,0x5a2f,
	0x5a2e,0x5a2d,0x5a2a,0x5a29,0x5a26,0x5a24,0x5a23,0x5a20,0x5a1f,0x5a1e,0x5a1b,0x5a1a,0x5a18,0x5a16,0x5a14,0x5a13,
	0x5a10,0x5a0f,0x5a0e,0x5a0b,0x5a09,0x5a07,0x5a05,0x5a04,0x5a01,0x5a00,0x59ff,0x59fc,0x59fb,0x59f9,0x59f7,0x59f5,
	0x59f4,0x59f1,0x59f0,0x59ee,0x59ec,0x59ea,0x59e9,0x59e6,0x59e5,0x59e2,0x59e1,0x59e0,0x59dd,0x59dc,0x59da,0x59d7,
	0x59d6,0x59d5,0x59d2,0x59d1,0x59cf,0x59cd,0x59cb,0x59ca,0x59c7,0x59c6,0x59c3,0x59c2,0x59c0,0x59be,0x59bc,0x59bb,
	0x59b8,0x59b7,0x59b6,0x59b3,0x59b2,0x59b0,0x59ae,0x59ac,0x59ab,0x59a8,0x59a7,0x59a5,0x59a3,0x59a1,0x599f,0x599d,
	0x599c,0x5999,0x5998,0x5997,0x5994,0x5993,0x5991,0x598e,0x598d,0x598c,0x5989,0x5988,0x5986,0x5984,0x5982,0x5980,
	0x597e,0x597d,0x597a,0x5979,0x5977,0x5975,0x5973,0x5972,0x596f,0x596e,0x596d,0x596a,0x5969,0x5967,0x5965,0x5963,
	0x5961,0x5960,0x595e,0x595c,0x595a,0x5959,0x5956,0x5955,0x5952,0x5951,0x5950,0x594d,0x594b,0x594a,0x5947,0x5946,
	0x5945,0x5942,0x5941,0x593f,0x593d,0x593b,0x593a,0x5937,0x5936,0x5935,0x5932,0x5931,0x592f,0x592c,0x592b,0x5928,
	0x5927,0x5926,0x5923,0x5922,0x5920,0x591e,0x591c,0x591b,0x5918,0x5917,0x5916,0x5913,0x5912,0x5910,0x590d,0x590c,
	0x590b,0x5908,0x5907,0x5905,0x5903,0x5901,0x58ff,0x58fd,0x58fc,0x58f9,0x58f8,0x58f7,0x58f4,0x58f3,0x58f1,0x58ee,
	0x58ed,0x58ec,0x58e9,0x58e8,0x58e6,0x58e4,0x58e2,0x58e1,0x58de,0x58dd,0x58dc,0x58d9,0x58d8,0x58d5,0x58d4,0x58d2,
	0x58cf,0x58ce,0x58cd,0x58ca,0x58c9,0x58c7,0x58c5,0x58c3,0x58c2,0x58bf,0x58be,0x58bd,0x58ba,0x58b9,0x58b7,0x58b5,
	0x58b3,0x58b2,0x58af,0x58ae,0x58ab,0x58aa,0x58a8,0x58a6,0x58a4,0x58a3,0x58a0,0x589f,0x589e,0x589b,0x589a,0x5898,
	0x5896,0x5894,0x5893,0x5890,0x588f,0x588d,0x588b,0x5889,0x5888,0x5885,0x5884,0x5881,0x5880,0x587f,0x587c,0x587b,
	0x5879,0x5878,0x5875,0x5874,0x5872,0x5870,0x586e,0x586d,0x586a,0x5869,0x5867,0x5865,0x5863,0x5862,0x585f,0x585e,
	0x585d,0x585a,0x5859,0x5856,0x5855,0x5853,0x5851,0x584f,0x584e,0x584b,0x584a,0x5849,0x5846,0x5845,0x5843,0x5841,
	0x583f,0x583e,0x583b,0x583a,0x5838,0x5836,0x5834,0x5833,0x5830,0x582f,0x582e,0x582b,0x582a,0x5828,0x5826,0x5824,
	0x5823,0x5820,0x581f,0x581c,0x581b,0x581a,0x5817,0x5816,0x5814,0x5811,0x5810,0x580f,0x580c,0x580b,0x5809,0x5807,
	0x5805,0x5804,0x5801,0x5800,0x57ff,0x57fc,0x57fb,0x57f9,0x57f7,0x57f5,0x57f4,0x57f1,0x57f0,0x57ef,0x57ec,0x57eb,
	0x57e9,0x57e6,0x57e5,0x57e2,0x57e1,0x57e0,0x57dd,0x57dc,0x57da,0x57d8,0x57d6,0x57d5,0x57d2,0x57d1,0x57d0,0x57cd,
	0x57cc,0x57ca,0x57c8,0x57c6,0x57c5,0x57c2,0x57c1,0x57c0,0x57bd,0x57bb,0x57ba,0x57b7,0x57b6,0x57b5,0x57b2,0x57b1,
	0x57af,0x57ad,0x57ab,0x57a9,0x57a7,0x57a6,0x57a3,0x57a2,0x57a1,0x579e,0x579d,0x579b,0x5799,0x5797,0x5796,0x5793,
	0x5792,0x578f,0x578e,0x578d,0x578a,0x5789,0x5787,0x5785,0x5783,0x5782,0x577f,0x577e,0x577d,0x577a,0x5779,0x5777,
	0x5775,0x5773,0x5772,0x576f,0x576e,0x576d,0x576a,0x5769,0x5767,0x5764,0x5763,0x5762,0x575f,0x575e,0x575c,0x575a,
	0x5758,0x5757,0x5754,0x5753,0x5752,0x574f,0x574e,0x574b,0x574a,0x5748,0x5746,0x5744,0x5743,0x5740,0x573f,0x573e,
	0x573b,0x573a,0x5738,0x5736,0x5734,0x5733,0x5730,0x572f,0x572e,0x572b,0x572a,0x5728,0x5726,0x5724,0x5723,0x5720,
	0x571f,0x571d,0x571b,0x5719,0x5718,0x5715,0x5714,0x5713,0x5710,0x570f,0x570d,0x570b,0x5709,0x5708,0x5705,0x5704,
	0x5703,0x5700,0x56ff,0x56fd,0x56fb,0x56f9,0x56f8,0x56f5,0x56f4,0x56f3,0x56f0,0x56ef,0x56ec,0x56eb,0x56e9,0x56e7,
	0x56e5,0x56e4,0x56e1,0x56e0,0x56df,0x56dc,0x56da,0x56d9,0x56d6,0x56d5,0x56d4,0x56d1,0x56d0,0x56ce,0x56cc,0x56ca,
	0x56c9,0x56c6,0x56c5,0x56c4,0x56c1,0x56c0,0x56be,0x56bc,0x56ba,0x56b9,0x56b6,0x56b5,0x56b4,0x56b1,0x56b0,0x56ae,
	0x56ac,0x56aa,0x56a8,0x56a6,0x56a4,0x56a3,0x56a0,0x569f,0x569e,0x569b,0x569a,0x5698,0x5696,0x5694,0x5693,0x5690,
	0x568f,0x568e,0x568b,0x568a,0x5688,0x5686,0x5684,0x5683,0x5680,0x567f,0x567e,0x567b,0x567a,0x5678,0x5676,0x5674,
	0x5673,0x5670,0x566f,0x566e,0x566b,0x566a,0x5668,0x5666,0x5664,0x5663,0x5660,0x565f,0x565e,0x565b,0x565a,0x5658,
	0x5656,0x5654,0x5653,0x5650,0x564f,0x564d,0x564b,0x5649,0x5648,0x5645,0x5644,0x5643,0x5640,0x563f,0x563d,0x563b,
	0x5639,0x5638,0x5635,0x5634,0x5633,0x5630,0x562f,0x562d,0x562b,0x5629,0x5628,0x5625,0x5624,0x5623,0x5620,0x561f,
	0x561d,0x561b,0x5619,0x5618,0x5615,0x5614,0x5613,0x5610,0x560f,0x560d,0x560b,0x5609,0x5608,0x5605,0x5604,0x5603,
	0x5600,0x55ff,0x55fd,0x55fb,0x55f9,0x55f8,0x55f5,0x55f4,0x55f3,0x55f0,0x55ef,0x55ed,0x55eb,0x55e9,0x55e8,0x55e5,
	0x55e4,0x55e3,0x55e0,0x55df,0x55dd,0x55db,0x55d9,0x55d8,0x55d5,0x55d4,0x55d3,0x55d0,0x55cf,0x55cd,0x55cb,0x55c9,
	0x55c6,0x55c5,0x55c4,0x55c1,0x55c0,0x55be,0x55bc,0x55ba,0x55b9,0x55b6,0x55b5,0x55b4,0x55b1,0x55b0,0x55ae,0x55ac,
	0x55ab,0x55a9,0x55a7,0x55a5,0x55a4,0x55a1,0x55a0,0x559f,0x559c,0x559b,0x5599,0x5597,0x5595,0x5594,0x5591,0x5590,
	0x558f,0x558c,0x558b,0x5589,0x5587,0x5585,0x5584,0x5581,0x5580,0x557f,0x557c,0x557b,0x5579,0x5577,0x5575,0x5574,
	0x5571,0x5570,0x556f,0x556c,0x556b,0x5569,0x5567,0x5565,0x5564,0x5561,0x5560,0x555f,0x555c,0x555b,0x5559,0x5557,
	0x5555,0x5554,0x5551,0x5550,0x554f,0x554c,0x554b,0x5549,0x5547,0x5545,0x5544,0x5541,0x5540,0x553f,0x553c,0x553b,
	0x5539,0x5537,0x5535,0x5534,0x5531,0x5530,0x552f,0x552c,0x552b,0x5529,0x5527,0x5525,0x5524,0x5521,0x5520,0x551f,
	0x551c,0x551b,0x5519,0x5517,0x5515,0x5514,0x5511,0x5510,0x550f,0x550c,0x550b,0x5509,0x5507,0x5505,0x5504,0x5501,
	0x5500,0x54ff,0x54fc,0x54fb,0x54f9,0x54f7,0x54f5,0x54f4,0x54f1,0x54f0,0x54ef,0x54ec,0x54eb,0x54e9,0x54e7,0x54e5,
	0x54e3,0x54e2,0x54df,0x54de,0x54dc,0x54da,0x54d8,0x54d7,0x54d5,0x54d3,0x54d2,0x54cf,0x54ce,0x54cd,0x54ca,0x54c9,
	0x54c7,0x54c5,0x54c3,0x54c2,0x54bf,0x54be,0x54bd,0x54ba,0x54b9,0x54b7,0x54b5,0x54b3,0x54b2,0x54af,0x54ae,0x54ad,
	0x54ab,0x54a9,0x54a7,0x54a6,0x54a3,0x54a2,0x54a1,0x549e,0x549d,0x549b,0x5499,0x5497,0x5496,0x5493,0x5492,0x5491,
	0x548e,0x548d,0x548b,0x5489,0x5487,0x5486,0x5483,0x5482,0x5481,0x547e,0x547d,0x547c,0x5479,0x5478,0x5476,0x5474,
	0x5472,0x5471,0x546e,0x546d,0x546c,0x5469,0x5468,0x5466,0x5464,0x5462,0x5461,0x545e,0x545d,0x545c,0x5459,0x5458,
	0x5456,0x5454,0x5452,0x5451,0x544e,0x544d,0x544c,0x5449,0x5448,0x5446,0x5444,0x5442,0x5441,0x543e,0x543d,0x543c,
	0x543a,0x5438,0x5436,0x5435,0x5432,0x5431,0x5430,0x542d,0x542c,0x542b,0x5428,0x5427,0x5425,0x5423,0x5421,0x5420,
	0x541d,0x541c,0x541b,0x5418,0x5417,0x5415,0x5413,0x5411,0x5410,0x540d,0x540c,0x540b,0x5408,0x5407,0x5405,0x5403,
	0x5401,0x5400,0x53fd,0x53fc,0x53fb,0x53f8,0x53f7,0x53f5,0x53f3,0x53f1,0x53f0,0x53ed,0x53ec,0x53eb,0x53e8,0x53e7,
	0x53e5,0x53e3,0x53e1,0x53e0,0x53dd,0x53dc,0x53db,0x53d8,0x53d7,0x53d6,0x53d3,0x53d2,0x53d0,0x53ce,0x53cc,0x53cb,
	0x53ca,0x53c7,0x53c6,0x53c4,0x53c2,0x53c0,0x53bf,0x53bc,0x53bb,0x53ba,0x53b7,0x53b6,0x53b4,0x53b2,0x53b0,0x53af,
	0x53ac,0x53ab,0x53aa,0x53a7,0x53a6,0x53a4,0x53a2,0x53a0,0x539f,0x539c,0x539b,0x539a,0x5397,0x5396,0x5394,0x5392,
	0x5390,0x538f,0x538c,0x538b,0x538a,0x5387,0x5386,0x5384,0x5382,0x5381,0x537f,0x537d,0x537b,0x537a,0x5377,0x5376,
	0x5375,0x5372,0x5371,0x536f,0x536d,0x536b,0x536a,0x5367,0x5366,0x5365,0x5362,0x5361,0x535f,0x535d,0x535b,0x535a,
	0x5359,0x5356,0x5355,0x5353,0x5351,0x534f,0x534e,0x534b,0x534a,0x5349,0x5346,0x5345,0x5343,0x5341,0x533f,0x533e,
	0x533b,0x533a,0x5339,0x5336,0x5335,0x5333,0x5331,0x532f,0x532e,0x532c,0x532a,0x5329,0x5326,0x5325,0x5324,0x5321,
	0x5320,0x531e,0x531d,0x531a,0x5319,0x5318,0x5315,0x5314,0x5313,0x5310,0x530f,0x530d,0x530b,0x5309,0x5308,0x5305,
	0x5304,0x5303,0x5300,0x52ff,0x52fd,0x52fb,0x52f9,0x52f8,0x52f5,0x52f4,0x52f3,0x52f0,0x52ef,0x52ed,0x52ec,0x52e9,
	0x52e8,0x52e7,0x52e4,0x52e3,0x52e2,0x52df,0x52de,0x52dc,0x52da,0x52d8,0x52d7,0x52d4,0x52d3,0x52d2,0x52cf,0x52ce,
	0x52cc,0x52ca,0x52c8,0x52c7,0x52c4,0x52c3,0x52c2,0x52bf,0x52be,0x52bc,0x52ba,0x52b8,0x52b7,0x52b5,0x52b3,0x52b2,
	0x52b1,0x52ae,0x52ad,0x52ab,0x52a9,0x52a7,0x52a6,0x52a3,0x52a2,0x52a1,0x529e,0x529d,0x529b,0x5299,0x5297,0x5296,
	0x5293,0x5292,0x5291,0x528e,0x528d,0x528b,0x5289,0x5287,0x5286,0x5284,0x5282,0x5281,0x527e,0x527d,0x527c,0x5279,
	0x5278,0x5276,0x5274,0x5272,0x5271,0x5270,0x526d,0x526c,0x526a,0x5268,0x5266,0x5265,0x5262,0x5261,0x5260,0x525d,
	0x525c,0x525a,0x5258,0x5257,0x5255,0x5253,0x5251,0x5250,0x524d,0x524c,0x524b,0x5248,0x5247,0x5245,0x5243,0x5241,
	0x5241,0x523e,0x523d,0x523b,0x5239,0x5237,0x5236,0x5233,0x5232,0x5231,0x522e,0x522d,0x522c,0x5229,0x5228,0x5226,
	0x5224,0x5222,0x5221,0x521e,0x521d,0x521c,0x5219,0x5218,0x5216,0x5214,0x5212,0x5211,0x5210,0x520d,0x520c,0x520b,
	0x5208,0x5207,0x5205,0x5203,0x5201,0x5200,0x51fd,0x51fc,0x51fb,0x51f8,0x51f7,0x51f5,0x51f3,0x51f1,0x51f0,0x51ee,
	0x51ec,0x51eb,0x51e8,0x51e7,0x51e6,0x51e4,0x51e2,0x51e0,0x51df,0x51dc,0x51db,0x51da,0x51d7,0x51d6,0x51d4,0x51d2,
	0x51d0,0x51cf,0x51cd,0x51cb,0x51ca,0x51c7,0x51c6,0x51c5,0x51c2,0x51c1,0x51bf,0x51bd,0x51bb,0x51ba,0x51b9,0x51b6,
	0x51b5,0x51b3,0x51b1,0x51af,0x51ae,0x51ac,0x51aa,0x51a9,0x51a6,0x51a5,0x51a4,0x51a1,0x51a0,0x519e,0x519c,0x519a,
	0x5199,0x5196,0x5195,0x5194,0x5191,0x5190,0x518e,0x518d,0x518a,0x5189,0x5188,0x5185,0x5184,0x5183,0x5180,0x517f,
	0x517d,0x517b,0x5179,0x5178,0x5175,0x5174,0x5173,0x5170,0x516f,0x516d,0x516b,0x5169,0x5168,0x5166,0x5164,0x5163,
	0x5161,0x515f,0x515d,0x515c,0x515a,0x5158,0x5157,0x5156,0x5153,0x5152,0x5150,0x514e,0x514c,0x514b,0x5148,0x5147,
	0x5146,0x5143,0x5142,0x5141,0x513e,0x513d,0x513b,0x5139,0x5137,0x5136,0x5135,0x5132,0x5131,0x512f,0x512d,0x512b,
	0x512a,0x5127,0x5126,0x5125,0x5122,0x5121,0x5120,0x511d,0x511c,0x511a,0x5118,0x5116,0x5115,0x5114,0x5111,0x5110,
	0x510e,0x510c,0x510b,0x5109,0x5107,0x5105,0x5104,0x5101,0x5100,0x50ff,0x50fc,0x50fb,0x50f9,0x50f7,0x50f5,0x50f4,
	0x50f3,0x50f0,0x50ef,0x50ee,0x50eb,0x50ea,0x50e8,0x50e6,0x50e4,0x50e3,0x50e0,0x50df,0x50de,0x50db,0x50da,0x50d9,
	0x50d6,0x50d5,0x50d3,0x50d1,0x50cf,0x50ce,0x50cd,0x50ca,0x50c9,0x50c7,0x50c5,0x50c3,0x50c2,0x50bf,0x50be,0x50bd,
	0x50ba,0x50b9,0x50b8,0x50b5,0x50b4,0x50b2,0x50b0,0x50ae,0x50ad,0x50ac,0x50a9,0x50a8,0x50a6,0x50a4,0x50a3,0x50a1,
	0x509f,0x509d,0x509c,0x5099,0x5098,0x5097,0x5094,0x5093,0x5091,0x508f,0x508d,0x508c,0x508b,0x5088,0x5087,0x5086,
	0x5083,0x5082,0x507f,0x507e,0x507d,0x507a,0x5079,0x5077,0x5076,0x5074,0x5072,0x5071,0x506e,0x506d,0x506c,0x5069,
	0x5068,0x5066,0x5064,0x5062,0x5061,0x505f,0x505d,0x505c,0x505b,0x5058,0x5057,0x5055,0x5053,0x5051,0x5050,0x504d,
	0x504c,0x504b,0x5048,0x5047,0x5046,0x5043,0x5042,0x5040,0x503f,0x503c,0x503b,0x503a,0x5037,0x5036,0x5035,0x5032,
	0x5031,0x502f,0x502d,0x502b,0x502a,0x5027,0x5026,0x5025,0x5023,0x5021,0x5020,0x501e,0x501c,0x501a,0x5019,0x5016,
	0x5015,0x5014,0x5011,0x5010,0x500e,0x500c,0x500b,0x5009,0x5008,0x5005,0x5004,0x5003,0x5000,0x4fff,0x4ffd,0x4ffb,
	0x4ff9,0x4ff8,0x4ff6,0x4ff4,0x4ff3,0x4ff0,0x4fef,0x4fee,0x4fec,0x4fea,0x4fe8,0x4fe7,0x4fe4,0x4fe3,0x4fe2,0x4fdf,
	0x4fde,0x4fdd,0x4fda,0x4fd9,0x4fd7,0x4fd5,0x4fd3,0x4fd2,0x4fd1,0x4fce,0x4fcd,0x4fcc,0x4fc9,0x4fc8,0x4fc6,0x4fc4,
	0x4fc2,0x4fc1,0x4fbe,0x4fbd,0x4fbc,0x4fb9,0x4fb8,0x4fb7,0x4fb5,0x4fb3,0x4fb1,0x4fb0,0x4fad,0x4fac,0x4fab,0x4fa8,
	0x4fa6,0x4fa5,0x4fa4,0x4fa1,0x4fa0,0x4f9e,0x4f9c,0x4f9a,0x4f99,0x4f96,0x4f95,0x4f94,0x4f93,0x4f90,0x4f8f,0x4f8d,
	0x4f8b,0x4f89,0x4f88,0x4f85,0x4f84,0x4f83,0x4f80,0x4f7f,0x4f7e,0x4f7c,0x4f7a,0x4f78,0x4f77,0x4f74,0x4f73,0x4f72,
	0x4f6f,0x4f6e,0x4f6d,0x4f6a,0x4f69,0x4f67,0x4f66,0x4f63,0x4f62,0x4f61,0x4f5e,0x4f5d,0x4f5c,0x4f59,0x4f58,0x4f56,
	0x4f54,0x4f52,0x4f51,0x4f50,0x4f4d,0x4f4c,0x4f4b,0x4f48,0x4f47,0x4f45,0x4f43,0x4f41,0x4f40,0x4f3e,0x4f3c,0x4f3b,
	0x4f3a,0x4f37,0x4f36,0x4f34,0x4f32,0x4f30,0x4f2f,0x4f2c,0x4f2b,0x4f2a,0x4f27,0x4f26,0x4f25,0x4f22,0x4f21,0x4f1f,
	0x4f1e,0x4f1b,0x4f1a,0x4f19,0x4f16,0x4f15,0x4f14,0x4f11,0x4f10,0x4f0e,0x4f0c,0x4f0a,0x4f09,0x4f08,0x4f05,0x4f04,
	0x4f03,0x4f00,0x4eff,0x4efd,0x4efb,0x4ef9,0x4ef8,0x4ef6,0x4ef4,0x4ef3,0x4ef2,0x4eef,0x4eee,0x4eec,0x4eea,0x4ee8,
	0x4ee7,0x4ee5,0x4ee3,0x4ee2,0x4edf,0x4ede,0x4edd,0x4edb,0x4ed9,0x4ed7,0x4ed6,0x4ed4,0x4ed2,0x4ed1,0x4ece,0x4ecd,
	0x4ecc,0x4ec9,0x4ec8,0x4ec6,0x4ec4,0x4ec2,0x4ec1,0x4ebe,0x4ebd,0x4ebc,0x4eb9,0x4eb8,0x4eb7,0x4eb5,0x4eb3,0x4eb1,
	0x4eb0,0x4eae,0x4eac,0x4eab,0x4ea8,0x4ea7,0x4ea6,0x4ea3,0x4ea2,0x4ea0,0x4e9f,0x4e9d,0x4e9b,0x4e9a,0x4e97,0x4e96,
	0x4e95,0x4e92,0x4e91,0x4e90,0x4e8e,0x4e8c,0x4e8a,0x4e89,0x4e86,0x4e85,0x4e84,0x4e81,0x4e80,0x4e7f,0x4e7c,0x4e7b,
	0x4e79,0x4e78,0x4e75,0x4e74,0x4e73,0x4e70,0x4e6f,0x4e6e,0x4e6b,0x4e6a,0x4e68,0x4e67,0x4e64,0x4e63,0x4e62,0x4e5f,
	0x4e5e,0x4e5d,0x4e5a,0x4e59,0x4e57,0x4e55,0x4e54,0x4e52,0x4e51,0x4e4e,0x4e4d,0x4e4c,0x4e49,0x4e48,0x4e46,0x4e44,
	0x4e43,0x4e41,0x4e40,0x4e3d,0x4e3c,0x4e3b,0x4e38,0x4e37,0x4e35,0x4e33,0x4e32,0x4e30,0x4e2e,0x4e2c,0x4e2b,0x4e2a,
	0x4e27,0x4e26,0x4e25,0x4e22,0x4e21,0x4e1f,0x4e1d,0x4e1b,0x4e1a,0x4e19,0x4e16,0x4e15,0x4e14,0x4e11,0x4e10,0x4e0e,
	0x4e0c,0x4e0a,0x4e09,0x4e07,0x4e05,0x4e04,0x4e03,0x4e00,0x4dff,0x4dfd,0x4dfb,0x4df9,0x4df8,0x4df6,0x4df4,0x4df3,
	0x4df2,0x4def,0x4dee,0x4dec,0x4dea,0x4de9,0x4de7,0x4de5,0x4de3,0x4de2,0x4ddf,0x4dde,0x4ddd,0x4ddb,0x4dd9,0x4dd8,
	0x4dd6,0x4dd4,0x4dd2,0x4dd1,0x4dce,0x4dcd,0x4dcc,0x4dc9,0x4dc8,0x4dc7,0x4dc5,0x4dc3,0x4dc1,0x4dc0,0x4dbd,0x4dbc,
	0x4dbb,0x4db8,0x4db7,0x4db6,0x4db4,0x4db2,0x4db0,0x4daf,0x4dad,0x4dab,0x4daa,0x4da7,0x4da6,0x4da5,0x4da2,0x4da1,
	0x4d9f,0x4d9e,0x4d9c,0x4d9a,0x4d99,0x4d96,0x4d95,0x4d94,0x4d91,0x4d90,0x4d8e,0x4d8d,0x4d8b,0x4d89,0x4d88,0x4d85,
	0x4d84,0x4d83,0x4d80,0x4d7f,0x4d7e,0x4d7b,0x4d7a,0x4d78,0x4d77,0x4d74,0x4d73,0x4d72,0x4d6f,0x4d6e,0x4d6d,0x4d6a,
	0x4d69,0x4d67,0x4d66,0x4d63,0x4d62,0x4d61,0x4d5e,0x4d5d,0x4d5c,0x4d59,0x4d58,0x4d56,0x4d54,0x4d52,0x4d51,0x4d50,
	0x4d4d,0x4d4c,0x4d4b,0x4d48,0x4d47,0x4d45,0x4d43,0x4d42,0x4d40,0x4d3f,0x4d3c,0x4d3b,0x4d3a,0x4d37,0x4d36,0x4d34,
	0x4d32,0x4d31,0x4d2f,0x4d2d,0x4d2b,0x4d2a,0x4d29,0x4d26,0x4d25,0x4d24,0x4d21,0x4d20,0x4d1e,0x4d1c,0x4d1a,0x4d19,
	0x4d17,0x4d16,0x4d15,0x4d12,0x4d11,0x4d10,0x4d0d,0x4d0c,0x4d0a,0x4d09,0x4d07,0x4d05,0x4d04,0x4d01,0x4d00,0x4cff,
	0x4cfc,0x4cfb,0x4cf9,0x4cf8,0x4cf6,0x4cf4,0x4cf3,0x4cf0,0x4cef,0x4cee,0x4ceb,0x4cea,0x4ce9,0x4ce7,0x4ce5,0x4ce3,
	0x4ce2,0x4cdf,0x4cde,0x4cdd,0x4cda,0x4cd9,0x4cd8,0x4cd6,0x4cd4,0x4cd2,0x4cd1,0x4ccf,0x4ccd,0x4ccc,0x4cc9,0x4cc8,
	0x4cc7,0x4cc5,0x4cc3,0x4cc2,0x4cc0,0x4cbe,0x4cbc,0x4cbb,0x4cb8,0x4cb7,0x4cb6,0x4cb5,0x4cb2,0x4cb1,0x4caf,0x4cad,
	0x4cab,0x4caa,0x4ca8,0x4ca6,0x4ca5,0x4ca2,0x4ca1,0x4ca0,0x4c9e,0x4c9c,0x4c9b,0x4c99,0x4c97,0x4c95,0x4c94,0x4c91,
	0x4c90,0x4c8f,0x4c8e,0x4c8b,0x4c8a,0x4c88,0x4c86,0x4c84,0x4c83,0x4c81,0x4c7f,0x4c7e,0x4c7d,0x4c7a,0x4c79,0x4c77,
	0x4c75,0x4c74,0x4c72,0x4c70,0x4c6e,0x4c6d,0x4c6c,0x4c69,0x4c68,0x4c67,0x4c64,0x4c63,0x4c61,0x4c5f,0x4c5d,0x4c5c,
	0x4c5b,0x4c58,0x4c57,0x4c56,0x4c53,0x4c52,0x4c50,0x4c4e,0x4c4c,0x4c4b,0x4c4a,0x4c47,0x4c46,0x4c45,0x4c42,0x4c41,
	0x4c40,0x4c3e,0x4c3c,0x4c3a,0x4c39,0x4c38,0x4c35,0x4c34,0x4c33,0x4c30,0x4c2f,0x4c2d,0x4c2b,0x4c2a,0x4c28,0x4c27,
	0x4c24,0x4c23,0x4c22,0x4c1f,0x4c1e,0x4c1d,0x4c1a,0x4c19,0x4c17,0x4c16,0x4c13,0x4c12,0x4c11,0x4c0e,0x4c0d,0x4c0c,
	0x4c0a,0x4c08,0x4c06,0x4c05,0x4c03,0x4c01,0x4c00,0x4bfd,0x4bfc,0x4bfb,0x4bfa,0x4bf7,0x4bf6,0x4bf4,0x4bf2,0x4bf0,
	0x4bef,0x4bed,0x4beb,0x4bea,0x4be9,0x4be6,0x4be5,0x4be3,0x4be1,0x4be0,0x4bde,0x4bdc,0x4bda,0x4bd9,0x4bd8,0x4bd5,
	0x4bd4,0x4bd3,0x4bd0,0x4bcf,0x4bcd,0x4bcc,0x4bca,0x4bc8,0x4bc7,0x4bc4,0x4bc3,0x4bc2,0x4bbf,0x4bbe,0x4bbd,0x4bbb,
	0x4bb9,0x4bb7,0x4bb6,0x4bb3,0x4bb2,0x4bb1,0x4bae,0x4bad,0x4bac,0x4baa,0x4ba8,0x4ba7,0x4ba5,0x4ba3,0x4ba1,0x4ba0,
	0x4b9f,0x4b9c,0x4b9b,0x4b9a,0x4b97,0x4b96,0x4b94,0x4b92,0x4b90,0x4b8f,0x4b8e,0x4b8b,0x4b8a,0x4b89,0x4b86,0x4b85,
	0x4b84,0x4b81,0x4b80,0x4b7e,0x4b7d,0x4b7a,0x4b79,0x4b78,0x4b75,0x4b74,0x4b73,0x4b70,0x4b6f,0x4b6d,0x4b6c,0x4b6a,
	0x4b69,0x4b66,0x4b65,0x4b64,0x4b62,0x4b60,0x4b5e,0x4b5d,0x4b5b,0x4b59,0x4b58,0x4b57,0x4b54,0x4b53,0x4b51,0x4b4f,
	0x4b4e,0x4b4c,0x4b4a,0x4b48,0x4b47,0x4b46,0x4b43,0x4b42,0x4b41,0x4b3e,0x4b3d,0x4b3b,0x4b3a,0x4b38,0x4b36,0x4b35,
	0x4b32,0x4b31,0x4b30,0x4b2d,0x4b2c,0x4b2b,0x4b29,0x4b27,0x4b25,0x4b24,0x4b22,0x4b20,0x4b1f,0x4b1e,0x4b1b,0x4b1a,
	0x4b19,0x4b16,0x4b15,0x4b13,0x4b12,0x4b10,0x4b0e,0x4b0d,0x4b0a,0x4b09,0x4b08,0x4b05,0x4b04,0x4b03,0x4b01,0x4aff,
	0x4afd,0x4afc,0x4afa,0x4af8,0x4af7,0x4af6,0x4af3,0x4af2,0x4af0,0x4aee,0x4aed,0x4aeb,0x4ae9,0x4ae7,0x4ae6,0x4ae5,
	0x4ae2,0x4ae1,0x4ae0,0x4add,0x4adc,0x4adb,0x4ad9,0x4ad7,0x4ad5,0x4ad4,0x4ad1,0x4ad0,0x4acf,0x4ace,0x4acb,0x4aca,
	0x4ac8,0x4ac6,0x4ac5,0x4ac3,0x4ac1,0x4abf,0x4abe,0x4abd,0x4aba,0x4ab9,0x4ab8,0x4ab5,0x4ab4,0x4ab2,0x4ab1,0x4aaf,
	0x4aad,0x4aac,0x4aa9,0x4aa8,0x4aa7,0x4aa4,0x4aa3,0x4aa2,0x4aa0,0x4a9e,0x4a9c,0x4a9b,0x4a99,0x4a97,0x4a96,0x4a95,
	0x4a93,0x4a90,0x4a8f,0x4a8e,0x4a8b,0x4a8a,0x4a89,0x4a86,0x4a85,0x4a83,0x4a82,0x4a80,0x4a7e,0x4a7d,0x4a7a,0x4a79,
	0x4a78,0x4a77,0x4a74,0x4a73,0x4a71,0x4a6f,0x4a6e,0x4a6c,0x4a6b,0x4a68,0x4a67,0x4a66,0x4a63,0x4a62,0x4a61,0x4a5f,
	0x4a5d,0x4a5c,0x4a5a,0x4a58,0x4a56,0x4a55,0x4a54,0x4a51,0x4a50,0x4a4f,0x4a4c,0x4a4b,0x4a49,0x4a47,0x4a46,0x4a44,
	0x4a43,0x4a40,0x4a3f,0x4a3e,0x4a3b,0x4a3a,0x4a39,0x4a37,0x4a35,0x4a34,0x4a32,0x4a30,0x4a2e,0x4a2d,0x4a2c,0x4a29,
	0x4a28,0x4a27,0x4a24,0x4a23,0x4a22,0x4a20,0x4a1e,0x4a1c,0x4a1b,0x4a18,0x4a17,0x4a16,0x4a15,0x4a12,0x4a11,0x4a0f,
	0x4a0d,0x4a0c,0x4a0a,0x4a09,0x4a06,0x4a05,0x4a04,0x4a01,0x4a00,0x49ff,0x49fc,0x49fb,0x49fa,0x49f8,0x49f6,0x49f4,
	0x49f3,0x49f1,0x49ef,0x49ee,0x49ed,0x49ea,0x49e9,0x49e8,0x49e5,0x49e4,0x49e2,0x49e1,0x49de,0x49dd,0x49dc,0x49d9,
	0x49d8,0x49d7,0x49d5,0x49d3,0x49d2,0x49d0,0x49ce,0x49cc,0x49cb,0x49ca,0x49c7,0x49c6,0x49c5,0x49c2,0x49c1,0x49c0,
	0x49bd,0x49bc,0x49ba,0x49b8,0x49b6,0x49b5,0x49b4,0x49b1,0x49b0,0x49af,0x49ac,0x49ab,0x49aa,0x49a8,0x49a6,0x49a5,
	0x49a3,0x49a1,0x499f,0x499e,0x499d,0x499a,0x4999,0x4998,0x4995,0x4994,0x4993,0x4991,0x498f,0x498d,0x498c,0x498a,
	0x4988,0x4987,0x4986,0x4983,0x4982,0x4981,0x497e,0x497d,0x497b,0x497a,0x4978,0x4976,0x4975,0x4972,0x4971,0x4970,
	0x496f,0x496c,0x496b,0x4969,0x4967,0x4966,0x4964,0x4963,0x4960,0x495f,0x495e,0x495b,0x495a,0x4959,0x4957,0x4955,
	0x4954,0x4952,0x4950,0x494e,0x494d,0x494c,0x4949,0x4948,0x4947,0x4944,0x4943,0x4942,0x4940,0x493e,0x493c,0x493b,
	0x4939,0x4937,0x4936,0x4935,0x4932,0x4931,0x4930,0x492d,0x492c,0x492a,0x4929,0x4927,0x4925,0x4924,0x4921,0x4920,
	0x491f,0x491e,0x491b,0x491a,0x4918,0x4916,0x4915,0x4913,0x4912,0x490f,0x490e,0x490d,0x490a,0x4909,0x4908,0x4906,
	0x4904,0x4903,0x4901,0x48ff,0x48fd,0x48fc,0x48fb,0x48f8,0x48f7,0x48f6,0x48f3,0x48f2,0x48f1,0x48ef,0x48ed,0x48eb,
	0x48ea,0x48e8,0x48e6,0x48e5,0x48e4,0x48e1,0x48e0,0x48df,0x48dc,0x48db,0x48d9,0x48d8,0x48d6,0x48d4,0x48d3,0x48d0,
	0x48cf,0x48ce,0x48cd,0x48ca,0x48c9,0x48c7,0x48c5,0x48c4,0x48c2,0x48c1,0x48be,0x48bd,0x48bc,0x48b9,0x48b8,0x48b7,
	0x48b5,0x48b3,0x48b2,0x48b0,0x48ae,0x48ac,0x48ab,0x48aa,0x48a7,0x48a6,0x48a5,0x48a2,0x48a1,0x48a0,0x489e,0x489c,
	0x489a,0x4899,0x4897,0x4895,0x4894,0x4893,0x4890,0x488f,0x488e,0x488b,0x488a,0x4888,0x4887,0x4885,0x4883,0x4882,
	0x4881,0x487e,0x487d,0x487c,0x4879,0x4878,0x4876,0x4875,0x4873,0x4871,0x4870,0x486d,0x486c,0x486b,0x486a,0x4867,
	0x4866,0x4864,0x4862,0x4861,0x485f,0x485e,0x485c,0x485a,0x4859,0x4856,0x4855,0x4854,0x4853,0x4850,0x484f,0x484d,
	0x484b,0x484a,0x4848,0x4847,0x4844,0x4843,0x4842,0x483f,0x483e,0x483d,0x483b,0x4839,0x4838,0x4836,0x4834,0x4832,
	0x4831,0x4830,0x482d,0x482c,0x482b,0x4828,0x4827,0x4826,0x4824,0x4822,0x4820,0x481f,0x481d,0x481b,0x481a,0x4819,
	0x4817,0x4816,0x4813,0x4812,0x4811,0x480f,0x480d,0x480c,0x480a,0x4808,0x4806,0x4805,0x4804,0x4801,0x4800,0x47ff,
	0x47fd,0x47fb,0x47fa,0x47f8,0x47f6,0x47f4,0x47f3,0x47f2,0x47ef,0x47ee,0x47ed,0x47ea,0x47e9,0x47e8,0x47e6,0x47e4,
	0x47e3,0x47e1,0x47df,0x47dd,0x47dc,0x47db,0x47d8,0x47d7,0x47d6,0x47d3,0x47d2,0x47d1,0x47cf,0x47cd,0x47cb,0x47ca,
	0x47c9,0x47c6,0x47c5,0x47c4,0x47c1,0x47c0,0x47bf,0x47bd,0x47bb,0x47ba,0x47b8,0x47b6,0x47b4,0x47b3,0x47b2,0x47af,
	0x47ae,0x47ad,0x47aa,0x47a9,0x47a8,0x47a6,0x47a4,0x47a2,0x47a1,0x479f,0x479d,0x479c,0x479b,0x4798,0x4797,0x4796,
	0x4794,0x4792,0x4791,0x478f,0x478d,0x478b,0x478a,0x4789,0x4786,0x4785,0x4784,0x4781,0x4780,0x477f,0x477d,0x477b,
	0x4779,0x4778,0x4776,0x4774,0x4773,0x4772,0x476f,0x476e,0x476d,0x476a,0x4769,0x4768,0x4766,0x4764,0x4762,0x4761,
	0x4760,0x475d,0x475c,0x475b,0x4758,0x4757,0x4756,0x4754,0x4752,0x4750,0x474f,0x474d,0x474b,0x474a,0x4749,0x4746,
	0x4745,0x4744,0x4742,0x4740,0x473f,0x473d,0x473c,0x4739,0x4738,0x4737,0x4734,0x4733,0x4732,0x4731,0x472e,0x472d,
	0x472b,0x472a,0x4728,0x4726,0x4725,0x4722,0x4721,0x4720,0x471f,0x471c,0x471b,0x471a,0x4717,0x4716,0x4714,0x4713,
	0x4711,0x470f,0x470e,0x470d,0x470a,0x4709,0x4708,0x4705,0x4704,0x4703,0x4701,0x46ff,0x46fd,0x46fc,0x46fa,0x46f8,
	0x46f7,0x46f6,0x46f3,0x46f2,0x46f1,0x46ef,0x46ed,0x46ec,0x46ea,0x46e8,0x46e6,0x46e5,0x46e4,0x46e1,0x46e0,0x46df,
	0x46de,0x46db,0x46da,0x46d8,0x46d6,0x46d5,0x46d3,0x46d2,0x46cf,0x46ce,0x46cd,0x46ca,0x46c9,0x46c8,0x46c7,0x46c4,
	0x46c3,0x46c1,0x46c0,0x46be,0x46bc,0x46bb,0x46b8,0x46b7,0x46b6,0x46b5,0x46b2,0x46b1,0x46b0,0x46ad,0x46ac,0x46aa,
	0x46a9,0x46a7,0x46a5,0x46a4,0x46a3,0x46a0,0x469f,0x469e,0x469b,0x469a,0x4699,0x4697,0x4695,0x4693,0x4692,0x4690,
	0x468e,0x468d,0x468c,0x4689,0x4688,0x4687,0x4685,0x4683,0x4682,0x4680,0x467e,0x467c,0x467b,0x467a,0x4677,0x4676,
	0x4675,0x4673,0x4671,0x4670,0x466e,0x466c,0x466b,0x466a,0x4667,0x4666,0x4665,0x4662,0x4661,0x4660,0x465e,0x465c,
	0x465a,0x4659,0x4658,0x4655,0x4654,0x4653,0x4650,0x464f,0x464e,0x464c,0x464a,0x4649,0x4647,0x4646,0x4643,0x4642,
	0x4641,0x463e,0x463d,0x463c,0x463b,0x4638,0x4637,0x4635,0x4634,0x4632,0x4630,0x462f,0x462d,0x462b,0x462a,0x4629,
	0x4626,0x4625,0x4624,0x4622,0x4620,0x461f,0x461d,0x461b,0x4619,0x4618,0x4617,0x4614,0x4613,0x4612,0x4611,0x460e,
	0x460d,0x460b,0x4609,0x4608,0x4606,0x4605,0x4603,0x4601,0x4600,0x45ff,0x45fc,0x45fb,0x45fa,0x45f7,0x45f6,0x45f5,
	0x45f3,0x45f1,0x45ef,0x45ee,0x45ed,0x45ea,0x45e9,0x45e8,0x45e5,0x45e4,0x45e3,0x45e1,0x45df,0x45de,0x45dc,0x45db,
	0x45d9,0x45d7,0x45d6,0x45d3,0x45d2,0x45d1,0x45d0,0x45cd,0x45cc,0x45cb,0x45c9,0x45c7,0x45c5,0x45c4,0x45c2,0x45c0,
	0x45bf,0x45be,0x45bb,0x45ba,0x45b9,0x45b7,0x45b5,0x45b4,0x45b2,0x45b0,0x45af,0x45ad,0x45ac,0x45a9,0x45a8,0x45a7,
	0x45a5,0x45a3,0x45a1,0x45a0,0x459f,0x459c,0x459b,0x459a,0x4598,0x4596,0x4595,0x4593,0x4592,0x458f,0x458e,0x458d,
	0x458a,0x4589,0x4588,0x4587,0x4584,0x4583,0x4581,0x4580,0x457e,0x457c,0x457b,0x4579,0x4577,0x4576,0x4575,0x4572,
	0x4571,0x4570,0x456e,0x456c,0x456b,0x4569,0x4568,0x4566,0x4564,0x4563,0x4560,0x455f,0x455e,0x455d,0x455a,0x4559,
	0x4558,0x4556,0x4554,0x4553,0x4551,0x454f,0x454d,0x454c,0x454b,0x4548,0x4547,0x4546,0x4545,0x4542,0x4541,0x453f,
	0x453e,0x453c,0x453a,0x4539,0x4537,0x4535,0x4534,0x4533,0x4530,0x452f,0x452e,0x452c,0x452a,0x4529,0x4527,0x4525,
	0x4524,0x4522,0x4521,0x451e,0x451d,0x451c,0x451b,0x4518,0x4517,0x4516,0x4513,0x4512,0x4511,0x450f,0x450d,0x450b,
	0x450a,0x4509,0x4506,0x4505,0x4504,0x4503,0x4500,0x44ff,0x44fd,0x44fb,0x44fa,0x44f8,0x44f7,0x44f5,0x44f3,0x44f2,
	0x44f1,0x44ee,0x44ed,0x44ec,0x44e9,0x44e8,0x44e7,0x44e5,0x44e3,0x44e2,0x44e0,0x44df,0x44dc,0x44db,0x44da,0x44d9,
	0x44d6,0x44d5,0x44d2,0x44d1,0x44d0,0x44ce,0x44cc,0x44cb,0x44c9,0x44c8,0x44c5,0x44c4,0x44c3,0x44c2,0x44bf,0x44be,
	0x44bd,0x44ba,0x44b9,0x44b8,0x44b6,0x44b4,0x44b2,0x44b1,0x44b0,0x44ad,0x44ac,0x44ab,0x44aa,0x44a7,0x44a6,0x44a5,
	0x44a3,0x44a1,0x449f,0x449e,0x449c,0x449a,0x4499,0x4498,0x4495,0x4494,0x4493,0x4492,0x448f,0x448e,0x448c,0x448b,
	0x4489,0x4487,0x4486,0x4484,0x4482,0x4481,0x4480,0x447d,0x447c,0x447b,0x447a,0x4477,0x4476,0x4474,0x4473,0x4471,
	0x446f,0x446e,0x446c,0x446a,0x4469,0x4468,0x4465,0x4464,0x4463,0x4461,0x445f,0x445e,0x445c,0x445b,0x4459,0x4457,
	0x4456,0x4454,0x4452,0x4451,0x4450,0x444d,0x444c,0x444b,0x4449,0x4447,0x4446,0x4444,0x4443,0x4441,0x443f,0x443e,
	0x443d,0x443a,0x4439,0x4438,0x4435,0x4434,0x4433,0x4431,0x442f,0x442e,0x442c,0x442b,0x4429,0x4427,0x4426,0x4425,
	0x4422,0x4421,0x4420,0x441d,0x441c,0x441b,0x4419,0x4417,0x4416,0x4414,0x4413,0x4410,0x440f,0x440e,0x440d,0x440a,
	0x4409,0x4408,0x4405,0x4404,0x4403,0x4401,0x43ff,0x43fd,0x43fc,0x43fb,0x43f8,0x43f7,0x43f6,0x43f5,0x43f2,0x43f1,
	0x43f0,0x43ed,0x43ec,0x43ea,0x43e9,0x43e7,0x43e5,0x43e4,0x43e3,0x43e0,0x43df,0x43de,0x43dd,0x43da,0x43d9,0x43d8,
	0x43d6,0x43d4,0x43d2,0x43d1,0x43cf,0x43cd,0x43cc,0x43cb,0x43c8,0x43c7,0x43c6,0x43c5,0x43c2,0x43c1,0x43bf,0x43be,
	0x43bc,0x43ba,0x43b9,0x43b7,0x43b5,0x43b4,0x43b3,0x43b0,0x43af,0x43ae,0x43ac,0x43aa,0x43a9,0x43a7,0x43a6,0x43a4,
	0x43a2,0x43a1,0x439f,0x439d,0x439c,0x439b,0x4398,0x4397,0x4396,0x4394,0x4392,0x4391,0x438f,0x438e,0x438c,0x438a,
	0x4389,0x4387,0x4385,0x4384,0x4383,0x4380,0x437f,0x437e,0x437c,0x437a,0x4379,0x4377,0x4376,0x4374,0x4372,0x4371,
	0x4370,0x436d,0x436c,0x436b,0x4368,0x4367,0x4366,0x4364,0x4362,0x4361,0x435f,0x435e,0x435b,0x435a,0x4359,0x4358,
	0x4355,0x4354,0x4353,0x4350,0x434f,0x434e,0x434c,0x434a,0x4348,0x4347,0x4346,0x4343,0x4342,0x4341,0x4340,0x433d,
	0x433d,0x433a,0x4339,0x4338,0x4336,0x4334,0x4332,0x4331,0x4330,0x432d,0x432c,0x432b,0x432a,0x4327,0x4326,0x4325,
	0x4323,0x4321,0x4320,0x431e,0x431d,0x431a,0x4319,0x4318,0x4315,0x4314,0x4313,0x4312,0x430f,0x430e,0x430d,0x430b,
	0x4309,0x4308,0x4306,0x4305,0x4303,0x4301,0x4300,0x42ff,0x42fc,0x42fb,0x42fa,0x42f7,0x42f6,0x42f5,0x42f3,0x42f1,
	0x42f0,0x42ee,0x42ed,0x42eb,0x42e9,0x42e8,0x42e7,0x42e4,0x42e3,0x42e2,0x42e0,0x42de,0x42dd,0x42db,0x42da,0x42d8,
	0x42d6,0x42d5,0x42d3,0x42d1,0x42d0,0x42cf,0x42cc,0x42cb,0x42ca,0x42c8,0x42c6,0x42c5,0x42c3,0x42c2,0x42c0,0x42be,
	0x42bd,0x42bc,0x42b9,0x42b8,0x42b7,0x42b6,0x42b3,0x42b2,0x42b1,0x42ae,0x42ad,0x42ab,0x42aa,0x42a8,0x42a6,0x42a5,
	0x42a4,0x42a1,0x42a0,0x429f,0x429e,0x429b,0x429a,0x4299,0x4297,0x4295,0x4293,0x4292,0x4290,0x428e,0x428d,0x428c,
	0x4289,0x4288,0x4287,0x4286,0x4283,0x4282,0x4281,0x427f,0x427d,0x427c,0x427a,0x4279,0x4276,0x4275,0x4274,0x4273,
	0x4271,0x426f,0x426d,0x426c,0x426a,0x4268,0x4267,0x4266,0x4263,0x4262,0x4261,0x4260,0x425d,0x425c,0x425b,0x4259,
	0x4257,0x4256,0x4254,0x4253,0x4251,0x424f,0x424e,0x424d,0x424a,0x4249,0x4248,0x4246,0x4244,0x4243,0x4241,0x4240,
	0x423e,0x423c,0x423b,0x4239,0x4237,0x4236,0x4235,0x4232,0x4231,0x4230,0x422f,0x422c,0x422b,0x422a,0x4228,0x4226,
	0x4224,0x4223,0x4222,0x421f,0x421e,0x421d,0x421c,0x4219,0x4218,0x4217,0x4215,0x4213,0x4212,0x4210,0x420f,0x420d,
	0x420b,0x420a,0x4208,0x4206,0x4205,0x4204,0x4201,0x4200,0x41ff,0x41fd,0x41fb,0x41fa,0x41f8,0x41f7,0x41f5,0x41f3,
	0x41f2,0x41f1,0x41ee,0x41ed,0x41ec,0x41eb,0x41e8,0x41e7,0x41e6,0x41e4,0x41e2,0x41e1,0x41df,0x41de,0x41dc,0x41da,
	0x41d9,0x41d6,0x41d5,0x41d4,0x41d3,0x41d0,0x41cf,0x41ce,0x41cc,0x41ca,0x41c9,0x41c7,0x41c6,0x41c4,0x41c2,0x41c1,
	0x41c0,0x41bd,0x41bc,0x41bb,0x41ba,0x41b7,0x41b6,0x41b5,0x41b3,0x41b1,0x41b0,0x41ae,0x41ad,0x41aa,0x41a9,0x41a8,
	0x41a6,0x41a4,0x41a3,0x41a1,0x419f,0x419e,0x419d,0x419a,0x4199,0x4198,0x4196,0x4194,0x4193,0x4191,0x4190,0x418e,
	0x418c,0x418b,0x418a,0x4187,0x4186,0x4185,0x4184,0x4181,0x4180,0x417f,0x417d,0x417b,0x417a,0x4178,0x4177,0x4175,
	0x4173,0x4172,0x4171,0x416e,0x416d,0x416c,0x416b,0x4168,0x4167,0x4166,0x4164,0x4162,0x4161,0x415f,0x415e,0x415c,
	0x415a,0x4159,0x4158,0x4155,0x4154,0x4153,0x4152,0x414f,0x414e,0x414c,0x414b,0x4149,0x4147,0x4146,0x4145,0x4142,
	0x4141,0x4140,0x413d,0x413c,0x413b,0x413a,0x4137,0x4136,0x4135,0x4133,0x4131,0x4130,0x412e,0x412d,0x412b,0x4129,
	0x4128,0x4127,0x4124,0x4123,0x4122,0x4121,0x411e,0x411d,0x411c,0x411a,0x4118,0x4117,0x4115,0x4114,0x4112,0x4110,
	0x410f,0x410e,0x410b,0x410a,0x4109,0x4108,0x4105,0x4104,0x4103,0x4101,0x40ff,0x40fd,0x40fc,0x40fb,0x40f8,0x40f7,
	0x40f6,0x40f5,0x40f2,0x40f1,0x40f0,0x40ee,0x40ec,0x40eb,0x40e9,0x40e8,0x40e6,0x40e4,0x40e3,0x40e2,0x40df,0x40de,
	0x40dc,0x40db,0x40da,0x40d7,0x40d6,0x40d5,0x40d3,0x40d1,0x40d0,0x40ce,0x40cd,0x40cb,0x40c9,0x40c8,0x40c7,0x40c4,
	0x40c3,0x40c2,0x40c1,0x40be,0x40bd,0x40bc,0x40ba,0x40b8,0x40b7,0x40b5,0x40b4,0x40b2,0x40b0,0x40af,0x40ae,0x40ab,
	0x40aa,0x40a9,0x40a8,0x40a5,0x40a4,0x40a3,0x40a1,0x409f,0x409e,0x409c,0x409b,0x4099,0x4097,0x4096,0x4095,0x4092,
	0x4091,0x4090,0x408f,0x408c,0x408b,0x408a,0x4088,0x4086,0x4085,0x4083,0x4082,0x4080,0x407e,0x407d,0x407c,0x4079,
	0x4078,0x4077,0x4076,0x4073,0x4072,0x4071,0x406f,0x406d,0x406c,0x406a,0x4069,0x4067,0x4065,0x4064,0x4063,0x4060,
	0x405f,0x405e,0x405d,0x405a,0x4059,0x4058,0x4056,0x4054,0x4053,0x4051,0x4050,0x404e,0x404c,0x404b,0x404a,0x4047,
	0x4046,0x4045,0x4044,0x4041,0x4040,0x403f,0x403d,0x403b,0x403a,0x4038,0x4037,0x4035,0x4033,0x4032,0x4031,0x402e,
	0x402d,0x402c,0x402b,0x4028,0x4027,0x4026,0x4024,0x4022,0x4021,0x401f,0x401e,0x401c,0x401a,0x4019,0x4018,0x4015,
	0x4014,0x4013,0x4012,0x400f,0x400e,0x400d,0x400b,0x4009,0x4008,0x4006,0x4005,0x4003,0x4001,0x4000,0x3fff,0x3ffc,
	0x3ffb,0x3ffa,0x3ff9,0x3ff6,0x3ff5,0x3ff4,0x3ff2,0x3ff0,0x3fef,0x3fed,0x3fec,0x3fea,0x3fe8,0x3fe7,0x3fe6,0x3fe3,
	0x3fe2,0x3fe1,0x3fe0,0x3fdd,0x3fdc,0x3fdb,0x3fd9,0x3fd7,0x3fd6,0x3fd4,0x3fd3,0x3fd1,0x3fcf,0x3fce,0x3fcd,0x3fca,
	0x3fc9,0x3fc8,0x3fc7,0x3fc4,0x3fc3,0x3fc2,0x3fc0,0x3fbe,0x3fbd,0x3fbb,0x3fba,0x3fb8,0x3fb6,0x3fb5,0x3fb4,0x3fb1,
	0x3fb0,0x3faf,0x3fae,0x3fab,0x3faa,0x3fa9,0x3fa7,0x3fa5,0x3fa4,0x3fa2,0x3fa1,0x3f9f,0x3f9d,0x3f9c,0x3f9b,0x3f98,
	0x3f97,0x3f96,0x3f95,0x3f92,0x3f91,0x3f90,0x3f8e,0x3f8c,0x3f8b,0x3f89,0x3f88,0x3f86,0x3f84,0x3f83,0x3f82,0x3f7f,
	0x3f7e,0x3f7d,0x3f7c,0x3f79,0x3f78,0x3f77,0x3f75,0x3f73,0x3f72,0x3f70,0x3f6f,0x3f6d,0x3f6b,0x3f6a,0x3f69,0x3f66,
	0x3f65,0x3f64,0x3f63,0x3f60,0x3f5f,0x3f5e,0x3f5c,0x3f5a,0x3f59,0x3f57,0x3f56,0x3f54,0x3f52,0x3f51,0x3f50,0x3f4d,
	0x3f4c,0x3f4b,0x3f4a,0x3f48,0x3f46,0x3f45,0x3f43,0x3f42,0x3f40,0x3f3e,0x3f3d,0x3f3c,0x3f39,0x3f38,0x3f37,0x3f36,
	0x3f33,0x3f32,0x3f31,0x3f2f,0x3f2d,0x3f2c,0x3f2a,0x3f29,0x3f27,0x3f25,0x3f24,0x3f23,0x3f20,0x3f1f,0x3f1e,0x3f1d,
	0x3f1a,0x3f19,0x3f18,0x3f16,0x3f14,0x3f13,0x3f11,0x3f10,0x3f0e,0x3f0c,0x3f0b,0x3f0a,0x3f07,0x3f06,0x3f05,0x3f04,
	0x3f01,0x3f00,0x3eff,0x3efe,0x3efb,0x3efa,0x3ef9,0x3ef7,0x3ef5,0x3ef4,0x3ef2,0x3ef1,0x3eef,0x3eed,0x3eec,0x3eeb,
	0x3eea,0x3ee7,0x3ee6,0x3ee5,0x3ee3,0x3ee1,0x3ee0,0x3ede,0x3edd,0x3edb,0x3ed9,0x3ed8,0x3ed7,0x3ed4,0x3ed3,0x3ed2,
	0x3ed1,0x3ece,0x3ecd,0x3ecc,0x3eca,0x3ec8,0x3ec7,0x3ec5,0x3ec4,0x3ec2,0x3ec0,0x3ebf,0x3ebe,0x3ebb,0x3eba,0x3eb9,
	0x3eb8,0x3eb5,0x3eb4,0x3eb3,0x3eb1,0x3eaf,0x3eae,0x3ead,0x3eab,0x3ea9,0x3ea8,0x3ea6,0x3ea5,0x3ea3,0x3ea1,0x3ea0,
	0x3e9f,0x3e9c,0x3e9b,0x3e9a,0x3e99,0x3e96,0x3e95,0x3e94,0x3e92,0x3e90,0x3e8f,0x3e8d,0x3e8c,0x3e8a,0x3e88,0x3e87,
	0x3e86,0x3e84,0x3e82,0x3e81,0x3e80,0x3e7d,0x3e7c,0x3e7b,0x3e7a,0x3e77,0x3e76,0x3e75,0x3e74,0x3e71,0x3e70,0x3e6f,
	0x3e6d,0x3e6b,0x3e6a,0x3e68,0x3e67,0x3e65,0x3e63,0x3e62,0x3e61,0x3e5e,0x3e5d,0x3e5c,0x3e5b,0x3e58,0x3e57,0x3e56,
	0x3e54,0x3e53,0x3e51,0x3e50,0x3e4e,0x3e4d,0x3e4b,0x3e49,0x3e48,0x3e47,0x3e44,0x3e43,0x3e42,0x3e41,0x3e3e,0x3e3d,
	0x3e3c,0x3e3a,0x3e38,0x3e37,0x3e35,0x3e34,0x3e32,0x3e30,0x3e2f,0x3e2e,0x3e2b,0x3e2a,0x3e29,0x3e28,0x3e25,0x3e24,
	0x3e23,0x3e22,0x3e20,0x3e1e,0x3e1d,0x3e1b,0x3e1a,0x3e18,0x3e16,0x3e15,0x3e14,0x3e11,0x3e10,0x3e0f,0x3e0e,0x3e0b,
	0x3e0a,0x3e09,0x3e07,0x3e05,0x3e04,0x3e02,0x3e01,0x3dff,0x3dfe,0x3dfc,0x3dfb,0x3df9,0x3df7,0x3df6,0x3df5,0x3df2,
	0x3df1,0x3df0,0x3def,0x3ded,0x3deb,0x3dea,0x3de8,0x3de7,0x3de5,0x3de3,0x3de2,0x3de1,0x3dde,0x3ddd,0x3ddc,0x3ddb,
	0x3dd8,0x3dd7,0x3dd6,0x3dd5,0x3dd2,0x3dd1,0x3dd0,0x3dce,0x3dcc,0x3dcb,0x3dc9,0x3dc8,0x3dc6,0x3dc4,0x3dc3,0x3dc2,
	0x3dc0,0x3dbf,0x3dbc,0x3dbb,0x3dba,0x3db9,0x3db6,0x3db5,0x3db4,0x3db2,0x3db0,0x3daf,0x3dad,0x3dac,0x3daa,0x3da8,
	0x3da7,0x3da6,0x3da3,0x3da2,0x3da1,0x3da0,0x3d9f,0x3d9c,0x3d9b,0x3d9a,0x3d98,0x3d96,0x3d95,0x3d93,0x3d92,0x3d90,
	0x3d8e,0x3d8d,0x3d8c,0x3d89,0x3d88,0x3d87,0x3d86,0x3d83,0x3d82,0x3d81,0x3d80,0x3d7e,0x3d7c,0x3d7b,0x3d79,0x3d78,
	0x3d76,0x3d74,0x3d73,0x3d72,0x3d6f,0x3d6e,0x3d6d,0x3d6c,0x3d69,0x3d68,0x3d67,0x3d66,0x3d63,0x3d62,0x3d61,0x3d5f,
	0x3d5e,0x3d5c,0x3d5a,0x3d59,0x3d58,0x3d55,0x3d54,0x3d53,0x3d52,0x3d4f,0x3d4e,0x3d4d,0x3d4c,0x3d49,0x3d48,0x3d47,
	0x3d45,0x3d43,0x3d42,0x3d40,0x3d3f,0x3d3d,0x3d3b,0x3d3a,0x3d39,0x3d38,0x3d35,0x3d34,0x3d33,0x3d32,0x3d2f,0x3d2e,
	0x3d2d,0x3d2b,0x3d29,0x3d28,0x3d26,0x3d25,0x3d23,0x3d21,0x3d20,0x3d1f,0x3d1c,0x3d1b,0x3d1a,0x3d19,0x3d18,0x3d15,
	0x3d14,0x3d13,0x3d11,0x3d0f,0x3d0e,0x3d0c,0x3d0b,0x3d09,0x3d07,0x3d06,0x3d05,0x3d02,0x3d01,0x3d00,0x3cff,0x3cfc,
	0x3cfb,0x3cfa,0x3cf9,0x3cf7,0x3cf5,0x3cf4,0x3cf2,0x3cf1,0x3cef,0x3ced,0x3cec,0x3ceb,0x3ce8,0x3ce7,0x3ce6,0x3ce5,
	0x3ce2,0x3ce1,0x3ce0,0x3cdf,0x3cdc,0x3cdb,0x3cda,0x3cd8,0x3cd7,0x3cd5,0x3cd3,0x3cd2,0x3cd1,0x3cce,0x3ccd,0x3ccc,
	0x3ccb,0x3cc8,0x3cc7,0x3cc6,0x3cc5,0x3cc2,0x3cc1,0x3cc0,0x3cbe,0x3cbc,0x3cbb,0x3cb9,0x3cb8,0x3cb7,0x3cb4,0x3cb3,
	0x3cb2,0x3cb1,0x3cae,0x3cad,0x3cac,0x3cab,0x3ca8,0x3ca7,0x3ca6,0x3ca4,0x3ca2,0x3ca1,0x3c9f,0x3c9e,0x3c9c,0x3c9a,
	0x3c99,0x3c98,0x3c97,0x3c94,0x3c93,0x3c92,0x3c91,0x3c8e,0x3c8d,0x3c8c,0x3c8a,0x3c88,0x3c87,0x3c85,0x3c84,0x3c82,
	0x3c80,0x3c7f,0x3c7e,0x3c7c,0x3c7a,0x3c79,0x3c78,0x3c77,0x3c74,0x3c73,0x3c72,0x3c70,0x3c6e,0x3c6d,0x3c6b,0x3c6a,
	0x3c68,0x3c66,0x3c65,0x3c64,0x3c62,0x3c60,0x3c5f,0x3c5e,0x3c5b,0x3c5a,0x3c59,0x3c58,0x3c56,0x3c54,0x3c53,0x3c51,
	0x3c50,0x3c4e,0x3c4c,0x3c4b,0x3c4a,0x3c47,0x3c46,0x3c45,0x3c44,0x3c41,0x3c40,0x3c3f,0x3c3e,0x3c3b,0x3c3a,0x3c39,
	0x3c38,0x3c36,0x3c35,0x3c33,0x3c31,0x3c30,0x3c2f,0x3c2c,0x3c2b,0x3c2a,0x3c29,0x3c26,0x3c25,0x3c24,0x3c23,0x3c20,
	0x3c1f,0x3c1e,0x3c1c,0x3c1b,0x3c19,0x3c17,0x3c16,0x3c15,0x3c13,0x3c11,0x3c10,0x3c0f,0x3c0c,0x3c0b,0x3c0a,0x3c09,
	0x3c07,0x3c05,0x3c04,0x3c02,0x3c01,0x3bff,0x3bfe,0x3bfc,0x3bfb,0x3bf9,0x3bf7,0x3bf6,0x3bf5,0x3bf2,0x3bf1,0x3bf0,
	0x3bef,0x3bed,0x3beb,0x3bea,0x3be9,0x3be7,0x3be5,0x3be4,0x3be2,0x3be1,0x3bdf,0x3bdd,0x3bdc,0x3bdb,0x3bd9,0x3bd7,
	0x3bd6,0x3bd5,0x3bd4,0x3bd1,0x3bd0,0x3bcf,0x3bcd,0x3bcb,0x3bca,0x3bc8,0x3bc7,0x3bc5,0x3bc4,0x3bc2,0x3bc1,0x3bbf,
	0x3bbd,0x3bbc,0x3bbb,0x3bba,0x3bb7,0x3bb6,0x3bb5,0x3bb3,0x3bb1,0x3bb0,0x3baf,0x3bad,0x3bab,0x3baa,0x3ba8,0x3ba7,
	0x3ba6,0x3ba3,0x3ba2,0x3ba1,0x3ba0,0x3b9d,0x3b9c,0x3b9b,0x3b9a,0x3b97,0x3b96,0x3b95,0x3b93,0x3b91,0x3b90,0x3b8e,
	0x3b8d,0x3b8c,0x3b8a,0x3b88,0x3b87,0x3b86,0x3b83,0x3b82,0x3b81,0x3b80,0x3b7d,0x3b7c,0x3b7b,0x3b79,0x3b77,0x3b76,
	0x3b75,0x3b74,0x3b71,0x3b70,0x3b6f,0x3b6e,0x3b6b,0x3b6a,0x3b69,0x3b67,0x3b65,0x3b64,0x3b62,0x3b61,0x3b60,0x3b5e,
	0x3b5c,0x3b5b,0x3b5a,0x3b57,0x3b56,0x3b55,0x3b54,0x3b51,0x3b50,0x3b4f,0x3b4e,0x3b4c,0x3b4a,0x3b49,0x3b47,0x3b46,
	0x3b44,0x3b42,0x3b41,0x3b40,0x3b3e,0x3b3c,0x3b3b,0x3b3a,0x3b39,0x3b36,0x3b35,0x3b34,0x3b32,0x3b30,0x3b2f,0x3b2e,
	0x3b2c,0x3b2a,0x3b29,0x3b27,0x3b26,0x3b25,0x3b22,0x3b21,0x3b20,0x3b1f,0x3b1c,0x3b1b,0x3b1a,0x3b19,0x3b16,0x3b15,
	0x3b14,0x3b12,0x3b11,0x3b0f,0x3b0e,0x3b0c,0x3b0b,0x3b09,0x3b07,0x3b06,0x3b05,0x3b02,0x3b01,0x3b00,0x3aff,0x3afe,
	0x3afb,0x3afa,0x3af9,0x3af7,0x3af5,0x3af4,0x3af2,0x3af1,0x3aef,0x3aee,0x3aec,0x3aeb,0x3aea,0x3ae7,0x3ae6,0x3ae5,
	0x3ae4,0x3ae1,0x3ae0,0x3adf,0x3ade,0x3adb,0x3ada,0x3ad9,0x3ad7,0x3ad6,0x3ad4,0x3ad2,0x3ad1,0x3ad0,0x3ace,0x3acc,
	0x3acb,0x3aca,0x3ac7,0x3ac6,0x3ac5,0x3ac4,0x3ac2,0x3ac0,0x3abf,0x3abe,0x3abc,0x3aba,0x3ab9,0x3ab7,0x3ab6,0x3ab4,
	0x3ab3,0x3ab1,0x3aaf,0x3aae,0x3aad,0x3aaa,0x3aa9,0x3aa8,0x3aa7,0x3aa4,0x3aa3,0x3aa2,0x3aa1,0x3a9f,0x3a9d,0x3a9c,
	0x3a9b,0x3a99,0x3a97,0x3a96,0x3a94,0x3a93,0x3a92,0x3a90,0x3a8e,0x3a8d,0x3a8c,0x3a89,0x3a88,0x3a87,0x3a86,0x3a83,
	0x3a82,0x3a81,0x3a80,0x3a7e,0x3a7c,0x3a7b,0x3a79,0x3a78,0x3a76,0x3a75,0x3a73,0x3a72,0x3a71,0x3a6e,0x3a6d,0x3a6c,
	0x3a6b,0x3a68,0x3a67,0x3a66,0x3a65,0x3a62,0x3a61,0x3a60,0x3a5e,0x3a5d,0x3a5b,0x3a5a,0x3a58,0x3a57,0x3a55,0x3a53,
	0x3a52,0x3a51,0x3a50,0x3a4d,0x3a4c,0x3a4b,0x3a4a,0x3a47,0x3a46,0x3a45,0x3a43,0x3a42,0x3a40,0x3a3f,0x3a3d,0x3a3c,
	0x3a3a,0x3a38,0x3a37,0x3a36,0x3a34,0x3a32,0x3a31,0x3a30,0x3a2f,0x3a2c,0x3a2b,0x3a2a,0x3a28,0x3a26,0x3a25,0x3a24,
	0x3a22,0x3a21,0x3a1f,0x3a1d,0x3a1c,0x3a1b,0x3a19,0x3a17,0x3a16,0x3a15,0x3a12,0x3a11,0x3a10,0x3a0f,0x3a0d,0x3a0b,
	0x3a0a,0x3a09,0x3a07,0x3a05,0x3a04,0x3a02,0x3a01,0x3a00,0x39fe,0x39fc,0x39fb,0x39fa,0x39f7,0x39f6,0x39f5,0x39f4,
	0x39f3,0x39f0,0x39ef,0x39ee,0x39ec,0x39ea,0x39e9,0x39e7,0x39e6,0x39e4,0x39e3,0x39e1,0x39e0,0x39df,0x39dc,0x39db,
	0x39da,0x39d9,0x39d6,0x39d5,0x39d4,0x39d3,0x39d1,0x39cf,0x39ce,0x39cc,0x39cb,0x39c9,0x39c8,0x39c6,0x39c5,0x39c3,
	0x39c1,0x39c0,0x39bf,0x39be,0x39bb,0x39ba,0x39b9,0x39b8,0x39b5,0x39b4,0x39b3,0x39b2,0x39b0,0x39ae,0x39ad,0x39ab,
	0x39aa,0x39a8,0x39a6,0x39a5,0x39a4,0x39a2,0x39a0,0x399f,0x399e,0x399d,0x399a,0x3999,0x3998,0x3997,0x3994,0x3993,
	0x3992,0x3990,0x398f,0x398d,0x398b,0x398a,0x3989,0x3987,0x3985,0x3984,0x3983,0x3982,0x397f,0x397e,0x397d,0x397c,
	0x3979,0x3978,0x3977,0x3975,0x3973,0x3972,0x3970,0x396f,0x396e,0x396c,0x396a,0x3969,0x3968,0x3965,0x3964,0x3963,
	0x3962,0x3961,0x395e,0x395d,0x395c,0x395a,0x3958,0x3957,0x3956,0x3954,0x3952,0x3951,0x394f,0x394e,0x394d,0x394a,
	0x3949,0x3948,0x3947,0x3944,0x3943,0x3942,0x3941,0x393f,0x393d,0x393c,0x393b,0x3939,0x3937,0x3936,0x3934,0x3933,
	0x3932,0x3931,0x392e,0x392d,0x392c,0x392b,0x3928,0x3927,0x3926,0x3925,0x3923,0x3921,0x3920,0x391f,0x391d,0x391b,
	0x391a,0x3918,0x3917,0x3916,0x3914,0x3912,0x3911,0x3910,0x390d,0x390c,0x390b,0x390a,0x3909,0x3906,0x3905,0x3904,
	0x3902,0x3900,0x38ff,0x38fe,0x38fc,0x38fb,0x38f9,0x38f7,0x38f6,0x38f5,0x38f3,0x38f1,0x38f0,0x38ef,0x38ee,0x38eb,
	0x38ea,0x38e9,0x38e8,0x38e5,0x38e4,0x38e3,0x38e1,0x38e0,0x38de,0x38dd,0x38db,0x38da,0x38d8,0x38d6,0x38d5,0x38d4,
	0x38d3,0x38d0,0x38cf,0x38ce,0x38cd,0x38ca,0x38c9,0x38c8,0x38c7,0x38c4,0x38c3,0x38c2,0x38c0,0x38bf,0x38bd,0x38bc,
	0x38ba,0x38b9,0x38b7,0x38b5,0x38b4,0x38b3,0x38b2,0x38af,0x38ae,0x38ad,0x38ac,0x38a9,0x38a8,0x38a7,0x38a6,0x38a4,
	0x38a2,0x38a1,0x389f,0x389e,0x389c,0x389b,0x3899,0x3898,0x3897,0x3894,0x3893,0x3892,0x3891,0x388e,0x388d,0x388c,
	0x388b,0x3889,0x3887,0x3886,0x3885,0x3883,0x3881,0x3880,0x387e,0x387d,0x387c,0x387a,0x3878,0x3877,0x3876,0x3873,
	0x3873,0x3871,0x3870,0x386e,0x386c,0x386b,0x386a,0x3868,0x3866,0x3865,0x3864,0x3863,0x3860,0x385f,0x385e,0x385d,
	0x385b,0x3859,0x3858,0x3857,0x3855,0x3853,0x3852,0x3850,0x384f,0x384e,0x384c,0x384a,0x3849,0x3848,0x3845,0x3844,
	0x3843,0x3842,0x3841,0x383e,0x383d,0x383c,0x383b,0x3838,0x3837,0x3836,0x3834,0x3833,0x3831,0x3830,0x382e,0x382d,
	0x382c,0x3829,0x3828,0x3827,0x3826,0x3823,0x3822,0x3821,0x3820,0x381e,0x381c,0x381b,0x381a,0x3818,0x3816,0x3815,
	0x3814,0x3812,0x3811,0x380f,0x380d,0x380c,0x380b,0x3809,0x3807,0x3806,0x3805,0x3804,0x3801,0x3800,0x37ff,0x37fe,
	0x37fc,0x37fa,0x37f9,0x37f7,0x37f6,0x37f4,0x37f3,0x37f1,0x37f0,0x37ef,0x37ed,0x37eb,0x37ea,0x37e9,0x37e6,0x37e5,
	0x37e4,0x37e3,0x37e2,0x37df,0x37de,0x37dd,0x37db,0x37d9,0x37d8,0x37d7,0x37d5,0x37d4,0x37d2,0x37d0,0x37cf,0x37ce,
	0x37cd,0x37ca,0x37c9,0x37c8,0x37c7,0x37c4,0x37c3,0x37c2,0x37c1,0x37bf,0x37bd,0x37bc,0x37bb,0x37b9,0x37b7,0x37b6,
	0x37b4,0x37b3,0x37b1,0x37b0,0x37ae,0x37ad,0x37ab,0x37aa,0x37a9,0x37a6,0x37a5,0x37a4,0x37a3,0x37a0,0x379f,0x379e,
	0x379d,0x379c,0x3799,0x3798,0x3797,0x3795,0x3794,0x3792,0x3791,0x378f,0x378e,0x378c,0x378a,0x3789,0x3788,0x3787,
	0x3784,0x3783,0x3782,0x3781,0x3780,0x377d,0x377c,0x377b,0x3779,0x3777,0x3776,0x3775,0x3773,0x3772,0x3770,0x376f,
	0x376d,0x376c,0x376b,0x3768,0x3767,0x3766,0x3765,0x3762,0x3761,0x3760,0x375f,0x375e,0x375b,0x375a,0x3759,0x3757,
	0x3756,0x3754,0x3753,0x3751,0x3750,0x374e,0x374d,0x374b,0x374a,0x3749,0x3746,0x3745,0x3744,0x3743,0x3740,0x373f,
	0x373e,0x373d,0x373c,0x3739,0x3738,0x3737,0x3735,0x3734,0x3732,0x3731,0x372f,0x372e,0x372c,0x372b,0x3729,0x3728,
	0x3727,0x3724,0x3723,0x3722,0x3721,0x3720,0x371d,0x371c,0x371b,0x371a,0x3717,0x3716,0x3715,0x3713,0x3712,0x3710,
	0x370f,0x370d,0x370c,0x370b,0x3709,0x3707,0x3706,0x3705,0x3702,0x3701,0x3700,0x36ff,0x36fe,0x36fb,0x36fa,0x36f9,
	0x36f7,0x36f6,0x36f4,0x36f3,0x36f1,0x36f0,0x36ee,0x36ed,0x36eb,0x36ea,0x36e9,0x36e6,0x36e5,0x36e4,0x36e3,0x36e2,
	0x36df,0x36de,0x36dd,0x36dc,0x36d9,0x36d8,0x36d7,0x36d5,0x36d4,0x36d2,0x36d1,0x36cf,0x36ce,0x36cd,0x36cb,0x36c9,
	0x36c8,0x36c7,0x36c4,0x36c3,0x36c2,0x36c1,0x36c0,0x36bd,0x36bc,0x36bb,0x36ba,0x36b8,0x36b6,0x36b5,0x36b3,0x36b2,
	0x36b0,0x36af,0x36ad,0x36ac,0x36ab,0x36a9,0x36a7,0x36a6,0x36a5,0x36a4,0x36a1,0x36a0,0x369f,0x369e,0x369b,0x369a,
	0x3699,0x3698,0x3696,0x3694,0x3693,0x3691,0x3690,0x368e,0x368d,0x368b,0x368a,0x3689,0x3687,0x3685,0x3684,0x3683,
	0x3682,0x367f,0x367e,0x367d,0x367c,0x3679,0x3678,0x3677,0x3676,0x3674,0x3672,0x3671,0x366f,0x366e,0x366d,0x366b,
	0x3669,0x3668,0x3667,0x3665,0x3663,0x3662,0x3661,0x3660,0x365d,0x365c,0x365b,0x365a,0x3658,0x3656,0x3655,0x3653,
	0x3652,0x3650,0x364f,0x364d,0x364c,0x364b,0x3649,0x3647,0x3646,0x3645,0x3644,0x3641,0x3640,0x363f,0x363e,0x363b,
	0x363b,0x3639,0x3638,0x3636,0x3635,0x3633,0x3632,0x3630,0x362f,0x362e,0x362c,0x362a,0x3629,0x3628,0x3627,0x3624,
	0x3623,0x3622,0x3621,0x361e,0x361d,0x361c,0x361b,0x3619,0x3617,0x3616,0x3615,0x3613,0x3612,0x3610,0x360f,0x360d,
	0x360c,0x360b,0x3608,0x3607,0x3606,0x3605,0x3602,0x3601,0x3600,0x35ff,0x35fe,0x35fb,0x35fa,0x35f9,0x35f8,0x35f6,
	0x35f4,0x35f3,0x35f1,0x35f0,0x35ef,0x35ed,0x35eb,0x35ea,0x35e9,0x35e7,0x35e5,0x35e4,0x35e3,0x35e2,0x35df,0x35de,
	0x35dd,0x35dc,0x35da,0x35d8,0x35d7,0x35d6,0x35d4,0x35d3,0x35d1,0x35d0,0x35ce,0x35cd,0x35cb,0x35ca,0x35c8,0x35c7,
	0x35c6,0x35c3,0x35c2,0x35c1,0x35c0,0x35bf,0x35bc,0x35bb,0x35ba,0x35b9,0x35b7,0x35b5,0x35b4,0x35b3,0x35b1,0x35af,
	0x35ae,0x35ac,0x35ab,0x35aa,0x35a8,0x35a6,0x35a5,0x35a4,0x35a3,0x35a0,0x359f,0x359e,0x359d,0x359b,0x3599,0x3598,
	0x3597,0x3595,0x3593,0x3592,0x3591,0x358f,0x358e,0x358c,0x358b,0x3589,0x3588,0x3587,0x3584,0x3583,0x3582,0x3581,
	0x357f,0x357e,0x357c,0x357b,0x3579,0x3577,0x3576,0x3575,0x3574,0x3571,0x3570,0x356f,0x356e,0x356d,0x356a,0x3569,
	0x3568,0x3567,0x3565,0x3563,0x3562,0x3561,0x355f,0x355d,0x355c,0x355b,0x3559,0x3558,0x3556,0x3554,0x3553,0x3552,
	0x3551,0x354e,0x354d,0x354c,0x354b,0x354a,0x3547,0x3546,0x3545,0x3544,0x3542,0x3540,0x353f,0x353e,0x353c,0x353b,
	0x3539,0x3538,0x3536,0x3535,0x3533,0x3531,0x3530,0x352f,0x352e,0x352b,0x352a,0x3529,0x3528,0x3527,0x3524,0x3523,
	0x3522,0x3521,0x351f,0x351d,0x351c,0x351b,0x3519,0x3518,0x3516,0x3515,0x3513,0x3512,0x3511,0x350e,0x350d,0x350c,
	0x350b,0x3508,0x3507,0x3506,0x3505,0x3504,0x3501,0x3500,0x34ff,0x34fe,0x34fc,0x34fa,0x34f9,0x34f8,0x34f6,0x34f5,
	0x34f3,0x34f2,0x34f0,0x34ef,0x34ee,0x34eb,0x34ea,0x34e9,0x34e8,0x34e7,0x34e4,0x34e3,0x34e2,0x34e1,0x34de,0x34dd,
	0x34dc,0x34db,0x34d9,0x34d7,0x34d6,0x34d5,0x34d3,0x34d2,0x34d0,0x34cf,0x34cd,0x34cc,0x34cb,0x34c8,0x34c7,0x34c6,
	0x34c4,0x34c3,0x34c2,0x34bf,0x34be,0x34bd,0x34bc,0x34ba,0x34b8,0x34b7,0x34b6,0x34b4,0x34b3,0x34b1,0x34b0,0x34ae,
	0x34ad,0x34ac,0x34aa,0x34a8,0x34a7,0x34a6,0x34a5,0x34a2,0x34a1,0x34a0,0x349f,0x349e,0x349b,0x349a,0x3499,0x3498,
	0x3496,0x3494,0x3493,0x3492,0x3490,0x348e,0x348d,0x348c,0x348a,0x3489,0x3487,0x3486,0x3484,0x3483,0x3482,0x3480,
	0x347e,0x347d,0x347c,0x347b,0x3478,0x3477,0x3476,0x3475,0x3474,0x3471,0x3470,0x346f,0x346d,0x346c,0x346a,0x3469,
	0x3467,0x3466,0x3465,0x3463,0x3461,0x3460,0x345f,0x345e,0x345b,0x345a,0x3459,0x3458,0x3457,0x3454,0x3453,0x3452,
	0x3451,0x344f,0x344d,0x344c,0x344b,0x3449,0x3448,0x3446,0x3445,0x3443,0x3442,0x3441,0x343f,0x343d,0x343c,0x343b,
	0x343a,0x3437,0x3436,0x3435,0x3434,0x3431,0x3430,0x342f,0x342e,0x342d,0x342a,0x3429,0x3428,0x3427,0x3425,0x3423,
	0x3422,0x3420,0x341f,0x341e,0x341c,0x341a,0x3419,0x3418,0x3417,0x3414,0x3413,0x3412,0x3411,0x3410,0x340d,0x340c,
	0x340b,0x340a,0x3408,0x3406,0x3405,0x3404,0x3402,0x3401,0x33ff,0x33fe,0x33fc,0x33fb,0x33fa,0x33f8,0x33f6,0x33f5,
	0x33f4,0x33f3,0x33f0,0x33ef,0x33ee,0x33ed,0x33ec,0x33e9,0x33e8,0x33e7,0x33e6,0x33e4,0x33e2,0x33e1,0x33e0,0x33de,
	0x33dd,0x33db,0x33da,0x33d8,0x33d7,0x33d5,0x33d3,0x33d2,0x33d1,0x33d0,0x33cd,0x33cc,0x33cb,0x33ca,0x33c9,0x33c6,
	0x33c5,0x33c4,0x33c3,0x33c1,0x33bf,0x33be,0x33bd,0x33bb,0x33ba,0x33b8,0x33b7,0x33b5,0x33b4,0x33b3,0x33b1,0x33af,
	0x33ae,0x33ad,0x33ac,0x33a9,0x33a8,0x33a7,0x33a6,0x33a5,0x33a2,0x33a1,0x33a0,0x339f,0x339d,0x339b,0x339a,0x3399,
	0x3397,0x3396,0x3394,0x3393,0x3391,0x3390,0x338f,0x338d,0x338b,0x338a,0x3389,0x3388,0x3385,0x3384,0x3383,0x3382,
	0x3380,0x337e,0x337d,0x337c,0x337a,0x3378,0x3377,0x3376,0x3374,0x3373,0x3371,0x3370,0x336e,0x336d,0x336c,0x336a,
	0x3368,0x3367,0x3366,0x3365,0x3362,0x3361,0x3360,0x335f,0x335e,0x335b,0x335a,0x3359,0x3358,0x3356,0x3354,0x3353,
	0x3352,0x3350,0x334f,0x334e,0x334c,0x334a,0x3349,0x3348,0x3347,0x3344,0x3343,0x3342,0x3341,0x3340,0x333d,0x333c,
	0x333b,0x333a,0x3338,0x3336,0x3335,0x3334,0x3332,0x3331,0x332f,0x332e,0x332c,0x332b,0x332a,0x3328,0x3326,0x3325,
	0x3324,0x3323,0x3320,0x331f,0x331e,0x331d,0x331c,0x3319,0x3318,0x3317,0x3316,0x3314,0x3312,0x3311,0x3310,0x330e,
	0x330d,0x330b,0x330a,0x3308,0x3307,0x3306,0x3304,0x3302,0x3301,0x3300,0x32ff,0x32fc,0x32fb,0x32fa,0x32f9,0x32f8,
	0x32f6,0x32f4,0x32f3,0x32f2,0x32f0,0x32ef,0x32ed,0x32ec,0x32ea,0x32e9,0x32e8,0x32e6,0x32e4,0x32e3,0x32e2,0x32e1,
	0x32de,0x32dd,0x32dc,0x32db,0x32da,0x32d7,0x32d6,0x32d5,0x32d4,0x32d2,0x32d0,0x32cf,0x32ce,0x32cc,0x32cb,0x32c9,
	0x32c8,0x32c6,0x32c5,0x32c4,0x32c2,0x32c0,0x32bf,0x32be,0x32bd,0x32ba,0x32b9,0x32b8,0x32b7,0x32b6,0x32b3,0x32b2,
	0x32b1,0x32b0,0x32ae,0x32ac,0x32ab,0x32aa,0x32a8,0x32a7,0x32a5,0x32a4,0x32a2,0x32a1,0x32a0,0x329e,0x329c,0x329b,
	0x329a,0x3298,0x3297,0x3295,0x3294,0x3293,0x3291,0x328f,0x328e,0x328d,0x328c,0x3289,0x3288,0x3287,0x3286,0x3285,
	0x3282,0x3281,0x3280,0x327f,0x327e,0x327b,0x327a,0x3279,0x3278,0x3276,0x3274,0x3273,0x3272,0x3270,0x326f,0x326d,
	0x326c,0x326a,0x3269,0x3268,0x3266,0x3264,0x3263,0x3262,0x3261,0x325e,0x325d,0x325c,0x325b,0x325a,0x3258,0x3256,
	0x3255,0x3254,0x3252,0x3251,0x324f,0x324e,0x324d,0x324b,0x324a,0x3248,0x3247,0x3245,0x3244,0x3243,0x3241,0x323f,
	0x323e,0x323d,0x323c,0x3239,0x3238,0x3237,0x3236,0x3235,0x3232,0x3231,0x3230,0x322f,0x322d,0x322b,0x322a,0x3229,
	0x3227,0x3226,0x3224,0x3223,0x3221,0x3220,0x321f,0x321e,0x321b,0x321a,0x3219,0x3218,0x3217,0x3214,0x3213,0x3212,
	0x3211,0x3210,0x320d,0x320c,0x320b,0x320a,0x3208,0x3206,0x3205,0x3204,0x3202,0x3201,0x31ff,0x31fe,0x31fc,0x31fb,
	0x31fa,0x31f8,0x31f6,0x31f5,0x31f4,0x31f3,0x31f0,0x31ef,0x31ee,0x31ed,0x31ec,0x31e9,0x31e8,0x31e7,0x31e6,0x31e5,
	0x31e3,0x31e1,0x31e0,0x31df,0x31dd,0x31dc,0x31da,0x31d9,0x31d7,0x31d6,0x31d5,0x31d3,0x31d1,0x31d0,0x31cf,0x31ce,
	0x31cb,0x31ca,0x31c9,0x31c8,0x31c7,0x31c4,0x31c3,0x31c2,0x31c1,0x31bf,0x31bd,0x31bc,0x31bb,0x31b9,0x31b8,0x31b6,
	0x31b5,0x31b4,0x31b2,0x31b1,0x31af,0x31ae,0x31ac,0x31ab,0x31aa,0x31a8,0x31a6,0x31a5,0x31a4,0x31a3,0x31a2,0x319f,
	0x319e,0x319d,0x319c,0x319a,0x3198,0x3197,0x3196,0x3194,0x3193,0x3191,0x3190,0x318e,0x318d,0x318c,0x318a,0x3188,
	0x3187,0x3186,0x3185,0x3182,0x3181,0x3180,0x317f,0x317e,0x317b,0x317a,0x3179,0x3178,0x3177,0x3174,0x3173,0x3172,
	0x3171,0x316f,0x316d,0x316c,0x316b,0x3169,0x3168,0x3167,0x3165,0x3163,0x3162,0x3161,0x3160,0x315d,0x315c,0x315b,
	0x315a,0x3159,0x3156,0x3155,0x3154,0x3153,0x3151,0x314f,0x314e,0x314d,0x314c,0x314a,0x3148,0x3147,0x3146,0x3144,
	0x3143,0x3141,0x3140,0x313e,0x313d,0x313c,0x313a,0x3138,0x3137,0x3136,0x3135,0x3132,0x3131,0x3130,0x312f,0x312e,
	0x312c,0x312b,0x312a,0x3129,0x3126,0x3125,0x3124,0x3123,0x3121,0x3120,0x311e,0x311d,0x311b,0x311a,0x3119,0x3117,
	0x3115,0x3114,0x3113,0x3112,0x310f,0x310e,0x310d,0x310c,0x310b,0x3108,0x3107,0x3106,0x3105,0x3104,0x3102,0x3100,
	0x30ff,0x30fe,0x30fc,0x30fb,0x30f9,0x30f8,0x30f6,0x30f5,0x30f4,0x30f2,0x30f1,0x30ef,0x30ee,0x30ed,0x30eb,0x30e9,
	0x30e8,0x30e7,0x30e6,0x30e3,0x30e2,0x30e1,0x30e0,0x30df,0x30dd,0x30db,0x30da,0x30d9,0x30d7,0x30d6,0x30d4,0x30d3,
	0x30d2,0x30d0,0x30cf,0x30cd,0x30cc,0x30ca,0x30c9,0x30c8,0x30c6,0x30c4,0x30c3,0x30c2,0x30c1,0x30c0,0x30bd,0x30bc,
	0x30bb,0x30ba,0x30b9,0x30b6,0x30b5,0x30b4,0x30b3,0x30b1,0x30af,0x30ae,0x30ad,0x30ab,0x30aa,0x30a8,0x30a7,0x30a5,
	0x30a4,0x30a3,0x30a1,0x309f,0x309e,0x309d,0x309c,0x309b,0x3098,0x3097,0x3096,0x3095,0x3094,0x3091,0x3090,0x308f,
	0x308e,0x308c,0x308a,0x3089,0x3088,0x3086,0x3085,0x3083,0x3082,0x3081,0x307f,0x307e,0x307d,0x307b,0x3079,0x3078,
	0x3077,0x3076,0x3074,0x3072,0x3071,0x3070,0x306e,0x306d,0x306c,0x306a,0x3069,0x3067,0x3066,0x3065,0x3063,0x3061,
	0x3060,0x305f,0x305e,0x305b,0x305a,0x3059,0x3058,0x3057,0x3056,0x3053,0x3052,0x3051,0x3050,0x304e,0x304c,0x304b,
	0x304a,0x3048,0x3047,0x3045,0x3044,0x3043,0x3041,0x3040,0x303f,0x303d,0x303b,0x303a,0x3039,0x3038,0x3035,0x3034,
	0x3033,0x3032,0x3031,0x302e,0x302d,0x302c,0x302b,0x302a,0x3028,0x3026,0x3025,0x3024,0x3022,0x3021,0x301f,0x301e,
	0x301d,0x301b,0x301a,0x3018,0x3017,0x3015,0x3014,0x3013,0x3012,0x300f,0x300e,0x300d,0x300c,0x300b,0x3008,0x3007,
	0x3006,0x3005,0x3004,0x3001,0x3000,0x2fff,0x2ffe,0x2ffc,0x2ffb,0x2ff9,0x2ff8,0x2ff7,0x2ff5,0x2ff4,0x2ff2,0x2ff1,
	0x2fef,0x2fee,0x2fed,0x2feb,0x2fe9,0x2fe8,0x2fe7,0x2fe6,0x2fe5,0x2fe2,0x2fe1,0x2fe0,0x2fdf,0x2fde,0x2fdb,0x2fda,
	0x2fd9,0x2fd8,0x2fd6,0x2fd4,0x2fd3,0x2fd2,0x2fd1,0x2fcf,0x2fce,0x2fcc,0x2fcb,0x2fc9,0x2fc8,0x2fc7,0x2fc5,0x2fc3,
	0x2fc2,0x2fc1,0x2fbf,0x2fbd,0x2fbc,0x2fbb,0x2fba,0x2fb7,0x2fb6,0x2fb5,0x2fb4,0x2fb3,0x2fb2,0x2faf,0x2fae,0x2fad,
	0x2fac,0x2faa,0x2fa8,0x2fa7,0x2fa6,0x2fa5,0x2fa3,0x2fa2,0x2fa0,0x2f9f,0x2f9d,0x2f9c,0x2f9b,0x2f99,0x2f98,0x2f96,
	0x2f95,0x2f94,0x2f93,0x2f90,0x2f8f,0x2f8e,0x2f8d,0x2f8c,0x2f89,0x2f88,0x2f87,0x2f86,0x2f85,0x2f82,0x2f81,0x2f80,
	0x2f7f,0x2f7d,0x2f7c,0x2f7a,0x2f79,0x2f78,0x2f76,0x2f75,0x2f73,0x2f72,0x2f70,0x2f6f,0x2f6e,0x2f6d,0x2f6b,0x2f69,
	0x2f68,0x2f67,0x2f66,0x2f63,0x2f62,0x2f61,0x2f60,0x2f5f,0x2f5e,0x2f5b,0x2f5a,0x2f59,0x2f58,0x2f56,0x2f54,0x2f53,
	0x2f52,0x2f50,0x2f4f,0x2f4e,0x2f4c,0x2f4b,0x2f49,0x2f48,0x2f47,0x2f45,0x2f43,0x2f42,0x2f41,0x2f40,0x2f3e,0x2f3c,
	0x2f3b,0x2f3a,0x2f39,0x2f38,0x2f35,0x2f34,0x2f33,0x2f32,0x2f31,0x2f2e,0x2f2d,0x2f2c,0x2f2b,0x2f29,0x2f28,0x2f26,
	0x2f25,0x2f24,0x2f22,0x2f21,0x2f1f,0x2f1e,0x2f1c,0x2f1b,0x2f1a,0x2f19,0x2f16,0x2f15,0x2f14,0x2f13,0x2f12,0x2f0f,
	0x2f0e,0x2f0d,0x2f0c,0x2f0b,0x2f09,0x2f07,0x2f06,0x2f05,0x2f04,0x2f02,0x2f00,0x2eff,0x2efe,0x2efc,0x2efb,0x2ef9,
	0x2ef8,0x2ef7,0x2ef5,0x2ef4,0x2ef3,0x2ef1,0x2eef,0x2eee,0x2eed,0x2eec,0x2eea,0x2ee8,0x2ee7,0x2ee6,0x2ee5,0x2ee4,
	0x2ee1,0x2ee0,0x2edf,0x2ede,0x2edc,0x2eda,0x2ed9,0x2ed8,0x2ed7,0x2ed5,0x2ed4,0x2ed2,0x2ed1,0x2ecf,0x2ece,0x2ecd,
	0x2ecb,0x2eca,0x2ec8,0x2ec7,0x2ec6,0x2ec5,0x2ec2,0x2ec1,0x2ec0,0x2ebf,0x2ebe,0x2ebb,0x2eba,0x2eb9,0x2eb8,0x2eb7,
	0x2eb5,0x2eb3,0x2eb2,0x2eb1,0x2eb0,0x2eae,0x2eac,0x2eab,0x2eaa,0x2ea8,0x2ea7,0x2ea5,0x2ea4,0x2ea2,0x2ea1,0x2ea0,
	0x2e9f,0x2e9d,0x2e9b,0x2e9a,0x2e99,0x2e98,0x2e95,0x2e94,0x2e93,0x2e92,0x2e91,0x2e90,0x2e8d,0x2e8c,0x2e8b,0x2e8a,
	0x2e88,0x2e86,0x2e85,0x2e84,0x2e83,0x2e81,0x2e80,0x2e7e,0x2e7d,0x2e7b,0x2e7a,0x2e79,0x2e77,0x2e76,0x2e74,0x2e73,
	0x2e72,0x2e71,0x2e6e,0x2e6d,0x2e6c,0x2e6b,0x2e6a,0x2e67,0x2e66,0x2e65,0x2e64,0x2e63,0x2e60,0x2e5f,0x2e5e,0x2e5d,
	0x2e5c,0x2e5a,0x2e58,0x2e57,0x2e56,0x2e55,0x2e53,0x2e51,0x2e50,0x2e4f,0x2e4e,0x2e4d,0x2e4a,0x2e49,0x2e48,0x2e47,
	0x2e46,0x2e43,0x2e42,0x2e41,0x2e40,0x2e3f,0x2e3d,0x2e3b,0x2e3a,0x2e39,0x2e37,0x2e36,0x2e34,0x2e33,0x2e32,0x2e30,
	0x2e2f,0x2e2e,0x2e2c,0x2e2a,0x2e29,0x2e28,0x2e27,0x2e25,0x2e23,0x2e22,0x2e21,0x2e20,0x2e1f,0x2e1c,0x2e1b,0x2e1a,
	0x2e19,0x2e18,0x2e16,0x2e14,0x2e13,0x2e12,0x2e11,0x2e0f,0x2e0d,0x2e0c,0x2e0b,0x2e09,0x2e08,0x2e07,0x2e05,0x2e04,
	0x2e02,0x2e01,0x2e00,0x2dfe,0x2dfc,0x2dfb,0x2dfa,0x2df9,0x2df8,0x2df5,0x2df4,0x2df3,0x2df2,0x2df1,0x2dee,0x2ded,
	0x2dec,0x2deb,0x2dea,0x2de8,0x2de6,0x2de5,0x2de4,0x2de3,0x2de1,0x2de0,0x2dde,0x2ddd,0x2ddb,0x2dda,0x2dd9,0x2dd7,
	0x2dd6,0x2dd4,0x2dd3,0x2dd2,0x2dd1,0x2dce,0x2dcd,0x2dcc,0x2dcb,0x2dca,0x2dc7,0x2dc6,0x2dc5,0x2dc4,0x2dc3,0x2dc1,
	0x2dbf,0x2dbe,0x2dbd,0x2dbc,0x2dba,0x2db8,0x2db7,0x2db6,0x2db5,0x2db3,0x2db2,0x2db0,0x2daf,0x2dad,0x2dac,0x2dab,
	0x2da9,0x2da8,0x2da6,0x2da5,0x2da3,0x2da2,0x2da1,0x2da0,0x2d9e,0x2d9c,0x2d9b,0x2d9a,0x2d99,0x2d97,0x2d95,0x2d94,
	0x2d93,0x2d92,0x2d91,0x2d8e,0x2d8d,0x2d8c,0x2d8b,0x2d8a,0x2d88,0x2d86,0x2d85,0x2d84,0x2d83,0x2d81,0x2d7f,0x2d7e,
	0x2d7d,0x2d7b,0x2d7a,0x2d79,0x2d77,0x2d76,0x2d74,0x2d73,0x2d72,0x2d71,0x2d6f,0x2d6d,0x2d6c,0x2d6b,0x2d6a,0x2d67,
	0x2d66,0x2d65,0x2d64,0x2d63,0x2d62,0x2d5f,0x2d5e,0x2d5d,0x2d5c,0x2d5b,0x2d59,0x2d57,0x2d56,0x2d55,0x2d53,0x2d52,
	0x2d51,0x2d4f,0x2d4e,0x2d4c,0x2d4b,0x2d4a,0x2d48,0x2d47,0x2d45,0x2d44,0x2d43,0x2d42,0x2d3f,0x2d3e,0x2d3d,0x2d3c,
	0x2d3b,0x2d3a,0x2d37,0x2d36,0x2d35,0x2d34,0x2d33,0x2d30,0x2d2f,0x2d2e,0x2d2d,0x2d2c,0x2d2a,0x2d28,0x2d27,0x2d26,
	0x2d24,0x2d23,0x2d22,0x2d20,0x2d1f,0x2d1d,0x2d1c,0x2d1b,0x2d19,0x2d18,0x2d16,0x2d15,0x2d14,0x2d13,0x2d10,0x2d0f,
	0x2d0e,0x2d0d,0x2d0c,0x2d0b,0x2d08,0x2d07,0x2d06,0x2d05,0x2d04,0x2d01,0x2d00,0x2cff,0x2cfe,0x2cfc,0x2cfb,0x2cf9,
	0x2cf8,0x2cf7,0x2cf5,0x2cf4,0x2cf3,0x2cf1,0x2cf0,0x2cee,0x2ced,0x2cec,0x2cea,0x2ce8,0x2ce7,0x2ce6,0x2ce5,0x2ce4,
	0x2ce1,0x2ce0,0x2cdf,0x2cde,0x2cdd,0x2cdc,0x2cd9,0x2cd8,0x2cd7,0x2cd6,0x2cd4,0x2cd3,0x2cd1,0x2cd0,0x2ccf,0x2ccd,
	0x2ccc,0x2cca,0x2cc9,0x2cc8,0x2cc6,0x2cc5,0x2cc4,0x2cc2,0x2cc1,0x2cbf,0x2cbe,0x2cbd,0x2cbc,0x2cb9,0x2cb8,0x2cb7,
	0x2cb6,0x2cb5,0x2cb2,0x2cb1,0x2cb0,0x2caf,0x2cae,0x2cad,0x2caa,0x2ca9,0x2ca8,0x2ca7,0x2ca5,0x2ca4,0x2ca2,0x2ca1,
	0x2ca0,0x2c9e,0x2c9d,0x2c9b,0x2c9a,0x2c99,0x2c97,0x2c96,0x2c95,0x2c93,0x2c91,0x2c90,0x2c8f,0x2c8e,0x2c8d,0x2c8a,
	0x2c89,0x2c88,0x2c87,0x2c86,0x2c83,0x2c82,0x2c81,0x2c80,0x2c7f,0x2c7d,0x2c7b,0x2c7a,0x2c79,0x2c78,0x2c76,0x2c75,
	0x2c73,0x2c72,0x2c71,0x2c6f,0x2c6e,0x2c6c,0x2c6b,0x2c6a,0x2c68,0x2c67,0x2c66,0x2c64,0x2c62,0x2c61,0x2c60,0x2c5f,
	0x2c5e,0x2c5b,0x2c5a,0x2c59,0x2c58,0x2c57,0x2c56,0x2c53,0x2c52,0x2c51,0x2c50,0x2c4e,0x2c4c,0x2c4b,0x2c4a,0x2c49,
	0x2c47,0x2c46,0x2c45,0x2c43,0x2c42,0x2c40,0x2c3f,0x2c3e,0x2c3d,0x2c3b,0x2c39,0x2c38,0x2c37,0x2c36,0x2c35,0x2c32,
	0x2c31,0x2c30,0x2c2f,0x2c2e,0x2c2d,0x2c2a,0x2c29,0x2c28,0x2c27,0x2c26,0x2c24,0x2c22,0x2c21,0x2c20,0x2c1e,0x2c1d,
	0x2c1c,0x2c1a,0x2c19,0x2c17,0x2c16,0x2c15,0x2c13,0x2c12,0x2c10,0x2c0f,0x2c0e,0x2c0d,0x2c0b,0x2c09,0x2c08,0x2c07,
	0x2c06,0x2c05,0x2c02,0x2c01,0x2c00,0x2bff,0x2bfe,0x2bfc,0x2bfa,0x2bf9,0x2bf8,0x2bf7,0x2bf5,0x2bf4,0x2bf2,0x2bf1,
	0x2bf0,0x2bee,0x2bed,0x2beb,0x2bea,0x2be9,0x2be7,0x2be6,0x2be5,0x2be3,0x2be2,0x2be0,0x2bdf,0x2bde,0x2bdd,0x2bdb,
	0x2bd9,0x2bd8,0x2bd7,0x2bd6,0x2bd5,0x2bd2,0x2bd1,0x2bd0,0x2bcf,0x2bce,0x2bcc,0x2bca,0x2bc9,0x2bc8,0x2bc7,0x2bc5,
	0x2bc4,0x2bc2,0x2bc1,0x2bc0,0x2bbe,0x2bbd,0x2bbb,0x2bba,0x2bb9,0x2bb7,0x2bb6,0x2bb5,0x2bb3,0x2bb1,0x2bb0,0x2baf,
	0x2bae,0x2bad,0x2baa,0x2ba9,0x2ba8,0x2ba7,0x2ba6,0x2ba5,0x2ba2,0x2ba1,0x2ba0,0x2b9f,0x2b9e,0x2b9c,0x2b9a,0x2b99,
	0x2b98,0x2b97,0x2b94,0x2b93,0x2b92,0x2b91,0x2b90,0x2b8e,0x2b8c,0x2b8b,0x2b8a,0x2b89,0x2b87,0x2b86,0x2b84,0x2b83,
	0x2b82,0x2b80,0x2b7f,0x2b7e,0x2b7c,0x2b7b,0x2b79,0x2b78,0x2b77,0x2b76,0x2b74,0x2b72,0x2b71,0x2b70,0x2b6f,0x2b6e,
	0x2b6b,0x2b6a,0x2b69,0x2b68,0x2b67,0x2b66,0x2b63,0x2b62,0x2b61,0x2b60,0x2b5f,0x2b5d,0x2b5b,0x2b5a,0x2b59,0x2b58,
	0x2b56,0x2b55,0x2b53,0x2b52,0x2b51,0x2b4f,0x2b4e,0x2b4d,0x2b4b,0x2b4a,0x2b48,0x2b47,0x2b46,0x2b45,0x2b43,0x2b41,
	0x2b40,0x2b3f,0x2b3e,0x2b3c,0x2b3a,0x2b39,0x2b38,0x2b37,0x2b36,0x2b33,0x2b32,0x2b31,0x2b30,0x2b2f,0x2b2e,0x2b2b,
	0x2b2a,0x2b29,0x2b28,0x2b27,0x2b25,0x2b23,0x2b22,0x2b21,0x2b20,0x2b1e,0x2b1d,0x2b1b,0x2b1a,0x2b19,0x2b17,0x2b16,
	0x2b15,0x2b13,0x2b12,0x2b10,0x2b0f,0x2b0e,0x2b0d,0x2b0b,0x2b09,0x2b08,0x2b07,0x2b06,0x2b05,0x2b02,0x2b01,0x2b00,
	0x2aff,0x2afe,0x2afc,0x2afa,0x2af9,0x2af8,0x2af7,0x2af5,0x2af4,0x2af2,0x2af1,0x2af0,0x2aee,0x2aed,0x2aec,0x2aea,
	0x2ae9,0x2ae7,0x2ae6,0x2ae5,0x2ae4,0x2ae2,0x2ae0,0x2adf,0x2ade,0x2add,0x2adc,0x2ad9,0x2ad8,0x2ad7,0x2ad6,0x2ad5,
	0x2ad4,0x2ad1,0x2ad0,0x2acf,0x2ace,0x2acd,0x2acb,0x2ac9,0x2ac8,0x2ac7,0x2ac6,0x2ac4,0x2ac3,0x2ac1,0x2ac0,0x2abf,
	0x2abd,0x2abc,0x2abb,0x2ab9,0x2ab8,0x2ab6,0x2ab5,0x2ab4,0x2ab3,0x2ab1,0x2aaf,0x2aae,0x2aad,0x2aac,0x2aab,0x2aa8,
	0x2aa7,0x2aa6,0x2aa5,0x2aa4,0x2aa3,0x2aa0,0x2a9f,0x2a9e,0x2a9d,0x2a9c,0x2a9a,0x2a98,0x2a97,0x2a96,0x2a95,0x2a93,
	0x2a92,0x2a90,0x2a8f,0x2a8e,0x2a8c,0x2a8b,0x2a89,0x2a88,0x2a87,0x2a85,0x2a84,0x2a83,0x2a81,0x2a80,0x2a7e,0x2a7d,
	0x2a7c,0x2a7b,0x2a79,0x2a77,0x2a76,0x2a75,0x2a74,0x2a73,0x2a70,0x2a6f,0x2a6e,0x2a6d,0x2a6c,0x2a6b,0x2a68,0x2a67,
	0x2a66,0x2a65,0x2a63,0x2a62,0x2a60,0x2a5f,0x2a5e,0x2a5c,0x2a5b,0x2a5a,0x2a58,0x2a57,0x2a55,0x2a54,0x2a53,0x2a52,
	0x2a50,0x2a4e,0x2a4d,0x2a4c,0x2a4b,0x2a4a,0x2a47,0x2a46,0x2a45,0x2a44,0x2a43,0x2a42,0x2a3f,0x2a3e,0x2a3d,0x2a3c,
	0x2a3b,0x2a39,0x2a38,0x2a37,0x2a36,0x2a34,0x2a33,0x2a31,0x2a30,0x2a2f,0x2a2d,0x2a2c,0x2a2b,0x2a29,0x2a28,0x2a26,
	0x2a25,0x2a24,0x2a23,0x2a21,0x2a1f,0x2a1e,0x2a1d,0x2a1c,0x2a1b,0x2a18,0x2a17,0x2a16,0x2a15,0x2a14,0x2a13,0x2a10,
	0x2a0f,0x2a0e,0x2a0d,0x2a0c,0x2a0a,0x2a08,0x2a07,0x2a06,0x2a05,0x2a03,0x2a02,0x2a00,0x29ff,0x29fe,0x29fd,0x29fb,
	0x29fa,0x29f8,0x29f7,0x29f6,0x29f4,0x29f3,0x29f2,0x29f0,0x29ef,0x29ed,0x29ec,0x29eb,0x29ea,0x29e8,0x29e6,0x29e5,
	0x29e4,0x29e3,0x29e2,0x29e1,0x29de,0x29dd,0x29dc,0x29db,0x29da,0x29d8,0x29d6,0x29d5,0x29d4,0x29d3,0x29d1,0x29d0,
	0x29ce,0x29cd,0x29cc,0x29ca,0x29c9,0x29c8,0x29c6,0x29c5,0x29c3,0x29c2,0x29c1,0x29c0,0x29be,0x29bc,0x29bb,0x29ba,
	0x29b9,0x29b8,0x29b6,0x29b4,0x29b3,0x29b2,0x29b1,0x29b0,0x29ad,0x29ac,0x29ab,0x29aa,0x29a9,0x29a8,0x29a5,0x29a4,
	0x29a3,0x29a2,0x29a1,0x299f,0x299d,0x299c,0x299b,0x299a,0x2998,0x2997,0x2995,0x2994,0x2993,0x2991,0x2990,0x298f,
	0x298e,0x298c,0x298a,0x2989,0x2988,0x2987,0x2985,0x2984,0x2982,0x2981,0x2980,0x297e,0x297d,0x297c,0x297b,0x2979,
	0x2977,0x2976,0x2975,0x2974,0x2973,0x2971,0x296f,0x296e,0x296d,0x296c,0x296b,0x2968,0x2967,0x2966,0x2965,0x2964,
	0x2963,0x2960,0x295f,0x295e,0x295d,0x295c,0x295a,0x2959,0x2957,0x2956,0x2955,0x2954,0x2952,0x2951,0x294f,0x294e,
	0x294d,0x294b,0x294a,0x2949,0x2947,0x2946,0x2944,0x2943,0x2942,0x2941,0x293f,0x293d,0x293c,0x293b,0x293a,0x2939,
	0x2938,0x2935,0x2934,0x2933,0x2932,0x2931,0x2930,0x292d,0x292c,0x292b,0x292a,0x2929,0x2927,0x2925,0x2924,0x2923,
	0x2922,0x2920,0x291f,0x291d,0x291c,0x291b,0x291a,0x2918,0x2917,0x2915,0x2914,0x2913,0x2911,0x2910,0x290f,0x290e,
	0x290c,0x290a,0x2909,0x2908,0x2907,0x2906,0x2903,0x2902,0x2901,0x2900,0x28ff,0x28fe,0x28fb,0x28fa,0x28f9,0x28f8,
	0x28f7,0x28f6,0x28f3,0x28f2,0x28f1,0x28f0,0x28ef,0x28ed,0x28ec,0x28ea,0x28e9,0x28e8,0x28e6,0x28e5,0x28e4,0x28e2,
	0x28e1,0x28e0,0x28de,0x28dd,0x28dc,0x28da,0x28d9,0x28d7,0x28d6,0x28d5,0x28d4,0x28d2,0x28d0,0x28cf,0x28ce,0x28cd,
	0x28cc,0x28cb,0x28c8,0x28c7,0x28c6,0x28c5,0x28c4,0x28c3,0x28c0,0x28bf,0x28be,0x28bd,0x28bc,0x28ba,0x28b8,0x28b7,
	0x28b6,0x28b5,0x28b3,0x28b2,0x28b0,0x28af,0x28ae,0x28ac,0x28ab,0x28aa,0x28a9,0x28a7,0x28a6,0x28a4,0x28a3,0x28a2,
	0x28a1,0x289f,0x289d,0x289c,0x289b,0x289a,0x2899,0x2896,0x2895,0x2894,0x2893,0x2892,0x2891,0x288e,0x288d,0x288c,
	0x288b,0x288a,0x2889,0x2887,0x2885,0x2884,0x2883,0x2882,0x2880,0x287f,0x287d,0x287c,0x287b,0x2879,0x2878,0x2877,
	0x2875,0x2874,0x2872,0x2871,0x2870,0x286f,0x286d,0x286c,0x286a,0x2869,0x2868,0x2867,0x2866,0x2863,0x2862,0x2861,
	0x2860,0x285f,0x285e,0x285b,0x285a,0x2859,0x2858,0x2857,0x2855,0x2853,0x2852,0x2851,0x2850,0x284e,0x284d,0x284b,
	0x284a,0x2849,0x2848,0x2846,0x2845,0x2844,0x2842,0x2841,0x283f,0x283e,0x283d,0x283c,0x283a,0x2838,0x2837,0x2836,
	0x2835,0x2834,0x2833,0x2832,0x282f,0x282e,0x282d,0x282c,0x282b,0x282a,0x2827,0x2826,0x2825,0x2824,0x2823,0x2822,
	0x2820,0x281e,0x281d,0x281c,0x281b,0x2819,0x2818,0x2816,0x2815,0x2814,0x2813,0x2811,0x2810,0x280e,0x280d,0x280c,
	0x280a,0x2809,0x2808,0x2807,0x2805,0x2803,0x2802,0x2801,0x2800,0x27ff,0x27fd,0x27fb,0x27fa,0x27f9,0x27f8,0x27f7,
	0x27f6,0x27f3,0x27f2,0x27f1,0x27f0,0x27ef,0x27ee,0x27eb,0x27ea,0x27e9,0x27e8,0x27e7,0x27e5,0x27e3,0x27e2,0x27e1,
	0x27e0,0x27de,0x27dd,0x27dc,0x27da,0x27d9,0x27d8,0x27d6,0x27d5,0x27d4,0x27d2,0x27d1,0x27cf,0x27ce,0x27cd,0x27cc,
	0x27cb,0x27c9,0x27c7,0x27c6,0x27c5,0x27c4,0x27c3,0x27c0,0x27bf,0x27be,0x27bd,0x27bc,0x27bb,0x27b8,0x27b7,0x27b6,
	0x27b5,0x27b4,0x27b3,0x27b1,0x27af,0x27ae,0x27ad,0x27ac,0x27aa,0x27a9,0x27a7,0x27a6,0x27a5,0x27a4,0x27a2,0x27a1,
	0x27a0,0x279e,0x279d,0x279b,0x279a,0x2799,0x2798,0x2796,0x2794,0x2793,0x2792,0x2791,0x2790,0x278e,0x278c,0x278b,
	0x278a,0x2789,0x2787,0x2786,0x2785,0x2783,0x2782,0x2781,0x2780,0x277e,0x277c,0x277b,0x277a,0x2779,0x2778,0x2776,
	0x2774,0x2773,0x2772,0x2771,0x2770,0x276f,0x276c,0x276b,0x276a,0x2769,0x2768,0x2767,0x2764,0x2763,0x2762,0x2761,
	0x2760,0x275e,0x275d,0x275b,0x275a,0x2759,0x2758,0x2756,0x2755,0x2754,0x2752,0x2751,0x2750,0x274e,0x274d,0x274c,
	0x274a,0x2749,0x2747,0x2746,0x2745,0x2744,0x2743,0x2741,0x273f,0x273e,0x273d,0x273c,0x273b,0x2738,0x2737,0x2736,
	0x2735,0x2734,0x2733,0x2732,0x272f,0x272e,0x272d,0x272c,0x272b,0x2729,0x2727,0x2726,0x2725,0x2724,0x2723,0x2721,
	0x2720,0x271e,0x271d,0x271c,0x271b,0x2719,0x2718,0x2716,0x2715,0x2714,0x2712,0x2711,0x2710,0x270f,0x270d,0x270c,
	0x270a,0x2709,0x2708,0x2707,0x2706,0x2703,0x2702,0x2701,0x2700,0x26ff,0x26fe,0x26fb,0x26fa,0x26f9,0x26f8,0x26f7,
	0x26f6,0x26f4,0x26f2,0x26f1,0x26f0,0x26ef,0x26ee,0x26ec,0x26ea,0x26e9,0x26e8,0x26e7,0x26e5,0x26e4,0x26e3,0x26e1,
	0x26e0,0x26df,0x26dd,0x26dc,0x26db,0x26d9,0x26d8,0x26d7,0x26d5,0x26d4,0x26d3,0x26d2,0x26d0,0x26ce,0x26cd,0x26cc,
	0x26cb,0x26ca,0x26c9,0x26c6,0x26c5,0x26c4,0x26c3,0x26c2,0x26c1,0x26be,0x26bd,0x26bc,0x26bb,0x26ba,0x26b9,0x26b7,
	0x26b5,0x26b4,0x26b3,0x26b2,0x26b0,0x26af,0x26ad,0x26ac,0x26ab,0x26aa,0x26a8,0x26a7,0x26a6,0x26a4,0x26a3,0x26a2,
	0x26a0,0x269f,0x269e,0x269c,0x269b,0x2699,0x2698,0x2697,0x2696,0x2695,0x2693,0x2691,0x2690,0x268f,0x268e,0x268d,
	0x268c,0x2689,0x2688,0x2687,0x2686,0x2685,0x2684,0x2681,0x2680,0x267f,0x267e,0x267d,0x267b,0x267a,0x2678,0x2677,
	0x2676,0x2675,0x2673,0x2672,0x2670,0x266f,0x266e,0x266d,0x266b,0x266a,0x2669,0x2667,0x2666,0x2664,0x2663,0x2662,
	0x2661,0x265f,0x265e,0x265c,0x265b,0x265a,0x2659,0x2658,0x2655,0x2654,0x2653,0x2652,0x2651,0x2650,0x264d,0x264c,
	0x264b,0x264a,0x2649,0x2648,0x2646,0x2644,0x2643,0x2642,0x2641,0x2640,0x263e,0x263d,0x263b,0x263a,0x2639,0x2637,
	0x2637,0x2635,0x2634,0x2633,0x2631,0x2630,0x262f,0x262d,0x262c,0x262a,0x2629,0x2628,0x2627,0x2626,0x2624,0x2622,
	0x2621,0x2620,0x261f,0x261e,0x261d,0x261a,0x2619,0x2618,0x2617,0x2616,0x2615,0x2612,0x2611,0x2610,0x260f,0x260e,
	0x260d,0x260b,0x2609,0x2608,0x2607,0x2606,0x2605,0x2603,0x2602,0x2600,0x25ff,0x25fe,0x25fd,0x25fb,0x25fa,0x25f9,
	0x25f7,0x25f6,0x25f5,0x25f3,0x25f2,0x25f1,0x25ef,0x25ee,0x25ec,0x25eb,0x25ea,0x25e9,0x25e8,0x25e6,0x25e4,0x25e3,
	0x25e2,0x25e1,0x25e0,0x25df,0x25dc,0x25db,0x25da,0x25d9,0x25d8,0x25d7,0x25d6,0x25d3,0x25d2,0x25d1,0x25d0,0x25cf,
	0x25cd,0x25cb,0x25ca,0x25c9,0x25c8,0x25c7,0x25c5,0x25c4,0x25c2,0x25c1,0x25c0,0x25bf,0x25bd,0x25bc,0x25bb,0x25b9,
	0x25b8,0x25b7,0x25b5,0x25b4,0x25b3,0x25b2,0x25b0,0x25ae,0x25ad,0x25ac,0x25ab,0x25aa,0x25a8,0x25a6,0x25a5,0x25a4,
	0x25a3,0x25a2,0x25a1,0x259e,0x259d,0x259c,0x259b,0x259a,0x2599,0x2597,0x2595,0x2594,0x2593,0x2592,0x2591,0x258f,
	0x258e,0x258d,0x258a,0x2589,0x2588,0x2587,0x2586,0x2585,0x2584,0x2581,0x2580,0x257f,0x257e,0x257d,0x257c,0x257a,
	0x2578,0x2577,0x2576,0x2575,0x2574,0x2572,0x2571,0x256f,0x256e,0x256d,0x256c,0x256a,0x2569,0x2568,0x2566,0x2565,
	0x2564,0x2562,0x2561,0x2560,0x255f,0x255d,0x255c,0x255a,0x2559,0x2558,0x2557,0x2556,0x2554,0x2552,0x2551,0x2550,
	0x254f,0x254e,0x254d,0x254a,0x2549,0x2548,0x2547,0x2546,0x2545,0x2544,0x2541,0x2540,0x253f,0x253e,0x253d,0x253c,
	0x253a,0x2538,0x2537,0x2536,0x2535,0x2534,0x2532,0x2530,0x252f,0x252e,0x252d,0x252b,0x252a,0x2529,0x2527,0x2526,
	0x2525,0x2523,0x2522,0x2521,0x2520,0x251e,0x251d,0x251b,0x251a,0x2519,0x2518,0x2517,0x2515,0x2513,0x2512,0x2511,
	0x2510,0x250f,0x250e,0x250b,0x250a,0x2509,0x2508,0x2507,0x2506,0x2505,0x2502,0x2501,0x2500,0x24ff,0x24fe,0x24fd,
	0x24fb,0x24f9,0x24f8,0x24f7,0x24f6,0x24f5,0x24f3,0x24f2,0x24f0,0x24ef,0x24ee,0x24ed,0x24eb,0x24ea,0x24e9,0x24e7,
	0x24e6,0x24e5,0x24e3,0x24e2,0x24e1,0x24e0,0x24de,0x24dd,0x24db,0x24da,0x24d9,0x24d8,0x24d7,0x24d5,0x24d3,0x24d2,
	0x24d1,0x24d0,0x24cf,0x24ce,0x24cb,0x24ca,0x24c9,0x24c8,0x24c7,0x24c6,0x24c3,0x24c2,0x24c1,0x24c0,0x24bf,0x24be,
	0x24bc,0x24ba,0x24b9,0x24b8,0x24b7,0x24b6,0x24b4,0x24b3,0x24b1,0x24b0,0x24af,0x24ae,0x24ac,0x24ab,0x24aa,0x24a8,
	0x24a7,0x24a6,0x24a4,0x24a3,0x24a2,0x24a1,0x249f,0x249e,0x249c,0x249b,0x249a,0x2499,0x2498,0x2496,0x2494,0x2493,
	0x2492,0x2491,0x2490,0x248f,0x248c,0x248b,0x248a,0x2489,0x2488,0x2487,0x2486,0x2483,0x2482,0x2481,0x2480,0x247f,
	0x247e,0x247c,0x247a,0x2479,0x2478,0x2477,0x2476,0x2474,0x2473,0x2471,0x2470,0x246f,0x246e,0x246c,0x246b,0x246a,
	0x2468,0x2467,0x2465,0x2464,0x2463,0x2462,0x2461,0x245f,0x245d,0x245c,0x245b,0x245a,0x2459,0x2457,0x2455,0x2454,
	0x2453,0x2452,0x2451,0x2450,0x244d,0x244c,0x244b,0x244a,0x2449,0x2448,0x2447,0x2444,0x2443,0x2442,0x2441,0x2440,
	0x243f,0x243e,0x243c,0x243b,0x2439,0x2438,0x2437,0x2436,0x2434,0x2433,0x2432,0x2430,0x242f,0x242e,0x242c,0x242b,
	0x242a,0x2429,0x2427,0x2426,0x2425,0x2423,0x2422,0x2421,0x2420,0x241e,0x241d,0x241b,0x241a,0x2419,0x2418,0x2417,
	0x2416,0x2413,0x2412,0x2411,0x2410,0x240f,0x240e,0x240d,0x240a,0x2409,0x2408,0x2407,0x2406,0x2405,0x2403,0x2401,
	0x2400,0x23ff,0x23fe,0x23fd,0x23fb,0x23fa,0x23f8,0x23f7,0x23f6,0x23f5,0x23f3,0x23f2,0x23f1,0x23ef,0x23ee,0x23ed,
	0x23eb,0x23ea,0x23e9,0x23e8,0x23e6,0x23e5,0x23e3,0x23e2,0x23e1,0x23e0,0x23df,0x23dd,0x23dc,0x23da,0x23d9,0x23d8,
	0x23d7,0x23d6,0x23d4,0x23d2,0x23d1,0x23d0,0x23cf,0x23ce,0x23cd,0x23ca,0x23c9,0x23c8,0x23c7,0x23c6,0x23c5,0x23c4,
	0x23c2,0x23c0,0x23bf,0x23be,0x23bd,0x23bc,0x23ba,0x23b9,0x23b7,0x23b6,0x23b5,0x23b4,0x23b2,0x23b1,0x23b0,0x23ae,
	0x23ad,0x23ac,0x23aa,0x23a9,0x23a8,0x23a7,0x23a5,0x23a4,0x23a2,0x23a1,0x23a0,0x239f,0x239e,0x239c,0x239a,0x2399,
	0x2398,0x2397,0x2395,0x2394,0x2393,0x2392,0x2390,0x238f,0x238e,0x238d,0x238b,0x238a,0x2388,0x2387,0x2386,0x2385,
	0x2384,0x2382,0x2380,0x237f,0x237e,0x237d,0x237c,0x237b,0x2378,0x2377,0x2376,0x2375,0x2374,0x2373,0x2372,0x2371,
	0x236e,0x236d,0x236c,0x236b,0x236a,0x2369,0x2367,0x2365,0x2364,0x2363,0x2362,0x2361,0x235f,0x235e,0x235c,0x235b,
	0x235a,0x2359,0x2358,0x2356,0x2355,0x2354,0x2352,0x2351,0x2350,0x234e,0x234d,0x234c,0x234b,0x2349,0x2348,0x2346,
	0x2345,0x2344,0x2343,0x2342,0x2340,0x233f,0x233d,0x233c,0x233b,0x233a,0x2339,0x2337,0x2335,0x2334,0x2333,0x2332,
	0x2331,0x2330,0x232f,0x232c,0x232b,0x232a,0x2329,0x2328,0x2327,0x2326,0x2323,0x2322,0x2321,0x2320,0x231f,0x231e,
	0x231c,0x231a,0x2319,0x2318,0x2317,0x2316,0x2314,0x2313,0x2312,0x2310,0x230f,0x230e,0x230d,0x230b,0x230a,0x2309,
	0x2307,0x2306,0x2305,0x2303,0x2302,0x2301,0x2300,0x22fe,0x22fd,0x22fb,0x22fa,0x22f9,0x22f8,0x22f7,0x22f6,0x22f3,
	0x22f2,0x22f1,0x22f0,0x22ef,0x22ee,0x22ed,0x22ea,0x22e9,0x22e8,0x22e7,0x22e6,0x22e5,0x22e4,0x22e1,0x22e0,0x22df,
	0x22de,0x22dd,0x22dc,0x22da,0x22d9,0x22d7,0x22d6,0x22d5,0x22d4,0x22d3,0x22d1,0x22d0,0x22ce,0x22cd,0x22cc,0x22cb,
	0x22c9,0x22c8,0x22c7,0x22c5,0x22c4,0x22c3,0x22c1,0x22c0,0x22bf,0x22be,0x22bd,0x22bb,0x22ba,0x22b8,0x22b7,0x22b6,
	0x22b5,0x22b4,0x22b2,0x22b0,0x22af,0x22ae,0x22ad,0x22ac,0x22ab,0x22a8,0x22a7,0x22a6,0x22a5,0x22a4,0x22a3,0x22a2,
	0x22a1,0x229e,0x229d,0x229c,0x229b,0x229a,0x2299,0x2297,0x2295,0x2294,0x2293,0x2292,0x2291,0x228f,0x228e,0x228c,
	0x228b,0x228a,0x2289,0x2288,0x2286,0x2285,0x2283,0x2282,0x2281,0x2280,0x227e,0x227d,0x227c,0x227b,0x2279,0x2278,
	0x2276,0x2275,0x2274,0x2273,0x2272,0x2270,0x226f,0x226d,0x226c,0x226b,0x226a,0x2269,0x2267,0x2265,0x2264,0x2263,
	0x2262,0x2261,0x2260,0x225f,0x225c,0x225b,0x225a,0x2259,0x2258,0x2257,0x2255,0x2253,0x2252,0x2251,0x2250,0x224f,
	0x224e,0x224c,0x224b,0x224a,0x2249,0x2247,0x2246,0x2245,0x2243,0x2242,0x2241,0x223f,0x223e,0x223d,0x223c,0x223b,
	0x2239,0x2238,0x2236,0x2235,0x2234,0x2233,0x2232,0x2230,0x222e,0x222d,0x222c,0x222b,0x222a,0x2229,0x2228,0x2225,
	0x2224,0x2223,0x2222,0x2221,0x2220,0x221f,0x221c,0x221b,0x221a,0x2219,0x2218,0x2217,0x2216,0x2214,0x2212,0x2211,
	0x2210,0x220f,0x220e,0x220c,0x220b,0x2209,0x2208,0x2207,0x2206,0x2205,0x2203,0x2202,0x2201,0x21ff,0x21fe,0x21fd,
	0x21fb,0x21fa,0x21f9,0x21f8,0x21f6,0x21f5,0x21f4,0x21f2,0x21f1,0x21f0,0x21ef,0x21ee,0x21ec,0x21ea,0x21e9,0x21e8,
	0x21e7,0x21e6,0x21e5,0x21e3,0x21e1,0x21e0,0x21df,0x21de,0x21dd,0x21dc,0x21db,0x21d8,0x21d7,0x21d6,0x21d5,0x21d4,
	0x21d3,0x21d2,0x21cf,0x21ce,0x21cd,0x21cc,0x21cb,0x21ca,0x21c8,0x21c7,0x21c5,0x21c4,0x21c3,0x21c2,0x21c1,0x21bf,
	0x21be,0x21bc,0x21bb,0x21ba,0x21b9,0x21b8,0x21b6,0x21b5,0x21b4,0x21b2,0x21b1,0x21b0,0x21ae,0x21ad,0x21ac,0x21ab,
	0x21a9,0x21a8,0x21a6,0x21a5,0x21a4,0x21a3,0x21a1,0x21a0,0x219f,0x219e,0x219c,0x219b,0x219a,0x2198,0x2197,0x2196,
	0x2195,0x2194,0x2192,0x2191,0x218f,0x218e,0x218d,0x218c,0x218b,0x218a,0x2187,0x2186,0x2185,0x2184,0x2183,0x2182,
	0x2181,0x217e,0x217d,0x217c,0x217b,0x217a,0x2179,0x2178,0x2177,0x2174,0x2173,0x2172,0x2171,0x2170,0x216f,0x216e,
	0x216c,0x216a,0x2169,0x2168,0x2167,0x2166,0x2164,0x2163,0x2162,0x2160,0x215f,0x215e,0x215d,0x215b,0x215a,0x2159,
	0x2158,0x2156,0x2155,0x2154,0x2152,0x2151,0x2150,0x214f,0x214d,0x214c,0x214b,0x2149,0x2148,0x2147,0x2146,0x2145,
	0x2143,0x2141,0x2140,0x213f,0x213e,0x213d,0x213c,0x213b,0x2138,0x2137,0x2136,0x2135,0x2134,0x2133,0x2132,0x2131,
	0x212e,0x212d,0x212c,0x212b,0x212a,0x2129,0x2128,0x2125,0x2124,0x2123,0x2122,0x2121,0x2120,0x211e,0x211d,0x211b,
	0x211a,0x2119,0x2118,0x2117,0x2115,0x2114,0x2113,0x2111,0x2110,0x210f,0x210e,0x210c,0x210b,0x210a,0x2109,0x2107,
	0x2106,0x2105,0x2103,0x2102,0x2101,0x2100,0x20fe,0x20fd,0x20fb,0x20fa,0x20f9,0x20f8,0x20f7,0x20f6,0x20f4,0x20f2,
	0x20f1,0x20f0,0x20ef,0x20ee,0x20ed,0x20ec,0x20e9,0x20e8,0x20e7,0x20e6,0x20e5,0x20e4,0x20e3,0x20e2,0x20df,0x20de,
	0x20dd,0x20dc,0x20db,0x20da,0x20d8,0x20d6,0x20d5,0x20d4,0x20d3,0x20d2,0x20d1,0x20cf,0x20ce,0x20cc,0x20cb,0x20ca,
	0x20c9,0x20c8,0x20c6,0x20c5,0x20c4,0x20c2,0x20c1,0x20c0,0x20bf,0x20bd,0x20bc,0x20bb,0x20ba,0x20b8,0x20b7,0x20b5,
	0x20b4,0x20b3,0x20b2,0x20b1,0x20af,0x20ae,0x20ac,0x20ab,0x20aa,0x20a9,0x20a8,0x20a7,0x20a5,0x20a3,0x20a2,0x20a1,
	0x20a0,0x209f,0x209e,0x209d,0x209a,0x2099,0x2098,0x2097,0x2096,0x2095,0x2094,0x2092,0x2090,0x208f,0x208e,0x208d,
	0x208c,0x208b,0x2089,0x2087,0x2086,0x2085,0x2084,0x2083,0x2082,0x2080,0x207f,0x207d,0x207c,0x207b,0x207a,0x2079,
	0x2077,0x2076,0x2075,0x2073,0x2072,0x2071,0x2070,0x206e,0x206d,0x206c,0x206b,0x2069,0x2068,0x2066,0x2065,0x2064,
	0x2063,0x2062,0x2061,0x2060,0x205f,0x205c,0x205b,0x205a,0x2059,0x2058,0x2057,0x2056,0x2054,0x2052,0x2051,0x2050,
	0x204f,0x204e,0x204d,0x204b,0x204a,0x2048,0x2047,0x2046,0x2045,0x2044,0x2042,0x2041,0x2040,0x203e,0x203d,0x203c,
	0x203b,0x2039,0x2038,0x2037,0x2036,0x2034,0x2033,0x2032,0x2030,0x202f,0x202e,0x202d,0x202c,0x202a,0x2029,0x2027,
	0x2026,0x2025,0x2024,0x2023,0x2022,0x2020,0x201e,0x201d,0x201c,0x201b,0x201a,0x2019,0x2018,0x2015,0x2014,0x2013,
	0x2012,0x2011,0x2010,0x200f,0x200e,0x200b,0x200a,0x2009,0x2008,0x2007,0x2006,0x2005,0x2003,0x2001,0x2000,0x1fff,
	0x1ffe,0x1ffd,0x1ffc,0x1ffa,0x1ff9,0x1ff7,0x1ff6,0x1ff5,0x1ff4,0x1ff3,0x1ff1,0x1ff0,0x1fef,0x1fed,0x1fec,0x1feb,
	0x1fea,0x1fe8,0x1fe7,0x1fe6,0x1fe5,0x1fe3,0x1fe2,0x1fe1,0x1fdf,0x1fde,0x1fdd,0x1fdc,0x1fdb,0x1fd9,0x1fd8,0x1fd6,
	0x1fd5,0x1fd4,0x1fd3,0x1fd2,0x1fd1,0x1fcf,0x1fcd,0x1fcc,0x1fcb,0x1fca,0x1fc9,0x1fc8,0x1fc7,0x1fc4,0x1fc3,0x1fc2,
	0x1fc1,0x1fc0,0x1fbf,0x1fbc,0x1fbb,0x1fba,0x1fb9,0x1fb8,0x1fb7,0x1fb6,0x1fb4,0x1fb2,0x1fb1,0x1fb0,0x1faf,0x1fae,
	0x1fad,0x1fac,0x1faa,0x1fa8,0x1fa7,0x1fa6,0x1fa5,0x1fa4,0x1fa3,0x1fa1,0x1fa0,0x1f9e,0x1f9d,0x1f9c,0x1f9b,0x1f9a,
	0x1f98,0x1f97,0x1f96,0x1f95,0x1f93,0x1f92,0x1f91,0x1f90,0x1f8e,0x1f8d,0x1f8c,0x1f8b,0x1f89,0x1f88,0x1f87,0x1f85,
	0x1f84,0x1f83,0x1f82,0x1f81,0x1f7f,0x1f7e,0x1f7c,0x1f7b,0x1f7a,0x1f79,0x1f78,0x1f77,0x1f75,0x1f73,0x1f72,0x1f71,
	0x1f70,0x1f6f,0x1f6e,0x1f6d,0x1f6c,0x1f69,0x1f68,0x1f67,0x1f66,0x1f65,0x1f64,0x1f63,0x1f62,0x1f5f,0x1f5e,0x1f5d,
	0x1f5c,0x1f5b,0x1f5a,0x1f59,0x1f57,0x1f55,0x1f54,0x1f53,0x1f52,0x1f51,0x1f50,0x1f4e,0x1f4d,0x1f4b,0x1f4a,0x1f49,
	0x1f48,0x1f47,0x1f46,0x1f44,0x1f43,0x1f42,0x1f40,0x1f3f,0x1f3e,0x1f3d,0x1f3b,0x1f3a,0x1f39,0x1f38,0x1f36,0x1f35,
	0x1f34,0x1f32,0x1f31,0x1f30,0x1f2f,0x1f2e,0x1f2c,0x1f2b,0x1f29,0x1f28,0x1f27,0x1f26,0x1f25,0x1f24,0x1f22,0x1f21,
	0x1f1f,0x1f1e,0x1f1d,0x1f1c,0x1f1b,0x1f1a,0x1f19,0x1f16,0x1f15,0x1f14,0x1f13,0x1f12,0x1f11,0x1f10,0x1f0f,0x1f0c,
	0x1f0b,0x1f0a,0x1f09,0x1f08,0x1f07,0x1f06,0x1f04,0x1f02,0x1f01,0x1f00,0x1eff,0x1efe,0x1efd,0x1efc,0x1efa,0x1ef8,
	0x1ef7,0x1ef6,0x1ef5,0x1ef4,0x1ef3,0x1ef1,0x1ef0,0x1eef,0x1eed,0x1eec,0x1eeb,0x1eea,0x1ee8,0x1ee7,0x1ee6,0x1ee5,
	0x1ee3,0x1ee2,0x1ee1,0x1edf,0x1ede,0x1edd,0x1edc,0x1edb,0x1ed9,0x1ed8,0x1ed7,0x1ed5,0x1ed4,0x1ed3,0x1ed2,0x1ed1,
	0x1ed0,0x1ece,0x1ecc,0x1ecb,0x1eca,0x1ec9,0x1ec8,0x1ec7,0x1ec6,0x1ec3,0x1ec2,0x1ec1,0x1ec0,0x1ebf,0x1ebe,0x1ebd,
	0x1ebc,0x1eb9,0x1eb8,0x1eb7,0x1eb6,0x1eb5,0x1eb4,0x1eb3,0x1eb2,0x1eaf,0x1eae,0x1ead,0x1eac,0x1eab,0x1eaa,0x1ea9,
	0x1ea7,0x1ea6,0x1ea4,0x1ea3,0x1ea2,0x1ea1,0x1ea0,0x1e9e,0x1e9d,0x1e9c,0x1e9a,0x1e99,0x1e98,0x1e97,0x1e95,0x1e94,
	0x1e93,0x1e92,0x1e90,0x1e8f,0x1e8e,0x1e8d,0x1e8b,0x1e8a,0x1e89,0x1e88,0x1e86,0x1e85,0x1e84,0x1e82,0x1e81,0x1e80,
	0x1e7f,0x1e7d,0x1e7c,0x1e7b,0x1e7a,0x1e79,0x1e78,0x1e76,0x1e75,0x1e73,0x1e72,0x1e71,0x1e70,0x1e6f,0x1e6d,0x1e6c,
	0x1e6b,0x1e6a,0x1e68,0x1e67,0x1e66,0x1e65,0x1e63,0x1e62,0x1e61,0x1e60,0x1e5e,0x1e5d,0x1e5c,0x1e5a,0x1e59,0x1e58,
	0x1e57,0x1e56,0x1e55,0x1e53,0x1e52,0x1e50,0x1e4f,0x1e4e,0x1e4d,0x1e4c,0x1e4b,0x1e49,0x1e48,0x1e46,0x1e45,0x1e44,
	0x1e43,0x1e42,0x1e41,0x1e40,0x1e3d,0x1e3c,0x1e3b,0x1e3a,0x1e39,0x1e38,0x1e37,0x1e36,0x1e33,0x1e32,0x1e31,0x1e30,
	0x1e2f,0x1e2e,0x1e2d,0x1e2c,0x1e2a,0x1e28,0x1e27,0x1e26,0x1e25,0x1e24,0x1e23,0x1e22,0x1e20,0x1e1e,0x1e1d,0x1e1c,
	0x1e1b,0x1e1a,0x1e19,0x1e17,0x1e16,0x1e15,0x1e13,0x1e12,0x1e11,0x1e10,0x1e0f,0x1e0d,0x1e0c,0x1e0b,0x1e09,0x1e08,
	0x1e07,0x1e06,0x1e04,0x1e03,0x1e02,0x1e01,0x1e00,0x1dfe,0x1dfd,0x1dfc,0x1dfa,0x1df9,0x1df8,0x1df7,0x1df6,0x1df4,
	0x1df3,0x1df1,0x1df0,0x1def,0x1dee,0x1ded,0x1dec,0x1deb,0x1de9,0x1de7,0x1de6,0x1de5,0x1de4,0x1de3,0x1de2,0x1de1,
	0x1de0,0x1ddd,0x1ddc,0x1ddb,0x1dda,0x1dd9,0x1dd8,0x1dd7,0x1dd6,0x1dd3,0x1dd2,0x1dd1,0x1dd0,0x1dcf,0x1dce,0x1dcd,
	0x1dcb,0x1dca,0x1dc8,0x1dc7,0x1dc6,0x1dc5,0x1dc4,0x1dc3,0x1dc1,0x1dc0,0x1dbe,0x1dbd,0x1dbc,0x1dbb,0x1dba,0x1db9,
	0x1db7,0x1db6,0x1db5,0x1db3,0x1db2,0x1db1,0x1db0,0x1dae,0x1dad,0x1dac,0x1dab,0x1da9,0x1da8,0x1da7,0x1da6,0x1da4,
	0x1da3,0x1da2,0x1da1,0x1da0,0x1d9e,0x1d9d,0x1d9b,0x1d9a,0x1d99,0x1d98,0x1d97,0x1d96,0x1d94,0x1d93,0x1d91,0x1d90,
	0x1d8f,0x1d8e,0x1d8d,0x1d8c,0x1d8b,0x1d88,0x1d87,0x1d86,0x1d85,0x1d84,0x1d83,0x1d82,0x1d81,0x1d7e,0x1d7d,0x1d7c,
	0x1d7b,0x1d7a,0x1d79,0x1d78,0x1d77,0x1d75,0x1d73,0x1d72,0x1d71,0x1d70,0x1d6f,0x1d6e,0x1d6d,0x1d6b,0x1d69,0x1d68,
	0x1d67,0x1d66,0x1d65,0x1d64,0x1d62,0x1d61,0x1d60,0x1d5e,0x1d5d,0x1d5c,0x1d5b,0x1d5a,0x1d58,0x1d57,0x1d56,0x1d54,
	0x1d53,0x1d52,0x1d51,0x1d4f,0x1d4e,0x1d4d,0x1d4c,0x1d4b,0x1d49,0x1d48,0x1d47,0x1d45,0x1d44,0x1d43,0x1d42,0x1d41,
	0x1d3f,0x1d3e,0x1d3d,0x1d3c,0x1d3b,0x1d3a,0x1d39,0x1d37,0x1d35,0x1d34,0x1d33,0x1d32,0x1d31,0x1d30,0x1d2f,0x1d2e,
	0x1d2b,0x1d2a,0x1d29,0x1d28,0x1d27,0x1d26,0x1d25,0x1d24,0x1d23,0x1d20,0x1d1f,0x1d1e,0x1d1d,0x1d1c,0x1d1b,0x1d1a,
	0x1d18,0x1d17,0x1d15,0x1d14,0x1d13,0x1d12,0x1d11,0x1d10,0x1d0e,0x1d0d,0x1d0b,0x1d0a,0x1d09,0x1d08,0x1d07,0x1d06,
	0x1d04,0x1d03,0x1d02,0x1d00,0x1cff,0x1cfe,0x1cfd,0x1cfc,0x1cfa,0x1cf9,0x1cf8,0x1cf7,0x1cf5,0x1cf4,0x1cf3,0x1cf2,
	0x1cf0,0x1cef,0x1cee,0x1ced,0x1cec,0x1cea,0x1ce9,0x1ce8,0x1ce6,0x1ce5,0x1ce4,0x1ce3,0x1ce2,0x1ce1,0x1cdf,0x1cdd,
	0x1cdc,0x1cdb,0x1cda,0x1cd9,0x1cd8,0x1cd7,0x1cd6,0x1cd3,0x1cd2,0x1cd1,0x1cd0,0x1ccf,0x1cce,0x1ccd,0x1ccc,0x1cc9,
	0x1cc8,0x1cc7,0x1cc6,0x1cc5,0x1cc4,0x1cc3,0x1cc2,0x1cc1,0x1cbe,0x1cbd,0x1cbc,0x1cbb,0x1cba,0x1cb9,0x1cb8,0x1cb7,
	0x1cb5,0x1cb3,0x1cb2,0x1cb1,0x1cb0,0x1caf,0x1cae,0x1cad,0x1cab,0x1caa,0x1ca8,0x1ca7,0x1ca6,0x1ca5,0x1ca4,0x1ca2,
	0x1ca1,0x1ca0,0x1c9f,0x1c9e,0x1c9c,0x1c9a,0x1c99,0x1c98,0x1c97,0x1c96,0x1c95,0x1c94,0x1c92,0x1c91,0x1c8f,0x1c8e,
	0x1c8d,0x1c8c,0x1c8b,0x1c8a,0x1c88,0x1c87,0x1c86,0x1c84,0x1c83,0x1c82,0x1c81,0x1c80,0x1c7e,0x1c7d,0x1c7c,0x1c7b,
	0x1c7a,0x1c78,0x1c77,0x1c76,0x1c74,0x1c73,0x1c72,0x1c71,0x1c70,0x1c6f,0x1c6d,0x1c6c,0x1c6b,0x1c69,0x1c68,0x1c67,
	0x1c66,0x1c65,0x1c64,0x1c62,0x1c61,0x1c5f,0x1c5e,0x1c5d,0x1c5c,0x1c5b,0x1c5a,0x1c59,0x1c57,0x1c55,0x1c54,0x1c53,
	0x1c52,0x1c51,0x1c50,0x1c4f,0x1c4e,0x1c4b,0x1c4a,0x1c49,0x1c48,0x1c47,0x1c46,0x1c45,0x1c44,0x1c43,0x1c40,0x1c3f,
	0x1c3e,0x1c3d,0x1c3c,0x1c3b,0x1c3a,0x1c39,0x1c37,0x1c35,0x1c34,0x1c33,0x1c32,0x1c31,0x1c30,0x1c2f,0x1c2d,0x1c2c,
	0x1c2b,0x1c29,0x1c28,0x1c27,0x1c26,0x1c25,0x1c24,0x1c22,0x1c21,0x1c20,0x1c1e,0x1c1d,0x1c1c,0x1c1b,0x1c1a,0x1c18,
	0x1c17,0x1c16,0x1c15,0x1c13,0x1c12,0x1c11,0x1c10,0x1c0e,0x1c0d,0x1c0c,0x1c0b,0x1c0a,0x1c08,0x1c07,0x1c06,0x1c04,
	0x1c03,0x1c02,0x1c01,0x1c00,0x1bff,0x1bfd,0x1bfc,0x1bfa,0x1bf9,0x1bf8,0x1bf7,0x1bf6,0x1bf5,0x1bf4,0x1bf2,0x1bf0,
	0x1bef,0x1bee,0x1bed,0x1bec,0x1beb,0x1bea,0x1be9,0x1be6,0x1be5,0x1be4,0x1be3,0x1be2,0x1be1,0x1be0,0x1bdf,0x1bde,
	0x1bdd,0x1bda,0x1bd9,0x1bd8,0x1bd7,0x1bd6,0x1bd5,0x1bd4,0x1bd3,0x1bd1,0x1bcf,0x1bce,0x1bcd,0x1bcc,0x1bcb,0x1bca,
	0x1bc9,0x1bc7,0x1bc6,0x1bc4,0x1bc3,0x1bc2,0x1bc1,0x1bc0,0x1bbf,0x1bbd,0x1bbc,0x1bbb,0x1bb9,0x1bb8,0x1bb7,0x1bb6,
	0x1bb5,0x1bb3,0x1bb2,0x1bb1,0x1bb0,0x1bae,0x1bad,0x1bac,0x1bab,0x1ba9,0x1ba8,0x1ba7,0x1ba6,0x1ba5,0x1ba3,0x1ba2,
	0x1ba1,0x1b9f,0x1b9e,0x1b9d,0x1b9c,0x1b9b,0x1b9a,0x1b98,0x1b97,0x1b96,0x1b94,0x1b93,0x1b92,0x1b91,0x1b90,0x1b8f,
	0x1b8e,0x1b8c,0x1b8a,0x1b89,0x1b88,0x1b87,0x1b86,0x1b85,0x1b84,0x1b83,0x1b80,0x1b7f,0x1b7e,0x1b7d,0x1b7c,0x1b7b,
	0x1b7a,0x1b79,0x1b78,0x1b75,0x1b74,0x1b73,0x1b72,0x1b71,0x1b70,0x1b6f,0x1b6e,0x1b6c,0x1b6a,0x1b69,0x1b68,0x1b67,
	0x1b66,0x1b65,0x1b63,0x1b62,0x1b61,0x1b60,0x1b5f,0x1b5e,0x1b5c,0x1b5b,0x1b5a,0x1b58,0x1b57,0x1b56,0x1b55,0x1b54,
	0x1b53,0x1b51,0x1b50,0x1b4e,0x1b4d,0x1b4c,0x1b4b,0x1b4a,0x1b49,0x1b48,0x1b47,0x1b45,0x1b43,0x1b42,0x1b41,0x1b40,
	0x1b3f,0x1b3e,0x1b3d,0x1b3c,0x1b39,0x1b38,0x1b37,0x1b36,0x1b35,0x1b34,0x1b33,0x1b32,0x1b31,0x1b30,0x1b2d,0x1b2c,
	0x1b2b,0x1b2a,0x1b29,0x1b28,0x1b27,0x1b26,0x1b24,0x1b22,0x1b21,0x1b20,0x1b1f,0x1b1e,0x1b1d,0x1b1c,0x1b1b,0x1b19,
	0x1b18,0x1b16,0x1b15,0x1b14,0x1b13,0x1b12,0x1b11,0x1b0f,0x1b0e,0x1b0d,0x1b0b,0x1b0a,0x1b09,0x1b08,0x1b07,0x1b06,
	0x1b04,0x1b03,0x1b02,0x1b00,0x1aff,0x1afe,0x1afd,0x1afc,0x1afa,0x1af9,0x1af8,0x1af7,0x1af6,0x1af4,0x1af3,0x1af2,
	0x1af1,0x1aef,0x1aee,0x1aed,0x1aec,0x1aeb,0x1ae9,0x1ae8,0x1ae7,0x1ae5,0x1ae4,0x1ae3,0x1ae2,0x1ae1,0x1ae0,0x1adf,
	0x1add,0x1adc,0x1ada,0x1ad9,0x1ad8,0x1ad7,0x1ad6,0x1ad5,0x1ad4,0x1ad2,0x1ad0,0x1acf,0x1ace,0x1acd,0x1acc,0x1acb,
	0x1aca,0x1ac7,0x1ac6,0x1ac5,0x1ac4,0x1ac3,0x1ac2,0x1ac1,0x1ac0,0x1abf,0x1abc,0x1abb,0x1aba,0x1ab9,0x1ab8,0x1ab7,
	0x1ab6,0x1ab5,0x1ab4,0x1ab2,0x1ab0,0x1aaf,0x1aae,0x1aad,0x1aac,0x1aab,0x1aaa,0x1aa9,0x1aa7,0x1aa6,0x1aa4,0x1aa3,
	0x1aa2,0x1aa1,0x1aa0,0x1a9f,0x1a9e,0x1a9c,0x1a9b,0x1a9a,0x1a98,0x1a97,0x1a96,0x1a95,0x1a94,0x1a93,0x1a91,0x1a90,
	0x1a8f,0x1a8e,0x1a8c,0x1a8b,0x1a8a,0x1a89,0x1a87,0x1a86,0x1a85,0x1a84,0x1a83,0x1a82,0x1a80,0x1a7f,0x1a7e,0x1a7c,
	0x1a7b,0x1a7a,0x1a79,0x1a78,0x1a77,0x1a75,0x1a74,0x1a73,0x1a71,0x1a70,0x1a6f,0x1a6e,0x1a6d,0x1a6c,0x1a6b,0x1a69,
	0x1a68,0x1a66,0x1a65,0x1a64,0x1a63,0x1a62,0x1a61,0x1a60,0x1a5f,0x1a5d,0x1a5b,0x1a5a,0x1a59,0x1a58,0x1a57,0x1a56,
	0x1a55,0x1a54,0x1a53,0x1a50,0x1a4f,0x1a4e,0x1a4d,0x1a4c,0x1a4b,0x1a4a,0x1a49,0x1a48,0x1a47,0x1a44,0x1a43,0x1a42,
	0x1a41,0x1a40,0x1a3f,0x1a3e,0x1a3d,0x1a3b,0x1a3a,0x1a38,0x1a37,0x1a36,0x1a35,0x1a34,0x1a33,0x1a32,0x1a30,0x1a2f,
	0x1a2e,0x1a2c,0x1a2b,0x1a2a,0x1a29,0x1a28,0x1a27,0x1a25,0x1a24,0x1a23,0x1a21,0x1a20,0x1a1f,0x1a1e,0x1a1d,0x1a1c,
	0x1a1a,0x1a19,0x1a18,0x1a17,0x1a15,0x1a14,0x1a13,0x1a12,0x1a11,0x1a0f,0x1a0e,0x1a0d,0x1a0c,0x1a0b,0x1a09,0x1a08,
	0x1a07,0x1a06,0x1a04,0x1a03,0x1a02,0x1a01,0x1a00,0x19ff,0x19fd,0x19fc,0x19fa,0x19f9,0x19f8,0x19f7,0x19f6,0x19f5,
	0x19f4,0x19f3,0x19f1,0x19ef,0x19ee,0x19ed,0x19ec,0x19eb,0x19ea,0x19e9,0x19e8,0x19e7,0x19e4,0x19e3,0x19e2,0x19e1,
	0x19e0,0x19df,0x19de,0x19dd,0x19dc,0x19d9,0x19d8,0x19d7,0x19d6,0x19d5,0x19d4,0x19d3,0x19d2,0x19d1,0x19d0,0x19cd,
	0x19cc,0x19cb,0x19ca,0x19c9,0x19c8,0x19c7,0x19c6,0x19c5,0x19c3,0x19c1,0x19c0,0x19bf,0x19be,0x19bd,0x19bc,0x19bb,
	0x19ba,0x19b8,0x19b7,0x19b5,0x19b4,0x19b3,0x19b2,0x19b1,0x19b0,0x19ae,0x19ad,0x19ac,0x19ab,0x19a9,0x19a8,0x19a7,
	0x19a6,0x19a5,0x19a3,0x19a2,0x19a1,0x19a0,0x199f,0x199d,0x199c,0x199b,0x199a,0x1998,0x1997,0x1996,0x1995,0x1994,
	0x1992,0x1991,0x1990,0x198f,0x198e,0x198d,0x198b,0x198a,0x1988,0x1987,0x1986,0x1985,0x1984,0x1983,0x1982,0x1980,
	0x197f,0x197e,0x197c,0x197b,0x197a,0x1979,0x1978,0x1977,0x1975,0x1974,0x1973,0x1972,0x1971,0x196f,0x196e,0x196d,
	0x196c,0x196a,0x1969,0x1968,0x1967,0x1966,0x1965,0x1963,0x1962,0x1961,0x195f,0x195e,0x195d,0x195c,0x195b,0x195a,
	0x1959,0x1957,0x1956,0x1955,0x1953,0x1952,0x1951,0x1950,0x194f,0x194e,0x194d,0x194b,0x194a,0x1948,0x1947,0x1946,
	0x1945,0x1944,0x1943,0x1942,0x1941,0x1940,0x193d,0x193c,0x193b,0x193a,0x1939,0x1938,0x1937,0x1936,0x1935,0x1934,
	0x1931,0x1930,0x192f,0x192e,0x192d,0x192c,0x192b,0x192a,0x1929,0x1928,0x1925,0x1924,0x1923,0x1922,0x1921,0x1920,
	0x191f,0x191e,0x191d,0x191b,0x1919,0x1918,0x1917,0x1916,0x1915,0x1914,0x1913,0x1912,0x1910,0x190f,0x190e,0x190c,
	0x190b,0x190a,0x1909,0x1908,0x1907,0x1905,0x1904,0x1903,0x1902,0x1900,0x18ff,0x18fe,0x18fd,0x18fc,0x18fb,0x18f9,
	0x18f8,0x18f7,0x18f6,0x18f4,0x18f3,0x18f2,0x18f1,0x18f0,0x18ee,0x18ed,0x18ec,0x18eb,0x18ea,0x18e8,0x18e7,0x18e6,
	0x18e5,0x18e3,0x18e2,0x18e1,0x18e0,0x18df,0x18de,0x18dd,0x18db,0x18da,0x18d8,0x18d7,0x18d6,0x18d5,0x18d4,0x18d3,
	0x18d2,0x18d1,0x18cf,0x18ce,0x18cc,0x18cb,0x18ca,0x18c9,0x18c8,0x18c7,0x18c6,0x18c5,0x18c3,0x18c1,0x18c0,0x18bf,
	0x18be,0x18bd,0x18bc,0x18bb,0x18ba,0x18b9,0x18b6,0x18b5,0x18b4,0x18b3,0x18b2,0x18b1,0x18b0,0x18af,0x18ae,0x18ad,
	0x18ab,0x18a9,0x18a8,0x18a7,0x18a6,0x18a5,0x18a4,0x18a3,0x18a2,0x18a1,0x189f,0x189d,0x189c,0x189b,0x189a,0x1899,
	0x1898,0x1897,0x1896,0x1894,0x1893,0x1891,0x1890,0x188f,0x188e,0x188d,0x188c,0x188b,0x1889,0x1888,0x1887,0x1885,
	0x1884,0x1883,0x1882,0x1881,0x1880,0x187e,0x187d,0x187c,0x187b,0x1879,0x1878,0x1877,0x1876,0x1875,0x1874,0x1872,
	0x1871,0x1870,0x186f,0x186e,0x186c,0x186b,0x186a,0x1869,0x1867,0x1866,0x1865,0x1864,0x1863,0x1862,0x1860,0x185f,
	0x185e,0x185d,0x185c,0x185b,0x185a,0x1859,0x1856,0x1855,0x1854,0x1853,0x1852,0x1851,0x1850,0x184f,0x184e,0x184d,
	0x184b,0x1849,0x1848,0x1847,0x1846,0x1845,0x1844,0x1843,0x1842,0x1841,0x183f,0x183d,0x183c,0x183b,0x183a,0x1839,
	0x1838,0x1837,0x1836,0x1835,0x1833,0x1832,0x1830,0x182f,0x182e,0x182d,0x182c,0x182b,0x182a,0x1828,0x1827,0x1826,
	0x1825,0x1823,0x1822,0x1821,0x1820,0x181f,0x181e,0x181c,0x181b,0x181a,0x1819,0x1817,0x1816,0x1815,0x1814,0x1813,
	0x1812,0x1810,0x180f,0x180e,0x180d,0x180c,0x180a,0x1809,0x1808,0x1807,0x1805,0x1804,0x1803,0x1802,0x1801,0x1800,
	0x17ff,0x17fd,0x17fc,0x17fb,0x17f9,0x17f8,0x17f7,0x17f6,0x17f5,0x17f4,0x17f3,0x17f1,0x17f0,0x17ef,0x17ed,0x17ec,
	0x17eb,0x17ea,0x17e9,0x17e8,0x17e7,0x17e6,0x17e4,0x17e2,0x17e1,0x17e0,0x17df,0x17de,0x17dd,0x17dc,0x17db,0x17da,
	0x17d9,0x17d6,0x17d5,0x17d4,0x17d3,0x17d2,0x17d1,0x17d0,0x17cf,0x17ce,0x17cd,0x17ca,0x17c9,0x17c8,0x17c7,0x17c6,
	0x17c4,0x17c3,0x17c2,0x17c1,0x17c0,0x17bf,0x17be,0x17bd,0x17bb,0x17b9,0x17b8,0x17b7,0x17b6,0x17b5,0x17b4,0x17b3,
	0x17b2,0x17b1,0x17af,0x17ae,0x17ac,0x17ab,0x17aa,0x17a9,0x17a8,0x17a7,0x17a6,0x17a5,0x17a3,0x17a2,0x17a1,0x179f,
	0x179e,0x179d,0x179c,0x179b,0x179a,0x1799,0x1797,0x1796,0x1795,0x1794,0x1792,0x1791,0x1790,0x178f,0x178e,0x178d,
	0x178c,0x178a,0x1789,0x1788,0x1787,0x1785,0x1784,0x1783,0x1782,0x1781,0x1780,0x177e,0x177d,0x177c,0x177b,0x177a,
	0x1779,0x1777,0x1776,0x1775,0x1774,0x1772,0x1771,0x1770,0x176f,0x176e,0x176d,0x176c,0x176a,0x1769,0x1768,0x1766,
	0x1765,0x1764,0x1763,0x1762,0x1761,0x1760,0x175f,0x175d,0x175c,0x175a,0x1759,0x1758,0x1757,0x1756,0x1755,0x1754,
	0x1753,0x1752,0x1750,0x174e,0x174d,0x174c,0x174b,0x174a,0x1749,0x1748,0x1747,0x1746,0x1745,0x1742,0x1741,0x1740,
	0x173f,0x173e,0x173d,0x173c,0x173b,0x173a,0x1739,0x1738,0x1735,0x1734,0x1733,0x1732,0x1731,0x1730,0x172f,0x172e,
	0x172d,0x172c,0x172b,0x1728,0x1727,0x1726,0x1725,0x1724,0x1723,0x1722,0x1721,0x1720,0x171f,0x171d,0x171b,0x171a,
	0x1719,0x1718,0x1717,0x1716,0x1715,0x1714,0x1713,0x1711,0x1710,0x170f,0x170d,0x170c,0x170b,0x170a,0x1709,0x1708,
	0x1707,0x1705,0x1704,0x1703,0x1702,0x1700,0x16ff,0x16fe,0x16fd,0x16fc,0x16fb,0x16f9,0x16f8,0x16f7,0x16f6,0x16f5,
	0x16f3,0x16f2,0x16f1,0x16f0,0x16ef,0x16ed,0x16ec,0x16eb,0x16ea,0x16e9,0x16e8,0x16e6,0x16e5,0x16e4,0x16e3,0x16e1,
	0x16e0,0x16df,0x16de,0x16dd,0x16dc,0x16db,0x16d9,0x16d8,0x16d7,0x16d5,0x16d4,0x16d3,0x16d2,0x16d1,0x16d0,0x16cf,
	0x16ce,0x16cc,0x16cb,0x16ca,0x16c8,0x16c7,0x16c6,0x16c5,0x16c4,0x16c3,0x16c2,0x16c1,0x16bf,0x16be,0x16bc,0x16bb,
	0x16ba,0x16b9,0x16b8,0x16b7,0x16b6,0x16b5,0x16b4,0x16b2,0x16b0,0x16af,0x16ae,0x16ad,0x16ac,0x16ab,0x16aa,0x16a9,
	0x16a8,0x16a7,0x16a6,0x16a3,0x16a2,0x16a1,0x16a0,0x169f,0x169e,0x169d,0x169c,0x169b,0x169a,0x1698,0x1696,0x1695,
	0x1695,0x1693,0x1692,0x1690,0x168f,0x168e,0x168d,0x168c,0x168b,0x168a,0x1689,0x1688,0x1686,0x1684,0x1683,0x1682,
	0x1681,0x1680,0x167f,0x167e,0x167d,0x167c,0x167b,0x167a,0x1677,0x1676,0x1675,0x1674,0x1673,0x1672,0x1671,0x1670,
	0x166f,0x166e,0x166d,0x166a,0x1669,0x1668,0x1667,0x1666,0x1665,0x1664,0x1663,0x1662,0x1661,0x1660,0x165e,0x165c,
	0x165b,0x165a,0x1659,0x1658,0x1657,0x1656,0x1655,0x1654,0x1653,0x1651,0x1650,0x164e,0x164d,0x164c,0x164b,0x164a,
	0x1649,0x1648,0x1647,0x1646,0x1644,0x1643,0x1641,0x1640,0x163f,0x163e,0x163d,0x163c,0x163b,0x163a,0x1638,0x1637,
	0x1636,0x1635,0x1633,0x1632,0x1631,0x1630,0x162f,0x162e,0x162d,0x162b,0x162a,0x1629,0x1628,0x1627,0x1625,0x1624,
	0x1623,0x1622,0x1621,0x161f,0x161e,0x161d,0x161c,0x161b,0x161a,0x1618,0x1617,0x1616,0x1615,0x1614,0x1612,0x1611,
	0x1610,0x160f,0x160e,0x160d,0x160c,0x160a,0x1609,0x1608,0x1607,0x1605,0x1604,0x1603,0x1602,0x1601,0x1600,0x15ff,
	0x15fe,0x15fc,0x15fb,0x15f9,0x15f8,0x15f7,0x15f6,0x15f5,0x15f4,0x15f3,0x15f2,0x15f1,0x15ef,0x15ee,0x15ec,0x15eb,
	0x15ea,0x15e9,0x15e8,0x15e7,0x15e6,0x15e5,0x15e4,0x15e3,0x15e1,0x15df,0x15de,0x15dd,0x15dc,0x15db,0x15da,0x15d9,
	0x15d8,0x15d7,0x15d6,0x15d3,0x15d2,0x15d1,0x15d0,0x15cf,0x15ce,0x15cd,0x15cc,0x15cb,0x15ca,0x15c9,0x15c8,0x15c5,
	0x15c4,0x15c3,0x15c2,0x15c1,0x15c0,0x15bf,0x15be,0x15bd,0x15bc,0x15bb,0x15b9,0x15b7,0x15b6,0x15b5,0x15b4,0x15b3,
	0x15b2,0x15b1,0x15b0,0x15af,0x15ad,0x15ac,0x15aa,0x15a9,0x15a8,0x15a7,0x15a6,0x15a5,0x15a4,0x15a3,0x15a2,0x15a0,
	0x159f,0x159e,0x159c,0x159b,0x159a,0x1599,0x1598,0x1597,0x1596,0x1594,0x1593,0x1592,0x1591,0x1590,0x158e,0x158d,
	0x158c,0x158b,0x158a,0x1589,0x1587,0x1586,0x1585,0x1584,0x1583,0x1581,0x1580,0x157f,0x157e,0x157d,0x157c,0x157a,
	0x1579,0x1578,0x1577,0x1576,0x1575,0x1573,0x1572,0x1571,0x1570,0x156e,0x156d,0x156c,0x156b,0x156a,0x1569,0x1568,
	0x1567,0x1566,0x1565,0x1564,0x1561,0x1560,0x155f,0x155e,0x155d,0x155c,0x155b,0x155a,0x1559,0x1558,0x1557,0x1555,
	0x1553,0x1552,0x1551,0x1550,0x154f,0x154e,0x154d,0x154c,0x154b,0x154a,0x1548,0x1547,0x1545,0x1544,0x1543,0x1542,
	0x1541,0x1540,0x153f,0x153e,0x153d,0x153c,0x153a,0x1539,0x1537,0x1536,0x1535,0x1534,0x1533,0x1532,0x1531,0x1530,
	0x152f,0x152d,0x152c,0x152b,0x1529,0x1528,0x1527,0x1526,0x1525,0x1524,0x1523,0x1522,0x1520,0x151f,0x151e,0x151d,
	0x151c,0x151a,0x1519,0x1518,0x1517,0x1516,0x1515,0x1513,0x1512,0x1511,0x1510,0x150f,0x150e,0x150c,0x150b,0x150a,
	0x1509,0x1508,0x1506,0x1505,0x1504,0x1503,0x1502,0x1501,0x1500,0x14fe,0x14fd,0x14fc,0x14fb,0x14fa,0x14f8,0x14f7,
	0x14f6,0x14f5,0x14f4,0x14f3,0x14f2,0x14f0,0x14ef,0x14ee,0x14ed,0x14eb,0x14ea,0x14e9,0x14e8,0x14e7,0x14e6,0x14e5,
	0x14e4,0x14e2,0x14e1,0x14e0,0x14de,0x14dd,0x14dc,0x14db,0x14da,0x14d9,0x14d8,0x14d7,0x14d6,0x14d4,0x14d3,0x14d1,
	0x14d0,0x14cf,0x14ce,0x14cd,0x14cc,0x14cb,0x14ca,0x14c9,0x14c8,0x14c7,0x14c4,0x14c3,0x14c2,0x14c1,0x14c0,0x14bf,
	0x14be,0x14bd,0x14bc,0x14bb,0x14ba,0x14b9,0x14b6,0x14b5,0x14b4,0x14b3,0x14b2,0x14b1,0x14b0,0x14af,0x14ae,0x14ad,
	0x14ac,0x14ab,0x14a8,0x14a7,0x14a6,0x14a5,0x14a4,0x14a3,0x14a2,0x14a1,0x14a0,0x149f,0x149e,0x149c,0x149a,0x1499,
	0x1498,0x1497,0x1496,0x1495,0x1494,0x1493,0x1492,0x1491,0x148f,0x148e,0x148c,0x148b,0x148a,0x1489,0x1488,0x1487,
	0x1486,0x1485,0x1484,0x1482,0x1481,0x1480,0x147f,0x147d,0x147c,0x147b,0x147a,0x1479,0x1478,0x1477,0x1476,0x1474,
	0x1473,0x1472,0x1471,0x146f,0x146e,0x146d,0x146c,0x146b,0x146a,0x1469,0x1467,0x1466,0x1465,0x1464,0x1463,0x1461,
	0x1460,0x145f,0x145e,0x145d,0x145c,0x145a,0x1459,0x1458,0x1457,0x1456,0x1455,0x1453,0x1452,0x1451,0x1450,0x144f,
	0x144d,0x144c,0x144b,0x144a,0x1449,0x1448,0x1447,0x1445,0x1444,0x1443,0x1442,0x1440,0x143f,0x143e,0x143d,0x143c,
	0x143b,0x143a,0x1439,0x1438,0x1437,0x1436,0x1435,0x1433,0x1432,0x1431,0x142f,0x142e,0x142d,0x142c,0x142b,0x142a,
	0x1429,0x1428,0x1427,0x1426,0x1424,0x1423,0x1421,0x1420,0x141f,0x141e,0x141d,0x141c,0x141b,0x141a,0x1419,0x1418,
	0x1417,0x1414,0x1413,0x1412,0x1411,0x1410,0x140f,0x140e,0x140d,0x140c,0x140b,0x140a,0x1409,0x1406,0x1405,0x1404,
	0x1403,0x1402,0x1401,0x1400,0x13ff,0x13fe,0x13fd,0x13fc,0x13fb,0x13fa,0x13f7,0x13f6,0x13f5,0x13f4,0x13f3,0x13f2,
	0x13f1,0x13f0,0x13ef,0x13ee,0x13ed,0x13ec,0x13ea,0x13e8,0x13e7,0x13e6,0x13e5,0x13e4,0x13e3,0x13e2,0x13e1,0x13e0,
	0x13df,0x13de,0x13dc,0x13db,0x13d9,0x13d8,0x13d7,0x13d6,0x13d5,0x13d4,0x13d3,0x13d2,0x13d1,0x13cf,0x13ce,0x13cd,
	0x13cc,0x13ca,0x13c9,0x13c8,0x13c7,0x13c6,0x13c5,0x13c4,0x13c3,0x13c1,0x13c0,0x13bf,0x13be,0x13bc,0x13bb,0x13ba,
	0x13b9,0x13b8,0x13b7,0x13b6,0x13b5,0x13b3,0x13b2,0x13b1,0x13b0,0x13af,0x13ad,0x13ac,0x13ab,0x13aa,0x13a9,0x13a8,
	0x13a7,0x13a5,0x13a4,0x13a3,0x13a1,0x13a0,0x139f,0x139e,0x139d,0x139c,0x139b,0x139a,0x1399,0x1397,0x1396,0x1395,
	0x1394,0x1392,0x1391,0x1390,0x138f,0x138e,0x138d,0x138c,0x138b,0x138a,0x1388,0x1387,0x1386,0x1385,0x1384,0x1382,
	0x1381,0x1380,0x137f,0x137e,0x137d,0x137c,0x137a,0x1379,0x1378,0x1377,0x1376,0x1375,0x1373,0x1372,0x1371,0x1370,
	0x136f,0x136e,0x136c,0x136b,0x136a,0x1369,0x1368,0x1367,0x1366,0x1365,0x1363,0x1362,0x1361,0x1360,0x135f,0x135d,
	0x135c,0x135b,0x135a,0x1359,0x1358,0x1357,0x1356,0x1354,0x1353,0x1352,0x1351,0x134f,0x134e,0x134d,0x134c,0x134b,
	0x134a,0x1349,0x1348,0x1347,0x1346,0x1344,0x1343,0x1342,0x1340,0x133f,0x133e,0x133d,0x133c,0x133b,0x133a,0x1339,
	0x1338,0x1337,0x1335,0x1334,0x1332,0x1331,0x1330,0x132f,0x132e,0x132d,0x132c,0x132b,0x132a,0x1329,0x1328,0x1327,
	0x1325,0x1323,0x1322,0x1321,0x1320,0x131f,0x131e,0x131d,0x131c,0x131b,0x131a,0x1319,0x1318,0x1315,0x1314,0x1313,
	0x1312,0x1311,0x1310,0x130f,0x130e,0x130d,0x130c,0x130b,0x130a,0x1309,0x1308,0x1305,0x1304,0x1303,0x1302,0x1301,
	0x1300,0x12ff,0x12fe,0x12fd,0x12fc,0x12fb,0x12fa,0x12f8,0x12f6,0x12f5,0x12f4,0x12f3,0x12f2,0x12f1,0x12f0,0x12ef,
	0x12ee,0x12ed,0x12ec,0x12eb,0x12e9,0x12e8,0x12e6,0x12e5,0x12e4,0x12e3,0x12e2,0x12e1,0x12e0,0x12df,0x12de,0x12dd,
	0x12db,0x12da,0x12d9,0x12d7,0x12d6,0x12d5,0x12d4,0x12d3,0x12d2,0x12d1,0x12d0,0x12cf,0x12ce,0x12cc,0x12cb,0x12ca,
	0x12c9,0x12c7,0x12c6,0x12c5,0x12c4,0x12c3,0x12c2,0x12c1,0x12c0,0x12be,0x12bd,0x12bc,0x12bb,0x12ba,0x12b8,0x12b7,
	0x12b6,0x12b5,0x12b4,0x12b3,0x12b2,0x12b1,0x12af,0x12ae,0x12ad,0x12ac,0x12ab,0x12a9,0x12a8,0x12a7,0x12a6,0x12a5,
	0x12a4,0x12a3,0x12a1,0x12a0,0x129f,0x129e,0x129d,0x129c,0x129b,0x1299,0x1298,0x1297,0x1296,0x1295,0x1294,0x1292,
	0x1291,0x1290,0x128f,0x128e,0x128d,0x128c,0x128a,0x1289,0x1288,0x1287,0x1286,0x1284,0x1283,0x1282,0x1281,0x1280,
	0x127f,0x127d,0x127c,0x127b,0x127a,0x1279,0x1278,0x1277,0x1276,0x1275,0x1274,0x1273,0x1271,0x1270,0x126f,0x126d,
	0x126c,0x126b,0x126a,0x1269,0x1268,0x1267,0x1266,0x1265,0x1264,0x1262,0x1261,0x1260,0x125f,0x125d,0x125c,0x125b,
	0x125a,0x1259,0x1258,0x1257,0x1256,0x1255,0x1253,0x1252,0x1251,0x1250,0x124f,0x124d,0x124c,0x124b,0x124a,0x1249,
	0x1248,0x1247,0x1246,0x1245,0x1243,0x1242,0x1241,0x1240,0x123f,0x123d,0x123c,0x123b,0x123a,0x1239,0x1238,0x1237,
	0x1236,0x1234,0x1233,0x1232,0x1231,0x1230,0x122f,0x122e,0x122c,0x122b,0x122a,0x1229,0x1228,0x1227,0x1225,0x1224,
	0x1223,0x1222,0x1221,0x1220,0x121f,0x121e,0x121c,0x121b,0x121a,0x1219,0x1218,0x1216,0x1215,0x1214,0x1213,0x1212,
	0x1211,0x1210,0x120f,0x120e,0x120c,0x120b,0x120a,0x1209,0x1207,0x1206,0x1205,0x1204,0x1203,0x1202,0x1201,0x1200,
	0x11ff,0x11fe,0x11fc,0x11fb,0x11fa,0x11f9,0x11f7,0x11f6,0x11f5,0x11f4,0x11f3,0x11f2,0x11f1,0x11f0,0x11ef,0x11ee,
	0x11ed,0x11eb,0x11ea,0x11e8,0x11e7,0x11e6,0x11e5,0x11e4,0x11e3,0x11e2,0x11e1,0x11e0,0x11df,0x11de,0x11dd,0x11db,
	0x11d9,0x11d8,0x11d7,0x11d6,0x11d5,0x11d4,0x11d3,0x11d2,0x11d1,0x11d0,0x11cf,0x11ce,0x11cd,0x11ca,0x11c9,0x11c8,
	0x11c7,0x11c6,0x11c5,0x11c4,0x11c3,0x11c2,0x11c1,0x11c0,0x11bf,0x11be,0x11bd,0x11ba,0x11b9,0x11b8,0x11b7,0x11b6,
	0x11b5,0x11b4,0x11b3,0x11b2,0x11b1,0x11b0,0x11af,0x11ae,0x11ad,0x11aa,0x11a9,0x11a8,0x11a7,0x11a6,0x11a5,0x11a4,
	0x11a3,0x11a2,0x11a1,0x11a0,0x119f,0x119e,0x119c,0x119b,0x1199,0x1198,0x1197,0x1196,0x1195,0x1194,0x1193,0x1192,
	0x1191,0x1190,0x118f,0x118d,0x118c,0x118b,0x1189,0x1188,0x1187,0x1186,0x1185,0x1184,0x1183,0x1182,0x1181,0x1180,
	0x117e,0x117d,0x117c,0x117b,0x1179,0x1178,0x1177,0x1176,0x1175,0x1174,0x1173,0x1172,0x1171,0x1170,0x116e,0x116d,
	0x116c,0x116b,0x1169,0x1168,0x1167,0x1166,0x1165,0x1164,0x1163,0x1162,0x1161,0x115f,0x115e,0x115d,0x115c,0x115b,
	0x115a,0x1159,0x1157,0x1156,0x1155,0x1154,0x1153,0x1152,0x1151,0x1150,0x114f,0x114e,0x114c,0x114b,0x114a,0x1149,
	0x1147,0x1146,0x1145,0x1144,0x1143,0x1142,0x1141,0x1140,0x113f,0x113e,0x113d,0x113b,0x113a,0x1139,0x1137,0x1136,
	0x1135,0x1134,0x1133,0x1132,0x1131,0x1130,0x112f,0x112e,0x112d,0x112c,0x112a,0x1129,0x1127,0x1126,0x1125,0x1124,
	0x1123,0x1122,0x1121,0x1120,0x111f,0x111e,0x111d,0x111c,0x111b,0x1119,0x1117,0x1116,0x1115,0x1114,0x1113,0x1112,
	0x1111,0x1110,0x110f,0x110e,0x110d,0x110c,0x110b,0x110a,0x1107,0x1106,0x1105,0x1104,0x1103,0x1102,0x1101,0x1100,
	0x10ff,0x10fe,0x10fd,0x10fc,0x10fb,0x10fa,0x10f9,0x10f6,0x10f5,0x10f4,0x10f3,0x10f2,0x10f1,0x10f0,0x10ef,0x10ee,
	0x10ed,0x10ec,0x10eb,0x10ea,0x10e9,0x10e7,0x10e5,0x10e4,0x10e3,0x10e2,0x10e1,0x10e0,0x10df,0x10de,0x10dd,0x10dc,
	0x10db,0x10da,0x10d9,0x10d7,0x10d6,0x10d4,0x10d3,0x10d2,0x10d1,0x10d0,0x10cf,0x10ce,0x10cd,0x10cc,0x10cb,0x10ca,
	0x10c8,0x10c7,0x10c6,0x10c5,0x10c4,0x10c2,0x10c1,0x10c0,0x10bf,0x10bd,0x10bc,0x10bb,0x10ba,0x10b9,0x10b8,0x10b7,
	0x10b6,0x10b5,0x10b4,0x10b3,0x10b1,0x10b0,0x10af,0x10ae,0x10ad,0x10ab,0x10aa,0x10a9,0x10a8,0x10a7,0x10a6,0x10a5,
	0x10a4,0x10a3,0x10a2,0x10a0,0x109f,0x109e,0x109d,0x109c,0x109b,0x1099,0x1098,0x1097,0x1096,0x1095,0x1094,0x1093,
	0x1092,0x1091,0x108f,0x108e,0x108d,0x108c,0x108b,0x108a,0x1089,0x1087,0x1086,0x1085,0x1084,0x1083,0x1082,0x1081,
	0x1080,0x107e,0x107d,0x107c,0x107b,0x107a,0x1079,0x1078,0x1077,0x1075,0x1074,0x1073,0x1072,0x1071,0x1070,0x106f,
	0x106d,0x106c,0x106b,0x106a,0x1069,0x1068,0x1067,0x1066,0x1065,0x1063,0x1062,0x1061,0x1060,0x105f,0x105e,0x105c,
	0x105b,0x105a,0x1059,0x1058,0x1057,0x1056,0x1055,0x1054,0x1053,0x1051,0x1050,0x104f,0x104e,0x104d,0x104b,0x104a,
	0x1049,0x1048,0x1047,0x1046,0x1045,0x1044,0x1043,0x1042,0x1041,0x103f,0x103e,0x103d,0x103c,0x103a,0x1039,0x1038,
	0x1037,0x1036,0x1035,0x1034,0x1033,0x1032,0x1031,0x1030,0x102f,0x102d,0x102c,0x102b,0x1029,0x1028,0x1027,0x1026,
	0x1025,0x1024,0x1023,0x1022,0x1021,0x1020,0x101f,0x101e,0x101d,0x101b,0x101a,0x1018,0x1017,0x1016,0x1015,0x1014,
	0x1013,0x1012,0x1011,0x1010,0x100f,0x100e,0x100d,0x100c,0x100b,0x1009,0x1007,0x1006,0x1005,0x1004,0x1003,0x1002,
	0x1001,0x1000,0xfff,0xffe,0xffd,0xffc,0xffb,0xffa,0xff9,0xff6,0xff5,0xff4,0xff3,0xff2,0xff1,0xff0,
	0xfef,0xfee,0xfed,0xfec,0xfeb,0xfea,0xfe9,0xfe8,0xfe7,0xfe4,0xfe3,0xfe2,0xfe1,0xfe0,0xfdf,0xfde,
	0xfdd,0xfdc,0xfdb,0xfda,0xfd9,0xfd8,0xfd7,0xfd6,0xfd4,0xfd2,0xfd1,0xfd0,0xfcf,0xfce,0xfcd,0xfcc,
	0xfcb,0xfca,0xfc9,0xfc8,0xfc7,0xfc6,0xfc5,0xfc3,0xfc2,0xfc0,0xfbf,0xfbe,0xfbd,0xfbc,0xfbb,0xfba,
	0xfb9,0xfb8,0xfb7,0xfb6,0xfb5,0xfb4,0xfb2,0xfb1,0xfb0,0xfae,0xfad,0xfac,0xfab,0xfaa,0xfa9,0xfa8,
	0xfa6,0xfa5,0xfa4,0xfa3,0xfa2,0xfa1,0xfa0,0xf9f,0xf9e,0xf9d,0xf9c,0xf9b,0xf99,0xf97,0xf96,0xf95,
	0xf94,0xf93,0xf92,0xf91,0xf90,0xf8f,0xf8e,0xf8d,0xf8c,0xf8b,0xf8a,0xf89,0xf88,0xf86,0xf84,0xf83,
	0xf82,0xf81,0xf80,0xf7f,0xf7e,0xf7d,0xf7c,0xf7b,0xf7a,0xf79,0xf78,0xf77,0xf76,0xf74,0xf73,0xf71,
	0xf70,0xf6f,0xf6e,0xf6d,0xf6c,0xf6b,0xf6a,0xf69,0xf68,0xf67,0xf66,0xf65,0xf64,0xf62,0xf61,0xf60,
	0xf5f,0xf5d,0xf5c,0xf5b,0xf5a,0xf59,0xf58,0xf57,0xf56,0xf55,0xf54,0xf53,0xf52,0xf50,0xf4f,0xf4e,
	0xf4d,0xf4c,0xf4a,0xf49,0xf48,0xf47,0xf46,0xf45,0xf44,0xf43,0xf42,0xf41,0xf40,0xf3e,0xf3d,0xf3c,
	0xf3b,0xf3a,0xf39,0xf37,0xf36,0xf35,0xf34,0xf33,0xf32,0xf31,0xf30,0xf2f,0xf2e,0xf2c,0xf2b,0xf2a,
	0xf29,0xf28,0xf27,0xf26,0xf24,0xf23,0xf22,0xf21,0xf20,0xf1f,0xf1e,0xf1d,0xf1c,0xf1a,0xf19,0xf18,
	0xf17,0xf16,0xf15,0xf14,0xf13,0xf11,0xf10,0xf0f,0xf0e,0xf0d,0xf0c,0xf0b,0xf0a,0xf08,0xf07,0xf06,
	0xf05,0xf04,0xf03,0xf02,0xf01,0xf00,0xefe,0xefd,0xefc,0xefb,0xefa,0xef9,0xef8,0xef6,0xef5,0xef4,
	0xef3,0xef2,0xef1,0xef0,0xeef,0xeee,0xeed,0xeeb,0xeea,0xee9,0xee8,0xee7,0xee6,0xee4,0xee3,0xee2,
	0xee1,0xee0,0xedf,0xede,0xedd,0xedc,0xedb,0xeda,0xed8,0xed7,0xed6,0xed5,0xed4,0xed2,0xed1,0xed0,
	0xecf,0xece,0xecd,0xecc,0xecb,0xeca,0xec9,0xec8,0xec7,0xec5,0xec4,0xec3,0xec2,0xec0,0xebf,0xebe,
	0xebd,0xebc,0xebb,0xeba,0xeb9,0xeb8,0xeb7,0xeb6,0xeb5,0xeb4,0xeb2,0xeb1,0xeb0,0xeae,0xead,0xeac,
	0xeab,0xeaa,0xea9,0xea8,0xea7,0xea6,0xea5,0xea4,0xea3,0xea2,0xea1,0xe9f,0xe9e,0xe9c,0xe9b,0xe9a,
	0xe99,0xe98,0xe97,0xe96,0xe95,0xe94,0xe93,0xe92,0xe91,0xe90,0xe8f,0xe8e,0xe8c,0xe8a,0xe89,0xe88,
	0xe88,0xe86,0xe85,0xe84,0xe83,0xe82,0xe81,0xe80,0xe7f,0xe7e,0xe7c,0xe7b,0xe7a,0xe79,0xe78,0xe77,
	0xe76,0xe75,0xe74,0xe72,0xe71,0xe70,0xe6f,0xe6e,0xe6d,0xe6c,0xe6b,0xe69,0xe68,0xe67,0xe66,0xe65,
	0xe64,0xe63,0xe62,0xe61,0xe60,0xe5e,0xe5d,0xe5c,0xe5b,0xe5a,0xe59,0xe58,0xe56,0xe55,0xe54,0xe53,
	0xe52,0xe51,0xe50,0xe4f,0xe4e,0xe4d,0xe4c,0xe4a,0xe49,0xe48,0xe47,0xe46,0xe45,0xe44,0xe42,0xe41,
	0xe40,0xe3f,0xe3e,0xe3d,0xe3c,0xe3b,0xe3a,0xe39,0xe38,0xe37,0xe35,0xe34,0xe33,0xe32,0xe31,0xe2f,
	0xe2e,0xe2d,0xe2c,0xe2b,0xe2a,0xe29,0xe28,0xe27,0xe26,0xe25,0xe24,0xe23,0xe21,0xe20,0xe1f,0xe1e,
	0xe1c,0xe1b,0xe1a,0xe19,0xe18,0xe17,0xe16,0xe15,0xe14,0xe13,0xe12,0xe11,0xe10,0xe0f,0xe0d,0xe0c,
	0xe0b,0xe09,0xe08,0xe07,0xe06,0xe05,0xe04,0xe03,0xe02,0xe01,0xe00,0xdff,0xdfe,0xdfd,0xdfc,0xdfb,
	0xdfa,0xdf8,0xdf7,0xdf5,0xdf4,0xdf3,0xdf2,0xdf1,0xdf0,0xdef,0xdee,0xded,0xdec,0xdeb,0xdea,0xde9,
	0xde8,0xde7,0xde6,0xde4,0xde2,0xde1,0xde0,0xddf,0xdde,0xddd,0xddc,0xddb,0xdda,0xdd9,0xdd8,0xdd7,
	0xdd6,0xdd5,0xdd4,0xdd3,0xdd2,0xdcf,0xdce,0xdcd,0xdcc,0xdcb,0xdca,0xdc9,0xdc8,0xdc7,0xdc6,0xdc5,
	0xdc4,0xdc3,0xdc2,0xdc1,0xdc0,0xdbf,0xdbe,0xdbb,0xdba,0xdb9,0xdb8,0xdb7,0xdb6,0xdb5,0xdb4,0xdb3,
	0xdb2,0xdb1,0xdb0,0xdaf,0xdae,0xdad,0xdac,0xdab,0xdaa,0xda8,0xda6,0xda5,0xda4,0xda3,0xda2,0xda1,
	0xda0,0xd9f,0xd9e,0xd9d,0xd9c,0xd9b,0xd9a,0xd99,0xd98,0xd97,0xd95,0xd94,0xd92,0xd91,0xd90,0xd8f,
	0xd8e,0xd8d,0xd8c,0xd8b,0xd8a,0xd89,0xd88,0xd87,0xd86,0xd85,0xd84,0xd82,0xd81,0xd80,0xd7e,0xd7d,
	0xd7c,0xd7b,0xd7a,0xd79,0xd78,0xd77,0xd76,0xd75,0xd74,0xd73,0xd72,0xd71,0xd70,0xd6e,0xd6d,0xd6c,
	0xd6b,0xd6a,0xd69,0xd68,0xd67,0xd65,0xd64,0xd63,0xd62,0xd61,0xd60,0xd5f,0xd5e,0xd5c,0xd5b,0xd5a,
	0xd59,0xd58,0xd57,0xd56,0xd55,0xd54,0xd53,0xd52,0xd51,0xd4f,0xd4e,0xd4d,0xd4c,0xd4b,0xd4a,0xd49,
	0xd47,0xd46,0xd45,0xd44,0xd43,0xd42,0xd41,0xd40,0xd3f,0xd3e,0xd3d,0xd3c,0xd3b,0xd39,0xd38,0xd37,
	0xd36,0xd35,0xd33,0xd32,0xd31,0xd30,0xd2f,0xd2e,0xd2d,0xd2c,0xd2b,0xd2a,0xd29,0xd28,0xd27,0xd26,
	0xd25,0xd23,0xd22,0xd21,0xd20,0xd1e,0xd1d,0xd1c,0xd1b,0xd1a,0xd19,0xd18,0xd17,0xd16,0xd15,0xd14,
	0xd13,0xd12,0xd11,0xd10,0xd0e,0xd0d,0xd0c,0xd0b,0xd09,0xd08,0xd07,0xd06,0xd05,0xd04,0xd03,0xd02,
	0xd01,0xd00,0xcff,0xcfe,0xcfd,0xcfc,0xcfb,0xcfa,0xcf8,0xcf7,0xcf5,0xcf4,0xcf3,0xcf2,0xcf1,0xcf0,
	0xcef,0xcee,0xced,0xcec,0xceb,0xcea,0xce9,0xce8,0xce7,0xce6,0xce5,0xce4,0xce2,0xce0,0xcdf,0xcde,
	0xcdd,0xcdc,0xcdb,0xcda,0xcd9,0xcd8,0xcd7,0xcd6,0xcd5,0xcd4,0xcd3,0xcd2,0xcd1,0xcd0,0xccf,0xccd,
	0xccb,0xcca,0xcc9,0xcc8,0xcc7,0xcc6,0xcc5,0xcc4,0xcc3,0xcc2,0xcc1,0xcc0,0xcbf,0xcbe,0xcbd,0xcbc,
	0xcbb,0xcba,0xcb9,0xcb6,0xcb5,0xcb4,0xcb3,0xcb2,0xcb1,0xcb0,0xcaf,0xcae,0xcad,0xcac,0xcab,0xcaa,
	0xca9,0xca8,0xca7,0xca6,0xca5,0xca4,0xca2,0xca0,0xc9f,0xc9e,0xc9d,0xc9c,0xc9b,0xc9a,0xc99,0xc98,
	0xc97,0xc96,0xc95,0xc94,0xc93,0xc92,0xc91,0xc90,0xc8f,0xc8d,0xc8b,0xc8a,0xc89,0xc88,0xc87,0xc86,
	0xc85,0xc84,0xc83,0xc82,0xc81,0xc80,0xc7f,0xc7e,0xc7d,0xc7c,0xc7b,0xc7a,0xc78,0xc77,0xc75,0xc74,
	0xc73,0xc72,0xc71,0xc70,0xc6f,0xc6e,0xc6d,0xc6c,0xc6b,0xc6a,0xc69,0xc68,0xc67,0xc66,0xc64,0xc63,
	0xc62,0xc61,0xc5f,0xc5e,0xc5d,0xc5c,0xc5b,0xc5a,0xc59,0xc58,0xc57,0xc56,0xc55,0xc54,0xc53,0xc52,
	0xc51,0xc50,0xc4f,0xc4d,0xc4c,0xc4b,0xc4a,0xc49,0xc48,0xc47,0xc46,0xc44,0xc43,0xc42,0xc41,0xc40,
	0xc3f,0xc3e,0xc3d,0xc3c,0xc3b,0xc3a,0xc39,0xc37,0xc36,0xc35,0xc34,0xc33,0xc32,0xc31,0xc30,0xc2f,
	0xc2e,0xc2c,0xc2b,0xc2a,0xc29,0xc28,0xc27,0xc26,0xc25,0xc24,0xc23,0xc22,0xc20,0xc1f,0xc1e,0xc1d,
	0xc1c,0xc1b,0xc1a,0xc19,0xc18,0xc17,0xc15,0xc14,0xc13,0xc12,0xc11,0xc10,0xc0f,0xc0e,0xc0d,0xc0c,
	0xc0a,0xc09,0xc08,0xc07,0xc06,0xc05,0xc04,0xc03,0xc02,0xc01,0xc00,0xbfe,0xbfd,0xbfc,0xbfb,0xbfa,
	0xbf9,0xbf8,0xbf7,0xbf6,0xbf4,0xbf3,0xbf2,0xbf1,0xbf0,0xbef,0xbee,0xbed,0xbec,0xbeb,0xbea,0xbe9,
	0xbe8,0xbe6,0xbe5,0xbe4,0xbe3,0xbe2,0xbe1,0xbe0,0xbdf,0xbdd,0xbdc,0xbdb,0xbda,0xbd9,0xbd8,0xbd7,
	0xbd6,0xbd5,0xbd4,0xbd3,0xbd2,0xbd1,0xbcf,0xbce,0xbcd,0xbcc,0xbcb,0xbca,0xbc9,0xbc7,0xbc6,0xbc5,
	0xbc4,0xbc3,0xbc2,0xbc1,0xbc0,0xbbf,0xbbd,0xbbc,0xbbb,0xbba,0xbb9,0xbb8,0xbb7,0xbb5,0xbb4,0xbb3,
	0xbb2,0xbb1,0xbb0,0xbaf,0xbae,0xbad,0xbac,0xbab,0xbaa,0xba9,0xba8,0xba7,0xba6,0xba5,0xba3,0xba2,
	0xba1,0xba0,0xb9f,0xb9d,0xb9c,0xb9b,0xb9a,0xb99,0xb98,0xb97,0xb96,0xb95,0xb94,0xb93,0xb92,0xb91,
	0xb90,0xb8f,0xb8e,0xb8d,0xb8c,0xb8a,0xb89,0xb88,0xb87,0xb86,0xb84,0xb83,0xb82,0xb81,0xb80,0xb7f,
	0xb7e,0xb7d,0xb7c,0xb7b,0xb7a,0xb79,0xb78,0xb77,0xb76,0xb75,0xb74,0xb73,0xb71,0xb70,0xb6f,0xb6e,
	0xb6c,0xb6b,0xb6a,0xb69,0xb68,0xb67,0xb66,0xb65,0xb64,0xb63,0xb62,0xb61,0xb60,0xb5f,0xb5e,0xb5d,
	0xb5c,0xb5b,0xb5a,0xb58,0xb57,0xb56,0xb54,0xb53,0xb52,0xb51,0xb50,0xb4f,0xb4e,0xb4d,0xb4c,0xb4b,
	0xb4a,0xb49,0xb48,0xb47,0xb46,0xb45,0xb44,0xb43,0xb42,0xb41,0xb40,0xb3e,0xb3c,0xb3b,0xb3a,0xb39,
	0xb38,0xb37,0xb36,0xb35,0xb34,0xb33,0xb32,0xb31,0xb30,0xb2f,0xb2e,0xb2d,0xb2c,0xb2b,0xb2a,0xb29,
	0xb28,0xb27,0xb25,0xb23,0xb22,0xb21,0xb20,0xb1f,0xb1e,0xb1d,0xb1c,0xb1b,0xb1a,0xb19,0xb18,0xb17,
	0xb16,0xb15,0xb14,0xb13,0xb12,0xb11,0xb10,0xb0f,0xb0e,0xb0b,0xb0a,0xb09,0xb08,0xb07,0xb06,0xb05,
	0xb04,0xb03,0xb02,0xb01,0xb00,0xaff,0xafe,0xafd,0xafc,0xafb,0xafa,0xaf9,0xaf8,0xaf7,0xaf6,0xaf5,
	0xaf3,0xaf1,0xaf0,0xaef,0xaee,0xaed,0xaec,0xaeb,0xaea,0xae9,0xae8,0xae7,0xae6,0xae5,0xae4,0xae3,
	0xae2,0xae1,0xae0,0xadf,0xade,0xadd,0xadb,0xada,0xad8,0xad7,0xad6,0xad5,0xad4,0xad3,0xad2,0xad1,
	0xad0,0xacf,0xace,0xacd,0xacc,0xacb,0xaca,0xac9,0xac8,0xac7,0xac6,0xac5,0xac4,0xac2,0xac1,0xabf,
	0xabe,0xabd,0xabc,0xabb,0xaba,0xab9,0xab8,0xab7,0xab6,0xab5,0xab4,0xab3,0xab2,0xab1,0xab0,0xaaf,
	0xaae,0xaab,0xaaa,0xaa9,0xaa8,0xaa7,0xaa6,0xaa5,0xaa4,0xaa3,0xaa2,0xaa1,0xaa0,0xa9f,0xa9e,0xa9d,
	0xa9c,0xa9b,0xa9a,0xa99,0xa98,0xa97,0xa96,0xa95,0xa94,0xa93,0xa90,0xa8f,0xa8e,0xa8d,0xa8c,0xa8b,
	0xa8a,0xa89,0xa88,0xa87,0xa86,0xa85,0xa84,0xa83,0xa82,0xa81,0xa80,0xa7f,0xa7e,0xa7d,0xa7c,0xa7b,
	0xa7a,0xa79,0xa77,0xa76,0xa74,0xa73,0xa72,0xa71,0xa70,0xa6f,0xa6e,0xa6d,0xa6c,0xa6b,0xa6a,0xa69,
	0xa68,0xa67,0xa66,0xa65,0xa64,0xa63,0xa62,0xa61,0xa60,0xa5f,0xa5e,0xa5c,0xa5b,0xa59,0xa58,0xa57,
	0xa56,0xa55,0xa54,0xa53,0xa52,0xa51,0xa50,0xa4f,0xa4e,0xa4d,0xa4c,0xa4b,0xa4a,0xa49,0xa48,0xa47,
	0xa46,0xa45,0xa44,0xa43,0xa41,0xa40,0xa3f,0xa3d,0xa3c,0xa3b,0xa3a,0xa39,0xa38,0xa37,0xa36,0xa35,
	0xa34,0xa33,0xa32,0xa31,0xa30,0xa2f,0xa2e,0xa2d,0xa2c,0xa2b,0xa2a,0xa29,0xa27,0xa26,0xa25,0xa24,
	0xa23,0xa21,0xa20,0xa1f,0xa1e,0xa1d,0xa1c,0xa1b,0xa1a,0xa19,0xa18,0xa17,0xa16,0xa15,0xa14,0xa13,
	0xa12,0xa11,0xa10,0xa0f,0xa0e,0xa0c,0xa0b,0xa0a,0xa09,0xa08,0xa06,0xa05,0xa04,0xa03,0xa02,0xa01,
	0xa00,0x9ff,0x9fe,0x9fd,0x9fc,0x9fb,0x9fa,0x9f9,0x9f8,0x9f7,0x9f6,0x9f5,0x9f4,0x9f2,0x9f1,0x9f0,
	0x9ef,0x9ee,0x9ed,0x9ec,0x9ea,0x9e9,0x9e8,0x9e7,0x9e6,0x9e5,0x9e4,0x9e3,0x9e2,0x9e1,0x9e0,0x9df,
	0x9de,0x9dd,0x9dc,0x9db,0x9da,0x9d9,0x9d7,0x9d6,0x9d5,0x9d4,0x9d3,0x9d2,0x9d1,0x9cf,0x9ce,0x9cd,
	0x9cc,0x9cb,0x9ca,0x9c9,0x9c8,0x9c7,0x9c6,0x9c5,0x9c4,0x9c3,0x9c2,0x9c1,0x9c0,0x9bf,0x9be,0x9bc,
	0x9bb,0x9ba,0x9b9,0x9b8,0x9b7,0x9b6,0x9b5,0x9b3,0x9b2,0x9b1,0x9b0,0x9af,0x9ae,0x9ad,0x9ac,0x9ab,
	0x9aa,0x9a9,0x9a8,0x9a7,0x9a6,0x9a5,0x9a4,0x9a2,0x9a1,0x9a0,0x99f,0x99e,0x99d,0x99c,0x99b,0x99a,
	0x998,0x997,0x996,0x995,0x994,0x993,0x992,0x991,0x990,0x98f,0x98e,0x98d,0x98c,0x98b,0x98a,0x989,
	0x988,0x987,0x986,0x985,0x983,0x982,0x980,0x97f,0x97e,0x97d,0x97c,0x97b,0x97a,0x979,0x978,0x977,
	0x976,0x975,0x974,0x973,0x972,0x971,0x970,0x96f,0x96e,0x96d,0x96c,0x96b,0x96a,0x969,0x968,0x966,
	0x965,0x964,0x962,0x961,0x960,0x95f,0x95e,0x95d,0x95c,0x95b,0x95a,0x959,0x958,0x957,0x956,0x955,
	0x954,0x953,0x952,0x951,0x950,0x94f,0x94e,0x94d,0x94c,0x94b,0x949,0x948,0x947,0x946,0x945,0x943,
	0x942,0x941,0x940,0x93f,0x93e,0x93d,0x93c,0x93b,0x93a,0x939,0x938,0x937,0x936,0x935,0x934,0x933,
	0x932,0x931,0x930,0x92f,0x92e,0x92d,0x92b,0x92a,0x929,0x928,0x927,0x925,0x924,0x923,0x922,0x921,
	0x920,0x91f,0x91e,0x91d,0x91c,0x91b,0x91a,0x919,0x918,0x917,0x916,0x915,0x914,0x913,0x912,0x911,
	0x910,0x90e,0x90d,0x90c,0x90b,0x90a,0x909,0x908,0x906,0x905,0x904,0x903,0x902,0x901,0x900,0x8ff,
	0x8fe,0x8fd,0x8fc,0x8fb,0x8fa,0x8f9,0x8f8,0x8f7,0x8f6,0x8f5,0x8f4,0x8f3,0x8f2,0x8f0,0x8ef,0x8ee,
	0x8ed,0x8ec,0x8eb,0x8ea,0x8e8,0x8e7,0x8e6,0x8e5,0x8e4,0x8e3,0x8e2,0x8e1,0x8e0,0x8df,0x8de,0x8dd,
	0x8dc,0x8db,0x8da,0x8d9,0x8d8,0x8d7,0x8d6,0x8d5,0x8d3,0x8d2,0x8d1,0x8d0,0x8cf,0x8ce,0x8cd,0x8cc,
	0x8cb,0x8c9,0x8c8,0x8c7,0x8c6,0x8c5,0x8c4,0x8c3,0x8c2,0x8c1,0x8c0,0x8bf,0x8be,0x8bd,0x8bc,0x8bb,
	0x8ba,0x8b9,0x8b8,0x8b7,0x8b5,0x8b4,0x8b3,0x8b2,0x8b1,0x8b0,0x8af,0x8ae,0x8ad,0x8ab,0x8aa,0x8a9,
	0x8a8,0x8a7,0x8a6,0x8a5,0x8a4,0x8a3,0x8a2,0x8a1,0x8a0,0x89f,0x89e,0x89d,0x89c,0x89b,0x89a,0x898,
	0x897,0x896,0x895,0x894,0x893,0x892,0x891,0x890,0x88f,0x88d,0x88c,0x88b,0x88a,0x889,0x888,0x887,
	0x886,0x885,0x884,0x883,0x882,0x881,0x880,0x87f,0x87e,0x87d,0x87c,0x87b,0x87a,0x879,0x878,0x877,
	0x876,0x874,0x872,0x871,0x870,0x86f,0x86e,0x86d,0x86c,0x86b,0x86a,0x869,0x868,0x867,0x866,0x865,
	0x864,0x863,0x862,0x861,0x860,0x85f,0x85e,0x85d,0x85c,0x85b,0x85a,0x859,0x858,0x857,0x856,0x855,
	0x854,0x851,0x850,0x84f,0x84e,0x84d,0x84c,0x84b,0x84a,0x849,0x848,0x847,0x846,0x845,0x844,0x843,
	0x842,0x841,0x840,0x83f,0x83e,0x83d,0x83c,0x83b,0x83a,0x839,0x838,0x837,0x836,0x835,0x834,0x833,
	0x832,0x82f,0x82e,0x82d,0x82c,0x82b,0x82a,0x829,0x828,0x827,0x826,0x825,0x824,0x823,0x822,0x821,
	0x820,0x81f,0x81e,0x81d,0x81c,0x81b,0x81a,0x819,0x818,0x817,0x816,0x815,0x814,0x813,0x812,0x811,
	0x80f,0x80d,0x80c,0x80b,0x80a,0x809,0x808,0x807,0x806,0x805,0x804,0x803,0x802,0x801,0x800,0x7ff,
	0x7fe,0x7fd,0x7fc,0x7fb,0x7fa,0x7f9,0x7f8,0x7f7,0x7f6,0x7f5,0x7f4,0x7f3,0x7f2,0x7f1,0x7f0,0x7ee,
	0x7ed,0x7eb,0x7ea,0x7e9,0x7e8,0x7e7,0x7e6,0x7e5,0x7e4,0x7e3,0x7e2,0x7e1,0x7e0,0x7df,0x7de,0x7dd,
	0x7dc,0x7db,0x7da,0x7d9,0x7d8,0x7d7,0x7d6,0x7d5,0x7d4,0x7d3,0x7d2,0x7d1,0x7d0,0x7cf,0x7cd,0x7cc,
	0x7cb,0x7c9,0x7c8,0x7c7,0x7c6,0x7c5,0x7c4,0x7c3,0x7c2,0x7c1,0x7c0,0x7bf,0x7be,0x7bd,0x7bc,0x7bb,
	0x7ba,0x7b9,0x7b8,0x7b7,0x7b6,0x7b5,0x7b4,0x7b3,0x7b2,0x7b1,0x7b0,0x7af,0x7ae,0x7ac,0x7ab,0x7aa,
	0x7a9,0x7a7,0x7a6,0x7a5,0x7a4,0x7a3,0x7a2,0x7a1,0x7a0,0x79f,0x79e,0x79d,0x79c,0x79b,0x79a,0x799,
	0x798,0x797,0x796,0x795,0x794,0x793,0x792,0x791,0x790,0x78f,0x78e,0x78d,0x78b,0x78a,0x789,0x788,
	0x787,0x785,0x784,0x783,0x782,0x781,0x780,0x77f,0x77e,0x77d,0x77c,0x77b,0x77a,0x779,0x778,0x777,
	0x776,0x775,0x774,0x773,0x772,0x771,0x770,0x76f,0x76e,0x76d,0x76b,0x76a,0x769,0x768,0x767,0x766,
	0x765,0x764,0x763,0x762,0x761,0x760,0x75f,0x75d,0x75c,0x75b,0x75a,0x759,0x758,0x757,0x756,0x755,
	0x754,0x753,0x752,0x751,0x750,0x74f,0x74e,0x74d,0x74c,0x74b,0x74a,0x749,0x748,0x747,0x745,0x744,
	0x743,0x742,0x741,0x740,0x73f,0x73e,0x73d,0x73c,0x73b,0x73a,0x739,0x737,0x736,0x735,0x734,0x733,
	0x732,0x731,0x730,0x72f,0x72e,0x72d,0x72c,0x72b,0x72a,0x729,0x728,0x727,0x726,0x725,0x724,0x723,
	0x722,0x721,0x720,0x71e,0x71d,0x71c,0x71b,0x71a,0x719,0x718,0x717,0x716,0x715,0x714,0x712,0x711,
	0x710,0x70f,0x70e,0x70d,0x70c,0x70b,0x70a,0x709,0x708,0x707,0x706,0x705,0x704,0x703,0x702,0x701,
	0x700,0x6ff,0x6fe,0x6fd,0x6fc,0x6fb,0x6fa,0x6f8,0x6f7,0x6f6,0x6f5,0x6f4,0x6f3,0x6f2,0x6f1,0x6f0,
	0x6ef,0x6ee,0x6ec,0x6eb,0x6ea,0x6e9,0x6e8,0x6e7,0x6e6,0x6e5,0x6e4,0x6e3,0x6e2,0x6e1,0x6e0,0x6df,
	0x6de,0x6dd,0x6dc,0x6db,0x6da,0x6d9,0x6d8,0x6d7,0x6d6,0x6d5,0x6d4,0x6d3,0x6d1,0x6d0,0x6cf,0x6ce,
	0x6cd,0x6cc,0x6cb,0x6ca,0x6c9,0x6c7,0x6c6,0x6c5,0x6c4,0x6c3,0x6c2,0x6c1,0x6c0,0x6bf,0x6be,0x6bd,
	0x6bc,0x6bb,0x6ba,0x6b9,0x6b8,0x6b7,0x6b6,0x6b5,0x6b4,0x6b3,0x6b2,0x6b1,0x6b0,0x6af,0x6ae,0x6ad,
	0x6ac,0x6aa,0x6a9,0x6a8,0x6a7,0x6a6,0x6a5,0x6a4,0x6a3,0x6a1,0x6a0,0x69f,0x69e,0x69d,0x69c,0x69b,
	0x69a,0x699,0x698,0x697,0x696,0x695,0x694,0x693,0x692,0x691,0x690,0x68f,0x68e,0x68d,0x68c,0x68b,
	0x68a,0x689,0x688,0x687,0x686,0x684,0x683,0x682,0x681,0x680,0x67f,0x67e,0x67d,0x67b,0x67a,0x679,
	0x678,0x677,0x676,0x675,0x674,0x673,0x672,0x671,0x670,0x66f,0x66e,0x66d,0x66c,0x66b,0x66a,0x669,
	0x668,0x666,0x665,0x664,0x663,0x662,0x661,0x660,0x65f,0x65e,0x65d,0x65c,0x65b,0x65a,0x659,0x658,
	0x657,0x656,0x655,0x654,0x653,0x652,0x651,0x650,0x64f,0x64e,0x64d,0x64c,0x64b,0x64a,0x649,0x648,
	0x647,0x646,0x645,0x644,0x643,0x642,0x640,0x63f,0x63e,0x63d,0x63c,0x63a,0x639,0x638,0x637,0x636,
	0x635,0x634,0x633,0x632,0x631,0x630,0x62f,0x62e,0x62d,0x62c,0x62b,0x62a,0x629,0x628,0x627,0x626,
	0x625,0x624,0x623,0x622,0x621,0x620,0x61f,0x61e,0x61d,0x61c,0x61b,0x61a,0x619,0x618,0x617,0x615,
	0x614,0x613,0x612,0x611,0x610,0x60f,0x60d,0x60c,0x60b,0x60a,0x609,0x608,0x607,0x606,0x605,0x604,
	0x603,0x602,0x601,0x600,0x5ff,0x5fe,0x5fd,0x5fc,0x5fb,0x5fa,0x5f9,0x5f8,0x5f7,0x5f6,0x5f5,0x5f4,
	0x5f3,0x5f2,0x5f1,0x5f0,0x5ef,0x5ee,0x5ed,0x5ec,0x5eb,0x5e9,0x5e8,0x5e7,0x5e6,0x5e5,0x5e4,0x5e3,
	0x5e2,0x5e0,0x5df,0x5de,0x5dd,0x5dc,0x5db,0x5da,0x5d9,0x5d8,0x5d7,0x5d6,0x5d5,0x5d4,0x5d3,0x5d2,
	0x5d1,0x5d0,0x5cf,0x5ce,0x5cd,0x5cc,0x5cb,0x5ca,0x5c9,0x5c8,0x5c7,0x5c6,0x5c5,0x5c4,0x5c3,0x5c2,
	0x5c1,0x5c0,0x5bf,0x5bd,0x5bc,0x5bb,0x5ba,0x5b9,0x5b8,0x5b7,0x5b6,0x5b4,0x5b3,0x5b2,0x5b1,0x5b0,
	0x5af,0x5ae,0x5ad,0x5ac,0x5ab,0x5aa,0x5a9,0x5a8,0x5a7,0x5a6,0x5a5,0x5a4,0x5a3,0x5a2,0x5a1,0x5a0,
	0x59f,0x59e,0x59d,0x59c,0x59b,0x59a,0x599,0x598,0x597,0x596,0x595,0x594,0x592,0x591,0x590,0x58f,
	0x58e,0x58d,0x58c,0x58b,0x58a,0x589,0x587,0x586,0x585,0x584,0x583,0x582,0x581,0x580,0x57f,0x57e,
	0x57d,0x57c,0x57b,0x57a,0x579,0x578,0x577,0x576,0x575,0x574,0x573,0x572,0x571,0x570,0x56f,0x56e,
	0x56d,0x56c,0x56b,0x56a,0x569,0x568,0x566,0x565,0x564,0x563,0x562,0x561,0x560,0x55f,0x55e,0x55d,
	0x55c,0x55b,0x55a,0x559,0x558,0x557,0x556,0x555,0x554,0x553,0x552,0x551,0x550,0x54f,0x54d,0x54c,
	0x54b,0x54a,0x549,0x548,0x547,0x546,0x545,0x544,0x543,0x542,0x541,0x540,0x53f,0x53e,0x53d,0x53c,
	0x53b,0x53a,0x539,0x538,0x537,0x536,0x535,0x533,0x532,0x531,0x530,0x52f,0x52e,0x52d,0x52c,0x52b,
	0x52a,0x529,0x528,0x527,0x526,0x525,0x524,0x523,0x522,0x521,0x520,0x51f,0x51e,0x51d,0x51c,0x51b,
	0x519,0x518,0x517,0x516,0x515,0x514,0x513,0x512,0x511,0x510,0x50f,0x50e,0x50d,0x50c,0x50b,0x50a,
	0x509,0x508,0x507,0x506,0x505,0x504,0x503,0x502,0x501,0x500,0x4fe,0x4fd,0x4fc,0x4fb,0x4fa,0x4f9,
	0x4f8,0x4f7,0x4f6,0x4f5,0x4f4,0x4f3,0x4f2,0x4f1,0x4f0,0x4ef,0x4ee,0x4ed,0x4ec,0x4eb,0x4ea,0x4e9,
	0x4e8,0x4e7,0x4e5,0x4e4,0x4e3,0x4e2,0x4e1,0x4e0,0x4df,0x4de,0x4dd,0x4dc,0x4db,0x4da,0x4d9,0x4d8,
	0x4d7,0x4d6,0x4d5,0x4d4,0x4d3,0x4d2,0x4d1,0x4d0,0x4cf,0x4ce,0x4cd,0x4cc,0x4cb,0x4c9,0x4c8,0x4c7,
	0x4c6,0x4c5,0x4c4,0x4c3,0x4c2,0x4c1,0x4c0,0x4bf,0x4be,0x4bd,0x4bc,0x4bb,0x4ba,0x4b9,0x4b8,0x4b7,
	0x4b6,0x4b5,0x4b4,0x4b3,0x4b1,0x4b0,0x4af,0x4ae,0x4ad,0x4ac,0x4ab,0x4aa,0x4a9,0x4a8,0x4a7,0x4a6,
	0x4a5,0x4a4,0x4a3,0x4a2,0x4a1,0x4a0,0x49f,0x49e,0x49d,0x49c,0x49b,0x49a,0x499,0x498,0x497,0x496,
	0x495,0x493,0x492,0x491,0x490,0x48f,0x48e,0x48d,0x48c,0x48b,0x48a,0x489,0x488,0x487,0x486,0x485,
	0x484,0x483,0x482,0x481,0x480,0x47f,0x47e,0x47c,0x47b,0x47a,0x479,0x478,0x477,0x476,0x475,0x474,
	0x473,0x472,0x471,0x470,0x46f,0x46e,0x46d,0x46c,0x46b,0x46a,0x469,0x468,0x467,0x466,0x465,0x464,
	0x463,0x462,0x461,0x460,0x45e,0x45d,0x45c,0x45b,0x45a,0x459,0x458,0x457,0x456,0x455,0x454,0x453,
	0x452,0x451,0x450,0x44f,0x44e,0x44d,0x44c,0x44b,0x44a,0x449,0x448,0x447,0x446,0x445,0x444,0x443,
	0x442,0x441,0x440,0x43f,0x43e,0x43d,0x43c,0x43b,0x43a,0x439,0x438,0x437,0x436,0x435,0x434,0x433,
	0x432,0x430,0x42f,0x42e,0x42d,0x42c,0x42b,0x42a,0x429,0x428,0x427,0x426,0x425,0x424,0x423,0x422,
	0x421,0x41f,0x41e,0x41d,0x41c,0x41b,0x41a,0x419,0x418,0x417,0x416,0x415,0x414,0x413,0x412,0x411,
	0x410,0x40f,0x40e,0x40d,0x40c,0x40b,0x40a,0x409,0x408,0x407,0x406,0x405,0x404,0x403,0x402,0x401,
	0x400,0x3ff,0x3fe,0x3fd,0x3fc,0x3fb,0x3fa,0x3f9,0x3f8,0x3f7,0x3f6,0x3f5,0x3f4,0x3f3,0x3f2,0x3f1,
	0x3f0,0x3ee,0x3ed,0x3ec,0x3eb,0x3ea,0x3e9,0x3e8,0x3e7,0x3e6,0x3e5,0x3e4,0x3e3,0x3e2,0x3e1,0x3e0,
	0x3de,0x3dd,0x3dc,0x3db,0x3da,0x3d9,0x3d8,0x3d7,0x3d6,0x3d5,0x3d4,0x3d3,0x3d2,0x3d1,0x3d0,0x3cf,
	0x3ce,0x3cd,0x3cc,0x3cb,0x3ca,0x3c9,0x3c8,0x3c7,0x3c6,0x3c5,0x3c4,0x3c3,0x3c2,0x3c1,0x3c0,0x3bf,
	0x3be,0x3bd,0x3bc,0x3bb,0x3ba,0x3b9,0x3b8,0x3b7,0x3b6,0x3b5,0x3b4,0x3b3,0x3b2,0x3b1,0x3b0,0x3af,
	0x3ae,0x3ac,0x3ab,0x3aa,0x3a9,0x3a8,0x3a7,0x3a6,0x3a5,0x3a4,0x3a3,0x3a2,0x3a1,0x3a0,0x39f,0x39d,
	0x39c,0x39b,0x39a,0x399,0x398,0x397,0x396,0x395,0x394,0x393,0x392,0x391,0x390,0x38f,0x38e,0x38d,
	0x38c,0x38b,0x38a,0x389,0x388,0x387,0x386,0x385,0x384,0x383,0x382,0x381,0x380,0x37f,0x37e,0x37d,
	0x37c,0x37b,0x37a,0x379,0x378,0x377,0x376,0x375,0x374,0x373,0x372,0x371,0x370,0x36f,0x36e,0x36d,
	0x36c,0x36a,0x369,0x368,0x367,0x366,0x365,0x364,0x363,0x362,0x361,0x360,0x35f,0x35e,0x35c,0x35b,
	0x35a,0x359,0x358,0x357,0x356,0x355,0x354,0x353,0x352,0x351,0x350,0x34f,0x34e,0x34d,0x34c,0x34b,
	0x34b,0x34a,0x349,0x348,0x347,0x346,0x345,0x344,0x343,0x342,0x341,0x340,0x33f,0x33e,0x33d,0x33c,
	0x33b,0x33a,0x339,0x338,0x337,0x336,0x334,0x333,0x332,0x331,0x330,0x32f,0x32e,0x32d,0x32c,0x32a,
	0x329,0x328,0x327,0x326,0x325,0x324,0x323,0x322,0x321,0x320,0x31f,0x31e,0x31d,0x31c,0x31b,0x31a,
	0x319,0x318,0x317,0x316,0x315,0x314,0x313,0x312,0x311,0x310,0x30f,0x30e,0x30d,0x30c,0x30b,0x30a,
	0x309,0x308,0x307,0x306,0x305,0x304,0x303,0x302,0x301,0x300,0x2ff,0x2fe,0x2fd,0x2fc,0x2fb,0x2fa,
	0x2f9,0x2f8,0x2f7,0x2f6,0x2f5,0x2f4,0x2f3,0x2f2,0x2f1,0x2f0,0x2ef,0x2ee,0x2ed,0x2ec,0x2eb,0x2ea,
	0x2e9,0x2e8,0x2e7,0x2e6,0x2e5,0x2e4,0x2e3,0x2e2,0x2e1,0x2e0,0x2df,0x2dd,0x2dc,0x2db,0x2da,0x2d9,
	0x2d8,0x2d7,0x2d6,0x2d4,0x2d3,0x2d2,0x2d1,0x2d0,0x2cf,0x2ce,0x2cd,0x2cc,0x2cb,0x2ca,0x2c9,0x2c8,
	0x2c7,0x2c6,0x2c5,0x2c4,0x2c3,0x2c2,0x2c1,0x2c0,0x2bf,0x2be,0x2bd,0x2bc,0x2bb,0x2ba,0x2b9,0x2b8,
	0x2b7,0x2b6,0x2b5,0x2b4,0x2b3,0x2b2,0x2b1,0x2b0,0x2af,0x2ae,0x2ad,0x2ac,0x2ab,0x2aa,0x2a9,0x2a8,
	0x2a7,0x2a6,0x2a5,0x2a4,0x2a3,0x2a2,0x2a1,0x2a0,0x29f,0x29e,0x29d,0x29c,0x29b,0x29a,0x299,0x298,
	0x297,0x296,0x295,0x294,0x293,0x292,0x291,0x290,0x28f,0x28e,0x28d,0x28c,0x28b,0x28a,0x289,0x288,
	0x287,0x285,0x284,0x283,0x282,0x281,0x280,0x27f,0x27d,0x27c,0x27b,0x27a,0x279,0x278,0x277,0x276,
	0x275,0x274,0x273,0x272,0x271,0x270,0x26f,0x26e,0x26d,0x26c,0x26b,0x26a,0x269,0x268,0x267,0x266,
	0x265,0x264,0x263,0x262,0x261,0x260,0x25f,0x25e,0x25d,0x25c,0x25b,0x25a,0x259,0x258,0x257,0x256,
	0x255,0x254,0x253,0x252,0x251,0x250,0x24f,0x24e,0x24d,0x24c,0x24b,0x24a,0x249,0x248,0x247,0x246,
	0x246,0x244,0x243,0x242,0x241,0x23f,0x23e,0x23d,0x23c,0x23b,0x23a,0x239,0x238,0x237,0x236,0x235,
	0x234,0x233,0x232,0x231,0x230,0x22f,0x22e,0x22d,0x22c,0x22b,0x22a,0x229,0x228,0x227,0x226,0x225,
	0x224,0x223,0x222,0x221,0x220,0x21f,0x21e,0x21d,0x21c,0x21b,0x21a,0x219,0x218,0x217,0x216,0x215,
	0x214,0x213,0x212,0x211,0x210,0x20f,0x20e,0x20d,0x20c,0x20b,0x20a,0x209,0x208,0x207,0x206,0x205,
	0x204,0x203,0x202,0x201,0x200,0x1ff,0x1fe,0x1fd,0x1fc,0x1fb,0x1fa,0x1f9,0x1f8,0x1f7,0x1f6,0x1f5,
	0x1f4,0x1f3,0x1f2,0x1f1,0x1f0,0x1ef,0x1ee,0x1ed,0x1ec,0x1eb,0x1ea,0x1e9,0x1e8,0x1e7,0x1e6,0x1e5,
	0x1e4,0x1e3,0x1e2,0x1e1,0x1e0,0x1df,0x1de,0x1dd,0x1dc,0x1db,0x1da,0x1d9,0x1d8,0x1d7,0x1d6,0x1d5,
	0x1d4,0x1d3,0x1d2,0x1d1,0x1d0,0x1cf,0x1ce,0x1cd,0x1cc,0x1cb,0x1ca,0x1c9,0x1c8,0x1c7,0x1c6,0x1c5,
	0x1c4,0x1c2,0x1c1,0x1c0,0x1be,0x1bd,0x1bc,0x1bb,0x1ba,0x1b9,0x1b8,0x1b7,0x1b6,0x1b5,0x1b4,0x1b3,
	0x1b2,0x1b1,0x1b0,0x1af,0x1ae,0x1ad,0x1ac,0x1ab,0x1aa,0x1a9,0x1a8,0x1a7,0x1a6,0x1a5,0x1a4,0x1a3,
	0x1a2,0x1a1,0x1a0,0x19f,0x19e,0x19d,0x19c,0x19b,0x19a,0x199,0x198,0x197,0x196,0x195,0x194,0x193,
	0x192,0x191,0x190,0x18f,0x18e,0x18d,0x18c,0x18b,0x18a,0x189,0x188,0x187,0x186,0x185,0x184,0x183,
	0x182,0x181,0x180,0x17f,0x17e,0x17d,0x17c,0x17b,0x17a,0x179,0x178,0x177,0x176,0x175,0x174,0x173,
	0x172,0x171,0x170,0x16f,0x16e,0x16d,0x16c,0x16b,0x16a,0x169,0x168,0x167,0x166,0x165,0x164,0x163,
	0x162,0x161,0x160,0x15f,0x15e,0x15d,0x15c,0x15b,0x15a,0x159,0x158,0x157,0x156,0x155,0x154,0x153,
	0x152,0x151,0x150,0x14f,0x14e,0x14d,0x14c,0x14b,0x14a,0x149,0x148,0x147,0x146,0x145,0x144,0x143,
	0x141,0x140,0x13f,0x13e,0x13d,0x13c,0x13b,0x13a,0x139,0x138,0x137,0x136,0x135,0x134,0x133,0x132,
	0x131,0x130,0x12f,0x12e,0x12d,0x12c,0x12b,0x12a,0x129,0x128,0x127,0x126,0x125,0x124,0x123,0x122,
	0x121,0x120,0x11f,0x11e,0x11d,0x11c,0x11b,0x11a,0x119,0x118,0x117,0x116,0x115,0x114,0x113,0x112,
	0x111,0x110,0x10f,0x10e,0x10d,0x10c,0x10b,0x10a,0x109,0x108,0x107,0x106,0x105,0x104,0x103,0x102,
	0x101,0x100,0xff,0xfe,0xfd,0xfc,0xfb,0xfa,0xf9,0xf8,0xf7,0xf6,0xf5,0xf4,0xf3,0xf2,
	0xf1,0xf0,0xef,0xee,0xed,0xec,0xeb,0xea,0xe9,0xe8,0xe7,0xe6,0xe5,0xe4,0xe3,0xe2,
	0xe1,0xe0,0xdf,0xde,0xdd,0xdc,0xdb,0xda,0xd9,0xd8,0xd7,0xd6,0xd5,0xd4,0xd3,0xd2,
	0xd1,0xd0,0xcf,0xce,0xcd,0xcc,0xcb,0xca,0xc9,0xc8,0xc7,0xc6,0xc5,0xc4,0xc3,0xc2,
	0xc1,0xc0,0xbf,0xbe,0xbd,0xbc,0xbb,0xba,0xb9,0xb8,0xb7,0xb6,0xb5,0xb4,0xb3,0xb2,
	0xb1,0xb0,0xaf,0xae,0xad,0xac,0xab,0xaa,0xa9,0xa8,0xa7,0xa6,0xa5,0xa4,0xa3,0xa2,
	0xa1,0xa0,0x9f,0x9e,0x9d,0x9c,0x9b,0x9a,0x99,0x98,0x97,0x96,0x95,0x94,0x93,0x92,
	0x91,0x90,0x8f,0x8e,0x8d,0x8c,0x8b,0x8a,0x89,0x88,0x87,0x86,0x85,0x84,0x83,0x82,
	0x81,0x7e,0x7d,0x7c,0x7b,0x7a,0x79,0x78,0x77,0x76,0x75,0x74,0x73,0x72,0x71,0x70,
	0x6f,0x6e,0x6d,0x6c,0x6b,0x6a,0x69,0x68,0x67,0x66,0x65,0x64,0x63,0x62,0x61,0x60,
	0x5f,0x5e,0x5d,0x5c,0x5b,0x5a,0x59,0x58,0x57,0x56,0x55,0x54,0x53,0x52,0x51,0x50,
	0x4f,0x4e,0x4d,0x4c,0x4b,0x4a,0x49,0x48,0x47,0x46,0x45,0x44,0x43,0x42,0x41,0x40,
	0x3f,0x3e,0x3d,0x3c,0x3b,0x3a,0x39,0x38,0x37,0x36,0x35,0x34,0x33,0x32,0x31,0x30,
	0x2f,0x2e,0x2d,0x2c,0x2b,0x2a,0x29,0x28,0x27,0x26,0x25,0x24,0x23,0x22,0x21,0x20,
	0x1f,0x1e,0x1d,0x1c,0x1b,0x1a,0x19,0x18,0x17,0x16,0x15,0x14,0x13,0x12,0x11,0x10,
	0xf,0xe,0xd,0xc,0xb,0xa,0x9,0x8,0x7,0x6,0x5,0x4,0x3,0x2,0x1,0x0,
};

INLINE UINT32 gte_divide( INT16 numerator, UINT16 denominator )
{
	if( numerator >= 0 && numerator < ( denominator * 2 ) )
	{
		UINT32 offset = denominator;
		int shift = 0;
		UINT64 reciprocal;
		while( offset <= 0x8000 )
		{
			offset <<= 1;
			shift++;
		}

		reciprocal = (UINT64)( 0x10000 | reciprocals[ offset & 0x7fff ] ) << shift;

		return (UINT32)( ( ( reciprocal * numerator ) + 0x8000 ) >> 16 );
	}

	return 0xffffffff;
}

/* Setting bits 12 & 19-22 in FLAG does not set bit 31 */

#define A1( a ) BOUNDS( psxcpu, ( a ), 0x7fffffff, ( 1 << 30 ), -(INT64)0x80000000, ( 1 << 31 ) | ( 1 << 27 ) )
#define A2( a ) BOUNDS( psxcpu, ( a ), 0x7fffffff, ( 1 << 29 ), -(INT64)0x80000000, ( 1 << 31 ) | ( 1 << 26 ) )
#define A3( a ) BOUNDS( psxcpu, ( a ), 0x7fffffff, ( 1 << 28 ), -(INT64)0x80000000, ( 1 << 31 ) | ( 1 << 25 ) )
#define Lm_B1( a, l ) LIM( psxcpu, ( a ), 0x7fff, -0x8000 * !l, ( 1 << 31 ) | ( 1 << 24 ) )
#define Lm_B2( a, l ) LIM( psxcpu, ( a ), 0x7fff, -0x8000 * !l, ( 1 << 31 ) | ( 1 << 23 ) )
#define Lm_B3( a, l ) LIM( psxcpu, ( a ), 0x7fff, -0x8000 * !l, ( 1 << 22 ) )
#define Lm_C1( a ) LIM( psxcpu, ( a ), 0x00ff, 0x0000, ( 1 << 21 ) )
#define Lm_C2( a ) LIM( psxcpu, ( a ), 0x00ff, 0x0000, ( 1 << 20 ) )
#define Lm_C3( a ) LIM( psxcpu, ( a ), 0x00ff, 0x0000, ( 1 << 19 ) )
#define Lm_D( a ) LIM( psxcpu, ( a ), 0xffff, 0x0000, ( 1 << 31 ) | ( 1 << 18 ) )

INLINE UINT32 Lm_E( psxcpu_state *psxcpu, UINT32 result )
{
	if( result > 0x1ffff )
	{
		FLAG |= ( 1 << 31 ) | ( 1 << 17 );
		return 0x1ffff;
	}

	return result;
}

#define F( a ) BOUNDS( psxcpu, ( a ), 0x7fffffff, ( 1 << 31 ) | ( 1 << 16 ), -(INT64)0x80000000, ( 1 << 31 ) | ( 1 << 15 ) )
#define Lm_G1( a ) LIM( psxcpu, ( a ), 0x3ff, -0x400, ( 1 << 31 ) | ( 1 << 14 ) )
#define Lm_G2( a ) LIM( psxcpu, ( a ), 0x3ff, -0x400, ( 1 << 31 ) | ( 1 << 13 ) )
#define Lm_H( a ) LIM( psxcpu, ( a ), 0xfff, 0x000, ( 1 << 12 ) )

static void docop2( psxcpu_state *psxcpu, int gteop )
{
	int shift;
	int v;
	int lm;
	int cv;
	int mx;
	INT32 h_over_sz3;
	INT64 mac0;

	switch( GTE_FUNCT( gteop ) )
	{
	case 0x01:
		if( gteop == 0x0180001 )
		{
			GTELOG( "RTPS" );
			FLAG = 0;

			MAC1 = A1( ( ( (INT64) TRX << 12 ) + ( R11 * VX0 ) + ( R12 * VY0 ) + ( R13 * VZ0 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) TRY << 12 ) + ( R21 * VX0 ) + ( R22 * VY0 ) + ( R23 * VZ0 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) TRZ << 12 ) + ( R31 * VX0 ) + ( R32 * VY0 ) + ( R33 * VZ0 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 0 );
			IR2 = Lm_B2( MAC2, 0 );
			IR3 = Lm_B3( MAC3, 0 );
			SZ0 = SZ1;
			SZ1 = SZ2;
			SZ2 = SZ3;
			SZ3 = Lm_D( MAC3 );
			h_over_sz3 = Lm_E( psxcpu, gte_divide( H, SZ3 ) );
			SXY0 = SXY1;
			SXY1 = SXY2;
			SX2 = Lm_G1( F( (INT64) OFX + ( (INT64) IR1 * h_over_sz3 ) ) >> 16 );
			SY2 = Lm_G2( F( (INT64) OFY + ( (INT64) IR2 * h_over_sz3 ) ) >> 16 );
			MAC0 = F( (INT64) DQB + ( (INT64) DQA * h_over_sz3 ) );
			IR0 = Lm_H( MAC0 >> 12 );
			return;
		}
		break;

	case 0x06:
		GTELOG( "NCLIP" );
		FLAG = 0;

		MAC0 = F( (INT64)( SX0 * SY1 ) + ( SX1 * SY2 ) + ( SX2 * SY0 ) - ( SX0 * SY2 ) - ( SX1 * SY0 ) - ( SX2 * SY1 ) );
		return;

	case 0x0c:
		GTELOG( "OP" );
		FLAG = 0;

		shift = 12 * GTE_SF( gteop );
		lm = GTE_LM( gteop );

		MAC1 = A1( ( (INT64) ( R22 * IR3 ) - ( R33 * IR2 ) ) >> shift );
		MAC2 = A2( ( (INT64) ( R33 * IR1 ) - ( R11 * IR3 ) ) >> shift );
		MAC3 = A3( ( (INT64) ( R11 * IR2 ) - ( R22 * IR1 ) ) >> shift );
		IR1 = Lm_B1( MAC1, lm );
		IR2 = Lm_B2( MAC2, lm );
		IR3 = Lm_B3( MAC3, lm );
		return;

	case 0x10:
		GTELOG( "DPCS" );
		FLAG = 0;

		shift = 12 * GTE_SF( gteop );
		lm = GTE_LM( gteop );

		MAC1 = ( ( R << 16 ) + ( IR0 * Lm_B1( A1( (INT64) RFC - ( R << 4 ) ) << ( 12 - shift ), 0 ) ) ) >> shift;
		MAC2 = ( ( G << 16 ) + ( IR0 * Lm_B2( A2( (INT64) GFC - ( G << 4 ) ) << ( 12 - shift ), 0 ) ) ) >> shift;
		MAC3 = ( ( B << 16 ) + ( IR0 * Lm_B3( A3( (INT64) BFC - ( B << 4 ) ) << ( 12 - shift ), 0 ) ) ) >> shift;
		IR1 = Lm_B1( MAC1, lm );
		IR2 = Lm_B2( MAC2, lm );
		IR3 = Lm_B3( MAC3, lm );
		RGB0 = RGB1;
		RGB1 = RGB2;
		CD2 = CODE;
		R2 = Lm_C1( MAC1 >> 4 );
		G2 = Lm_C2( MAC2 >> 4 );
		B2 = Lm_C3( MAC3 >> 4 );
		return;

	case 0x11:
		GTELOG( "INTPL" );
		FLAG = 0;

		shift = 12 * GTE_SF( gteop );
		lm = GTE_LM( gteop );

		MAC1 = ( ( IR1 << 12 ) + ( IR0 * Lm_B1( A1( (INT64) RFC - IR1 ) << ( 12 - shift ), 0 ) ) ) >> shift;
		MAC2 = ( ( IR2 << 12 ) + ( IR0 * Lm_B2( A2( (INT64) GFC - IR2 ) << ( 12 - shift ), 0 ) ) ) >> shift;
		MAC3 = ( ( IR3 << 12 ) + ( IR0 * Lm_B3( A3( (INT64) BFC - IR3 ) << ( 12 - shift ), 0 ) ) ) >> shift;
		IR1 = Lm_B1( MAC1, lm );
		IR2 = Lm_B2( MAC2, lm );
		IR3 = Lm_B3( MAC3, lm );
		RGB0 = RGB1;
		RGB1 = RGB2;
		CD2 = CODE;
		R2 = Lm_C1( MAC1 >> 4 );
		G2 = Lm_C2( MAC2 >> 4 );
		B2 = Lm_C3( MAC3 >> 4 );
		return;

	case 0x12:
		if( GTE_OP( gteop ) == 0x04 )
		{
			GTELOG( "MVMVA" );
			shift = 12 * GTE_SF( gteop );
			mx = GTE_MX( gteop );
			v = GTE_V( gteop );
			cv = GTE_CV( gteop );
			lm = GTE_LM( gteop );

			FLAG = 0;

			MAC1 = A1( ( ( (INT64) CV1( cv ) << 12 ) + ( MX11( mx ) * VX( v ) ) + ( MX12( mx ) * VY( v ) ) + ( MX13( mx ) * VZ( v ) ) ) >> shift );
			MAC2 = A2( ( ( (INT64) CV2( cv ) << 12 ) + ( MX21( mx ) * VX( v ) ) + ( MX22( mx ) * VY( v ) ) + ( MX23( mx ) * VZ( v ) ) ) >> shift );
			MAC3 = A3( ( ( (INT64) CV3( cv ) << 12 ) + ( MX31( mx ) * VX( v ) ) + ( MX32( mx ) * VY( v ) ) + ( MX33( mx ) * VZ( v ) ) ) >> shift );

			IR1 = Lm_B1( MAC1, lm );
			IR2 = Lm_B2( MAC2, lm );
			IR3 = Lm_B3( MAC3, lm );
			return;
		}
		break;
	case 0x13:
		if( gteop == 0x0e80413 )
		{
			GTELOG( "NCDS" );
			FLAG = 0;

			MAC1 = A1( ( ( (INT64) L11 * VX0 ) + ( L12 * VY0 ) + ( L13 * VZ0 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) L21 * VX0 ) + ( L22 * VY0 ) + ( L23 * VZ0 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) L31 * VX0 ) + ( L32 * VY0 ) + ( L33 * VZ0 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			MAC1 = A1( ( ( ( (INT64) R << 4 ) * IR1 ) + ( IR0 * Lm_B1( RFC - ( ( R * IR1 ) >> 8 ), 0 ) ) ) >> 12 );
			MAC2 = A2( ( ( ( (INT64) G << 4 ) * IR2 ) + ( IR0 * Lm_B2( GFC - ( ( G * IR2 ) >> 8 ), 0 ) ) ) >> 12 );
			MAC3 = A3( ( ( ( (INT64) B << 4 ) * IR3 ) + ( IR0 * Lm_B3( BFC - ( ( B * IR3 ) >> 8 ), 0 ) ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			CD0 = CD1;
			CD1 = CD2;
			CD2 = CODE;
			R0 = R1;
			R1 = R2;
			R2 = Lm_C1( MAC1 >> 4 );
			G0 = G1;
			G1 = G2;
			G2 = Lm_C2( MAC2 >> 4 );
			B0 = B1;
			B1 = B2;
			B2 = Lm_C3( MAC3 >> 4 );
			return;
		}
		break;
	case 0x14:
		if( gteop == 0x1280414 )
		{
			GTELOG( "CDP" );
			FLAG = 0;

			MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			MAC1 = A1( ( ( ( (INT64) R << 4 ) * IR1 ) + ( IR0 * Lm_B1( RFC - ( ( R * IR1 ) >> 8 ), 0 ) ) ) >> 12 );
			MAC2 = A2( ( ( ( (INT64) G << 4 ) * IR2 ) + ( IR0 * Lm_B2( GFC - ( ( G * IR2 ) >> 8 ), 0 ) ) ) >> 12 );
			MAC3 = A3( ( ( ( (INT64) B << 4 ) * IR3 ) + ( IR0 * Lm_B3( BFC - ( ( B * IR3 ) >> 8 ), 0 ) ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			CD0 = CD1;
			CD1 = CD2;
			CD2 = CODE;
			R0 = R1;
			R1 = R2;
			R2 = Lm_C1( MAC1 >> 4 );
			G0 = G1;
			G1 = G2;
			G2 = Lm_C2( MAC2 >> 4 );
			B0 = B1;
			B1 = B2;
			B2 = Lm_C3( MAC3 >> 4 );
			return;
		}
		break;
	case 0x16:
		if( gteop == 0x0f80416 )
		{
			GTELOG( "NCDT" );
			FLAG = 0;

			for( v = 0; v < 3; v++ )
			{
				MAC1 = A1( ( ( (INT64) L11 * VX( v ) ) + ( L12 * VY( v ) ) + ( L13 * VZ( v ) ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) L21 * VX( v ) ) + ( L22 * VY( v ) ) + ( L23 * VZ( v ) ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) L31 * VX( v ) ) + ( L32 * VY( v ) ) + ( L33 * VZ( v ) ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				MAC1 = A1( ( ( ( (INT64) R << 4 ) * IR1 ) + ( IR0 * Lm_B1( RFC - ( ( R * IR1 ) >> 8 ), 0 ) ) ) >> 12 );
				MAC2 = A2( ( ( ( (INT64) G << 4 ) * IR2 ) + ( IR0 * Lm_B2( GFC - ( ( G * IR2 ) >> 8 ), 0 ) ) ) >> 12 );
				MAC3 = A3( ( ( ( (INT64) B << 4 ) * IR3 ) + ( IR0 * Lm_B3( BFC - ( ( B * IR3 ) >> 8 ), 0 ) ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				CD0 = CD1;
				CD1 = CD2;
				CD2 = CODE;
				R0 = R1;
				R1 = R2;
				R2 = Lm_C1( MAC1 >> 4 );
				G0 = G1;
				G1 = G2;
				G2 = Lm_C2( MAC2 >> 4 );
				B0 = B1;
				B1 = B2;
				B2 = Lm_C3( MAC3 >> 4 );
			}
			return;
		}
		break;
	case 0x1b:
		if( gteop == 0x108041b || gteop == 0x118041b )
		{
			GTELOG( "NCCS" );
			FLAG = 0;

			MAC1 = A1( ( ( (INT64) L11 * VX0 ) + ( L12 * VY0 ) + ( L13 * VZ0 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) L21 * VX0 ) + ( L22 * VY0 ) + ( L23 * VZ0 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) L31 * VX0 ) + ( L32 * VY0 ) + ( L33 * VZ0 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			MAC1 = A1( ( (INT64) R * IR1 ) >> 8 );
			MAC2 = A2( ( (INT64) G * IR2 ) >> 8 );
			MAC3 = A3( ( (INT64) B * IR3 ) >> 8 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			CD0 = CD1;
			CD1 = CD2;
			CD2 = CODE;
			R0 = R1;
			R1 = R2;
			R2 = Lm_C1( MAC1 >> 4 );
			G0 = G1;
			G1 = G2;
			G2 = Lm_C2( MAC2 >> 4 );
			B0 = B1;
			B1 = B2;
			B2 = Lm_C3( MAC3 >> 4 );
			return;
		}
		break;
	case 0x1c:
		if( gteop == 0x138041c )
		{
			GTELOG( "CC" );
			FLAG = 0;

			MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			MAC1 = A1( ( (INT64) R * IR1 ) >> 8 );
			MAC2 = A2( ( (INT64) G * IR2 ) >> 8 );
			MAC3 = A3( ( (INT64) B * IR3 ) >> 8 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			CD0 = CD1;
			CD1 = CD2;
			CD2 = CODE;
			R0 = R1;
			R1 = R2;
			R2 = Lm_C1( MAC1 >> 4 );
			G0 = G1;
			G1 = G2;
			G2 = Lm_C2( MAC2 >> 4 );
			B0 = B1;
			B1 = B2;
			B2 = Lm_C3( MAC3 >> 4 );
			return;
		}
		break;
	case 0x1e:
		if( gteop == 0x0c8041e )
		{
			GTELOG( "NCS" );
			FLAG = 0;

			MAC1 = A1( ( ( (INT64) L11 * VX0 ) + ( L12 * VY0 ) + ( L13 * VZ0 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) L21 * VX0 ) + ( L22 * VY0 ) + ( L23 * VZ0 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) L31 * VX0 ) + ( L32 * VY0 ) + ( L33 * VZ0 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
			MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
			MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
			IR1 = Lm_B1( MAC1, 1 );
			IR2 = Lm_B2( MAC2, 1 );
			IR3 = Lm_B3( MAC3, 1 );
			CD0 = CD1;
			CD1 = CD2;
			CD2 = CODE;
			R0 = R1;
			R1 = R2;
			R2 = Lm_C1( MAC1 >> 4 );
			G0 = G1;
			G1 = G2;
			G2 = Lm_C2( MAC2 >> 4 );
			B0 = B1;
			B1 = B2;
			B2 = Lm_C3( MAC3 >> 4 );
			return;
		}
		break;
	case 0x20:
		if( gteop == 0x0d80420 )
		{
			GTELOG( "NCT" );
			FLAG = 0;

			for( v = 0; v < 3; v++ )
			{
				MAC1 = A1( ( ( (INT64) L11 * VX( v ) ) + ( L12 * VY( v ) ) + ( L13 * VZ( v ) ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) L21 * VX( v ) ) + ( L22 * VY( v ) ) + ( L23 * VZ( v ) ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) L31 * VX( v ) ) + ( L32 * VY( v ) ) + ( L33 * VZ( v ) ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				CD0 = CD1;
				CD1 = CD2;
				CD2 = CODE;
				R0 = R1;
				R1 = R2;
				R2 = Lm_C1( MAC1 >> 4 );
				G0 = G1;
				G1 = G2;
				G2 = Lm_C2( MAC2 >> 4 );
				B0 = B1;
				B1 = B2;
				B2 = Lm_C3( MAC3 >> 4 );
			}
			return;
		}
		break;

	case 0x28:
		GTELOG( "SQR" );
		FLAG = 0;

		shift = 12 * GTE_SF( gteop );
		lm = GTE_LM( gteop );

		MAC1 = A1( ( IR1 * IR1 ) >> shift );
		MAC2 = A2( ( IR2 * IR2 ) >> shift );
		MAC3 = A3( ( IR3 * IR3 ) >> shift );
		IR1 = Lm_B1( MAC1, lm );
		IR2 = Lm_B2( MAC2, lm );
		IR3 = Lm_B3( MAC3, lm );
		return;

	// DCPL 0x29
	case 0x2a:
		if( gteop == 0x0f8002a )
		{
			GTELOG( "DPCT" );
			FLAG = 0;

			for( v = 0; v < 3; v++ )
			{
				MAC1 = A1( ( ( (INT64) R0 << 16 ) + ( (INT64) IR0 * ( Lm_B1( RFC - ( R0 << 4 ), 0 ) ) ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) G0 << 16 ) + ( (INT64) IR0 * ( Lm_B1( GFC - ( G0 << 4 ), 0 ) ) ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) B0 << 16 ) + ( (INT64) IR0 * ( Lm_B1( BFC - ( B0 << 4 ), 0 ) ) ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 0 );
				IR2 = Lm_B2( MAC2, 0 );
				IR3 = Lm_B3( MAC3, 0 );
				CD0 = CD1;
				CD1 = CD2;
				CD2 = CODE;
				R0 = R1;
				R1 = R2;
				R2 = Lm_C1( MAC1 >> 4 );
				G0 = G1;
				G1 = G2;
				G2 = Lm_C2( MAC2 >> 4 );
				B0 = B1;
				B1 = B2;
				B2 = Lm_C3( MAC3 >> 4 );
			}
			return;
		}
		break;

	case 0x2d:
		GTELOG( "AVSZ3" );
		FLAG = 0;

		mac0 = F( (INT64) ( ZSF3 * SZ1 ) + ( ZSF3 * SZ2 ) + ( ZSF3 * SZ3 ) );
		OTZ = Lm_D( mac0 >> 12 );

		MAC0 = mac0;
		return;

	case 0x2e:
		GTELOG( "AVSZ4" );
		FLAG = 0;

		mac0 = F( (INT64) ( ZSF4 * SZ0 ) + ( ZSF4 * SZ1 ) + ( ZSF4 * SZ2 ) + ( ZSF4 * SZ3 ) );
		OTZ = Lm_D( mac0 >> 12 );

		MAC0 = mac0;
		return;

	case 0x30:
		if( gteop == 0x0280030 )
		{
			GTELOG( "RTPT" );
			FLAG = 0;

			for( v = 0; v < 3; v++ )
			{
				MAC1 = A1( ( ( (INT64) TRX << 12 ) + ( R11 * VX( v ) ) + ( R12 * VY( v ) ) + ( R13 * VZ( v ) ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) TRY << 12 ) + ( R21 * VX( v ) ) + ( R22 * VY( v ) ) + ( R23 * VZ( v ) ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) TRZ << 12 ) + ( R31 * VX( v ) ) + ( R32 * VY( v ) ) + ( R33 * VZ( v ) ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 0 );
				IR2 = Lm_B2( MAC2, 0 );
				IR3 = Lm_B3( MAC3, 0 );
				SZ0 = SZ1;
				SZ1 = SZ2;
				SZ2 = SZ3;
				SZ3 = Lm_D( MAC3 );
				h_over_sz3 = Lm_E( psxcpu, gte_divide( H, SZ3 ) );
				SXY0 = SXY1;
				SXY1 = SXY2;
				SX2 = Lm_G1( F( ( (INT64) OFX + ( (INT64) IR1 * h_over_sz3 ) ) >> 16 ) );
				SY2 = Lm_G2( F( ( (INT64) OFY + ( (INT64) IR2 * h_over_sz3 ) ) >> 16 ) );
				MAC0 = F( (INT64) DQB + ( (INT64) DQA * h_over_sz3 ) );
				IR0 = Lm_H( MAC0 >> 12 );
			}
			return;
		}
		break;
	case 0x3d:
		if( GTE_OP( gteop ) == 0x09 ||
			GTE_OP( gteop ) == 0x19 )
		{
			GTELOG( "GPF" );
			shift = 12 * GTE_SF( gteop );
			FLAG = 0;

			MAC1 = A1( ( (INT64) IR0 * IR1 ) >> shift );
			MAC2 = A2( ( (INT64) IR0 * IR2 ) >> shift );
			MAC3 = A3( ( (INT64) IR0 * IR3 ) >> shift );
			IR1 = Lm_B1( MAC1, 0 );
			IR2 = Lm_B2( MAC2, 0 );
			IR3 = Lm_B3( MAC3, 0 );
			CD0 = CD1;
			CD1 = CD2;
			CD2 = CODE;
			R0 = R1;
			R1 = R2;
			R2 = Lm_C1( MAC1 >> 4 );
			G0 = G1;
			G1 = G2;
			G2 = Lm_C2( MAC2 >> 4 );
			B0 = B1;
			B1 = B2;
			B2 = Lm_C3( MAC3 >> 4 );
			return;
		}
		break;
	case 0x3e:
		if( GTE_OP( gteop ) == 0x1a )
		{
			GTELOG( "GPL" );
			shift = 12 * GTE_SF( gteop );
			FLAG = 0;

			MAC1 = A1( ( ( (INT64) MAC1 << shift ) + ( IR0 * IR1 ) ) >> shift );
			MAC2 = A2( ( ( (INT64) MAC2 << shift ) + ( IR0 * IR2 ) ) >> shift );
			MAC3 = A3( ( ( (INT64) MAC3 << shift ) + ( IR0 * IR3 ) ) >> shift );
			IR1 = Lm_B1( MAC1, 0 );
			IR2 = Lm_B2( MAC2, 0 );
			IR3 = Lm_B3( MAC3, 0 );
			CD0 = CD1;
			CD1 = CD2;
			CD2 = CODE;
			R0 = R1;
			R1 = R2;
			R2 = Lm_C1( MAC1 >> 4 );
			G0 = G1;
			G1 = G2;
			G2 = Lm_C2( MAC2 >> 4 );
			B0 = B1;
			B1 = B2;
			B2 = Lm_C3( MAC3 >> 4 );
			return;
		}
		break;
	case 0x3f:
		if( gteop == 0x108043f ||
			gteop == 0x118043f )
		{
			GTELOG( "NCCT" );
			FLAG = 0;

			for( v = 0; v < 3; v++ )
			{
				MAC1 = A1( ( ( (INT64) L11 * VX( v ) ) + ( L12 * VY( v ) ) + ( L13 * VZ( v ) ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) L21 * VX( v ) ) + ( L22 * VY( v ) ) + ( L23 * VZ( v ) ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) L31 * VX( v ) ) + ( L32 * VY( v ) ) + ( L33 * VZ( v ) ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				MAC1 = A1( ( ( (INT64) RBK << 12 ) + ( LR1 * IR1 ) + ( LR2 * IR2 ) + ( LR3 * IR3 ) ) >> 12 );
				MAC2 = A2( ( ( (INT64) GBK << 12 ) + ( LG1 * IR1 ) + ( LG2 * IR2 ) + ( LG3 * IR3 ) ) >> 12 );
				MAC3 = A3( ( ( (INT64) BBK << 12 ) + ( LB1 * IR1 ) + ( LB2 * IR2 ) + ( LB3 * IR3 ) ) >> 12 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				MAC1 = A1( ( (INT64) R * IR1 ) >> 8 );
				MAC2 = A2( ( (INT64) G * IR2 ) >> 8 );
				MAC3 = A3( ( (INT64) B * IR3 ) >> 8 );
				IR1 = Lm_B1( MAC1, 1 );
				IR2 = Lm_B2( MAC2, 1 );
				IR3 = Lm_B3( MAC3, 1 );
				CD0 = CD1;
				CD1 = CD2;
				CD2 = CODE;
				R0 = R1;
				R1 = R2;
				R2 = Lm_C1( MAC1 >> 4 );
				G0 = G1;
				G1 = G2;
				G2 = Lm_C2( MAC2 >> 4 );
				B0 = B1;
				B1 = B2;
				B2 = Lm_C3( MAC3 >> 4 );
			}
			return;
		}
		break;
	}
	popmessage( "unknown GTE op %08x", gteop );
	logerror( "%08x: unknown GTE op %08x\n", psxcpu->pc, gteop );
	mips_stop( psxcpu );
}

/**************************************************************************
 * Generic set_info
 **************************************************************************/

static CPU_SET_INFO( psxcpu )
{
	psxcpu_state *psxcpu = get_safe_token(device);
	switch (state)
	{
		/* --- the following bits of info are set as 64-bit signed integers --- */
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ0:		set_irq_line( psxcpu, PSXCPU_IRQ0, info->i );	break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ1:		set_irq_line( psxcpu, PSXCPU_IRQ1, info->i );	break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ2:		set_irq_line( psxcpu, PSXCPU_IRQ2, info->i );	break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ3:		set_irq_line( psxcpu, PSXCPU_IRQ3, info->i );	break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ4:		set_irq_line( psxcpu, PSXCPU_IRQ4, info->i );	break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ5:		set_irq_line( psxcpu, PSXCPU_IRQ5, info->i );	break;

		case CPUINFO_INT_PC:							mips_set_pc( psxcpu, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_PC:			mips_set_pc( psxcpu, info->i );			break;
		case CPUINFO_INT_SP:							/* no stack */							break;
		case CPUINFO_INT_REGISTER + PSXCPU_DELAYV:		psxcpu->delayv = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_DELAYR:		if( info->i <= PSXCPU_DELAYR_NOTPC ) psxcpu->delayr = info->i; break;
		case CPUINFO_INT_REGISTER + PSXCPU_HI:			psxcpu->hi = info->i;					break;
		case CPUINFO_INT_REGISTER + PSXCPU_LO:			psxcpu->lo = info->i;					break;
		case CPUINFO_INT_REGISTER + PSXCPU_BIU:			psxcpu->biu = info->i;					break;
		case CPUINFO_INT_REGISTER + PSXCPU_R0:			psxcpu->r[ 0 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R1:			psxcpu->r[ 1 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R2:			psxcpu->r[ 2 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R3:			psxcpu->r[ 3 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R4:			psxcpu->r[ 4 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R5:			psxcpu->r[ 5 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R6:			psxcpu->r[ 6 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R7:			psxcpu->r[ 7 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R8:			psxcpu->r[ 8 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R9:			psxcpu->r[ 9 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R10:			psxcpu->r[ 10 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R11:			psxcpu->r[ 11 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R12:			psxcpu->r[ 12 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R13:			psxcpu->r[ 13 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R14:			psxcpu->r[ 14 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R15:			psxcpu->r[ 15 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R16:			psxcpu->r[ 16 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R17:			psxcpu->r[ 17 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R18:			psxcpu->r[ 18 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R19:			psxcpu->r[ 19 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R20:			psxcpu->r[ 20 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R21:			psxcpu->r[ 21 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R22:			psxcpu->r[ 22 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R23:			psxcpu->r[ 23 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R24:			psxcpu->r[ 24 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R25:			psxcpu->r[ 25 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R26:			psxcpu->r[ 26 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R27:			psxcpu->r[ 27 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R28:			psxcpu->r[ 28 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R29:			psxcpu->r[ 29 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R30:			psxcpu->r[ 30 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R31:			psxcpu->r[ 31 ] = info->i;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R0:			mips_set_cp0r( psxcpu, 0, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R1:			mips_set_cp0r( psxcpu, 1, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R2:			mips_set_cp0r( psxcpu, 2, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R3:			mips_set_cp0r( psxcpu, 3, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R4:			mips_set_cp0r( psxcpu, 4, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R5:			mips_set_cp0r( psxcpu, 5, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R6:			mips_set_cp0r( psxcpu, 6, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R7:			mips_set_cp0r( psxcpu, 7, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R8:			mips_set_cp0r( psxcpu, 8, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R9:			mips_set_cp0r( psxcpu, 9, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R10:		mips_set_cp0r( psxcpu, 10, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R11:		mips_set_cp0r( psxcpu, 11, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R12:		mips_set_cp0r( psxcpu, 12, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R13:		mips_set_cp0r( psxcpu, 13, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R14:		mips_set_cp0r( psxcpu, 14, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R15:		mips_set_cp0r( psxcpu, 15, info->i );			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR0:		psxcpu->cp2dr[ 0 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR1:		psxcpu->cp2dr[ 1 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR2:		psxcpu->cp2dr[ 2 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR3:		psxcpu->cp2dr[ 3 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR4:		psxcpu->cp2dr[ 4 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR5:		psxcpu->cp2dr[ 5 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR6:		psxcpu->cp2dr[ 6 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR7:		psxcpu->cp2dr[ 7 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR8:		psxcpu->cp2dr[ 8 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR9:		psxcpu->cp2dr[ 9 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR10:		psxcpu->cp2dr[ 10 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR11:		psxcpu->cp2dr[ 11 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR12:		psxcpu->cp2dr[ 12 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR13:		psxcpu->cp2dr[ 13 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR14:		psxcpu->cp2dr[ 14 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR15:		psxcpu->cp2dr[ 15 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR16:		psxcpu->cp2dr[ 16 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR17:		psxcpu->cp2dr[ 17 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR18:		psxcpu->cp2dr[ 18 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR19:		psxcpu->cp2dr[ 19 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR20:		psxcpu->cp2dr[ 20 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR21:		psxcpu->cp2dr[ 21 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR22:		psxcpu->cp2dr[ 22 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR23:		psxcpu->cp2dr[ 23 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR24:		psxcpu->cp2dr[ 24 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR25:		psxcpu->cp2dr[ 25 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR26:		psxcpu->cp2dr[ 26 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR27:		psxcpu->cp2dr[ 27 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR28:		psxcpu->cp2dr[ 28 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR29:		psxcpu->cp2dr[ 29 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR30:		psxcpu->cp2dr[ 30 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR31:		psxcpu->cp2dr[ 31 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR0:		psxcpu->cp2cr[ 0 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR1:		psxcpu->cp2cr[ 1 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR2:		psxcpu->cp2cr[ 2 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR3:		psxcpu->cp2cr[ 3 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR4:		psxcpu->cp2cr[ 4 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR5:		psxcpu->cp2cr[ 5 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR6:		psxcpu->cp2cr[ 6 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR7:		psxcpu->cp2cr[ 7 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR8:		psxcpu->cp2cr[ 8 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR9:		psxcpu->cp2cr[ 9 ].d = info->i;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR10:		psxcpu->cp2cr[ 10 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR11:		psxcpu->cp2cr[ 11 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR12:		psxcpu->cp2cr[ 12 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR13:		psxcpu->cp2cr[ 13 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR14:		psxcpu->cp2cr[ 14 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR15:		psxcpu->cp2cr[ 15 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR16:		psxcpu->cp2cr[ 16 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR17:		psxcpu->cp2cr[ 17 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR18:		psxcpu->cp2cr[ 18 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR19:		psxcpu->cp2cr[ 19 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR20:		psxcpu->cp2cr[ 20 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR21:		psxcpu->cp2cr[ 21 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR22:		psxcpu->cp2cr[ 22 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR23:		psxcpu->cp2cr[ 23 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR24:		psxcpu->cp2cr[ 24 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR25:		psxcpu->cp2cr[ 25 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR26:		psxcpu->cp2cr[ 26 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR27:		psxcpu->cp2cr[ 27 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR28:		psxcpu->cp2cr[ 28 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR29:		psxcpu->cp2cr[ 29 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR30:		psxcpu->cp2cr[ 30 ].d = info->i;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR31:		psxcpu->cp2cr[ 31 ].d = info->i;		break;
	}
}



/**************************************************************************
 * Generic get_info
 **************************************************************************/

CPU_GET_INFO( psxcpu )
{
	psxcpu_state *psxcpu = (device != NULL && device->token != NULL) ? get_safe_token(device) : NULL;
	switch (state)
	{
		/* --- the following bits of info are returned as 64-bit signed integers --- */
		case CPUINFO_INT_CONTEXT_SIZE:					info->i = sizeof(psxcpu_state);			break;
		case CPUINFO_INT_INPUT_LINES:					info->i = 6;							break;
		case CPUINFO_INT_DEFAULT_IRQ_VECTOR:			info->i = 0;							break;
		case DEVINFO_INT_ENDIANNESS:					info->i = ENDIANNESS_LITTLE;					break;
		case CPUINFO_INT_CLOCK_MULTIPLIER:				info->i = 1;							break;
		case CPUINFO_INT_CLOCK_DIVIDER:					info->i = 2 * 2;							break;
		case CPUINFO_INT_MIN_INSTRUCTION_BYTES:			info->i = 4;							break;
		case CPUINFO_INT_MAX_INSTRUCTION_BYTES:			info->i = 4;							break;
		case CPUINFO_INT_MIN_CYCLES:					info->i = 1;							break;
		case CPUINFO_INT_MAX_CYCLES:					info->i = 40;							break;

		case CPUINFO_INT_DATABUS_WIDTH_PROGRAM:	info->i = 32;					break;
		case CPUINFO_INT_ADDRBUS_WIDTH_PROGRAM: info->i = 32;					break;
		case CPUINFO_INT_ADDRBUS_SHIFT_PROGRAM: info->i = 0;					break;
		case CPUINFO_INT_DATABUS_WIDTH_DATA:	info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_WIDTH_DATA: 	info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_SHIFT_DATA: 	info->i = 0;					break;
		case CPUINFO_INT_DATABUS_WIDTH_IO:		info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_WIDTH_IO: 		info->i = 0;					break;
		case CPUINFO_INT_ADDRBUS_SHIFT_IO: 		info->i = 0;					break;

		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ0:		info->i = (psxcpu->cp0r[ CP0_CAUSE ] & 0x400) ? ASSERT_LINE : CLEAR_LINE; break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ1:		info->i = (psxcpu->cp0r[ CP0_CAUSE ] & 0x800) ? ASSERT_LINE : CLEAR_LINE; break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ2:		info->i = (psxcpu->cp0r[ CP0_CAUSE ] & 0x1000) ? ASSERT_LINE : CLEAR_LINE; break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ3:		info->i = (psxcpu->cp0r[ CP0_CAUSE ] & 0x2000) ? ASSERT_LINE : CLEAR_LINE; break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ4:		info->i = (psxcpu->cp0r[ CP0_CAUSE ] & 0x4000) ? ASSERT_LINE : CLEAR_LINE; break;
		case CPUINFO_INT_INPUT_STATE + PSXCPU_IRQ5:		info->i = (psxcpu->cp0r[ CP0_CAUSE ] & 0x8000) ? ASSERT_LINE : CLEAR_LINE; break;

		case CPUINFO_INT_PREVIOUSPC:					/* not implemented */					break;

		case CPUINFO_INT_PC:							info->i = psxcpu->pc;					break;
		case CPUINFO_INT_REGISTER + PSXCPU_PC:			info->i = psxcpu->pc;					break;
		case CPUINFO_INT_SP:
			/* because there is no hardware stack and the pipeline causes the cpu to execute the
            instruction after a subroutine call before the subroutine is executed there is little
            chance of cmd_step_over() in mamedbg.c working. */
								info->i = 0;													break;
		case CPUINFO_INT_REGISTER + PSXCPU_DELAYV:		info->i = psxcpu->delayv;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_DELAYR:		info->i = psxcpu->delayr;				break;
		case CPUINFO_INT_REGISTER + PSXCPU_HI:			info->i = psxcpu->hi;					break;
		case CPUINFO_INT_REGISTER + PSXCPU_LO:			info->i = psxcpu->lo;					break;
		case CPUINFO_INT_REGISTER + PSXCPU_BIU:			info->i = psxcpu->biu;					break;
		case CPUINFO_INT_REGISTER + PSXCPU_R0:			info->i = psxcpu->r[ 0 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R1:			info->i = psxcpu->r[ 1 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R2:			info->i = psxcpu->r[ 2 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R3:			info->i = psxcpu->r[ 3 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R4:			info->i = psxcpu->r[ 4 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R5:			info->i = psxcpu->r[ 5 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R6:			info->i = psxcpu->r[ 6 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R7:			info->i = psxcpu->r[ 7 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R8:			info->i = psxcpu->r[ 8 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R9:			info->i = psxcpu->r[ 9 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R10:			info->i = psxcpu->r[ 10 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R11:			info->i = psxcpu->r[ 11 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R12:			info->i = psxcpu->r[ 12 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R13:			info->i = psxcpu->r[ 13 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R14:			info->i = psxcpu->r[ 14 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R15:			info->i = psxcpu->r[ 15 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R16:			info->i = psxcpu->r[ 16 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R17:			info->i = psxcpu->r[ 17 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R18:			info->i = psxcpu->r[ 18 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R19:			info->i = psxcpu->r[ 19 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R20:			info->i = psxcpu->r[ 20 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R21:			info->i = psxcpu->r[ 21 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R22:			info->i = psxcpu->r[ 22 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R23:			info->i = psxcpu->r[ 23 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R24:			info->i = psxcpu->r[ 24 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R25:			info->i = psxcpu->r[ 25 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R26:			info->i = psxcpu->r[ 26 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R27:			info->i = psxcpu->r[ 27 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R28:			info->i = psxcpu->r[ 28 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R29:			info->i = psxcpu->r[ 29 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R30:			info->i = psxcpu->r[ 30 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_R31:			info->i = psxcpu->r[ 31 ];				break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R0:		info->i = psxcpu->cp0r[ 0 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R1:		info->i = psxcpu->cp0r[ 1 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R2:		info->i = psxcpu->cp0r[ 2 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R3:		info->i = psxcpu->cp0r[ 3 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R4:		info->i = psxcpu->cp0r[ 4 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R5:		info->i = psxcpu->cp0r[ 5 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R6:		info->i = psxcpu->cp0r[ 6 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R7:		info->i = psxcpu->cp0r[ 7 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R8:		info->i = psxcpu->cp0r[ 8 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R9:		info->i = psxcpu->cp0r[ 9 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R10:		info->i = psxcpu->cp0r[ 10 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R11:		info->i = psxcpu->cp0r[ 11 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R12:		info->i = psxcpu->cp0r[ 12 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R13:		info->i = psxcpu->cp0r[ 13 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R14:		info->i = psxcpu->cp0r[ 14 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP0R15:		info->i = psxcpu->cp0r[ 15 ];			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR0:		info->i = psxcpu->cp2dr[ 0 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR1:		info->i = psxcpu->cp2dr[ 1 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR2:		info->i = psxcpu->cp2dr[ 2 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR3:		info->i = psxcpu->cp2dr[ 3 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR4:		info->i = psxcpu->cp2dr[ 4 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR5:		info->i = psxcpu->cp2dr[ 5 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR6:		info->i = psxcpu->cp2dr[ 6 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR7:		info->i = psxcpu->cp2dr[ 7 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR8:		info->i = psxcpu->cp2dr[ 8 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR9:		info->i = psxcpu->cp2dr[ 9 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR10:		info->i = psxcpu->cp2dr[ 10 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR11:		info->i = psxcpu->cp2dr[ 11 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR12:		info->i = psxcpu->cp2dr[ 12 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR13:		info->i = psxcpu->cp2dr[ 13 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR14:		info->i = psxcpu->cp2dr[ 14 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR15:		info->i = psxcpu->cp2dr[ 15 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR16:		info->i = psxcpu->cp2dr[ 16 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR17:		info->i = psxcpu->cp2dr[ 17 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR18:		info->i = psxcpu->cp2dr[ 18 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR19:		info->i = psxcpu->cp2dr[ 19 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR20:		info->i = psxcpu->cp2dr[ 20 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR21:		info->i = psxcpu->cp2dr[ 21 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR22:		info->i = psxcpu->cp2dr[ 22 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR23:		info->i = psxcpu->cp2dr[ 23 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR24:		info->i = psxcpu->cp2dr[ 24 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR25:		info->i = psxcpu->cp2dr[ 25 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR26:		info->i = psxcpu->cp2dr[ 26 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR27:		info->i = psxcpu->cp2dr[ 27 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR28:		info->i = psxcpu->cp2dr[ 28 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR29:		info->i = psxcpu->cp2dr[ 29 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR30:		info->i = psxcpu->cp2dr[ 30 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2DR31:		info->i = psxcpu->cp2dr[ 31 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR0:		info->i = psxcpu->cp2cr[ 0 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR1:		info->i = psxcpu->cp2cr[ 1 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR2:		info->i = psxcpu->cp2cr[ 2 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR3:		info->i = psxcpu->cp2cr[ 3 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR4:		info->i = psxcpu->cp2cr[ 4 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR5:		info->i = psxcpu->cp2cr[ 5 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR6:		info->i = psxcpu->cp2cr[ 6 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR7:		info->i = psxcpu->cp2cr[ 7 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR8:		info->i = psxcpu->cp2cr[ 8 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR9:		info->i = psxcpu->cp2cr[ 9 ].d;			break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR10:		info->i = psxcpu->cp2cr[ 10 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR11:		info->i = psxcpu->cp2cr[ 11 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR12:		info->i = psxcpu->cp2cr[ 12 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR13:		info->i = psxcpu->cp2cr[ 13 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR14:		info->i = psxcpu->cp2cr[ 14 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR15:		info->i = psxcpu->cp2cr[ 15 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR16:		info->i = psxcpu->cp2cr[ 16 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR17:		info->i = psxcpu->cp2cr[ 17 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR18:		info->i = psxcpu->cp2cr[ 18 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR19:		info->i = psxcpu->cp2cr[ 19 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR20:		info->i = psxcpu->cp2cr[ 20 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR21:		info->i = psxcpu->cp2cr[ 21 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR22:		info->i = psxcpu->cp2cr[ 22 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR23:		info->i = psxcpu->cp2cr[ 23 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR24:		info->i = psxcpu->cp2cr[ 24 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR25:		info->i = psxcpu->cp2cr[ 25 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR26:		info->i = psxcpu->cp2cr[ 26 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR27:		info->i = psxcpu->cp2cr[ 27 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR28:		info->i = psxcpu->cp2cr[ 28 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR29:		info->i = psxcpu->cp2cr[ 29 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR30:		info->i = psxcpu->cp2cr[ 30 ].d;		break;
		case CPUINFO_INT_REGISTER + PSXCPU_CP2CR31:		info->i = psxcpu->cp2cr[ 31 ].d;		break;

		/* --- the following bits of info are returned as pointers to data or functions --- */
		case CPUINFO_FCT_SET_INFO:						info->setinfo = CPU_SET_INFO_NAME(psxcpu);			break;
		case CPUINFO_FCT_INIT:							info->init = CPU_INIT_NAME(psxcpu);					break;
		case CPUINFO_FCT_RESET:							info->reset = CPU_RESET_NAME(psxcpu);				break;
		case CPUINFO_FCT_EXIT:							info->exit = CPU_EXIT_NAME(psxcpu);					break;
		case CPUINFO_FCT_EXECUTE:						info->execute = CPU_EXECUTE_NAME(psxcpu);			break;
		case CPUINFO_FCT_BURN:							info->burn = NULL;						break;
		case CPUINFO_FCT_DISASSEMBLE:					info->disassemble = CPU_DISASSEMBLE_NAME(psxcpu);	break;
		case CPUINFO_PTR_INSTRUCTION_COUNTER:			info->icount = &psxcpu->icount;			break;

		case CPUINFO_PTR_INTERNAL_MEMORY_MAP_PROGRAM: info->internal_map32 = ADDRESS_MAP_NAME(psxcpu_internal_map); break;
		case CPUINFO_PTR_INTERNAL_MEMORY_MAP_DATA:    info->internal_map32 = 0; break;
		case CPUINFO_PTR_INTERNAL_MEMORY_MAP_IO:      info->internal_map32 = 0; break;

		/* --- the following bits of info are returned as NULL-terminated strings --- */
		case DEVINFO_STR_NAME:							strcpy(info->s, "PSX CPU"); break;
		case DEVINFO_STR_FAMILY:					strcpy(info->s, "MIPS"); break;
		case DEVINFO_STR_VERSION:					strcpy(info->s, "2.0"); break;
		case DEVINFO_STR_SOURCE_FILE:						strcpy(info->s, __FILE__); break;
		case DEVINFO_STR_CREDITS:					strcpy(info->s, "Copyright 2008 smf"); break;

		case CPUINFO_STR_FLAGS:							strcpy(info->s, " "); break;

		case CPUINFO_STR_REGISTER + PSXCPU_PC:			sprintf( info->s, "pc      :%08x", psxcpu->pc ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_DELAYV:		sprintf( info->s, "delayv  :%08x", psxcpu->delayv ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_DELAYR:		sprintf( info->s, "delayr  :%02x %s", psxcpu->delayr, delayn[ psxcpu->delayr ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_HI:			sprintf( info->s, "hi      :%08x", psxcpu->hi ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_LO:			sprintf( info->s, "lo      :%08x", psxcpu->lo ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_BIU:			sprintf( info->s, "biu     :%08x", psxcpu->biu ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R0:			sprintf( info->s, "zero    :%08x", psxcpu->r[ 0 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R1:			sprintf( info->s, "at      :%08x", psxcpu->r[ 1 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R2:			sprintf( info->s, "v0      :%08x", psxcpu->r[ 2 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R3:			sprintf( info->s, "v1      :%08x", psxcpu->r[ 3 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R4:			sprintf( info->s, "a0      :%08x", psxcpu->r[ 4 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R5:			sprintf( info->s, "a1      :%08x", psxcpu->r[ 5 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R6:			sprintf( info->s, "a2      :%08x", psxcpu->r[ 6 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R7:			sprintf( info->s, "a3      :%08x", psxcpu->r[ 7 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R8:			sprintf( info->s, "t0      :%08x", psxcpu->r[ 8 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R9:			sprintf( info->s, "t1      :%08x", psxcpu->r[ 9 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R10:			sprintf( info->s, "t2      :%08x", psxcpu->r[ 10 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R11:			sprintf( info->s, "t3      :%08x", psxcpu->r[ 11 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R12:			sprintf( info->s, "t4      :%08x", psxcpu->r[ 12 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R13:			sprintf( info->s, "t5      :%08x", psxcpu->r[ 13 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R14:			sprintf( info->s, "t6      :%08x", psxcpu->r[ 14 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R15:			sprintf( info->s, "t7      :%08x", psxcpu->r[ 15 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R16:			sprintf( info->s, "s0      :%08x", psxcpu->r[ 16 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R17:			sprintf( info->s, "s1      :%08x", psxcpu->r[ 17 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R18:			sprintf( info->s, "s2      :%08x", psxcpu->r[ 18 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R19:			sprintf( info->s, "s3      :%08x", psxcpu->r[ 19 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R20:			sprintf( info->s, "s4      :%08x", psxcpu->r[ 20 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R21:			sprintf( info->s, "s5      :%08x", psxcpu->r[ 21 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R22:			sprintf( info->s, "s6      :%08x", psxcpu->r[ 22 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R23:			sprintf( info->s, "s7      :%08x", psxcpu->r[ 23 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R24:			sprintf( info->s, "t8      :%08x", psxcpu->r[ 24 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R25:			sprintf( info->s, "t9      :%08x", psxcpu->r[ 25 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R26:			sprintf( info->s, "k0      :%08x", psxcpu->r[ 26 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R27:			sprintf( info->s, "k1      :%08x", psxcpu->r[ 27 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R28:			sprintf( info->s, "gp      :%08x", psxcpu->r[ 28 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R29:			sprintf( info->s, "sp      :%08x", psxcpu->r[ 29 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R30:			sprintf( info->s, "fp      :%08x", psxcpu->r[ 30 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_R31:			sprintf( info->s, "ra      :%08x", psxcpu->r[ 31 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R0:		sprintf( info->s, "!Index  :%08x", psxcpu->cp0r[ 0 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R1:		sprintf( info->s, "!Random :%08x", psxcpu->cp0r[ 1 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R2:		sprintf( info->s, "!EntryLo:%08x", psxcpu->cp0r[ 2 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R3:		sprintf( info->s, "BPC     :%08x", psxcpu->cp0r[ 3 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R4:		sprintf( info->s, "!Context:%08x", psxcpu->cp0r[ 4 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R5:		sprintf( info->s, "BDA     :%08x", psxcpu->cp0r[ 5 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R6:		sprintf( info->s, "TAR     :%08x", psxcpu->cp0r[ 6 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R7:		sprintf( info->s, "DCIC    :%08x", psxcpu->cp0r[ 7 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R8:		sprintf( info->s, "BadA    :%08x", psxcpu->cp0r[ 8 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R9:		sprintf( info->s, "BDAM    :%08x", psxcpu->cp0r[ 9 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R10:		sprintf( info->s, "!EntryHi:%08x", psxcpu->cp0r[ 10 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R11:		sprintf( info->s, "BPCM    :%08x", psxcpu->cp0r[ 11 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R12:		sprintf( info->s, "SR      :%08x", psxcpu->cp0r[ 12 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R13:		sprintf( info->s, "Cause   :%08x", psxcpu->cp0r[ 13 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R14:		sprintf( info->s, "EPC     :%08x", psxcpu->cp0r[ 14 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP0R15:		sprintf( info->s, "PRId    :%08x", psxcpu->cp0r[ 15 ] ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR0:		sprintf( info->s, "vxy0    :%08x", psxcpu->cp2dr[ 0 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR1:		sprintf( info->s, "vz0     :%08x", psxcpu->cp2dr[ 1 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR2:		sprintf( info->s, "vxy1    :%08x", psxcpu->cp2dr[ 2 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR3:		sprintf( info->s, "vz1     :%08x", psxcpu->cp2dr[ 3 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR4:		sprintf( info->s, "vxy2    :%08x", psxcpu->cp2dr[ 4 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR5:		sprintf( info->s, "vz2     :%08x", psxcpu->cp2dr[ 5 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR6:		sprintf( info->s, "rgb     :%08x", psxcpu->cp2dr[ 6 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR7:		sprintf( info->s, "otz     :%08x", psxcpu->cp2dr[ 7 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR8:		sprintf( info->s, "ir0     :%08x", psxcpu->cp2dr[ 8 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR9:		sprintf( info->s, "ir1     :%08x", psxcpu->cp2dr[ 9 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR10:		sprintf( info->s, "ir2     :%08x", psxcpu->cp2dr[ 10 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR11:		sprintf( info->s, "ir3     :%08x", psxcpu->cp2dr[ 11 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR12:		sprintf( info->s, "sxy0    :%08x", psxcpu->cp2dr[ 12 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR13:		sprintf( info->s, "sxy1    :%08x", psxcpu->cp2dr[ 13 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR14:		sprintf( info->s, "sxy2    :%08x", psxcpu->cp2dr[ 14 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR15:		sprintf( info->s, "sxyp    :%08x", psxcpu->cp2dr[ 15 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR16:		sprintf( info->s, "sz0     :%08x", psxcpu->cp2dr[ 16 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR17:		sprintf( info->s, "sz1     :%08x", psxcpu->cp2dr[ 17 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR18:		sprintf( info->s, "sz2     :%08x", psxcpu->cp2dr[ 18 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR19:		sprintf( info->s, "sz3     :%08x", psxcpu->cp2dr[ 19 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR20:		sprintf( info->s, "rgb0    :%08x", psxcpu->cp2dr[ 20 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR21:		sprintf( info->s, "rgb1    :%08x", psxcpu->cp2dr[ 21 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR22:		sprintf( info->s, "rgb2    :%08x", psxcpu->cp2dr[ 22 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR23:		sprintf( info->s, "res1    :%08x", psxcpu->cp2dr[ 23 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR24:		sprintf( info->s, "mac0    :%08x", psxcpu->cp2dr[ 24 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR25:		sprintf( info->s, "mac1    :%08x", psxcpu->cp2dr[ 25 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR26:		sprintf( info->s, "mac2    :%08x", psxcpu->cp2dr[ 26 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR27:		sprintf( info->s, "mac3    :%08x", psxcpu->cp2dr[ 27 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR28:		sprintf( info->s, "irgb    :%08x", psxcpu->cp2dr[ 28 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR29:		sprintf( info->s, "orgb    :%08x", psxcpu->cp2dr[ 29 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR30:		sprintf( info->s, "lzcs    :%08x", psxcpu->cp2dr[ 30 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2DR31:		sprintf( info->s, "lzcr    :%08x", psxcpu->cp2dr[ 31 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR0:		sprintf( info->s, "r11r12  :%08x", psxcpu->cp2cr[ 0 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR1:		sprintf( info->s, "r13r21  :%08x", psxcpu->cp2cr[ 1 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR2:		sprintf( info->s, "r22r23  :%08x", psxcpu->cp2cr[ 2 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR3:		sprintf( info->s, "r31r32  :%08x", psxcpu->cp2cr[ 3 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR4:		sprintf( info->s, "r33     :%08x", psxcpu->cp2cr[ 4 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR5:		sprintf( info->s, "trx     :%08x", psxcpu->cp2cr[ 5 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR6:		sprintf( info->s, "try     :%08x", psxcpu->cp2cr[ 6 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR7:		sprintf( info->s, "trz     :%08x", psxcpu->cp2cr[ 7 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR8:		sprintf( info->s, "l11l12  :%08x", psxcpu->cp2cr[ 8 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR9:		sprintf( info->s, "l13l21  :%08x", psxcpu->cp2cr[ 9 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR10:		sprintf( info->s, "l22l23  :%08x", psxcpu->cp2cr[ 10 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR11:		sprintf( info->s, "l31l32  :%08x", psxcpu->cp2cr[ 11 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR12:		sprintf( info->s, "l33     :%08x", psxcpu->cp2cr[ 12 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR13:		sprintf( info->s, "rbk     :%08x", psxcpu->cp2cr[ 13 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR14:		sprintf( info->s, "gbk     :%08x", psxcpu->cp2cr[ 14 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR15:		sprintf( info->s, "bbk     :%08x", psxcpu->cp2cr[ 15 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR16:		sprintf( info->s, "lr1lr2  :%08x", psxcpu->cp2cr[ 16 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR17:		sprintf( info->s, "lr31g1  :%08x", psxcpu->cp2cr[ 17 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR18:		sprintf( info->s, "lg2lg3  :%08x", psxcpu->cp2cr[ 18 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR19:		sprintf( info->s, "lb1lb2  :%08x", psxcpu->cp2cr[ 19 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR20:		sprintf( info->s, "lb3     :%08x", psxcpu->cp2cr[ 20 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR21:		sprintf( info->s, "rfc     :%08x", psxcpu->cp2cr[ 21 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR22:		sprintf( info->s, "gfc     :%08x", psxcpu->cp2cr[ 22 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR23:		sprintf( info->s, "bfc     :%08x", psxcpu->cp2cr[ 23 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR24:		sprintf( info->s, "ofx     :%08x", psxcpu->cp2cr[ 24 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR25:		sprintf( info->s, "ofy     :%08x", psxcpu->cp2cr[ 25 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR26:		sprintf( info->s, "h       :%08x", psxcpu->cp2cr[ 26 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR27:		sprintf( info->s, "dqa     :%08x", psxcpu->cp2cr[ 27 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR28:		sprintf( info->s, "dqb     :%08x", psxcpu->cp2cr[ 28 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR29:		sprintf( info->s, "zsf3    :%08x", psxcpu->cp2cr[ 29 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR30:		sprintf( info->s, "zsf4    :%08x", psxcpu->cp2cr[ 30 ].d ); break;
		case CPUINFO_STR_REGISTER + PSXCPU_CP2CR31:		sprintf( info->s, "flag    :%08x", psxcpu->cp2cr[ 31 ].d ); break;
	}
}



CPU_GET_INFO( cxd8661r )
{
	switch (state)
	{
		case CPUINFO_PTR_INTERNAL_MEMORY_MAP_PROGRAM: info->internal_map32 = ADDRESS_MAP_NAME(cxd8661r_internal_map); break;

			/* --- the following bits of info are returned as NULL-terminated strings --- */
		case DEVINFO_STR_NAME:							strcpy(info->s, "CXD8661R"); break;

		default:
			CPU_GET_INFO_CALL(psxcpu);
			break;
	}
}

