/**
 *  ServingXML
 *  
 *  Copyright (C) 2006  Daniel Parker
 *    daniel.parker@servingxml.com 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License. 
 * 
 **/

package com.servingxml.components.flatfile.recordtype;

import java.io.IOException;

import com.servingxml.app.Flow;
import com.servingxml.app.ServiceContext;
import com.servingxml.components.flatfile.RecordInput;
import com.servingxml.components.flatfile.options.FlatFileOptions;
import com.servingxml.util.Name;
import com.servingxml.util.ServingXmlException;
import com.servingxml.util.StringHelper;
import com.servingxml.util.record.RecordBuilder;
import com.servingxml.util.record.Record;
import com.servingxml.expr.substitution.IntegerSubstitutionExpr;
import com.servingxml.components.parameter.DefaultValue;
import com.servingxml.components.flatfile.options.Delimiter;

public class PositionalFieldReader implements FlatRecordFieldReader {
  private final Name name;
  private final int startPosition;
  private final IntegerSubstitutionExpr fieldWidthExpr;
  private final boolean trimLeading;
  private final boolean trimTrailing;
  private final DefaultValue defaultValue;

  public PositionalFieldReader(Name name, int startPosition, IntegerSubstitutionExpr fieldWidthExpr, 
                               DefaultValue defaultValue, 
                               FlatFileOptions flatFileOptions) {
    this.name = name;
    this.startPosition = startPosition;
    this.fieldWidthExpr = fieldWidthExpr;
    this.trimLeading = flatFileOptions.isTrimLeading();
    this.trimTrailing = flatFileOptions.isTrimTrailing();
    this.defaultValue = defaultValue;
  }

  public void readField(ServiceContext context, 
                        Flow flow, 
                        final RecordInput recordInput, 
                        Delimiter[] recordDelimiters, 
                        int recordDelimiterStart, int recordDelimiterCount, 
                        RecordBuilder recordBuilder) {

    try {
      //System.out.println(getClass().getName()+".readField startPosition="+startPosition+",initial position="+recordInput.getPosition());
      if (startPosition != -1) {
        recordInput.setPosition(startPosition);
      }
      //System.out.println(getClass().getName()+".readField final position="+recordInput.getPosition());

      int fieldWidth = fieldWidthExpr.evaluateAsInt(flow.getParameters(),recordBuilder);
      if (fieldWidth > 0) {
        //System.out.println(getClass().getName()+".readField *** position=" + recordInput.getPosition() 
        //   + ", last= " + recordInput.getLast()  + ", fieldWidth=" + fieldWidth);
        String value = recordInput.readString(fieldWidth);
        value = StringHelper.trim(value, trimLeading, trimTrailing); 
        //System.out.println(getClass().getName()+".readField name="+name+",length="+value.length()+", value="+value);
        if (value.length() == 0) {
          value = defaultValue.evaluateString(context,flow);
          //System.out.println(getClass().getName()+".readField defaultValue="+value);
        }
        recordBuilder.setString(name,value);
        //System.out.println(getClass().getName()+".readField position=" + recordInput.getPosition() 
        //  + ", last= " + recordInput.getLast() + ", length = "
        //  + recordInput.length());
      }
    } catch (IOException e) {
      throw new ServingXmlException(e.getMessage(), e);
    }
  }

  public int getFixedEndPosition(Record parameters, Record currentRecord, int currentPosition) {
    int fieldWidth = fieldWidthExpr.evaluateAsInt(parameters, currentRecord);
    return startPosition >= 0 ? startPosition+fieldWidth : currentPosition+fieldWidth;
  }
}
