/**
 *  ServingXML
 *  
 *  Copyright (C) 2006  Daniel Parker
 *    daniel.parker@servingxml.com 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License. 
 * 
 **/

package com.servingxml.components.flatfile.scanner.bytes;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import com.servingxml.components.flatfile.RecordInput;
import com.servingxml.components.flatfile.options.DelimiterByteChecker;
import com.servingxml.components.flatfile.options.FlatFileOptions;
import com.servingxml.components.flatfile.options.QuoteSymbolByteChecker;
import com.servingxml.components.flatfile.options.ByteTrimmer;
import com.servingxml.util.ByteArrayBuilder;
import com.servingxml.util.CharsetHelper;
import com.servingxml.util.ServingXmlException;

public class ByteRecordInput implements RecordInput {
  private final byte[] data;
  private final int start;
  private final int length;
  private int currentIndex;
  private int endIndex;
  private final Charset charset;

  public ByteRecordInput(byte[] data, int start, int length, Charset charset) {
    this.data = data;
    this.start = start;
    this.length = length;
    this.currentIndex = 0;
    this.endIndex = 0;
    this.charset = (charset == null) ? Charset.defaultCharset() : charset;
    //System.out.println(charset.toString());
  }

  public ByteRecordInput(byte[] data, Charset charset) {
    this.data = data;
    this.start = 0;
    this.length = data.length;
    this.currentIndex = 0;
    this.endIndex = 0;
    this.charset = (charset == null) ? Charset.defaultCharset() : charset;
  }

  public byte[] toByteArray() {
    byte[] newData = new byte[length];
    System.arraycopy(data,start,newData,0,length);
    return newData;
  }

  public char[] toCharArray() {
    return CharsetHelper.bytesToCharacters(data, start, length, charset);
  }

  public boolean done() {
    return currentIndex >= length - start;
  }

  public int readBytes(byte[] value) {
    int len = value.length <= length - currentIndex ? value.length : length - currentIndex;
    if (len > 0) {
      System.arraycopy(data, start+currentIndex, value, 0, len);
      currentIndex += len;
    }
    updateLast();
    return len;
  }

  public String readString(int width) {
    //System.out.println(getClass().getName()+".readString enter");
    int len = width <= length - currentIndex ? width : length - currentIndex;
    String s;
    if (len > 0) {
      //System.out.println(getClass().getName()+".readString width="+width+", length="+length+", currentIndex="
      //  +currentIndex + ", len="+len);
      s = CharsetHelper.bytesToString(data, start+currentIndex, len, charset);
      currentIndex += len;
    } else {
      s = "";
    }
    updateLast();
    //System.out.println(getClass().getName()+".readString currentIndex="+currentIndex);
    return s;
  }

  public String readString(int maxLength, FlatFileOptions flatFileOptions) 
  throws IOException {
    QuoteSymbolByteChecker quoteSymbolChecker = flatFileOptions.getQuoteSymbolByteChecker();
    DelimiterByteChecker[] fieldDelimiterCheckers = flatFileOptions.getFieldDelimiterByteCheckers();
    boolean omitFinalFieldDelimiter = flatFileOptions.isOmitFinalFieldDelimiter();
    boolean trimLeading = flatFileOptions.isTrimLeading();
    boolean trimTrailing = flatFileOptions.isTrimTrailing();
    ByteTrimmer byteTrimmer = flatFileOptions.getByteTrimmer();

    int maxLen = (maxLength >= 0 && maxLength <= length - currentIndex) ? maxLength : length - currentIndex;
    int end = currentIndex + maxLen;

    boolean inQuotes = false;
    boolean delimiterFound = false;

    ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();

    //  Initialize endFieldPosition
    int endFieldPosition = length;
    if (maxLength >= 0) {
      endFieldPosition = currentIndex+maxLength < length ? currentIndex+maxLength : length;
    }

    //  Initialize leadingCount and leftLimitTrailingSpace
    int leadingCount = 0;
    if (trimLeading) {
      leadingCount = byteTrimmer.countLeadingWhitespace(data, start+currentIndex, endFieldPosition-currentIndex);
    }
    int leftLimitTrailingSpace = leadingCount;

    while (!delimiterFound && currentIndex < end) {
      int n = 0;
      if (inQuotes) {
        n = quoteSymbolChecker.foundEscapedQuoteSymbol(data, start+currentIndex, end-currentIndex);
      }
      if (n > 0) {
        currentIndex += (n - quoteSymbolChecker.length());
        byteArrayBuilder.append(data, start+currentIndex, quoteSymbolChecker.length());
        currentIndex += quoteSymbolChecker.length();
      } else {
        n = quoteSymbolChecker.foundQuoteSymbol(data, start+currentIndex, end-currentIndex);
        if (n > 0) {
          //System.out.println("readQuotedField-foundQuoteSymbol " + new String(data,start+currentIndex,length-currentIndex));
          inQuotes = !inQuotes;
          currentIndex += n;
        } else if (inQuotes) {
          byteArrayBuilder.append(data[start+currentIndex]);
          //System.out.println("Appending " + new String(data,start+currentIndex,1) + ", buflen=" + byteArrayBuilder.length());
          ++currentIndex;
          leftLimitTrailingSpace = byteArrayBuilder.length();
        } else {
          for (int i = 0; !delimiterFound && i < fieldDelimiterCheckers.length; ++i) {
            int delimiterLength = fieldDelimiterCheckers[i].foundEndDelimiter(data, start+currentIndex, end-currentIndex);
            if (delimiterLength > 0) {
              currentIndex += delimiterLength;
              delimiterFound = true;
            }
          }
          if (!delimiterFound) {
            byteArrayBuilder.append(data[start+currentIndex]);
            //System.out.println("Appending " + new String(data,start+currentIndex,1) + ", buflen=" + byteArrayBuilder.length());
            ++currentIndex;
          }
        }
      }
    }
    String value = null;
    if (delimiterFound || omitFinalFieldDelimiter) {
      value = "";
      int len = byteArrayBuilder.length() - leadingCount;
      int trailingCount = 0;
      if (trimTrailing) {
        trailingCount = byteTrimmer.countTrailingWhitespace(byteArrayBuilder.buffer(), leftLimitTrailingSpace, 
                                                           byteArrayBuilder.length()-leftLimitTrailingSpace);
        //System.out.println(getClass().getName() + ".readField "
        //  + "leftLimitTrailingSpace="+leftLimitTrailingSpace 
        //  + ", buflen=" + byteArrayBuilder.length()
        //  + ", trailingCount=" + trailingCount
        //  + ", leadingCount=" + leadingCount);
      }
      if (len-trailingCount > 0) {
        value = CharsetHelper.bytesToString(byteArrayBuilder.buffer(), leadingCount, len-trailingCount, charset);
        //System.out.println(getClass().getName() + ".readField "
        // + ".readField buffer=" + new String(byteArrayBuilder.buffer()) + "." 
        // + ", value=" + value + "." + " leadingCount = " + leadingCount + ", len = " + len + ", trailingCount= " + trailingCount);
        //System.out.println("value = " + value);
        //value = StringHelper.trim(value, false, trimTrailing); 
      }
    }
    updateLast();
    return value;
  }

  public String[] readStringArray(int maxLength, FlatFileOptions flatFileOptions) 
  throws IOException {
    QuoteSymbolByteChecker quoteSymbolChecker = flatFileOptions.getQuoteSymbolByteChecker();
    DelimiterByteChecker[] fieldDelimiterCheckers = flatFileOptions.getFieldDelimiterByteCheckers();
    DelimiterByteChecker[] subfieldDelimiterCheckers = flatFileOptions.getSubfieldDelimiterByteCheckers();
    boolean omitFinalFieldDelimiter = flatFileOptions.isOmitFinalFieldDelimiter();
    boolean trimLeading = flatFileOptions.isTrimLeading();
    boolean trimTrailing = flatFileOptions.isTrimTrailing();
    ByteTrimmer byteTrimmer = flatFileOptions.getByteTrimmer();

    int maxLen = (maxLength >= 0 && maxLength <= length - currentIndex) ? maxLength : length - currentIndex;
    int end = currentIndex + maxLen;

    boolean inQuotes = false;
    boolean delimiterFound = false;

    ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();
    ArrayList<String> valueList = new ArrayList<String>();

    //  Initialize endFieldPosition
    int endFieldPosition = length;
    if (maxLength >= 0) {
      endFieldPosition = currentIndex+maxLength < length ? currentIndex+maxLength : length;
    }

    //  Initialize leadingCount and leftLimitTrailingSpace
    int leadingCount = 0;
    if (trimLeading) {
      leadingCount = byteTrimmer.countLeadingWhitespace(data, start+currentIndex, endFieldPosition-currentIndex);
    }
    int leftLimitTrailingSpace = leadingCount;

    while (!delimiterFound && currentIndex < end) {
      int n = 0;
      if (inQuotes) {
        n = quoteSymbolChecker.foundEscapedQuoteSymbol(data, start+currentIndex, end-currentIndex);
      }
      if (n > 0) {
        currentIndex += (n-quoteSymbolChecker.length());
        byteArrayBuilder.append(data, start+currentIndex, quoteSymbolChecker.length());
        currentIndex += quoteSymbolChecker.length();
      } else {
        n = quoteSymbolChecker.foundQuoteSymbol(data, start+currentIndex, end-currentIndex);
        if (n > 0) {
          //System.out.println("readQuotedField-foundQuoteSymbol " + new String(data,start+currentIndex,length-currentIndex));
          inQuotes = !inQuotes;
          currentIndex += n;
        } else if (inQuotes) {
          byteArrayBuilder.append(data[start+currentIndex]);
          //System.out.println("Appending " + new String(data,start+currentIndex,1) + ", buflen=" + byteArrayBuilder.length());
          ++currentIndex;
          leftLimitTrailingSpace = byteArrayBuilder.length();
        } else {
          for (int i = 0; !delimiterFound && i < fieldDelimiterCheckers.length; ++i) {
            int delimiterLength = fieldDelimiterCheckers[i].foundEndDelimiter(data, start+currentIndex, end-currentIndex);
            if (delimiterLength > 0) {
              currentIndex += delimiterLength;
              delimiterFound = true;
            }
          }
          if (!delimiterFound) {
            boolean subfieldDone = false;
            for (int i = 0; !subfieldDone && i < subfieldDelimiterCheckers.length; ++i) {
              int delimiterLength = subfieldDelimiterCheckers[i].foundEndDelimiter(data, start+currentIndex, end-currentIndex);
              if (delimiterLength > 0) {
                //String s;
                //if (byteArrayBuilder.length() > 0) {
                //  s = CharsetHelper.bytesToString(byteArrayBuilder.buffer(), byteArrayBuilder.start(),
                //        byteArrayBuilder.length(), charset);
                //} else {
                //  s = "";
                //}
                if (byteArrayBuilder.length() > 0) {
                  //System.out.println(getClass().getName()+" addingValue");
                  extractValue(byteTrimmer,trimTrailing,leadingCount,leftLimitTrailingSpace,byteArrayBuilder,valueList);
                } else {
                  // Adjacent delimiters, add empty value 
                  //System.out.println(getClass().getName()+" adding empty value");
                  valueList.add("");
                }
                byteArrayBuilder.clear();
                currentIndex += delimiterLength;
                if (trimLeading) {
                  leadingCount = byteTrimmer.countLeadingWhitespace(data, start+currentIndex, 
                                                                   endFieldPosition-currentIndex);
                }
                leftLimitTrailingSpace = leadingCount;
                subfieldDone = true;
              }
            }
            if (!subfieldDone) {
              byteArrayBuilder.append(data[start+currentIndex]);
              ++currentIndex;
            }
          }
        }
      }
    }

    String[] values = null;
    if (delimiterFound || omitFinalFieldDelimiter) {
      if (byteArrayBuilder.length() > 0) {
        //String s = CharsetHelper.bytesToString(byteArrayBuilder.buffer(), byteArrayBuilder.start(),
        //             byteArrayBuilder.length(), charset );
        //valueList.add(s);
        extractValue(byteTrimmer,trimTrailing,leadingCount,leftLimitTrailingSpace,byteArrayBuilder,valueList);
      }
      values = new String[valueList.size()];
      values = (String[])valueList.toArray(values); 
    }
    //System.out.println(getClass().getName()+" value count="+values.length);
    updateLast();
    return values;
  }

  private void extractValue(ByteTrimmer byteTrimmer, boolean trimTrailing, int leadingCount, 
                            int leftLimitTrailingSpace, ByteArrayBuilder byteArrayBuilder, List<String> valueList) {
    String value="";
    int len = byteArrayBuilder.length() - leadingCount;
    int trailingCount = 0;
    if (trimTrailing) {
      trailingCount = byteTrimmer.countTrailingWhitespace(byteArrayBuilder.buffer(), leftLimitTrailingSpace, 
                                                         byteArrayBuilder.length()-leftLimitTrailingSpace);
      //System.out.println("leftLimitTrailingSpace="+leftLimitTrailingSpace + ", buflen=" + byteArrayBuilder.length()
      //  + ", trailingCount=" + trailingCount);
    }
    if (len-trailingCount > 0) {
      value = CharsetHelper.bytesToString(byteArrayBuilder.buffer(), leadingCount, len-trailingCount, charset);
      //System.out.println("value = " + value);
      //value = StringHelper.trim(value, false, trimTrailing); 
    }
    valueList.add(value);
  }

  public int getPosition() {
    return currentIndex;
  }

  public int getLast() {
    return endIndex;
  }

  public void setPosition(int index) {
    //System.out.println(getClass()+".setPosition index="+index);
    if (index >= 0) {
      if (index <= length) {
        currentIndex = index;
      } else {
        currentIndex = length;
      }
      updateLast();
    }
  }

  public void updateLast() {
    if (endIndex < currentIndex) {
      endIndex = currentIndex;
    }
  }

  public Charset getCharset() {
    return charset;
  }

  public void wipe() throws IOException {
  }

  public void readRepeatingGroup(int count, 
                                 FlatFileOptions flatFileOptions,
                                 List<RecordInput> children) {

    //System.out.println(getClass().getName()+".readRepeatingGroup start="+start()+",length="+length() );
    try {
      QuoteSymbolByteChecker quoteSymbolChecker = flatFileOptions.getQuoteSymbolByteChecker();
      DelimiterByteChecker[] segmentDelimiterCheckers = flatFileOptions.getSegmentDelimiterByteCheckers();
      DelimiterByteChecker[] repeatDelimiterCheckers = flatFileOptions.getRepeatDelimiterByteCheckers();
      ByteTrimmer byteTrimmer = flatFileOptions.getByteTrimmer();
      boolean omitFinalRepeatDelimiter = flatFileOptions.isOmitFinalRepeatDelimiter();

      ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();
      boolean inQuotes = false;
      boolean delimiterFound = false;
      boolean startDelimiterFound = false;
      boolean endDelimiterFound = false;
      boolean startRepeatDelimiterFound = false;

      int counter = 0;
      while (!delimiterFound && currentIndex < length && counter < count) {
        //System.out.println(getClass().getName()+".readRepeatingGroup inQuotes=" + inQuotes +", start="+start()+",length="+length());
        int n = 0;
        if (inQuotes) {
          n = quoteSymbolChecker.foundEscapedQuoteSymbol(data, start+currentIndex, length-currentIndex);
        }
        if (n > 0) {
          int len = n;
          byteArrayBuilder.append(data, start+currentIndex, len);
          currentIndex += len;
        } else {
          n = quoteSymbolChecker.foundQuoteSymbol(data, start+currentIndex, length-currentIndex);
          //System.out.println(getClass().getName()+".readRepeatingGroup 30 " + "start="+start+",currentIndex="+currentIndex+",length="+length);
          //System.out.println(getClass().getName()+".readRepeatingGroup 30 " + new String(data,start+currentIndex,length-currentIndex) + n);
          if (n > 0) {
            inQuotes = !inQuotes;
            byteArrayBuilder.append(data, start+currentIndex, n);
            currentIndex += n;
            // Need to leave quotes in here, for field reader
          } else if (inQuotes) {
            byteArrayBuilder.append(data[start+currentIndex]);
            //System.out.println(getClass().getName()+".readRepeatingGroup " + "In quotes, buffering");
            ++currentIndex;
          } else {
            for (int j = 0; !startDelimiterFound && j < segmentDelimiterCheckers.length; ++j) {
              int delimiterLength = segmentDelimiterCheckers[j].testStart(data, start+currentIndex, length-currentIndex);
              if (delimiterLength > 0) {
                currentIndex += delimiterLength;
                startDelimiterFound = true;
                byteArrayBuilder.clear();
                readBracketedRepeatingGroup(quoteSymbolChecker, 
                                            segmentDelimiterCheckers[j],
                                            repeatDelimiterCheckers, 
                                            count, byteTrimmer, 
                                            children);
                delimiterFound = true;
                //System.out.println(getClass().getName()+".readRepeatingGroup found segment delimiter counter="+counter+", count=" +count);
              }
            }
            if (!startDelimiterFound) {
              for (int j = 0; !delimiterFound && j < segmentDelimiterCheckers.length; ++j) {
                int delimiterLength = segmentDelimiterCheckers[j].foundEndDelimiter(data, start+currentIndex, length-currentIndex);
                if (delimiterLength > 0) {
                  currentIndex += delimiterLength;
                  delimiterFound = true;
                  //System.out.println(getClass().getName()+".readRepeatingGroup found segment delimiter counter="+counter+", count=" +count);
                }
              }
            }
            if (!delimiterFound) {
              boolean repeatDone = false;
              for (int j = 0; !repeatDone && j < repeatDelimiterCheckers.length; ++j) {
                DelimiterByteChecker repeatDelimiter = repeatDelimiterCheckers[j];
                int repeatDelimiterLength = repeatDelimiter.testStart(data, start+currentIndex, length-currentIndex);
                if (repeatDelimiterLength > 0) {
                  startRepeatDelimiterFound = true;
                  if (byteArrayBuilder.length() > 0 && !byteTrimmer.isAllWhitespace(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length())) {
                    //System.out.println(getClass().getName()+".readRepeatingGroup "+new String(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length()));
                    //System.out.println (getClass().getName()+".readRepeatingGroup before readRecord 10");
                    RecordInput recordInput = new ByteRecordInput(byteArrayBuilder.toByteArray(),
                                                                           charset);
                    children.add(recordInput);
                    //System.out.println (getClass().getName()+".readRepeatingGroup after readRecord, before clear 10");
                    byteArrayBuilder.clear();
                    //System.out.println (getClass().getName()+".readRepeatingGroup after readRecord 10");
                  }
                  currentIndex += repeatDelimiterLength;
                  readGroup(quoteSymbolChecker,repeatDelimiter,byteTrimmer,children,byteArrayBuilder);
                  repeatDone = true;
                } else {
                  repeatDelimiterLength = repeatDelimiter.foundEndDelimiter(data, start+currentIndex, length-currentIndex);
                  if (repeatDelimiterLength > 0) {
                    //System.out.println(getClass().getName()+".readRepeatingGroup found repeat delimiter counter="+counter+", count=" +count);
                    currentIndex += repeatDelimiterLength;
                    repeatDone = true;
                    ++counter;                         
                    //if (byteArrayBuilder.length() > 0 && !byteTrimmer.isAllWhitespace(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length())) {
                    //System.out.println(getClass().getName()+".readRepeatingGroup " + new String(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length()));
                    //System.out.println (getClass().getName()+".readRepeatingGroup before readRecord 50");
                    RecordInput recordInput = new ByteRecordInput(byteArrayBuilder.toByteArray(),
                                                                           charset);
                    children.add(recordInput);
                    //System.out.println (getClass().getName()+".readRepeatingGroup after readRecord, before clear 50");
                    byteArrayBuilder.clear();
                    //System.out.println (getClass().getName()+".readRepeatingGroup after readRecord 50");
                    //}
                  }
                }
              }
              if (!repeatDone) {
                byteArrayBuilder.append(data[start+currentIndex]);
                ++currentIndex;
              }
            }
          }
        }
      }

      //if (byteArrayBuilder.length() > 0 && !byteTrimmer.isAllWhitespace(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length())) {
      //System.out.println(getClass().getName()+".readRepeatingGroup finishing with byteArrayBuilder counter="+counter+", count=" +count);
      //System.out.println (getClass().getName()+".readRepeatingGroup before readRecord 100");
      //System.out.println("omitFinalRepeatDelimiter="+omitFinalRepeatDelimiter + ", startDelimiterFound="+startDelimiterFound);
      if (omitFinalRepeatDelimiter && !startRepeatDelimiterFound && byteArrayBuilder.length() > 0) {
        RecordInput recordInput = new ByteRecordInput(byteArrayBuilder.toByteArray(),
                                                               charset);
        children.add(recordInput);
        //System.out.println (getClass().getName()+".readRepeatingGroup after readRecord 100");
      }
    } catch (IOException e) {
      throw new ServingXmlException(e.getMessage(), e);
    }
  }

  public void readGroup(QuoteSymbolByteChecker quoteSymbolChecker, 
                        DelimiterByteChecker repeatDelimiter, 
                        ByteTrimmer byteTrimmer, 
                        List<RecordInput> children, 
                        ByteArrayBuilder byteArrayBuilder) 
  throws IOException {
    //System.out.println(getClass().getName()+".readGroup start="+start() + ",length = " + length() );

    boolean inQuotes = false;

    //ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();
    int level = 1;

    while (level > 0 && currentIndex < length) {
      int n = 0;
      if (inQuotes) {
        n = quoteSymbolChecker.foundEscapedQuoteSymbol(data, start+currentIndex, length-currentIndex);
      }
      if (n > 0) {
        int len = n;
        byteArrayBuilder.append(data, start+currentIndex, len);
        currentIndex += len;
      } else {
        n = quoteSymbolChecker.foundQuoteSymbol(data, start+currentIndex, length-currentIndex);
        //System.out.println(getClass().getName()+".readGroup "+new String(data,start+currentIndex,length-currentIndex) + n);
        if (n > 0) {
          inQuotes = !inQuotes;
          byteArrayBuilder.append(data, start+currentIndex, n);
          currentIndex += n;
          // Need to leave quotes in here, for field reader
        } else if (inQuotes) {
          byteArrayBuilder.append(data[start+currentIndex]);
          //System.out.println("In quotes, buffering");
          ++currentIndex;
        } else {
          int repeatDelimiterLength = repeatDelimiter.testStart(data, start+currentIndex, length-currentIndex);
          if (repeatDelimiterLength > 0) {
            if (level > 0) {
              byteArrayBuilder.append(data,currentIndex,repeatDelimiterLength);
            }
            //System.out.println("found start level="+level);
            currentIndex += repeatDelimiterLength;
            ++level;
          } else {
            repeatDelimiterLength = repeatDelimiter.foundEndDelimiter(data, start+currentIndex, length-currentIndex);
            if (repeatDelimiterLength > 0) {
              //System.out.println("found end level="+level);
              --level;
              if (level == 0) {
                //System.out.println (getClass().getName()+".readGroup before readRecord");
                if (byteArrayBuilder.length() > 0) {
                  RecordInput recordInput = new ByteRecordInput(byteArrayBuilder.toByteArray(),
                                                                         charset);
                  children.add(recordInput);
                  //System.out.println (getClass().getName()+".readGroup after readRecord, before clear");
                  byteArrayBuilder.clear();
                } 
                //System.out.println (getClass().getName()+".readGroup after readRecord");
              } else {
                byteArrayBuilder.append(data,currentIndex,repeatDelimiterLength);
              }
              currentIndex += repeatDelimiterLength;
            } else if (currentIndex < length) {
              byteArrayBuilder.append(data[start+currentIndex]);
              ++currentIndex;
            }
          }
        }
      }
    }
    //System.out.println (getClass().getName()+".readGroup leave");
    //System.out.println("readGroup leave");
  }

  public void readBracketedRepeatingGroup(QuoteSymbolByteChecker quoteSymbolChecker, 
                                          DelimiterByteChecker segmentDelimiter,
                                          DelimiterByteChecker[] repeatDelimiterCheckers, 
                                          int count, ByteTrimmer byteTrimmer, 
                                          List<RecordInput> children) 
  throws IOException {

    ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();
    int counter = 0;
    boolean inQuotes = false;

    //ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();
    int level = 1;

    //System.out.println (getClass().getName()+".readGroup enter");
    while (level > 0 && currentIndex < length && counter < count) {
      int n = 0;
      if (inQuotes) {
        n = quoteSymbolChecker.foundEscapedQuoteSymbol(data, start+currentIndex, length-currentIndex);
      }
      if (n > 0) {
        int len = n;
        byteArrayBuilder.append(data, start+currentIndex, len);
        currentIndex += len;
      } else {
        n = quoteSymbolChecker.foundQuoteSymbol(data, start+currentIndex, length-currentIndex);
        //System.out.println(new String(data,start+currentIndex,length-currentIndex) + n);
        if (n > 0) {
          inQuotes = !inQuotes;
          byteArrayBuilder.append(data, start+currentIndex, n);
          currentIndex += n;
          // Need to leave quotes in here, for field reader
        } else if (inQuotes) {
          byteArrayBuilder.append(data[start+currentIndex]);
          //System.out.println("In quotes, buffering");
          ++currentIndex;
        } else {
          int delimiterLength = segmentDelimiter.testStart(data, start+currentIndex, length-currentIndex);
          if (delimiterLength > 0) {
            ++level;
          } else {
            delimiterLength = segmentDelimiter.foundEndDelimiter(data, start+currentIndex, length-currentIndex);
            if (delimiterLength > 0) {
              --level;
              if (level == 0) {
                byteArrayBuilder.clear();  //DAP
              } else {
                byteArrayBuilder.append(data,currentIndex,delimiterLength);
              }
              currentIndex += delimiterLength;
            }
          }
          if (level > 0) {
            boolean repeatDone = false;
            for (int j = 0; !repeatDone && j < repeatDelimiterCheckers.length; ++j) {
              DelimiterByteChecker repeatDelimiter = repeatDelimiterCheckers[j];
              int repeatDelimiterLength = repeatDelimiter.testStart(data, start+currentIndex, length-currentIndex);
              if (repeatDelimiterLength > 0) {
                if (byteArrayBuilder.length() > 0 && !byteTrimmer.isAllWhitespace(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length())) {
                  //System.out.println(new String(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length()));
                  //System.out.println("Calling readRecord on " + flatRecordReader.getClass().getName());
                  //System.out.println (getClass().getName()+".readField before readRecord 10");
                  RecordInput recordInput = new ByteRecordInput(byteArrayBuilder.toByteArray(),
                                                                         charset);
                  children.add(recordInput);
                  //System.out.println (getClass().getName()+".readField after readRecord, before clear 10");
                  byteArrayBuilder.clear();
                  //System.out.println (getClass().getName()+".readField after readRecord 10");
                }
                currentIndex += repeatDelimiterLength;
                readGroup(quoteSymbolChecker,repeatDelimiter,byteTrimmer,children,byteArrayBuilder);
                repeatDone = true;
              } else {
                repeatDelimiterLength = repeatDelimiter.foundEndDelimiter(data, start+currentIndex, length-currentIndex);
                if (repeatDelimiterLength > 0) {
                  //System.out.println(getClass().getName()+".readField found repeat delimiter counter="+counter+", count=" +count);
                  currentIndex += repeatDelimiterLength;
                  repeatDone = true;
                  ++counter;                         
                  //if (byteArrayBuilder.length() > 0 && !byteTrimmer.isAllWhitespace(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length())) {
                  //System.out.println(new String(byteArrayBuilder.buffer(), 0, byteArrayBuilder.length()));
                  //System.out.println("Calling readRecord on " + flatRecordReader.getClass().getName());
                  //System.out.println (getClass().getName()+".readField before readRecord 50");
                  RecordInput recordInput = new ByteRecordInput(byteArrayBuilder.toByteArray(),
                                                                         charset);
                  children.add(recordInput);
                  //System.out.println (getClass().getName()+".readField after readRecord, before clear 50");
                  byteArrayBuilder.clear();
                  //System.out.println (getClass().getName()+".readField after readRecord 50");
                  //}
                } else {
                  ++currentIndex; //DAP
                }
              }
            }
            if (!repeatDone && currentIndex < length) {
              byteArrayBuilder.append(data[start+currentIndex]);
              ++currentIndex;
            }
          }
        }
      }
    }
    //System.out.println (getClass().getName()+".readGroup leave");
  }

  public RecordInput readSegment(int segmentLength) {
    byte[] data = new byte[segmentLength];
    int length = readBytes(data);
    RecordInput segmentInput = new ByteRecordInput(data, 0, length, getCharset());
    return segmentInput;
  }

  public RecordInput readSegment(FlatFileOptions flatFileOptions) {

    QuoteSymbolByteChecker quoteSymbolChecker = flatFileOptions.getQuoteSymbolByteChecker();
    DelimiterByteChecker[] segmentDelimiterCheckers = flatFileOptions.getSegmentDelimiterByteCheckers();
    ByteTrimmer byteTrimmer = flatFileOptions.getByteTrimmer();

    try {
      ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();
      boolean inQuotes = false;
      boolean endDelimiterFound = false;

      int counter = 0;
      while (!endDelimiterFound && currentIndex < length) {
        //System.out.println("inQuotes=" + inQuotes);
        int n = 0;
        if (inQuotes) {
          n = quoteSymbolChecker.foundEscapedQuoteSymbol(data, start+currentIndex, length-currentIndex);
        }
        if (n > 0) {
          int len = n;
          byteArrayBuilder.append(data, start+currentIndex, len);
          currentIndex += len;
        } else {
          n = quoteSymbolChecker.foundQuoteSymbol(data, start+currentIndex, length-currentIndex);
          //System.out.println(new String(data,start+currentIndex,length-currentIndex) + n);
          if (n > 0) {
            inQuotes = !inQuotes;
            byteArrayBuilder.append(data, start+currentIndex, n);
            currentIndex += n;
            // Need to leave quotes in here, for field reader
          } else if (inQuotes) {
            byteArrayBuilder.append(data[start+currentIndex]);
            //System.out.println("In quotes, buffering");
            ++currentIndex;
          } else {
            if (!endDelimiterFound) {
              for (int j = 0; !endDelimiterFound && j < segmentDelimiterCheckers.length; ++j) {
                DelimiterByteChecker segmentDelimiter = segmentDelimiterCheckers[j];
                int delimiterLength = segmentDelimiter.testStart(data, start+currentIndex, length-currentIndex);
                if (delimiterLength > 0) {
                  currentIndex += delimiterLength;
                  readToEndOfSegment(segmentDelimiter,byteArrayBuilder);
                  endDelimiterFound = true;
                } else {
                  delimiterLength = segmentDelimiter.foundEndDelimiter(data, start+currentIndex, length-currentIndex);
                  if (delimiterLength > 0) {
                    //System.out.println(getClass().getName()+".readField found repeat delimiter counter="+counter+", count=" +count);
                    currentIndex += delimiterLength;
                    endDelimiterFound = true;
                    ++counter;                         
                  }
                }
              }
              if (!endDelimiterFound) {
                byteArrayBuilder.append(data[start+currentIndex]);
                ++currentIndex;
              }
            }
          }
        }
      }
      RecordInput recordInput = new ByteRecordInput(byteArrayBuilder.toByteArray(),
                                                             charset);
      return recordInput;
    } catch (IOException e) {
      throw new ServingXmlException(e.getMessage(), e);
    }
  }


  public void readToEndOfSegment(DelimiterByteChecker segmentDelimiter, ByteArrayBuilder byteArrayBuilder) 
  throws IOException {

    //ByteArrayBuilder byteArrayBuilder = new ByteArrayBuilder();
    int level = 1;

    //System.out.println (getClass().getName()+".readGroup enter");
    while (level > 0 && currentIndex < length) {
      int delimiterLength = segmentDelimiter.testStart(data, start+currentIndex, length-currentIndex);
      if (delimiterLength > 0) {
        byteArrayBuilder.append(data,start+currentIndex,delimiterLength);
        currentIndex += delimiterLength;
        ++level;
      } else {
        delimiterLength = segmentDelimiter.foundEndDelimiter(data, start+currentIndex, length-currentIndex);
        if (delimiterLength > 0) {
          --level;
          if (level > 0) {
            byteArrayBuilder.append(data,start+currentIndex,delimiterLength);
          }
          currentIndex += delimiterLength;
        } else {
          byteArrayBuilder.append(data[start+currentIndex]);
          ++currentIndex;
        }
      }
    }
    //System.out.println (getClass().getName()+".readGroup leave");
  }


  public RecordInput concatentate(RecordInput ri) {
    RecordInput recordInput = this;
    byte[] rhBytes = ri.toByteArray();
    if (rhBytes.length > 0) {
      int capacity = (length + rhBytes.length)*2;
      byte[] newBuffer = new byte[capacity];
      System.arraycopy(data,start,newBuffer,0,length);
      System.arraycopy(rhBytes,0,newBuffer,length,rhBytes.length);
      recordInput = new ByteRecordInput(newBuffer, 0, length+rhBytes.length, 
                                                 getCharset());
    }
    return recordInput;
  }

  public String toString() {
    try {
      return new String(data,start,length,charset.name());
    } catch (java.io.UnsupportedEncodingException e) {
      throw new ServingXmlException(e.getMessage(),e);
    }
  }
}




