/**
 *  ServingXML
 *  
 *  Copyright (C) 2006  Daniel Parker
 *    daniel.parker@servingxml.com 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License. 
 * 
 **/

package com.servingxml.components.flatfile.recordtype;

import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXTransformerFactory;
import java.nio.charset.Charset;

import com.servingxml.app.AppContext;
import com.servingxml.app.DefaultAppContext;
import com.servingxml.app.DefaultServiceContext;
import com.servingxml.app.Flow;
import com.servingxml.app.FlowImpl;
import com.servingxml.app.ServiceContext;
import com.servingxml.components.flatfile.RecordInput;
import com.servingxml.components.flatfile.options.FlatFileOptionsImpl;
import com.servingxml.components.flatfile.options.Delimiter;
import com.servingxml.components.flatfile.options.RecordDelimiter;
import com.servingxml.components.flatfile.options.FieldDelimiter;
import com.servingxml.components.quotesymbol.QuoteSymbol;
import com.servingxml.components.flatfile.scanner.bytes.ByteRecordInput;
import com.servingxml.components.parameter.DefaultValue;
import com.servingxml.ioc.resources.SimpleIocContainer;
import com.servingxml.util.MutableNameTable;
import com.servingxml.util.Name;
import com.servingxml.util.NameTableImpl;
import com.servingxml.util.QualifiedName;
import com.servingxml.util.record.Record;
import com.servingxml.util.record.RecordBuilder;
import com.servingxml.util.system.Logger;
import com.servingxml.util.system.SystemConfiguration;
import com.servingxml.app.Environment;
import com.servingxml.util.SimpleQnameContext;
import com.servingxml.app.ParameterDescriptor;

import junit.framework.TestCase;
import junit.framework.TestFailure;
import junit.framework.TestResult;
import junit.framework.TestSuite;

/**
 *
 * 
 * @author Daniel A. Parker (daniel.parker@servingxml.com)
 */

public class DelimitedFieldReaderTest extends TestCase {

  private Logger logger = SystemConfiguration.getLogger();
  private AppContext appContext;
  private MutableNameTable nameTable = new NameTableImpl();
  private ServiceContext context;
  private Flow flow;
  private Delimiter[] recordDelimiters = new RecordDelimiter[]{RecordDelimiter.CRLF,RecordDelimiter.LF};
  private int recordDelimiterStart = 0;
  private int recordDelimiterCount = 2;

  public DelimitedFieldReaderTest(String name) {
    super(name);
  }

  protected void setUp() throws Exception {
    SAXTransformerFactory transformerFactory = (SAXTransformerFactory)TransformerFactory.newInstance();
    SimpleIocContainer resources = new SimpleIocContainer(nameTable, transformerFactory);
    appContext = new DefaultAppContext("",resources,logger);
    Environment env = new Environment(ParameterDescriptor.EMPTY_PARAMETER_DESCRIPTOR_ARRAY,new SimpleQnameContext());
    flow = new FlowImpl(env, context, Record.EMPTY, Record.EMPTY);
  }

  public void testTrimWithinQuotes() throws Exception {
    Name recordTypeName = new QualifiedName("book");
    Name fieldName = new QualifiedName("description");
    int maxWidth = -1;
    int offset = -1;

    Delimiter fieldDelimiter = new FieldDelimiter(",");
    Delimiter[] fieldDelimiters = new Delimiter[]{fieldDelimiter};

    FlatFileOptionsImpl flatFileOptions = new FlatFileOptionsImpl(Charset.defaultCharset(),true,true);
    flatFileOptions.setCountPositionsInBytes(true);
    flatFileOptions.setQuoteSymbol(new QuoteSymbol('"',"\"\""));
    flatFileOptions.setFieldDelimiters(fieldDelimiters);

    FlatRecordFieldReader reader = new DelimitedFieldReader(fieldName,offset,
      maxWidth, DefaultValue.EMPTY, flatFileOptions);

    String str = "xx  \" A good book. \"  ";
    byte[] data = str.getBytes();
    RecordBuilder recordBuilder = new RecordBuilder(recordTypeName);
    RecordInput recordInput = new ByteRecordInput(data, 2, data.length-2, Charset.defaultCharset());
    reader.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    Record record = recordBuilder.toRecord();

    //System.out.println("description=" + record.getString(fieldName) + ".");

    String s = record.getString(fieldName);
    assertTrue(" A good book. =|"+s+"|", s.equals(" A good book. "));
  }

  public void testTrimNonQuoted() throws Exception {
    Name recordTypeName = new QualifiedName("book");
    Name fieldName = new QualifiedName("description");
    int maxWidth = -1;
    int offset = -1;

    Delimiter fieldDelimiter = new FieldDelimiter(",");
    Delimiter[] fieldDelimiters = new Delimiter[]{fieldDelimiter};

    FlatFileOptionsImpl flatFileOptions = new FlatFileOptionsImpl(Charset.defaultCharset(),true,true);
    flatFileOptions.setCountPositionsInBytes(true);
    flatFileOptions.setQuote(false);
    flatFileOptions.setFieldDelimiters(fieldDelimiters);

    FlatRecordFieldReader reader = new DelimitedFieldReader(fieldName,offset,      
      maxWidth,DefaultValue.EMPTY,flatFileOptions);

    String str = "xx   A good book.   ";
    byte[] data = str.getBytes();
    RecordBuilder recordBuilder = new RecordBuilder(recordTypeName);
    RecordInput recordInput = new ByteRecordInput(data, 2, data.length-2, Charset.defaultCharset());
    reader.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    Record record = recordBuilder.toRecord();

    //System.out.println("description=" + record.getString(fieldName) + ".");

    String s = record.getString(fieldName);
    assertTrue("A good book.="+s+".", s.equals("A good book."));
  }

  public void testTrimNonQuotedDelimited() throws Exception {
    Name recordTypeName = new QualifiedName("book");
    Name fieldName = new QualifiedName("description");
    int maxWidth = -1;
    int offset = -1;

    Delimiter fieldDelimiter = new FieldDelimiter(",");
    Delimiter[] fieldDelimiters = new Delimiter[]{fieldDelimiter};

    FlatFileOptionsImpl flatFileOptions = new FlatFileOptionsImpl(Charset.defaultCharset(),true,true);
    flatFileOptions.setCountPositionsInBytes(true);
    flatFileOptions.setQuote(false);
    flatFileOptions.setFieldDelimiters(fieldDelimiters);

    FlatRecordFieldReader reader = new DelimitedFieldReader(fieldName,offset,
      maxWidth, DefaultValue.EMPTY, flatFileOptions);

    String str = "xx   A good book.   , A bad book";
    byte[] data = str.getBytes();
    RecordBuilder recordBuilder = new RecordBuilder(recordTypeName);
    RecordInput recordInput = new ByteRecordInput(data, 2, data.length-2, Charset.defaultCharset());
    reader.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    Record record = recordBuilder.toRecord();

    //System.out.println("description=" + record.getString(fieldName) + ".");

    String s = record.getString(fieldName);
    assertTrue("A good book.="+s+".", s.equals("A good book."));
  }

  public void testTrimWithinQuotesMaxWidth11() throws Exception {
    Name recordTypeName = new QualifiedName("book");
    Name fieldName = new QualifiedName("description");
    int maxWidth = 11;
    int offset = -1;

    Delimiter fieldDelimiter = new FieldDelimiter(",");
    Delimiter[] fieldDelimiters = new Delimiter[]{fieldDelimiter};

    FlatFileOptionsImpl flatFileOptions = new FlatFileOptionsImpl(Charset.defaultCharset(),true,true);
    flatFileOptions.setCountPositionsInBytes(true);
    flatFileOptions.setQuoteSymbol(new QuoteSymbol('"',"\"\""));
    flatFileOptions.setFieldDelimiters(fieldDelimiters);

    FlatRecordFieldReader reader = new DelimitedFieldReader(fieldName,offset,
      maxWidth, DefaultValue.EMPTY, flatFileOptions);

    String str = "xx  \" A good book. \"  ";
    byte[] data = str.getBytes();
    RecordBuilder recordBuilder = new RecordBuilder(recordTypeName);
    RecordInput recordInput = new ByteRecordInput(data, 2, data.length-2, Charset.defaultCharset());
    reader.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    Record record = recordBuilder.toRecord();

    //System.out.println("description=" + record.getString(fieldName) + ".");

    String s = record.getString(fieldName);
    assertTrue(" A good="+s+".", s.equals(" A good "));
  }

  public void testTrimWithinQuotesMaxWidth10() throws Exception {
    Name recordTypeName = new QualifiedName("book");
    Name fieldName = new QualifiedName("description");
    int maxWidth = 10;
    int offset = -1;

    Delimiter fieldDelimiter = new FieldDelimiter(",");
    Delimiter[] fieldDelimiters = new Delimiter[]{fieldDelimiter};

    FlatFileOptionsImpl flatFileOptions = new FlatFileOptionsImpl(Charset.defaultCharset(),true,true);
    flatFileOptions.setCountPositionsInBytes(true);
    flatFileOptions.setQuoteSymbol(new QuoteSymbol('"',"\"\""));
    flatFileOptions.setFieldDelimiters(fieldDelimiters);

    FlatRecordFieldReader reader = new DelimitedFieldReader(fieldName,offset,
      maxWidth,DefaultValue.EMPTY,flatFileOptions);

    String str = "  \" A good book. \"  ";
    byte[] data = str.getBytes();
    RecordBuilder recordBuilder = new RecordBuilder(recordTypeName);
    RecordInput recordInput = new ByteRecordInput(data, 0, data.length, Charset.defaultCharset());
    reader.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    Record record = recordBuilder.toRecord();

    //System.out.println("description=" + record.getString(fieldName) + ".");

    String s = record.getString(fieldName);
    assertTrue(" A good="+s+".", s.equals(" A good"));
  }

  public void testQuotedComma() throws Exception {
    Name recordTypeName = new QualifiedName("INDEX_VALUE");
    Name fieldName1 = new QualifiedName("defaultSource");
    Name fieldName2 = new QualifiedName("sources");
    int maxWidth = -1;
    int offset = -1;

    Delimiter fieldDelimiter = new FieldDelimiter(",");
    Delimiter[] fieldDelimiters = new Delimiter[]{fieldDelimiter};

    FlatFileOptionsImpl flatFileOptions = new FlatFileOptionsImpl(Charset.defaultCharset(),true,true);
    flatFileOptions.setCountPositionsInBytes(true);
    flatFileOptions.setQuoteSymbol(new QuoteSymbol('"',"\"\""));
    flatFileOptions.setFieldDelimiters(fieldDelimiters);

    FlatRecordFieldReader reader1 = new DelimitedFieldReader(fieldName1,offset,
      maxWidth, DefaultValue.EMPTY, flatFileOptions);

    FlatRecordFieldReader reader2 = new DelimitedFieldReader(fieldName2,offset,
      maxWidth, DefaultValue.EMPTY, flatFileOptions);

    String str = "BBA,\"BBA,LIBO\"";
    byte[] data = str.getBytes();
    RecordBuilder recordBuilder = new RecordBuilder(recordTypeName);
    RecordInput recordInput = new ByteRecordInput(data, 0, data.length, Charset.defaultCharset());
    reader1.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    reader2.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    Record record = recordBuilder.toRecord();
    //System.out.println("record="+record.toXmlString(context));

    //System.out.println("description=" + record.getString(fieldName) + ".");

    String expected1="BBA";
    String expected2="BBA,LIBO";

    String value1 = record.getString(fieldName1);
    String value2 = record.getString(fieldName2);
    assertTrue(value1+"="+expected1, value1.equals(expected1));
    assertTrue(value2+"="+expected2, value2.equals(expected2));
  }

  public void testEmptyQuoted() throws Exception {
    Name recordTypeName = new QualifiedName("INDEX_VALUE");
    Name fieldName1 = new QualifiedName("a");
    Name fieldName2 = new QualifiedName("b");
    int maxWidth = -1;
    int offset = -1;

    Delimiter fieldDelimiter = new FieldDelimiter("~~");
    Delimiter[] fieldDelimiters = new Delimiter[]{fieldDelimiter};

    FlatFileOptionsImpl flatFileOptions = new FlatFileOptionsImpl(Charset.defaultCharset(),true,true);
    flatFileOptions.setCountPositionsInBytes(true);
    flatFileOptions.setQuote(false);
    flatFileOptions.setFieldDelimiters(fieldDelimiters);
    flatFileOptions.setOmitFinalFieldDelimiter(true);

    FlatRecordFieldReader reader1 = new DelimitedFieldReader(fieldName1,offset,
      maxWidth, DefaultValue.EMPTY, flatFileOptions);

    FlatRecordFieldReader reader2 = new DelimitedFieldReader(fieldName2,offset,
      maxWidth, DefaultValue.EMPTY, flatFileOptions);

    String str = "\"\"~~\"";
    byte[] data = str.getBytes();
    RecordBuilder recordBuilder = new RecordBuilder(recordTypeName);
    RecordInput recordInput = new ByteRecordInput(data, 0, data.length, Charset.defaultCharset());
    reader1.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    reader2.readField(context, flow, recordInput, recordDelimiters,  recordDelimiterStart, recordDelimiterCount, maxRecordWidth, recordBuilder);
    Record record = recordBuilder.toRecord();
    //System.out.println("record="+record.toXmlString(context));

    //System.out.println("description=" + record.getString(fieldName) + ".");

    String expected1="\"\"";
    String expected2="\"";

    String value1 = record.getString(fieldName1);
    String value2 = record.getString(fieldName2);
    assertTrue(value1+"="+expected1, value1.equals(expected1));
    assertTrue(value2+"="+expected2, value2.equals(expected2));
  }
}                    

