<?php

// forms.php -- basic standard functions dealing with html forms
//
//  $Id: forms.php,v 1.1 2006/10/03 08:33:55 fstuurman Exp $
//
//  Copyright (C) 2004 Peter Fokker, <peter@berestijn.nl>
//
//  This file is part of Janitor.
//
//  Janitor is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  Janitor is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

//  Based on code in MySQL/PHP Database Applications, Jay Greenspan & Brad Bulger, ISBN 0-7645-3537-4

/* This file contains some useful standard functions dealing with forms.
 *
 * string start_form ($action, $attributes)
 * string end_form(void)
 * string text_field ($name, $value, $size, $maximum_length)
 * string number_field ($name, $value, $size, $maximum_length)
 * string textarea_field ($name, $value, $cols, $rows, $wrap_mode)
 * string password_field ($name, $value, $size, $maximum_length)
 * string hidden_field ($name, $value)
 * string file_field ($name)
 * string submit_field ($name, $value)
 * string image_field ($name, $src, $value)
 * string reset_field ($name, $value)
 * string checkbox_field ($name, $value, $label, $match)
 * string radio_field ($name, $value, $label, $match)
 * string select_field ($name, $items, $default_value)
 * string db_select_field ($name, $table, $value_field, $label_field, $sort_field, $match, $where)
 * string db_radio_field ($name, $table, $value_field, $label_field, $sort_field, $match, $where)
 *
 */

function start_form($action = "", $atts = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML <form> tag. If the first argument
	 * is empty, the value of the global Apache variable SCRIPT_NAME
	 * is used for the 'action' attribute of the <form> tag. Other
	 * attributes for the form can be specified in the optional second
	 * argument; the default method of the form is "post".
	 *
	 * The behavior of this function on servers other than Apache is
	 * not known. It's likely that it will work, as SCRIPT_NAME is
	 * part of the CGI 1.1 specification.
	 */
	global $SCRIPT_NAME;

	if (empty ($action))
	{
		$action = $SCRIPT_NAME;
	}

	$attlist = get_attlist($atts, array (
		"method" => "post"
	));
	$output = "\n<form action=\"$action\" $attlist>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* start_form() */

function end_form()
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML </form> tag.
	 */
	$output = "\n</form>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* end_form() */

function text_field($name = "", $value = "", $size = 10, $maxlen = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML text input field. The default size
	 * of the field is 10. A value and maximum data length for the field
	 * may be supplied.
	 */
	$maxatt = empty ($maxlen) ? "" : " maxlength=\"$maxlen\"";
	$output = "<input type=\"text\" name=\"$name\" value=\"$value\" size=\"$size\"$maxatt>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* text_field() */

function number_field($name = "", $value = "", $size = 10, $maxlen = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML text input field. The default size
	 * of the field is 10. A value and maximum data length for the field
	 * may be supplied. The field is right-aligned. (2001-07-24/HPA)
	 */
	$maxatt = empty ($maxlen) ? "" : " maxlength=\"$maxlen\"";
	$output = "<input type=\"text\" name=\"$name\" value=\"$value\" size=\"$size\" align=\"right\"$maxatt>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* number_field() */

function textarea_field($name = "", $value = "", $cols = 50, $rows = 10, $wrap = "soft")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML textarea field. The default size is
	 * 50 columns and 10 rows, and the default wrap mode is 'soft', which means 
	 * no hard newline characters will be inserted after line breaks in what
	 * the user types into the field. The alternative wrap mode is 'hard',
	 * which means that hard newlines will be inserted.
	 */
	$output = "<textarea name=\"$name\" cols=\"$cols\" rows=\"$rows\" wrap=\"$wrap\">$value</textarea>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* textarea_field() */

function password_field($name = "", $value = "", $size = 10, $maxlen = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML password field. This is like a text field,
	 * but the value of the field is obscured (only stars or bullets are visible
	 * for each character).  The default size of the field is 10.  A starting
	 * value and maximum data length may be supplied.
	 */
	$maxatt = empty ($maxlen) ? "" : " maxlength=\"$maxlen\"";
	$output = "<input type=\"password\" name=\"$name\" value=\"$value\" size=\"$size\"$maxatt>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* pasword_field() */

function hidden_field($name = "", $value = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML hidden field. A value may be supplied.
	 */
	$output = "<input type=\"hidden\" name=\"$name\" value=\"$value\">\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* hidden_field() */

function file_field($name = "", $value = "", $size = 10, $maxlen = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML file field. These are used to specify
	 * files on the user's local hard drive, typically for uploading as
	 * part of the form. (See http://www.zend.com/manual/features.file-upload.php
	 * for more information about this subject.)
	 */
	$maxatt = empty ($maxlen) ? "" : " maxlength=\"$maxlen\"";
	$output = "<input type=\"file\" name=\"$name\" value=\"$value\" size=\"$size\"$maxatt>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* file_field() */

function submit_field($name = "", $value = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML submit field. The value of the field
	 * will be the string displayed by the button displayed by the user's
	 * browser. The default value is "Submit".
	 */
	if (empty ($value))
	{
		$value = "Submit";
	}

	$output = "<input type=\"submit\" name=\"$name\" value=\"$value\">\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* submit_field()*/

function image_field($name = "", $src = "", $value = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML image field. An image field works
	 * likes a submit field, except that the image specified by the URL
	 * given in the second argument is displayed instead of a button.
	 */
	if (empty ($value))
	{
		$value = $name;
	}

	$output = "<input type=\"image\" name=\"$name\" value=\"$value\" src=\"$src\">\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* image_field() */

function reset_field($name = "reset", $value = "Reset")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML reset field. A reset field returns
	 * the current form to its original state.
	
		$output = "<input type=\"reset\" name=\"$name\" value=\"$value\">\n";
		return $output;
	/*-----------------------------------------------------------------------------*/
} /* reset_field() */

function checkbox_field($name = "", $value = "", $label = "", $match = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML checkbox field. The optional third argument
	 * will be included immediately after the checkbox field, and the pair
	 * is included inside a HTML <nobr> tag - meaning that they will be
	 * displayed together on the same line.  If the value of the
	 * second or third argument matches that of the fourth argument,
	 * the checkbox will be 'checked' (i.e., flipped on).
	 */
	$checked = ($value == $match || $label == $match) ? " checked" : "";
	$output = "<nobr><input type=\"checkbox\" name=\"$name\" value=\"$value\"$checked> $label</nobr>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* checkbox_field() */

function radio_field($name = "", $value = "", $label = "", $match = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML radio button field. The optional third 
	 * argument will be included immediately after the radio button, and the pair
	 * is included inside a HTML <nobr> tag - meaning that they will be
	 * displayed together on the same line.  If the value of the
	 * second or third argument matches that of the fourth argument,
	 * the radio button will be 'checked' (i.e., flipped on).
	 */
	$checked = ($value == $match || $label == $match) ? " checked" : "";
	$output = "<nobr><input type=\"radio\" name=\"$name\" value=\"$value\"$checked> $label</nobr>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* radio_field() */

function select_field($name = "", $array = "", $value = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML select field (a popup field).
	 * If the optional second argument is an array, each key in the array
	 * will be set to the value of an option of the select field, and
	 * the corresponding value from the array will be the displayed string
	 * for that option. If the key or the value from the array matches
	 * the optional third argument, that option will be designated as the default
	 * value of the select field.
	 */
	$output = "<select name=\"$name\">\n";
	if (is_array($array))
	{
		while (list ($avalue, $alabel) = each($array))
		{
			$selected = ($avalue == $value || $alabel == $value) ? " selected" : "";
			$output .= "<option value=\"$avalue\"$selected>$alabel</option>\n";
		}
	}
	$output .= "</select>\n";
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* select_field() */

function db_select_field($name = "", $table = "", $value_field = "", $label_field = "", $sort_field = "", $match = "", $where = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns an HTML select field (popup field), based
	 * on the values in the MySQL database table specified by the second argument,
	 * as returned by the db_values_array() function (defined in db.php).
	 */
	$values = db_values_array($table, $value_field, $label_field, $sort_field, $where);
	$output = select_field($name, $values, $match);
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* db_select() */

function db_radio_field($name = "", $table = "", $value_field = "", $label_field = "", $sort_field = "", $match = "", $where = "")
{
	/*-----------------------------------------------------------------------------
	 * This function returns a list of HTML radio button fields, separated
	 * by a non-breaking space HTML entity (&nbsp;) and a newline, based
	 * on the values in the MySQL database table named by the second
	 * argument, as returned by the db_values_array() function (defined in db.php).
	 */
	$values = db_values_array($table, $value_field, $label_field, $sort_field, $where);
	$output = "";
	while (list ($value, $label) = each($values))
	{
		$output .= radio_field($name, $value, $label, $match) . "&nbsp;\n";
	}
	return $output;
	/*-----------------------------------------------------------------------------*/
} /* db_radio_field() */

/*----- forms.php -----*/
?>
