package jp.gr.java_conf.jaba2.awt;

import java.awt.LayoutManager2;
import java.awt.Component;
import java.awt.Container;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Insets;
import jp.gr.java_conf.jaba2.util.Debug;
import jp.gr.java_conf.jaba2.util.D2Array;

/**
 * TableLayout
 *
 * @since   Apr. 19, 1998
 * @version Nov. 20, 1998
 * @author ASAMI, Tomoharu (tasami@ibm.net)
 */
public class TableLayout extends AbstractLayoutManager {
    /**
     * policy
     *
     * HBASE
     *   HBASE_LEFT
     *   HBASE_CENTER
     *   HBASE_RIGHT
     *   HBASE_FILL
     * VBASE
     *   VBASE_TOP
     *   VBASE_CENTER
     *   VBASE_BOTTOM
     *   VBASE_FILL
     * HALIGN
     *   HALIGN_LEFT
     *   HALIGN_CENTER
     *   HALIGN_RIGHT
     *   HALIGN_FILL
     *   HALIGN_PREFERRED (use component's x alignment) // XXX
     * VALIGN
     *   VALIGN_TOP
     *   VALIGN_CENTER
     *   VALIGN_BOTTOM
     *   VALIGN_FILL
     *   VALIGN_PREFERRED (use component's y alignment) // XXX
     * WIDTH
     *   WIDTH_PREFERRED
     *   WIDTH_EQUAL
     *   WIDTH_RELATIVE
     *   WIDTH_ABSOLUTE
     * HEIGHT
     *   HEIGHT_PREFERRED
     *   HEIGHT_EQUAL
     *   HEIGHT_RELATIVE
     *   HEIGHT_ABSOLUTE
     * RESIZE
     *   RESIZE_NONE
     *   RESIZE_WIDTH
     *   RESIZE_HEIGHT
     *   RESIZE_BOUND
     */

    // HBASE
    private static final int HBASE_MASK = 0x0000000f;
    public static final int HBASE_DEFAULT = 0x00000000;
    public static final int HBASE_LEFT = 0x00000001;
    public static final int HBASE_CENTER = 0x00000002; // default
    public static final int HBASE_RIGHT = 0x00000003;
    public static final int HBASE_FILL = 0x00000004;
    // VBASE
    private static final int VBASE_MASK = 0x000000f0;
    public static final int VBASE_DEFAULT = 0x00000000;
    public static final int VBASE_TOP = 0x00000010;
    public static final int VBASE_CENTER = 0x00000020; // default
    public static final int VBASE_BOTTOM = 0x00000030;
    public static final int VBASE_FILL = 0x00000040;
    // HALIGN
    private static final int HALIGN_MASK = 0x0000f00;
    public static final int HALIGN_DEFAULT = 0x00000000;
    public static final int HALIGN_LEFT = 0x00000100;
    public static final int HALIGN_CENTER = 0x00000200;
    public static final int HALIGN_RIGHT = 0x00000300;
    public static final int HALIGN_FILL = 0x00000400;
    public static final int HALIGN_PREFERRED = 0x00000500; // default
    // VALIGN
    private static final int VALIGN_MASK = 0x0000f000;
    public static final int VALIGN_DEFAULT = 0x00000000;
    public static final int VALIGN_TOP = 0x00001000;
    public static final int VALIGN_CENTER = 0x00002000;
    public static final int VALIGN_BOTTOM = 0x00003000;
    public static final int VALIGN_FILL = 0x00004000;
    public static final int VALIGN_PREFERRED = 0x00005000; // default
    // WIDTH
    private static final int WIDTH_MASK = 0x000f0000;
    public static final int WIDTH_DEFAULT = 0x00000000;
    public static final int WIDTH_PREFERRED = 0x00010000; // default
    public static final int WIDTH_EQUAL = 0x00020000;
    public static final int WIDTH_FILL = 0x00030000;
    public static final int WIDTH_RELATIVE = 0x00040000; // use attr
    public static final int WIDTH_ABSOLUTE = 0x00050000; // use attr
    // HEIGHT
    private static final int HEIGHT_MASK = 0x00f00000;
    public static final int HEIGHT_DEFAULT = 0x00000000;
    public static final int HEIGHT_PREFERRED = 0x00100000; // default
    public static final int HEIGHT_EQUAL = 0x00200000;
    public static final int HEIGHT_FILL = 0x00300000;
    public static final int HEIGHT_RELATIVE = 0x00400000; // use attr
    public static final int HEIGHT_ABSOLUTE = 0x00500000; // use attr
    // RESIZE
    private static final int RESIZE_MASK = 0x0f000000;
    public static final int RESIZE_DEFAULT = 0x00000000;
    public static final int RESIZE_NONE = 0x01000000;
    public static final int RESIZE_WIDTH = 0x02000000;
    public static final int RESIZE_HEIGHT = 0x03000000;
    public static final int RESIZE_BOUND = 0x04000000; // default

    protected int hbasePolicy_;
    protected int vbasePolicy_;
    protected int halignPolicy_;
    protected int valignPolicy_;
    protected int widthPolicy_;
    protected int heightPolicy_;
    protected int resizePolicy_;
    protected int x_ = 0;
    protected int y_ = 0;
    protected int width_ = 0;
    protected int height_ = 0;
    protected D2Array comps_ = new D2Array(); // D2Array<Component, Component>

    public TableLayout() {
	this(
	    HBASE_DEFAULT |
	    VBASE_DEFAULT |
	    HALIGN_DEFAULT |
	    VALIGN_DEFAULT |
	    WIDTH_DEFAULT |
	    HEIGHT_DEFAULT |
	    RESIZE_DEFAULT
	);
    }

    public TableLayout(int policy) {
	this(policy, 5, 5);
    }

    public TableLayout(int policy, int hgap, int vgap) {
	super(hgap, vgap);
	hbasePolicy_ = policy & HBASE_MASK;
	if (hbasePolicy_ == HBASE_DEFAULT) {
	    hbasePolicy_ = HBASE_CENTER;
	}
	vbasePolicy_ = policy & VBASE_MASK;
	if (vbasePolicy_ == VBASE_DEFAULT) {
	    vbasePolicy_ = VBASE_CENTER;
	}
	halignPolicy_ = policy & HALIGN_MASK;
	if (halignPolicy_ == HALIGN_DEFAULT) {
	    halignPolicy_ = HALIGN_PREFERRED;
	}
	valignPolicy_ = policy & VALIGN_MASK;
	if (valignPolicy_ == VALIGN_DEFAULT) {
	    valignPolicy_ = VALIGN_PREFERRED;
	}
	widthPolicy_ = policy & WIDTH_MASK;
	if (widthPolicy_ == WIDTH_DEFAULT) {
	    widthPolicy_ = WIDTH_PREFERRED;
	}
	heightPolicy_ = policy & HEIGHT_MASK;
	if (heightPolicy_ == HEIGHT_DEFAULT) {
	    heightPolicy_ = HEIGHT_PREFERRED;
	}
	resizePolicy_ = policy & RESIZE_MASK;
	if (resizePolicy_ == RESIZE_DEFAULT) {
	    resizePolicy_ = RESIZE_BOUND;
	}
	if (!((hbasePolicy_ == HBASE_LEFT ||
	       hbasePolicy_ == HBASE_CENTER ||
	       hbasePolicy_ == HBASE_RIGHT ||
	       hbasePolicy_ == HBASE_FILL) &&
	      (vbasePolicy_ == VBASE_TOP ||
	       vbasePolicy_ == VBASE_CENTER ||
	       vbasePolicy_ == VBASE_BOTTOM ||
	       vbasePolicy_ == VBASE_FILL) &&
	      (halignPolicy_ == HALIGN_LEFT ||
	       halignPolicy_ == HALIGN_CENTER ||
	       halignPolicy_ == HALIGN_RIGHT ||
	       halignPolicy_ == HALIGN_FILL ||
	       halignPolicy_ == HALIGN_PREFERRED) &&
	      (valignPolicy_ == VALIGN_TOP ||
	       valignPolicy_ == VALIGN_CENTER ||
	       valignPolicy_ == VALIGN_BOTTOM ||
	       valignPolicy_ == VALIGN_FILL ||
	       valignPolicy_ == VALIGN_PREFERRED) &&
	      (widthPolicy_ == WIDTH_PREFERRED ||
	       widthPolicy_ == WIDTH_EQUAL ||
	       widthPolicy_ == WIDTH_FILL ||
	       widthPolicy_ == WIDTH_RELATIVE ||
	       widthPolicy_ == WIDTH_ABSOLUTE) &&
	      (heightPolicy_ == HEIGHT_PREFERRED ||
	       heightPolicy_ == HEIGHT_EQUAL) &&
	      (resizePolicy_ == RESIZE_NONE ||
	       resizePolicy_ == RESIZE_WIDTH ||
	       resizePolicy_ == RESIZE_HEIGHT ||
	       resizePolicy_ == RESIZE_BOUND))) {
	    throw (new IllegalArgumentException(
		"policy : " + Integer.toHexString(policy)));
	}
    }

    public void addLayoutComponent(Component comp, Object constraints) {
	TableLayoutConstraints info;
	if (constraints == null) {
	    info = new TableLayoutConstraints();
	} else if (constraints.equals("next")) {
	    info = new TableLayoutConstraints();
	} else if (constraints.equals("last")) {
	    info = TableLayoutConstraints.makeLast();
	} else if (constraints.equals("first")) {
	    info = TableLayoutConstraints.makeFirst();
	} else {
	    info = (TableLayoutConstraints)constraints;
	}
	switch (info.x) {

	case TableLayoutConstraints.POSITION_CURRENT:
	    // do nothing
	    break;
	case TableLayoutConstraints.POSITION_NEXT:
	    x_++;
	    break;
	case TableLayoutConstraints.POSITION_PREVIOUS:
	    x_--;
	    break;
	default:
	    x_ = info.x;
	}
	switch (info.y) {

	case TableLayoutConstraints.POSITION_CURRENT:
	    // do nothing
	    break;
	case TableLayoutConstraints.POSITION_NEXT:
	    y_++;
	    break;
	case TableLayoutConstraints.POSITION_PREVIOUS:
	    y_--;
	    break;
	default:
	    y_ = info.y;
	}
	comps_.put(x_, y_, comp);
	width_ = comps_.getWidth();
	height_ = comps_.getHeight();
	switch(info.xaction) {

	case TableLayoutConstraints.ACTION_STAY:
	    // do nothing
	    break;
	case TableLayoutConstraints.ACTION_NEXT:
	    x_++;
	    break;
	case TableLayoutConstraints.ACTION_FIRST:
	    x_ = 0;
	    break;
	default:
	    throw (new IllegalArgumentException(
		"Invalid xaction : " + info.xaction));
	}
	switch(info.yaction) {

	case TableLayoutConstraints.ACTION_STAY:
	    // do nothing
	    break;
	case TableLayoutConstraints.ACTION_NEXT:
	    y_++;
	    break;
	case TableLayoutConstraints.ACTION_FIRST:
	    y_ = 0;
	    break;
	default:
	    throw (new IllegalArgumentException(
		"Invalid yaction : " + info.yaction));
	}
    }

    // AbstractLayoutManager
    public void removeLayoutComponent(Component comp) {
	super.removeLayoutComponent(comp);
	for (int x = 0;x < width_;x++) {
	    for (int y = 0;y < height_;y++) {
		if (comps_.get(x, y) == comp) {
		    comps_.put(x, y, null);
		}
	    }
	}
    }

    public Dimension preferredLayoutSize(Container target) {
	Insets insets = target.getInsets();
	if (target.getComponentCount() == 0) {
	    return (new Dimension(_getLeftRightGaps(insets),
				  _getTopBottomGaps(insets)));
	}
	LayoutManagerHelper[] rowHelpers = makeRowHelpers(target);
	LayoutManagerHelper[] columnHelpers = makeColumnHelpers(target);

	// calculates the width
	int width = 0;
	switch (widthPolicy_) {

	case WIDTH_PREFERRED:
	    for (int x = 0;x < width_;x++) {
		width += columnHelpers[x].maxPref.width;
	    }
	    break;
	case WIDTH_EQUAL:
	    int maxWidth = 0;
	    for (int x = 0;x < width_;x++) {
		maxWidth = Math.max(columnHelpers[x].maxPref.width, maxWidth);
	    }
	    width = maxWidth * width_;
	    break;
	default:
	    throw (new InternalError(
		"Invalid width policy : " +
		Integer.toHexString(widthPolicy_)));
	}

	// calculates the height
	int height = 0;
	switch (heightPolicy_) {

	case HEIGHT_PREFERRED:
	    for (int y = 0;y < height_;y++) {
		height += rowHelpers[y].maxPref.height;
	    }
	    break;
	case HEIGHT_EQUAL:
	    int maxHeight = 0;
	    for (int y = 0;y < height_;y++) {
		maxHeight = Math.max(rowHelpers[y].maxPref.height, maxHeight);
	    }
	    height = maxHeight * height_;
	    break;
	default:
	    throw (new InternalError(
		"Invalid height policy : " +
		Integer.toHexString(heightPolicy_)));
	}
	return (
	    _adjustBorder(
		new Dimension(
		    width + _getHInnerGaps(width_),
		    height + _getVInnerGaps(height_)
		),
		insets
	    )
	);
    }

    public Dimension minimumLayoutSize(Container target) {
	return (preferredLayoutSize(target));
    }

    public void layoutContainer(Container target) {
	if (target.getComponentCount() == 0) {
	    return;
	}
	LayoutManagerHelper[] rowHelpers = makeRowHelpers(target);
	LayoutManagerHelper[] columnHelpers = makeColumnHelpers(target);
	Dimension size = target.getSize();
	Insets insets = target.getInsets();
	if (size.width == 0 || size.height == 0) {
	    return;
	}

	ArrayMatrix areas = new ArrayMatrix(width_, height_);
	for (int x = 0;x < width_;x++) {
	    for (int y = 0;y < height_;y++) {
		areas.put(x, y, new Rectangle(0, 0, 0, 0));
	    }
	}

	Debug.log(3, this, "components : " + comps_);
	Debug.log(3, this, "after initialize : " + areas);

	// calc area widths and relative x positions
	switch (widthPolicy_) {

	case WIDTH_PREFERRED: {
	    int xPos = 0;
	    for (int x = 0;x < width_;x++) {
		int width = columnHelpers[x].maxPref.width;
		for (int y = 0;y < height_;y++) {
		    Rectangle area = (Rectangle)areas.get(x, y);
		    area.x = xPos;
		    area.width = width;
		}
		xPos += width + hInnerGap_;
	    }
	    break;
	}
	case WIDTH_EQUAL: {
	    int xPos = 0;
	    int maxWidth = 0;
	    for (int x = 0; x < width_;x++) {
		maxWidth = Math.max(columnHelpers[x].maxPref.width, maxWidth);
	    }
	    for (int x = 0;x < width_;x++) {
		for (int y = 0;y < height_;y++) {
		    Rectangle area = (Rectangle)areas.get(x, y);
		    area.x = xPos;
		    area.width = maxWidth;
		}
		xPos += maxWidth + hInnerGap_;
	    }
	    break;
	}
	default:
	    throw (new InternalError(
		"Invalid width policy : " +
		Integer.toHexString(widthPolicy_)));
	}

	Debug.log(3, this,
		  "after relative x position and width : " + areas);

	// calc area heights and relative y positions
	switch (heightPolicy_) {

	case HEIGHT_PREFERRED: {
	    int yPos = 0;
	    for (int y = 0;y < height_;y++) {
		int height = rowHelpers[y].maxPref.height;
		for (int x = 0;x < width_;x++) {
		    Rectangle area = (Rectangle)areas.get(x, y);
		    area.y = yPos;
		    area.height = height;
		}
		yPos += height + vInnerGap_;
	    }
	    break;
	}
	case HEIGHT_EQUAL: {
	    int yPos = 0;
	    int maxHeight = 0;
	    for (int y = 0; y < height_;y++) {
		maxHeight = Math.max(rowHelpers[y].maxPref.height, maxHeight);
	    }
	    for (int y = 0;y < height_;y++) {
		for (int x = 0;x < width_;x++) {
		    Rectangle area = (Rectangle)areas.get(x, y);
		    area.y = yPos;
		    area.height = maxHeight;
		}
		yPos += maxHeight + vInnerGap_;
	    }
	    break;
	}
	default:
	    throw (new InternalError(
		"Invalid height policy : " +
		Integer.toHexString(heightPolicy_)));
	}

	Debug.log(3, this,
		  "after relative y position and height : " + areas);

	// calc absolute x positions
	Rectangle last = (Rectangle)areas.get(width_ - 1, height_ - 1);
	int compsWidth = last.x + last.width;
	int compsHeight = last.y + last.height;
	int viewWidth = size.width - _getLeftRightGaps(insets);
	int xGap;
	switch(hbasePolicy_) {

	case HBASE_LEFT: {
	    xGap = _getLeftGap(insets);
	    break;
	}
	case HBASE_CENTER:
	    if (viewWidth > compsWidth) {
		xGap = _getLeftGap(insets)
		    + (viewWidth / 2) - (compsWidth / 2);
	    } else {
		xGap = _getLeftGap(insets);
	    }
	    break;
	case HBASE_RIGHT:
	    if (viewWidth > compsWidth) {
		xGap = _getLeftGap(insets) + viewWidth - compsWidth;
	    } else {
		xGap = _getLeftGap(insets);
	    }
	    break;
	default:
	    throw (new InternalError(
		"Invalid hbase policy : " +
		Integer.toHexString(hbasePolicy_)));
	}

	// calc absolute y positions
	int viewHeight = size.height - _getTopBottomGaps(insets);
	int yGap;
	switch(vbasePolicy_) {

	case VBASE_TOP: {
	    yGap = _getTopGap(insets);
	    break;
	}
	case VBASE_CENTER:
	    if (viewHeight > compsHeight) {
		yGap = _getTopGap(insets)
		    + (viewHeight / 2) - (compsHeight / 2);
	    } else {
		yGap = _getTopGap(insets);
	    }
	    break;
	case VBASE_BOTTOM:
	    if (viewHeight > compsHeight) {
		yGap = _getTopGap(insets) + viewHeight - compsHeight;
	    } else {
		yGap = _getTopGap(insets);
	    }
	    break;
	default:
	    throw (new InternalError(
		"Invalid vbase policy : " +
		Integer.toHexString(vbasePolicy_)));
	}

	// translate
	for (int x = 0;x < width_;x++) {
	    for (int y = 0;y < height_;y++) {
		Rectangle area = (Rectangle)areas.get(x, y);
		area.translate(xGap, yGap);
	    }
	}

	Debug.log(3, this, "after translate : " + areas);

	// calc resize
	switch (resizePolicy_) {

	case RESIZE_NONE: {
	    for (int x = 0;x < width_;x++) {
		for (int y = 0;y < height_;y++) {
		    Rectangle area = (Rectangle)areas.get(x, y);
		    Dimension compsize = columnHelpers[x].prefs[y];
		    Point point = UAWT.calcPointToDrawCenter(area, compsize);
		    area.x = point.x;
		    area.y = point.y;
		}
	    }
	    break;
	}
	case RESIZE_WIDTH: {
	    for (int x = 0;x < width_;x++) {
		for (int y = 0;y < height_;y++) {
		    Rectangle area = (Rectangle)areas.get(x, y);
		    Dimension compsize = columnHelpers[x].prefs[y];
		    Point point = UAWT.calcPointToDrawCenter(area, compsize);
		    area.width = compsize.width;
		    area.y = point.y;
		}
	    }
	    break;
	}
	case RESIZE_HEIGHT: {
	    for (int x = 0;x < width_;x++) {
		for (int y = 0;y < height_;y++) {
		    Rectangle area = (Rectangle)areas.get(x, y);
		    Dimension compsize = columnHelpers[x].prefs[y];
		    Point point = UAWT.calcPointToDrawCenter(area, compsize);
		    area.x = point.x;
		    area.height = compsize.height;
		}
	    }
	    break;
	}
	case RESIZE_BOUND:
	    // do nothing
	    break;
	default:
	    throw (new InternalError(
		"Bad resizePolicy_ : " +
		Integer.toHexString(resizePolicy_)));
	}

	// should do align : XXX

	Debug.log(3, this, "after calculation : " + areas);

	// do move
	Component[] compList = new Component[width_ * height_];
	Rectangle[] areaList = new Rectangle[width_ * height_];
	int index = 0;
	for (int y = 0;y < height_;y++) {
	    for (int x = 0;x < width_;x++) {
		Component comp = (Component)comps_.get(x, y);
		if (comp == null) {
		    compList[index] = new NullComponent();
		} else {
		    compList[index] = comp;
		}
		areaList[index] = (Rectangle)areas.get(x, y);
		index++;
	    }
	}

	_moveComponents(compList, areaList);
    }

    public Component getComponent(int x, int y) {
	return ((Component)comps_.get(x, y));
    }

    protected LayoutManagerHelper[] makeRowHelpers(Container target) {
	LayoutManagerHelper[] helpers = new LayoutManagerHelper[height_];
	Component[] columns = new Component[width_];
	for (int y = 0;y < height_;y++) {
	    for (int x = 0;x < width_;x++) {
		Component comp = (Component)comps_.get(x, y);
		if (comp == null) {
		    columns[x] = new NullComponent();
		} else {
		    columns[x] = comp;
		}
	    }
	    helpers[y] = new LayoutManagerHelper(target, columns);
	    helpers[y].setup();
	}
	return (helpers);
    }

    protected LayoutManagerHelper[] makeColumnHelpers(Container target) {
	LayoutManagerHelper[] helpers = new LayoutManagerHelper[width_];
	Component[] rows = new Component[height_];
	for (int x = 0;x < width_;x++) {
	    for (int y = 0;y < height_;y++) {
		Component comp = (Component)comps_.get(x, y);
		if (comp == null) {
		    rows[y] = new NullComponent();
		} else {
		    rows[y] = comp;
		}
	    }
	    helpers[x] = new LayoutManagerHelper(target, rows);
	    helpers[x].setup();
	}
	return (helpers);
    }

    static class ArrayMatrix {	// XXX : replace D2Array
	protected Object[] data_;
	protected int width_;
	protected int height_;

	ArrayMatrix(int width, int height) {
	    data_ = new Object[width * height];
	    width_ = width;
	    height_ = height;
	}

	Object get(int x, int y) {
	    return (data_[y * width_ + x]);
	}

	void put(int x, int y, Object obj) {
	    data_[y * width_ + x] = obj;
	}

	Object[] getComps() {
	    return (data_);
	}

	// Object
	public String toString() {
	    StringBuffer buffer = new StringBuffer();
	    buffer.append('[');
	    for (int y = 0;y < height_;y++) {
		buffer.append('[');
		for (int x = 0;x < width_;x++) {
		    buffer.append(get(x, y));
		    buffer.append(',');
		}
		buffer.append(']');
	    }
	    buffer.append(']');
	    return (buffer.toString());
	}
    }

    public static class NullComponent extends Component {
    }

    // test driver
    public static void main(String[] args) {
	jp.gr.java_conf.jaba2.util.Debug.setClass("TableLayout");
	javax.swing.JPanel panel = new javax.swing.JPanel();
	panel.setLayout(new TableLayout());
	panel.add(new javax.swing.JLabel("1-1"));
	panel.add(new javax.swing.JLabel("1-2"));
	panel.add(new javax.swing.JLabel("1-3"), "last");
	panel.add(new javax.swing.JLabel("2-1"));
	panel.add(new javax.swing.JLabel("2-2"));
	panel.add(new javax.swing.JLabel("2-3"), "last");
	new JTestFrame("TableLayout test", panel);
    }
}
