/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2000  ASAMI, Tomoharu (asami@zeomtech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.jmodel.lang;

import jp.gr.java_conf.jaba2.text.UString;

/**
 * ULJModel
 *
 * @since   Jan. 18, 2000
 * @version Sep.  2, 2000
 * @author  ASAMI, Tomoharu (asami@zeomtech.com)
 */
public final class ULJModel implements IAccessMode {
    public static String getCode(LJInterface ljinterface) {
	int indent = 4;
	StringBuffer buffer = new StringBuffer();
	_makePrologueCode(ljinterface, buffer);
	getTitle(ljinterface, buffer);
	// interface
	getAccessMode(ljinterface.getAccessMode(), buffer);
	buffer.append(" interface ");
	buffer.append(ljinterface.getName());
	LJInterface[] parents = ljinterface.getParentInterfaces();
	if (parents.length > 0) {
	    buffer.append(" extends ");
	    buffer.append(parents[0].getName());
	    for (int i = 1;i < parents.length;i++) {
		buffer.append(", ");
		buffer.append(parents[i].getName());
	    }
	}
	buffer.append(" {\n");
	LJVariable[] variables = ljinterface.getVariables();
	for (int i = 0;i < variables.length;i++) {
	    LJVariable variable = variables[i];
	    LJType type = variable.getType();
	    buffer.append("@");
	    buffer.append(type.getName());
	    buffer.append(" ");
	    buffer.append(variable.getName());
	    buffer.append(";\n");
	}
	if (variables.length > 0) {
	    buffer.append("\n");
	}
	LJMethod[] methods = ljinterface.getMethods();
	if (methods.length > 0) {
	    getInterfaceSignature(methods[0], buffer);
	    for (int i = 1;i < methods.length;i++) {
		buffer.append("\n");
		getInterfaceSignature(methods[i], buffer);
	    }
	}
	buffer.append("}\n");
	// indent
	return (_adjustIndent(buffer));
    }

    public static String getCode(LJClass ljclass) {
	int indent = 4;
	StringBuffer buffer = new StringBuffer();
	_makePrologueCode(ljclass, buffer);
	getTitle(ljclass, buffer);
	// class
	getAccessMode(ljclass.getAccessMode(), buffer);
	if (ljclass.isAbstract()) {
	    buffer.append(" abstract");
	}
	buffer.append(" class ");
	buffer.append(ljclass.getName());
	LJClass parent = ljclass.getParentClass();
	if (parent != null) {
	    buffer.append(" extends ");
	    buffer.append(parent.getName());
	}
	LJInterface[] interfaces = ljclass.getImplementsInterfaces();
	if (interfaces.length > 0) {
	    buffer.append(" implements ");
	    buffer.append(interfaces[0].getName());
	    for (int i = 1;i < interfaces.length;i++) {
		buffer.append(", ");
		buffer.append(interfaces[i].getName());
	    }
	}
	buffer.append(" {\n");
	LJVariable[] variables = ljclass.getVariables();
	for (int i = 0;i < variables.length;i++) {
	    LJVariable variable = variables[i];
	    LJType type = variable.getType();
	    String defaultValue = variable.getDefaultValue();
	    if (type instanceof LJListType) {
		buffer.append("@");
		buffer.append("// List<");
		buffer.append(((LJListType)type).getContainee().getName());
		buffer.append(">\n");
	    }
	    buffer.append("@");
	    getAccessMode(variable.getAccessMode(), buffer);
	    buffer.append(" ");
	    buffer.append(type.getName());
	    buffer.append(" ");
	    buffer.append(variable.getName());
	    if (type instanceof LJListType) {
		buffer.append(" = new java.util.ArrayList()");
	    } else if (defaultValue != null) {
		buffer.append(" = ");
		buffer.append(makeLiteralValue(defaultValue, type));
	    }
	    buffer.append(";\n");
	}
	variables = ljclass.getClassVariables();
	for (int i = 0;i < variables.length;i++) {
	    LJVariable variable = variables[i];
	    LJType type = variable.getType();
	    String defaultValue = variable.getDefaultValue();
	    if (type instanceof LJListType) {
		buffer.append("@");
		buffer.append("// List<");
		buffer.append(((LJListType)type).getContainee().getName());
		buffer.append(">\n");
	    }
	    buffer.append("@");
	    getAccessMode(variable.getAccessMode(), buffer);
	    buffer.append(" static ");
	    buffer.append(type.getName());
	    buffer.append(" ");
	    buffer.append(variable.getName());
	    if (type instanceof LJListType) {
		buffer.append(" = new java.util.ArrayList()");
	    } else if (defaultValue != null) {
		buffer.append(" = ");
		buffer.append(makeLiteralValue(defaultValue, type));
	    }
	    buffer.append(";\n");
	}
	// XXX : adjust space
	LJMethod[] methods = ljclass.getConstructors();
	for (int i = 0;i < methods.length;i++) {
	    buffer.append("\n");
	    getMethod(methods[i], buffer);
	}
	methods = ljclass.getMethods();
	for (int i = 0;i < methods.length;i++) {
	    buffer.append("\n");
	    getMethod(methods[i], buffer);
	}
	methods = ljclass.getClassMethods();
	for (int i = 0;i < methods.length;i++) {
	    buffer.append("\n");
	    getMethod(methods[i], buffer);
	}
	buffer.append("}\n");
	// indent
	return (_adjustIndent(buffer));
    }

    public static String makeLiteralValue(
	String defaultValue,
	LJType type
    ) {
	String typeName = type.getName();
	if ("String".equals(typeName) ||
	    "java.lang.String".equals(typeName)) {

	    return ("\"" + defaultValue + "\"");
	} else if ("Byte".equals(typeName) ||
		   "java.lang.Byte".equals(typeName)) {

	    return ("new Byte(" + defaultValue + ")");
	} else if ("Boolean".equals(typeName) ||
		   "java.lang.Boolean".equals(typeName)) {

	    return ("new Boolean(" + defaultValue + ")");
	} else if ("Short".equals(typeName) ||
		   "java.lang.Short".equals(typeName)) {

	    return ("new Short(" + defaultValue + ")");
	} else if ("Integer".equals(typeName) ||
		   "java.lang.Integer".equals(typeName)) {

	    return ("new Integer(" + defaultValue + ")");
	} else if ("Long".equals(typeName) ||
		   "java.lang.Long".equals(typeName)) {

	    return ("new Long(" + defaultValue + ")");
	} else if ("Float".equals(typeName) ||
		   "java.lang.Float".equals(typeName)) {

	    return ("new Float(" + defaultValue + ")");
	} else if ("Double".equals(typeName) ||
		   "java.lang.Double".equals(typeName)) {

	    return ("new Double(" + defaultValue + ")");
	} else if ("BigDecimal".equals(typeName) ||
		   "java.math.BigDecimal".equals(typeName)) {

	    return ("new BigDecimal(\"" + defaultValue + "\")");
	} else if ("BigInteger".equals(typeName) ||
		   "java.math.BigInteger".equals(typeName)) {

	    return ("new BigInteger(\"" + defaultValue + "\")");
	} else if ("URL".equals(typeName) ||
		   "java.net.URL".equals(typeName)) {

	    return ("new URL(\"" + defaultValue + "\")");
	} else if ("Locale".equals(typeName) ||
		   "java.util.Locale".equals(typeName)) {

	    throw (new UnsupportedOperationException());
	} else if ("Date".equals(typeName) ||
		   "java.util.Date".equals(typeName)) {

	    throw (new UnsupportedOperationException());
	} else {
	    return (defaultValue);
	}
    }

    private static void _makePrologueCode(
	LJClassifier ljClassifier,
	StringBuffer buffer
    ) {
	// pacakge
	LJPackage ljPackage = ljClassifier.getPackage();
	if (ljPackage != null) {
	    if (!"".equals(ljPackage.getName())) {
		buffer.append("package ");
		buffer.append(ljPackage.getName());
		buffer.append(";\n");
		buffer.append("\n");
	    }
	}
	// import
	String[] importPackages = ljClassifier.getImportPackages();
	for (int i = 0;i < importPackages.length;i++) {
	    buffer.append("import ");
	    buffer.append(importPackages[i]);
	    buffer.append(";\n");
	}
	if (importPackages.length > 0) {
	    buffer.append("\n");
	}
    }

    private static String _adjustIndent(StringBuffer buffer) {
	StringBuffer result = new StringBuffer();
	boolean lineStart = true;
	int size = buffer.length();
	for (int i = 0;i < size;i++) {
	    char c = buffer.charAt(i);
	    if (lineStart) {
		if (c == '@') {
		    result.append("    "); // XXX
		    continue;
		}
	    }
	    if (c == '\n' || c == '\r') {
		lineStart = true;
	    } else {
		lineStart = false;
	    }
	    result.append(c);
	}
	return (new String(result));
    }

    public static void getTitle(
	LJClassifier classifier,
	StringBuffer buffer
    ) {
	String desc = classifier.getDescription();
	String version = classifier.getVersion();
	String author = classifier.getAuthor();
	if (desc == null && version == null && author == null) {
	    return;
	}
	buffer.append("/**\n");
	if (desc != null) {
	    _getDesc(desc, " * ", buffer);
	    buffer.append(" *\n");
	}
	if (version != null) {
	    buffer.append(" * @version ");
	    buffer.append(version);
	    buffer.append("\n");
	}
	if (author != null) {
	    buffer.append(" * @author  ");
	    buffer.append(author);
	    buffer.append("\n");
	}
	buffer.append(" */\n");
    }

    public static void getMethodTitle(LJMethod method, StringBuffer buffer) {
	String desc = method.getDescription();
	LJArgument[] arguments = method.getArguments();
	LJType[] exceptions = method.getExceptions();
	LJType returnType = method.getReturnType();
	if (desc == null &&
	    (arguments == null || arguments.length == 0) &&
	    exceptions == null &&
	    returnType == null) {

	    return;
	}
	buffer.append("@/**\n");
	if (desc != null) {
	    _getDesc(desc, "@ * ", buffer);
	    buffer.append("@ *\n");
	}
	if (arguments != null) {
	    for (int i = 0;i < arguments.length;i++) {
		LJArgument arg = arguments[i];
		String adesc = arg.getDescription();
		buffer.append("@ * @param ");
		buffer.append(arg.getName());
		if (adesc != null) {
		    buffer.append(adesc); // XXX
		}
		buffer.append("\n");
	    }
	}
	if (exceptions != null) {
	    for (int i = 0;i < exceptions.length;i++) {
		LJType e = exceptions[i];
		buffer.append("@ * @exception ");
		buffer.append(e.getName());
		// XXX : desc
		buffer.append("\n");
	    }
	}
	if (returnType != null) {
	    String typeName = returnType.getName();
	    if (!"void".equals(typeName)) {
		buffer.append("@ * @return ");
		buffer.append(typeName);
		buffer.append("\n");
	    }
	}
	buffer.append("@ */\n");
    }

    private static void _getDesc(
	String text,
	String prefix,
	StringBuffer buffer
    ) {
	String[] list = UString.makeStringList(text);
	for (int i = 0;i < list.length;i++) {
	    buffer.append(prefix);
	    buffer.append(list[i]);
	    buffer.append("\n");
	}
    }

    public static void getAccessMode(int access, StringBuffer buffer) {
	switch (access) {

	case ACCESS_PUBLIC:
	    buffer.append("public");
	    break;
	case ACCESS_PROTECTED:
	    buffer.append("protected");
	    break;
	case ACCESS_PRIVATE:
	    buffer.append("private");
	    break;
	default:
	    throw (new InternalError());
	}
    }

    public static void getMethod(LJMethod method, StringBuffer buffer) {
	getMethodTitle(method, buffer);
	buffer.append("@");
	getAccessMode(method.getAccessMode(), buffer);
	buffer.append(" ");
	if (method.isFinal()) {
	    buffer.append("final ");
	}
	if (method.isStatic()) {
	    buffer.append("static ");
	}
	LJType type = method.getReturnType();
	if (type != null) {
	    buffer.append(type.getName());
	    buffer.append(" ");
	}
	buffer.append(method.getName());
	buffer.append("(");
	LJArgument[] arguments = method.getArguments();
	if (arguments != null) {
	    if (arguments.length > 0) {
		LJArgument arg = arguments[0];
		buffer.append(arg.getType().getName());
		buffer.append(" ");
		buffer.append(arg.getName());
		for (int i = 1;i < arguments.length;i++) {
		    arg = arguments[i];
		    buffer.append(", ");
		    buffer.append(arg.getType().getName());
		    buffer.append(" ");
		    buffer.append(arg.getName());
		}
	    }
	}
	buffer.append(")");
	LJType[] exceptions = method.getExceptions();
	if (exceptions != null) {
	    if (exceptions.length > 0) {
		buffer.append(" throws ");
		buffer.append(exceptions[0].getName());
		for (int i = 1;i < exceptions.length;i++) {
		    buffer.append(", ");
		    buffer.append(exceptions[i].getName());
		}
	    }
	}
	buffer.append(" {\n");
	String impl = method.getImplementation();
	if (impl != null) {
	    String[] lines = UString.makeStringList(impl);
	    for (int j = 0;j < lines.length;j++) {
		buffer.append("@@");
		buffer.append(lines[j]);
		buffer.append("\n");
	    }
	}
	buffer.append("@}\n");
    }

    public static void getInterfaceSignature(
	LJMethod method,
	StringBuffer buffer
    ) {
	getMethodTitle(method, buffer);
	buffer.append("@");
	LJType type = method.getReturnType();
	if (type != null) {
	    buffer.append(type.getName());
	    buffer.append(" ");
	}
	buffer.append(method.getName());
	buffer.append("(");
	LJArgument[] arguments = method.getArguments();
	if (arguments != null) {
	    if (arguments.length > 0) {
		LJArgument arg = arguments[0];
		buffer.append(arg.getType().getName());
		buffer.append(" ");
		buffer.append(arg.getName());
		for (int i = 1;i < arguments.length;i++) {
		    arg = arguments[i];
		    buffer.append(", ");
		    buffer.append(arg.getType().getName());
		    buffer.append(" ");
		    buffer.append(arg.getName());
		}
	    }
	}
	buffer.append(")");
	LJType[] exceptions = method.getExceptions();
	if (exceptions != null) {
	    if (exceptions.length > 0) {
		buffer.append(" throws ");
		buffer.append(exceptions[0].getName());
		for (int i = 1;i < exceptions.length;i++) {
		    buffer.append(", ");
		    buffer.append(exceptions[i].getName());
		}
	    }
	}
	buffer.append(";\n");
    }
}
