/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2001  ASAMI, Tomoharu (asami@zeomtech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.xml.relaxng;

import java.util.*;
import java.io.IOException;
import java.net.URL;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.xml.*;
import jp.gr.java_conf.jaba2.xml.relaxng.rRelaxng.*;
import jp.gr.java_conf.jaba2.xml.relax.cooked.*;
import jp.gr.java_conf.jaba2.xml.relax.expanded.*;

/**
 * RNGGrammar
 *
 * @since   Jun. 15, 2001
 * @version Aug. 25, 2001
 * @author  ASAMI, Tomoharu (asami@zeomtech.com)
 */
public class RNGGrammar {
    private static final int OCCURS_ONE = 1;
    private static final int OCCURS_ZEROONE = 2;
    private static final int OCCURS_ONEMORE = 3;
    private static final int OCCURS_ZEROMORE = 4;
    private String baseURI_;
    private Map modules_ = new HashMap(); // Map<String, RNGModule>
    private Map nsByLabel_ = new HashMap(); // Map<String, String>

    public RNGGrammar(RNGrammar grammar, String baseURI) {
	baseURI_ = baseURI;
	RNDefine[] defines = grammar.getDefine();
        for (int i = 0;i < defines.length;i++) {
	    RNDefine define = defines[i];
	    _buildDefine(define);
	}
	RNStart start = grammar.getStart();
	_buildStart(start);
    }

    public RNGModule[] getModules() {
	Collection values = modules_.values();
	RNGModule[] modules = new RNGModule[values.size()];
	return ((RNGModule[])values.toArray(modules));
    }

    private void _buildStart(RNStart start) {
	IRNElementHedgeChoice[] nodes = start.getElementHedge();
	_buildStart(nodes);
    }

    private void _buildStart(IRNElementHedgeChoice[] nodes) {
	for (int i = 0;i < nodes.length;i++) {
	    IRNElementHedgeChoice node = nodes[i];
	    if (node instanceof RNElement) {
		RNElement element = (RNElement)node;
		String labelName = _makeLabelName("root", element.getName());
		_buildElement((RNElement)node, labelName);
		RNGModule module = _getModule(element);
		module.addExportLabel(labelName);
	    } else if (node instanceof RNRef) {
		RNRef ref = (RNRef)node;
		String name = ref.getName();
		RNGModule module = _getModule(ref);
		module.addExportLabel(name);
	    }
	}
    }

    private void _buildDefine(RNDefine define) {
	IRNElementHedgeChoice[] nodes = define.getElementHedge();
	if (nodes.length == 1 &&
	    nodes[0] instanceof RNElement) {

	    _buildElementRule(define, (RNElement)nodes[0]);
	} else {
	    _buildHedgeRule(define, nodes);
	}
    }

    private RNGModule _getModule(RNDefine define) {
	return (_getModule(""));
    }

    private RNGModule _getModule(RNElement element) {
	return (_getModule(_findNamespace(element)));
    }

    private RNGModule _getModule(IRNode node) {
	return (_getModule(_findNamespace(node)));
    }

    private String _findNamespace(IRNode node) {
	for (;;) {
	    if (node instanceof RNElement) {
		RNElement element = (RNElement)node;
		String ns = element.getNs();
		if (ns != null) {
		    return (ns);
		}
	    }
	    IRNode parent = node.getParentRNode();
	    if (parent == null) {
		return ("");
	    }
	    node = parent;
	}
    }

    private RNGModule _getModule(RNRef ref) {
	String name = ref.getName();
	String ns = (String)nsByLabel_.get(name);
	if (ns == null) {
	    throw (new InternalError(name));
	}
	return (_getModule(ns));
    }

    private RNGModule _getModule(RNAttribute attr) {
	String ns = attr.getNs();
	// XXX global?
	if (ns == null) {
	    return (_getModule(""));
	} else {
	    return (_getModule(ns));
	}
    }

    private RNGModule _getModule(String namespaceURI) {
	RNGModule module = (RNGModule)modules_.get(namespaceURI);
	if (module == null) {
	    module = new RNGModule(namespaceURI);
	    module.setBaseURI(baseURI_);
	    modules_.put(namespaceURI, module);
	}
	return (module);
    }

    private void _buildElementRule(RNDefine define, RNElement element) {
	String labelName = define.getName();
	_buildElement(element, labelName);
    }

    private CRefWithLabel _buildElementWithRef(
	RNElement element,
	String context,
	int occurs
    ) {
	String labelName = _makeLabelName(context, element.getName());
	_buildElement(element, labelName);
	CRefWithLabel ref = new CRefWithLabel(labelName, _makeCOccurs(occurs));
	ref.setBase(element.getXMLElement());
	return (ref);
    }

    private void _buildElement(
	RNElement element,
	String labelName
    ) {
	String elementName = element.getName();
	IRNElementHedgeChoice[] contents = element.getElementHedge();
	RNGModule ngModule = _getModule(element);
	CModule cModule = ngModule.getCModule();
	CTagPattern cTag = new CTagPattern(labelName, elementName, cModule);
	cTag.setBase(element.getXMLElement());
	CSequence cSequence = new CSequence();
	String typeName = null;
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    _makeLabelName(labelName, elementName),
		    OCCURS_ONE
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		RNRef ngRef = (RNRef)content;
		String refLabelName = ngRef.getName();
		CRefWithLabel cRef = new CRefWithLabel(refLabelName, "");
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String value = _getValueText(ngAttr);
		if (value != null) {
		    cTag.addContent(new CValue(value));
		} else {
		    String type = _getTypeName(ngAttr);
		    CAttribute cAttr = new CAttribute(
			ngAttr.getName(),
			type,
			cModule
		    );
		    cAttr.setRequired(true);
		    cTag.addContent(cAttr);
		}
	    } else if (content instanceof RNText) {
		typeName = "string";
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		typeName = data.getType();
	    } else if (content instanceof RNValue) {
	    } else if (content instanceof RNGroup) {
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, labelName, cChoice);
		cSequence.addParticle(cChoice);
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    labelName,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cTag.addContent(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    labelName,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    labelName,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError());
	    }
	}
	if (typeName != null) {
	    CTypeElementRule cRule = new CTypeElementRule(
		labelName,
		typeName,
		cModule
	    );
	    cRule.setBase(element.getXMLElement());
	    cModule.addElementRule(cRule);
	    nsByLabel_.put(labelName, cModule.getTargetNamespace());
	} else {
	    if (cSequence.getParticleCount() == 0) {
		CEmptyElementRule cRule = new CEmptyElementRule(
		    labelName,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
	    } else {
		CSequenceElementRule cRule = new CSequenceElementRule(
		    labelName,
		    cSequence,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
	    }
	}
	cModule.addTagPattern(cTag);
    }

    private String _getValueText(RNAttribute attr) {
	IRNAttributeHedgeChoice content = attr.getAttributeHedge();
	if (content instanceof RNValue) {
	    RNValue value = (RNValue)content;
	    return (value.getContent());
	} else {
	    return (null);
	}
    }

    private String _getTypeName(RNAttribute attr) {
	IRNAttributeHedgeChoice content = attr.getAttributeHedge();
	if (content instanceof RNText) {
	    return ("string");
	} else if (content instanceof RNData) {
	    RNData data = (RNData)content;
	    return (data.getType());
	} else {
	    throw (new InternalError());
	}
    }

    private void _buildGroup(	// XXX : unify generic reduction logic
	RNGroup group,
	String context,
	CSequence cSequence
    ) {
	RNGModule ngModule = _getModule(group);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = group.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		RNRef ngRef = (RNRef)content;
		String refLabelName = ngRef.getName();
		CRefWithLabel cRef = new CRefWithLabel(refLabelName, "");
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cSequence.addParticle(cAttr);
	    } else if (content instanceof RNText) {
	    } else if (content instanceof RNData) {
	    } else if (content instanceof RNValue) {
	    } else if (content instanceof RNGroup) {
	    } else if (content instanceof RNChoice) {
/*
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, cChoice);
		cChoice.addParticle(cChoice);
*/
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cSequence.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildChoice(
	RNChoice ngChoice,
	String context,
	CChoice cChoice
    ) {
	RNGModule ngModule = _getModule(ngChoice);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = ngChoice.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cChoice.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		RNRef ngRef = (RNRef)content;
		String refLabelName = ngRef.getName();
		CRefWithLabel cRef = new CRefWithLabel(refLabelName, "");
		cChoice.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cChoice.addParticle(cAttr);
	    } else if (content instanceof RNText) {
	    } else if (content instanceof RNData) {
	    } else if (content instanceof RNValue) {
	    } else if (content instanceof RNGroup) {
	    } else if (content instanceof RNChoice) {
/*
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, cChoice);
		cChoice.addParticle(cChoice);
*/
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cChoice.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildOptional(
	RNOptional optional,
	String context,
	List result,
	List attrs
    ) {
	IRNElementHedgeChoice[] contents = optional.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROONE
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNRef) {
		RNRef ngRef = (RNRef)content;
		String labelName = ngRef.getName();
		CRefWithLabel cRef = new CRefWithLabel(labelName, "?");
		result.add(cRef);
		return;
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		attrs.add(cAttr);
		return;
	    } else if (content instanceof RNText) {
		throw (new InternalError());
	    } else if (content instanceof RNData) {
		throw (new InternalError());
	    } else if (content instanceof RNValue) {
		throw (new InternalError());
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence("?");
		_buildGroup((RNGroup)content, context, cSequence);
		result.add(cSequence);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("?");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildOneOrMore(
	RNOneOrMore oneOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = oneOrMore.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ONEMORE
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNRef) {
		RNRef ngRef = (RNRef)content;
		String labelName = ngRef.getName();
		CRefWithLabel cRef = new CRefWithLabel(labelName, "+");
		result.add(cRef);
		return;
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(true);
		result.add(cAttr);
		return;
	    } else if (content instanceof RNText) {
		throw (new InternalError());
	    } else if (content instanceof RNData) {
		throw (new InternalError());
	    } else if (content instanceof RNValue) {
		throw (new InternalError());
	    } else if (content instanceof RNGroup) {
		throw (new InternalError());
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("+");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildZeroOrMore(
	RNZeroOrMore zeroOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = zeroOrMore.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROMORE
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNRef) {
		RNRef ngRef = (RNRef)content;
		String labelName = ngRef.getName();
		CRefWithLabel cRef = new CRefWithLabel(labelName, "*");
		result.add(cRef);
		return;
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		result.add(cAttr);
		return;
	    } else if (content instanceof RNText) {
		throw (new InternalError());
	    } else if (content instanceof RNData) {
		throw (new InternalError());
	    } else if (content instanceof RNValue) {
		throw (new InternalError());
	    } else if (content instanceof RNGroup) {
		throw (new InternalError());
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("*");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private String _makeLabelName(String context, String name) {
	return (context + UString.capitalize(name));
    }

    private String _makeCOccurs(int occurs) {
	switch (occurs) {

	case OCCURS_ONE:
	    return ("");
	case OCCURS_ZEROONE:
	    return ("?");
	case OCCURS_ONEMORE:
	    return ("+");
	case OCCURS_ZEROMORE:
	    return ("*");
	default:
	    throw (new InternalError());
	}
    }

    private void _buildHedgeRule(
	RNDefine define,
	IRNElementHedgeChoice[] nodes
    ) {
	String labelName = define.getName();
	RNGModule ngModule = _getModule(define);
	CModule cModule = ngModule.getCModule();
	List list = new ArrayList();
	for (int i = 0;i < nodes.length;i++) {
	    IRNElementHedgeChoice content = nodes[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    _makeLabelName(labelName, element.getName()),
		    OCCURS_ONE
		);
		list.add(cRef);
	    } else if (content instanceof RNRef) {
		RNRef ngRef = (RNRef)content;
		String refLabelName = ngRef.getName();
		CRefWithLabel cRef = new CRefWithLabel(refLabelName, "");
		list.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		list.add(cAttr);
	    } else if (content instanceof RNText) {
		throw (new InternalError());
	    } else if (content instanceof RNData) {
		throw (new InternalError());
	    } else if (content instanceof RNValue) {
		throw (new InternalError());
	    } else if (content instanceof RNGroup) {
		throw (new InternalError());
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, labelName, cChoice);
		list.add(cChoice);
	    } else if (content instanceof RNOptional) {
		_buildOptional(
		    (RNOptional)content,
		    labelName,
		    list,
		    list
		);
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    labelName,
		    list
		);
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    labelName,
		    list
		);
	    } else {
		throw (new InternalError());
	    }
	}
	if (list.size() == 0) {
	    throw (new InternalError("Unidentified spec"));
	}
	CSequence cSequence = new CSequence();
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    cSequence.addParticle((ICParticle)list.get(i));
	}
	CSequenceContentRule cRule = new CSequenceContentRule(
	    labelName,
	    cSequence,
	    cModule
	);
	cModule.addContentRule(cRule);
	cRule.setBase(define.getXMLElement());
	nsByLabel_.put(labelName, cModule.getTargetNamespace());
/*
	if (_canAttPool(list)) {
	    CAttListPattern attPool = new CAttListPattern(
		labelName,
		cModule
	    );
	    int size = list.size();
	    for (int i = 0;i < size;i++) {
		Object node = list.get(i);
		if (!(node instanceof CAttribute)) {
		    throw (new InternalError());
		}
		attPool.addContent((ICPatternContent)node);
	    }
	    cModule.addAttListPattern(attPool);
	} else {
	    CSequence cSequence = new CSequence();
	    int size = list.size();
	    for (int i = 0;i < size;i++) {
		cSequence.addParticle((ICParticle)list.get(i));
	    }
	    CSequenceContentRule cRule = new CSequenceContentRule(
		labelName,
		cSequence,
		cModule
	    );
	    cModule.addContentRule(cRule);
	}
*/
    }
/*
    private boolean _canAttPool(List list) {
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    Object node = list.get(i);
	    if (!(node instanceof CAttribute)) {
		return (false);
	    }
	}
	return (true);
    }
*/
}
