// Copyright (c) 2011 RealSolid. Please read license.txt for applicable licensing.

#ifndef SCBLOCK_H
#define SCBLOCK_H


#include "headers.h"
#include "db.h"
#include "net.h"
#include "init.h"
#include "cryptopp/sha.h"
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>

//
// Nodes collect new transactions into a block, hash them into a hash tree,
// and scan through nonce values to make the block's hash satisfy proof-of-work
// requirements.  When they solve the proof-of-work, they broadcast the block
// to everyone and the block is added to the block chain.  The first transaction
// in the block is a special one that creates a new coin owned by the creator
// of the block.
//
// Blocks are appended to blk0001.dat files on disk.  Their location on disk
// is indexed by CBlockIndex objects in memory.

int64 Block_GetTotalNumEstimate(void);
bool Block_IsInitialDownload(void);
void Block_InvalidChainFound(CBlockIndex* pindexNew);
bool Block_LoadIndex(bool fAllowNew=true);
void Block_PrintTree();
bool Block_Reorganize(CTxDB& txdb, CBlockIndex* pindexNew);
uint256 Block_GetOrphanRoot(const CBlock* pblock);
bool Block_Process(CNode* pfrom, CBlock* pblock);
CBlock* Block_Create(CWallet *pWallet, const std::string &mining_id,CReserveKey& reservekey);
int64 Block_GetCoinBaseValue(uint32 dwWorkBits, int64 nHeight);
CBlockIndex* Block_InsertBlockIndex(uint256 hash);
unsigned int Block_GetNextWorkRequired(const CBlockIndex* pindexLast);
bool Block_CheckProofOfWork(uint256 hash, unsigned int nBits);
bool Block_CheckWork(CBlock* pblock, CWallet& wallet, CReserveKey& reservekey, uint256 *outhash);
CBigNum GetBlockWork(uint32 dwBits);
bool Block_IsPowerBlock(const uint256 &trustedhash,const uint256 &userhash);

extern std::map<uint256, CBlockIndex*> g_BlockIndexMap;
extern std::map<uint256, CBlock*> g_BlockOrphanMap;
extern std::multimap<uint256, CBlock*> g_BlockOrphanMapByPrev;
extern uint256 g_BlockGenesisHash;
extern CBlockIndex* g_pBlockGenesisIndex;
extern CBlockIndex* g_pBlockBestIndex;
extern uint256 g_BlockBestChainHash;
extern CBigNum g_bnBlockBestChainWork;
extern CBigNum g_bnBlockBestInvalidWork ;
extern CBigNum g_bnBlockProofOfWorkLimit;
//


class CBlock
{
public:
    // header
    BLOCK_DATA blk;
    unsigned char m_WorkArea[384];     //we hash 512bytes so this needs to be here

    bool m_bPower;

    // network and disk
    std::vector<CTransaction> vtx;

    // memory only
    mutable std::vector<uint256> vMerkleTree;


    CBlock()
    {
        SetNull();
    }

    IMPLEMENT_SERIALIZE
    (
        READWRITE(this->blk.nVersion);
        nVersion = this->blk.nVersion;
        READWRITE(blk.hashPrevBlock);
        READWRITE(blk.hashMerkleRoot);
        READWRITE(blk.nBlockNum);
        READWRITE(blk.nTime);
        READWRITE(blk.nNonce1);
        READWRITE(blk.nNonce2);
        READWRITE(blk.nNonce3);
        READWRITE(blk.nNonce4);
        READWRITE(FLATDATA(blk.miner_id));
        READWRITE(blk.dwBits);

        // ConnectBlock depends on vtx being last so it can calculate offset
        if (!(nType & (SER_GETHASH|SER_BLOCKHEADERONLY)))   READWRITE(vtx);
        else if (fRead)                                     const_cast<CBlock*>(this)->vtx.clear();
    )

    void SetNull()
    {
        blk.nVersion = 1;
        blk.hashPrevBlock = 0;
        blk.hashMerkleRoot = 0;
        blk.nBlockNum = 0;
        blk.nTime = 0;
        blk.dwBits = 0;
        blk.nNonce1 = 0;
        blk.nNonce2 = 0;
        blk.nNonce3 = 0;
        blk.nNonce4 = 0;
        memset(blk.miner_id,0,12);
        vtx.clear();
        vMerkleTree.clear();
        m_bPower=false;
    }

    bool IsNull() const
    {
        return (blk.dwBits == 0);
    }

    uint256 GetHash() const
    {
        uint256 final_hash;
        BlockHash_1(final_hash,(unsigned char*)&blk);
        return final_hash;
    }

    int GetSigOpCount() const
    {
        int n = 0;
        BOOST_FOREACH(const CTransaction& tx, vtx)
            n += tx.GetSigOpCount();
        return n;
    }


    uint256 BuildMerkleTree() const
    {
        vMerkleTree.clear();
        BOOST_FOREACH(const CTransaction& tx, vtx)
            vMerkleTree.push_back(tx.GetHash());
        int j = 0;
        for (int nSize = vtx.size(); nSize > 1; nSize = (nSize + 1) / 2)
        {
            for (int i = 0; i < nSize; i += 2)
            {
                int i2 = std::min(i+1, nSize-1);
                vMerkleTree.push_back(Hash(BEGIN(vMerkleTree[j+i]),  END(vMerkleTree[j+i]),
                                           BEGIN(vMerkleTree[j+i2]), END(vMerkleTree[j+i2])));
            }
            j += nSize;
        }
        return (vMerkleTree.empty() ? 0 : vMerkleTree.back());
    }

    std::vector<uint256> GetMerkleBranch(int nIndex) const
    {
        if (vMerkleTree.empty())
            BuildMerkleTree();
        std::vector<uint256> vMerkleBranch;
        int j = 0;
        for (int nSize = vtx.size(); nSize > 1; nSize = (nSize + 1) / 2)
        {
            int i = std::min(nIndex^1, nSize-1);
            vMerkleBranch.push_back(vMerkleTree[j+i]);
            nIndex >>= 1;
            j += nSize;
        }
        return vMerkleBranch;
    }

    static uint256 CheckMerkleBranch(uint256 hash, const std::vector<uint256>& vMerkleBranch, int nIndex)
    {
        if (nIndex == -1)
            return 0;
        BOOST_FOREACH(const uint256& otherside, vMerkleBranch)
        {
            if (nIndex & 1)
                hash = Hash(BEGIN(otherside), END(otherside), BEGIN(hash), END(hash));
            else
                hash = Hash(BEGIN(hash), END(hash), BEGIN(otherside), END(otherside));
            nIndex >>= 1;
        }
        return hash;
    }


    bool WriteToDisk(unsigned int& nFileRet, unsigned int& nBlockPosRet)
    {
        // Open history file to append
        CAutoFile fileout = AppendBlockFile(nFileRet);
        if (!fileout)
            return error("CBlock::WriteToDisk() : AppendBlockFile failed");

        // Write index header
        unsigned int nSize = fileout.GetSerializeSize(*this);
        fileout << FLATDATA(pchMessageStart) << nSize;

        // Write block
        nBlockPosRet = ftell(fileout);
        if (nBlockPosRet == -1)
            return error("CBlock::WriteToDisk() : ftell failed");
        fileout << *this;

        // Flush stdio buffers and commit to disk before returning
        fflush(fileout);
        if (!Block_IsInitialDownload() || (g_qBlockBestHeight+1) % 500 == 0)
        {
#ifdef __WXMSW__
            _commit(_fileno(fileout));
#else
            fsync(fileno(fileout));
#endif
        }

        return true;
    }

    bool ReadFromDisk(unsigned int nFile, unsigned int nBlockPos, bool fReadTransactions=true)
    {
        SetNull();

        // Open history file to read
        CAutoFile filein = OpenBlockFile(nFile, nBlockPos, "rb");
        if (!filein)
            return error("CBlock::ReadFromDisk() : OpenBlockFile failed");
        if (!fReadTransactions)
            filein.nType |= SER_BLOCKHEADERONLY;

        // Read block
        filein >> *this;

        // Check the header
        if (!Block_CheckProofOfWork(GetHash(), blk.dwBits))
            return error("CBlock::ReadFromDisk() : errors in block header");

        return true;
    }



    void print() const
    {
        printf("CBlock(hash=%s, ver=%d, hashPrevBlock=%s, hashMerkleRoot=%s, nTime=%u, nBits=%08x, nBlockNum=%"PRI64d", nNonce1=%"PRI64d", nNonce2=%"PRI64d" vtx=%d)\n",
            GetHash().ToString().substr(0,20).c_str(),
            blk.nVersion,
            blk.hashPrevBlock.ToString().substr(0,20).c_str(),
            blk.hashMerkleRoot.ToString().substr(0,10).c_str(),
            (int)blk.nTime, blk.dwBits,blk.nBlockNum, blk.nNonce1,blk.nNonce2,
            vtx.size());
        for (int i = 0; i < vtx.size(); i++)
        {
            printf("  ");
            vtx[i].print();
        }
        printf("  vMerkleTree: ");
        for (int i = 0; i < vMerkleTree.size(); i++)
            printf("%s ", vMerkleTree[i].ToString().substr(0,10).c_str());
        printf("\n");
    }


    bool DisconnectBlock(CTxDB& txdb, CBlockIndex* pindex);
    bool ConnectBlock(CTxDB& txdb, CBlockIndex* pindex, bool bAllowOldBlockChecking);
    bool ReadFromDisk(const CBlockIndex* pindex, bool fReadTransactions=true);
    bool SetBestChain(CTxDB& txdb, CBlockIndex* pindexNew);
    bool AddToBlockIndex(unsigned int nFile, unsigned int nBlockPos);
    bool VerifyBlock() const;
    void CalculateChainWork(CBlockIndex *pIndex);
    bool AcceptBlock();
};

// The block chain is a tree shaped structure starting with the
// genesis block at the root, with each block potentially having multiple
// candidates to be the next block.  pprev and pnext link a path through the
// main/longest chain.  A blockindex may have multiple pprev pointing back
// to it, but pnext will only point forward to the longest branch, or will
// be null if the block is not part of the longest chain.
//
class CBlockIndex
{
public:
    const uint256* phashBlock;
    CBlockIndex* pprev;
    CBlockIndex* pnext;
    unsigned int nFile;
    unsigned int nBlockPos;
    CBigNum bnChainWork;
    BLOCK_DATA blk;

    CBlockIndex();
    CBlockIndex(unsigned int nFileIn, unsigned int nBlockPosIn, CBlock& block);
    CBlock GetBlockHeader() const;
    uint256 GetBlockHash() const;
    bool IsInMainChain() const;
    bool CheckIndex() const;
    bool EraseBlockFromDisk();
    std::string ToString() const;
    void print() const;
};



//
// Used to marshal pointers into hashes for db storage.
//
class CDiskBlockIndex : public CBlockIndex
{
public:
    uint256 m_hashPrev;
    uint256 m_hashNext;

    CDiskBlockIndex()
    {
        m_hashPrev = 0;
        m_hashNext = 0;
    }

    explicit CDiskBlockIndex(CBlockIndex* pindex) : CBlockIndex(*pindex)
    {
        m_hashPrev = (pprev ? pprev->GetBlockHash() : 0);
        m_hashNext = (pnext ? pnext->GetBlockHash() : 0);
    }

    IMPLEMENT_SERIALIZE
    (
        if (!(nType & SER_GETHASH))
            READWRITE(nVersion);

        READWRITE(m_hashNext);
        READWRITE(nFile);
        READWRITE(nBlockPos);

        // block header
        READWRITE(this->blk.nVersion);
        READWRITE(m_hashPrev);
        READWRITE(blk.hashMerkleRoot);
        READWRITE(blk.nBlockNum);
        READWRITE(blk.nTime);
        READWRITE(blk.nNonce1);
        READWRITE(blk.nNonce2);
        READWRITE(blk.nNonce3);
        READWRITE(blk.nNonce4);
        READWRITE(FLATDATA(blk.miner_id));
        READWRITE(blk.dwBits);
    )

    uint256 GetBlockHash() const
    {
        CBlock block;
        block.blk=blk;
        block.blk.hashPrevBlock   = m_hashPrev;
        return block.GetHash();
    }


    std::string ToString() const
    {
        std::string str = "CDiskBlockIndex(";
        str += CBlockIndex::ToString();
        str += strprintf("\n                hashBlock=%s, hashPrev=%s, hashNext=%s)",
            GetBlockHash().ToString().c_str(),
            m_hashPrev.ToString().substr(0,20).c_str(),
            m_hashNext.ToString().substr(0,20).c_str());
        return str;
    }

    void print() const
    {
        printf("%s\n", ToString().c_str());
    }
};








//
// Describes a place in the block chain to another node such that if the
// other node doesn't have the same branch, it can find a recent common trunk.
// The further back it is, the further before the fork it may be.
//
class CBlockLocator
{
protected:
    std::vector<uint256> vHave;
public:

    CBlockLocator();
    explicit CBlockLocator(const CBlockIndex* pindex);
    explicit CBlockLocator(uint256 hashBlock);

    IMPLEMENT_SERIALIZE
    (
        if (!(nType & SER_GETHASH))
            READWRITE(nVersion);
        READWRITE(vHave);
    )

    void SetNull();
    bool IsNull();
    void Set(const CBlockIndex* pindex);
    int GetDistanceBack();
    CBlockIndex* GetBlockIndex();
    uint256 GetBlockHash();
};




#endif
