// Copyright (c) 2011 RealSolid. Please read license.txt for applicable licensing.

#include "headers.h"
#include "db.h"
#include "net.h"
#include "init.h"
#include "cryptopp/sha.h"
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>

extern CCriticalSection cs_mapTransactions;
extern std::map<COutPoint, CInPoint> mapNextTx;
extern CScript g_TrustedScript[TRUST_FUND_NUM];
extern CScript g_GenScript;
extern CSolidCoinAddress g_TrustPubKeysHash[TRUST_FUND_NUM];
extern char *g_TrustPubKeys[TRUST_FUND_NUM];

bool CTransaction::ReadFromDisk(CTxDB& txdb, COutPoint prevout, CTxIndex& txindexRet)
{
    SetNull();
    if (!txdb.ReadTxIndex(prevout.hash, txindexRet))    return false;
    if (!ReadFromDisk(txindexRet.pos))                  return false;
    if (prevout.n >= vout.size())
    {
        SetNull();
        return false;
    }
    return true;
}

bool CTransaction::ReadFromDisk(CTxDB& txdb, COutPoint prevout)
{
    CTxIndex txindex;
    return ReadFromDisk(txdb, prevout, txindex);
}

bool CTransaction::ReadFromDisk(COutPoint prevout)
{
    CTxDB txdb("r");
    CTxIndex txindex;
    return ReadFromDisk(txdb, prevout, txindex);
}



int64 CMerkleTx::SetMerkleBranch(const CBlock* pblock)
{
    if (fClient)
    {
        if (hashBlock == 0)
            return 0;
    }
    else
    {
        CBlock blockTmp;
        if (pblock == NULL)
        {
            // Load the block this tx is in
            CTxIndex txindex;
            if (!CTxDB("r").ReadTxIndex(GetHash(), txindex))
                return 0;
            if (!blockTmp.ReadFromDisk(txindex.pos.nFile, txindex.pos.nBlockPos))
                return 0;
            pblock = &blockTmp;
        }

        // Update the tx's hashBlock
        hashBlock = pblock->GetHash();

        // Locate the transaction
        for (nIndex = 0; nIndex < pblock->vtx.size(); nIndex++)
            if (pblock->vtx[nIndex] == *(CTransaction*)this)
                break;
        if (nIndex == pblock->vtx.size())
        {
            vMerkleBranch.clear();
            nIndex = -1;
            printf("ERROR: SetMerkleBranch() : couldn't find tx in block\n");
            return 0;
        }

        // Fill in merkle branch
        vMerkleBranch = pblock->GetMerkleBranch(nIndex);
    }

    // Is the tx in a block that's in the main chain
    std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hashBlock);
    if (mi == g_BlockIndexMap.end())
        return 0;
    CBlockIndex* pindex = (*mi).second;
    if (!pindex || !pindex->IsInMainChain())
        return 0;

    return g_pBlockBestIndex->blk.nBlockNum - pindex->blk.nBlockNum + 1;
}

bool CTransaction::CheckTransaction(int nBlockHeight) const
{
    // Basic checks that don't depend on any context
    if (vin.empty() || vout.empty())
        return error("CTransaction::CheckTransaction() : vin or vout empty");

    unsigned int dwTXSize=::GetSerializeSize(*this, SER_NETWORK);

    // Size limits
    if (::GetSerializeSize(*this, SER_NETWORK) > MAX_BLOCK_SIZE)
        return error("CTransaction::CheckTransaction() : size limits failed");

    if(dwTXSize > MAX_BLOCK_SIZE)                   return error("CTransaction::CheckTransaction() : block size limits failed");
    //if(nBlockHeight>28500 && dwTXSize>MAX_TX_SIZE)  return error("CTransaction::CheckTransaction() : tx size limits failed");

    // Check for negative or overflow output values
    int64 nValueOut = 0;
    BOOST_FOREACH(const CTxOut& txout, vout)
    {
        if (txout.nValue <= 0)      return error("CTransaction::CheckTransaction() : txout.nValue negative or 0");
        nValueOut += txout.nValue;
        if (nValueOut<=0) return error("CTransaction::CheckTransaction() : txout total out of range");
    }

    // Check for duplicate inputs
    std::set<COutPoint> vInOutPoints;
    BOOST_FOREACH(const CTxIn& txin, vin)
    {
        if (vInOutPoints.count(txin.prevout))
            return false;
        vInOutPoints.insert(txin.prevout);
    }

    if (IsCoinBase())
    {
        if (vin[0].scriptSig.size() < 2 || vin[0].scriptSig.size() > 100)
            return error("CTransaction::CheckTransaction() : coinbase script size");
    }
    else
    {
        BOOST_FOREACH(const CTxIn& txin, vin)
            if (txin.prevout.IsNull())
                return error("CTransaction::CheckTransaction() : prevout is null");
    }

    return true;
}

bool CTransaction::AcceptToMemoryPool(int nBlockHeight, CTxDB& txdb, bool fCheckInputs, bool* pfMissingInputs)
{
    if (pfMissingInputs)    *pfMissingInputs = false;

    if (!CheckTransaction(nBlockHeight))
        return error("AcceptToMemoryPool() : CheckTransaction failed");

    // Coinbase is only valid in a block, not as a loose transaction
    if (IsCoinBase())
        return error("AcceptToMemoryPool() : coinbase as individual tx");

    // To help v0.1.5 clients who would see it as a negative number
    if ((int64)nLockTime > INT_MAX)
        return error("AcceptToMemoryPool() : not accepting nLockTime beyond 2038 yet");

    // Safety limits
    unsigned int nSize = ::GetSerializeSize(*this, SER_NETWORK);
    // Checking ECDSA signatures is a CPU bottleneck, so to avoid denial-of-service
    // attacks disallow transactions with more than one SigOp per 34 bytes.
    // 34 bytes because a TxOut is:
    //   20-byte address + 8 byte solidcoin amount + 5 bytes of ops + 1 byte script length
    if (GetSigOpCount() > nSize / 34 || nSize < 100)
        return error("AcceptToMemoryPool() : nonstandard transaction");

    // Rather not work on nonstandard transactions (unless -testnet)
    if (!fTestNet && !IsStandard())
        return error("AcceptToMemoryPool() : nonstandard transaction type");

    // Do we already have it?
    uint256 hash = GetHash();
    {
        MUTEX_LOCK(cs_mapTransactions);
        if (mapTransactions.count(hash))    return false;
    }
    if (fCheckInputs)
        if (txdb.ContainsTx(hash))
            return false;

    // Check for conflicts with in-memory transactions
    CTransaction* ptxOld = NULL;
    for (int i = 0; i < vin.size(); i++)
    {
        COutPoint outpoint = vin[i].prevout;
        if (mapNextTx.count(outpoint))
        {
            // Disable replacement feature for now
            return false;

            // Allow replacing with a newer version of the same transaction
            if (i != 0) return false;
            ptxOld = mapNextTx[outpoint].ptx;
            if (ptxOld->IsFinal())      return false;
            if (!IsNewerThan(*ptxOld))  return false;
            for (int i = 0; i < vin.size(); i++)
            {
                COutPoint outpoint = vin[i].prevout;
                if (!mapNextTx.count(outpoint) || mapNextTx[outpoint].ptx != ptxOld)    return false;
            }
            break;
        }
    }

    if (fCheckInputs)
    {
        // Check against previous transactions
        std::map<uint256, CTxIndex> mapUnused;
        int64 nFees = 0;
        if (!ConnectInputs(txdb, mapUnused, CDiskTxPos(1,1,1), g_pBlockBestIndex, nFees, false, false,false))
        {
            if (pfMissingInputs)    *pfMissingInputs = true;
            return error("AcceptToMemoryPool() : ConnectInputs failed %s", hash.ToString().substr(0,10).c_str());
        }

        // Don't accept it if it can't get into a block
        if (nFees < GetMinFee())
        {
            return error("AcceptToMemoryPool() : not enough fees");
        }
    }

    // Store transaction in memory

    {
        MUTEX_LOCK(cs_mapTransactions);
        if (ptxOld)
        {
            printf("AcceptToMemoryPool() : replacing tx %s with new version\n", ptxOld->GetHash().ToString().c_str());
            ptxOld->RemoveFromMemoryPool();
        }
        AddToMemoryPoolUnchecked();
    }

    ///// are we sure this is ok when loading transactions or restoring block txes
    // If updated, erase old tx from wallet
    if (ptxOld)
    {
        LOCK_WALLET_ACCESS();
        Wallet_Erase(ptxOld->GetHash());
    }

    printf("AcceptToMemoryPool(): accepted %s\n", hash.ToString().substr(0,10).c_str());
    return true;
}

bool CTransaction::AcceptToMemoryPool(int blockheight, bool fCheckInputs, bool* pfMissingInputs)
{
    CTxDB txdb("r");
    return AcceptToMemoryPool(blockheight,txdb, fCheckInputs, pfMissingInputs);
}

bool CTransaction::AddToMemoryPoolUnchecked()
{
    // Add to memory pool without checking anything.  Don't call this directly,
    // call AcceptToMemoryPool to properly check the transaction first.
    MUTEX_LOCK(cs_mapTransactions);
    uint256 hash = GetHash();
    mapTransactions[hash] = *this;
    for (int i = 0; i < vin.size(); i++)    mapNextTx[vin[i].prevout] = CInPoint(&mapTransactions[hash], i);
    nTransactionsUpdated++;
    return true;
}


bool CTransaction::RemoveFromMemoryPool()
{
    // Remove transaction from memory pool
    MUTEX_LOCK(cs_mapTransactions);

    BOOST_FOREACH(const CTxIn& txin, vin)
    {
        mapNextTx.erase(txin.prevout);
    }
    mapTransactions.erase(GetHash());
    nTransactionsUpdated++;
    return true;
}

int64 CMerkleTx::GetDepthInMainChain(int64& nHeightRet) const
{
    if (hashBlock == 0 || nIndex == -1) return 0;

    // Find the block it claims to be in
    std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hashBlock);
    if (mi == g_BlockIndexMap.end())
        return 0;
    CBlockIndex* pindex = (*mi).second;
    if (!pindex || !pindex->IsInMainChain())
        return 0;

    // Make sure the merkle branch connects to this block
    if (!fMerkleVerified)
    {
        if (CBlock::CheckMerkleBranch(GetHash(), vMerkleBranch, nIndex) != pindex->blk.hashMerkleRoot)
            return 0;
        fMerkleVerified = true;
    }

    nHeightRet = pindex->blk.nBlockNum;
    return g_pBlockBestIndex->blk.nBlockNum - pindex->blk.nBlockNum + 1;
}

int64 CMerkleTx::GetBlocksToMaturity() const
{
    if (!IsCoinBase())  return 0;
    int64 nDiff=(COINBASE_MATURITY+2) - GetDepthInMainChain();
    if(nDiff>0) return nDiff;
    return 0;
}


bool CMerkleTx::AcceptToMemoryPool(CTxDB& txdb, bool fCheckInputs)
{
    int64 nBlockHeight=0;
    GetDepthInMainChain(nBlockHeight);
    if(nBlockHeight==0) nBlockHeight=g_qBlockBestHeight;

    if (fClient)
    {
        if (!IsInMainChain() && !ClientConnectInputs()) return false;
        return CTransaction::AcceptToMemoryPool(nBlockHeight,txdb, false);
    }
    else
    {
        return CTransaction::AcceptToMemoryPool(nBlockHeight,txdb, fCheckInputs);
    }
}

bool CMerkleTx::AcceptToMemoryPool()
{
    CTxDB txdb("r");
    return AcceptToMemoryPool(txdb);
}



bool CWalletTx::AcceptWalletTransaction(CTxDB& txdb, bool fCheckInputs)
{
    MUTEX_LOCK(cs_mapTransactions);

    // Add previous supporting transactions first
    BOOST_FOREACH(CMerkleTx& tx, vtxPrev)
    {
        if (!tx.IsCoinBase())
        {
            uint256 hash = tx.GetHash();
            if (!mapTransactions.count(hash) && !txdb.ContainsTx(hash))
                tx.AcceptToMemoryPool(txdb, fCheckInputs);
        }
    }
    return AcceptToMemoryPool(txdb, fCheckInputs);
}

bool CWalletTx::AcceptWalletTransaction()
{
    CTxDB txdb("r");
    return AcceptWalletTransaction(txdb);
}

CBlockIndex* CTxIndex::GetBlockIndex()
{
    CBlock prevTXBlock;
    if(prevTXBlock.ReadFromDisk(pos.nFile, pos.nBlockPos,false))
    {
        std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(prevTXBlock.GetHash());
        if (mi != g_BlockIndexMap.end())
        {
            return (*mi).second;
        }
    }
    return 0;
}

int64 CTxIndex::GetDepthInMainChain() const
{
    // Read block header
    CBlock block;
    if (!block.ReadFromDisk(pos.nFile, pos.nBlockPos, false))
        return 0;
    // Find the block in the index
    std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(block.GetHash());
    if (mi == g_BlockIndexMap.end())
        return 0;
    CBlockIndex* pindex = (*mi).second;
    if (!pindex || !pindex->IsInMainChain())    return 0;
    return 1 + g_qBlockBestHeight - pindex->blk.nBlockNum;
}



bool CTransaction::DisconnectInputs(CTxDB& txdb)
{
    // Relinquish previous transactions' spent pointers
    if (!IsCoinBase())
    {
        BOOST_FOREACH(const CTxIn& txin, vin)
        {
            COutPoint prevout = txin.prevout;   // Get prev txindex from disk
            CTxIndex txindex;
            if (!txdb.ReadTxIndex(prevout.hash, txindex))   return error("DisconnectInputs() : ReadTxIndex failed");
            if (prevout.n >= txindex.vSpent.size())         return error("DisconnectInputs() : prevout.n out of range");
            txindex.vSpent[prevout.n].SetNull();    // Mark outpoint as not spent
            if (!txdb.UpdateTxIndex(prevout.hash, txindex)) return error("DisconnectInputs() : UpdateTxIndex failed");  // Write back
        }
    }
    // Remove transaction from index
    if (!txdb.EraseTxIndex(*this))  return error("DisconnectInputs() : EraseTxPos failed");
    return true;
}


bool CTransaction::ConnectInputs(CTxDB& txdb, std::map<uint256, CTxIndex>& mapTestPool, CDiskTxPos posThisTx,CBlockIndex* pindexBlock, int64& nFees, bool fBlock, bool fMiner, bool bTrustedTX)
{
    // Take over previous transactions' spent pointers
    if (!IsCoinBase())
    {
        int64 nValueIn = 0;
        for (int i = 0; i < vin.size(); i++)
        {
            COutPoint prevout = vin[i].prevout;
            printf("connect inputs prevout: ");
            prevout.print();

            // Read txindex
            CTxIndex txindex;
            bool fFound = true;
            if ((fBlock || fMiner) && mapTestPool.count(prevout.hash))
            {
                txindex = mapTestPool[prevout.hash];    // Get txindex from current proposed changes
            }
            else
            {
                fFound = txdb.ReadTxIndex(prevout.hash, txindex);   // Read txindex from txdb
            }
            if (!fFound && (fBlock || fMiner))
                return fMiner ? false : error("ConnectInputs() : %s prev tx %s index entry not found", GetHash().ToString().substr(0,10).c_str(),  prevout.hash.ToString().substr(0,10).c_str());

            // Read txPrev
            CTransaction txPrev;
            if (!fFound || txindex.pos == CDiskTxPos(1,1,1))
            {
                // Get prev tx from single transactions in memory
                {
                    MUTEX_LOCK(cs_mapTransactions);
                    if (!mapTransactions.count(prevout.hash))   return error("ConnectInputs() : %s mapTransactions prev not found %s", GetHash().ToString().substr(0,10).c_str(),  prevout.hash.ToString().substr(0,10).c_str());
                    txPrev = mapTransactions[prevout.hash];
                }
                if (!fFound)    txindex.vSpent.resize(txPrev.vout.size());
            }
            else
            {
                // Get prev tx from disk
                if (!txPrev.ReadFromDisk(txindex.pos))
                    return error("ConnectInputs() : %s ReadFromDisk prev tx %s failed", GetHash().ToString().substr(0,10).c_str(),  prevout.hash.ToString().substr(0,10).c_str());
            }

            if (prevout.n >= txPrev.vout.size() || prevout.n >= txindex.vSpent.size())
                return error("ConnectInputs() : %s prevout.n out of range %d %d %d prev tx %s\n%s", GetHash().ToString().substr(0,10).c_str(), prevout.n, txPrev.vout.size(), txindex.vSpent.size(), prevout.hash.ToString().substr(0,10).c_str(), txPrev.ToString().c_str());

            if(bTrustedTX)
            {
                if(!::IsStandard(txPrev.vout[prevout.n].scriptPubKey))              return error("ConnectInputs() : trusted tx isnt standard");
                if(txPrev.vout[prevout.n].nValue<(TRUST_FUND_AMOUNT*COIN))          return error("ConnectInputs() : not enough SC for a trusted block");
                if(txPrev.vout[prevout.n].scriptPubKey != vout[0].scriptPubKey)     return error("ConnectInputs() : trusted tx pubkey does not match previous");
                int64 valDiff = txPrev.vout[prevout.n].nValue-vout[0].nValue;
                int64 blockValue;
                if(pindexBlock->blk.nBlockNum<42000 || (pindexBlock->blk.nBlockNum%2)==1)
                {
                    blockValue = Block_GetCoinBaseValue(pindexBlock->blk.dwBits, pindexBlock->blk.nBlockNum);
                }
                else
                {
                    //proper way to get trusted generate value, use last blocks work rate to find real 5%
                    blockValue = Block_GetCoinBaseValue(pindexBlock->pprev->blk.dwBits, pindexBlock->blk.nBlockNum);
                }
                if(valDiff<blockValue)                                              return error("ConnectInputs() : trusted tx payment less than CPF");
            }
            else
            {
                CSolidCoinAddress addr=txPrev.vout[prevout.n].scriptPubKey.GetSolidCoinAddress();
                if(addr.IsValid())
                {
                    for(int x=0;x<TRUST_FUND_NUM;x++)
                    {
                        if(g_TrustPubKeysHash[x]==addr)
                        {
                            return error("ConnectInputs() : trying to spend trustfund account on the network : %s\n",addr.ToString().c_str());
                        }
                    }
                }
                else
                {
                    std::string txstr = txPrev.vout[prevout.n].scriptPubKey.ToString();
                    for(int x=0;x<TRUST_FUND_NUM;x++)
                    {
                        if(txstr.find(g_TrustPubKeys[x])!=std::string::npos)
                        {
                            return error("ConnectInputs() : trying to spend trustfund account on the network : %s\n",txstr.c_str());
                        }
                    }
                }
            }

            // If prev is coinbase, check that it's matured
            if (txPrev.IsCoinBase())
            {
                bool bTrusted=false;

                //to get around the first ~10 blocks needing to use the trust generates we check here and allow them to be used before maturity
                if(pindexBlock->blk.nBlockNum<=COINBASE_MATURITY)
                {
                    for(int x=0;x<TRUST_FUND_NUM;x++)
                    {
                        if(txPrev.vout[0].scriptPubKey==g_TrustedScript[x])
                        {
                            bTrusted=true;
                            break;
                        }
                    }
                    if(txPrev.vout[0].scriptPubKey==g_GenScript)   bTrusted=true;
                }
                if(!bTrusted)
                {
                    for (CBlockIndex* pindex = pindexBlock; pindex && pindexBlock->blk.nBlockNum - pindex->blk.nBlockNum < COINBASE_MATURITY; pindex = pindex->pprev)
                        if (pindex->nBlockPos == txindex.pos.nBlockPos && pindex->nFile == txindex.pos.nFile)
                            return error("ConnectInputs() : tried to spend coinbase at depth %"PRI64d"", pindexBlock->blk.nBlockNum - pindex->blk.nBlockNum);
                }

                //check ntime of block to see if generate was in a block with malformed time, deny the use of the generate if so
                //if the block the generate was came 1 or 2 seconds after the previous, and trusted block is more than 40s off from trusted time
                //things to test, that the generate is from 20000+
                if(pindexBlock->blk.nBlockNum>=40000)
                {
                    CBlockIndex *pPrevBlockIndex=txindex.GetBlockIndex();
                    if (pPrevBlockIndex && pPrevBlockIndex->IsInMainChain() && pPrevBlockIndex->blk.nBlockNum>30000 && (pPrevBlockIndex->blk.nBlockNum%2)==1 && pPrevBlockIndex->pprev && pPrevBlockIndex->pnext)
                    {
                        if( pPrevBlockIndex->blk.nTime-pPrevBlockIndex->pprev->blk.nTime<3 &&  (pPrevBlockIndex->pnext->blk.nTime-pPrevBlockIndex->blk.nTime ) > 40)
                        {
                            return error("ConnectInputs() : tried to spend coinbase before maturity");
                        }
                    }
                }
            }

            if (!VerifySignature(txPrev, *this, i)) return error("ConnectInputs() : %s VerifySignature failed", GetHash().ToString().substr(0,10).c_str());

            // Check for conflicts
            if (!txindex.vSpent[prevout.n].IsNull())    return fMiner ? false : error("ConnectInputs() : %s prev tx already used at %s", GetHash().ToString().substr(0,10).c_str(), txindex.vSpent[prevout.n].ToString().c_str());

            // Check for negative or overflow input values
            nValueIn += txPrev.vout[prevout.n].nValue;
            if (txPrev.vout[prevout.n].nValue<=0 || nValueIn<=0)    return error("ConnectInputs() : txin values out of range");

            txindex.vSpent[prevout.n] = posThisTx;  // Mark outpoints as spent

            // Write back
            if (fBlock || fMiner)
            {
                mapTestPool[prevout.hash] = txindex;
            }
        }

        if (nValueIn < GetValueOut())   return error("ConnectInputs() : %s value in < value out", GetHash().ToString().substr(0,10).c_str());

        // Tally transaction fees
        int64 nTxFee = nValueIn - GetValueOut();
        if (nTxFee < 0)         return error("ConnectInputs() : %s nTxFee < 0", GetHash().ToString().substr(0,10).c_str());
        if (nTxFee < GetMinFee())   return false;
        nFees += nTxFee;
        if (nFees<=0) return error("ConnectInputs() : nFees out of range");
    }

    if (fBlock)
    {
       // Add transaction to changes
        mapTestPool[GetHash()] = CTxIndex(posThisTx, vout.size());
    }
    else if (fMiner)
    {
        // Add transaction to test pool
        mapTestPool[GetHash()] = CTxIndex(CDiskTxPos(1,1,1), vout.size());
    }

    return true;
}


bool CTransaction::ClientConnectInputs()
{
    if (IsCoinBase())
        return false;

    // Take over previous transactions' spent pointers
    MUTEX_LOCK(cs_mapTransactions);
    int64 nValueIn = 0;
    for (int i = 0; i < vin.size(); i++)
    {
        // Get prev tx from single transactions in memory
        COutPoint prevout = vin[i].prevout;
        if (!mapTransactions.count(prevout.hash))
            return false;
        CTransaction& txPrev = mapTransactions[prevout.hash];

        if (prevout.n >= txPrev.vout.size())
            return false;

        // Verify signature
        if (!VerifySignature(txPrev, *this, i))
            return error("ConnectInputs() : VerifySignature failed");

        ///// this is redundant with the mapNextTx stuff, not sure which I want to get rid of
        ///// this has to go away now that posNext is gone
        // // Check for conflicts
        // if (!txPrev.vout[prevout.n].posNext.IsNull())
        //     return error("ConnectInputs() : prev tx already used");
        //
        // // Flag outpoints as used
        // txPrev.vout[prevout.n].posNext = posThisTx;

        nValueIn += txPrev.vout[prevout.n].nValue;

        if (txPrev.vout[prevout.n].nValue<=0 || nValueIn<=0)
            return error("ClientConnectInputs() : txin values out of range");
    }
    if (GetValueOut() > nValueIn)   return false;

    return true;
}

