// Copyright (c) 2011 RealSolid. Please read license.txt for applicable licensing.

#include "headers.h"
#include "cryptopp/sha.h"
#include "db.h"
#include "net.h"
#include "init.h"
#include "wallet_func.h"
#undef printf
#include <boost/asio.hpp>
#include <boost/iostreams/concepts.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/algorithm/string.hpp>
#ifdef USE_SSL
#include <boost/asio/ssl.hpp>
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>
typedef boost::asio::ssl::stream<boost::asio::ip::tcp::socket> SSLStream;
#endif
#include "json/json_spirit_reader_template.h"
#include "json/json_spirit_writer_template.h"
#include "json/json_spirit_utils.h"
#define printf OutputDebugStringF
// MinGW 3.4.5 gets "fatal error: had to relocate PCH" if the json headers are
// precompiled in headers.h.  The problem might be when the pch file goes over
// a certain size around 145MB.  If we need access to json_spirit outside this
// file, we could use the compiled json_spirit option.

using namespace std;
using namespace boost;
using namespace boost::asio;
using namespace json_spirit;

void ThreadRPCServer2(void* parg);
void Mining_AddPool(MINING_POOL *pPool);
void Mining_RemovePool(int id);
typedef Value(*rpcfn_type)(const Array& params, bool fHelp);
extern map<string, rpcfn_type> mapCallTable;
extern CCriticalSection cs_main;

extern CCriticalSection cs_mapTransactions;



int64 g_MiningShareTargetInt=0;
int64 g_MiningTestWorkShares=0;
int64 g_MiningTestWorkInvalids=0;

int64 g_qShareTimeWindow=8;
uint256 g_MiningShareTargetHash;
string g_MiningIDString;
vector<CBlock*> g_NewBlockList;
static map<uint256, pair<CBlock*, uint32> > g_mapNewBlock;
CCriticalSection cs_getwork;
CReserveKey g_MiningReserveKey;


Object JSONRPCError(int code, const string& message)
{
    Object error;
    error.push_back(Pair("code", code));
    error.push_back(Pair("message", message));
    return error;
}


void PrintConsole(const char* format, ...)
{
    char buffer[50000];
    int limit = sizeof(buffer);
    va_list arg_ptr;
    va_start(arg_ptr, format);
    int ret = _vsnprintf(buffer, limit, format, arg_ptr);
    va_end(arg_ptr);
    if (ret < 0 || ret >= limit)
    {
        ret = limit - 1;
        buffer[limit-1] = 0;
    }
    printf("%s", buffer);
#if defined(__WXMSW__) && defined(GUI)
    MyMessageBox(buffer, "SolidCoin", wxOK | wxICON_EXCLAMATION);
#else
    fprintf(stdout, "%s", buffer);
#endif
}


int64 AmountFromValue(const Value& value)
{
    double dAmount = value.get_real();
    if (dAmount <= 0.0) throw JSONRPCError(-3, "Invalid amount");
    int64 nAmount = roundint64(dAmount * COIN);
    if (nAmount<=0) throw JSONRPCError(-3, "Invalid amount");
    return nAmount;
}

int64 AmountFromValueStr(const Value& value)
{
    int64 nAmount = boost::lexical_cast<int64>(value.get_str());
    if(nAmount<=0)   throw JSONRPCError(-3, "Invalid amount");
    return nAmount;
}

Value ValueFromAmount(int64 amount)
{
    return (double)amount / (double)COIN;
}

Value ValueFromAmountStr(int64 amount)
{
    string result = boost::lexical_cast<string>(amount);
    return result;
    //boost::format("%");
    //return (double)amount / (double)COIN;
}

void WalletTxToJSON(const CWalletTx& wtx, Object& entry)
{
    entry.push_back(Pair("confirmations", wtx.GetDepthInMainChain()));
    entry.push_back(Pair("txid", wtx.GetHash().GetHex()));
    entry.push_back(Pair("time", (boost::int64_t)wtx.GetTxTime()));
    BOOST_FOREACH(const PAIRTYPE(string,string)& item, wtx.mapValue)
        entry.push_back(Pair(item.first, item.second));
}

string AccountFromValue(const Value& value)
{
    string strAccount = value.get_str();
    if (strAccount == "*")
        throw JSONRPCError(-11, "Invalid account name");
    return strAccount;
}

Value help(const Array& params, bool fHelp)
{
    if (fHelp || params.size() > 1)
        throw runtime_error(
            "help [command]\n"
            "List commands, or get help for a command.");

    string strCommand;
    if (params.size() > 0)
        strCommand = params[0].get_str();

    string strRet;
    set<rpcfn_type> setDone;
    for (map<string, rpcfn_type>::iterator mi = mapCallTable.begin(); mi != mapCallTable.end(); ++mi)
    {
        string strMethod = (*mi).first;
        // We already filter duplicates, but these deprecated screw up the sort order
        if (strMethod == "getamountreceived" ||
            strMethod == "getallreceived" ||
            (strMethod.find("label") != string::npos))
            continue;
        if (strCommand != "" && strMethod != strCommand)
            continue;
        try
        {
            Array params;
            rpcfn_type pfn = (*mi).second;
            if (setDone.insert(pfn).second)
                (*pfn)(params, true);
        }
        catch (std::exception& e)
        {
            // Help text is returned in an exception
            string strHelp = string(e.what());
            if (strCommand == "")
                if (strHelp.find('\n') != -1)
                    strHelp = strHelp.substr(0, strHelp.find('\n'));
            strRet += strHelp + "\n";
        }
    }
    if (strRet == "")
        strRet = strprintf("help: unknown command: %s\n", strCommand.c_str());
    strRet = strRet.substr(0,strRet.size()-1);
    return strRet;
}


Value stop(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "stop\n"
            "Stop solidcoin server.");

    // Shutdown will take long enough that the response should get back
    CreateThread(Shutdown, NULL);
    return "solidcoin server stopping";
}

Value getblockcount(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "getblockcount\n"
            "Returns the number of blocks in the longest block chain.");

    return g_qBlockBestHeight;
}

Value getblocknumber(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "getblocknumber\n"
            "Returns the block number of the latest block in the longest block chain.");

    return g_qBlockBestHeight;
}

Value getconnectioncount(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "getconnectioncount\n"
            "Returns the number of connections to other nodes.");

    return (int)vNodes.size();
}


double GetDifficulty(uint32 dwBits)
{
    // Floating point number that is a multiple of the minimum difficulty minimum difficulty = 1.0.
    uint32 dwDiff1=g_bnBlockProofOfWorkLimit.GetCompact();
    if(dwBits==0)
    {
        if(g_pBlockBestIndex == NULL) return 1.0;
        if((g_pBlockBestIndex->blk.nBlockNum%2) == 0 )
        {
            if(g_pBlockBestIndex->pprev==0) return 1.0;
            dwBits = g_pBlockBestIndex->pprev->blk.dwBits;
        }
        else dwBits = g_pBlockBestIndex->blk.dwBits;
    }
    int nShift = (dwBits >> 24) & 0xff;
    double dDiff =  (double)(dwDiff1&0x00FFFFFF) / (double)(dwBits & 0x00ffffff);
    while (nShift < ((dwDiff1>>24)&0xFF) )
    {
        dDiff *= 256.0;
        nShift++;
    }
    while (nShift > ((dwDiff1>>24)&0xFF))
    {
        dDiff /= 256.0;
        nShift--;
    }
    return dDiff;
}

Value getdifficulty(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "getdifficulty\n"
            "Returns the proof-of-work difficulty as a multiple of the minimum difficulty.");

    return GetDifficulty(0);
}


Value sc_getmining(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "sc_getmining\n"
            "Returns status of all miners.");

    MUTEX_LOCK(cs_main);
    Array ret;

    static int64 qLastMilliTime=0;
    static int64 qLastTestworkShares=0;
    #define DIFF1   131072


    int64 qTime = GetTimeMillis()-qLastMilliTime;
    qLastMilliTime=GetTimeMillis();
    int64 qShares = (g_MiningTestWorkShares-qLastTestworkShares)*DIFF1;
    qLastTestworkShares=g_MiningTestWorkShares;

    Object testwork;

    testwork.push_back(Pair("testwork_totalhash",    (g_MiningTestWorkShares*DIFF1) ));
    testwork.push_back(Pair("testwork_invalids",    g_MiningTestWorkInvalids));
    testwork.push_back(Pair("testwork_hashpersec",   (int64)(qShares/((double)qTime/1000.0)) ));
    ret.push_back(testwork);


    for(int x=0;x<g_MiningPoolList.size();x++)
    {
        Object miner;
        MINING_POOL *pPool=&g_MiningPoolList[x];
        miner.push_back(Pair("name",       pPool->name));
        miner.push_back(Pair("running",    pPool->nRunning?true:false));
        miner.push_back(Pair("hashpersec",    pPool->qLastHashPerSec));
        miner.push_back(Pair("totalhash",    pPool->qTotalHash));
        miner.push_back(Pair("totalshares",    pPool->qTotalShares));
        ret.push_back(miner);
    }
    return ret;
}


Value sc_setmining(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 2)
        throw runtime_error(
            "sc_setmining <poolid> <on>\n"
            "<poolid> A number which specifies the pool to use as defined in solidcoin.conf\n"
            "<generate> is true or false to turn generation on or off.\n");


    int nPoolID = params[0].get_int();
    bool bON = params[1].get_bool();
    {
        MUTEX_LOCK(cs_main);
        if(nPoolID<0 || nPoolID>=g_MiningPoolList.size())  throw runtime_error("sc_setmining - incorrect pool ID.");
        MINING_POOL *pPool=&g_MiningPoolList[nPoolID];
        if(!bON && pPool->nRunning)         //close her down
        {
            Mining_RemovePool(pPool->id);
            pPool->nRunning=0;
            pPool->qLastHashPerSec=0;
            pPool->qTotalHash=0;
            pPool->qTotalShares=0;
        }
        else if(bON && !pPool->nRunning)    //start her up
        {
            pPool->id=nPoolID;
            Mining_AddPool(pPool);
            pPool->nRunning=1;
            pPool->qLastHashPerSec=0;
            pPool->qTotalHash=0;
            pPool->qTotalShares=0;
        }
    }
    UIThreadCall(boost::bind(CalledUpdateMiningUI));
    return Value::null;
}


Value real_getinfo(bool bSC, CWallet *pWallet, const Array& params, bool fHelp)
{
    Object obj;
    obj.push_back(Pair("version",       (int)VERSION));
    obj.push_back(Pair("balance",       ValueFromAmount(pWallet->GetBalance())));
    obj.push_back(Pair("blocks",        g_qBlockBestHeight));
    obj.push_back(Pair("connections",   (int)vNodes.size()));
    obj.push_back(Pair("proxy",         (g_bUseProxy ? addrProxy.ToStringIPPort() : string())));
    obj.push_back(Pair("generate",      (bool)false));
    obj.push_back(Pair("genproclimit",  (int)(-1)));
    obj.push_back(Pair("difficulty",    (double)GetDifficulty(0)));
    obj.push_back(Pair("testnet",       fTestNet));
    obj.push_back(Pair("keypoololdest", (boost::int64_t)pWallet->GetOldestKeyPoolTime()));
    obj.push_back(Pair("keypoolsize",   pWallet->GetKeyPoolSize()));
    obj.push_back(Pair("mintxfee",      ValueFromAmount(MIN_TX_FEE)));
    if (pWallet->IsCrypted())           obj.push_back(Pair("unlocked_until", (boost::int64_t)pWallet->m_nWalletLockTime));
    obj.push_back(Pair("errors",        GetWarnings("statusbar")));
    if(bSC)
    {
        obj.push_back(Pair("int_balance", ValueFromAmountStr(pWallet->GetBalance())));
        obj.push_back(Pair("int_minfee", ValueFromAmountStr(MIN_TX_FEE)));
    }
    return obj;
}

Value getinfo(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "getinfo\n"
            "Returns an object containing various state info.");
    LOCK_WALLET_ACCESS();
    return real_getinfo(false,Wallet_GetDefault(),params, fHelp);
}

Value sc_getinfo(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;

    if (fHelp || params.size() != 1 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getinfo <wallet>\n"
            "Returns an object containing various state info for the given wallet.");

    return real_getinfo(true,pWallet, params, fHelp);
}

Value getnewaddress(const Array& params, bool fHelp)
{
    if (fHelp || params.size() > 1)
        throw runtime_error(
            "getnewaddress [account]\n"
            "Returns a new solidcoin address for receiving payments.  "
            "If [account] is specified (recommended), it is added to the address book "
            "so payments received with the address will be credited to [account].");

    if(!Wallet_GetDefault()->IsLocked())           Wallet_GetDefault()->TopUpKeyPool();
    if(Wallet_GetDefault()->GetKeyPoolSize() < 1)  throw JSONRPCError(-12, "Error: Keypool ran out, please call keypoolrefill first");

    string strAccount;  // Parse the account first so we don't generate a key if there's an error
    if (params.size() > 0)  strAccount = AccountFromValue(params[0]);
    CSolidCoinAddress address(Wallet_GetDefault()->GetOrReuseKeyFromPool());    // Generate a new key that is added to wallet
    Wallet_GetDefault()->SetAddressBookName(address, strAccount);   // This could be done in the same main CS as GetKeyFromKeyPool.
    return address.ToString();
}

Value sc_getnewaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size()<1 || params.size() > 2 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "getnewaddress <wallet> [account]\n"
            "Returns a new solidcoin address for receiving payments from [wallet].  "
            "If [account] is specified (recommended), it is added to the address book "
            "so payments received with the address will be credited to [account].");

    if(!pWallet->IsLocked())           pWallet->TopUpKeyPool();
    if(pWallet->GetKeyPoolSize() < 1)  throw JSONRPCError(-12, "Error: Keypool ran out, please call keypoolrefill first");

    string strAccount;  // Parse the account first so we don't generate a key if there's an error
    if (params.size() > 1)  strAccount = AccountFromValue(params[1]);
    CSolidCoinAddress address(pWallet->GetOrReuseKeyFromPool());    // Generate a new key that is added to wallet
    pWallet->SetAddressBookName(address, strAccount);   // This could be done in the same main CS as GetKeyFromKeyPool.
    return address.ToString();
}


// requires cs_main, cs_mapWallet, cs_mapAddressBook locks
CSolidCoinAddress GetAccountAddress(CWallet *pWallet, string strAccount, bool bForceNew=false)
{
    CWalletDB walletdb(pWallet->strWalletFile);
    CAccount account;
    {
        walletdb.ReadAccount(strAccount, account);
        bool bKeyUsed = false;
        if (!account.vchPubKey.empty())     // Check if the current key has been used
        {
            CScript scriptPubKey;
            scriptPubKey.SetSolidCoinAddress(account.vchPubKey);
            for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin();
                 it != pWallet->mapWallet.end() && !account.vchPubKey.empty();
                 ++it)
            {
                const CWalletTx& wtx = (*it).second;
                BOOST_FOREACH(const CTxOut& txout, wtx.vout)
                    if (txout.scriptPubKey == scriptPubKey)
                        bKeyUsed = true;
            }
        }

        // Generate a new key
        if (account.vchPubKey.empty() || bForceNew || bKeyUsed)
        {
            if (pWallet->GetKeyPoolSize() < 1)
            {
                if (bKeyUsed || bForceNew)
                    throw JSONRPCError(-12, "Error: Keypool ran out, please call topupkeypool first");
            }
            else
            {
                account.vchPubKey = pWallet->GetOrReuseKeyFromPool();
                pWallet->SetAddressBookName(CSolidCoinAddress(account.vchPubKey), strAccount);
                walletdb.WriteAccount(strAccount, account);
            }
        }
    }
    return CSolidCoinAddress(account.vchPubKey);
}

Value getaccountaddress(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "getaccountaddress <account>\n"
            "Returns the current solidcoin address for receiving payments to this account.");

    string strAccount = AccountFromValue(params[0]);    // Parse the account first so we don't generate a key if there's an error
    Value ret;
    {
        MUTEX_LOCK(cs_main);
        LOCK_WALLET_ACCESS();
        ret = GetAccountAddress(Wallet_GetDefault(),strAccount).ToString();
    }
    return ret;
}

Value sc_getaccountaddress(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getaccountaddress <wallet> <account>\n"
            "Returns the current solidcoin address for receiving payments to this account.");

    string strAccount = AccountFromValue(params[0]);    // Parse the account first so we don't generate a key if there's an error
    Value ret = GetAccountAddress(pWallet,strAccount).ToString();
    return ret;
}

Value setaccount(const Array& params, bool fHelp)
{
    if (fHelp || params.size() < 1 || params.size() > 2)
        throw runtime_error(
            "setaccount <solidcoinaddress> <account>\n"
            "Sets the account associated with the given address.");

    CSolidCoinAddress address(params[0].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");

    string strAccount;
    if (params.size() > 1)  strAccount = AccountFromValue(params[1]);
    // Detect when changing the account of an address that is the 'unused current key' of another account:
    {
        MUTEX_LOCK(cs_main);
        LOCK_WALLET_ACCESS();
        CWallet *pWallet=Wallet_GetDefault();
        if (pWallet->mapAddressBook.count(address))
        {
            string strOldAccount = pWallet->mapAddressBook[address];
            if (address == GetAccountAddress(pWallet,strOldAccount))
                GetAccountAddress(pWallet,strOldAccount, true);
        }
        pWallet->SetAddressBookName(address, strAccount);
    }
    return Value::null;
}

Value sc_setaccount(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || (params.size()!=2 && params.size()!=3) || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_setaccount <wallet> <solidcoinaddress> <account>\n"
            "Sets the account associated with the given address.");

    CSolidCoinAddress address(params[1].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");

    string strAccount;
    if (params.size() > 2)  strAccount = AccountFromValue(params[2]);
    // Detect when changing the account of an address that is the 'unused current key' of another account:
    if (pWallet->mapAddressBook.count(address))
    {
        string strOldAccount = pWallet->mapAddressBook[address];
        if (address == GetAccountAddress(pWallet,strOldAccount))    GetAccountAddress(pWallet,strOldAccount, true);
    }
    pWallet->SetAddressBookName(address, strAccount);
    return Value::null;
}


Value getaccount(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "getaccount <solidcoinaddress>\n"
            "Returns the account associated with the given address.");

    CSolidCoinAddress address(params[0].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");

    string strAccount;
    LOCK_WALLET_ACCESS();
    map<CSolidCoinAddress, string>::iterator mi = Wallet_GetDefault()->mapAddressBook.find(address);
    if (mi != Wallet_GetDefault()->mapAddressBook.end() && !(*mi).second.empty())   strAccount = (*mi).second;
    return strAccount;
}


Value sc_getaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()!=2  || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getaccount <wallet> <solidcoinaddress>\n"
            "Returns the account associated with the given address.");

    CSolidCoinAddress address(params[1].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");
    string strAccount;
    map<CSolidCoinAddress, string>::iterator mi = pWallet->mapAddressBook.find(address);
    if (mi != pWallet->mapAddressBook.end() && !(*mi).second.empty())   strAccount = (*mi).second;
    return strAccount;
}


Value getaddressesbyaccount(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "getaddressesbyaccount <account>\n"
            "Returns the list of addresses for the given account.");

    string strAccount = AccountFromValue(params[0]);
    Array ret;  // Find all addresses that have the given account
    LOCK_WALLET_ACCESS();
    BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& item, Wallet_GetDefault()->mapAddressBook)
    {
        const CSolidCoinAddress& address = item.first;
        const string& strName = item.second;
        if (strName == strAccount)  ret.push_back(address.ToString());
    }
    return ret;
}

Value sc_getaddressesbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()!=2  || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "getaddressesbyaccount <wallet> <account>\n"
            "Returns the list of addresses for the given account in given wallet.");

    string strAccount = AccountFromValue(params[1]);
    Array ret;  // Find all addresses that have the given account
    BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& item, pWallet->mapAddressBook)
    {
        const CSolidCoinAddress& address = item.first;
        const string& strName = item.second;
        if (strName == strAccount)  ret.push_back(address.ToString());
    }
    return ret;
}

Value sc_gettxfee(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()!=2  || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_gettxfee <wallet> <amount>\n"
            "Returns the transaction fee for an amount to be sent");

    int64 nAmount = params[1].get_int64();
    int64 nFees=0;
    if(!pWallet->GetFees(nAmount,nFees)) nFees=0;
    Object obj;
    obj.push_back(Pair("int_fee",ValueFromAmountStr(nFees)));
    return obj;
}

Value real_sendtoaddress(CWallet *pWallet, bool bSC, const std::string &addr, const Array& params)
{
    CSolidCoinAddress address(addr);
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");

    // Amount
    int64 nAmount=0,nMinFee=0;
    if(bSC)
    {
        nAmount = params[2].get_int64();
        if(params.size() > 3)  nMinFee=params[3].get_int64();
    }
    else
    {
        nAmount = AmountFromValue(params[1]);
    }

    // Wallet comments
    CWalletTx wtx;
    int nCommentID=2;
    if(bSC) nCommentID=3;

    //if (params.size() > nCommentID && params[nCommentID].type() != null_type && !params[nCommentID].get_str().empty()) wtx.mapValue["comment"] = params[nCommentID].get_str();
    //if (params.size() > (nCommentID+1) && params[(nCommentID+1)].type() != null_type && !params[(nCommentID+1)].get_str().empty()) wtx.mapValue["to"]      = params[(nCommentID+1)].get_str();

    if(pWallet->IsLocked()) throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");

    string strError = pWallet->SendMoneyToSolidCoinAddress(address, nAmount, wtx,nMinFee);
    if (strError != "") throw JSONRPCError(-4, strError);
    return wtx.GetHash().GetHex();
}

Value sendtoaddress(const Array& params, bool fHelp)
{
    if (fHelp || params.size() < 2 || params.size() > 4)
        throw runtime_error(
            "sendtoaddress <solidcoinaddress> <amount>\n"
            "<amount> is a real and is rounded to the nearest 0.00000001\n"
            "requires wallet passphrase to be set with walletpassphrase first");

    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    return real_sendtoaddress(Wallet_GetDefault(),false,params[0].get_str(), params);
}

Value sc_sendtoaddress(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if ( fHelp || params.size() < 3 || params.size() > 4 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_sendtoaddress <wallet> <solidcoinaddress> <amount> [maxfee]\n"
            "<amount> is an integer where 10000 = 1SC\n"
            "[maxfee] is an integer specifying the max fee you will pay. If fee is over that the send won't happen\n");
    return real_sendtoaddress(pWallet,true,params[1].get_str(), params);
}

Value real_getreceivedbyaddress(CWallet *pWallet, bool bSC,const std::string &addr,const Array& params)
{
    // SolidCoin address
    CSolidCoinAddress address = CSolidCoinAddress(addr);
    CScript scriptPubKey;
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");
    scriptPubKey.SetSolidCoinAddress(address);
    if (!IsMine(*pWallet,scriptPubKey))
    {
        if(bSC) return ValueFromAmountStr(0);
        return ValueFromAmount(0);
    }

    // Minimum confirmations
    int64 nMinDepth = 1;
    if (bSC && params.size() > 2)  nMinDepth = params[2].get_int();
    if (!bSC && params.size() > 1)  nMinDepth = params[1].get_int();

    // Tally
    int64 nAmount = 0;
    {
        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            if (wtx.IsCoinBase() || !wtx.IsFinal()) continue;

            BOOST_FOREACH(const CTxOut& txout, wtx.vout)
            {
                if (txout.scriptPubKey == scriptPubKey && wtx.GetDepthInMainChain() >= nMinDepth)   nAmount += txout.nValue;
            }
        }
    }

    if(bSC) return ValueFromAmountStr(nAmount);
    return  ValueFromAmount(nAmount);
}

Value getreceivedbyaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() < 1 || params.size() > 2)
        throw runtime_error(
            "getreceivedbyaddress <solidcoinaddress> [minconf=1]\n"
            "Returns the total amount received by <solidcoinaddress> in transactions with at least [minconf] confirmations.");
    return real_getreceivedbyaddress(Wallet_GetDefault(),false,params[0].get_str(),params);
}

Value sc_getreceivedbyaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 2 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getreceivedbyaddress <wallet> <solidcoinaddress> [minconf=1]\n"
            "Returns the total amount received by <solidcoinaddress> in transactions with at least [minconf] confirmations.");
    return real_getreceivedbyaddress(pWallet,true,params[1].get_str(),params);
}


void GetAccountAddresses(CWallet *pWallet, string strAccount, set<CSolidCoinAddress>& setAddress)
{
    BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& item, pWallet->mapAddressBook)
    {
        const CSolidCoinAddress& address = item.first;
        const string& strName = item.second;
        if (strName == strAccount)
            setAddress.insert(address);
    }
}


Value real_getreceivedbyaccount(CWallet *pWallet,bool bSC, const Array& params)
{
    // Minimum confirmations
    int64 nMinDepth = 1;

    // Get the set of pub keys that have the label
    string strAccount;
    if(bSC)
    {
        strAccount = AccountFromValue(params[1]);
        if (params.size() > 2)  nMinDepth = params[2].get_int();
    }
    else
    {
        strAccount = AccountFromValue(params[0]);
        if (params.size() > 1)  nMinDepth = params[1].get_int();
    }
    set<CSolidCoinAddress> setAddress;
    GetAccountAddresses(pWallet,strAccount, setAddress);

    // Tally
    int64 nAmount = 0;
    for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
    {
        const CWalletTx& wtx = (*it).second;
        if (wtx.IsCoinBase() || !wtx.IsFinal()) continue;
        BOOST_FOREACH(const CTxOut& txout, wtx.vout)
        {
            CSolidCoinAddress address;
            if (ExtractAddress(txout.scriptPubKey, pWallet, address) && setAddress.count(address))
            {
                if (wtx.GetDepthInMainChain() >= nMinDepth) nAmount += txout.nValue;
            }
        }
    }

    return (double)nAmount / (double)COIN;
}

Value getreceivedbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() < 1 || params.size() > 2)
        throw runtime_error(
            "getreceivedbyaccount <account> [minconf=1]\n"
            "Returns the total amount received by addresses with <account> in transactions with at least [minconf] confirmations.");

    real_getreceivedbyaccount(Wallet_GetDefault(),false,params);
}

Value sc_getreceivedbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 2 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "getreceivedbyaccount <wallet> <account> [minconf=1]\n"
            "Returns the total amount received by addresses with <account> in transactions with at least [minconf] confirmations.");

    real_getreceivedbyaccount(pWallet,true,params);
}

int64 GetAccountBalance(CWallet *pWallet, CWalletDB& walletdb, const string& strAccount, int64 nMinDepth)
{
    int64 nBalance = 0;
    // Tally wallet transactions
    for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
    {
        const CWalletTx& wtx = (*it).second;
        if (!wtx.IsFinal()) continue;

        int64 nGenerated, nReceived, nSent, nFee;
        wtx.GetAccountAmounts(strAccount, nGenerated, nReceived, nSent, nFee);

        if (nReceived != 0 && wtx.GetDepthInMainChain() >= nMinDepth)
            nBalance += nReceived;
        nBalance += nGenerated - nSent - nFee;
    }

    nBalance += walletdb.GetAccountCreditDebit(strAccount);     // Tally internal accounting entries
    return nBalance;
}

int64 GetAccountBalance(CWallet *pWallet, const string& strAccount, int64 nMinDepth)
{
    CWalletDB walletdb(pWallet->strWalletFile);
    return GetAccountBalance(pWallet,walletdb, strAccount, nMinDepth);
}

Value real_getbalance(CWallet *pWallet,bool bSC, const Array& params)
{
    string strAccount;
    int64 nMinDepth = 1;
    if(bSC)
    {
        if(params.size()==1)   return ValueFromAmountStr(pWallet->GetBalance());
        if(params.size() > 2)  nMinDepth = params[2].get_int();
        strAccount=params[1].get_str();
    }
    else
    {
        if(params.size()==0)    return  ValueFromAmount(pWallet->GetBalance());
        if(params.size() > 1)   nMinDepth = params[1].get_int();
        strAccount=params[0].get_str();
    }


    if (strAccount == "*") {
        // Calculate total balance a different way from GetBalance()
        // (GetBalance() sums up all unspent TxOuts)
        // getbalance and getbalance '*' should always return the same number.
        int64 nBalance = 0;
        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            if (!wtx.IsFinal()) continue;

            int64 allGeneratedImmature, allGeneratedMature, allFee;
            allGeneratedImmature = allGeneratedMature = allFee = 0;
            string strSentAccount;
            list<pair<CSolidCoinAddress, int64> > listReceived;
            list<pair<CSolidCoinAddress, int64> > listSent;
            wtx.GetAmounts(allGeneratedImmature, allGeneratedMature, listReceived, listSent, allFee, strSentAccount);
            if (wtx.GetDepthInMainChain() >= nMinDepth)
            {
                BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress,int64)& r, listReceived)
                {
                    nBalance += r.second;
                }
            }
            BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress,int64)& r, listSent)
            {
                nBalance -= r.second;
            }
            nBalance -= allFee;
            nBalance += allGeneratedMature;
        }
        if(bSC) return  ValueFromAmountStr(nBalance);
        else    return  ValueFromAmount(nBalance);
    }
    if(bSC) strAccount = AccountFromValue(params[1]);
    else    strAccount = AccountFromValue(params[0]);

    int64 nBalance = GetAccountBalance(pWallet,strAccount, nMinDepth);
    if(bSC) return  ValueFromAmountStr(nBalance);
    return ValueFromAmount(nBalance);
}

Value getbalance(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() > 2)
    {
        throw runtime_error("getbalance [account] [minconf=1]\n"
            "If [account] is not specified, returns the server's total available balance.\n"
            "If [account] is specified, returns the balance in the account.");
    }
    return real_getbalance(Wallet_GetDefault(),false,params);
}

Value sc_getbalance(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()<1 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())) )
    {
        throw runtime_error("sc_getbalance <wallet> [account] [minconf=1]\n"
            "If [account] is not specified, returns the server's total available balance.\n"
            "If [account] is specified, returns the balance in the account.");
    }
    return real_getbalance(pWallet,true,params);
}

Value real_movecmd(CWallet *pWallet, bool bSC, const Array& params)
{
    string strComment;
    string strFrom;
    string strTo;
    int64 nAmount =0;
    if(bSC)
    {
        strFrom = AccountFromValue(params[1]);
        strTo = AccountFromValue(params[2]);
        nAmount = params[3].get_int64();
        if (params.size() > 3)  strComment = params[4].get_str();
    }
    else
    {
        strFrom = AccountFromValue(params[0]);
        strTo = AccountFromValue(params[1]);
        nAmount = AmountFromValue(params[2]);
        if (params.size() > 3)  strComment = params[3].get_str();
    }

    CWalletDB walletdb(pWallet->strWalletFile);
    walletdb.TxnBegin();
    int64 nNow = SolidTime_Get();

    CAccountingEntry debit;
    debit.strAccount = strFrom;
    debit.nCreditDebit = -nAmount;
    debit.nTime = nNow;
    debit.strOtherAccount = strTo;
    debit.strComment = strComment;
    walletdb.WriteAccountingEntry(debit);

    CAccountingEntry credit;
    credit.strAccount = strTo;
    credit.nCreditDebit = nAmount;
    credit.nTime = nNow;
    credit.strOtherAccount = strFrom;
    credit.strComment = strComment;
    walletdb.WriteAccountingEntry(credit);
    walletdb.TxnCommit();
    return true;
}

Value movecmd(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() < 3 || params.size() > 4)
        throw runtime_error(
            "move <fromaccount> <toaccount> <amount> [comment]\n"
            "Move from one account in your wallet to another.");
    return real_movecmd(Wallet_GetDefault(),false,params);
}

Value sc_movecmd(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()<4 || params.size() > 5 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_move <wallet> <fromaccount> <toaccount> <amount> [comment]\n"
            "Move from one account in your wallet to another.");
    return real_movecmd(pWallet, true,params);
}

Value real_sendfrom(CWallet *pWallet,bool bSC, const Array& params)
{
    string strAccount;
    CSolidCoinAddress address;
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");
    int64 nAmount = 0;
    int nMinDepth = 1;
    CWalletTx wtx;
    if(bSC)
    {
        strAccount = AccountFromValue(params[1]);
        wtx.strFromAccount = strAccount;
        address.SetString((params[2].get_str()));
        nAmount=params[3].get_int64();
        if (params.size() > 4)  nMinDepth = params[4].get_int();
        if (params.size() > 5 && params[5].type() != null_type && !params[5].get_str().empty()) wtx.mapValue["comment"] = params[5].get_str();
        if (params.size() > 6 && params[6].type() != null_type && !params[6].get_str().empty()) wtx.mapValue["to"]      = params[6].get_str();
    }
    else
    {
        strAccount = AccountFromValue(params[0]);
        wtx.strFromAccount = strAccount;
        address.SetString((params[1].get_str()));
        nAmount=AmountFromValue(params[2]);
        if (params.size() > 3)  nMinDepth = params[3].get_int();
        if (params.size() > 4 && params[4].type() != null_type && !params[4].get_str().empty()) wtx.mapValue["comment"] = params[4].get_str();
        if (params.size() > 5 && params[5].type() != null_type && !params[5].get_str().empty()) wtx.mapValue["to"]      = params[5].get_str();
    }

    if(pWallet->IsLocked()) throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");

    int64 nBalance = GetAccountBalance(pWallet,strAccount, nMinDepth);  // Check funds
    if (nAmount > nBalance) throw JSONRPCError(-6, "Account has insufficient funds");

    string strError = pWallet->SendMoneyToSolidCoinAddress(address, nAmount, wtx,0);
    if (strError != "") throw JSONRPCError(-4, strError);
    return wtx.GetHash().GetHex();
}

Value sendfrom(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() < 3 || params.size() > 6)
        throw runtime_error(
            "sendfrom <fromaccount> <tosolidcoinaddress> <amount> [minconf=1] [comment] [comment-to]\n"
            "<amount> is a real and is rounded to the nearest 0.00000001\n"
            "requires wallet passphrase to be set with walletpassphrase if wallet is encrypted");

    return real_sendfrom(Wallet_GetDefault(),false,params);
}


Value sc_sendfrom(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 4 || params.size() > 7 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_sendfrom <wallet> <fromaccount> <tosolidcoinaddress> <amount> [minconf=1] [comment] [comment-to]\n"
            "<amount> is an integer. 10000 units = 1SC\n"
            "requires wallet passphrase to be set with walletpassphrase first if wallet is encrypted");

    return real_sendfrom(pWallet,true,params);
}


Value real_sendmany(CWallet *pWallet, bool bSC, const Array& params)
{
    CWalletTx wtx;
    string strAccount;
    Object sendTo;
    int nMinDepth = 1;
    if(bSC)
    {
        strAccount = AccountFromValue(params[1]);
        wtx.strFromAccount = strAccount;
        sendTo = params[2].get_obj();
        if (params.size() > 3)  nMinDepth = params[3].get_int();
        if (params.size() > 4 && params[4].type() != null_type && !params[4].get_str().empty()) wtx.mapValue["comment"] = params[4].get_str();
    }
    else
    {
        strAccount = AccountFromValue(params[0]);
        wtx.strFromAccount = strAccount;
        sendTo = params[1].get_obj();
        if (params.size() > 2)  nMinDepth = params[2].get_int();
        if (params.size() > 3 && params[3].type() != null_type && !params[3].get_str().empty()) wtx.mapValue["comment"] = params[3].get_str();
    }

    set<CSolidCoinAddress> setAddress;
    vector<pair<CScript, int64> > vecSend;

    int64 totalAmount = 0;
    BOOST_FOREACH(const Pair& s, sendTo)
    {
        CSolidCoinAddress address(s.name_);
        if(!address.IsValid())             throw JSONRPCError(-5, string("Invalid solidcoin address:")+s.name_);
        if(setAddress.count(address))      throw JSONRPCError(-8, string("Invalid parameter, duplicated address: ")+s.name_);
        setAddress.insert(address);

        CScript scriptPubKey;
        scriptPubKey.SetSolidCoinAddress(address);
        int64 nAmount = 0;
        if(bSC) nAmount=AmountFromValueStr(s.value_);
        else    nAmount=AmountFromValue(s.value_);
        totalAmount += nAmount;
        vecSend.push_back(make_pair(scriptPubKey, nAmount));
    }

    if(pWallet->IsLocked()) throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");

    // Check funds
    int64 nBalance = GetAccountBalance(pWallet,strAccount, nMinDepth);
    if (totalAmount > nBalance) throw JSONRPCError(-6, "Account has insufficient funds");

    // Send
    CReserveKey keyChange;
    keyChange.SetWallet(pWallet);
    int64 nFeeRequired = 0;
    bool fCreated = pWallet->CreateTransaction(vecSend, wtx, keyChange, nFeeRequired);
    if (!fCreated)
    {
        if (totalAmount + nFeeRequired > pWallet->GetBalance()) throw JSONRPCError(-6, "Insufficient funds");
        throw JSONRPCError(-4, "Transaction creation failed");
    }
    if (!pWallet->CommitTransaction(wtx, keyChange))    throw JSONRPCError(-4, "Transaction commit failed");
    return wtx.GetHash().GetHex();
}

Value sendmany(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();

    if (fHelp || params.size() < 2 || params.size() > 4)
        throw runtime_error(
            "sendmany <fromaccount> {address:amount,...} [minconf=1] [comment]\n"
            "amounts are double-precision floating point numbers\n"
            "requires wallet passphrase to be set with walletpassphrase if wallet is encrypted");

    return real_sendmany(Wallet_GetDefault(),false, params);
}

Value sc_sendmany(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 3 || params.size() > 5 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_sendmany <wallet> <fromaccount> {address:amount,...} [minconf=1] [comment]\n"
            "amounts are an integer\n"
            "requires wallet passphrase to be set with walletpassphrase if wallet is encrypted");
   return real_sendmany(pWallet,true, params);
}


struct tallyitem
{
    int64 nAmount;
    int64 nConf;
    tallyitem()
    {
        nAmount = 0;
        nConf = INT64_MAX;
    }
};

Value ListReceived(CWallet *pWallet, bool bSC, const Array& params, bool fByAccounts)
{
    int nMinDepth = 1;              // Minimum confirmations
    bool fIncludeEmpty = false;     // Whether to include empty accounts
    if(bSC)
    {
        if (params.size() > 1)  nMinDepth = params[1].get_int();
        if (params.size() > 2)  fIncludeEmpty = params[2].get_bool();

    }
    else
    {
        if (params.size() > 0)  nMinDepth = params[0].get_int();
        if (params.size() > 1)  fIncludeEmpty = params[1].get_bool();
    }

    // Tally
    map<CSolidCoinAddress, tallyitem> mapTally;
    {
        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            if (wtx.IsCoinBase() || !wtx.IsFinal()) continue;

            int64 nDepth = wtx.GetDepthInMainChain();
            if (nDepth < nMinDepth)     continue;

            BOOST_FOREACH(const CTxOut& txout, wtx.vout)
            {
                CSolidCoinAddress address;
                if (!ExtractAddress(txout.scriptPubKey, pWallet, address) || !address.IsValid())    continue;
                tallyitem& item = mapTally[address];
                item.nAmount += txout.nValue;
                item.nConf = min(item.nConf, nDepth);
            }
        }
    }

    // Reply
    Array ret;
    map<string, tallyitem> mapAccountTally;
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& item, pWallet->mapAddressBook)
        {
            const CSolidCoinAddress& address = item.first;
            const string& strAccount = item.second;
            map<CSolidCoinAddress, tallyitem>::iterator it = mapTally.find(address);
            if (it == mapTally.end() && !fIncludeEmpty)
                continue;

            int64 nAmount = 0;
            int64 nConf = INT64_MAX;
            if (it != mapTally.end())
            {
                nAmount = (*it).second.nAmount;
                nConf = (*it).second.nConf;
            }

            if (fByAccounts)
            {
                tallyitem& item = mapAccountTally[strAccount];
                item.nAmount += nAmount;
                item.nConf = min(item.nConf, nConf);
            }
            else
            {
                Object obj;
                obj.push_back(Pair("address",       address.ToString()));
                obj.push_back(Pair("account",       strAccount));
                obj.push_back(Pair("label",         strAccount)); // deprecated
                if(bSC) obj.push_back(Pair("int_amount",    ValueFromAmountStr(nAmount)));  //only show int amount with SolidCoin API
                else    obj.push_back(Pair("amount",        ValueFromAmount(nAmount)));
                obj.push_back(Pair("confirmations", (nConf == INT64_MAX ? 0 : nConf)));
                ret.push_back(obj);
            }
        }
    }

    if (fByAccounts)
    {
        for (map<string, tallyitem>::iterator it = mapAccountTally.begin(); it != mapAccountTally.end(); ++it)
        {
            int64 nAmount = (*it).second.nAmount;
            int64 nConf = (*it).second.nConf;
            Object obj;
            obj.push_back(Pair("account",       (*it).first));
            obj.push_back(Pair("label",         (*it).first)); // deprecated
            if(bSC) obj.push_back(Pair("int_amount",    ValueFromAmountStr(nAmount)));      //only show int amount with SolidCoin API
            else    obj.push_back(Pair("amount",        ValueFromAmount(nAmount)));
            obj.push_back(Pair("confirmations", (nConf == INT64_MAX ? (int64)0 : nConf)));
            ret.push_back(obj);
        }
    }

    return ret;
}

Value listreceivedbyaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() > 2)
        throw runtime_error(
            "listreceivedbyaddress [minconf=1] [includeempty=false]\n"
            "[minconf] is the minimum number of confirmations before payments are included.\n"
            "[includeempty] whether to include addresses that haven't received any payments.\n"
            "Returns an array of objects containing:\n"
            "  \"address\" : receiving address\n"
            "  \"account\" : the account of the receiving address\n"
            "  \"amount\" : total amount received by the address\n"
            "  \"confirmations\" : number of confirmations of the most recent transaction included");

    return ListReceived(Wallet_GetDefault(),false, params, false);
}

Value sc_listreceivedbyaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 1 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listreceivedbyaddress <wallet> [minconf=1] [includeempty=false]\n"
            "[minconf] is the minimum number of confirmations before payments are included.\n"
            "[includeempty] whether to include addresses that haven't received any payments.\n"
            "Returns an array of objects containing:\n"
            "  \"address\" : receiving address\n"
            "  \"account\" : the account of the receiving address\n"
            "  \"amount\" : total amount received by the address\n"
            "  \"confirmations\" : number of confirmations of the most recent transaction included");

    return ListReceived(pWallet,true, params, false);
}

Value listreceivedbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() > 2)
        throw runtime_error(
            "listreceivedbyaccount [minconf=1] [includeempty=false]\n"
            "[minconf] is the minimum number of confirmations before payments are included.\n"
            "[includeempty] whether to include accounts that haven't received any payments.\n"
            "Returns an array of objects containing:\n"
            "  \"account\" : the account of the receiving addresses\n"
            "  \"amount\" : total amount received by addresses with this account\n"
            "  \"confirmations\" : number of confirmations of the most recent transaction included");

    return ListReceived(Wallet_GetDefault(),false, params, true);
}

Value sc_listreceivedbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 1 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listreceivedbyaccount <wallet> [minconf=1] [includeempty=false]\n"
            "[minconf] is the minimum number of confirmations before payments are included.\n"
            "[includeempty] whether to include accounts that haven't received any payments.\n"
            "Returns an array of objects containing:\n"
            "  \"account\" : the account of the receiving addresses\n"
            "  \"amount\" : total amount received by addresses with this account\n"
            "  \"confirmations\" : number of confirmations of the most recent transaction included");

    return ListReceived(pWallet,true, params, true);
}

void ListTransactions(CWallet *pWallet, bool bSC, const CWalletTx& wtx, const string& strAccount, int nMinDepth, bool fLong, Array& ret)
{
    int64 nGeneratedImmature, nGeneratedMature, nFee;
    string strSentAccount;
    list<pair<CSolidCoinAddress, int64> > listReceived;
    list<pair<CSolidCoinAddress, int64> > listSent;
    wtx.GetAmounts(nGeneratedImmature, nGeneratedMature, listReceived, listSent, nFee, strSentAccount);

    bool fAllAccounts = (strAccount == string("*"));

    // Generated blocks assigned to account ""
    if ((nGeneratedMature+nGeneratedImmature) != 0 && (fAllAccounts || strAccount == ""))
    {
        Object entry;
        entry.push_back(Pair("account", string("")));
        if (nGeneratedImmature)
        {
            entry.push_back(Pair("category", wtx.GetDepthInMainChain() ? "immature" : "orphan"));
            if(bSC) entry.push_back(Pair("int_amount", ValueFromAmountStr(nGeneratedImmature)));    //only show int amount with SolidCoin API
            entry.push_back(Pair("amount", ValueFromAmount(nGeneratedImmature)));
        }
        else
        {
            entry.push_back(Pair("category", "generate"));
            if(bSC) entry.push_back(Pair("int_amount", ValueFromAmountStr(nGeneratedMature)));  //only show int amount with SolidCoin API
            entry.push_back(Pair("amount", ValueFromAmount(nGeneratedMature)));
        }
        if (fLong)
            WalletTxToJSON(wtx, entry);
        ret.push_back(entry);
    }

    // Sent
    if ((!listSent.empty() || nFee != 0) && (fAllAccounts || strAccount == strSentAccount))
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& s, listSent)
        {
            Object entry;
            entry.push_back(Pair("account", strSentAccount));
            entry.push_back(Pair("address", s.first.ToString()));
            entry.push_back(Pair("category", "send"));
            if(bSC) //only show int amount with SolidCoin API
            {
                entry.push_back(Pair("int_amount", ValueFromAmountStr(-s.second)));
                entry.push_back(Pair("int_fee", ValueFromAmountStr(-nFee)));
            }

            entry.push_back(Pair("amount", ValueFromAmount(-s.second)));
            entry.push_back(Pair("fee", ValueFromAmount(-nFee)));


            if (fLong)
                WalletTxToJSON(wtx, entry);
            ret.push_back(entry);
        }
    }

    // Received
    if (listReceived.size() > 0 && wtx.GetDepthInMainChain() >= nMinDepth)
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& r, listReceived)
        {
            string account;
            if (pWallet->mapAddressBook.count(r.first)) account = pWallet->mapAddressBook[r.first];
            if (fAllAccounts || (account == strAccount))
            {
                Object entry;
                entry.push_back(Pair("account", account));
                entry.push_back(Pair("address", r.first.ToString()));
                entry.push_back(Pair("category", "receive"));
                if(bSC) entry.push_back(Pair("int_amount", ValueFromAmountStr(r.second)));      //only show int amount with SolidCoin API
                entry.push_back(Pair("amount", ValueFromAmount(r.second)));
                if (fLong)  WalletTxToJSON(wtx, entry);
                ret.push_back(entry);
            }
        }
    }

}

void AcentryToJSON(bool bSC, const CAccountingEntry& acentry, const string& strAccount, Array& ret)
{
    bool fAllAccounts = (strAccount == string("*"));

    if (fAllAccounts || acentry.strAccount == strAccount)
    {
        Object entry;
        entry.push_back(Pair("account", acentry.strAccount));
        entry.push_back(Pair("category", "move"));
        entry.push_back(Pair("time", (boost::int64_t)acentry.nTime));
        if(bSC) entry.push_back(Pair("int_amount", ValueFromAmountStr(acentry.nCreditDebit)));      //only show int amount with SolidCoin API
        entry.push_back(Pair("amount", ValueFromAmount(acentry.nCreditDebit)));
        entry.push_back(Pair("otheraccount", acentry.strOtherAccount));
        entry.push_back(Pair("comment", acentry.strComment));
        ret.push_back(entry);
    }
}

Value real_listtransactions(CWallet *pWallet, bool bSC, const Array& params)
{
    string strAccount = "*";
    int nCount = 10;
    int nFrom = 0;
    if(bSC)
    {
        if (params.size() > 1)  strAccount = params[1].get_str();
        if (params.size() > 2)  nCount = params[2].get_int();
        if (params.size() > 3)  nFrom = params[3].get_int();
    }
    else
    {
        if (params.size() > 0)  strAccount = params[0].get_str();
        if (params.size() > 1)  nCount = params[1].get_int();
        if (params.size() > 2)  nFrom = params[2].get_int();
    }

    Array ret;
    CWalletDB walletdb(pWallet->strWalletFile);

    // Firs: get all CWalletTx and CAccountingEntry into a sorted-by-time multimap:
    typedef pair<CWalletTx*, CAccountingEntry*> TxPair;
    typedef multimap<int64, TxPair > TxItems;
    TxItems txByTime;

    for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
    {
        CWalletTx* wtx = &((*it).second);
        txByTime.insert(make_pair(wtx->GetTxTime(), TxPair(wtx, (CAccountingEntry*)0)));
    }
    list<CAccountingEntry> acentries;
    walletdb.ListAccountCreditDebit(strAccount, acentries);
    BOOST_FOREACH(CAccountingEntry& entry, acentries)
    {
        txByTime.insert(make_pair(entry.nTime, TxPair((CWalletTx*)0, &entry)));
    }

    // Now: iterate backwards until we have nCount items to return:
    TxItems::reverse_iterator it = txByTime.rbegin();
    if (txByTime.size() > nFrom) std::advance(it, nFrom);
    for (; it != txByTime.rend(); ++it)
    {
        CWalletTx *const pwtx = (*it).second.first;
        if (pwtx != 0)  ListTransactions(pWallet,bSC,*pwtx, strAccount, 0, true, ret);
        CAccountingEntry *const pacentry = (*it).second.second;
        if (pacentry != 0)  AcentryToJSON(bSC,*pacentry, strAccount, ret);
        if (ret.size() >= nCount) break;
    }
    // ret is now newest to oldest

    // Make sure we return only last nCount items (sends-to-self might give us an extra):
    if (ret.size() > nCount)
    {
        Array::iterator last = ret.begin();
        std::advance(last, nCount);
        ret.erase(last, ret.end());
    }
    std::reverse(ret.begin(), ret.end()); // oldest to newest

    return ret;
}

Value listtransactions(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() > 3)
        throw runtime_error(
            "listtransactions [account] [count=10] [from=0]\n"
            "Returns up to [count] most recent transactions skipping the first [from] transactions for account [account].");
    return real_listtransactions(Wallet_GetDefault(),false,params);
}

Value sc_listtransactions(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 1 || params.size() > 4 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listtransactions <wallet> [account] [count=10] [from=0]\n"
            "Returns up to [count] most recent transactions skipping the first [from] transactions for account [account].");
    return real_listtransactions(pWallet,true,params);
}


Value real_listaccounts(CWallet *pWallet, bool bSC,const Array& params)
{
    int64 nMinDepth = 1;
    if(bSC && params.size() > 1)  nMinDepth = params[1].get_int();
    else if(!bSC && params.size() > 0)  nMinDepth = params[0].get_int();

    map<string, int64> mapAccountBalances;
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& entry, pWallet->mapAddressBook) {
            if (pWallet->HaveKey(entry.first)) // This address belongs to me
                mapAccountBalances[entry.second] = 0;
        }

        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            int64 nGeneratedImmature, nGeneratedMature, nFee;
            string strSentAccount;
            list<pair<CSolidCoinAddress, int64> > listReceived;
            list<pair<CSolidCoinAddress, int64> > listSent;
            wtx.GetAmounts(nGeneratedImmature, nGeneratedMature, listReceived, listSent, nFee, strSentAccount);
            mapAccountBalances[strSentAccount] -= nFee;
            BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& s, listSent)
                mapAccountBalances[strSentAccount] -= s.second;
            if (wtx.GetDepthInMainChain() >= nMinDepth)
            {
                mapAccountBalances[""] += nGeneratedMature;
                BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& r, listReceived)
                    if (pWallet->mapAddressBook.count(r.first))
                        mapAccountBalances[pWallet->mapAddressBook[r.first]] += r.second;
                    else
                        mapAccountBalances[""] += r.second;
            }
        }
    }

    list<CAccountingEntry> acentries;
    CWalletDB(pWallet->strWalletFile).ListAccountCreditDebit("*", acentries);
    BOOST_FOREACH(const CAccountingEntry& entry, acentries)
        mapAccountBalances[entry.strAccount] += entry.nCreditDebit;

    Object ret;
    BOOST_FOREACH(const PAIRTYPE(string, int64)& accountBalance, mapAccountBalances) {
        if(bSC) ret.push_back(Pair(accountBalance.first, ValueFromAmountStr(accountBalance.second)));
        else    ret.push_back(Pair(accountBalance.first, ValueFromAmount(accountBalance.second)));
    }
    return ret;
}
Value listaccounts(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() > 1)
        throw runtime_error(
            "listaccounts [minconf=1]\n"
            "Returns Object that has account names as keys, account balances as values.");
    return real_listaccounts(Wallet_GetDefault(),false,params);
}

Value sc_listaccounts(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()<1 || params.size() > 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listaccounts <wallet> [minconf=1]\n"
            "Returns Object that has account names as keys, account balances as values.");
    return real_listaccounts(pWallet,true,params);
}

Value real_gettransaction(CWallet *pWallet, bool bSC, const Array& params)
{
    uint256 hash;
    if(bSC) hash.SetHex(params[1].get_str());
    else    hash.SetHex(params[0].get_str());

    Object entry;
    {
        if (!pWallet->mapWallet.count(hash))    throw JSONRPCError(-5, "Invalid or non-wallet transaction id");
        const CWalletTx& wtx = pWallet->mapWallet[hash];

        int64 nCredit = wtx.GetCredit();
        int64 nDebit = wtx.GetDebit();
        int64 nNet = nCredit - nDebit;
        int64 nFee = (wtx.IsFromMe() ? wtx.GetValueOut() - nDebit : 0);

        entry.push_back(Pair("amount", ValueFromAmount(nNet - nFee)));
        if (wtx.IsFromMe())
            entry.push_back(Pair("fee", ValueFromAmount(nFee)));

        WalletTxToJSON(pWallet->mapWallet[hash], entry);

        Array details;
        ListTransactions(pWallet,bSC,pWallet->mapWallet[hash], "*", 0, false, details);
        entry.push_back(Pair("details", details));
    }

    return entry;

}
Value gettransaction(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "gettransaction <txid>\n"
            "Get detailed information about <txid>");
    return real_gettransaction(Wallet_GetDefault(),false,params);
}
Value sc_gettransaction(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_gettransaction <wallet> <txid>\n"
            "Get detailed information about <txid>");
    return real_gettransaction(pWallet,true,params);
}


Value backupwallet(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "backupwallet <destination>\n"
            "Safely copies wallet.dat to destination, which can be a directory or a path with filename.");

    string strDest = params[0].get_str();
    BackupWallet(*Wallet_GetDefault(), strDest);

    return Value::null;
}

Value sc_backupwallet(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_backupwallet <wallet> <destination>\n"
            "Safely copies <wallet> to destination, which can be a directory or a path with filename.");

    string strDest = params[1].get_str();
    BackupWallet(*pWallet, strDest);
    return Value::null;
}

Value keypoolrefill(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if(fHelp || params.size() > 0)
        throw runtime_error(
            "keypoolrefill\n"
            "Fills the keypool, requires wallet passphrase to be set if wallet is encrypted.");

    if (Wallet_GetDefault()->IsLocked())    throw JSONRPCError(-13, "Error: Please enter the wallet passphrase with walletpassphrase first.");
    Wallet_GetDefault()->TopUpKeyPool();
    if (Wallet_GetDefault()->GetKeyPoolSize() < Setting_GetINT64("keypool"))    throw JSONRPCError(-4, "Error refreshing keypool.");
    return Value::null;
}

Value sc_keypoolrefill(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 1 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "keypoolrefill <wallet>\n"
            "Fills the keypool, requires wallet passphrase to be set if wallet is encrypted.");

    if (pWallet->IsLocked())    throw JSONRPCError(-13, "Error: Please enter the wallet passphrase with walletpassphrase first.");
    pWallet->TopUpKeyPool();
    if (pWallet->GetKeyPoolSize() < Setting_GetINT64("keypool"))    throw JSONRPCError(-4, "Error refreshing keypool.");
    return Value::null;
}

struct WALLET_SLEEP
{
    int nTime;
    std::string walletname;
};
void ThreadCleanWalletPassphrase(void* parg)
{
    WALLET_SLEEP *pSleep = (WALLET_SLEEP*)parg;
    while(1)
    {
        {
            LOCK_WALLET_ACCESS();
            CWallet *pWallet=Wallet_Get(pSleep->walletname);
            if(pWallet)
            {
                if(SolidTime_Get()>pWallet->m_nWalletLockTime)
                {
                    if(pWallet->m_nWalletLockTime!=0) pWallet->Lock();  //if its still positive, we should lock
                    break;
                }
            }
        }
        Sleep(1000);
    }
    delete pSleep;
}

void real_walletpassphrase(CWallet *pWallet, const std::string &pass, int nSleepTime)
{
    if (!pWallet->IsCrypted())  throw JSONRPCError(-15, "Error: running with an unencrypted wallet, but walletpassphrase was called.");
    if (!pWallet->IsLocked())   throw JSONRPCError(-17, "Error: Wallet is already unlocked.");

    // Note that the walletpassphrase is stored in params[0] which is not mlock()ed
    string strWalletPass;
    strWalletPass.reserve(100);
    mlock(&strWalletPass[0], strWalletPass.capacity());
    strWalletPass = pass;

    if (strWalletPass.length() > 0)
    {
        if (!pWallet->Unlock(strWalletPass))
        {
            fill(strWalletPass.begin(), strWalletPass.end(), '\0');
            munlock(&strWalletPass[0], strWalletPass.capacity());
            throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");
        }
        fill(strWalletPass.begin(), strWalletPass.end(), '\0');
        munlock(&strWalletPass[0], strWalletPass.capacity());
    }
    else
        throw runtime_error(
            "walletpassphrase <passphrase> <timeout>\n"
            "Stores the wallet decryption key in memory for <timeout> seconds.");

    pWallet->TopUpKeyPool();
    WALLET_SLEEP *pSleep= new WALLET_SLEEP;
    pWallet->m_nWalletLockTime=nSleepTime+SolidTime_Get();
    pSleep->nTime=nSleepTime;
    pSleep->walletname=pWallet->GetWalletName();
    CreateThread(ThreadCleanWalletPassphrase, pSleep);
}

Value walletpassphrase(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() != 2)
        throw runtime_error(
            "walletpassphrase <passphrase> <timeout>\n"
            "Stores the wallet decryption key in memory for <timeout> seconds.");

    real_walletpassphrase(Wallet_GetDefault(),params[0].get_str(),params[1].get_int());
    return Value::null;
}

Value sc_walletpassphrase(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "walletpassphrase <wallet> <passphrase> <timeout>\n"
            "Stores the wallet decryption key in memory for <timeout> seconds.");

    real_walletpassphrase(pWallet,params[0].get_str(),params[1].get_int());
    return Value::null;
}

void real_walletpassphrasechange(CWallet *pWallet, const std::string &oldpass, const std::string &newpass)
{
    if (!pWallet->IsCrypted())  throw JSONRPCError(-15, "Error: running with an unencrypted wallet, but walletpassphrasechange was called.");

    string strOldWalletPass;
    strOldWalletPass.reserve(100);
    mlock(&strOldWalletPass[0], strOldWalletPass.capacity());
    strOldWalletPass = oldpass;

    string strNewWalletPass;
    strNewWalletPass.reserve(100);
    mlock(&strNewWalletPass[0], strNewWalletPass.capacity());
    strNewWalletPass = newpass;

    if (strOldWalletPass.length() < 1 || strNewWalletPass.length() < 1)
        throw runtime_error(
            "walletpassphrasechange <oldpassphrase> <newpassphrase>\n"
            "Changes the wallet passphrase from <oldpassphrase> to <newpassphrase>.");

    if (!pWallet->ChangeWalletPassphrase(strOldWalletPass, strNewWalletPass))
    {
        fill(strOldWalletPass.begin(), strOldWalletPass.end(), '\0');
        fill(strNewWalletPass.begin(), strNewWalletPass.end(), '\0');
        munlock(&strOldWalletPass[0], strOldWalletPass.capacity());
        munlock(&strNewWalletPass[0], strNewWalletPass.capacity());
        throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");
    }
    fill(strNewWalletPass.begin(), strNewWalletPass.end(), '\0');
    fill(strOldWalletPass.begin(), strOldWalletPass.end(), '\0');
    munlock(&strOldWalletPass[0], strOldWalletPass.capacity());
    munlock(&strNewWalletPass[0], strNewWalletPass.capacity());
}

Value walletpassphrasechange(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() != 2)
        throw runtime_error(
            "walletpassphrasechange <oldpassphrase> <newpassphrase>\n"
            "Changes the wallet passphrase from <oldpassphrase> to <newpassphrase>.");
    real_walletpassphrasechange(Wallet_GetDefault(),params[0].get_str(),params[1].get_str());
    return Value::null;
}

Value sc_walletpassphrasechange(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "walletpassphrasechange <wallet> <oldpassphrase> <newpassphrase>\n"
            "Changes the wallet passphrase from <oldpassphrase> to <newpassphrase>.");
    real_walletpassphrasechange(pWallet,params[1].get_str(),params[2].get_str());
    return Value::null;
}

Value walletlock(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "walletlock\n"
            "Removes the wallet encryption key from memory, locking the wallet.\n"
            "After calling this method, you will need to call walletpassphrase again\n"
            "before being able to call any methods which require the wallet to be unlocked.");
    if (!Wallet_GetDefault()->IsCrypted())  throw JSONRPCError(-15, "Error: running with an unencrypted wallet, but walletlock was called.");
    Wallet_GetDefault()->m_nWalletLockTime = 0;
    Wallet_GetDefault()->Lock();
    return Value::null;
}

Value sc_walletlock(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 1 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "walletlock <wallet>\n"
            "Removes the wallet encryption key from memory, locking the wallet.\n"
            "After calling this method, you will need to call walletpassphrase again\n"
            "before being able to call any methods which require the wallet to be unlocked.");
    if (!pWallet->IsCrypted())  throw JSONRPCError(-15, "Error: running with an unencrypted wallet, but walletlock was called.");
    pWallet->m_nWalletLockTime = 0;
    pWallet->Lock();
    return Value::null;
}

void real_encryptwallet(CWallet *pWallet, const std::string &pass)
{
    if (pWallet->IsCrypted())   throw JSONRPCError(-15, "Error: running with an encrypted wallet, but encryptwallet was called.");

    string strWalletPass;
    strWalletPass.reserve(100);
    mlock(&strWalletPass[0], strWalletPass.capacity());
    strWalletPass = pass;

    if (strWalletPass.length() < 1)
        throw runtime_error(
            "encryptwallet <passphrase>\n"
            "Encrypts the wallet with <passphrase>.");

    if (!pWallet->EncryptWallet(strWalletPass))
    {
        fill(strWalletPass.begin(), strWalletPass.end(), '\0');
        munlock(&strWalletPass[0], strWalletPass.capacity());
        throw JSONRPCError(-16, "Error: Failed to encrypt the wallet.");
    }
    fill(strWalletPass.begin(), strWalletPass.end(), '\0');
    munlock(&strWalletPass[0], strWalletPass.capacity());

}
Value encryptwallet(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "encryptwallet <passphrase>\n"
            "Encrypts the wallet with <passphrase>.");
    real_encryptwallet(Wallet_GetDefault(),params[0].get_str());
    return Value::null;
}

Value sc_encryptwallet(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "encryptwallet <wallet> <passphrase>\n"
            "Encrypts the wallet with <passphrase>.");
    real_encryptwallet(pWallet,params[1].get_str());
    return Value::null;
}

Value real_validateaddress(CWallet *pWallet, const std::string &addr)
{
    CSolidCoinAddress address(addr);
    bool isValid = address.IsValid();

    Object ret;
    ret.push_back(Pair("isvalid", isValid));
    if (isValid)
    {
        // Call Hash160ToAddress() so we always return current ADDRESSVERSION version of the address:
        string currentAddress = address.ToString();
        ret.push_back(Pair("address", currentAddress));
        ret.push_back(Pair("ismine", (pWallet->HaveKey(address) > 0)));
        {
            if (pWallet->mapAddressBook.count(address))
                ret.push_back(Pair("account", pWallet->mapAddressBook[address]));
        }
    }
    return ret;
}

Value validateaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "validateaddress <solidcoinaddress>\n"
            "Return information about <solidcoinaddress>.");
    return real_validateaddress(Wallet_GetDefault(),params[0].get_str());
}

Value sc_validateaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_validateaddress <wallet> <solidcoinaddress>\n"
            "Return information about <solidcoinaddress>.");
    return real_validateaddress(pWallet,params[1].get_str());
}

Value sc_gethash(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "sc_gethash [data]\n"
            "  \"data\" : block data, must be 128 bytes\n"
            "Returns Object named hash.");

    vector<unsigned char> vchData = ParseHex(params[0].get_str());
    if (vchData.size() != 128)  throw JSONRPCError(-8, "Invalid parameter");

    CBlock data;
    data.blk = *((BLOCK_DATA*)&vchData[0]);
    uint256 rethash=data.GetHash();

    Object result;
    result.push_back(Pair("hash",   HexNumStr(BEGIN(rethash), END(rethash))));
    return result;


}

Value sc_testwork(const Array& params, bool fHelp)
{
    if (fHelp || params.size()!=1)
        throw runtime_error(
            "sc_testwork [data]\n"
            "Tries to solve the block and returns true if it was successful.\n"
            "  \"data\" : block data\n"
            "  \"target_share\" : little endian hash target of a share\n"
            "  \"target_real\" : little endian hash target of the real diff\n");


    vector<unsigned char> vchData = ParseHex(params[0].get_str());
    if (vchData.size()%128 != 0 )  throw JSONRPCError(-8, "Invalid parameter");

    Object result;
    Array works;

    int nCount=vchData.size()/128;
    for(int x=0;x<nCount;x++)
    {
        Object work;
        CBlock data;
        data.blk = *((BLOCK_DATA*)&vchData[x*128]);
        bool bBlockValid=false,bShareValid=false;
        bool bPowerBlock=false,bFoundBlock=false;
        uint256 txid;
        uint256 blockhash;

        {
            MUTEX_LOCK(cs_getwork);
            BOOST_FOREACH(CBlock* pBlock, g_NewBlockList)
            {
                int64 nTimeDiff =SolidTime_Get()-data.blk.nTime;
                if(nTimeDiff<0) nTimeDiff*=-1;
                if(data.blk.hashMerkleRoot==pBlock->blk.hashMerkleRoot)
                {
                    if(nTimeDiff<g_qShareTimeWindow)
                    {
                        pBlock->blk.nTime = data.blk.nTime;
                        pBlock->blk.nNonce1 = data.blk.nNonce1;
                        pBlock->blk.nNonce2 = data.blk.nNonce2;
                        pBlock->blk.nNonce3 = data.blk.nNonce3;
                        pBlock->blk.nNonce4 = data.blk.nNonce4;
                        bPowerBlock=pBlock->m_bPower;
                        {
                            MUTEX_LOCK(cs_main);
                            LOCK_WALLET_ACCESS();
                            bShareValid=bBlockValid=Block_CheckWork(pBlock, *Wallet_GetDefault(), g_MiningReserveKey,&blockhash);
                        }
                        txid=pBlock->vtx[0].GetHash();
                        //not a valid block, see if its a valid share
                        if(bBlockValid==false && g_MiningShareTargetInt>0 && pBlock->GetHash()<=g_MiningShareTargetHash)    bShareValid=true;
                        if(bShareValid) g_MiningTestWorkShares+=g_MiningShareTargetInt;
                        else            g_MiningTestWorkInvalids++;
                    }
                    bFoundBlock=true;
                    break;
                }
            }
        }
        if(bFoundBlock==false)
        {
            blockhash=data.GetHash();
            g_MiningTestWorkInvalids++;
        }

        work.push_back(Pair("share_valid",     bShareValid));
        work.push_back(Pair("block_valid",     bBlockValid));
        work.push_back(Pair("block_hash",       blockhash.ToString().c_str()));
        if(bBlockValid)
        {
            work.push_back(Pair("block_txid",     txid.ToString().c_str()));
            work.push_back(Pair("block_power",     bPowerBlock));
        }
        works.push_back(work);
    }
    result.push_back(Pair("work", works));
    //obj.push_back(Pair("generate",      (bool)false));
    //result.push_back(Pair("block_value",   HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real))));
    //result.push_back(Pair("target_share",   HexNumStr(BEGIN(hashTarget_share), END(hashTarget_share))));
    //0011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233
    return result;

}
Value sc_getwork(const Array& params, bool fHelp)
{
    if (fHelp || params.size()>0)
        throw runtime_error(
            "sc_getwork\n"
            "Returns formatted hash data to work on:\n"
            "  \"data\" : block data\n"
            "  \"target_share\" : little endian hash target of a share\n"
            "  \"target_real\" : little endian hash target of the real diff\n");

    if(!fTestNet)
    {
        if (vNodes.empty())             throw JSONRPCError(-9, "SolidCoin is not connected!");
        if (Block_IsInitialDownload())   throw JSONRPCError(-10, "SolidCoin is downloading blocks...");
    }

    static unsigned int nTransactionsUpdatedLast=0;
    static CBlockIndex* pindexPrev=0;
    static int64 nStart=0;
    BLOCK_DATA blk;

    {
        MUTEX_LOCK(cs_getwork);
        CBlock *pBlock=0;
        if (pindexPrev != g_pBlockBestIndex || (nTransactionsUpdated != nTransactionsUpdatedLast && SolidTime_Get() - nStart > 20))
        {
            if (pindexPrev != g_pBlockBestIndex)
            {
                std::vector<CBlock*>::iterator iter;
                for(iter=g_NewBlockList.begin();iter!=g_NewBlockList.end();)
                {
                    if( (*iter)->blk.nBlockNum < g_qBlockBestHeight-3)
                    {
                        CBlock *pBlock=*iter;
                        iter=g_NewBlockList.erase(iter);
                        delete pBlock;

                    }
                    else
                    {
                        ++iter;
                    }
                }
            }
            nTransactionsUpdatedLast = nTransactionsUpdated;
            pindexPrev = g_pBlockBestIndex;
            extern string g_TrustedNodePublicKey;
            if((pindexPrev->blk.nBlockNum%2)==1 && g_TrustedNodePublicKey.length())
            {
                //Sleep( (40+(rand()%30)) * 1000);  //supernode testing
            }
            nStart = SolidTime_Get();
            {
                MUTEX_LOCK(cs_main);
                MUTEX_LOCK2(cs_mapTransactions);
                LOCK_WALLET_ACCESS();
                g_MiningReserveKey.SetWallet(Wallet_GetDefault());
                pBlock = Block_Create(Wallet_GetDefault(), g_MiningIDString,g_MiningReserveKey);
            }
            g_NewBlockList.push_back(pBlock);
        }
        else
        {
            pBlock = g_NewBlockList[g_NewBlockList.size()-1];  //get last item as its most recent
        }
        blk=pBlock->blk;
    }

    static uint64 nonce2=0;
    blk.nTime = SolidTime_Get();
    blk.nNonce1 = 0;
    blk.nNonce2 = nonce2++;
    blk.nNonce3 = 0;
    blk.nNonce4 = 0;

    //char pmidstate[32];char pdata[128];char phash1[64];
    Object result;
    result.push_back(Pair("data",     HexStr((unsigned char*)&blk,((unsigned char*)&blk)+128 )));
    uint256 hashTarget_real = CBigNum().SetCompact(blk.dwBits).getuint256();
    if(g_MiningShareTargetInt<=0)   result.push_back(Pair("target_share",   HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real))));
    else                            result.push_back(Pair("target_share",   HexNumStr(BEGIN(g_MiningShareTargetHash), END(g_MiningShareTargetHash))));
    result.push_back(Pair("target_real",   HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real))));
    //string str1=HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real));
    //string str2=HexNumStr(BEGIN(g_MiningShareTargetHash), END(g_MiningShareTargetHash));
    //printf("hash1:%s\nhash2:%s",str1.c_str(),str2.c_str());
    return result;
}

Value sc_getblockbynumber(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "sc_getblockbynumber height\n"
            "Dumps the block existing at specified height");

    int64 height = params[0].get_int64();
    if (height > g_qBlockBestHeight)
        throw runtime_error(
            "sc_getblockbynumber height\n"
            "Dumps the block existing at specified height");

    string blkname = strprintf("blk%d", height);

    CBlockIndex* pindex=0;
    bool found = false;

    for (map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.begin();mi != g_BlockIndexMap.end(); ++mi)
    {
    	pindex = (*mi).second;
        if ((pindex->blk.nBlockNum == height) && (pindex->IsInMainChain()))
        {
            found = true;
            break;
        }
    }

    if (!found)
        throw runtime_error(
            "sc_getblockbynumber height\n"
            "Dumps the block existing at specified height");

    CBlock block;
    block.ReadFromDisk(pindex);
    block.BuildMerkleTree();
    char miner_id[13]={0};
    memcpy(miner_id,block.blk.miner_id,12);
    for(int x=0;x<12;x++) if(miner_id[x]<0) miner_id[x]=' ';

    Object obj;
    obj.push_back(Pair("hash", block.GetHash().ToString().c_str()));
    obj.push_back(Pair("version", block.blk.nVersion));
    obj.push_back(Pair("prev_block", block.blk.hashPrevBlock.ToString().c_str()));
    obj.push_back(Pair("mrkl_root", block.blk.hashMerkleRoot.ToString().c_str()));
    obj.push_back(Pair("time", (uint64_t)block.blk.nTime));
    obj.push_back(Pair("bits", (uint64_t)block.blk.dwBits));
    obj.push_back(Pair("blocknum", (uint64_t)block.blk.nBlockNum));
    obj.push_back(Pair("nonce1", (uint64_t)block.blk.nNonce1));
    obj.push_back(Pair("nonce2", (uint64_t)block.blk.nNonce2));
    obj.push_back(Pair("nonce3", (uint64_t)block.blk.nNonce3));
    obj.push_back(Pair("nonce4", (uint64_t)block.blk.nNonce4));
    obj.push_back(Pair("miner_id", miner_id));
    obj.push_back(Pair("miner_idhex", HexNumStr(block.blk.miner_id, block.blk.miner_id+12)));

    obj.push_back(Pair("n_tx", (int)block.vtx.size()));
    obj.push_back(Pair("size", (int)::GetSerializeSize(block, SER_NETWORK)));

    Array tx;
    for (int i = 0; i < block.vtx.size(); i++)
    {
    	Object txobj;
        txobj.push_back(Pair("hash", block.vtx[i].GetHash().ToString().c_str()));
        txobj.push_back(Pair("version", block.vtx[i].nVersion));
        txobj.push_back(Pair("lock_time", (uint64_t)block.vtx[i].nLockTime));
        txobj.push_back(Pair("size",(int)::GetSerializeSize(block.vtx[i], SER_NETWORK)));

        Array tx_vin;
        for (int j = 0; j < block.vtx[i].vin.size(); j++)
        {
            Object vino;

            Object vino_outpt;

            vino_outpt.push_back(Pair("hash",block.vtx[i].vin[j].prevout.hash.ToString().c_str()));
            vino_outpt.push_back(Pair("n", (uint64_t)block.vtx[i].vin[j].prevout.n));

            vino.push_back(Pair("prev_out", vino_outpt));

            if (block.vtx[i].vin[j].prevout.IsNull())
                vino.push_back(Pair("coinbase", HexStr(block.vtx[i].vin[j].scriptSig.begin(),block.vtx[i].vin[j].scriptSig.end(), false).c_str()));
            else
                vino.push_back(Pair("scriptSig",block.vtx[i].vin[j].scriptSig.ToString().c_str()));

            if (block.vtx[i].vin[j].nSequence != UINT_MAX)
                vino.push_back(Pair("sequence", (uint64_t)block.vtx[i].vin[j].nSequence));

            tx_vin.push_back(vino);
        }

        Array tx_vout;
        for (int j = 0; j < block.vtx[i].vout.size(); j++) {
            Object vouto;

            vouto.push_back(Pair("value",
                (double)block.vtx[i].vout[j].nValue / (double)COIN));
            vouto.push_back(Pair("scriptPubKey",
            block.vtx[i].vout[j].scriptPubKey.ToString().c_str()));

            tx_vout.push_back(vouto);
        }

        txobj.push_back(Pair("in", tx_vin));
        txobj.push_back(Pair("out", tx_vout));

        tx.push_back(txobj);
    }

    obj.push_back(Pair("tx", tx));

    Array mrkl;
    for (int i = 0; i < block.vMerkleTree.size(); i++)
    	mrkl.push_back(block.vMerkleTree[i].ToString().c_str());

    obj.push_back(Pair("mrkl_tree", mrkl));

    return obj;
}

// Call Table

pair<string, rpcfn_type> pCallTable[] =
{
    make_pair("help",                   &help),
    make_pair("stop",                   &stop),
    make_pair("getblockcount",          &getblockcount),
    make_pair("getblocknumber",         &getblocknumber),
    make_pair("getconnectioncount",     &getconnectioncount),
    make_pair("getdifficulty",          &getdifficulty),

    make_pair("backupwallet",           &backupwallet),
    make_pair("encryptwallet",          &encryptwallet),
    make_pair("getbalance",             &getbalance),
    make_pair("getinfo",                &getinfo),
    make_pair("getnewaddress",          &getnewaddress),
    make_pair("getaccountaddress",      &getaccountaddress),
    make_pair("getaccount",             &getaccount),
    make_pair("getaddressesbyaccount",  &getaddressesbyaccount),
    make_pair("getreceivedbyaddress",   &getreceivedbyaddress),
    make_pair("getreceivedbyaccount",   &getreceivedbyaccount),
    make_pair("gettransaction",         &gettransaction),
    make_pair("keypoolrefill",          &keypoolrefill),
    make_pair("listreceivedbyaddress",  &listreceivedbyaddress),
    make_pair("listreceivedbyaccount",  &listreceivedbyaccount),
    make_pair("listtransactions",       &listtransactions),
    make_pair("listaccounts",           &listaccounts),
    make_pair("move",                   &movecmd),
    make_pair("sendfrom",               &sendfrom),
    make_pair("sendmany",               &sendmany),
    make_pair("sendtoaddress",          &sendtoaddress),
    make_pair("setaccount",             &setaccount),
    make_pair("validateaddress",        &validateaddress),
    make_pair("walletpassphrase",       &walletpassphrase),
    make_pair("walletpassphrasechange", &walletpassphrasechange),
    make_pair("walletlock",             &walletlock),

    make_pair("sc_backupwallet",            &sc_backupwallet),
    make_pair("sc_encryptwallet",           &sc_encryptwallet),
    make_pair("sc_getaccountaddress",       &sc_getaccountaddress),
    make_pair("sc_getaccount",              &sc_getaccount),
    make_pair("sc_getaddressesbyaccount",   &sc_getaddressesbyaccount),
    make_pair("sc_getblockbynumber",        &sc_getblockbynumber),
    make_pair("sc_getnewaddress",           &sc_getnewaddress),
    make_pair("sc_getreceivedbyaccount",    &sc_getreceivedbyaccount),
    make_pair("sc_getwork",                 &sc_getwork),
    make_pair("sc_gethash",                 &sc_gethash),
    make_pair("sc_getinfo",                 &sc_getinfo),
    make_pair("sc_gettxfee",                &sc_gettxfee),
    make_pair("sc_gettransaction",          &sc_gettransaction),
    make_pair("sc_getreceivedbyaddress",    &sc_getreceivedbyaddress),
    make_pair("sc_getbalance",              &sc_getbalance),
    make_pair("sc_getmining",               &sc_getmining),
    make_pair("sc_keypoolrefill",           &sc_keypoolrefill),
    make_pair("sc_listaccounts",            &sc_listaccounts),
    make_pair("sc_listreceivedbyaccount",   &sc_listreceivedbyaccount),
    make_pair("sc_listreceivedbyaddress",   &sc_listreceivedbyaddress),
    make_pair("sc_listtransactions",        &sc_listtransactions),
    make_pair("sc_move",                    &sc_movecmd),
    make_pair("sc_sendtoaddress",           &sc_sendtoaddress),
    make_pair("sc_sendfrom",                &sc_sendfrom),
    make_pair("sc_sendmany",                &sc_sendmany),
    make_pair("sc_setaccount",              &sc_setaccount),
    make_pair("sc_setmining",               &sc_setmining),
    make_pair("sc_testwork",                &sc_testwork),
    make_pair("sc_validateaddress",         &sc_validateaddress),
    make_pair("sc_walletpassphrase",        &sc_walletpassphrase),
    make_pair("sc_walletpassphrasechange",  &sc_walletpassphrasechange),
    make_pair("sc_walletlock",              &sc_walletlock),
};
map<string, rpcfn_type> mapCallTable(pCallTable, pCallTable + sizeof(pCallTable)/sizeof(pCallTable[0]));

string pAllowInSafeMode[] =
{
    "help",
    "stop",
    "getblockcount",
    "getblocknumber",
    "getconnectioncount",
    "getdifficulty",
    "sc_getmining",
    "sc_setmining",
    "getinfo",
    "getnewaddress",
    "getaccountaddress",
    "setlabel", // deprecated
    "getaccount",
    "getlabel", // deprecated
    "getaddressesbyaccount",
    "getaddressesbylabel", // deprecated
    "backupwallet",
    "keypoolrefill",
    "walletpassphrase",
    "walletlock",
    "validateaddress",
    "sc_getnewaddress",
    "sc_getwork",
    "sc_getaccount",
    "sc_testwork",
    "sc_gethash",
    "sc_getinfo",
};
set<string> setAllowInSafeMode(pAllowInSafeMode, pAllowInSafeMode + sizeof(pAllowInSafeMode)/sizeof(pAllowInSafeMode[0]));




//
// HTTP protocol
//
// This ain't Apache.  We're just using HTTP header for the length field
// and to be compatible with other JSON-RPC implementations.
//

string HTTPPost(const string& strMsg, const map<string,string>& mapRequestHeaders)
{
    ostringstream s;
    s << "POST / HTTP/1.1\r\n"
      << "User-Agent: solidcoin-json-rpc/" << FormatFullVersion() << "\r\n"
      << "Host: 127.0.0.1\r\n"
      << "Content-Type: application/json\r\n"
      << "Content-Length: " << strMsg.size() << "\r\n"
      << "Accept: application/json\r\n";
    BOOST_FOREACH(const PAIRTYPE(string, string)& item, mapRequestHeaders)
        s << item.first << ": " << item.second << "\r\n";
    s << "\r\n" << strMsg;

    return s.str();
}

string rfc1123Time()
{
    char buffer[64];
    time_t now;
    time(&now);
    struct tm* now_gmt = gmtime(&now);
    string locale(setlocale(LC_TIME, NULL));
    setlocale(LC_TIME, "C"); // we want posix (aka "C") weekday/month strings
    strftime(buffer, sizeof(buffer), "%a, %d %b %Y %H:%M:%S +0000", now_gmt);
    setlocale(LC_TIME, locale.c_str());
    return string(buffer);
}

static string HTTPReply(int nStatus, const string& strMsg)
{
    if (nStatus == 401)
        return strprintf("HTTP/1.0 401 Authorization Required\r\n"
            "Date: %s\r\n"
            "Server: solidcoin-json-rpc/%s\r\n"
            "WWW-Authenticate: Basic realm=\"jsonrpc\"\r\n"
            "Content-Type: text/html\r\n"
            "Content-Length: 296\r\n"
            "\r\n"
            "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\"\r\n"
            "\"http://www.w3.org/TR/1999/REC-html401-19991224/loose.dtd\">\r\n"
            "<HTML>\r\n"
            "<HEAD>\r\n"
            "<TITLE>Error</TITLE>\r\n"
            "<META HTTP-EQUIV='Content-Type' CONTENT='text/html; charset=ISO-8859-1'>\r\n"
            "</HEAD>\r\n"
            "<BODY><H1>401 Unauthorized.</H1></BODY>\r\n"
            "</HTML>\r\n", rfc1123Time().c_str(), FormatFullVersion().c_str());
    string strStatus;
         if (nStatus == 200) strStatus = "OK";
    else if (nStatus == 400) strStatus = "Bad Request";
    else if (nStatus == 403) strStatus = "Forbidden";
    else if (nStatus == 404) strStatus = "Not Found";
    else if (nStatus == 500) strStatus = "Internal Server Error";
    return strprintf(
            "HTTP/1.1 %d %s\r\n"
            "Date: %s\r\n"
            "Connection: close\r\n"
            "Content-Length: %d\r\n"
            "Content-Type: application/json\r\n"
            "Server: solidcoin-json-rpc/%s\r\n"
            "\r\n"
            "%s",
        nStatus,
        strStatus.c_str(),
        rfc1123Time().c_str(),
        strMsg.size(),
        FormatFullVersion().c_str(),
        strMsg.c_str());
}

int ReadHTTPStatus(std::basic_istream<char>& stream)
{
    string str;
    getline(stream, str);
    vector<string> vWords;
    boost::split(vWords, str, boost::is_any_of(" "));
    if (vWords.size() < 2)
        return 500;
    return atoi(vWords[1].c_str());
}

int ReadHTTPHeader(std::basic_istream<char>& stream, map<string, string>& mapHeadersRet)
{
    int nLen = 0;
    loop
    {
        string str;
        std::getline(stream, str);
        if (str.empty() || str == "\r")
            break;
        string::size_type nColon = str.find(":");
        if (nColon != string::npos)
        {
            string strHeader = str.substr(0, nColon);
            boost::trim(strHeader);
            boost::to_lower(strHeader);
            string strValue = str.substr(nColon+1);
            boost::trim(strValue);
            mapHeadersRet[strHeader] = strValue;
            if (strHeader == "content-length")
                nLen = atoi(strValue.c_str());
        }
    }
    return nLen;
}

int ReadHTTP(std::basic_istream<char>& stream, map<string, string>& mapHeadersRet, string& strMessageRet)
{
    mapHeadersRet.clear();
    strMessageRet = "";

    // Read status
    int nStatus = ReadHTTPStatus(stream);

    // Read header
    int nLen = ReadHTTPHeader(stream, mapHeadersRet);
    if (nLen < 0 || nLen > MAX_SIZE)
        return 500;

    // Read message
    if (nLen > 0)
    {
        vector<char> vch(nLen);
        stream.read(&vch[0], nLen);
        strMessageRet = string(vch.begin(), vch.end());
    }

    return nStatus;
}

string EncodeBase64(string s)
{
    BIO *b64, *bmem;
    BUF_MEM *bptr;

    b64 = BIO_new(BIO_f_base64());
    BIO_set_flags(b64, BIO_FLAGS_BASE64_NO_NL);
    bmem = BIO_new(BIO_s_mem());
    b64 = BIO_push(b64, bmem);
    BIO_write(b64, s.c_str(), s.size());
    BIO_flush(b64);
    BIO_get_mem_ptr(b64, &bptr);

    string result(bptr->data, bptr->length);
    BIO_free_all(b64);

    return result;
}

string DecodeBase64(string s)
{
    BIO *b64, *bmem;

    char* buffer = static_cast<char*>(calloc(s.size(), sizeof(char)));

    b64 = BIO_new(BIO_f_base64());
    BIO_set_flags(b64, BIO_FLAGS_BASE64_NO_NL);
    bmem = BIO_new_mem_buf(const_cast<char*>(s.c_str()), s.size());
    bmem = BIO_push(b64, bmem);
    BIO_read(bmem, buffer, s.size());
    BIO_free_all(bmem);

    string result(buffer);
    free(buffer);
    return result;
}

bool HTTPAuthorized(map<string, string>& mapHeaders)
{
    string strAuth = mapHeaders["authorization"];
    if (strAuth.substr(0,6) != "Basic ")
        return false;
    string strUserPass64 = strAuth.substr(6); boost::trim(strUserPass64);
    string strUserPass = DecodeBase64(strUserPass64);
    string::size_type nColon = strUserPass.find(":");
    if (nColon == string::npos)
        return false;
    string strUser = strUserPass.substr(0, nColon);
    string strPassword = strUserPass.substr(nColon+1);
    return (strUser == Setting_Get("rpcuser") && strPassword == Setting_Get("rpcpassword"));
}

//
// JSON-RPC protocol.  SolidCoin speaks version 1.0 for maximum compatibility,
// but uses JSON-RPC 1.1/2.0 standards for parts of the 1.0 standard that were
// unspecified (HTTP errors and contents of 'error').
//
// 1.0 spec: http://json-rpc.org/wiki/specification
// 1.2 spec: http://groups.google.com/group/json-rpc/web/json-rpc-over-http
// http://www.codeproject.com/KB/recipes/JSON_Spirit.aspx
//

string JSONRPCRequest(const string& strMethod, const Array& params, const Value& id)
{
    Object request;
    request.push_back(Pair("method", strMethod));
    request.push_back(Pair("params", params));
    request.push_back(Pair("id", id));
    return write_string(Value(request), false) + "\n";
}

string JSONRPCReply(const Value& result, const Value& error, const Value& id)
{
    Object reply;
    if (error.type() != null_type)
        reply.push_back(Pair("result", Value::null));
    else
        reply.push_back(Pair("result", result));
    reply.push_back(Pair("error", error));
    reply.push_back(Pair("id", id));
    return write_string(Value(reply), false) + "\n";
}

void ErrorReply(std::ostream& stream, const Object& objError, const Value& id)
{
    // Send error reply from json-rpc error object
    int nStatus = 500;
    int code = find_value(objError, "code").get_int();
    if (code == -32600) nStatus = 400;
    else if (code == -32601) nStatus = 404;
    string strReply = JSONRPCReply(Value::null, objError, id);
    stream << HTTPReply(nStatus, strReply) << std::flush;
}

bool ClientAllowed(const string& strAddress)
{
    if (strAddress == asio::ip::address_v4::loopback().to_string())
        return true;
    const vector<string>& vAllow = Setting_GetVector("rpcallowip");
    BOOST_FOREACH(string strAllow, vAllow)
        if (WildcardMatch(strAddress, strAllow))
            return true;
    return false;
}

#ifdef USE_SSL
//
// IOStream device that speaks SSL but can also speak non-SSL
//
class SSLIOStreamDevice : public iostreams::device<iostreams::bidirectional> {
public:
    SSLIOStreamDevice(SSLStream &streamIn, bool fUseSSLIn) : stream(streamIn)
    {
        fUseSSL = fUseSSLIn;
        fNeedHandshake = fUseSSLIn;
    }

    void handshake(ssl::stream_base::handshake_type role)
    {
        if (!fNeedHandshake) return;
        fNeedHandshake = false;
        stream.handshake(role);
    }
    std::streamsize read(char* s, std::streamsize n)
    {
        handshake(ssl::stream_base::server); // HTTPS servers read first
        if (fUseSSL) return stream.read_some(asio::buffer(s, n));
        return stream.next_layer().read_some(asio::buffer(s, n));
    }
    std::streamsize write(const char* s, std::streamsize n)
    {
        handshake(ssl::stream_base::client); // HTTPS clients write first
        if (fUseSSL) return asio::write(stream, asio::buffer(s, n));
        return asio::write(stream.next_layer(), asio::buffer(s, n));
    }
    bool connect(const std::string& server, const std::string& port)
    {
        ip::tcp::resolver resolver(stream.get_io_service());
        ip::tcp::resolver::query query(server.c_str(), port.c_str());
        ip::tcp::resolver::iterator endpoint_iterator = resolver.resolve(query);
        ip::tcp::resolver::iterator end;
        boost::system::error_code error = asio::error::host_not_found;
        while (error && endpoint_iterator != end)
        {
            stream.lowest_layer().close();
            stream.lowest_layer().connect(*endpoint_iterator++, error);
        }
        if (error)
            return false;
        return true;
    }

private:
    bool fNeedHandshake;
    bool fUseSSL;
    SSLStream& stream;
};
#endif

void ThreadRPCServer(void* parg)
{
    IMPLEMENT_RANDOMIZE_STACK(ThreadRPCServer(parg));
    try
    {
        vnThreadsRunning[4]++;
        ThreadRPCServer2(parg);
        vnThreadsRunning[4]--;
    }
    catch (std::exception& e) {
        vnThreadsRunning[4]--;
        PrintException(&e, "ThreadRPCServer()");
    } catch (...) {
        vnThreadsRunning[4]--;
        PrintException(NULL, "ThreadRPCServer()");
    }
    printf("ThreadRPCServer exiting\n");
}

void ThreadRPCServer2(void* parg)
{
    printf("ThreadRPCServer started\n");

    if (Setting_Get("rpcuser") == "" && Setting_Get("rpcpassword") == "")
    {
        string strWhatAmI = "To use solidcoind";
        if (Setting_GetBOOL("server"))
            strWhatAmI = strprintf(_("To use the %s option"), "\"-server\"");
        else if (Setting_GetBOOL("daemon"))
            strWhatAmI = strprintf(_("To use the %s option"), "\"-daemon\"");
        PrintConsole(
            _("Warning: %s, you must set rpcpassword=<password>\nin the configuration file: %s\n"
              "If the file does not exist, create it with owner-readable-only file permissions.\n"),
                strWhatAmI.c_str(),
                GetConfigFile().c_str());
        CreateThread(Shutdown, NULL);
        return;
    }

    bool fUseSSL = Setting_GetBOOL("rpcssl");
    asio::ip::address bindAddress = Setting_Exist("rpcallowip") ? asio::ip::address_v4::any() : asio::ip::address_v4::loopback();

    asio::io_service io_service;
    ip::tcp::endpoint endpoint(bindAddress, Setting_GetINT64("rpcport"));
    ip::tcp::acceptor acceptor(io_service, endpoint);

    acceptor.set_option(boost::asio::ip::tcp::acceptor::reuse_address(true));

#ifdef USE_SSL
    ssl::context context(io_service, ssl::context::sslv23);
    if (fUseSSL)
    {
        context.set_options(ssl::context::no_sslv2);
        filesystem::path certfile = Setting_Get("rpcsslcertificatechainfile");
        if (!certfile.is_complete()) certfile = filesystem::path(GetDataDir()) / certfile;
        if (filesystem::exists(certfile)) context.use_certificate_chain_file(certfile.string().c_str());
        else printf("ThreadRPCServer ERROR: missing server certificate file %s\n", certfile.string().c_str());
        filesystem::path pkfile = Setting_Get("rpcsslprivatekeyfile");
        if (!pkfile.is_complete()) pkfile = filesystem::path(GetDataDir()) / pkfile;
        if (filesystem::exists(pkfile)) context.use_private_key_file(pkfile.string().c_str(), ssl::context::pem);
        else printf("ThreadRPCServer ERROR: missing server private key file %s\n", pkfile.string().c_str());

        string ciphers = Setting_Get("rpcsslciphers");
        SSL_CTX_set_cipher_list(context.impl(), ciphers.c_str());
    }
#else
    if (fUseSSL)
        throw runtime_error("-rpcssl=1, but solidcoin compiled without full openssl libraries.");
#endif

    loop
    {
        // Accept connection
#ifdef USE_SSL
        SSLStream sslStream(io_service, context);
        SSLIOStreamDevice d(sslStream, fUseSSL);
        iostreams::stream<SSLIOStreamDevice> stream(d);
#else
        ip::tcp::iostream stream;
#endif

        ip::tcp::endpoint peer;
        vnThreadsRunning[4]--;
#ifdef USE_SSL
        acceptor.accept(sslStream.lowest_layer(), peer);
#else
        acceptor.accept(*stream.rdbuf(), peer);
#endif
        vnThreadsRunning[4]++;
        if (fShutdown)
            return;

        // Restrict callers by IP
        if (!ClientAllowed(peer.address().to_string()))
        {
            // Only send a 403 if we're not using SSL to prevent a DoS during the SSL handshake.
            if (!fUseSSL)
                stream << HTTPReply(403, "") << std::flush;
            continue;
        }

        map<string, string> mapHeaders;
        string strRequest;

        boost::thread api_caller(ReadHTTP, boost::ref(stream), boost::ref(mapHeaders), boost::ref(strRequest));
        if (!api_caller.timed_join(boost::posix_time::seconds(Setting_GetINT64("rpctimeout"))))
        {   // Timed out:
            acceptor.cancel();
            printf("ThreadRPCServer ReadHTTP timeout\n");
            continue;
        }

        // Check authorization
        if (mapHeaders.count("authorization") == 0)
        {
            stream << HTTPReply(401, "") << std::flush;
            continue;
        }
        if (!HTTPAuthorized(mapHeaders))
        {
            // Deter brute-forcing short passwords
            if (Setting_Get("rpcpassword").size() < 15)
                Sleep(50);

            stream << HTTPReply(401, "") << std::flush;
            printf("ThreadRPCServer incorrect password attempt\n");
            continue;
        }

        Value id = Value::null;
        try
        {
            // Parse request
            Value valRequest;
            if (!read_string(strRequest, valRequest) || valRequest.type() != obj_type)
                throw JSONRPCError(-32700, "Parse error");
            const Object& request = valRequest.get_obj();

            // Parse id now so errors from here on will have the id
            id = find_value(request, "id");

            // Parse method
            Value valMethod = find_value(request, "method");
            if (valMethod.type() == null_type)
                throw JSONRPCError(-32600, "Missing method");
            if (valMethod.type() != str_type)
                throw JSONRPCError(-32600, "Method must be a string");
            string strMethod = valMethod.get_str();
            if (strMethod != "sc_getwork")
                printf("ThreadRPCServer method=%s\n", strMethod.c_str());

            // Parse params
            Value valParams = find_value(request, "params");
            Array params;
            if (valParams.type() == array_type)
                params = valParams.get_array();
            else if (valParams.type() == null_type)
                params = Array();
            else
                throw JSONRPCError(-32600, "Params must be an array");

            // Find method
            map<string, rpcfn_type>::iterator mi = mapCallTable.find(strMethod);
            if (mi == mapCallTable.end())
                throw JSONRPCError(-32601, "Method not found");

            // Observe safe mode
            string strWarning = GetWarnings("rpc");
            if (strWarning != "" && !Setting_GetBOOL("disablesafemode") && !setAllowInSafeMode.count(strMethod))
                throw JSONRPCError(-2, string("Safe mode: ") + strWarning);

            try
            {
                // Execute
                Value result = (*(*mi).second)(params, false);

                // Send reply
                string strReply = JSONRPCReply(result, Value::null, id);
                stream << HTTPReply(200, strReply) << std::flush;
            }
            catch (std::exception& e)
            {
                ErrorReply(stream, JSONRPCError(-1, e.what()), id);
            }
        }
        catch (Object& objError)
        {
            ErrorReply(stream, objError, id);
        }
        catch (std::exception& e)
        {
            ErrorReply(stream, JSONRPCError(-32700, e.what()), id);
        }
    }
}


Object CallRPC(const string& host, const string& port, const string& base64UserString, const string& strMethod, const Array& params)
{

    // Connect to localhost
    bool fUseSSL = Setting_GetBOOL("rpcssl");
#ifdef USE_SSL
    asio::io_service io_service;
    ssl::context context(io_service, ssl::context::sslv23);
    context.set_options(ssl::context::no_sslv2);
    SSLStream sslStream(io_service, context);
    SSLIOStreamDevice d(sslStream, fUseSSL);
    iostreams::stream<SSLIOStreamDevice> stream(d);
    if (!d.connect(host,port))
        throw runtime_error("couldn't connect to server ("+host+":"+port+")");
#else
    if (fUseSSL)
        throw runtime_error("-rpcssl=1, but solidcoin compiled without full openssl libraries.");

    ip::tcp::iostream stream(host, port);
    if (stream.fail())
        throw runtime_error("couldn't connect to server ("+host+":"+port+")");
#endif


    // HTTP basic authentication
    map<string, string> mapRequestHeaders;
    mapRequestHeaders["Authorization"] = string("Basic ") + base64UserString;

    // Send request
    string strRequest = JSONRPCRequest(strMethod, params, 1);
    string strPost = HTTPPost(strRequest, mapRequestHeaders);
    stream << strPost << std::flush;

    // Receive reply
    map<string, string> mapHeaders;
    string strReply;
    int nStatus = ReadHTTP(stream, mapHeaders, strReply);
    if (nStatus == 401)
        throw runtime_error("incorrect rpcuser or rpcpassword (authorization failed)");
    else if (nStatus >= 400 && nStatus != 400 && nStatus != 404 && nStatus != 500)
        throw runtime_error(strprintf("server returned HTTP error %d", nStatus));
    else if (strReply.empty())
        throw runtime_error("no response from server");

    // Parse reply
    Value valReply;
    if (!read_string(strReply, valReply))
        throw runtime_error("couldn't parse reply from server");
    const Object& reply = valReply.get_obj();
    if (reply.empty())
        throw runtime_error("expected reply to have result, error and id properties");

    return reply;
}


template<typename T>
void ConvertTo(Value& value)
{
    if (value.type() == str_type)
    {
        // reinterpret string as unquoted json value
        Value value2;
        if (!read_string(value.get_str(), value2))
            throw runtime_error("type mismatch");
        value = value2.get_value<T>();
    }
    else
    {
        value = value.get_value<T>();
    }
}

int CommandLineRPC(int argc, std::string *argv)
{
    string strPrint;
    int nRet = 0;
    try
    {
        // Skip switches
        while (argc > 1 && IsSwitchChar(argv[1][0]))
        {
            argc--;
            argv++;
        }

        // Method
        if (argc < 2)
            throw runtime_error("too few parameters");
        string strMethod = argv[1];

        // Parameters default to strings
        Array params;
        for (int i = 2; i < argc; i++)
            params.push_back(argv[i]);
        int n = params.size();

        //
        // Special case non-string parameter types
        //
        if (strMethod == "sc_setmining"            && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "sc_setmining"            && n > 1) ConvertTo<bool>(params[1]);
        if (strMethod == "sendtoaddress"          && n > 1) ConvertTo<double>(params[1]);
        if (strMethod == "getreceivedbyaddress"   && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "getreceivedbyaccount"   && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "listreceivedbyaddress"  && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "listreceivedbyaddress"  && n > 1) ConvertTo<bool>(params[1]);
        if (strMethod == "listreceivedbyaccount"  && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "listreceivedbyaccount"  && n > 1) ConvertTo<bool>(params[1]);
        if (strMethod == "getbalance"             && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "move"                   && n > 2) ConvertTo<double>(params[2]);
        if (strMethod == "sendfrom"               && n > 2) ConvertTo<double>(params[2]);
        if (strMethod == "sendfrom"               && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "listtransactions"       && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "listtransactions"       && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "listaccounts"           && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "walletpassphrase"       && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_gettxfee"            && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_getblockbynumber"     && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "sc_getreceivedbyaddress"&& n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_getreceivedbyaccount"   && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_getbalance"          && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_listaccounts"        && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_listtransactions"    && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_listtransactions"    && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_listreceivedbyaddress"  && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_listreceivedbyaddress"  && n > 2) ConvertTo<bool>(params[2]);
        if (strMethod == "sc_listreceivedbyaccount"  && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_listreceivedbyaccount"  && n > 2) ConvertTo<bool>(params[2]);
        if (strMethod == "sc_sendtoaddress"          && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_sendtoaddress"          && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_move"                   && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_sendfrom"               && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_sendfrom"               && n > 4) ConvertTo<boost::int64_t>(params[4]);
        if (strMethod == "sendmany" && n > 1)
        {
            string s = params[1].get_str();
            Value v;
            if (!read_string(s, v) || v.type() != obj_type) throw runtime_error("type mismatch");
            params[1] = v.get_obj();
        }
        if (strMethod == "sendmany"                 && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_sendmany" && n > 2)
        {
            string s = params[2].get_str();
            Value v;
            if (!read_string(s, v) || v.type() != obj_type) throw runtime_error("type mismatch");
            params[2] = v.get_obj();
        }
        if (strMethod == "sc_sendmany"              && n > 3) ConvertTo<boost::int64_t>(params[3]);

        // Execute

        if (Setting_Get("rpcuser") == "" && Setting_Get("rpcpassword") == "")
            throw runtime_error(strprintf(
                _("You must set rpcpassword=<password> in the configuration file:\n%s\n"
                "If the file does not exist, create it with owner-readable-only file permissions."),
                    GetConfigFile().c_str()));

        string strUserPass64 = EncodeBase64(Setting_Get("rpcuser") + ":" + Setting_Get("rpcpassword"));
        string host=Setting_Get("rpcconnect");
        string port=Setting_Get("rpcport");
        Object reply = CallRPC(host,port,strUserPass64,strMethod, params);

        // Parse reply
        const Value& result = find_value(reply, "result");
        const Value& error  = find_value(reply, "error");

        if (error.type() != null_type)
        {
            // Error
            strPrint = "error: " + write_string(error, false);
            int code = find_value(error.get_obj(), "code").get_int();
            nRet = abs(code);
        }
        else
        {
            // Result
            if (result.type() == null_type)         strPrint = "";
            else if (result.type() == str_type)     strPrint = result.get_str();
            else                                    strPrint = write_string(result, true);
        }
    }
    catch (std::exception& e)
    {
        strPrint = string("error: ") + e.what();
        nRet = 87;
    }
    catch (...)
    {
        PrintException(NULL, "CommandLineRPC()");
    }

    if (strPrint != "")
    {
#if defined(__WXMSW__) && defined(GUI)
        // Windows GUI apps can't print to command line, so settle for a message box yuck
        MyMessageBox(strPrint, "SolidCoin", wxOK);
#else
        fprintf((nRet == 0 ? stdout : stderr), "%s\n", strPrint.c_str());
#endif
    }
    return nRet;
}




#ifdef TEST
int main(int argc, char *argv[])
{
#ifdef _MSC_VER
    // Turn off microsoft heap dump noise
    _CrtSetReportMode(_CRT_WARN, _CRTDBG_MODE_FILE);
    _CrtSetReportFile(_CRT_WARN, CreateFile("NUL", GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, 0));
#endif
    setbuf(stdin, NULL);
    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    try
    {
        if (argc >= 2 && string(argv[1]) == "-server")
        {
            printf("server ready\n");
            ThreadRPCServer(NULL);
        }
        else
        {
            return CommandLineRPC(argc, argv);
        }
    }
    catch (std::exception& e) {
        PrintException(&e, "main()");
    } catch (...) {
        PrintException(NULL, "main()");
    }
    return 0;
}
#endif


CCriticalSection g_MineControl,g_MineStatusControl;
std::vector<MINING_POOL*> g_MinePoolList;
std::vector<std::string> g_MineStatusList;
int64 g_MiningRPCDelay=60000;
void Mining_AddPool(MINING_POOL *pPool)
{
    MINING_POOL *pNewPool= new MINING_POOL;
    *pNewPool=*pPool;
    pNewPool->pThreads=0;
    pNewPool->nRunning=0;
    pNewPool->qLastHeight=-1;
    pNewPool->qLastGetWork=0;
    pNewPool->pCallRPC=0;
    pNewPool->nQuit=0;
    pNewPool->nFirstWork=0;
    pNewPool->pCRIT = new CCriticalSection;



    pNewPool->strUserPass64 = EncodeBase64(pNewPool->user + ":" + pNewPool->pass);


    {
        MUTEX_LOCK(g_MineControl);
        Mining_RemovePool(pNewPool->id);
        g_MinePoolList.push_back(pNewPool);
    }
}

void Mining_RemovePool(int id)
{
    MUTEX_LOCK(g_MineControl);

    for(int x=0;x<g_MinePoolList.size();x++)
    {
        if(id==g_MinePoolList[x]->id)
        {
            g_MinePoolList[x]->nQuit=1;
            for(int z=0;z<g_MinePoolList[x]->nRunning;z++)  g_MinePoolList[x]->pThreads[z].nQuit=1;
            g_MinePoolList[x]->id=-1;
        }
    }
}


void SolidCoinMiner(void* parg)
{
    unsigned char *blockdata = new unsigned char[512];
    int nCountAtOnce=0;
    uint256 hashTarget;
    uint256 endhash;
    MINING_POOL_THREAD *pThreadPool=(MINING_POOL_THREAD*)parg;

    bool bGetNewData=true;
    SetThreadPriority(THREAD_PRIORITY_LOWEST);

    BLOCK_DATA *pBlock=(BLOCK_DATA*)blockdata;

    int64 nStartTime=0;
    int64 nMilliStart=0;

    while (1)
    {
        if (pThreadPool->nQuit)
        {
            pThreadPool->nQuit++;
            delete[] blockdata;
            return;
        }
        if (fShutdown)
        {
            delete[] blockdata;
            return;
        }
        if(pThreadPool->nHasWork==0)
        {
            Sleep(1);
            continue;
        }

        if(pThreadPool->nHasWork==1)
        {
            MUTEX_LOCK(*pThreadPool->pCRIT);
            *pBlock = *pThreadPool->pBlockData;
            nStartTime = pBlock->nTime+1;
            pBlock->nNonce1 = ((uint64)pThreadPool->nID<<32);
            hashTarget=*pThreadPool->pTarget;
            pThreadPool->nHasWork=2;
            nMilliStart=GetTimeMillis();

        }



        pBlock->nTime=nStartTime+ ((GetTimeMillis()-nMilliStart)/1000);    //update time

        nCountAtOnce=0;
        while(++nCountAtOnce<200)
        {
            ++pBlock->nNonce1;
            BlockHash_1(endhash,blockdata);
            if(endhash < hashTarget)
            {
                pThreadPool->qShareCount++;     //increase share count
                MUTEX_LOCK(*pThreadPool->pCRIT);
                pThreadPool->pBlockFoundList->push_back(*pBlock); //submit solution
                break;
            }
        }
        pThreadPool->qCurHashCount+=nCountAtOnce;
    }
}

struct CALLRPC_THREAD
{
    std::string name;
    std::string host;
    std::string port;
    std::string userpass64;
    std::string method;
    Array params;

    Object result;
    std::string error;
    bool bFinished;
};
void GetRPCThread(void* parg)
{
    CALLRPC_THREAD *pThread = (CALLRPC_THREAD*)parg;
    try
    {
        pThread->result = CallRPC(pThread->host,pThread->port,pThread->userpass64,pThread->method,pThread->params);
    }
    catch (std::exception& e)
    {
        pThread->error = str(boost::format("%s - %s error: %s") % pThread->name % pThread->method % e.what());
    }
    catch (Object& objError)
    {
        pThread->error = str(boost::format("%s - %s error") % pThread->name % pThread->method);
    }
    pThread->bFinished=true;
}

CALLRPC_THREAD* DoCallRPC(const std::string &method, Array &params, MINING_POOL *pPool)
{
    CALLRPC_THREAD *pThread = new CALLRPC_THREAD;

    pThread->name = pPool->name;
    pThread->host = pPool->host;
    pThread->port = itostr(pPool->nPort);
    pThread->userpass64 = pPool->strUserPass64;
    pThread->method=method;
    pThread->params = params;
    pThread->bFinished=false;
    CreateThread(GetRPCThread, pThread);
    return pThread;
}

void HandlePool(MINING_POOL *pPool)
{
    //see if blockcount has changed since we last got a share
    CALLRPC_THREAD *pRPCThread = (CALLRPC_THREAD*)pPool->pCallRPC;

    if(pRPCThread && pRPCThread->bFinished)
    {
        if(pRPCThread->error.length())
        {
            g_MineStatusList.push_back(pRPCThread->error);
            goto END_RPC;
        }

        try
        {
            if(pRPCThread->method=="sc_getwork")
            {
                const Value& result = find_value(pRPCThread->result, "result");
                const Value& error  = find_value(pRPCThread->result, "error");
                if (error.type() != null_type)
                {
                    Object obj = error.get_obj();
                    Value msg = find_value(obj, "message");
                    throw runtime_error(msg.get_str());
                }
                else if (result.type() == obj_type)
                {
                    Object obj = result.get_obj();
                    Value data = find_value(obj, "data");
                    Value target = find_value(obj, "target_share");
                    vector<unsigned char> vchData = ParseHex(data.get_str());
                    if (vchData.size() != 128)
                    {
                        throw runtime_error("Wrong data size returned");
                    }

                    CBlock block1;
                    block1.blk = *((BLOCK_DATA*)&vchData[0]);
                    CBigNum bn;
                    bn.SetHex(target.get_str());

                    {
                        MUTEX_LOCK(*pPool->pCRIT);
                        pPool->hashTarget = bn.getuint256();
                        pPool->blk = block1.blk;
                        pPool->qLastHeight=g_qBlockBestHeight;

                        if(!pPool->pThreads)
                        {
                            if(pPool->nFirstWork<2)
                            {
                                g_MineStatusList.push_back(str(boost::format("First work received at %s") % pPool->name));
                                pPool->nFirstWork++;
                            }


                            pPool->pThreads = new MINING_POOL_THREAD[pPool->nThreads];
                            while(pPool->nRunning<pPool->nThreads)
                            {
                                MINING_POOL_THREAD *pMineThread=&pPool->pThreads[pPool->nRunning];
                                pMineThread->nID=pPool->nRunning;
                                pMineThread->qShareCount=0;
                                pMineThread->nHasWork=0;
                                pMineThread->nQuit=0;
                                pMineThread->qLastHashCount=0;
                                pMineThread->qCurHashCount=0;
                                pMineThread->pTarget=&pPool->hashTarget;
                                pMineThread->pCRIT=pPool->pCRIT;
                                pMineThread->pBlockData=&pPool->blk;
                                pMineThread->pBlockFoundList=&pPool->blockFoundList;
                                CreateThread(SolidCoinMiner,pMineThread);
                                pPool->nRunning++;
                            }
                        }

                        //printf("GETWORK: Got work (%s prevhash) %"PRI64d" height:\n",pPool->blk.hashPrevBlock.ToString().c_str(),g_qBlockBestHeight);
                        int x=0;
                        while(x<pPool->nThreads)
                        {
                            pPool->pThreads[x].nHasWork=1;
                            x++;
                        }
                    }
                }
            }
        }
        catch (std::exception& e)
        {
            printf("%s - %s error: %s \n",pPool->name.c_str(),pRPCThread->method.c_str() ,e.what());
            pRPCThread->error = str(boost::format("%s - %s error: %s") % pRPCThread->name % pRPCThread->method % e.what());
            g_MineStatusList.push_back(pRPCThread->error);
        }
END_RPC:
        delete pRPCThread;
        pRPCThread=0;
        pPool->pCallRPC=0;
    }

    if((GetTimeMillis()-pPool->qLastGetWork)>=g_MiningRPCDelay || ((GetTimeMillis()-pPool->qLastGetWork)>=1000 && pPool->qLastHeight!=g_qBlockBestHeight) )
    {
        if(pPool->qLastHeight!=g_qBlockBestHeight)
        {
            //stop mining threads until we get new work
            MUTEX_LOCK(*pPool->pCRIT);
            int x=0;
            while(x<pPool->nRunning)    pPool->pThreads[x++].nHasWork=0;
        }
        pPool->qLastGetWork=GetTimeMillis();

        if(pRPCThread==0)
        {
            Array params;

            if(pPool->nFirstWork==0)
            {
                g_MineStatusList.push_back(str(boost::format("Starting miners at %s") % pPool->name));
                pPool->nFirstWork++;
            }
            pPool->pCallRPC=pRPCThread=DoCallRPC("sc_getwork",params,pPool);
        }
    }

    //submit any shares found

    {
        MUTEX_LOCK(*pPool->pCRIT);
        if(pPool->blockFoundList.size())
        {
            unsigned char *pBlockData = new unsigned char[128*pPool->blockFoundList.size()];
            for(int x=0;x<pPool->blockFoundList.size();x++) memcpy(pBlockData+(x*128),&pPool->blockFoundList[x],128);
            Array params;
            params.push_back(HexStr(pBlockData,pBlockData+(128*pPool->blockFoundList.size()) ));
            if(pRPCThread==0)
            {
                pPool->pCallRPC=pRPCThread=DoCallRPC("sc_testwork",params,pPool);
            }
            pPool->blockFoundList.clear();
        }
    }
}
void SolidCoinMineController(void* parg)
{
    int64 nHPSTimerStart=0;
    int64 nMilliTime=0;
    int64 nMiningStart=GetTimeMillis();
    int64 qLastShares=0;
    int64 qLastInvalids=0;
    bool bAutoStartCheck=true;
    std::vector<MINING_SPEEDS> *speedlist=0;
    std::vector<MINING_SPEEDS> testworkspeeds;

    g_MiningRPCDelay = Setting_GetINT64("mining_workmaxperiod")*1000;
    g_MiningShareTargetInt = Setting_GetINT64("mining_sharetarget");
    g_MiningIDString = Setting_Get("mining_id");
    g_qShareTimeWindow= Setting_GetINT64("mining_servertimewindow");
    if(g_qShareTimeWindow<1)    g_qShareTimeWindow=1;
    if(g_qShareTimeWindow>30)   g_qShareTimeWindow=30;


    if(g_MiningShareTargetInt>0)
    {
        CBigNum bn=g_bnBlockProofOfWorkLimit;
        bn= bn/g_MiningShareTargetInt;
        g_MiningShareTargetHash = bn.getuint256();
    }

    while(1)
    {
        if(fShutdown) return;
        bool bUpdateGUI=false;
        speedlist=0;

        if (bAutoStartCheck && GetTimeMillis() - nMiningStart > 30000)
        {
            MUTEX_LOCK(cs_main);
            for(int x=0;x<g_MiningPoolList.size();x++)
            {
                MINING_POOL *pPool=&g_MiningPoolList[x];

                if(pPool->bAutoStart==false || pPool->nRunning) continue;
                pPool->id=x;
                Mining_AddPool(pPool);
                pPool->nRunning=1;
                pPool->qLastHashPerSec=0;
                pPool->qTotalHash=0;
                pPool->qTotalShares=0;
            }
            bAutoStartCheck=false;
        }

        if (GetTimeMillis() - nHPSTimerStart > 500)
        {
            nMilliTime = GetTimeMillis()-nHPSTimerStart;
            speedlist = new std::vector<MINING_SPEEDS>;
            bUpdateGUI=true;
            nHPSTimerStart=GetTimeMillis();
        }

        {
            MUTEX_LOCK(g_MineControl);
            for(int x=0;x<g_MinePoolList.size();x++)
            {
                if(g_MinePoolList[x]->nQuit)
                {
                    int nCount=0;   //make sure all worker threads have quit
                    for(int z=0;z<g_MinePoolList[x]->nRunning;z++)
                    {
                        if(g_MinePoolList[x]->pThreads[z].nQuit==2) nCount++;
                    }
                    if(nCount==g_MinePoolList[x]->nRunning && g_MinePoolList[x]->nQuit==1)
                    {
                        CalledUpdateMiningStatus(str(boost::format("All mining threads closed at %s") % g_MinePoolList[x]->name));
                        g_MinePoolList[x]->nQuit=2;
                    }

                    CALLRPC_THREAD *pRPCThread = (CALLRPC_THREAD*)g_MinePoolList[x]->pCallRPC;
                    if(pRPCThread && pRPCThread->bFinished)
                    {
                        delete pRPCThread;
                        g_MinePoolList[x]->pCallRPC=0;

                    }
                    if(g_MinePoolList[x]->nQuit==2 && g_MinePoolList[x]->pCallRPC==0)
                    {
                        delete g_MinePoolList[x];
                        g_MinePoolList.erase(g_MinePoolList.begin()+x);     //remove pool
                        x--;    //since we've removed an item , need to decrement x
                    }
                    continue;
                }

                HandlePool(g_MinePoolList[x]);
                if(bUpdateGUI)
                {
                    MINING_SPEEDS speed;
                    speed.nID=g_MinePoolList[x]->id;
                    speed.qHash=0;
                    speed.qShares=0;
                    for(int z=0;z<g_MinePoolList[x]->nRunning;z++)
                    {
                        MINING_POOL_THREAD*pThreadPool=&g_MinePoolList[x]->pThreads[z];
                        speed.qShares+=pThreadPool->qShareCount;
                        if(pThreadPool->qCurHashCount>pThreadPool->qLastHashCount)
                        {
                            speed.qHash+=pThreadPool->qCurHashCount-pThreadPool->qLastHashCount;
                            pThreadPool->qLastHashCount=pThreadPool->qCurHashCount;
                        }
                    }
                    speedlist->push_back(speed);
                }
            }
        }

        BOOST_FOREACH(std::string s,g_MineStatusList)
        {
            UIThreadCall(boost::bind(CalledUpdateMiningStatus,s));
        }
        g_MineStatusList.clear();

        if(bUpdateGUI)
        {
            MINING_SPEEDS speed;
            if(g_MiningTestWorkShares!=qLastShares || g_MiningTestWorkInvalids!=qLastInvalids)
            {
                int64 qShares=g_MiningTestWorkShares-qLastShares;
                qLastShares=g_MiningTestWorkShares;
                qLastInvalids=g_MiningTestWorkInvalids;
                speed.qTime=GetTimeMillis();
                speed.qShares=qShares;
                testworkspeeds.push_back(speed);
            }

            speed.nID=-1;
            speed.qShares=0;

            std::vector<MINING_SPEEDS>::iterator iter;
            for(iter=testworkspeeds.begin();iter!=testworkspeeds.end();)
            {
                if(GetTimeMillis()-iter->qTime > 30000)
                {
                    iter = testworkspeeds.erase(iter);
                }
                else
                {
                    speed.qShares+=iter->qShares;
                    ++iter;
                }
            }

            speed.qHash=(speed.qShares*131072)/30;
            speed.qShares=g_MiningTestWorkShares;
            speed.qInvalid=g_MiningTestWorkInvalids;

            speedlist->push_back(speed);
        }

        if(bUpdateGUI)
        {
            UIThreadCall(boost::bind(CalledUpdateMiningHash,nMilliTime,speedlist));
        }
        Sleep(20);
    }


}
