//Please read license.txt for licensing and copyright information

#include "headers.h"
#include "cryptopp/sha.h"
#include "db.h"
#include "net.h"
#include "init.h"
#include "wallet_func.h"
#undef printf
#include <boost/asio.hpp>
#include <boost/iostreams/concepts.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/algorithm/string.hpp>
#ifdef USE_SSL
#include <boost/asio/ssl.hpp>
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>
typedef boost::asio::ssl::stream<boost::asio::ip::tcp::socket> SSLStream;
#endif
#include "json/json_spirit_reader_template.h"
#include "json/json_spirit_writer_template.h"
#include "json/json_spirit_utils.h"
#define printf OutputDebugStringF

using namespace std;
using namespace boost;
using namespace boost::asio;
using namespace json_spirit;



typedef Value(*rpcfn_type)(const Array& params, bool fHelp);
void ThreadRPCServer2(void* parg);
void ThreadRPCServer3(void* parg);
Object JSONRPCError(int code, const string& message);

bool g_bAllowKeepAlive = false;
string g_RPCUser;
string g_RPCPass;

extern map<string, rpcfn_type> mapCallTable;
extern set<string> setAllowInSafeMode;
//
// HTTP protocol
//
// This ain't Apache.  We're just using HTTP header for the length field
// and to be compatible with other JSON-RPC implementations.
//

string HTTPPost(const string& strMsg, const map<string,string>& mapRequestHeaders)
{
    ostringstream s;
    s << "POST / HTTP/1.1\r\n"
      << "User-Agent: solidcoin-json-rpc/" << FormatFullVersion() << "\r\n"
      << "Host: 127.0.0.1\r\n"
      << "Content-Type: application/json\r\n"
      << "Content-Length: " << strMsg.size() << "\r\n"
      << "Accept: application/json\r\n";
    BOOST_FOREACH(const PAIRTYPE(string, string)& item, mapRequestHeaders)
        s << item.first << ": " << item.second << "\r\n";
    s << "\r\n" << strMsg;

    return s.str();
}

string rfc1123Time()
{
    char buffer[64];
    time_t now;
    time(&now);
    struct tm* now_gmt = gmtime(&now);
    string locale(setlocale(LC_TIME, NULL));
    setlocale(LC_TIME, "C"); // we want posix (aka "C") weekday/month strings
    strftime(buffer, sizeof(buffer), "%a, %d %b %Y %H:%M:%S +0000", now_gmt);
    setlocale(LC_TIME, locale.c_str());
    return string(buffer);
}

static string HTTPReply(int nStatus, const string& strMsg, bool keepalive)
{
    if (nStatus == 401)
        return strprintf("HTTP/1.0 401 Authorization Required\r\n"
            "Date: %s\r\n"
            "Server: solidcoin-json-rpc/%s\r\n"
            "WWW-Authenticate: Basic realm=\"jsonrpc\"\r\n"
            "Content-Type: text/html\r\n"
            "Content-Length: 296\r\n"
            "\r\n"
            "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\"\r\n"
            "\"http://www.w3.org/TR/1999/REC-html401-19991224/loose.dtd\">\r\n"
            "<HTML>\r\n"
            "<HEAD>\r\n"
            "<TITLE>Error</TITLE>\r\n"
            "<META HTTP-EQUIV='Content-Type' CONTENT='text/html; charset=ISO-8859-1'>\r\n"
            "</HEAD>\r\n"
            "<BODY><H1>401 Unauthorized.</H1></BODY>\r\n"
            "</HTML>\r\n", rfc1123Time().c_str(), FormatFullVersion().c_str());
    string strStatus;
         if (nStatus == 200) strStatus = "OK";
    else if (nStatus == 400) strStatus = "Bad Request";
    else if (nStatus == 403) strStatus = "Forbidden";
    else if (nStatus == 404) strStatus = "Not Found";
    else if (nStatus == 500) strStatus = "Internal Server Error";
    return strprintf(
            "HTTP/1.1 %d %s\r\n"
            "Date: %s\r\n"
            "Connection: %s\r\n"
            "Content-Length: %d\r\n"
            "Content-Type: application/json\r\n"
            "Server: solidcoin-json-rpc/%s\r\n"
            "\r\n"
            "%s",
        nStatus,
        strStatus.c_str(),
        rfc1123Time().c_str(),
		keepalive ? "keep-alive" : "close",
        strMsg.size(),
        FormatFullVersion().c_str(),
        strMsg.c_str());
}

int ReadHTTPStatus(std::basic_istream<char>& stream, int &proto)
{
    string str;
    getline(stream, str);
    vector<string> vWords;
    boost::split(vWords, str, boost::is_any_of(" "));
    if (vWords.size() < 2)
        return 500;
    proto = 0;
    const char *ver = strstr(str.c_str(), "HTTP/1.");
    if (ver != NULL)
        proto = atoi(ver+7);
	return atoi(vWords[1].c_str());
}

int ReadHTTPHeader(std::basic_istream<char>& stream, map<string, string>& mapHeadersRet)
{
    int nLen = 0;
    loop
    {
        string str;
        std::getline(stream, str);
        if (str.empty() || str == "\r")
            break;
        string::size_type nColon = str.find(":");
        if (nColon != string::npos)
        {
            string strHeader = str.substr(0, nColon);
            boost::trim(strHeader);
            boost::to_lower(strHeader);
            string strValue = str.substr(nColon+1);
            boost::trim(strValue);
            mapHeadersRet[strHeader] = strValue;
            if (strHeader == "content-length")
                nLen = atoi(strValue.c_str());
        }
    }
    return nLen;
}

int ReadHTTP(std::basic_istream<char>& stream, map<string, string>& mapHeadersRet, string& strMessageRet)
{
    mapHeadersRet.clear();
    strMessageRet = "";

    // Read status
	int nProto;
    int nStatus = ReadHTTPStatus(stream, nProto);

    // Read header
    int nLen = ReadHTTPHeader(stream, mapHeadersRet);
    if (nLen < 0 || nLen > MAX_SIZE)
        return 500;

    // Read message
    if (nLen > 0)
    {
        vector<char> vch(nLen);
        stream.read(&vch[0], nLen);
        strMessageRet = string(vch.begin(), vch.end());
    }

	string sConHdr=mapHeadersRet["connection"];

    if ( (sConHdr != "close") && (sConHdr != "keep-alive") )
    {
        if(nProto >= 1)
            mapHeadersRet["connection"]="keep-alive";
        else
            mapHeadersRet["connection"]="close";
    }

    return nStatus;
    return nStatus;
}

bool HTTPAuthorized(map<string, string>& mapHeaders)
{
    string strAuth = mapHeaders["authorization"];
    if (strAuth.substr(0,6) != "Basic ")
        return false;
    string strUserPass64 = strAuth.substr(6); boost::trim(strUserPass64);
    string strUserPass = DecodeBase64(strUserPass64);
    string::size_type nColon = strUserPass.find(":");
    if (nColon == string::npos)
        return false;
    string strUser = strUserPass.substr(0, nColon);
    string strPassword = strUserPass.substr(nColon+1);
	return (strUser == g_RPCUser && strPassword == g_RPCPass);
}

//
// JSON-RPC protocol.  SolidCoin speaks version 1.0 for maximum compatibility,
// but uses JSON-RPC 1.1/2.0 standards for parts of the 1.0 standard that were
// unspecified (HTTP errors and contents of 'error').
//
// 1.0 spec: http://json-rpc.org/wiki/specification
// 1.2 spec: http://groups.google.com/group/json-rpc/web/json-rpc-over-http
// http://www.codeproject.com/KB/recipes/JSON_Spirit.aspx
//

void PrintConsole(const char* format, ...)
{
    char buffer[50000];
    int limit = sizeof(buffer);
    va_list arg_ptr;
    va_start(arg_ptr, format);
    int ret = _vsnprintf(buffer, limit, format, arg_ptr);
    va_end(arg_ptr);
    if (ret < 0 || ret >= limit)
    {
        ret = limit - 1;
        buffer[limit-1] = 0;
    }
    debugprintf(INFO, "%s", buffer);
#if defined(__WXMSW__) && defined(GUI)
    MyMessageBox(buffer, "SolidCoin", wxOK | wxICON_EXCLAMATION);
#else
    fprintf(stdout, "%s", buffer);
#endif
}


string JSONRPCRequest(const string& strMethod, const Array& params, const Value& id)
{
    Object request;
    request.push_back(Pair("method", strMethod));
    request.push_back(Pair("params", params));
    request.push_back(Pair("id", id));
    return write_string(Value(request), false) + "\n";
}

string JSONRPCReply(const Value& result, const Value& error, const Value& id)
{
    Object reply;
    if (error.type() != null_type)
        reply.push_back(Pair("result", Value::null));
    else
        reply.push_back(Pair("result", result));
    reply.push_back(Pair("error", error));
    reply.push_back(Pair("id", id));
    return write_string(Value(reply), false) + "\n";
}

void ErrorReply(std::ostream& stream, const Object& objError, const Value& id)
{
    // Send error reply from json-rpc error object
    int nStatus = 500;
    int code = find_value(objError, "code").get_int();
    if (code == -32600) nStatus = 400;
    else if (code == -32601) nStatus = 404;
    string strReply = JSONRPCReply(Value::null, objError, id);
    stream << HTTPReply(nStatus, strReply, false) << std::flush;
}

bool ClientAllowed(const string& strAddress)
{
    if (strAddress == asio::ip::address_v4::loopback().to_string())
        return true;
    const vector<string>& vAllow = Setting_GetVector("rpcallowip");
    BOOST_FOREACH(string strAllow, vAllow)
        if (WildcardMatch(strAddress, strAllow))
            return true;
    return false;
}

#ifdef USE_SSL
//
// IOStream device that speaks SSL but can also speak non-SSL
//
class SSLIOStreamDevice : public iostreams::device<iostreams::bidirectional> {
public:
    SSLIOStreamDevice(SSLStream &streamIn, bool fUseSSLIn) : stream(streamIn)
    {
        fUseSSL = fUseSSLIn;
        fNeedHandshake = fUseSSLIn;
    }

    void handshake(ssl::stream_base::handshake_type role)
    {
        if (!fNeedHandshake) return;
        fNeedHandshake = false;
        stream.handshake(role);
    }
    std::streamsize read(char* s, std::streamsize n)
    {
        handshake(ssl::stream_base::server); // HTTPS servers read first
        if (fUseSSL) return stream.read_some(asio::buffer(s, n));
        return stream.next_layer().read_some(asio::buffer(s, n));
    }
    std::streamsize write(const char* s, std::streamsize n)
    {
        handshake(ssl::stream_base::client); // HTTPS clients write first
        if (fUseSSL) return asio::write(stream, asio::buffer(s, n));
        return asio::write(stream.next_layer(), asio::buffer(s, n));
    }
    bool connect(const std::string& server, const std::string& port)
    {
        ip::tcp::resolver resolver(stream.get_io_service());
        ip::tcp::resolver::query query(server.c_str(), port.c_str());
        ip::tcp::resolver::iterator endpoint_iterator = resolver.resolve(query);
        ip::tcp::resolver::iterator end;
        boost::system::error_code error = asio::error::host_not_found;
        while (error && endpoint_iterator != end)
        {
            stream.lowest_layer().close();
            stream.lowest_layer().connect(*endpoint_iterator++, error);
        }
        if (error)
            return false;
        return true;
    }

private:
    bool fNeedHandshake;
    bool fUseSSL;
    SSLStream& stream;
};
#endif

class AcceptedConnection
{
    public:
#ifdef USE_SSL
    SSLStream sslStream;
    SSLIOStreamDevice d;
    iostreams::stream<SSLIOStreamDevice> stream;
#else
    ip::tcp::iostream stream;
#endif

    ip::tcp::endpoint peer;

#ifdef USE_SSL
    AcceptedConnection(asio::io_service &io_service, ssl::context &context,bool fUseSSL) : sslStream(io_service, context), d(sslStream, fUseSSL),stream(d)
    {
        ;
    }
#else
    AcceptedConnection(void)
    {
        ;
    }
#endif
};

void ThreadRPCServer(void* parg)
{
    IMPLEMENT_RANDOMIZE_STACK(ThreadRPCServer(parg));
    try
    {
        vnThreadsRunning[4]++;
        ThreadRPCServer2(parg);
        vnThreadsRunning[4]--;
    }
    catch (std::exception& e) {
        vnThreadsRunning[4]--;
        PrintException(&e, "ThreadRPCServer()");
    } catch (...) {
        vnThreadsRunning[4]--;
        PrintException(NULL, "ThreadRPCServer()");
    }
    debugprintf(ALWAYS, "ThreadRPCServer exiting\n");
}

void ThreadRPCServer2(void* parg)
{
    debugprintf(ALWAYS, "ThreadRPCServer started\n");

    if (Setting_Get("rpcuser") == "" && Setting_Get("rpcpassword") == "")
    {
        string strWhatAmI = "To use solidcoind";
        if (Setting_GetBOOL("server"))
            strWhatAmI = strprintf(_("To use the %s option"), "\"-server\"");
        else if (Setting_GetBOOL("daemon"))
            strWhatAmI = strprintf(_("To use the %s option"), "\"-daemon\"");
        PrintConsole(
            _("Warning: %s, you must set rpcpassword=<password>\nin the configuration file: %s\n"
              "If the file does not exist, create it with owner-readable-only file permissions.\n"),
                strWhatAmI.c_str(),
                GetConfigFile().c_str());
        CreateThread(Shutdown, NULL);
        return;
    }

    bool fUseSSL = Setting_GetBOOL("rpcssl");
	g_bAllowKeepAlive = Setting_GetBOOL("rpcallowkeepalive");
	g_RPCUser = Setting_Get("rpcuser");
	g_RPCPass = Setting_Get("rpcpassword");


    asio::ip::address bindAddress = Setting_Exist("rpcallowip") ? asio::ip::address_v4::any() : asio::ip::address_v4::loopback();

    asio::io_service io_service;
    ip::tcp::endpoint endpoint(bindAddress, Setting_GetINT64("rpcport"));
    ip::tcp::acceptor acceptor(io_service, endpoint);

    acceptor.set_option(boost::asio::ip::tcp::acceptor::reuse_address(true));

#ifdef USE_SSL
    ssl::context context(io_service, ssl::context::sslv23);
    if (fUseSSL)
    {
        context.set_options(ssl::context::no_sslv2);
        filesystem::path certfile = Setting_Get("rpcsslcertificatechainfile");
        if (!certfile.is_complete()) certfile = filesystem::path(GetDataDir()) / certfile;
        if (filesystem::exists(certfile)) context.use_certificate_chain_file(certfile.string().c_str());
        else debugprintf(ERR, "ThreadRPCServer ERROR: missing server certificate file %s\n", certfile.string().c_str());
        filesystem::path pkfile =  Setting_Get("rpcsslprivatekeyfile");
        if (!pkfile.is_complete()) pkfile = filesystem::path(GetDataDir()) / pkfile;
        if (filesystem::exists(pkfile)) context.use_private_key_file(pkfile.string().c_str(), ssl::context::pem);
        else debugprintf(ERR, "ThreadRPCServer ERROR: missing server private key file %s\n", pkfile.string().c_str());

        string ciphers = Setting_Get("rpcsslciphers");
        SSL_CTX_set_cipher_list(context.impl(), ciphers.c_str());
    }
#else
    if (fUseSSL)
        throw runtime_error("-rpcssl=1, but bitcoin compiled without full openssl libraries.");
#endif

    loop
    {
// Accept connection
#ifdef USE_SSL
        AcceptedConnection *conn=new AcceptedConnection(io_service, context, fUseSSL);
#else
        AcceptedConnection *conn=new AcceptedConnection();
#endif

        vnThreadsRunning[4]--;
#ifdef USE_SSL
        acceptor.accept(conn->sslStream.lowest_layer(), conn->peer);
#else
        acceptor.accept(*conn->stream.rdbuf(), conn->peer);
#endif
        vnThreadsRunning[4]++;

        if (fShutdown)
        {
            delete conn;
            return;
        }

        // Restrict callers by IP
        if (!ClientAllowed(conn->peer.address().to_string()))
        {
            // Only send a 403 if we're not using SSL to prevent a DoS during the SSL handshake.
            if (!fUseSSL)
                conn->stream << HTTPReply(403, "", false) << std::flush;
            delete conn;
        }
        else
            CreateThread(ThreadRPCServer3, (void *) conn);
    }
}

void ThreadRPCServer3(void* parg)
{
    IMPLEMENT_RANDOMIZE_STACK(ThreadRPCServer3(parg));
    ++vaMultiThreads1;
    AcceptedConnection *conn=(AcceptedConnection *) parg;

	bool fKeepAlive = g_bAllowKeepAlive;

    do
    {
		if (fShutdown)
        {
           conn->stream.close();
           break;
        }

        map<string, string> mapHeaders;
        string strRequest;

        ReadHTTP(conn->stream, mapHeaders, strRequest);

        // Check authorization
        if (mapHeaders.count("authorization") == 0)
        {
            conn->stream << HTTPReply(401, "", false) << std::flush;
            break;
        }
        if (!HTTPAuthorized(mapHeaders))
        {
            // Deter brute-forcing short passwords
            if (Setting_Get("rpcpassword").size() < 15)
                Sleep(50);

            conn->stream << HTTPReply(401, "", false) << std::flush;
            debugprintf(WARN, "ThreadRPCServer incorrect password attempt\n");
            break;
        }

		//check client-side request for keep-alives
		if (mapHeaders["connection"] == "close")
            fKeepAlive = false;

        //ThreadUnsafeRPC();

        Value id = Value::null;
        try
        {
            // Parse request
            Value valRequest;
            if (!read_string(strRequest, valRequest) || valRequest.type() != obj_type)
                throw JSONRPCError(-32700, "Parse error");
            const Object& request = valRequest.get_obj();

            // Parse id now so errors from here on will have the id
            id = find_value(request, "id");

            // Parse method
            Value valMethod = find_value(request, "method");
            if (valMethod.type() == null_type)      throw JSONRPCError(-32600, "Missing method");
            if (valMethod.type() != str_type)       throw JSONRPCError(-32600, "Method must be a string");
            string strMethod = valMethod.get_str();
            //if (strMethod != "getwork")             debugprintf(INFO, "ThreadRPCServer method=%s\n", strMethod.c_str());

            // Parse params
            Value valParams = find_value(request, "params");
            Array params;
            if (valParams.type() == array_type)         params = valParams.get_array();
            else if (valParams.type() == null_type)     params = Array();
            else                                        throw JSONRPCError(-32600, "Params must be an array");

            // Find method
            map<string, rpcfn_type>::iterator mi = mapCallTable.find(strMethod);
            if (mi == mapCallTable.end())               throw JSONRPCError(-32601, "Method not found");

            // Observe safe mode
            string strWarning = GetWarnings("rpc");
            if (strWarning != "" && !Setting_GetBOOL("disablesafemode") && !setAllowInSafeMode.count(strMethod))
                throw JSONRPCError(-2, string("Safe mode: ") + strWarning);

            try
            {
                Value result = (*(*mi).second)(params, false);              // Execute
                string strReply = JSONRPCReply(result, Value::null, id);    // Send reply
                conn->stream << HTTPReply(200, strReply, fKeepAlive) << std::flush;
            }
            catch (std::exception& e)
            {
                ErrorReply(conn->stream, JSONRPCError(-1, e.what()), id);
				fKeepAlive = false;
            }
            catch (Object& e)
            {
                ErrorReply(conn->stream, e, id);
				fKeepAlive = false;
            }
        }
        catch (Object& objError)
        {
            ErrorReply(conn->stream, objError, id);
            break;
        }
        catch (std::exception& e)
        {
            ErrorReply(conn->stream, JSONRPCError(-32700, e.what()), id);
            break;
        }
    }
    while (fKeepAlive);
    delete conn;
    //delete fUnsafe.release();
    --vaMultiThreads1;
}



Object CallRPC(const string& host, const string& port, const string& base64UserString, const string& strMethod, const Array& params)
{
    // Connect to localhost
    bool fUseSSL = Setting_GetBOOL("rpcssl");
#ifdef USE_SSL
    asio::io_service io_service;
    ssl::context context(io_service, ssl::context::sslv23);
    context.set_options(ssl::context::no_sslv2);
    SSLStream sslStream(io_service, context);
    SSLIOStreamDevice d(sslStream, fUseSSL);
    iostreams::stream<SSLIOStreamDevice> stream(d);
    if (!d.connect(host,port))
        throw runtime_error("couldn't connect to server ("+host+":"+port+")");
#else
    if (fUseSSL)
        throw runtime_error("-rpcssl=1, but solidcoin compiled without full openssl libraries.");

    ip::tcp::iostream stream(host, port);
    if (stream.fail())
        throw runtime_error("couldn't connect to server ("+host+":"+port+")");
#endif


    // HTTP basic authentication
    map<string, string> mapRequestHeaders;
    mapRequestHeaders["Authorization"] = string("Basic ") + base64UserString;

    // Send request
    string strRequest = JSONRPCRequest(strMethod, params, 1);
    string strPost = HTTPPost(strRequest, mapRequestHeaders);
    stream << strPost << std::flush;

    // Receive reply
    map<string, string> mapHeaders;
    string strReply;
    int nStatus = ReadHTTP(stream, mapHeaders, strReply);
    if (nStatus == 401)
        throw runtime_error("incorrect rpcuser or rpcpassword (authorization failed)");
    else if (nStatus >= 400 && nStatus != 400 && nStatus != 404 && nStatus != 500)
        throw runtime_error(strprintf("server returned HTTP error %d", nStatus));
    else if (strReply.empty())
        throw runtime_error("no response from server");

    // Parse reply
    Value valReply;
    if (!read_string(strReply, valReply))
        throw runtime_error("couldn't parse reply from server");
    const Object& reply = valReply.get_obj();
    if (reply.empty())
        throw runtime_error("expected reply to have result, error and id properties");

    return reply;
}


template<typename T>
void ConvertTo(Value& value)
{
    if (value.type() == str_type)
    {
        // reinterpret string as unquoted json value
        Value value2;
        if (!read_string(value.get_str(), value2))
            throw runtime_error("type mismatch");
        value = value2.get_value<T>();
    }
    else
    {
        value = value.get_value<T>();
    }
}

int CommandLineRPC(int argc, std::string *argv)
{
    string strPrint;
    int nRet = 0;
    try
    {
        // Skip switches
        while (argc > 1 && IsSwitchChar(argv[1][0]))
        {
            argc--;
            argv++;
        }

        // Method
        if (argc < 2)
            throw runtime_error("too few parameters");
        string strMethod = argv[1];

        // Parameters default to strings
        Array params;
        for (int i = 2; i < argc; i++)
            params.push_back(argv[i]);
        int n = params.size();

        //
        // Special case non-string parameter types
        //
        if (strMethod == "sc_setmining"            && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "sc_setmining"            && n > 1) ConvertTo<bool>(params[1]);
        if (strMethod == "sendtoaddress"          && n > 1) ConvertTo<double>(params[1]);
        if (strMethod == "getreceivedbyaddress"   && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "getreceivedbyaccount"   && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "listreceivedbyaddress"  && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "listreceivedbyaddress"  && n > 1) ConvertTo<bool>(params[1]);
        if (strMethod == "listreceivedbyaccount"  && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "listreceivedbyaccount"  && n > 1) ConvertTo<bool>(params[1]);
        if (strMethod == "getbalance"             && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "move"                   && n > 2) ConvertTo<double>(params[2]);
        if (strMethod == "sendfrom"               && n > 2) ConvertTo<double>(params[2]);
        if (strMethod == "sendfrom"               && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "listtransactions"       && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "listtransactions"       && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "listaccounts"           && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "walletpassphrase"       && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_gettxfee"            && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_getblockbynumber"     && n > 0) ConvertTo<boost::int64_t>(params[0]);
        if (strMethod == "sc_getreceivedbyaddress"&& n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_getreceivedbyaccount"   && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_getbalance"          && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_listaccounts"        && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_listtransactions"    && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_listtransactions"    && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_listreceivedbyaddress"  && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_listreceivedbyaddress"  && n > 2) ConvertTo<bool>(params[2]);
        if (strMethod == "sc_listreceivedbyaccount"  && n > 1) ConvertTo<boost::int64_t>(params[1]);
        if (strMethod == "sc_listreceivedbyaccount"  && n > 2) ConvertTo<bool>(params[2]);
        if (strMethod == "sc_sendtoaddress"          && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_sendtoaddress"          && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_move"                   && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_sendfrom"               && n > 3) ConvertTo<boost::int64_t>(params[3]);
        if (strMethod == "sc_sendfrom"               && n > 4) ConvertTo<boost::int64_t>(params[4]);
        if (strMethod == "sendmany" && n > 1)
        {
            string s = params[1].get_str();
            Value v;
            if (!read_string(s, v) || v.type() != obj_type) throw runtime_error("type mismatch");
            params[1] = v.get_obj();
        }
        if (strMethod == "sendmany"                 && n > 2) ConvertTo<boost::int64_t>(params[2]);
        if (strMethod == "sc_sendmany" && n > 2)
        {
            string s = params[2].get_str();
            Value v;
            if (!read_string(s, v) || v.type() != obj_type) throw runtime_error("type mismatch");
            params[2] = v.get_obj();
        }
        if (strMethod == "sc_sendmany"              && n > 3) ConvertTo<boost::int64_t>(params[3]);

        // Execute

        if (Setting_Get("rpcuser") == "" && Setting_Get("rpcpassword") == "")
            throw runtime_error(strprintf(
                _("You must set rpcpassword=<password> in the configuration file:\n%s\n"
                "If the file does not exist, create it with owner-readable-only file permissions."),
                    GetConfigFile().c_str()));

        string strUserPass64 = EncodeBase64(Setting_Get("rpcuser") + ":" + Setting_Get("rpcpassword"));
        string host=Setting_Get("rpcconnect");
        string port=Setting_Get("rpcport");
        Object reply = CallRPC(host,port,strUserPass64,strMethod, params);

        // Parse reply
        const Value& result = find_value(reply, "result");
        const Value& error  = find_value(reply, "error");

        if (error.type() != null_type)
        {
            // Error
            strPrint = "error: " + write_string(error, false);
            int code = find_value(error.get_obj(), "code").get_int();
            nRet = abs(code);
        }
        else
        {
            // Result
            if (result.type() == null_type)         strPrint = "";
            else if (result.type() == str_type)     strPrint = result.get_str();
            else                                    strPrint = write_string(result, true);
        }
    }
    catch (std::exception& e)
    {
        strPrint = string("error: ") + e.what();
        nRet = 87;
    }
    catch (...)
    {
        PrintException(NULL, "CommandLineRPC()");
    }

    if (strPrint != "")
    {
#if defined(__WXMSW__) && defined(GUI)
        // Windows GUI apps can't print to command line, so settle for a message box yuck
        MyMessageBox(strPrint, "SolidCoin", wxOK);
#else
        fprintf((nRet == 0 ? stdout : stderr), "%s\n", strPrint.c_str());
#endif
    }
    return nRet;
}




#ifdef TEST
int main(int argc, char *argv[])
{
#ifdef _MSC_VER
    // Turn off microsoft heap dump noise
    _CrtSetReportMode(_CRT_WARN, _CRTDBG_MODE_FILE);
    _CrtSetReportFile(_CRT_WARN, CreateFile("NUL", GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, 0));
#endif
    setbuf(stdin, NULL);
    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    try
    {
        if (argc >= 2 && string(argv[1]) == "-server")
        {
            debugprintf(INFO, "server ready\n");
            ThreadRPCServer(NULL);
        }
        else
        {
            return CommandLineRPC(argc, argv);
        }
    }
    catch (std::exception& e) {
        PrintException(&e, "main()");
    } catch (...) {
        PrintException(NULL, "main()");
    }
    return 0;
}
#endif


