//Please read license.txt for licensing and copyright information

#include "headers.h"
#include "irc.h"
#include "db.h"
#include "net.h"
#include "init.h"
#include "strlcpy.h"

#ifdef __WXMSW__
#include <string.h>
// This file can be downloaded as a part of the Windows Platform SDK
// and is required for SolidCoin binaries to work properly on versions
// of Windows before XP.  If you are doing builds of SolidCoin for
// public release, you should uncomment this line.
//#include <WSPiApi.h>
#endif

#ifdef USE_UPNP
#include <miniupnpc/miniwget.h>
#include <miniupnpc/miniupnpc.h>
#include <miniupnpc/upnpcommands.h>
#include <miniupnpc/upnperrors.h>
#endif

using namespace std;
using namespace boost;

static const int MAX_OUTBOUND_CONNECTIONS = 12;

void ThreadMessageHandler2(void* parg);
void ThreadSocketHandler2(void* parg);
void ThreadOpenConnections2(void* parg);
#ifdef USE_UPNP
void ThreadMapPort2(void* parg);
#endif
bool OpenNetworkConnection(const CAddress& addrConnect);





//
// Global state variables
//
bool fClient = false;
bool fAllowDNS = false;
uint64 nLocalServices = (fClient ? 0 : NODE_NETWORK);
CAddress addrLocalHost("0.0.0.0", 0, false, nLocalServices);
CNode* pnodeLocalHost = NULL;
uint64 nLocalHostNonce = 0;
array<int, 10> vnThreadsRunning;
boost::detail::atomic_count vaMultiThreads1(0);
SOCKET hListenSocket = INVALID_SOCKET;

vector<CNode*> vNodes;
CCriticalSection cs_vNodes;
map<vector<unsigned char>, CAddress> mapAddresses;
CCriticalSection cs_mapAddresses;
map<CInv, CDataStream> mapRelay;
deque<pair<int64, CInv> > vRelayExpiration;
CCriticalSection cs_mapRelay;
map<CInv, int64> mapAlreadyAskedFor;

// Settings
bool g_bUseProxy = false;
int nConnectTimeout = 5000;
CAddress addrProxy("127.0.0.1",9050);
int64 g_nBestBlockCountExternal=0;




unsigned short GetListenPort()
{
    return (unsigned short)GetDefaultPort();
}

void CNode::PushGetBlocks(CBlockIndex* pindexBegin, uint256 hashEnd)
{
    // Filter out duplicate requests
    if (pindexBegin == pindexLastGetBlocksBegin && hashEnd == hashLastGetBlocksEnd)
        return;
    pindexLastGetBlocksBegin = pindexBegin;
    hashLastGetBlocksEnd = hashEnd;

    PushMessage("getblocks", CBlockLocator(pindexBegin), hashEnd);
}





bool ConnectSocket(const CAddress& addrConnect, SOCKET& hSocketRet, int nTimeout)
{
    hSocketRet = INVALID_SOCKET;

    SOCKET hSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (hSocket == INVALID_SOCKET)
        return false;
#ifdef SO_NOSIGPIPE
    int set = 1;
    setsockopt(hSocket, SOL_SOCKET, SO_NOSIGPIPE, (void*)&set, sizeof(int));
#endif

    bool fProxy = (g_bUseProxy && addrConnect.IsRoutable());
    struct sockaddr_in sockaddr = (fProxy ? addrProxy.GetSockAddr() : addrConnect.GetSockAddr());

#ifdef __WXMSW__
    u_long fNonblock = 1;
    if (ioctlsocket(hSocket, FIONBIO, &fNonblock) == SOCKET_ERROR)
#else
    int fFlags = fcntl(hSocket, F_GETFL, 0);
    if (fcntl(hSocket, F_SETFL, fFlags | O_NONBLOCK) == -1)
#endif
    {
        closesocket(hSocket);
        return false;
    }


    if (connect(hSocket, (struct sockaddr*)&sockaddr, sizeof(sockaddr)) == SOCKET_ERROR)
    {
        // WSAEINVAL is here because some legacy version of winsock uses it
        if (WSAGetLastError() == WSAEINPROGRESS || WSAGetLastError() == WSAEWOULDBLOCK || WSAGetLastError() == WSAEINVAL)
        {
            struct timeval timeout;
            timeout.tv_sec  = nTimeout / 1000;
            timeout.tv_usec = (nTimeout % 1000) * 1000;

            fd_set fdset;
            FD_ZERO(&fdset);
            FD_SET(hSocket, &fdset);
            int nRet = select(hSocket + 1, NULL, &fdset, NULL, &timeout);
            if (nRet == 0)
            {
                debugprintf(INFO, "connection timeout\n");
                closesocket(hSocket);
                return false;
            }
            if (nRet == SOCKET_ERROR)
            {
                debugprintf(INFO, "select() for connection failed: %i\n",WSAGetLastError());
                closesocket(hSocket);
                return false;
            }
            socklen_t nRetSize = sizeof(nRet);
#ifdef __WXMSW__
            if (getsockopt(hSocket, SOL_SOCKET, SO_ERROR, (char*)(&nRet), &nRetSize) == SOCKET_ERROR)
#else
            if (getsockopt(hSocket, SOL_SOCKET, SO_ERROR, &nRet, &nRetSize) == SOCKET_ERROR)
#endif
            {
                debugprintf(INFO, "getsockopt() for connection failed: %i\n",WSAGetLastError());
                closesocket(hSocket);
                return false;
            }
            if (nRet != 0)
            {
                debugprintf(INFO, "connect() failed after select(): %s\n",strerror(nRet));
                closesocket(hSocket);
                return false;
            }
        }
#ifdef __WXMSW__
        else if (WSAGetLastError() != WSAEISCONN)
#else
        else
#endif
        {
            debugprintf(INFO, "connect() failed: %i\n",WSAGetLastError());
            closesocket(hSocket);
            return false;
        }
    }

    /*
    this isn't even strictly necessary
    CNode::ConnectNode immediately turns the socket back to non-blocking
    but we'll turn it back to blocking just in case
    */
#ifdef __WXMSW__
    fNonblock = 0;
    if (ioctlsocket(hSocket, FIONBIO, &fNonblock) == SOCKET_ERROR)
#else
    fFlags = fcntl(hSocket, F_GETFL, 0);
    if (fcntl(hSocket, F_SETFL, fFlags & !O_NONBLOCK) == SOCKET_ERROR)
#endif
    {
        closesocket(hSocket);
        return false;
    }

    if (fProxy)
    {
        debugprintf(ALWAYS, "proxy connecting %s\n", addrConnect.ToString().c_str());
        char pszSocks4IP[] = "\4\1\0\0\0\0\0\0user";
        memcpy(pszSocks4IP + 2, &addrConnect.port, 2);
        memcpy(pszSocks4IP + 4, &addrConnect.ip, 4);
        char* pszSocks4 = pszSocks4IP;
        int nSize = sizeof(pszSocks4IP);

        int ret = send(hSocket, pszSocks4, nSize, MSG_NOSIGNAL);
        if (ret != nSize)
        {
            closesocket(hSocket);
            return error("Error sending to proxy");
        }
        char pchRet[8];
        if (recv(hSocket, pchRet, 8, 0) != 8)
        {
            closesocket(hSocket);
            return error("Error reading proxy response");
        }
        if (pchRet[1] != 0x5a)
        {
            closesocket(hSocket);
            if (pchRet[1] != 0x5b)
                debugprintf(ERR, "ERROR: Proxy returned error %d\n", pchRet[1]);
            return false;
        }
        debugprintf(INFO, "proxy connected %s\n", addrConnect.ToString().c_str());
    }

    hSocketRet = hSocket;
    return true;
}

// portDefault is in host order
bool Lookup(const char *pszName, vector<CAddress>& vaddr, int nServices, int nMaxSolutions, bool fAllowLookup, int portDefault, bool fAllowPort)
{
    vaddr.clear();
    if (pszName[0] == 0)
        return false;
    int port = portDefault;
    char psz[256];
    char *pszHost = psz;
    strlcpy(psz, pszName, sizeof(psz));
    if (fAllowPort)
    {
        char* pszColon = strrchr(psz+1,':');
        char *pszPortEnd = NULL;
        int portParsed = pszColon ? strtoul(pszColon+1, &pszPortEnd, 10) : 0;
        if (pszColon && pszPortEnd && pszPortEnd[0] == 0)
        {
            if (psz[0] == '[' && pszColon[-1] == ']')
            {
                // Future: enable IPv6 colon-notation inside []
                pszHost = psz+1;
                pszColon[-1] = 0;
            }
            else
                pszColon[0] = 0;
            port = portParsed;
            if (port < 0 || port > USHRT_MAX)
                port = USHRT_MAX;
        }
    }

    unsigned int addrIP = inet_addr(pszHost);
    if (addrIP != INADDR_NONE)
    {
        // valid IP address passed
        vaddr.push_back(CAddress(addrIP, port, nServices));
        return true;
    }

    if (!fAllowLookup)
        return false;

    struct hostent* phostent = gethostbyname(pszHost);
    if (!phostent)
        return false;

    if (phostent->h_addrtype != AF_INET)
        return false;

    char** ppAddr = phostent->h_addr_list;
    while (*ppAddr != NULL && vaddr.size() != nMaxSolutions)
    {
        CAddress addr(((struct in_addr*)ppAddr[0])->s_addr, port, nServices);
        if (addr.IsValid())
            vaddr.push_back(addr);
        ppAddr++;
    }

    return (vaddr.size() > 0);
}

// portDefault is in host order
bool Lookup(const char *pszName, CAddress& addr, int nServices, bool fAllowLookup, int portDefault, bool fAllowPort)
{
    vector<CAddress> vaddr;
    bool fRet = Lookup(pszName, vaddr, nServices, 1, fAllowLookup, portDefault, fAllowPort);
    if (fRet)
        addr = vaddr[0];
    return fRet;
}

bool DownloadFile(const std::string &geturl, std::string &strout)
{
    std::string host,dir;
    CAddress addrConnect;
	int npos=geturl.find("http://");
	if(npos!=0) return false;
	host=geturl.substr(7,geturl.length());
	npos=host.find("/");
	if(npos==(int)std::string::npos) return false;
	dir=host.substr(npos,host.length());
	host.erase(npos,host.length());

	CAddress addrIP(host, 80, true);
	addrConnect = addrIP;

    string getrequest = "GET "+dir+" HTTP/1.1\r\n"
                     "Host: "+host+"\r\n"
                     "User-Agent: Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1)\r\n"
                     "Connection: close\r\n"
                     "\r\n";

    SOCKET hSocket;
    if (!ConnectSocket(addrConnect, hSocket))   return error("DownloadFile() : connection to %s failed", addrConnect.ToString().c_str());

    send(hSocket, getrequest.c_str(), getrequest.length(), MSG_NOSIGNAL);

    string strLine;
    while(RecvLine(hSocket, strLine))
    {
        if (strLine.empty()==false) continue;   // HTTP response is separated from headers by blank line
        if (!RecvLine(hSocket, strLine))    break;
        strout+=strLine;
    }
    closesocket(hSocket);
    if(strout.length()) return true;
    return error("DownloadFile() : Error downloading");
}

bool GetMyExternalIP2(const CAddress& addrConnect, const char* pszGet, const char* pszKeyword, unsigned int& ipRet)
{
    SOCKET hSocket;
    if (!ConnectSocket(addrConnect, hSocket))
        return error("GetMyExternalIP() : connection to %s failed", addrConnect.ToString().c_str());

    send(hSocket, pszGet, strlen(pszGet), MSG_NOSIGNAL);

    string strLine;
    while (RecvLine(hSocket, strLine))
    {
        if (strLine.empty()) // HTTP response is separated from headers by blank line
        {
            loop
            {
                if (!RecvLine(hSocket, strLine))
                {
                    closesocket(hSocket);
                    return false;
                }
                if (pszKeyword == NULL)
                    break;
                if (strLine.find(pszKeyword) != -1)
                {
                    strLine = strLine.substr(strLine.find(pszKeyword) + strlen(pszKeyword));
                    break;
                }
            }
            closesocket(hSocket);
            if (strLine.find("<") != -1)
                strLine = strLine.substr(0, strLine.find("<"));
            strLine = strLine.substr(strspn(strLine.c_str(), " \t\n\r"));
            while (strLine.size() > 0 && isspace(strLine[strLine.size()-1]))
                strLine.resize(strLine.size()-1);
            CAddress addr(strLine,0,true);
            debugprintf(ALWAYS, "GetMyExternalIP() received [%s] %s\n", strLine.c_str(), addr.ToString().c_str());
            if (addr.ip == 0 || addr.ip == INADDR_NONE || !addr.IsRoutable())
                return false;
            ipRet = addr.ip;
            return true;
        }
    }
    closesocket(hSocket);
    return error("GetMyExternalIP() : connection closed");
}

// We now get our external IP from the IRC server first and only use this as a backup
bool GetMyExternalIP(unsigned int& ipRet)
{
    CAddress addrConnect;
    const char* pszGet=0;
    const char* pszKeyword=0;

    if (g_bUseProxy)
        return false;

    for (int nLookup = 0; nLookup <= 1; nLookup++)
    for (int nHost = 1; nHost <= 2; nHost++)
    {
        // We should be phasing out our use of sites like these.  If we need
        // replacements, we should ask for volunteers to put this simple
        // php file on their webserver that prints the client IP:
        //  <?php echo $_SERVER["REMOTE_ADDR"]; ?>
        if (nHost == 1)
        {
            addrConnect = CAddress("91.198.22.70",80); // checkip.dyndns.org

            if (nLookup == 1)
            {
                CAddress addrIP("checkip.dyndns.org", 80, true);
                if (addrIP.IsValid())
                    addrConnect = addrIP;
            }

            pszGet = "GET / HTTP/1.1\r\n"
                     "Host: checkip.dyndns.org\r\n"
                     "User-Agent: Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1)\r\n"
                     "Connection: close\r\n"
                     "\r\n";

            pszKeyword = "Address:";
        }
        else if (nHost == 2)
        {
            addrConnect = CAddress("74.208.43.192", 80); // www.showmyip.com

            if (nLookup == 1)
            {
                CAddress addrIP("www.showmyip.com", 80, true);
                if (addrIP.IsValid())
                    addrConnect = addrIP;
            }

            pszGet = "GET /simple/ HTTP/1.1\r\n"
                     "Host: www.showmyip.com\r\n"
                     "User-Agent: Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1)\r\n"
                     "Connection: close\r\n"
                     "\r\n";

            pszKeyword = NULL; // Returns just IP address
        }

        if (GetMyExternalIP2(addrConnect, pszGet, pszKeyword, ipRet))
            return true;
    }

    return false;
}

void ThreadGetMyExternalIP(void* parg)
{
    // Wait for IRC to get it first
    if (!Setting_GetBOOL("noirc"))
    {
        for (int i = 0; i < 2 * 60; i++)
        {
            Sleep(1000);
            if (fGotExternalIP || fShutdown)
                return;
        }
    }

    // Fallback in case IRC fails to get it
    if (GetMyExternalIP(addrLocalHost.ip))
    {
        debugprintf(ALWAYS, "GetMyExternalIP() returned %s\n", addrLocalHost.ToStringIP().c_str());
        if (addrLocalHost.IsRoutable())
        {
            // If we already connected to a few before we had our IP, go back and addr them.
            // setAddrKnown automatically filters any duplicate sends.
            CAddress addr(addrLocalHost);
            addr.nTime = SolidTime_Get();
            MUTEX_LOCK(cs_vNodes);
            BOOST_FOREACH(CNode* pnode, vNodes)
            {
                pnode->PushAddress(addr);
            }
        }
    }
}

void ThreadGetLatestBlockCount(void* parg)
{
    if(fTestNet) return;

    int nNumUrl=1,nCount=0;
    const char*urlList[]= { "http://node.solidcointalk.org/blockcount.txt"};
    std::string data;
    //while(!DownloadFile(urlList[rand()%nNumUrl],data) && nCount<2)
    while(!DownloadFile(urlList[rand()%nNumUrl],data) && nCount<2)
    {
        Sleep(10000);   //sleep 10 seconds and try again
        nCount++;
    }
    if(nCount==2) return;
    nCount = strtol(data.c_str(),0,10);
    if(nCount<=0 || nCount>(Block_GetTotalNumEstimate()*10) ) return;  //bounds checking
    g_nBestBlockCountExternal=nCount;
    MainFrameRepaint();
}

bool AddAddress(CAddress addr, int64 nTimePenalty, CAddrDB *pAddrDB)
{
    if (!addr.IsRoutable())
        return false;
    if (addr.ip == addrLocalHost.ip)
        return false;
    addr.nTime = max((int64)0, (int64)addr.nTime - nTimePenalty);

    MUTEX_LOCK(cs_mapAddresses);

    map<vector<unsigned char>, CAddress>::iterator it = mapAddresses.find(addr.GetKey());
    if (it == mapAddresses.end())
    {
        // New address
        debugprintf(INFO, "AddAddress(%s)\n", addr.ToString().c_str());
        mapAddresses.insert(make_pair(addr.GetKey(), addr));
        if (pAddrDB)    pAddrDB->WriteAddress(addr);
        else            CAddrDB().WriteAddress(addr);
        return true;
    }
    else
    {
        bool fUpdated = false;
        CAddress& addrFound = (*it).second;
        if ((addrFound.nServices | addr.nServices) != addrFound.nServices)
        {
            // Services have been added
            addrFound.nServices |= addr.nServices;
            fUpdated = true;
        }
        bool fCurrentlyOnline = (SolidTime_Get() - addr.nTime < 24 * 60 * 60);
        int64 nUpdateInterval = (fCurrentlyOnline ? 60 * 60 : 24 * 60 * 60);
        if (addrFound.nTime < addr.nTime - nUpdateInterval)
        {
            // Periodically update most recently seen time
            addrFound.nTime = addr.nTime;
            fUpdated = true;
        }
        if (fUpdated)
        {
            if (pAddrDB)    pAddrDB->WriteAddress(addrFound);
            else            CAddrDB().WriteAddress(addrFound);
        }
    }
    return false;
}

void AddressCurrentlyConnected(const CAddress& addr)
{
    MUTEX_LOCK(cs_mapAddresses);

    // Only if it's been published already
    map<vector<unsigned char>, CAddress>::iterator it = mapAddresses.find(addr.GetKey());
    if (it != mapAddresses.end())
    {
        CAddress& addrFound = (*it).second;
        int64 nUpdateInterval = 20 * 60;
        if (addrFound.nTime < SolidTime_Get() - nUpdateInterval)
        {
            // Periodically update most recently seen time
            addrFound.nTime = SolidTime_Get();
            CAddrDB addrdb;
            addrdb.WriteAddress(addrFound);
        }
    }
}





void AbandonRequests(void (*fn)(void*, CDataStream&), void* param1)
{
    // If the dialog might get closed before the reply comes back,
    // call this in the destructor so it doesn't get called after it's deleted.
    MUTEX_LOCK(cs_vNodes);

    BOOST_FOREACH(CNode* pnode, vNodes)
    {
        MUTEX_LOCK(pnode->cs_mapRequests);
        for (map<uint256, CRequestTracker>::iterator mi = pnode->mapRequests.begin(); mi != pnode->mapRequests.end();)
        {
            CRequestTracker& tracker = (*mi).second;
            if (tracker.fn == fn && tracker.param1 == param1)   pnode->mapRequests.erase(mi++);
            else                                                mi++;
        }
    }
}







//
// Subscription methods for the broadcast and subscription system.
// Channel numbers are message numbers, i.e. MSG_TABLE and MSG_PRODUCT.
//
// The subscription system uses a meet-in-the-middle strategy.
// With 100,000 nodes, if senders broadcast to 1000 random nodes and receivers
// subscribe to 1000 random nodes, 99.995% (1 - 0.99^1000) of messages will get through.
//

bool AnySubscribed(unsigned int nChannel)
{
    if (pnodeLocalHost->IsSubscribed(nChannel)) return true;
    MUTEX_LOCK(cs_vNodes);
    BOOST_FOREACH(CNode* pnode, vNodes)
    {
        if (pnode->IsSubscribed(nChannel))return true;
    }
    return false;
}

bool CNode::IsSubscribed(unsigned int nChannel)
{
    if (nChannel >= vfSubscribe.size())
        return false;
    return vfSubscribe[nChannel];
}

void CNode::Subscribe(unsigned int nChannel, unsigned int nHops)
{
    if (nChannel >= vfSubscribe.size())
        return;

    if (!AnySubscribed(nChannel))
    {
        // Relay subscribe
        MUTEX_LOCK(cs_vNodes);
        BOOST_FOREACH(CNode* pnode, vNodes)
        {
            if (pnode != this)  pnode->PushMessage("subscribe", nChannel, nHops);
        }
    }

    vfSubscribe[nChannel] = true;
}

void CNode::CancelSubscribe(unsigned int nChannel)
{
    if (nChannel >= vfSubscribe.size()) return;

    // Prevent from relaying cancel if wasn't subscribed
    if (!vfSubscribe[nChannel]) return;
    vfSubscribe[nChannel] = false;

    if (!AnySubscribed(nChannel))
    {
        // Relay subscription cancel
        MUTEX_LOCK(cs_vNodes);
        BOOST_FOREACH(CNode* pnode, vNodes)
        {
            if (pnode != this)  pnode->PushMessage("sub-cancel", nChannel);
        }
    }
}









CNode* FindNode(unsigned int ip)
{
    MUTEX_LOCK(cs_vNodes);
    BOOST_FOREACH(CNode* pnode, vNodes)
    {
        if (pnode->addr.ip == ip)   return (pnode);
    }
    return NULL;
}

CNode* FindNode(CAddress addr)
{
    MUTEX_LOCK(cs_vNodes);
    BOOST_FOREACH(CNode* pnode, vNodes)
    {
        if (pnode->addr == addr)    return (pnode);
    }
    return NULL;
}

CNode* ConnectNode(CAddress addrConnect, int64 nTimeout)
{
    if (addrConnect.ip == addrLocalHost.ip)
        return NULL;

    // Look for an existing connection
    CNode* pnode = FindNode(addrConnect.ip);
    if (pnode)
    {
        if (nTimeout != 0)  pnode->AddRef(nTimeout);
        else                pnode->AddRef();
        return pnode;
    }

    /// debug print
    debugprintf(INFO, "trying connection %s lastseen=%.1fhrs lasttry=%.1fhrs\n",addrConnect.ToString().c_str(),(double)(addrConnect.nTime - SolidTime_Get())/3600.0, (double)(addrConnect.nLastTry - SolidTime_Get())/3600.0);
    {
        MUTEX_LOCK(cs_mapAddresses);
        mapAddresses[addrConnect.GetKey()].nLastTry = SolidTime_Get();
    }

    // Connect
    SOCKET hSocket;
    if (ConnectSocket(addrConnect, hSocket))
    {
        /// debug print
        debugprintf(ALWAYS, "connected %s\n", addrConnect.ToString().c_str());

        // Set to nonblocking
#ifdef __WXMSW__
        u_long nOne = 1;
        if (ioctlsocket(hSocket, FIONBIO, &nOne) == SOCKET_ERROR)
            debugprintf(ERR, "ConnectSocket() : ioctlsocket nonblocking setting failed, error %d\n", WSAGetLastError());
#else
        if (fcntl(hSocket, F_SETFL, O_NONBLOCK) == SOCKET_ERROR)
            debugprintf(ERR, "ConnectSocket() : fcntl nonblocking setting failed, error %d\n", errno);
#endif

        // Add node
        CNode* pnode = new CNode(hSocket, addrConnect, false);
        if (nTimeout != 0)  pnode->AddRef(nTimeout);
        else                pnode->AddRef();

        {
            MUTEX_LOCK(cs_vNodes);
            vNodes.push_back(pnode);
        }
        pnode->nTimeConnected = SolidTime_Get();
        return pnode;
    }
    else
    {
        return NULL;
    }
}

void CNode::CloseSocketDisconnect()
{
    fDisconnect = true;
    if (hSocket != INVALID_SOCKET)
    {
        if (fDebug)
            debugprintf(INFO, "%s ", DateTimeStrFormat("%x %H:%M:%S", SolidTime_Get()).c_str());
        debugprintf(ALWAYS, "disconnecting node %s\n", addr.ToString().c_str());
        closesocket(hSocket);
        hSocket = INVALID_SOCKET;
    }
}

void CNode::Cleanup()
{
    // All of a nodes broadcasts and subscriptions are automatically torn down
    // when it goes down, so a node has to stay up to keep its broadcast going.

    // Cancel subscriptions
    for (unsigned int nChannel = 0; nChannel < vfSubscribe.size(); nChannel++)
        if (vfSubscribe[nChannel])
            CancelSubscribe(nChannel);
}













void ThreadSocketHandler(void* parg)
{
    IMPLEMENT_RANDOMIZE_STACK(ThreadSocketHandler(parg));
    try
    {
        vnThreadsRunning[0]++;
        ThreadSocketHandler2(parg);
        vnThreadsRunning[0]--;
    }
    catch (std::exception& e) {
        vnThreadsRunning[0]--;
        PrintException(&e, "ThreadSocketHandler()");
    } catch (...) {
        vnThreadsRunning[0]--;
        throw; // support pthread_cancel()
    }
    debugprintf(INFO, "ThreadSocketHandler exiting\n");
}

void ThreadSocketHandler2(void* parg)
{
    debugprintf(INFO, "ThreadSocketHandler started\n");
    list<CNode*> vNodesDisconnected;
    int nPrevNodeCount = 0;

    loop
    {
        //
        // Disconnect nodes
        //

        {
            MUTEX_LOCK(cs_vNodes);
            // Disconnect unused nodes
            vector<CNode*> vNodesCopy = vNodes;
            BOOST_FOREACH(CNode* pnode, vNodesCopy)
            {
                if (pnode->fDisconnect ||
                    (pnode->GetRefCount() <= 0 && pnode->vRecv.empty() && pnode->vSend.empty()))
                {
                    // remove from vNodes
                    vNodes.erase(remove(vNodes.begin(), vNodes.end(), pnode), vNodes.end());

                    // close socket and cleanup
                    pnode->CloseSocketDisconnect();
                    pnode->Cleanup();

                    // hold in disconnected pool until all refs are released
                    pnode->nReleaseTime = max(pnode->nReleaseTime, SolidTime_Get() + 15 * 60);
                    if (pnode->fNetworkNode || pnode->fInbound)
                        pnode->Release();
                    vNodesDisconnected.push_back(pnode);
                }
            }

            // Delete disconnected nodes
            list<CNode*> vNodesDisconnectedCopy = vNodesDisconnected;
            BOOST_FOREACH(CNode* pnode, vNodesDisconnectedCopy)
            {
                // wait until threads are done using it
                if (pnode->GetRefCount() <= 0)
                {
                    bool fDelete = false;
                    TRY_CRITICAL_BLOCK(pnode->cs_vSend)
                     TRY_CRITICAL_BLOCK(pnode->cs_vRecv)
                      TRY_CRITICAL_BLOCK(pnode->cs_mapRequests)
                       TRY_CRITICAL_BLOCK(pnode->cs_inventory)
                        fDelete = true;
                    if (fDelete)
                    {
                        vNodesDisconnected.remove(pnode);
                        delete pnode;
                    }
                }
            }
        }
        if (vNodes.size() != nPrevNodeCount)
        {
            nPrevNodeCount = vNodes.size();
            MainFrameRepaint();
        }


        //
        // Find which sockets have data to receive
        //
        struct timeval timeout;
        timeout.tv_sec  = 0;
        timeout.tv_usec = 50000; // frequency to poll pnode->vSend

        fd_set fdsetRecv;
        fd_set fdsetSend;
        fd_set fdsetError;
        FD_ZERO(&fdsetRecv);
        FD_ZERO(&fdsetSend);
        FD_ZERO(&fdsetError);
        SOCKET hSocketMax = 0;

        if(hListenSocket != INVALID_SOCKET)
            FD_SET(hListenSocket, &fdsetRecv);
        hSocketMax = max(hSocketMax, hListenSocket);

        {
            MUTEX_LOCK(cs_vNodes);
            BOOST_FOREACH(CNode* pnode, vNodes)
            {
                if (pnode->hSocket == INVALID_SOCKET)
                    continue;
                FD_SET(pnode->hSocket, &fdsetRecv);
                FD_SET(pnode->hSocket, &fdsetError);
                hSocketMax = max(hSocketMax, pnode->hSocket);
                TRY_CRITICAL_BLOCK(pnode->cs_vSend)
                    if (!pnode->vSend.empty())
                        FD_SET(pnode->hSocket, &fdsetSend);
            }
        }

        vnThreadsRunning[0]--;
        int nSelect = select(hSocketMax + 1, &fdsetRecv, &fdsetSend, &fdsetError, &timeout);
        vnThreadsRunning[0]++;
        if (fShutdown)
            return;
        if (nSelect == SOCKET_ERROR)
        {
            int nErr = WSAGetLastError();
            if (hSocketMax > -1)
            {
                debugprintf(ERR, "socket select error %d\n", nErr);
                for (int i = 0; i <= hSocketMax; i++)
                    FD_SET(i, &fdsetRecv);
            }
            FD_ZERO(&fdsetSend);
            FD_ZERO(&fdsetError);
            Sleep(timeout.tv_usec/1000);
        }


        //
        // Accept new connections
        //
        if (hListenSocket != INVALID_SOCKET && FD_ISSET(hListenSocket, &fdsetRecv))
        {
            struct sockaddr_in sockaddr;
            socklen_t len = sizeof(sockaddr);
            SOCKET hSocket = accept(hListenSocket, (struct sockaddr*)&sockaddr, &len);
            CAddress addr(sockaddr);
            int nInbound = 0;
            {
                MUTEX_LOCK(cs_vNodes);
                BOOST_FOREACH(CNode* pnode, vNodes)
                {
                    if (pnode->fInbound)   nInbound++;
                }
            }

            if (hSocket == INVALID_SOCKET)
            {
                if (WSAGetLastError() != WSAEWOULDBLOCK)    debugprintf(ERR, "socket error accept failed: %d\n", WSAGetLastError());
            }
            else if (nInbound >= Setting_GetINT64("maxconnections") - MAX_OUTBOUND_CONNECTIONS)
            {
                closesocket(hSocket);
            }
            else
            {
                debugprintf(ALWAYS, "accepted connection %s\n", addr.ToString().c_str());
                CNode* pnode = new CNode(hSocket, addr, true);
                pnode->AddRef();

                {
                    MUTEX_LOCK(cs_vNodes);
                    vNodes.push_back(pnode);
                }
            }
        }


        //
        // Service each socket
        //
        vector<CNode*> vNodesCopy;

        {
            MUTEX_LOCK(cs_vNodes);
            vNodesCopy = vNodes;
            BOOST_FOREACH(CNode* pnode, vNodesCopy)
            {
                pnode->AddRef();
            }
        }
        BOOST_FOREACH(CNode* pnode, vNodesCopy)
        {
            if (fShutdown)  return;

            // Receive
            if (pnode->hSocket == INVALID_SOCKET)   continue;
            if (FD_ISSET(pnode->hSocket, &fdsetRecv) || FD_ISSET(pnode->hSocket, &fdsetError))
            {
                TRY_CRITICAL_BLOCK(pnode->cs_vRecv)
                {
                    CDataStream& vRecv = pnode->vRecv;
                    unsigned int nPos = vRecv.size();

                    if (nPos > ReceiveBufferSize()) {
                        if (!pnode->fDisconnect)
                            debugprintf(WARN, "socket recv flood control disconnect (%d bytes)\n", vRecv.size());
                        pnode->CloseSocketDisconnect();
                    }
                    else {
                        // typical socket buffer is 8K-64K
                        char pchBuf[0x10000];
                        int nBytes = recv(pnode->hSocket, pchBuf, sizeof(pchBuf), MSG_DONTWAIT);
                        if (nBytes > 0)
                        {
                            vRecv.resize(nPos + nBytes);
                            memcpy(&vRecv[nPos], pchBuf, nBytes);
                            pnode->nLastRecv = SolidTime_Get();
                            pnode->m_bNeedProcess = true;
                        }
                        else if (nBytes == 0)
                        {
                            // socket closed gracefully
                            if (!pnode->fDisconnect)
                                debugprintf(INFO, "socket closed\n");
                            pnode->CloseSocketDisconnect();
                        }
                        else if (nBytes < 0)
                        {
                            // error
                            int nErr = WSAGetLastError();
                            if (nErr != WSAEWOULDBLOCK && nErr != WSAEMSGSIZE && nErr != WSAEINTR && nErr != WSAEINPROGRESS)
                            {
                                if (!pnode->fDisconnect)
                                    debugprintf(ERR, "socket recv error %d\n", nErr);
                                pnode->CloseSocketDisconnect();
                            }
                        }
                    }
                }
            }

            //
            // Send
            //
            if (pnode->hSocket == INVALID_SOCKET)
                continue;
            if (FD_ISSET(pnode->hSocket, &fdsetSend))
            {
                TRY_CRITICAL_BLOCK(pnode->cs_vSend)
                {
                    CDataStream& vSend = pnode->vSend;
                    if (!vSend.empty())
                    {
                        int nBytes = send(pnode->hSocket, &vSend[0], vSend.size(), MSG_NOSIGNAL | MSG_DONTWAIT);
                        if (nBytes > 0)
                        {
                            vSend.erase(vSend.begin(), vSend.begin() + nBytes);
                            pnode->nLastSend = SolidTime_Get();
                        }
                        else if (nBytes < 0)
                        {
                            // error
                            int nErr = WSAGetLastError();
                            if (nErr != WSAEWOULDBLOCK && nErr != WSAEMSGSIZE && nErr != WSAEINTR && nErr != WSAEINPROGRESS)
                            {
                                debugprintf(ERR, "socket send error %d\n", nErr);
                                pnode->CloseSocketDisconnect();
                            }
                        }
                        if (vSend.size() > SendBufferSize()) {
                            if (!pnode->fDisconnect)
                                debugprintf(WARN, "socket send flood control disconnect (%d bytes)\n", vSend.size());
                            pnode->CloseSocketDisconnect();
                        }
                    }
                }
            }

            //
            // Inactivity checking
            //
            if (pnode->vSend.empty())
                pnode->nLastSendEmpty = SolidTime_Get();
            if (SolidTime_Get() - pnode->nTimeConnected > 60)
            {
                if (pnode->nLastRecv == 0 || pnode->nLastSend == 0)
                {
                    debugprintf(WARN, "socket no message in first 60 seconds, %d %d\n", pnode->nLastRecv != 0, pnode->nLastSend != 0);
                    pnode->fDisconnect = true;
                }
                else if (SolidTime_Get() - pnode->nLastSend > 90*60 && SolidTime_Get() - pnode->nLastSendEmpty > 90*60)
                {
                    debugprintf(WARN, "socket not sending\n");
                    pnode->fDisconnect = true;
                }
                else if (SolidTime_Get() - pnode->nLastRecv > 90*60)
                {
                    debugprintf(WARN, "socket inactivity timeout\n");
                    pnode->fDisconnect = true;
                }
            }
        }

        {
            MUTEX_LOCK(cs_vNodes);
            BOOST_FOREACH(CNode* pnode, vNodesCopy)
            {
                pnode->Release();
            }
        }

        Sleep(10);
    }
}









#ifdef USE_UPNP
void ThreadMapPort(void* parg)
{
    IMPLEMENT_RANDOMIZE_STACK(ThreadMapPort(parg));
    try
    {
        vnThreadsRunning[5]++;
        ThreadMapPort2(parg);
        vnThreadsRunning[5]--;
    }
    catch (std::exception& e) {
        vnThreadsRunning[5]--;
        PrintException(&e, "ThreadMapPort()");
    } catch (...) {
        vnThreadsRunning[5]--;
        PrintException(NULL, "ThreadMapPort()");
    }
    debugprintf(ALWAYS, "ThreadMapPort exiting\n");
}

void ThreadMapPort2(void* parg)
{
    debugprintf(ALWAYS, "ThreadMapPort started\n");

    char port[6];
    sprintf(port, "%d", GetListenPort());

    const char * rootdescurl = 0;
    const char * multicastif = 0;
    const char * minissdpdpath = 0;
    struct UPNPDev * devlist = 0;
    char lanaddr[64];

    devlist = upnpDiscover(2000, multicastif, minissdpdpath, 0);

    struct UPNPUrls urls;
    struct IGDdatas data;
    int r;

    r = UPNP_GetValidIGD(devlist, &urls, &data, lanaddr, sizeof(lanaddr));
    if (r == 1)
    {
        char intClient[16];
        char intPort[6];
        string strDesc = "SolidCoin " + FormatFullVersion();

#ifndef __WXMSW__
        r = UPNP_AddPortMapping(urls.controlURL, data.first.servicetype,
	                        port, port, lanaddr, strDesc.c_str(), "TCP", 0);
#else
        r = UPNP_AddPortMapping(urls.controlURL, data.first.servicetype,
	                        port, port, lanaddr, strDesc.c_str(), "TCP", 0, "0");
#endif
        if(r!=UPNPCOMMAND_SUCCESS)
            debugprintf(ERR, "AddPortMapping(%s, %s, %s) failed with code %d (%s)\n",
                port, port, lanaddr, r, strupnperror(r));
        else
            debugprintf(ALWAYS, "UPnP Port Mapping successful.\n");
        loop {
            if (fShutdown || !fUseUPnP)
            {
                r = UPNP_DeletePortMapping(urls.controlURL, data.first.servicetype, port, "TCP", 0);
                debugprintf(INFO, "UPNP_DeletePortMapping() returned : %d\n", r);
                freeUPNPDevlist(devlist); devlist = 0;
                FreeUPNPUrls(&urls);
                return;
            }
            Sleep(2000);
        }
    } else {
        debugprintf(WARN, "No valid UPnP IGDs found\n");
        freeUPNPDevlist(devlist); devlist = 0;
        if (r != 0)
            FreeUPNPUrls(&urls);
        loop {
            if (fShutdown || !fUseUPnP)
                return;
            Sleep(2000);
        }
    }
}

void MapPort(bool fMapPort)
{
    if (fUseUPnP != fMapPort)
    {
        fUseUPnP = fMapPort;
        //WriteSetting("fUseUPnP", fUseUPnP);
    }
    if (fUseUPnP && vnThreadsRunning[5] < 1)
    {
        if (!CreateThread(ThreadMapPort, NULL))
            debugprintf(ERR, "Error: ThreadMapPort(ThreadMapPort) failed\n");
    }
}
#else
void MapPort(bool /* unused fMapPort */)
{
    // Intentionally left blank.
}
#endif










static const char *strDNSSeed[] = {
    "bitseed.xf2.org",
    "bitseed.bitcoin.org.uk",
    "dnsseed.bluematt.me",
};

void DNSAddressSeed()
{
    int found = 0;

    if (!fTestNet)
    {
        debugprintf(ALWAYS, "Loading addresses from DNS seeds (could take a while)\n");
        CAddrDB addrDB;
        addrDB.TxnBegin();

        for (int seed_idx = 0; seed_idx < ARRAYLEN(strDNSSeed); seed_idx++) {
            vector<CAddress> vaddr;
            if (Lookup(strDNSSeed[seed_idx], vaddr, NODE_NETWORK, -1, true))
            {
                BOOST_FOREACH (CAddress& addr, vaddr)
                {
                    if (addr.GetByte(3) != 127)
                    {
                        addr.nTime = 0;
                        AddAddress(addr, 0, &addrDB);
                        found++;
                    }
                }
            }
        }

        addrDB.TxnCommit();  // Save addresses (it's ok if this fails)
    }

    debugprintf(ALWAYS, "%d addresses found from DNS seeds\n", found);
}


//123.243.184.249       murray
//91.228.199.245        coinotron
//88.198.14.120         alpha-mine-us
//206.217.130.93        backend.tecnode.org


void ThreadOpenConnections(void* parg)
{
    IMPLEMENT_RANDOMIZE_STACK(ThreadOpenConnections(parg));
    try
    {
        vnThreadsRunning[1]++;
        ThreadOpenConnections2(parg);
        vnThreadsRunning[1]--;
    }
    catch (std::exception& e) {
        vnThreadsRunning[1]--;
        PrintException(&e, "ThreadOpenConnections()");
    } catch (...) {
        vnThreadsRunning[1]--;
        PrintException(NULL, "ThreadOpenConnections()");
    }
    debugprintf(ALWAYS, "ThreadOpenConnections exiting\n");
}

void ThreadOpenConnections2(void* parg)
{
    bool bOwnAddNodes=false;
    debugprintf(ALWAYS, "ThreadOpenConnections started\n");

    // Connect to specific addresses
    if (Setting_Exist("connect"))
    {
        for (int64 nLoop = 0;; nLoop++)
        {
            BOOST_FOREACH(string strAddr, Setting_GetVector("connect"))
            {
                CAddress addr(strAddr, fAllowDNS);
                if (addr.IsValid())
                    OpenNetworkConnection(addr);
                for (int i = 0; i < 10 && i < nLoop; i++)
                {
                    Sleep(500);
                    if (fShutdown)
                        return;
                }
            }
        }
    }

    // Connect to manually added nodes first
    vector<std::string> nodes;
    vector<CAddress> addnodes;

    if(fAllowDNS)
    {
        nodes.push_back("node.solidcointalk.org");
        nodes.push_back("coinotron.com");
    }
    nodes.push_back("88.198.14.120");
    if (Setting_Exist("addnode"))
    {
        vector<std::string> tempnodes=Setting_GetVector("addnode");
        if(tempnodes.size())
        {
            nodes=tempnodes;
            bOwnAddNodes=true;
        }
    }

    BOOST_FOREACH(string strAddr, nodes)
    {
        CAddress addr(strAddr, fAllowDNS);
        if (addr.IsValid())
        {
            addnodes.push_back(addr);
            OpenNetworkConnection(addr);
            if (fShutdown)  return;
        }
    }

    // Initiate network connections
    int64 nStart = SolidTime_Get();
    int64 qAddNodeStart = SolidTime_Get();
    loop
    {
        // Limit outbound connections
        vnThreadsRunning[1]--;
        Sleep(500);

        loop
        {
            //if the user supplied their own nodes to connect to, always ensure we are connected to them
            //look every 2 minutes
            if(bOwnAddNodes && SolidTime_Get()-qAddNodeStart > 120)
            {
                qAddNodeStart=SolidTime_Get();
                BOOST_FOREACH(const CAddress &addr, addnodes)
                {
                    bool bFound=false;
                    {
                        MUTEX_LOCK(cs_vNodes);
                        BOOST_FOREACH(CNode* pnode, vNodes)
                        {
                            if(pnode->addr.ip == addr.ip)
                            {
                                bFound=true;
                                break;
                            }
                        }
                    }
                    if(bFound==false)
                    {
                        //readd an addnode connection
                        OpenNetworkConnection(addr);
                    }

                }
            }

            int nOutbound = 0;
            {
                MUTEX_LOCK(cs_vNodes);
                BOOST_FOREACH(CNode* pnode, vNodes)
                {
                    if (!pnode->fInbound)   nOutbound++;
                }
            }
            int nMaxOutboundConnections = MAX_OUTBOUND_CONNECTIONS;
            nMaxOutboundConnections = min(nMaxOutboundConnections, (int)Setting_GetINT64("maxconnections"));
            if (nOutbound < nMaxOutboundConnections)    break;
            Sleep(2000);
            if (fShutdown)  return;
        }

        vnThreadsRunning[1]++;
        if (fShutdown)
            return;


        // Choose an address to connect to based on most recently seen
        CAddress addrConnect;
        int64 nBest = INT64_MIN;

        // Only connect to one address per a.b.?.? range.
        // Do this here so we don't have to critsect vNodes inside mapAddresses critsect.
        set<unsigned int> setConnected;
        {
            MUTEX_LOCK(cs_vNodes);
            BOOST_FOREACH(CNode* pnode, vNodes)
            {
                setConnected.insert(pnode->addr.ip & 0x0000ffff);
            }
        }


        {
            MUTEX_LOCK(cs_mapAddresses);
            BOOST_FOREACH(const PAIRTYPE(vector<unsigned char>, CAddress)& item, mapAddresses)
            {
                const CAddress& addr = item.second;
                if (!addr.IsIPv4() || !addr.IsValid() || setConnected.count(addr.ip & 0x0000ffff))
                    continue;
                int64 nSinceLastSeen = SolidTime_Get() - addr.nTime;
                int64 nSinceLastTry = SolidTime_Get() - addr.nLastTry;

                // Randomize the order in a deterministic way, putting the standard port first
                int64 nRandomizer = (uint64)(nStart * 4951 + addr.nLastTry * 9567851 + addr.ip * 7789) % (2 * 60 * 60);
                if (addr.port != htons(GetDefaultPort()))
                    nRandomizer += 2 * 60 * 60;

                // Last seen  Base retry frequency
                //   <1 hour   10 min
                //    1 hour    1 hour
                //    4 hours   2 hours
                //   24 hours   5 hours
                //   48 hours   7 hours
                //    7 days   13 hours
                //   30 days   27 hours
                //   90 days   46 hours
                //  365 days   93 hours
                int64 nDelay = (int64)(3600.0 * sqrt(fabs((double)nSinceLastSeen) / 3600.0) + nRandomizer);

                // Fast reconnect for one hour after last seen
                if (nSinceLastSeen < 60 * 60)   nDelay = 10 * 60;

                // Limit retry frequency
                if (nSinceLastTry < nDelay) continue;

                // If we have IRC, we'll be notified when they first come online,
                // and again every 24 hours by the refresh broadcast.
                if (nGotIRCAddresses > 0 && vNodes.size() >= 2 && nSinceLastSeen > 24 * 60 * 60)    continue;

                // Only try the old stuff if we don't have enough connections
                if (vNodes.size() >= 8 && nSinceLastSeen > 24 * 60 * 60)    continue;

                // If multiple addresses are ready, prioritize by time since
                // last seen and time since last tried.
                int64 nScore = min(nSinceLastTry, (int64)24 * 60 * 60) - nSinceLastSeen - nRandomizer;
                if (nScore > nBest)
                {
                    nBest = nScore;
                    addrConnect = addr;
                }
            }
        }

        if (addrConnect.IsValid())
            OpenNetworkConnection(addrConnect);
    }
}

bool OpenNetworkConnection(const CAddress& addrConnect)
{
    // Initiate outbound network connection
    if (addrConnect.ip == addrLocalHost.ip || !addrConnect.IsIPv4() || FindNode(addrConnect.ip))
        return false;

    vnThreadsRunning[1]--;
    CNode* pnode = ConnectNode(addrConnect);
    vnThreadsRunning[1]++;
    if (!pnode)     return false;
    pnode->fNetworkNode = true;
    return true;
}


void ThreadMessageHandler(void* parg)
{
    IMPLEMENT_RANDOMIZE_STACK(ThreadMessageHandler(parg));
    try
    {
        vnThreadsRunning[2]++;
        ThreadMessageHandler2(parg);
        vnThreadsRunning[2]--;
    }
    catch (std::exception& e) {
        vnThreadsRunning[2]--;
        PrintException(&e, "ThreadMessageHandler()");
    } catch (...) {
        vnThreadsRunning[2]--;
        PrintException(NULL, "ThreadMessageHandler()");
    }
    debugprintf(ALWAYS, "ThreadMessageHandler exiting\n");
}

void ThreadMessageHandler2(void* parg)
{
    debugprintf(ALWAYS, "ThreadMessageHandler started\n");
    //SetThreadPriority(THREAD_PRIORITY_BELOW_NORMAL);
    while (!fShutdown)
    {
        vector<CNode*> vNodesCopy;

        {
            MUTEX_LOCK(cs_vNodes);
            vNodesCopy = vNodes;
            BOOST_FOREACH(CNode* pnode, vNodesCopy)
                pnode->AddRef();
        }

        // Poll the connected nodes for messages
        CNode* pnodeTrickle = NULL;
        if (!vNodesCopy.empty())
            pnodeTrickle = vNodesCopy[GetRand(vNodesCopy.size())];
        BOOST_FOREACH(CNode* pnode, vNodesCopy)
        {
            // Receive messages
            TRY_CRITICAL_BLOCK(pnode->cs_vRecv)
            {
                if (pnode->m_bNeedProcess)
                {
                    pnode->m_bNeedProcess = false;
                    ProcessMessages(pnode);
                }
            }

            if (fShutdown)  return;

            // Send messages
            TRY_CRITICAL_BLOCK(pnode->cs_vSend)
            {
                SendMessages(pnode, pnode == pnodeTrickle);
            }
            if (fShutdown)  return;
        }


        {
            MUTEX_LOCK(cs_vNodes);
            BOOST_FOREACH(CNode* pnode, vNodesCopy)
            {
                pnode->Release();
            }
        }

        // Wait and allow messages to bunch up.
        // Reduce vnThreadsRunning so StopNode has permission to exit while
        // we're sleeping, but we must always check fShutdown after doing this.
        vnThreadsRunning[2]--;
        Sleep(100);
        if (fRequestShutdown)   Shutdown(NULL);
        vnThreadsRunning[2]++;
        if (fShutdown) return;
    }
}






bool BindListenPort(string& strError)
{
    strError = "";
    int nOne = 1;
    addrLocalHost.port = htons(GetListenPort());

#ifdef __WXMSW__
    // Initialize Windows Sockets
    WSADATA wsadata;
    int ret = WSAStartup(MAKEWORD(2,2), &wsadata);
    if (ret != NO_ERROR)
    {
        strError = strprintf("Error: TCP/IP socket library failed to start (WSAStartup returned error %d)", ret);
        debugprintf(ERR, "%s\n", strError.c_str());
        return false;
    }
#endif

    // Create socket for listening for incoming connections
    hListenSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (hListenSocket == INVALID_SOCKET)
    {
        strError = strprintf("Error: Couldn't open socket for incoming connections (socket returned error %d)", WSAGetLastError());
        debugprintf(ERR, "%s\n", strError.c_str());
        return false;
    }

#ifdef SO_NOSIGPIPE
    // Different way of disabling SIGPIPE on BSD
    setsockopt(hListenSocket, SOL_SOCKET, SO_NOSIGPIPE, (void*)&nOne, sizeof(int));
#endif

#ifndef __WXMSW__
    // Allow binding if the port is still in TIME_WAIT state after
    // the program was closed and restarted.  Not an issue on windows.
    setsockopt(hListenSocket, SOL_SOCKET, SO_REUSEADDR, (void*)&nOne, sizeof(int));
#endif

#ifdef __WXMSW__
    // Set to nonblocking, incoming connections will also inherit this
    if (ioctlsocket(hListenSocket, FIONBIO, (u_long*)&nOne) == SOCKET_ERROR)
#else
    if (fcntl(hListenSocket, F_SETFL, O_NONBLOCK) == SOCKET_ERROR)
#endif
    {
        strError = strprintf("Error: Couldn't set properties on socket for incoming connections (error %d)", WSAGetLastError());
        debugprintf(ERR, "%s\n", strError.c_str());
        return false;
    }

    // The sockaddr_in structure specifies the address family,
    // IP address, and port for the socket that is being bound
    struct sockaddr_in sockaddr;
    memset(&sockaddr, 0, sizeof(sockaddr));
    sockaddr.sin_family = AF_INET;
    sockaddr.sin_addr.s_addr = INADDR_ANY; // bind to all IPs on this computer
    sockaddr.sin_port = htons(GetListenPort());
    if (::bind(hListenSocket, (struct sockaddr*)&sockaddr, sizeof(sockaddr)) == SOCKET_ERROR)
    {
        int nErr = WSAGetLastError();
        if (nErr == WSAEADDRINUSE)
            strError = strprintf(_("Unable to bind to port %d on this computer.  SolidCoin is probably already running."), ntohs(sockaddr.sin_port));
        else
            strError = strprintf("Error: Unable to bind to port %d on this computer (bind returned error %d)", ntohs(sockaddr.sin_port), nErr);
        debugprintf(ERR, "%s\n", strError.c_str());
        return false;
    }
    debugprintf(ALWAYS, "Bound to port %d\n", ntohs(sockaddr.sin_port));

    // Listen for incoming connections
    if (listen(hListenSocket, SOMAXCONN) == SOCKET_ERROR)
    {
        strError = strprintf("Error: Listening for incoming connections failed (listen returned error %d)", WSAGetLastError());
        debugprintf(ERR, "%s\n", strError.c_str());
        return false;
    }

    return true;
}

void StartNode(void* parg)
{
    if (pnodeLocalHost == NULL)
        pnodeLocalHost = new CNode(INVALID_SOCKET, CAddress("127.0.0.1", 0, false, nLocalServices));

#ifdef __WXMSW__
    // Get local host ip
    char pszHostName[1000] = "";
    if (gethostname(pszHostName, sizeof(pszHostName)) != SOCKET_ERROR)
    {
        vector<CAddress> vaddr;
        if (Lookup(pszHostName, vaddr, nLocalServices, -1, true))
            BOOST_FOREACH (const CAddress &addr, vaddr)
                if (addr.GetByte(3) != 127)
                {
                    addrLocalHost = addr;
                    break;
                }
    }
#else
    // Get local host ip
    struct ifaddrs* myaddrs;
    if (getifaddrs(&myaddrs) == 0)
    {
        for (struct ifaddrs* ifa = myaddrs; ifa != NULL; ifa = ifa->ifa_next)
        {
            if (ifa->ifa_addr == NULL) continue;
            if ((ifa->ifa_flags & IFF_UP) == 0) continue;
            if (strcmp(ifa->ifa_name, "lo") == 0) continue;
            if (strcmp(ifa->ifa_name, "lo0") == 0) continue;
            char pszIP[100];
            if (ifa->ifa_addr->sa_family == AF_INET)
            {
                struct sockaddr_in* s4 = (struct sockaddr_in*)(ifa->ifa_addr);
                if (inet_ntop(ifa->ifa_addr->sa_family, (void*)&(s4->sin_addr), pszIP, sizeof(pszIP)) != NULL)
                    debugprintf(INFO, "ipv4 %s: %s\n", ifa->ifa_name, pszIP);

                // Take the first IP that isn't loopback 127.x.x.x
                CAddress addr(*(unsigned int*)&s4->sin_addr, GetListenPort(), nLocalServices);
                if (addr.IsValid() && addr.GetByte(3) != 127)
                {
                    addrLocalHost = addr;
                    break;
                }
            }
            else if (ifa->ifa_addr->sa_family == AF_INET6)
            {
                struct sockaddr_in6* s6 = (struct sockaddr_in6*)(ifa->ifa_addr);
                if (inet_ntop(ifa->ifa_addr->sa_family, (void*)&(s6->sin6_addr), pszIP, sizeof(pszIP)) != NULL)
                    debugprintf(INFO, "ipv6 %s: %s\n", ifa->ifa_name, pszIP);
            }
        }
        freeifaddrs(myaddrs);
    }
#endif
    debugprintf(ALWAYS, "addrLocalHost = %s\n", addrLocalHost.ToString().c_str());

    if (g_bUseProxy || Setting_Exist("connect") || fNoListen)
    {
        // Proxies can't take incoming connections
        addrLocalHost.ip = CAddress("0.0.0.0").ip;
        debugprintf(ALWAYS, "addrLocalHost = %s\n", addrLocalHost.ToString().c_str());
    }
    else
    {
        CreateThread(ThreadGetMyExternalIP, NULL);
    }

    CreateThread(ThreadGetLatestBlockCount, NULL);


    //
    // Start threads
    //

    // Map ports with UPnP
    MapPort(Setting_GetBOOL("upnp"));

    // Get addresses from IRC and advertise ours
    if (!CreateThread(ThreadIRCSeed, NULL))
        debugprintf(ERR, "Error: CreateThread(ThreadIRCSeed) failed\n");

    // Send and receive from sockets, accept connections
    CreateThread(ThreadSocketHandler, NULL, true);

    // Initiate outbound connections
    if (!CreateThread(ThreadOpenConnections, NULL))
        debugprintf(ERR, "Error: CreateThread(ThreadOpenConnections) failed\n");

    // Process messages
    if (!CreateThread(ThreadMessageHandler, NULL))
        debugprintf(ERR, "Error: CreateThread(ThreadMessageHandler) failed\n");
}

bool StopNode()
{
    debugprintf(ALWAYS, "StopNode()\n");
    fShutdown = true;
    nTransactionsUpdated++;
    int64 nStart = SolidTime_Get();
    while (vnThreadsRunning[0] > 0 || vnThreadsRunning[2] > 0 || vnThreadsRunning[3] > 0 || vnThreadsRunning[4] > 0 || vaMultiThreads1>0
#ifdef USE_UPNP
        || vnThreadsRunning[5] > 0
#endif
    )
    {
        if (SolidTime_Get() - nStart > 20)
            break;
        Sleep(20);
    }
    if (vnThreadsRunning[0] > 0) debugprintf(WARN, "ThreadSocketHandler still running\n");
    if (vnThreadsRunning[1] > 0) debugprintf(WARN, "ThreadOpenConnections still running\n");
    if (vnThreadsRunning[2] > 0) debugprintf(WARN, "ThreadMessageHandler still running\n");
    if (vnThreadsRunning[3] > 0) debugprintf(WARN, "ThreadSolidCoinMiner still running\n");
    if (vnThreadsRunning[4] > 0) debugprintf(WARN, "ThreadRPCListener still running\n");
    if (vaMultiThreads1 > 0) debugprintf(WARN, "ThreadsRPCServer still running\n");
    if (fHaveUPnP && vnThreadsRunning[5] > 0) debugprintf(WARN, "ThreadMapPort still running\n");
    while (vnThreadsRunning[2] > 0 || vnThreadsRunning[4] > 0)
        Sleep(20);
    Sleep(50);

    return true;
}

class CNetCleanup
{
public:
    CNetCleanup()
    {
    }
    ~CNetCleanup()
    {
        // Close sockets
        BOOST_FOREACH(CNode* pnode, vNodes)
            if (pnode->hSocket != INVALID_SOCKET)
                closesocket(pnode->hSocket);
        if (hListenSocket != INVALID_SOCKET)
            if (closesocket(hListenSocket) == SOCKET_ERROR)
                debugprintf(ERR, "closesocket(hListenSocket) failed with error %d\n", WSAGetLastError());

#ifdef __WXMSW__
        // Shutdown Windows Sockets
        WSACleanup();
#endif
    }
}
instance_of_cnetcleanup;
