/*
 * Copyright 2002-2011 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.web.method.support;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.lang.reflect.Method;

import org.junit.Before;
import org.junit.Test;
import org.springframework.core.MethodParameter;

/**
 * Test fixture with {@link HandlerMethodArgumentResolverComposite}.
 * 
 * @author Rossen Stoyanchev
 */
public class HandlerMethodArgumentResolverCompositeTests {

	private HandlerMethodArgumentResolverComposite resolvers;

	private MethodParameter paramInt;

	private MethodParameter paramStr;

	@Before
	public void setUp() throws Exception {
		resolvers = new HandlerMethodArgumentResolverComposite();
		
		Method method = getClass().getDeclaredMethod("handle", Integer.class, String.class);
		paramInt = new MethodParameter(method, 0);
		paramStr = new MethodParameter(method, 1);
	}
	
	@Test
	public void supportsParameter() throws Exception {
		registerResolver(Integer.class, null);
		
		assertTrue(this.resolvers.supportsParameter(paramInt));
		assertFalse(this.resolvers.supportsParameter(paramStr));
	}
	
	@Test
	public void resolveArgument() throws Exception {
		registerResolver(Integer.class, Integer.valueOf(55));
		Object resolvedValue = this.resolvers.resolveArgument(paramInt, null, null, null);

		assertEquals(Integer.valueOf(55), resolvedValue);
	}
	
	@Test
	public void checkArgumentResolverOrder() throws Exception {
		registerResolver(Integer.class, Integer.valueOf(1));
		registerResolver(Integer.class, Integer.valueOf(2));
		Object resolvedValue = this.resolvers.resolveArgument(paramInt, null, null, null);

		assertEquals("Didn't use the first registered resolver", Integer.valueOf(1), resolvedValue);
	}
	
	@Test(expected=IllegalArgumentException.class)
	public void noSuitableArgumentResolver() throws Exception {
		this.resolvers.resolveArgument(paramStr, null, null, null);
	}

	protected StubArgumentResolver registerResolver(Class<?> supportedType, Object stubValue) {
		StubArgumentResolver resolver = new StubArgumentResolver(supportedType, stubValue);
		this.resolvers.addResolver(resolver);
		return resolver;
	}

	@SuppressWarnings("unused")
	private void handle(Integer arg1, String arg2) {
	}

}