/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the UpdateDialog class
** File name: updatedialog.cpp
**
****************************************************************/

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QMessageBox>
#include <QSqlQuery>
#include <QSettings>
#include <QTextStream>
#include <QString>
#include <QtNetwork>
#include <QHash>
#include <QHashIterator>
#include <QSizePolicy>
#include <QApplication>

#include "updatedialog.h"
#include "def/defines.h"
#include "def/errordefines.h"
#include "errormessage.h"
#include "def/defines.h"

// Constructor
UpdateDialog::UpdateDialog(QWidget *parent) : QDialog(parent) {
	// Init flag default value: no new version available
    newVersion = false;

    // QHttp object provides interface to HTTP
    http = new QHttp(this);

	// Create dialog content
	createProgressinfo();
	createButtons();
	createLayout();
	createConnections();

	// Proxy settings
	readSettings();

	// Dialog defaults
    setWindowTitle(QObject::tr("Aktualisierung"));
	setWindowIcon(QIcon("img/icon.ico"));
    buttonUpdate->setFocus();
}

void UpdateDialog::showHelp() {
	helpBrowser = new HelpBrowser("update.html", this);
	helpBrowser->show();
}

void UpdateDialog::createProgressinfo() {
    labelStatus = new QLabel(QObject::tr("Hier knnen Sie eine Aktualisierung "
    	"der Lektionen durchfhren.\nFr die Aktualisierung ist eine "
    	"Internetverbindung erforderlich."));
    progressBar = new QProgressBar(this);
    progressBar->setVisible(false);
}

void UpdateDialog::createButtons() {
	//Buttons
    buttonClose = new QPushButton(QObject::tr("Schlieen"));
    buttonUpdate = new QPushButton(QObject::tr("Aktualisierung starten"));
	buttonHelp = new QPushButton(QObject::tr("&Hilfe"));
    checkProxy = new QCheckBox(QObject::tr("ber einen Proxyserver verbinden"));
    txtProxyServer = new QLineEdit();
    txtProxyServer->setShown(false);
    txtProxyPort = new QLineEdit();
    txtProxyPort->setShown(false);
    labelProxyServer = new QLabel(QObject::tr("Server:"));
    labelProxyServer->setShown(false);
    labelProxyPort = new QLabel(QObject::tr("Port:"));
    labelProxyPort->setShown(false);
    buttonUpdate->setDefault(true);
}

void UpdateDialog::createLayout() {
    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget(buttonClose);
    buttonLayout->addSpacing(10);
    buttonLayout->addWidget(buttonHelp);
    buttonLayout->addWidget(buttonUpdate);

    QHBoxLayout *proxyLayout = new QHBoxLayout;
    proxyLayout->addWidget(labelProxyServer);
    proxyLayout->addWidget(txtProxyServer);
    proxyLayout->addWidget(labelProxyPort);
    proxyLayout->addWidget(txtProxyPort);

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(labelStatus);
    mainLayout->addWidget(progressBar);
    mainLayout->addWidget(checkProxy);
    mainLayout->addLayout(proxyLayout);
    mainLayout->addLayout(buttonLayout);
    mainLayout->setSizeConstraint(QLayout::SetFixedSize);
    setLayout(mainLayout);
}

void UpdateDialog::createConnections() {
    connect(http, SIGNAL(responseHeaderReceived(const QHttpResponseHeader &)),
    	this, SLOT(readResponseHeader(const QHttpResponseHeader &)));
    connect(http, SIGNAL(done(bool)), this, SLOT(httpDownloadFinished(bool)));
    connect(http, SIGNAL(dataReadProgress(int, int)), this,
    	SLOT(updateDataReadProgress(int, int)));
    connect(buttonUpdate, SIGNAL(clicked()), this,
    	SLOT(downloadVersionFile()));
    connect(buttonClose, SIGNAL(clicked()), this, SLOT(writeSettings()));
    connect(buttonClose, SIGNAL(clicked()), this, SLOT(close()));
    connect(checkProxy, SIGNAL(toggled(bool)), txtProxyServer, SLOT(setShown(bool)));
    connect(checkProxy, SIGNAL(toggled(bool)), txtProxyPort, SLOT(setShown(bool)));
    connect(checkProxy, SIGNAL(toggled(bool)), labelProxyServer, SLOT(setShown(bool)));
    connect(checkProxy, SIGNAL(toggled(bool)), labelProxyPort, SLOT(setShown(bool)));
    connect(buttonHelp, SIGNAL(clicked()), this, SLOT(showHelp()));
}

void UpdateDialog::downloadVersionFile() {
	// Save proxy settings
	writeSettings();

	labelStatus->setText(
		QObject::tr("Versionsinformationen herunterladen..."));
	progressBar->setVisible(true);
	qApp->processEvents();

    tempVersionFile = new QTemporaryFile;
    if (!tempVersionFile->open()) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_TEMP_FILE_CREATION, TYPE_CRITICAL,
			CANCEL_UPDATE);
        delete tempVersionFile;
        progressBar->setVisible(false);
        buttonUpdate->setEnabled(true);
        return;
    }

    buttonUpdate->setEnabled(false);

    checkProxy->setVisible(false);
    labelProxyServer->setVisible(false);
    txtProxyServer->setVisible(false);
    labelProxyPort->setVisible(false);
    txtProxyPort->setVisible(false);

    http->setHost(UPDATE_URL);

    // Proxy server?
    if (checkProxy->isChecked()) {
    	http->setProxy(txtProxyServer->text(), txtProxyPort->text().toInt());
	}

    http->get(UPDATE_URL_VERSION, tempVersionFile);
}

void UpdateDialog::downloadSqlFile() {
	http->abort();
	progressBar->setVisible(true);
	labelStatus->setText(QObject::tr("SQL-Datei herunterladen..."));
	qApp->processEvents();
	tempSqlFile = new QTemporaryFile;
    if (!tempSqlFile->open()) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_TEMP_FILE_CREATION, TYPE_CRITICAL,
			CANCEL_UPDATE);
        delete tempSqlFile;
		close();
        return;
    }
    buttonUpdate->setEnabled(false);

	http->setHost(UPDATE_URL);

    // Proxy server?
    if (checkProxy->isChecked()) {
    	http->setProxy(txtProxyServer->text(), txtProxyPort->text().toInt());
	}

	http->get(UPDATE_URL_SQL, tempSqlFile);
}

void UpdateDialog::updateDataReadProgress(int bytesRead, int totalBytes) {
    progressBar->setMaximum(totalBytes);
    progressBar->setValue(bytesRead);
}

void UpdateDialog::readResponseHeader(const QHttpResponseHeader
		&responseHeader) {
    if (responseHeader.statusCode() != 200) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_UPDATE_EXECUTION, TYPE_CRITICAL,
			CANCEL_UPDATE);
        http->abort();
        return;
    }
}

void UpdateDialog::httpDownloadFinished(bool error) {
	// Download finished
	if (error) {
		// Error message + additional error information
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_UPDATE_EXECUTION, TYPE_CRITICAL,
			CANCEL_UPDATE, http->errorString());
		close();
		return;
    }

	if (!newVersion) {
		// First check the database version
		labelStatus->setText(QObject::tr("Version berprfen..."));
		qApp->processEvents();
		if (checkVersionFile()) {
			// DB Version is new
			// -> download sql file
			delete tempVersionFile;
			newVersion = true;
			downloadSqlFile();
		}
	} else {
		// Execute sql file and analyze current text in DB
		labelStatus->setText(APP_NAME + QObject::tr(" Datenbank "
			"aktualisieren..."));
		labelStatus->update();
		qApp->processEvents();
		if (executeSqlFile()) {
			if (analyzeLessons("lesson")) {
				if (analyzeLessons("open")) {
					labelStatus->setText(APP_NAME + QObject::tr(" wurde "
						"erfolgreich aktualisiert!"));
					buttonClose->setFocus();
				} else {
					close();
				}
			} else {
				close();
			}
		} else {
			close();
		}
	}
}

bool UpdateDialog::checkVersionFile() {
	// Go to the beginning of the version file
    tempVersionFile->seek(0);

	QSqlQuery query;
	QTextStream in(tempVersionFile);
	// Read only the first line (server DB version)
	QString updateVersion = in.readLine();
	if (updateVersion.isNull()) {
		// Can't read line
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_ONLINE_VERSION_READABLE, TYPE_CRITICAL,
			CANCEL_UPDATE);
        close();
        return false;
	} else {
		// Check DB version of software
		if (!query.exec("SELECT * FROM db_version ORDER BY version DESC;")) {
			// Error message
			ErrorMessage *errorMessage = new ErrorMessage(this);
			errorMessage->showMessage(ERR_DB_VERSION_READABLE,
				TYPE_CRITICAL, CANCEL_UPDATE);
			close();
			return false;
		} else {
			if (!query.first()) {
				// Error message
				ErrorMessage *errorMessage = new ErrorMessage(this);
				errorMessage->showMessage(ERR_DB_VERSION_READABLE,
					TYPE_CRITICAL, CANCEL_UPDATE);
				close();
				return false;
			} else {
				// Server DB version is 0
				// -> software is too old to update
				QString softwareVersion = query.value(0).toString();
				if (updateVersion.trimmed() == "0") {
					QMessageBox::information(this, APP_NAME,
						QString(QObject::tr("Ihre Version der Software ist "
						"veraltet und nicht mehr aktualisierungsfhig.\nBitte "
						"laden Sie die neueste Version unter\n%1\nherunter. "
						"Vielen Dank!")).arg(APP_URL));
					close();
					return false;
				}
				// Check whether ther is a new DB version on the server
				if (softwareVersion.trimmed() != updateVersion.trimmed()) {
					labelStatus->setText(QObject::tr("Es stehen Updates fr "
						"Tipp10 zur Verfgung..."));
					return true;
				}
			}
		}
	}
	labelStatus->setText(APP_NAME + QObject::tr(" befindet sich bereits auf "
		"dem aktuellsten Stand.\nEs stehen derzeit keine Aktualisierungen zur "
		"Verfgung."));
	return false;

}

bool UpdateDialog::executeSqlFile() {
    QSqlQuery query;
	QString line = "";
	int bytesRead = 0;
	int totalBytes = tempSqlFile->size();

	// Go to the beginning of the version file
    tempSqlFile->seek(0);

	QTextStream in(tempSqlFile);

	// Execute all sql command of the downloaded file
	while (!in.atEnd()) {
		line = in.readLine();
		bytesRead += line.size();
		// Without error handling, because DROP-Statements are allowed to
		// be invalid (there exist also a IF EXISTS statement in the SQLite
		// library which suppresses an error, but it didn't work when I try it)
		if (!query.exec(line) && !line.contains("drop", Qt::CaseInsensitive)) {
			// Error message + failed sql string
			ErrorMessage *errorMessage = new ErrorMessage(this);
			errorMessage->showMessage(ERR_UPDATE_SQL_EXECUTION, TYPE_CRITICAL,
				CANCEL_UPDATE, line);
			return false;
		}
		updateDataReadProgress(bytesRead, totalBytes);
	}
	delete tempSqlFile;
	updateDataReadProgress(totalBytes, totalBytes);
	return true;
}

bool UpdateDialog::analyzeLessons(QString lessonType) {
    QSqlQuery mainQuery;
	QSqlQuery subQuery;
	QString queryString;
	QString indexString;
	QString lessonString;
	int lessonLen;
	QHash<int, int> unicodeErrorHash;
	QHash<int, int> unicodeErrorHashClean;
	int errorNum = 0;
	int rowsRead = 0;
	int totalRows = 0;
	int i;

	if (!mainQuery.exec("SELECT char_unicode FROM lesson_chars;")) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_ERROR_DEFINES_EXIST, TYPE_CRITICAL,
			CANCEL_UPDATE);
		return false;
	}

	while (mainQuery.next()) {
		unicodeErrorHashClean[mainQuery.value(0).toInt()] = 0;
		errorNum++;
	}

	// 1. Delete current lessonanalysis table
	// (no error message, table could not exist)
	mainQuery.exec("DROP TABLE " + lessonType + "_analysis;");

	// 2a. Create new analysis table query with new error definitions as columns
	queryString = "CREATE TABLE " + lessonType + "_analysis (analysis_content "
		"INTEGER primary key";
	indexString = "CREATE INDEX " + lessonType + "_analysis_index ON " +
		lessonType + "_analysis (analysis_content";

	QHashIterator<int, int> hashIteratorClean(unicodeErrorHashClean);
    while (hashIteratorClean.hasNext()) {
        hashIteratorClean.next();
        queryString += ", analysis_char_"
        	+ QString::number(hashIteratorClean.key()) + " INTEGER";
        indexString += ", analysis_char_"
        	+ QString::number(hashIteratorClean.key());
        //cout << i.key() << ": " << i.value() << endl;
    }

	queryString += ");";
	indexString += ");";

	// 2b. Execute new analysis table query
	if (!mainQuery.exec(queryString)) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_ANALYZE_TABLE_CREATION, TYPE_CRITICAL,
			CANCEL_UPDATE);
		return false;
	}

	// 2c. Execute new analysis table query
	if (!mainQuery.exec(indexString)) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_ANALYZE_INDEX_CREATION, TYPE_CRITICAL,
			CANCEL_UPDATE);
		return false;
	}

	// 3. Count number of lesson text
	if (!mainQuery.exec("SELECT COUNT(content_id) FROM " + lessonType +
		"_content;")) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSON_CONTENT_EXIST, TYPE_CRITICAL,
			CANCEL_UPDATE);
		return false;
	}
	mainQuery.first();
	totalRows = mainQuery.value(0).toInt();

	// 4. Query all lesson text and analyze it
	if (!mainQuery.exec("SELECT content_id, content_text FROM " + lessonType +
		"_content ORDER BY content_id;")) {
		// Error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_LESSON_CONTENT_EXIST, TYPE_CRITICAL,
			CANCEL_UPDATE);
		return false;
	}

	while (mainQuery.next()) {
		unicodeErrorHash = unicodeErrorHashClean;
		lessonString = mainQuery.value(1).toString();
		lessonLen = lessonString.length();

        // Now count all error chars and put them into the hash table
		for (i = 0; i < lessonString.size(); i++) {
			if (unicodeErrorHash.contains(lessonString[i].unicode())) {
				unicodeErrorHash[lessonString[i].unicode()]++;
			}
		}

		queryString = "INSERT INTO " + lessonType + "_analysis VALUES(";
		queryString += mainQuery.value(0).toString();
		QHashIterator<int, int> hashIterator(unicodeErrorHash);
		//hashIterator.toFront();
		while (hashIterator.hasNext()) {
			hashIterator.next();
			// Weighted number of chars (char ratio) to SQL string
			queryString += ", " + QString::number(((double) hashIterator.value() /
				(double) lessonLen) * 100.0, 'f', 2);
		}
		queryString += ");";

		// Execute new analysis table query
		if (!subQuery.exec(queryString)) {
			// Error message
			ErrorMessage *errorMessage = new ErrorMessage(this);
			errorMessage->showMessage(ERR_ANALYZE_TABLE_FILL, TYPE_CRITICAL,
				CANCEL_UPDATE);
			return false;
		}
		rowsRead++;
		updateDataReadProgress(rowsRead, totalRows);
	}

	updateDataReadProgress(totalRows, totalRows);
	return true;
}

void UpdateDialog::readSettings() {
	QSettings settings;
	settings.beginGroup("proxy");
	checkProxy->setChecked(settings.value("check_proxy", false).toBool());
	txtProxyServer->setText(settings.value("proxy_server", "").toString());
	txtProxyPort->setText(settings.value("proxy_port", "").toString());
	settings.endGroup();
	// Show proxy settings if proxy is checked
    if (checkProxy->isChecked()) {
	    labelProxyServer->setVisible(true);
	    txtProxyServer->setVisible(true);
	    labelProxyPort->setVisible(true);
   		txtProxyPort->setVisible(true);
	}
}

void UpdateDialog::writeSettings() {
	QSettings settings;
	settings.beginGroup("proxy");
	settings.setValue("check_proxy", checkProxy->isChecked());
	settings.setValue("proxy_server", txtProxyServer->text());
	settings.setValue("proxy_port", txtProxyPort->text());
	settings.endGroup();
}
