/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the KeyBoard class
** File name: progressionwidget.cpp
**
****************************************************************/

#include <QPainter>
#include <QSettings>
#include <QSqlQuery>
#include <QSqlDriver>
#include <QVariant>
#include <QSqlDatabase>
#include <QLineF>
#include <QPen>
#include <QDateTime>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QRectF>

#include "progressionwidget.h"

ProgressionWidget::ProgressionWidget(QWidget *parent) : QWidget(parent) {
    // Fix the size of this class because of using fix sized images
    //setFixedSize(600, 310);
	whereClausel = "";
	orderClausel = "ORDER BY user_lesson_timestamp";

	xAxis = "Zeitpunkt";
	xAxisColumn = 1;

	// Create filter headline
	labelFilter = new QLabel(QObject::tr("Zeige: "));
	comboFilter = new QComboBox();
	comboFilter->insertItem(0, QObject::tr("Alle Lektionen"));
	comboFilter->insertItem(1, QObject::tr("bungslektionen"));
	comboFilter->insertItem(2, QObject::tr("Freie Lektionen"));
	comboFilter->insertItem(3, QObject::tr("Eigene Lektionen"));
	labelOrder = new QLabel(QObject::tr("X-Achse sortieren nach: "));
	comboOrder = new QComboBox();
	comboOrder->insertItem(0, QObject::tr("Zeitpunkt"));
	comboOrder->insertItem(1, QObject::tr("Lektion"));
	//comboOrder->insertItem(2, QObject::tr("A/min"));
	//comboOrder->insertItem(3, QObject::tr("Bewertung"));

	// Set a horizonal layout
	QHBoxLayout *filterLayout = new QHBoxLayout;
	filterLayout->addStretch(1);
	filterLayout->addWidget(labelOrder);
	filterLayout->addWidget(comboOrder);
	filterLayout->addSpacing(20);
	filterLayout->addWidget(labelFilter);
	filterLayout->addWidget(comboFilter);
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addLayout(filterLayout);
	mainLayout->addStretch(1);
	// Pass layout to parent widget (this)
	this->setLayout(mainLayout);

	connect(comboFilter, SIGNAL(activated(int)), this, SLOT(changeFilter(int)));
	connect(comboOrder, SIGNAL(activated(int)), this, SLOT(changeOrder(int)));
	getChartValues();
}

void ProgressionWidget::getChartValues() {
	QSqlQuery query;
	QString lessonNumber;
	lessonCounter = 0;
	lessonAv = 0;
	lessonsNumbers.clear();
	lessonsTimestamps.clear();
	lessonsGrades.clear();
	lessonsType.clear();
	// SQL: all lessons sorted by id and a left joint to the number of
	// lessons done by the user
	if (!query.exec("SELECT user_lesson_lesson, user_lesson_timestamp, "
			"ROUND(((user_lesson_strokesnum - (20 * user_lesson_errornum)) / "
			"(user_lesson_timelen / 60.0)) * 0.4, 0) AS user_lesson_grade, "
			"user_lesson_id, ROUND(user_lesson_strokesnum / "
			"(user_lesson_timelen / 60.0), 0), user_lesson_type, "
			"user_lesson_name AS user_lesson_cpm FROM user_lesson_list " +
			whereClausel + orderClausel + ";")) {
		return;
	}
	lessonGradeMax = 0;
	// Read all datasets to list items
	while (query.next()) {
		// Number of the lesson
		lessonsNumbers.append(query.value(0).toString());
		// Timestamp of the lesson
		if (xAxisColumn == 1) {
			lessonsTimestamps.append(
				QDateTime::fromString(query.value(xAxisColumn).toString(),
				"yyyyMMddhhmmss").toString(DATE_FORMAT));
		} else {
			if (xAxisColumn == 2 && query.value(2).toInt() < 0) {
				lessonsTimestamps.append("0");
			} else {
				lessonsTimestamps.append(query.value(xAxisColumn).toString());
			}
		}
		// Type of lesson
		lessonsType.append(query.value(5).toInt());
		// Grade of the lesson
		lessonsGrades.append(query.value(2).toInt());

		// Maximum
		if (query.value(2).toInt() > lessonGradeMax) {
			lessonGradeMax = query.value(2).toInt();
		}

		// Average
		if (query.value(2).toInt() > 0) {
			lessonAv += query.value(2).toInt() ;
		}
		lessonCounter++;
	}
	if (lessonGradeMax > 120) {
		lessonGradeMax += 10;
	} else {
		lessonGradeMax = 120;
	}
	if (lessonCounter != 0) {
		lessonAv = lessonAv / lessonCounter;
	}
	update();
}

void ProgressionWidget::paintEvent(QPaintEvent *event) {
	if (lessonCounter > 1) {
		drawGrid();
		drawGraph();
	} else {
		drawNothing();
	}
}

void ProgressionWidget::drawGrid() {
	QPainter painter(this);
	//painter.setRenderHint(QPainter::Antialiasing);
	int widgetLeft = 20;
	int widgetTop = 40;
	int widgetWidth = this->width() - 20;
	int widgetHeight = this->height();
	int lessonCounterTemp;
	if (lessonCounter > 1) {
		lessonCounterTemp = lessonCounter;
	} else {
		lessonCounterTemp = 2;
	}
	int xUnit = (widgetWidth - 120) / (lessonCounterTemp - 1);
	int yUnit = (widgetHeight - 110) / lessonGradeMax; //120;
	QPen penDashLine;

	painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
	painter.setPen(QColor(0, 0, 0));
	// Text y axis
	painter.drawText(widgetLeft,
		widgetTop + widgetHeight - 113 - (lessonGradeMax * yUnit),
		50,
		10,
		Qt::AlignCenter | Qt::AlignVCenter,
		QObject::tr("Punkte"));
	// Text x axis
	painter.drawText(widgetLeft + 60 + ((lessonCounterTemp - 1) * xUnit),
		widgetTop + widgetHeight - 91,
		60,
		20,
		Qt::AlignLeft | Qt::AlignVCenter,
		xAxis);
	// y axis
	painter.drawLine(widgetLeft + 30,
		widgetTop + widgetHeight - 95 - (lessonGradeMax * yUnit),
		widgetLeft + 30,
		widgetTop + widgetHeight - 80);
	// y axis arrow
	painter.drawLine(widgetLeft + 28,
		widgetTop + widgetHeight - 93 - (lessonGradeMax * yUnit),
		widgetLeft + 30,
		widgetTop + widgetHeight - 95 - (lessonGradeMax * yUnit));
	painter.drawLine(widgetLeft + 32,
		widgetTop + widgetHeight - 93 - (lessonGradeMax * yUnit),
		widgetLeft + 30,
		widgetTop + widgetHeight - 95 - (lessonGradeMax * yUnit));
	// x axis
	painter.drawLine(widgetLeft + 28,
		widgetTop + widgetHeight - 80,
		widgetLeft + 50 + ((lessonCounterTemp - 1) * xUnit),
		widgetTop + widgetHeight - 80);
	// x axis arrow
	painter.drawLine(widgetLeft+ 48 + ((lessonCounterTemp - 1) * xUnit),
		widgetTop + widgetHeight - 82,
		widgetLeft + 50 + ((lessonCounterTemp - 1) * xUnit),
		widgetTop + widgetHeight - 80);
	painter.drawLine(widgetLeft + 48 + ((lessonCounterTemp - 1) * xUnit),
		widgetTop + widgetHeight - 78,
		widgetLeft + 50 + ((lessonCounterTemp - 1) * xUnit),
		widgetTop + widgetHeight - 80);
	// Bottom unit line y axis
	painter.drawText(widgetLeft + 4,
		widgetTop + widgetHeight - 85,
		20,
		10,
		Qt::AlignCenter | Qt::AlignVCenter,
		"0");
	// Unit lines and text y axis
	for (int x = 1; x <= (lessonGradeMax / 10); x++) {
		painter.drawLine(widgetLeft + 28,
			widgetTop + widgetHeight - 80 - (x * yUnit * 10),
			widgetLeft + 50 + ((lessonCounterTemp - 1) * xUnit),
			widgetTop + widgetHeight - 80 - (x * yUnit * 10));
		painter.drawText(widgetLeft + 4,
			widgetTop + widgetHeight - 85 - (x * yUnit * 10),
			20,
			10,
			Qt::AlignCenter | Qt::AlignVCenter,
			QString::number(x * 10));
	}
	// Text x axis
	if (lessonCounter > 0) {
		painter.drawText(widgetLeft - 20,
			widgetTop + widgetHeight - 74,
			120,
			30,
			Qt::AlignCenter | Qt::AlignVCenter | Qt::TextWordWrap,
			lessonsTimestamps[0]);
	}
	if (lessonCounter > 1) {
		painter.drawText(widgetLeft - 20 + ((lessonCounter - 1) * xUnit),
			widgetTop + widgetHeight - 74,
			120,
			30,
			Qt::AlignCenter | Qt::AlignVCenter | Qt::TextWordWrap,
			lessonsTimestamps[lessonCounter - 1]);
	}
	// Average
	penDashLine.setBrush(QColor(249, 126, 50));
	penDashLine.setStyle(Qt::DashLine);
    painter.setPen(penDashLine);
	painter.drawLine(widgetLeft + 28,
		widgetTop + widgetHeight - 80 - (lessonAv * yUnit),
		widgetLeft + 50 + ((lessonCounterTemp - 1) * xUnit),
		widgetTop + widgetHeight - 80 - (lessonAv * yUnit));
	penDashLine.setStyle(Qt::SolidLine);
    painter.setPen(penDashLine);
	painter.drawEllipse (widgetLeft - 10,
		widgetTop + (widgetHeight - 80) - (lessonAv * yUnit) - 3,
		6,
		6);
	painter.drawLine(widgetLeft - 5,
		widgetTop + widgetHeight - 80 - (lessonAv * yUnit) - 5,
		widgetLeft - 10,
		widgetTop + widgetHeight - 80 - (lessonAv * yUnit) + 5);
}

void ProgressionWidget::drawGraph() {
	QPainter painter(this);
	painter.setRenderHint(QPainter::Antialiasing);
	QPen penOrange;
	QPen penBlack;
	QPen penGrey;
	QPen penWhite;
	QLineF line;
	int currentValue;
	int nextValue;
	int widgetLeft = 20;
	int widgetTop = 40;
	int widgetWidth = this->width() - 20;
	int widgetHeight = this->height();
	int lessonCounterTemp;
	if (lessonCounter > 1) {
		lessonCounterTemp = lessonCounter;
	} else {
		lessonCounterTemp = 2;
	}
	int xUnit = (widgetWidth - 120) / (lessonCounterTemp - 1);
	int yUnit = (widgetHeight - 110) / lessonGradeMax;

	painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));

    penOrange.setStyle(Qt::SolidLine);
	penOrange.setBrush(QColor(249, 126, 50));
    penOrange.setCapStyle(Qt::RoundCap);
    penOrange.setJoinStyle(Qt::RoundJoin);
	penOrange.setWidth(2);

	penBlack.setStyle(Qt::SolidLine);
    penBlack.setBrush(QColor(0, 0, 0));
    penBlack.setCapStyle(Qt::RoundCap);
    penBlack.setJoinStyle(Qt::RoundJoin);
	penBlack.setWidth(2);

	penGrey.setStyle(Qt::SolidLine);
    penGrey.setBrush(QColor(60, 60, 180));
    penGrey.setCapStyle(Qt::RoundCap);
    penGrey.setJoinStyle(Qt::RoundJoin);
	penGrey.setWidth(2);

	penWhite.setStyle(Qt::SolidLine);
    penWhite.setBrush(QColor(60, 180, 60));
    penWhite.setCapStyle(Qt::RoundCap);
    penWhite.setJoinStyle(Qt::RoundJoin);
	penWhite.setWidth(2);

	// Legend
	if (whereClausel == "") {
		//Mac Version:
		//-----------
		/*
		painter.setPen(penBlack);
		painter.drawEllipse (widgetLeft + 60,
			widgetTop - 20,
			4,
			4);
		painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
		painter.drawText(widgetLeft + 70,
			widgetTop - 24,
			90,
			12,
			Qt::AlignLeft | Qt::AlignVCenter,
			QObject::tr("bungslektion"));

		painter.setPen(penGrey);
		painter.drawEllipse (widgetLeft + 60,
			widgetTop - 5,
			4,
			4);
		painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
		painter.drawText(widgetLeft + 70,
			widgetTop - 9,
			90,
			12,
			Qt::AlignLeft | Qt::AlignVCenter,
			QObject::tr("Freie Lektion"));

		painter.setPen(penWhite);
		painter.drawEllipse (widgetLeft + 60,
			widgetTop + 10,
			4,
			4);
		painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
		painter.drawText(widgetLeft + 70,
			widgetTop + 6,
			90,
			12,
			Qt::AlignLeft | Qt::AlignVCenter,
			QObject::tr("Eigene Lektion"));
		*/
		//Win Version:
		//-----------
		painter.setPen(penBlack);
		painter.drawEllipse (widgetLeft + 70,
			widgetTop - 20,
			4,
			4);
		painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
		painter.drawText(widgetLeft + 80,
			widgetTop - 24,
			90,
			12,
			Qt::AlignLeft | Qt::AlignVCenter,
			QObject::tr("bungslektion"));

		painter.setPen(penGrey);
		painter.drawEllipse (widgetLeft + 70,
			widgetTop - 5,
			4,
			4);
		painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
		painter.drawText(widgetLeft + 80,
			widgetTop - 9,
			90,
			12,
			Qt::AlignLeft | Qt::AlignVCenter,
			QObject::tr("Freie Lektion"));

		painter.setPen(penWhite);
		painter.drawEllipse (widgetLeft + 70,
			widgetTop + 10,
			4,
			4);
		painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
		painter.drawText(widgetLeft + 80,
			widgetTop + 6,
			90,
			12,
			Qt::AlignLeft | Qt::AlignVCenter,
			QObject::tr("Eigene Lektion"));
	}

	// Number of lines
	for (int x = 0; x < lessonCounter; x++) {
		painter.setPen(penOrange);
		currentValue = lessonsGrades[x];
		if (currentValue < 0) {
			currentValue = 0;
		}
		if (x < lessonCounter - 1) {
			nextValue = lessonsGrades[x + 1];
			if (nextValue < 0) {
				nextValue = 0;
			}
			painter.drawLine(widgetLeft + 40 + (x * xUnit),
				widgetTop + (widgetHeight - 80) - (currentValue * yUnit),
				widgetLeft + 40 + ((x + 1) * xUnit),
				widgetTop + (widgetHeight - 80) - (nextValue * yUnit));
		}
		switch (lessonsType[x]) {
			case 0:
			default:
				painter.setPen(penBlack);
				break;
			case 1:
				painter.setPen(penGrey);
				break;
			case 2:
				painter.setPen(penWhite);
				break;
		}
		painter.drawEllipse (widgetLeft + 40 + (x * xUnit) - 2,
			widgetTop + (widgetHeight - 80) - (currentValue * yUnit) - 2,
			4,
			4);
		//painter.setPen(QColor(255, 255, 255));
    	//painter.drawLine(60 + (x * xUnit), widgetHeight - 80,
		//	60 + (x * xUnit), (widgetHeight - 80) - (currentValue * yUnit));
		if (lessonsType[x] == 0) {
			painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS_LESSON));
			painter.drawText(widgetLeft + 30 + (x * xUnit),
				widgetTop + (widgetHeight - 96) - (currentValue * yUnit),
				20,
				10,
				Qt::AlignCenter | Qt::AlignVCenter,
				lessonsNumbers[x]);
		}
	}
}

void ProgressionWidget::drawNothing() {
	QPainter painter(this);
	//painter.setRenderHint(QPainter::Antialiasing);
	int widgetWidth = this->width();
	int widgetHeight = this->height();

	painter.setFont(QFont(FONT_STANDARD, FONT_SIZE_PROGRESS));
	painter.setPen(QColor(0, 0, 0));
	// Text y axis
	painter.drawText(0,
		0,
		widgetWidth,
		widgetHeight,
		Qt::AlignCenter | Qt::AlignVCenter,
		QObject::tr("Der Verlauf wird erst nach der zweiten absolvierten Lektion sichtbar."));
}

void ProgressionWidget::changeFilter(int rowindex) {
	// Select columnname from columnindex
	switch (rowindex) {
		case 0:
			whereClausel = "";
			break;
		case 1:
		default:
			whereClausel = "WHERE user_lesson_type = 0 ";
			break;
		case 2:
			whereClausel = "WHERE user_lesson_type = 1 ";
			break;
		case 3:
			whereClausel = "WHERE user_lesson_type = 2 ";
			break;
	}
	// Call new query
	getChartValues();
}

void ProgressionWidget::changeOrder(int rowindex) {
	// Select columnname from columnindex
	switch (rowindex) {
		case 0:
			orderClausel = "ORDER BY user_lesson_timestamp";
			xAxis = "Zeitpunkt";
			xAxisColumn = 1;
			break;
		case 1:
		default:
			orderClausel = "ORDER BY user_lesson_type, user_lesson_lesson";
			xAxis = "Lektion";
			xAxisColumn = 6;
			break;
		case 2:
			orderClausel = "ORDER BY user_lesson_cpm";
			xAxis = "A/min";
			xAxisColumn = 4;
			break;
		case 3:
			orderClausel = "ORDER BY user_lesson_grade";
			xAxis = "Bewertung";
			xAxisColumn = 2;
			break;
	}
	// Call new query
	getChartValues();
}
