/*************************************************************************
vStrip by [maven] (maven@maven.de)
aspi.c: routines for ASPI support (win32), refs: mmc3r09.pdf, adaptec aspi-sdk
(tabsize 2)
*************************************************************************/

#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include "aspi.h"

#ifdef aspi_USE_ASPI

static HINSTANCE	aspi_DLL = NULL;
static DWORD (*aspi_lpGetSupportInfo)(void);
static DWORD (*aspi_lpSendCommand)(LPSRB lpsrc);
static int aspi_DLLRefCount = 0;

VSTRIP_DLL_API bool VSTRIP_DLL_CC aspi_Init(void)
{
	if (aspi_DLL)
	{
		aspi_DLLRefCount++;
		return TRUE;
	}
	else if (!(aspi_DLL = LoadLibrary(aspi_DLL_NAME)))
		return FALSE;
	
	if (!((FARPROC)aspi_lpGetSupportInfo = GetProcAddress(aspi_DLL, "GetASPI32SupportInfo")))
	{
		FreeLibrary(aspi_DLL);
		aspi_DLL = NULL;
		return FALSE;
	}
	
	if (!((FARPROC)aspi_lpSendCommand = GetProcAddress(aspi_DLL, "SendASPI32Command")))
	{
		FreeLibrary(aspi_DLL);
		aspi_DLL = NULL;
		return FALSE;
	}
	
	aspi_DLLRefCount++;
	return TRUE;
}

VSTRIP_DLL_API void VSTRIP_DLL_CC aspi_Done(void)
{
	if (--aspi_DLLRefCount <= 0)
	{
		aspi_lpSendCommand = NULL;
		aspi_lpGetSupportInfo = NULL;
		if (aspi_DLL)
		{
			FreeLibrary(aspi_DLL);
			aspi_DLL = NULL;
		}
	}
}

DWORD aspi_GetSupportInfo(void)
{
	if (aspi_lpGetSupportInfo != NULL)
		return aspi_lpGetSupportInfo();
	return 0;
}

DWORD aspi_SendCommand(LPSRB srb)
{
	HANDLE hOldEvent = NULL, hNewEvent = NULL;
	DWORD	returnVal;
	SRB_ExecSCSICmd	*scsicmd;
	
	if (aspi_SendCommand == NULL)
		return 0;
	
	// Setup event handling if needed
	scsicmd = (LPSRB_ExecSCSICmd)srb;
	if ((scsicmd->SRB_Flags & SRB_EVENT_NOTIFY) == SRB_EVENT_NOTIFY)
	{
		hNewEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
		if (hNewEvent)
		{
			hOldEvent = (HANDLE)scsicmd->SRB_PostProc;
			scsicmd->SRB_PostProc = (LPVOID)hNewEvent;
		}
	}
	
	// Execute the SCSI command
	returnVal = aspi_lpSendCommand(srb);
	
	// Wait for completion event if needed
	if (hNewEvent != NULL)
	{
		if (returnVal == SS_PENDING)
			WaitForSingleObject(hNewEvent, INFINITE);
		CloseHandle(hNewEvent);
	}
	
	if (hOldEvent != NULL)
		SetEvent(hOldEvent);
	
	return scsicmd->SRB_Status;
}

DWORD aspi_SendCommandAsync(LPSRB srb, HANDLE* handle)
{
	HANDLE hNewEvent = NULL;
	DWORD	returnVal;
	SRB_ExecSCSICmd* scsicmd;
	
	if (aspi_SendCommand == NULL)
		return 0;
	
	// Setup event handling
	scsicmd = (LPSRB_ExecSCSICmd)srb;
	scsicmd->SRB_Flags |= SRB_EVENT_NOTIFY;
	hNewEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
	scsicmd->SRB_PostProc = (LPVOID)hNewEvent;
	
	// Execute the SCSI command
	returnVal = aspi_lpSendCommand(srb);
	if (returnVal != SS_COMP && returnVal != SS_PENDING)
	{
		CloseHandle(hNewEvent);
		hNewEvent = NULL;
	}
	*handle = hNewEvent;
	return returnVal;
}

DWORD aspi_WaitCommandAsync(LPSRB srb, HANDLE handle)
{
	if (handle != NULL)
	{
		WaitForSingleObject(handle, INFINITE);
		CloseHandle(handle);
	}
	return ((SRB_ExecSCSICmd*)srb)->SRB_Status;
}

byte aspi_GetWinVersion(void)
{
	dword version = GetVersion();
	
	if (version < 0x80000000) // Windows NT/2000/Whistler
		return 2;
	else if (LOBYTE(LOWORD(version)) >= 4) // Windows 95/98/Me
		return 1;
	return 0;
}

VSTRIP_DLL_API bool VSTRIP_DLL_CC aspi_GetDriveAddress(char drive_letter, dword *address)
{
	byte ver = aspi_GetWinVersion();
	
	drive_letter = toupper(drive_letter);
	switch (ver)
	{
	case 2: // NT
		{
			char DrivePath[7] = "\\\\.\\x:";
			HANDLE h;
			unsigned long	bytesReturned;
			SCSI_ADDRESS dataBuffer;
			BOOL res;
			
			DrivePath[4] = drive_letter;
			h = CreateFile(DrivePath, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL);
			if (!h)
				return FALSE;
			
			memset(&dataBuffer, 0, sizeof dataBuffer);
			res = DeviceIoControl(h, IOCTL_SCSI_GET_ADDRESS, NULL, 0, &dataBuffer, sizeof dataBuffer, &bytesReturned, FALSE);
			CloseHandle(h);
			if (res)
			{
				*address = (((dword)dataBuffer.PortNumber) << 16) + (((dword)dataBuffer.TargetId) << 8) + (dword)dataBuffer.Lun;
				return TRUE;
			}
			else
				return FALSE;
		}
		break;
	case 1: // 9x
		{
			int	ASPI32Status = aspi_GetSupportInfo();
			int drive = toupper(drive_letter) - toupper('A');
			int AdapterCount, i, j, k;
			
			AdapterCount = (LOBYTE(LOWORD(ASPI32Status)));
			for (i=0; i < AdapterCount; i++)
			{
				SRB_HAInquiry srbHAInquiry;
				int TargetCount;
				
				memset(&srbHAInquiry, 0, sizeof srbHAInquiry);
				srbHAInquiry.SRB_Cmd = SC_HA_INQUIRY;
				srbHAInquiry.SRB_HaId = i;
				
				aspi_SendCommand((LPSRB)&srbHAInquiry);
				if (srbHAInquiry.SRB_Status == SS_COMP)
				{
					TargetCount = srbHAInquiry.HA_Unique[3];
					if (TargetCount != 8 && TargetCount != 16)
						TargetCount = 8;
					for (j = 0; j < TargetCount; j++)
						for (k = 0; k < 8; k++)
						{
							SRB_GetDiskInfo	GDI;
							
							GDI.SRB_Cmd = SC_GET_DISK_INFO;
							GDI.SRB_HaId = i;
							GDI.SRB_Flags = SRB_EVENT_NOTIFY;
							GDI.SRB_Hdr_Rsvd = 0;
							GDI.SRB_Target = j;
							GDI.SRB_Lun = k;
							ASPI32Status = aspi_SendCommand((LPSRB)&GDI);
							if (ASPI32Status == SS_COMP && GDI.SRB_Int13HDriveInfo == drive)
							{
								*address = (((dword)i) << 16) + (((dword)j) << 8) + (dword)k;
								return TRUE;
							}
						}
				}
			}
			return FALSE;
		}
		break;
	}
	return FALSE;
}

VSTRIP_DLL_API bool VSTRIP_DLL_CC aspi_SetTimeout(const dword address, const dword seconds)
{
	dword err;
	SRB_GetSetTimeouts c;
	
	memset(&c, 0, sizeof c);
	c.SRB_Cmd = SC_GETSET_TIMEOUTS;
	c.SRB_HaId = aspi_GetHost(address);
	c.SRB_Flags = SRB_DIR_OUT;
	c.SRB_Target = aspi_GetTarget(address);
	c.SRB_Lun = aspi_GetLun(address);
	c.SRB_Timeout = seconds * 2;
	err = aspi_SendCommand(&c);
	return err == SS_COMP;
}

dword aspi_ExecSCSICommand(const dword address, int tries, const byte flags, const byte *data, const dword datalen, byte *buf, const dword buflen)
{
	dword err;
	SRB_ExecSCSICmd c;
	
	memset(&c, 0, sizeof c);
	c.SRB_Cmd = SC_EXEC_SCSI_CMD;
	c.SRB_HaId = aspi_GetHost(address);
	c.SRB_Flags = flags | SRB_EVENT_NOTIFY;
	c.SRB_Target = aspi_GetTarget(address);
	c.SRB_Lun = aspi_GetLun(address);
	c.SRB_BufLen = buflen;
	c.SRB_BufPointer = buf;
	c.SRB_SenseLen = SENSE_LEN;
	c.SRB_CDBLen = datalen;
	memcpy(c.CDBByte, data, min(sizeof c.CDBByte, datalen));
	do
	{
		err = aspi_SendCommand(&c);
		if (err != SS_COMP && (c.SRB_TargStat != STATUS_CHKCOND || (c.SenseArea[2] & 0x0F) != KEY_UNITATT))
			tries--;
		else
			tries = 0;
	} while (tries > 0);
	return err;
}

VSTRIP_DLL_API bool VSTRIP_DLL_CC aspi_GetSectorInfo(const dword address, dword *secSize, dword *maxSec)
{
	dword err;
	byte CDB[10], Cap[8];
	
	memset(CDB, 0, sizeof CDB);
	CDB[0] = SCSI_RD_CAPAC;
	err = aspi_ExecSCSICommand(address, 2, SRB_DIR_IN, CDB, sizeof CDB, Cap, sizeof Cap);
	if (err == SS_COMP)
	{
		aspi_MOVESCSIDWORD(&Cap[0], maxSec);
		aspi_MOVESCSIDWORD(&Cap[4], secSize);
		return TRUE;
	}
	return FALSE;
}

dword aspi_ExecSCSICommandAsync(SRB_ExecSCSICmd *c, const dword address, const byte flags, const byte *data, const dword datalen, byte *buf, const dword buflen, HANDLE *handle)
{
	memset(c, 0, sizeof *c);
	c->SRB_Cmd = SC_EXEC_SCSI_CMD;
	c->SRB_HaId = aspi_GetHost(address);
	c->SRB_Flags = flags | SRB_EVENT_NOTIFY;
	c->SRB_Target = aspi_GetTarget(address);
	c->SRB_Lun = aspi_GetLun(address);
	c->SRB_BufLen = buflen;
	c->SRB_BufPointer = buf;
	c->SRB_SenseLen = SENSE_LEN;
	c->SRB_CDBLen = datalen;
	memcpy(c->CDBByte, data, min(sizeof c->CDBByte, datalen));
	return aspi_SendCommandAsync(c, handle);
}

dword aspi_WaitSCSICommandAsync(SRB_ExecSCSICmd *c, int tries, HANDLE handle)
{
	dword err = aspi_WaitCommandAsync(c, handle);
	
	if (err != SS_COMP && (c->SRB_TargStat != STATUS_CHKCOND || (c->SenseArea[2] & 0x0F) != KEY_UNITATT))
	{
		tries--;
		while (tries > 0)
		{
			err = aspi_SendCommand(&c);
			if (err != SS_COMP && (c->SRB_TargStat != STATUS_CHKCOND || (c->SenseArea[2] & 0x0F) != KEY_UNITATT))
				tries--;
			else
				tries = 0;
		}
	}
	return err;
}

dword aspi_ReadSectors(const dword address, const dword cur_lba, const dword num_lba, const dword sec_size, byte *buf, tp_aspi_async asr)
{
	dword err;
	byte CDB[10];
	bool do_read_ahead = asr != NULL && num_lba * sec_size <= 65536;
	
	//	printf("aspi_ReadSectors: lba: %u num: %u\n", cur_lba, num_lba);
	memset(CDB, 0, sizeof CDB);
	CDB[0] = SCSI_READ10;
	if (asr && asr->handle && (asr->last_read_address != address ||
		asr->last_read_lba != cur_lba || asr->last_read_num_lba != num_lba))
	{ // had an (incorrect) pending read
		//		puts("throwing away incorrect pending");
		aspi_WaitSCSICommandAsync(&asr->cmd, 1, asr->handle); // ignore old result
		asr->handle = NULL;
	}
	if (!do_read_ahead || (asr && !asr->handle))
	{
		dword num = num_lba, lba = cur_lba;
		dword already_read = 0, max_transfer = 65536 / sec_size;
		
		//		puts("normal read");
		do
		{
			word this_time = (word)min(num, max_transfer);
			
			aspi_MOVESCSIDWORD(&lba, &CDB[2]);
			aspi_MOVESCSIWORD(&this_time, &CDB[7]);
			
			err = aspi_ExecSCSICommand(address, 2, SRB_DIR_IN, CDB, sizeof CDB, buf, this_time * sec_size);
			num -= this_time;
			lba += this_time;
			buf += this_time * sec_size;
		} while (err == SS_COMP && num > 0);
	}
	
	if (do_read_ahead)
	{
		if (asr->handle)
		{ // do we have a (correct) read pending?
			//			puts("waiting for pending read");
			err = aspi_WaitSCSICommandAsync(&asr->cmd, 2, asr->handle);
			if (err == SS_COMP)
				memcpy(buf, asr->buffer, num_lba * sec_size);
			asr->handle = NULL;
		}
		if (err == SS_COMP)
		{ // everything went fine, so set up the next read
			int num = min(num_lba, asr->max_lba - (cur_lba + num_lba) + 1);
			
			if (num > 0)
			{ // sectors left to read?
				word this_time = num;
				
				asr->last_read_lba = cur_lba + num_lba;
				asr->last_read_num_lba = num;
				asr->last_read_address = address;
				//				printf("setting up read ahead: lba: %u num: %u\n", asr->last_read_lba, num_lba);
				aspi_MOVESCSIDWORD(&asr->last_read_lba, &CDB[2]);
				aspi_MOVESCSIWORD(&this_time, &CDB[7]);
				aspi_ExecSCSICommandAsync(&asr->cmd, address, SRB_DIR_IN, CDB, sizeof CDB,
					asr->buffer, num * sec_size, &asr->handle);
			}
		}
	}
	return err;
}


/*	byte CDB[12];
dword already_read = 0, max_transfer = 65536 / sec_size;
dword err;

  memset(CDB, 0, sizeof CDB);
  CDB[0] = SCSI_READ12;
  do
  {
		dword this_time = (word)min(num, max_transfer);
		
		  aspi_MOVESCSIDWORD(&lba, &CDB[2]);
		  aspi_MOVESCSIDWORD(&this_time, &CDB[6]);
		  
			err = aspi_ExecSCSICommand(address, 2, SRB_DIR_IN, CDB, sizeof CDB, buf, this_time * sec_size);
			num -= this_time;
			lba += this_time;
			buf += this_time * sec_size;
			} while (err == SS_COMP && num > 0);
return err;*/
/*	byte CDB[10];
dword already_read = 0, max_transfer = 65536 / sec_size;
dword err;

  memset(CDB, 0, sizeof CDB);
  CDB[0] = SCSI_READ10;
  
	do
	{
	word this_time = (word)min(num, max_transfer);
	
	  aspi_MOVESCSIDWORD(&lba, &CDB[2]);
	  aspi_MOVESCSIWORD(&this_time, &CDB[7]);
	  
		err = aspi_ExecSCSICommand(address, 2, SRB_DIR_IN, CDB, sizeof CDB, buf, this_time * sec_size);
		num -= this_time;
		lba += this_time;
		buf += this_time * sec_size;
		} while (err == SS_COMP && num > 0);
		return err;*/
		
		
#endif // aspi_USE_ASPI