/*************************************************************************
	vStrip: Parses VOB- and IFO-Files and extracts data from them.
	Copyright (C) 1999-2002 [maven] (maven@maven.de)

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA

	vstrip.c: vob-demuxer core w/ user-function support (tabsize ?)

	references:
	- Frank A. Stephenson: "Cryptanalysis of Contents Scrambling System",	http://crypto.gq.nu
	- Andreas Bogk: "DVD-Schutz gegen Anfnger",							c't 08/00 (Magazin fr Computertechnik)
	- Bruce Schneier: "Applied Cryptography" (2nd Edition), 				ISBN 0-471-11709-9
	- Derek Fawkus: css-descramble.c, 										http://www.linuxvideo.org
	- Thomas Mirlacher: lib_ifo 											http://www.linuxvideo.org

	0.8f:
	- added DVD2AVI-project file writing support... (cmdline -2).
		- i've tested this with progressive ntsc (princess mononoke r1), normal ntsc (gokudo-kun mannyuki r1),
		  fudgy ntsc (interlaced & progressive) (ah my goddess movie r1) and pal (5th element, r2 de).
		  in all cases output was identical to dvd2avi (after lots of bugfixing). phew.
		- i write hard-coded values for the iDCT (normal MMX), Field-Flags (none). these two can be
		  changed afterwards, as they do not change the data DVD2AVI needs to frameserve.
		- thanks to tron for the good idea ;)
	- changed t_vs_streaminfo (and some other structure) to include out_lba and out_flags
	- user-functions are now called separately for each output (but at least once)
	
	0.8e2:
	- improved IFO-parsing (now _should_ have correct audio & subtitle track display), after
	  reading one of Derrow's post at http://www.ultimateboard.de
	  i want to repeat that i don't trust IFOs further than i can spit ^_^
	
	0.8e:
	- when splitting output, even the 1st file gets a number (DVD2AVI friendly)
	- end-lba set from command-line using -w {LBA}
	
	0.8d:
	- added demacrovision (enabled by default)
	- removed r1-patriot specific program_stream_2 fix (i hope the general resync takes care of this)
	- improved PROGRAM_STREAM_2 parsing a bit (no more "last_system" funkyness)
	- added info to subp & audio info (in [] -> in which program chains they are active)
		- changed SDDS to DTS
	- added (ifo) video info
		
	0.8c:
	- implemented read-ahead for aspi, now it is as fast as windows/clib read
		(subtracting the time needed for seeks on the disc for parsing the udf-filesystem)
	- fixed incorrect output to stderr (aspect ratios)
	- fixed a (STUPID!) bug in the aspi-seek routines (used offset where i should have used origin)
		this should correct strange behaviour in aspi-files after key-search etc.
		
	0.8b:
	- improved 1gb handling (using flags) to use the extent_flags as size (bit 31)
	- corrected a few exports/statics definitions
	- udf-LBAs are now returned absolute (not relative to partition start)
	- fixed _BAD_ listmode bug when using ASPI (ups... got the boolean negation wrong!)
	
	0.8a:
	- fixed command-line parsing bug
	
	0.8:
	- fiddled a bit with language mapping (thx 2 qbert, tsunami, bruce)
	- now it _really_ should compile under linux (maybe not anymore, due to aspi/udf)
	- works on non 2048 byte streams , 'e' switch (only had 1 for testing) [big hack]
	- udf-parsing (using aspi), direct dvd access (t & u switches)
	
	0.7c:
	- added ifo audio & subpicture info (lang)
	- revived linux (cmdline) support (due to Arne Zellentin (arne@unix-ag.org))
	
	0.7b:
	- incorporated a few (interface) changes by Ingo Korb (ik@akana.de)
		- removed frame-rate descriptions and display them in 2 columns in the usage screen (to fit better)
		- -o and -i now accept an optional space (and -i warns if no param given)
	- renamed ofs_packets & num_packets to start_lba & end_lba
	- made it stop after (not on) end_lba, therefore now 0xffffffff means process all (and is default)
	- fixed ifo-parsing times (hex -> dec conversion)
	- tweaked DeCSS+ Keysearch settings
		
	0.7a:
	- fixed problem with multi-cell keysearch (would never really start looking)
	- fixed problem with empty/non existent list-file
	- improved ifo-multi-angle support (with tips from fu2k (thanks!))

	0.7:
	- fixed (many) use-the-same variable twice in the IFO-code (wasn't *that* clean after all),
		which lead to crashes and unpredictable behaviour (mixture of stupidity & cut/paste)
	- added support for up to 8 outputs at the same time (not useable from cmdline)
		(didn't want to cause too much hassle to DannyDin & doom9 which different cmdline-switches)
		
	0.6i:
	- enabled user_func even on do_output
	- sti.lba etc. always set
	- fixed a bug that closed a file twice on some occasions (splitting)
	- dll support
	- cleaned up ifo a lot (newer livid version, better encapsulation)
	- delphi GUI
		
	0.6h2:
	- fixed *stupid* rearrangement of code (display WRONG PTS-times)

	0.6h:
	- updated IFO-Parser a bit, added multi-angle support (seems to work on Disney R1 now)
		- added 'j' option to select angle (set to 0 to ignore angle flag), only works when IFO-Parsing
	- fixed a problem where IFO-Parsing might have resulted in skipping alternating VOB/CELL-IDs
	- rearranged loops a little

	0.6g:
	- changed compiler to VisualC (don't know whether it still compiles under linux)
	- changed file-io to _open (using _O_SEQUENTIAL hint under Visual C)
	- added string-list to file_io (list of input-files, terminated by \0\0), as opposed to string-file
		- moved stream-list parsing to main.c (vstrip only recognizes string-list ':' and single files now)
		- made '@' optional when ".LST" extension is used
	- added PTS-Info
	- added delay info w/ respect to VIDEO_STREAM_0 (0xe0) (based on DVD2AVI)
	- added errormessage on out-of-diskspace
	- fixed wrong lba printed when using -sX (seek)
	- fixed missing errormessage when no keys (at all) were found
	- improved DeCSS+ Method to be bit-based (as in DeCSS+2pre2), which is ~70% better than before
	- correctly (hopefully) redirected output to stdout/stderr (what about usage-infos?)

	0.6f:
	- added error message to empty file (also not authenticated)

	0.6e:
	- fixed bug in IFO-parsing (didn't alloc enought mem), was introduced in 0.6d, thanks to Johan N.

	0.6d:
	- changed into a one block/call method (vs_strip_one_block), the old interface is still available, though (internal)
	- fixed bug in resync-code (was not used w/ the old default value of 0)
	- added a default value of 2048 to main.c for number of resync bytes (instead of 0)
	- added check for "invalid" program streams: PROGRAM_STREAM_2 packets that either have
		invalid packet length or a missing padding stream (depends on how you look at it), so i skip that packet
		(fixed length of 1024 bytes assumed for PCI & DSI packets), thanks to robert m. for test-data.
		this should fix "Lost Sync [@ LBA 0]" errors that some people have reported (i thought they were due to the
		clib not being able to open files under certain circumstances).

	0.6c:
	- fixed incorrect CSS-detection
	- added key probability-counting based on repetative patterns (as in DeCSS+)
		- added 'm' option to set the value of a padding-stream derived key in comparison to a normal guess (=32)
		- added 'n' option to set the number of times a key needs to repeat (=32)
		- added 'q' option to set the percentage a candidate key has to have of all keys (=90)
	- added key refinding per vob-id

	0.6b:
	- added 'l' option to print cell-list (when ifo parsing)

	0.6a:
	- fixed stupid command-line parsing bug (e.g. vstrip in.vob out.vob (which is not valid anymore))

	0.6:
	- parsing vob-/cell-id as words now (not bytes anymore)
	- completely changed argument parsing (using - for options now)
		- renamed 'r' option to 'a'
		- added output file splitting support (-$N, where N is the size in megs)
		- output-file is an option now (-oOutName)
	- ifo-parsing (-i option for filename, -p for program-chain-selection)

	0.5e:
	- improved SPLIT option

	0.5d:
	- bugfix for accidentally deleting output-files

	0.5c:
	- added only keep GOPs option (producing small VOBs)

	0.5b:
	- fixed fio_close() bug if filelen == 0

	0.5a:
	- used VobDec lfsr-reversal code
	- changed filename-# to 2-digits

	0.5:
	- now under GPL v2
	- added support for writing raw PCM-Audio files (skipping 7 byte header)
		(see vs_KEEP_PCM_IDENT_BYTES), new default behaviour
	- when splitting by CELL/VOB-ID, any "we don't know where to put it" data
		is written to a out_PRE_outext file
	- the LBA for the CELL/VOB-IDs is now printed out
	- added lba-seeking (ofs_packets), see s-option
	- added num_packets in flags
	- decryption option when key is known
	- user_functions return bool value (cont. stripping)
	- fixed stupid "eof"-bug in list-mode (when exactly hitting the buffer boundary)
		(thx QWERT (the files didn't help, though ;))

	0.4b:
	- fixed too many open files-bug

	how-to-use-in-your-own-code:
	1. initialize t_vs_data structure with input-file and flags
		 (vs_DEMUX has to be set for user_functions to be called),
		 offsets, num_packets, etc. (see vstrip.h)
	2. initialize t_vs_streamflags array for 256-streams and 256
		 substreams, setting the 'save'-member to true for the streams/
		 substreams you are interested in and setting the correct user_func
		 and user_data (if AC3 is your business, you might be interested in
		 the KEEP_AC3_IDENT_BYTES)
	3. call vs_strip(&vsd, streams, substreams);
	4. done! (look at the vstrip.h-file for more hints/definitions)
		 your user-function will be called for every block in that
		 particular stream.

	main.c is a simple example that shows how to use vstrip.c (although it does
	not use user_functions)

	if you have got questions about how to use this, or have some improvements/
	fixes to make to the code (or interface (which is quite a hack, but a lot
	better than it was before)), mail me!
*************************************************************************/

#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <math.h>
#include "s_types.h"
#include "file_io.h"
#include "in_buffer.h"
#include "vstrip.h"
#include "aspi.h"
#include "dvd2avi_plugin.h"

#define vs_VERSION 0x00086601

#ifdef vs_DECRYPT

static const byte csstab1[256]=
{
	0x33,0x73,0x3b,0x26,0x63,0x23,0x6b,0x76,0x3e,0x7e,0x36,0x2b,0x6e,0x2e,0x66,0x7b,
		0xd3,0x93,0xdb,0x06,0x43,0x03,0x4b,0x96,0xde,0x9e,0xd6,0x0b,0x4e,0x0e,0x46,0x9b,
		0x57,0x17,0x5f,0x82,0xc7,0x87,0xcf,0x12,0x5a,0x1a,0x52,0x8f,0xca,0x8a,0xc2,0x1f,
		0xd9,0x99,0xd1,0x00,0x49,0x09,0x41,0x90,0xd8,0x98,0xd0,0x01,0x48,0x08,0x40,0x91,
		0x3d,0x7d,0x35,0x24,0x6d,0x2d,0x65,0x74,0x3c,0x7c,0x34,0x25,0x6c,0x2c,0x64,0x75,
		0xdd,0x9d,0xd5,0x04,0x4d,0x0d,0x45,0x94,0xdc,0x9c,0xd4,0x05,0x4c,0x0c,0x44,0x95,
		0x59,0x19,0x51,0x80,0xc9,0x89,0xc1,0x10,0x58,0x18,0x50,0x81,0xc8,0x88,0xc0,0x11,
		0xd7,0x97,0xdf,0x02,0x47,0x07,0x4f,0x92,0xda,0x9a,0xd2,0x0f,0x4a,0x0a,0x42,0x9f,
		0x53,0x13,0x5b,0x86,0xc3,0x83,0xcb,0x16,0x5e,0x1e,0x56,0x8b,0xce,0x8e,0xc6,0x1b,
		0xb3,0xf3,0xbb,0xa6,0xe3,0xa3,0xeb,0xf6,0xbe,0xfe,0xb6,0xab,0xee,0xae,0xe6,0xfb,
		0x37,0x77,0x3f,0x22,0x67,0x27,0x6f,0x72,0x3a,0x7a,0x32,0x2f,0x6a,0x2a,0x62,0x7f,
		0xb9,0xf9,0xb1,0xa0,0xe9,0xa9,0xe1,0xf0,0xb8,0xf8,0xb0,0xa1,0xe8,0xa8,0xe0,0xf1,
		0x5d,0x1d,0x55,0x84,0xcd,0x8d,0xc5,0x14,0x5c,0x1c,0x54,0x85,0xcc,0x8c,0xc4,0x15,
		0xbd,0xfd,0xb5,0xa4,0xed,0xad,0xe5,0xf4,0xbc,0xfc,0xb4,0xa5,0xec,0xac,0xe4,0xf5,
		0x39,0x79,0x31,0x20,0x69,0x29,0x61,0x70,0x38,0x78,0x30,0x21,0x68,0x28,0x60,0x71,
		0xb7,0xf7,0xbf,0xa2,0xe7,0xa7,0xef,0xf2,0xba,0xfa,0xb2,0xaf,0xea,0xaa,0xe2,0xff
};

static bool css_tables_ready = FALSE;
static byte bit_reverse[256];

static void vs_init_css_tables(void)
{
	if (!css_tables_ready)
	{
		dword i;
		byte j, b, c;
		
		for (i = 0; i < 256; i++)
		{
			for (j = 0, b = 0, c = i; j < 8; j++, c >>= 1)
				b |= (c & 1) << (7 - j);
			bit_reverse[i] = b;
		}
		css_tables_ready = TRUE;
	}
}

// try to recover a set of lfsrs from the stream of length bytes, but rewind lfsrs reverse bytes
static int vs_recover_lfsrs(const byte *stream, const dword length, const dword reverse, dword *plfsr1, dword *plfsr2)
{
	dword ntry, i;
	dword lfsr1, lfsr2, guess;
	dword candidate, val;
	byte o_lfsr1, o_lfsr2;
	
	*plfsr1 = *plfsr2 = 0;
	
	if (length < CSS_BYTES_NEEDED)
		return FALSE;
	
	for (ntry = 0; ntry < 0x40000; ntry++)
	{
		val = (ntry & 0x20000) >> 17; // don't know whether there was a carry from the last val
		lfsr1 = ntry & 0x1ffff;	// cannot rely on the fact that bit 9 is set *somewhere* in the key-stream
		lfsr2 = 0; // not needed
		
		// iterate cipher 4 times to reconstruct LFSR2
		for (i = 0; i < 4; i++)
		{
			// advance LFSR1 normally
			o_lfsr1 = ((lfsr1 >> 14) & 7) ^ lfsr1;
			o_lfsr1 ^= (o_lfsr1 << 3) ^ (o_lfsr1 << 6);
			lfsr1 = (lfsr1 >> 8) ^ (o_lfsr1 << 9);
			o_lfsr1 = ~o_lfsr1;
			
			// deduce guess
			guess = stream[i];
			if (val)
				guess = (guess + 0xff) & 0xff; // remove carry
			if (guess < o_lfsr1)
				guess += 0x100;
			guess -= o_lfsr1;
			val += o_lfsr1 + guess;
			// feed / advance lfsr2 / val
			lfsr2 = (lfsr2 >> 8) ^ (guess << 17);
			val >>= 8;
		}
		candidate = lfsr2;
		
		// keep on iterating to validate candidate key
		for (; i < length; i++)
		{
			o_lfsr1 = ((lfsr1 >> 14) & 7) ^ lfsr1;
			o_lfsr1 ^= (o_lfsr1 << 3) ^ (o_lfsr1 << 6);
			lfsr1 = (lfsr1 >> 8) ^ (o_lfsr1 << 9);
			o_lfsr1 = ~o_lfsr1;
			
			o_lfsr2 = (lfsr2 >> 12) ^ (lfsr2 >> 4) ^	(lfsr2 >> 3) ^ lfsr2;
			lfsr2 = (lfsr2 >> 8) ^ (o_lfsr2 << 17);
			
			val += o_lfsr1 + o_lfsr2;
			if ((val & 0xff) != stream[i])
				break;
			val >>= 8;
		}
		
		if (i == length)
		{
			lfsr1 = ntry & 0x1ffff;
			lfsr2 = candidate;
			// Do backwards steps of iterating both lfsrs to deduce initial state, replaced brute force by code from VobDec 0.3s
			for (i = 0; i < 4 + reverse; i++)
			{
				dword temp0, temp1;
				
				if (i >= 4)
					lfsr1 = ((lfsr1 << 8) ^ ((((lfsr1 >> 3) ^ lfsr1) >> 6) & 0xff)) & ((1 << 17) - 1); // reverse lfsr1
				
				temp0 = ((lfsr2 >> 17) ^ (lfsr2 >> 4)) & 0xff;
				temp1 = (lfsr2 << 5) | (temp0 >> 3);
				temp1 = ((temp1 >> 1) ^ temp1) & 0xff;
				lfsr2 = ((lfsr2 << 8) | ((((((temp1 >> 2) ^ temp1) >> 1) ^ temp1) >> 3) ^ temp1 ^ temp0)) & ((1 << 25) - 1);
			}
			if ((lfsr1 & 0x100) != 0 && (lfsr2 & 0x200000) != 0)
			{
				//				printf("lfrs1: 0x%08x lfrs2: 0x%08x val: %01u\n", lfsr1, lfsr2, ntry >> 17);
				*plfsr1 = lfsr1;
				*plfsr2 = lfsr2;
				return TRUE;
			}
		}
	}
	return FALSE;
}

static bool vs_initial_key_add(const byte *sec, const dword lfsr1, const dword lfsr2, const dword count, tp_vs_keysearch ks)
{ // returns whether we are over the treshold (--> valid key)
	dword flfsr2;
	tp_vs_4key key;
	t_vs_key_union ku;
	byte k0;
	
#define SALTED(i) sec[0x54 + (i)]
	k0 = bit_reverse[lfsr1 >> 9] ^ SALTED(0);
	ku.b[0] = bit_reverse[lfsr1 & 0xff] ^ SALTED(1);
	
	flfsr2 = bit_reverse[lfsr2 >> 17] | (bit_reverse[(lfsr2 >> 9) & 0xff] << 8)
		| (bit_reverse[(lfsr2 >> 1) & 0xff] << 16) | ((lfsr2 & 1) << 24);
	
	ku.b[1] = (((flfsr2 & 0x1f0) >> 1) | (flfsr2 & 7)) ^ SALTED(2);
	ku.b[2] = ((flfsr2 >> 9) & 0xff) ^ SALTED(3);
	ku.b[3] = (flfsr2 >> 17) ^ SALTED(4);
#undef SALTED
	
	//	printf("lfrs1: 0x%08x lfrs2: 0x%08x -> key: 0x%02X%02X%02X%02X%02X\n", lfsr1, lfsr2, k0, ku.b[0], ku.b[1], ku.b[2], ku.b[3]);
	
	ks->total_keys += count;
	if (!ks->keys)
		ks->keys = calloc(256, sizeof *ks->keys);
	for (key = ks->keys[k0]; key; key = (tp_vs_4key)key->next)
		if (key->key.d == ku.d)
		{
			key->found += count;
			break;
		}
		if (!key)
		{
			key = calloc(1, sizeof *key);
			key->key.d = ku.d;
			key->found = count;
			key->next = ks->keys[k0];
			ks->keys[k0] = key;
		}
		if (key->found >= ks->same_guess && (key->found * 100) / ks->total_keys >= ks->pc_guess)
		{
			ks->k0 = k0;
			ks->key = key;
			return TRUE;
		}
		else
			return FALSE;
}

static bool vs_find_vulnerable_sector(byte *data, tp_vs_streaminfo const si, void *user_data)
{ // user_data points to key
	if (si->encrypted && si->idx <= CSS_OFFSET && si->idx + si->length >= CSS_OFFSET)
	{
		tp_vs_keysearch ks = (tp_vs_keysearch)user_data;
		dword left = fio_SECTOR_SIZE - (si->idx + si->length);
		dword i;
		
		if (left >= CSS_BYTES_NEEDED && left <= CSS_MAX_ASSUME_PADDING)
		{ // unfilled space at the end of the sector -> needs padding stream
			dword lfsr1, lfsr2;
			byte pad_header[6] = {0x00,0x00,0x01,PADDING_STREAM,0x00,0x00};
			byte stream[CSS_MAX_ASSUME_PADDING];
			byte *b = data + si->length;
			
			pad_header[4] = (left - sizeof pad_header) >> 8;
			pad_header[5] = (left - sizeof pad_header) & 0xff; // length of padding stream
			for (i = 0; i < sizeof pad_header; i++)
				stream[i] = csstab1[*b++] ^ pad_header[i];
			for (; i < left; i++)
				stream[i] = csstab1[*b++] ^ 0xff; // padding bytes
			
			if (vs_recover_lfsrs(stream, left, (si->idx + si->length) - CSS_OFFSET, &lfsr1, &lfsr2))
				if (vs_initial_key_add(data - si->idx, lfsr1, lfsr2, ks->pad_guess, ks))
					return FALSE; // and stop looking
		}
		else
		{ // search for repetative data and hope it behaves the same way for just a little longer
			bool bits[CSS_OFFSET * 8];
			byte *sec_start = data - si->idx;
			dword repeat_length = 0;
			dword repeat_period = 0;
			dword j;
			
			// copy & reverse into bit array (so we can search forward)
			for (i = 0; i < CSS_OFFSET; i++)
				for (j = 0; j < 8; j++)
					bits[CSS_OFFSET * 8 - 1 - (i * 8 + j)] = (sec_start[i] & (0x80 >> j)) != 0;
				
				for (i = CSS_MIN_GUESS_PERIOD * 8; i < CSS_MAX_GUESS_PERIOD * 8; i++)
				{
					for(j = i; j < CSS_OFFSET * 8; j++)
						if (bits[j] != bits[j % i])
							break;
						if (j > repeat_length && (j / i) >= CSS_MIN_GUESS_REPEAT)
						{
							repeat_length = j;
							repeat_period = i;
						}
				}
				if (repeat_length > 0)
				{
					dword lfsr1, lfsr2;
					byte stream[CSS_BYTES_NEEDED];
					
					for (i = 0; i < CSS_BYTES_NEEDED; i++)
					{
						stream[i] = 0;
						for (j = 0; j < 8; j++)
							stream[i] |= bits[repeat_period - 1 - ((i * 8 + j) % repeat_period)] << (7 - j);
						stream[i] ^= csstab1[sec_start[CSS_OFFSET + i]];
					}
					
					if (vs_recover_lfsrs(stream, CSS_BYTES_NEEDED, 0, &lfsr1, &lfsr2))
						if (vs_initial_key_add(data - si->idx, lfsr1, lfsr2, 1, ks))
							return FALSE; // and stop looking
				}
		}
	}
	return TRUE;
}

static bool vs_search_key(const tp_vs_data vsd, const tp_inb_buffer bb, const dword vob_id, const dword cell_id)
{ // call vs_strip again with a user-function for looking for vulnerable sectors, starting at the current ofs
	dword 						i;
	t_vs_data 				c_vsd;
	t_vs_streamflags	c_streams[256], c_substreams[256];
	t_vs_errorcode		ec;
	t_vs_keysearch		ks;
	t_vs_vobcellid		vcid;
	t_vs_flags				flag_mask = vs_NO_VOB;
	
	if ((vsd->flags & vs_PRINT_INFO) != 0)
		fprintf(vs_STDOUT, "Encountered encrypted sector, attempting key recovery [@LBA %u]\n", bb->lba);
	memset(&c_vsd, 0, sizeof c_vsd);
	c_vsd.infile = vsd->infile;
	c_vsd.num_outputs = 0;
	c_vsd.max_sync_bytes = vsd->max_sync_bytes;
#ifdef aspi_USE_ASPI
	flag_mask |= vs_USE_ASPI | vs_PREFER_ASPI | vs_SUPPORT_1GB;
#endif
	c_vsd.flags = (vs_IGNORE_UNKNOWN_CELLS | vs_DONT_CRACK) | (vsd->flags & flag_mask);
	c_vsd.aspectratio = c_vsd.framerate = -1;
	c_vsd.start_lba = bb->lba; // this is the first encrypted block
	c_vsd.end_lba = 0xffffffff; // allow to seach the whole file for vulnerable block
	memset(&ks, 0, sizeof ks);
	ks.same_guess = vsd->same_guess;
	ks.pad_guess = vsd->pad_guess;
	ks.pc_guess = vsd->pc_guess;
	for (i = 0; i < 256; i++)
	{
		c_streams[i].remap_to = i;
		c_streams[i].save = TRUE;
		c_streams[i].user_func = vs_find_vulnerable_sector;
		c_streams[i].user_data = &ks;
		c_substreams[i].remap_to = i;
		c_substreams[i].save = TRUE;
		c_substreams[i].user_func = vs_find_vulnerable_sector;
		c_substreams[i].user_data = &ks;
	}
	if ((vob_id != -1) || (cell_id != -1))
	{
		vcid.vob_id = vob_id;
		vcid.cell_id = cell_id;
		ec = vs_strip(&c_vsd, c_streams, c_substreams, 1, &vcid); // only check inside vob-id
	}
	else
		ec = vs_strip(&c_vsd, c_streams, c_substreams, 0, NULL); // we can use all vob-/cell-ids to deduce the key
	if (ec == vse_USER_FUNC_EXIT)
	{
		vsd->key[0] = ks.k0;
		vsd->key[1] = ks.key->key.b[0];
		vsd->key[2] = ks.key->key.b[1];
		vsd->key[3] = ks.key->key.b[2];
		vsd->key[4] = ks.key->key.b[3];
		if ((vsd->flags & vs_PRINT_INFO) != 0)
			fprintf(vs_STDOUT, "  Deduced key: 0x%02X%02X%02X%02X%02X (%u/%u vkey(s))\n",
			vsd->key[0], vsd->key[1], vsd->key[2], vsd->key[3], vsd->key[4], ks.key->found, ks.total_keys);
	}
	else if (ks.keys)
	{ // look for the one that occured most often
		dword maxf = 0;
		tp_vs_4key maxk = NULL, k, k2;
		byte maxi = 0;
		
		for (i = 0; i < 256; i++)
		{
			for (k = ks.keys[i]; k; k = (tp_vs_4key)k->next)
				if (k->found > maxf)
				{
					maxk = k;
					maxf = k->found;
					maxi = i;
				}
		}
		if (maxf >= 1)
		{
			vsd->key[0] = maxi;
			vsd->key[1] = maxk->key.b[0];
			vsd->key[2] = maxk->key.b[1];
			vsd->key[3] = maxk->key.b[2];
			vsd->key[4] = maxk->key.b[3];
			if ((vsd->flags & vs_PRINT_INFO) != 0)
				fprintf(vs_STDOUT, "  Guessed key: 0x%02X%02X%02X%02X%02X (%u/%u vkey(s))\n",
				vsd->key[0], vsd->key[1], vsd->key[2], vsd->key[3], vsd->key[4], maxf, ks.total_keys);
			ec = vse_USER_FUNC_EXIT; // so can continue
		}
		
		// free mem from keys
		for (i = 0; i < 256; i++)
			for (k = ks.keys[i]; k; k = k2)
			{
				k2 = (tp_vs_4key)k->next;
				free(k);
			}
			free(ks.keys);
	}
	if ((vsd->flags & vs_PRINT_ERROR) != 0 && ec != vse_USER_FUNC_EXIT)
		fputs("\n* Unable to recover key\n", vs_STDERR);
	return ec == vse_USER_FUNC_EXIT;
}

static void vs_decss(const tp_inb_buffer bb, const byte key[5])
{
	if (bb->encrypted && bb->num_bytes > 128)
	{
		byte *sec = bb->bytes;
		dword lfsr1, lfsr2, val;
		byte o_lfsr1, o_lfsr2;
		byte *end = sec + bb->num_bytes;
		
		bb->encrypted = FALSE;
		bb->bytes[20] &= ~inb_ENCRYPTED_MASK; // set encrypted flag to false
		
#define SALTED(i) (key[i] ^ sec[0x54 + (i)])
		lfsr1 = (bit_reverse[SALTED(0)] << 9) | 0x100 | (bit_reverse[SALTED(1)]);
		lfsr2 = ((SALTED(4) << 17) | (SALTED(3) << 9) | (SALTED(2) << 1)) + 8 - (SALTED(2) & 7);
		lfsr2 = (bit_reverse[lfsr2 & 0xff] << 17) | (bit_reverse[(lfsr2 >> 8) & 0xff] << 9)
			| (bit_reverse[(lfsr2 >> 16) & 0xff] << 1) | (lfsr2 >> 24);
#undef SALTED
		
		//		printf("%i: key: 0x%02X%02X%02X%02X%02X\n", bb->lba, SALTED(0), SALTED(1), SALTED(2), SALTED(3), SALTED(4));
		sec += 128;
		val = 0;
		while (sec != end)
		{
			o_lfsr1 = ((lfsr1 >> 14) & 7) ^ lfsr1;
			o_lfsr1 ^= (o_lfsr1 << 3) ^ (o_lfsr1 << 6);
			lfsr1 = (lfsr1 >> 8) ^ (o_lfsr1 << 9);
			
			o_lfsr2 = (lfsr2 >> 12) ^ (lfsr2 >> 4) ^	(lfsr2 >> 3) ^ lfsr2;
			lfsr2 = (lfsr2 >> 8) ^ (o_lfsr2 << 17);
			
			val += (byte)~o_lfsr1 + o_lfsr2;
			*sec++ = csstab1[*sec] ^ (val & 0xff);
			val >>= 8;
		}
	}
}
#endif

VSTRIP_DLL_API const char* VSTRIP_DLL_CC vs_get_stream_name(const dword id)
{
	static char buffer[256];
	
	switch (id)
	{
	case PROGRAM_STREAM_MAP:
		return "Program Map";
	case PRIVATE_STREAM_1:
		return "Private 1";
	case PRIVATE_STREAM_2:
		return "Private 2";
	case ECM_STREAM:
		return "ECM";
	case EMM_STREAM:
		return "EMM";
	case PROGRAM_STREAM_DIRECTORY:
		return "Program Directory";
	case DSMCC_STREAM:
		return "DSMCC";
	case ITUTRECH222TYPEE_STREAM:
		return "ITU-T Rec. H.222.1 type E";
	case PADDING_STREAM:
		return "Padding";
	default:
		if ((id >= VIDEO_STREAM) && (id <= MAX_VIDEO_STREAM))
			sprintf(buffer, "Video %u", id - VIDEO_STREAM);
		else if ((id >= AUDIO_STREAM) && (id <= MAX_AUDIO_STREAM))
			sprintf(buffer, "MPEG1 Audio %u", id - AUDIO_STREAM);
		else
			strcpy(buffer, "other");
		return buffer;
	}
}

VSTRIP_DLL_API const char* VSTRIP_DLL_CC vs_get_framerate(const dword id)
{
	switch (id)
	{
	case 1:
		return "[1] 23.976 (24000/1001) fps";
	case 2:
		return "[2] 24 fps";
	case 3:
		return "[3] 25 fps";
	case 4:
		return "[4] 29.97 (30000/1001) fps";
	case 5:
		return "[5] 30 fps";
	case 6:
		return "[6] 50 fps";
	case 7:
		return "[7] 59.94 (60000/1001) fps";
	case 8:
		return "[8] 60 fps";
	default:
		return "[?] unknown frame-rate";
	}
}

VSTRIP_DLL_API const char* VSTRIP_DLL_CC vs_get_aspectratio(const dword id)
{
	switch (id)
	{
	case 1:
		return "[1] square pixels";
	case 2:
		return "[2] 4:3 display";
	case 3:
		return "[3] 16:9 display";
	case 4:
		return "[4] 2.21:1 display";
	default:
		return "[?] unknown aspect-ratio";
	}
}

VSTRIP_DLL_API const char* VSTRIP_DLL_CC vs_get_time(const double time)
{
	static char buffer[64];
	double mt = time + ((time < 0.0) ? -0.5 : 0.5);
	double amt = fabs(mt);
	
	if (mt > 0.0)
		sprintf(buffer, "%i:%02i:%02i.%03i", (int)(amt / 3600000.0), (int)(fmod(amt, 3600000.0) / 60000.0), (int)(fmod(amt, 60000.0) / 1000.0), (int)fmod(amt, 1000.0));
	else
		sprintf(buffer, "-%i:%02i:%02i.%03i", (int)(amt / 3600000.0), (int)(fmod(amt, 3600000.0) / 60000.0), (int)(fmod(amt, 60000.0) / 1000.0), (int)fmod(amt, 1000.0));
	return buffer;
}

static void print_streaminfo(const bool print, const dword lba, const tp_vs_streamflags streams, const tp_vs_streamflags substreams,
							 const dword stream_id, const dword substream_id)
{
	if ((!streams[stream_id].found) || ((stream_id == PRIVATE_STREAM_1) && (!substreams[substream_id].found)))
	{
		if (print)
		{
			static const char vs_found_save[2][7] = {"Found", "Saving"};
			double pts = 0.0;
			
			if (stream_id == PRIVATE_STREAM_1)
			{
				fprintf(vs_STDOUT, "%s 0x%02X = %s, sub 0x%02X", vs_found_save[substreams[substream_id].save != 0x00], stream_id, vs_get_stream_name(stream_id), substream_id);
				substreams[substream_id].found = TRUE;
				if ((substreams[substream_id].save != 0x00) && (substreams[substream_id].remap_to != substream_id))
					fprintf(vs_STDOUT, " -> 0x%02X", substreams[substream_id].remap_to);
				pts = substreams[substream_id].pts;
			}
			else
			{
				fprintf(vs_STDOUT, "%s 0x%02X = %s", vs_found_save[streams[stream_id].save != 0x00], stream_id, vs_get_stream_name(stream_id));
				if ((streams[stream_id].save != 0x00) && (streams[stream_id].remap_to != stream_id))
					fprintf(vs_STDOUT, " -> 0x%02X", streams[stream_id].remap_to);
				pts = streams[stream_id].pts;
			}
			fprintf(vs_STDOUT, " [");
			if (pts > 0.0)
				fprintf(vs_STDOUT, "PTS %s ", vs_get_time(pts));
			fprintf(vs_STDOUT, "@LBA %u]\n", lba);
		}
		streams[stream_id].found = TRUE;
	}
}

static void vs_print_summary(const dword pack_packets, const dword system_packets, const tp_vs_streamflags streams, const tp_vs_streamflags substreams)
{
	dword i, j;
	
	fputs("\nSummary:\n", vs_STDOUT);
	fprintf(vs_STDOUT, "MPEG Packs = %u\n", pack_packets);
	if (system_packets)
		fprintf(vs_STDOUT, "System headers = %u\n", system_packets);
	for (i = 0; i < 256; i++)
	{
		if (streams[i].packets)
		{
			fprintf(vs_STDOUT, "%s packets = %u, total bytes = %u\n", vs_get_stream_name(i), streams[i].packets, streams[i].bytes);
			if (i == PRIVATE_STREAM_1)
			{
				for (j = 0; j < 256; j++)
				{
					if (substreams[j].packets)
					{
						fprintf(vs_STDOUT, "  Sub 0x%02X packets = %u, total bytes = %u", j, substreams[j].packets, substreams[j].bytes);
						if (streams[VIDEO_STREAM].pts != 0.0 && substreams[j].pts != 0.0 && streams[VIDEO_STREAM].pts != substreams[j].pts)
							fprintf(vs_STDOUT, " (delay %s)\n", vs_get_time(substreams[j].pts - streams[VIDEO_STREAM].pts));
						else
							fputs("\n", vs_STDOUT);
					}
				}
			}
		}
	}
}

/*void vs_log(char *str)
{
FILE *f = fopen("c:\\_vs_log.txt", "at");
if (f)
{
fputs(str, f);
fclose(f);
}
}*/

static bool vs_change_vobcells(const dword vob_id, const dword cell_id, const dword num_idl, const tp_vs_vobcellid idl, tp_vs_data vsd)
{ // returns save status (of this vob- and cell-id)
	bool save_status = TRUE;
	dword i;
	
	// ---- check whether we want this one ----
	if (num_idl > 0 && idl)
	{
		
		for (i = 0; i < num_idl; i++)
			if ((idl[i].vob_id == vob_id && idl[i].cell_id == cell_id) || (idl[i].vob_id == vob_id && idl[i].cell_id == 0xff) || (idl[i].vob_id == 0xffff && idl[i].cell_id == cell_id))
				break;
			if (i >= num_idl)
				save_status = FALSE; // not found -> we don't want it
	}
	if ((vsd->flags & vs_PRINT_INFO) != 0)
	{
		static const char vs_found_skip[2][9] = {"Skipping", "Found"};
		
		fprintf(vs_STDOUT, "  %s VOB-ID: %02u/CELL-ID: %02u [@LBA %u]\n", vs_found_skip[save_status], vob_id, cell_id, vsd->_in.buffer->lba);
	}
	
	for (i = 0; i < vsd->num_outputs; i++)
		if ((vsd->outputs[i].flags & (vso_SPLIT_VOBID | vso_SPLIT_CELLID)) != 0 && ((vsd->outputs[i].flags & vso_SPLIT_CELLID) != 0 || vsd->_in.sti.vob_id != vob_id))
		{
			if (vsd->_in.buffer->outfp[i])
			{ // close old-file
				fio_close(vsd->_in.buffer->outfp[i]);
				vsd->_in.buffer->outfp[i] = NULL;
			}
			if (save_status)
			{
				char fname[256];
				
				if ((vsd->outputs[i].flags & vso_SPLIT_CELLID) != 0)
					sprintf(fname, "%s_V%02uC%02u%s", vsd->_in.outfileprep[i], vob_id, cell_id, vsd->_in.outfileext[i]);
				else
					sprintf(fname, "%s_V%02uC__%s", vsd->_in.outfileprep[i], vob_id, vsd->_in.outfileext[i]);
				vsd->_in.buffer->outfp[i] = fio_open(fname, fio_WRITEABLE | ((((vsd->_in.opened[vob_id * MAX_CELLIDS + cell_id].opened & (1 << i)) == (1 << i)) || (vsd->outputs[i].flags & vso_APPEND) != 0) ? fio_APPEND : 0), vsd->outputs[i].split_output);
				vsd->_in.opened[vob_id * MAX_CELLIDS + cell_id].opened |= 1 << i;
			}
		} 
		vsd->_in.sti.vob_id = vob_id;
		vsd->_in.sti.cell_id = cell_id;
		return save_status;
}

VSTRIP_DLL_API t_vs_errorcode VSTRIP_DLL_CC vs_init(tp_vs_data vsd, t_vs_streamflags streams[256], t_vs_streamflags substreams[256])
{
	dword i;
	
	vsd->_in.init_ok = FALSE;
	vsd->_in.pack_packets = vsd->_in.system_packets = 0;
	vsd->_in.opened = NULL;
	vsd->_in.dvd2avi = NULL;
	vsd->_in.dvd2avi_idx = -1;
	vsd->_in.did_video_info = FALSE;
	vsd->_in.save_cell = (vsd->flags & vs_IGNORE_UNKNOWN_CELLS) == 0;
	
	// init some often used flags
	vsd->_in.framerate = vsd->_in.aspectratio = -1;
	if (vsd->framerate != -1)
		vsd->_in.framerate = vsd->framerate % MAX_FRAMERATE;
	if (vsd->aspectratio != -1)
		vsd->_in.aspectratio = (vsd->aspectratio % MAX_ASPECTRATIO) << 4;
	vsd->_in.did_video_info = (vsd->flags & vs_PRINT_INFO) == 0;
	
#ifdef vs_DECRYPT
	vsd->_in.vob_key = (vsd->flags & vs_CRACK_EACH_VOB_ID) != 0;
	vsd->_in.cell_key = (vsd->flags & vs_CRACK_EACH_CELL_ID) != 0;
	vsd->_in.valid_key = vsd->key[0] != 0 || vsd->key[1] != 0 || vsd->key[2] != 0 || vsd->key[3] != 0 || vsd->key[4] != 0;
	vs_init_css_tables(); // so we have the tables ready for the decrypting
#endif
	
	// init stream-info
	vsd->_in.sti.stream_id = vsd->_in.sti.substream_id = 0;
	vsd->_in.sti.cell_id = vsd->_in.sti.vob_id = -1;
	vsd->_in.sti.out_flags = NULL;
	vsd->_in.sti.out_lba = 0;
	vsd->_in.sti.out_file_num = 0;
	vsd->_in.sti.encrypted = FALSE;
	for (i = 0; i < 256; i++)
	{
		streams[i].packets = substreams[i].packets = 0;
		streams[i].bytes = substreams[i].bytes = 0;
		streams[i].found = substreams[i].found = FALSE;
		streams[i].pts = substreams[i].pts = 0.0;
	}
	i = 0;
#ifdef aspi_USE_ASPI
	if ((vsd->flags & vs_USE_ASPI) != 0 && !aspi_Init())
		vsd->flags = vsd->flags & ~vs_USE_ASPI;
	if ((vsd->flags & vs_USE_ASPI) != 0)
		i |= fio_USE_ASPI;
	if ((vsd->flags & vs_PREFER_ASPI) != 0)
		i |= fio_PREFER_ASPI;
	if ((vsd->flags & vs_SUPPORT_1GB) != 0)
		i |= fio_SUPPORT_1GB;
#endif
	vsd->_in.buffer = inb_init(vsd->infile, i, (bool)((vsd->flags & vs_NO_VOB) == 0));
	if (!vsd->_in.buffer || (vsd->_in.buffer && vsd->_in.buffer->status != inb_OK))
	{
		if (vsd->_in.buffer)
		{
			if ((vsd->flags & vs_PRINT_ERROR) != 0)
#ifdef vs_DECRYPT
				fprintf(vs_STDERR, "* Unable to read from \"%s\". Maybe not authenticated?\n", vsd->infile);
#else
			fprintf(vs_STDERR, "* Unable to read from \"%s\"\n", vsd->infile);
#endif
			inb_done(vsd->_in.buffer);
		}
		else if ((vsd->flags & vs_PRINT_ERROR) != 0)
			fprintf(vs_STDERR, "* Unable to open \"%s\" for input\n", vsd->infile);
		return vse_CANT_OPEN_INPUT;
	}
	
	vsd->_in.only_gop_mask = 0xff; // assume KeepOnlyGOP is false for all
	// init output-files
	for (i = 0; i < vsd->num_outputs; i++)
		if (vsd->outputs[i].outfile[0])
		{
			tp_fio_file outfp = NULL;
			char *outman;
			char outname[256];
			
			if (vsd->_in.dvd2avi == NULL && (vsd->outputs[i].flags & vso_DVD2AVI) != 0 &&
				(vsd->outputs[i].flags & vso_ONLY_KEEP_GOPS) == 0 && (streams[0xe0].save & (1 << i)) != 0)
			{
				vsd->_in.dvd2avi = dvd2avi_init();
				vsd->_in.dvd2avi_idx = i;
				streams[0xe0].user_func = dvd2avi_process;
				streams[0xe0].user_data = vsd->_in.dvd2avi;
			}
			if ((vsd->outputs[i].flags & vso_ONLY_KEEP_GOPS) != 0)
				vsd->_in.only_gop_mask &= ~(1 << i);
			if ((vsd->outputs[i].flags & (vso_SPLIT_CELLID | vso_SPLIT_VOBID)) == 0) // don't split by cell- or vob-ids
				outman = vsd->outputs[i].outfile;
			else
			{ // prepare the filename parts for allocation by vob-/cell-ids
				strcpy(vsd->_in.outfileprep[i], vsd->outputs[i].outfile);
				strcpy(vsd->_in.outfileext[i], ".");
				outman = strrchr(vsd->_in.outfileprep[i], '.');
				if (outman)
				{
					strcpy(vsd->_in.outfileext[i], outman);
					*outman = 0;
				}
				sprintf(outname, "%s_V__C__%s", vsd->_in.outfileprep[i], vsd->_in.outfileext[i]);
				outman = outname;
			}
			outfp = fio_open(outman, fio_WRITEABLE | (((vsd->outputs[i].flags & vso_APPEND) != 0) ? fio_APPEND : 0), vsd->outputs[i].split_output);
			if (!outfp)
			{
				int j;
				
				for (j = i - 1; j >= 0; j--)
					if (vsd->_in.buffer->outfp[j])
					{
						fio_close(vsd->_in.buffer->outfp[j]);
						vsd->_in.buffer->outfp[j] = NULL;
					}
					if ((vsd->flags & vs_PRINT_ERROR) != 0)
						fprintf(vs_STDERR, "* Unable to open \"%s\" for output\n", outman);
					if (vsd->_in.buffer)
						inb_done(vsd->_in.buffer);
					return vse_CANT_CREATE_OUTPUT;
			}
			else
				vsd->_in.buffer->outfp[i] = outfp;
		}
	vsd->_in.opened = calloc(MAX_VOBIDS, MAX_CELLIDS * sizeof *vsd->_in.opened);
	if (vsd->start_lba > 0)
		inb_skip(vsd->start_lba, vsd->_in.buffer);
	vsd->_in.init_ok = TRUE;
	return vse_OK;
}

static void vs_close_all(tp_vs_data vsd)
{
	if (vsd->_in.init_ok)
	{
		if (vsd->_in.buffer)
		{
			dword i;
			
			for (i = 0; i < vsd->num_outputs; i++)
			{
				fio_close(vsd->_in.buffer->outfp[i]);
				vsd->_in.buffer->outfp[i] = NULL;
				if (vsd->_in.dvd2avi && i == vsd->_in.dvd2avi_idx)
				{
					dvd2avi_write_free(vsd->outputs[i].outfile, (t_dvd2avi*)vsd->_in.dvd2avi);
					vsd->_in.dvd2avi = NULL;
				}
			}
		}
		if (vsd->_in.opened)
		{
			free(vsd->_in.opened);
			vsd->_in.opened = NULL;
		}
		inb_done(vsd->_in.buffer);
		vsd->_in.buffer = NULL;
		vsd->_in.init_ok = FALSE;
	}
}

VSTRIP_DLL_API t_vs_errorcode VSTRIP_DLL_CC vs_done(tp_vs_data vsd, t_vs_streamflags streams[256], t_vs_streamflags substreams[256])
{
	if (vsd->_in.init_ok)
	{
		if ((vsd->flags & vs_PRINT_SUMMARY) != 0)
			vs_print_summary(vsd->_in.pack_packets, vsd->_in.system_packets, streams, substreams);
		vs_close_all(vsd);
#ifdef aspi_USE_ASPI
		if ((vsd->flags & vs_USE_ASPI) != 0)
			aspi_Done();
#endif
	}
	return vse_OK;
}

VSTRIP_DLL_API t_vs_errorcode VSTRIP_DLL_CC vs_strip_one_block(tp_vs_data vsd, t_vs_streamflags streams[256], t_vs_streamflags substreams[256], dword num_idl, tp_vs_vobcellid idl)
{
	dword i, j, packet_length, header_length;
	bool	user_func_stay = TRUE;
	byte save_this_one = 0x00;
	
	if (!vsd->_in.init_ok)
		return vse_INIT_FAILED;
	if (vsd->_in.buffer->idx >= CSS_OFFSET && vsd->_in.buffer->encrypted)
		inb_fresh(vsd->_in.buffer); // skip the rest of this one because it's encrypted (only encrypted if vob)
	
	i = inb_get_bytes(4, vsd->_in.buffer);
	switch (i)
	{
	case PACK_START_CODE:
		vsd->_in.pack_packets++;
		inb_get_bytes(9, vsd->_in.buffer);
		j = inb_get_bytes(1, vsd->_in.buffer) & 7;
		inb_get_bytes(j, vsd->_in.buffer);
		break;
	case SYSTEM_HEADER_START_CODE:
		vsd->_in.system_packets++;
		vsd->_in.buffer->save = 0x00;
		for (j = 0; j < vsd->num_outputs; j++)
			vsd->_in.buffer->save |= ((vsd->outputs[j].flags & vso_DEMUX) == 0 && vsd->_in.save_cell) << j;
		vsd->_in.buffer->has_system = TRUE;
		inb_get_bytes(8, vsd->_in.buffer);
		while ((inb_peek_byte(0, vsd->_in.buffer) & 0xf0) != 0)
			inb_get_bytes(3, vsd->_in.buffer);
		break;
	case MPEG_PROGRAM_END_CODE:
		break;
	default:
		if ((i >> 8) != PACKET_START_CODE_PREFIX)
		{
			dword ctr;
			
			if (vsd->max_sync_bytes > 0)
			{
				i = inb_get_bytes(3, vsd->_in.buffer);
				for (ctr = 0; ctr < vsd->max_sync_bytes && i != PACKET_START_CODE_PREFIX; ctr++)
					i = ((i << 8) & 0xffffff) | inb_get_bytes(1, vsd->_in.buffer);
			}
			if (ctr >= vsd->max_sync_bytes)
			{ // didn't make it
				if ((vsd->flags & vs_PRINT_ERROR) != 0)
					fprintf(vs_STDERR, "\n* Lost sync [@LBA %u], was expecting PACKET_START_CODE_PREFIX!\n", vsd->_in.buffer->lba);
				vs_close_all(vsd);
				return vse_LOST_SYNC;
			}
			else
			{
				i = ((i << 8) & 0xffffff) | inb_get_bytes(1, vsd->_in.buffer);
				if ((vsd->flags & vs_PRINT_ERROR) != 0)
					fprintf(vs_STDERR, "Skipped %u bytes for resync [@LBA %u]\n", ctr, vsd->_in.buffer->lba);
			}
		}
		// we are sync'ed
		vsd->_in.sti.stream_id = i & 0x000000FF;
		vsd->_in.sti.substream_id = 0;
		if (vsd->_in.sti.stream_id == PRIVATE_STREAM_1)
		{ // let's have a peek at the substream-id
			byte header_skip;
			
			header_skip = inb_peek_byte(4, vsd->_in.buffer);
			vsd->_in.sti.substream_id = inb_peek_byte(5 + header_skip, vsd->_in.buffer);
			streams[vsd->_in.sti.stream_id].packets++;
			vsd->_in.cur_stream = &substreams[vsd->_in.sti.substream_id];
		}
		else
		{
			vsd->_in.cur_stream = &streams[vsd->_in.sti.stream_id];
			inb_poke_byte(-1, vsd->_in.cur_stream->remap_to, vsd->_in.buffer);
			save_this_one |= vsd->_in.cur_stream->save;
		}
		packet_length = inb_get_bytes(2, vsd->_in.buffer);
		vsd->_in.sti.lba = vsd->_in.buffer->lba;
		vsd->_in.sti.idx = vsd->_in.buffer->idx;
		vsd->_in.sti.length = packet_length;
		vsd->_in.cur_stream->packets++;
		
#ifdef vs_DECRYPT
		if (vsd->_in.buffer->encrypted)
		{
			bool old_valid_key = vsd->_in.valid_key;
			
			if (vsd->_in.vob_key && vsd->_in.sti.vob_id != -1 && vsd->_in.cell_key && vsd->_in.sti.cell_id != -1)
			{ // one key for each VOB-/CELL-ID combination
				vsd->_in.valid_key = vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS + vsd->_in.sti.cell_id].valid_key;
				if (vsd->_in.valid_key)
					memcpy(vsd->key, vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS + vsd->_in.sti.cell_id].key, sizeof vsd->key);
			}
			else if (vsd->_in.vob_key && vsd->_in.sti.vob_id != -1)
			{ // ony key for each VOB-ID
				vsd->_in.valid_key = vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS].valid_key;
				if (vsd->_in.valid_key)
					memcpy(vsd->key, vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS].key, sizeof vsd->key);
			}
			else if (vsd->_in.cell_key && vsd->_in.sti.cell_id != -1)
			{ // one key for each CELL-ID
				vsd->_in.valid_key = vsd->_in.opened[vsd->_in.sti.cell_id].valid_key;
				if (vsd->_in.valid_key)
					memcpy(vsd->key, vsd->_in.opened[vsd->_in.sti.cell_id].key, sizeof vsd->key);
			}
			if (!vsd->_in.valid_key && (vsd->flags & vs_DONT_CRACK) == 0)
			{
				vsd->_in.valid_key = vs_search_key(vsd, vsd->_in.buffer, vsd->_in.vob_key ? vsd->_in.sti.vob_id : -1, vsd->_in.cell_key ? vsd->_in.sti.cell_id : -1);
				if (!vsd->_in.valid_key)
					vsd->_in.valid_key = old_valid_key; // try the old one
				if (!vsd->_in.valid_key)
				{ // errormessage was in keysearch
					vs_close_all(vsd);
					return vse_CANT_CRACK;
				}
				else
				{
					if (vsd->_in.vob_key && vsd->_in.sti.vob_id != -1 && vsd->_in.cell_key && vsd->_in.sti.cell_id != -1)
					{ // one key for each VOB-/CELL-ID combination
						vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS + vsd->_in.sti.cell_id].valid_key = TRUE;
						memcpy(vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS + vsd->_in.sti.cell_id].key, vsd->key, sizeof vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS + vsd->_in.sti.cell_id].key);
					}
					else if (vsd->_in.vob_key && vsd->_in.sti.vob_id != -1)
					{ // ony key for each VOB-ID
						vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS].valid_key = TRUE;
						memcpy(vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS].key, vsd->key, sizeof vsd->_in.opened[vsd->_in.sti.vob_id * MAX_CELLIDS].key);
					}
					else if (vsd->_in.cell_key && vsd->_in.sti.cell_id != -1)
					{ // one key for each CELL-ID
						vsd->_in.opened[vsd->_in.sti.cell_id].valid_key = TRUE;
						memcpy(vsd->_in.opened[vsd->_in.sti.cell_id].key, vsd->key, sizeof vsd->_in.opened[vsd->_in.sti.cell_id].key);
					}
				}
			}
			if (vsd->_in.valid_key)
				vs_decss(vsd->_in.buffer, vsd->key);
		}
#endif
		vsd->_in.sti.encrypted = vsd->_in.buffer->encrypted;
		
		switch (vsd->_in.sti.stream_id)
		{
		case PRIVATE_STREAM_2:
			{
				// get the VOB- and CELL-ID tags/remove macrovision
				dword vob_id, cell_id;
				byte macro_flag;
				
				switch (inb_peek_byte(0, vsd->_in.buffer))
				{
				case 0: // PCI-packet (remove macrovision)
					if ((vsd->flags & vs_DEMACRO) != 0 && ((macro_flag = inb_peek_byte(5, vsd->_in.buffer)) & 0x80) != 0)
						inb_poke_byte(5, (byte)(macro_flag & (~vs_MACROVISION_BITS)), vsd->_in.buffer);
					break;
				case 1: // DSI-packet (get cell/vob ids)
					vob_id = inb_peek_bytes(25, 2, vsd->_in.buffer);
					cell_id = inb_peek_bytes(28, 1, vsd->_in.buffer);
					if (vsd->_in.sti.vob_id != vob_id || vsd->_in.sti.cell_id != cell_id)
					{
						vsd->_in.save_cell = vs_change_vobcells(vob_id, cell_id, num_idl, idl, vsd);
						if (!vsd->_in.save_cell)
							vsd->_in.buffer->save = 0x00; // save none
						else
							for (j = 0; j < vsd->num_outputs; j++)
								vsd->_in.buffer->save |= (vsd->_in.buffer->has_system && (vsd->outputs[j].flags & vso_DEMUX) == 0) << j;
					}
					break;
				}
				/*						if ((vsd->flags & vs_NO_VOB) == 0 && inb_peek_bytes(packet_length, 3, vsd->_in.buffer) != PACKET_START_CODE_PREFIX)
				inb_get_bytes((fio_SECTOR_SIZE - (vsd->_in.buffer->idx + packet_length)) & (fio_SECTOR_SIZE / 2 - 1), vsd->_in.buffer); // skip rest of this chunk in the program-stream*/
			}
		case PROGRAM_STREAM_MAP:
		case ECM_STREAM:
		case EMM_STREAM:
		case PROGRAM_STREAM_DIRECTORY:
		case DSMCC_STREAM:
		case ITUTRECH222TYPEE_STREAM:
		case PADDING_STREAM:
			vsd->_in.cur_stream->bytes += packet_length;
			print_streaminfo((bool)((vsd->flags & vs_PRINT_INFO) != 0), vsd->_in.buffer->lba, streams, substreams, vsd->_in.sti.stream_id, 0);
			if (save_this_one != 0x00 && vsd->_in.save_cell)
			{
				for (j = 0; j < vsd->num_outputs; j++)
				{
					if ((save_this_one & (1 << j)) != 0)
					{
						if (vsd->_in.cur_stream->user_func)
						{
							vsd->_in.sti.out_lba = 0;
							vsd->_in.sti.out_file_num = 0;
							vsd->_in.sti.out_flags = NULL;
							if (vsd->_in.buffer->outfp[j])
							{
								vsd->_in.sti.out_lba = vsd->_in.buffer->outfp[j]->lba;
								vsd->_in.sti.out_file_num = vsd->_in.buffer->outfp[j]->file_idx;
								vsd->_in.sti.out_flags = &vsd->outputs[j];
							}
							user_func_stay = vsd->_in.cur_stream->user_func(&vsd->_in.buffer->bytes[vsd->_in.buffer->idx], &vsd->_in.sti, vsd->_in.cur_stream->user_data);
						}
						if (vsd->_in.buffer->outfp[j])
						{
							if ((vsd->outputs[j].flags & vso_DEMUX) != 0)
							{
								if (fio_write(&vsd->_in.buffer->bytes[vsd->_in.buffer->idx], packet_length, vsd->_in.buffer->outfp[j]) != packet_length)
									vsd->_in.buffer->status = inb_CANT_WRITE;
							}
							else
								vsd->_in.buffer->save |= (1 << j);
						}
					}
				}
				if (j == 0 && vsd->_in.cur_stream->user_func)
					user_func_stay = vsd->_in.cur_stream->user_func(&vsd->_in.buffer->bytes[vsd->_in.buffer->idx], &vsd->_in.sti, vsd->_in.cur_stream->user_data); // also call if no outputs
			}
			inb_get_bytes(packet_length, vsd->_in.buffer);
			break;
		default:
			inb_get_bytes(1, vsd->_in.buffer); // skip flags
			i = inb_get_bytes(1, vsd->_in.buffer);
			header_length = inb_get_bytes(1, vsd->_in.buffer);
			if (i >= 0x80 && vsd->_in.cur_stream->pts == 0.0)
			{ // PTS
				dword pts;
				
				pts = (inb_peek_byte(0, vsd->_in.buffer) & 0x0e) >> 1;
				if (pts & 0x04)
					vsd->_in.cur_stream->pts = 4294967296.0;
				else
					vsd->_in.cur_stream->pts = 0.0;
				pts = pts << 30;
				pts |= (inb_peek_bytes(1, 2, vsd->_in.buffer) & 0xfffe) << 14;
				pts |= (inb_peek_bytes(3, 2, vsd->_in.buffer) >> 1) & 0x7fff;
				vsd->_in.cur_stream->pts += (double)pts;
				vsd->_in.cur_stream->pts /= 90.0;
			}
			inb_get_bytes(header_length + 1, vsd->_in.buffer); // +1 = skip substream_id, already got that
			print_streaminfo((bool)((vsd->flags & vs_PRINT_INFO) != 0), vsd->_in.buffer->lba, streams, substreams, vsd->_in.sti.stream_id, vsd->_in.sti.substream_id);
			streams[vsd->_in.sti.stream_id].bytes += packet_length - header_length - 3;
			if (vsd->_in.sti.stream_id == PRIVATE_STREAM_1)
			{
				vsd->_in.cur_stream->bytes += packet_length - header_length - 3;
				inb_poke_byte(-1, vsd->_in.cur_stream->remap_to, vsd->_in.buffer);
				save_this_one |= vsd->_in.cur_stream->save;
			}
			else if ((vsd->_in.sti.stream_id >= VIDEO_STREAM) && (vsd->_in.sti.stream_id <= MAX_VIDEO_STREAM))
			{ 
				if (inb_peek_bytes(-1, 4, vsd->_in.buffer) == SEQUENCE_HEADER_CODE)
				{ // change framerate, aspectratio
					byte frameaspect;
					
					frameaspect = inb_peek_byte(6, vsd->_in.buffer);
					if (!vsd->_in.did_video_info)
					{
						fprintf(vs_STDOUT, "  Width = %u\n", inb_peek_bytes(3, 3, vsd->_in.buffer) >> 12);
						fprintf(vs_STDOUT, "  Height = %u\n", inb_peek_bytes(3, 3, vsd->_in.buffer) & 0xfff);
						fprintf(vs_STDOUT, "  Aspect-ratio = %s\n", vs_get_aspectratio(frameaspect >> 4));
						fprintf(vs_STDOUT, "  Frame-rate = %s\n", vs_get_framerate(frameaspect & 0xf));
						vsd->_in.did_video_info = TRUE;
					}
					if (vsd->_in.framerate != -1 || vsd->_in.aspectratio != -1)
					{
						if (vsd->_in.framerate != -1)
							frameaspect = (frameaspect & 0xf0) | vsd->_in.framerate;
						if (vsd->_in.aspectratio != -1)
							frameaspect = (frameaspect & 0xf) | vsd->_in.aspectratio;
						inb_poke_byte(6, frameaspect, vsd->_in.buffer);
					}
				}
				else
					save_this_one &= vsd->_in.only_gop_mask;
			}
			
			if (save_this_one != 0x00 && vsd->_in.save_cell)
			{
				vsd->_in.sti.idx = vsd->_in.buffer->idx - 1;
				vsd->_in.sti.length = packet_length - header_length - 3;
				vsd->_in.sti.lba = vsd->_in.buffer->lba;
				
				for (j = 0; j < vsd->num_outputs; j++)
					if ((save_this_one & (1 << j)) != 0)
					{
						if (vsd->_in.cur_stream->user_func)
						{
							vsd->_in.sti.out_lba = 0;
							vsd->_in.sti.out_file_num = 0;
							vsd->_in.sti.out_flags = NULL;
							if (vsd->_in.buffer->outfp[j])
							{
								vsd->_in.sti.out_lba = vsd->_in.buffer->outfp[j]->lba;
								vsd->_in.sti.out_file_num = vsd->_in.buffer->outfp[j]->file_idx;
								vsd->_in.sti.out_flags = &vsd->outputs[j];
							}
							user_func_stay = vsd->_in.cur_stream->user_func(&vsd->_in.buffer->bytes[vsd->_in.sti.idx], &vsd->_in.sti, vsd->_in.cur_stream->user_data);
						}
						if (vsd->_in.buffer->outfp[j])
						{
							if ((vsd->outputs[j].flags & vso_DEMUX) != 0)
							{
								dword w_idx = vsd->_in.buffer->idx - 1;
								dword w_len = packet_length - header_length - 3;

								if ((vsd->outputs[j].flags & vso_KEEP_AC3_IDENT_BYTES) == 0 && (vsd->_in.sti.stream_id == PRIVATE_STREAM_1) && (vsd->_in.sti.substream_id >= SUBSTREAM_AC3) && (vsd->_in.sti.substream_id <= MAX_SUBSTREAM_AC3))
								{ // skip the AC3 identifier bytes
									w_idx += 4;
									w_len -= 4;
								}
								else if ((vsd->outputs[j].flags & vso_KEEP_PCM_IDENT_BYTES) == 0 && (vsd->_in.sti.stream_id == PRIVATE_STREAM_1) && (vsd->_in.sti.substream_id >= SUBSTREAM_PCM) && (vsd->_in.sti.substream_id <= MAX_SUBSTREAM_PCM))
								{ // skip the PCM identifier bytes
									w_idx += 7;
									w_len -= 7;
								}
								if (fio_write(&vsd->_in.buffer->bytes[w_idx], w_len, vsd->_in.buffer->outfp[j]) != w_len)
									vsd->_in.buffer->status = inb_CANT_WRITE;
							}
							else
								vsd->_in.buffer->save |= (1 << j);
						}
					}
					if (j == 0 && vsd->_in.cur_stream->user_func)
						user_func_stay = vsd->_in.cur_stream->user_func(&vsd->_in.buffer->bytes[vsd->_in.sti.idx], &vsd->_in.sti, vsd->_in.cur_stream->user_data); // also call if no outputs
			}
			inb_get_bytes(packet_length - header_length - 4, vsd->_in.buffer);
			break;
			}
			break;
	}
	if (user_func_stay == FALSE)
	{
		vs_done(vsd, streams, substreams);
		return vse_USER_FUNC_EXIT;
	}
	if (vsd->_in.buffer->status == inb_OK && vsd->_in.buffer->lba > vsd->end_lba)
		vsd->_in.buffer->status = inb_EMPTY;
	if (vsd->_in.buffer->status == inb_OK)
		return vse_OK;
	else if (vsd->_in.buffer->status == inb_EMPTY)
	{
		vs_done(vsd, streams, substreams);
		return vse_DONE;
	}
	else
	{ // inb_CANT_WRITE
		if ((vsd->flags & vs_PRINT_ERROR) != 0)
			fprintf(vs_STDERR, "Could not write to output-file(s). Maybe disk full?\n");
		vs_close_all(vsd);
		return vse_CANT_WRITE_OUTPUT;
	}
}

VSTRIP_DLL_API t_vs_errorcode VSTRIP_DLL_CC vs_strip(tp_vs_data vsd, t_vs_streamflags streams[256], t_vs_streamflags substreams[256], dword num_idl, tp_vs_vobcellid idl)
{
	t_vs_errorcode ec = vse_INIT_FAILED;
	
	ec = vs_init(vsd, streams, substreams);
	if (ec == vse_OK)
		for (; ec == vse_OK; ec = vs_strip_one_block(vsd, streams, substreams, num_idl, idl)) ;
		return ec; // vs_done not neccessary as we call vs_strip_one_block until it was done
}

VSTRIP_DLL_API dword VSTRIP_DLL_CC vs_get_version(t_vs_versionflags* flags)
{
	if (flags)
	{
		*flags = 0;
#ifdef vs_DECRYPT
		*flags |= vsv_DECRYPT;
#endif
	}
	return vs_VERSION;
}

VSTRIP_DLL_API const char* VSTRIP_DLL_CC vs_get_author(void)
{
#ifdef vs_DECRYPT
	return "[maven] (maven@maven.de), CSS-code by R0TfL";
#else
	return "[maven] (maven@maven.de)";
#endif
}
