package iaik.pkcs.pkcs11.objects;

import iaik.pkcs.pkcs11.Session;
import iaik.pkcs.pkcs11.TokenException;
import iaik.pkcs.pkcs11.wrapper.Constants;

import java.io.UnsupportedEncodingException;

/**
 * Objects of this class represent a clock as specified by PKCS#11
 * v2.11. Remind that this is a snapshot; this means that this object does not
 * get the values from the token on demand it gets them uppon instanciation.
 *
 * @author Karl Scheibelhofer
 * @version 1.0
 * @invariants (value_ <> null)
 */
public class Clock extends HardwareFeature {

	/**
	 * The value attribute of this clock; i.e. the value, this object was created.
	 */
	protected ByteArrayAttribute value_;

	/**
	 * Deafult Constructor.
	 *
	 * @preconditions
	 * @postconditions
	 */
	public Clock() {
		super();
		hardwareFeatureType_.setLongValue(FeatureType.CLOCK);
	}

	/**
	 * Called by getInstance to create an instance of a PKCS#11 clock.
	 *
	 * @param session The session to use for reading attributes.
	 *                This session must have the appropriate rights; i.e.
	 *                it must be a user-session, if it is a private object.
	 * @param objectHandle The object handle as given from the PKCS#111 module.
	 * @exception TokenException If getting the attributes failed.
	 * @preconditions (session <> null)
	 * @postconditions
	 */
	protected Clock(Session session, long objectHandle)
	    throws TokenException
	{
		super(session, objectHandle);
		hardwareFeatureType_.setLongValue(FeatureType.CLOCK);
	}

	/**
	 * The getInstance method of the HardwareFeature class uses this method to
	 * create an instance of a PKCS#11 clock.
	 *
	 * @param session The session to use for reading attributes.
	 *                This session must have the appropriate rights; i.e.
	 *                it must be a user-session, if it is a private object.
	 * @param objectHandle The object handle as given from the PKCS#111 module.
	 * @return The object representing the PKCS#11 object.
	 *         The returned object can be casted to the
	 *         according sub-class.
	 * @exception TokenException If getting the attributes failed.
	 * @preconditions (session <> null)
	 * @postconditions (result <> null) 
	 */
	public static Object getInstance(Session session, long objectHandle)
	    throws TokenException
	{
		return new Clock(session, objectHandle);
	}

	/**
	 * Put all attributes of the given object into the attributes table of this
	 * object. This method is only static to be able to access invoke the
	 * implementation of this method for each class separately (see use in
	 * clone()).
	 *
	 * @param object The object to handle.
	 * @preconditions (object <> null)
	 * @postconditions
	 */
	protected static void putAttributesInTable(Clock object) {
		if (object == null) {
			throw new NullPointerException("Argument \"object\" must not be null.");
		}

		object.attributeTable_.put(Attribute.VALUE, object.value_);
	}

	/**
	 * Allocates the attribute objects for this class and adds them to the
	 * attribute table.
	 *
	 * @preconditions
	 * @postconditions
	 */
	protected void allocateAttributes() {
		super.allocateAttributes();

		value_ = new ByteArrayAttribute(Attribute.VALUE);

		putAttributesInTable(this);
	}

	/**
	 * Create a (deep) clone of this object.
	 *
	 * @return A clone of this object.
	 * @preconditions
	 * @postconditions (result <> null)
	 *                 and (result instanceof Clock)
	 *                 and (result.equals(this))
	 */
	public java.lang.Object clone() {
		Clock clone = (Clock) super.clone();

		clone.value_ = (ByteArrayAttribute) this.value_.clone();

		putAttributesInTable(clone); // put all cloned attributes into the new table

		return clone;
	}

	/**
	 * Compares all member variables of this object with the other object.
	 * Returns only true, if all are equal in both objects.
	 *
	 * @param otherObject The other object to compare to.
	 * @return True, if other is an instance of Info and all member variables of
	 *         both objects are equal. False, otherwise.
	 * @preconditions
	 * @postconditions
	 */
	public boolean equals(java.lang.Object otherObject) {
		boolean equal = false;

		if (otherObject instanceof Clock) {
			Clock other = (Clock) otherObject;
			equal = (this == other)
			    || (super.equals(other) && this.value_.equals(other.value_));
		}

		return equal;
	}

	/**
	 * Gets the value attribute of this clock object.
	 *
	 * @return The value attribute.
	 * @preconditions
	 * @postconditions (result <> null)
	 */
	public ByteArrayAttribute getValue() {
		return value_;
	}

	/**
	 * The overriding of this method should ensure that the objects of this class
	 * work correctly in a hashtable.
	 *
	 * @return The hash code of this object.
	 * @preconditions
	 * @postconditions
	 */
	public int hashCode() {
		return value_.hashCode();
	}

	/**
	 * Read the values of the attributes of this object from the token.
	 *
	 * @param session The session handle to use for reading attributes.
	 *                This session must have the appropriate rights; i.e.
	 *                it must be a user-session, if it is a private object.
	 * @exception TokenException If getting the attributes failed.
	 * @preconditions (session <> null)
	 * @postconditions
	 */
	public void readAttributes(Session session)
	    throws TokenException
	{
		super.readAttributes(session);

		Object.getAttributeValue(session, objectHandle_, value_);
	}

	/**
	 * This method returns a string representation of the current object. The
	 * output is only for debugging purposes and should not be used for other
	 * purposes.
	 *
	 * @return A string presentation of this object for debugging output.
	 * @preconditions
	 * @postconditions (result <> null)
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer(256);

		buffer.append(super.toString());

		buffer.append(Constants.NEWLINE);
		buffer.append(Constants.INDENT);
		buffer.append("Value: ");
		try {
			buffer.append(new String(value_.getByteArrayValue(), "ASCII"));
		} catch (UnsupportedEncodingException ex) {
			buffer.append(new String(value_.getByteArrayValue()));
		}

		return buffer.toString();
	}

}
