package iaik.pkcs.pkcs11.parameters;

import iaik.pkcs.pkcs11.TokenRuntimeException;
import iaik.pkcs.pkcs11.wrapper.Constants;
import iaik.pkcs.pkcs11.wrapper.Functions;
import iaik.pkcs.pkcs11.wrapper.PKCS11Constants;

/**
 * This abstract class encapsulates parameters for the DH mechanisms
 * Mechanism.ECDH1_DERIVE, Mechanism.ECDH1_COFACTOR_DERIVE,
 * Mechanism.ECMQV_DERIVE, Mechanism.X9_42_DH_DERIVE ,
 * Mechanism.X9_42_DH_HYBRID_DERIVE and Mechanism.X9_42_MQV_DERIVE.
 *
 * @author Karl Scheibelhofer
 * @version 1.0
 * @invariants (keyDerivationFunction_ == KeyDerivationFunctionType.NULL)
 *              or (keyDerivationFunction_ == KeyDerivationFunctionType.SHA1_KDF)
 *              or (keyDerivationFunction_ == KeyDerivationFunctionType.SHA1_KDF_ASN1)
 *              or (keyDerivationFunction_ == KeyDerivationFunctionType.SHA1_KDF_CONCATENATE))
 *             and (publicData_ <> null)
 */
abstract public class DHKeyDerivationParameters implements Parameters {

	/**
	 * This interface defines the available key derivation function types as
	 * defined by PKCS#11: CKD_NULL, CKD_SHA1_KDF, CKD_SHA1_KDF_ASN1,
	 * CKD_SHA1_KDF_CONCATENATE.
	 *
	 * @author Karl Scheibelhofer
	 * @version 1.0
	 * @invariants
	 */
	public interface KeyDerivationFunctionType {

		/**
		 * The indentifier for CKD_NULL.
		 */
		static public final long NULL = PKCS11Constants.CKD_NULL;

		/**
		 * The indentifier for CKD_SHA1_KDF.
		 */
		static public final long SHA1_KDF = PKCS11Constants.CKD_SHA1_KDF;

		/**
		 * The indentifier for CKD_SHA1_KDF_ASN1.
		 */
		static public final long SHA1_KDF_ASN1 = PKCS11Constants.CKD_SHA1_KDF_ASN1;

		/**
		 * The indentifier for CKD_SHA1_KDF_CONCATENATE.
		 */
		static public final long SHA1_KDF_CONCATENATE = PKCS11Constants.CKD_SHA1_KDF_CONCATENATE;

	}

	/**
	 * The key derivation function used on the shared secret value.
	 */
	protected long keyDerivationFunction_;

	/**
	 * The other partie's public key value.
	 */
	protected byte[] publicData_;

	/**
	 * Create a new DHKeyDerivationParameters object with the given attributes.
	 *
	 * @param keyDerivationFunction The key derivation function used on the shared
	 *                              secret value.
	 *                              One of the values defined in
	 *                              KeyDerivationFunctionType.
	 * @param publicData The other partie's public key value.
	 * @preconditions ((keyDerivationFunction == KeyDerivationFunctionType.NULL)
	 *                 or (keyDerivationFunction == KeyDerivationFunctionType.SHA1_KDF)
	 *                 or (keyDerivationFunction == KeyDerivationFunctionType.SHA1_KDF_ASN1)
	 *                 or (keyDerivationFunction == KeyDerivationFunctionType.SHA1_KDF_CONCATENATE))
	 *                and (publicData <> null)
	 * @postconditions
	 */
	protected DHKeyDerivationParameters(long keyDerivationFunction, byte[] publicData) {
		if ((keyDerivationFunction != KeyDerivationFunctionType.NULL)
		    && (keyDerivationFunction != KeyDerivationFunctionType.SHA1_KDF)
		    && (keyDerivationFunction != KeyDerivationFunctionType.SHA1_KDF_ASN1)
		    && (keyDerivationFunction != KeyDerivationFunctionType.SHA1_KDF_CONCATENATE)) {
			throw new IllegalArgumentException(
			    "Illegal value for argument\"keyDerivationFunction\": "
			        + Functions.toHexString(keyDerivationFunction));
		}
		if (publicData == null) {
			throw new NullPointerException("Argument \"publicData\" must not be null.");
		}
		keyDerivationFunction_ = keyDerivationFunction;
		publicData_ = publicData;
	}

	/**
	 * Create a (deep) clone of this object.
	 *
	 * @return A clone of this object.
	 * @preconditions
	 * @postconditions (result <> null)
	 *                 and (result instanceof DHKeyDerivationParameters)
	 *                 and (result.equals(this))
	 */
	public java.lang.Object clone() {
		DHKeyDerivationParameters clone;

		try {
			clone = (DHKeyDerivationParameters) super.clone();

			clone.publicData_ = (byte[]) this.publicData_.clone();
		} catch (CloneNotSupportedException ex) {
			// this must not happen, because this class is cloneable
			throw new TokenRuntimeException("An unexpected clone exception occurred.", ex);
		}

		return clone;
	}

	/**
	 * Get the key derivation function used on the shared secret value.
	 *
	 * @return The key derivation function used on the shared secret value.
	 *         One of the values defined in KeyDerivationFunctionType.
	 * @preconditions
	 * @postconditions
	 */
	public long getKeyDerivationFunction() {
		return keyDerivationFunction_;
	}

	/**
	 * Get the other partie's public key value.
	 *
	 * @return The other partie's public key value.
	 * @preconditions
	 * @postconditions (result <> null)
	 */
	public byte[] getPublicData() {
		return publicData_;
	}

	/**
	 * Set the ey derivation function used on the shared secret value.
	 *
	 * @param keyDerivationFunction The key derivation function used on the shared
	 *                              secret value.
	 *                              One of the values defined in
	 *                              KeyDerivationFunctionType.
	 * @preconditions (keyDerivationFunction == KeyDerivationFunctionType.NULL)
	 *                or (keyDerivationFunction == KeyDerivationFunctionType.SHA1_KDF))
	 *                or (keyDerivationFunction == KeyDerivationFunctionType.SHA1_KDF_ASN1))
	 *                or (keyDerivationFunction == KeyDerivationFunctionType.SHA1_KDF_CONCATENATE))
	 * @postconditions
	 */
	public void setKeyDerivationFunction(long keyDerivationFunction) {
		if ((keyDerivationFunction != KeyDerivationFunctionType.NULL)
		    && (keyDerivationFunction != KeyDerivationFunctionType.SHA1_KDF)
		    && (keyDerivationFunction != KeyDerivationFunctionType.SHA1_KDF_ASN1)
		    && (keyDerivationFunction != KeyDerivationFunctionType.SHA1_KDF_CONCATENATE)) {
			throw new IllegalArgumentException(
			    "Illegal value for argument\"keyDerivationFunction\": "
			        + Functions.toHexString(keyDerivationFunction));
		}
		keyDerivationFunction_ = keyDerivationFunction;
	}

	/**
	 * Set the other partie's public key value.
	 *
	 * @param publicData The other partie's public key value.
	 * @preconditions (publicData <> null)
	 * @postconditions
	 */
	public void setPublicData(byte[] publicData) {
		if (publicData == null) {
			throw new NullPointerException("Argument \"publicData\" must not be null.");
		}
		publicData_ = publicData;
	}

	/**
	 * Returns the string representation of this object. Do not parse data from
	 * this string, it is for debugging only.
	 *
	 * @return A string representation of this object.
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();

		buffer.append(Constants.INDENT);
		buffer.append("Key Derivation Function: ");
		if (keyDerivationFunction_ == KeyDerivationFunctionType.NULL) {
			buffer.append("NULL");
		} else if (keyDerivationFunction_ == KeyDerivationFunctionType.SHA1_KDF) {
			buffer.append("SHA1_KDF");
		} else if (keyDerivationFunction_ == KeyDerivationFunctionType.SHA1_KDF_ASN1) {
			buffer.append("SHA1_KDF_ASN1");
		} else if (keyDerivationFunction_ == KeyDerivationFunctionType.SHA1_KDF_CONCATENATE) {
			buffer.append("SHA1_KDF_CONCATENATE");
		} else {
			buffer.append("<unknown>");
		}
		buffer.append(Constants.NEWLINE);

		buffer.append(Constants.INDENT);
		buffer.append("Public Data: ");
		buffer.append(Functions.toHexString(publicData_));
		// buffer.append(Constants.NEWLINE);

		return buffer.toString();
	}

	/**
	 * Compares all member variables of this object with the other object.
	 * Returns only true, if all are equal in both objects.
	 *
	 * @param otherObject The other object to compare to.
	 * @return True, if other is an instance of this class and all member
	 *         variables of both objects are equal. False, otherwise.
	 * @preconditions
	 * @postconditions
	 */
	public boolean equals(java.lang.Object otherObject) {
		boolean equal = false;

		if (otherObject instanceof DHKeyDerivationParameters) {
			DHKeyDerivationParameters other = (DHKeyDerivationParameters) otherObject;
			equal = (this == other)
			    || ((this.keyDerivationFunction_ == other.keyDerivationFunction_) && Functions
			        .equals(this.publicData_, other.publicData_));
		}

		return equal;
	}

	/**
	 * The overriding of this method should ensure that the objects of this class
	 * work correctly in a hashtable.
	 *
	 * @return The hash code of this object.
	 * @preconditions
	 * @postconditions
	 */
	public int hashCode() {
		return ((int) keyDerivationFunction_) ^ Functions.hashCode(publicData_);
	}

}
