/*
 * Copyright (c) 2004, 2005, id Quantique SA, Switzerland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of id Quantique nor the names of its contributors may be
 * used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


/**
 * File:              quantis_pci_common.h
 * Last modification: 4 May 2010
 * Description:
 * ------------
 * This file defines all the functions, types and operation used by all the Unix
 * drivers. 
 *
 * Few elements which are system dependant are defined as opaque type
 * or extern definitions and should be implemented in the system dependant file
 * called "quantis_pci_OSName.c". The rest of the functions are mainly the manipulation
 * of the Quantis card (read the data, enable modules, read config etc).
 *
*/

#ifndef QUANTIS_PCI_COMMON_H
#define QUANTIS_PCI_COMMON_H

#include "quantis_pci.h"

QUANTIS_EXTERN_C_START

#ifdef __linux__
# include <linux/types.h>
#endif

/* If the code for getting FIFO availability is using the hardware interrupts of the 
   Quantis card,  the define "USE_INTERRUPTS" should
   be set and the function "quantis_rng_wait_fifo" should be implemented
   in the system dependant file (which has the name "quantis_pci_MYOS.c").

   Since this function will probably not be written at the same time for all OS,
   (the managing of interruptions is not a very simple task in a driver),
   this define should be done on the compiler command line.
*/ 
/* #define USE_INTERRUPTS */

#include "quantis_pci_reg.h"
typedef int boolean;
#define false 0
#define true 1

/**
 * Size of the internal buffer used to read random data.
 * This corresponds to the maximal length that can be read by quantis_rng_read
 * at once.
 * @see quantis_rng_read
 */
#define QUANTIS_DEVICE_BUFFER_SIZE  (32 * QUANTIS_FIFO_SIZE)

#define QUANTIS_DRIVER_VERSION 12

/**
 * Structure representing a Quantis PCI device. This is an opaque type
 * that must be defined in quantis_pci_MYOS.c
 */
typedef struct quantis_pci_device quantis_pci_device;


/**
 * Get the value of a register. This function is OS-specific and must be 
 * implemented in quantis_pci_MYOS.c
 * @param qdev 
 * @param reg which is the offset in bytes since the begin of register base.
 * @return the value of a register
 */
extern QuantisRegValue quantis_reg_get(quantis_pci_device* qdev, QuantisRegister reg);


/**
 * Set the value of a register. This function is OS-specific and must be
 * implemented in quantis_pci_MYOS.c
 * @param qdev
 * @param reg which is the offset in bytes since the begin of register base.
 * @param value compatible with a Quantis register
 */
extern void quantis_reg_set(quantis_pci_device* qdev, QuantisRegister reg, QuantisRegValue value);

/**
 * Wait until the FIFO has some random data available or a timeout occur.
 * 
 * @return the number of bytes of random data available in the FIFO or a 
 * negative value on timeout.
 */
#ifdef USE_INTERRUPTS
/* When using interrupts instead of doing light (?) active wait, this function
   should be implemented in system dependant file. */
extern int quantis_rng_wait_fifo(quantis_pci_device* qdev);
#else
/* This forward decalration is useful to avoid a warning for implicit
   function declaration in the following code and to avoid to have
   the code of the function "quantis_rng_fifo_bytes" at a not too convenient
   place in this file.
*/
static inline int quantis_rng_fifo_bytes(quantis_pci_device* qdev);
static inline int quantis_rng_wait_fifo(quantis_pci_device* qdev)
{
  unsigned int timeout = 10000;
  QuantisRegValue available_bytes = 0;

  while(timeout > 0)
  {
    available_bytes = quantis_rng_fifo_bytes(qdev);
    if (available_bytes > 0)
    {
      return available_bytes;
    }
    timeout--;

    // TODO
    // (in function 'quantis_rng_wait_fifo'): add sleep or
    // use the interrupt of the cards and define 'USE_INTERRUPTS'
    // Active sleep is bad, while passive sleep (like relinquish
    // the CPU for a tick) is too much time, so there isn't a solution.
  }

  /* Wait timed out*/
  return -1;
}
#endif

/**
 * 
 * @param mask an integer with in the bits 0-3 set if the bit "type" should be set
 *             for the corresponding Quantis module.
 * @param type The bit to set for each module 
 * @return A value suitable to be put in a Quantis register with the bit "type" correctly set for each Quantis 
*          module specified in "mask".
 */
static inline QuantisRegValue mask2reg(ModuleMask mask, u_int32_t type)
{
  QuantisRegValue reg = (mask & MODULE1 ? 1 << 0  : 0)
                | (mask & MODULE2 ? 1 << 8  : 0)
                | (mask & MODULE3 ? 1 << 16 : 0)
                | (mask & MODULE4 ? 1 << 24 : 0);
  return reg << type;
}


/**
 *
 * @param reg is a register describing many Quantis modules. 
 * @param type is a bit which should be checked for any of the module
 * @return a positive value where the bits 0-3 are set if the
 *         bit "type" was set for the corresponding module. The result can be used
 *         as a boolean value to know if any of the bit was set for a module.
 */
static inline ModuleMask reg2mask(QuantisRegister reg, u_int32_t type)
{
  reg >>= type;
  return (ModuleMask)
         (reg & (1 << 0)  ? MODULE1 : 0)
    |    (reg & (1 << 8)  ? MODULE2 : 0)
    |    (reg & (1 << 16) ? MODULE3 : 0)
    |    (reg & (1 << 24) ? MODULE4 : 0);
}


/**
 * Returns the FIFO status
 * @param qdev
 * @return the value of the FIFO status register
 */
static inline QuantisRegValue quantis_get_fifo_status(quantis_pci_device* qdev)
{
  return quantis_reg_get(qdev, FS_RR);
}


/**
 * Flush the FIFO.
 * @param qdev
 */
static inline void quantis_flush_fifo(quantis_pci_device* qdev)
{
  quantis_reg_set(qdev, FS_CA, 0);
}


/**
 *
 * @param qdev
 * @return a boolean indicating if any of the modules is in error.
 */
static inline boolean quantis_rng_error(quantis_pci_device* qdev)
{
  QuantisRegValue reg    = quantis_reg_get(qdev, MX_SR);
  ModuleMask test   = reg2mask(reg, MX_TM);
  ModuleMask status = reg2mask(reg, MX_HST);
  ModuleMask enable = reg2mask(reg, MX_SEN);

  if (test)
  {
    return false;
  }
  else
  {
    return (enable & status) == 0;
  }
}


/**
 * Returns the number of bytes of random data available in the FIFO.
 * @param qdev
 * @return the number of bytes of random data available in the FIFO.
 */
static inline int quantis_rng_fifo_bytes(quantis_pci_device* qdev)
{
  QuantisRegValue status = quantis_get_fifo_status(qdev);

  /* */
#ifdef __linux__
  static int once = 1;

  if (once)
  {
      schedule();
      once = 0;
  }
#endif

  if (status & QUANTIS_FIFO_FULL)
  {
    return QUANTIS_FIFO_SIZE;
  }
  if (status & QUANTIS_FIFO_FL3)
  {
    return ((QUANTIS_FIFO_SIZE / 4) * 3);
  }
  if (status & QUANTIS_FIFO_FL2)
  {
    return (QUANTIS_FIFO_SIZE / 2);
  }
  if (status & QUANTIS_FIFO_FL1)
  {
    return (QUANTIS_FIFO_SIZE / 4);
  }
  return 0;
}


/**
 * Do a soft reset of the Quantis card. The FIFO is emptied and error are cleared.
 */
static inline void quantis_rng_reset(quantis_pci_device* qdev)
{
  LOG_DEBUG0("In quantis_rng_reset.\n");
  quantis_reg_set(qdev, CC_ER, 1);
  quantis_reg_set(qdev, CC_DR, 1);
}


/**
 * Returns the version of the hardware of the card.
 */
static inline QuantisRegValue quantis_rng_version(quantis_pci_device* qdev)
{
  return quantis_reg_get(qdev, CV_SR);
}

/**
 * Enable module(s) speficified by mask
 * @param qdev
 * @param mask
 */
static inline void quantis_rng_enable_modules(quantis_pci_device* qdev, ModuleMask mask)
{
  /* enable modules */
  quantis_reg_set(qdev, MX_ER, mask2reg(mask, MX_SEN));

  /* Flush FIFO */
  quantis_flush_fifo(qdev);
}


/**
 * Disable module(s) speficified by mask
 * @param qdev
 * @param mask
 */
static inline void quantis_rng_disable_modules(quantis_pci_device* qdev, ModuleMask mask)
{
  /* Disable modules */
  quantis_reg_set(qdev, MX_DR, mask2reg(mask, MX_SEN));

  /* Flush FIFO */
  quantis_flush_fifo(qdev);
}


/**
 *
 * @param qdev
 * @return a mask indicating which Quantis module
 *         is ready.
 */
static inline ModuleMask quantis_rng_modules_status(quantis_pci_device* qdev)
{
  return reg2mask(quantis_reg_get(qdev, MX_SR), MX_SEN)
      &  reg2mask(quantis_reg_get(qdev, MX_SR), MX_HST);
}


/**
 * Return the modules built on the device as a mask of 4 bits. These
 * modules are not necessarly activated.
 * @param qdev
 * @return
 */
static inline ModuleMask quantis_rng_modules_mask(quantis_pci_device* qdev)
{
  return reg2mask(quantis_reg_get(qdev, MX_SR), MX_HEN);
}


/**
 * Fills buffer with random bytes.
 * @param qdev
 * @param buffer a pointer to a buffer which size is at least
 *               QUANTIS_DEVICE_BUFFER_SIZE.
 * @param length the requested number of bytes to fill into the buffer.
 * @return the number of random bytes filled into the buffer or a negative
 * value on error.
 * @warning if (length > QUANTIS_DEVICE_BUFFER_SIZE), only
 * QUANTIS_DEVICE_BUFFER_SIZE will be filled!
 */
static inline int quantis_rng_read(quantis_pci_device* qdev,
                                   unsigned char* buffer,
                                   int length)
{
  int read_bytes = 0;
  int fifo_in_error = 0; /* Number of times the FIFO of Quantis card was in error and
                            the card should be reset */
  const int MAX_ERROR=30;
  /* Verify at least one module is enabled and everything works correctly */
  if (quantis_rng_error(qdev))
  {
    /* Module status error (is at least one module enabled?) */
    return -1;
  }

  /* Read random bytes */
  while (read_bytes < length)
  {
    /* Get the number of bytes available on the FIFO */
    int available_bytes = quantis_rng_wait_fifo(qdev);
    if (available_bytes < 0)
    {
      /* Timeout while waiting random bytes */
      return -2;
    }
    else if (available_bytes > (length - read_bytes))
    {
      available_bytes = (length - read_bytes);
    }

    while (available_bytes > 0)
    {
      int random_data_length = sizeof(RandomData);
      /* this can potentially cause endianness problems */
      RandomData random_data = (RandomData) quantis_reg_get(qdev, FD_RR);
      QuantisRegValue fifo_status = quantis_get_fifo_status(qdev);

      if (fifo_status & QUANTIS_FIFO_ERROR)
      {
        /* The FIFO has overflown, reset it */
        quantis_flush_fifo(qdev);
        available_bytes = 0;
        fifo_in_error++;
        if (fifo_in_error > MAX_ERROR)
        {
           return -3;
        }
        break;
      }

      /* Checks we don't read too much data */
      if (random_data_length > available_bytes)
      {
        random_data_length = available_bytes;
      }

      /* Avoid buffer overflow */
      if ((read_bytes + random_data_length) > QUANTIS_DEVICE_BUFFER_SIZE)
      {
        return read_bytes;
      }

      /* copy random data to the buffer */
      memcpy(buffer + read_bytes, &random_data, random_data_length);
      available_bytes -= random_data_length;
      read_bytes += random_data_length;
    } // while (available_bytes > 0)
  } // while (read_bytes < length)

  return read_bytes;
}

QUANTIS_EXTERN_C_END

#endif

