/*
 * Copyright (c) 2004, 2010, id Quantique SA, Switzerland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of id Quantique nor the names of its contributors may be
 * used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * File:              quantis_reg.h
 * Last modification: 1 May 2010
 * Description:
 * ------------
 * This file defines all the data structures used to describe the Quantis registers.
 * Some types are structure with bits definition and it is not always a good idea
 * to use them in code (since sometime the generated code could be wrong
 * or be different of what is expected). Nonetheless these types are very useful to document
 * the internal structures of the Quantis card and can be useful if used with caution.
 *
*/

#ifndef QUANTIS_PCI_REG_H
#define QUANTIS_PCI_REG_H

QUANTIS_EXTERN_C_START

/* Type describing the values used to select a register of a Quantis board. A register is alway described
   by its offset (expressed in bytes) from the base of the registers. Since a Quantis
   register contains always unsigned value of 32 bits, a value of this type should always
   be divisible 4.
   */
typedef u_int32_t QuantisRegister, *PQuantisRegister;

/*
  This is the type of the value contained in a Quantis register (which is always an unsigned 32 bits value).
*/
typedef u_int32_t QuantisRegValue;
typedef QuantisRegValue RandomData;
typedef QuantisRegValue NotUsed; /* Register not used or reserved on the Quantis board */


/*
 *
 * Quantis PCI Registers
 *
 * The Quantis PCI device has a memory register page 20 words long,
 * consisting of 13 hardware registers. These registers are split into the
 * following functional groups:
 *  -> Core Control group:
 *      - Enable Register `CC_ER'
 *      - Disable Register `CC_DR' 
 *      - Status Register `CC_SR'.
 *  -> Core Version group:
 *      - Status Register `CV_SR', which is used to get the board version.
 *  -> FIFO Status group:
 *      - FIFO Flush Register `FS_CA', 
 *      - FIFO Status Register `FS_RR'
 *      - FIFO Data Read Register `FD_RR'.
 *  -> Module Group:
 *      - Enable Register `MX_ER'
 *      - Disable Register `MX_DR'
 *      - Status Register `MX_SR'
 * used to enable and disable card modules,
 *  -> Interrupt group
 *      - Enable Register `IR_ER'
 *      - Disable Register `IR_DR'
 *      - Status Register `IR_SR'
 * used to enable or disable interrupt for specific conditions
 *
 * All the other registers are "reserved" and should not be used.
 **/

/* Offset (in bytes) of a register in the register array. */
#define CC_ER   0     /* Core enable */
#define CC_DR   4     /* Core disable */
#define CC_SR   12    /* Core status */
#define CV_SR   28    /* Core version  */
#define FS_CA   32    /* FIFO flush */
#define FS_RR   40    /* FIFO status */
#define FD_RR   44    /* FIFO data */
#define MX_ER   48    /* Module enable */
#define MX_DR   52    /* Module disable */
#define MX_SR   60    /* Module status */
#define IR_ER   64    /* Interrupts enable */
#define IR_DR   68    /* Interrupts disable */
#define IR_SR   76    /* Interrupts status */

/**
 * The FIFO of the random number generator is 4096 bytes big (the FIFO
 * is however always read by integer). It can be either
 * empty, 1/4 full, half-full, 3/4 full or completely full.
 */
#define QUANTIS_FIFO_SIZE     4096

typedef struct _QUANTIS_REG {
    QuantisRegValue    CC_ER_r;  /* 0, Write only, IDQ CORE CONTROL Enable Register */
    QuantisRegValue    CC_DR_r;  /* 1, Write only, IDQ CORE CONTROL Disable Register */
    NotUsed            Res1;     /* 2 */
    QuantisRegValue    CC_SR_r;  /* 3, Read only,  IDQ CORE CONTROL Status Register */
    NotUsed            Res2;     /* 4 */
    NotUsed            Res3;     /* 5 */
    NotUsed            Res4;     /* 6 */
    QuantisRegValue    CV_SR_r;  /* 7, Read only, IDQ CORE VERSION Status Register */
    QuantisRegValue    FC_CA_r;  /* 8, Write only, IDQ FIFO CONTROL Clear Access */
                                 /*    Writing any value on this register will CLEAR the FIFO machine */
                                 /*    and the on-chip tester. */
    NotUsed            Res5;     /* 9 */
    QuantisRegValue    FS_RR_r;  /* 10, Read only, IDQ FIFO STATUS Read Register */
    RandomData         FD_RR_r;  /* 11, Read only, IDQ FIFO DATA Read Register */
    QuantisRegValue    MX_ER_r;  /* 12, Write only, IDQ MODULE X Enable Register */
    QuantisRegValue    MX_DR_r;  /* 13, Write only, IDQ MODULE X Disable Register */
    NotUsed            Res6;     /* 14 */
    QuantisRegValue    MX_SR_r;  /* 15, Read only, IDQ MODULE X Status Register */
    QuantisRegValue    IR_ER_r;  /* 16, Write only, IDQ IRQ Enable Register */
    QuantisRegValue    IR_DR_r;  /* 17, Write only, IDQ IRQ Disable Register */
    NotUsed            Res7;     /* 18 */
    QuantisRegValue    IR_SR_r;  /* 19, Read only, IDQ IRQ Status Register */
} QUANTIS_REG, * PQUANTIS_REG;

/* Size in bytes of all hardware registers */
#define QUANTIS_REG_LENGTH (sizeof(QUANTIS_REG))

/*
*-----------------------------------------------------------------------------
* Define the Interrupt Control/Status Register (INTCSR) and the general
* status register (depending if the system is driven by interrupts or by polling)
*-----------------------------------------------------------------------------
*/
typedef struct _INTCSR_REG {
    QuantisRegValue  IrEmp       : 1;  /* bit 0 Interrupt Request FIFO Empty mas */
    QuantisRegValue  IrFl1       : 1;  /* bit 1 Interrupt Request FIFO Flag 1 mask (buffer 1/4 full */
    QuantisRegValue  IrFl2       : 1;  /* bit 2 Interrupt Request FIFO Flag 2 mask (buffer 1/2 full */
    QuantisRegValue  IrFl3       : 1;  /* bit 3 Interrupt Request FIFO Flag 3 mask (buffer 3/4 full */
    NotUsed          NotUsed1    : 2;  /* bits 4-5 Reserved */
    QuantisRegValue  IrFull      : 1;  /* bit 6 Interrupt Request FIFO ful */
    QuantisRegValue  IrErr       : 1;  /* bit 7 Interrupt Request FIFO Error mas */
    QuantisRegValue  IrErrWR     : 1;  /* bit 8 Interrupt Request FIFO Write Error mas */
    QuantisRegValue  IrErrRD     : 1;  /* bit 9 Interrupt Request FIFO READ Error mas */
    NotUsed          NotUsed2    : 22; /* bits 10-32 Reserved */
} INTCSR_REG, * PINTCSR_REG;

/* Define when a interrupt has arose (all the one bits for intterupt) */
#define IS_QUANTIS_INTERRUPT 0x3CF
/* Bits for the individual flags in the FIFO read status register and interrupt register */
#define QUANTIS_FIFO_EMPTY        1
#define QUANTIS_FIFO_FL1          (1 << 1)
#define QUANTIS_FIFO_FL2          (1 << 2)
#define QUANTIS_FIFO_FL3          (1 << 3)
#define QUANTIS_FIFO_FULL         (1 << 6)
#define QUANTIS_FIFO_ERROR        (1 << 7)
#define QUANTIS_FIFO_WRITE_ERROR  (1 << 8)
#define QUANTIS_FIFO_READ_ERROR   (1 << 9)

/* Define a value representing all the possible interrupts */
#define IS_QUANTIS_INTERRUPT 0x3CF

/**
 * Some registers used to describe or specify capabilities for quantis modules
 * are composed of four 1-byte for each of the (possible) module.
 * 
 *  31     24 23     16 15      8 7       0   bits
 * +---------+---------+---------+---------+
 * | Module3 | Module2 | Module1 | Module0 |
 * +---------+---------+---------+---------+
*/ 

/* This type is very specific to types describing the Quantis modules. 
   Only the bits 0-3 of the integer are used and they
   represent Quantis modules (they can be at most 4 modules per card)
*/
typedef u_int32_t ModuleMask;

/* Value to detect the different module in a mask */
#define MODULE1 1
#define MODULE2 2
#define MODULE3 4
#define MODULE4 8

#ifndef _KERNEL
/* This small function returns the number of modules in a mask */
static unsigned char GetModule(ModuleMask mask){
    unsigned char NbModules=0;
    if (mask & MODULE1) {NbModules++;}
    if (mask & MODULE2) {NbModules++;};
    if (mask & MODULE3) {NbModules++;};
    if (mask & MODULE4) {NbModules++;};
   
    return NbModules;
}
#endif

/*
*-----------------------------------------------------------------------------
* Define the Quantis modules status register (QMSR)
* a board can be made of 1 to 4 Quantis number generators. Each module
* can be hardware activated (it is activated when the module is build on the board)
* and hardware in low or high status (low status is a transitional status when
* random numbers can not be used during hardware initialisation). In addition
* to the hardware status, each module can be deactivated by setting a flag in a register.
* A transitionnal status can then occure before random generator can be used again.
* Finally, a software test mode is available.
*
* The Quantis modules status register describes all the status possible
* by devoting one char (8bits) to each modules (at most 4 modules are
* available by card)
* -----------------------------------------------------------------------------
*/

typedef struct _QMS_REG {
    QuantisRegValue M0SD             : 1; /* bit 0 Software Shutdown of the Module 0
                                          * 0 = Software Shutdown of Module 0 is disabled
                                          * 1 = Software Shutdown of Module 0 is enabled */
    QuantisRegValue M0EN             : 1; /* bit 1 Module 0 Software Enable
                                          * 0 = Data Acquisition of the Module 0 is deactivated
                                          * 1 = Data Acquisition of the Module 0 is activated */
    QuantisRegValue M0TM             : 1; /* bit 2 Test Mode Enable on the Module 0
                                          * 0 = Test Mode of Module 0 is disabled
                                          * 1 = Test Mode of Module 0 is enabled */
    NotUsed         Res0             : 3; /* bit 3-5 Reserved */
    QuantisRegValue M0HEN            : 1; /* bit 6 Hardware Enable of Module 0
                                          * 0 = Hardware Module 0 is not present
                                          * 1 = Hardware Module 0 is present */
    QuantisRegValue M0HST            : 1; /* bit 7 Hardware Status of Module 0
                                          * 0 = Hardware Status of Module 0 is low
                                          * 1 = Hardware Status of Module 0 is high */
    /* Similar values for the second module */
    QuantisRegValue M1SD             : 1;  /* bit 8 */
    QuantisRegValue M1EN             : 1;  /* bit 9 */
    QuantisRegValue M1TM             : 1;  /* bit 10 */
    NotUsed         Res1             : 3;  /* bit 11-13 */
    QuantisRegValue M1HEN            : 1;  /* bit 14 */
    QuantisRegValue M1HST            : 1;  /* bit 15 */
    /* Similar values for the third module */
    QuantisRegValue M2SD             : 1;  /* bit 16 */
    QuantisRegValue M2EN             : 1;  /* bit 17 */
    QuantisRegValue M2TM             : 1;  /* bit 18 */
    NotUsed         Res2             : 3;  /* bit 19-21 */
    QuantisRegValue M2HEN            : 1;  /* bit 22 */
    QuantisRegValue M2HST            : 1;  /* bit 23 */
    /* Similar values for the fourth module */
    QuantisRegValue M3SD             : 1;  /* bit 24 */
    QuantisRegValue M3EN             : 1;  /* bit 25 */
    QuantisRegValue M3TM             : 1;  /* bit 26 */
    NotUsed         Res3             : 3;  /* bit 27-29 */
    QuantisRegValue M3HEN            : 1;  /* bit 30 */
    QuantisRegValue M3HST            : 1;  /* bit 31 */
} QMS_REG, * PQMS_REG;

/*
  This type of register is similar to the Quantis Module Status Register,
  but the harware bits are not defined since they can not be set by software.
  The "Quantis Module Register" matches the "IDQ MODULE X CONTROL Enable/Disable Register".
  In the "disable" register, the value 0 has no effect and the value 1 disable the 
  corresponding bit while the opposite is true in the "enable" register.
*/
typedef struct _QM_REG {
    QuantisRegValue M0SD             : 1;  /* bit 0 Software Reset of the Module 0 */
    QuantisRegValue M0SEN            : 1;  /* bit 1 Module 0 Software Enable/Disable  of the module 0 */
    QuantisRegValue M0TM             : 1;  /* bit 2 Test Mode Enable/Disable on the Module 0 */
    NotUsed         Res0             : 5;  /* bit 3-7 Reserved */
    /* Similar values for the second module */
    QuantisRegValue M1SD             : 1;  /* bit 8 */
    QuantisRegValue M1EN             : 1;  /* bit 9 */
    QuantisRegValue M1TM             : 1;  /* bit 10 */
    NotUsed         Res1             : 5;  /* bit 11-15 */
    /* Similar values for the third module */
    QuantisRegValue M2SD             : 1;  /* bit 16 */
    QuantisRegValue M2EN             : 1;  /* bit 17 */
    QuantisRegValue M2TM             : 1;  /* bit 18 */
    NotUsed         Res2             : 5;  /* bit 19-23 */
    /* Similar values for the fourth module */
    QuantisRegValue M3SD             : 1;  /* bit 24 */
    QuantisRegValue M3EN             : 1;  /* bit 25 */
    QuantisRegValue M3TM             : 1;  /* bit 26 */
    NotUsed         Res3             : 5;  /* bit 27-31 */
} QM_REG, * PQM_REG;
 
/* Bit number of each bit with a specific meaning for a module in a byte of a register */
#define MX_SD 0  /* Software shutdown */
#define MX_SEN 1 /* Software enable */
#define MX_TM 2  /* Test mode enable */
#define MX_HEN 6 /* Hardware enable */
#define MX_HST 7 /* Hardware status */

QUANTIS_EXTERN_C_END

#endif
