//-----------------------------------------------------------------------------
// Gestion de fichiers
//-----------------------------------------------------------------------------

#ifndef __FILES_H__
#define __FILES_H__

#include "cake.h"
#include "consts.h"
#include <stdio.h>

#ifdef WIN32
  #include <io.h>
#else
  #include <dirent.h>
  #include <time.h>

  /**
   * Following stuff was taken from io.h
   */
  #ifndef _FSIZE_T_DEFINED
    typedef unsigned long _fsize_t; /**< Could be 64 bits for Win32 */
    #define _FSIZE_T_DEFINED
  #endif

  struct _finddata_t
  {
    unsigned    attrib;
    time_t      time_create;    /**< -1 for FAT file systems */
    time_t      time_access;    /**< -1 for FAT file systems */
    time_t      time_write;
    _fsize_t    size;
    char        name[260];
  };
#endif

#define MAX_FILEPATH    1024
#define FILE_NAME     256

typedef struct PackedFile_s
{
  char      name[1024];     /**< @test Static allocation */
  int       offset;
  int       size;
} PackedFile;

typedef struct Pack_s
{
  char *    name;
  int     nFiles;
  PackedFile *files;
  Pack_s *  next;
} Pack;

//-----------------------------------------------------------------------------

extern char* UnZip(FILE *h);

#ifdef WIN32
  #pragma pack( push, before_ZIPStructs )
  #pragma pack(1)
#else
  #pragma pack(push, 1)
#endif

#define ZIPHeaderSig  0x04034b50
typedef struct
{
  LONG  Signature;       //  (0x04034b50)
  WORD  Version;
  WORD  Flags;
  WORD  Method;
  LONG  LastMod;
  LONG  CRC32;
  LONG  CompressedSize;
  LONG  UnCompressedSize;
  WORD  FileNameLength;
  WORD  ExtraLength;
} ZIPHeader;

#define ZIPCtrlHeaderSig  0x02014b50
typedef struct
{
  LONG  Signature;       //  (0x02014b50)
  WORD  VersionMade;
  WORD  VersionNeeded;
  WORD  Flags;
  WORD  Method;
  LONG  LastMod;
  LONG  CRC32;
  LONG  CompressedSize;
  LONG  UnCompressedSize;
  WORD  FileNameLength;
  WORD  ExtraLength;
  WORD  CommentLength;
  WORD  StartDisk;
  WORD  IniternalAttribs;
  LONG  ExternalAttribs;
  LONG  Offset;
} ZIPCtrlHeader;

#define ZIPEndSig     0x06054b50
typedef struct
{
  LONG  Signature;       //  (0x06054b50)
  WORD  DiskNumber;
  WORD  StartDiskNumber;
  WORD  FilesOnDisk;
  WORD  Files;
  LONG  Size;
  LONG  Offset;
  WORD  CommentLength;
} ZIPEnd;

#ifdef WIN32
  #pragma pack( pop, before_ZIPStructs )
#else
  #pragma pack( pop )
#endif

char * f_Name(char *path);
void f_StripName(char *path);
char * f_Extension(char *name);
void f_StripExtension (char *path);

/**
 * Display in console the content of pak files.
 * FileEnvironnment must be created and initialized.
 */
void getPAKContent(char *pak_name = NULL,
           char *file_type = ".*",
           char *grep = NULL);

/**
 * Write a list of all maps available with levelshot in console.
 * FileEnvironnment must be created and initialized.
 */
void logPreviews(void);

/**
 * Checks if a file exists, inside or outside any package.
 * The file is searched from current path.
 * @param file_name the file name to search
 * @param pak_search an integer value that indicates if the function must look in pak files
 * @param strip_extension the function must strip the extension for search (note
 *        that it as only effect with files searched in pak files)
 * @return an integer (0 or 1) that indicates if file exists (1) or not (0)
 */
int FileExist(char *file_name, int pak_search = 1, int strip_extension = 1);

inline void f_CorrectSlashInName(char *name);

//-----------------------------------------------------------------------------
// SearchFile
//-----------------------------------------------------------------------------

/**
 * Typical string list.
 */
typedef struct _s
{
  char path[PATH_LENGTH];
  _s *next;
} search_path;

/**
 * Searchs for a filenames given a pattern.
 */
class SearchFile
{
  public:
    SearchFile(char *pattern="*");
    ~SearchFile(void);

    char *FirstFile(void);
    char *NextFile(void);

    char *FirstFile(const char *path);
    char *NextFile(const char *path);

  private:
    #ifdef WIN32
      int handle;           /**< io handle */
    #else
      DIR *handle;
    #endif
    char *pattern;
    char *temp_pat;         /**< last pattern used. */
    search_path *last;        /**< last path searched. */
    char lastpath[MAX_FILEPATH];  /**< this will save the name of the last path until f_FindNext is called. */
};

//-----------------------------------------------------------------------------
//  FileEnvironment
//-----------------------------------------------------------------------------
/**
 * Global file acces functions
 */
class FileEnvironment
{
  public:
    static void Init(void);
    static void Shut(void);
    static void AddPath(const char *path);
    static void LoadPacks(void);
    static search_path *Search;
    static char main_path[MAX_FILEPATH];

    static void dumpEnvironment(void);
};

//-----------------------------------------------------------------------------
// VFile
//-----------------------------------------------------------------------------
/**
 * Read-only virtual file
 */
class VFile : public FileEnvironment
{
  public:
    VFile(const char *name, bool pak_search=true, const char *pakname=NULL);
    ~VFile(void);
    char  fname[FILE_NAME];
    BYTE *  mem;
    int   size;
    int   error;
};

//-----------------------------------------------------------------------------
// LumpFile
//-----------------------------------------------------------------------------

/**
 * Read-only virtual file with lump header and parsing support (tipical q1,
 * q2 or q3 files).
 */
class LumpFile : public VFile
{
  public:
    LumpFile(const char *name, int id, int ver, const int num_lumps);
    ~LumpFile(void);
    
    int ReadLump(int Lump, void** mem, size_t elem);

  protected:

    struct header
    {
      int id, ver;
      struct { int fileofs, filelen; } lump[];  // is that allowed on most compilers ???
    } *head;
};

//-----------------------------------------------------------------------------
// Config File
//-----------------------------------------------------------------------------
class ConfigFile : public VFile
{
  public:
    ConfigFile(char *name);
};

//-----------------------------------------------------------------------------
// Shader loading
//-----------------------------------------------------------------------------

#if SHADER_USE_REFERENCES
  /** Structure that stores the informations about shaders. */
  typedef struct _shader_ref_list
  {
    char shader_name[SHADER_NAME_MAX_LENGTH];
    char file_name[SHADER_NAME_MAX_LENGTH];
    char pak_name[SHADER_NAME_MAX_LENGTH];
    int offset;           /**< Position of the beginning of the
                     *   shader. First char should be '{' */
    _shader_ref_list *next;
  } shader_ref_list;
  shader_ref_list *RefForShader(const char *shader);
#else
  /** Structure that stores the name of all shader files. */
  typedef struct _shaders_file_list
  {
    char file_name[SHADER_NAME_MAX_LENGTH];
    _shaders_file_list *next;
  } shaders_file_list;
  char *GetFirstShadersFile(void);
  char *NextShadersFile(void);
#endif

void InitializeShaders();
void DestroyShaderList(void);

//-----------------------------------------------------------------------------
// Bitmap file writer
//-----------------------------------------------------------------------------
// Copyright (C) 2000 Nicholas Anderson
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

typedef unsigned char BYTE;
typedef short int SHORT;
typedef long int LONG;
typedef unsigned short int WORD;
typedef unsigned long int DWORD;

typedef struct TEXTURE_BMP{
  int w, h;
  unsigned char *data;
} TEXTURE_BMP;

typedef struct _Win3xBitmapHeader
{
  WORD  ImageFileType;
  DWORD FileSize;
  WORD  Reserved1;
  WORD  Reserved2;
  DWORD ImageDataOffset;

} WIN3XHEAD;

typedef struct _Win3xBitmapInfoHeader
{
  DWORD HeaderSize;
  LONG  ImageWidth;
  LONG  ImageHeight;
  WORD  NumberOfImagePlanes;
  WORD  BitsPerPixel;
  DWORD CompressionMethod;
  DWORD SizeOfBitmap;
  LONG  HorzResolution;
  LONG  VertResolution;
  DWORD NumColorsUsed;
  DWORD NumSignificantColors;

} WIN3XINFOHEAD;

typedef struct _Win3xPixelData
{
  BYTE r;
  BYTE g;
  BYTE b;
} PAL;
int WriteBitmapFile(char *filename, int width, int height, int bpp, unsigned char *imageData);

#endif  /* __FILES_H__ */
