//-----------------------------------------------------------------------------
// Fonctions mathmatiques
//-----------------------------------------------------------------------------

#ifndef __MATH_H__
#define __MATH_H__

#include "types.h"
#include <math.h>
#include <stdlib.h>   // for rand

// Enable assembly code for win32 compiler
#ifdef WIN32
  #define ASM 1
#else
  #define ASM 0
#endif

#ifndef M_PI      // may already been defined
  #define M_PI  3.1415926535897932384626433832795f    // matches value in gcc v2 math.h
#endif
#define M_TWO_PI  6.283185307179586476925286766559f
#define M_180_PI  57.295779513082320876798154814105f
#define M_PI_180  0.017453292519943295769236907684886f

#define DEG2RAD(a) (a*M_PI_180)
#define RAD2DEG(a) (a*M_180_PI)

#define MAKERGB(v,r,g,b) (v[0]=r;v[1]=g;v[2]=b)
#define MAKERGBA(v,r,g,b,a) (v[0]=r;v[1]=g;v[2]=b;v[3]=a)

// Integer Math
/**
 * Find the closest power of 2 that is >= N.
 */
DWORD NextPowerOfTwo(DWORD N);
DWORD Log2(DWORD val);

// vector encoding and decoding
//#define NUMVERTEXNORMALS  216
//extern  vec3_t  bytedirs[NUMVERTEXNORMALS];
//int DirToByte( vec3_t dir );
//void ByteToDir( int b, vec3_t dir );

// Floating Point Math
#define FastTan(a) (FastSin(a)/FastCos(a))
#ifdef WIN32
  float __fastcall FastAbs(float a);
  float __fastcall FastSin(float a);
  float __fastcall FastCos(float a);
  float __fastcall InverseSqrt(float a);
#else
  float FastAbs(float a);
  float FastSin(float a);
  float FastCos(float a);
  float InverseSqrt(float a);
#endif

#ifndef min
  #define min(a,b) (a<b?a:b)
#endif

#ifndef max
  #define max(a,b) (a>b?a:b)
#endif

#define bound(a,b,c) ((a) >= (c) ? (a) : (b) < (a) ? (a) : (b) > (c) ? (c) : (b))

// clamps a (must be lvalue) to [b..c] range
#define clamp(a,b,c) ((b) >= (c) ? (a)=(b) : (a) < (b) ? (a)=(b) : (a) > (c) ? (a)=(c) : (a))

// Reciprocal square root
float rSqrt(float number);

// Aproximations:
#define Sqrt(n)   (n*rSqrt(n))
#ifdef WIN32
  float __fastcall FastSqrt(float a);
#else
  float FastSqrt(float a);
#endif

#define rnd(a,b) ((b-(a))*((long double) rand()/32767)+(a))
#define random()  ((rand () & 0x7fff) / ((float)0x7fff))
#define crandom() (2.0 * (random() - 0.5))

// Vector Math

//  3d Vector macros:
#define PlaneDiff(point,plane) (((plane)->type < 3 ? (point)[(plane)->type] : DotProduct((point), (plane)->normal)) - (plane)->dist)

#define DotProduct(x,y)     ((x)[0]*(y)[0]+(x)[1]*(y)[1]+(x)[2]*(y)[2])
#define CrossProduct(v1,v2,cross) ((cross)[0]=(v1)[1]*(v2)[2]-(v1)[2]*(v2)[1],(cross)[1]=(v1)[2]*(v2)[0]-(v1)[0]*(v2)[2],(cross)[2]=(v1)[0]*(v2)[1]-(v1)[1]*(v2)[0])

#define PlaneDiff(point,plane) (((plane)->type < 3 ? (point)[(plane)->type] : DotProduct((point), (plane)->normal)) - (plane)->dist)

#define VectorDot(x,y)        ((x)[0]*(y)[0]+(x)[1]*(y)[1]+(x)[2]*(y)[2])
#define VectorMult(a,b,c)     ((c)[0]=(a)[2]*(b)[3]-(a)[3]*(b)[2],(c)[1]=(a)[3]*(b)[1]-(a)[1]*(b)[3],(c)[2]=(a)[1]*(b)[2]-(a)[2]*(b)[1])
#define VectorSub(a,b,c)      ((c)[0]=(a)[0]-(b)[0],(c)[1]=(a)[1]-(b)[1],(c)[2]=(a)[2]-(b)[2])
#define VectorAdd(a,b,c)      ((c)[0]=(a)[0]+(b)[0],(c)[1]=(a)[1]+(b)[1],(c)[2]=(a)[2]+(b)[2])
#define VectorCopy(a,b)       ((b)[0]=(a)[0],(b)[1]=(a)[1],(b)[2]=(a)[2])
#define VectorScale(v, s, o)    ((o)[0]=(v)[0]*(s),(o)[1]=(v)[1]*(s),(o)[2]=(v)[2]*(s))
#define VectorMA(v, s, b, o)    ((o)[0]=(v)[0]+(b)[0]*(s),(o)[1]=(v)[1]+(b)[1]*(s),(o)[2]=(v)[2]+(b)[2]*(s))
#define VectorAvrg(a,b,c)     ((c)[0]=((a)[0]+(b)[0])/2,(c)[1]=((a)[1]+(b)[1])/2,(c)[2]=((a)[2]+(b)[2])/2)
#define VectorFakeMod(v)      (v[0]*v[0]+v[1]*v[1]+v[2]*v[2])
#define VectorMDot(a, b, c)     (a[0]*(b[0]-c[0])+a[1]*(b[1]-c[1])+a[2]*(b[2]-c[2]))
#define VectorClear(a)        ((a)[0]=(a)[1]=(a)[2]=0)
#define VectorNegate(a,b)     ((b)[0]=-(a)[0],(b)[1]=-(a)[1],(b)[2]=-(a)[2])
#define VectorSet(v, x, y, z)   ((v)[0]=(x), (v)[1]=(y), (v)[2]=(z))
#define VectorCompare(v1,v2)    ((v1)[0]==(v2)[0] && (v1)[1]==(v2)[1] && (v1)[2]==(v2)[2])
#define VectorLength(v)       (sqrtf(DotProduct((v),(v))))
#define VectorInverse(v)      ((v)[0]=-(v)[0],(v)[1]=-(v)[1],(v)[2]=-(v)[2])
#define Vector4Set(v, a, b, c, d) ((v)[0]=(a),(v)[1]=(b),(v)[2]=(c),(v)[3]=(d))
#define Vector4Copy(a,b)      ((b)[0]=(a)[0],(b)[1]=(a)[1],(b)[2]=(a)[2],(b)[3]=(a)[3])
#define Vector4Scale(v,s,o)     ((o)[0]=(v)[0]*(s),(o)[1]=(v)[1]*(s),(o)[2]=(v)[2]*(s),(o)[3]=(v)[3]*(s))
#define Vector4Add(a,b,c)     ((c)[0]=(((a[0])+(b[0]))),(c)[1]=(((a[1])+(b[1]))),(c)[2]=(((a[2])+(b[2]))),(c)[3]=(((a[3])+(b[3])))) 
#define Vector4Avg(a,b,c)     ((c)[0]=(((a[0])+(b[0]))*0.5f),(c)[1]=(((a[1])+(b[1]))*0.5f),(c)[2]=(((a[2])+(b[2]))*0.5f),(c)[3]=(((a[3])+(b[3]))*0.5f)) 
#define Vector4Clear(a)       ((a)[0]=(a)[1]=(a)[2]=(a)[3]=0)

#define Vector2Set(v, x, y)     ((v)[0]=(x),(v)[1]=(y))
#define Vector2Clear(a)       ((a)[0]=(a)[1]=0)
#define Vector2Copy(a,b)      ((b)[0]=(a)[0],(b)[1]=(a)[1])
#define Vector2Avg(a,b,c)     ((c)[0]=(((a[0])+(b[0]))*0.5f),(c)[1]=(((a[1])+(b[1]))*0.5f)) 
#define SnapVector(v)       {v[0]=(int)v[0];v[1]=(int)v[1];v[2]=(int)v[2];}

//  2d Vector macros:
#define TexcoordAdd(a,b,c)      ((c)[0]=(a)[0]+(b)[0],(c)[1]=(a)[1]+(b)[1])
#define TexcoordScale(v, s, o)    ((o)[0]=(v)[0]*(s),(o)[1]=(v)[1]*(s))
#define TexcoordCopy(a,b)     ((b)[0]=(a)[0],(b)[1]=(a)[1])
#define TexcoordClear(a)      ((a)[0]=(a)[1]=0)
#define TexcoordSet(v, x, y)    ((v)[0]=(x),(v)[1]=(y))

//  4d Vector macros:
#define ColorAdd(a,b,c)       ((c)[0]=(a)[0]+(b)[0],(c)[1]=(a)[1]+(b)[1],(c)[2]=(a)[2]+(b)[2],(c)[3]=(a)[3]+(b)[3])
#define ColorScale(v, s, o)     ((o)[0]=(v)[0]*(s),(o)[1]=(v)[1]*(s),(o)[2]=(v)[2]*(s),(o)[3]=(v)[3]*(s))
#define ColorCopy(a,c)        ((c)[0]=(a)[0],(c)[1]=(a)[1],(c)[2]=(a)[2],(c)[3]=(a)[3])
#define ColorSet(v, a, b, c, d)   ((v)[0]=(a),(v)[1]=(b),(v)[2]=(c),(v)[3]=(d))
#define ColorClear(a)       ((a)[0]=(a)[1]=(a)[2]=(a)[3]=0)

// BBox macros:
#define BoxCopy(a,b)        ((b)[0]=(a)[0],(b)[1]=(a)[1],(b)[2]=(a)[2],(b)[3]=(a)[3],(b)[4]=(a)[4],(b)[5]=(a)[5])

#define min3(a,b,c)         (min(min(a,b),c))
#define max3(a,b,c)         (max(max(a,b),c))

// optimized dot product
#if 0
#ifndef DotProduct
  #if ASM
    #pragma warning (disable: 4035)
    //__declspec( naked )
    float __cdecl DotProduct(const vec3_t v1, const vec3_t v2);
    #pragma warning( default: 4035 )
  #else
    #ifdef WIN32
      float __cdecl DotProduct(const vec3_t v1, const vec3_t v2);
    #else
      float DotProduct(const vec3_t v1, const vec3_t v2);
    #endif
  #endif
#endif
#endif

// Matrix initialisation
void InitMat3x3(float *A);
void InitMat4x4(float *A);

// Matrix multiplication
void MultMat3x3(float *A, float *B, float *C);
void MultMat4x4(float *A, float *B, float *C);

// Vector and matrix multiplication
void MultVect3x3(float *A, float *v, float *dest);
void MultVect4x4(float *A, float *v, float *dest);

/**
 * Fast normalization of 3 component vector (does not test if the vector has 0 length)
 */
void FastNormVect3(float *v);

/**
 * Fast normalization of 2 component vector (does not test if the vector has 0 length)
 */
void FastNormVect2(float *v);

/**
 * Slow normalization that returns the norm
 */
vec_t VectorNormalize(vec3_t v);
vec_t VectorNormalize2(vec3_t v, vec3_t out);
void VectorNormalizeFast(vec3_t v);
float ColorNormalize(vec3_t in, vec3_t out);

/**
 * Cross Product
 */
#ifndef CrossProduct
void CrossProduct(const vec3_t v1, const vec3_t v2, vec3_t cross);
#endif

//-----------------------------------------------------------------------------
// Bezier curves (bugged?) functions
//-----------------------------------------------------------------------------

// Three control point Bezier interpolation
// mu ranges from 0 to 1, start to end of the curve
void BezierCurve3(vec3_t p1, vec3_t p2, vec3_t p3, float mu, vec3_t dest);
// Four control point Bezier interpolation
// mu ranges from 0 to 1, start to end of curve
void BezierCurve4(vec3_t p1, vec3_t p2, vec3_t p3, vec3_t p4, float mu, vec3_t dest);

// General Bezier curve
// Number of control points is n
// 0 <= mu < 1    IMPORTANT, the last point is not computed
void BezierCurveN(vec3_t *p, int n, float mu, vec3_t dest);

//-----------------------------------------------------------------------------
// Bounding box related functions
//-----------------------------------------------------------------------------

/**
 * Calculate the bounding box of a mesh surface
 * @param surf the concerned mesh surface
 */
void CalcFaceBounds(Surface *surf);

/**
 * Redefine a bounding box with a new vertex.
 * The bounding box is only updated if the vertex is out of the box.
 * @param v The vertex to check with bounding box
 * @param bbox The bounding box that will be updated if needed.
 */
void AddPointToBounds(vec3_t v, bboxf_t bbox);
void AddPointToBounds(vec3_t v, vec3_t mins, vec3_t maxs);

/**
 * Clears the values of a bounding box.
 * Note: After having been cleared, a bounding box can't be used for frustum
 * test.
 * @param bbox The bounding box to clear.
 */
void ClearBounds(bboxf_t bbox);
void ClearBounds(vec3_t mins, vec3_t maxs);
bool BoundsIntersect(bboxf_t bbox, vec3_t mins2, vec3_t maxs2);
bool BoundsIntersect(vec3_t mins1, vec3_t maxs1, vec3_t mins2, vec3_t maxs2);
bool BoundsAndSphereIntersect(bboxf_t bbox, vec3_t centre, float radius);
bool BoundsAndSphereIntersect(vec3_t mins, vec3_t maxs, vec3_t centre, float radius);
bool PointInBounds(vec3_t point, bboxf_t bbox);
bool GetIntersection(bboxf_t bbox1, bboxf_t bbox2, bboxf_t dest);

//-----------------------------------------------------------------------------
// Plane operations
//-----------------------------------------------------------------------------

/**
 * Generate a plane given 3 points.
 * The normal will point out of the clock for clockwise ordered points.
 * @return false if the triangle is degenrate
 */
bool PlaneFromPoints(vec4_t plane, const vec3_t a, const vec3_t b, const vec3_t c);
void PlaneFromPoints(vertex_t verts[3], cplane_t *plane);
void PlaneFromPoints(vec3_t verts[3], cplane_t *plane);

void CategorizePlane(cplane_t *plane);

// returns the side of the plane in wich the box is
#if !ASM
  int BoxOnPlaneSide(vec3_t emins, vec3_t emaxs, struct cplane_s *p);
#else
  #pragma warning( disable: 4035 )
  int BoxOnPlaneSide(vec3_t emins, vec3_t emaxs, struct cplane_s *p);
  #pragma warning( default: 4035 )
#endif
#define BOX_ON_PLANE_SIDE(emins, emaxs, p)  \
  (((p)->type < 3)?           \
  (                   \
    ((p)->dist <= (emins)[(p)->type])?  \
      1               \
    :                 \
    (                 \
      ((p)->dist >= (emaxs)[(p)->type])?\
        2             \
      :               \
        3             \
    )                 \
  )                   \
  :                   \
    BoxOnPlaneSide((emins), (emaxs), (p)))

/**
 * Get the distance between a point and a plane along the plane normal.
 * @param v The point vector.
 * @param p The concerned plane.
 * @return The distance between point and plane along the plane normal.
 */
float PointDistance(vec3_t v, cplane_t* p);

// matrix classes

/**
 * 3x2 matrix in homogeneus coordinates.
 * Used for texture transformations.
 */
class Mat3x2
{
  public:

    Mat3x2(void)
    {
    }
    
    Mat3x2(float *p)
    {
      data[0][0] = p[0];
      data[0][1] = p[1];
      data[1][0] = p[2];
      data[1][1] = p[3];
      data[2][0] = p[4];
      data[2][1] = p[5];
    }

    Mat3x2(const Mat3x2& src)
    {
      data[0][0] = src.data[0][0];
      data[0][1] = src.data[0][1];
      data[1][0] = src.data[1][0];
      data[1][1] = src.data[1][1];
      data[2][0] = src.data[2][0];
      data[2][1] = src.data[2][1];
    }

    ~Mat3x2(void)
    {
    }

    const Mat3x2 &operator=(const Mat3x2 &src)
    {
      Copy(src);
      return *this;
    }
    
    const float *GetData(void) const
    {
      return &data[0][0];
    }

    float *GetData(void)
    {
      return &data[0][0];
    }

    void Set(int i1, int i2, float f)
    {
      data[i1][i2] = f;
    }

    float Get(int i1, int i2) const
    {
      return data[i1][i2];
    }

    void Copy (const Mat3x2 &src)
    {
      data[0][0] = src.data[0][0];
      data[0][1] = src.data[0][1];
      data[1][0] = src.data[1][0];
      data[1][1] = src.data[1][1];
      data[2][0] = src.data[2][0];
      data[2][1] = src.data[2][1];
    }

    void Copy (const float *p)
    {
      data[0][0] = p[0];
      data[0][1] = p[1];
      data[1][0] = p[2];
      data[1][1] = p[3];
      data[2][0] = p[4];
      data[2][1] = p[5];
    }

    void Identity(void)
    {
      data[0][0] = 1.0f;
      data[0][1] = 0.0f;
      data[1][0] = 0.0f;
      data[1][1] = 1.0f;
      data[2][0] = 0.0f;
      data[2][1] = 0.0f;
    }

    void Zero(void)
    {
      data[0][0] = 0.0f;
      data[0][1] = 0.0f;
      data[1][0] = 0.0f;
      data[1][1] = 0.0f;
      data[2][0] = 0.0f;
      data[2][1] = 0.0f;
    }

    void Add(const Mat3x2 &a, const Mat3x2 &b)
    {
      data[0][0] = a.data[0][0] + b.data[0][0];
      data[0][1] = a.data[0][1] + b.data[0][1];
      data[1][0] = a.data[1][0] + b.data[1][0];
      data[1][1] = a.data[1][1] + b.data[1][1];
      data[2][0] = a.data[2][0] + b.data[2][0];
      data[2][1] = a.data[2][1] + b.data[2][1];
    }

    void Sub(const Mat3x2 &a, const Mat3x2 &b)
    {
      data[0][0] = a.data[0][0] - b.data[0][0];
      data[0][1] = a.data[0][1] - b.data[0][1];
      data[1][0] = a.data[1][0] - b.data[1][0];
      data[1][1] = a.data[1][1] - b.data[1][1];
      data[2][0] = a.data[2][0] - b.data[2][0];
      data[2][1] = a.data[2][1] - b.data[2][1];
    }

    void ScaleMatrix(const vec2_t v)
    {
      data[0][0] = v[0];
      data[0][1] = 0.0f;
      data[1][0] = 0.0f;
      data[1][1] = v[1];
      data[2][0] = 0.0f;
      data[2][1] = 0.0f;
    }

    void ScaleMatrix(float f)
    {
      data[0][0] = f;
      data[0][1] = 0.0f;
      data[1][0] = 0.0f;
      data[1][1] = f;
      data[2][0] = 0.0f;
      data[2][1] = 0.0f;
    }

    void TranslationMatrix(const vec2_t v)
    {
      data[0][0] = 1.0f;
      data[0][1] = 0.0f;
      data[1][0] = 0.0f;
      data[1][1] = 1.0f;
      data[2][0] = v[0];
      data[2][1] = v[1];
    }

    void RotationMatrix(float theta)
    {
      float cost, sint;

      cost = FastCos(theta);
      sint = FastSin(theta);

      data[0][0] = cost;
      data[0][1] = -sint;
      data[1][0] = sint;
      data[1][1] = cost;
      data[2][0] = 0.0f;
      data[2][1] = 0.0f;
    }

    void Multiply(const Mat3x2 &a, const Mat3x2 &b)
    {
      data[0][0] = a.data[0][0] * b.data[0][0] + a.data[0][1] * b.data[1][0];
      data[0][1] = a.data[0][0] * b.data[0][1] + a.data[0][1] * b.data[1][1];

      data[1][0] = a.data[1][0] * b.data[0][0] + a.data[1][1] * b.data[1][0];
      data[1][1] = a.data[1][0] * b.data[0][1] + a.data[1][1] * b.data[1][1];

      data[2][0] = a.data[2][0] + b.data[2][0];
      data[2][1] = a.data[2][1] + b.data[2][1];
    }
      
    // transform 2d point
    void Mat3x2::TransformPoint(const vec2_t v, vec2_t dest)
    {
      #ifndef WIN32
        dest[0] = v[0]*data[0][0] + v[1]*data[1][0] + data[2][0];
        dest[1] = v[0]*data[0][1] + v[1]*data[1][1] + data[2][1];
      #else
        __asm{
          mov eax, DWORD PTR this
          mov ebx, DWORD PTR v
          mov ecx, DWORD PTR dest

          fld   DWORD PTR [ebx]
          fmul  DWORD PTR [eax+4]     // v[0] * data[0][1]
          fld   DWORD PTR [ebx]
          fmul  DWORD PTR [eax]     // v[0] * data[0][0]
          fld   DWORD PTR [ebx+4]
          fmul  DWORD PTR [eax+12]    // v[1] * data[1][1]
          fld   DWORD PTR [ebx+4]
          fmul  DWORD PTR [eax+8]     // v[1] * data[1][0]
          faddp ST(2), ST         // v[0] * data[1][0] + v[1] * data[0][0]
          faddp ST(2), ST         // v[0] * data[1][1] + v[1] * data[0][1]
          fadd  DWORD PTR [eax+16]    
          fstp  DWORD PTR [ecx]     // dest[0] = v[0]*data[0][0] + v[1]*data[1][0] + data[2][0];
          fadd  DWORD PTR [eax+20]
          fstp  DWORD PTR [ecx+4]     // dest[0] = v[0]*data[0][0] + v[1]*data[1][0] + data[2][0];
        }
      #endif
    }

    // transform 2d point with turbulences
    void Mat3x2::TransformPointX(const vec3_t v, vec2_t dest)
    {
      #ifndef WIN32
        dest[0] = v[0]*data[0][0] + v[1]*data[1][0] + FastSin(v[0])*data[2][0];
        dest[1] = v[0]*data[0][1] + v[1]*data[1][1] + FastSin(v[1])*data[2][1];
      #else
        __asm
        {
          mov eax, DWORD PTR this
          mov ebx, DWORD PTR v
          mov ecx, DWORD PTR dest

          //fld   DWORD PTR [ebx]       // x
          //fld   DWORD PTR [ebx+4]     // x y
          //fld   st(0)           // x y y
          //fxch  st(2)           // y y x
          //fld   st(0)           // y y x x

          fld   DWORD PTR [ebx]
          fmul  DWORD PTR [eax]     // v[0] * data[0][0]

          fld   DWORD PTR [ebx]
          fmul  DWORD PTR [eax+4]     // v[0] * data[0][1]

          fld   DWORD PTR [ebx+4]
          fmul  DWORD PTR [eax+8]     // v[1] * data[1][0]

          fld   DWORD PTR [ebx+4]
          fmul  DWORD PTR [eax+12]    // v[1] * data[1][1]

          fld   DWORD PTR [ebx]
          fsin
          fmul  DWORD PTR [eax+16]    // sin(v[0]) * data[2][0]

          fld   DWORD PTR [ebx+4]
          fsin
          fmul  DWORD PTR [eax+20]    // sin(v[0]) * data[2][1]

          faddp ST(2), ST
          faddp ST(2), ST
          faddp ST(2), ST
          faddp ST(2), ST
          fxch  ST(1)

          fstp  DWORD PTR [ecx]     // dest[0] = v[0]*data[0][0] + v[1]*data[1][0] + data[2][0];
          fstp  DWORD PTR [ecx+4]     // dest[0] = v[0]*data[0][0] + v[1]*data[1][0] + data[2][0];
        }
      #endif
    }

  protected:
    float data[3][2];
};

/**
 * 3x3 matrix class.
 */
class Mat3x3
{
  public:

    Mat3x3(void);
    Mat3x3(float*);
    Mat3x3(const Mat3x3& rhs);
    ~Mat3x3(void);
    const Mat3x3 &operator=(const Mat3x3 &src);

    const float *GetData(void) const
    {
      return &data[0][0];
    }

    float *GetData(void)
    {
      return &data[0][0];
    }

    void  Set(int i1, int i2, float);
    float Get(int i1, int i2) const;


  protected:
    float data[3][3];
};

/**
 * 4x4 matrix class.
 */
class Mat4x4
{
  public:

    Mat4x4(void);
    Mat4x4(float*);
    Mat4x4(const Mat4x4& rhs);
    ~Mat4x4(void);
    const Mat4x4 &operator=(const Mat4x4 &src);
    
    const float *GetData(void) const
    {
      return &data[0][0];
    }

    float *GetData(void)
    {
      return &data[0][0];
    }

    void  Set(int i1, int i2, float);
    float Get(int i1, int i2) const;


  protected:
    float data[4][4];
};

#endif  /* __MATH_H__ */
