#include "md3.h"
#include "entity.h"
#include "files.h"
#include "console.h"
#include "logfile.h"
#include "math.h"
#include "definitions.h"
#include "system.h"

#define MD3_MAGICWORD "IDP3"
#define MD3_VERSION   15

MD3Loader::MD3Loader(Q3BSP *bsp)
{
  pBSP = bsp;
}

MD3Loader::~MD3Loader()
{
  pBSP = NULL;
}

Entity* MD3Loader::LoadMD3(const char *filename)
{
  if (!filename) return NULL;

  char tempname[FILENAME_LENGTH];
  strcpy(tempname, filename);
  int len = (int) strlen(tempname)-4;

  Entity *ent = new Entity;
  if (!ent) ThrowException(ALLOCATION_ERROR, "MD3Loader::LoadMD3.ent");

  // Initialize the entity
  ent->enttype = ET_ITEM;

  cmodel_t* model = NULL;
  int lod = 0;
  while ((model = ImportMD3(tempname)))
  {
    ent->SetModel(model, lod++);  // add the new lod model into entity
  
    strcpy(tempname, filename);
    tempname[len] = '\0';
    sprintf(tempname, "%s_%d.md3", tempname, lod);
  }

  if (!ent->GetModel(0))
  {
    delete ent;
    ent = NULL;
  }

  return ent;
}

void MD3Loader::DumpMD3(const char *filename)
{
  if (!filename) return;

  VFile* file = new VFile(filename);
  if (!file) ThrowException(ALLOCATION_ERROR, "MD3Loader::DumpMD3.file");
  if (!file->mem) return;

  MD3HEADER *head = (MD3HEADER*) file->mem;

  // dump md3 content
  gLogFile->OpenFile();
  gConsole->Insertln("MD3 content (file %s)", filename);
  gConsole->Insertln("\tfileID: %s", head->fileID);
  gConsole->Insertln("\tversion: %d", head->version);
  gConsole->Insertln("\tstrFile: %s", head->strFile);
  gConsole->Insertln("\tnumFrames: %d", head->numFrames);
  gConsole->Insertln("\tnumMeshes: %d", head->numMeshes);
  gConsole->Insertln("\tnumTags: %d", head->numTags);
  gConsole->Insertln("\tnumMaxSkins: %d", head->numMaxSkins);
  gConsole->Insertln("\theaderSize: %d", head->headerSize);
  gConsole->Insertln("\ttagStart: %d", head->tagStart);
  gConsole->Insertln("\ttagEnd: %d", head->tagEnd);
  gConsole->Insertln("\tfileSize: %d", head->fileSize);
  gLogFile->CloseFile();

  delete file;
}

cmodel_t* MD3Loader::ImportMD3(const char *filename)
{
  cmodel_t *model = NULL;
  // TODO: Check if already loaded

  // Loads the given file
    MD3file = new VFile(filename);
  if (!MD3file) ThrowException(ALLOCATION_ERROR ,"MD3Loader::ImportMD3.MD3file");
  if (!MD3file->mem) return NULL;

  // Read the header
  memcpy(&header, MD3file->mem, sizeof(MD3HEADER));

  // Check for compatibility
  if (!header.numFrames) gConsole->Insertln("^5WARNING: Unexpected number of boneframes: %d", header.numFrames);
  if (!strstr(header.fileID, MD3_MAGICWORD)) gConsole->Insertln("^5WARNING: Unexpected magic word: %s", header.fileID);
  if (header.version != MD3_VERSION) gConsole->Insertln("^5WARNING: Unsupported version: %d", header.version);

  // Create the destination model
  int modelnum;
  model = pBSP->CreateNewModel(&modelnum);
  if (!model) ThrowException(NULL_POINTER, "MD3Loader::ImportMD3.model");

  // Initialize the model
  model->numfaces = 0; 
  model->numbrushes = 0;
  model->firstface = 0;
  model->firstbrush = 0;
  model->headnode = 0;
  model->render = true;
  model->entity = NULL;

  // Read all the meshes and add them into the model struct
  ReadMD3Data(model, modelnum);

  // model->mergeVertexBuffers();
  // model->optimize();

  // Calculate the model bounding box
  pBSP->GenerateBoundingBox(model->bbox, model->firstface, model->numfaces);

  delete MD3file;

  return model;
}

void MD3Loader::ReadMD3Data(cmodel_t *model, int modelnum)
{
  if (!model || !MD3file) return;

  float sin_a, cos_a, sin_b, cos_b;

  // We skip bones, tags and don't create any links [not used for items ;] (should we take care of them ??)
  BYTE* meshpos = MD3file->mem+header.tagEnd;     // Store the position of mesh starts

  int facenum;

  for (int m = 0; m < header.numMeshes; ++m)
  {
    MD3MESHINFO meshheader;
    BYTE* filepos = meshpos;
    int i;
    memcpy(&meshheader, filepos, sizeof(MD3MESHINFO));

    // Variables used to get data from file
    MD3SKIN *skins      = new MD3SKIN[meshheader.numSkins];
    if (!skins) ThrowException(ALLOCATION_ERROR, "MD3Loader::ReadMD3Data.skins");

    MD3FACE *triangles    = new MD3FACE[meshheader.numTriangles];
    if (!triangles) ThrowException(ALLOCATION_ERROR, "MD3Loader::ReadMD3Data.triangles");

    MD3TEXCOORD *texcoords  = new MD3TEXCOORD[meshheader.numVertices];
    if (!texcoords) ThrowException(ALLOCATION_ERROR, "MD3Loader::ReadMD3Data.texcoords");

    MD3TRIANGLE *vertices = new MD3TRIANGLE[meshheader.numVertices*meshheader.numMeshFrames];
    if (!vertices) ThrowException(ALLOCATION_ERROR, "MD3Loader::ReadMD3Data.vertices");

    // Read in the skin information
    filepos = meshpos + meshheader.headerSize;
    memcpy(skins, filepos, meshheader.numSkins*sizeof(MD3SKIN));
    
    // Read in the triangle/face data
    filepos = meshpos + meshheader.triStart;
    memcpy(triangles, filepos, meshheader.numTriangles*sizeof(MD3FACE));

    // Read in the texture coordinates
    filepos = meshpos + meshheader.uvStart;
    memcpy(texcoords, filepos, meshheader.numVertices*sizeof(MD3TEXCOORD));

    // Read in the vertex/face information
    filepos = meshpos + meshheader.vertexStart;
    memcpy(vertices, filepos, meshheader.numVertices*meshheader.numMeshFrames*sizeof(MD3TRIANGLE));

    // Create our destination mesh structure (it is actually only one surface)
    Surface *mesh = pBSP->CreateNewSurface(FACETYPE_MESH, &facenum);
    if (!mesh)
    {
      gConsole->Insertln("^1ERROR: BSP returned a NULL surface");
      return;
    }
    mesh->Init(meshheader.numMeshFrames);   // Create the levels
    mesh->model = modelnum;

    for (int l = 0; l < meshheader.numMeshFrames; ++l)
    {
      // Create destination buffers
      mesh->numverts[l] = meshheader.numVertices;
      mesh->firstvert[l] = new vertex_t[mesh->numverts[l]];
      mesh->numelems[l] = 3*meshheader.numTriangles;
      mesh->firstelem[l] = new int[mesh->numelems[l]];

      // Copy source data to destination data
      for (i = 0; i < meshheader.numVertices; ++i)
      {
        mesh->firstvert[l][i].v_point[0] = vertices[l*meshheader.numMeshFrames+i].vertex[0]/64.f;
        mesh->firstvert[l][i].v_point[1] = vertices[l*meshheader.numMeshFrames+i].vertex[1]/64.f;
        mesh->firstvert[l][i].v_point[2] = vertices[l*meshheader.numMeshFrames+i].vertex[2]/64.f;
  
        sin_a = (float)vertices[l*meshheader.numMeshFrames+i].normal[0]*M_PI/128.f;
        cos_a = FastCos(sin_a);
        sin_a = FastSin(sin_a);

        sin_b = (float)vertices[l*meshheader.numMeshFrames+i].normal[1]*M_PI/128.f;
        cos_b = FastCos(sin_b);
        sin_b = FastSin(sin_b);

        VectorSet(mesh->firstvert[l][i].v_norm, cos_b*sin_a, sin_b*sin_a, cos_a);

        mesh->firstvert[l][i].colour[0] = 
        mesh->firstvert[l][i].colour[1] = 
        mesh->firstvert[l][i].colour[2] = 
        mesh->firstvert[l][i].colour[3] = 255;
        mesh->firstvert[l][i].lm_st[0] = mesh->firstvert[l][i].lm_st[1] = 0;

        mesh->firstvert[l][i].tex_st[0] = texcoords[i][0];
        mesh->firstvert[l][i].tex_st[1] = texcoords[i][1];
      }

      for (i = 0; i < meshheader.numTriangles; ++i)
      {
        mesh->firstelem[l][3*i+0] = triangles[i][0];
        mesh->firstelem[l][3*i+1] = triangles[i][1];
        mesh->firstelem[l][3*i+2] = triangles[i][2];
      }
    }

    // Multiskin is not supported (why is it used for ??)
    // Surface can only have one shader, other shaders are loaded all the same
    if (meshheader.numSkins)
    {
      for (int s = 0; s < meshheader.numSkins; ++s)
      {
        if (strlen(skins[s]))
        {
          f_StripExtension(skins[s]);
          mesh->shader = pBSP->world->shaders.AddShader(skins[s], 0, 0, mesh->facetype);
        }
        else
        {
          mesh->shader = LoadSkin(meshheader.strName);
        }
      }
    }

    // We don't need this any more
    delete [] skins;
    delete [] triangles;
    delete [] texcoords;
    delete [] vertices;

    // Calc bounding box
    CalcFaceBounds(mesh);

    // Update the model
    ++model->numfaces;

    // Next mesh position
    meshpos += meshheader.meshSize;
  }

  model->firstface = facenum - header.numMeshes + 1;
}

int MD3Loader::LoadSkin(const char *name)
{
  return -1;
}
