//-----------------------------------------------------------------------------
// BSP Structures
//-----------------------------------------------------------------------------

#ifndef __Q3BSP_H__
#define __Q3BSP_H__

#include "cake.h"
#include "types.h"
#include "skybox.h"
#include "surfaceflags.h"
#include "entity.h"
#include "files.h"
#include "bezierpatch.h"
#include "world.h"

#define BSPHEADERID  (*(int*)"IBSP")
#define BSPVERSION 46

// lightmaps
#define LIGHTMAP_BYTES      3
#define LIGHTMAP_WIDTH      128
#define LIGHTMAP_HEIGHT     128
#define LIGHTMAP_SIZE     (LIGHTMAP_WIDTH*LIGHTMAP_HEIGHT*LIGHTMAP_BYTES)

#define MAX_MAP_LIGHTMAPS   (MAX_MAP_LIGHTING / LIGHTMAP_SIZE)

// there shouldn't be any problem with increasing these values at the
// expense of more memory allocation in the utilities
#define MAX_MAP_MODELS    0x400
#define MAX_MAP_BRUSHES   0x8000
#define MAX_MAP_ENTITIES  0x800
#define MAX_MAP_ENTSTRING 0x40000
#define MAX_MAP_SHADERS   0x400

#define MAX_MAP_AREAS   0x100
#define MAX_MAP_FOGS    0x100
#define MAX_MAP_PLANES    0x20000
#define MAX_MAP_NODES   0x20000
#define MAX_MAP_BRUSHSIDES  0x30000
#define MAX_MAP_LEAFS   0x20000
#define MAX_MAP_VERTEXES  0x80000
#define MAX_MAP_FACES   0x20000
#define MAX_MAP_LEAFFACES 0x20000
#define MAX_MAP_LEAFBRUSHES 0x40000
#define MAX_MAP_PORTALS   0x20000
#define MAX_MAP_INDICES   0x80000
#define MAX_MAP_LIGHTING  0x800000
#define MAX_MAP_VISIBILITY  0x200000

#define MAX_CM_AREAS      (MAX_MAP_AREAS)
#define MAX_CM_BRUSHSIDES   (MAX_MAP_BRUSHSIDES << 1)
#define MAX_CM_SHADERS      (MAX_MAP_SHADERS)
#define MAX_CM_PLANES     (MAX_MAP_PLANES << 2)
#define MAX_CM_NODES      (MAX_MAP_NODES)
#define MAX_CM_LEAFS      (MAX_MAP_LEAFS)
#define MAX_CM_LEAFBRUSHES    (MAX_MAP_LEAFBRUSHES)
#define MAX_CM_MODELS     (MAX_MAP_MODELS)
#define MAX_CM_BRUSHES      (MAX_MAP_BRUSHES << 1)
#define MAX_CM_VISIBILITY   (MAX_MAP_VISIBILITY)
#define MAX_CM_FACES      (MAX_MAP_FACES)
#define MAX_CM_LEAFFACES    (MAX_MAP_LEAFFACES)
#define MAX_CM_VERTEXES     (MAX_MAP_VERTEXES)
#define MAX_CM_PATCHES      (0x10000)
#define MAX_CM_PATCH_VERTS    (4096)
#define MAX_CM_ENTSTRING    (MAX_MAP_ENTSTRING)

//-----------------------------------------------------------------------------

/**
 * Q3BSP class.
 * The q3bsp is the main bsp manager. 
 * @bug The map loading bugs sometimes. For example sky in q3dm10 isn't
 *      correctly loaded in some cases. The reload of the map often shows
 *      other loading bugs. Try to see for eventual memory overfilling.
 */
class Q3BSP : public LumpFile
{
  public:
    World*      world;

    /**
     * Constructor will load data from a file and store it in internal
     * variables.
     * @param name the name of file to read
     * @param w a pointer to the parent world object
     * @param res a pointer to an integer that will receive the operation
     *        result (1 if success, 0 if error)
     */
    Q3BSP(char *name, World* w, int *res);
    ~Q3BSP(void);

    /**
     * More initialization.
     * The function must treat data before being capable to draw it. It
     * is also responsible to generate sky.
     */
    void Init(void);

    void Update(void);    /**< Update all the BSP data for a new frame. */

    void Report(void);    /**< Write a report of BSP elements in console. */

    /**
     * Renders the map.
     */
    void Render(void);

    /**
     * Get the entity manager.
     * The entity manager is used for entities support, loading, update, etc.
     * It contains all entities. See EntityManager class for more info.
     * @return A pointer to entities manager structure.
     */
    EntityManager* GetEntities(void);

    /**
     * Check a movement for collisions.
     */
    trace_t CheckMove(vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end, int head_node = 0, int brushmask = MASK_PLAYERSOLID);

    /**
     * Link an entity in the BSP tree.
     * Needs to be called any time an entity changes origin, mins, maxs,
     * or solid. Automatically unlinks if needed.
     * @todo This is not supported yet
     * @param ent The entity that will be added to BSP tree.
     */
    void LinkEntity(Entity* ent);

    /**
     * Unling an entity from the BSP tree.
     * Call before removing an entity, and before trying to move one,
     * so it doesn't clip agains itself.
     * @todo This is not supported yet
     * @param ent The entity that will be removed from BSP tree.
     */
    void UnlinkEntity(Entity *ent);

    /**
     * Return a pointer to an unused model in BSP models
     * This function can be used to add new models into BSP models structure.
     * This is used by md3 loader to directly put the loaded models in the
     * main models table.
     * @param modelnum The index of returned model in BSP models table
     * @return The pointer to allocated model.
     */
    cmodel_t* CreateNewModel(int *modelnum);

    /**
     * Return a pointer to an unused surface in BSP surfaces.
     * This function works on the same manner than the CreateNewModel() function.
     * @param facetype The surface type
     * @param facenum The index of returned surface in BSP surfaces table
     * @return The pointer to allocated surface.
     */
    Surface* CreateNewSurface(int facetype, int *facenum);

    /**
     * Generates a bounding box using BSP faces
     * @param destbox The destination bounding box to build
     * @param firstface The index of first faces used for bbox building
     * @param numfaces The number of faces used for bbox building
     */
    void GenerateBoundingBox(bboxf_t destbox, int firstface, int numfaces);

    /**
     * Display the bounding box of a specified face.
     * This function is only for debug.
     * @param bbox The bounding box to render.
     * @param colorline The color of bounding box lines.
     * @param colorface The color of bounding box faces.
     */
    static void DrawBoundingBox(bboxf_t bbox, colour_t colorline, colour_t colorface);

    /**
     * Display a message on screen.
     * This function is only usable when bsp state is not RUNNING. This is mainly
     * used to display informations on the loading process. The function displays
     * the levelshot and put a message in forground. The message is centered on
     * the line.
     * @param msg The message to display
     * @param line The line to write the message on (-1 => centered on screen height)
     */
    void DisplayLoadingMessage(const char *msg, int line = -1);

    void SetWorldName(const char *name);

  private:

    int FindLeaf(vec3_t p, int headnode = 0); 
    void MarkLeafs(int vis_cluster);
    void WalkTree(int idx, bool checkFrustum);
    void UpdateFaces(void);     /**< Update the faces. */
    void UpdateSky(void);     /**< Update the skybox. */

    void DrawFaces(void);     /**< Draws the faces. */
    void DrawSky(void);       /**< Draws the sky box. */
    void DrawModels(void);      /**< Draws the models. */
    void DrawBoundingBoxes(void); /**< Display the bounding box of patches and meshes. */
    void DrawNormals(void);     /**< Draw the vertice normal */

    void ClearLink(link_t *l);
    void RemoveLink(link_t *l);
    void InsertLinkBefore(link_t *l, link_t *before);


    /**
     * Generates the skybox.
     * The function will find sky shader and associate it with the sky
     * box. It will then store a reference to all surfaces that have
     * sky shader to prepare sky surface list for sky rendering
     * optimization.
     * @see SkyBox
     */
    void GenerateSkyBox(void);

    /**
     * Initialize the surfaces.
     * The function creates the patches and the patch grounps.
     */
    void InitSurfaces(void);

    /**
     * Creates the patches groups.
     * Patches are organized in groups. Every patch of a group is near to
     * the others, so tesselation changes are made together for each patch
     * of the group. This is done for progressive tesselation.
     * @see PatchesGroup
     */
    void CreatePatchesGroups(void);

    /**
     * Callback to sort the faces.
     */
    static int FaceCmp(const void *a, const void *b);

    void LoadEntities(void);      /**< Load entities from BSP file */
    void LoadShaders(void);       /**< Load shaders from BSP file */
    void LoadPlanes(void);        /**< Load planes from BSP file */
    void LoadNodes(void);       /**< Load nodes from BSP file */
    void LoadLeafs(void);       /**< Load leafs from BSP file */
    void LoadLFaces(void);        /**< Load lfaces from BSP file */
    void LoadLBrushes(void);      /**< Load lbrushes from BSP file */
    void LoadModels(void);        /**< Load models from BSP file */
    void LoadBrushes(void);       /**< Load brushes from BSP file */
    void LoadBrushSides(void);      /**< Load brushsides from BSP file */
    void LoadVerts(void);       /**< Load verts from BSP file */
    void LoadElems(void);       /**< Load elems from BSP file */
    void LoadFaces(void);       /**< Load faces from BSP file */
    void LoadLightmaps(void);     /**< Load lightmaps from BSP file */
    void LoadVisibility(void);      /**< Load visibility from BSP file */
    void LoadEffects(void);       /**< Load visibility from BSP file
                       *   @todo Finish effect management. */
    void LoadLightVols(void);     /**< Load visibility from BSP file
                       *   @todo Finish lightvols management */

    int levelshotdetail, levelshot;   /**< Levelshot and levelshotdetails shader index */

    /**
     * Load the levelshot in the framework shaders table.
     * @param map_name The map that is currently being loaded.
     */
    void LoadLevelshot(const char* map_name);

    /**
     * Display the current defined levelshot.
     * The function only map the levelshot on render surface but does not
     * execute buffers swap.
     */
    void DisplayLevelshot(void);

    void SetParent(int node, cnode_t *parent);

    /**
     * Resets the surface rotation.
     * The function resets the surface rotation so the surface has a
     * default orientation. The surface must have normal colliding with
     * Oy axis and logically be in the Oxz plane.
     * <pre>
     *   Note on the Quake 3 axis system
     *
     *  Z /|\     _  Y
     *     |      /|
     *     |  ___/___
     *     | |       |
     *     | |       |  The surface must finally be in the Oxz plane,
     *     | |       |  with normal pointing in Oy direction.
     *     | |_______|
     *     |/              \
     *   --+---------------- X
     *    /|               /
     *
     * </pre>
     * The function is used with surfaces that have Autosprite or
     * Autosprite2 deformation vertex. The goal is to reorient the face
     * to be then able to rotate it correctly. The function also update
     * surface axis for autosprite2 rotation.
     * @param surf the surface to analyse
     * @param type the autosprite type
     */
    void InitAutosprite(Surface *surf, int type);

    /**
     * Check if a point is inside a brush.
     * @param brushnum The index of tested brush.
     * @param p The point position
     * @return A boolean value that is true if the point is in the brush, false if not
     */
    bool PointInsideBrush(int brushnum, vec3_t p);

    /**
     * Generate a bezier patch.
     * The function creates several bezier patches (depending on subdivision levels).
     * @param surf The destination surface (surface that will become bezier patch)
     * @param curvefactor Factor used to perform flatness test.
     */
    void BuildBezierPatch(Surface* surf, int curvefactor);

    // Collision detection stuff
    // @todo Organize this when coldet is finished
    int   leaf_count, leaf_maxcount;
    int   *leaf_list;
    float *leaf_mins, *leaf_maxs;
    int   leaf_topnode;
    void RecursiveHullCheck(float sf, float ef, const vec3_t sp, const vec3_t ep, int node, trace_t *trace_trace);
    void TraceToLeaf(int leafnum, trace_t *trace_trace);
    void TestInLeaf(int leafnum, trace_t *trace_trace);
    void ClipBoxToBrush(cbrush_t *brush, trace_t *trace);
    void ClipBoxToPatch(cbrush_t *brush, trace_t *trace);
    void TestBoxInBrush(cbrush_t *brush, trace_t *trace);
    void TestBoxInPatch(cbrush_t *brush, trace_t *trace);
    int BoxLeafnums_headnode(vec3_t mins, vec3_t maxs, int *list, int listsize, int headnode, int *topnode);
    void BoxLeafnums_r(int nodenum);

    void CreateBrush(cbrush_t *brush, vec3_t *verts, int surface_shader);
    void CreatePatch(cpatch_t *patch, int numverts, vec4_t *verts, int *patch_cp);
    void CreatePatchesForLeafs(void);

    // sizes of the loaded data
    // The sizes are defined by 2 int. The first value is the currently number
    // of data and the second value is the size of allocated memory. The allocated
    // memory can be bigger than what is required.
    int       r_nummodels[2];
    int       r_numverts[2];
    int       r_numplanes[2];
    int       r_numleafs[2];
    int       r_numnodes[2];
    int       r_numshaders[2];
    int       r_numsurfaces[2];   /**< total number of surfaces */
    int       r_numleaffaces[2];
    int       r_numelems[2];
    int       r_numlightmaps[2];
    int       r_numleafbrushes[2];
    int       r_numbrushes[2];
    int       r_numbrushsides[2];
    int       r_numlightvols[2];
    int       r_numeffects[2];
    int       r_numvisibility[2];
    int       r_numleafpatches[2];
    int       r_numfaces[2];
    int       r_numpatches[2];
    int       r_numcpatches[2];
    int       r_nummeshes[2];
    int       r_numflares[2];

    // pointers to loaded data
    vec3_t *    v_xyz;
    texcoord_t *  v_tex_st;
    texcoord_t *  v_lm_st;
    vec3_t *    v_norm;
    colour_t *    v_colour;

    vertex_t *    r_verts;
    cmodel_t *    r_models;
    cleaf_t *   r_leafs;
    cnode_t *   r_nodes;
    shaderref_t * r_shaderrefs;
    int *     r_leaffaces;
    int *     r_elems;
    visibility_t *  r_visibility;
    texinfo *   r_lightmapinfos;
    int *     r_leafbrushes;
    cbrush_t *    r_brushes;
    cbrushside_t *  r_brushsides;
    cplane_t *    r_planes;
    ceffect_t *   r_effects;
    lightvols_t * r_lightvols;
    int *     r_leafpatches;
    cpatch_t *    r_cpatches;     /**< structure used for collision detection with patches */
        
    render_face_s * face_list;

    Surface *   r_surfaces;     /**< table conntaining all surfaces */
    Surface **    r_patches;      /**< table countaining patches surfaces pointers */
    Surface **    r_faces;      /**< table countaining faces surfaces pointers */
    Surface **    r_meshes;     /**< table countaining meshes surfaces pointers */
    Surface **    r_flares;     /**< table countaining flares surfaces pointers */
    int       r_numpatchesgroups[2];
    PatchesGroup *  r_patchesgroups;
    SkyBox *    r_skybox;
    EntityManager * r_entities;

    bboxf_t     bbox;       /**< bsp bounding box */
    
    int       emptyleaf;

    char*     worldname;

    /**
     * Visibility counter.
     * The visibility counter is updated each frame. It is used to know
     * if a node or a leag must be walked or drawn.
     */
    int viscount;
    
    /**
     * Node counter.
     * The node counter is different because nodes only are updated when
     * the camera leaf changes
     */
    int nodecount;

    /**
     * Check counter.
     * This is used to avoid repeated testings in collision tests.
     */
    int checkcount;
};

#endif  /* __Q3BSP_H__ */
