//-----------------------------------------------------------------------------
// Render
//-----------------------------------------------------------------------------

#include "render.h"
#include "commands.h"
#include "console.h"
#include "logfile.h"
#include "overlay.h"
#include "camera.h"
#include "client.h"
#include "timer.h"
#include "vars.h"
#include "math.h"
#include "surfaceflags.h"
#include "mem.h"
#include "system.h"
#include <assert.h>

#include "glsetup/glutils.h"
#ifdef WIN32
  #include "glsetup/glext.h"
#else
  #include <GL/glx.h>
  #include <GL/glext.h>
#endif

PFNGLLOCKARRAYSEXTPROC            glLockArraysEXT;
PFNGLUNLOCKARRAYSEXTPROC          glUnlockArraysEXT;

#ifdef WIN32
  PFNGLACTIVETEXTUREARBPROC       glActiveTextureARB;
  PFNGLCLIENTACTIVETEXTUREARBPROC     glClientActiveTextureARB;
  #define lglGetProcAddress(p)      wglGetProcAddress(p)

  // size of float may change from one platform to one other
  #define XYZ_OFFSET            16    // 4 * sizeof(float)
#else
  #if !defined(GLX_VERSION_1_4)
    #define lglGetProcAddress(p)    glXGetProcAddressARB((byte*) p);
  #else
    #define lglGetProcAddress(p)    glXGetProcAddress((byte*) p);
    #define glActiveTextureARB      glActiveTexture
    #define glClientActiveTextureARB  glClientActiveTexture
  #endif

  // size of float may change from one platform to one other
  #define XYZ_OFFSET            16    // 4 * sizeof(float)
#endif

Var gl_vendor("gl_vendor", "", VF_SYSTEM);
Var gl_renderer("gl_renderer", "", VF_SYSTEM);
Var gl_version("gl_version", "", VF_SYSTEM);
Var gl_extensions("gl_extensions", "", VF_SYSTEM);
Var gl_maxTextureSize("gl_maxTextureSize", 0, VF_SYSTEM);
Var gl_maxTextureUnits("gl_maxTextureUnits", 0, VF_SYSTEM);
Var r_ext_compiledVertexArray("r_ext_compiled_vertex_array", 0, VF_SYSTEM);
Var r_ext_multitexture("r_ext_multitexture", 0, VF_SYSTEM);
Var r_ext_texture_edge_clamp("r_ext_texture_edge_clamp", 0, VF_SYSTEM);
Var r_usedTextureUnits("r_usedTextureUnits", 0, VF_SYSTEM);
Var r_vertexLight("r_vertexLight", 0, VF_PERSISTENT);
Var r_texturebits("r_texturebits", 0);
Var r_clear("r_clear", 0, VF_PERSISTENT);
Var r_filter("r_filter", 1, VF_PERSISTENT);
Var r_quality("r_quality", 2, VF_PERSISTENT | VF_LATCH);
Var r_picmip("r_picmip", 0, VF_PERSISTENT | VF_LATCH);
Var gl_dither("gl_dither", 1, VF_PERSISTENT);
Var v_overbrightbits("v_overbrightbits", 3, VF_LATCH);
Var v_gamma("v_gamma", 1.3375f, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var r_nolightmap("r_nolightmap", 0);
Var r_noshader("r_noshader", 0);
Var r_novertexcolor("r_novertexcolor", 0);
Var r_nocull("r_nocull", 0);
Var r_nodeform("r_nodeform", 0);
Var r_nofog("r_nofog", 0);
Var v_fullscreen("v_fullscreen", 0, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_colorBits("v_colorBits", 32, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_hz("v_hz", 60, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_height("v_height", 480, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_width("v_width", 640, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_left("v_left", 0, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_top("v_top", 0, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_aspect("v_aspect", (float)4/3, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_depthBits("v_depthBits", 32, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_stencilBits("v_stencilBits", 0, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var v_lightness("v_lightness", 0.0f, VF_LATCH);
Var v_mode("v_mode", 0, VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var r_subdivision("r_subdivision", 2, VF_LATCH | VF_PERSISTENT);
Var r_curvefactor("r_curvefactor", 6, VF_LATCH | VF_PERSISTENT);
Var wnd_handle("wnd_handle", 0);
Var sys_buffersize("sys_buffersize", 0, VF_SYSTEM); // size of dynamic buffer
Var sys_maxelems("sys_maxelems", 0, VF_SYSTEM);   // number of elements
Var v_camnear("v_camnear", 4);
Var v_camfar("v_camfar", 50000);

void cmd_setclearcolor(int argc, char *argv[])
{
  if (argc > 4)
  {
    glClearColor((float) atof(argv[1]), (float) atof(argv[2]), (float) atof(argv[3]), (float) atof(argv[4]));
  }
  else gConsole->Insertln("usage: %s <value> <value> <value> <value>", argv[0]);
}

Shader *  State::curr_shader = NULL;
Shader *  State::curr_effect = NULL;
int       State::curr_depthFunc = GL_LEQUAL;
int       State::curr_depthWrite = true;
int       State::curr_blenddst = GL_ONE;
int       State::curr_blendsrc = GL_ZERO;
int       State::curr_culling = CULL_NONE;
int       State::curr_alphaFunc;
int       State::curr_polygonmode = GL_FILL;
int       State::skipping = 0;
int       State::curr_texture[];
int       State::curr_lightmap = ~0;
int       State::pass = 0;
int       State::last_tcgen[MAX_TEXTURE_UNITS];

GLboolean lighting, texture, depthtest;
int texture_unit;

// buffer states
enum
{
  BF_EMPTY,
  BF_FILLING,
  BF_LOCKED
};

// Constructor
Render::Render()
{
  ready                 = 0;

  fogtexture.mem        = NULL;

  swapbuffersfunc       = NULL;

  gridpass              = 0;

  for (int i = 0; i < MAX_TEXTURE_UNITS; ++i)
  {
    buffer[i].st = NULL;
    buffer[i].c = NULL;
  }
  input.c             = NULL;
  input.elems         = NULL;
  input.fog           = NULL;
  input.lm_st         = NULL;
  input.normal        = NULL;
  input.numelems      = 0;
  input.numverts      = 0;
  input.tex_st        = NULL;
  input.xyz           = NULL;

  current_camera      = NULL;

  // Registering variables
  gVars->RegisterVar(gl_vendor);
  gVars->RegisterVar(gl_renderer);
  gVars->RegisterVar(gl_version);
  gVars->RegisterVar(gl_extensions);
  gVars->RegisterVar(gl_maxTextureSize);
  gVars->RegisterVar(gl_maxTextureUnits);
  gVars->RegisterVar(gl_dither);
  gVars->RegisterVar(r_ext_compiledVertexArray);
  gVars->RegisterVar(r_ext_multitexture);
  gVars->RegisterVar(r_ext_texture_edge_clamp);
  gVars->RegisterVar(r_usedTextureUnits);
  gVars->RegisterVar(r_vertexLight);
  gVars->RegisterVar(r_texturebits);
  gVars->RegisterVar(r_quality);
  gVars->RegisterVar(r_picmip);
  gVars->RegisterVar(r_nolightmap);
  gVars->RegisterVar(r_noshader);
  gVars->RegisterVar(r_novertexcolor);
  gVars->RegisterVar(r_filter);
  gVars->RegisterVar(r_nocull);
  gVars->RegisterVar(r_nofog);
  gVars->RegisterVar(v_depthBits);
  gVars->RegisterVar(v_stencilBits);
  gVars->RegisterVar(v_mode);
  gVars->RegisterVar(wnd_handle);
  gVars->RegisterVar(v_overbrightbits);
  gVars->RegisterVar(v_gamma);
  gVars->RegisterVar(v_lightness);
  gVars->RegisterVar(r_nodeform);
  gVars->RegisterVar(r_clear);
  gVars->RegisterVar(v_aspect);
  gVars->RegisterVar(v_fullscreen);
  gVars->RegisterVar(v_hz);
  gVars->RegisterVar(v_colorBits);
  gVars->RegisterVar(v_height);
  gVars->RegisterVar(v_width);
  gVars->RegisterVar(v_top);
  gVars->RegisterVar(v_left);
  gVars->RegisterVar(r_subdivision);
  gVars->RegisterVar(r_curvefactor);
  gVars->RegisterVar(sys_buffersize);
  gVars->RegisterVar(sys_maxelems);
  gVars->RegisterVar(v_camnear);
  gVars->RegisterVar(v_camfar);

  gCommands->AddCommand("setclearcolor", cmd_setclearcolor, "Define the clear color (need to be enabled with r_clear variable).");
}

// Destructor
Render::~Render(void)
{
  for (int i = 0; i < MAX_TEXTURE_UNITS; ++i)
  {
    if (buffer[i].st) cake_free(buffer[i].st);
    if (buffer[i].c) cake_free(buffer[i].c);
  }
  if (input.xyz)    cake_free(input.xyz); input.xyz = NULL;
  if (input.tex_st) cake_free(input.tex_st); input.tex_st = NULL;
  if (input.lm_st)  cake_free(input.lm_st); input.lm_st = NULL;
  if (input.normal) cake_free(input.normal); input.normal = NULL;
  if (input.c)      cake_free(input.c); input.c = NULL;
  if (input.fog)    cake_free(input.fog); input.fog = NULL;
  if (input.elems)  cake_free(input.elems); input.elems = NULL;

  sys_buffersize = 0;
  sys_maxelems = 0;

  if (fogtexture.mem) delete [] fogtexture.mem;
  fogtexture.mem = NULL;

  // Unregister variables
  gVars->UnregisterVar(gl_vendor);
  gVars->UnregisterVar(gl_renderer);
  gVars->UnregisterVar(gl_version);
  gVars->UnregisterVar(gl_extensions);
  gVars->UnregisterVar(gl_maxTextureSize);
  gVars->UnregisterVar(gl_maxTextureUnits);
  gVars->UnregisterVar(gl_dither);
  gVars->UnregisterVar(r_ext_compiledVertexArray);
  gVars->UnregisterVar(r_ext_multitexture);
  gVars->UnregisterVar(r_ext_texture_edge_clamp);
  gVars->UnregisterVar(r_usedTextureUnits);
  gVars->UnregisterVar(r_vertexLight);
  gVars->UnregisterVar(r_texturebits);
  gVars->UnregisterVar(r_clear);
  gVars->UnregisterVar(r_quality);
  gVars->UnregisterVar(r_picmip);
  gVars->UnregisterVar(r_nolightmap);
  gVars->UnregisterVar(r_noshader);
  gVars->UnregisterVar(r_novertexcolor);
  gVars->UnregisterVar(r_filter);
  gVars->UnregisterVar(r_nocull);
  gVars->UnregisterVar(r_nofog);
  gVars->UnregisterVar(v_fullscreen);
  gVars->UnregisterVar(v_width);
  gVars->UnregisterVar(v_height);
  gVars->UnregisterVar(v_top);
  gVars->UnregisterVar(v_left);
  gVars->UnregisterVar(v_aspect);
  gVars->UnregisterVar(v_colorBits);
  gVars->UnregisterVar(v_hz);
  gVars->UnregisterVar(v_depthBits);
  gVars->UnregisterVar(v_stencilBits);
  gVars->UnregisterVar(v_gamma);
  gVars->UnregisterVar(v_mode);
  gVars->UnregisterVar(wnd_handle);
  gVars->UnregisterVar(v_overbrightbits);
  gVars->UnregisterVar(v_lightness);
  gVars->UnregisterVar(r_nodeform);
  gVars->UnregisterVar(r_subdivision);
  gVars->UnregisterVar(r_curvefactor);
  gVars->UnregisterVar(sys_buffersize);
  gVars->UnregisterVar(sys_maxelems);
  gVars->UnregisterVar(v_camnear);
  gVars->UnregisterVar(v_camfar);
  gCommands->RemoveCommand("setclearcolor");
}

// Initialize the render
int Render::Init(swapcommand_t func)
{
  // Initializing driver
  int temp;

  swapbuffersfunc = func;

  gConsole->Insertln("<br/><b>^6----- Initializing driver -------------------------------------</b>");

  // Creates buffers
  gConsole->Insertln("<b>Creating buffers...</b>");
  IncreaseBuffer(BUFFER_SIZE_BLOC<<3);
  IncreaseElems(ELEMS_BLOC<<2);
  //IncreaseBuffer(1000000);
  //IncreaseElems(4000000);

  // get driver infos
  gVars->SetKeyValue("gl_vendor",     (char*)glGetString(GL_VENDOR));
  gVars->SetKeyValue("gl_renderer",   (char*)glGetString(GL_RENDERER));
  gVars->SetKeyValue("gl_version",    (char*)glGetString(GL_VERSION));
  gVars->SetKeyValue("gl_extensions", (char*)glGetString(GL_EXTENSIONS));

  // Check float size
  if (4*sizeof(float) != XYZ_OFFSET)
    ThrowException(DEFAULT_EXCEPTION, "gRender->Init - float should be 4 bytes wide");

  // Report some infos
  glGetIntegerv(GL_MAX_TEXTURE_SIZE, &temp);
  CheckGLError(5);
  gl_maxTextureSize = temp;

  // Initialize extensions:

  // compiled vertex array
  if (CheckExtension("GL_EXT_compiled_vertex_array"))
  {
    glLockArraysEXT = (PFNGLLOCKARRAYSEXTPROC) lglGetProcAddress("glLockArraysEXT");
    glUnlockArraysEXT = (PFNGLUNLOCKARRAYSEXTPROC) lglGetProcAddress("glUnlockArraysEXT");
    r_ext_compiledVertexArray = 1;
    CheckGLError(2);
  }
  else
  {
    r_ext_compiledVertexArray = 0;
    gConsole->Insertln("^1Compiled vertex array extension (GL_EXT_compiled_vertex_array) not supported");
  }

  // multitexture
  if (CheckExtension("GL_ARB_multitexture"))
  {
    r_ext_multitexture = 1;

    #ifdef WIN32
      glActiveTextureARB    = (PFNGLACTIVETEXTUREARBPROC) lglGetProcAddress("glActiveTextureARB");
      glClientActiveTextureARB= (PFNGLCLIENTACTIVETEXTUREARBPROC) lglGetProcAddress("glClientActiveTextureARB");
    #endif
    glGetIntegerv(GL_MAX_TEXTURE_UNITS_ARB, &temp);
    gl_maxTextureUnits = temp;
    r_usedTextureUnits = min(temp, MAX_TEXTURE_UNITS);
    CheckGLError(3);
  }
  else
  {
    r_ext_multitexture = 0;
    gl_maxTextureUnits = 1;
    r_usedTextureUnits = 1;
    gConsole->Insertln("^1ARB multitexture extension (GL_ARB_multitexture) not supported");
  }

  // texture clamp
  if (CheckExtension("EXT_texture_edge_clamp"))
  {
    r_ext_texture_edge_clamp = 1;
  }
  else
  {
    r_ext_texture_edge_clamp = 0;
    gConsole->Insertln("^1Texture edge clamp extension (EXT_texture_edge_clamp) not supported");
  }

  // Display OpenGL infos
  gConsole->Insertln("<b>OpenGL Driver Infos :</b>");
  gConsole->Insertln("\tgl_vendor :");      gConsole->Insertln("\t\t%s", gl_vendor.svalue);
  gConsole->Insertln("\tgl_renderer :");    gConsole->Insertln("\t\t%s", gl_renderer.svalue);
  gConsole->Insertln("\tgl_version :");     gConsole->Insertln("\t\t%s", gl_version.svalue);
  gConsole->Insertln("\tgl_extensions :");

  char *exts = gl_extensions.svalue;
  char buffer[4096] = {'\0'};
  char tmp[2] = {'\0'};
  int lexts = (int) strlen(exts);
  for (int i = 0; i < lexts; ++i)
  {
    if (exts[i] == ' ' && i != lexts-1)
    {
      gConsole->Insertln("\t\t%s", buffer);
      memset(buffer, '\0', 4096);
    }
    else
    {
      tmp[0] = exts[i]; tmp[1] = '\0';
      strcat(buffer, tmp);
    }
  }
  gConsole->Insertln("\t\t%s", buffer);
  gConsole->Insertln("\tgl_max_texture_size : %s", gVars->StringForKey("gl_maxTextureSize"));
  gConsole->Insertln("\tgl_max_texture_units : %s (used : %s)",
  gVars->StringForKey("gl_maxTextureUnits"), gVars->StringForKey("r_usedTextureUnits"));

  // Generate fog map
  CreateFogTexture(&fogtexture);
  //WriteBitmapFile("fogtexture.bmp", FOG_TEXTURE_WIDTH, FOG_TEXTURE_HEIGHT, FOG_TEXTURE_BPP, fogtexture.mem);

  // Initialize state
  State::SetInitialState(r_ext_multitexture.ivalue);

  //_controlfp(_PC_24, _MCW_PC);
  state = BF_EMPTY;
  input.numverts = 0;
  input.numelems = 0;

  ready = 1;

  return 1;
}

void Render::Shut(void)
{
  ready = 0;

  State::setDepthWrite(GL_TRUE);                // must be true before clearing!
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  CheckGLError();
  
  /** @test order? */
  //wglMakeCurrent(0,0);  // -> generate an opengl error (0x502)
  //HGLRC app_rc = wglGetCurrentContext();
  //wglDeleteContext(app_rc);
  //CheckGLError(3);
}

void Render::IncreaseBuffer(unsigned int newsize)
{
  if (newsize <= (unsigned) sys_buffersize.ivalue) return;

  int i;

  if (sys_buffersize.ivalue)
  {
    for (i = 0; i < MAX_TEXTURE_UNITS; ++i)
    {
      buffer[i].st = (float*)   cake_realloc(buffer[i].st, (newsize<<1)*sizeof(float), "gRender->IncreaseBuffer.buffer.st");
      buffer[i].c  = (byte*)    cake_realloc(buffer[i].c,  (newsize<<2)*sizeof(byte), "gRender->IncreaseBuffer.buffer.c");
    }
    input.xyz    = (float*)     cake_realloc(input.xyz,    (newsize<<2)*sizeof(float), "gRender->IncreaseBuffer.xyz");
    input.tex_st = (float*)     cake_realloc(input.tex_st, (newsize<<1)*sizeof(float), "gRender->IncreaseBuffer.tex_st");
    input.lm_st  = (float*)     cake_realloc(input.lm_st,  (newsize<<1)*sizeof(float), "gRender->IncreaseBuffer.lm_st");
    input.normal = (float*)     cake_realloc(input.normal, (newsize<<2)*sizeof(float), "gRender->IncreaseBuffer.normal");
    input.c      = (byte*)      cake_realloc(input.c,      (newsize<<2)*sizeof(byte), "gRender->IncreaseBuffer.c");
#if MODIF_FOG
    input.fog    = (cplane_t**) cake_realloc(input.fog,    newsize*sizeof(cplane_t*), "gRender->IncreaseBuffer.fog");
#else
    input.fog    = (float*)     cake_realloc(input.fog,    5*newsize*sizeof(float), "gRender->IncreaseBuffer.fog");
#endif
  }
  else
  {
    for (i = 0; i < MAX_TEXTURE_UNITS; ++i)
    {
      buffer[i].st = (float*)   cake_malloc((newsize<<1)*sizeof(float), "gRender->IncreaseBuffer.buffer.st");
      buffer[i].c  = (byte*)    cake_malloc((newsize<<2)*sizeof(byte), "gRender->IncreaseBuffer.buffer.c");
    }
    input.xyz    = (float*)     cake_malloc((newsize<<2)*sizeof(float), "gRender->IncreaseBuffer.xyz");
    input.tex_st = (float*)     cake_malloc((newsize<<1)*sizeof(float), "gRender->IncreaseBuffer.tex_st");
    input.lm_st  = (float*)     cake_malloc((newsize<<1)*sizeof(float), "gRender->IncreaseBuffer.lm_st");
    input.normal = (float*)     cake_malloc((newsize<<2)*sizeof(float), "gRender->IncreaseBuffer.normal");
    input.c      = (byte*)      cake_malloc((newsize<<2)*sizeof(byte), "gRender->IncreaseBuffer.c");
#if MODIF_FOG
    input.fog    = (cplane_t**) cake_malloc(newsize*sizeof(cplane_t*), "gRender->IncreaseBuffer.fog");
#else
    input.fog    = (float*)     cake_malloc(5*newsize*sizeof(float), "gRender->IncreaseBuffer.fog");
#endif
  }

  sys_buffersize = (int) newsize;

  // Check for memory fails
  bool failed = false;
  for (i = 0; i < MAX_TEXTURE_UNITS; ++i)
    if (!buffer[i].st || !buffer[i].c)
    {
      failed = true;
      break;
    }

  failed = (!failed && (!input.xyz || !input.tex_st || !input.lm_st || !input.normal || !input.c || !input.fog));

  if (failed)
  {
    if (gConsole) gConsole->Insertln("^1gRender->IncreaseBuffer: Cannot allocate required size");
    else if (gLogFile) gLogFile->Insert("^1gRender->IncreaseBuffer: Cannot allocate required size\n");
  }
  else if (gLogFile) gLogFile->Insert("\tInput buffer increased to %d.\n", newsize);
}

void Render::IncreaseElems(unsigned int newsize)
{
  if (newsize <= (unsigned) sys_maxelems.ivalue) return;

  if (sys_maxelems.ivalue)
    input.elems = (unsigned int*) cake_realloc(input.elems, newsize*sizeof(unsigned int), "gRender->IncreaseElems.elems");
  else
    input.elems = (unsigned int*) cake_malloc(newsize*sizeof(unsigned int), "gRender->IncreaseElems.elems");

  sys_maxelems = (int) newsize;

  if (!input.elems)
  {
    if (gConsole) gConsole->Insertln("^1gRender->IncreaseElems: Cannot allocate required size");
    else if (gLogFile) gLogFile->Insert("^1gRender->IncreaseElems: Cannot allocate required size\n");
  }
  else if (gLogFile) gLogFile->Insert("\tElems buffer increased to %d.\n", newsize);
}

void Render::SetWindowSettings(GLsizei w, GLsizei h, GLint c, GLint f)
{
  if (w > 0) v_width = w;
  if (h > 0) v_height = h;
  if (c > 0) v_colorBits = c;
  if (f > 0) v_hz = f;

  gConsole->Resize(v_width.ivalue - 2*gConsole->GetLeft(), (int) (v_height.ivalue/2));
}

int Render::GetWidth(void)
{
  return v_width.ivalue;
}

int Render::GetHeight(void)
{
  return v_height.ivalue;
}

int Render::GetTop(void)
{
  return v_top.ivalue;
}

int Render::GetLeft(void)
{
  return v_left.ivalue;
}

void Render::CreateFogTexture(texinfo *t)
{
  t->num = -1;
  t->width = FOG_TEXTURE_WIDTH;
  t->height = FOG_TEXTURE_HEIGHT;
  t->bpp = FOG_TEXTURE_BPP;
  t->mem = new byte[FOG_TEXTURE_WIDTH*FOG_TEXTURE_HEIGHT*FOG_TEXTURE_BPP];

  float tw = 1.0f / ((float)t->width - 1.0f);
  float th = 1.0f / ((float)t->height - 1.0f);
  float tx = 0.0f;
  float ty = 0.0f;
  float a;
  for (int y = 1; y < t->height; ++y, ty += th)
  {
    tx = 0.0f;
    for (int x = 0; x < t->width; ++x, tx += tw)
    {
      //a = tx * 255.0f;  // LINEAR
      // squared fog-thickness proportional to linear distance seems to be best looking
      a = sqrtf(tx) * 255.0f;  //SQUARE
      if (a > 255.0f) a = 255.0f;
      for (int i = 0; i < t->bpp; ++i)
        t->mem[(y*t->width+x)*t->bpp+i] = (byte) a;
    }
  }

  for (int x = 0; x < t->width*t->bpp; ++x)
  {
    t->mem[x] = 0;
  }

  RegisterTexInfo(t, LF_NOMIPMAP|LF_NOPICMIP|LF_CLAMP);
}


//-----------------------------------------------------------------------------
// Drawing
//-----------------------------------------------------------------------------

//  Begin Frame
int Render::BeginFrame(void)
{
  State::setDepthWrite(GL_TRUE);                // must be true before clearing!
  
  if (r_clear.ivalue)
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  else 
    glClear(GL_DEPTH_BUFFER_BIT);

  State::setPolygonMode(GL_FILL);

  glEnable(GL_TEXTURE_2D);

  CheckGLError(3);

  state = BF_EMPTY;

  Timer::Refresh();                 // update the timer for the new frame

  // initialize counters
  num_tris = 0;
  num_flush = 0;
  num_verts = 0;
  num_apicalls = 0;

  return 1;
}

//  Ends Frame
int Render::EndFrame(void)
{
  // Flush all the buffers
  if (state != BF_EMPTY) Flush();

  return 1;
}

// Determine if an extension is supported, must be called after a valid
// context has been created
bool Render::CheckExtension(char *extName)
{
  char *extensions = (char *) glGetString(GL_EXTENSIONS);
  CheckGLError();
  if (strstr(extensions, extName)) return true;
  return false;
}

// Register Texinfo by loading the texture into video memory
int Render::RegisterTexInfo(texinfo *tex, int flags)
{
  int picmip = (flags & LF_NOPICMIP)?0:r_picmip.ivalue;
  if (picmip < 0) picmip = 0;
  int newwidth = (1 << Log2(tex->width)) >> picmip;
  int newheight = (1 << Log2(tex->height)) >> picmip;
  if (newwidth <= 0) newwidth = 1;
  if (newheight <= 0) newheight = 1;
  int format, outformat;
  float fform;
  byte *mem = tex->mem;

  if (tex->num != ~0) return 1;   // FIXME: 1?

  switch (tex->bpp)
  {
    case 1:
      format = GL_ALPHA;
      outformat = GL_ALPHA;
      break;
    case 2:
      format = GL_LUMINANCE_ALPHA;    // do I support that??
      outformat = GL_LUMINANCE_ALPHA;
      break;
    case 3:
      format = GL_RGB;
      fform = r_texturebits.fvalue;
      switch ((int) fform)
      {
        // FIXME: float to int conversion !!!
        case 16:
          outformat = GL_RGB5;
          break;
        case 32:
          outformat = GL_RGB8;
          break;
        default:
          outformat = GL_RGB;
          break;
      }
      break;
    case 4:
      format = GL_RGBA;
      fform = r_texturebits.fvalue;
      switch ((int)fform)
      {
        // FIXME: float to int conversion !!!
        case 16:
          outformat = GL_RGBA4;
          break;
        case 32:
          outformat = GL_RGBA8;
          break;
        default:
          outformat = GL_RGBA;
          break;
      }
      break;
    default:
      format = GL_RGB;
      outformat = GL_RGB;
      break;
  }
  
  if (newheight > gl_maxTextureSize.ivalue) newheight = gl_maxTextureSize.ivalue;
  if (newwidth > gl_maxTextureSize.ivalue) newwidth = gl_maxTextureSize.ivalue;

  if (tex->height != newheight || tex->width != newwidth)
  {
    mem = (byte *) cake_malloc(newheight*newwidth*tex->bpp, "gRender->RegisterTexInfo.mem");
    ScaleImage(format, tex->width, tex->height, tex->mem, newwidth, newheight, mem);
  }

  glGenTextures(1, (GLuint*) &tex->num);
  glBindTexture(GL_TEXTURE_2D, tex->num);
  CheckGLError(2);

  if (flags & LF_NOMIPMAP)
  {
    glTexImage2D(GL_TEXTURE_2D, 0, outformat, newwidth, newheight, 0, format, GL_UNSIGNED_BYTE, mem);
    CheckGLError();
  }
  else
    BuildMipmaps(outformat, newwidth, newheight, format, mem);

  if (flags & LF_CLAMP)
  {
    if (r_ext_texture_edge_clamp.ivalue)
    {
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    }
    else
    {
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    }
  }
  else
  {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  }
  CheckGLError(2);
  
  //float filter = Import.cvars->ValueForKey("gl_filter");
  
  if (r_filter.ivalue == 0)
  {
    // nearest
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    if (flags & LF_NOMIPMAP)
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    else
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
    CheckGLError(2);
  }
  else if (r_filter.ivalue == 2)
  {
    // trilinear
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    if (flags & LF_NOMIPMAP)
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    else
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    CheckGLError(2);
  }
  else /*if (r_filter.ivalue == 1)*/
  {
    // bilinear
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    if (flags & LF_NOMIPMAP)
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    else
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);
    CheckGLError(2);
  }

  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE); CheckGLError();
  //glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE); CheckGLError();

  if (mem != tex->mem) cake_free(mem);
  return 1;
}

// Deletes texture memory
void Render::UnregisterTexInfo(texinfo *tex)
{
  glDeleteTextures(1, (GLuint*) &tex->num);
  CheckGLError();
}

// Sets the given render state
void Render::SetRenderState(Shader *s, int lightmap, Shader *e)
{
  // FIXME: If State::curr_shader is NULL, State::skipping will be true. Then this
  // will have as consequence that Flushing will be aborted and reported to next
  // time where State::curr_shader is not NULL and faces that are already in input
  // struct may have wrong shader when rendered. So what is better ? Render them
  // with wrong shader or not render them at all ?
  if (state != BF_EMPTY) Flush();

  State::curr_shader = s;

  if (r_nolightmap.ivalue) State::curr_lightmap = ~0;
  else State::curr_lightmap = lightmap;

  State::curr_effect = e;

  State::skipping = (!s || s->surface_flags & SURF_SKIP || s->surface_flags & SURF_NODRAW);
}

void Render::InitializeViewPort(void)
{
  if (state != BF_EMPTY) Flush();

  glViewport(0, 0, v_width.ivalue, v_height.ivalue);
  glScissor(0, 0, v_width.ivalue, v_height.ivalue);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  glOrtho(0, v_width.ivalue, v_height.ivalue, 0, 0.0f, 1.0f);

  CheckGLError(7);

  // Initializing state
  State::setBlend(GL_ONE, GL_ZERO);
  State::setAlphaFunc(GL_ALWAYS, 0);
  State::setDepthFunc(GL_LEQUAL);
  State::setDepthWrite(GL_TRUE);
  State::setCulling(CULL_NONE);
}

// Write vertexes in the input, Flush when full
void Render::PushTriangles(Surface *surf, int *visible)
{
  if (!surf) return;
  if (State::skipping) return;

  int level = surf->currentlevel;
  if (level < 0 || level >= surf->levels)
  {
    gConsole->Insertln("^1ERROR: Invalid surface level !");
    return;
  }
  if (surf->numelems[level] == 0 || surf->numverts[level] == 0) return;

  // Force flushing before resizing buffer to avoid problems
  if (input.numverts + surf->numverts[level] >= sys_buffersize.ivalue ||
	  input.numelems + surf->numelems[level] >= sys_maxelems.ivalue)
	ForceFlush();

  // Check buffers overflow
  while (input.numverts + surf->numverts[level] >= sys_buffersize.ivalue) 
    IncreaseBuffer(input.numverts + surf->numverts[level] + BUFFER_SIZE_BLOC);

  while (input.numelems + surf->numelems[level] >= sys_maxelems.ivalue) 
    IncreaseElems(input.numelems + surf->numelems[level] + ELEMS_BLOC);

  float * xyz;
  float * st;
  vertex_t * v;
  int i;

  state = BF_FILLING; // filling the buffer
  
  // adding elements to input data
  if (visible)
  {
    for (i = 0; i < surf->numelems[level]; ++i)
      if (visible[i])
        input.elems[input.numelems++] = input.numverts + surf->firstelem[level][i];
  }
  else
  {
    for (i = 0; i < surf->numelems[level]; ++i)
      input.elems[input.numelems++] = input.numverts + surf->firstelem[level][i];
  }

  // vertices
  xyz = &input.xyz[input.numverts<<2];
  v = surf->firstvert[level], i = surf->numverts[level];
  do
  {
    VectorCopy(v->v_point, xyz);
    xyz += 4; // 4 coordinates for a vertex
    ++v;
  } while (--i);

  // texture coordinates
  st = &input.tex_st[input.numverts<<1];
  v = surf->firstvert[level], i = surf->numverts[level];
  do
  {
    TexcoordCopy(v->tex_st, st);
    st += 2;  // 2 coordinates s & t
    ++v;
  } while (--i);

  if (State::curr_shader->flags & SF_HASLIGHTMAP)
  {
    st = &input.lm_st[input.numverts<<1];
    v = surf->firstvert[level], i = surf->numverts[level];
    do
    {
      TexcoordCopy(v->lm_st, st);
      st += 2;
      ++v;
    } while (--i);
  }

  if (State::curr_shader->flags & SF_USENORMALS)
  {
    float *n = &input.normal[input.numverts<<2];
    v = surf->firstvert[level], i = surf->numverts[level];
    do
    {
      VectorCopy(v->v_norm, n);
      n += 4;
      ++v;
    } while (--i);
  }

  if (State::curr_shader->flags & SF_HASCOLORS)
  {
    byte *c = &input.c[input.numverts<<2];
    v = surf->firstvert[level], i = surf->numverts[level];
    do
    {
      ColorCopy(v->colour, c);
      c += 4;
      ++v;
    } while (--i);
  }

  SetVertexCoords(surf, level); // Apply vertex coordinates transformations for the surface

#if MODIF_FOG
  // Get fog plane pointer
  //if (surf->fogplane)
  {
    cplane_t** f = &input.fog[input.numverts];
    i = surf->numverts[level];
    do
    {
      *f++ = surf->fogplane;
    } while(--i);
  }
#else
  // Store triangle normal and distance in input data (for fog)
  float *f = &input.fog[input.numverts * 5];
  i = surf->numverts[level];
  do
  {
    memcpy(f, surf->fog, 5*sizeof(float));
    f += 5;
  } while (--i);
#endif

  input.numverts += surf->numverts[level];
}

// Flush a buffer setting the appropriate state
void Render::Flush(void)
{
  Shader* s = State::curr_shader;

  if (!input.numelems /*|| !s->num_layers*/) return;
  if (State::skipping) return;

  // set buffer states:
  state = BF_LOCKED;
  buffer[0].st_content = -1;    // the buffers are always dirty the first time.
  buffer[1].st_content = -1;
  buffer[0].c_content = -1;
  buffer[1].c_content = -1;

  if (!gridpass)
  {
    State::SetupShader();

    // select render path
    if (r_vertexLight.ivalue)
      FlushVertexLit();
    else if (r_ext_compiledVertexArray.ivalue)
    {
      if (r_ext_multitexture.ivalue && (s->flags & SF_MULTITEX))
        FlushMtxCVA();
      else if (s->num_layers == 1)
        FlushSingleCVA();
      else
        FlushGenericCVA();
    }
    else
    {
      if (r_ext_multitexture.ivalue && (s->flags & SF_MULTITEX))
        FlushMtx();
      else if (s->num_layers == 1)
        FlushSingle();
      else
        FlushGeneric();
    }

    if (!r_nofog.ivalue &&
        State::curr_effect &&
        State::curr_effect->content_flags & CONTENTS_FOG)
    {
      // Render fog pass
      Layer *l = State::curr_effect->fog_params.layer;
      if (l)
      {
        glMatrixMode(GL_TEXTURE);
        glPushMatrix();
        SetTextureCoordinates(0, l);
        SetVertexColors(0, l);
        State::BindTexture(l, 0);
        State::setState(l);
    //  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);
        if (r_ext_compiledVertexArray.ivalue)
        {
          glDisableClientState(GL_TEXTURE_COORD_ARRAY);
          glDisableClientState(GL_COLOR_ARRAY);
          glLockArraysEXT(0, input.numverts);
          glEnableClientState(GL_TEXTURE_COORD_ARRAY);
          glEnableClientState(GL_COLOR_ARRAY);
          glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);
          glUnlockArraysEXT();
          CheckGLError(7);
        }
        else
        {
          Stripmine(input.numelems, input.elems);
        }
        glPopMatrix();
        CheckGLError(3);
      }
    }
  }
  else
  {
    if (r_ext_compiledVertexArray.ivalue) FlushGridCVA();
    else FlushGrid();
  }

  // update counters
  num_tris += input.numelems/3;
  num_verts += input.numverts;
  ++num_flush;

  // clear buffer state:
  state = BF_EMPTY;
  input.numelems = 0;
  input.numverts = 0;
}

void Render::ForceFlush(void)
{
  if (state != BF_EMPTY) Flush();         // Flush all the buffers
}

// Flush multitexture faces using CVAs
void Render::FlushMtxCVA(void)
{
  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);

  glDisableClientState(GL_TEXTURE_COORD_ARRAY); // don't allow to cache this!
  glDisableClientState(GL_COLOR_ARRAY);
  glLockArraysEXT(0, input.numverts);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glEnableClientState(GL_COLOR_ARRAY);

  Layer *l;

  for (State::pass = 0; State::pass < State::curr_shader->num_layers; ++State::pass)
  {
    l = State::curr_shader->layer[State::pass];

    SetTextureCoordinates(0, l);
    SetVertexColors(0, l);

    State::BindTexture(l, 0);
    State::setState(l);

    if (l->flags & LF_MULTITEX)
    {
      l = State::curr_shader->layer[++State::pass];

      glActiveTextureARB(GL_TEXTURE1_ARB);
      glClientActiveTextureARB(GL_TEXTURE1_ARB);
      glEnable(GL_TEXTURE_2D);
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);

      SetTextureCoordinates(1, l);

      State::BindTexture(l, 1);

      glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);

      glDisable(GL_TEXTURE_2D);
      glActiveTextureARB(GL_TEXTURE0_ARB);
      glClientActiveTextureARB(GL_TEXTURE0_ARB);
      CheckGLError(8);
    }
    else
    {
      glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);
      CheckGLError();
    }
  }

  glUnlockArraysEXT();
  CheckGLError(7);
}

// Flush multitexture faces with lightmap
void Render::FlushMtx(void)
{
  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);
  CheckGLError();

  Layer *l;

  for (State::pass = 0; State::pass < State::curr_shader->num_layers; ++State::pass)
  {
    l = State::curr_shader->layer[State::pass];
  
    SetTextureCoordinates(0, l);
    SetVertexColors(0, l);

    State::BindTexture(l, 0);
    State::setState(l);
    if (l->flags & LF_MULTITEX)
    {
      l = State::curr_shader->layer[++State::pass];
      
      glActiveTextureARB(GL_TEXTURE1_ARB);
      glClientActiveTextureARB(GL_TEXTURE1_ARB);
      glEnable(GL_TEXTURE_2D);
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);
      
      SetTextureCoordinates(1, l);

      State::BindTexture(l, 1);
      
      glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);
      
      glDisable(GL_TEXTURE_2D);
      glActiveTextureARB(GL_TEXTURE0_ARB);
      glClientActiveTextureARB(GL_TEXTURE0_ARB);
      CheckGLError(8);
    }
    else
    {
      Stripmine(input.numelems, input.elems);
    }
  }
}

// Flush faces without lightmap using CVAs
void Render::FlushGenericCVA(void)
{
  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);

  glDisableClientState(GL_TEXTURE_COORD_ARRAY); // don't allow to cache this!
  glDisableClientState(GL_COLOR_ARRAY);
  glLockArraysEXT(0, input.numverts);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glEnableClientState(GL_COLOR_ARRAY);

  Layer *l;

  for (State::pass = 0; State::pass < State::curr_shader->num_layers; ++State::pass)
  {
    l = State::curr_shader->layer[State::pass];

    SetTextureCoordinates(0, l);
    SetVertexColors(0, l);

    State::BindTexture(l, 0);
    State::setState(l);
    glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);
  }

  glUnlockArraysEXT();
  CheckGLError(7+State::curr_shader->num_layers);
}

// Flush faces without lightmap
void Render::FlushGeneric(void)
{
  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);
  CheckGLError();

  Layer *l;

  for (State::pass = 0; State::pass < State::curr_shader->num_layers; ++State::pass)
  {
    l = State::curr_shader->layer[State::pass];

    SetTextureCoordinates(0, l);
    SetVertexColors(0, l);

    State::BindTexture(l, 0);
    State::setState(l);
    Stripmine(input.numelems, input.elems);
  }
}

// Flush faces with a single layer using CVAs
void Render::FlushSingleCVA(void)
{
  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);
  glLockArraysEXT(0, input.numverts);

  State::pass = 0;
  Layer *l = State::curr_shader->layer[0];

  SetTextureCoordinates(0, l);
  SetVertexColors(0, l);

  State::BindTexture(l, 0);
  State::setState(l);
  glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);
  glUnlockArraysEXT();
  CheckGLError(4);
}

// Flush faces with a single layer without using CVAs
void Render::FlushSingle(void)
{
  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);
  CheckGLError();

  State::pass = 0;
  Layer *l = State::curr_shader->layer[0];
  
  SetTextureCoordinates(0, l);
  SetVertexColors(0, l);

  State::BindTexture(l, 0);
  State::setState(l);
  Stripmine(input.numelems, input.elems);
}

void Render::FlushGridCVA(void)
{
  State::setBlend(GL_ONE, GL_ZERO);
  State::setAlphaFunc(GL_ALWAYS);

  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);
  glLockArraysEXT(0, input.numverts);

  glColorPointer(4, GL_UNSIGNED_BYTE, 0, buffer[0].c);
  memset(buffer[0].c, 255, (input.numverts<<2));
  glColor4f(1.f, 1.f, 1.f, 1.f);

  glGetBooleanv(GL_TEXTURE_2D, &texture);
  glGetBooleanv(GL_LIGHTING, &lighting);
  glGetBooleanv(GL_DEPTH_TEST, &depthtest);
  
  State::setPolygonMode(GL_LINE);

  glDisable(GL_TEXTURE_2D);
  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);

  glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);

  if (texture) { glEnable(GL_TEXTURE_2D); CheckGLError(); }
  if (lighting) { glEnable(GL_LIGHTING); CheckGLError(); }
  if (depthtest) { glEnable(GL_DEPTH_TEST); CheckGLError(); }

  State::setPolygonMode(GL_FILL);

  glUnlockArraysEXT();
  CheckGLError(12);
}

void Render::FlushGrid(void)
{
  State::setBlend(GL_ONE, GL_ZERO);
  State::setAlphaFunc(GL_ALWAYS);

  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);

  glColorPointer(4, GL_UNSIGNED_BYTE, 0, buffer[0].c);
  memset(buffer[0].c, 255, (input.numverts<<2));
  glColor4f(1.f, 1.f, 1.f, 1.f);

  glGetBooleanv(GL_TEXTURE_2D, &texture);
  glGetBooleanv(GL_LIGHTING, &lighting);
  glGetBooleanv(GL_DEPTH_TEST, &depthtest);

  State::setPolygonMode(GL_LINE);

  glDisable(GL_TEXTURE_2D);
  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);

  glDrawElements(GL_TRIANGLES, input.numelems, GL_UNSIGNED_INT, input.elems);

  if (texture) { glEnable(GL_TEXTURE_2D); CheckGLError(); }
  if (lighting) { glEnable(GL_LIGHTING); CheckGLError(); }
  if (depthtest) { glEnable(GL_DEPTH_TEST); CheckGLError(); }
  
  State::setPolygonMode(GL_FILL);

  CheckGLError(12);
}

// FIXME: this is only experimental
void Render::FlushVertexLit(void)
{
  Layer *l;

  glVertexPointer(3, GL_FLOAT, XYZ_OFFSET, input.xyz);
  CheckGLError();

  State::pass = 0;
  if (State::curr_shader->layer[0]->flags & LF_LIGHTMAP)
  {
    l = State::curr_shader->layer[++State::pass];

    SetTextureCoordinates(0, l);
  
//  glColorPointer(4, GL_UNSIGNED_BYTE, 0, &input.c); CheckGLError();

    State::BindTexture(l, 0);
    
    if (State::curr_blendsrc != GL_ONE || State::curr_blenddst != GL_ZERO)
    {
      glDisable(GL_BLEND);
      CheckGLError();
      State::curr_blenddst = l->blenddst;
      State::curr_blendsrc = l->blendsrc;
    }
    
    Stripmine(input.numelems, input.elems);
    ++State::pass;
  }

  for (; State::pass < State::curr_shader->num_layers; ++State::pass)
  {
    l = State::curr_shader->layer[State::pass];
    if (l->flags & LF_LIGHTMAP) continue;

    SetTextureCoordinates(0, l);
    SetVertexColors(0, l);
  
    State::BindTexture(l, 0);
    State::setState(l);
    Stripmine(input.numelems, input.elems);
  }
}

void Render::SwapBuffers(void)
{
  if (state != BF_EMPTY) Flush();

  if (!swapbuffersfunc)
    ThrowException(NULL_POINTER, "Undefined swap function pointer for render.");
  swapbuffersfunc();
}

// that have been taken from aftershock:  (FIXME: can I use it?)
void Render::Stripmine(int numelems, unsigned int *elems)
{
  bool toggle;
  int elem = 0;
  unsigned int a, b;

  while (elem < numelems)
  {
    toggle = true;
    glBegin(GL_TRIANGLE_STRIP);
  
    glArrayElement(elems[elem++]);
    b = elems[elem++]; glArrayElement(b);
    a = elems[elem++]; glArrayElement(a);

    while (elem < numelems)
    {
      if (a != elems[elem] || b != elems[elem+1]) break;
      if (toggle) { b = elems[elem+2]; glArrayElement(b); }
      else        { a = elems[elem+2]; glArrayElement(a); }
      elem += 3;
      toggle = !toggle;
      CheckGLError(1);
    }
    glEnd();
    CheckGLError(5);
  }
}

void Render::SetVertexCoords(Surface *surf, int level)
{
  if (!State::curr_shader->num_deforms || r_nodeform.ivalue) return;

  float args[5], startoff = 0.f, off, wavesize = 1.f, inc;
  const float speed = 2;

  for (int i = 0; i < MAX_DEFORMS; ++i)
  {
    switch (State::curr_shader->deformVertexes[i].type)
    {
      case DEFORMVERTEXES_WAVE:
        {
          float *xyz = &input.xyz[input.numverts<<2];
          vertex_t *v = surf->firstvert[level];
          int nverts = surf->numverts[level];
          vec3_t p;

          args[0]  = State::curr_shader->deformVertexes[i].params[0]; // sin
          wavesize = State::curr_shader->deformVertexes[i].params[1];
          args[1]  = State::curr_shader->deformVertexes[i].params[2];
          args[2]  = State::curr_shader->deformVertexes[i].params[3];
          args[4]  = State::curr_shader->deformVertexes[i].params[5];
          startoff = State::curr_shader->deformVertexes[i].params[4];

          do
          {
            p[0] = *xyz; p[1] = *(xyz+1); p[2] = *(xyz+2);
            off = (p[0]+p[1]+p[2])/wavesize;
            args[3] = startoff + off;
            inc = State::Eval(&args[0]);
            *xyz     = p[0]+inc*v->v_norm[0];
            *(xyz+1) = p[1]+inc*v->v_norm[1];
            *(xyz+2) = p[2]+inc*v->v_norm[2];
            ++v;
            xyz += 4;
          } while (--nverts);
        }
        break;
      case DEFORMVERTEXES_NORMAL:
        {
          float *normal = &input.normal[input.numverts<<2];
          float *xyz = &input.xyz[input.numverts<<2];
          vertex_t *v = surf->firstvert[level];
          int nverts = surf->numverts[level];
          vec3_t p, n;

          args[0] = FUNC_SIN;
          args[2] = State::curr_shader->deformVertexes[i].params[0];
          args[4] = State::curr_shader->deformVertexes[i].params[1];
          float lat, lng, sinlng;

          do
          {
            p[0] = *xyz; p[1] = *(xyz+1); p[2] = *(xyz+2);
            n[0] = *normal; n[1] = *(normal+1); n[2] = *(normal+2);
#if 1
            // diesel method
            args[1] = atan2f(p[2], -p[0]);
            args[3] = p[0]+p[1];
            lat = State::Eval(&args[0]);

            args[1] = acosf(n[1]);
            args[3] = p[2]+p[1];
            lng = State::Eval(&args[0]);

            sinlng = FastSin(lng);
            *(normal  ) = -FastCos(lat)*sinlng;
            *(normal+1) =  FastCos(lng);
            *(normal+2) =  FastSin(lat)*sinlng;
#else
            // another method (real method is not known ;)
            off = (p[0]+p[1]+p[2])/(speed*wavesize);
            args[3] = startoff + off;
            inc = State::Eval(&args[0]);
            *(normal  ) = n[0]+inc*v->v_norm[0];
            *(normal+1) = n[1]+inc*v->v_norm[1];
            *(normal+2) = n[2]+inc*v->v_norm[2];
#endif
            ++v;
            normal += 4;
            xyz += 4;
          } while (--nverts);
        }
        break;
      case DEFORMVERTEXES_BULGE:
        {
          if (surf->facetype != FACETYPE_PATCH) break;  // are bulge applied on other face types ?
          float *xyz = &input.xyz[input.numverts<<2];
          float *st = &input.tex_st[input.numverts<<1];
          vertex_t *v = surf->firstvert[level];
          int nverts = surf->numverts[level];

          args[0] = FUNC_SIN; // sin
          args[1] = 0;
          args[2] = State::curr_shader->deformVertexes[i].params[1];
          args[4] = 1/(speed*State::curr_shader->deformVertexes[i].params[0]);

          do
          {
            args[3] = *st*speed*State::curr_shader->deformVertexes[i].params[2];
            inc = State::Eval(&args[0]);
            *xyz     += inc*v->v_norm[0];
            *(xyz+1) += inc*v->v_norm[1];
            *(xyz+2) += inc*v->v_norm[2];
            ++v;
            xyz += 4;
            st += 2;
          } while (--nverts);
        }
        break;
      case DEFORMVERTEXES_MOVE:
        {
          float *xyz = &input.xyz[input.numverts<<2];
          vertex_t *v = surf->firstvert[level];
          int nverts = surf->numverts[level];

          args[0] = State::curr_shader->deformVertexes[i].params[0];  // sin
          args[1] = State::curr_shader->deformVertexes[i].params[4];
          args[2] = State::curr_shader->deformVertexes[i].params[5];
          args[3] = State::curr_shader->deformVertexes[i].params[6];
          args[4] = State::curr_shader->deformVertexes[i].params[7];

          do
          {
            inc = State::Eval(&args[0]);
            *xyz     += inc*State::curr_shader->deformVertexes[i].params[1];
            *(xyz+1) += inc*State::curr_shader->deformVertexes[i].params[2];
            *(xyz+2) += inc*State::curr_shader->deformVertexes[i].params[3];
            ++v;
            xyz += 4;
          } while (--nverts);
        }
        break;
      case DEFORMVERTEXES_AUTOSPRITE:
        {
          float *xyz = &input.xyz[input.numverts<<2], p[3], d[3], *xyzB;
          vertex_t *v = surf->firstvert[level], *vB;
          int nverts = surf->numverts[level], incr, nvertsB;
          vec3_t middle;

          while (nverts)
          {
            // Compute middle of quad
            incr = 4;
            xyzB = xyz;
            vB = v;
            nvertsB = nverts;

            middle[0] = middle[1] = middle[2] = 0.f;

            do
            {
              middle[0] += *xyz;
              middle[1] += *(xyz+1);
              middle[2] += *(xyz+2);
              ++v;
              xyz += 4;
            } while (--nverts && --incr);
            VectorScale(middle, 0.25f, middle);

            // Rotate each vertex
            incr = 4;
            xyz = xyzB;
            v = vB;
            nverts = nvertsB;
            do
            {
              p[0] = *xyz    -middle[0];
              p[2] = *(xyz+1)-middle[1];
              p[1] = *(xyz+2)-middle[2];
              MultVect3x3(current_camera->rotation, p, d);
              *xyz     = d[0]+middle[0];
              *(xyz+1) = d[1]+middle[1];
              *(xyz+2) = d[2]+middle[2];
              ++v;
              xyz += 4;
            } while (--nverts && --incr);
          }
        }
        break;
      case DEFORMVERTEXES_AUTOSPRITE2:
        {
          float *xyz = &input.xyz[input.numverts<<2], p[3], d[3], *xyzB;
          vertex_t *v = surf->firstvert[level], *vB;
          int nverts = surf->numverts[level], incr, nvertsB;
          vec3_t middle;

          float m[9], temp[16];

          while (nverts)
          {
            // Compute middle of quad
            incr = 4;
            xyzB = xyz;
            vB = v;
            nvertsB = nverts;

            middle[0] = middle[1] = middle[2] = 0.f;

            do
            {
              middle[0] += *xyz;
              middle[1] += *(xyz+1);
              middle[2] += *(xyz+2);

              ++v;
              xyz += 4;
            } while (--nverts && --incr);
            VectorScale(middle, 0.25f, middle);
  
            glMatrixMode(GL_MODELVIEW);
            glPushMatrix();
              glLoadIdentity();
              glRotatef(-90, 1, 0, 0); 
              if (surf->autosprite2_axis == 0)
              {
                // @todo Terminate this stuff
              }
              else if (surf->autosprite2_axis == 2)
              {
                glRotatef(-current_camera->rot[1], 0, 0, 1);
                CheckGLError();
              }
              glGetFloatv(GL_MODELVIEW_MATRIX, temp);
            glPopMatrix();
            CheckGLError(6);

            m[0] = temp[0];
            m[1] = temp[1];
            m[2] = temp[2];
            m[3] = temp[4];
            m[4] = temp[5];
            m[5] = temp[6];
            m[6] = temp[8];
            m[7] = temp[9];
            m[8] = temp[10];

            // Rotate each vertex
            incr = 4;
            xyz = xyzB;
            v = vB;
            nverts = nvertsB;
            do
            {
              p[0] = *xyz    -middle[0];
              p[2] = *(xyz+1)-middle[1];
              p[1] = *(xyz+2)-middle[2];
              MultVect3x3(m, p, d);
              *xyz     = d[0]+middle[0];
              *(xyz+1) = d[1]+middle[1];
              *(xyz+2) = d[2]+middle[2];
              ++v;
              xyz += 4;
            } while (--nverts && --incr);
          }
        }
        break;
    //  case DEFORMVERTEXES_NONE:
    //    break;
      default:
        break;
    }
  }
}

void Render::SetTextureCoordinates(int tmu, Layer *l)
{
  glTexCoordPointer(2, GL_FLOAT, 0, buffer[tmu].st);
  CheckGLError();

  if (r_noshader.ivalue)
  {
    memcpy(buffer[tmu].st, input.lm_st, sizeof(float)*(input.numverts<<1));
    return;
  }

  // if the contents are the same, and the buffer isn't dirty don't fill it again.
  if (buffer[tmu].st_content != l->tcGen || buffer[tmu].st_content == -1)
  {
    buffer[tmu].st_content = l->tcGen;

    switch (l->tcGen)
    {
      case TCGEN_LIGHTMAP:
        memcpy(buffer[tmu].st, input.lm_st, sizeof(float)*(input.numverts<<1));
        break;
      case TCGEN_ENVIRONMENT:
        {
          vec3_t dir, pos;
          vec_t m[16];
          glMatrixMode(GL_MODELVIEW);
          glGetFloatv(GL_MODELVIEW_MATRIX, m);
          vec3_t trans;
          // Invert the matrix
          VectorCopy(&m[12], trans);
          m[12] = -DotProduct(trans, &m[0]);
          m[13] = -DotProduct(trans, &m[4]);
          m[14] = -DotProduct(trans, &m[8]);
          float t = m[1];
          m[1] = m[4];
          m[4] = t;
          t = m[2];
          m[2] = m[8];
          m[8] = t;
          t = m[9];
          m[9] = m[6];
          m[6] = t;
          VectorCopy(&m[12], pos);
          for (int i = 0; i < input.numverts; ++i)
          {
            VectorSub(pos, &input.xyz[i<<2], dir);

            FastNormVect3(dir);

            buffer[tmu].st[ i<<1   ] = dir[2]-input.normal[(i<<2)+2];
            buffer[tmu].st[(i<<1)+1] = dir[1]-input.normal[(i<<2)+1];
          }
        }
        break;
      case TCGEN_VECTOR:
      case TCGEN_BASE:
        memcpy(buffer[tmu].st, input.tex_st, sizeof(float)*(input.numverts<<1));
        break;
      case TCGEN_FOG:
#if MODIF_FOG
        {
          cplane_t** f = input.fog;
          float *xyz = input.xyz;
          byte *c = input.c;
          float fogdistance, inormu, d1, d2, d3;
          byte r, g, b, a = 255;
          vec3_t d, u, v;
          cplane_t* fogplane = NULL;

          if (State::curr_effect)
          {
            // Get and set the fog parameters in effect shader
            r = (int)(255.f*State::curr_effect->fog_params.params[0]);
            g = (int)(255.f*State::curr_effect->fog_params.params[1]);
            b = (int)(255.f*State::curr_effect->fog_params.params[2]);
            fogdistance = State::curr_effect->fog_params.params[4];
          }
          else
          {
            // Get and set the fog parameters in current shader
            r = (int)(255.f*State::curr_shader->fog_params.params[0]);
            g = (int)(255.f*State::curr_shader->fog_params.params[1]);
            b = (int)(255.f*State::curr_shader->fog_params.params[2]);
            fogdistance = State::curr_shader->fog_params.params[4];
          }

          for (int i = 0; i < input.numverts; ++i, ++f, c += 4, xyz += 4)
          {
            if (*f) fogplane = *f;
            else __asm int 3;   // @bug Search why fogplane is NULL sometimes !!
            if (!fogplane) continue;

            v[0] = *(xyz+0);
            v[1] = *(xyz+1);
            v[2] = *(xyz+2);

            d[0] = v[0] - (current_camera->pos[0] + v_camnear.fvalue*current_camera->eyedir[0]);
            d[1] = v[1] - (current_camera->pos[1] + v_camnear.fvalue*current_camera->eyedir[1]);
            d[2] = v[2] - (current_camera->pos[2] + v_camnear.fvalue*current_camera->eyedir[2]);
            VectorCopy(current_camera->eyedir, u);
            inormu = InverseSqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2]);
            VectorScale(u, inormu, u);

            d1 = PointDistance(current_camera->pos, fogplane);

            if (d1 < 0)
            {
              // the viewer is below the fog-surface (inside the fog volume)
              buffer[tmu].st[ i<<1   ] =  DotProduct(d, u);
              buffer[tmu].st[(i<<1)+1] = -PointDistance(v, fogplane);
            }
            else
            {
              d2 = PointDistance(v, fogplane);
              buffer[tmu].st[(i<<1)+1] = -d2;
              if (d2 < 0.0f)
              {
                d3 = d2/(d2-d1);
                d3 *= DotProduct(d, u);
                buffer[tmu].st[i<<1] = d3;
              }
              else
              {
                buffer[tmu].st[i<<1] = 0.0f;
              }
            }

            *(c+0) = r;
            *(c+1) = g;
            *(c+2) = b;
            *(c+3) = a;
          }

          glTranslatef(0, 1.5f/(float)FOG_TEXTURE_HEIGHT, 0);
          glScalef(1.0f/fogdistance, 1.0f/fogdistance, 1.0f); // map [0.0, fogdistance] to [0.0, 1.0]
          CheckGLError(2);
        }
#else
        {
          float *f = input.fog;
          float *xyz = input.xyz;
          byte *c = input.c;
          cplane_t fogplane;
          float fogdistance, inormu, d1, d2, d3;
          byte r, g, b, a = 255;
          vec3_t d, u, v;

          // default plane
          fogplane.normal[0] = 0;
          fogplane.normal[1] = 0;
          fogplane.normal[2] = 1;
          fogplane.dist = 0;
          fogplane.type = 3;

          if (State::curr_effect)
          {
            // Get and set the fog parameters in effect shader
            r = (int)(255.f*State::curr_effect->fog_params.params[0]);
            g = (int)(255.f*State::curr_effect->fog_params.params[1]);
            b = (int)(255.f*State::curr_effect->fog_params.params[2]);
            fogdistance = State::curr_effect->fog_params.params[4];
          }
          else
          {
            // Get and set the fog parameters in current shader
            r = (int)(255.f*State::curr_shader->fog_params.params[0]);
            g = (int)(255.f*State::curr_shader->fog_params.params[1]);
            b = (int)(255.f*State::curr_shader->fog_params.params[2]);
            fogdistance = State::curr_shader->fog_params.params[4];
          }

          for (int i = 0; i < input.numverts; ++i)
          {
            if (fogplane.type < 4)
            {
              fogplane.normal[0] = *(f+0);
              fogplane.normal[1] = *(f+1);
              fogplane.normal[2] = *(f+2);
              fogplane.dist = *(f+3);
              fogplane.type = (unsigned int) *(f+4);
            }
            //else gConsole->Insertln("invalid vertex associated fog plane");

            v[0] = *(xyz+0);
            v[1] = *(xyz+1);
            v[2] = *(xyz+2);

            d[0] = v[0] - current_camera->pos[0];
            d[1] = v[1] - current_camera->pos[1];
            d[2] = v[2] - current_camera->pos[2];
            VectorCopy(current_camera->forward, u);
            inormu = 1.f / (float) sqrt(u[0]*u[0]+u[1]*u[1]+u[2]*u[2]);
            VectorScale(u, inormu, u);

            d1 = PointDistance(current_camera->pos, &fogplane);

            if (d1 < 0)
            {
              // the viewer is below the fog-surface (inside the fog volume)
              buffer[tmu].st[ i<<1   ] =  DotProduct(d, u);
              buffer[tmu].st[(i<<1)+1] = -PointDistance(v, &fogplane);
            }
            else
            {
              d2 = PointDistance(v, &fogplane);
              buffer[tmu].st[(i<<1)+1] = -d2;
              if (d2 < 0.0f)
              {
                d3 = d2/(d2-d1);
                d3 *= DotProduct(d, u);
                buffer[tmu].st[i<<1] = d3;
              }
              else
              {
                buffer[tmu].st[i<<1] = 0.0f;
              }
            }

            *(c+0) = r;
            *(c+1) = g;
            *(c+2) = b;
            *(c+3) = a;

            c += 4;
            f += 5;
            xyz += 4;
          }

          glTranslatef(0, 1.5f/(float)FOG_TEXTURE_HEIGHT, 0);
          glScalef(1.0f/fogdistance, 1.0f/fogdistance, 1.0f); // map [0.0, fogdistance] to [0.0, 1.0]
          CheckGLError(2);
        }
#endif
        break;
        default:
          break;
    }
  }
 
  if (l->tcMod)
  {
    buffer[tmu].st_content = -1;  // texcoord transforms always destroy the content

    int i;
    float *st, t1, t2;

    for (funclist *f = l->tcMod; f; f = f->next)
    {
      st = buffer[tmu].st;
      switch (f->func)
      {
        case TCMOD_ROTATE:
          {
            float angulo = -f->p[0]*(float)Timer::fTime/M_180_PI;
            float sint = FastSin(angulo);
            float cost = FastCos(angulo);
            for (i = 0; i < input.numverts; ++i, st += 2)
            {
              t1 = cost * (st[0] - 0.5f) - sint * (st[1] - 0.5f) + 0.5f;
              t2 = cost * (st[1] - 0.5f) + sint * (st[0] - 0.5f) + 0.5f;
              st[0] = t1;
              st[1] = t2;
            }
          }
          break;
        case TCMOD_SCALE:
          {
            for (i = 0; i < input.numverts; ++i, st += 2)
            {
              st[0] = (st[0]/*-0.5f*/) * f->p[0]/* + 0.5f*/;
              st[1] = (st[1]/*-0.5f*/) * f->p[1]/* + 0.5f*/;
            }
          }
          break;
        case TCMOD_SCROLL:
          {
            t1 = f->p[0]*(float)Timer::fTime;
            t2 = f->p[1]*(float)Timer::fTime;
            t1 -= floorf(t1);
            t2 -= floorf(t2);
            for (i = 0; i < input.numverts; ++i, st += 2)
            {
              st[0] += t1;
              st[1] += t2;
            }
          }
          break;
        case TCMOD_STRETCH:
          {
            float factor = 1/State::Eval(f->p);
            for (i = 0; i < input.numverts; ++i, st += 2)
            {
              st[0] = (st[0] - 0.5f) * factor + 0.5f;
              st[1] = (st[1] - 0.5f) * factor + 0.5f;
            }
          }
          break;
        case TCMOD_TRANSFORM:
          {
            for (i = 0; i < input.numverts; ++i, st += 2)
            {
              t1 = st[0];
              t2 = st[1];
              st[0] = t1*f->p[0] + t2*f->p[2] + f->p[4];
              st[1] = t2*f->p[1] + t1*f->p[3] + f->p[5];
            }
          }
          break;
        case TCMOD_TURB:
          {
            t1 = f->p[2] + (float)Timer::fTime * f->p[3];
            t2 = f->p[1];
            for (i = 0; i < input.numverts; ++i, st += 2)
            {
              st[0] += FastSin(st[0]*t2+t1)*t2;
              st[1] += FastSin(st[1]*t2+t1)*t2;
            }
          }
          break;
        default:
          break;
      }
    }
  }
}

void Render::SetVertexColors(int tmu, Layer *l)
{
  if (r_novertexcolor.ivalue)
  {
    memset(buffer[tmu].c, 255, (input.numverts<<2));
    return;
  }

  if (State::curr_shader->num_layers)
  {
    glColorPointer(4, GL_UNSIGNED_BYTE, 0, buffer[tmu].c);
    CheckGLError();

//    if (buffer[tmu].c_content == -1 || buffer[tmu].c_content != l->rgbGen)
//    {
//      buffer[tmu].c_content = l->rgbGen;

      switch (l->rgbGen)
      {
        case RGBGEN_NONE:
        case RGBGEN_IDENTITY:
          memset(buffer[tmu].c, 255, (input.numverts<<2));
          break;
        case RGBGEN_IDENTITY_LIGHTING:
          {
            for (int i = 0; i < input.numverts; ++i)
            {
              buffer[tmu].c[(i<<2)  ] = 127;
              buffer[tmu].c[(i<<2)+1] = 127;
              buffer[tmu].c[(i<<2)+2] = 127;
              buffer[tmu].c[(i<<2)+3] = 255;
            }
          }
          break;
        case RGBGEN_VERTEX:
        case RGBGEN_EXACTVERTEX:
          memcpy(buffer[tmu].c, input.c, input.numverts<<2);
          break;
        case RGBGEN_ONE_MINUS_VERTEX:
          {
            for (int i = 0; i < input.numverts; ++i)
            {
              buffer[tmu].c[(i<<2)  ] = 255 - input.c[(i<<2)];
              buffer[tmu].c[(i<<2)+1] = 255 - input.c[(i<<2)+1];
              buffer[tmu].c[(i<<2)+2] = 255 - input.c[(i<<2)+2];
              buffer[tmu].c[(i<<2)+3] = 255;
            }
          }
          break;
        case RGBGEN_LIGHTING_DIFFUSE:
          memcpy(buffer[tmu].c, input.c, input.numverts<<2);
          break;
        case RGBGEN_WAVE:
          {
            int c = (int) (255.0f*State::Eval(l->rgbGenParams));
            if (c < 0) c = 0;
            if (c > 255) c = 255;
            buffer[tmu].c_content = -1;
            for (int i = 0; i < input.numverts; ++i)
            {
              buffer[tmu].c[(i<<2)  ] = c;
              buffer[tmu].c[(i<<2)+1] = c;
              buffer[tmu].c[(i<<2)+2] = c;
              buffer[tmu].c[(i<<2)+3] = 255;
            }
          }
          break;
        case RGBGEN_FOG:
          memcpy(buffer[tmu].c, input.c, input.numverts<<2);
          break;
        default:  // TODO: finish other rgbgen types
          memcpy(buffer[tmu].c, input.c, input.numverts<<2);
          break;
      }

      switch (l->alphaGen)
      {
        case ALPHAGEN_IDENTITY:
          {
            for (int i = 0; i < input.numverts; ++i)
              buffer[tmu].c[(i<<2)+3] = 255;
          }
          break;
        case ALPHAGEN_CONST:
          {
            // TODO: implement const support
          }
          break;
        case ALPHAGEN_WAVE:
          {
            int c = (int) (255.0f*State::Eval(l->alphaGenParams));
            if (c < 0) c = 0;
            if (c > 255) c = 255;
            for (int i = 0; i < input.numverts; ++i)
              buffer[tmu].c[(i<<2)+3] = c;
          }
          break;
        case ALPHAGEN_PORTAL:
          {/*
            VectorAdd ( vertexArray[0], currententity->cam.pos, v );
            VectorSubtract ( r_origin, v, t );
            a = VectorLength ( t ) * (1.0 / 255.0);
            clamp ( a, 0.0f, 1.0f );
            b = FloatToByte ( a );

            for ( i = 0; i < numColors; i++, bArray += 4 ) {
              bArray[3] = b;
            }*/
          }
          break;
        case ALPHAGEN_VERTEX:
          {
            for (int i = 0; i < input.numverts; ++i)
              buffer[tmu].c[(i<<2)+3] = input.c[(i<<2)+3];
          }
          break;

        case ALPHAGEN_ENTITY:
          {/*
            for (int i = 0; i < input.numverts; ++i)
              buffer[tmu].c[(i<<2)+3] = currententity->color[3];
            }*/
          }
          break;
        case ALPHAGEN_SPECULAR:
          {/*
            VectorSubtract ( r_origin, currententity->cam.pos, t );

            if (!Matrix3_Compare (currententity->axis, axis_identity) ) {
              Matrix3_Multiply_Vec3 ( currententity->axis, t, v );
            } else {
              VectorCopy ( t, v );
            }

            for (i = 0; i < numColors; i++, bArray += 4 ) {
              VectorSubtract ( v, vertexArray[i], t );
              a = DotProduct( t, normalsArray[i] ) * rSqrt ( DotProduct(t,t) );
              a = a * a * a * a * a;
              bArray[3] = FloatToByte ( bound (0.0f, a, 1.0f) );
            }*/
          }
          break;
        default:
          break;
      }
//    }
  }
  else
  {
    memcpy(buffer[tmu].c, input.c, input.numverts<<2);
  }
}

void Render::Perspective(float fov, float aspect)
{
  float xmin, xmax, ymin, ymax;
  float one_deltax, one_deltay, one_deltaz, doubleznear;
  float m[16];

  xmax = v_camnear.fvalue * FastTan(fov/2.f);
  xmin = -xmax;

  ymax = xmax / aspect;
  ymin = -ymax;

  //glFrustum(xmin, xmax, ymin, ymax, zNear, zFar); CheckGLError();

  doubleznear = 2.f * v_camnear.fvalue;
  one_deltax = (xmax - xmin);
  one_deltay = (ymax - ymin);
  one_deltaz = (v_camfar.fvalue - v_camnear.fvalue);

  m[0] = (float) (doubleznear / one_deltax);
  m[1] = 0.f;
  m[2] = 0.f;
  m[3] = 0.f;
  m[4] = 0.f;
  m[5] = (float) (doubleznear / one_deltay);
  m[6] = 0.f;
  m[7] = 0.f;
  m[8] = (float) ((xmax + xmin) / one_deltax);
  m[9] = (float) ((ymax + ymin) / one_deltay);
  m[10] = (float) (-(v_camfar.fvalue + v_camnear.fvalue) / one_deltaz);
  m[11] = -1.f;
  m[12] = 0.f;
  m[13] = 0.f;
  m[14] = (float) (-(v_camfar.fvalue * doubleznear) / one_deltaz);
  m[15] = 0.f;

  glLoadMatrixf(m);
  CheckGLError();
}

void Render::DumpGLState(void)
{
  GLboolean bool_buffer;
  GLfloat float_buffer;
  GLint int_buffer;

  gLogFile->OpenFile();

  // blending
  gConsole->Insertln("State report:");
  glGetBooleanv(GL_BLEND, &bool_buffer);
  gConsole->Insertln("\tblending: %s", bool_buffer?"enabled":"disabled");
  glGetIntegerv(GL_BLEND_SRC, &int_buffer);
  gConsole->Insertln("\t\tblend src: 0x%x", int_buffer);
  if (int_buffer != State::curr_blendsrc) gConsole->Insertln("\t\t\t-> ^1state mismatch (%d)", State::curr_blendsrc);
  glGetIntegerv(GL_BLEND_DST, &int_buffer);
  gConsole->Insertln("\t\tblend dst: 0x%x", int_buffer);
  if (int_buffer != State::curr_blenddst) gConsole->Insertln("\t\t\t-> ^1state mismatch (%d)", State::curr_blenddst);

  // culling
  glGetBooleanv(GL_CULL_FACE, &bool_buffer);
  gConsole->Insertln("\tculling: %s", bool_buffer?"enabled":"disabled");
  if (State::curr_culling == CULL_NONE && bool_buffer) gConsole->Insertln("\t\t-> ^1state mismatch (%d)", State::curr_culling);
  glGetIntegerv(GL_CULL_FACE_MODE, &int_buffer);
  gConsole->Insertln("\t\tculling face mode: 0x%x", int_buffer);
  if (State::curr_culling != CULL_NONE &&
    ((State::curr_culling == CULL_FRONT && int_buffer != GL_FRONT) ||
     (State::curr_culling == CULL_BACK && int_buffer != GL_BACK)))
    gConsole->Insertln("\t\t\t-> ^1state mismatch (%d)", State::curr_culling);

  // depth
  glGetBooleanv(GL_DEPTH_TEST, &bool_buffer);
  gConsole->Insertln("\tdepth test: %s", bool_buffer?"enabled":"disabled");
  glGetBooleanv(GL_DEPTH_WRITEMASK, &bool_buffer);
  gConsole->Insertln("\t\tdepthwrite mask: %s", bool_buffer?"enabled":"disabled");
  if (bool_buffer != State::curr_depthWrite) gConsole->Insertln("\t\t-> ^1state mismatch (%s)", State::curr_depthWrite?"enabled":"disabled");
  glGetIntegerv(GL_DEPTH_FUNC, &int_buffer);
  gConsole->Insertln("\t\tdepth func: 0x%x", int_buffer);
  if (int_buffer != State::curr_depthFunc) gConsole->Insertln("\t\t\t-> ^1state mismatch (%d)", State::curr_depthFunc);

  // fog
  glGetBooleanv(GL_FOG, &bool_buffer);
  gConsole->Insertln("\tfog: %s", bool_buffer?"enabled":"disabled");
  glGetFloatv(GL_FOG_DENSITY, &float_buffer);
  gConsole->Insertln("\t\tfog density: %f", float_buffer);
  glGetFloatv(GL_FOG_START, &float_buffer);
  gConsole->Insertln("\t\tfog start: %f", float_buffer);
  glGetFloatv(GL_FOG_END, &float_buffer);
  gConsole->Insertln("\t\tfog end: %f", float_buffer);
  glGetIntegerv(GL_FOG_MODE, &int_buffer);
  gConsole->Insertln("\t\tfog mode: 0x%x", int_buffer);

  // lighting
  glGetBooleanv(GL_LIGHTING, &bool_buffer);
  gConsole->Insertln("\tlighting: %s", bool_buffer?"enabled":"disabled");

  // 1D texturing
  glGetBooleanv(GL_TEXTURE_1D, &bool_buffer);
  gConsole->Insertln("\t1D texturing: %s", bool_buffer?"enabled":"disabled");

  // 2D texturing
  glGetBooleanv(GL_TEXTURE_2D, &bool_buffer);
  gConsole->Insertln("\t2D texturing: %s", bool_buffer?"enabled":"disabled");

  // render mode
  glGetIntegerv(GL_RENDER_MODE, &int_buffer);
  gConsole->Insertln("\trendering mode: 0x%x", int_buffer);

  CheckGLError(18);

  gLogFile->CloseFile();
}

void Render::CheckGLError(int ncalls)
{
  // FIXME: Should we stop or throw an exception in release mode ?
  #ifdef _DEBUG
    GLenum glerr = glGetError();
    if (glerr)
    {
      /*
        GL_NO_ERROR                       0
        GL_INVALID_ENUM                   0x0500
        GL_INVALID_VALUE                  0x0501
        GL_INVALID_OPERATION              0x0502
        GL_STACK_OVERFLOW                 0x0503
        GL_STACK_UNDERFLOW                0x0504
        GL_OUT_OF_MEMORY                  0x0505
      */
      char errDescr[32] = { '\0' };
      switch (glerr)
      {
        case GL_INVALID_ENUM:       sprintf(errDescr, "GL_INVALID_ENUM"); break;
        case GL_INVALID_VALUE:      sprintf(errDescr, "GL_INVALID_VALUE"); break;
        case GL_INVALID_OPERATION:  sprintf(errDescr, "GL_INVALID_OPERATION"); break;
        case GL_STACK_OVERFLOW:     sprintf(errDescr, "GL_STACK_OVERFLOW"); break;
        case GL_STACK_UNDERFLOW:    sprintf(errDescr, "GL_STACK_UNDERFLOW"); break;
        case GL_OUT_OF_MEMORY:      sprintf(errDescr, "GL_OUT_OF_MEMORY"); break;
        default :                   sprintf(errDescr, "UNDEFINED_ERROR"); break;
      }

      gLogFile->Insert("^1OpenGL error detected: %s\n", errDescr);

      //__asm int 3;
    }
  #endif
  num_apicalls += ncalls;
}

//-----------------------------------------------------------------------------
// State
//-----------------------------------------------------------------------------

void State::SetInitialState(int multitexture)
{
  // client state:
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);

  // texturing:
  glEnable(GL_TEXTURE_2D);
  curr_texture[0] = -1;
  
  // second texture unit
  if (multitexture)
  {
    glActiveTextureARB(GL_TEXTURE1_ARB);
    glClientActiveTextureARB(GL_TEXTURE1_ARB);
    glDisable(GL_TEXTURE_2D);
    glActiveTextureARB(GL_TEXTURE0_ARB);
    glClientActiveTextureARB(GL_TEXTURE0_ARB);
    curr_texture[1] = -1;
    gRender->CheckGLError(5);
  }
  
  // backface culling:
  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);
  curr_culling = CULL_BACK;

  // color:
  glShadeModel(GL_SMOOTH);
  glEnableClientState(GL_COLOR_ARRAY);

  if (gl_dither.ivalue) glEnable(GL_DITHER);
  else glDisable(GL_DITHER);

  // blending:
  glDisable(GL_BLEND);
  curr_blendsrc = GL_ONE;
  curr_blenddst = GL_ZERO;

  // alpha func
  glDisable(GL_ALPHA_TEST);
  glAlphaFunc(GL_ALWAYS, 1.0f);

  // depth test:
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glDepthMask(GL_TRUE);
  curr_depthFunc = GL_LEQUAL;
  curr_depthWrite = true;

  glEnable(GL_SCISSOR_TEST);
  glDisable(GL_POLYGON_OFFSET_FILL);
  glPolygonOffset(-1.f, -2.f);
  State::setPolygonMode(GL_FILL);

  switch (r_quality.ivalue)
  {
    case 0:
      glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
      break;
    case 1:
      glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_DONT_CARE);
      break;
    case 2:
    default:
      glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
      break;
  }

  glReadBuffer(GL_BACK);
  glDrawBuffer(GL_BACK);

  // clearing:
  glClearColor(0.2f, 0.3f, 0.4f, 1.0f);
  glClearDepth(1.0f);
  
  gRender->CheckGLError(21);
}

// not done
void State::SetDefaultState(void)
{
  pass = 0;
}

// binds the texture of the given layer
void State::BindTexture(Layer *l, int unit)
{
  int num;

  if (l->flags & LF_LIGHTMAP || r_noshader.ivalue)
  {
    if (curr_lightmap != ~0) num = curr_lightmap;
    else return;
  }
  else if (!l->map) return; // FIXME: that happens???
  else if (l->num_maps > 1)
  {
    int frame = ((int) floor(Timer::fTime*l->anim_speed)) % l->num_maps;
    clamp(frame, 0, l->num_maps-1);
    num = l->map[frame]->num;
  }
  else if (l->map[0]) num = l->map[0]->num;
  else return;        // FIXME: that happens???
  
  if (State::curr_texture[unit] != num)
  {
    glBindTexture(GL_TEXTURE_2D, num);
    gRender->CheckGLError();
    State::curr_texture[unit] = num;
  }
}

// set the appropiate state to render the layer
void State::setState(Layer *l)
{
  State::setBlend(l->blendsrc, l->blenddst);

  State::setAlphaFunc(l->alphafunc, l->alphafuncref);

  State::setDepthWrite(l->depthWrite);
  State::setDepthFunc(l->depthFunc);
}

void State::SetupShader(void)
{
  State::setCulling(State::curr_shader->cull);

  if (State::curr_shader->flags & SF_POLYGONOFFSET)
    glEnable(GL_POLYGON_OFFSET_FILL);
  else
    glDisable(GL_POLYGON_OFFSET_FILL);
  gRender->CheckGLError();
}

// evaluates a deform function
float State::Eval(float *p)
{
    float x, y = 0;

    // Evaluate a number of time based periodic functions
    // y = p[1] + p[2] * func((time + p[3]) * p[4])
    
  x = ((float)Timer::fTime + p[3]) * p[4];  //x = (Timer::fTime) * p[4] + p[3];
  x -= floorf(x);   // normalized

    switch ((int) (p[0]))
  {
    case FUNC_SIN:
      y = FastSin(x * M_TWO_PI);
      break;
    case FUNC_SQUARE:
      if (x < 0.5f) y = 1.0f;
      else y = -1.0f;
      break;
    case FUNC_SAWTOOTH:
      y = x;
      break;
    case FUNC_INVERSESAWTOOTH:
      y = 1.0f - x;
      break;
    case FUNC_TRIANGLE:
      if (x < 0.5f) y = 4.0f * x - 1.0f;
        else y = -4.0f * x + 3.0f;
      break;
    case FUNC_NOISE:
      y = (((float)rand()+(float)rand())/2.f)/(float)RAND_MAX;
      break;
    default:
      break;
    }

    return y * p[2] + p[1];
}

void State::setDepthFunc(int val)
{
  if (State::curr_depthFunc != val)
  {
    if (val == GL_NEVER) glDisable(GL_DEPTH_TEST);
    else if (State::curr_depthFunc == GL_NEVER) glEnable(GL_DEPTH_TEST);
    glDepthFunc(val);
    gRender->CheckGLError(2);
    State::curr_depthFunc = val;
  }
}
void State::setDepthWrite(int val)
{
  if (State::curr_depthWrite != val)
  {
    glDepthMask(val);
    gRender->CheckGLError();
    State::curr_depthWrite = val;
  }
}

void State::setBlend(int src, int dst)
{
  if (State::curr_blendsrc != src || State::curr_blenddst != dst)
  {
    if (src == GL_ONE && dst == GL_ZERO)
      glDisable(GL_BLEND);
    else
    {
      if (State::curr_blendsrc == GL_ONE && State::curr_blenddst == GL_ZERO)
      {
        glEnable(GL_BLEND);
        gRender->CheckGLError();
      }
      glBlendFunc(src, dst);
    }
    State::curr_blenddst = dst;
    State::curr_blendsrc = src;
    gRender->CheckGLError();
  }
}

void State::setCulling(int val)
{
  if (r_nocull.ivalue)
  {
    if (State::curr_culling != CULL_NONE)
    {
      State::curr_culling = CULL_NONE;
      glDisable(GL_CULL_FACE);
      gRender->CheckGLError();
    }
  }
  else if (val != State::curr_culling)
  {
    if (val)
    {
      glEnable(GL_CULL_FACE);
      if (val < 0) glCullFace(GL_FRONT);
      else glCullFace(GL_BACK);
      gRender->CheckGLError(2);
    }
    else
    {
      glDisable(GL_CULL_FACE);
      gRender->CheckGLError();
    }
    State::curr_culling = val;
  }
}

void State::setAlphaFunc(int func, float funcref)
{
  if (func != State::curr_alphaFunc)
  {
    if (func == GL_ALWAYS)
    {
      glDisable(GL_ALPHA_TEST);
      gRender->CheckGLError();
    }
    else
    {
      glEnable(GL_ALPHA_TEST);
      glAlphaFunc(func, funcref);
      gRender->CheckGLError(2);
    }
    State::curr_alphaFunc = func;
  }
}

void State::setPolygonMode(int val)
{
  if (val != State::curr_polygonmode)
  {
    switch (val)
    {
      case GL_POINT:
        glEnable(GL_POLYGON_OFFSET_POINT);
        glDisable(GL_POLYGON_OFFSET_LINE);
        glDisable(GL_POLYGON_OFFSET_FILL);
        break;
      case GL_LINE:
        glDisable(GL_POLYGON_OFFSET_POINT);
        glEnable(GL_POLYGON_OFFSET_LINE);
        glDisable(GL_POLYGON_OFFSET_FILL);
        break;
      case GL_FILL:
        glDisable(GL_POLYGON_OFFSET_POINT);
        glDisable(GL_POLYGON_OFFSET_LINE);
        glEnable(GL_POLYGON_OFFSET_FILL);
        break;
      default:
        return;   // invalid enum
    }

    if (State::curr_culling == CULL_BACK)
      glPolygonMode(GL_BACK, val);
    else if (State::curr_culling == CULL_FRONT)
      glPolygonMode(GL_FRONT, val);
    else
      glPolygonMode(GL_FRONT_AND_BACK, val);

    gRender->CheckGLError();
    State::curr_polygonmode = val;
  }
}
