//-----------------------------------------------------------------------------
// Shader
//-----------------------------------------------------------------------------

#include "shader.h"
#include "console.h"
#include "logfile.h"
#include "parser.h"
#include "surfaceflags.h"
#include "render.h"
#include "mem.h"
#include "definitions.h"

#include <assert.h>

// Constructor: allocate space for the hashes
ShaderManager::ShaderManager(const int num)
{
  num_refs = 0;
  max_refs = num;
  ShaderTable = (ShaderHash*) cake_malloc(num*sizeof(ShaderHash), "ShaderManager::ShaderManager.ShaderTable");
  
  // initialize the shadertable
  for (int i = 0; i < max_refs; ++i)
  {
    ShaderTable[i].contents = 0;
    ShaderTable[i].flags = 0;
    ShaderTable[i].list = NULL;
    memset(ShaderTable[i].name, '\0', 64*sizeof(char));
    ShaderTable[i].num_nodes = 0;
    ShaderTable[i].ref_count = 0;
    ShaderTable[i].shader = NULL;
    ShaderTable[i].type = 0;
  }
}

ShaderManager::~ShaderManager(void)
{
  cake_free(ShaderTable);
}

// Adds a reference to a shader
int ShaderManager::AddShader(const char *name, int flags, int contents, int type)
{
  int i;

  // search for name
  for (i = 0; i < num_refs; ++i)
  {
    if (!ShaderTable[i].name) continue;   // free slot

    if (!stricmp(ShaderTable[i].name, name))
    {
      // allready referenced, increase reference counter
      ++ShaderTable[i].ref_count;
      return i;
    }
  }

  if (num_refs >= max_refs)
  {
    // TODO: do something better here !!!
    gConsole->Insertln("^1ERROR: Out of shader refs slots!");
    return -1;
  }

  // find a free reference and allocate it
  for (i = 0; i < max_refs; ++i)
  {
    if (ShaderTable[i].ref_count == 0 && ShaderTable[i].shader == NULL)
    {
      ShaderTable[i].ref_count = 1;
      ShaderTable[i].shader = NULL;
      memset(ShaderTable[i].name, '\0', 64*sizeof(char));
      strncpy(ShaderTable[i].name, name, 64);
      ShaderTable[i].flags = flags;
      ShaderTable[i].contents = contents;
      ShaderTable[i].type = type;

      ++num_refs;
      return i;
    }
  }
  return -1;
}

// Deletes a reference, and his shader
// FIXME: not tested!!!
// FIXME: why not decrease the ref_count and update later???
void ShaderManager::DeleteShader(const int num)
{
  // decrease reference counter
  if (num >= 0 && ShaderTable[num].ref_count)
  {
    --ShaderTable[num].ref_count;

    // delete if we reached 0
    if (!ShaderTable[num].ref_count)
    {
      delete ShaderTable[num].shader; ShaderTable[num].shader = NULL;
      --num_refs;
    }
  }
}

Shader* ShaderManager::GetShader(const int num)
{
  return (num_refs&&num>=0)?ShaderTable[num].shader:NULL;
}

char* ShaderManager::GetShaderName(const int num)
{
  return (num_refs&&num>=0)?ShaderTable[num].name:NULL;
}

int ShaderManager::GetShaderNum(const char* name)
{
  for (int i = 0; i < num_refs; ++i)
    if (!stricmp(ShaderTable[i].name, name)) return i;

  return -1;
}

int ShaderManager::GetShadersNum(void)
{
  return num_refs;
}

// reset all reference counters
void ShaderManager::ResetAll(void)
{
  gConsole->Insertln("--- Unreferencing shaders...");
  for (int i = 0; i < max_refs; ++i)
    ShaderTable[i].ref_count = 0;

  textures.ResetAll();
}

// loads unloaded shaders, deletes unreferenced shaders
void ShaderManager::Update(bool pak_search)
{
  int i;

  gLogFile->OpenFile();

#if SHADER_USE_REFERENCES
  shader_ref_list *shaderref;
  for (i = 0; i < max_refs; ++i)
  {
    if (!ShaderTable[i].shader && ShaderTable[i].ref_count)
    {
      if (!(shaderref = RefForShader(ShaderTable[i].name)))
      {
        // Not referenced as shader
        // Should be a texture or reference has changed during runtime
        continue;
      }
      assert(shaderref->offset >= 0);

      gConsole->Insertln("...loading \"%s\" (from \"%s\")", ShaderTable[i].name, shaderref->file_name);
      
      // TODO: Optimize here to avoid looking in all packages
      // (this was already done during initialisation and package is known)
      ShaderFile file((char *) shaderref->file_name, shaderref->pak_name);
      file.SetShaderPos(shaderref->offset);

      // create blank shader
      Shader *shad = ShaderTable[i].shader = CreateShader();
      
      // set the appropiate flags
      shad->surface_flags = ShaderTable[i].flags;
      shad->content_flags = ShaderTable[i].contents;
      
      // set the appropriate name
      strncpy(shad->shadername, shaderref->shader_name, 64);

      // parse it, and fill shader structure
      file.Parse(shad, &layers, &textures);
      
      // precalculate multitexture collapsing
      if (shad->num_layers > 1)
      {
        for (int i = 0; i < shad->num_layers - 1; ++i)
        {
          if (shad->layer[i]->blendsrc == IntForConst("GL_ONE") && 
            shad->layer[i]->blenddst == IntForConst("GL_ZERO"))
          {
            if ((shad->layer[i+1]->blendsrc == IntForConst("GL_DST_COLOR")  && 
                 shad->layer[i+1]->blenddst == IntForConst("GL_ZERO"))      ||
                (shad->layer[i+1]->blendsrc == IntForConst("GL_ZERO")       && 
                 shad->layer[i+1]->blenddst == IntForConst("GL_SRC_COLOR")))
            {
              shad->layer[i]->flags |= LF_MULTITEX;
              shad->flags |= SF_MULTITEX;
              //gConsole->Insertln("^5WARNING: two layers collapsed!");
            }
          }
        }
      }
    }
  }
#else
  char *filename, *shadername;

  // don't use the shader references
  for (filename = GetFirstShadersFile(); filename; filename = NextShadersFile())
  {
    // opens the file
    ShaderFile file((char *) filename);

    // parse each shader
    while ((shadername = file.GetShaderName()))
    {
      // store position of shader in the file
      int shaderpos = file.GetShaderPos();

      for (i = 0; i < max_refs; ++i)
      {                         // load it if
        if (!ShaderTable[i].shader &&         // not a free slot
          ShaderTable[i].ref_count &&         // has a reference
          !stricmp(ShaderTable[i].name, shadername))  // the names are equal
        {
          gConsole->Insertln("...loading \"%s\" (from \"%s\")", shadername, filename);

          // create blank shader
          Shader *shad = ShaderTable[i].shader = CreateShader();

          // set the appropiate flags
          shad->surface_flags = ShaderTable[i].flags;
          shad->content_flags = ShaderTable[i].contents;

          // set the appropriate name
          strncpy(shad->shadername, shadername, 64);

          // parse it, and fill shader structure
          file.Parse(shad, &layers, &textures);

          // precalculate multitexture collapsing
          if (shad->num_layers > 1)
          {
            for (int i = 0; i < shad->num_layers - 1; ++i)
            {
              if (shad->layer[i]->blendsrc == IntForConst("GL_ONE") && 
                shad->layer[i]->blenddst == IntForConst("GL_ZERO"))
              {
                if ((shad->layer[i+1]->blendsrc == IntForConst("GL_DST_COLOR")  && 
                     shad->layer[i+1]->blenddst == IntForConst("GL_ZERO"))      ||
                    (shad->layer[i+1]->blendsrc == IntForConst("GL_ZERO")       && 
                     shad->layer[i+1]->blenddst == IntForConst("GL_SRC_COLOR")))
                {
                  shad->layer[i]->flags |= LF_MULTITEX;
                  shad->flags |= SF_MULTITEX;
                  //gConsole->Insertln("^5WARNING: two layers collapsed!");
                }
              }
            }
          }

          // restore the shader start position
          file.SetShaderPos(shaderpos);
        }
      }

      // go to next shader
      file.Skip();
    } /* while loop */
  }
#endif

  for (i = 0; i < max_refs; ++i)
  {
    if (ShaderTable[i].ref_count)
    {
      // load referenced shaders that havent been found using default shader
      if (ShaderTable[i].shader == NULL)
      {
        ShaderTable[i].shader = CreateShader();
        strncpy(ShaderTable[i].shader->shadername, ShaderTable[i].name, 64);
        ShaderTable[i].shader->surface_flags = ShaderTable[i].flags;
        ShaderTable[i].shader->content_flags = ShaderTable[i].contents;
        DefaultShader(i);
      }
    }
    else
    {
      // delete non-referenced shaders
      DeleteReference(i);
    }
  }
  gLogFile->CloseFile();

  gConsole->Insertln("ShaderMng: %d shader%s in memory", num_refs, num_refs>1?"s":"");

  // reallocate surface buffers
  for (i = 0; i < max_refs; ++i)
  {
    ShaderTable[i].num_nodes = 0;

    if (ShaderTable[i].list) delete [] ShaderTable[i].list;
    ShaderTable[i].list = NULL;

    if (ShaderTable[i].ref_count)
    {
      ShaderTable[i].list = new Node[ShaderTable[i].ref_count];
    }
  }

  textures.Update();
}

// deletes all the shaders
void ShaderManager::DeleteAll(void)
{
}

// loads the default shader and puts it in ShaderTable[i]
void ShaderManager::DefaultShader(int i)
{
  ShaderHash *r = &ShaderTable[i];
  r->shader->surface_flags = r->flags;
  r->shader->content_flags = r->contents;

  if (r->type == FACETYPE_MESH)
  {
    r->shader->flags = SF_HASCOLORS;
    r->shader->num_layers = 1;
    r->shader->layer[0] = layers.CreateLayer();
    r->shader->layer[0]->depthWrite = 1;
    r->shader->layer[0]->blendsrc = IntForConst("GL_ONE");
    r->shader->layer[0]->blenddst = IntForConst("GL_ZERO");
    r->shader->layer[0]->rgbGen = RGBGEN_VERTEX;
    r->shader->layer[0]->flags = 0;

    r->shader->sky_params.num_layers = 0;
    
    // load texture into memory
    r->shader->layer[0]->map[0] = textures.AddTexinfo(r->name, r->shader->layer[0]->flags);
  }
  else
  {
    r->shader->flags = SF_HASLIGHTMAP | SF_MULTITEX;
    
    r->shader->num_layers = 2;
    r->shader->layer[0] = layers.CreateLayer();
    r->shader->layer[0]->depthWrite = 1;
    r->shader->layer[0]->blendsrc = IntForConst("GL_ONE");
    r->shader->layer[0]->blenddst = IntForConst("GL_ZERO");
    r->shader->layer[0]->flags = LF_LIGHTMAP | LF_NOMIPMAP | LF_NOPICMIP | LF_CLAMP | LF_MULTITEX;
    r->shader->layer[0]->tcGen = TCGEN_LIGHTMAP;
    r->shader->layer[0]->rgbGen = RGBGEN_IDENTITY;
    
    r->shader->layer[1] = layers.CreateLayer();
    r->shader->layer[1]->depthWrite = 1;
    r->shader->layer[1]->blendsrc = IntForConst("GL_DST_COLOR");
    r->shader->layer[1]->blenddst = IntForConst("GL_ZERO");
    r->shader->layer[1]->flags = 0;

    r->shader->sky_params.num_layers = 0;

    // load texture into memory
    r->shader->layer[1]->map[0] = textures.AddTexinfo(r->name, r->shader->layer[1]->flags);
  }
}

// CreateShader allocates a shader and fills it with default values
// TODO: we could improve that using a linear allocator to store the shaders
Shader *ShaderManager::CreateShader(void)
{
  // allocate shader.
  Shader *shad = new Shader;
  int i;

  // fill shader with default values of an empty shader.
  memset(shad->shadername, 0, 64*sizeof(char));
  shad->surface_flags = 0;
  shad->content_flags = 0;
  shad->cull = CULL_BACK;
  shad->sortvalue = SORT_NORMAL;
  shad->flags = 0;
  shad->num_layers = 0;

  for (i = 0; i < MAX_LAYERS; ++i) shad->layer[i] = NULL;
  for (i = 0; i < MAX_DEFORMS; ++i)
  {
    shad->deformVertexes[i].type = DEFORMVERTEXES_NONE;
    for (int j = 0; j < 8; ++j) shad->deformVertexes[i].params[j] = 0;
  }
  shad->num_deforms = 0;

  // Fog parameters
  shad->fog_params.layer = NULL;
  shad->fog_params.params[0] = 0;
  shad->fog_params.params[1] = 0;
  shad->fog_params.params[2] = 0;
  shad->fog_params.params[3] = 0;
  shad->fog_params.params[4] = 0;

  // Sky parameters
  shad->sky_params.cloudheight = 512;
  shad->sky_params.num_layers = 0;

  return shad;
}

void ShaderManager::DeleteReference(int ref)
{
  if (ShaderTable[ref].shader)
  {
    int i;
    for (i = 0; i < MAX_LAYERS; ++i)
    {
      if (!ShaderTable[ref].shader->layer[i]) break;
      
      layers.DeleteLayer(ShaderTable[ref].shader->layer[i]);
      ShaderTable[ref].shader->layer[i] = NULL;
    }

    // Delete sky layers
    for (i = 0; i < ShaderTable[ref].shader->sky_params.num_layers; ++i)
    {
      for (int j = 0; j < 6; ++j)
      {
        layers.DeleteLayer(ShaderTable[ref].shader->sky_params.layer[i][j]);
        ShaderTable[ref].shader->sky_params.layer[i][j] = NULL;
      }
    }

    ShaderTable[ref].shader->sky_params.num_layers = 0;

    delete ShaderTable[ref].shader;
    ShaderTable[ref].shader = NULL;
    --num_refs;
  }
}

//-----------------------------------------------------------------------------
// ShaderFile
//-----------------------------------------------------------------------------

ShaderFile::ShaderFile(const char *name, const char *pakname):VFile(name, true, pakname)
{
  Parser::StartParseBuffer(mem, size);
}

ShaderFile::~ShaderFile()
{
  Parser::StopParseFile();
}

char* ShaderFile::GetShaderName(void)
{
  if (mem)
  {
    if (Parser::GetToken(true))
      return Parser::token;
  }
  return NULL;
}

void ShaderFile::SetShaderPos(int pos)
{
  if (mem)
    Parser::GoTo(pos);
}

int ShaderFile::GetShaderPos(void)
{
  if (mem)
    return Parser::GetOffset();
  else
    return -1;
}

void ShaderFile::Parse(Shader *shad, LayerManager* layers, TextureManager* textures)
{
  shad->num_layers = 0;
  shad->sky_params.num_layers = 0;

  if (Parser::GetToken(true))
  {
    if (strcmp(Parser::token, "{"))
    {
      gConsole->Insertln("^5WARNING: '{' expected in line %d (found \"%s\")", Parser::scriptline, Parser::token);
      return;
    }
  }

  while (Parser::GetToken(true))  // while not EOF
  {
    if (!strcmp(Parser::token, "{"))
    {
      shad->layer[shad->num_layers] = layers->CreateLayer();
      ParseLayer(shad, shad->layer[shad->num_layers], textures);
      ++shad->num_layers;

      continue;
    }

    else if (!strcmp(Parser::token, "}")) break;  // ends shader
    
    else if (!stricmp(Parser::token, "cull"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }

      if (!stricmp(Parser::token, "front")) shad->cull = CULL_FRONT;
      else if (!stricmp(Parser::token, "back")) shad->cull = CULL_BACK;
      else if (!stricmp(Parser::token, "none") ||
               !stricmp(Parser::token, "disable") ||
               !stricmp(Parser::token, "twosided")) shad->cull = CULL_NONE;
      else
      {
        // unknown parameter !!
        gConsole->Insertln("^5WARNING: unknown parameter \"%s\" in line %d", Parser::token, Parser::scriptline);
      }
    }

    else if (!stricmp(Parser::token, "nopicmip"))
    {
      shad->flags |= SF_NOPICMIP;
    }
    
    else if (!stricmp(Parser::token, "nomipmaps"))
    {
      shad->flags |= SF_NOMIPMAP;
    }
    
    else if (!stricmp(Parser::token, "polygonOffset"))
    {
      shad->flags |= SF_POLYGONOFFSET;
    }
    
    else if (!stricmp(Parser::token, "portal"))
    {
      shad->flags |= SF_PORTAL;
      shad->sortvalue = SORT_PORTAL;
    }
    
    else if (!stricmp(Parser::token, "sort"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      
      if (!stricmp(Parser::token, "portal"))
      {
        shad->flags |= SF_PORTAL;
        shad->sortvalue = SORT_PORTAL;
      }
      else if (!stricmp(Parser::token, "sky")) shad->sortvalue = SORT_SKY;
      else if (!stricmp(Parser::token, "opaque")) shad->sortvalue = SORT_OPAQUE;
      else if (!stricmp(Parser::token, "banner")) shad->sortvalue = SORT_BANNER;
      else if (!stricmp(Parser::token, "underwater")) shad->sortvalue = SORT_UNDERWATER;
      else if (!stricmp(Parser::token, "additive")) shad->sortvalue = SORT_ADDITIVE;
      else if (!stricmp(Parser::token, "nearest")) shad->sortvalue = SORT_NEAREST;
      else // integer value
      {
        shad->sortvalue = atoi(Parser::token);
        if (shad->sortvalue == 0)
        {
          // FIXME: does atoi return 0 when no isdigit?
          shad->sortvalue = SORT_OPAQUE;
          // unknown parameter !!
          gConsole->Insertln("^5WARNING: invalid parameter \"%s\" in line %d", Parser::token, Parser::scriptline);
        }
      }
    }
    else if (!stricmp(Parser::token, "deformvertexes"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      
      if (!stricmp(Parser::token, "wave"))
      {
        // Search for a free indice
        int idx;
        for (idx = 0; idx < MAX_DEFORMS; ++idx)
          if (shad->deformVertexes[idx].type == DEFORMVERTEXES_NONE) break;
        if (idx >= MAX_DEFORMS)
        {
          gConsole->Insertln("Unable to allocate more than %d deformvertexes", MAX_DEFORMS);
          continue;
        }

        shad->deformVertexes[idx].type = DEFORMVERTEXES_WAVE;
        ++shad->num_deforms;
        if (!Parser::GetToken(false))
        {
          gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
          continue;
        }
        shad->deformVertexes[idx].params[1] = (float) atof(Parser::token);
        shad->deformVertexes[idx].params[0] = ParseFunc();
        for (int i = 2; i < 6; ++i)
        {
          if (!Parser::GetToken(false))
          {
            gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
            break;
          }
          shad->deformVertexes[idx].params[i] = (float) atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "normal"))
      {
        // Search for a free indice
        int idx;
        for (idx = 0; idx < MAX_DEFORMS; ++idx)
          if (shad->deformVertexes[idx].type == DEFORMVERTEXES_NONE) break;
        if (idx >= MAX_DEFORMS)
        {
          gConsole->Insertln("Unable to allocate more than %d deformvertexes", MAX_DEFORMS);
          continue;
        }

        shad->deformVertexes[idx].type = DEFORMVERTEXES_NORMAL;
        ++shad->num_deforms;

        // Reading amplitude and frequency
        for (int i = 0; i < 2; ++i)
        {
          if (!Parser::GetToken(false))
          {
            gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
            break;
          }
          shad->deformVertexes[idx].params[i] = (float) atof(Parser::token);
        }       
      }
      else if (!stricmp(Parser::token, "bulge"))
      {
        // Search for a free indice
        int idx;
        for (idx = 0; idx < MAX_DEFORMS; ++idx)
          if (shad->deformVertexes[idx].type == DEFORMVERTEXES_NONE) break;
        if (idx >= MAX_DEFORMS)
        {
          gConsole->Insertln("Unable to allocate more than %d deformvertexes", MAX_DEFORMS);
          continue;
        }

        shad->deformVertexes[idx].type = DEFORMVERTEXES_BULGE;
        ++shad->num_deforms;
        if (!Parser::GetToken(false)) break;
        shad->deformVertexes[idx].params[0] = (float) atof(Parser::token);
        if (!Parser::GetToken(false)) break;
        shad->deformVertexes[idx].params[1] = (float) atof(Parser::token);
        if (!Parser::GetToken(false)) break;
        shad->deformVertexes[idx].params[2] = (float) atof(Parser::token);
      }
      else if (!stricmp(Parser::token, "move"))
      {
        // Search for a free indice
        int idx, i;
        for (idx = 0; idx < MAX_DEFORMS; ++idx)
          if (shad->deformVertexes[idx].type == DEFORMVERTEXES_NONE) break;
        if (idx >= MAX_DEFORMS)
        {
          gConsole->Insertln("Unable to allocate more than %d deformvertexes", MAX_DEFORMS);
          continue;
        }

        // FIXME: This was based on .shader content, but it doesn't respect Q3A Shader Manual !!
        shad->deformVertexes[idx].type = DEFORMVERTEXES_MOVE;
        ++shad->num_deforms;
        for (i = 1; i < 4; ++i)
        {
          if (!Parser::GetToken(false)) continue;
          shad->deformVertexes[idx].params[i] = (float) atof(Parser::token);
        }
        shad->deformVertexes[idx].params[0] = ParseFunc();
        for (i = 4; i < 8; ++i)
        {
          if (!Parser::GetToken(false)) continue;
          shad->deformVertexes[idx].params[i] = (float) atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "autosprite"))
      {
        // Search for a free indice
        int idx;
        for (idx = 0; idx < MAX_DEFORMS; ++idx)
          if (shad->deformVertexes[idx].type == DEFORMVERTEXES_NONE) break;
        if (idx >= MAX_DEFORMS)
        {
          gConsole->Insertln("Unable to allocate more than %d deformvertexes", MAX_DEFORMS);
          continue;
        }

        shad->deformVertexes[idx].type = DEFORMVERTEXES_AUTOSPRITE;
        ++shad->num_deforms;
      }
      else if (!stricmp(Parser::token, "autosprite2"))
      {
        // Search for a free indice
        int idx;
        for (idx = 0; idx < MAX_DEFORMS; ++idx)
          if (shad->deformVertexes[idx].type == DEFORMVERTEXES_NONE) break;
        if (idx >= MAX_DEFORMS)
        {
          gConsole->Insertln("Unable to allocate more than %d deformvertexes", MAX_DEFORMS);
          continue;
        }

        shad->deformVertexes[idx].type = DEFORMVERTEXES_AUTOSPRITE2;
        ++shad->num_deforms;
      }
      else
      {
        // unknown parameter !!
        gConsole->Insertln("^5WARNING: unknown parameter \"%s\" in line %d", Parser::token, Parser::scriptline);
      }
    }

    else if (!stricmp(Parser::token, "surfaceparm"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (!stricmp(Parser::token, "alphashadow"))
      {
        shad->surface_flags |= SURF_ALPHASHADOW;
      }
      else if (!stricmp(Parser::token, "areaportal"))
      {
        shad->content_flags |= CONTENTS_AREAPORTAL;
      }
      else if (!stricmp(Parser::token, "clusterportal"))
      {
        shad->content_flags |= CONTENTS_CLUSTERPORTAL;
      }
      else if (!stricmp(Parser::token, "donotenter"))
      {
        shad->content_flags |= CONTENTS_DONOTENTER;
      }
      else if (!stricmp(Parser::token, "flesh"))
      {
        shad->surface_flags |= SURF_FLESH;
      }
      else if (!stricmp(Parser::token, "fog"))
      {
        shad->content_flags |= CONTENTS_FOG;
      }
      else if (!stricmp(Parser::token, "lava"))
      {
        shad->content_flags |= CONTENTS_LAVA;
      }
      else if (!stricmp(Parser::token, "metalsteps"))
      {
        shad->surface_flags |= SURF_METALSTEPS;
      }
      else if (!stricmp(Parser::token, "nodamage"))
      {
        shad->surface_flags |= SURF_NODAMAGE;
      }
      else if (!stricmp(Parser::token, "nodlight"))
      {
        shad->surface_flags |= SURF_NODLIGHT;
      }
      else if (!stricmp(Parser::token, "nodraw"))
      {
        shad->surface_flags |= SURF_NODRAW;
      }
      else if (!stricmp(Parser::token, "nodrop"))
      {
        shad->content_flags |= CONTENTS_NODROP;
      }
      else if (!stricmp(Parser::token, "noimpact"))
      {
        shad->surface_flags |= SURF_NOIMPACT;
      }
      else if (!stricmp(Parser::token, "nomarks"))
      {
        shad->surface_flags |= SURF_NOMARKS;
      }
      else if (!stricmp(Parser::token, "nolightmap"))
      {
        shad->surface_flags |= SURF_NOLIGHTMAP;
      }
      else if (!stricmp(Parser::token, "nosteps"))
      {
        shad->surface_flags |= SURF_NOSTEPS;
      }
      else if (!stricmp(Parser::token, "nonsolid"))
      {
        shad->surface_flags |= SURF_NONSOLID;
      }
      else if (!stricmp(Parser::token, "origin"))
      {
        shad->content_flags |= CONTENTS_ORIGIN;
      }
      else if (!stricmp(Parser::token, "playerclip"))
      {
        shad->content_flags |= CONTENTS_PLAYERCLIP;
      }
      else if (!stricmp(Parser::token, "slick"))
      {
        shad->surface_flags |= SURF_SLICK;
      }
      else if (!stricmp(Parser::token, "slime"))
      {
        shad->content_flags |= CONTENTS_SLIME;
      }
      else if (!stricmp(Parser::token, "structural"))
      {
        shad->content_flags |= CONTENTS_STRUCTURAL;
      }
      else if (!stricmp(Parser::token, "trans"))
      {
        shad->content_flags |= CONTENTS_TRANSLUCENT;
      }
      else if (!stricmp(Parser::token, "water"))
      {
        shad->content_flags |= CONTENTS_WATER;
      }
      else if (!stricmp(Parser::token, "nomipmaps"))
      {
      //  gConsole->Insertln("^5WARNING: unexpected \"nomipmaps\" parameter found in line %d", Parser::scriptline);
        shad->flags |= LF_NOMIPMAP;
      }
      else if (!stricmp(Parser::token, "sky"))
      {
      //  gConsole->Insertln("^5WARNING: unexpected \"sky\" parameter found in line %d", Parser::scriptline);
        shad->surface_flags |= SURF_SKY;
      }
      else
      {
        gConsole->Insertln("^5WARNING: unknown parameter \"%s\" in line %d", Parser::token, Parser::scriptline);
      }
    }
    else if (!stricmp(Parser::token, "fogparms"))
    { // syntax: fogparms ( <red> <green> <blue> ) <distance to Opaque>
      shad->content_flags |= CONTENTS_FOG;

      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }

      if (strcmp(Parser::token, "("))
      {
        gConsole->Insertln("^5WARNING: expected \"(\" character in line %d", Parser::scriptline);
        continue;
      }

      if (!Parser::GetToken(false)) break;
      shad->fog_params.params[0] = (float) atof(Parser::token);
      if (!Parser::GetToken(false)) break;
      shad->fog_params.params[1] = (float) atof(Parser::token);
      if (!Parser::GetToken(false)) break;
      shad->fog_params.params[2] = (float) atof(Parser::token);
      shad->fog_params.params[3] = 1.f;

      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }

      if (strcmp(Parser::token, ")"))
      {     
        gConsole->Insertln("^5WARNING: ) expected in line %d", Parser::scriptline);
        continue;
      }

      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }

      // This parameter represents 1/density
      shad->fog_params.params[4] = (float) atof(Parser::token);

      // Create the new fog layer
      shad->fog_params.layer = layers->CreateLayer();
      shad->fog_params.layer->blendsrc = IntForConst("GL_SRC_ALPHA");
      shad->fog_params.layer->blenddst = IntForConst("GL_ONE_MINUS_SRC_ALPHA");
      shad->fog_params.layer->depthFunc = IntForConst("GL_LEQUAL");
      shad->fog_params.layer->num_maps = 1;
      shad->fog_params.layer->map[0] = &gRender->fogtexture;
      shad->fog_params.layer->alphafunc = IntForConst("GL_GREATER");
      shad->fog_params.layer->alphafuncref = 0.05f;
      shad->fog_params.layer->rgbGen = RGBGEN_FOG;
      shad->fog_params.layer->tcGen = TCGEN_FOG;
//    shad->fog_params.layer->depthWrite = false;
      shad->sortvalue = SORT_FOG;
    }
    else if (!stricmp(Parser::token, "skyparms"))
    {
      bool hasFarBox = false;

      // farbox parameter
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: farbox parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (strcmp(Parser::token, "-"))
      {
        hasFarBox = true;

        // a farbox is present - we need to load the 6 env textures
        for (int i = 0; i < 6; ++i)
        {
          shad->sky_params.layer[shad->sky_params.num_layers][i] = layers->CreateLayer();
          shad->sky_params.layer[shad->sky_params.num_layers][i]->blendsrc = IntForConst("GL_ONE");
          shad->sky_params.layer[shad->sky_params.num_layers][i]->blenddst = IntForConst("GL_ZERO");
          shad->sky_params.layer[shad->sky_params.num_layers][i]->depthWrite = false;
          shad->sky_params.layer[shad->sky_params.num_layers][i]->alphafunc = IntForConst("GL_GEQUAL");
          shad->sky_params.layer[shad->sky_params.num_layers][i]->alphafuncref = 0.5f;
        }

        char texname[64] = { '\0' };
        sprintf(texname, "%s_rt", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][0]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_bk", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][1]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_up", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][2]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_lf", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][3]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_ft", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][4]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_dn", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][5]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        ++shad->sky_params.num_layers;
      }

      // cloudheight parameter
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: cloudheight parameter expected in line %d", Parser::scriptline);
        continue;
      }

      if (strcmp(Parser::token, "-"))
        shad->sky_params.cloudheight = (int) atoi(Parser::token);
      else
        shad->sky_params.cloudheight = 128; // default value

      // nearbox parameter
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: nearbox parameter expected in line %d", Parser::scriptline);
        continue;
      }
  
      if (strcmp(Parser::token, "-"))
      {
        // a nearbox is present - we need to load the 6 env textures
        for (int i = 0; i < 6; ++i)
        {
          shad->sky_params.layer[shad->sky_params.num_layers][i] = layers->CreateLayer();
          shad->sky_params.layer[shad->sky_params.num_layers][i]->blendsrc = IntForConst("GL_ONE");
          if (hasFarBox)
            shad->sky_params.layer[shad->sky_params.num_layers][i]->blenddst = IntForConst("GL_ONE");
          else  // no blending if no farbox
            shad->sky_params.layer[shad->sky_params.num_layers][i]->blenddst = IntForConst("GL_ZERO");
          shad->sky_params.layer[shad->sky_params.num_layers][i]->depthWrite = false;
          shad->sky_params.layer[shad->sky_params.num_layers][i]->alphafunc = IntForConst("GL_GEQUAL");
          shad->sky_params.layer[shad->sky_params.num_layers][i]->alphafuncref = 0.5f;
        }

        char texname[64] = { '\0' };
        sprintf(texname, "%s_rt", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][0]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_bk", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][1]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_up", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][2]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_lf", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][3]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_ft", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][4]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        sprintf(texname, "%s_dn", Parser::token); shad->sky_params.layer[shad->sky_params.num_layers][5]->map[0] = textures->AddTexinfo(texname, LF_CLAMP);
        ++shad->sky_params.num_layers;
      }
    }

    // skip non-used commands
    else if (!strnicmp(Parser::token, "qer_", 4))
    { // ignore
      while (Parser::GetToken(false)){};
      continue;
    }
    else if (!strnicmp(Parser::token, "q3map_", 6))
    { // ignore
      while (Parser::GetToken(false)){};
      continue;
    }
    else if (!stricmp(Parser::token, "light"))
    { // ignore
      while (Parser::GetToken(false)){};
      continue;
    }
    else if (!stricmp(Parser::token, "tessSize"))
    { // ignore
      while (Parser::GetToken(false)){};
      continue;
    }
    else // unknown command
      gConsole->Insertln("^5WARNING: unknown command \"%s\" in line %d", Parser::token, Parser::scriptline);

    // unknown keyword skip line
    while (Parser::GetToken(false)) // while not EOL of EOF
      gConsole->Insertln("^5WARNING: skipping \"%s\" in line %d", Parser::token, Parser::scriptline);
  }
}

void ShaderFile::Skip(void)   // skip this shader
{
  int depth = 0;
  
  while (Parser::GetToken(true))
  {
    if (!strcmp(Parser::token, "{")) ++depth;
    else if (!strcmp(Parser::token, "}")) --depth;
    
    if (depth == 0) break;
  }
}

// si hago lo que dije de no reservar memoria para todos los shaders a la vez, parse, se
// debera encargar de hacerlo y devolver un puntero.

void ShaderFile::ParseLayer(Shader *shad, Layer *layer, TextureManager* textures)
{
  if (shad->flags & SF_NOMIPMAP) layer->flags |= LF_NOMIPMAP;
  if (shad->flags & SF_NOPICMIP) layer->flags |= LF_NOPICMIP;

  while (Parser::GetToken(true))
  {
    if (!strcmp(Parser::token, "{"))
    {
      // display error, token unexpected!
      break;
    }
    
    else if (!strcmp(Parser::token, "}")) return; // ends layer!
    
    else if (!stricmp(Parser::token, "map"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }

      if (!stricmp(Parser::token, "$lightmap"))
      {
        shad->flags |= SF_HASLIGHTMAP;
        layer->flags = LF_LIGHTMAP;
        layer->tcGen = TCGEN_LIGHTMAP;
        layer->map[0] = NULL;
        layer->rgbGen = RGBGEN_IDENTITY;
      }
      else if (!stricmp(Parser::token, "$whiteimage"))
      {
        gConsole->Insertln("^5WARNING: Using unsupported $whiteimage map");
      }
      else layer->map[0] = textures->AddTexinfo(Parser::token, layer->flags);

      continue;
    }
    
    else if (!stricmp(Parser::token, "clampmap"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      layer->flags |= LF_CLAMP;
      layer->map[0] = textures->AddTexinfo(Parser::token, layer->flags);
      
      continue;
    }
    
    else if (!stricmp(Parser::token, "animmap"))
    {
      int num_maps = 0;
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      layer->anim_speed = (float) atof(Parser::token);
      
      while (Parser::GetToken(false))
      {
        layer->map[num_maps] = textures->AddTexinfo(Parser::token, layer->flags);
        ++num_maps;
      }
      layer->num_maps = num_maps;
      continue;
    }
    else if (!stricmp(Parser::token, "blendfunc"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }

      if (!stricmp(Parser::token, "add"))
      {
        layer->blendsrc = layer->blenddst = IntForConst("GL_ONE");
        layer->depthWrite = false;
      }
      else if (!stricmp(Parser::token, "filter"))
      {
        layer->blendsrc = IntForConst("GL_ZERO");
        layer->blenddst = IntForConst("GL_SRC_COLOR");
        layer->depthWrite = false;
      }
      else if (!stricmp(Parser::token, "blend"))
      {
        layer->blendsrc = IntForConst("GL_SRC_ALPHA");
        layer->blenddst = IntForConst("GL_ONE_MINUS_SRC_ALPHA");
        layer->depthWrite = false;
      }
      else
      {
        // TODO: analize better the blendfunc!!
        layer->blendsrc = IntForConst(Parser::token);
        if (!Parser::GetToken(false))
        {
          gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
          continue;
        } 
        layer->blenddst = IntForConst(Parser::token);
        layer->depthWrite = (layer->blendsrc == 1 && layer->blenddst == 0 && shad->num_layers == 0);
      }

      if (shad->num_layers == 0 && shad->sortvalue == SORT_OPAQUE/*default*/)
      {
        // FIXME: thats a hack to work around texture sorting problems, id docs say
        //      that any shader with blendfunc has SORT_ADDITIVE by default.
        if (layer->blendsrc == 1 && layer->blenddst == 1 && shad->num_layers == 0) shad->sortvalue = SORT_ADDITIVE;
        else shad->sortvalue = SORT_SEETHROUGH;
      }
      //if (layer->blendsrc == IntForConst("GL_SRC_ALPHA") && layer->blenddst == IntForConst("GL_ONE_MINUS_SRC_ALPHA")) layer->rgbGen = RGBGEN_IDENTITY_LIGHTING;
      //else if (layer->blendsrc == IntForConst("GL_ONE") && layer->blenddst == IntForConst("GL_ONE")) layer->rgbGen = RGBGEN_IDENTITY_LIGHTING;
      if (layer->blendsrc == IntForConst("GL_ZERO") && layer->blenddst == IntForConst("GL_SRC_COLOR")) layer->rgbGen = RGBGEN_IDENTITY;
      else if (layer->blendsrc == IntForConst("GL_DST_COLOR") && layer->blenddst == IntForConst("GL_ZERO")) layer->rgbGen = RGBGEN_IDENTITY;
    }
    
    else if (!stricmp(Parser::token, "depthfunc"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (!stricmp(Parser::token, "lequal")) layer->depthFunc = IntForConst("GL_LEQUAL");
      else if (!stricmp(Parser::token, "equal")) layer->depthFunc = IntForConst("GL_EQUAL");
      else
      {
        // warning!!
        gConsole->Insertln("^5WARNING: unknown parameter \"%s\" in line %d", Parser::token, Parser::scriptline);
        continue;
      }
    }
    
    else if (!stricmp(Parser::token, "depthwrite"))
    {
      layer->depthWrite = true;
    }
    
    else if (!stricmp(Parser::token, "detail"))
    {
      layer->flags |= LF_DETAIL;
    }
    
    else if (!stricmp(Parser::token, "alphafunc"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (!stricmp(Parser::token, "gt0"))
      {
        layer->alphafunc = IntForConst("GL_GREATER");
        layer->alphafuncref = 0.0f;
        //layer->depthWrite = false;
      }
      else if (!stricmp(Parser::token, "lt128"))
      {
        layer->alphafunc = IntForConst("GL_LESS");
        layer->alphafuncref = 0.5f;
        //layer->depthWrite = false;
      }
      else if (!stricmp(Parser::token, "ge128"))
      {
        layer->alphafunc = IntForConst("GL_GEQUAL");
        layer->alphafuncref = 0.5f;
        //layer->depthWrite = false;
      }
      else
      {
        gConsole->Insertln("^5WARNING: unknown parameter \"%s\" in line %d", Parser::token, Parser::scriptline);
        continue;
      }
    }

    else if (!stricmp(Parser::token, "rgbgen"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (!stricmp(Parser::token, "vertex"))
      {
        shad->flags |= SF_HASCOLORS;
        layer->rgbGen = RGBGEN_VERTEX;
      }
      else if (!stricmp(Parser::token, "lightingdiffuse"))
      {
        shad->flags |= SF_HASCOLORS;
        layer->rgbGen = RGBGEN_LIGHTING_DIFFUSE;
      }
      else if (!stricmp(Parser::token, "oneminusvertex"))
      {
        shad->flags |= SF_HASCOLORS;
        layer->rgbGen = RGBGEN_ONE_MINUS_VERTEX;
      }
      else if (!stricmp(Parser::token, "identity"))
      {
        layer->rgbGen = RGBGEN_IDENTITY;
      }
      else if (!stricmp(Parser::token, "identitylighting"))
      {
        layer->rgbGen = RGBGEN_IDENTITY_LIGHTING;
      }
      else if (!stricmp(Parser::token, "wave"))
      {
        layer->rgbGen = RGBGEN_WAVE;
        layer->rgbGenParams[0] = ParseFunc();
        for (int i = 1; i < 5; ++i)
        {
          if (!Parser::GetToken(false))
          {
            gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
            break;
          }
          layer->rgbGenParams[i] = (float) atof(Parser::token);
        }
      }

      while (Parser::GetToken(false))
        gConsole->Insertln("^5WARNING: skipping \"%s\" in line %d", Parser::token, Parser::scriptline);

      continue;
    }
    else if (!stricmp(Parser::token, "alphagen"))
    {
      // TODO: add distance atenuation !!
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (!stricmp(Parser::token, "portal"))
      {
        layer->alphaGen = ALPHAGEN_PORTAL;

        if (!Parser::GetToken(false))
        {
          gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
          continue;
        }
        layer->alphaGenParams[0] = (float) atof(Parser::token);
      }
      else if (!stricmp(Parser::token, "vertex"))
      {
        //shad->flags |= SF_HASCOLORS;
        layer->alphaGen = ALPHAGEN_VERTEX;
      }
      else if (!stricmp(Parser::token, "identity"))
      {
        layer->alphaGen = ALPHAGEN_IDENTITY;
      }
      else if (!stricmp(Parser::token, "entity"))
      {
        layer->alphaGen = ALPHAGEN_ENTITY;
      }
      else if (!stricmp(Parser::token, "lightingspecular"))
      {
        layer->alphaGen = ALPHAGEN_SPECULAR;
      }
      else if (!stricmp(Parser::token, "wave"))
      {
        layer->alphaGen = ALPHAGEN_WAVE;
        layer->alphaGenParams[0] = ParseFunc();
        for (int i = 1; i < 5; ++i)
        {
          if (!Parser::GetToken(false))
          {
            gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
            break;
          }
          layer->alphaGenParams[i] = (float) atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "const") || !stricmp(Parser::token, "constant"))
      {
        gConsole->Insertln("^5WARNING: Constant alphagen not supported");
        layer->alphaGen = ALPHAGEN_CONST;
        layer->aGen[0].func = FUNC_CONST;
        layer->aGen[0].p[0] = 0;
      }

      while (Parser::GetToken(false))
        gConsole->Insertln("^5WARNING: skipping \"%s\" in line %d", Parser::token, Parser::scriptline);

      continue;
    }

    else if (!stricmp(Parser::token, "tcgen"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (!stricmp(Parser::token, "environment"))
      {
        layer->tcGen = TCGEN_ENVIRONMENT;
        layer->flags |= LF_USENORMALS;
        shad->flags |= SF_USENORMALS;
      }

      while (Parser::GetToken(false))
        gConsole->Insertln("^5WARNING: skipping \"%s\" in line %d", Parser::token, Parser::scriptline);

      continue;
    }
    
    else if (!stricmp(Parser::token, "tcmod"))
    {
      if (!Parser::GetToken(false))
      {
        gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
        continue;
      }
      if (!stricmp(Parser::token, "rotate"))
      {
        funclist *f;
        layer->flags |= LF_HASTCMOD;
        shad->flags |= SF_HASTCMOD;
        if (!layer->tcMod) f = layer->tcMod = new funclist;
        else
        {
          for (f = layer->tcMod; f->next; f = f->next){}
          f = f->next = new funclist;
        }
        f->next = NULL;
        f->func = TCMOD_ROTATE;
        f->parameters = 1;
        for (int i = 0; i < MAX_PARAMS; ++i)
        {
          if (!Parser::GetToken(false)) break;
          f->p[i] = (float) atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "scale"))
      {
        funclist *f;
        layer->flags |= LF_HASTCMOD;
        shad->flags |= SF_HASTCMOD;
        if (!layer->tcMod) f = layer->tcMod = new funclist;
        else
        {
          for (f = layer->tcMod; f->next; f = f->next){}
          f = f->next = new funclist;
        }
        f->next = NULL;
        f->func = TCMOD_SCALE;
        f->parameters = 2;
        for (int i = 0; i < MAX_PARAMS; ++i)
        {
          if (!Parser::GetToken(false)) break;
          f->p[i] = (float)atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "scroll"))
      {
        funclist *f;
        layer->flags |= LF_HASTCMOD;
        shad->flags |= SF_HASTCMOD;
        if (!layer->tcMod) f = layer->tcMod = new funclist;
        else
        {
          for (f = layer->tcMod; f->next; f = f->next){}
          f = f->next = new funclist;
        }
        f->next = NULL;
        f->func = TCMOD_SCROLL;
        f->parameters = 2;
        for (int i = 0; i < MAX_PARAMS; ++i)
        {
          if (!Parser::GetToken(false)) break;
          f->p[i] = (float)atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "stretch"))
      {
        funclist *f;
        layer->flags |= LF_HASTCMOD;
        shad->flags |= SF_HASTCMOD;
        if (!layer->tcMod) f = layer->tcMod = new funclist;
        else
        {
          for (f = layer->tcMod; f->next; f = f->next){}
          f=f->next = new funclist;
        }
        f->next = NULL;
        f->func = TCMOD_STRETCH;
        f->parameters = 5;
        f->p[0] = ParseFunc();
        for (int i = 1; i < MAX_PARAMS; ++i)
        {
          if (!Parser::GetToken(false)) break;
          f->p[i] = (float)atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "transform"))
      {
        funclist *f;
        layer->flags |= LF_HASTCMOD;
        shad->flags |= SF_HASTCMOD;
        if (!layer->tcMod) f = layer->tcMod = new funclist;
        else
        {
          for (f = layer->tcMod; f->next; f = f->next){}
          f=f->next = new funclist;
        }
        f->next = NULL;
        f->func = TCMOD_TRANSFORM;
        f->parameters = 6;
        for (int i = 0; i < MAX_PARAMS; ++i)
        {
          if (!Parser::GetToken(false)) break;
          f->p[i] = (float)atof(Parser::token);
        }
      }
      else if (!stricmp(Parser::token, "turb"))
      {
        funclist *f;
        layer->flags |= LF_HASTCMOD;
        shad->flags |= SF_HASTCMOD;
        if (!layer->tcMod) f = layer->tcMod=new funclist;
        else
        {
          for (f = layer->tcMod; f->next; f = f->next){}
          f = f->next = new funclist;
        }
        f->next = NULL;
        f->func = TCMOD_TURB;
        f->parameters = 4;
        for (int i = 0; i < MAX_PARAMS; ++i)
        {
          if (!Parser::GetToken(false)) break;
          f->p[i] = (float)atof(Parser::token);
        }
      }

      while (Parser::GetToken(false))
        gConsole->Insertln("^5WARNING: skipping \"%s\" in line %d", Parser::token, Parser::scriptline);

      continue;
    }
    else
      gConsole->Insertln("^5WARNING: unknown parameter \"%s\" in line %d", Parser::token, Parser::scriptline);

    // unknown keyword skip line
    while (Parser::GetToken(false))
      gConsole->Insertln("^5WARNING: skipping \"%s\" in line %d", Parser::token, Parser::scriptline);
  }
}

//void ParseParameters(Shader *shader, Layer *pass, const shaderkey_t *keys);
float ShaderFile::ParseFunc(void)
{
  if (!Parser::GetToken(false))
  {
    gConsole->Insertln("^5WARNING: parameter expected in line %d", Parser::scriptline);
    return 0;
  }
  
  if (!stricmp(Parser::token, "sin")) return FUNC_SIN;
  else if (!stricmp(Parser::token, "triangle")) return FUNC_TRIANGLE;
  else if (!stricmp(Parser::token, "square")) return FUNC_SQUARE;
  else if (!stricmp(Parser::token, "sawtooth")) return FUNC_SAWTOOTH;
  else if (!stricmp(Parser::token, "inversesawtooth")) return FUNC_INVERSESAWTOOTH;
  else if (!stricmp(Parser::token, "noise")) return FUNC_NOISE;
  else
  {
    gConsole->Insertln("^5WARNING: unknown function \"%s\" in line %d", Parser::token, Parser::scriptline);
    return 0;
  }
}
