//-----------------------------------------------------------------------------
// Shader
//-----------------------------------------------------------------------------

#ifndef __SHADER_H__
#define __SHADER_H__

#include "texture.h"
#include "layer.h"
#include "files.h"

/**
 * Shader manager.
 * The shader manager is used to store every shader used by the current
 * opened BSP.
 * @todo Add q3render shaders
 * @todo Finish shader support (support portal shaders and other
 *       shaders features)
 * @todo Try to change the shadermanager so we don't need to have 2 shader
 *       references in surface class.
 */
class ShaderManager
{
  public:

    /**
     * Allocate space for the references.
     * @param num the shader table size
     */
    ShaderManager(const int num = MAX_NSHADERS);
    ~ShaderManager(void);

    /**
     * Adds shader to the references.
     * @param name the shader name
     * @param flags the flags for shaders
     * @param contents the shader content
     * @param type the shader type
     * @return the shader index in shader manager
     */
    int AddShader(const char * name,
            int flags = 0,
            int contents = 0,
            int type = FACETYPE_NORMAL);  
    
    /**
     * Un-adds a shader reference.
     * @param num the shader to unreference
     */
    void DeleteShader(const int num);
    
    /**
     * Return shader of the reference.
     * @param num the shader index to get
     * @return a pointer to existing shader or
     *         NULL pointer if shader does not exist
     * @test This function was inline before
     */
    Shader *GetShader(const int num);

    /**
     * Gets the name of a registered shader.
     * @param num the shader index
     * @return the shader name is shader number exists, NULL pointer if not
     */
    char* GetShaderName(const int num);

    /**
     * Gets the index of a shader in the list.
     * @param name the shader name
     * @return the shader index
     */
    int GetShaderNum(const char* name);

    /**
     * Gets tje number of registered shaders.
     * @return the number of registered shaders
     */
    int GetShadersNum(void);

    /**
     * Reset all reference counts.
     */
    void ResetAll(void);
    
    /**
     * Updates the shaders acording to the references.
     * @param pak_search enable the shaders search in pak files if true
     */
    void Update(bool pak_search = true);
    
    /**
     * Delete all the shaders.
     */
    void DeleteAll(void);

  private:

    void DefaultShader(int i);
    Shader * CreateShader(void);
    Layer * CreateLayer(void);
    void DeleteReference(int i);

    ShaderHash *ShaderTable;
    int max_refs;
    int num_refs;

    LayerManager  layers;
    TextureManager  textures;
};

//-----------------------------------------------------------------------------
// ShaderFile
//-----------------------------------------------------------------------------

class ShaderFile : public VFile
{
  public:
    ShaderFile(const char *name, const char *pakname = "");
    ~ShaderFile();
    
    char *GetShaderName(void);
    void SetShaderPos(int pos);
    int GetShaderPos(void);
    void Parse(Shader *shad, LayerManager* layers, TextureManager* textures);
    void Skip(void);
    
    // si hago lo que dije de no reservar memoria para todos los shaders a la vez, parse, se
    // debera encargar de hacerlo y devolver un puntero.

  private:
    void ParseLayer(Shader *shad, Layer *llayer, TextureManager* textures);
    
    //void ParseParameters(Shader *shader, Layer *pass, const shaderkey_t *keys);
    float ParseFunc(void);
};

#endif
