//-----------------------------------------------------------------------------
// Timer
//-----------------------------------------------------------------------------

#include "timer.h"
#include "console.h"
#ifdef WIN32
#include <time.h>
#else
#include <sys/time.h>
#endif

double          Timer::flastTime;     /**< last recorded time */
double          Timer::frametime = 0;   /**< time elapsed in the last frame */
int           Timer::frames = 0;      /**< number of frames */
double          Timer::fTime = 0.0f;    /**< time since window started */
LARGE_INTEGER     Timer::tFrequency;
double          Timer::tResolution;

static int        lowshift;
static byte       tPerformanceTimerEnabled;

static unsigned int   oldTime;
static int        sametimecount;  /**< counter for frames with the same time. */

void Timer::Init(void)
{
  gConsole->Insert("Performance Counter available... ");
  #ifdef WIN32
    if (!QueryPerformanceFrequency((LARGE_INTEGER *) &tFrequency))
    {
      // no performance counter available, you might be using w98 osr1
      gConsole->Insertln("failed");
      tResolution = 0.00000025f;  // FIXME: find the good value
      tPerformanceTimerEnabled = 0;
      //ThrowException(-1); // I usually exit in this situation
    }
    else
    { 
      // performance counter is available, use it instead of multimedia timer
      gConsole->Insertln("ok");
      tPerformanceTimerEnabled = 1;

      // get 32 out of the 64 time bits such that we have around
      // 1 microsecond resolution
      unsigned int lowpart = (unsigned int)tFrequency.LowPart;
      unsigned int highpart = (unsigned int)tFrequency.HighPart;
      lowshift = 0;

      while (highpart || (lowpart > 2000000.0))
      {
        ++lowshift;
        lowpart >>= 1;
        lowpart |= (highpart & 1) << 31;
        highpart >>= 1;
      }
      tResolution = 1.0/(double)lowpart;
    }
  #else
    tResolution = 0.001f;              // 1 ms
    tPerformanceTimerEnabled = 0;
    gConsole->Insertln("unsupported (not a win32 platform)");
  #endif
}

//-----------------------------------------------------------------------------
// Refresh - updates timers and counters (should be called for each frame)
//-----------------------------------------------------------------------------
int Timer::Refresh(void)
{
  unsigned int  temp, t2;

  #ifdef WIN32
    LARGE_INTEGER count;
    if (tPerformanceTimerEnabled)
    {
      QueryPerformanceCounter((LARGE_INTEGER *) &count);

      temp = ((unsigned int)count.LowPart >> lowshift) |
        ((unsigned int)count.HighPart << (32 - lowshift));
    }
    else
    {
      temp = timeGetTime();
    }
  #else
    struct timeval tt;

    gettimeofday(&tt, (struct timezone *)0);
    temp = (unsigned int) ((double) tt.tv_sec*1000 + (double) tt.tv_usec/1000);
  #endif

  // check for firstframe, turnover or backward time
  if ((frames == 0) || (temp <= oldTime))
  {
    oldTime = temp;
  }
  else
  {
    t2 = temp - oldTime;
    frametime = (double)t2 * tResolution;
    fTime += frametime;

    oldTime = temp;

    if (fTime == flastTime)
    {
      ++sametimecount;

      if (sametimecount > 100000)
      {
        fTime += 1.0;
        sametimecount = 0;
      }
    }
    else sametimecount = 0;

    flastTime = fTime;
  }

  ++frames;
  return frames;
}
