/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2000  ASAMI, Tomoharu (asami@zeomtech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.dombus;

import java.util.*;
import java.io.IOException;
import java.net.URL;
import javax.xml.parsers.*;
import org.xml.sax.SAXException;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.io.UURL;
import jp.gr.java_conf.jaba2.dombus.guard.AllGuard;

/**
 * DOMBus
 *
 * @since   Mar. 27, 2000
 * @version Sep.  1, 2000
 * @author  ASAMI, Tomoharu (asami@zeomtech.com)
 */
public class DOMBus implements IDOMBusConstants {
    private DOMBusConfig config_;
    private IDOMBusContext defaultContext_;
    private Map contexts_ = new HashMap();
    private List slots_ = new ArrayList();
    private DOMBusQueue queue_ = new DOMBusQueue();
    private DOMBusMonitor monitor_;
    private String[] args_;

    public DOMBus(String[] args, URL config)
	throws IOException, ParserConfigurationException, SAXException {

	args_ = args;
	monitor_ = new DOMBusMonitor();
	config_ = new DOMBusConfig(config);
	IDOMBusContext[] contexts = config_.getContexts();
	if (contexts != null) {
	    defaultContext_ = contexts[0]; // XXX
	    for (int i = 0;i < contexts.length;i++) {
		IDOMBusContext context = contexts[i];
		context.setup(args);
		addContextSlot(context);
	    }
	} else {
	    _ensureDefaultContext();
	}
	addComponent(new AllGuard(), monitor_);
	IComponentSlot[] slots = config_.getSlots();
	for (int i = 0;i < slots.length;i++) {
	    addComponentSlot(slots[i]);
	}
	_startup();
    }

    private void _startup() {
	if (_isHelp()) {
	    monitor_.println(_getCopyright());
	    monitor_.println(_getVersion());
	    String[] lines = UString.makeStringList(config_.getHelp());
	    for (int i = 0;i < lines.length;i++) {
		monitor_.println(lines[i]);
	    }
	    System.exit(0);
	} else {
	    monitor_.info(_getCopyright());
	    monitor_.info(_getVersion());
	}
    }

    private boolean _isHelp() {
	for (int i = 0;i < args_.length;i++) {
	    String arg = args_[i];
	    if ("-help".equals(arg)) {
		return (true);
	    }
	}
	return (false);
    }

    private String _getCopyright() {
	String author = config_.getAuthor();
	String year = config_.getYear();
	String help = config_.getHelp();
	StringBuffer copyrightBuffer = new StringBuffer();
	copyrightBuffer.append("Copyright(c) ");
	copyrightBuffer.append(year);
	copyrightBuffer.append(" ");
	copyrightBuffer.append(author);
	copyrightBuffer.append(". All rights reserved.");
	return (new String(copyrightBuffer));
    }

    private String _getVersion() {
	String name = config_.getName();
	String version = config_.getVersion();
	String build = config_.getBuild();
	String provider = config_.getProvider();
	StringBuffer versionBuffer = new StringBuffer();
	versionBuffer.append(name);
	versionBuffer.append(" Version ");
	versionBuffer.append(version);
	if (build != null) {
	    versionBuffer.append(" (");
	    versionBuffer.append(build);
	    versionBuffer.append(")");
	}
	versionBuffer.append(" by ");
	versionBuffer.append(provider);
	return (new String(versionBuffer));
    }

    private String _getUsage() {
	StringBuffer buffer = new StringBuffer();
	buffer.append("Usage: ");
	buffer.append(config_.getUsage());
	return (new String(buffer));
    }

    public void init() throws DOMBusException {
	boolean hasActive = false;
	int size = slots_.size();
	for (int i = 0;i < size;i++) {
	    IComponentSlot slot = (IComponentSlot)slots_.get(i);
	    IDOMBusComponent comp = slot.getComponent();
	    if (comp.init()) {
		hasActive = true;
	    }
	}
	if (!hasActive) {
	    monitor_.println(_getCopyright());
	    monitor_.println(_getVersion());
	    monitor_.println();
	    monitor_.println(_getUsage());
	    monitor_.println("  for more information, use -help option");
	    System.exit(1);
	}
    }

    public void start() {
	int size = slots_.size();
	for (int i = 0;i < size;i++) {
	    IComponentSlot slot = (IComponentSlot)slots_.get(i);
	    IDOMBusComponent comp = slot.getComponent();
	    comp.start();
	}
	stimulus(new DOMBusSignal("dombus:start"));
    }

    public void stop() {
	int size = slots_.size();
	for (int i = 0;i < size;i++) {
	    IComponentSlot slot = (IComponentSlot)slots_.get(i);
	    IDOMBusComponent comp = slot.getComponent();
	    comp.stop();

	}
    }

    public void destroy() {
	int size = slots_.size();
	for (int i = 0;i < size;i++) {
	    IComponentSlot slot = (IComponentSlot)slots_.get(i);
	    IDOMBusComponent comp = slot.getComponent();
	    comp.destroy();
	}
    }

    public void addComponent(
	IDOMBusGuard guard,
	IDOMBusComponent component
    ) {
	addComponentSlot(new ComponentSlot(component, guard));
    }

    public void addComponentSlot(IComponentSlot slot) {
	IDOMBusComponent comp = slot.getComponent();
	String contextName = slot.getContextName();
	String contextRole = slot.getContextRole();
	comp.setup(this);
	comp.setup(args_);
	DOMBusContext context = (DOMBusContext)getContext(contextName);	// XXX
	if (context != null) {
	    comp.setup(context);
	    if (contextRole != null) {
		context.setComponent(contextRole, comp);
	    } else {
		context.addComponent(comp);
	    }
	}
	slots_.add(slot);
    }

    public void addContextSlot(IDOMBusContext context) {
	String name = context.getName();
	if (contexts_.containsKey(name)) {
	    throw (new InternalError());
	}
	contexts_.put(name, context);
    }

    public IDOMBusContext getContext(String name) {
	if (name.equals(DEFAULT_CONTEXT_NAME)) {
	    return (defaultContext_);
	}
	return ((IDOMBusContext)contexts_.get(name));
    }

    public void stimulus(DOMBusSignal signal) {
	queue_.putSignal(signal);
    }

    public void schedule() {
	for (;;) {
	    DOMBusSignal signal = queue_.getSignalWait();
	    if ("dombus:finish".equals(signal.getLabel())) {
		_scheduleFinish();
		stop();
		destroy();
		System.exit(0);
	    }
	    broadcast(signal);
	}
    }

    private void _scheduleFinish() {
	DOMBusSignal signal = queue_.getSignal();
	if (signal == null) {
	    return;
	}
	broadcast(signal);
    }

    public void broadcast(DOMBusSignal signal) {
	monitor_.debug("broadcast: " + signal);
	int count = 0;
	int size = slots_.size();
	signal.setNParticipants(size);
	for (int i = 0;i < size;i++) {
	    IComponentSlot slot = (IComponentSlot)slots_.get(i);
	    if (slot.stimulus(signal)) {
		count++;
		monitor_.debug("  accept: " + slot);
	    } else {
		monitor_.debug("  ignore: " + slot);
	    }
	}
	if (count == 0) {
	    monitor_.debug("  missing receiver:" + signal);
	}
    }

    private void _ensureDefaultContext() {
	defaultContext_ = new DOMBusContext();
	contexts_.put(DEFAULT_CONTEXT_NAME, defaultContext_);
    }

    public IDOMBusComponent[] getComponents() {
	int size = slots_.size();
	IDOMBusComponent[] comps = new IDOMBusComponent[size];
	for (int i = 0;i < size;i++) {
	    IComponentSlot slot = (IComponentSlot)slots_.get(i);
	    comps[i] = slot.getComponent();
	}
	return (comps);
    }

    public IDOMBusComponent[] getComponents(Class targetClass) {
	IDOMBusComponent[] comps = getComponents();
	List list = new ArrayList(comps.length);
	for (int i = 0;i < comps.length;i++) {
	    IDOMBusComponent comp = comps[i];
	    if (targetClass.isInstance(comp)) {
		list.add(comp);
	    }
	}
	IDOMBusComponent[] result = new IDOMBusComponent[list.size()];
	return ((IDOMBusComponent[])list.toArray(result));
    }

    public static void main(String[] args) {
	try {
	    URL config = UURL.getURLFromResourceName(args[0], DOMBus.class);
	    String[] realArgs = new String[args.length - 1];
	    for (int i = 1;i < args.length;i++) {
		realArgs[i - 1] = args[i];
	    }
	    DOMBus bus = new DOMBus(realArgs, config);
	    __setDOMBus(bus);
	    bus.init();
	    bus.start();
	    bus.schedule();
	} catch (ParserConfigurationException e) {
	    e.printStackTrace();
	} catch (SAXException e) {
	    e.printStackTrace();
	} catch (IOException e) {
	    e.printStackTrace();
	}
    }

    private static void __setDOMBus(DOMBus bus) {
	if (bus__ != null) {
	    throw (new InternalError());
	}
	bus__ = bus;
    }

    public static DOMBus getDOMBus() {
	return (bus__);
    }

    private static DOMBus bus__;
}
