/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2002  ASAMI, Tomoharu (asami@AsamiOffice.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.xml.relaxng;

import java.util.*;
import java.io.IOException;
import java.net.URL;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.datatype.XMLFacet;
import jp.gr.java_conf.jaba2.xml.*;
import jp.gr.java_conf.jaba2.xml.relaxng.rRelaxng.*;
import jp.gr.java_conf.jaba2.xml.relax.cooked.*;
import jp.gr.java_conf.jaba2.xml.relax.expanded.*;

/**
 * RNGGrammar
 *
 * @since   Jun. 15, 2001
 * @version Jun. 30, 2002
 * @author  ASAMI, Tomoharu (asami@AsamiOffice.com)
 */
public class RNGGrammar {
    private static final int OCCURS_ONE = 1;
    private static final int OCCURS_ZEROONE = 2;
    private static final int OCCURS_ONEMORE = 3;
    private static final int OCCURS_ZEROMORE = 4;
    private String baseURI_;
    private Map modules_ = new HashMap(); // Map<String, RNGModule>
    private Map nsByLabel_ = new HashMap(); // Map<String, String>
    private List unresolvedRefs_ = new ArrayList();

    public RNGGrammar(RNGrammar grammar, String baseURI) {
	baseURI_ = baseURI;
	IRNGrammarContentChoice[] contents = grammar.getGrammarContent();
	List defineList = new ArrayList();
	List includeList = new ArrayList();
	List startList = new ArrayList();
	for (int i = 0;i < contents.length;i++) {
	    IRNGrammarContentChoice content = contents[i];
	    if (content instanceof RNDefine) {
		defineList.add(content);
	    } else if (content instanceof RNInclude) {
		includeList.add(content);
	    } else if (content instanceof RNStart) {
		startList.add(content);
	    } else if (content instanceof RNDiv) {
		_buildDiv((RNDiv)content, startList, defineList, includeList);
	    } else {
		throw (new IllegalArgumentException("syntax error"));
	    }
	}
// System.out.println("l = " + defineList.size());
	RNDefine[] defines = _getDefines(defineList);
        for (int i = 0;i < defines.length;i++) {
	    RNDefine define = defines[i];
// System.out.println("Z = " + define.getName());
	    _buildDefine(define);
	}
	RNStart start = _getStart(startList);
	_buildStart(start);
	_resolveRefs();
    }

    private void _buildDiv(
	RNDiv div,
	List startList,
	List defineList,
	List includeList
    ) {
	IRNGrammarContentChoice[] contents = div.getGrammarContent();
	for (int i = 0;i < contents.length;i++) {
	    IRNGrammarContentChoice content = contents[i];
	    if (content instanceof RNDefine) {
		defineList.add(content);
	    } else if (content instanceof RNInclude) {
		includeList.add(content);
	    } else if (content instanceof RNStart) {
		startList.add(content);
	    } else if (content instanceof RNDiv) {
		_buildDiv((RNDiv)content, startList, defineList, includeList);
	    } else {
		throw (new IllegalArgumentException("syntax error"));
	    }
	}
    }

    private RNDefine[] _getDefines(List defineList) {
	RNDefine[] defines = new RNDefine[defineList.size()];
	return ((RNDefine[])defineList.toArray(defines));
    }
    
    private RNStart _getStart(List startList) {
	switch (startList.size()) {

	case 0:
	    throw (new IllegalArgumentException("syntax error"));
	case 1:
	    return ((RNStart)startList.get(0));
	default:
	    throw (new UnsupportedOperationException("multiple start"));
	}
    }

    public RNGModule[] getModules() {
	Collection values = modules_.values();
	RNGModule[] modules = new RNGModule[values.size()];
	return ((RNGModule[])values.toArray(modules));
    }

    private void _buildStart(RNStart start) {
	IRNElementHedgeChoice[] nodes = start.getElementHedge();
	_buildStart(nodes);
    }

    private void _buildStart(IRNElementHedgeChoice[] nodes) {
	for (int i = 0;i < nodes.length;i++) {
	    IRNElementHedgeChoice node = nodes[i];
	    if (node instanceof RNElement) {
		_buildStartElement((RNElement)node);
	    } else if (node instanceof RNRef) {
		_buildStartRef((RNRef)node);
	    } else if (node instanceof RNChoice) {
		_buildStartChoice((RNChoice)node);
	    }
	}
    }

    private void _buildStartElement(RNElement element) {
	String labelName = _makeLabelName("root", element.getName());
	_buildElement(element, labelName);
	RNGModule module = _getModule(element);
	module.addExportLabel(labelName);
    }

    private void _buildStartRef(RNRef ref) {
	String name = ref.getName();
	RNGModule module = _getTargetModule(ref);
	module.addExportLabel(name);
    }

    private void _buildStartChoice(RNChoice choice) {
	IRNElementHedgeChoice[] contents = choice.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		_buildStartElement((RNElement)content);
	    } else if (content instanceof RNRef) {
		_buildStartRef((RNRef)content);
	    } else if (content instanceof RNChoice) {
		_buildStartChoice((RNChoice)content);
	    }
	}
    }

    private void _resolveRefs() {
	int size = unresolvedRefs_.size();
	for (int i = 0;i < size;i++) {
	    RefPair pair = (RefPair)unresolvedRefs_.get(i);
	    RNGModule contextModule = _getModule(pair.ngRef);
	    RNGModule targetModule = _getTargetModule(pair.ngRef);
	    if (contextModule != targetModule) {
		pair.cRef.setNamespace(targetModule.getNamespaceURI());
	    }
	}
    }

    private void _buildDefine(RNDefine define) {
	IRNElementHedgeChoice[] nodes = define.getElementHedge();
	if (nodes.length == 1 &&
	    nodes[0] instanceof RNElement) {

	    _buildElementRule(define, (RNElement)nodes[0]);
	} else {
	    _buildHedgeRule(define, nodes);
	}
    }

    private RNGModule _getModule(RNDefine define) {
	return (_getModule(""));
    }

    private RNGModule _getModule(RNElement element) {
	String name = element.getName();
	int index = name.indexOf(":");
	if (index == -1) {
	    return (_getModule(_findNamespace(element)));
	} else {
	    throw (new UnsupportedOperationException());
	}
    }

    private RNGModule _getModule(RNAttribute attribute) {
	String name = attribute.getName();
	int index = name.indexOf(":");
	if (index == -1) {
	    return (_getModule(_findNamespace(attribute)));
	} else {
	    throw (new UnsupportedOperationException());
	}
    }

    private RNGModule _getModule(IRNode node) {
	return (_getModule(_findNamespace(node)));
    }

    private String _findNamespace(IRNode node) {
	for (;;) {
	    if (node instanceof RNElement) {
		RNElement element = (RNElement)node;
		String ns = element.getNs();
		if (ns != null) {
		    return (ns);
		}
	    }
	    IRNode parent = node.getParentRNode();
	    if (parent == null) {
		return ("");
	    }
	    node = parent;
	}
    }

    private RNGModule _getTargetModule(RNRef ref) {
	String name = ref.getName();
	String ns = (String)nsByLabel_.get(name);
	if (ns == null) {
	    throw (new InternalError(name));
	}
	return (_getModule(ns));
    }

    private RNGModule _getModule(RNAttribute attr) {
	String ns = attr.getNs();
	// XXX global?
	if (ns == null) {
	    return (_getModule(""));
	} else {
	    return (_getModule(ns));
	}
    }

    private RNGModule _getModule(String namespaceURI) {
	RNGModule module = (RNGModule)modules_.get(namespaceURI);
	if (module == null) {
	    module = new RNGModule(namespaceURI);
	    module.setBaseURI(baseURI_);
	    modules_.put(namespaceURI, module);
	}
	return (module);
    }

    private void _buildElementRule(RNDefine define, RNElement element) {
	String labelName = define.getName();
	_buildElement(element, labelName);
    }

    private CRefWithLabel _buildElementWithRef(
	RNElement element,
	String context,
	int occurs
    ) {
	String labelName = _makeLabelName(context, element.getName());
	_buildElement(element, labelName);
	CRefWithLabel ref = new CRefWithLabel(labelName, _makeCOccurs(occurs));
	ref.setBase(element.getXMLElement());
	return (ref);
    }

    private void _buildElement(
	RNElement element,
	String labelName
    ) {
	String elementName = _getElementName(element);
	IRNElementHedgeChoice[] contents = element.getElementHedge();
	RNGModule ngModule = _getModule(element);
	CModule cModule = ngModule.getCModule();
	CTagPattern cTag = new CTagPattern(labelName, elementName, cModule);
	cTag.setBase(element.getXMLElement());
	CSequence cSequence = new CSequence();
	String typeName = null;
	XMLFacet[] facets = null;
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
//		    _makeLabelName(labelName, elementName),
		    labelName,
		    OCCURS_ONE
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		_buildCTag(ngAttr, true, cTag, cModule);
/*
		String value = _getValueText(ngAttr);
		if (value != null) {
		    cTag.addContent(new CValue(value));
		} else {
		    String type = _getTypeName(ngAttr);
		    CAttribute cAttr = new CAttribute(
			ngAttr.getName(),
			type,
			cModule
		    );
		    cAttr.setRequired(true);
		    cTag.addContent(cAttr);
		}
*/
	    } else if (content instanceof RNText) {
		typeName = "string";
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		typeName = data.getType();
		facets = _makeFacets(data);
	    } else if (content instanceof RNValue) {
	    } else if (content instanceof RNGroup) {
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, labelName, cChoice);
		cSequence.addParticle(cChoice);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave();
		_buildInterleave(
		    (RNInterleave)content,
		    labelName,
		    cInterleave
		);
		cSequence.addParticle(cInterleave);
	    } else if (content instanceof RNMixed) {
		CMixed cMixed = new CMixed();
		_buildMixed((RNMixed)content, labelName, cMixed);
		CMixedElementRule cRule = new CMixedElementRule(
		    labelName,
		    cMixed,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
		cModule.addTagPattern(cTag);
		return;
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    labelName,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cTag.addContent(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    labelName,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    labelName,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError());
	    }
	}
	if (typeName != null) {
	    CTypeElementRule cRule = new CTypeElementRule(
		    labelName,
		    typeName,
		    facets,
		    cModule
	    );
	    cRule.setBase(element.getXMLElement());
	    cModule.addElementRule(cRule);
	    nsByLabel_.put(labelName, cModule.getTargetNamespace());
	} else {
	    if (cSequence.getParticleCount() == 0) {
		CEmptyElementRule cRule = new CEmptyElementRule(
		    labelName,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
	    } else {
		CSequenceElementRule cRule = new CSequenceElementRule(
		    labelName,
		    cSequence,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
	    }
	}
	cModule.addTagPattern(cTag);
    }

    private String _getElementName(RNElement element) {
	String elementName = element.getName();
	if (elementName != null) {
	    int index = elementName.indexOf(":");
	    if (index == -1) {
		return (elementName);
	    } else {
		return (elementName.substring(index + 1));
	    }
	}
	IRNNameClassHedgeChoice child = element.getNameClassHedge();
	if (child instanceof RNName) {
	    RNName name = (RNName)child;
	    return (name.getContent());
	} else if (child instanceof RNAnyName) {
	    return (child.makeTextDocument());
	} else if (child instanceof RNNsName) {
	    return (child.makeTextDocument());
	} else if (child instanceof RNNameChoice) {
	    return (child.makeTextDocument());
	} else {
	    return (null);	// XXX : syntax error
	}
    }

    private String _getAttributeName(RNAttribute attribute) {
	String attributeName = attribute.getName();
	if (attributeName != null) {
	    return (attributeName);
	}
	IRNNameClassHedgeChoice child = attribute.getNameClassHedge();
	if (child instanceof RNName) {
	    RNName name = (RNName)child;
	    return (name.getContent());
	} else if (child instanceof RNAnyName) {
	    return (child.makeTextDocument());
	} else if (child instanceof RNNsName) {
	    return (child.makeTextDocument());
	} else if (child instanceof RNNameChoice) {
	    return (child.makeTextDocument());
	} else {
	    return (null);	// XXX : syntax error
	}
    }

    private void _buildCTag(
	RNAttribute ngAttr,
	boolean isRequired,
	CTagPattern cTag,
	CModule cModule
    ) {
	CAttribute cAttr = _makeCAttribute(ngAttr, isRequired, cModule);
	cTag.addContent(cAttr);
    }

    private CAttribute _makeCAttribute(
	RNAttribute ngAttr,
	boolean isRequired,
	CModule cModule
    ) {
	IRNAttributeHedgeChoice content = ngAttr.getAttributeHedge();
	if (content instanceof RNRef) {
	    return (
		_makeCAttributeRef(
		    ngAttr,
		    (RNRef)content,
		    isRequired,
		    cModule
		)
	    );
	} else if (content instanceof RNText) {
	    return (_makeCAttributeText(ngAttr, isRequired, cModule));
	} else if (content instanceof RNData) {
	    return (
		_makeCAttributeData(
		    ngAttr,
		    (RNData)content,
		    isRequired,
		    cModule
		)
	    );
	} else if (content instanceof RNValue) {
	    return (
		_makeCAttributeValue(
		    ngAttr,
		    (RNValue)content,
		    isRequired,
		    cModule
		)
	    );
	} else if (content instanceof RNAttributeChoice) {
	    return (
		_makeCAttributeChoice(
		    ngAttr,
		    (RNAttributeChoice)content,
		    isRequired,
		    cModule
		)
	    );
	} else {
	    throw (new InternalError());
	}
    }

    private CAttribute _makeCAttributeRef(
	RNAttribute ngAttr,
	RNRef ngRef,
	boolean isRequired,
	CModule cModule
    ) {
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    ngRef.makeTextDocument(),
	    cModule
	);
/*
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    new CRefWithPattern(ngAttr.getName(), cModule),
	    cModule
	);
*/
	cAttr.setRequired(isRequired);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	return (cAttr);
    }

    private CAttribute _makeCAttributeText(
	RNAttribute ngAttr,
	boolean isRequired,
	CModule cModule
    ) {
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    "string",
	    cModule
	);
	cAttr.setRequired(isRequired);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	return (cAttr);
    }

    private CAttribute _makeCAttributeData(
	RNAttribute ngAttr,
	RNData data,
	boolean isRequired,
	CModule cModule
    ) {
	String type = data.getType();
	XMLFacet[] facets = _makeFacets(data);
	if (facets != null) {
	    CAttribute cAttr = new CAttribute(
		_getAttributeName(ngAttr),
		type,
		facets,
		cModule
	    );
	    cAttr.setRequired(isRequired);
	    cAttr.setAnd(true);
	    cAttr.setBase(ngAttr.getXMLElement());
	    return (cAttr);
	} else {
	    CAttribute cAttr = new CAttribute(
		_getAttributeName(ngAttr),
		type,
		cModule
	    );
	    cAttr.setRequired(isRequired);
	    cAttr.setAnd(true);
	    cAttr.setBase(ngAttr.getXMLElement());
	    return (cAttr);
	}
    }

    private XMLFacet[] _makeFacets(RNData data) {
	RNParam[] params = data.getParam();
	if (params == null || params.length == 0) {
	    return (null);
	}
	XMLFacet[] facets = new XMLFacet[params.length];
	for (int i = 0;i < params.length;i++) {
	    RNParam param = params[i];
	    facets[i] = new XMLFacet(param.getName(), param.getContent());
	}
	return (facets);
    }

    private CAttribute _makeCAttributeValue(
	RNAttribute ngAttr,
	RNValue ngValue,
	boolean isRequired,
	CModule cModule
    ) {
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    ngValue.makeTextDocument(),
	    cModule
	);
/*
	String value = ngValue.getContent();
	String type = ngValue.getType();
	CValue cValue;
	if (type != null) {
	    cValue = new CValue(value, type);
	} else {
	    cValue = new CValue(value);
	}
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    cValue,
	    cModule
	);
*/
	cAttr.setRequired(isRequired);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	return (cAttr);
    }

    private CAttribute _makeCAttributeChoice(
	RNAttribute ngAttr,
	RNAttributeChoice ngChoice,
	boolean isRequired,
	CModule cModule
    ) {
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    ngChoice.makeTextDocument(),
	    cModule
	);
	cAttr.setRequired(isRequired);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	return (cAttr);
    }

/*
    private String _getValueText(RNAttribute attr) {
	IRNAttributeHedgeChoice content = attr.getAttributeHedge();
	if (content instanceof RNValue) {
	    RNValue value = (RNValue)content;
	    return (value.getContent());
	} else {
	    return (null);
	}
    }

    private String _getTypeName(RNAttribute attr) {
	IRNAttributeHedgeChoice content = attr.getAttributeHedge();
	if (content instanceof RNText) {
	    return ("string");
	} else if (content instanceof RNData) {
	    RNData data = (RNData)content;
	    return (data.getType());
	} else if (content instanceof RNAttributeChoice) {
	    throw (new InternalError());
	} else {
	    throw (new InternalError());
	}
    }
*/

    private void _buildGroup(	// XXX : unify generic reduction logic
	RNGroup group,
	String context,
	CSequence cSequence
    ) {
	RNGModule ngModule = _getModule(group);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = group.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		cSequence.addParticle(_makeCAttribute(ngAttr, true, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cSequence.addParticle(cAttr);
*/
	    } else if (content instanceof RNText) {
	    } else if (content instanceof RNData) {
	    } else if (content instanceof RNValue) {
	    } else if (content instanceof RNGroup) {
	    } else if (content instanceof RNChoice) {
/*
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, cChoice);
		cChoice.addParticle(cChoice);
*/
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cSequence.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildChoice(
	RNChoice ngChoice,
	String context,
	CChoice cChoice
    ) {
	RNGModule ngModule = _getModule(ngChoice);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = ngChoice.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cChoice.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cChoice.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		cChoice.addParticle(_makeCAttribute(ngAttr, true, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cChoice.addParticle(cAttr);
*/
	    } else if (content instanceof RNText) {
		cChoice.addParticle(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		cChoice.addParticle(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		cChoice.addParticle(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		throw (new UnsupportedOperationException());
	    } else if (content instanceof RNChoice) {
/*
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, cChoice);
		cChoice.addParticle(cChoice);
*/
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cChoice.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildInterleave(
	RNInterleave ngInterleave,
	String context,
	CInterleave cInterleave
    ) {
	RNGModule ngModule = _getModule(ngInterleave);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = ngInterleave.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cInterleave.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cInterleave.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		cInterleave.addParticle(_makeCAttribute(ngAttr, true, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cInterleave.addParticle(cAttr);
*/
	    } else if (content instanceof RNText) {
		cInterleave.addParticle(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		cInterleave.addParticle(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		cInterleave.addParticle(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		throw (new UnsupportedOperationException());
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, context, cChoice);
		cInterleave.addParticle(cChoice);
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cInterleave.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cInterleave.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cInterleave.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cInterleave.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildMixed(
	RNMixed ngMixed,
	String context,
	CMixed cMixed
    ) {
	RNGModule ngModule = _getModule(ngMixed);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice content = ngMixed.getElementHedge();
	if (content instanceof RNElement) {
	    RNElement element = (RNElement)content;
	    CRefWithLabel cRef = _buildElementWithRef(
		element,
		context,
		OCCURS_ONE
	    );
	    cMixed.addParticle(cRef);
	} else if (content instanceof RNRef) {
	    CRefWithLabel cRef = _makeCRefWithLabel(
		(RNRef)content,
		ngModule,
		""
	    );
	    cMixed.addParticle(cRef);
	} else if (content instanceof RNAttribute) {
	    RNAttribute ngAttr = (RNAttribute)content;
	    cMixed.addParticle(_makeCAttribute(ngAttr, true, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cMixed.addParticle(cAttr);
*/
	} else if (content instanceof RNText) {
	    cMixed.addParticle(new CData("string", cModule));
	} else if (content instanceof RNData) {
	    RNData data = (RNData)content;
	    String typeName = data.getType();
	    XMLFacet[] facets = _makeFacets(data);
	    cMixed.addParticle(new CData(typeName, facets, cModule));
	} else if (content instanceof RNValue) {
	    RNValue value = (RNValue)content;
	    String typeName = value.getType();
	    cMixed.addParticle(new CValue(value.getContent(), typeName));
	} else if (content instanceof RNGroup) {
	    throw (new UnsupportedOperationException());
	} else if (content instanceof RNChoice) {
	    CChoice cChoice = new CChoice();
	    _buildChoice((RNChoice)content, context, cChoice);
	    cMixed.addParticle(cChoice);
	} else if (content instanceof RNOptional) {
	    List optionContents = new ArrayList();
	    List optionAttrs = new ArrayList();
	    _buildOptional(
		(RNOptional)content,
		context,
		optionContents,
		optionAttrs
	    );
	    int size = optionContents.size();
	    for (int j = 0;j < size;j++) {
		ICParticle cParticle = (ICParticle)optionContents.get(j);
		cMixed.addParticle(cParticle);
	    }
	    size = optionAttrs.size();
	    for (int j = 0;j < size;j++) {
		CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		cMixed.addParticle(cAttr);
	    }
	} else if (content instanceof RNZeroOrMore) {
	    List optionContents = new ArrayList();
	    _buildZeroOrMore(
		(RNZeroOrMore)content,
		context,
		optionContents
	    );
	    int size = optionContents.size();
	    for (int j = 0;j < size;j++) {
		ICParticle cParticle = (ICParticle)optionContents.get(j);
		cMixed.addParticle(cParticle);
	    }
	} else if (content instanceof RNOneOrMore) {
	    List optionContents = new ArrayList();
	    _buildOneOrMore(
		(RNOneOrMore)content,
		context,
		optionContents
	    );
	    int size = optionContents.size();
	    for (int j = 0;j < size;j++) {
		ICParticle cParticle = (ICParticle)optionContents.get(j);
		cMixed.addParticle(cParticle);
	    }
	} else {
	    throw (new InternalError());
	}
    }

    private void _buildOptional(
	RNOptional optional,
	String context,
	List result,
	List attrs
    ) {
	IRNElementHedgeChoice[] contents = optional.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROONE
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(optional),
		    "?"
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		attrs.add(_makeCAttribute(ngAttr, false, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		attrs.add(cAttr);
*/
		return;
	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence("?");
		_buildGroup((RNGroup)content, context, cSequence);
		result.add(cSequence);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("?");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("?");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildOneOrMore(
	RNOneOrMore oneOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = oneOrMore.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ONEMORE
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(oneOrMore),
		    "+"
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		result.add(_makeCAttribute(ngAttr, true, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(true);
		result.add(cAttr);
*/
		return;

	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		throw (new InternalError());
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("+");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("+");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildZeroOrMore(
	RNZeroOrMore zeroOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = zeroOrMore.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROMORE
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(zeroOrMore),
		    "*"
		);
		result.add(cRef);
		return;
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		result.add(_makeCAttribute(ngAttr, true, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		result.add(cAttr);
*/
		return;

	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		throw (new InternalError());
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("*");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("*");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private String _makeLabelName(String context, String name) {
	return (context + UString.capitalize(name));
    }

    private String _makeCOccurs(int occurs) {
	switch (occurs) {

	case OCCURS_ONE:
	    return ("");
	case OCCURS_ZEROONE:
	    return ("?");
	case OCCURS_ONEMORE:
	    return ("+");
	case OCCURS_ZEROMORE:
	    return ("*");
	default:
	    throw (new InternalError());
	}
    }

    private void _buildHedgeRule(
	RNDefine define,
	IRNElementHedgeChoice[] nodes
    ) {
	String labelName = define.getName();
	RNGModule ngModule = _getModule(define);
	CModule cModule = ngModule.getCModule();
	List list = new ArrayList();
	for (int i = 0;i < nodes.length;i++) {
	    IRNElementHedgeChoice content = nodes[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    _makeLabelName(labelName, element.getName()),
		    OCCURS_ONE
		);
		list.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		list.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		list.add(_makeCAttribute(ngAttr, true, cModule));
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		list.add(cAttr);
*/
	    } else if (content instanceof RNText) {
		list.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		list.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		list.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		throw (new InternalError());
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave();
		_buildInterleave(
		    (RNInterleave)content,
		    labelName,
		    cInterleave
		);
		list.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, labelName, cChoice);
		list.add(cChoice);
	    } else if (content instanceof RNOptional) {
		_buildOptional(
		    (RNOptional)content,
		    labelName,
		    list,
		    list
		);
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    labelName,
		    list
		);
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    labelName,
		    list
		);
	    } else {
		throw (new InternalError());
	    }
	}
	if (list.size() == 0) {
	    throw (new InternalError("Unidentified spec"));
	}
	CSequence cSequence = new CSequence();
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    cSequence.addParticle((ICParticle)list.get(i));
	}
	CSequenceContentRule cRule = new CSequenceContentRule(
	    labelName,
	    cSequence,
	    cModule
	);
	cModule.addContentRule(cRule);
	cRule.setBase(define.getXMLElement());
	nsByLabel_.put(labelName, cModule.getTargetNamespace());
/*
	if (_canAttPool(list)) {
	    CAttListPattern attPool = new CAttListPattern(
		labelName,
		cModule
	    );
	    int size = list.size();
	    for (int i = 0;i < size;i++) {
		Object node = list.get(i);
		if (!(node instanceof CAttribute)) {
		    throw (new InternalError());
		}
		attPool.addContent((ICPatternContent)node);
	    }
	    cModule.addAttListPattern(attPool);
	} else {
	    CSequence cSequence = new CSequence();
	    int size = list.size();
	    for (int i = 0;i < size;i++) {
		cSequence.addParticle((ICParticle)list.get(i));
	    }
	    CSequenceContentRule cRule = new CSequenceContentRule(
		labelName,
		cSequence,
		cModule
	    );
	    cModule.addContentRule(cRule);
	}
*/
    }
/*
    private boolean _canAttPool(List list) {
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    Object node = list.get(i);
	    if (!(node instanceof CAttribute)) {
		return (false);
	    }
	}
	return (true);
    }
*/

    private CRefWithLabel _makeCRefWithLabel(
	RNRef ngRef,
	RNGModule ngModule,
	String occurs
    ) {
	String refLabelName = ngRef.getName();
	CRefWithLabel cRef = new CRefWithLabel(refLabelName, occurs);
	unresolvedRefs_.add(new RefPair(ngRef, cRef));
	return (cRef);
/*
	RNGModule targetModule = _getModule(ngRef);
	if (ngModule == targetModule) {
	    return (new CRefWithLabel(refLabelName, occurs));
	} else {
	    String namespace = (String)nsByLabel_.get(refLabelName);
	    return (new CRefWithLabel(refLabelName, namespace, occurs));
	}
*/
    }

    private static class RefPair {
	RNRef ngRef;
	CRefWithLabel cRef;

	RefPair(RNRef ngRef, CRefWithLabel cRef) {
	    this.ngRef = ngRef;
	    this.cRef = cRef;
	}
    }
}
