/*
 * RelaxerOrg class library
 *  Copyright (C) 2000,2001  ASAMI, Tomoharu (asami@zeomtech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package org.relaxer.mom;

import java.util.*;
import java.io.IOException;
import javax.jms.*;
import org.relaxer.tabular.*;
import org.relaxer.util.D2Array;

/**
 * QueueFacade
 *
 * @since   Jul. 17, 2001
 * @version Jul. 17, 2001
 * @author  ASAMI, Tomoharu (asami@zeomtech.com)
 */
public class QueueFacade extends AbstractJmsFacade {
    private QueueConnectionFactory factory_ = null;
    private Queue queue_ = null;
    private QueueConnection connection_ = null;
    private QueueSession session_ = null;
    private QueueSender sender_ = null;
    private QueueReceiver receiver_ = null;

    public QueueFacade(
	QueueConnectionFactory factory,
	Object destination
    ) throws JMSException {
	_setup(factory, destination);
    }

    public void close() {
	_release();
    }

    protected QueueFacade() {
    }

    protected void _setup(
	QueueConnectionFactory factory,
	Object destination
    ) throws JMSException {
	if (!(destination instanceof Queue)) {
	    throw (new JMSException("no queue = " + destination));
	}
	try {
	    factory_ = factory;
	    queue_ = (Queue)destination;
	    connection_ = factory.createQueueConnection();
	    session_ = connection_.createQueueSession(
		false,
		Session.AUTO_ACKNOWLEDGE
	    );
	} catch (JMSException e) {
	    _release();
	    throw (e);
	}
    }

    private void _release() {
	try {
	    if (sender_ != null) {
		sender_.close();
	    }
	} catch (JMSException e) {
	}
	try {
	    if (receiver_ != null) {
		receiver_.close();
	    }
	} catch (JMSException e) {
	}
	try {
	    if (session_ != null) {
		session_.close();
	    }
	} catch (JMSException e) {
	}
	try {
	    if (connection_ != null) {
		connection_.close();
	    }
	} catch (JMSException e) {
	}
	factory_ = null;
	queue_ = null;
	connection_ = null;
	session_ = null;
	sender_ = null;
	receiver_ = null;
    }

    public Message read() throws JMSException {
	if (receiver_ == null) {
	    receiver_ = session_.createReceiver(queue_);
	}
	return (receiver_.receiveNoWait());
    }

    public void write(Message message) throws JMSException {
	if (sender_ == null) {
	    sender_ = session_.createSender(queue_);
	}
	sender_.send(message);
    }

    public ITabular getTableOfContents() throws JMSException {
	QueueBrowser browser = session_.createBrowser(queue_);
	D2Array array = new D2Array();
	int y = 0;
	Enumeration enum = browser.getEnumeration();
	while (enum.hasMoreElements()) {
	    Message message = (Message)enum.nextElement();
	    _makeRecord(message, array, y);
	    y++;
	}
	browser.close();
	return (new D2ArrayTabular(array));
    }

    private void _makeRecord(Message message, D2Array array, int y)
	throws JMSException {

	String correlationId = message.getJMSCorrelationID();
	int deliveryMode = message.getJMSDeliveryMode();
	Destination destination = message.getJMSDestination();
	long expiration = message.getJMSExpiration();
	String messageId = message.getJMSMessageID();
	int priority = message.getJMSPriority();
	boolean redelivered = message.getJMSRedelivered();
	Destination replyTo = message.getJMSReplyTo();
	long timestamp = message.getJMSTimestamp();
	String type = message.getJMSType();
	array.put(0, y, messageId);
	array.put(1, y, Integer.toString(priority));
	array.put(2, y, Long.toString(timestamp));
	array.put(3, y, Long.toString(expiration));
    }

    public TextMessage createTextMessage() throws JMSException {
	return (session_.createTextMessage());
    }

    protected Session _getSession() {
	return (session_);
    }
}
