/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2002  ASAMI, Tomoharu (asami@relaxer.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.xml.relax.expanded;

import java.util.*;
import org.w3c.dom.*;
import jp.gr.java_conf.jaba2.util.UArray;
import jp.gr.java_conf.jaba2.util.ElementCounter;
import jp.gr.java_conf.jaba2.util.PropertyList;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.datatype.IXMLDatatype;
import jp.gr.java_conf.jaba2.datatype.XMLFacet;
import jp.gr.java_conf.jaba2.xml.UElement;
import jp.gr.java_conf.jaba2.xml.UXML;
import jp.gr.java_conf.jaba2.xml.datatype.*;
import jp.gr.java_conf.jaba2.xml.relax.cooked.*;

/**
 * EModule
 *
 * @since   Jan.  2, 2000
 * @version Jul. 30, 2002
 * @author  ASAMI, Tomoharu (asami@relaxer.org)
 */
public class EModule implements IEOccurs {
    private CModule cmodule_;
    private ERelax model_;
    private ENode[] roots_;
    // Map<CElementRule, EElementNode>
    private Map elementNodes_ = new HashMap();
    // Map<CContentRule, EContentNode>
    private Map contentNodes_ = new HashMap();
    private PropertyList properties_ = new PropertyList();

    public EModule(CModule cmodule) {
	cmodule_ = cmodule;
	List list = new ArrayList();
	_buildRoots(list);
	ENode[] roots = new ENode[list.size()];
	roots_ = (ENode[])list.toArray(roots);
    }

    public void adjustTree() {
	_adjustTrees();
	_adjustTrees();		// XXX : two pass
	_adjustNames();
    }

    public CModule getCModule() {
	return (cmodule_);
    }

    public String getFileURI() {
	return (cmodule_.getFileURI());
    }

    public String getBaseURI() {
	return (cmodule_.getBaseURI());
    }

    public String getModuleVersion() {
	return (cmodule_.getModuleVersion());
    }

    public String getRelaxCoreVersion() {
	return (cmodule_.getRelaxCoreVersion());
    }

    public String getTargetNamespace() {
	return (cmodule_.getTargetNamespace());
    }

    public String[] getExportLabels() {
	return (cmodule_.getExportLabels());
    }

    public String[] getExportRoles() {
	return (cmodule_.getExportRoles());
    }

    public String[] getImportLabels() {
	return (cmodule_.getImportLabels());
    }

    public String[] getImportRoles() {
	return (cmodule_.getImportRoles());
    }

    public ERelax getModel() {
	return (model_);
    }

    public void setModel(ERelax model) {
	model_ = model;
    }

    public void resolveExternalRefs() {
	ERuleNode[] eNodes = getElementRules();
	for (int i = 0;i < eNodes.length;i++) {
	    _resolveExternalRefs(eNodes[i]);
	}
	ERuleNode[] cNodes = getContentRules();
	for (int i = 0;i < cNodes.length;i++) {
	    _resolveExternalRefs(cNodes[i]);
	}
    }

    private void _resolveExternalRefs(ENode node) {
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EExternalRefNode) {
		children[i] = _resolveExternalRef((EExternalRefNode)child);
	    } else {
		_resolveExternalRefs(child);
	    }
	}
	boolean immutable = node.isImmutable();
	node.setImmutable(false);
	node.clear();
	node.addChildren(children);
	node.setImmutable(immutable);
    }

    private ENode _resolveExternalRef(EExternalRefNode node) {
	return (model_.resolveExternalRef(node));
    }

    public void resolveExternalRoleRefs() {
	ERuleNode[] eNodes = getElementRules();
	for (int i = 0;i < eNodes.length;i++) {
	    _resolveExternalRoleRefs(eNodes[i]);
	}
    }

    private void _resolveExternalRoleRefs(ENode node) {
	ENode[] children = node.getChildren();
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EExternalRoleRefNode) {
		ENode[] nodes =
		    _resolveExternalRoleRef((EExternalRoleRefNode)child);
		for (int j = 0;j < nodes.length;j++) {
		    list.add(nodes[j]);
		}
	    } else {
		list.add(child);
		_resolveExternalRoleRefs(child);
	    }
	}
	boolean immutable = node.isImmutable();
	node.setImmutable(false);
	node.clear();
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    node.addChild((ENode)list.get(i));
	}
	node.setImmutable(immutable);
    }

    private ENode[] _resolveExternalRoleRef(EExternalRoleRefNode node) {
	return (model_.resolveExternalRoleRef(node));
    }

    public ERuleNode[] getRootRules() {
	List list = new ArrayList();
	for (int i = 0;i < roots_.length;i++) {	// XXX : really needs?
	    ENode node = roots_[i];
	    if (!node.isNone()) {
		list.add(node);
//System.out.println("getRootRules:" + node);
	    }
	}
	ERuleNode[] nodes = new ERuleNode[list.size()];
	return ((ERuleNode[])list.toArray(nodes));
    }

    public ERuleNode[] getElementRules() { // XXX : return old impl?
	Collection col = elementNodes_.values();
	ENode[] tmp = new ENode[col.size()];
	tmp = (ENode[])col.toArray(tmp);
	List list = new ArrayList();
	for (int i = 0;i < tmp.length;i++) {
	    ENode node = tmp[i];
// System.out.println("---");
// System.out.println(node);
	    if (node instanceof EElementNode) {
	        EElementNode eNode = (EElementNode)node;
		if (!eNode.isNone()) {
		    list.add(eNode);
		}
	    } else if (node instanceof EChoiceNode) {
//System.out.println("EChoiceNode");
		ENode[] children = node.getChildren();
		for (int j = 0;j < children.length;j++) {
		    ENode child = children[j];
//System.out.println(child);
		    if (child instanceof EElementNode) {
			EElementNode eNode = (EElementNode)child;
			if (!eNode.isNone()) {
			    list.add(eNode);
			}
		    } else if (child instanceof EElementRefNode) {
			// XXX : needs precise check
			EElementRefNode erNode = (EElementRefNode)children[j];
			EElementNode eNode
			    = (EElementNode)erNode.getElementNode();
			if (!eNode.isNone()) {
			    list.add(eNode);
			}
		    }
		}
	    } else {
		throw (new InternalError());
	    }
	}
	ERuleNode[] result = new ERuleNode[list.size()];
	return ((ERuleNode[])list.toArray(result));
    }

    // XXX : address choice
    public ERuleNode[] getContentRules() {
	Collection col = contentNodes_.values();
	ERuleNode[] tmp = new ERuleNode[col.size()];
	tmp = (ERuleNode[])col.toArray(tmp);
	List list = new ArrayList();
	for (int i = 0;i < tmp.length;i++) {
	    EContentNode cNode = (EContentNode)tmp[i];
	    if (!cNode.isNone()) {
		list.add(cNode);
	    }
	}
	ERuleNode[] result = new ERuleNode[list.size()];
	return ((ERuleNode[])list.toArray(result));
    }

    public ERuleNode[] getActiveElementRules() {
	List eRules = new ArrayList();
	List cRules = new ArrayList();
	ERuleNode[] rootRules = getRootRules();
	_collectActiveRules(rootRules, eRules, cRules);
	ERuleNode[] result = new ERuleNode[eRules.size()];
	return ((ERuleNode[])eRules.toArray(result));
    }

    public ERuleNode[] getActiveContentRules() {
	List eRules = new ArrayList();
	List cRules = new ArrayList();
	ERuleNode[] rootRules = getRootRules();
	_collectActiveRules(rootRules, eRules, cRules);
	ERuleNode[] result = new ERuleNode[cRules.size()];
	return ((ERuleNode[])cRules.toArray(result));
    }

    private void _collectActiveRules(
	ENode[] eNodes,
	List eRules,
	List cRules
    ) {
	for (int i = 0;i < eNodes.length;i++) {
	    _collectActiveRules(eNodes[i], eRules, cRules);
	}
    }

    private void _collectActiveRules(ENode eNode, List eRules, List cRules) {
	if (eNode instanceof EElementNode) {
	    if (eRules.contains(eNode)) {
		return;
	    }
	    eRules.add(eNode);
	} else if (eNode instanceof EContentNode) {
	    if (cRules.contains(eNode)) {
		return;
	    }
	    cRules.add(eNode);
	}
	ENode[] children = eNode.getChildren();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementRefNode) {
		EElementRefNode eRef = (EElementRefNode)child;
		_collectActiveRules(eRef.getElementNode(), eRules, cRules);
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode cRef = (EContentRefNode)child;
		_collectActiveRules(cRef.getContentNode(), eRules, cRules);
	    } else {
		_collectActiveRules(child, eRules, cRules);
	    }
	}
    }

    public EElementNode[] resolveElementLabel(String label) {
	CElementRule[] crules = cmodule_.getElementRules(label);
	if (crules == null || crules.length == 0) {
	    return (null);
	}
	EElementNode[] erules = new EElementNode[crules.length];
	for (int i = 0;i < crules.length;i++) {
	    EElementNode rule = (EElementNode)elementNodes_.get(crules[i]);
	    if (rule == null) {
		throw (new InternalError());
	    }
	    erules[i] = rule;
	}
	return (erules);
    }

    public EContentNode[] resolveContentLabel(String label) {
	CContentRule[] crules = cmodule_.getContentRules(label);
	if (crules == null || crules.length == 0) {
	    return (null);
	}
	EContentNode[] erules = new EContentNode[crules.length];
	for (int i = 0;i < crules.length;i++) {
	    EContentNode rule = (EContentNode)contentNodes_.get(crules[i]);
	    if (rule == null) {
		throw (new InternalError());
	    }
	    erules[i] = rule;
	}
	return (erules);
    }

    public ENode[] resolveAttListRole(String role) {
	ENode node = _buildPattern(role);
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    children[i].setModule(this);
	}
	return (children);
    }

    // called from ERelax
    public void setup(OptimizerController controller) {
	ERuleNode[] eRules = getContentRules();
	ERuleNode[] cRules = getElementRules();
	for (int i = 0;i < eRules.length;i++) {
	    controller.addThread(eRules[i], this);
	}
	for (int i = 0;i < cRules.length;i++) {
	    controller.addThread(cRules[i], this);
	}
    }

    // called from optimizer
    public void addOptimizedContentRule(EContentNode cNode) {
	contentNodes_.put(new CAnonymousContentRule(), cNode);
    }

    private void _buildRoots(List list) {
	String[] exportLabels = cmodule_.getExportLabels();
	for (int i = 0;i < exportLabels.length;i++) {
	    CElementRule[] rules
		= cmodule_.getElementRules(exportLabels[i]);
	    if (rules != null) {
		for (int j = 0;j < rules.length;j++) {
		    ENode eRule = _buildRoot(rules[j]);
//		    if (!eRule.isNone()) { // XXX : Insufficient?
//			list.add(eRule);
//		    }
//System.out.println("buildRules:" + eRule);
		    list.add(eRule);
		}
	    } else {
		CContentRule[] cRules
		    = cmodule_.getContentRules(exportLabels[i]);
		if (cRules == null) {
		    return;
//		    throw (new ESyntaxErrorException(
//			"invalid label \"" + exportLabels[i] + "\""));
		}
		for (int j = 0;j < cRules.length;j++) {
		    ENode cRule = _buildRoot(cRules[j]);
//		    if (!cRule.isNone()) { // XXX : Insufficient?
//			list.add(cRule);
//		    }
		    list.add(cRule);
//System.out.println("buildRules:" + cRule);
		}
	    }
	}
    }

    private ENode _buildRoot(CElementRule root) {
	ENode eNode = _buildElementRule(root);
	if (eNode instanceof EChoiceNode) {
	    throw (new ESyntaxErrorException(
		"\"" + root.getLabel() + "\" is non deterministic root"
	    ));
	}
	return (eNode);
    }

    private ENode _buildRoot(CContentRule root) {
	ENode cNode = _buildContentRule(root);
	if (cNode instanceof EChoiceNode) {
	    throw (new ESyntaxErrorException(
		"\"" + root.getLabel() + "\" is non deterministic root"
	    ));
	}
	return (cNode);
    }

    private ENode _buildElementRule(CElementRule rule) {
	ENode node = (ENode)elementNodes_.get(rule);
	if (node != null) {
	    return (node);
	}
	node = _buildPattern(rule.getPattern(), rule.getLabel());
// System.out.println("_buildElementRule");
// System.out.println(rule);
// System.out.println(node);
	if (elementNodes_.containsKey(rule)) {
	    throw (new InternalError());
	}
	elementNodes_.put(rule, node); // XXX : temporaly
	ENode[] elements;
	if (node instanceof EElementNode) {
	    elements = new ENode[] { node };
	} else if (node instanceof EChoiceNode) {
	    elements = node.getChildren();
	} else {
	    throw (new InternalError());
	}
	for (int i = 0;i < elements.length;i++) {
	    EElementNode element = (EElementNode)elements[i];
	    element.setAppinfo(_makeAppinfo(rule));
	    element.setBase(rule.getBase());
	    if (rule instanceof CRelaxNgElementRule) {
		CRelaxNgElementRule crule = (CRelaxNgElementRule)rule;
		_setDatatype(element, crule.getDatatype());
		element.addChild(_buildSequence(crule.getSequence()));
	    } else if (rule instanceof CTypeElementRule) {
		CTypeElementRule crule = (CTypeElementRule)rule;
		_setDatatype(element, crule.getDatatype());
/*
	    } else if (rule instanceof CTypeElementRule) {
		// XXX : unify normal procedure
		CTypeElementRule crule = (CTypeElementRule)rule;
		EElementSlotNode esNode = new EElementSlotNode(
		    element.getElementName(),
		    element.getLabel()
		);
//		esNode.setTypeName(crule.getType());
		_setDatatype(esNode, crule.getDatatype());
		esNode.addChildren(element.getChildren());
		esNode.setAppinfo(_makeAppinfo(crule));
		esNode.setBase(crule.getBase());
		esNode.setAttrBase(element.getAttrBase());
//		esNode.setUniqueName(element.getLabel()); // XXX
		esNode.setModule(this);
		elements[i] = esNode;
		if (node instanceof EElementNode) {
		    elementNodes_.put(rule, esNode); // XXX
		}
*/
	    } else if (rule instanceof CSequenceElementRule) {
		element.addChildren(
		    _setupSequenceElementRule((CSequenceElementRule)rule)
		);
	    } else if (rule instanceof CChoiceElementRule) {
		element.addChildren(
		    _setupChoiceElementRule((CChoiceElementRule)rule)
		);
	    } else if (rule instanceof CRefElementRule) {
		element.addChildren(
		    _setupRefElementRule((CRefElementRule)rule)
		);
	    } else if (rule instanceof CElementElementRule) {
		element.addChildren(
		    _setupElementElementRule((CElementElementRule)rule)
		);
	    } else if (rule instanceof CEmptyElementRule) {
		// do nothing
	    } else if (rule instanceof CNoneElementRule) {
		element.addChild(new ENoneNode());
//		element.addNoneElementNode(
//		    (EElementNode)_buildElementRule(rule)
//		);
	    } else if (rule instanceof CMixedElementRule) {
		element.addChildren(
		    _setupMixedElementRule((CMixedElementRule)rule)
		);
	    } else {
		throw (new InternalError());
	    }
	}
	if (elements.length == 1) {
	    node = elements[0];
	} else {
	    EChoiceNode choice = new EChoiceNode();
	    choice.setConflict(true); // XXX
	    node = choice;
	    node.addChildren(elements);
	}
	node.setModule(this);
	node.setImmutable(true);
	elementNodes_.put(rule, node);
// System.out.println("end of _buildElementRule");
// System.out.println(node);
	return (node);
    }

    private ENode[] _setupSequenceElementRule(
	CSequenceElementRule rule
    ) {				// XXX : sequence occurs="*"
	CSequence sequence = rule.getSequence();
	ENode node = _buildSequence(sequence);
	if (node instanceof ESequenceNode) {
	    ESequenceNode esNode = (ESequenceNode)node;
	    if (esNode.getOccurs() == OCCURS_ONE) {
		return (node.getChildren()); // XXX : adjust tree
	    } else {
		return (new ENode[] { esNode });
	    }
	} else {
	    throw (new InternalError());
	}
    }

    private ENode[] _setupChoiceElementRule(
	CChoiceElementRule rule
    ) {
	return (new ENode[] { _buildChoice(rule.getChoice()) });
    }

    private ENode[] _setupRefElementRule(
	CRefElementRule rule
    ) {
	String label = rule.getRefLabel();
	int occurs = UERule.getOccurs(rule.getOccurs());
	String namespace = rule.getNamespace();
	ENode node;
	if (namespace == null) {
	    node = _makeNodeByLabel(label, occurs);
	} else {
	    node = _makeNodeByNamespace(namespace, label, occurs);
	}
	if (node == null) {
	    return (new ENode[0]);
	} else {
	    return (new ENode[] { node });
	}
    }

    private ENode[] _setupElementElementRule(
	CElementElementRule rule
    ) {
	return (new ENode[] { _buildElement(rule.getElement()) } );
    }

    private ENode[] _setupMixedElementRule(
	CMixedElementRule rule
    ) {
	return (new ENode[] { _buildMixed(rule.getMixed()) } );
    }

    private ENode _buildContentRule(CContentRule rule) {
	EContentNode node = (EContentNode)contentNodes_.get(rule);
	if (node != null) {
	    return (node);
	}
	node = new EContentNode(rule.getLabel());
	contentNodes_.put(rule, node);
	node.setAppinfo(_makeAppinfo(rule));
	node.setBase(rule.getBase());
	if (rule instanceof CSequenceContentRule) {
	    node.addChildren(
		_setupSequenceContentRule((CSequenceContentRule)rule)
	    );
	} else if (rule instanceof CChoiceContentRule) {
	    node.addChildren(
		_setupChoiceContentRule((CChoiceContentRule)rule)
	    );
	} else if (rule instanceof CRefContentRule) {
	    node.addChildren(
		_setupRefContentRule((CRefContentRule)rule)
	    );
	} else if (rule instanceof CElementContentRule) {
	    node.addChildren(
		_setupElementContentRule((CElementContentRule)rule)
	    );
	} else if (rule instanceof CEmptyContentRule) {
	    // do nothing
	} else if (rule instanceof CNoneContentRule) {
	    node.addChild(new ENoneNode());
	    node.setNone(true);
	} else if (rule instanceof CMixedContentRule) {
	    node.addChildren(
		_setupMixedContentRule((CMixedContentRule)rule)
	    );
	} else {
	    throw (new InternalError());
	}
	node.setModule(this);
	return (node);
    }

    private ENode[] _setupSequenceContentRule(
	CSequenceContentRule rule
    ) {				// XXX : sequence occurs="*"
	CSequence sequence = rule.getSequence();
	ENode node = _buildSequence(sequence);
	return (node.getChildren()); // XXX
    }

    private ENode[] _setupChoiceContentRule(
	CChoiceContentRule rule
    ) {
	return (new ENode[] { _buildChoice(rule.getChoice()) });
    }

    private ENode[] _setupRefContentRule(
	CRefContentRule rule
    ) {
	CRefWithLabel ref = rule.getRef();
	ENode node = _buildRefWithLabel(ref);
	if (node == null) {
	    return (new ENode[0]);
	} else {
	    return (new ENode[] { node });
	}
    }

    private ENode[] _setupElementContentRule(
	CElementContentRule rule
    ) {
	return (new ENode[] { _buildElement(rule.getElement()) } );
    }

    private ENode[] _setupMixedContentRule(
	CMixedContentRule rule
    ) {
	return (new ENode[] { _buildMixed(rule.getMixed()) } );
    }

    private EAttributeSlot _buildAttributeSlot(CAttribute cAttr) {
	IXMLDatatype datatype = cAttr.getDatatype();
	_adjustDatatype(datatype);
	EAttributeSlot aSlot = new EAttributeSlot(
	    cAttr.getName(),
	    cAttr.getDatatype()
	);
	aSlot.setRequired(cAttr.isRequired());
	aSlot.setAnd(cAttr.isAnd());
	aSlot.setEnumerations(cAttr.getEnumerations());
	aSlot.setAppinfo(_makeAppinfo(cAttr));
	aSlot.setBase(cAttr.getBase());
	aSlot.setNamespace(cAttr.getNamespace());
	return (aSlot);
    }

    private void _adjustDatatype(IXMLDatatype datatype) {
	if (datatype instanceof DComplex) {
	    DComplex complex = (DComplex)datatype;
	    if (complex.getProperty() == null) {
		String expr = complex.getExpr();
		Document doc = complex.getExprDoc();
		ENode node = _adjustExprTree(doc.getDocumentElement());
		complex.setProperty(node);
	    }
	}
    }

    private ENode _adjustExprTree(Element node) {
	String localName = node.getLocalName();
	if ("data".equals(localName)) {
	    return (_adjustExprTreeData(node));
	} else if ("value".equals(localName)) {
	    return (_adjustExprTreeValue(node));
	} else if ("ref".equals(localName)) {
	    return (_adjustExprTreeRef(node));
	} else if ("list".equals(localName)) {
	    return (_adjustExprTreeList(node));
	} else if ("choice".equals(localName)) {
	    return (_adjustExprTreeChoice(node));
	} else {
	    throw (new InternalError(localName));
	}
    }

    private ENode _adjustExprTreeData(Element data) {
	String typeName = data.getAttribute("type");
	Datatype datatype = DatatypeFactory.getDatatype(typeName);
	XMLFacet[] facets = _makeRELAXNGFacets(data);
	if (facets != null) {
	    datatype.addFacets(facets);
	}
	EDataSlot slot = new EDataSlot(datatype);
	return (slot);
    }

    private XMLFacet[] _makeRELAXNGFacets(Element data) {
	Element[] params = UElement.getElements(
	    data,
	    "http://relaxng.org/ns/structure/1.0",
	    "param"
	);
	XMLFacet[] facets = new XMLFacet[params.length];
	for (int i = 0;i < params.length;i++) {
	    Element param = params[i];
	    String value = UXML.element2Data(param);
	    String name = param.getAttribute("name");
	    facets[i] = new XMLFacet(name, value);
	}
	return (facets);
    }

    private ENode _adjustExprTreeValue(Element value) {
	String text = UXML.element2Data(value);
	String typeName = value.getAttribute("type");
	if (typeName != null && !"".equals(typeName)) {
	    return (new EValueSlot(text, typeName));
	} else {
	    return (new EValueSlot(text));
	}
    }

    private ENode _adjustExprTreeRef(Element ref) {
	String refName = ref.getAttribute("name");
	return (_adjustExprTreeRef(refName));
    }

    private ENode _adjustExprTreeRef(String refName) {
	ENode cRef = _makeNodeByLabel(refName, OCCURS_ONE);
	if (!(cRef instanceof EContentRefNode)) {
	    throw (new InternalError());
	}
	return (cRef);
    }

    private ENode _adjustExprTreeList(Element textList) {
	List list = _makeTextChildrenList(textList);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	case 1:
	    return ((ENode)list.get(0));
	default:
	    ESequenceNode result = new ESequenceNode();	// XXX : EListNode
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeChoice(Element choice) {
	List list = _makeTextChildrenList(choice);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	case 1:
	    return ((ENode)list.get(0));
	default:
	    EChoiceNode result = new EChoiceNode();
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeOptional(Element optional) {
	List list = _makeTextChildrenList(optional);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	default:
	    ESequenceNode result = new ESequenceNode(OCCURS_ZEROONE);
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeOneOrMore(Element oneOrMore) {
	List list = _makeTextChildrenList(oneOrMore);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	default:
	    ESequenceNode result = new ESequenceNode(OCCURS_ONEMORE);
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeZeroOrMore(Element zeroOrMore) {
	List list = _makeTextChildrenList(zeroOrMore);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	default:
	    ESequenceNode result = new ESequenceNode(OCCURS_ZEROMORE);
	    result.addChildren(list);
	    return (result);
	}
    }

    private List _makeTextChildrenList(Element parent) {
	Element[] children = UElement.getElements(parent);
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    Element child = children[i];
	    String localName = child.getLocalName();
	    if ("data".equals(localName)) {
		list.add(_adjustExprTreeData(child));
	    } else if ("value".equals(localName)) {
		list.add(_adjustExprTreeValue(child));
	    } else if ("ref".equals(localName)) {
		list.add(_adjustExprTreeRef(child));
	    } else if ("list".equals(localName)) {
		list.add(_adjustExprTreeList(child));
	    } else if ("optional".equals(localName)) {
		list.add(_adjustExprTreeOptional(child));
	    } else if ("oneOrMore".equals(localName)) {
		list.add(_adjustExprTreeOneOrMore(child));
	    } else if ("zeroOrMore".equals(localName)) {
		list.add(_adjustExprTreeZeroOrMore(child));
	    } else if ("choice".equals(localName)) {
		list.add(_adjustExprTreeChoice(child));
	    } else {
		throw (new UnsupportedOperationException());
	    }
	}
	return (list);
    }

    private ENode _buildSequence(CSequence sequence) {
	int occurs = UERule.getOccurs(sequence.getOccurs());
	ESequenceNode node = new ESequenceNode(occurs);
	ICParticle[] particles = sequence.getParticles();
	for (int j = 0;j < particles.length;j++) {
	    ICParticle particle = particles[j];
	    ENode newNode;
	    if (particle instanceof CSequence) {
		newNode = _buildSequence((CSequence)particle);
	    } else if (particle instanceof CChoice) {
		newNode = _buildChoice((CChoice)particle);
	    } else if (particle instanceof CInterleave) {
		newNode = _buildInterleave((CInterleave)particle);
	    } else if (particle instanceof CRefWithLabel) {
		newNode = _buildRefWithLabel((CRefWithLabel)particle);
	    } else if (particle instanceof CRefImportedLabel) {
		newNode = _buildRefImportedLabel((CRefImportedLabel)particle);
	    } else if (particle instanceof CElement) {
		newNode = _buildElement((CElement)particle);
	    } else if (particle instanceof CEmpty) {
		newNode = null;
	    } else if (particle instanceof CNone) {
		return (new ENoneNode());
	    } else if (particle instanceof CRefWithPattern) {
		throw (new InternalError());
	    } else if (particle instanceof CAnyOtherElement) {
		newNode = _buildAnyOtherElement((CAnyOtherElement)particle);
	    } else if (particle instanceof CAttribute) {
		newNode = _buildAttributeNode((CAttribute)particle); // XXX
	    } else if (particle instanceof CData) {
		newNode = _buildDataNode((CData)particle);
	    } else if (particle instanceof CValue) {
		newNode = _buildValueNode((CValue)particle);
	    } else {
		throw (new InternalError(particle.toString()));
	    }
	    if (newNode != null) {
		node.addChild(newNode);
	    }
	}
	return (node);
    }

    private ENode _buildChoice(CChoice choice) {
	COccurs occurs = choice.getOccurs();
	EChoiceNode node = new EChoiceNode(UERule.getOccurs(occurs));
	ICParticle[] particles = choice.getParticles();
	for (int j = 0;j < particles.length;j++) {
	    ICParticle particle = particles[j];
	    ENode newNode;
	    if (particle instanceof CSequence) {
		newNode = _buildSequence((CSequence)particle);
	    } else if (particle instanceof CChoice) {
		newNode = _buildChoice((CChoice)particle);
	    } else if (particle instanceof CRefWithLabel) {
		newNode = _buildRefWithLabel((CRefWithLabel)particle);
	    } else if (particle instanceof CRefImportedLabel) {
		newNode = _buildRefImportedLabel((CRefImportedLabel)particle);
	    } else if (particle instanceof CElement) {
		newNode = _buildElement((CElement)particle);
	    } else if (particle instanceof CEmpty) {
		newNode = null;
	    } else if (particle instanceof CNone) {
		return (new ENoneNode());
	    } else if (particle instanceof CRefWithPattern) {
		throw (new InternalError());
	    } else if (particle instanceof CAttribute) {
		newNode = _buildAttributeNode((CAttribute)particle);
	    } else if (particle instanceof CData) {
		newNode = _buildDataNode((CData)particle);
	    } else if (particle instanceof CValue) {
		newNode = _buildValueNode((CValue)particle);
	    } else {
		throw (new InternalError());
	    }
	    if (newNode != null) {
		node.addChild(newNode);
	    }
	}
	return (node);
    }

    private ENode _buildInterleave(CInterleave interleave) {
	COccurs occurs = interleave.getOccurs();
	EInterleaveNode node = new EInterleaveNode(UERule.getOccurs(occurs));
	ICParticle[] particles = interleave.getParticles();
	for (int j = 0;j < particles.length;j++) {
	    ICParticle particle = particles[j];
	    ENode newNode;
	    if (particle instanceof CSequence) {
		newNode = _buildSequence((CSequence)particle);
	    } else if (particle instanceof CChoice) {
		newNode = _buildChoice((CChoice)particle);
	    } else if (particle instanceof CInterleave) {
		newNode = _buildInterleave((CInterleave)particle);
	    } else if (particle instanceof CRefWithLabel) {
		newNode = _buildRefWithLabel((CRefWithLabel)particle);
	    } else if (particle instanceof CRefImportedLabel) {
		newNode = _buildRefImportedLabel((CRefImportedLabel)particle);
	    } else if (particle instanceof CElement) {
		newNode = _buildElement((CElement)particle);
	    } else if (particle instanceof CEmpty) {
		newNode = null;
	    } else if (particle instanceof CNone) {
		return (new ENoneNode());
	    } else if (particle instanceof CRefWithPattern) {
		throw (new InternalError());
	    } else if (particle instanceof CAttribute) {
		newNode = _buildAttributeNode((CAttribute)particle);
	    } else if (particle instanceof CData) {
		newNode = _buildDataNode((CData)particle);
	    } else if (particle instanceof CValue) {
		newNode = _buildValueNode((CValue)particle);
	    } else {
		throw (new InternalError());
	    }
	    if (newNode != null) {
		node.addChild(newNode);
	    }
	}
	return (node);
    }

    private ENode _buildRefWithLabel(CRefWithLabel cref) {
	String label = cref.getLabel();
	int occurs = UERule.getOccurs(cref.getOccurs());
	String namespace = cref.getNamespace();
	if (namespace == null) {
	    return (_makeNodeByLabel(label, occurs));
	} else {
	    return (_makeNodeByNamespace(namespace, label, occurs));
	}
    }

    private ENode _buildRefImportedLabel(CRefImportedLabel cref) {
	String label = cref.getLabel();
	int occurs = UERule.getOccurs(cref.getOccurs());
	try {
	    return (_makeNodeByLabel(label, occurs));
	} catch (ESyntaxErrorException e) { // XXX
	    return (null);
	}
    }

    private ENode _buildElement(CElement celement) {
	String name = celement.getName();
	String type = celement.getType();
	String label = celement.getLabel();
	COccurs coccurs = celement.getOccurs();
	String pattern = celement.getPattern();
	CTagPattern[] tagPatterns;
	if (type != null && label != null) {
	    throw (new ESyntaxErrorException(
		"element \"" + celement.getName() +
		"\" has both type and label"
	    ));
	}
	int occurs = UERule.getOccurs(coccurs);
	if (pattern != null) {	// XXX : obsolate
	    throw (new InternalError());
//	    return (_buildPattern(pattern));
	} else if (name != null) {
	    if (type != null) {
		String tmpLabel = "element" + _getSequenceNo();
		EElementSlotNode element
		    = new EElementSlotNode(name, tmpLabel);
//		element.setTypeName(type);
		_setDatatype(element, celement.getDatatype());
		element.setBaseName(name); // XXX : base name semantics
		element.setUniqueName(
		    _adjustUniqueName(
			name + UString.capitalize(tmpLabel)
		    )
		); // XXX
		element.setAppinfo(_makeAppinfo(celement));
		element.setBase(celement.getBase());
		element.setModule(this);
		elementNodes_.put(new CAnonymousElementRule(), element);
		return (new EElementRefNode(element, occurs));
/*
		if (occurs == OCCURS_ONE) {
		    EElementSlot eSlot = new EElementSlot(name, type);
		    Datatype datatype = celement.getDatatype();
		    eSlot.setDatatype(datatype);
		    return (eSlot);
		} else {
		    ESequenceNode eSeq = new ESequenceNode(occurs);
		    EElementSlot eSlot = new EElementSlot(name, type);
		    Datatype datatype = celement.getDatatype();
		    eSlot.setDatatype(datatype);
		    eSeq.addChild(eSlot);
		    return (eSeq);
		}
*/
	    } else if (label != null) {	// deprecated procedure
		String tmpLabel = label + _getSequenceNo(); // XXX
		EElementNode element = new EElementNode(name, tmpLabel);
		element.setBaseName(name);
		// XXX
		element.setAppinfo(_makeAppinfo(celement));
		element.setBase(celement.getBase());
		element.setModule(this);
		element.addChild(
		    _makeNodeByLabel(label, OCCURS_ONE)	// XXX : would be null
		);
		elementNodes_.put(new CAnonymousElementRule(), element);
		return (
		    new EElementRefNode(element, occurs)
		);
	    } else {
		throw (new InternalError());
	    }
	} else {
	    throw (new ESyntaxErrorException("element has no name"));
	}
    }

    private ENode _buildMixed(CMixed mixed) {
	EMixedNode node = new EMixedNode();
	ICParticle[] particles = mixed.getParticles();
	for (int j = 0;j < particles.length;j++) {
	    ICParticle particle = particles[j];
	    ENode newNode;
	    if (particle instanceof CSequence) {
		newNode = _buildSequence((CSequence)particle);
	    } else if (particle instanceof CChoice) {
		newNode = _buildChoice((CChoice)particle);
	    } else if (particle instanceof CInterleave) {
		newNode = _buildInterleave((CInterleave)particle);
	    } else if (particle instanceof CRefWithLabel) {
		newNode = _buildRefWithLabel((CRefWithLabel)particle);
	    } else if (particle instanceof CRefImportedLabel) {
		newNode = _buildRefImportedLabel((CRefImportedLabel)particle);
	    } else if (particle instanceof CElement) {
		newNode = _buildElement((CElement)particle);
	    } else if (particle instanceof CEmpty) {
		newNode = null;
	    } else if (particle instanceof CNone) {
		return (null);
	    } else if (particle instanceof CRefWithPattern) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	    if (newNode != null) {
		node.addChild(newNode);
	    }
	}
	return (node);
    }

    private ENode _buildAnyOtherElement(CAnyOtherElement any) {
	EAnyOtherElementNode node = new EAnyOtherElementNode(any);
	node.setOccurs(UERule.getOccurs(any.getOccurs()));
	node.setIncludeNamespaces(any.getIncludeNamespaces());
	node.setExcludeNamespaces(any.getExcludeNamespaces());
	node.setBase(any.getBase());
	return (node);
    }

    private ENode _buildAnyOtherAttribute(CAnyOtherAttribute any) {
	EAnyOtherAttributeNode node = new EAnyOtherAttributeNode(any);
	node.setIncludeNamespaces(any.getIncludeNamespaces());
	node.setExcludeNamespaces(any.getExcludeNamespaces());
	node.setBase(any.getBase());
	return (node);
    }

/*
    // only used in RELAXNG
    private ENode _buildAttribute(CAttribute attribute) {// XXX : obsolate
	return (
	    new EAttributeSlot(attribute.getName(), attribute.getDatatype())
	);
    }
*/

    // only used in RELAXNG
    private ENode _buildAttributeNode(CAttribute attribute) {
	ESequenceNode node = new ESequenceNode();
	EAttributeSlot aSlot = _buildAttributeSlot(attribute);
	aSlot.setRequired(true);// must set for choice
	aSlot.setAnd(true);	// must set for choice
	node.addChild(aSlot);
	return (node);
/*
	Datatype datatype = attribute.getDatatype();
	CValue value = attribute.getValue();
	CRefWithPattern ref = attribute.getRef();
	if (datatype == null &&
	    value == null &&
	    ref == null) {

	    throw (new InternalError());
	}
	// XXX
*/
/*
	ESequenceNode node = new ESequenceNode();
	EAttributeSlot aSlot = new EAttributeSlot(
	    attribute.getName(),
	    attribute.getDatatype()
	);
	aSlot.setRequired(true);// must set for choice
	aSlot.setAnd(true);	// must set for choice
	node.addChild(aSlot);
	return (node);
//	return (new EAttributeLeaf(attribute));
*/
    }

    private ENode _buildDataNode(CData data) {
	return (new EDataSlot(data));
    }

    private ENode _buildValueNode(CValue value) {
	return (new EValueSlot(value));
    }

    private ENode _buildPattern(String pattern, String label) {
	ENode node = _buildPattern(pattern);
	if (node instanceof ERuleNode) {
	    ERuleNode eNode = (ERuleNode)node;
	    eNode.setLabel(label);
	} else if (node instanceof EChoiceNode) {
	    ENode[] children = node.getChildren();
	    for (int i = 0;i < children.length;i++) {
		ENode child = children[i];
		if (child instanceof ERuleNode) {
		    ERuleNode eNode = (ERuleNode)child;
		    eNode.setLabel(label);
/*
		    eNode.setBaseName(pattern);	// XXX
System.out.println(pattern);
*/
		} else {
		    throw (new InternalError());
		}
	    }
	} else {
	    throw (new ESyntaxErrorException(
		"role \"" + pattern + "\" has no tag name"
	    ));
	}
	return (node);
    }

    private ENode _buildPattern(String pattern) {
	CTagPattern[] tagPatterns = cmodule_.getTagPatterns(pattern);
	CAttListPattern[] attListPatterns
	    = cmodule_.getAttListPatterns(pattern);
	ENode node = new EChoiceNode();
	((EChoiceNode)node).setConflict(true);
	if (tagPatterns != null) {
	    for (int i = 0;i < tagPatterns.length;i++) {
		ENode child = _buildTagPattern(tagPatterns[i]);
		_setupBase(child, tagPatterns[i]);
		node.addChild(child);
	    }
	}
	if (attListPatterns != null) {
	    for (int i = 0;i < attListPatterns.length;i++) {
		ENode child = _buildAttListPattern(attListPatterns[i]);
		_setupBase(child, attListPatterns[i]);
		node.addChild(child);
	    }
	}
	switch (node.getSize()) {

	case 0:
	    if (!cmodule_.isImportRole(pattern)) {
		throw (new ESyntaxErrorException(
		    "void role \"" + pattern + "\""));
	    } else {
		return (node);
	    }
	case 1:
	    return ((ENode)node.getFirstChild());
	default:
/*
	    ENode[] children = node.getChildren();
	    for (int i = 0;i < children.length;i++) {
		ENode child = children[i];
		if (child instanceof ERuleNode) {
		    ERuleNode rNode = (ERuleNode)child;
System.out.println(pattern);
		    rNode.setBaseName(pattern);	// XXX
		}
	    }
*/
	    return (node);
	}
    }

    private void _setupBase(ENode node, CPattern pattern) {
	Element base = pattern.getBase();
	if (base == null) {
	    return;
	}
	_setupBase(node, base);
    }

    private void _setupBase(ENode node, Element base) {
	if (node instanceof EElementNode) {
	    EElementNode eNode = (EElementNode)node;
	    eNode.setAttrBase(base);
	} else {
	    ENode[] children = node.getChildren();
	    for (int i = 0;i < children.length;i++) {
		_setupBase(children[i], base);
	    }
	}
    }

    private ENode _buildTagPattern(CTagPattern ctagpattern) {
	String tagName = ctagpattern.getTagName();
	if (tagName == null) {
	    tagName = ctagpattern.getName();
	}
	ENode node = new EElementNode(tagName);
	ICPatternContent[] contents = ctagpattern.getContents();
	for (int i = 0;i < contents.length;i++) {
	    ICPatternContent content = contents[i];
	    if (content instanceof CAttribute) {
		EAttributeSlot aSlot
		    = _buildAttributeSlot((CAttribute)content);
/*
		CAttribute ca = ((CAttribute)content);
		EAttributeSlot aSlot = new EAttributeSlot(
			ca.getName(),
			ca.getDatatype()
		);
//		aSlot.setDatatype(ca.getDatatype());
		aSlot.setRequired(ca.isRequired());
		aSlot.setAnd(ca.isAnd());
		aSlot.setEnumerations(ca.getEnumerations());
		aSlot.setAppinfo(_makeAppinfo(ca));
		aSlot.setBase(ca.getBase());
		aSlot.setNamespace(ca.getNamespace());
*/
		node.addChild(aSlot);
	    } else if (content instanceof CRefWithPattern) {
		CRefWithPattern cref = (CRefWithPattern)content;
		String namespace = cref.getNamespace();
		if (namespace == null) {
		    ENode expanded = _buildPattern(cref.getPattern());
		    node.addChildren(expanded);
		} else {
		    node.addChild(
			new EExternalRoleRefNode(namespace, cref.getPattern())
		    );
		}
	    } else if (content instanceof CAnyOtherAttribute) {
		node.addChild(
		    _buildAnyOtherAttribute((CAnyOtherAttribute)content)
		);
	    } else {
		throw (new InternalError());
	    }
	}
	return (node);
    }

/*
    private ENode _buildAttListPattern(CAttListPattern calpattern) {
	String namespace = calpattern.getNamespace();
	if (namespace != null) {
	    return (
		new EExternalRoleRefNode(
		    namespace,
		    calpattern.getName(),
		    calpattern
		)
	    );
	}
	return (_makeAttListPattern(calpattern));
    }
*/

    private ENode _buildAttListPattern(CAttListPattern calpattern) {
	List list = new ArrayList();
	String tagName = null;
	ICPatternContent[] contents = calpattern.getContents();
	for (int i = 0;i < contents.length;i++) {
	    ICPatternContent content = contents[i];
	    if (content instanceof CAttribute) {
		EAttributeSlot aSlot
		    = _buildAttributeSlot((CAttribute)content);
/*
		CAttribute ca = ((CAttribute)content);
		EAttributeSlot aSlot = new EAttributeSlot(
		    ca.getName(),
		    ca.getDatatype()
		);
//		aSlot.setDatatype(ca.getDatatype());
		aSlot.setRequired(ca.isRequired());
		aSlot.setAnd(ca.isAnd());
		aSlot.setEnumerations(ca.getEnumerations());
		aSlot.setAppinfo(_makeAppinfo(ca));
		aSlot.setBase(ca.getBase());
		aSlot.setNamespace(ca.getNamespace());
*/
		list.add(aSlot);
	    } else if (content instanceof CRefWithPattern) {
		CRefWithPattern cref = (CRefWithPattern)content;
		String namespace = cref.getNamespace();
		if (namespace == null) {
		    ENode expanded = _buildPattern(cref.getPattern());
		    ENode[] expandedContents = expanded.getChildren();
		    for (int j = 0;j < expandedContents.length;j++) {
			ENode child = expandedContents[j];
			if (child instanceof EElementNode) {
			    tagName = ((EElementNode)child).getTagName();
			}
			list.add(child);
		    }
		} else {
		    list.add(
			new EExternalRoleRefNode(namespace, cref.getPattern())
		    );
		}
	    } else {
		throw (new InternalError());
	    }
	}
	ENode node;
	if (tagName != null) {
	    node = new EElementNode(tagName);
	} else {
	    node = new ESequenceNode();
	}
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    node.addChild((ENode)list.get(i));
	}
	return (node);
    }

    // make node

    private ENode _makeNodeByLabel(String label, int occurs) {
	CElementRule[] erules = cmodule_.getElementRules(label);
	CContentRule[] crules = cmodule_.getContentRules(label);
	if (erules == null) {
	    erules = new CElementRule[0];
	}
	if (crules == null) {
	    crules = new CContentRule[0];
	}
	ENode[] eNodes = new ENode[erules.length];
	for (int i = 0;i < erules.length;i++) {
	    eNodes[i] = _buildElementRule(erules[i]);
	}
	EContentNode[] cNodes = new EContentNode[crules.length];
	for (int i = 0;i < crules.length;i++) {
	    cNodes[i] = (EContentNode)_buildContentRule(crules[i]);
	}
	boolean unique = true;
	if (eNodes.length + cNodes.length > 1) {
	    unique = false;
	}
	for (int i = 0; i < eNodes.length;i++) {
	    if (eNodes[i] instanceof EChoiceNode) {
		unique = false;
		break;
	    }
	}
	if (!unique) {
	    ElementCounter counter = new ElementCounter();
	    for (int i = 0;i < eNodes.length;i++) {
		ENode node = eNodes[i];
		ENode[] nodes;
		if (node instanceof EElementNode) {
		    nodes = new ENode[] { node };
		} else if (node instanceof EChoiceNode) {
		    nodes = node.getChildren();
		} else {
		    throw (new InternalError());
		}
		for (int j = 0;j < nodes.length;j++) {
		    EElementNode eNode = (EElementNode)nodes[j];
		    String eLabel = eNode.getLabel();
		    String eRole = eNode.getElementName();
		    String uniqueName;
		    if (eLabel.equalsIgnoreCase(eRole)) {
			uniqueName = eLabel;
		    } else {
			uniqueName = eRole + UString.capitalize(eLabel);
		    }
		    uniqueName = _adjustUniqueName(uniqueName);
		    eNode.setUniqueName(uniqueName);
		    counter.add(uniqueName);
		}
	    }
	    ElementCounter memo = new ElementCounter();
	    for (int i = 0;i < eNodes.length;i++) {
		ENode node = eNodes[i];
		ENode[] nodes;
		if (node instanceof EElementNode) {
		    nodes = new ENode[] { node };
		} else if (node instanceof EChoiceNode) {
		    nodes = node.getChildren();
		} else {
		    throw (new InternalError());
		}
		for (int j = 0;j < nodes.length;j++) {
		    EElementNode eNode = (EElementNode)nodes[j];
		    String uniqueName = eNode.getUniqueName();
		    switch (counter.getCount(uniqueName)) {

		    case 0:
			throw (new InternalError());
		    case 1:
			// do nothing
			break;
		    default:
			int no = memo.getCount(uniqueName);
			memo.add(uniqueName);
			eNode.setUniqueName(
			    _adjustUniqueName(uniqueName + (no + 1))
			);
		    }
		}
	    }
	    for (int i = 0;i < cNodes.length;i++) {
		EContentNode cNode = cNodes[i];
		String cLabel = cNode.getLabel();
		if (cNode.getUniqueName().equals(cLabel)) {
		    cNode.setUniqueName(
			_adjustUniqueName(cLabel + (i + 1))
		    );
		}
	    }
	}
	if (erules.length == 0 && crules.length == 0) {
	    throw (new ESyntaxErrorException(
		"label \"" + label + "\" has no rule"
	    ));
	} else if (erules.length == 1 && crules.length == 0) {
	    return (
		_elementNode2ElementRefNode(eNodes[0], occurs)
	    );
	} else if (erules.length == 0 && crules.length == 1) {
	    return (
		new EContentRefNode(cNodes[0], occurs)
	    );
	} else {
	    EChoiceNode chNode = new EChoiceNode(occurs);
	    chNode.setConflict(true);
	    // XXX : duplicate <choice> and label confliction
	    for (int i = 0;i < erules.length;i++) {
		chNode.addChild(
		    _elementNode2ElementRefNode(eNodes[i], occurs)
		);
	    }
	    for (int i = 0;i < crules.length;i++) {
		chNode.addChild(
		    new EContentRefNode(cNodes[i], occurs)
		);
	    }
	    return (chNode);
	}
    }

    private ENode _makeNodeByNamespace(
	String namespace,
	String label,
	int occurs
    ) {
	// use EExternalRefNode temporarily
	return (new EExternalRefNode(namespace, label, occurs));
    }

    private EAppinfo _makeAppinfo(CNode node) {
	Element base = node.getBase();
	if (base == null) {
	    return (null);
	}
	Element annotation = UElement.getOnlyElement(base, "annotation");
	if (annotation == null) {
	    return (null);
	}
	Element appinfo = UElement.getOnlyElement(annotation, "appinfo");
	if (appinfo == null) {
	    return (null);
	}
	return (new EAppinfo(appinfo));
    }

    private void _setDatatype(EElementNode eNode, Datatype datatype) {
	eNode.setDatatype(datatype);
    }

    private ENode _elementNode2ElementRefNode(ENode node, int occurs) {
	if (node instanceof EElementNode) {
	    return (new EElementRefNode((EElementNode)node, occurs));
	} else if (node instanceof EChoiceNode) {
	    EChoiceNode chNode = (EChoiceNode)node;
	    ENode[] children = chNode.getChildren();
	    EChoiceNode newNode = new EChoiceNode(occurs); // XXX : occurs
	    newNode.setConflict(chNode.isConflict());
	    for (int i = 0;i < children.length;i++) {
		newNode.addChild(
		    new EElementRefNode((EElementNode)children[i], occurs)
		);
	    }
// System.out.println("_elementNode2ElementRefNode");
// System.out.println(chNode);
	    return (newNode);
	} else {
	    throw (new InternalError());
	}
    }

    // adjust
    // XXX : divide normalization and optimization

    private void _adjustTrees() {
	ERuleNode[] eNodes = getElementRules();
	for (int i = 0;i < eNodes.length;i++) {
	    _adjustTree((EElementNode)eNodes[i]);
	}
	ERuleNode[] cNodes = getContentRules();
	for (int i = 0;i < cNodes.length;i++) {
	    _adjustTree((EContentNode)cNodes[i]);
	}
    }

    private void _adjustTree(ERuleNode node) {
//System.out.println("debug(before): " + node);
	node.setImmutable(false);
	ENode[] children = node.getChildren();
	node.clear();
	for (int i = 0;i < children.length;i++) {
	    ENode child = _adjustTree(children[i]);
	    if (child != null) {
		if (child instanceof EElementSlot) {
		    EElementSlot eSlot = (EElementSlot)child;
//		    String typeName = eSlot.getTypeName();
		    String typeName = eSlot.getDatatype().getName();
		    if ("none".equals(typeName)) {
			node.addNoneElementSlot(eSlot);
		    } else {
			node.addChild(child);
		    }
		} else if (child instanceof EAttributeSlot) {
		    EAttributeSlot aSlot = (EAttributeSlot)child;
//		    String typeName = aSlot.getTypeName();
		    String typeName = aSlot.getDatatype().getName();
		    if ("none".equals(typeName)) {
			node.addNoneAttributeSlot(aSlot);
		    } else {
			node.addChild(child);
		    }
		} else if (child instanceof EElementRefNode) {
		    // XXX
		    EElementRefNode eRef = (EElementRefNode)child;
		    EElementNode eNode = eRef.getElementNode();
		    if (!eNode.isNone()) {
			node.addChild(child);
		    }
		} else if (child instanceof EMixedNode) {
		    if ((node instanceof EElementNode) &&
			(child.getSize() == 0)) {

			EElementNode eNode = (EElementNode)node;
//			eNode.setTypeName("string");
			eNode.setDatatype(new DString());
		    } else {
			node.addChild(child);
		    }
		} else if (child instanceof ETempNode) {
		    node.addChildren(child);
		} else {
		    node.addChild(child);
		}
	    }
	}
	node.setImmutable(true);
//System.out.println("debug(after): " + node);
    }

    private ENode _adjustTree(ENode node) {
	if (node instanceof EElementNode) {
	    throw (new InternalError());
	} else if (node instanceof EContentNode) {
	    throw (new InternalError());
	} else if (node instanceof EElementSlot) {
	    return (node);
	} else if (node instanceof EAttributeSlot) {
	    return (node);
	} else if (node instanceof ENoneNode) {
	    return (node);
	} else if (node instanceof EElementRefNode) {
	    EElementRefNode eRef = (EElementRefNode)node;
	    int occurs = eRef.getOccurs();
	    EElementNode eNode = eRef.getElementNode();
	    if (eNode.isNone()) {
		return (new ENoneNode());
	    }
	    if (eNode instanceof EElementSlotNode) {
		EElementSlotNode esNode = (EElementSlotNode)eNode;
		if (esNode.getSize() > 0) {
		    return (node);
		}
//		if ("emptyString".equals(esNode.getTypeName())) {
		if ("emptyString".equals(esNode.getDatatype().getName())) {
		    return (node);
		}
		ENode parent = esNode.getParent();
		if (parent instanceof EChoiceNode ||
		    parent instanceof EMixedNode) {

		    return (node);
		}
		EElementSlot eSlot = new EElementSlot(
		    esNode.getElementName(),
//		    esNode.getTypeName(),
		    esNode.getDatatype(),
		    occurs
		);
//		eSlot.setDatatype(esNode.getDatatype());
		eSlot.setAppinfo(esNode.getAppinfo());
		eSlot.setBase(esNode.getBase());
		eSlot.setModule(this);
		return (eSlot);
	    } else {
		return (node);
	    }
	} else if (node instanceof EContentRefNode) {
	    EContentRefNode cRef = (EContentRefNode)node;
	    int occurs = cRef.getOccurs();
	    EContentNode cNode = cRef.getContentNode();
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE) {
		    return (new ENoneNode());
		} else {
		    return (null);
		}
	    }
	    if (occurs != OCCURS_ONE) {
		return (node);
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return (null);
	    }
	    ENode parent = cRef.getParent();
	    if (parent instanceof EChoiceNode ||
		parent instanceof EMixedNode) {

		if (children.length == 1) {
		    return (
			_adjustTreeImportContentOverChoiceMixed(
			    children[0]
			)
		    );
		} else {
		    ETempNode temp = new ETempNode();
		    for (int i = 0;i < children.length;i++) {
			temp.addChild(
			    _adjustTreeImportContentOverChoiceMixed(
				children[i]
			    )
			);
		    }
		    return (temp);
		}
	    } else {
		if (children.length == 1) {
		    return (_adjustTree(_deepCopy(children[0])));
		} else {
		    ETempNode temp = new ETempNode();
		    for (int i = 0;i < children.length;i++) {
			temp.addChild(_deepCopy(_adjustTree(children[i])));
		    }
		    return (temp);
		}
	    }
	} else if (node instanceof ESequenceNode) {
	    int occurs = ((ESequenceNode)node).getOccurs();
	    ENode[] children = _adjustChildren(node);
	    if (children.length == 0) {
		return (null);
	    } else if (children.length == 1 && occurs == OCCURS_ONE) {
		return (_adjustTree(children[0]));
	    } else {
		ENode parent = node.getParent();
		if (occurs != OCCURS_ONE ||
		    parent instanceof EChoiceNode ||
		    parent instanceof EMixedNode) {

		    // XXX : precise name
		    EContentNode cNode = new EContentNode(
			"sequence" + _getSequenceNo()
		    );
		    _addChildren(cNode, children);
		    cNode.setModule(this);
		    contentNodes_.put(new CAnonymousContentRule(), cNode);
		    return (new EContentRefNode(cNode, occurs));
		} else {
		    node.clear();
		    _addChildren(node, children);
		    return (node);
		}
	    }
	} else if (node instanceof EChoiceNode) {
	    ENode adjusted = _adjustChoice((EChoiceNode)node);
	    if (!(adjusted instanceof EChoiceNode)) {
		return (adjusted);
	    }
	    EChoiceNode choice = (EChoiceNode)adjusted;
	    if (UERule.hasChildChoice(choice)) {
		int occurs = choice.getOccurs();
		// XXX : precise name
		EContentNode cNode = new EContentNode(
		    "choice" + _getSequenceNo()
		);
		EChoiceNode contentChoice = new EChoiceNode(OCCURS_ONE);
		contentChoice.addChildren(choice);
		cNode.addChild(contentChoice);
		cNode.setModule(this);
		contentNodes_.put(new CAnonymousContentRule(), cNode);
		return (new EContentRefNode(cNode, occurs));
	    } else {
		return (choice);
	    }
	} else if (node instanceof EMixedNode) {
	    ENode[] children = node.getChildren();
	    if (children.length == 0) {
//		return (null);  -- no children mixed is simple PCDATA
		return (node);
	    } else {
		node.clear();
		for (int i = 0;i < children.length;i++) {
		    ENode child = children[i];
		    if (child instanceof EChoiceNode) {
			_addChildren(node, child);
		    } else {
			_addChild(node, child);
		    }
		}
/*
		for (int i = 0;i < children.length;i++) {
		    ENode child = children[i];
		    if (child instanceof EElementRefNode ||
			child instanceof EContentRefNode) {

			node.addChild(child);
		    } else {
			node.addChild(_adjustTree(child));
		    }
		}
*/
		return (node);
	    }
	} else if (node instanceof EExternalRefNode ||
		   node instanceof EExternalContentRefNode) {
	    return (node);
	} else {
	    throw (new InternalError());
	}
    }

    private ENode _adjustChoice(EChoiceNode choice) {
//System.out.println("adjustChoice(before): " + choice);
	ENode node = _adjustChoiceNone(choice);
	if (node == null) {
//System.out.println("adjustChoice(after): null");
	    return (null);
	}
	if (!(node instanceof EChoiceNode)) {
//System.out.println("adjustChoice(after): " + node);
	    return (node);
	}
	choice = (EChoiceNode)node;
	int occurs = choice.getOccurs();
	ENode[] children = choice.getChildren();
	if (children.length == 0) {
//System.out.println("adjustChoice(after): null");
	    return (null);
	} else if (children.length == 1 && occurs == OCCURS_ONE) {
//System.out.println("adjustChoice(after): " + _adjustTree(children[0]));
	    return (_adjustTree(children[0]));
	} else {
//System.out.println("adjustChoice(after): " + _adjustChoiceDuplicate(choice));
	    return (_adjustChoiceDuplicate(choice));
	}
    }

    private EChoiceNode _adjustChoiceChildren(
	EChoiceNode choice,
	ENode[] children
    ) {
//System.out.println("adjustChoiceChildren(before): " + choice);
	choice.clear();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
//System.out.println("adjustChoiceChildren(in): " + child);
	    if (child instanceof EElementRefNode) {
		EElementRefNode eRef = (EElementRefNode)child;
		if (UERule.canUnification(
		    choice.getOccurs(),
		    eRef.getOccurs())) {

		    eRef.setOccurs(OCCURS_ONE);
		}
		choice.addChild(child);
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode eRef = (EExternalRefNode)child;
		if (UERule.canUnification(
		    choice.getOccurs(),
		    eRef.getOccurs())) {

		    eRef.setOccurs(OCCURS_ONE);
		}
		choice.addChild(child);
	    } else if (child instanceof EExternalContentRefNode) {
		EContentRefNode eRef = (EContentRefNode)child;
		if (UERule.canUnification(
		    choice.getOccurs(),
		    eRef.getOccurs())) {

		    eRef.setOccurs(OCCURS_ONE);
		}
		choice.addChild(child);
	    } else if (child instanceof EChoiceNode) {
		EChoiceNode childChoice = (EChoiceNode)child;
		ENode adjustedNode = _adjustChoice(childChoice);
		if (adjustedNode instanceof EChoiceNode) {
		    EChoiceNode adjustedChoice = (EChoiceNode)adjustedNode;
		    if (UERule.canUnification(
			choice.getOccurs(),
			adjustedChoice.getOccurs())) {

			_addChildren(choice, adjustedChoice);
		    } else {
			_addChild(choice, adjustedChoice);
		    }
		} else {
		    _addChild(choice, adjustedNode);
		}
	    } else {
		_addChild(choice, child);
	    }
	}
	return (choice);
    }

    private ENode _adjustChoiceNone(EChoiceNode choice) {
//System.out.println("choice before: " + choice);
	ENode[] children = choice.getChildren();
	int baseChildCount = children.length;
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof ENoneNode) {
		if (choice.isConflict()) {
		    // do nothing
		} else {
		    return (child);
		}
	    } else if (child instanceof EElementRefNode) {
		EElementRefNode ref = (EElementRefNode)child;
		int occurs = ref.getOccurs();
		EElementNode eNode = ref.getElementNode();
		if (eNode.isNone()) {
		    if (choice.isConflict()) {
			// do nothing
		    } else if (occurs == OCCURS_ONE ||
			       occurs == OCCURS_ONEMORE) {
			return (new ENoneNode());
		    } else {
			// do nothing
		    }
		} else {
		    list.add(child);
		}
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode ref = (EContentRefNode)child;
		int occurs = ref.getOccurs();
		EContentNode cNode = ref.getContentNode();
		if (cNode.isNone()) {
		    if (choice.isConflict()) {
			// do nothing
		    } else if (occurs == OCCURS_ONE ||
			       occurs == OCCURS_ONEMORE) {
			return (new ENoneNode());
		    } else {
			// do nothing
		    }
		} else {
		    list.add(child);
		}
	    } else {
		list.add(child);
	    }
	}
	children = new ENode[list.size()];
	children = (ENode[])list.toArray(children);
	choice.clear();
	choice.addChildren(children);
//System.out.println("choice after: " + choice);
	return (choice);
/*
	choice = _adjustChoiceChildren(choice, children);
	int childrenChildCount = choice.getSize();
	if (baseChildCount != childrenChildCount) {
	    return (_adjustChoice(choice));
	} else {
	    return (choice);
	}
*/
    }

    private ENode _adjustChoiceDuplicate(EChoiceNode choice) {
//System.out.println("duplicate before: " + choice);
	ENode[] children = choice.getChildren();
	int baseChildCount = children.length;
	Set memo = new HashSet();
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementRefNode) {
		EElementRefNode ref = (EElementRefNode)child;
		String href = "#element" + ref.getElementNode().getId();
		if (!memo.contains(href)) {
		    list.add(child);
		    memo.add(href);
		}
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode ref = (EContentRefNode)child;
		String href = "#content" + ref.getContentNode().getId();
		if (!memo.contains(href)) {
		    list.add(child);
		    memo.add(href);
		}
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode ref = (EExternalRefNode)child;
		String href
		    = "#external" + ref.getNamespace() + "#" + ref.getLabel();
		if (!memo.contains(href)) {
		    list.add(child);
		    memo.add(href);
		}
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode ref = (EExternalContentRefNode)child;
		String href
		    = "#externalContent" + ref.getNamespace() +
		    "#" + ref.getLabel();
		if (!memo.contains(href)) {
		    list.add(child);
		    memo.add(href);
		}
	    } else {
		list.add(child);
	    }
	}
	children = new ENode[list.size()];
	children = (ENode[])list.toArray(children);
	int duplicatedChildCount = list.size();
	choice = _adjustChoiceChildren(choice, children);
	int childrenChildCount = choice.getSize();
	if (baseChildCount != duplicatedChildCount ||
	    duplicatedChildCount != childrenChildCount) {

//System.out.println("duplicate after(+): " + choice);
	    return (_adjustChoice(choice));
	} else {
//System.out.println("duplicate after: " + choice);
	    return (choice);
	}
    }

    private ENode _adjustTreeImportContentOverChoiceMixed(ENode node) {
	if (node instanceof EElementRefNode ||
	    node instanceof EContentRefNode ||
	    node instanceof EExternalRefNode ||
	    node instanceof EExternalContentRefNode) {

	    return (_deepCopy(node));
	} else {
	    return (_adjustTree(_deepCopy(node)));
	}
    }

    private ENode[] _adjustChildren(ENode node) {
	ENode[] children = node.getChildren();
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    list.add(_adjustTree(children[i]));
	}
	ENode[] result = new ENode[list.size()];
	return ((ENode[])list.toArray(result));
    }

    private void _addChildren(ENode parent, ENode children) {
	_addChildren(parent, children.getChildren());
    }

    private void _addChildren(ENode parent, ENode[] children) {
	for (int i = 0;i < children.length;i++) {
	    _addChild(parent, children[i]);
	}
    }

    private void _addChild(ENode parent, ENode child) {
	if (child instanceof EElementRefNode ||
	    (child instanceof EContentRefNode &&
	     parent instanceof EMixedNode) || // XXX : EMixed can't have EChoiceNode
	    child instanceof EExternalRefNode ||
	    child instanceof EExternalContentRefNode) {

	    parent.addChild(child);
	} else if (child instanceof ETempNode) { // XXX : move to ENode
	    parent.addChildren(child);
	} else {
	    parent.addChild(_adjustTree(child));
	}
    }

    // XXX : divide indispensable opration and optimized operaton
    private ENode _deepCopy(ENode node) {
	if (node instanceof EElementSlot) {
	    return (new EElementSlot((EElementSlot)node));
	} else if (node instanceof EAttributeSlot) {
	    return (new EAttributeSlot((EAttributeSlot)node));
	} else if (node instanceof EElementRefNode) {
	    return (new EElementRefNode((EElementRefNode)node));
	} else if (node instanceof EContentRefNode) {
	    return (new EContentRefNode((EContentRefNode)node));
	} else if (node instanceof EChoiceNode) {
	    return (new EChoiceNode((EChoiceNode)node)); // XXX : modify
	} else if (node instanceof ENoneNode) {
	    return (new ENoneNode());
	} else if (node instanceof EExternalRefNode) {
	    return (new EExternalRefNode((EExternalRefNode)node));
	} else if (node instanceof EExternalContentRefNode) {
	    return (
		new EExternalContentRefNode((EExternalContentRefNode)node)
	    );
	} else {
	    throw (new InternalError());
	}
    }

    private void _adjustNames() {
	// XXX : adjust it
    }

    public final void setProperty(String key, Object value) {
	properties_.put(key, value);
    }

    public final Object getProperty(String key) {
	return (properties_.get(key));
    }

    // utility

/*
    // XXX : need?
    private ENode _adjustElementSlot(ENode node, int occurs) {
	if (node instanceof EElementNode) {
	    EElementNode element = (EElementNode)node;
	    throw (new InternalError());
//	    return (
	} else if (node instanceof EContentNode) {
	    EContentNode content = (EContentNode)node;
	    throw (new InternalError());
//	    return (
	} else if (node instanceof EChoiceNode) {
	    return (node);
	} else {
	    throw (new InternalError());
	}
    }
*/

    // XXX
    private ENode _adjustByOccurs(ENode node, int occurs) {
	switch (occurs) {

	case OCCURS_ONE:
	    return (node);
	case OCCURS_ZEROONE:
	case OCCURS_ONEMORE:
	case OCCURS_ZEROMORE:
	    ENode newNode = new ESequenceNode(occurs);
	    newNode.addChild(node);
	    return (newNode);
	default:
	    throw (new InternalError());
	}
    }

    private boolean _isCollection(int occurs) {
	switch (occurs) {
	case OCCURS_ONE:
	case OCCURS_ZEROONE:
	    return (false);
	case OCCURS_ONEMORE:
	case OCCURS_ZEROMORE:
	    return (true);
	default:
	    throw (new InternalError());
	}
    }

    private String _adjustUniqueName(String name) {
	return (UString.uncapitalize(name));
    }

    //
    private static int seqNo__ = 0;

    private static int _getSequenceNo() {
	return (++seqNo__);
    }
}
