/*
 * The RelaxerOrg class library
 *  Copyright (C) 1997-2001  ASAMI, Tomoharu (asami@relaxer.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package org.relaxer.xml;

import java.util.*;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.SAXException;
import org.relaxer.xml.visitor.UDOMVisitor;
import com.AsamiOffice.io.UURL;
import com.AsamiOffice.io.UFile;

/**
 * UDOM
 *
 * @since   Jul.  1, 1998
 * @version Jul.  8, 2002
 * @author  ASAMI, Tomoharu (asami@relaxer.org)
 */
public final class UDOM {
    public static String makeTextDocument(Node node) {
	XMLMaker maker = new XMLMaker();
	UDOMVisitor.traverse(node, maker);
	return (maker.getText());
    }

    public static String makeTextDocumentVisual(Node node) {
	XMLMaker maker = new XMLMaker();
	maker.setVisual(true);
	UDOMVisitor.traverse(node, maker);
	return (maker.getText());
    }

    public static boolean isParsedEntity(EntityReference entityRef) {
	String name = entityRef.getNodeName();
	Document doc = entityRef.getOwnerDocument();
	DocumentType doctype = doc.getDoctype();
	if (doctype == null) {
	    return (false);
	}
	NamedNodeMap entities = doctype.getEntities();
	Entity entity = (Entity)entities.getNamedItem(name);
	if (entity == null) {
	    return (false);
	}
	return (entity.getNotationName() == null);
    }

    public static String escape(String string) {
	if (string.indexOf('<') == -1 &&
	    string.indexOf('>') == -1 &&
	    string.indexOf('&') == -1 &&
	    string.indexOf('"') == -1 &&
	    string.indexOf('\'') == -1) {

	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '<') {
		buffer.append("&lt;");
	    } else if (c == '>') {
		buffer.append("&gt;");
	    } else if (c == '&') {
		buffer.append("&amp;");
	    } else if (c == '"') {
		buffer.append("&quot;");
	    } else if (c == '\'') {
		buffer.append("&apos;");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    public static String escapeEntityQuot(String string) {
	if (string.indexOf('%') == -1 &&
	    string.indexOf('&') == -1 &&
	    string.indexOf('"') == -1) {

	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '%') {
		buffer.append("&---;");
	    } else if (c == '&') {
		buffer.append("&amp;");
	    } else if (c == '"') {
		buffer.append("&quot;");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    public static String escapeEntityApos(String string) {
	if (string.indexOf('%') == -1 &&
	    string.indexOf('&') == -1 &&
	    string.indexOf('\'') == -1) {

	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '%') {
		buffer.append("&#x25;");
	    } else if (c == '&') {
		buffer.append("&amp;");
	    } else if (c == '\'') {
		buffer.append("&apos;");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    public static String escapeAttrQuot(String string) {
	if (string.indexOf('<') == -1 &&
	    string.indexOf('&') == -1 &&
	    string.indexOf('"') == -1) {

	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '<') {
		buffer.append("&lt;");
	    } else if (c == '&') {
		buffer.append("&amp;");
	    } else if (c == '"') {
		buffer.append("&quot;");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    public static String escapeAttrApos(String string) {
	if (string.indexOf('<') == -1 &&
	    string.indexOf('&') == -1 &&
	    string.indexOf('\'') == -1) {

	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '<') {
		buffer.append("&lt;");
	    } else if (c == '&') {
		buffer.append("&amp;");
	    } else if (c == '\'') {
		buffer.append("&apos;");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    public static String escapeSystemQuot(String string) {
	if (string.indexOf('"') == -1) {
	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '"') {
		buffer.append("&quot;");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    public static String escapeSystemApos(String string) {
	if (string.indexOf('\'') == -1) {
	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '\'') {
		buffer.append("&apos;");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    public static String escapeCharData(String string) {
	if (string.indexOf('<') == -1 &&
	    string.indexOf('&') == -1 &&
	    string.indexOf("]]>") == -1) {

	    return (string);
	}
	StringBuffer buffer = new StringBuffer();
	int nBrackets = 0;
	int size = string.length();
	for (int i = 0;i < size;i++) {
	    char c = string.charAt(i);
	    if (c == '<') {
		buffer.append("&lt;");
	    } else if (c == '&') {
		buffer.append("&amp;");
	    } else if (c == '>' && nBrackets >= 2) {
		buffer.append("&gt;");
	    } else {
		buffer.append(c);
	    }
	    if (c == ']') {
		nBrackets++;
	    } else {
		nBrackets = 0;
	    }
	}
	return (new String(buffer));
    }

    public static Element[] getElements(Node element) {
	NodeList children = element.getChildNodes();
	List list = new ArrayList();
	int size = children.getLength();
	for (int i = 0;i < size;i++) {
	    Node child = children.item(i);
	    if (child instanceof Element) {
		list.add(child);
	    }
	}
	Element[] array = new Element[list.size()];
	return ((Element[])list.toArray(array));
    }

    public static Element[] getElements(Node element, String localName) {
	NodeList children = element.getChildNodes();
	List list = new ArrayList();
	int size = children.getLength();
// System.out.println("getElements - '" + localName + "' : " + size);
	for (int i = 0;i < size;i++) {
	    Node child = children.item(i);
	    if (child instanceof Element) {
		if (localName.equals(getLocalName((Element)child))) {
		    list.add(child);
		}
	    }
	}
	Element[] array = new Element[list.size()];
	return ((Element[])list.toArray(array));
    }

    public static Element getFirstElement(Node element, String localName) {
	NodeList children = element.getChildNodes();
	List list = new ArrayList();
	int size = children.getLength();
	for (int i = 0;i < size;i++) {
	    Node child = children.item(i);
	    if (child instanceof Element) {
		if (localName.equals(getLocalName((Element)child))) {
		    return ((Element)child);
		}
	    }
	}
	return (null);
    }

    public static boolean hasElement(Node element) {
	NodeList children = element.getChildNodes();
	List list = new ArrayList();
	int size = children.getLength();
	for (int i = 0;i < size;i++) {
	    Node child = children.item(i);
	    if (child instanceof Element) {
		return (true);
	    }
	}
	return (false);
    }

    public static String getTextValue(Element element) {
	StringBuffer buffer = new StringBuffer();
	getTextValue(element, buffer);
	return (new String(buffer));
    }

    public static void getTextValue(Element element, StringBuffer buffer) {
	NodeList children = element.getChildNodes();
	List list = new ArrayList();
	int size = children.getLength();
	for (int i = 0;i < size;i++) {
	    Node child = children.item(i);
	    if (child instanceof Element) {
		getTextValue((Element)child, buffer);
	    } else if (child instanceof Text) {
		buffer.append(child.getNodeValue());
	    }
	}
    }

    public static String getLocalName(Element element) {
	String localName = element.getLocalName();
	if (localName != null) {
	    return (localName);
	} else {
	    return (element.getTagName()); // XXX
	}
    }

    // I/O
    public static Document loadDocument(String uri)
	throws IOException,
	       ParserConfigurationException,
	       SAXException {

	URL url = UURL.getURLFromUri(uri);
	DocumentBuilderFactory factory
	    = DocumentBuilderFactory.newInstance();
	DocumentBuilder builder = factory.newDocumentBuilder();
	return (builder.parse(url.toExternalForm()));
    }

    public static Document loadDocumentNs(String uri)
	throws IOException,
	       ParserConfigurationException,
	       SAXException {

	URL url = UURL.getURLFromUri(uri);
	DocumentBuilderFactory factory
	    = DocumentBuilderFactory.newInstance();
	factory.setNamespaceAware(true);
	DocumentBuilder builder = factory.newDocumentBuilder();
	return (builder.parse(url.toExternalForm()));
    }

    public static void saveDocument(
	String uri,
	Document doc
    ) throws IOException {
	URL url = UURL.getURLFromUri(uri);
	if (!"file".equals(url.getProtocol())) {
	    throw (new IOException("bad uri = " + uri));
	}
	UFile.createFile(
	    new File(url.getFile()),
	    makeTextDocumentVisual(doc)
	);
    }
}
