/*
 * The RelaxerOrg class library
 *  Copyright (C) 1997-2001  ASAMI, Tomoharu (asami@zeomtech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package org.relaxer.xml;

import java.io.*;
import org.w3c.dom.*;
import org.relaxer.bus.RBusException;
import org.relaxer.xml.UDOM;
import org.relaxer.xml.visitor.IDOMVisitor;
import org.relaxer.xml.visitor.UDOMVisitor;

/**
 * XMLMaker
 *
 * @since   Oct. 27, 2000
 * @version Aug. 11, 2001
 * @author  ASAMI, Tomoharu (asami@zeomtech.com)
 */
public class XMLMaker implements IDOMVisitor {
    private Writer buffer_;
    private StringWriter stringWriter_;
    private String encoding_ = "UTF-8";
    private String publicId_ = null;
    private String systemId_ = null;
    private boolean dom2_ = false;
    private boolean expandEntityReference_ = false;
    private boolean emptyElementTag_ = false;
    private boolean visual_ = false;
    private int indent_ = 2;
    private int indentCount_ = 0;
    private String lineSeparator_ = System.getProperty("line.separator");

    public XMLMaker() {
	buffer_ = stringWriter_ = new StringWriter();
    }

    public XMLMaker(Writer writer) {
	buffer_ = writer;
	stringWriter_ = null;
    }

    public void setEncoding(String encoding) {
	encoding_ = encoding;
    }

    public void setPublicId(String id) {
	publicId_ = id;
    }

    public void setSystemId(String id) {
	systemId_ = id;
    }

    public void setDOM2(boolean dom2) {
	dom2_ = dom2;
    }

    public void setExpandEntityReference(boolean expand) {
	expandEntityReference_ = expand;
    }

    public void setEmptyElementTag(boolean empty) {
	emptyElementTag_ = empty;
    }

    public void setVisual(boolean visual) {
	visual_ = visual;
    }

    public void setLineSeparator(String separator) {
	lineSeparator_ = separator;
    }

    public String getText() {
	return (stringWriter_.toString());
    }

    public boolean enter(Element element) {
	try {
	    if (visual_) {
		_writeIndent();
	    }
	    String tag = element.getTagName();
	    buffer_.write("<");
	    buffer_.write(tag);
	    NamedNodeMap attrs = element.getAttributes();
	    int nAttrs = attrs.getLength();
	    for (int i = 0;i < nAttrs;i++) {
		Attr attr = (Attr)attrs.item(i);
		if (attr.getSpecified()) {
		    buffer_.write(' ');
		    enter(attr);
		    leave(attr);
		}
	    }
	    if (element.getFirstChild() == null && emptyElementTag_) {
		buffer_.write("/>");
		if (visual_) {
		    buffer_.write(lineSeparator_);
		}
		return (false);
	    } else {
		buffer_.write(">");
		if (_isIndent(element)) {
		    buffer_.write(lineSeparator_);
		    _indentUp();
		}
		return (true);
	    }
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    private boolean _isIndent(Element element) {
	if (!visual_) {
	    return (false);
	}
	if (_hasElement(element)) {
	    return (true);
	} else {
	    return (false);
	}
    }

    private boolean _hasElement(Element element) {
	NodeList children = element.getChildNodes();
	int size = children.getLength();
	for (int i = 0;i < size;i++) {
	    if (children.item(i) instanceof Element) {
		return (true);
	    }
	}
	return (false);
    }

    public void leave(Element element) {
	String tag = element.getTagName();
	try {
	    if (_isIndent(element)) {
		_indentDown();
		_writeIndent();
		buffer_.write("</" + tag + ">");
		buffer_.write(lineSeparator_);
	    } else if (visual_) {
		buffer_.write("</" + tag + ">");
		buffer_.write(lineSeparator_);
	    } else {
		buffer_.write("</" + tag + ">");
	    }
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public boolean enter(Attr attr) {
	try {
	    buffer_.write(attr.getName());
	    buffer_.write("=\"");
	    buffer_.write(UDOM.escapeAttrQuot(attr.getValue()));
	    buffer_.write('\"');
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
	return (true);
    }

    public void leave(Attr attr) {
    }

    public boolean enter(Text text) {
	try {
	    buffer_.write(UDOM.escapeCharData(text.getData()));
	} catch(IOException e) {
	    throw (new RBusException(e));
	}
	return (true);
    }

    public void leave(Text text) {
    }

    public boolean enter(CDATASection cdata) {
	try {
	    buffer_.write("<![CDATA[");
	    buffer_.write(cdata.getData());
	    buffer_.write("]]>");
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
	return (true);
    }

    public void leave(CDATASection cdata) {
    }

    public boolean enter(EntityReference entityRef) {
	try {
	    if (expandEntityReference_ && UDOM.isParsedEntity(entityRef)) {
		return (true);
	    } else {
		buffer_.write("&");
		buffer_.write(entityRef.getNodeName());
		buffer_.write(";");
		return (false);
	    }
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public void leave(EntityReference entityRef) {
    }

    public boolean enter(Entity entity) {
	try {
	    String name = entity.getNodeName();
	    String pid = entity.getPublicId();
	    String sid = entity.getSystemId();
	    String notation = entity.getNotationName();
	    if (sid != null) {
		UXMLMaker.makeUnparsedEntity(name, pid, sid, notation, buffer_);
	    } else {
		buffer_.write("<!ENTITY ");
		buffer_.write(name);
		buffer_.write(" \"");
		XMLMaker entityMaker = new XMLMaker();
		UDOMVisitor.traverseChildren(entity, entityMaker);
		buffer_.write(UDOM.escapeEntityQuot(entityMaker.getText()));
		buffer_.write("\"");
		buffer_.write(">");
	    }
	    return (false);
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public void leave(Entity entity) {
    }

    public boolean enter(ProcessingInstruction pi) {
	try {
	    UXMLMaker.makeProcessingInstruction(
		pi.getTarget(),
		pi.getData(),
		buffer_
	    );
	    return (true);
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public void leave(ProcessingInstruction pi) {
    }

    public boolean enter(Comment comment) {
	try {
	    buffer_.write("<!--");
	    buffer_.write(comment.getData());
	    buffer_.write("-->");
	    return (true);
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public void leave(Comment comment) {
    }

    public boolean enter(Document doc) {
	String name = doc.getDocumentElement().getTagName();
	try {
	    buffer_.write("<?xml version=\"1.0\" encoding=\"");
	    buffer_.write(encoding_);
	    buffer_.write("\" ?>");
	    buffer_.write(lineSeparator_);
	    if (systemId_ != null) {
		buffer_.write("<!DOCTYPE ");
		buffer_.write(name);
		if (publicId_ != null) {
		    buffer_.write(" PUBLIC \"");
		    buffer_.write(publicId_);
		    buffer_.write("\"");
		    buffer_.write(" \"");
		    buffer_.write(systemId_);
		    buffer_.write("\"");
		} else {
		    buffer_.write(" SYSTEM \"");
		    buffer_.write(systemId_);
		    buffer_.write("\"");
		}
		buffer_.write(">");
		buffer_.write(lineSeparator_);
	    }
	    return (true);
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public void leave(Document doc) {
    }

    public boolean enter(DocumentType doctype) {
	if (systemId_ != null) {
	    return (true);
	}
	try {
	    if (dom2_) {
		String name = doctype.getName();
		String publicId = doctype.getPublicId();
		String systemId = doctype.getSystemId();
		String internalSubset = doctype.getInternalSubset();
		buffer_.write("<!DOCTYPE ");
		buffer_.write(name);
		if (publicId != null) {
		    buffer_.write(" PUBLIC \"");
		    buffer_.write(publicId);
		    buffer_.write("\"");
		}
		if (systemId != null) {
		    buffer_.write(" SYSTEM \"");
		    buffer_.write(systemId);
		    buffer_.write("\"");
		}
		if (internalSubset != null) {
		    buffer_.write(" [");
		    buffer_.write(internalSubset);
		    buffer_.write("]");
		}
		buffer_.write(">");
		buffer_.write(lineSeparator_);
		return (true);
	    } else {
		String name = doctype.getName();
		NamedNodeMap entities = doctype.getEntities();
		NamedNodeMap notations = doctype.getNotations();
		buffer_.write("<!DOCTYPE ");
		buffer_.write(name);
		if (entities != null && entities.getLength() > 0 ||
		    notations != null && notations.getLength() > 0) {

		    buffer_.write(" [");
		    int nEntities = entities.getLength();
		    for (int i = 0;i < nEntities;i++) {
			XMLMaker entityMaker = new XMLMaker();
			UDOMVisitor.traverse(entities.item(i), entityMaker);
			buffer_.write(entityMaker.getText());
		    }
		    int nNotations = notations.getLength();
		    for (int i = 0;i < nNotations;i++) {
			enter((Notation)notations.item(i));
			leave((Notation)notations.item(i));
		    }
		    buffer_.write("]");
		}
		buffer_.write(">");
		buffer_.write(lineSeparator_);
	    }
	    return (true);
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public void leave(DocumentType doctype) {
    }

    public boolean enter(DocumentFragment docfrag) {
	return (true);
    }

    public void leave(DocumentFragment docfrag) {
    }

    public boolean enter(Notation notation) {
	try {
	    String name = notation.getNodeName();
	    String pid = notation.getPublicId();
	    String sid = notation.getSystemId();
	    UXMLMaker.makeNotation(name, pid, sid, buffer_);
	    return (true);
	} catch (IOException e) {
	    throw (new RBusException(e));
	}
    }

    public void leave(Notation notation) {
    }

    public boolean enter(Node node) {
	throw (new InternalError());
    }

    public void leave(Node node) {
	throw (new InternalError());
    }

    private void _indentUp() {
	indentCount_ += indent_;
    }

    private void _indentDown() {
	indentCount_ -= indent_;
    }

    private void _writeIndent() throws IOException {
	for (int i = 0;i < indentCount_;i++) {
	    buffer_.write(" ");
	}
    }
}
