/*
 * The JabaJaba class library
 *  Copyright (C) 1997-1999  ASAMI, Tomoharu (tasami@ibm.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.awt;

import java.util.Observer;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import jp.gr.java_conf.jaba2.awt.graph.*;
import jp.gr.java_conf.jaba2.awt.lwlayout.*;

/**
 * GraphEditPanel
 *
 * @since   Nov. 13, 1998
 * @version Feb. 26, 1999
 * @author  ASAMI, Tomoharu (tasami@ibm.net)
 */
public class GraphEditPanel extends JComponent implements Observer {
    public final static Integer BASE_LAYER = JLayeredPane.DEFAULT_LAYER;
    public final static Integer LINK_LAYER = new Integer(10);
    public final static Integer NODE_LAYER = new Integer(20);

    public final static int MOVE_MODE = 1;
    public final static int LINK_MODE = 2;

    protected GraphModel model_;
    protected MouseEventHandler handler_;
    protected int grid_ = 1;
    protected JLayeredPane layer_;
    protected BaseLayer baseLayer_;
    protected LinkLayer linkLayer_;
    protected NodeLayer nodeLayer_;

    public GraphEditPanel(GraphModel model) {
	setLayout(new GraphEditPanelLayout());
	layer_ = new JLayeredPane();
	layer_.setLayout(new LayerLayout()); // XXX
	baseLayer_ = new BaseLayer();
	linkLayer_ = new LinkLayer();
	nodeLayer_ = new NodeLayer();
	layer_.add(baseLayer_, BASE_LAYER);
	layer_.add(linkLayer_, LINK_LAYER);
	layer_.add(nodeLayer_, NODE_LAYER);
	add(layer_);
	model_ = model;
	model_.addObserver(this);
	handler_ = new MouseEventHandler();
	addMouseListener(handler_);
	addMouseMotionListener(handler_);
    }

    public void setMetaLayoutManagerOnNodeLayer(MetaLayoutManager layout) {
	nodeLayer_.setMetaLayoutManager(layout);
    }

    public void setMetaLayoutManagerOnBaseLayer(MetaLayoutManager layout) {
	baseLayer_.setMetaLayoutManager(layout);
    }

    public void addLightWeightLayoutManagerOnNodeLayer(
	String name,
	LightWeightLayoutManager layout,
	Object constraints
    ) {
	nodeLayer_.addLightWeightLayoutManager(name, layout, constraints);
    }

    public void addLightWeightLayoutManagerOnBaseLayer(
	String name,
	LightWeightLayoutManager layout,
	Object constraints
    ) {
	baseLayer_.addLightWeightLayoutManager(name, layout, constraints);
    }

    public void setMode(int mode) {
	handler_.setMode(mode);
    }

    public void setGrid(int grid) {
	grid_ = grid;
    }

    public int getGrid() {
	return (grid_);
    }

    public void paint(Graphics g) {
	super.paint(g);
	handler_.paint(g);
    }

    public void update(Observable o, Object arg) {
	baseLayer_.removeAll();
	linkLayer_.removeAll();
	nodeLayer_.removeAll();
	GraphNode[] nodes = model_.getNodes();
	for (int i = 0;i < nodes.length;i++) {
	    nodeLayer_.addGraphObject(nodes[i]);
	}
	GraphLink[] links = model_.getLinks();
	for (int i = 0;i < links.length;i++) {
	    linkLayer_.add(links[i]);
	}
	GraphDock[] docks = model_.getDocks();
	for (int i = 0;i < docks.length;i++) {
	    baseLayer_.addGraphObject(docks[i]);
	}
	validate();
	repaint();
    }

    protected Component _findGraphNode(int x, int y) {
	Component[] comps = nodeLayer_.getComponents();
	for (int i = 0;i < comps.length;i++) {
	    Component comp = comps[i];
	    if (comp instanceof GraphNode) {
		Point pos = comp.getLocation();
		if (comp.contains(x - pos.x, y - pos.y)) {
		    return (comp);
		}
	    }
	}
	return (null);
    }

    protected Component _findActionGraphObject(int x, int y) {
	Component[] comps = getComponents();
	for (int i = 0;i < comps.length;i++) {
	    Component comp = comps[i];
	    if (comp instanceof GraphObject) {
		GraphObject gobj = (GraphObject)comp;
		Point pos = comp.getLocation();
		int offx = x - pos.x;
		int offy = y - pos.y;
		if (comp.contains(offx, offy)) {
		    if (gobj.isValidPosition(offx, offy)) {
			if (gobj.isAction()) {
			    return (comp);
			}
			if (gobj.isRemovable()) {
			    return (comp);
			}
		    }
		}
	    }
	}
	return (null);
    }

    abstract class NodeHandler extends MouseAdapter
        implements MouseMotionListener {

	public void paint(Graphics g) {
	}

	public abstract void reset();
    }

    class MoveNodeHandler extends NodeHandler {
	Dimension offset_ = null;
	GraphNode node_ = null;
	GraphNode hilight_ = null;

	public void mousePressed(MouseEvent evt) {
	    mouseDragged(evt);
	}

	public void mouseReleased(MouseEvent evt) {
	    int x = evt.getX();
	    int y = evt.getY();
	    if (node_ != null) {
		Point pos = UAWT.calcGridedPosition(x, y, grid_);
		node_.setLocation(new Point(pos.x - offset_.width,
					    pos.y - offset_.height));
		node_.setSelected(false);
		repaint();
	    }
	    offset_ = null;
	    node_ = null;
	}

	public void mouseDragged(MouseEvent evt) {
	    int x = evt.getX();
	    int y = evt.getY();
	    if (node_ == null) {
		Component comp = _findGraphNode(x, y); // XXX
		if (comp == null) {
		    return;
		}
		if (!(comp instanceof GraphNode)) {
		    return;
		}
		Point point = comp.getLocation();
		offset_ = new Dimension(x - point.x, y - point.y);
		node_ = (GraphNode)comp;
		node_.setSelected(true);
		repaint();
	    } else {
//		Rectangle bounds = node_.getBounds();
		node_.setLocation(new Point(x - offset_.width,
					    y - offset_.height));
//		bounds.union(node_.getRepaintBounds());
//		repaint(bounds.x, bounds.y, bounds.width, bounds.height);
		repaint();	// XXX
	    }
	}

	public void mouseMoved(MouseEvent evt) {
	    boolean needRepaint = false;
	    try {
		int x = evt.getX();
		int y = evt.getY();
		if (hilight_ != null) {
		    hilight_.setHilight(false);
		    hilight_ = null;
		    needRepaint = true;
		}
		Component comp = _findGraphNode(x, y); // XXX
		if (comp == null) {
		    return;
		}
		if (!(comp instanceof GraphNode)) {
		    return;
		}
		GraphNode node = (GraphNode)comp;
		node.setHilight(true);
		needRepaint = true;
		hilight_ = node;
	    } finally {
		if (needRepaint) {
		    repaint();
		}
	    }
	}

	public void reset() {
	    if (hilight_ != null) {
		hilight_.setHilight(false);
	    }
	    offset_ = null;
	    node_ = null;
	    hilight_ = null;
	    repaint();
	}
    }

    class LinkNodeHandler extends NodeHandler {
	Point start_ = null;
	Point end_ = null;
	GraphNode base_ = null;
	GraphNode hilight_ = null;

	public void mousePressed(MouseEvent evt) {
	    mouseDragged(evt);
	}

	public void mouseReleased(MouseEvent evt) {
	    if (start_ != null) {
		if (base_ != null) {
		    Component comp = _findGraphNode(evt.getX(), evt.getY());
		    if (comp != null && comp instanceof GraphNode) {
			GraphNode node = (GraphNode)comp;
			if (node != base_) {
			    GraphLink link = model_.createLink(base_, node);
			    model_.sync();
			}
		    }
		    base_.setSelected(false);
		}
		reset();
		repaint();
	    }
	}

	public void mouseDragged(MouseEvent evt) {
	    boolean needRepaint = false;
	    int x = evt.getX();
	    int y = evt.getY();
	    try {
		needRepaint = _processHilight(x, y);
		if (base_ == null) {
		    Component comp = _findGraphNode(x, y);
		    if (comp == null) {
			return;
		    }
		    if (!(comp instanceof GraphNode)) {
			return;
		    }
		    base_ = (GraphNode)comp;
		    base_.setSelected(true);
		    start_ = new Point(x, y);
		    end_ = start_;
		    needRepaint = true;
		} else {
		    end_ = new Point(x, y);
		    needRepaint = true;
		}
	    } finally {
		if (needRepaint) {
		    repaint();
		}
	    }
	}

	public void mouseMoved(MouseEvent evt) {
	    if (_processHilight(evt.getX(), evt.getY())) {
		repaint();
	    }
	}

	protected boolean _processHilight(int x, int y) {
	    Component comp = _findGraphNode(x, y); // XXX
	    if (hilight_ == null) {
		if (comp == null) {
		    return (false);
		}
		if (!(comp instanceof GraphNode)) {
		    return (false);
		}
		hilight_ = (GraphNode)comp;
		hilight_.setHilight(true);
		return (true);
	    } else {
		if (comp == null) {
		    hilight_.setHilight(false);
		    hilight_ = null;
		    return (true);
		}
		if (comp == hilight_) {
		    return (false);
		}
		hilight_.setHilight(false);
		if (comp instanceof GraphNode) {
		    hilight_ = (GraphNode)comp;
		    hilight_.setHilight(true);
		} else {
		    hilight_ = null;
		}
		return (true);
	    }
	}

	// NodeHandler
	public void paint(Graphics g) {
	    if (start_ != null) {
		g.setColor(Color.black);
		g.drawLine(start_.x, start_.y, end_.x, end_.y);
	    }
	}

	// NodeHandler
	public void reset() {
	    if (base_ != null) {
		base_.setHilight(false);
	    }
	    if (hilight_ != null) {
		hilight_.setHilight(false);
	    }
	    start_ = null;
	    end_ = null;
	    base_ = null;
	    hilight_ = null;
	}
    }

    class MouseEventHandler extends MouseAdapter
        implements MouseMotionListener {

	protected MoveNodeHandler moveHandler_;
	protected LinkNodeHandler linkHandler_;
	protected NodeHandler handler_;

	public MouseEventHandler() {
	    moveHandler_ = new MoveNodeHandler();
	    linkHandler_ = new LinkNodeHandler();
	    handler_ = moveHandler_;
	}

	public void paint(Graphics g) {
	    handler_.paint(g);
	}

	public void setMode(int mode) {
	    handler_.reset();
	    switch (mode) {

	    case MOVE_MODE:
		handler_ = moveHandler_;
		break;
	    case LINK_MODE:
		handler_ = linkHandler_;
		break;
	    default:
		throw (new InternalError());
	    }
	}

	public void mouseClicked(MouseEvent evt) {
//System.out.println(evt); // XXX
	}

	public void mouseReleased(MouseEvent evt) {
	    handler_.mouseReleased(evt);
	}

	public void mouseDragged(MouseEvent evt) {
	    handler_.mouseDragged(evt);
	}

	public void mouseMoved(MouseEvent evt) {
	    handler_.mouseMoved(evt);
	}

	public void mousePressed(MouseEvent evt) {
	    final int x = evt.getX();
	    final int y = evt.getY();
//System.out.println(evt);
	    if (evt.isPopupTrigger()) {
		Component comp = _findActionGraphObject(x, y);
		if (comp != null) {
		    final GraphObject gobj = (GraphObject)comp;
		    Action[] actions = gobj.getActions();
		    if (actions == null) {
			actions = new Action[0];
		    }
		    if (actions.length > 0 || gobj.isRemovable()) {
			JPopupMenu popup = new JPopupMenu("node operation");
			for (int i = 0;i < actions.length;i++) {
			    Action action = actions[i];
			    JMenuItem item = new JMenuItem(
				(String)action.getValue(Action.NAME)
			    );
			    item.addActionListener(action);
			    popup.add(item);
			}
			if (gobj.isRemovable()) {
			    if (actions.length > 0) {
				popup.addSeparator();
			    }
			    JMenuItem item = new JMenuItem("remove");
			    item.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent evt) {
				    model_.remove(gobj);
				    model_.sync();
				}
			    });
			    popup.add(item);
			}
			popup.show(evt.getComponent(), x, y);
		    }
		}
	    } else {
		handler_.mousePressed(evt);
	    }
	}
    }

    static class LayerBase extends LightWeightLayoutPanel {
	public void addGraphObject(GraphObject gobject) {
	    if (!(gobject instanceof Component)) {
		throw (new IllegalArgumentException(gobject.toString()));
	    }
	    String layoutName = gobject.getLayoutName();
	    Object constraints = gobject.getLayoutConstraints();
	    if (layoutName != null) {
		add(layoutName, (Component)gobject, constraints);
	    } else {
		add((Component)gobject, constraints);
	    }
	}
    }

    static class BaseLayer extends LayerBase {
/*
	public void paint(Graphics g) {
	    g.setColor(Color.pink);
	    Dimension size = getSize();
	    g.fillRect(0, 0, size.width, size.height);
	    super.paint(g);
	}
*/
    }

    static class LinkLayer extends JComponent {
	public LinkLayer() {
	    setLayout(new LinkLayout());
	}

/*
	public void paint(Graphics g) {
	    g.setColor(Color.cyan);
	    Dimension size = getSize();
	    g.fillRect(0, 0, size.width, size.height);
	    super.paint(g);
	    System.out.println("layer : " + getComponentCount());
	}
*/

	static class LinkLayout extends AbstractLayoutManager {
	    public void layoutContainer(Container parent) {
System.out.println("LinkLayout - layoutContainer");
		Dimension size = parent.getSize();
		Component[] comps = parent.getComponents();
		for (int i = 0;i < comps.length;i++) {
		    comps[i].setBounds(0, 0, size.width, size.height);
		}
	    }

	    public Dimension preferredLayoutSize(Container parent) {
System.out.println("LinkLayout - preferredLayoutSize");
		return (new Dimension(0, 0));
	    }
	}
    }

    static class NodeLayer extends LayerBase {
/*
	public void paint(Graphics g) {
	    g.setColor(Color.magenta);
	    Dimension size = getSize();
	    g.fillRect(0, 0, size.width, size.height);
	    super.paint(g);
	}
*/
    }

    class GraphEditPanelLayout extends AbstractLayoutManager {
	public Dimension preferredLayoutSize(Container parent) {
	    Dimension size;
	    size = baseLayer_.getPreferredSize();
	    int width = size.width;
	    int height = size.height;
	    size = linkLayer_.getPreferredSize();
	    width = Math.max(width, size.width);
	    height = Math.max(height, size.height);
	    size = nodeLayer_.getPreferredSize();
	    width = Math.max(width, size.width);
	    height = Math.max(height, size.height);
	    return (new Dimension(width, height));
	}

	public void layoutContainer(Container parent) {
	    Dimension size = parent.getSize();
	    layer_.setBounds(0, 0, size.width, size.height);
//	    baseLayer_.setBounds(0, 0, size.width, size.height); // XXX
//	    linkLayer_.setBounds(0, 0, size.width, size.height);
//	    nodeLayer_.setBounds(0, 0, size.width, size.height);
	}
    }

    class LayerLayout extends AbstractLayoutManager {
	public Dimension preferredLayoutSize(Container parent) {
System.out.println("LayerLayout - preferredLayoutSize");
	    int width = 0;
	    int height = 0;
	    Component[] components = parent.getComponents();
	    for (int i = 0;i < components.length;i++) {
		Dimension size = components[i].getPreferredSize();
		width = Math.max(width, size.width);
		height = Math.max(height, size.height);
	    }
	    return (new Dimension(width, height));
	}

	public void layoutContainer(Container parent) {
System.out.println("LayerLayout - layoutContainer");
	    Dimension size = parent.getSize();
	    Component[] components = parent.getComponents();
	    for (int i = 0;i < components.length;i++) {
		components[i].setBounds(0, 0, size.width, size.height);
	    }
	}
    }

    // test driver
    public static void main(String[] args) throws Exception {
	new JTestFrame("GraphEditPanel test", new Sample());
    }
}
