/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2003  ASAMI, Tomoharu (asami@AsamiOffice.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.xml.relax.expanded;

import java.util.*;
import java.io.IOException;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.*;
import org.xml.sax.SAXException;
import jp.gr.java_conf.jaba2.util.UArray;
import jp.gr.java_conf.jaba2.util.ElementCounter;
import jp.gr.java_conf.jaba2.util.LooseList;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.datatype.IXMLDatatype;
import jp.gr.java_conf.jaba2.datatype.XMLFacet;
import jp.gr.java_conf.jaba2.xml.UElement;
import jp.gr.java_conf.jaba2.xml.UXML;
import jp.gr.java_conf.jaba2.xml.datatype.*;
import jp.gr.java_conf.jaba2.xml.relax.cooked.*;
import jp.gr.java_conf.jaba2.xml.relax.expanded.*;
import jp.gr.java_conf.jaba2.xml.relax.expanded.rText.*;

/**
 * OptimizerThread1
 *
 * @since   Dec. 14, 2000
 * @version Jan.  8, 2003
 * @author  ASAMI, Tomoharu (asami@AsamiOffice.com)
 */
public class OptimizerThread1 extends Thread implements IOptimizerThread {
    private ERuleNode rule_;
    private EModule module_;
    private OptimizerController controller_;
    private SyncCounter sync_ = null;
    private String command_ = null;
    private boolean fragile_ = false;

    public OptimizerThread1(
	ThreadGroup group,
	ERuleNode node,
	EModule module
    ) {
	super(group, "optimizer-" + node.getLabel());
	rule_ = node;
	module_ = module;
    }

    public void setup() {
	try {
	    rule_.lockWrite();
	} catch (InterruptedException e) {
	    throw (new InternalError());
	}
    }

    public synchronized void prepare(SyncCounter sync) {
//System.out.println("prepare");
	command_ = "prepare";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void cyclicResolve(SyncCounter sync) {
//System.out.println("cyclicResolve");
	command_ = "cyclicResolve";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void noneResolve(SyncCounter sync) {
//System.out.println("noneResolve");
	command_ = "noneResolve";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void optimize(SyncCounter sync) {
//System.out.println("optimize");
	command_ = "optimize";
	sync_ = sync;
	notifyAll();
    }

    public synchronized void naming(SyncCounter sync) {
//System.out.println("naming");
	command_ = "naming";
	sync_ = sync;
	notifyAll();
    }

    public void setController(OptimizerController controller) {
	controller_ = controller;
    }

    public synchronized void run() {
//System.out.println("start");
	for (;;) {
	    try {
		while (command_ == null) {
		    wait();
		}
//System.out.println("command = " + command_);
		if ("prepare".equals(command_)) {
		    _prepare();
		} else if ("cyclicResolve".equals(command_)) {
		    _cyclicResolve();
		} else if ("noneResolve".equals(command_)) {
		    _noneResolve();
		} else if ("optimize".equals(command_)) {
		    _optimize();
		} else if ("naming".equals(command_)) {
		    _naming();
		} else {
		    throw (new InternalError(command_));
		}
		command_ = null;
	    } catch (InterruptedException e) {
		throw (new InternalError());
	    }
	}
    }

    private void _prepare() throws InterruptedException {
	try {
	    _prepare(rule_);
	} finally {
	    _post();
	}
    }

    private void _prepare(ENode node) {
	node.setHometown(rule_);
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    _prepare(children[i]);
	}
    }

    private void _cyclicResolve() throws InterruptedException {
	try {
	    Set refs = new HashSet();
	    _cyclicResolve(rule_, refs);
	    rule_.setCyclicRefs(refs);
	} finally {
	    _post();
	}
    }

    private void _cyclicResolve(ENode node, Set refs) {
	if (node instanceof EElementNode) {
	    if (!refs.contains(node)) {
		refs.add(node);
		_cyclicResolveChildren(node, refs);
	    }
	} else if (node instanceof EContentNode) {
	    if (!refs.contains(node)) {
		refs.add(node);
		_cyclicResolveChildren(node, refs);
	    }
	} else if (node instanceof EAttributeNode) {
	    // XXX
	} else if (node instanceof EElementRefNode) {
	    EElementRefNode ref = (EElementRefNode)node;
	    _cyclicResolve(ref.getElementNode(), refs);
	} else if (node instanceof EContentRefNode) {
	    EContentRefNode ref = (EContentRefNode)node;
	    _cyclicResolve(ref.getContentNode(), refs);
	} else if (node instanceof EExternalRefNode) {
	    EExternalRefNode ref = (EExternalRefNode)node;
	    _cyclicResolve(ref.getElementNode(), refs);
	} else if (node instanceof EExternalContentRefNode) {
	    EExternalContentRefNode ref = (EExternalContentRefNode)node;
	    _cyclicResolve(ref.getContentNode(), refs);
	} else if (node instanceof ESequenceNode) {
	    _cyclicResolveChildren(node, refs);
	} else if (node instanceof EInterleaveNode) {
	    _cyclicResolveChildren(node, refs);
	} else if (node instanceof EChoiceNode) {
	    _cyclicResolveChildren(node, refs);
	} else if (node instanceof EMixedNode) {
	    _cyclicResolveChildren(node, refs);
	}
    }

    private void _cyclicResolveChildren(ENode node, Set refs) {
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    _cyclicResolve(children[i], refs);
	}
    }

    private void _noneResolve() throws InterruptedException {
	try {
	    if (rule_.isNone()) {
		return;
	    }
	    ENode[] children = rule_.getChildren();
	    for (int i = 0;i < children.length;i++) {
		if (children[i] instanceof ENoneNode) {
		    controller_.needRetry();
		    rule_.setNone(true);
		    return;
		} else {
		    boolean isNone = _noneResolve(children[i]);
		    if (isNone) {
			controller_.needRetry();
			rule_.setNone(true);
		    }
		}
	    }
	    rule_.setNone(false);
	} finally {
	    _post();
	}
    }

    private boolean _noneResolve(ENode node) {
	return (false);		// XXX
    }

    private void _optimize() throws InterruptedException {
	_adjustTree(rule_);
	_post();
    }

    private void _naming() throws InterruptedException {
	_post();
    }

    private void _post() throws InterruptedException {
	rule_.unlockWrite();
	sync_.done();
	sync_ = null;
    }

    private void _adjustTree(ERuleNode node) throws InterruptedException {
	int count = 0;
	for (;;) {
	    fragile_ = false;
	    if (node instanceof EElementNode) {
		_adjustTree((EElementNode)node);
	    } else if (node instanceof EContentNode) {
		_adjustTree((EContentNode)node);
	    } else if (node instanceof EAttributeNode) {
		_adjustTree((EAttributeNode)node);
	    } else {
		throw (new InternalError());
	    }
	    if (!fragile_) {
		break;
	    }
	    count++;
	    if (count > 2) {
System.out.println("readjust(" + count + " : " + node);
_printContentRefs(node, node.getLabel());
	    }
	    if (count > 5){	// XXX
		throw (new InternalError(node.toString()));
	    }
	} 
	node.setImmutable(true);
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    children[i].setImmutable(true);
	}
    }

    private void _adjustTree(EElementNode node) throws InterruptedException {
	node.setImmutable(false);
	ENode[] children = node.getChildren();
	node.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTree(node, children[i]);
	}
//System.out.println("before - " + node);
	if (node.size() == 0) {
	    if (node.getDatatype() == null) {
		// adjust for no attributes and elements
		node.setDatatype(new DEmptyString());
	    }
	} else {
	    _adjustElementHedge(node);
	}
//System.out.println("after - " + node);
	_adjustZeroableContentRefs(node);
	IXMLDatatype datatype = node.getDatatype();
	if (datatype != null) {
	    _adjustDatatype(node);
	}
    }

    private void _adjustElementHedge(EElementNode node)
	throws InterruptedException {

	ENode[] children = node.getChildren();
	node.clear();
	IXMLDatatype datatype = node.getDatatype();
	node.setDatatype(null);
	int elementSlotCount = 0;
	int contentCount = 0;
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		elementSlotCount++;
	    } else if (child instanceof EAttributeSlot) {
		node.addChild(child);
	    } else if (child instanceof EElementRefNode) {
		contentCount++;
	    } else if (child instanceof EContentRefNode) {
		contentCount++;
	    } else if (child instanceof EExternalRefNode) {
		contentCount++;
	    } else if (child instanceof EExternalContentRefNode) {
		contentCount++;
	    } else if (child instanceof ESequenceNode) {
		contentCount++;
	    } else if (child instanceof EInterleaveNode) {
		_assertInterleaveInElement((EInterleaveNode)child);
		contentCount++;
	    } else if (child instanceof EChoiceNode) {
		contentCount++;
	    } else if (child instanceof EMixedNode) {
		contentCount++;
	    } else if (child instanceof EAnyOtherElementNode) {
		contentCount++;
	    } else if (child instanceof EDataSlot) {
		if (node.getDatatype() != null) {
		    throw (new InternalError()); // XXX : syntax error
		}
		datatype = _makeDatatype((EDataSlot)child);
	    } else if (child instanceof EValueSlot) {
		if (node.getDatatype() != null) {
		    throw (new InternalError()); // XXX : syntax error
		}
		datatype = _makeDatatype((EValueSlot)child);
	    } else if (child instanceof EAttributeRefNode) {
		_adjustTreeInElementNode(node, (EAttributeRefNode)child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	if (elementSlotCount == 0 && contentCount == 0) {
	    node.setDatatype(datatype);
	} else if (datatype != null) { // XXX : in case of value
	    EMixedNode mixed = new EMixedNode();
	    for (int i = 0;i < children.length;i++) {
		ENode child = children[i];
		if (child instanceof EElementSlot) {
		    EElementSlot eSlot = (EElementSlot)child;
		    EElementNode eNode = eSlot.getElementNode();
		    if (eNode == null) {
			throw (new InternalError());
		    }
		    EModule eModule = eNode.getModule();
		    eModule.activateOptimizedElementRule(eNode);
		    mixed.addChild(
			new EElementRefNode(
			    eNode,
			    eSlot.getOccurs()
			)
		    );
		} else if (child instanceof EAttributeSlot) {
		    // do nothing
		} else if (child instanceof EElementRefNode) {
		    mixed.addChild(child);
		} else if (child instanceof EContentRefNode) {
		    mixed.addChild(child);
		} else if (child instanceof EExternalRefNode) {
		    mixed.addChild(child);
		} else if (child instanceof EExternalContentRefNode) {
		    mixed.addChild(child);
		} else if (child instanceof ESequenceNode) {
		    mixed.addChild(child);
		} else if (child instanceof EInterleaveNode) {
		    mixed.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    mixed.addChild(child);
		} else if (child instanceof EMixedNode) {
		    mixed.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    mixed.addChild(child);
		} else if (child instanceof EDataSlot) {
		    // do nothing
		} else if (child instanceof EValueSlot) {
		    // do nothing
		} else if (child instanceof EAttributeRefNode) {
		    // do nothing
		} else {
		    throw (new InternalError(child.toString()));
		}
	    }
	    node.addChild(mixed);
	} else {
	    for (int i = 0;i < children.length;i++) {
		ENode child = children[i];
		if (child instanceof EElementSlot) {
		    node.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    // do nothing
		} else if (child instanceof EElementRefNode) {
		    node.addChild(child);
		} else if (child instanceof EContentRefNode) {
		    node.addChild(child);
		} else if (child instanceof EExternalRefNode) {
		    node.addChild(child);
		} else if (child instanceof EExternalContentRefNode) {
		    node.addChild(child);
		} else if (child instanceof ESequenceNode) {
		    node.addChild(child);
		} else if (child instanceof EInterleaveNode) {
		    node.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    node.addChild(child);
		} else if (child instanceof EMixedNode) {
		    node.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    node.addChild(child);
		} else if (child instanceof EDataSlot) {
		    // do nothing
		} else if (child instanceof EValueSlot) {
		    // do nothing
		} else if (child instanceof EAttributeRefNode) {
		    // do nothing
		} else {
		    throw (new InternalError(child.toString()));
		}
	    }
	}
    }

    private void _adjustElementHedge0(EElementNode node)
	throws InterruptedException {

	ENode[] children = node.getChildren();
	node.clear();
	List elementSlots = new ArrayList();
	List elements = new ArrayList();
	List attrs = new ArrayList();
	IXMLDatatype datatype = null;
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		node.addChild(child);
	    } else if (child instanceof EAttributeSlot) {
		node.addChild(child);
	    } else if (child instanceof EElementRefNode) {
		node.addChild(child);
	    } else if (child instanceof EContentRefNode) {
		node.addChild(child);
	    } else if (child instanceof EExternalRefNode) {
		node.addChild(child);
	    } else if (child instanceof EExternalContentRefNode) {
		node.addChild(child);
	    } else if (child instanceof ESequenceNode) {
		node.addChild(child);
	    } else if (child instanceof EInterleaveNode) {
		_assertInterleaveInElement((EInterleaveNode)child);
		node.addChild(child);
	    } else if (child instanceof EChoiceNode) {
		node.addChild(child);
	    } else if (child instanceof EMixedNode) {
		node.addChild(child);
	    } else if (child instanceof EAnyOtherElementNode) {
		node.addChild(child);
	    } else if (child instanceof EDataSlot) {
		if (node.getDatatype() != null) {
		    throw (new InternalError()); // XXX : syntax error
		}
		node.setDatatype(_makeDatatype((EDataSlot)child));
	    } else if (child instanceof EValueSlot) {
		if (node.getDatatype() != null) {
		    throw (new InternalError()); // XXX : syntax error
		}
		node.setDatatype(_makeDatatype((EValueSlot)child));
	    } else if (child instanceof EAttributeRefNode) {
		_adjustTreeInElementNode(node, (EAttributeRefNode)child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
    }

/*
    private void _adjustElementHedgeMixed(EElementNode node)
	throws InterruptedException {

	ENode[] children = node.getChildren();
	if (children.length == 0) {
	    return;
	}
	node.setDatatype(null);
	node.clear();
	EMixedNode mixed = new EMixedNode();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		EElementSlot eSlot = (EElementSlot)child;
		EElementNode eNode = eSlot.getElementNode();
		if (eNode == null) {
		    throw (new InternalError());
		}
		EModule eModule = eNode.getModule();
		eModule.activateOptimizedElementRule(eNode);
		mixed.addChild(
		    new EElementRefNode(
			eNode,
			eSlot.getOccurs()
		    )
		);
	    } else if (child instanceof EAttributeSlot) {
		node.addChild(child);
	    } else if (child instanceof EElementRefNode) {
		mixed.addChild(child);
	    } else if (child instanceof EContentRefNode) {
		mixed.addChild(child);
	    } else if (child instanceof EExternalRefNode) {
		mixed.addChild(child);
	    } else if (child instanceof EExternalContentRefNode) {
		mixed.addChild(child);
	    } else if (child instanceof ESequenceNode) {
		mixed.addChild(child);
	    } else if (child instanceof EInterleaveNode) {
		_assertInterleaveInElement((EInterleaveNode)child);
		mixed.addChild(child);
	    } else if (child instanceof EChoiceNode) {
		mixed.addChild(child);
	    } else if (child instanceof EMixedNode) {
		mixed.addChild(child);
	    } else if (child instanceof EAnyOtherElementNode) {
		mixed.addChild(child);
	    } else if (child instanceof EDataSlot) {
		throw (new InternalError());
	    } else if (child instanceof EValueSlot) {
		throw (new InternalError());
	    } else if (child instanceof EAttributeRefNode) {
		throw (new InternalError());
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	node.addChild(mixed);
    }
*/

    private void _adjustTree(EContentNode node) throws InterruptedException {
	node.setImmutable(false);
	TextPattern textPattern = _makeTextPattern(node);
	if (textPattern != null) {
//System.out.println("before:" + textPattern.getPattern());
	    textPattern = _adjustTextPattern(textPattern);
//System.out.println("after:" + textPattern.getPattern());
	    node.clear();
	    node.setTextPattern(textPattern);
	    DComplex complex = new DComplex();
	    complex.setProperty(textPattern);
	    EDataSlot slot = new EDataSlot(complex);
	    node.addChild(slot);
	} else {
	    ENode[] children = node.getChildren();
	    node.clear();
	    for (int i = 0;i < children.length;i++) {
		_adjustTree(node, children[i]);
	    }
	    _adjustZeroableContentRefs(node);
	}
	_assertValidCNode(node);
    }

    // XXX : UTextPattern
    private TextPattern _makeTextPattern(EContentNode cNode)
	throws InterruptedException {

	PPattern pattern = new PPattern();
	ENode[] children = cNode.getChildren();
	IPTextPatternChoice[] result = _buildTextPattern(children);
	if (result == null) {
	    return (null);
	} else {
	    pattern.setTextPattern(result);
	    return (new TextPattern(pattern));
	}
    }

    public TextPattern _makeTextPattern(EContentRefNode cRef)
	throws InterruptedException {

	PPattern pattern = new PPattern();
	pattern.setTextPattern(_buildTextPattern(cRef));
	return (new TextPattern(pattern));
    }

    public TextPattern _makeTextPattern(EExternalContentRefNode cRef)
	throws InterruptedException {

	PPattern pattern = new PPattern();
	pattern.setTextPattern(_buildTextPattern(cRef));
	return (new TextPattern(pattern));
    }

    public TextPattern _makeTextPattern(ESequenceNode seq)
	throws InterruptedException {

	PPattern pattern = new PPattern();
	pattern.setTextPattern(_buildTextPattern(seq));
	return (new TextPattern(pattern));
    }

    public TextPattern _makeTextPattern(EChoiceNode choice)
	throws InterruptedException {

	PPattern pattern = new PPattern();
	pattern.setTextPattern(_buildTextPattern(choice));
	return (new TextPattern(pattern));
    }

    public TextPattern _makeTextPattern(EDataSlot data)
	throws InterruptedException {

	PPattern pattern = new PPattern();
	pattern.setTextPattern(_buildTextPattern(data));
	return (new TextPattern(pattern));
    }

    public TextPattern _makeTextPattern(EValueSlot vSlot)
	throws InterruptedException {

	PPattern pattern = new PPattern();
	pattern.setTextPattern(_buildTextPattern(vSlot));
	return (new TextPattern(pattern));
    }

    private TextPattern _makeTextPattern(String expr)
	throws InterruptedException {

	try {
	    PPattern pattern = new PPattern("<pattern>" + expr + "</pattern>");
	    return (new TextPattern(pattern));
	} catch (IOException e) {
	    throw (new InternalError());
	} catch (SAXException e) {
	    throw (new InternalError());
	} catch (ParserConfigurationException e) {
	    throw (new InternalError());
	}
    }

    private IPTextPatternChoice[] _buildTextPattern(
	ENode[] children
    ) throws InterruptedException {
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		return (null);
	    } else if (child instanceof EAttributeSlot) {
		return (null);
	    } else if (child instanceof EElementRefNode) {
		return (null);
	    } else if (child instanceof EContentRefNode) {
		IPTextPatternChoice[] result =
		    _buildTextPattern((EContentRefNode)child);
		if (result == null) {
		    return (null);
		}
		list.addAll(Arrays.asList(result));
	    } else if (child instanceof EExternalRefNode) {
		return (null);
	    } else if (child instanceof EExternalContentRefNode) {
		IPTextPatternChoice[] result =
		    _buildTextPattern((EExternalContentRefNode)child);
		if (result == null) {
		    return (null);
		}
		list.addAll(Arrays.asList(result));
	    } else if (child instanceof ESequenceNode) {
		IPTextPatternChoice[] result =
		    _buildTextPattern((ESequenceNode)child);
		if (result == null) {
		    return (null);
		}
		list.addAll(Arrays.asList(result));
	    } else if (child instanceof EInterleaveNode) {
		return (null);
	    } else if (child instanceof EChoiceNode) {
		IPTextPatternChoice[] result =
		    _buildTextPattern((EChoiceNode)child);
		if (result == null) {
		    return (null);
		}
		list.addAll(Arrays.asList(result));
	    } else if (child instanceof EMixedNode) {
		return (null);
	    } else if (child instanceof ENoneNode) {
		return (null);
	    } else if (child instanceof EAnyOtherAttributeNode) {
		return (null);
	    } else if (child instanceof EAnyOtherElementNode) {
		return (null);
	    } else if (child instanceof EDataSlot) {
		IPTextPatternChoice[] result =
		    _buildTextPattern((EDataSlot)child);
		if (result == null) {
		    return (null);
		}
		list.addAll(Arrays.asList(result));
	    } else if (child instanceof EValueSlot) {
		IPTextPatternChoice[] result =
		    _buildTextPattern((EValueSlot)child);
		if (result == null) {
		    return (null);
		}
		list.addAll(Arrays.asList(result));
	    } else if (child instanceof EAttributeRefNode) {
		return (null);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	IPTextPatternChoice[] result = new IPTextPatternChoice[list.size()];
	return ((IPTextPatternChoice[])list.toArray(result));
    }

    private IPTextPatternChoice[] _buildTextPattern(
	EContentRefNode cRef
    ) throws InterruptedException {
	EContentNode cNode = cRef.getContentNode();
	if (cNode.isNone()) {
	    return (null);
	}
	cNode.lockRead();
	try {
	    TextPattern targetPattern = cNode.getTextPattern();
	    if (targetPattern == null) {
		return (null);
	    }
	    return (targetPattern.getContents());
	} finally {
	    cNode.unlockRead();
	}
    }

    private IPTextPatternChoice[] _buildTextPattern(
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	EContentNode cNode = cRef.getContentNode();
	if (cNode.isNone()) {
	    return (null);
	}
	cNode.lockRead();
	try {
	    TextPattern targetPattern = cNode.getTextPattern();
	    if (targetPattern == null) {
		return (null);
	    }
	    return (targetPattern.getContents());
	} finally {
	    cNode.unlockRead();
	}
    }

    private IPTextPatternChoice[] _buildTextPattern(
	ESequenceNode seq
    ) throws InterruptedException {
	IPTextPatternChoice[] result = _buildTextPattern(seq.getChildren());
	if (result == null) {
	    return (null);
	}
	PList list = new PList();
	list.setTextPattern(result);
	return (new IPTextPatternChoice[] { list });
    }

    private IPTextPatternChoice[] _buildTextPattern(
	EChoiceNode choice
    ) throws InterruptedException {
	IPTextPatternChoice[] result = _buildTextPattern(choice.getChildren());
	if (result == null) {
	    return (null);
	}
	PChoice pChoice = new PChoice();
	pChoice.setTextPattern(result);
	return (new IPTextPatternChoice[] { pChoice });
    }

    private IPTextPatternChoice[] _buildTextPattern(
	EDataSlot data
    ) {
	PData pData = new PData();
	IXMLDatatype datatype = data.getDatatype();
	if (datatype instanceof DComplex) {
	    DComplex complex = (DComplex)datatype;
	    TextPattern textPattern = (TextPattern)complex.getProperty();
	    if (textPattern == null) {
		throw (new InternalError(complex.getExpr()));
	    }
	    PPattern pattern = textPattern.getPattern();
	    return (pattern.getTextPattern());
	} else {
	    pData.setType(datatype.getName());
	    return (new IPTextPatternChoice[] { pData });
	}
    }

    private IPTextPatternChoice[] _buildTextPattern(
	EValueSlot vSlot
    ) {
	PValue pValue = new PValue();
	pValue.setType(vSlot.getType());
	pValue.setContent(vSlot.getText());
	return (new IPTextPatternChoice[] { pValue });
    }

    private TextPattern _adjustTextPattern(TextPattern textPattern)
	throws InterruptedException {

	PPattern pattern = textPattern.getPattern();
	return (new TextPattern(_adjustTextPattern(pattern)));
    }

    private PPattern _adjustTextPattern(PPattern pattern)
	throws InterruptedException {

	IPTextPatternChoice[] result
	    = _adjustTextPattern(pattern.getTextPattern());
	pattern.setTextPattern(result);
	return (pattern);
    }

    private IPTextPatternChoice[] _adjustTextPattern(PText pattern)
	throws InterruptedException {

	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(PData pattern)
	throws InterruptedException {

	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(PValue pattern)
	throws InterruptedException {

	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(PGroup pattern)
	throws InterruptedException {

	IPTextPatternChoice[] result
	    = _adjustTextPattern(pattern.getTextPattern());
	pattern.setTextPattern(result);
	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(PList pattern)
	throws InterruptedException {

	IPTextPatternChoice[] result
	    = _adjustTextPattern(pattern.getTextPattern());
	pattern.setTextPattern(result);
	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(PChoice pattern)
	throws InterruptedException {

	IPTextPatternChoice[] result
	    = _adjustTextPattern(pattern.getTextPattern());
	pattern.setTextPattern(result);
	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(POptional pattern)
	throws InterruptedException {

	IPTextPatternChoice[] result
	    = _adjustTextPattern(pattern.getTextPattern());
	pattern.setTextPattern(result);
	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(POneOrMore pattern)
	throws InterruptedException {

	IPTextPatternChoice[] result
	    = _adjustTextPattern(pattern.getTextPattern());
	pattern.setTextPattern(result);
	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(PZeroOrMore pattern)
	throws InterruptedException {

	IPTextPatternChoice[] result
	    = _adjustTextPattern(pattern.getTextPattern());
	pattern.setTextPattern(result);
	return (new IPTextPatternChoice[] { pattern });
    }

    private IPTextPatternChoice[] _adjustTextPattern(PRef pattern)
	throws InterruptedException {

	    String name = pattern.getName();
	EContentNode[] cNodes = module_.resolveContentLabel(name);
	if (cNodes.length != 1) {
	    throw (new InternalError());
	}
	EContentNode cNode = cNodes[0];
	cNode.lockRead();
	try {
	    TextPattern targetPattern = cNode.getTextPattern();
	    if (targetPattern == null) {
		throw (new InternalError(cNode.toString()));
	    }
	    return (targetPattern.getContents());
	} finally {
	    cNode.unlockRead();
	}
    }

    private IPTextPatternChoice[] _adjustTextPattern(
	IPTextPatternChoice[] children
    ) throws InterruptedException {
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    IPTextPatternChoice child = children[i];
	    if (child instanceof PText) {
		list.addAll(Arrays.asList(_adjustTextPattern((PText)child)));
	    } else if (child instanceof PData) {
		list.addAll(Arrays.asList(_adjustTextPattern((PData)child)));
	    } else if (child instanceof PValue) {
		list.addAll(Arrays.asList(_adjustTextPattern((PValue)child)));
	    } else if (child instanceof PGroup) {
		list.addAll(Arrays.asList(_adjustTextPattern((PGroup)child)));
	    } else if (child instanceof PList) {
		list.addAll(Arrays.asList(_adjustTextPattern((PList)child)));
	    } else if (child instanceof PChoice) {
		list.addAll(Arrays.asList(_adjustTextPattern((PChoice)child)));
	    } else if (child instanceof POptional) {
		list.addAll(Arrays.asList(_adjustTextPattern((POptional)child)));
	    } else if (child instanceof POneOrMore) {
		list.addAll(Arrays.asList(_adjustTextPattern((POneOrMore)child)));
	    } else if (child instanceof PZeroOrMore) {
		list.addAll(Arrays.asList(_adjustTextPattern((PZeroOrMore)child)));
	    } else if (child instanceof PRef) {
		list.addAll(Arrays.asList(_adjustTextPattern((PRef)child)));
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	IPTextPatternChoice[] result = new IPTextPatternChoice[list.size()];
	return ((IPTextPatternChoice[])list.toArray(result));
    }

    //
    private void _adjustTree(EAttributeNode node)
	throws InterruptedException {

//System.out.println("zzz:" + node);
	_adjustDatatype(node);
    }

/*
    private void _adjustTree(ERuleNode parent, ENode child)
	throws InterruptedException {

	if (parent instanceof EElementNode) {
	    _adjustTree((EElementNode)parent, child);
	} else if (parent instanceof EContentNode) {
	    _adjustTree((EContentNode)parent, child);
	} else {
	    throw (new InternalError(child + " on " + parent));
	}
    }
*/

    private void _adjustTree(EElementNode parent, ENode child)
	throws InterruptedException {

	if (child instanceof EElementSlot) {
	    _adjustTreeInElementNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInElementNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInElementNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInElementNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInElementNode(parent, (EExternalRefNode)child);
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInElementNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInElementNode(parent, (ESequenceNode)child);
	} else if (child instanceof EInterleaveNode) {
	    _adjustTreeInElementNode(parent, (EInterleaveNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInElementNode(parent, (EChoiceNode)child);
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInElementNode(parent, (EMixedNode)child);
	} else if (child instanceof ENoneNode) {
	    parent.addChild(child);
	    parent.setNone(true);
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EDataSlot) {
	    parent.setDatatype(_makeDatatype((EDataSlot)child));
	} else if (child instanceof EValueSlot) {
	    parent.setDatatype(_makeDatatype((EValueSlot)child));
	} else if (child instanceof EAttributeRefNode) {
	    _adjustTreeInElementNode(parent, (EAttributeRefNode)child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTree(EContentNode parent, ENode child)
	throws InterruptedException {

	if (child instanceof EElementSlot) {
	    _adjustTreeInContentNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInContentNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInContentNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInContentNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInContentNode(parent, (EExternalRefNode)child);
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInContentNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInContentNode(parent, (ESequenceNode)child);
	} else if (child instanceof EInterleaveNode) {
	    _adjustTreeInContentNode(parent, (EInterleaveNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInContentNode(parent, (EChoiceNode)child);
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInContentNode(parent, (EMixedNode)child);
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EDataSlot) {
	    parent.addChild(child);
	} else if (child instanceof EValueSlot) {
	    parent.addChild(child);
	} else if (child instanceof EAttributeRefNode) {
	    _adjustTreeInContentNode(parent, (EAttributeRefNode)child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    // EElementNode context
    private void _adjustTreeInElementNode(
	EElementNode parent,
	EElementSlot eSlot
    ) throws InterruptedException {
	String typeName = eSlot.getDatatype().getName();
	if ("none".equals(typeName)) {
	    parent.addNoneElementSlot(eSlot);
	} else if (UERule.isNameClass(eSlot)) {
	    EElementRefNode eRef = _makeImplicitElementNode( // XXX
		_makeUniqueName("anyNameElement"),
		eSlot.getOccurs(),
		eSlot.getDatatype()
	    );
	    parent.addChild(eRef);
	} else {
	    _adjustDatatype(eSlot);
	    parent.addChild(eSlot);
	}
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EAttributeSlot aSlot
    ) throws InterruptedException {
	String typeName = aSlot.getDatatype().getName();
	if ("none".equals(typeName)) {
	    parent.addNoneAttributeSlot(aSlot);
	} else if (UERule.isNameClass(aSlot)) {
	    _adjustDatatype(aSlot);
	    EAttributeRefNode aRef = _makeImplicitAttributeNode(
		aSlot,
		"anyNameAttribute"
	    );
	    parent.addChild(aRef);
	} else {
	    _adjustDatatype(aSlot);
	    parent.addChild(aSlot);
	}
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EElementRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	_adjustElementRefNodeInRule(parent, eRef, eNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	EContentNode cNode = cRef.getContentNode();
	_adjustContentRefNodeInRule(parent, cRef, cNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EExternalRefNode eRef
    ) throws InterruptedException {
	EElementNode eNode = eRef.getElementNode();
	int occurs = eRef.getOccurs();
	_adjustExternalElementRefNodeInRule(parent, eRef, eNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	EContentNode cNode = cRef.getContentNode();
	_adjustExternalContentRefNodeInRule(parent, cRef, cNode, occurs);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	int occurs = sequence.getOccurs();
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) {
	    return;
	}
	String labelName;
	if (debug__) {
	    labelName = _makeUniqueName("sequencea");
	} else {
	    labelName = _makeUniqueName("sequence");
	}
	EContentRefNode cRef = _makeImplicitContentNode(
	    labelName,
	    occurs,
	    children
	);
	parent.addChild(cRef);
	fragile_ = true;
    }

    private void _adjustTreeInElementNode0(
	EElementNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	int occurs = sequence.getOccurs();
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) {
	    return;
	}
	if (sequence.getOccurs() == OCCURS_ONE) {
	    for (int i = 0;i < children.length;i++) {
		ENode child = children[i];
		if (child instanceof EElementSlot) {
		    _adjustTreeInElementNode(parent, (EElementSlot)child);
		} else if (child instanceof EAttributeSlot) {
		    _adjustTreeInElementNode(parent, (EAttributeSlot)child);
		} else if (child instanceof EElementRefNode) {
		    _adjustTreeInElementNode(parent, (EElementRefNode)child);
		} else if (child instanceof EContentRefNode) {
		    _adjustTreeInElementNode(parent, (EContentRefNode)child);
		} else if (child instanceof EExternalRefNode) {
		    _adjustTreeInElementNode(parent, (EExternalRefNode)child);
		} else if (child instanceof EExternalContentRefNode) {
		    _adjustTreeInElementNode(
			parent,
			(EExternalContentRefNode)child
		    );
		} else if (child instanceof ESequenceNode) {
		    _adjustTreeInElementNode(
			parent,
			(ESequenceNode)child
		    );
		} else if (child instanceof EInterleaveNode) {
		    _adjustTreeInElementNode(parent, (EInterleaveNode)child);
		} else if (child instanceof EChoiceNode) { // XXX : data choice
		    _adjustTreeInElementNode(parent, (EChoiceNode)child);
		} else if (child instanceof EMixedNode) {
		    _adjustTreeInElementNode(parent, (EMixedNode)child);
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else if (child instanceof EDataSlot) {
		    parent.setDatatype(_makeDatatype((EDataSlot)child));
		} else if (child instanceof EValueSlot) {
		    parent.setDatatype(_makeDatatype((EValueSlot)child));
		} else if (child instanceof EAttributeRefNode) {
		    _adjustTreeInElementNode(parent, (EAttributeRefNode)child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    }
	} else {
	    if (children.length == 1) {
		_makeOptimizedEntryForSequenceOne(
		    parent,
		    children[0],
		    occurs
		);
	    } else {
		String labelName;
		if (debug__) {
		    labelName = _makeUniqueName("sequencea");
		} else {
		    labelName = _makeUniqueName("sequence");
		}
		EContentRefNode cRef = _makeImplicitContentNode(
		    labelName,
		    occurs,
		    children
		);
		parent.addChild(cRef);
	    }
	}
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EInterleaveNode interleave
    ) throws InterruptedException {
	_adjustInterleaveNode(parent, interleave);
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	if (_isDataChoice(choice)) {
	    IXMLDatatype datatype = _makeDatatype((EChoiceNode)choice);
	    parent.setDatatype(datatype);
	} else {
	    _adjustChoiceNode(parent, choice);
	}
    }
/*
    private boolean _isTextChoice(EChoiceNode node) {
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (!(child instanceof EDataSlot ||
		  child instanceof EValueSlot)) {
		return (false);
	    }
	}
	return (true);
    }
*/

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	_adjustMixedNode(parent, mixed);
/*
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
*/
    }

    private void _adjustTreeInElementNode(
	EElementNode parent,
	EAttributeRefNode aRef
    ) throws InterruptedException {
	EAttributeNode aNode = aRef.getAttributeNode();
	int occurs = aRef.getOccurs();
	_adjustAttributeRefNodeInRule(parent, aRef, aNode, occurs);
    }

    // EContentNode context
    private void _adjustTreeInContentNode(
	EContentNode parent,
	EElementSlot eSlot
    ) throws InterruptedException {
	String typeName = eSlot.getDatatype().getName();
	if ("none".equals(typeName)) {
	    parent.addNoneElementSlot(eSlot);
	} else if (UERule.isNameClass(eSlot)) {
	    EElementRefNode eRef = _makeImplicitElementNode(
		eSlot,
		"anyNameElement"
	    );
	    parent.addChild(eRef);
	} else {
	    _adjustDatatype(eSlot);
	    parent.addChild(eSlot);
	}
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EAttributeSlot aSlot
    ) throws InterruptedException {
	String typeName = aSlot.getDatatype().getName();
	if ("none".equals(typeName)) {
	    parent.addNoneAttributeSlot(aSlot);
	} else if (UERule.isNameClass(aSlot)) {
	    EAttributeRefNode aRef = _makeImplicitAttributeNode(
		aSlot,
		"anyNameAttribute"
	    );
	    parent.addChild(aRef);
	} else {
	    _adjustDatatype(aSlot);
	    parent.addChild(aSlot);
	}
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EElementRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	// XXX : handle elementSlot more precisely
	_adjustElementRefNodeInRule(parent, eRef, eNode, occurs); // XXX
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	EContentNode cNode = cRef.getContentNode();
	// XXX : handle elementSlot more precisely
	_adjustContentRefNodeInRule(parent, cRef, cNode, occurs); // XXX
	_assertValidCNode(parent);
/*
	if (cRef.getOccurs() == OCCURS_ONE) {
	    EContentNode cNode = cRef.getContentNode();
	    cNode.lockRead();
	    try {
		parent.addChildren(
		    _deepCopyChildren(cNode.getChildren())
		);
	    } finally {
		cNode.unlockRead();
	    }
	} else {
	    parent.addChild(cRef);
	}
*/
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EExternalRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	_adjustExternalElementRefNodeInRule(parent, eRef, eNode, occurs);
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	EContentNode cNode = cRef.getContentNode();
	_adjustExternalContentRefNodeInRule(parent, cRef, cNode, occurs);
/*
	if (cRef.getOccurs() == OCCURS_ONE) {
	    EContentNode cNode = cRef.getContentNode();
	    cNode.lockRead();
	    try {
		parent.addChildren(
		    _deepCopyChildren(cNode.getChildren())
		);
	    } finally {
		cNode.unlockRead();
	    }
	} else {
	    parent.addChild(cRef);
	}
*/
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	if (sequence.getOccurs() == OCCURS_ONE) {
	    parent.addChildren(sequence);
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("sequenceb");
	    } else {
		labelName = _makeUniqueName("sequence");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		sequence.getOccurs(),
		sequence.getChildren()
	    );
	    parent.addChild(cRef);
	    fragile_ = true;
	}
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EInterleaveNode interleave
    ) throws InterruptedException {
	_adjustInterleaveNode(parent, interleave);
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_adjustChoiceNode(parent, choice);
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	_adjustMixedNode(parent, mixed);
/*
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	parent.addChild(mixed);
*/
    }

    private void _adjustTreeInContentNode(
	EContentNode parent,
	EAttributeRefNode aRef
    ) throws InterruptedException {
	int occurs = aRef.getOccurs();
	EAttributeNode aNode = aRef.getAttributeNode();
	aNode.lockRead();
	try {
	    _adjustAttributeRefNodeInRule(parent, aRef, aNode, occurs);
	} finally {
	    aNode.unlockRead();
	}
    }

    // ESequence context
    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    _adjustTreeInSequenceNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInSequenceNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInSequenceNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInSequenceNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInSequenceNode(parent, (EExternalRefNode)child);    
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInSequenceNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInSequenceNode(parent, (ESequenceNode)child);
	} else if (child instanceof EInterleaveNode) {
	    _adjustTreeInSequenceNode(parent, (EInterleaveNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInSequenceNode(parent, (EChoiceNode)child);    
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInSequenceNode(parent, (EMixedNode)child);    
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EDataSlot) {
	    parent.addChild(child);
	} else if (child instanceof EValueSlot) {
	    parent.addChild(child);
	} else if (child instanceof EAttributeRefNode) {
	    _adjustTreeInSequenceNode(parent, (EAttributeRefNode)child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EElementSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    // RELAX NG
    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EAttributeSlot aSlot
    ) throws InterruptedException {
	_adjustDatatype(aSlot);
	parent.addChild(aSlot);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EElementRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EContentRefNode child
    ) throws InterruptedException {
	parent.addChild(child);	// XXX
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EExternalRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EExternalContentRefNode child
    ) throws InterruptedException {
	parent.addChild(child);	// XXX
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) { // XXX : unify sequence child series
	    return;
	} else if (children.length == 1) {
	    int occurs = sequence.getOccurs();
	    ENode child = children[0];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		parent.addChild(child);
	    } else if (child instanceof EElementRefNode) {
		EElementRefNode oldRef = (EElementRefNode)child;
		EElementRefNode newRef = new EElementRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode oldRef = (EContentRefNode)child;
		EContentRefNode newRef = new EContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			_getContentRefNodeOccurs(oldRef),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode oldRef = (EExternalRefNode)child;
		EExternalRefNode newRef = new EExternalRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode oldRef
		    = (EExternalContentRefNode)child;
		EExternalContentRefNode newRef
		    = new EExternalContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			_getContentRefNodeOccurs(oldRef),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof ESequenceNode) {
		ESequenceNode childSequence = (ESequenceNode)child;
		sequence.setOccurs(
		    UERule.unifyOccurs(
			childSequence.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childSequence);
	    } else if (child instanceof EChoiceNode) {
		EChoiceNode childChoice = (EChoiceNode)child;
		childChoice.setOccurs(
		    UERule.unifyOccurs(
			childChoice.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childChoice);
	    } else if (child instanceof EMixedNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherAttributeNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherElementNode) {
		parent.addChild(child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("sequencec");
	    } else {
		labelName = _makeUniqueName("sequence");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		sequence.getOccurs(),
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EInterleaveNode interleave
    ) throws InterruptedException {
	_adjustInterleaveNode(parent, interleave);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_adjustChoiceNode(parent, choice);
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	_adjustMixedNode(parent, mixed);
/*
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	parent.addChild(mixed);
*/
    }

    private void _adjustTreeInSequenceNode(
	ESequenceNode parent,
	EAttributeRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    // EInterleave context
    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    _adjustTreeInInterleaveNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInInterleaveNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInInterleaveNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInInterleaveNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInInterleaveNode(parent, (EExternalRefNode)child);    
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInInterleaveNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInInterleaveNode(parent, (ESequenceNode)child);
	} else if (child instanceof EInterleaveNode) {
	    _adjustTreeInInterleaveNode(parent, (EInterleaveNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInInterleaveNode(parent, (EChoiceNode)child);    
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInInterleaveNode(parent, (EMixedNode)child);    
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EDataSlot) {
	    parent.addChild(child);
	} else if (child instanceof EValueSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAttributeRefNode) {
	    _adjustTreeInInterleaveNode(parent, (EAttributeRefNode)child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EElementSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EAttributeSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EElementRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	_adjustElementRefNodeInInterleave(parent, eRef, eNode, occurs);
//	_adjustElementRefNodeInRule(parent, eRef, eNode, occurs);
//	parent.addChild(eRef);
    }

    private void _adjustElementRefNodeInInterleave(
	EInterleaveNode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    return;
	}
	if (eRef instanceof EElementRefNode) {
	    EElementRefNode oldRef = (EElementRefNode)eRef;
	    EElementRefNode newRef = new EElementRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (eRef instanceof EExternalRefNode) {
	    EExternalRefNode oldRef = (EExternalRefNode)eRef;
	    EExternalRefNode newRef = new EExternalRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else {
	    throw (new InternalError(eRef.toString()));
	}
    }

    // XXX : same logic with _adjustElementRefNodeInChoice
    private void _adjustElementRefNodeInInterleave0(
	EInterleaveNode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    return;
	}
	if (UERule.canUnification(parent.getOccurs(), occurs)) {
	    parent.addChild(eRef);
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("refa");
	    } else {
		labelName = _makeUniqueName("ref");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		OCCURS_ONE,
		new ENode[] { eRef }
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	EContentNode cNode = cRef.getContentNode();
	_adjustContentRefNodeInInterleave(parent, cRef, cNode, occurs);
//	_adjustContentRefNodeInRule(parent, cRef, cNode, occurs);
//	parent.addChild(child);	// XXX
    }

    private void _adjustContentRefNodeInInterleave(
	EInterleaveNode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    return;
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		    return;
	    }
	    _assertValidCRefNode(cRef);
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (children.length == 1) {
		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    parent.addChild(cRef);
		} else if (child instanceof EAttributeSlot) {
		    EAttributeRefNode aRef
			= _makeImplicitAttributeNode(
			    (EAttributeSlot)child,
			    "attribute" // XXX
			);
		    aRef.setOccurs(
			UERule.unifyOccurs(aRef.getOccurs(), occurs)
		    );
		    parent.addChild(aRef);
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    _getContentRefNodeOccurs(oldRef),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		    fragile_ = true;
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    _getContentRefNodeOccurs(oldRef),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		    fragile_ = true;
		} else if (child instanceof ESequenceNode) {
		    ESequenceNode childSequence = (ESequenceNode)child;
		    if (childSequence.getSize() <= 1) {
			throw (new InternalError());
		    }
		    parent.addChild(cRef);
		} else if (child instanceof EInterleaveNode) {
		    EInterleaveNode childInterleave = (EInterleaveNode)child;
		    if (childInterleave.getSize() <= 1) {
			throw (new InternalError());
		    }
		    parent.addChild(cRef);
		} else if (child instanceof EChoiceNode) {
		    parent.addChild(cRef); // XXX : more optimize
		} else if (child instanceof EMixedNode) {
		    parent.addChild(cRef); // XXX : more optimize
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(cRef);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(cRef);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else if (occurs == OCCURS_ONE ||
		       (occurs == OCCURS_ZEROONE &&
			_isZeroableContent(cNode))) {
		children = cNode.getChildren();
		for (int i = 0;i < children.length;i++) {
		    ENode child = _deepCopy(children[i]);
		    if (child instanceof EElementSlot) {
			String labelName;
			if (debug__) {
			    labelName = _makeUniqueName("refb");
			} else {
			    labelName = _makeUniqueName("ref");
			}
			parent.addChild(
			    _makeImplicitElementNode(
				(EElementSlot)child,
				labelName
			    )
			);
		    } else if (child instanceof EAttributeSlot) {
			EAttributeRefNode aRef
			    = _makeImplicitAttributeNode(
				(EAttributeSlot)child,
				"attribute" // XXX
			    );
			parent.addChild(aRef);
		    } else if (child instanceof EElementRefNode) {
			parent.addChild(child);
		    } else if (child instanceof EContentRefNode) {
/*
			EExternalRefNode oldRef
			    = (EExternalRefNode)child;
			int oldOccurs = oldRef.getOccurs();
			// XXX: check zeroable hedge
			if (oldOccurs == OCCURS_ONE) {
			    EExternalRefNode newRef
				= new EExternalRefNode(oldRef);
			    newRef.setOccurs(
				UERule.unifyOccurs(
				    OCCURS_ZEROONE,
				    occurs
				)
			    );
			    parent.addChild(newRef);
			} else if (oldOccurs == OCCURS_ONEMORE) {
			    EExternalRefNode newRef
				= new EExternalRefNode(oldRef);
			    newRef.setOccurs(
				UERule.unifyOccurs(
				    OCCURS_ZEROMORE,
				    occurs
				)
			    );
			    parent.addChild(newRef);
			} else {
			    parent.addChild(child);
			}
*/
			parent.addChild(child);
			fragile_ = true;
		    } else if (child instanceof EExternalRefNode) {
			parent.addChild(child);
		    } else if (child instanceof EExternalContentRefNode) {
			parent.addChild(child);
			fragile_ = true;
		    } else if (child instanceof ESequenceNode) {
			throw (new InternalError());
		    } else if (child instanceof EInterleaveNode) {
			parent.addChildren(child);
		    } else if (child instanceof EChoiceNode) {
			EChoiceNode choice = (EChoiceNode)child;
			EContentRefNode newCRef
			    = _makeImplicitContentNodeChoiceInInterleave(
				choice.getOccurs(),
				_deepCopyChildren(choice.getChildren())
			    );
			parent.addChild(newCRef);
		    } else if (child instanceof EMixedNode) {
			throw (new InternalError());
		    } else if (child instanceof ENoneNode) {
			return;
		    } else if (child instanceof EAnyOtherAttributeNode) {
			parent.addChild(child);
		    } else if (child instanceof EAnyOtherElementNode) {
			parent.addChild(child);
		    } else if (child instanceof EDataSlot) {
			parent.addChild(child);
		    } else if (child instanceof EValueSlot) {
			parent.addChild(child);
		    } else if (child instanceof EAttributeRefNode) {
			parent.addChild(child);
		    } else {
			throw (new InternalError(child.toString()));
		    }
		}
	    } else {
		if (cRef instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)cRef;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (cRef instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)cRef;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else {
		    throw (new InternalError(cRef.toString()));
		}
/*
		if (UERule.canUnification(parent.getOccurs(), occurs)) {
		    parent.addChild(cRef);
		} else {
		    EContentRefNode newRef = _makeImplicitContentNode(
			_makeUniqueName("refd"),
			OCCURS_ONE,
			new ENode[] { cRef }
		    );
		    parent.addChild(newRef);
		}
*/
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EExternalRefNode eRef
    ) throws InterruptedException {
	int occurs = eRef.getOccurs();
	EElementNode eNode = eRef.getElementNode();
	_adjustExternalElementRefNodeInRule(parent, eRef, eNode, occurs);
//	parent.addChild(child);
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	EContentNode cNode = cRef.getContentNode();
	_adjustExternalContentRefNodeInRule(parent, cRef, cNode, occurs);
//	parent.addChild(child);	// XXX
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) {
	    return;
	}
	String labelName;
	if (debug__) {
	    labelName = _makeUniqueName("sequenced");
	} else {
	    labelName = _makeUniqueName("sequence");
	}
	EContentRefNode cRef = _makeImplicitContentNode(
	    labelName,
	    sequence.getOccurs(),
	    children
	);
	parent.addChild(cRef);
	fragile_ = true;
    }

    private void _adjustTreeInInterleaveNode0(
	EInterleaveNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) { // XXX : unify sequence child series
	    return;
	} else if (children.length == 1) {
	    int occurs = sequence.getOccurs();
	    ENode child = children[0];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		EAttributeSlot aSlot = (EAttributeSlot)child;
		EAttributeRefNode aRef = _makeImplicitAttributeNode(
		    aSlot,
		    "attribute",
		    UERule.unifyOccurs(
			occurs,
			aSlot.isRequired()?OCCURS_ONE:OCCURS_ZEROONE
		    )
		);
		parent.addChild(aRef);
	    } else if (child instanceof EElementRefNode) {
		EElementRefNode oldRef = (EElementRefNode)child;
		EElementRefNode newRef = new EElementRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode oldRef = (EContentRefNode)child;
		EContentRefNode newRef = new EContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			_getContentRefNodeOccurs(oldRef),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode oldRef = (EExternalRefNode)child;
		EExternalRefNode newRef = new EExternalRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode oldRef
		    = (EExternalContentRefNode)child;
		EExternalContentRefNode newRef
		    = new EExternalContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			_getContentRefNodeOccurs(oldRef),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof ESequenceNode) {
		ESequenceNode childSequence = (ESequenceNode)child;
		sequence.setOccurs(
		    UERule.unifyOccurs(
			childSequence.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childSequence);
	    } else if (child instanceof EChoiceNode) {
		EChoiceNode childChoice = (EChoiceNode)child;
		childChoice.setOccurs(
		    UERule.unifyOccurs(
			childChoice.getOccurs(),
			occurs
		    )
		);
		parent.addChild(childChoice);
	    } else if (child instanceof EMixedNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherAttributeNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherElementNode) {
		parent.addChild(child);
	    } else if (child instanceof EAttributeRefNode) {
		EAttributeRefNode oldRef = (EAttributeRefNode)child;
		EAttributeRefNode newRef = new EAttributeRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("sequenced");
	    } else {
		labelName = _makeUniqueName("sequence");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		sequence.getOccurs(),
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EInterleaveNode interleave
    ) throws InterruptedException {
	_adjustInterleaveNode(parent, interleave);
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_adjustChoiceNode(choice);
	ENode[] children = choice.getChildren();
	if (children.length == 0) {
	    return;
	} else if (children.length == 1) {
	    _makeOptimizedEntryForInterleaveOne(
		parent,
		children[0],
		choice.getOccurs()
	    );
	} else {
	    EContentRefNode cRef = _makeImplicitContentNodeChoiceInInterleave(
		choice.getOccurs(),
		children
	    );
/*
	    EContentRefNode cRef = _makeImplicitContentNode(
		_makeUniqueName("choice"),
		choice.getOccurs(),
		children
	    );
*/
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	_adjustMixedNode(parent, mixed);
/*
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	parent.addChild(mixed);
*/
    }

    private void _adjustTreeInInterleaveNode(
	EInterleaveNode parent,
	EAttributeRefNode aRef
    ) throws InterruptedException {
	parent.addChild(aRef);
    }

    // EChoiceNode context
    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    _adjustTreeInChoiceNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInChoiceNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInChoiceNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInChoiceNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInChoiceNode(parent, (EExternalRefNode)child);    
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInChoiceNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInChoiceNode(parent, (ESequenceNode)child);
	} else if (child instanceof EInterleaveNode) {
	    _adjustTreeInChoiceNode(parent, (EInterleaveNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInChoiceNode(parent, (EChoiceNode)child);    
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInChoiceNode(parent, (EMixedNode)child);
	} else if (child instanceof ENoneNode) {
	    // do nothing
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EDataSlot) {
	    parent.addChild(child);
	} else if (child instanceof EValueSlot) {
	    parent.addChild(child);
	} else if (child instanceof EAttributeRefNode) {
	    _adjustTreeInChoiceNode(parent, (EAttributeRefNode)child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EElementSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EAttributeSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EElementRefNode eRef
    ) throws InterruptedException {
	_adjustElementRefNodeInChoice(
	    parent,
	    eRef,
	    eRef.getElementNode(),
	    eRef.getOccurs()
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	_adjustContentRefNodeInChoice(
	    parent,
	    cRef,
	    cRef.getContentNode(),
	    occurs
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EExternalRefNode eRef
    ) throws InterruptedException {
	_adjustElementRefNodeInChoice(
	    parent,
	    eRef,
	    eRef.getElementNode(),
	    eRef.getOccurs()
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	int occurs = _getContentRefNodeOccurs(cRef);
	_adjustContentRefNodeInChoice(
	    parent,
	    cRef,
	    cRef.getContentNode(),
	    occurs
	);
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) { // XXX : unify Sequence Series
	    return;
	} else if (children.length == 1) {
	    int occurs = sequence.getOccurs();
	    ENode child = children[0];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		EAttributeSlot aSlot = (EAttributeSlot)child;
		EAttributeRefNode aRef = _makeImplicitAttributeNode(
		    aSlot,
		    "attribute",
		    UERule.unifyOccurs(
			occurs,
			aSlot.isRequired()?OCCURS_ONE:OCCURS_ZEROONE
		    )
		);
		parent.addChild(aRef);
	    } else if (child instanceof EElementRefNode) {
		EElementRefNode oldRef = (EElementRefNode)child;
		EElementRefNode newRef = new EElementRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode oldRef = (EContentRefNode)child;
		EContentRefNode newRef = new EContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			_getContentRefNodeOccurs(oldRef),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode oldRef = (EExternalRefNode)child;
		EExternalRefNode newRef = new EExternalRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			oldRef.getOccurs(),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode oldRef
		    = (EExternalContentRefNode)child;
		EExternalContentRefNode newRef
		    = new EExternalContentRefNode(oldRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			_getContentRefNodeOccurs(oldRef),
			occurs
		    )
		);
		parent.addChild(newRef);
	    } else if (child instanceof ESequenceNode) {
		String labelName;
		if (debug__) {
		    labelName = _makeUniqueName("sequencee");
		} else {
		    labelName = _makeUniqueName("sequence");
		}
		EContentRefNode cRef = _makeImplicitContentNode(
		    labelName,
		    UERule.unifyOccurs(
			((ESequenceNode)child).getOccurs(),
			occurs
		    ),
		    children
		);
		parent.addChild(cRef);
	    } else if (child instanceof EInterleaveNode) {
		EContentRefNode cRef = _makeImplicitContentNode(
		    _makeUniqueName("interleave"),
		    UERule.unifyOccurs(
			((EInterleaveNode)child).getOccurs(),
			occurs
		    ),
		    children
		);
		parent.addChild(cRef);
	    } else if (child instanceof EChoiceNode) {
		EContentRefNode cRef = _makeImplicitContentNode(
		    _makeUniqueName("choice"),
		    UERule.unifyOccurs(
			((EChoiceNode)child).getOccurs(),
			occurs
		    ),
		    children
		);
		parent.addChild(cRef);
	    } else if (child instanceof EMixedNode) {
		EContentRefNode cRef = _makeImplicitContentNode(
		    _makeUniqueName("mixed"),
		    occurs,
		    children
		);
		parent.addChild(cRef);
	    } else if (child instanceof EAnyOtherAttributeNode) {
		parent.addChild(child);
	    } else if (child instanceof EAnyOtherElementNode) {
		parent.addChild(child);
	    } else if (child instanceof EDataSlot) {
		parent.addChild(child);
	    } else if (child instanceof EValueSlot) {
		parent.addChild(child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("sequencef");
	    } else {
		labelName = _makeUniqueName("sequence");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		sequence.getOccurs(),
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EInterleaveNode interleave
    ) throws InterruptedException {
	throw (new UnsupportedOperationException());
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_adjustChoiceNode(choice);
//System.out.println("-- " + parent); // XXX : choice/choice
	ENode[] children = choice.getChildren();
	choice.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInChoiceNode(choice, children[i]);
	}
	children = choice.getChildren();
	choice.clear();
	if (children.length == 0) {
	    return;
	} else if (children.length == 1) {
	    _makeOptimizedEntryForChoiceChoiceOne(parent, children[0]);
	} else {
	    if (UERule.canUnification(
		parent.getOccurs(), choice.getOccurs())
	    ) {
		parent.addChildren(children);
	    } else {
/*
		EChoiceNode newChoice = new EChoiceNode();
		newChoice.setOccurs(choice.getOccurs());
		newChoice.setHometown(rule_); // XXX
		newChoice.addChildren(children);
		EContentRefNode cRef = _makeImplicitContentNode(
		    _makeUniqueName("choice"),
		    OCCURS_ONE,
		    new ENode[] { newChoice }
		);
*/
		EContentRefNode cRef = _makeImplicitContentNodeChoice(
		    choice.getOccurs(),
		    children
		);
		parent.addChild(cRef);
	    }
	}
    }

    private void _makeOptimizedEntryForChoiceChoiceOne(
	EChoiceNode parent,
	ENode child
    ) throws InterruptedException {
	int choiceOccurs = parent.getOccurs();
	if (child instanceof EElementSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    EElementRefNode childRef = (EElementRefNode)child;
	    int childOccurs = childRef.getOccurs();
	    childRef.setOccurs(
		UERule.unifyOccurs(choiceOccurs, childOccurs)
	    );
	    parent.addChild(childRef);
	} else if (child instanceof EContentRefNode) {
	    _adjustContentRefNodeForExpandInChoice(
		parent,
		(EContentRefNode)child,
		choiceOccurs
	    );
	} else if (child instanceof EExternalRefNode) {
	    EExternalRefNode childRef = (EExternalRefNode)child;
	    int childOccurs = childRef.getOccurs();
	    childRef.setOccurs(
		UERule.unifyOccurs(choiceOccurs, childOccurs)
	    );
	    parent.addChild(childRef);
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustExternalContentRefNodeForExpandInChoice(
		parent,
		(EExternalContentRefNode)child,
		choiceOccurs
	    );
	} else if (child instanceof ESequenceNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EInterleaveNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EChoiceNode) {
	    throw (new InternalError(child.toString()));
/*
	    EChoiceNode childChoice = (EChoiceNode)child;
	    int childOccurs = childChoice.getOccurs();
	    int newOccurs = UERule.unifyOccurs(choiceOccurs, childOccurs);
	    choice.setOccurs(newOccurs);
	    choice.addChildren(child);
	    parent.addChild(choice);
*/
	} else if (child instanceof EMixedNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof ENoneNode) {
	    // do nothing
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	_adjustMixedNode(mixed);
	if (mixed.size() == 0) {
	    parent.addChild(new EDataSlot(_makeDatatype("string")));
	} else {
	    EContentRefNode cRef = _makeImplicitContentNode(
		_makeUniqueName("mixed"),
		OCCURS_ONE,
		new ENode[] { mixed }
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInChoiceNode(
	EChoiceNode parent,
	EAttributeRefNode aRef
    ) throws InterruptedException {
	int occurs = aRef.getOccurs();
	if (UERule.canUnification(parent.getOccurs(), occurs)) {
	    parent.addChild(aRef);
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("refe");
	    } else {
		labelName = _makeUniqueName("ref");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		OCCURS_ONE,
		new ENode[] { aRef }
	    );
	    parent.addChild(cRef);
	}
    }

    // EMixed context
    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    _adjustTreeInMixedNode(parent, (EElementSlot)child);
	} else if (child instanceof EAttributeSlot) {
	    _adjustTreeInMixedNode(parent, (EAttributeSlot)child);
	} else if (child instanceof EElementRefNode) {
	    _adjustTreeInMixedNode(parent, (EElementRefNode)child);
	} else if (child instanceof EContentRefNode) {
	    _adjustTreeInMixedNode(parent, (EContentRefNode)child);
	} else if (child instanceof EExternalRefNode) {
	    _adjustTreeInMixedNode(parent, (EExternalRefNode)child);    
	} else if (child instanceof EExternalContentRefNode) {
	    _adjustTreeInMixedNode(parent, (EExternalContentRefNode)child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInMixedNode(parent, (ESequenceNode)child);
	} else if (child instanceof EInterleaveNode) {
	    _adjustTreeInMixedNode(parent, (EInterleaveNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInMixedNode(parent, (EChoiceNode)child);    
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInMixedNode(parent, (EMixedNode)child);    
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EAttributeRefNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EElementSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EAttributeSlot child
    ) throws InterruptedException {
	throw (new InternalError());
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EElementRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EContentRefNode cRef
    ) throws InterruptedException {
	_adjustContentRefNodeInMixed(
	    parent,
	    cRef,
	    cRef.getContentNode()
	);
//	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EExternalRefNode child
    ) throws InterruptedException {
	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EExternalContentRefNode cRef
    ) throws InterruptedException {
	_adjustContentRefNodeInMixed(
	    parent,
	    cRef,
	    cRef.getContentNode()
	);
//	parent.addChild(child);
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	ESequenceNode sequence
    ) throws InterruptedException {
	ENode[] children = sequence.getChildren();
	sequence.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInSequenceNode(sequence, children[i]);
	}
	children = sequence.getChildren();
	if (children.length == 0) {
	    return;
	} else if (children.length == 1) {
	    _makeOptimizedEntryForMixedSequenceOne(parent, children[0]);
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("sequenceg");
	    } else {
		labelName = _makeUniqueName("sequence");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		sequence.getOccurs(),
		children
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EInterleaveNode interleave
    ) throws InterruptedException {
	throw (new UnsupportedOperationException());
    }

    private void _makeOptimizedEntryForMixedSequenceOne(
	EMixedNode parent,
	ENode child
    ) {
	if (child instanceof EElementSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof ESequenceNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EInterleaveNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EChoiceNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EMixedNode) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	ENode[] children = choice.getChildren();
	choice.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInChoiceNode(choice, children[i]);
	}
	children = choice.getChildren();
	if (children.length == 0) {
	    return;
	} else {
	    _makeOptimizedEntryForMixed(parent, children);
	    parent.setOccurs(choice.getOccurs());
	}
    }

    private void _adjustTreeInMixedNode(
	EMixedNode parent,
	EMixedNode mixed
    ) throws InterruptedException {
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	children = mixed.getChildren();
	if (children.length == 0) {
	    return;
	} else {
	    _makeOptimizedEntryForMixed(parent, children);
	    parent.setOccurs(mixed.getOccurs());
	}
    }

    private void _makeOptimizedEntryForMixed(
	EMixedNode parent,
	ENode[] children
    ) throws InterruptedException {
	for (int i = 0;i < children.length;i++) {
	    _makeOptimizedEntryForMixed(parent, children[i]);
	}
    }

    private void _makeOptimizedEntryForMixed(
	EMixedNode parent,
	ENode child
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalRefNode) {
	    parent.addChild(child);
	} else if (child instanceof EExternalContentRefNode) {
	    parent.addChild(child);
	} else if (child instanceof ESequenceNode) {
	    _adjustTreeInMixedNode(parent, (ESequenceNode)child);
	} else if (child instanceof EInterleaveNode) {
	    _adjustTreeInMixedNode(parent, (EInterleaveNode)child);
	} else if (child instanceof EChoiceNode) {
	    _adjustTreeInMixedNode(parent, (EChoiceNode)child);
	} else if (child instanceof EMixedNode) {
	    _adjustTreeInMixedNode(parent, (EMixedNode)child);
	} else if (child instanceof ENoneNode) {
	    // do nothing
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EInterleaveNode) {
	    parent.addChild(child);
	} else if (child instanceof EDataSlot) {
	    // do nothing
//	    parent.addChild(child);
	} else if (child instanceof EValueSlot) {
	    // do nothing
//	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    // common procudures
    private void _adjustElementRefNodeInRule(
	ENode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    parent.addChild(new ENoneNode());
	    return;
	}
	if (eNode.isCyclic(rule_)) {
	    parent.addChild(eRef);
	    return;
	}
	eNode.lockRead();
	try {
	    if (eNode.isNone()) {
		parent.addChild(new ENoneNode());
		return;
	    }
	    if (eNode.getSize() > 0) {
		parent.addChild(eRef);
		return;
	    }
	    if (UERule.isNameClass(eNode)) {
		parent.addChild(eRef);
		return;
	    }
	    IXMLDatatype datatype = eNode.getDatatype();
	    if (datatype == null) {
		datatype = new DEmptyString();
	    }
	    if (datatype instanceof DEmptyString) {
		parent.addChild(eRef);
		return;
	    }
	    EElementSlot eSlot = new EElementSlot(
		eNode.getElementName(),
		datatype,
		occurs
	    );
	    eSlot.setAppinfo(eNode.getAppinfo());
	    eSlot.setBase(eNode.getBase());
	    eSlot.setModule(module_);
	    eSlot.setElementNode(eNode);
	    String datatypeName = datatype.getName();
	    if ("none".equals(datatypeName)) {
		parent.addNoneElementSlot(eSlot);
	    } else {
		_adjustDatatype(eSlot);
		parent.addChild(eSlot);
	    }
/*
	    if (eNode instanceof EElementSlotNode) {
		EElementSlotNode esNode = (EElementSlotNode)eNode;
		IXMLDatatype datatype = esNode.getDatatype();
		String datatypeName = datatype.getName();
		if (esNode.getSize() > 0) {
		    parent.addChild(eRef);
		    return;
		}
		if ("emptyString".equals(datatypeName)) {
		    parent.addChild(eRef);
		    return;
		}
		EElementSlot eSlot = new EElementSlot(
		    esNode.getElementName(),
		    datatype,
		    occurs
		);
		eSlot.setAppinfo(esNode.getAppinfo());
		eSlot.setBase(esNode.getBase());
		eSlot.setModule(module_);
		if ("none".equals(datatypeName)) {
		    parent.addNoneElementSlot(eSlot);
		} else {
		    parent.addChild(eSlot);
		}
	    } else {
		parent.addChild(eRef);
	    }
*/
	} finally {
	    eNode.unlockRead();
	}
    }

    private void _adjustExternalElementRefNodeInRule(
	ENode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    parent.addChild(new ENoneNode());
	    return;
	}
	if (eNode.isCyclic(rule_)) {
	    parent.addChild(eRef);
	    return;
	}
	eNode.lockRead();
	try {
	    if (eNode.isNone()) {
		parent.addChild(new ENoneNode());
		return;
	    }
	    parent.addChild(eRef);
	} finally {
	    eNode.unlockRead();
	}
    }

    private void _adjustContentRefNodeInRule(
	ENode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
	    }
	    return;
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		    parent.addChild(new ENoneNode());
		}
		return;
	    }
	    _assertValidCRefNode(cRef);
/*
	    if (cNode.isCyclic(rule_)) { // XXX
		parent.addChild(cRef);
		return;
	    }
*/
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (occurs == OCCURS_ONE ||
		       (occurs == OCCURS_ZEROONE &&
			_isZeroableContent(cNode))) {
		// differ against _adjustContentRefNode
		children = cNode.getChildren();
		for (int i = 0;i < children.length;i++) {
		    ENode child = _deepCopy(children[i]);
		    if (child instanceof EElementSlot) {
			_adjustDatatype((EElementSlot)child);
			parent.addChild(child);
		    } else if (child instanceof EAttributeSlot) {
			_adjustDatatype((EAttributeSlot)child);
			parent.addChild(child);
		    } else if (child instanceof EElementRefNode) {
			parent.addChild(child);
		    } else if (child instanceof EContentRefNode) {
			parent.addChild(child); // XXX : text ref
		    } else if (child instanceof EExternalRefNode) {
			parent.addChild(child);
		    } else if (child instanceof EExternalContentRefNode) {
			parent.addChild(child); // XXX : text ref
		    } else if (child instanceof ESequenceNode) {
			ESequenceNode seq = (ESequenceNode)child;
			if (seq.getOccurs() == OCCURS_ONE) {
			    parent.addChildren(seq);
			} else {
			    String labelName;
			    if (debug__) {
				labelName = _makeUniqueName("sequenceh");
			    } else {
				labelName = _makeUniqueName("sequence");
			    }
			    EContentRefNode newRef = _makeImplicitContentNode(
				labelName,
				UERule.unifyOccurs(
				    occurs,
				    seq.getOccurs()
				),
				seq.getChildren()
			    );
			    parent.addChild(newRef);
			}
		    } else if (child instanceof EInterleaveNode) {
			parent.addChild(child);
		    } else if (child instanceof EChoiceNode) {
			if (parent instanceof EElementNode) {
			    if (_isDataChoice((EChoiceNode)child)) {
				IXMLDatatype datatype
				    = _makeDatatype((EChoiceNode)child);
				((EElementNode)parent).setDatatype(datatype);
			    } else {
				parent.addChild(child);
			    }
			} else {
			    parent.addChild(child);
			}
		    } else if (child instanceof EMixedNode) {
			parent.addChild(child);
		    } else if (child instanceof ENoneNode) {
			return;
		    } else if (child instanceof EAnyOtherAttributeNode) {
			parent.addChild(child);
		    } else if (child instanceof EAnyOtherElementNode) {
			parent.addChild(child);
		    } else if (child instanceof EDataSlot) {
			if (parent instanceof EElementNode) {
			    ((EElementNode)parent).setDatatype(
				_makeDatatype((EDataSlot)child)
			    );
			} else {
			    parent.addChild(child);
			}
		    } else if (child instanceof EValueSlot) {
			if (parent instanceof EElementNode) {
			    ((EElementNode)parent).setDatatype(
				_makeDatatype((EValueSlot)child)
			    );
			} else {
			    parent.addChild(child);
			}
		    } else if (child instanceof EAttributeRefNode) {
			parent.addChild(child);
		    } else {
			throw (new InternalError(child.toString()));
		    }
		}
		fragile_ = true;
	    } else if (children.length == 1) {
		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    EElementSlot eSlot = (EElementSlot)child;
		    _adjustDatatype(eSlot);
		    eSlot.setOccurs(
			UERule.unifyOccurs(
			    occurs,
			    eSlot.getOccurs()
			)
		    );
		    parent.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    EAttributeSlot aSlot = (EAttributeSlot)child;
		    _adjustDatatype(aSlot);
		    if (occurs == OCCURS_ZEROONE) {
			aSlot.setRequired(false);
		    }
		    parent.addChild(child);
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof ESequenceNode) {
		    throw (new InternalError());
/*
		    ESequenceNode seq = (ESequenceNode)child;
		    EContentRefNode newRef = _makeImplicitContentNode(
			_makeUniqueName("sequence"),
			UERule.unifyOccurs(
			    occurs,
			    seq.getOccurs()
			),
			seq.getChildren()
		    );
		    parent.addChild(newRef);
*/
		} else if (child instanceof EInterleaveNode) {
		    parent.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode choice = (EChoiceNode)child;
		    choice.setOccurs(
			UERule.unifyOccurs(
			    choice.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(child);
		} else if (child instanceof EMixedNode) {
		    parent.addChild(child);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else if (child instanceof EAttributeRefNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
		fragile_ = true;
	    } else {
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustExternalContentRefNodeInRule(
	ENode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
	    }
	    return;
	}
	String parentNs = parent.getModule().getTargetNamespace();
	String contentNs = cNode.getModule().getTargetNamespace();
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		    parent.addChild(new ENoneNode());
		}
		return;
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (occurs == OCCURS_ONE) {
		cNode.lockRead();
		try {
		    // differ against _adjustContentRefNodeInRule
		    for (int i = 0;i < children.length;i++) {
			ENode child = _deepCopy(children[i]);
			if (child instanceof EElementSlot) {
			    EElementSlot eSlot = (EElementSlot)child;
			    EElementNode eNode = eSlot.getElementNode();
			    if (eNode == null) {
				throw (new InternalError());
			    }
			    EModule eModule = eNode.getModule();
			    eModule.activateOptimizedElementRule(eNode);
			    parent.addChild(
				new EExternalRefNode(
				    contentNs,
				    eNode,
				    eSlot.getOccurs()
				)
			    );
			} else if (child instanceof EElementRefNode) {
			    EElementRefNode oldRef = (EElementRefNode)child;
			    EExternalRefNode newRef = new EExternalRefNode(
				contentNs,
				oldRef.getElementNode(),
				UERule.unifyOccurs(
				    oldRef.getOccurs(),
				    occurs
				)
			    );
			    parent.addChild(newRef);
			} else if (child instanceof EContentRefNode) {
			    EContentRefNode oldRef = (EContentRefNode)child;
			    EExternalContentRefNode newRef
				= new EExternalContentRefNode(
				    contentNs,
				    oldRef.getContentNode(),
				    UERule.unifyOccurs(
					oldRef.getOccurs(),
					occurs
				    )
				);
			    parent.addChild(newRef);
			    fragile_ = true;
			} else if (child instanceof EExternalRefNode) {
			    EExternalRefNode oldRef = (EExternalRefNode)child;
			    if (parentNs.equals(oldRef.getNamespace())) {
				EElementRefNode newRef = new EElementRefNode(
				    oldRef.getElementNode(),
				    oldRef.getOccurs()
				);
				parent.addChild(newRef);
			    } else {
				parent.addChild(oldRef);
			    }
			} else if (child instanceof EExternalContentRefNode) {
			    EExternalContentRefNode oldRef
				= (EExternalContentRefNode)child;
			    if (parentNs.equals(oldRef.getNamespace())) {
				EContentRefNode newRef = new EContentRefNode(
				    oldRef.getContentNode(),
				    oldRef.getOccurs()
				);
				parent.addChild(newRef);
			    } else {
				parent.addChild(oldRef);
			    }
			    fragile_ = true;
			} else if (child instanceof EChoiceNode) {
			    _adjustExternalRef(child, parentNs, contentNs);
			    parent.addChild(child);
			} else if (child instanceof EMixedNode) {
			    _adjustExternalRef(child, parentNs, contentNs);
			    parent.addChild(child);
			} else {
			    parent.addChild(child);
			}
		    }
		} finally {
		    cNode.unlockRead();
		}
	    } else if (children.length == 1) {
		// differ against _adjustContentRefNodeInRule
//		ENode child = _deepCopyInContentRef(children[0]);
		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    _adjustDatatype((EElementSlot)child);
		    parent.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    EAttributeSlot aSlot = (EAttributeSlot)child;
		    _adjustDatatype(aSlot);
		    if (occurs == OCCURS_ZEROONE) {
			aSlot.setRequired(false);
		    }
		    parent.addChild(child);
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(
			contentNs,
			oldRef.getElementNode(),
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(
			    contentNs,
			    oldRef.getContentNode(),
			    UERule.unifyOccurs(
				oldRef.getOccurs(),
				occurs
			    )
			);
		    parent.addChild(newRef);
		    fragile_ = true;
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    if (parentNs.equals(oldRef.getNamespace())) {
			EElementRefNode newRef = new EElementRefNode(
			    oldRef.getElementNode(),
			    UERule.unifyOccurs(
				oldRef.getOccurs(),
				occurs
			    )
			);
			parent.addChild(newRef);
		    } else {
			EExternalRefNode newRef
			    = new EExternalRefNode(oldRef);
			newRef.setOccurs(
			    UERule.unifyOccurs(
				oldRef.getOccurs(),
				occurs
			    )
			);
			parent.addChild(newRef);
		    }
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    if (parentNs.equals(oldRef.getNamespace())) {
			EContentRefNode newRef = new EContentRefNode(
			    oldRef.getContentNode(),
			    UERule.unifyOccurs(
				oldRef.getOccurs(),
				occurs
			    )
			);
			parent.addChild(newRef);
		    } else {
			EExternalContentRefNode newRef
			    = new EExternalContentRefNode(oldRef);
			newRef.setOccurs(
			    UERule.unifyOccurs(
				oldRef.getOccurs(),
				occurs
			    )
			);
			parent.addChild(newRef);
		    }
		    fragile_ = true;
		} else if (child instanceof ESequenceNode) {
		    parent.addChild(child);
		} else if (child instanceof EInterleaveNode) {
		    parent.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    _adjustExternalRef(child, parentNs, contentNs);
		    EChoiceNode choice = (EChoiceNode)child;
		    choice.setOccurs(
			UERule.unifyOccurs(
			    choice.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(child);
		} else if (child instanceof EMixedNode) {
		    _adjustExternalRef(child, parentNs, contentNs);
		    parent.addChild(child);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else {
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustExternalRef(
	ENode parent,
	String parentNs,
	String contentNs
    ) {
	ENode[] children = parent.getChildren();
	parent.clear();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		EElementSlot eSlot = (EElementSlot)child;
		EElementNode eNode = eSlot.getElementNode();
		if (eNode == null) {
		    throw (new InternalError());
		}
		EModule eModule = eNode.getModule();
		eModule.activateOptimizedElementRule(eNode);
		parent.addChild(
		    new EExternalRefNode(
			contentNs,
			eNode,
			eSlot.getOccurs()
		    )
		);
	    } else if (child instanceof EElementRefNode) { // namespace
		EElementRefNode oldRef = (EElementRefNode)child;
		EElementNode eNode = oldRef.getElementNode();
		String targetNs = eNode.getModule().getTargetNamespace();
		if (parentNs.equals(targetNs)) {
		    parent.addChild(oldRef);
		} else {
		    EExternalRefNode newRef = new EExternalRefNode(
			targetNs,
			oldRef.getElementNode(),
			oldRef.getOccurs()
		    );
		    parent.addChild(newRef);
		}
	    } else if (child instanceof EContentRefNode) { // namespace
		EContentRefNode oldRef = (EContentRefNode)child;
		EContentNode cNode = oldRef.getContentNode();
		String targetNs = cNode.getModule().getTargetNamespace();
		if (parentNs.equals(targetNs)) {
		    parent.addChild(oldRef);
		} else {
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(
			    targetNs,
			    oldRef.getContentNode(),
			    oldRef.getOccurs()
			);
		    parent.addChild(newRef);
		}
		fragile_ = true;
	    } else if (child instanceof EExternalRefNode) {
		EExternalRefNode oldRef = (EExternalRefNode)child;
		if (parentNs.equals(oldRef.getNamespace())) {
		    EElementRefNode newRef = new EElementRefNode(
			oldRef.getElementNode(),
			oldRef.getOccurs()
		    );
		    parent.addChild(newRef);
		} else {
		    parent.addChild(oldRef);
		}
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode oldRef
		    = (EExternalContentRefNode)child;
		if (parentNs.equals(oldRef.getNamespace())) {
		    EContentRefNode newRef = new EContentRefNode(
			oldRef.getContentNode(),
			oldRef.getOccurs()
		    );
		    parent.addChild(newRef);
		} else {
		    parent.addChild(oldRef);
		}
		fragile_ = true;
	    } else if (child instanceof EChoiceNode) {
		_adjustExternalRef(child, parentNs, contentNs);
		parent.addChild(child);
	    } else if (child instanceof EMixedNode) {
		_adjustExternalRef(child, parentNs, contentNs);
		parent.addChild(child);
	    } else {
		parent.addChild(child);
	    }
	}
    }

    private void _adjustAttributeRefNodeInRule(
	ENode parent,
	ENode aRef,
	EAttributeNode aNode,
	int occurs
    ) throws InterruptedException {
	if (aNode.isNameClass()) {
	    parent.addChild(aRef);
	} else if (occurs == OCCURS_ONE) {
	    EAttributeSlot aSlot = new EAttributeSlot(
		aNode.getAttrName(),
		aNode.getDatatype()
	    );
	    aSlot.setNamespace(aNode.getNamespace());
//System.out.println("aaa : " + aSlot);
	    aSlot.setRequired(true);
	    aSlot.setAnd(aNode.isAnd());
	    aSlot.setAttributeNode(aNode);
	    parent.addChild(aSlot);
	} else if (occurs == OCCURS_ZEROONE) {
	    EAttributeSlot aSlot = new EAttributeSlot(
		aNode.getAttrName(),
		aNode.getDatatype()
	    );
	    aSlot.setNamespace(aNode.getNamespace());
//System.out.println("bbb : " + aSlot);
	    aSlot.setRequired(false);
	    aSlot.setAnd(aNode.isAnd());
	    aSlot.setAttributeNode(aNode);
	    parent.addChild(aSlot);
	} else if (occurs == OCCURS_ONEMORE) {
	    throw (new InternalError());
	} else if (occurs == OCCURS_ZEROMORE) {
	    throw (new InternalError());
	} else {
	    throw (new InternalError());
	}
    }

/*
    private ENode[] _deepCopyChildrenInContentRef(ENode[] children) {
	ENode[] result = new ENode[children.length];
	for (int i = 0;i < children.length;i++) {
	    result[i] = _deepCopyInContentRef(children[i]);
	}
	return (result);
    }

    private ENode _deepCopyInContentRef(ENode node) {
	if (node instanceof EElementSlot) {
	    EElementSlot eSlot = (EElementSlot)node;
	    EElementNode eNode = eSlot.getElementNode();
	    if (eNode == null) {
		throw (new InternalError());
	    }
	    return (new EElementRefNode(eNode, eSlot.getOccurs()));
	} else {
	    return (_deepCopy(node));
	}
    }
*/

    private void _adjustElementRefNodeInChoice(
	EChoiceNode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    return;
	}
	if (UERule.canUnification(parent.getOccurs(), occurs)) {
	    parent.addChild(eRef);
	} else {
	    String labelName;
	    if (debug__) {
		labelName = _makeUniqueName("reff");
	    } else {
		labelName = _makeUniqueName("ref");
	    }
	    EContentRefNode cRef = _makeImplicitContentNode(
		labelName,
		OCCURS_ONE,
		new ENode[] { eRef }
	    );
	    parent.addChild(cRef);
	}
    }

    private void _adjustElementRefNode(	// XXX : not used
	ENode parent,
	ENode eRef,
	EElementNode eNode,
	int occurs
    ) throws InterruptedException {
	if (eNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
		return;
	    }
	}
	parent.addChild(eRef);
    }

/*
    private void _adjustContentRefNode(	// XXX : not used
	ENode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		parent.addChild(new ENoneNode());
		return;
	    }
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		    parent.addChild(new ENoneNode());
		    return;
		}
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (children.length == 1) {
		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    _adjustDatatype((EElementSlot)child);
		    parent.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    throw (new InternalError(child.toString()));
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof ESequenceNode) {
		    parent.addChild(child);
		} else if (child instanceof EInterleaveNode) {
		    parent.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode choice = (EChoiceNode)child;
		    choice.setOccurs(
			UERule.unifyOccurs(
			    choice.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(child);
		} else if (child instanceof EMixedNode) {
		    parent.addChild(child);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else {
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }
*/

    private void _adjustContentRefNodeInMixed(
	EMixedNode parent,
	ENode cRef,
	EContentNode cNode
    ) throws InterruptedException {
	_adjustContentRefNodeInChoiceMixed(
	    parent,
	    cRef,
	    cNode,
	    OCCURS_ZEROMORE,
	    OCCURS_ZEROMORE
	);
    }	

    private void _adjustContentRefNodeInChoice(
	EChoiceNode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs
    ) throws InterruptedException {
	_adjustContentRefNodeInChoiceMixed(
	    parent,
	    cRef,
	    cNode,
	    occurs,
	    parent.getOccurs()
	);
    }

    private void _adjustContentRefNodeInChoiceMixed(
	ENode parent,
	ENode cRef,
	EContentNode cNode,
	int occurs,
	int parentOccurs
    ) throws InterruptedException {
	if (cNode.isNone()) {
	    return;
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		    return;
	    }
	    _assertValidCRefNode(cRef);
/*
	    if (cNode.isCyclic(rule_)) { // XXX
		parent.addChild(cRef);
		return;
	    }
*/
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return;
	    } else if (children.length == 1 &&
		       !(children[0] instanceof EAttributeSlot)) {

		ENode child = _deepCopy(children[0]);
		if (child instanceof EElementSlot) {
		    _adjustDatatype((EElementSlot)child);
		    parent.addChild(cRef);
		} else if (child instanceof EAttributeSlot) {
		    throw (new InternalError(child.toString()));
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode oldRef = (EElementRefNode)child;
		    EElementRefNode newRef = new EElementRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode oldRef = (EContentRefNode)child;
		    EContentRefNode newRef = new EContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode oldRef = (EExternalRefNode)child;
		    EExternalRefNode newRef = new EExternalRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode oldRef
			= (EExternalContentRefNode)child;
		    EExternalContentRefNode newRef
			= new EExternalContentRefNode(oldRef);
		    newRef.setOccurs(
			UERule.unifyOccurs(
			    oldRef.getOccurs(),
			    occurs
			)
		    );
		    parent.addChild(newRef);
		} else if (child instanceof ESequenceNode) {
		    ESequenceNode childSequence = (ESequenceNode)child;
		    if (childSequence.getSize() <= 1) {
			throw (new InternalError());
		    }
		    parent.addChild(cRef);
		} else if (child instanceof EInterleaveNode) {
		    EInterleaveNode childInterleave = (EInterleaveNode)child;
		    if (childInterleave.getSize() <= 1) {
			throw (new InternalError());
		    }
		    parent.addChild(cRef);
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode childChoice = (EChoiceNode)child;
		    int targetOccurs = UERule.unifyOccurs(
			occurs,
			childChoice.getOccurs()
		    );
		    if (UERule.canUnification(
			parentOccurs, targetOccurs)) {

			parent.addChildren(childChoice);
		    } else {
			parent.addChild(cRef);
		    }
		} else if (child instanceof EMixedNode) {
		    parent.addChild(cRef);
		} else if (child instanceof ENoneNode) {
		    return;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    parent.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    } else {
		if (UERule.canUnification(parentOccurs, occurs)) {
		    parent.addChild(cRef);
		} else {
		    String labelName;
		    if (debug__) {
			labelName = _makeUniqueName("refg");
		    } else {
			labelName = _makeUniqueName("ref");
		    }
		    EContentRefNode newRef = _makeImplicitContentNode(
			labelName,
			OCCURS_ONE,
			new ENode[] { cRef }
		    );
		    parent.addChild(newRef);
		}
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    private void _adjustContentRefNodeForExpandInChoice(
	ENode parent,
	EContentRefNode cRef,
	int currentOccurs
    ) throws InterruptedException {
	_assertValidCRefNode(cRef);
	EContentNode cNode = cRef.getContentNode();
/*
	if (cNode.isCyclic(rule_)) { // XXX
	    parent.addChild(cRef);
	    return;
	}
*/
	int refOccurs = _getContentRefNodeOccurs(cRef);
	cNode.lockRead();
	try {
	    if (cNode.getSize() == 0) {
		return;
	    } else if (cNode.getSize() == 1) {
		if (refOccurs == OCCURS_ONE) {
		    parent.addChildren(
			_deepCopyChildren(cNode.getChildren())
		    );
		} else {
		    parent.addChild(cRef);
		}
	    } else {
		cRef.setOccurs(	// XXX
		    UERule.unifyOccurs(currentOccurs, refOccurs)
		);
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    // XXX : change - _adjustExternalRef
    private void _adjustExternalContentRefNodeForExpandInChoice(
	ENode parent,
	EExternalContentRefNode cRef,
	int currentOccurs
    ) throws InterruptedException {
	EContentNode cNode = cRef.getContentNode();
	int refOccurs = _getContentRefNodeOccurs(cRef);
	cNode.lockRead();
	try {
	    if (cNode.getSize() == 0) {
		return;
	    } else if (cNode.getSize() == 1) {
		if (refOccurs == OCCURS_ONE) {
		    parent.addChildren(
			_deepCopyChildren(cNode.getChildren())
		    );
		} else {
		    parent.addChild(cRef);
		}
	    } else {
		cRef.setOccurs(
		    UERule.unifyOccurs(currentOccurs, refOccurs)
		);
		parent.addChild(cRef);
	    }
	} finally {
	    cNode.unlockRead();
	}
    }

    // _adjustTreeInElementNode
    // _adjustTreeInContentNode
    // _adjustTreeInSequenceNode
    // _adjustTreeInInterleaveNode
    private void _adjustInterleaveNode(
	ENode parent,
	EInterleaveNode interleave
    ) throws InterruptedException {
	int occurs = interleave.getOccurs();
	ENode[] children = interleave.getChildren();
	interleave.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInInterleaveNode(interleave, children[i]);
	}
	children = interleave.getChildren();
	interleave.clear();
/*
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EContentRefNode) {
		EContentRefNode cRef = (EContentRefNode)child;
		EContentNode cNode = cRef.getContentNode();
		cNode.lockRead();
		try {
		    ENode[] cChildren = cNode.getChildren();
		    if (cChildren.length == 1) {
			if (cChildren[0] instanceof EAttributeSlot) {
			    EAttributeSlot newASlot
				= new EAttributeSlot((EAttributeSlot)cChildren[0]);
			    if (occurs == OCCURS_ZEROONE ||
				occurs == OCCURS_ZEROMORE) {
				
				newASlot.setRequired(false);
			    }
			    parent.addChild(newASlot);
			    continue;
			}
		    }
		} finally {
		    cNode.unlockRead();
		}
	    }
	    interleave.addChild(child);
	}
	children = interleave.getChildren();
*/
	if (children.length == 0) {
	    return;
	}
	if (children.length == 2) { // Tutorial 8
	    if (children[0] instanceof EDataSlot) {
		EMixedNode mixed = new EMixedNode();
		_adjustTreeInMixedNode(mixed, children[1]);
		parent.addChild(mixed);
		return;
	    } else if (children[1] instanceof EDataSlot) {
		EMixedNode mixed = new EMixedNode();
		_adjustTreeInMixedNode(mixed, children[0]);
		parent.addChild(mixed);
		return;
	    }
	}
	if (occurs == OCCURS_ONE) {
	    for (int i = 0;i < children.length;i++) {
		ENode child = children[i];
		if (child instanceof EElementSlot) {
		    interleave.addChild(child);
		} else if (child instanceof EAttributeSlot) {
		    parent.addChild(child);
		} else if (child instanceof EElementRefNode) {
		    interleave.addChild(child);
		} else if (child instanceof EContentRefNode) {
		    interleave.addChild(child);
		} else if (child instanceof EExternalRefNode) {
		    interleave.addChild(child);
		} else if (child instanceof EExternalContentRefNode) {
		    interleave.addChild(child);
		} else if (child instanceof ESequenceNode) {
		    interleave.addChild(child);
		} else if (child instanceof EInterleaveNode) {
		    interleave.addChild(child);
		} else if (child instanceof EChoiceNode) {
		    interleave.addChild(child);
		} else if (child instanceof EMixedNode) {
		    interleave.addChild(child);
		} else if (child instanceof EAnyOtherElementNode) {
		    interleave.addChild(child);
		} else if (child instanceof EAttributeRefNode) {
		    parent.addChild(child);
		} else if (child instanceof EDataSlot) {
// XXX : mixed?	    System.out.println(child + " in " + parent);
		    parent.addChild(child);
		} else if (child instanceof EValueSlot) {
		    parent.addChild(child);
		} else {
		    throw (new InternalError(child.toString()));
		}
	    }
	    if (interleave.size() > 0) {
		if (parent instanceof ERuleNode) {
		    parent.addChild(interleave);
		} else {
		    EContentRefNode cRef = _makeImplicitContentNode(
			_makeUniqueName("interleave"),
			occurs,
			children
		    );
		    parent.addChild(cRef);
		}
	    }
	} else if (children.length == 1) {
	    _makeOptimizedEntryForInterleaveOne(
		parent,
		children[0],
		occurs
	    );
	} else {
	    if (parent instanceof ERuleNode) {
		parent.addChild(interleave);
	    } else {
		EContentRefNode cRef = _makeImplicitContentNode(
		    _makeUniqueName("interleave"),
		    occurs,
		    children
		);
		parent.addChild(cRef);
	    }
	}
    }

    // _adjustTreeInElementNode
    // _adjustTreeInContentNode
    // _adjustTreeInSequenceNode
    // _adjustTreeInChoiceNode
    private void _adjustChoiceNode(
	ENode parent,
	EChoiceNode choice
    ) throws InterruptedException {
	_printLog("before _adjustChoiceNode(2) : " + parent);
	_printLog("       _adjustChoiceNode(2) : " + choice);
	_adjustChoiceNode(choice);
	ENode[] children = choice.getChildren();
	choice.clear();
	if (children.length == 0) {
	    _printLog("after _adjustChoiceNode(2) : " + parent);
	    return;
	} else if (children.length == 1) {
	    _makeOptimizedEntryForChoiceOne(
		parent,
		children[0],
		choice.getOccurs()
	    );
	} else {
	    int occurs = choice.getOccurs();
	    ENode[] expandables = _getExpandableChoices(children, occurs);
	    if (expandables == null) {
//		if (parent instanceof EElementNode && // for data
//		    _isMixedChildren(children)) {
		if (_isMixedChildren(children)) {
		    EMixedNode mixed = new EMixedNode();
		    _makeOptimizedEntryForMixed(mixed, children);
		    mixed.setOccurs(choice.getOccurs());
		    parent.addChild(mixed);
		} else {
		    choice.addChildren(children);
		    parent.addChild(choice);
		}
	    } else {
//		if (parent instanceof EElementNode && // for data
//		    _isExpandableMixedChildren(expandables)) {
		if (_isExpandableMixedChildren(expandables)) {
//		if (_isMixedChildren(children)) {
		    EMixedNode mixed = new EMixedNode();
		    _makeExpandedMixed(mixed, expandables);
		    mixed.setOccurs(_getUnifyOccurs(mixed, occurs));
		    parent.addChild(mixed);
		} else {
		    _makeExpandedChoice(choice, expandables);
		    choice.setOccurs(_getUnifyOccurs(choice, occurs));
		    parent.addChild(choice);
		}
	    }
	}
        _printLog("after _adjustChoiceNode(2) : " + parent);
    }

    private int _getUnifyOccurs(ENode parent, int occurs) {
	ENode[] children = parent.getChildren();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EChoiceNode) {
		int targetOccurs = ((EChoiceNode)child).getOccurs();
		occurs = UERule.unifyOccurs(occurs, targetOccurs);
	    } else if (child instanceof EMixedNode) {
		int targetOccurs = ((EMixedNode)child).getOccurs();
		occurs = UERule.unifyOccurs(occurs, targetOccurs);
	    }
	}
	return (occurs);
    }

    private ENode[] _getExpandableChoices(
	ENode[] children,
	int occurs
    ) throws InterruptedException {
	List candidate = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EContentRefNode) {
		EContentRefNode cRef = (EContentRefNode)child;
		EContentNode cNode = cRef.getContentNode();
		cNode.lockRead();
		try {
		    ENode[] cChildren = cNode.getChildren();
		    if (cChildren.length == 1) {
			ENode cChild = cChildren[0];
			if (cChild instanceof EChoiceNode) {
			    EChoiceNode choice = (EChoiceNode)cChild;
			    if (!UERule.canUnification(
				occurs, choice.getOccurs())) {

				return (null);
			    }
			    candidate.add(cChild);
			} else if (cChild instanceof EMixedNode) {
			    candidate.add(cChild);
			} else if (cChild instanceof EDataSlot) {
			    candidate.add(cChild);
			} else if (cChild instanceof EValueSlot) {
			    candidate.add(cChild);
			} else {
			    return (null);
			}
		    } else {
			return (null);
		    }
		} finally {
		    cNode.unlockRead();
		}
	    } else if (child instanceof EElementRefNode ||
		       child instanceof EExternalRefNode ||
		       child instanceof EExternalContentRefNode) {

		candidate.add(child);
	    } else if (child instanceof EDataSlot ||
		       child instanceof EValueSlot) {

		candidate.add(child);
	    } else if (child instanceof EAttributeRefNode) {
		candidate.add(child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	int size = candidate.size();
	if (size == 0) {
	    return (null);
	}
	ENode[] result = new ENode[size];
	for (int i = 0;i < size;i++) {
	    result[i] = _deepCopy((ENode)candidate.get(i));
	}
	return (result);
    }

    private void _makeExpandedMixed(EMixedNode mixed, ENode[] children) {
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EChoiceNode) {
		mixed.addChildren(child);
	    } else if (child instanceof EMixedNode) {
		mixed.addChildren(child);
	    } else if (child instanceof EElementRefNode ||
		       child instanceof EExternalRefNode ||
		       child instanceof EExternalContentRefNode) {

		mixed.addChild(child);
	    } else if (child instanceof EDataSlot) {
		// do nothing
	    } else if (child instanceof EValueSlot) {
		// do nothing
	    } else if (child instanceof EAttributeRefNode) {
		mixed.addChild(child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
    }

    private void _makeExpandedChoice(EChoiceNode choice, ENode[] children) {
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EChoiceNode) {
		choice.addChildren(child);
	    } else if (child instanceof EMixedNode) {
		choice.addChildren(child);
	    } else if (child instanceof EElementRefNode ||
		       child instanceof EExternalRefNode ||
		       child instanceof EExternalContentRefNode) {

		choice.addChild(child);
	    } else if (child instanceof EDataSlot) {
		choice.addChild(child);
	    } else if (child instanceof EValueSlot) {
		choice.addChild(child);
	    } else if (child instanceof EAttributeRefNode) {
		choice.addChild(child);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
    }

    private boolean _isExpandableMixedChildren(ENode[] children)
	throws InterruptedException {

	boolean isText = false;
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EElementRefNode) {
		// do nothing
	    } else if (child instanceof EContentRefNode) {
		if (!_isDataContent((EContentRefNode)child)) {
		    return (false);
		}
	    } else if (child instanceof EExternalRefNode) {
		// do nothing
	    } else if (child instanceof EExternalContentRefNode) {
		if (!_isDataContent((EExternalContentRefNode)child)) {
		    return (false);
		}
	    } else if (child instanceof ESequenceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EInterleaveNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EChoiceNode) {
		// do nothing
	    } else if (child instanceof EMixedNode) {
		isText = true;
	    } else if (child instanceof EAnyOtherAttributeNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherElementNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EDataSlot ||
		       child instanceof EValueSlot) {
		isText = true;
	    } else if (child instanceof EAttributeRefNode) {
		// do nothing
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	return (isText);
    }

    private boolean _isMixedChildren(ENode[] children)
	throws InterruptedException {

	boolean isElement = false;
	boolean isText = false;
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		isElement = true;
	    } else if (child instanceof EAttributeSlot) {
		// do nothing
	    } else if (child instanceof EElementRefNode) {
		isElement = true;
	    } else if (child instanceof EContentRefNode) {
		if (!_isDataContent((EContentRefNode)child)) {
		    return (false);
		}
	    } else if (child instanceof EExternalRefNode) {
		isElement = true;
	    } else if (child instanceof EExternalContentRefNode) {
		if (!_isDataContent((EExternalContentRefNode)child)) {
		    return (false);
		}
	    } else if (child instanceof ESequenceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EInterleaveNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EChoiceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EMixedNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherAttributeNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherElementNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EDataSlot ||
		       child instanceof EValueSlot) {
		isText = true;
	    } else if (child instanceof EAttributeRefNode) {
		// do nothing
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	return (isElement && isText);
    }

    private void _adjustChoiceNode(
	EChoiceNode choice
    ) throws InterruptedException {
	_printLog("before _adjustChoiceNode : " + choice);
	ENode[] children = choice.getChildren();
	choice.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInChoiceNode(choice, children[i]);
	}
	children = _sortChoiceChildren(choice.getChildren());
	choice.clear();
	choice.addChildren(children);
	_printLog("after _adjustChoiceNode : " + choice);
    }

    private ENode[] _sortChoiceChildren(ENode[] children) {
	LooseList listByWidth = new LooseList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EElementRefNode ||
		       child instanceof EContentRefNode ||
		       child instanceof EExternalRefNode ||
		       child instanceof EExternalContentRefNode) {
		int width = _calcTargetWidth(child);
		_sortChoiceChildrenAddChild(child, width, listByWidth);
	    } else if (child instanceof ESequenceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EInterleaveNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EChoiceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EMixedNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherAttributeNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherElementNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EDataSlot ||
		       child instanceof EValueSlot) {
		_sortChoiceChildrenAddChild(child, 0, listByWidth);
	    } else if (child instanceof EAttributeRefNode) {
		_sortChoiceChildrenAddChild(child, 1, listByWidth);
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	List resultList = new ArrayList();
	for (int i = listByWidth.size() - 1;i >= 0;i--) {
	    List list = (List)listByWidth.get(i);
	    if (list != null) {
		resultList.addAll(list);
	    }
	}
	resultList = _unifyChoiceChildren(resultList);
	ENode[] result = new ENode[resultList.size()];
	return ((ENode[])resultList.toArray(result));
    }

    private void _sortChoiceChildrenAddChild(
	ENode child,
	int width,
	LooseList listByWidth
    ) {
	List list;
	if (listByWidth.size() <= width) {
	    list = new ArrayList();
	    listByWidth.put(width, list);
	} else {
	    list = (List)listByWidth.get(width);
	    if (list == null) {
		list = new ArrayList();
		listByWidth.put(width, list);
	    }
	}
	list.add(child);
    }

    private List _unifyChoiceChildren(List source) {
	Map memo = new HashMap();
	List result = new ArrayList();
	int size = source.size();
	for (int i = 0;i < size;i++) {
	    ENode child = (ENode)source.get(i);
	    if (child instanceof EDataSlot ||
		child instanceof EValueSlot) {

		result.add(child);
	    } else {
		ENode target = _getRefTarget(child);
		ENode ref = (ENode)memo.get(target);
		if (ref == null) {
		    memo.put(target, child);
		    result.add(child);
		} else {
		    if (_isStrongRef(child, ref)) {
			memo.put(target, child);
			result.remove(ref);
			result.add(child);
		    }
		}
	    }
	}
	return (result);
    }

    private boolean _isStrongRef(ENode newRef, ENode oldRef) {
	if (newRef.getClass() != oldRef.getClass()) {
	    return (false);
//	    throw (new InternalError());
	}
	int newOccurs = _getRefOccurs(newRef);
	int oldOccurs = _getRefOccurs(oldRef);
	return (UERule.canUnification(oldOccurs, newOccurs));
    }

    private int _getRefOccurs(ENode node) {
	if (node instanceof EElementRefNode) {
	    return (((EElementRefNode)node).getOccurs());
	} else if (node instanceof EContentRefNode) {
	    return (((EContentRefNode)node).getOccurs());
	} else if (node instanceof EExternalRefNode) {
	    return (((EExternalRefNode)node).getOccurs());
	} else if (node instanceof EExternalContentRefNode) {
	    return (((EExternalContentRefNode)node).getOccurs());
	} else {
	    throw (new InternalError());
	}
    }

    private int _calcTargetWidth(ENode node) {
	ENode target = _getRefTarget(node);
	return (target.getSize());
    }

    private ENode _getRefTarget(ENode node) {
	if (node instanceof EElementRefNode) {
	    return (((EElementRefNode)node).getElementNode());
	} else if (node instanceof EContentRefNode) {
	    return (((EContentRefNode)node).getContentNode());
	} else if (node instanceof EExternalRefNode) {
	    return (((EExternalRefNode)node).getElementNode());
	} else if (node instanceof EExternalContentRefNode) {
	    return (((EExternalContentRefNode)node).getContentNode());
	} else if (node instanceof EAttributeRefNode) {
	    return (((EAttributeRefNode)node).getAttributeNode());
	} else {
	    throw (new InternalError());
	}
    }

    private void _adjustMixedNode(ENode parent, EMixedNode mixed)
	throws InterruptedException {

	_adjustMixedNode(mixed);
	if (mixed.size() == 0) {
	    parent.addChild(new EDataSlot(_makeDatatype("string")));
	} else {
	    parent.addChild(mixed);
	}
    }

    private void _adjustMixedNode(
	EMixedNode mixed
    ) throws InterruptedException {
	_printLog("before _adjustMixedNode : " + mixed);
	ENode[] children = mixed.getChildren();
	mixed.clear();
	for (int i = 0;i < children.length;i++) {
	    _adjustTreeInMixedNode(mixed, children[i]);
	}
	children = _sortChoiceChildren(mixed.getChildren());
	mixed.clear();
	mixed.addChildren(children);
	_printLog("after _adjustMixedNode : " + mixed);
    }

    private void _makeOptimizedEntryForSequenceOne(
	ENode parent,
	ENode child,
	int occurs
    ) throws InterruptedException {
	_printLog("before _makeOptimizedEntryForSequenceOne : " + parent);
	_printLog("       _makeOptimizedEntryForSequenceOne : " + child);
	_printLog("       _makeOptimizedEntryForSequenceOne : " + occurs);
	_makeOptimizedEntryForOne(parent, child, occurs);
	_printLog("after _makeOptimizedEntryForSequenceOne : " + parent);
    }

    private void _makeOptimizedEntryForInterleaveOne(
	ENode parent,
	ENode child,
	int occurs
    ) throws InterruptedException {
	_printLog("before _makeOptimizedEntryForInterleaveOne : " + parent);
	_printLog("       _makeOptimizedEntryForInterleaveOne : " + child);
	_printLog("       _makeOptimizedEntryForInterleaveOne : " + occurs);
	_makeOptimizedEntryForSequenceOne(parent, child, occurs);
	_printLog("after _makeOptimizedEntryForInterleaveOne : " + parent);
    }

    private void _makeOptimizedEntryForChoiceOne(
	ENode parent,
	ENode child,
	int occurs
    ) throws InterruptedException {
	_printLog("before _makeOptimizedEntryForChoiceOne : " + parent);
	_printLog("       _makeOptimizedEntryForChoiceOne : " + child);
	_printLog("       _makeOptimizedEntryForChoiceOne : " + occurs);
	_makeOptimizedEntryForOne(parent, child, occurs);
	_printLog("after _makeOptimizedEntryForChoiceOne : " + parent);
    }

    private void _makeOptimizedEntryForOne(
	ENode parent,
	ENode child,
	int occurs
    ) throws InterruptedException {
	if (child instanceof EElementSlot) {
	    EElementSlot oldSlot = (EElementSlot)child;
	    EElementSlot newSlot = new EElementSlot(oldSlot);
	    _adjustDatatype(newSlot);
	    newSlot.setOccurs(
		UERule.unifyOccurs(
		    oldSlot.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newSlot);
	} else if (child instanceof EAttributeSlot) {
	    throw (new InternalError(child.toString()));
	} else if (child instanceof EElementRefNode) {
	    EElementRefNode oldRef = (EElementRefNode)child;
	    EElementRefNode newRef = new EElementRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (child instanceof EContentRefNode) {
	    EContentRefNode cRef = (EContentRefNode)child;
	    int cOccurs = _getContentRefNodeOccurs(cRef);
	    ENode[] children = _getDeepCopyFromRef((EContentRefNode)child);
	    if (children == null) {
		return;
	    } else if (children.length == 1) {
		ENode target = children[0];
		if (target instanceof ENoneNode) {
		    parent.addChild(target);
		    return;
		}
		_makeOptimizedEntryForOne(
		    parent,
		    target,
		    UERule.unifyOccurs(occurs, cOccurs)
		);
	    } else if (UERule.unifyOccurs(occurs, cOccurs) == OCCURS_ONE &&
		       !(parent instanceof EChoiceNode ||
			 parent instanceof EMixedNode)) {
		parent.addChildren(children);
	    } else {
		EContentRefNode newRef = new EContentRefNode(cRef);
		newRef.setOccurs(
		    UERule.unifyOccurs(
			_getContentRefNodeOccurs(cRef),
			occurs
		    )
		);
		parent.addChild(newRef);
	    }
	} else if (child instanceof EExternalRefNode) {
	    EExternalRefNode oldRef = (EExternalRefNode)child;
	    EExternalRefNode newRef = new EExternalRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (child instanceof EExternalContentRefNode) {
	    EExternalContentRefNode oldRef
		= (EExternalContentRefNode)child;
	    EExternalContentRefNode newRef
		= new EExternalContentRefNode(oldRef);
	    newRef.setOccurs(
		UERule.unifyOccurs(
		    oldRef.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newRef);
	} else if (child instanceof ESequenceNode) {
	    ESequenceNode oldSeq = (ESequenceNode)child;
	    ESequenceNode newSeq = new ESequenceNode(oldSeq);
	    newSeq.setOccurs(
		UERule.unifyOccurs(
		    oldSeq.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newSeq);
	} else if (child instanceof EInterleaveNode) {
	    EInterleaveNode oldInterleave = (EInterleaveNode)child;
	    EInterleaveNode newInterleave = new EInterleaveNode(oldInterleave);
	    newInterleave.setOccurs(
		UERule.unifyOccurs(
		    oldInterleave.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newInterleave);
	} else if (child instanceof EChoiceNode) {
	    EChoiceNode oldChoice = (EChoiceNode)child;
	    EChoiceNode newChoice = new EChoiceNode(oldChoice);
	    newChoice.setOccurs(
		UERule.unifyOccurs(
		    oldChoice.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newChoice);
	} else if (child instanceof EMixedNode) {
	    EMixedNode oldMixed = (EMixedNode)child;
	    EMixedNode newMixed = new EMixedNode(oldMixed);
	    newMixed.setOccurs(
		UERule.unifyOccurs(
		    oldMixed.getOccurs(),
		    occurs
		)
	    );
	    parent.addChild(newMixed);
	} else if (child instanceof ENoneNode) {
	    return;
	} else if (child instanceof EAnyOtherAttributeNode) {
	    parent.addChild(child);
	} else if (child instanceof EAnyOtherElementNode) {
	    parent.addChild(child);
	} else if (child instanceof EDataSlot) {
	    parent.addChild(child);
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private EContentRefNode _makeImplicitContentNodeChoice(
	int occurs,
	ENode[] children
    ) throws InterruptedException {
	EChoiceNode newChoice = new EChoiceNode();
	newChoice.setOccurs(occurs);
	newChoice.setHometown(rule_); // XXX
	newChoice.addChildren(children);
	EContentRefNode cRef = _makeImplicitContentNode(
	    _makeUniqueName("choice"),
	    OCCURS_ONE,
	    new ENode[] { newChoice }
	);
	return (cRef);
    }

    private EContentRefNode _makeImplicitContentNodeChoiceInInterleave(
	int occurs,
	ENode[] children
    ) throws InterruptedException {
	EChoiceNode newChoice = new EChoiceNode();
	newChoice.setHometown(rule_); // XXX
	newChoice.addChildren(children);
	EContentRefNode cRef = _makeImplicitContentNode(
	    _makeUniqueName("choice"),
	    occurs,
	    new ENode[] { newChoice }
	);
	return (cRef);
    }

    private EContentRefNode _makeImplicitContentNode(
	String name,
	int occurs,
	ENode[] children
    ) throws InterruptedException {
	EContentNode cNode = new EContentNode(name);
	for (int i = 0;i < children.length;i++) {
	    _adjustTree(cNode, children[i]);
	}
//	cNode.addChildren(children);
	cNode.setModule(module_);
	cNode.setCyclicRefs(new HashSet()); // XXX
	module_.addOptimizedContentRule(cNode);
	cNode.setImmutable(true);
	children = cNode.getChildren();
	for (int i = 0;i < children.length;i++) {
	    children[i].setImmutable(true);
	}
//	occurs = _adjustContentNodeOccurs(cNode, occurs);
	_assertValidCNode(cNode);
	return (new EContentRefNode(cNode, occurs));
    }

    private EElementRefNode _makeImplicitElementNode(
	EElementSlot eSlot,
	String labelName
    ) throws InterruptedException {
	EElementNode eNode = eSlot.getElementNode();
	if (eNode != null) {
	    return (new EElementRefNode(eNode, eSlot.getOccurs()));
	} else { // last resort
	    return (
		_makeImplicitElementNode(
		    _makeUniqueName(labelName),
		    eSlot.getOccurs(),
		    eSlot.getDatatype()
		)
	    );
	}
    }

    private EElementRefNode _makeImplicitElementNode(
	String name,
	int occurs,
	ENode[] children
    ) throws InterruptedException {
	EElementNode eNode = new EElementNode(name);
	for (int i = 0;i < children.length;i++) {
	    _adjustTree(eNode, children[i]);
	}
	eNode.setModule(module_);
	module_.addOptimizedElementRule(eNode);
	eNode.setImmutable(true);
	children = eNode.getChildren();
	for (int i = 0;i < children.length;i++) {
	    children[i].setImmutable(true);
	}
	return (new EElementRefNode(eNode, occurs));
    }

    private EElementRefNode _makeImplicitElementNode(
	String name,
	int occurs,
	IXMLDatatype datatype
    ) throws InterruptedException {
	EElementNode eNode = new EElementNode(name);
	eNode.setDatatype(datatype);
	eNode.setModule(module_);
	module_.addOptimizedElementRule(eNode);
	eNode.setImmutable(true);
	return (new EElementRefNode(eNode, occurs));
    }

/*
    private EAttributeRefNode _makeImplicitAttributeNode(
	String name,
	boolean isRequired,
	IXMLDatatype datatype
    ) throws InterruptedException {
	EAttributeNode aNode = new EAttributeNode(name, datatype);
	aNode.setModule(module_);
	module_.addOptimizedAttributeRule(aNode);
	aNode.setImmutable(true);
	if (isRequired) {
	    return (new EAttributeRefNode(aNode, OCCURS_ONE));
	} else {
	    return (new EAttributeRefNode(aNode, OCCURS_ZEROONE));
	}
    }

    private EAttributeRefNode _makeImplicitAttributeNode(
	String name,
	int occurs,
	IXMLDatatype datatype
    ) throws InterruptedException {
	EAttributeNode aNode = new EAttributeNode(name, datatype);
	aNode.setModule(module_);
	module_.addOptimizedAttributeRule(aNode);
	aNode.setImmutable(true);
	return (new EAttributeRefNode(aNode, occurs));
    }
*/

    private EAttributeRefNode _makeImplicitAttributeNode(
	EAttributeSlot aSlot,
	String labelName
    ) throws InterruptedException {
	return (
	    _makeImplicitAttributeNode(
		aSlot,
		labelName,
		aSlot.isRequired()?OCCURS_ONE:OCCURS_ZEROONE
	    )
	);
    }

    private EAttributeRefNode _makeImplicitAttributeNode(
	EAttributeSlot aSlot,
	String labelName,
	int occurs
    ) throws InterruptedException {
	EAttributeNode aNode = aSlot.getAttributeNode();
	if (aNode != null) {
	    return (new EAttributeRefNode(aNode, occurs));
	} else {
	    return (
		_makeImplicitAttributeNode(
		    aSlot.getAttrName(),
		    aSlot.getNamespace(),
		    _makeUniqueName(labelName),
		    occurs,
		    aSlot.getDatatype()
		)
	    );
	}
    }

    private EAttributeRefNode _makeImplicitAttributeNode(
	String name,
	String namespace,
	String labelName,
	int occurs,
	IXMLDatatype datatype
    ) throws InterruptedException {
	EAttributeNode aNode = new EAttributeNode(
	    name,
	    namespace,
	    labelName,
	    datatype
	);
//System.out.println("ccc: " + aNode);
	aNode.setModule(module_);
	module_.addOptimizedAttributeRule(aNode);
	aNode.setImmutable(true);
	return (new EAttributeRefNode(aNode, occurs));
    }

    private EAttributeRefNode _makeImplicitAttributeNode(
	String name,
	String namespace,
	String labelName,
	boolean isRequired,
	IXMLDatatype datatype
    ) throws InterruptedException {
	EAttributeNode aNode = new EAttributeNode(
	    name,
	    namespace,
	    labelName,
	    datatype
	);
//System.out.println("ddd: " + aNode);
	aNode.setModule(module_);
	module_.addOptimizedAttributeRule(aNode);
	aNode.setImmutable(true);
	if (isRequired) {
	    return (new EAttributeRefNode(aNode, OCCURS_ONE));
	} else {
	    return (new EAttributeRefNode(aNode, OCCURS_ZEROONE));
	}
    }

    private ENode _deepCopy(ENode node) {
	if (node instanceof EElementSlot) {
	    return (new EElementSlot((EElementSlot)node));
	} else if (node instanceof EAttributeSlot) {
	    return (new EAttributeSlot((EAttributeSlot)node));
	} else if (node instanceof EElementRefNode) {
	    return (new EElementRefNode((EElementRefNode)node));
	} else if (node instanceof EContentRefNode) {
	    return (new EContentRefNode((EContentRefNode)node));
	} else if (node instanceof EExternalRefNode) {
	    return (new EExternalRefNode((EExternalRefNode)node));
	} else if (node instanceof EExternalContentRefNode) {
	    return (
		new EExternalContentRefNode((EExternalContentRefNode)node)
	    );
	} else if (node instanceof ESequenceNode) {
	    return (new ESequenceNode((ESequenceNode)node));
	} else if (node instanceof EInterleaveNode) {
	    return (new EInterleaveNode((EInterleaveNode)node));
	} else if (node instanceof EChoiceNode) {
	    return (new EChoiceNode((EChoiceNode)node)); // XXX : modify
	} else if (node instanceof EMixedNode) {
	    return (new EMixedNode((EMixedNode)node));
	} else if (node instanceof ENoneNode) {
	    return (new ENoneNode());
	} else if (node instanceof EAnyOtherAttributeNode) {
	    return (new EAnyOtherAttributeNode((EAnyOtherAttributeNode)node));
	} else if (node instanceof EAnyOtherElementNode) {
	    return (new EAnyOtherElementNode((EAnyOtherElementNode)node));
	} else if (node instanceof EDataSlot) {
	    return (new EDataSlot((EDataSlot)node));
	} else if (node instanceof EValueSlot) {
	    return (new EValueSlot((EValueSlot)node));
	} else if (node instanceof EAttributeRefNode) {
	    return (new EAttributeRefNode((EAttributeRefNode)node));
	} else {
	    throw (new InternalError(node.toString()));
	}
    }

    private ENode[] _deepCopyChildren(ENode[] children) {
	ENode[] result = new ENode[children.length];
	for (int i = 0;i < children.length;i++) {
	    result[i] = _deepCopy(children[i]);
	}
	return (result);
    }

    private ENode[] _getDeepCopyFromRef(
	EContentRefNode cRef
    ) throws InterruptedException {
	int occurs = cRef.getOccurs();
	EContentNode cNode = cRef.getContentNode();
	if (cNode.isNone()) {
	    if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		return (new ENode[] { new ENoneNode() });
	    }
	    return (null);
	}
	cNode.lockRead();
	try {
	    if (cNode.isNone()) {
		if (occurs == OCCURS_ONE || occurs == OCCURS_ONEMORE) {
		    return (new ENode[] { new ENoneNode() });
		}
		return (null);
	    }
	    ENode[] children = cNode.getChildren();
	    if (children.length == 0) {
		return (null);
	    } else {
		return (_deepCopyChildren(children));
	    }
	} finally {
	    cNode.unlockRead();
	} 
    }

/*
    private IXMLDatatype _makeDatatype(ENode node) {
	if (node instanceof EDataSlot) {
	    return (_makeDatatype((EDataSlot)node));
	} else if (node instanceof ValueSlot) {
	    return (_makeDatatype((EValueSlot)node));
	} else if (node instanceof ChoiceSlot) {
	    return (_makeDatatype((EChoiceSlot)node));
	} else {
	    throw (new InternalError(node.toString()));
	}
    }
*/

    private IXMLDatatype _makeDatatype(String typeName) {
	return (DatatypeFactory.getDatatype(typeName));
    }

    private IXMLDatatype _makeDatatype(EDataSlot data) {
	IXMLDatatype datatype = data.getDatatype();
	return (_adjustDatatype(datatype));
/*
	Datatype datatype = DatatypeFactory.getDatatype(slot.getType());
	XMLFacet[] facets = slot.getFacets();
	if (facets != null) {
	    datatype.addFacets(facets);
	}
	return (datatype);
*/
    }

    private Datatype _makeDatatype(EValueSlot value) {
	return (new DValue(value.getText(), value.getType()));
//	String typeName = value.getType();
//	Datatype datatype = DatatypeFactory.getDatatype(typeName);
//	datatype.addFacet(new XMLFacet("enumeration", value.getText()));
//	return (datatype);
	// XXX : use DValue for choice selection
//	return (new DValue(value.getText(), value.getType()));
/*
	String typeName = slot.getType();
	StringBuffer buffer = new StringBuffer();
	buffer.append("<value");
	if (typeName != null) {
	    buffer.append(" type=\"");
	    buffer.append(typeName);
	    buffer.append("\"");
	}
	buffer.append(">");
	buffer.append(slot.getText());
	buffer.append("</value>");
	Datatype datatype = DatatypeFactory.getDatatype(new String(buffer));
	return (datatype);
*/
    }

    private IXMLDatatype _makeDatatype(EChoiceNode choice)
	throws InterruptedException {

	if (_isDataChoice(choice)) {
	    DComplex complex = new DComplex();
	    TextPattern textPattern = _makeTextPattern(choice);
	    if (textPattern == null) {
		throw (new InternalError());
	    }
	    complex.setProperty(textPattern);
//	    complex.setProperty(choice);
	    return (complex);
	} else {
	    throw (new InternalError());
	}
    }

    private boolean _isDataContent(EContentRefNode cRef)
	throws InterruptedException {

	EContentNode cNode = cRef.getContentNode();
	cNode.lockRead();
	try {
	    return (_isDataChildren(cNode.getChildren()));
	} finally {
	    cNode.unlockRead();
	}
    }

    private boolean _isDataContent(EExternalContentRefNode cRef)
	throws InterruptedException {

	EContentNode cNode = cRef.getContentNode();
	cNode.lockRead();
	try {
	    return (_isDataChildren(cNode.getChildren()));
	} finally {
	    cNode.unlockRead();
	}
    }

    private boolean _isDataChoice(EChoiceNode choice)
	throws InterruptedException {

	ENode[] children = choice.getChildren();
	if (children.length == 0) {
	    return (false);
	}
	return (_isDataChildren(children));
    }

    private boolean _isDataChoiceChild(EChoiceNode choice)
	throws InterruptedException {

//	if (choice.getOccurs() != OCCURS_ONE) {
//	    return (false);
//	}
	return (_isDataChildren(choice.getChildren()));
    }

    private boolean _isDataSequenceChild(ESequenceNode sequence)
	throws InterruptedException {

//	if (sequence.getOccurs() != OCCURS_ONE) {
//	    return (false);
//	}
	return (_isDataChildren(sequence.getChildren()));
    }

    private boolean _isDataChildren(ENode[] children)
	throws InterruptedException {

	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EDataSlot ||
		child instanceof EValueSlot) {

		continue;
	    } else if (child instanceof EChoiceNode) {
		if (!_isDataChoiceChild((EChoiceNode)child)) {
		    return (false);
		}
	    } else if (child instanceof ESequenceNode) {
		if (!_isDataSequenceChild((ESequenceNode)child)) {
		    return (false);
		}
	    } else if (child instanceof EContentRefNode) {
		EContentRefNode cRef = (EContentRefNode)child;
		if (!_isDataContent(cRef)) {
		    return (false);
		}
	    } else if (child instanceof EExternalContentRefNode) {
		EExternalContentRefNode cRef = (EExternalContentRefNode)child;
		if (!_isDataContent(cRef)) {
		    return (false);
		}
	    } else {
		return (false);
	    }
	}
	return (true);
    }

    private IXMLDatatype _makeDatatype(ESequenceNode sequence)
	throws InterruptedException {

	DComplex complex = new DComplex();
	complex.setList(true);
	TextPattern textPattern = _makeTextPattern(sequence);
	if (textPattern == null) {
	    throw (new InternalError());
	}
	complex.setProperty(textPattern);
//	complex.setProperty(sequence);
	return (complex);
    }

    private void _adjustDatatype(EElementNode eNode) {
	eNode.setDatatype(_adjustDatatype(eNode.getDatatype()));
    }

    private void _adjustDatatype(EAttributeNode aNode) {
/*
	IXMLDatatype x = aNode.getDatatype();
	if (x == null) {
	    throw (new InternalError("1"));
	}
	x = _adjustDatatype(x);
	if (x == null) {
	    throw (new InternalError("2"));
	}
	aNode.setDatatype(x);
*/
	aNode.setDatatype(_adjustDatatype(aNode.getDatatype()));
    }

    private void _adjustDatatype(ESlotNode slot) {
	slot.setDatatype(_adjustDatatype(slot.getDatatype()));
    }

    private IXMLDatatype _adjustDatatype(IXMLDatatype datatype) {
	if (datatype == null) {
	    throw (new InternalError());
	}
	if (datatype instanceof DComplex) {
	    try {
		DComplex complex = (DComplex)datatype;
		TextPattern pattern = (TextPattern)complex.getProperty();
		if (pattern == null) {
		    String expr = complex.getExpr();
		    pattern = _makeTextPattern(expr);
		    complex.setProperty(pattern);
		}
		pattern = _adjustTextPattern(pattern);
		complex.setProperty(pattern);
		return (_adjustComplexMore(complex, pattern));
	    } catch (InterruptedException e) {
		throw (new InternalError());
	    }
	}
	return (datatype);
    }

    private IXMLDatatype _adjustComplexMore(
	DComplex complex,
	TextPattern textPattern
    ) {
	PPattern pattern = textPattern.getPattern();
	IPTextPatternChoice[] children = pattern.getTextPattern();
	if (children.length != 1) {
	    throw (new InternalError("length = " + children.length));
	}
	IPTextPatternChoice child = children[0];
	child = _adjustComplexMoreTree(child);
	if (child instanceof PText) {
	    PData pData = (PData)child;
	    Datatype datatype
		= DatatypeFactory.getDatatype("string");
	    if (datatype == null) {
		throw (new InternalError(_getTypeName(pData)));
	    }
	    return (datatype);
	} else if (child instanceof PData) {
	    PData pData = (PData)child;
	    Datatype datatype
		= DatatypeFactory.getDatatype(_getTypeName(pData));
	    if (datatype == null) {
		throw (new InternalError(_getTypeName(pData)));
	    }
	    return (datatype);
	} else if (child instanceof PValue) {
	    PValue pValue = (PValue)child;
	    String content = pValue.getContent();
	    String type = _getTypeName(pValue);
	    return (new DValue(content, type));
	} else if (child instanceof PList) {
	    return (_adjustComplexMoreList((PList)child, complex));
	} else if (child instanceof PChoice) {
	    return (_adjustComplexMoreChoice((PChoice)child, complex));
	} else if (child instanceof PGroup) {
	    return (complex);
	} else if (child instanceof POptional) {
	    return (complex);	// XXX : Syntax Error
	} else if (child instanceof POneOrMore) {
	    return (complex);	// XXX : Syntax Error
	} else if (child instanceof PZeroOrMore) {
	    return (complex);	// XXX : Syntax Error
	} else {
	    throw (new InternalError(child.toString()));
	}
    }

    private IPTextPatternChoice _adjustComplexMoreTree(
	IPTextPatternChoice node
    ) {
	if (node instanceof PList) {
	    PList pList = (PList)node;
	    IPTextPatternChoice[] children = pList.getTextPattern();
	    if (children.length == 1) {
		IPTextPatternChoice child = children[0];
		if (child instanceof PText ||
		    child instanceof PData ||
		    child instanceof PValue) {

		    return (child);
		} else {
		    return (node);
		}
	    } else {
		return (node);
	    }
	} else if (node instanceof PGroup) {
	    PGroup pGroup = (PGroup)node;
	    IPTextPatternChoice[] children = pGroup.getTextPattern();
	    if (children.length == 1) {
		return (children[0]);
	    } else {
		return (node);
	    }
	} else {
	    return (node);
	}
    }

    private IXMLDatatype _adjustComplexMoreChoice(
	PChoice pChoice,
	DComplex complex
    ) {
	IPTextPatternChoice[] children = pChoice.getTextPattern();
	if (children.length == 0) {
	    throw (new InternalError());
	}
	IPTextPatternChoice child = children[0];
	if (child instanceof PValue) {
	    String[] values = new String[children.length];
	    PValue pValue = (PValue)child;
	    String typeName = _getTypeName(pValue);
	    values[0] = pValue.getContent();
	    for (int i = 1;i < children.length;i++) {
		child = children[i];
		if (!(child instanceof PValue)) {
		    return (complex);
		}
		pValue = (PValue)child;
		if (!typeName.equals(_getTypeName(pValue))) {
		    return (complex);
		}
		values[i] = pValue.getContent();
	    }
	    return (new DValue(values, typeName));
	} else {
	    return (complex);
	}
    }

    private IXMLDatatype _adjustComplexMoreList(
	PList pList,
	DComplex complex
    ) {
	complex.setList(true);
	IPTextPatternChoice[] children = pList.getTextPattern();
	if (children.length == 0) {
	    throw (new InternalError());
	}
	if (children.length == 1) {
	    IPTextPatternChoice child = children[0];
	    if (child instanceof PText ||
		child instanceof PData ||
		child instanceof PValue) {

		throw (new InternalError());
	    } else if (child instanceof PChoice) {
		return (complex);
	    } else if (child instanceof POptional) {
		return (complex);
	    } else if (child instanceof POneOrMore) {
		return (
		    _adjustComplexMoreListOneOrMore(
			(POneOrMore)child,
			complex
		    )
		);
	    } else if (child instanceof PZeroOrMore) {
		return (
		    _adjustComplexMoreListZeroOrMore(
			(PZeroOrMore)child,
			complex
		    )
		);
	    } else if (child instanceof PGroup) {
		throw (new InternalError());
	    } else if (child instanceof PRef) {
		throw (new InternalError());
	    } else {
		throw (new InternalError());
	    }
	}
	return (complex);
    }

    private IXMLDatatype _adjustComplexMoreListOneOrMore(
	POneOrMore oneOrMore,
	DComplex complex
    ) {
	return (
	    _adjustComplexMoreListMore(
		oneOrMore.getTextPattern(),
		complex
	    )
	);
    }

    private IXMLDatatype _adjustComplexMoreListZeroOrMore(
	PZeroOrMore zeroOrMore,
	DComplex complex
    ) {
	return (
	    _adjustComplexMoreListMore(
		zeroOrMore.getTextPattern(),
		complex
	    )
	);
    }

    private IXMLDatatype _adjustComplexMoreListMore(
	IPTextPatternChoice[] children,
	DComplex complex
    ) {
	if (children.length == 0) {
	    throw (new InternalError()); // XXX : SyntaxError
	}
	if (children.length != 1) {
	    return (complex);
	}
	IPTextPatternChoice child = children[0];
	if (child instanceof PText) {
	    return (complex);
	} else if (child instanceof PData) {
	    String typeName = _getTypeName((PData)child);
	    Datatype datatype = DatatypeFactory.getDatatype(typeName);
	    datatype.setList(true);
	    return (datatype);
	} else if (child instanceof PValue) {
	    return (complex);
	} else {
	    return (complex);
	}
    }

    private String _getTypeName(PData data) {
	String typeName = data.getType();
	if (typeName == null) {
	    return ("token");
	} else {
	    return (typeName);
	}
    }

    private String _getTypeName(PValue value) {
	String typeName = value.getType();
	if (typeName == null) {
	    return ("token");
	} else {
	    return (typeName);
	}
    }

/*
    private IXMLDatatype _adjustDatatype0(IXMLDatatype datatype) {
	if (datatype instanceof DComplex) {
	    try {
		DComplex complex = (DComplex)datatype;
		ENode node = (ENode)complex.getProperty();
		if (node == null) {
		    String expr = complex.getExpr();
		    Document doc = complex.getExprDoc();
		    node = _adjustExprTree(doc.getDocumentElement());
		    complex.setProperty(node);
		} else {
//System.out.println(Thread.currentThread() + "before = " + node);
		    node = _adjustExprTree(node);
//System.out.println(Thread.currentThread() + "after = " + node);
		    complex.setProperty(node);
		}
		return (_adjustComplexMore(complex));
	    } catch (InterruptedException e) {
		throw (new InternalError());
	    }
	}
	return (datatype);
    }

    private IXMLDatatype _adjustComplexMore0(DComplex complex)
	throws InterruptedException {

//System.out.println("expr = " + complex.getExpr());
	ENode node = (ENode)complex.getProperty();
//System.out.println("node = " + node);
	node = _resolveDataNode(node);
	complex.setProperty(node);
	if (node instanceof EDataSlot) {
	    EDataSlot data = (EDataSlot)node;
	    return (_makeDatatype(data));
	} else if (node instanceof EValueSlot) {
	    EValueSlot value = (EValueSlot)node;
	    return (_makeDatatype(value));
	} else if (node instanceof EContentRefNode) {
	    throw (new InternalError());
	} else if (node instanceof EChoiceNode) {
	    EChoiceNode choice = (EChoiceNode)node;
	    ENode[] children = choice.getChildren();
	    if (children.length == 0) {
		throw (new InternalError());
	    }
	    ENode child = children[0];
	    if (child instanceof EValueSlot) {
		String[] values = new String[children.length];
		EValueSlot value = (EValueSlot)child;
		String typeName = value.getType();
		values[0] = value.getText();
		for (int i = 1;i < children.length;i++) {
		    child = children[i];
		    if (!(child instanceof EValueSlot)) {
			return (complex);
		    }
		    value = (EValueSlot)child;
		    if (!typeName.equals(value.getType())) {
			return (complex);
		    }
		    values[i] = value.getText();
		}
		return (new DValue(values, typeName));
	    } else {
		return (complex);
	    }
	} else if (node instanceof ESequenceNode) {
	    ESequenceNode sequence = (ESequenceNode)node;
	    return (_makeDatatype(sequence));
	} else {
	    throw (new InternalError());
	}
    }

    private ENode _resolveDataNode(ENode node)
	throws InterruptedException {

	if (node instanceof EDataSlot) {
	    return (node);
	} else if (node instanceof EValueSlot) {
	    return (node);
	} else if (node instanceof EContentRefNode) {
	    EContentRefNode cRef = (EContentRefNode)node;
	    EContentNode cNode = cRef.getContentNode();
	    cNode.lockRead();
	    try {
		return (_resolveDataContentNode(cNode));
	    } finally {
		cNode.unlockRead();
	    }
	} else if (node instanceof EChoiceNode) {
	    return (_resolveDataChoice(node));
	} else if (node instanceof ESequenceNode) {
	    return (_resolveDataList(node));
	} else {
	    throw (new InternalError(node.toString()));
	}
    }

    private ENode _resolveDataContentNode(ENode node)
	throws InterruptedException {

	ENode[] children = node.getChildren();
	if (children.length == 0) {
	    throw (new InternalError());
	}
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EValueSlot ||
		child instanceof EDataSlot) {

		list.add(child);
	    } else if (child instanceof EChoiceNode ||
		       child instanceof EContentRefNode) {

		ENode result = _resolveDataNode(child);
		if (result instanceof EChoiceNode) {
		    list.addAll(Arrays.asList(result.getChildren()));
		} else {
		    list.add(result);
		}
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	switch (list.size()) {

	case 0:
	    throw (new InternalError());
	case 1:
	    return ((ENode)list.get(0));
	default:
	    throw (new InternalError());
	}
    }

    private ENode _resolveDataChoice(ENode node)
	throws InterruptedException {

	ENode[] children = node.getChildren();
	if (children.length == 0) {
	    throw (new InternalError());
	}
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EValueSlot ||
		child instanceof EDataSlot) {

		list.add(child);
	    } else if (child instanceof EChoiceNode ||
		       child instanceof EContentRefNode) {

		ENode result = _resolveDataNode(child);
		if (result instanceof EChoiceNode) {
		    list.addAll(Arrays.asList(result.getChildren()));
		} else {
		    list.add(result);
		}
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
	switch (list.size()) {

	case 0:
	    throw (new InternalError());
	case 1:
	    return ((ENode)list.get(0));
	default:
	    EChoiceNode choice = new EChoiceNode();
	    choice.addChildren(list);
	    return (choice);
	}
    }

    private ENode _resolveDataList(ENode node)
	throws InterruptedException {

	ENode[] children = node.getChildren();
	if (children.length == 0) {
	    throw (new InternalError());
	} else if (children.length == 1) {
	    ENode child = children[0];
	    ESequenceNode seq = (ESequenceNode)node;
	    switch (seq.getOccurs()) {

	    case OCCURS_ONE:
		return (_resolveDataNode(child));
	    case OCCURS_ZEROONE:
		return (node); // XXX
	    case OCCURS_ONEMORE: // continue
	    case OCCURS_ZEROMORE:
		return (_resolveDataListMore(seq, child));
	    default:
		throw (new InternalError());
	    }
	} else {
	    ENode child = children[0];
	    if (!(child instanceof EDataSlot)) {
		return (node);
	    }
	    EDataSlot dataSlot = (EDataSlot)child;
	    String typeName = dataSlot.getDatatype().getName();
	    for (int i = 1;i < children.length;i++) {
		child = children[i];
		if (!(child instanceof EDataSlot)) {
		    return (node);
		}
		dataSlot = (EDataSlot)child;
		if (!typeName.equals(dataSlot.getDatatype().getName())) {
		    return (node);
		}
	    }
	    Datatype datatype = DatatypeFactory.getDatatype(typeName);
	    datatype.setList(true);
	    datatype.setProperty(node);
	    return (new EDataSlot(datatype));
	}
    }

    private ENode _resolveDataListMore(ESequenceNode seq, ENode child)
	throws InterruptedException {

	ENode node = _resolveDataNode(child);
	if (node instanceof EDataSlot) {
	    EDataSlot dataSlot = (EDataSlot)node;
	    Datatype datatype = DatatypeFactory.getDatatype(
		(Datatype)dataSlot.getDatatype(),
		true
	    );
	    datatype.setProperty(seq);
	    return (new EDataSlot(datatype));
	} else {
	    return (seq);
	}
    }
*/

/*
    private IXMLDatatype _adjustComplexMore(DComplex complex) {
	ENode node = (ENode)complex.getProperty();
	if (node instanceof EDataSlot) {
	    EDataSlot data = (EDataSlot)node;
	    return (_makeDatatype(data));
	} else if (node instanceof EValueSlot) {
	    EValueSlot value = (EValueSlot)node;
	    return (_makeDatatype(value));
	} else if (node instanceof EContentRefNode) {
	    throw (new InternalError());
	} else if (node instanceof EChoiceNode) {
	    ENode[] children = node.getChildren();
	    if (children.length == 0) {
		throw (new InternalError());
	    }
	    ENode child = children[0];
	    if (!(child instanceof EValueSlot)) {
		return (complex);
	    }
	    XMLFacet[] facets = new XMLFacet[children.length];
	    EValueSlot value = (EValueSlot)child;
	    String typeName = value.getType();
	    facets[0] = new XMLFacet("enumeration", value.getText());
	    for (int i = 1;i < children.length;i++) {
		child = children[i];
		if (!(child instanceof EValueSlot)) {
		    return (complex);
		}
		value = (EValueSlot)child;
		if (!typeName.equals(value.getType())) {
		    return (complex);
		}
		facets[i] = new XMLFacet("enumeration", value.getText());
	    }
	    Datatype datatype = DatatypeFactory.getDatatype(typeName);
	    datatype.addFacets(facets);
	    return (datatype);
	} else {
	    throw (new InternalError());
	}
    }
*/

/*
    private ENode _adjustExprTree(ENode node) throws InterruptedException {
	if (node instanceof EDataSlot) {
	    return (_adjustExprTreeData((EDataSlot)node));
	} else if (node instanceof EValueSlot) {
	    return (_adjustExprTreeValue((EValueSlot)node));
	} else if (node instanceof EContentRefNode) {
	    return (_adjustExprTreeRef((EContentRefNode)node));
	} else if (node instanceof ESequenceNode) {
	    return (_adjustExprTreeList((ESequenceNode)node));
	} else if (node instanceof EChoiceNode) {
	    return (_adjustExprTreeChoice((EChoiceNode)node));
	} else {
	    throw (new InternalError());
	} 
    }

    private ENode _adjustExprTreeData(EDataSlot data) {
	return (data);
    }

    private ENode _adjustExprTreeValue(EValueSlot value) {
	return (value);
    }

    private ENode _adjustExprTreeRef(EContentRefNode ref)
	throws InterruptedException {

	EContentNode cNode = ref.getContentNode();
	cNode.lockRead();
	try {
	    ENode[] children = cNode.getChildren();
	    if (children.length != 1) {
		throw (new InternalError(ref.toString()));
	    }
	    return (_deepCopy(children[0]));
	} finally {
	    cNode.unlockRead();
	}
    }

    private ENode _adjustExprTreeList(ESequenceNode textList)
	throws InterruptedException {

	int occurs = textList.getOccurs();
	List list = _makeTextChildrenList(textList);
	int size = list.size();
	if (size == 0) {
	    throw (new InternalError(textList.toString()));
	} else if (size == 1 &&
		   (occurs == OCCURS_ONE ||
		    occurs == OCCURS_ZEROONE)) {
	    return ((ENode)list.get(0));
	} else {	// XXX : EListNode
	    ESequenceNode result = new ESequenceNode(occurs);
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeChoice(EChoiceNode choice)
	throws InterruptedException {

	List list = _makeTextChildrenList(choice);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	case 1:
	    return ((ENode)list.get(0));
	default:
	    EChoiceNode result = new EChoiceNode();
	    result.addChildren(list);
	    return (result);
	}
    }

    private List _makeTextChildrenList(ENode parent)
	throws InterruptedException {

	ENode[] children = parent.getChildren();
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    list.add(_adjustExprTree(child));
	}
	return (list);
    }

    private ENode _adjustExprTree(Element node) throws InterruptedException {
	String localName = node.getLocalName();
	if ("data".equals(localName)) {
	    return (_adjustExprTreeData(node));
	} else if ("value".equals(localName)) {
	    return (_adjustExprTreeValue(node));
	} else if ("ref".equals(localName)) {
	    return (_adjustExprTreeRef(node));
	} else if ("list".equals(localName)) {
	    return (_adjustExprTreeList(node));
	} else if ("choice".equals(localName)) {
	    return (_adjustExprTreeChoice(node));
	} else {
	    throw (new InternalError(localName));
	}
    }

    private ENode _adjustExprTreeData(Element data) {
	String typeName = data.getAttribute("type");
	Datatype datatype = DatatypeFactory.getDatatype(typeName);
	XMLFacet[] facets = _makeRELAXNGFacets(data);
	if (facets != null) {
	    datatype.addFacets(facets);
	}
	EDataSlot slot = new EDataSlot(datatype);
	return (slot);
    }

    private XMLFacet[] _makeRELAXNGFacets(Element data) {
	Element[] params = UElement.getElements(
	    data,
	    "http://relaxng.org/ns/structure/1.0",
	    "param"
	);
	XMLFacet[] facets = new XMLFacet[params.length];
	for (int i = 0;i < params.length;i++) {
	    Element param = params[i];
	    String value = UXML.element2Data(param);
	    String name = param.getAttribute("name");
	    facets[i] = new XMLFacet(name, value);
	}
	return (facets);
    }

    private ENode _adjustExprTreeValue(Element value) {
	String text = UXML.element2Data(value);
	String typeName = value.getAttribute("type");
	if (typeName != null && !"".equals(typeName)) {
	    return (new EValueSlot(text, typeName));
	} else {
	    return (new EValueSlot(text));
	}
    }

    private ENode _adjustExprTreeRef(Element ref)
	throws InterruptedException {

	String refName = ref.getAttribute("name");
	return (_adjustExprTreeRef(refName));
    }

    private ENode _adjustExprTreeRef(String refName)
	throws InterruptedException {

//System.out.println("refName = " + refName);
	EContentNode[] cNodes = module_.resolveContentLabel(refName);
	if (cNodes.length != 1) {
	    throw (new InternalError());
	}
	EContentNode cNode = cNodes[0];
	cNode.lockRead();
	try {
	    ENode[] children = cNode.getChildren();
	    if (children.length != 1) {
		throw (new InternalError());
	    }
	    return (_deepCopy(children[0]));
	} finally {
	    cNode.unlockRead();
	}
    }

    private ENode _adjustExprTreeList(Element textList)
	throws InterruptedException {

	List list = _makeTextChildrenList(textList);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	case 1:
	    return ((ENode)list.get(0));
	default:
	    ESequenceNode result = new ESequenceNode();	// XXX : EListNode
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeChoice(Element choice)
	throws InterruptedException {

	List list = _makeTextChildrenList(choice);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	case 1:
	    return ((ENode)list.get(0));
	default:
	    EChoiceNode result = new EChoiceNode();
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeOptional(Element optional)
	throws InterruptedException {

	List list = _makeTextChildrenList(optional);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	default:
	    ESequenceNode result = new ESequenceNode(OCCURS_ZEROONE);
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeOneOrMore(Element oneOrMore)
	throws InterruptedException {

	List list = _makeTextChildrenList(oneOrMore);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	default:
	    ESequenceNode result = new ESequenceNode(OCCURS_ONEMORE);
	    result.addChildren(list);
	    return (result);
	}
    }

    private ENode _adjustExprTreeZeroOrMore(Element zeroOrMore)
	throws InterruptedException {

	List list = _makeTextChildrenList(zeroOrMore);
	switch (list.size()) {

	case 0:
	    throw (new UnsupportedOperationException());
	default:
	    ESequenceNode result = new ESequenceNode(OCCURS_ZEROMORE);
	    result.addChildren(list);
	    return (result);
	}
    }

    private List _makeTextChildrenList(Element parent)
	throws InterruptedException {

	Element[] children = UElement.getElements(parent);
	List list = new ArrayList();
	for (int i = 0;i < children.length;i++) {
	    Element child = children[i];
	    String localName = child.getLocalName();
	    if ("data".equals(localName)) {
		list.add(_adjustExprTreeData(child));
	    } else if ("value".equals(localName)) {
		list.add(_adjustExprTreeValue(child));
	    } else if ("ref".equals(localName)) {
		list.add(_adjustExprTreeRef(child));
	    } else if ("list".equals(localName)) {
		list.add(_adjustExprTreeList(child));
	    } else if ("optional".equals(localName)) {
		list.add(_adjustExprTreeOptional(child));
	    } else if ("oneOrMore".equals(localName)) {
		list.add(_adjustExprTreeOneOrMore(child));
	    } else if ("zeroOrMore".equals(localName)) {
		list.add(_adjustExprTreeZeroOrMore(child));
	    } else if ("choice".equals(localName)) {
		list.add(_adjustExprTreeChoice(child));
	    } else {
		throw (new UnsupportedOperationException());
	    }
	}
	return (list);
    }
*/

    private int _getContentRefNodeOccurs(ENode node) {
	if (node instanceof EContentRefNode) {
	    EContentRefNode cRef = (EContentRefNode)node;
	    return (cRef.getOccurs());
	} else if (node instanceof EExternalContentRefNode) {
	    EExternalContentRefNode cRef = (EExternalContentRefNode)node;
	    return (cRef.getOccurs());
	} else {
	    throw (new InternalError(node.toString()));
	}
    }

    // post optimization process

    private void _adjustZeroableContentRefs(ENode node)
	throws InterruptedException {

	if (node instanceof EContentRefNode) {
	    EContentRefNode cRef = (EContentRefNode)node;
	    cRef.setOccurs(_adjustContentRefNodeOccurs(cRef));
	} else if (node instanceof EExternalContentRefNode) {
	    EExternalContentRefNode cRef = (EExternalContentRefNode)node;
	    cRef.setOccurs(_adjustContentRefNodeOccurs(cRef));
	}
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    _adjustZeroableContentRefs(children[i]);
	}
    }

    private int _adjustContentRefNodeOccurs(ENode node)
	throws InterruptedException {

	int occurs;
	EContentNode cNode;
	if (node instanceof EContentRefNode) {
	    EContentRefNode cRef = (EContentRefNode)node;
	    occurs = cRef.getOccurs();
	    cNode = cRef.getContentNode();
	} else if (node instanceof EExternalContentRefNode) {
	    EExternalContentRefNode cRef = (EExternalContentRefNode)node;
	    occurs = cRef.getOccurs();
	    cNode = cRef.getContentNode();
	} else {
	    throw (new InternalError(node.toString()));
	}
	return (_adjustContentNodeOccurs(cNode, occurs));
    }

    private int _adjustContentNodeOccurs(EContentNode cNode, int occurs)
	throws InterruptedException {

	if (!_isZeroableContent(cNode)) {
		return (occurs);
	}
	if (occurs == OCCURS_ONE) {
	    return (OCCURS_ZEROONE);
	} else if (occurs == OCCURS_ONEMORE) {
	    return (OCCURS_ZEROMORE);
	} else {
	    return (occurs);
	}
    }

    private boolean _isZeroableContent(EContentNode cNode)
	throws InterruptedException {

	cNode.lockRead();
	try {
	    ENode[] children = cNode.getChildren();
	    for (int i = 0;i < children.length;i++) {
		ENode child = children[i];
		int occurs;
		if (child instanceof EElementSlot) {
		    EElementSlot eSlot = (EElementSlot)child;
		    occurs = eSlot.getOccurs();
		} else if (child instanceof EAttributeSlot) {
		    EAttributeSlot aSlot = (EAttributeSlot)child;
		    occurs = aSlot.isRequired()?OCCURS_ONE:OCCURS_ZEROONE;
		} else if (child instanceof EElementRefNode) {
		    EElementRefNode eRef = (EElementRefNode)child;
		    occurs = eRef.getOccurs();
		} else if (child instanceof EContentRefNode) {
		    EContentRefNode cRef = (EContentRefNode)child;
		    occurs = cRef.getOccurs();
		} else if (child instanceof EExternalRefNode) {
		    EExternalRefNode eRef = (EExternalRefNode)child;
		    occurs = eRef.getOccurs();
		} else if (child instanceof EExternalContentRefNode) {
		    EExternalContentRefNode cRef
			= (EExternalContentRefNode)child;
		    occurs = cRef.getOccurs();
		} else if (child instanceof EInterleaveNode) {
		    EInterleaveNode interleave = (EInterleaveNode)child;
		    occurs = interleave.getOccurs();
		} else if (child instanceof EChoiceNode) {
		    EChoiceNode choice = (EChoiceNode)child;
		    occurs = choice.getOccurs();
		} else if (child instanceof EMixedNode) {
		    EMixedNode mixed = (EMixedNode)child;
		    continue;
		} else if (child instanceof EAnyOtherAttributeNode) {
		    // XXX
		    continue;
		} else if (child instanceof EAnyOtherElementNode) {
		    // XXX
		    continue;
		} else if (child instanceof EAttributeRefNode) {
		    EAttributeRefNode aRef = (EAttributeRefNode)child;
		    occurs = aRef.getOccurs();
		} else {
		    throw (new InternalError(child.toString()));
		}
		if (occurs == OCCURS_ONE ||
		    occurs == OCCURS_ONEMORE) {

		    return (false);
		}
	    }
	} finally {
	    cNode.unlockRead();
	}
	return (true);
    }

    //
    private void _syntaxError() {
	throw (new InternalError());
    }

    private void _assertValidCRefNode(ENode node) {
	if (node instanceof EContentRefNode) {
	    EContentRefNode cRef = (EContentRefNode)node;
	    _assertValidCNode(cRef.getContentNode());
	} else if (node instanceof EExternalContentRefNode) {
	    EExternalContentRefNode cRef = (EExternalContentRefNode)node;
	    _assertValidCNode(cRef.getContentNode());
	} else {
	    throw (new InternalError(node.toString()));
	}
    }

    private void _assertValidCNode(ENode node) {
	if (!(node instanceof EContentNode)) {
	    throw (new InternalError(node.toString()));
	}
	ENode[] children = node.getChildren();
	if (children.length > 1) {
	    for (int i = 0;i < children.length;i++) {
		if (children[i] instanceof EDataSlot) {
//System.out.println(node);		    
//		    throw (new InternalError(node.toString()));
		}
	    }
	}
    }

    private void _printContentRefs(ENode node, String label) {
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EContentRefNode) {
		EContentRefNode cRef = (EContentRefNode)child;
		System.out.println(label + "\t" + cRef.getContentNode());
	    } else {
		_printContentRefs(child, label);
	    }
	}
    }

    private void _assertInterleaveInElement(EInterleaveNode interleave) {
	ENode[] children = interleave.getChildren();
	for (int i = 0;i < children.length;i++) {
	    ENode child = children[i];
	    if (child instanceof EElementSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAttributeSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EElementRefNode) {
		// do nothing
	    } else if (child instanceof EContentRefNode) {
		// do nothing
	    } else if (child instanceof EExternalRefNode) {
		// do nothing
	    } else if (child instanceof EExternalContentRefNode) {
		// do nothing
	    } else if (child instanceof ESequenceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EInterleaveNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EChoiceNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EMixedNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EAnyOtherElementNode) {
		// do nothing
	    } else if (child instanceof EAttributeRefNode) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EDataSlot) {
		throw (new InternalError(child.toString()));
	    } else if (child instanceof EValueSlot) {
		throw (new InternalError(child.toString()));
	    } else {
		throw (new InternalError(child.toString()));
	    }
	}
    }	

    private void _dumpHedge(ENode node) {
	System.out.println("dump(" + Thread.currentThread() +  "):" + node);
	ENode[] children = node.getChildren();
	for (int i = 0;i < children.length;i++) {
	    System.out.println(children[i]);
	}
    }

    //
    private static ElementCounter nameCounter__ = new ElementCounter();

    private static String _makeUniqueName(String base) {
	nameCounter__.add(base);
	return (UString.uncapitalize(base + nameCounter__.getCount(base)));
    }

/*
    //
    private static int seqNo__ = 0;

    private static int _getSequenceNo() {
	return (++seqNo__);
    }
*/
    private static void _printLog(String message) {
//	System.out.println(message);
    }

    private static boolean debug__ = false;
}

/*
	if (child instanceof EElementSlot) {
	} else if (child instanceof EAttributeSlot) {
	} else if (child instanceof EElementRefNode) {
	} else if (child instanceof EContentRefNode) {
	} else if (child instanceof EExternalRefNode) {
	} else if (child instanceof EExternalContentRefNode) {
	} else if (child instanceof ESequenceNode) {
	} else if (child instanceof EInterleaveNode) {
	} else if (child instanceof EChoiceNode) {
	} else if (child instanceof EMixedNode) {
	} else if (child instanceof EAnyOtherElementNode) {
	} else if (child instanceof EAnyOtherAttributeNode) {
	} else if (child instanceof EAttributeRefNode) {
	} else if (child instanceof EDataSlot) {
	} else if (child instanceof EValueSlot) {
	} else {
	    throw (new InternalError(child.toString()));
	}
 */
