/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2000  ASAMI, Tomoharu (tasami@ibm.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.vfs;

import java.util.*;
import java.io.*;

/**
 * AbstractVFSRealm
 *
 * @since   Dec. 18, 1998
 * @version Feb.  8, 2000
 * @author  ASAMI, Tomoharu (tasami@ibm.net)
 */
public abstract class AbstractVFSRealm implements IVFSRealm {
    private static final int SETUP_NONE = 1;
    private static final int SETUP_SHALLOW = 2;
    private static final int SETUP_DEEP = 3;
    private IVFSRoot root_;
    private IFilter filter_;
    private int status_ = SETUP_NONE;

    protected void _shallowSetup(
	IVFSRoot root,
	IFilter filter
    ) throws IOException {
	if (status_ == SETUP_SHALLOW || status_ == SETUP_DEEP) {
	    return;
	}
	root.setRealm(this);
	root.shallowSetup(filter);
	filter_ = filter;
	root_ = root;
	status_ = SETUP_SHALLOW;
    }

    protected void _deepSetup(
	IVFSRoot root,
	IFilter filter
    ) throws IOException {
	if (status_ == SETUP_DEEP) {
	    return;
	}
	root.setRealm(this);
	root.deepSetup(filter);
	filter_ = filter;
	root_ = root;
	status_ = SETUP_DEEP;
    }

    protected void _assertDeepSetup() {
	if (!(status_ == SETUP_DEEP)) {
	    throw (new InternalError());
	}
    }

    protected void _assertShallowOrDeepSetup() {
	if (!(status_ == SETUP_DEEP || status_ == SETUP_SHALLOW)) {
	    throw (new InternalError());
	}
    }

    public void flush() throws IOException {
	// do nothing
    }

    public long getLastTimestamp() throws IOException {
	return (System.currentTimeMillis());
    }

    public IVFSRoot getRoot() {
	return (root_);
    }

    public String[] getTargetPathnames() {
	_assertDeepSetup();
	List list = new ArrayList();
	IVFSObject[] objects = root_.getObjectChildren();
	for (int i = 0;i < objects.length;i++) {
	    _collectValidPathnames(objects[i], list);
	}
	IVFSContainer[] containers = root_.getContainerChildren();
	for (int i = 0;i < containers.length;i++) {
	    _collectValidPathnames(containers[i], list);
	}
	String[] paths = new String[list.size()];
	return ((String[])list.toArray(paths));
    }

    protected void _collectValidPathnames(
	IVFSNode node,
	List list
    ) {
	if (node instanceof IVFSContainer) {
	    IVFSContainer container = (IVFSContainer)node;
	    if (container.isValid()) {
		list.add(container.getPathname());
	    }
	    IVFSObject[] objects = container.getObjectChildren();
	    for (int i = 0;i < objects.length;i++) {
		_collectValidPathnames(objects[i], list);
	    }
	    IVFSContainer[] containers
		= container.getContainerChildren();
	    for (int i = 0;i < containers.length;i++) {
		_collectValidPathnames(containers[i], list);
	    }
	} else if (node instanceof IVFSObject) {
	    if (node.isValid()) {
		list.add(node.getPathname());
	    }
	} else {
	    throw (new InternalError());
	}
    }

    public IVFSObject[] getTargetObjects() {
	_assertDeepSetup();
	List list = new ArrayList();
	IVFSObject[] objects = root_.getObjectChildren();
	for (int i = 0;i < objects.length;i++) {
	    _collectValidObjects(objects[i], list);
	}
	IVFSContainer[] containers = root_.getContainerChildren();
	for (int i = 0;i < containers.length;i++) {
	    _collectValidObjects(containers[i], list);
	}
	IVFSObject[] paths = new IVFSObject[list.size()];
	return ((IVFSObject[])list.toArray(paths));
    }

    protected void _collectValidObjects(
	IVFSNode node,
	List list
    ) {
	if (node instanceof IVFSContainer) {
	    IVFSContainer container = (IVFSContainer)node;
	    if (container.isValid()) {
		list.add(container);
	    }
	    IVFSObject[] objects = container.getObjectChildren();
	    for (int i = 0;i < objects.length;i++) {
		_collectValidObjects(objects[i], list);
	    }
	    IVFSContainer[] containers
		= container.getContainerChildren();
	    for (int i = 0;i < containers.length;i++) {
		_collectValidObjects(containers[i], list);
	    }
	} else if (node instanceof IVFSObject) {
	    if (node.isValid()) {
		list.add(node);
	    }
	} else {
	    throw (new InternalError());
	}
    }
}
