/*****************************************************************************
*
* File: PrecisionTimer.cpp
* Project: Osmose emulator.
*
* Description: This class handle accurate timer measure, using rdtsc pentium
* instruction. This code compile with gcc, under Linux, and windows. See in
* PrecisionTimer.h for more information about Sleep(millis)/sleep(sec) issues.
*
* Author: Vedder Bruno
* Date: 19/04/2004
* URL: http://bcz.emu-france.com/
*****************************************************************************/
#include "PrecisionTimer.h"

#define rdtsc(val) __asm__ __volatile__ ("rdtsc" : "=A" (val))

/*--------------------------------------------------------------------*/
/* Default Constructor initialise values to -1;                       */
/*--------------------------------------------------------------------*/
PrecisionTimer::PrecisionTimer()
{
    T0 = (unsigned long long)-1;
    T1 = (unsigned long long)-1;
    tick_per_second = (unsigned long long)-1;
    mode = ONE_SHOT;
    values_number = VALUES_BUFFER_SIZE;
    cumulative_buffer = NULL;
    cumul_buffer_index = 0;
    average_duration = 0.0;
    verbose = true;
}

/*--------------------------------------------------------------------*/
/* Calibration Process. s is in second. The longer is the calibration */
/* more accurate is the measure.                                      */
/*--------------------------------------------------------------------*/
void PrecisionTimer::calibrate(int s)
{
    unsigned long long start;
    unsigned long long stop;
    if (verbose)
    {    
	cout << "Calibrating:" << dec<< s << " seconds." << endl;
    }	
    rdtsc(start);
    PORTABLE_SLEEP_FUNC;
    rdtsc(stop);
    tick_per_second = (unsigned long long) ( stop - start );
    tick_per_second = (tick_per_second / s);
    if (verbose)
    {    
	cout << "tick_per_second is: " << dec << tick_per_second << endl;
	cout << "CPU seems to run at " << dec <<(tick_per_second / 1000000.0) << "Mhz." << endl;
    }
}

/*--------------------------------------------------------------------*/
/* Execute rdtsc pentium instruction store result into T0.            */
/*--------------------------------------------------------------------*/
void PrecisionTimer::start()
{
    rdtsc(T0);
}

/*--------------------------------------------------------------------*/
/* Execute rdtsc pentium instruction store result into T1.            */
/* This method returns true if a value is available from measure:     */
/* -Always true in ONE_SHOT mode                                      */
/* -True when CUMULATIVE mode has enought value to compute average    */
/*  duration else false.                                              */
/*--------------------------------------------------------------------*/
bool PrecisionTimer::stop()
{
    bool ret = false;
    
    rdtsc(T1);
    if (mode == CUMULATIVE)
    {
         cumulative_buffer[cumul_buffer_index] = getOneShotDuration();
	 cumul_buffer_index++;
	 
	// We have enough values: display average length.
	if (cumul_buffer_index == values_number)
	{
	    double total_duration = 0.0;
	    
	    for (unsigned int i =0; i < values_number; i++)
	    {
	        total_duration += cumulative_buffer[i];
	    }
	    
	    average_duration = total_duration / values_number;
	    if (verbose)
	    {
	        cout << "Average length = " << dec << average_duration << " ("<< dec << values_number << " values)." <<endl;
	    }
	    cumul_buffer_index = 0;
	    ret = true;
	}   
    }
    else
    {
        ret = true;
    }

    return ret;
}

/*--------------------------------------------------------------------*/
/* Compute tick delta between T1 and T0.                              */
/*--------------------------------------------------------------------*/
unsigned long long PrecisionTimer::getDeltaT1_T0()
{
    unsigned long long x;
    x = (unsigned long long) ( T1 - T0 );
    return x;
}

/*--------------------------------------------------------------------*/
/* Return number of tick per second.                                  */
/* This values is computed during calibration. If returned value is   */
/* -1, this mean that calibrate() method has not been called.         */
/*--------------------------------------------------------------------*/
unsigned long long PrecisionTimer::getTickPerSecond()
{
    return tick_per_second;
}

/*--------------------------------------------------------------------*/
/* Return duration from T0 to T1. Unit is second.                     */
/*--------------------------------------------------------------------*/
double PrecisionTimer::getOneShotDuration()
{
    double x;
    double y;
    x = ( T1 - T0 );
    y = ( (double)x / (double)tick_per_second );
    return y;
}

/*--------------------------------------------------------------------*/
/* Return duration from T0 to T1. Unit is second in ONE_SHOT mode     */
/* Return last average lenght computed, in CUMULATIVE mode.           */
/*--------------------------------------------------------------------*/
double PrecisionTimer::getDuration()
{
    double v = -1.0;
    
    if (mode == ONE_SHOT)
    {
       v = getOneShotDuration();
    }
    
    if (mode == CUMULATIVE)
    {
        v = average_duration;
    }
    return v;
}


/*--------------------------------------------------------------------*/
/* This method selects measure mode:                                  */
/* ONE_SHOT is T1-T0, converted in seconds.                           */
/* CUMULATIVE value_number sums of T1-T0, divided by value number.    */
/* its an average measure, on values_number values.                   */
/*--------------------------------------------------------------------*/
void PrecisionTimer::setMeasureMode(int mod)
{
    if (mod == ONE_SHOT)
    {
        mode = ONE_SHOT;
    }
    else
    {
        mode = CUMULATIVE;
	if (cumulative_buffer == NULL)
	{
	    cumulative_buffer = new double [values_number]; 
	    if (cumulative_buffer == NULL)
	    {
	        cerr << "PrecisionTimer: Unable to allocate buffer for cumulative measure."<<endl;
	        cerr << "Exiting"<<endl;		   
		exit (-1);
            }
	}
    }
}

/*--------------------------------------------------------------------*/
/* This method reallocate if needed array containing duration values. */
/* This is usefull if you want to change number of measure for        */
/* cumulative mode.                                                   */
/*--------------------------------------------------------------------*/
void PrecisionTimer::setCumulativeBufferSize(unsigned int buf_size)
{
    if (cumulative_buffer != NULL)
    {
        delete[] cumulative_buffer;
    }

    cumulative_buffer = new double[buf_size]; 
    values_number = buf_size;
    if (cumulative_buffer == NULL)
    {
	cerr << "PrecisionTimer: Unable to allocate buffer for cumulative measure."<<endl;
	cerr << "Exiting"<<endl;		   
	exit (-1);
    }
}

/*--------------------------------------------------------------------*/
/* This reset values, array of values, and index in array of values.  */
/*--------------------------------------------------------------------*/
void PrecisionTimer::reset()
{
    T0 = 0;
    T1 = 0;
    if (cumulative_buffer != NULL)
    {
	for (unsigned int i =0; i < values_number; i++)
	{
	    cumulative_buffer[i]=0;
	}
    }
    cumul_buffer_index = 0;
}

/*--------------------------------------------------------------------*/
/* If verbose is false Precision timer will never print out things on */
/* console.                                                           */
/*--------------------------------------------------------------------*/
void PrecisionTimer::setVerbose(bool v)
{
    verbose = v;
}
