//
// CustomTerrain class : an generic implementation of vtDynTerrainGeom
//
// Copyright (c) 2001 Virtual Terrain Project
// Free for all uses, see license.txt for details.
//

#include "vtlib/vtlib.h"
#include "CustomTerrain.h"

//
// Macro used to determine the index of a vertex (element of the height
// field) given it's (x,y) location in the grid
//
#define offset(x, y)  ((y) * m_iXPoints + (x))

//
// Macros used to generate vertex locations from a heightfield index
// (You don't have to use them, especially for height)
//
#define LOCX(index) m_fXLookup[index % m_iDim]
#define LOCY(index)	m_pData[index]*m_fZScale
#define LOCZ(index) m_fZLookup[index / m_iDim]
#define MAKE_XYZ(index) LOCX(index), LOCY(index), LOCZ(index)
#define MAKE_XYZ2(x,y) m_fXLookup[x], m_pData[offset(x,y)]*m_fZScale, m_fZLookup[y]


//
// Constructor/destructor
//
CustomTerrain::CustomTerrain() : vtDynTerrainGeom()
{
	m_pData = NULL;
}

CustomTerrain::~CustomTerrain()
{
	if (m_pData)
		delete m_pData;
}


//
// Initialize the terrain data
// fZScale converts from height values (meters) to world coordinates
//
bool CustomTerrain::Init(vtLocalGrid *pGrid, float fZScale,
					 float fOceanDepth, int &iError)
{
	// Initializes necessary field of the parent class
	BasicInit(pGrid);

	//
	// allocate array, copy data from supplied elevation grid
	//
	// (replace this with your own storage representation)
	//
	m_pData = new float[m_iXPoints * m_iYPoints];
	int i, j;
	for (i = 0; i < m_iXPoints; i++)
	{
		for (j = 0; j < m_iYPoints; j++)
			m_pData[offset(i,j)] = pGrid->GetFValue(i, j);
	}

	m_fZScale = fZScale;
	m_iDrawnTriangles = -1;
	return true;
}


//
// This will be called once per frame, during the culling pass.
//
void CustomTerrain::DoCulling(FPoint3 &eyepos_ogl, IPoint2 window_size, float fov)
{
	// Do your visibility testing here.
	// (Compute which detail will actually gets drawn)

	// Here are some handy methods to test against the view frustum:
#if 0
	// Tests a sphere or triangle, and return one of:
	//	0				- not in view
	//  VT_Visible		- partly in view
	//  VT_AllVisible	- entirely in view
	//
	int IsVisible(const FSphere &sphere) const;
	int IsVisible(const FPoint3 &point0,
				  const FPoint3 &point1,
				  const FPoint3 &point2,
				  const float fTolerance = 0.0f) const;
	int IsVisible(FPoint3 point, float radius);

	// Tests a single point, returns true if in view
	bool IsVisible(const FPoint3 &point) const;
#endif
}


void CustomTerrain::DoRender()
{
	// Prepare the render state for our OpenGL usage
	PreRender();

	// Render the triangles
	RenderSurface();

	// Clean up
	PostRender();
}


void CustomTerrain::LoadSingleMaterial()
{
	// single texture for the whole terrain
	vtMaterial *pMat = GetMaterial(0);
	if (pMat)
	{
		pMat->Apply();
		SetupTexGen(1.0f);
//		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}
}


void CustomTerrain::RenderSurface()
{
	LoadSingleMaterial();
	RenderPass();

	// here is an example of how to do a second rendering pass
	if (m_bDetailTexture)
	{
		// once again, with the detail texture material
		m_pDetailMat->Apply();

		// the uv tiling is different (usually highly repetitive)
		SetupTexGen(m_fDetailTiling);

		// draw the second pass
		glPolygonOffset(-1.0f, -1.0f);
		glEnable(GL_POLYGON_OFFSET_FILL);
		RenderPass();
		glDisable(GL_POLYGON_OFFSET_FILL);
	}
	DisableTexGen();
}


void CustomTerrain::RenderPass()
{
	//
	// Very naive code which draws the grid as immediate-mode
	// triangle strips.  (Replace with your own algorithm.)
	//
	int i, j;
	for (i = 0; i < m_iXPoints-5; i+=4)
	{
		glBegin(GL_TRIANGLE_STRIP);
		for (j = 0; j < m_iYPoints; j+=4)
		{
			glVertex3f(MAKE_XYZ2(i, j));
			glVertex3f(MAKE_XYZ2(i+4, j));
			m_iDrawnTriangles += 2;
		}
		glEnd();
	}
}

//
// GetLocation is called when the framework needs to know the surface
// position of the terrain at a given grid point.  Supply the height
// value from your own data structures.
//
void CustomTerrain::GetLocation(int iX, int iZ, FPoint3 &p)
{
	p.Set(MAKE_XYZ2(iX, iZ));
}


