//
// NavEngines.cpp
//
// Copyright (c) 2001 Virtual Terrain Project
// Free for all uses, see license.txt for details.
//

#include "vtlib/vtlib.h"
#include "NavEngines.h"


vtFlyer::vtFlyer(float fSpeed, bool bPreventRoll) : vtLastMouse()
{
	m_fSpeed = fSpeed;
	m_bPreventRoll = bPreventRoll;
}

void vtFlyer::SetSpeed(float fSpeed)
{
	m_fSpeed = fSpeed;
}

float vtFlyer::GetSpeed()
{
	return m_fSpeed;
}

//
//  This engine flies the viewpoint around using the mouse position.
//
//	Left button: forward-backward, yaw
//  Right button: up-down, left-right
//  Both buttons: pitch, roll
//
//  Position is considered relative to the center of the window.
//
void vtFlyer::Eval()
{
	if (!(m_buttons & VT_LEFT) && !(m_buttons & VT_RIGHT))
		return;

	vtTransform *pTarget = (vtTransform*) GetTarget();
	if (!pTarget)
		return;

	vtScene* scene = vtGetScene();
	IPoint2	WinSize = scene->GetWindowSize();
	float	mx = (float) m_pos.x / WinSize.x;
	float	my = (float) m_pos.y / WinSize.y;

	//	Left button: forward-backward, yaw
	if ((m_buttons & VT_LEFT) && !(m_buttons & VT_RIGHT))
	{
		float trans = (my - 0.5f) * m_fSpeed;
		float rotate = -(mx - 0.5f) / 15.0f;

		pTarget->TranslateLocal(FPoint3(0.0f, 0.0f, trans));
		if (m_bPreventRoll)
			pTarget->RotateParent(FPoint3(0.0f, 1.0f, 0.0f), rotate);
		else
			pTarget->RotateLocal(FPoint3(0.0f, 1.0f, 0.0f), rotate);
	}

	//  Right button: up-down, left-right
	if ((m_buttons & VT_RIGHT) && !(m_buttons & VT_LEFT))
	{
		FPoint3 pos = pTarget->GetTrans();

		float updown = -(my - 0.5f) * m_fSpeed;
		float leftright = (mx - 0.5f) * m_fSpeed;

		pTarget->TranslateLocal(FPoint3(leftright, updown, 0.0f));
	}

	//  Both buttons: pitch, roll
	if ((m_buttons & VT_LEFT) && (m_buttons & VT_RIGHT))
	{
		float updown = -(my - 0.5f) / 20.0f;
		float leftright = (mx - 0.5f) / 20.0f;
		if (fabs(updown) > fabs(leftright))
			pTarget->RotateLocal(FPoint3(1.0f, 0.0f, 0.0f), updown);
		else if (!m_bPreventRoll)
			pTarget->RotateLocal(FPoint3(0.0f, 0.0f, 1.0f), -leftright);
	}
}


//////////////////////////////////////////////////////////////////
//
// Simplified Fly engine specifically for following terrain
//
// uses absolute Yaw
//
//#define GRAVITY_CONSTANT 9.81*WORLD_SCALE  //g = 9.81 meters/sec^2
#define GRAVITY_CONSTANT 209.81f*WORLD_SCALE

vtTerrainFlyer::vtTerrainFlyer(float fSpeed, float fHeightAboveTerrain, bool bMin) : vtFlyer(fSpeed, true)
{
	m_fHeightAboveTerrain = fHeightAboveTerrain;
	m_bMin = bMin;
	m_bFollow = true;
	m_pHeightField = NULL;
	m_bMaintain = false;
	m_fMaintainHeight = 0;
}

void vtTerrainFlyer::FollowTerrain(bool bFollow)
{
	m_bFollow = bFollow;
}

void vtTerrainFlyer::Eval()
{
	vtFlyer::Eval();
	KeepAboveGround();
}

//
// Keep the target above the the terrain surface.
//
void vtTerrainFlyer::KeepAboveGround()
{
	if (!m_pHeightField)
		return;

	vtTransform *pTarget = (vtTransform*) GetTarget();
	if (!pTarget)
		return;

	FPoint3 pos = pTarget->GetTrans();

	float fAltitude;
	bool bOverTerrain = m_pHeightField->FindAltitudeAtPoint(pos, fAltitude);

	if (bOverTerrain)
	{
		// set y value based on location
		if (m_bMaintain)
		{
			if (m_fMaintainHeight == 0)
				m_fMaintainHeight = pos.y - fAltitude;
			pos.y = fAltitude + m_fMaintainHeight;
		}
		else if (!m_bMin)
			pos.y = fAltitude + m_fHeightAboveTerrain;
		else
		{
			if (pos.y <= fAltitude + m_fHeightAboveTerrain)
				pos.y = fAltitude + m_fHeightAboveTerrain;
		}
		pTarget->SetTrans(pos);
	}
}


//////////////////////
//
// VFlyer
//

VFlyer::VFlyer(float scale, float fHeightAboveTerrain, bool bMin)
 : vtTerrainFlyer(0.4f, fHeightAboveTerrain, bMin)	// hardcode scale override
{
	m_fTime = -1;
	m_fVelocity = 0.0f;
	m_fFlyVelocity = 0.0f;
}

void VFlyer::Eval()
{
	vtTransform *pTarget = (vtTransform*) GetTarget();
	if (!pTarget)
		return;

	vtScene* scene = vtGetScene();
	IPoint2	WinSize = scene->GetWindowSize();
	float	mx = (float) m_pos.x / WinSize.x;
	float	my = (float) m_pos.y / WinSize.y;

	//	Left button: forward-backward, yaw
	if ((m_buttons & VT_LEFT) && !(m_buttons & VT_RIGHT))
	{
		float trans = (my - 0.5f) * m_fSpeed;
		float rotate = -(mx - 0.5f) / 15.0f;

		m_fFlyVelocity += trans;
		if (m_bPreventRoll)
			pTarget->RotateParent(FPoint3(0.0f, 1.0f, 0.0f), rotate);
		else
			pTarget->RotateLocal(FPoint3(0.0f, 1.0f, 0.0f), rotate);
	}

	//  Right button: up-down, left-right
	if ((m_buttons & VT_RIGHT) && !(m_buttons & VT_LEFT))
	{
		FPoint3 pos = pTarget->GetTrans();

		float updown = -(my - 0.5f) * m_fSpeed;
		float leftright = (mx - 0.5f) * m_fSpeed;

		pTarget->TranslateLocal(FPoint3(leftright, updown, 0.0f));
	}

	//  Both buttons: pitch, roll
	if ((m_buttons & VT_LEFT) && (m_buttons & VT_RIGHT))
	{
		float updown = -(my - 0.5f) / 20.0f;
		float leftright = (mx - 0.5f) / 20.0f;
		if (fabs(updown) > fabs(leftright))
			pTarget->RotateLocal(FPoint3(1.0f, 0.0f, 0.0f), updown);
		else if (!m_bPreventRoll)
			pTarget->RotateLocal(FPoint3(0.0f, 0.0f, 1.0f), -leftright);
	}

	m_fFlyVelocity *= 0.9f;
	pTarget->TranslateLocal(FPoint3(0.0f, 0.0f, m_fFlyVelocity));

	KeepAboveGround();
}

void VFlyer::AdjustUpwardVelocity(float velocity)
{
	m_fVelocity += velocity;
}


///////////////////////////////////////////////
// Quake-style navigation
//
QuakeFlyer::QuakeFlyer(float scale, float fHeightAboveTerrain, bool bMin)
 : vtTerrainFlyer(0.4f, fHeightAboveTerrain, bMin)	// hardcode scale override
{
	m_fTime = -1;
	m_fVelocity = 0;
	m_fFlyVelocity = 0;
	m_sWrap = 0;
	m_bNavEnable = true;
}

void QuakeFlyer::Eval()
{
#if 0
	if (!m_bNavEnable)
		return false;

	if (!m_bFollow)
		return vtFlyer::Eval(t);

	if (!m_pHeightField)
		return false;

	// Find the camera (or other target) that we are going to move
	vtTransform *pTarget = (vtTransform*) GetTarget();
	if (!pTarget) return false;

	// Find the window we are navigating in
	vtScene* scene = vtGetScene();
	IPoint2	WinSize = scene->GetWindowSize();

	// save mousepoint
	if (m_LastMousePt.x == 0 && m_LastMousePt.y == 0)
	{
		m_LastMousePt.x = MousePos.x;
		m_LastMousePt.y = MousePos.y;
	}

	// mouse movement
	float leftright = -(MousePos.x - m_LastMousePt.x) / WinSize.x;
	float updown = -(MousePos.y - m_LastMousePt.y) / WinSize.y;

	// stop moving if mouse stopped
	if (m_flastmovex == leftright && m_flastmovey == updown)
		return true;

	m_flastmovex = leftright;
	m_flastmovey = updown;

	// save mouse points
	m_LastMousePt.x = MousePos.x;
	m_LastMousePt.y = MousePos.y;

	// Major hack job... delay waits 3 eval calls so frame doesn't jump
	if (m_sWrap > 0)
	{
		m_sWrap++;
		if (m_sWrap >= 3)
			m_sWrap = 0;
		return true;
	}

	pTarget->Rotate(FPoint3(0,1,0), leftright);
	pTarget->Rotate(FPoint3(1,0,0), updown);

	//  Left button down
	if ((m_buttons & VT_LEFT) && !(m_buttons & VT_RIGHT))
	{
		dummy.Translate(0.0f, -0.1f, 0.0f);
	}

	//  Right button down
	if ((m_buttons & VT_RIGHT) && !(m_buttons & VT_LEFT))
	{
		dummy.Translate(0.0f, 0.1f, 0.0f);
	}

	// conform to terrain
	KeepAboveGround();
#endif
}

#if 0
// unimplemented
bool QuakeFlyer::OnKey(int32 key, int32 flags, int32 repeat, float speed)
{
	char da_key = (char) key;

	if (!m_pHeightField)
		return false;

	vtTransform *pTarget = (vtTransform*) GetTarget();
	if (!pTarget) return false;

	switch (da_key)
	{
		case 't': // toggle
			m_bNavEnable = !m_bNavEnable;
			break;

		// flyer
		case 'w':  // forward
			MoveMatrix(dummy, 0, 0, -1 * speed);
			break;
		case 's':  // back
			MoveMatrix(dummy, 0, 0, speed);
			break;
		case 'a': // left
			MoveMatrix(dummy, -1 * speed, 0, 0);
			break;
		case 'd': // right
			MoveMatrix(dummy, speed, 0, 0);
			break;

		// shift held down
		case 'W':  // forward
			MoveMatrix(dummy, 0, 0, -3 * speed);
			break;
		case 'S':  // back
			MoveMatrix(dummy, 0, 0, 3 * speed);
			break;
		case 'A': // left
			MoveMatrix(dummy, -3 * speed, 0, 0);
			break;
		case 'D': // right
			MoveMatrix(dummy, 3 * speed, 0, 0);
			break;
	}


	switch (key)
	{
		// flyer

		case 294:  // forward
			MoveMatrix(dummy, 0, 0, -1 * speed);
			break;
		case 296:  // back
			MoveMatrix(dummy, 0, 0, speed);
			break;
		case 293: // left
			MoveMatrix(dummy, -1 * speed, 0, 0);
			break;
		case 295: // right
			MoveMatrix(dummy, speed, 0, 0);
			break;
	}

	// conform to terrain
	FPoint3 pos = GetTranslation(&dummy);

	float fAltitude;

	bool bOverTerrain = m_pHeightField->FindAltitudeAtPoint(pos, fAltitude);

	if (bOverTerrain)
	{
		// set y value based on location
		if (!m_bMin)
			pos.y = fAltitude + m_fHeightAboveTerrain;
		else
		{
			if (pos.y <= fAltitude + m_fHeightAboveTerrain)
				pos.y = fAltitude + m_fHeightAboveTerrain;
		}
		SetTranslation(&dummy, pos);
	}
	pTarget->SetTransform(&dummy);

	return true;
}
#endif


///////////////////////////////////////

void Trackball2::OnMouse(vtMouseEvent &event)
{
	if (event.type == VT_DOWN && event.button == VT_MIDDLE)
	{
		if (!m_bDrag)
		{
			m_Start = m_Pos;
			m_MouseStart = event.pos;
			m_bDrag = true;
			return;
		}
	}
	if (event.type == VT_MOVE && m_bDrag)
	{
		if (event.flags & VT_MF_SHIFT)
			m_Pos.z = m_Start.z - (event.pos.y - m_MouseStart.y) / 200.0f;
		else
		{
			m_Pos.x = m_Start.x + (event.pos.x - m_MouseStart.x) / 200.0f;
			m_Pos.y = m_Start.y + (event.pos.y - m_MouseStart.y) / 200.0f;
		}
	}
	if (event.type == VT_UP && event.button == VT_MIDDLE)
		m_bDrag = false;
}

void Trackball2::Eval()
{
	vtTransform *pTarget = (vtTransform *) GetTarget();
	if (!pTarget)
		return;

	pTarget->Identity();
	pTarget->Translate1(FPoint3(0.0, 0.0, m_Pos.z));
	pTarget->Rotate2(FPoint3(1.0f, 0.0f, 0.0f), -m_Pos.y);
	pTarget->Rotate2(FPoint3(0.0f, 1.0f, 0.0f), -m_Pos.x);
}


