/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: BottomUpPlace.java
 * Written by Steven M. Rubin
 *
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 */
package com.sun.electric.tool.placement.general;

import com.sun.electric.database.geometry.EGraphics;
import com.sun.electric.database.geometry.EPoint;
import com.sun.electric.database.geometry.ERectangle;
import com.sun.electric.database.geometry.Poly;
import com.sun.electric.database.geometry.PolyBase;
import com.sun.electric.database.geometry.PolyMerge;
import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.database.hierarchy.Library;
import com.sun.electric.database.prototype.NodeProto;
import com.sun.electric.database.topology.ArcInst;
import com.sun.electric.database.topology.NodeInst;
import com.sun.electric.database.topology.RTBounds;
import com.sun.electric.database.topology.RTNode;
import com.sun.electric.database.variable.TextDescriptor;
import com.sun.electric.technology.PrimitiveNode;
import com.sun.electric.technology.technologies.Artwork;
import com.sun.electric.technology.technologies.Generic;
import com.sun.electric.tool.Job;
import com.sun.electric.tool.placement.PlacementAdapter;
import com.sun.electric.tool.placement.PlacementAdapter.PlacementConnection;
import com.sun.electric.tool.placement.PlacementFrame;
import com.sun.electric.tool.placement.general.BottomUpPartition.PNPair;
import com.sun.electric.tool.simulation.test.TextUtils;
import com.sun.electric.util.math.DBMath;
import com.sun.electric.util.math.Orientation;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * Placement algorithms that does bottom-up placement.
 *
 * TO-DO list:
 *   Tune RotationTest.isBetter()
 */
public class BottomUpPlace extends PlacementFrame
{
	private static List<List<PNPair>> orderedPairGroups;
	private static Map<PlacementNode,List<PlacementConnection>> consOnNodes;

//	private final static boolean DEBUGMODE = false;
	private final static boolean DEBUGPROGRESS = false;
	private final static boolean DEBUGPLOW = false;

	private final static int GROUPPAIRSBYDENSITY = 0;
	private final static int GROUPPAIRSBYSIZE = 1;
	private final static int GROUPPAIRSFLAT = 2;
	private static int pairGrouping = GROUPPAIRSBYDENSITY;

	private static double boundWeight = 2, aspectRatioWeight = 2;
	private static int trellisWidth = 20;

	protected PlacementParameter numThreadsParam = new PlacementParameter("threads",
		"Number of threads:", 4);
	protected PlacementParameter maxRuntimeParam = new PlacementParameter("runtime",
		"Runtime (in seconds, 0 for no limit):", 240);
	protected PlacementParameter canRotate = new PlacementParameter("canRotate",
		"Allow node rotation", false);

	/**
	 * Method to return the name of this placement algorithm.
	 * @return the name of this placement algorithm.
	 */
	public String getAlgorithmName() { return "Bottom-Up-Place"; }

	public static void setWeights(double b, double ar, int tr)
	{
		boundWeight = b;
		aspectRatioWeight = ar;
		trellisWidth = tr;
	}

	/**
	 * Method to do placement.
	 * @param placementNodes a list of all nodes that are to be placed.
	 * @param allNetworks a list of all networks that connect the nodes.
	 * @param cellName the name of the cell being placed.
	 */
	public void runPlacement(List<PlacementNode> placementNodes, List<PlacementNetwork> allNetworks, String cellName, Job job)
	{
		setParamterValues(numThreadsParam.getIntValue(), maxRuntimeParam.getIntValue());

		// figure out the density of connections
		prepareClustering(placementNodes, allNetworks);

		// create the initial placement proposal
		ProposedPlacement initialProposal = new ProposedPlacement(placementNodes);
		List<ProposedPlacement> currentProposals = new ArrayList<ProposedPlacement>();
		currentProposals.add(initialProposal);

		// do the bottom-up clustering and make lists of node-pairs to be combined
		Set<PlacementNode> macroNodes = new HashSet<PlacementNode>();
		Map<Double,List<PNPair>> allDensities = BottomUpPartition.makeClusteredPairs(placementNodes, allNetworks, macroNodes);

		// analyze instance sizes and see if they can be split into big/small
		Map<Double,List<PlacementNode>> instSizes = new TreeMap<Double,List<PlacementNode>>();
		for(PlacementNode pn : placementNodes)
		{
			Double size = new Double(pn.getWidth() * pn.getHeight());
			List<PlacementNode> nodes = instSizes.get(size);
			if (nodes == null) instSizes.put(size, nodes = new ArrayList<PlacementNode>());
			nodes.add(pn);
		}
		int si = instSizes.size();
		double[] bigSizes = new double[si];
		for(Double size : instSizes.keySet()) bigSizes[--si] = size.doubleValue();
		Set<PlacementNode> bigNodes = new HashSet<PlacementNode>();
		for(int i=0; i<bigSizes.length; i++)
		{
			if (i != 0 && bigSizes[i] <= bigSizes[i-1]/2) break;
			List<PlacementNode> nodes = instSizes.get(new Double(bigSizes[i]));
			for(PlacementNode pn : nodes) bigNodes.add(pn);
		}
//for(Double size : instSizes.keySet()) System.out.println(instSizes.get(size)+" NODES ARE "+size+" BIG");

//// weight the densities by the size of the nodes in each pair
//Map<Double,List<PNPair>> newDensities = new TreeMap<Double,List<PNPair>>();
//for(Double density : allDensities.keySet())
//{
//	double d = density.doubleValue();
//	List<PNPair> pairs = allDensities.get(density);
//	for(PNPair pair : pairs)
//	{
//		double p1Size = pair.n1.getWidth() + pair.n1.getHeight();
//		double p2Size = pair.n2.getWidth() + pair.n2.getHeight();
//		double newDensity = (p1Size + p2Size) * d;
//		Double newDensityObj = new Double(newDensity);
//		List<PNPair> newPairs = newDensities.get(newDensityObj);
//		if (newPairs == null) newDensities.put(newDensityObj, newPairs = new ArrayList<PNPair>());
//		newPairs.add(pair);
//	}
//}
//allDensities = newDensities;

		// sort the pairs by density
		orderedPairGroups = new ArrayList<List<PNPair>>();
		int di = allDensities.size();
		double[] clusteringDensities = new double[di];
		for(Double density : allDensities.keySet()) clusteringDensities[--di] = density.doubleValue();
		switch (pairGrouping)
		{
			case GROUPPAIRSFLAT:
				List<PNPair> thisPairSet = new ArrayList<PNPair>();
				for(int i=0; i<allDensities.size(); i++)
				{
					List<PNPair> pairsAtLevel = allDensities.get(new Double(clusteringDensities[i]));
					for(PNPair pnp : pairsAtLevel) thisPairSet.add(pnp);
				}
				if (thisPairSet.size() > 0) orderedPairGroups.add(thisPairSet);
				break;

			case GROUPPAIRSBYSIZE:
				// first the big-size group
				thisPairSet = new ArrayList<PNPair>();
				for(int i=0; i<allDensities.size(); i++)
				{
					List<PNPair> pairsAtLevel = allDensities.get(new Double(clusteringDensities[i]));
					for(PNPair pnp : pairsAtLevel)
					{
						if (bigNodes.contains(pnp.n1)&& bigNodes.contains(pnp.n2)) thisPairSet.add(pnp);
					}
				}
				if (thisPairSet.size() > 0) orderedPairGroups.add(thisPairSet);

				// next the small-size group
				thisPairSet = new ArrayList<PNPair>();
				for(int i=0; i<allDensities.size(); i++)
				{
					List<PNPair> pairsAtLevel = allDensities.get(new Double(clusteringDensities[i]));
					for(PNPair pnp : pairsAtLevel)
					{
						if (!bigNodes.contains(pnp.n1)|| !bigNodes.contains(pnp.n2)) thisPairSet.add(pnp);
					}
				}
				if (thisPairSet.size() > 0) orderedPairGroups.add(thisPairSet);
				break;

			case GROUPPAIRSBYDENSITY:
				// first the big-size group
				for(int i=0; i<allDensities.size(); i++)
				{
					List<PNPair> pairsAtLevel = allDensities.get(new Double(clusteringDensities[i]));
					thisPairSet = new ArrayList<PNPair>();
					for(PNPair pnp : pairsAtLevel)
					{
						if (bigNodes.contains(pnp.n1)&& bigNodes.contains(pnp.n2))
							thisPairSet.add(pnp);
					}
					if (thisPairSet.size() > 0) orderedPairGroups.add(thisPairSet);
				}

				// next the small-size group
				for(int i=0; i<allDensities.size(); i++)
				{
					List<PNPair> pairsAtLevel = allDensities.get(new Double(clusteringDensities[i]));
					thisPairSet = new ArrayList<PNPair>();
					for(PNPair pnp : pairsAtLevel)
					{
						if (!bigNodes.contains(pnp.n1)|| !bigNodes.contains(pnp.n2))
							thisPairSet.add(pnp);
					}
					if (thisPairSet.size() > 0) orderedPairGroups.add(thisPairSet);
				}
				break;
		}

		System.out.println(bigNodes.size() + " out of " + placementNodes.size() + " nodes are 'big'");
		String groupingName = "";
		switch (pairGrouping)
		{
			case GROUPPAIRSFLAT:      groupingName = "FLAT";  break;
			case GROUPPAIRSBYSIZE:    groupingName = "SIZE";  break;
			case GROUPPAIRSBYDENSITY: groupingName = "DENSITY";  break;
		}
		if (Job.getDebug())
		{
			System.out.println("Parameters: bound-weight=" + boundWeight + " aspect-ratio-weight=" + aspectRatioWeight +
				" trellis-size=" + trellisWidth);
			System.out.println("Pair grouping mode: " + groupingName + ", number of groups = " + orderedPairGroups.size());
		}
//for(List<PNPair> pairlist : orderedPairGroups)
//{
//	System.out.print("PAIR LIST:");
//	for(PNPair pnp : pairlist) System.out.print(" "+pnp.n1+"--"+pnp.n2);
//	System.out.println();
//}

		// do the placement
//		if (DEBUGMODE)
//		{
//			initializeClusteringDebugging(placementNodes, currentProposal, allNetworks);
//			return;
//		}
		// Bottom-up placement
		for(int step=1; ; step++)
		{
			List<ProposedPlacement> newProposals = new ArrayList<ProposedPlacement>();
			for(ProposedPlacement curProp : currentProposals)
			{
				List<ProposedPlacement> nextProposals = curProp.clusteringPlacementStep();
				for(int i=0; i<nextProposals.size(); i++)
				{
					ProposedPlacement pp = nextProposals.get(i);
					pp.computeQuality(allNetworks);
					newProposals.add(pp);
				}
			}
			if (newProposals.size() == 0) break;

			// take only the best proposals for the next step
			Collections.sort(newProposals);
			while(newProposals.size() > trellisWidth)
				newProposals.remove(newProposals.size()-1);
			currentProposals = newProposals;
			if (DEBUGPROGRESS)
			{
				for(ProposedPlacement pp : currentProposals)
					showProposal(pp);
			}
		}

		// get the final placement
		ProposedPlacement finalProposal = currentProposals.get(0);

		// pack any unconnected clusters together
		if (finalProposal.allClusters.size() > 1)
			finalProposal.combineClusters();

		// apply the final placement to the actual nodes
		finalProposal.applyProposal(placementNodes);
	}

	private void prepareClustering(List<PlacementNode> placementNodes, List<PlacementNetwork> allNetworks)
	{
		// figure out what connections are on each node
		consOnNodes = new HashMap<PlacementNode,List<PlacementConnection>>();
		for(PlacementNetwork pNet : allNetworks)
		{
			List<PlacementConnection> cons = PlacementAdapter.getOptimalConnections(pNet);
			for(PlacementConnection con : cons)
			{
				PlacementNode n1 = con.getP1().getPlacementNode();
				PlacementNode n2 = con.getP2().getPlacementNode();
				List<PlacementConnection> consOnN1 = consOnNodes.get(n1);
				if (consOnN1 == null) consOnNodes.put(n1, consOnN1 = new ArrayList<PlacementConnection>());
				List<PlacementConnection> consOnN2 = consOnNodes.get(n2);
				if (consOnN2 == null) consOnNodes.put(n2, consOnN2 = new ArrayList<PlacementConnection>());
				consOnN1.add(con);
				if (n2 != n1) consOnN2.add(con);
			}
		}
	}

	private static int proposedPlacementIndex = 1;

	private class ProposedPlacement implements Comparable<ProposedPlacement>
    {
		/** list of proxy nodes */									private List<ProxyNode> nodesToPlace;
		/** map from original PlacementNodes to proxy nodes */		private Map<PlacementNode,ProxyNode> proxyMap;
		/** list of all proxy clusters */							private List<ProxyCluster> allClusters;
		/** map from ProxyNodes to their ProxyClusters */			private Map<ProxyNode,ProxyCluster> clusterMap;
		/** current grouping position being resolved */				private int groupPosition;
		/** remaining pairs to resolve in this group */				private List<PNPair> pairsToResolve;
		/** quality of this ProposedPlacement */					private double quality;
		/** unique index of this ProposedPlacement */				private int proposalNumber;
		/** unique index of this ProposedPlacement */				private int parentProposalNumber;
		/** the nodes that were moved */							private PNPair movedPair;
private List<String> furtherExplanation = new ArrayList<String>();

		ProposedPlacement()
		{
			nodesToPlace = new ArrayList<ProxyNode>();
			proxyMap = new HashMap<PlacementNode,ProxyNode>();
			allClusters = new ArrayList<ProxyCluster>();
			clusterMap = new HashMap<ProxyNode,ProxyCluster>();
			pairsToResolve = new ArrayList<PNPair>();
			proposalNumber = proposedPlacementIndex++;
		}

		ProposedPlacement(List<PlacementNode> placementNodes)
		{
			this();
			groupPosition = -1;
			for (int i=0; i<placementNodes.size(); i++)
			{
				// make a ProxyNode to shadow this PlacementNode
				PlacementNode p = placementNodes.get(i);
				ProxyNode proxy = new ProxyNode(p);
				nodesToPlace.add(proxy);
				proxyMap.put(p, proxy);

				// make a cluster for this ProxyNode
				ProxyCluster pCluster = new ProxyCluster();
				clusterMap.put(proxy, pCluster);
				pCluster.clusterIndex = i;
				allClusters.add(pCluster);
				pCluster.add(proxy);
			}
		}

		ProposedPlacement(ProposedPlacement copyIt)
		{
			this();
			for(ProxyNode pn : copyIt.nodesToPlace) nodesToPlace.add(pn);
			for(PlacementNode pn : copyIt.proxyMap.keySet()) proxyMap.put(pn, copyIt.proxyMap.get(pn));
			for(ProxyCluster pc : copyIt.allClusters) allClusters.add(pc);
			for(ProxyNode pn : copyIt.clusterMap.keySet()) clusterMap.put(pn, copyIt.clusterMap.get(pn));
			groupPosition = copyIt.groupPosition;
			for(PNPair pnp : copyIt.pairsToResolve) pairsToResolve.add(pnp);
			parentProposalNumber = copyIt.proposalNumber;
		}

		public int compareTo(ProposedPlacement pp)
        {
        	if (quality > pp.quality) return 1;
        	if (quality < pp.quality) return -1;
            return 0;
        }

		public void computeQuality(List<PlacementNetwork> allNetworks)
		{
			quality = 0;
			for(PlacementNetwork pNet : allNetworks)
			{
				double lXTot = Double.MAX_VALUE;
				double hXTot = -Double.MAX_VALUE;
				double lYTot = Double.MAX_VALUE;
				double hYTot = -Double.MAX_VALUE;
				for(PlacementPort pPort : pNet.getPortsOnNet())
				{
					ProxyNode pxn = proxyMap.get(pPort.getPlacementNode());
// if the next two lines are uncommented, then only placed nodes get considered in the quality metric
//ProxyCluster cluster = clusterMap.get(pxn);
//if (cluster.nodesInCluster.size() <= 1) continue;
					Orientation o = pxn.getOrientation();
					Point2D off = o.transformPoint(new Point2D.Double(pPort.getOffX(), pPort.getOffY()));
					double x = pxn.getX() + off.getX();
					double y = pxn.getY() + off.getY();
					if (x < lXTot) lXTot = x;
					if (x > hXTot) hXTot = x;
					if (y < lYTot) lYTot = y;
					if (y > hYTot) hYTot = y;
				}
				double hpwl = ((hXTot - lXTot) + (hYTot - lYTot)) / 2;
				quality += hpwl;
			}

// consider aspect-ratio factor
double aspectRatioFactor = 1;
for(ProxyCluster cluster : allClusters)
{
	if (cluster.getNodesInCluster().size() <= 1) continue;
	double aspectRatio = cluster.bound.getWidth() < cluster.bound.getHeight() ?
		cluster.bound.getHeight() / cluster.bound.getWidth() :
			cluster.bound.getWidth() / cluster.bound.getHeight();
	aspectRatioFactor *= aspectRatio;
}
// uncomment the next line to allow aspect-ratio considerations to blend with HPWL
//quality *= aspectRatioFactor;
		}

		/**
		 * Method to make a new version of a ProxyCluster.
		 * @param cluster the ProxyCluster to duplicate.
		 */
		public ProxyCluster copyCluster(ProxyCluster cluster)
		{
			ProxyCluster newCluster = new ProxyCluster();
			newCluster.clusterIndex = cluster.clusterIndex;
			allClusters.set(cluster.clusterIndex, newCluster);

			newCluster.bound = ERectangle.fromLambda(cluster.bound.getX(), cluster.bound.getY(),
				cluster.bound.getWidth(), cluster.bound.getHeight());

			List<ProxyNode> newClusterList = new ArrayList<ProxyNode>();
			for(ProxyNode pn : cluster.getNodesInCluster())
			{
				ProxyNode copyPN = new ProxyNode(pn);
				nodesToPlace.remove(pn);
				nodesToPlace.add(copyPN);

				proxyMap.put(pn.original, copyPN);

				newClusterList.add(copyPN);
				clusterMap.remove(pn);
				clusterMap.put(copyPN, newCluster);
			}
			newCluster.nodesInCluster = newClusterList;
			return newCluster;
		}

		private List<ProposedPlacement> clusteringPlacementStep()
		{
			// make a list of new choices
			List<ProposedPlacement> choices = new ArrayList<ProposedPlacement>();

			while (choices.size() == 0)
			{
				// get the list of possible pairs to merge
				if (pairsToResolve.size() == 0)
				{
					groupPosition++;
					if (groupPosition >= orderedPairGroups.size()) break;
					pairsToResolve = new ArrayList<PNPair>();
					for(PNPair pnp : orderedPairGroups.get(groupPosition)) pairsToResolve.add(pnp);
				}

				for(int pair=0; pair<pairsToResolve.size(); pair++)
				{
					if (choices.size() > trellisWidth/2) break;
					PNPair pnp = pairsToResolve.get(pair);
					ProxyNode pnp1 = proxyMap.get(pnp.n1);
					ProxyNode pnp2 = proxyMap.get(pnp.n2);
					ProxyCluster cluster1 = clusterMap.get(pnp1);
					ProxyCluster cluster2 = clusterMap.get(pnp2);
					if (cluster1 == cluster2) { pairsToResolve.remove(pnp);  pair--;  continue; }
String explanation = "MOVED NODE: "+pnp1.getNodeName() + " AT ("+pnp1.getX()+","+pnp1.getY()+") TO NODE " + pnp2.getNodeName() + " AT ("+pnp2.getX()+","+pnp2.getY()+")";

					List<RotationTest> allTests = new ArrayList<RotationTest>();
					for(int forceDir=0; forceDir<4; forceDir++)
					{
						RotationTest spinI = new RotationTest(this, pnp1, pnp2, Orientation.IDENT, forceDir);
						allTests.add(spinI);

						// test flips
						allTests.add(new RotationTest(this, pnp1, pnp2, Orientation.X, forceDir));
						allTests.add(new RotationTest(this, pnp1, pnp2, Orientation.Y, forceDir));
						allTests.add(new RotationTest(this, pnp1, pnp2, Orientation.RR, forceDir));
						if (canRotate.getBooleanValue())
						{
							// test rotations
							allTests.add(new RotationTest(this, pnp1, pnp2, Orientation.R, forceDir));
							allTests.add(new RotationTest(this, pnp1, pnp2, Orientation.RRR, forceDir));
							allTests.add(new RotationTest(this, pnp1, pnp2, Orientation.YR, forceDir));
							allTests.add(new RotationTest(this, pnp1, pnp2, Orientation.YRRR, forceDir));
						}
					}
					Collections.sort(allTests);
					RotationTest bestTest = allTests.get(0);
					ProposedPlacement newPP = new ProposedPlacement(this);
newPP.furtherExplanation.add(explanation);
//pushCluster(cluster1, cluster2, bestTest.originalDelta.getX(), bestTest.originalDelta.getY(),
//	bestTest.originalForceDir, newPP);
					makeProposal(newPP, bestTest, pnp, cluster1, cluster2);
					choices.add(newPP);
newPP.movedPair = pnp;

//					RotationTest secondBestTest = allTests.get(1);
//					newPP = makeProposal(secondBestTest, pnp, cluster1, cluster2);
//					choices.add(newPP);
				}
			}
			return choices;
		}

		private ProposedPlacement makeProposal(ProposedPlacement newPP, RotationTest bestTest, PNPair pnp, ProxyCluster cluster1, ProxyCluster cluster2)
		{
			double dX = bestTest.getDelta().getX();
			double dY = bestTest.getDelta().getY();

			// make a new ProposedPlacement for this change
			newPP.pairsToResolve.remove(pnp);
//System.out.println("  NEXT LEVEL DOWN ("+newPP.resolutionLevel+") WILL HAVE "+newPP.pairsToResolve.size()+" PAIRS TO EVALUATE");
//System.out.println("COPYING CLUSTERS "+cluster1.clusterIndex+" AND "+cluster2.clusterIndex);
			cluster1 = newPP.copyCluster(cluster1);
			cluster2 = newPP.copyCluster(cluster2);

			// move the cluster
			cluster1.rotate(bestTest.getOrientation());
			for(ProxyNode pNode : cluster1.getNodesInCluster())
			{
				pNode.setLocation(pNode.getX() + dX, pNode.getY() + dY);
newPP.furtherExplanation.add("MOVE "+pNode.getNodeName()+" ("+dX+","+dY+"), ROT="+bestTest.getOrientation()+" TO ("+pNode.getX()+","+pNode.getY()+") ROTATED "+bestTest.getOrientation());
//System.out.println("SO MOVED "+pNode+" TO ("+pNode.getX()+","+pNode.getY()+") ORIENTATION "+pNode.getOrientation());
			}

			// copy contents of cluster2 into cluster1
			for(ProxyNode pNode : cluster2.getNodesInCluster())
			{
				cluster1.add(pNode);
				newPP.clusterMap.put(pNode, cluster1);
			}
			cluster1.computeBounds();

			// delete cluster2
			int lastIndex = newPP.allClusters.size() - 1;
			if (cluster2.clusterIndex < lastIndex)
			{
				ProxyCluster newLastCluster = newPP.copyCluster(newPP.allClusters.get(lastIndex));
				if (cluster1.clusterIndex == lastIndex) cluster1 = newLastCluster;
				newLastCluster.clusterIndex = cluster2.clusterIndex;
				newPP.allClusters.set(newLastCluster.clusterIndex, newLastCluster);
			}
			newPP.allClusters.remove(lastIndex);

			// use RTree to make sure clusters don't overlap
			RTNode rTree = newPP.makeClusterRTree();
			int cluster1Index = cluster1.clusterIndex;
			rTree = newPP.plowCluster(cluster1Index, 0, 0, rTree, true, true, true, true);
			for(int i=0; i<newPP.allClusters.size(); i++)
			{
				if (i == cluster1Index) continue;
				rTree = newPP.plowCluster(i, 0, 0, rTree, true, true, true, true);
			}
			return newPP;
		}

		public void applyProposal(List<PlacementNode> placementNodes)
		{
			for (PlacementNode pn : placementNodes)
			{
				ProxyNode p = proxyMap.get(pn);
				pn.setPlacement(p.getX(), p.getY());
				pn.setOrientation(p.getOrientation());
			}
		}

		private RTNode makeClusterRTree()
		{
			RTNode root = RTNode.makeTopLevel();
			for(ProxyCluster pCluster : allClusters)
				root = RTNode.linkGeom(null, root, pCluster);
			return root;
		}

		private RTNode plowCluster(int clusterIndex, double dX, double dY, RTNode rTree,
			boolean lDir, boolean rDir, boolean uDir, boolean dDir)
		{
			// propose the node move
			ProxyCluster pCluster = allClusters.get(clusterIndex);
			ERectangle oldBounds = pCluster.getBounds();
			double prevX = oldBounds.getCenterX(), prevY = oldBounds.getCenterY();
			if (DEBUGPLOW) System.out.println("PLOWING "+pCluster+" BY ("+dX+","+dY+")");
furtherExplanation.add("PLOWING "+pCluster+" BY ("+TextUtils.formatDouble(dX)+","+TextUtils.formatDouble(dY)+")");

			// move the node in the R-Tree
			if (dX != 0 || dY != 0)
			{
				rTree = RTNode.unLinkGeom(null, rTree, pCluster);
				pCluster = copyCluster(pCluster);

				for(ProxyNode pNode : pCluster.getNodesInCluster())
				{
					pNode.setLocation(pNode.getX() + dX, pNode.getY() + dY);
				}
				pCluster.computeBounds();
furtherExplanation.add("CLUSTER BOUNDS NOW "+TextUtils.formatDouble(pCluster.getBounds().getMinX())+"<=X<="+TextUtils.formatDouble(pCluster.getBounds().getMaxX())+" AND "+
	TextUtils.formatDouble(pCluster.getBounds().getMinY())+"<=Y<="+TextUtils.formatDouble(pCluster.getBounds().getMaxY()));
				rTree = RTNode.linkGeom(null, rTree, pCluster);
			}
			ERectangle pBound = pCluster.getBounds();

			ERectangle search = ERectangle.fromLambda(Math.min(prevX, prevX+dX) - oldBounds.getWidth()/2,
				Math.min(prevY, prevY+dY) - oldBounds.getHeight()/2,
				oldBounds.getWidth() + Math.abs(dX), oldBounds.getHeight() + Math.abs(dY));
			boolean blocked = true;
			while (blocked)
			{
				blocked = false;

				// re-get the cluster bounds because it might have changed
				pBound = pCluster.getBounds();

				// look for an intersecting node
				for (RTNode.Search sea = new RTNode.Search(search, rTree, true); sea.hasNext();)
				{
					ProxyCluster inArea = (ProxyCluster)sea.next();
					if (inArea == pCluster) continue;
					ERectangle sBound = inArea.getBounds();
					if (pBound.getMinX() >= sBound.getMaxX() || pBound.getMaxX() <= sBound.getMinX() ||
						pBound.getMinY() >= sBound.getMaxY() || pBound.getMaxY() <= sBound.getMinY()) continue;

					// figure out which way to move the blocking node
					double leftMotion = sBound.getMaxX() - pBound.getMinX();
					double rightMotion = pBound.getMaxX() - sBound.getMinX();
					double downMotion = sBound.getMaxY() - pBound.getMinY();
					double upMotion = pBound.getMaxY() - sBound.getMinY();
					if (!lDir) leftMotion = Double.MAX_VALUE;
					if (!rDir) rightMotion = Double.MAX_VALUE;
					if (!dDir) downMotion = Double.MAX_VALUE;
					if (!uDir) upMotion = Double.MAX_VALUE;
String exp = "  INTERSECTS "+inArea+" LEFT=";
if (leftMotion == Double.MAX_VALUE) exp += "INFINITE"; else exp +=TextUtils.formatDouble(leftMotion);
exp += " RIGHT=";
if (rightMotion == Double.MAX_VALUE) exp += "INFINITE"; else exp +=TextUtils.formatDouble(rightMotion);
exp += " UP=";
if (upMotion == Double.MAX_VALUE) exp += "INFINITE"; else exp +=TextUtils.formatDouble(upMotion);
exp += " DOWN=";
if (downMotion == Double.MAX_VALUE) exp += "INFINITE"; else exp +=TextUtils.formatDouble(downMotion);
furtherExplanation.add(exp);
					double leastMotion = Math.min(Math.min(leftMotion, rightMotion), Math.min(upMotion, downMotion));
					if (leftMotion == leastMotion)
					{
						// move the other block left to keep it away
furtherExplanation.add("  MOVE "+inArea+" "+TextUtils.formatDouble(leftMotion)+" LEFT (Because its right edge is "+
	TextUtils.formatDouble(sBound.getMaxX())+" and "+pCluster+" left edge is "+TextUtils.formatDouble(pBound.getMinX())+")");
						rTree = plowCluster(inArea.clusterIndex, -leftMotion, 0, rTree, lDir, false, false, false);
					} else if (rightMotion == leastMotion)
					{
						// move the other block right to keep it away
furtherExplanation.add("  MOVE "+inArea+" "+TextUtils.formatDouble(rightMotion)+" RIGHT (Because its left edge is "+
	TextUtils.formatDouble(sBound.getMinX())+" and "+pCluster+" right edge is "+TextUtils.formatDouble(pBound.getMaxX())+")");
						rTree = plowCluster(inArea.clusterIndex, rightMotion, 0, rTree, false, rDir, false, false);
					} else if (upMotion == leastMotion)
					{
						// move the other block up to keep it away
furtherExplanation.add("  MOVE "+inArea+" "+TextUtils.formatDouble(upMotion)+" UP (Because its bottom edge is "+
	TextUtils.formatDouble(sBound.getMinY())+" and "+pCluster+" top edge is "+TextUtils.formatDouble(pBound.getMaxY())+")");
						rTree = plowCluster(inArea.clusterIndex, 0, upMotion, rTree, false, false, uDir, false);
					} else if (downMotion == leastMotion)
					{
						// move the other block down to keep it away
furtherExplanation.add("  MOVE "+inArea+" "+TextUtils.formatDouble(downMotion)+" DOWN (Because its top edge is "+
	TextUtils.formatDouble(sBound.getMaxY())+" and "+pCluster+" bottom edge is "+TextUtils.formatDouble(pBound.getMinY())+")");
						rTree = plowCluster(inArea.clusterIndex, 0, -downMotion, rTree, false, false, false, dDir);
					}
					blocked = true;
					break;
				}
			}
			return rTree;
		}

		/**
		 * Method to combine all clusters into one that is optimally sized.
		 * The assumption is that everything is clustered at this point,
		 * so if there are multiple clusters, they are not connected to each other.
		 */
		private void combineClusters()
		{
			// sort clusters by size
			Collections.sort(allClusters);

			// the largest one (first one) is the "main" cluster
			ProxyCluster mainCluster = allClusters.get(0);

			// iterate over remaining clusters from largest on down
			while (allClusters.size() > 1)
			{
				ProxyCluster otherCluster = allClusters.get(1);

				// gather all X and Y coordinates in the main cluster
				Set<Double> xCoords = new TreeSet<Double>();
				Set<Double> yCoords = new TreeSet<Double>();
				for(ProxyNode pNode : mainCluster.getNodesInCluster())
				{
					xCoords.add(new Double(pNode.getX() - pNode.getWidth()/2));
					xCoords.add(new Double(pNode.getX() + pNode.getWidth()/2));
					yCoords.add(new Double(pNode.getY() - pNode.getHeight()/2));
					yCoords.add(new Double(pNode.getY() + pNode.getHeight()/2));
				}

				// try all of these coordinates and find the one that expands the main cluster least
				double bestArea = -1;
				double bestDX = 0, bestDY = 0;
//				for(Double x : xCoords)
//				{
//					for(int forceDir=0; forceDir<4; forceDir++)
//					{
//						double dX = x.doubleValue() - otherCluster.bound.getMaxX();
//						double dY = mainCluster.bound.getCenterY() - otherCluster.bound.getCenterY();
//						EPoint newDelta = pushCluster(otherCluster, mainCluster, dX, dY, forceDir, null);
//						double area = mainCluster.getAreaWithOtherShiftedCluster(otherCluster, newDelta);
//						if (bestArea < 0 || area < bestArea)
//						{
//							bestArea = area;
//							bestDX = newDelta.getX();
//							bestDY = newDelta.getY();
//						}
//					}
//				}
				for(Double y : yCoords)
				{
					for(int forceDir=0; forceDir<4; forceDir++)
					{
						double dX = mainCluster.bound.getCenterX() - otherCluster.bound.getCenterX();
						double dY = y.doubleValue() - otherCluster.bound.getMaxY();
						EPoint newDelta = pushCluster(otherCluster, mainCluster, dX, dY, forceDir, null);
						double area = mainCluster.getAreaWithOtherShiftedCluster(otherCluster, newDelta);
						if (bestArea < 0 || area < bestArea)
						{
							bestArea = area;
							bestDX = newDelta.getX();
							bestDY = newDelta.getY();
						}
					}
				}

				// move cluster
				for(ProxyNode pNode : otherCluster.getNodesInCluster())
					pNode.setLocation(pNode.getX() + bestDX, pNode.getY() + bestDY);

				// merge the clusters
				for(ProxyNode pNode : otherCluster.getNodesInCluster())
				{
					mainCluster.add(pNode);
					clusterMap.put(pNode, mainCluster);
				}
				allClusters.remove(otherCluster);
				mainCluster.computeBounds();
			}
		}

//		public void describe(String title)
//		{
//			System.out.print(title+" PLACEMENT "+getObjName(this)+" HAS "+nodesToPlace.size()+" NODES:");
//			for(ProxyNode pn : nodesToPlace) System.out.print(" "+getObjName(pn)+"="+pn.getNodeName()+"/C="+getObjName(clusterMap.get(pn)));
//			System.out.println();
//			System.out.print("  HAS "+allClusters.size()+" CLUSTERS:");
//			for(ProxyCluster pc : allClusters)
//			{
//				System.out.print(" "+getObjName(pc));
//				String sep = "[";
//				for(ProxyNode pn : pc.nodesInCluster) { System.out.print(sep+getObjName(pn)+"="+pn.getNodeName());  sep = ", "; }
//				if (sep.equals("[")) System.out.print(sep);
//				System.out.print("]");
//			}
//			System.out.println();
//		}
	}

	/**
	 * Class to test a connection of two clusters
	 */
	private class RotationTest implements Comparable<RotationTest>
	{
		private double metric;
		private Point2D delta;
		private ERectangle overallBound;
		private Orientation spin;
		private ProxyNode beingMoved;
		ProxyCluster cluster1, cluster2;

		/**
		 * Constructor builds a test of the pairing of two nodes.
		 * @param pnp the pairing.
		 * @param spin the Orientation change of the first node.
		 */
		public RotationTest(ProposedPlacement pp, ProxyNode pn1, ProxyNode pn2, Orientation spin, int forceDir)
		{
			beingMoved = pn1;
			this.spin = spin;
			delta = new Point2D.Double();
			cluster1 = pp.clusterMap.get(pn1);
			cluster2 = pp.clusterMap.get(pn2);
			metric = bringTogether(pp, pn1, pn2, delta, spin, forceDir);
		}

		public String toString()
		{
			String msg = "MOVE "+beingMoved+" R="+spin.toString()+" TO ("+(beingMoved.getX()+delta.getX())+","+
				(beingMoved.getY()+delta.getY())+")";
			return msg;
		}

		public int compareTo(RotationTest other)
        {
//			return metric < other.metric;
			double metricImprovement = (other.metric - metric) / Math.max(other.metric, metric);

			double thisArea = overallBound.getWidth() * overallBound.getHeight();
			double otherArea = other.overallBound.getWidth() * other.overallBound.getHeight();
			double boundImprovement = (otherArea - thisArea) / Math.max(thisArea, otherArea);
			double aspectRatioThis = overallBound.getWidth() > overallBound.getHeight() ?
				overallBound.getHeight() / overallBound.getWidth() :
					overallBound.getWidth() / overallBound.getHeight();
			double aspectRatioOther = other.overallBound.getWidth() > other.overallBound.getHeight() ?
				other.overallBound.getHeight() / other.overallBound.getWidth() :
					other.overallBound.getWidth() / other.overallBound.getHeight();
			double aspectRatioImprovement = (aspectRatioThis - aspectRatioOther) / Math.max(aspectRatioOther, aspectRatioThis);
boundImprovement *= boundWeight;
aspectRatioImprovement *= aspectRatioWeight;
			double totalImprovement = metricImprovement + boundImprovement + aspectRatioImprovement;
			if (totalImprovement < 0) return 1;
			if (totalImprovement > 0) return -1;
			return 0;
		}

		public Point2D getDelta() { return delta; }

		public Orientation getOrientation() { return spin; }

		private double bringTogether(ProposedPlacement pp, ProxyNode px1, ProxyNode px2, Point2D delta, Orientation spin1, int forceDir)
		{
//System.out.println("ROTATING NODE "+pn1+" BY "+spin1.toString()+" AND MOVING IT TO NODE "+pn2);
			ComputeForce cf = new ComputeForce(pp, px1, px2);

			// save information for nodes in cluster 1
			for(ProxyNode pn : cluster1.nodesInCluster) pn.saveValues();

			// rotate cluster1 appropriately
			cluster1.rotate(spin1);
//for(ProxyNode pn : cluster1.nodesInCluster)
//	System.out.println(" SO NODE "+pn+" WAS "+pn.widthSave+"x"+pn.heightSave+" AT ("+pn.xSave+","+pn.ySave+") BUT NOW IS "+pn.width+","+pn.height+" AT ("+pn.x+","+pn.y+")");

			// figure out how to move these clusters together
			List<PlacementConnection> consOnC1 = new ArrayList<PlacementConnection>();
			for(ProxyNode pn : cluster1.nodesInCluster)
			{
				for(PlacementConnection pc : consOnNodes.get(pn.original)) consOnC1.add(pc);
			}
			Set<PlacementNode> nodesInC1 = new HashSet<PlacementNode>();
			for(ProxyNode pn : cluster1.nodesInCluster) nodesInC1.add(pn.original);
			Set<PlacementNode> nodesInC2 = new HashSet<PlacementNode>();
			for(ProxyNode pn : cluster2.nodesInCluster) nodesInC2.add(pn.original);

			List<PlacementConnection> validConnections = new ArrayList<PlacementConnection>();
			for(PlacementConnection con : consOnC1)
			{
				PlacementPort pp1 = con.getP1();
				PlacementPort pp2 = con.getP2();
				boolean linksTheNodes = false;
				if (nodesInC1.contains(pp1.getPlacementNode()) && nodesInC2.contains(pp2.getPlacementNode())) linksTheNodes = true;
				if (nodesInC1.contains(pp2.getPlacementNode()) && nodesInC2.contains(pp1.getPlacementNode())) linksTheNodes = true;
				if (linksTheNodes) validConnections.add(con);
			}

			for(PlacementConnection con : validConnections)
			{
				PlacementPort pp1 = con.getP1();
				PlacementPort pp2 = con.getP2();
				ProxyNode pxn1 = pp.proxyMap.get(pp1.getPlacementNode());
				Orientation o1 = pxn1.getOrientation();
				Point2D off1 = o1.transformPoint(new Point2D.Double(pp1.getOffX(), pp1.getOffY()));
				double x1 = pxn1.getX() + off1.getX();
				double y1 = pxn1.getY() + off1.getY();
				ProxyNode pxn2 = pp.proxyMap.get(pp2.getPlacementNode());
				Orientation o2 = pxn2.getOrientation();
				Point2D off2 = o2.transformPoint(new Point2D.Double(pp2.getOffX(), pp2.getOffY()));
				double x2 = pxn2.getX() + off2.getX();
				double y2 = pxn2.getY() + off2.getY();
				// move pn1 on top of pn2 and then back it off the minimal amount
				double dX = x2 - x1;
				double dY = y2 - y1;
				if (dX == 0 && dY == 0) continue;

				// if this is power/ground, double the force
				boolean isOnRail = pp1.getPlacementNetwork().isOnRail();

				if (pxn1 == px1)
				{
//System.out.println(" PORT OFFSET WENT FROM ("+TextUtils.formatDouble(pp1.getOffX())+","+TextUtils.formatDouble(pp1.getOffY())+") TO ("+
//	TextUtils.formatDouble(off1.getX())+","+TextUtils.formatDouble(off1.getY())+"), MOVING BY ("+TextUtils.formatDouble(dX)+","+TextUtils.formatDouble(dY)+")");
					cf.accumulateForce(dX, dY, isOnRail);
				} else if (pxn2 == px1)
				{
//System.out.println(" PORT OFFSET WENT FROM ("+TextUtils.formatDouble(pp2.getOffX())+","+TextUtils.formatDouble(pp2.getOffY())+") TO ("+
//	TextUtils.formatDouble(off2.getX())+","+TextUtils.formatDouble(off2.getY())+"), MOVING BY ("+TextUtils.formatDouble(-dX)+","+TextUtils.formatDouble(-dY)+")");
					cf.accumulateForce(-dX, -dY, isOnRail);
				}
			}

			// normalize the force vectors
			cf.normalizeForce();
			double dX = cf.getForceX();
			double dY = cf.getForceY();

			// propose the node move
			EPoint newDelta = pushCluster(cluster1, cluster2, dX, dY, forceDir, null);
//System.out.println("    SO WHEN MOVING NODE "+pn1+" TO "+pn2+" WITH ROTATION "+spin+", ORIGINAL VECTOR IS ("+dX+","+dY+") BUT GETS PUSHED TO BE ("+newDelta.getX()+","+newDelta.getY()+")");
			dX = newDelta.getX();
			dY = newDelta.getY();
			delta.setLocation(newDelta);

			// determine length of connections
			double totalLen = 0;
			for(PlacementConnection con : validConnections)
			{
				PlacementPort pp1 = con.getP1();
				PlacementPort pp2 = con.getP2();
				ProxyNode pxn1 = pp.proxyMap.get(pp1.getPlacementNode());
				ProxyNode pxn2 = pp.proxyMap.get(pp2.getPlacementNode());
				double x1 = pxn1.getX(), y1 = pxn1.getY();
				double x2 = pxn2.getX(), y2 = pxn2.getY();
				if (px1 == pxn1) { x1 += dX;  y1 += dY; } else { x2 += dX;  y2 += dY; }

				Orientation o1 = pxn1.getOrientation();
				Point2D off1 = o1.transformPoint(new Point2D.Double(pp1.getOffX(), pp1.getOffY()));
				x1 += off1.getX();
				y1 += off1.getY();

				Orientation o2 = pxn2.getOrientation();
				Point2D off2 = o2.transformPoint(new Point2D.Double(pp2.getOffX(), pp2.getOffY()));
				x2 += off2.getX();
				y2 += off2.getY();

				// move pn1 on top of pn2 and then back it off the minimal amount
				double dist = Math.sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1));

				// if this is not power/ground, double the distance
				if (!pp1.getPlacementNetwork().isOnRail())
					dist *= 2;
				totalLen += dist;
			}
//System.out.println("WHEN NODE "+pn1+" IS ROTATED "+spin1.toString()+" IT MOVES ("+TextUtils.formatDouble(dX)+","+TextUtils.formatDouble(dY)+") AND HAS TOTAL LENGTH "+totalLen);

			// determine the area of the combined clusters
			double lXTot = cluster2.getBounds().getMinX();
			double hXTot = cluster2.getBounds().getMaxX();
			double lYTot = cluster2.getBounds().getMinY();
			double hYTot = cluster2.getBounds().getMaxY();
			for(ProxyNode pn : cluster1.nodesInCluster)
			{
				double lX = pn.getX() - pn.getWidth()/2 + dX;
				double hX = lX + pn.getWidth();
				double lY = pn.getY() - pn.getHeight()/2 + dY;
				double hY = lY + pn.getHeight();
				if (lX < lXTot) lXTot = lX;
				if (hX > hXTot) hXTot = hX;
				if (lY < lYTot) lYTot = lY;
				if (hY > hYTot) hYTot = hY;
			}
			overallBound = ERectangle.fromLambda(lXTot, lYTot, hXTot-lXTot, hYTot-lYTot);
			
			// restore information for nodes in cluster
			for(ProxyNode pn : cluster1.nodesInCluster) pn.restoreValues();
			cluster1.computeBounds();

			return totalLen;
		}
	}

	/*************************************** Force Computation **********************************/

	class ComputeForce
	{
		private double dX, dY;						// forces on the node
		private int numMoved;						// number of forces on the node
		private double[] dXQ, dYQ;
		private int[] numMovedQ;
		private ProxyNode px1, px2;
		private List<EPoint> perfectAlignment;
		private List<EPoint> imperfectAlignment;

		public ComputeForce(ProposedPlacement pp, ProxyNode px1, ProxyNode px2)
		{
			this.px1 = px1;
			this.px2 = px2;
			dXQ = new double[8];
			dYQ = new double[8];
			numMovedQ = new int[8];
			perfectAlignment = new ArrayList<EPoint>();
			imperfectAlignment = new ArrayList<EPoint>();
		}

		public void accumulateForce(double addX, double addY, boolean isOnRail)
		{
//System.out.println("  FORCE COMPONENT MOVES "+px1+" ("+TextUtils.formatDouble(addX)+","+TextUtils.formatDouble(addY)+")");
			dX += addX;   dY += addY;   numMoved++;

			int quadrant = -1;
			if (addX == 0)
			{
				if (addY > 0) quadrant = 2; else
					if (addY < 0) quadrant = 6;
			} else if (addX > 0)
			{
				if (addY == 0) quadrant = 0; else
					if (addY > 0) quadrant = 1; else
						quadrant = 7;
			} else
			{
				if (addY == 0) quadrant = 4; else
					if (addY > 0) quadrant = 3; else
						quadrant = 5;
			}
			if (quadrant >= 0)
			{
				dXQ[quadrant] += addX;
				dYQ[quadrant] += addY;
				numMovedQ[quadrant]++;
			}
boolean alignedPerfectly = false, alignedImperfectly = false;
double top1 = px1.getY() + px1.getHeight()/2 + addY;
double bot1 = px1.getY() - px1.getHeight()/2 + addY;
double right1 = px1.getX() + px1.getWidth()/2 + addX;
double left1 = px1.getX() - px1.getWidth()/2 + addX;
double top2 = px2.getY() + px2.getHeight()/2;
double bot2 = px2.getY() - px2.getHeight()/2;
double right2 = px2.getX() + px2.getWidth()/2;
double left2 = px2.getX() - px2.getWidth()/2;
if (DBMath.areEquals(top1, top2) && DBMath.areEquals(bot1, bot2)) alignedPerfectly = true;
if (DBMath.areEquals(right1, right2) && DBMath.areEquals(left1, left2)) alignedPerfectly = true;
if (DBMath.areEquals(top1, top2) || DBMath.areEquals(bot1, bot2)) alignedImperfectly = true;
if (DBMath.areEquals(top1, top2) || DBMath.areEquals(bot1, bot2)) alignedImperfectly = true;
//System.out.println("   SO   FOR "+px1+", TOP="+top1+" BOTTOM="+bot1+" LEFT="+left1+" RIGHT="+right1);
//System.out.println("    AND FOR "+px2+", TOP="+top2+" BOTTOM="+bot2+" LEFT="+left2+" RIGHT="+right2);

//if (aligned && debug) System.out.println("SHIFT "+px1+" BY ("+addX+","+addY+") ALIGNS WITH "+px2);
if (isOnRail) alignedPerfectly = alignedImperfectly = true;
if (alignedPerfectly)
{
	perfectAlignment.add(new EPoint(addX, addY));
//	if (debug) System.out.println("SHIFT "+px1+" BY ("+newDelta.getX()+","+newDelta.getY()+") ALIGNS WITH "+px2);
//	if (debug) System.out.println("   WHICH IS AT "+(px2.x-px2.width/2)+"<=X<="+(px2.x+px2.width/2)+" AND "+(px2.y-px2.height/2)+"<=Y<="+(px2.y+px2.height/2));
}
//if (alignedImperfectly)
//	imperfectAlignment.add(new EPoint(addX, addY));
		}

		public void normalizeForce()
		{
			if (numMoved != 0)
			{
				dX /= numMoved;
				dY /= numMoved;
			}
			if (numMoved > 1)
			{
				int biggestQuadrant = -1;
				int numInBiggestQuadrant = 0;
//System.out.print("   QUADRANT CONNECTIONS ARE");
				for(int i=0; i<8; i++)
				{
//System.out.print(" ["+i+"]="+numMovedQ[i]);
					if (numMovedQ[i] > numInBiggestQuadrant)
					{
						numInBiggestQuadrant = numMovedQ[i];
						biggestQuadrant = i;
					}
					if (numMovedQ[i] > 0)
					{
						dXQ[i] /= numMovedQ[i];
						dYQ[i] /= numMovedQ[i];
					}
				}
				if (biggestQuadrant >= 0)
				{
					if (numInBiggestQuadrant*2 > numMoved)
					{
						dX = dXQ[biggestQuadrant];
						dY = dYQ[biggestQuadrant];
					}
				}
			}
//System.out.println(" SO MOVE BY ("+TextUtils.formatDouble(dX)+","+TextUtils.formatDouble(dY)+") WITH "+perfectAlignment.size()+" PERFECT CHOICES");
			if (perfectAlignment.size() > 0 || imperfectAlignment.size() > 0)
			{
				double bestDist = Double.MAX_VALUE;
				EPoint bestMove = null;
				for(EPoint move : perfectAlignment)
				{
					double offX = dX - move.getX();
					double offY = dY - move.getY();
					double dist = Math.sqrt(offX*offX + offY*offY);
					if (dist < bestDist)
					{
						bestDist = dist;
						bestMove = move;
					}
				}
				if (perfectAlignment.size() == 0)
				{
					for(EPoint move : imperfectAlignment)
					{
						double offX = dX - move.getX();
						double offY = dY - move.getY();
						double dist = Math.sqrt(offX*offX + offY*offY);
						if (dist < bestDist)
						{
							bestDist = dist;
							bestMove = move;
						}
					}
				}
				if (bestMove != null)
				{
					dX = bestMove.getX();
					dY = bestMove.getY();
//System.out.println("   WAIT! PERFECT ALIGNMENT MAKES IT ("+dX+","+dY+")");
				}
			}
		}

		public double getForceX() { return DBMath.round(dX); }

		public double getForceY() { return DBMath.round(dY); }
	}

	/**
	 * Method to determine how to adjust a cluster when a different one is moved.
	 * @param pc1 the cluster being moved.
	 * @param pc2 the cluster being affected.
	 * @param dX the X amount that the first cluster is being moved.
	 * @param dY the Y amount that the first cluster is being moved.
	 * @return an EPoint with the actual motion for the cluster being moved.
	 */
	private EPoint pushCluster(ProxyCluster pcMoving, ProxyCluster pcFixed, double dX, double dY, int forceDir, ProposedPlacement debug)
	{
		// make an R-Tree with the non-moving cluster
		RTNode root = RTNode.makeTopLevel();
		for(ProxyNode pnFixed : pcFixed.getNodesInCluster())
			root = RTNode.linkGeom(null, root, pnFixed);
//if (debug != null)
//{
//	debug.furtherExplanation.add("Best move is ("+dX+","+dY+") with these clusters:");
//	String msg = "   FIXED CLUSTER IS "+pcFixed.clusterIndex+" WHICH IS "+pcFixed.getBounds().getMinX()+"<=X<="+
//		pcFixed.getBounds().getMaxX()+" AND "+pcFixed.getBounds().getMinY()+"<=Y<="+pcFixed.getBounds().getMaxY()+" WITH";
//	for(ProxyNode pn : pcFixed.nodesInCluster) msg += " "+pn.getNodeName();
//	debug.furtherExplanation.add(msg);
//	for(ProxyNode pn : pcFixed.nodesInCluster)
//		debug.furtherExplanation.add(pn.getNodeName()+" IS AT ("+pn.getX()+","+pn.getY()+") SO IS "+pn.getBounds().getMinX()+"<=X<="+pn.getBounds().getMaxX()+" AND "+pn.getBounds().getMinY()+"<=Y<="+pn.getBounds().getMaxY());
//	msg = "   MOVING CLUSTER IS "+pcMoving.clusterIndex+" WHICH IS "+pcMoving.getBounds().getMinX()+"<=X<="+
//		pcMoving.getBounds().getMaxX()+" AND "+pcMoving.getBounds().getMinY()+"<=Y<="+pcMoving.getBounds().getMaxY()+" WITH";
//	for(ProxyNode pn : pcMoving.nodesInCluster) msg += " "+pn.getNodeName();
//	debug.furtherExplanation.add(msg);
//}
		double bestLeftMotion = 0, bestRightMotion = 0, bestUpMotion = 0, bestDownMotion = 0;
		boolean tryNegLeft = true, tryNegRight = true, tryNegUp = true, tryNegDown = true;
		boolean intersects = true;
		while (intersects)
		{
			intersects = false;
			for(ProxyNode pnMoving : pcMoving.getNodesInCluster())
			{
				// see if moving cluster must move left to avoid fixed cluster
				ERectangle boundMovingLeft = ERectangle.fromLambda(pnMoving.getX() + dX - bestLeftMotion - pnMoving.getWidth()/2,
					pnMoving.getY() + dY - pnMoving.getHeight()/2, pnMoving.getWidth(), pnMoving.getHeight());
				for (RTNode.Search sea = new RTNode.Search(boundMovingLeft, root, true); sea.hasNext();)
				{
					ERectangle boundFixed = ((ProxyNode)sea.next()).getBounds();
					if (boundFixed.getMaxX() > boundMovingLeft.getMinX() && boundFixed.getMinX() < boundMovingLeft.getMaxX() &&
						boundFixed.getMaxY() > boundMovingLeft.getMinY() && boundFixed.getMinY() < boundMovingLeft.getMaxY())
					{
						bestLeftMotion += boundMovingLeft.getMaxX() - boundFixed.getMinX();
						intersects = true;
						tryNegLeft = false;
						break;
					}
				}
				if (bestLeftMotion <= 0 && tryNegLeft)
				{
					// may have to shift right to pack closely
					bestLeftMotion -= pcMoving.getNodesInCluster().get(0).getWidth();
					if (pcMoving.getBounds().getMinX() - bestLeftMotion >= pcFixed.getBounds().getMaxX())
					{
						bestLeftMotion = 0;
						tryNegLeft = false;
					}
					intersects = true;
				}

				// see if moving cluster must move right to avoid fixed cluster
				ERectangle boundMovingRight = ERectangle.fromLambda(pnMoving.getX() + dX + bestRightMotion - pnMoving.getWidth()/2,
					pnMoving.getY() + dY - pnMoving.getHeight()/2, pnMoving.getWidth(), pnMoving.getHeight());
				for (RTNode.Search sea = new RTNode.Search(boundMovingRight, root, true); sea.hasNext();)
				{
					ERectangle boundFixed = ((ProxyNode)sea.next()).getBounds();
					if (boundFixed.getMaxX() > boundMovingRight.getMinX() && boundFixed.getMinX() < boundMovingRight.getMaxX() &&
						boundFixed.getMaxY() > boundMovingRight.getMinY() && boundFixed.getMinY() < boundMovingRight.getMaxY())
					{
						bestRightMotion += boundFixed.getMaxX() - boundMovingRight.getMinX();
						intersects = true;
						tryNegRight = false;
						break;
					}
				}
				if (bestRightMotion <= 0 && tryNegRight)
				{
					// may have to shift left to pack closely
					bestRightMotion -= pcMoving.getNodesInCluster().get(0).getWidth();
					if (pcMoving.getBounds().getMaxX() + bestRightMotion <= pcFixed.getBounds().getMinX())
					{
						bestRightMotion = 0;
						tryNegRight = false;
					}
					intersects = true;
				}

				// see if moving cluster must move up to avoid fixed cluster
				ERectangle boundMovingUp = ERectangle.fromLambda(pnMoving.getX() + dX - pnMoving.getWidth()/2,
					pnMoving.getY() + dY + bestUpMotion - pnMoving.getHeight()/2, pnMoving.getWidth(), pnMoving.getHeight());
				for (RTNode.Search sea = new RTNode.Search(boundMovingUp, root, true); sea.hasNext();)
				{
					ERectangle boundFixed = ((ProxyNode)sea.next()).getBounds();
					if (boundFixed.getMaxX() > boundMovingUp.getMinX() && boundFixed.getMinX() < boundMovingUp.getMaxX() &&
						boundFixed.getMaxY() > boundMovingUp.getMinY() && boundFixed.getMinY() < boundMovingUp.getMaxY())
					{
						bestUpMotion += boundFixed.getMaxY() - boundMovingUp.getMinY();
						intersects = true;
						tryNegUp = false;
						break;
					}
				}
				if (bestUpMotion <= 0 && tryNegUp)
				{
					// may have to shift down to pack closely
					bestUpMotion -= pcMoving.getNodesInCluster().get(0).getHeight();
					if (pcMoving.getBounds().getMaxY() + bestUpMotion <= pcFixed.getBounds().getMinY())
					{
						bestUpMotion = 0;
						tryNegUp = false;
					}
					intersects = true;
				}

				// see if moving cluster must move down to avoid fixed cluster
				ERectangle boundMovingDown = ERectangle.fromLambda(pnMoving.getX() + dX - pnMoving.getWidth()/2,
					pnMoving.getY() + dY - bestDownMotion - pnMoving.getHeight()/2, pnMoving.getWidth(), pnMoving.getHeight());
				for (RTNode.Search sea = new RTNode.Search(boundMovingDown, root, true); sea.hasNext();)
				{
					ProxyNode pnf = (ProxyNode)sea.next();
					ERectangle boundFixed = pnf.getBounds();
					if (boundFixed.getMaxX() > boundMovingDown.getMinX() && boundFixed.getMinX() < boundMovingDown.getMaxX() &&
						boundFixed.getMaxY() > boundMovingDown.getMinY() && boundFixed.getMinY() < boundMovingDown.getMaxY())
					{
//if (debug != null)
//{
//	debug.furtherExplanation.add("DOWN MOTION MOVES "+(boundMovingDown.getMaxY() - boundFixed.getMinY())+" MORE...");
//	debug.furtherExplanation.add("...BECAUSE FIXED BLOCK "+pnf.getNodeName()+" IS AT "+boundFixed.getMinX()+"<=X<="+boundFixed.getMaxX()+" AND "+
//		boundFixed.getMinY()+"<=Y<="+boundFixed.getMaxY());
//	debug.furtherExplanation.add("...AND MOVING BLOCK "+pnMoving.getNodeName()+" IS AT "+boundMovingDown.getMinX()+"<=X<="+boundMovingDown.getMaxX()+" AND "+
//		boundMovingDown.getMinY()+"<=Y<="+boundMovingDown.getMaxY());
//}
						bestDownMotion += boundMovingDown.getMaxY() - boundFixed.getMinY();
						intersects = true;
						tryNegDown = false;
						break;
					}
				}
				if (bestDownMotion <= 0 && tryNegDown)
				{
					// may have to shift up to pack closely
					bestDownMotion -= pcMoving.getNodesInCluster().get(0).getHeight();
					if (pcMoving.getBounds().getMinY() - bestDownMotion >= pcFixed.getBounds().getMaxY())
					{
						bestDownMotion = 0;
						tryNegDown = false;
					}
					intersects = true;
				}
	
				if (intersects) break;
			}
		}

//if (debug != null)
//	debug.furtherExplanation.add(" BEST MOTIONS ARE LEFT="+bestLeftMotion+" RIGHT="+bestRightMotion+" UP="+bestUpMotion+" DOWN="+bestDownMotion);
		double moveX = 0, moveY = 0;
		switch (forceDir)
		{
			case 0:			// left
				moveX = -bestLeftMotion;  break;
			case 1:			// right
				moveX = bestRightMotion;  break;
			case 2:			// up
				moveY = bestUpMotion;     break;
			case 3:			// down
				moveY = -bestDownMotion;  break;
			default:
				if (bestLeftMotion < bestRightMotion) moveX = -bestLeftMotion; else moveX = bestRightMotion;
				if (bestDownMotion < bestUpMotion) moveY = -bestDownMotion; else moveY = bestUpMotion;
				if (Math.abs(moveX) > Math.abs(moveY)) moveX = 0; else moveY = 0;
				break;
		}
//if (debug != null)
//	debug.furtherExplanation.add(" SO ADJUSTING MOVE BY ("+moveX+","+moveY+")");
		return new EPoint(moveX+dX, moveY+dY);
	}

    /*************************************** Proxy Cluster **********************************/

	private static class ProxyCluster implements RTBounds, Comparable<ProxyCluster>
	{
		private List<ProxyNode> nodesInCluster;
		private ERectangle bound;
		private int clusterIndex;

		ProxyCluster()
		{
			nodesInCluster = new ArrayList<ProxyNode>();
		}

		public int compareTo(ProxyCluster pcO)
        {
        	double sizeA = bound.getWidth() * bound.getHeight();
        	double sizeB = pcO.bound.getWidth() * pcO.bound.getHeight();
        	if (sizeA > sizeB) return -1;
        	if (sizeA < sizeB) return 1;
            return 0;
        }

		public ERectangle getBounds() { return bound; }

		public List<ProxyNode> getNodesInCluster() { return nodesInCluster; }

		public void computeBounds()
		{
			double lXAll=0, hXAll=0, lYAll=0, hYAll=0;
			for(int i=0; i<nodesInCluster.size(); i++)
			{
				ProxyNode pn = nodesInCluster.get(i);
				double lX = pn.getX() - pn.getWidth()/2;
				double hX = lX + pn.getWidth();
				double lY = pn.getY() - pn.getHeight()/2;
				double hY = lY + pn.getHeight();
				if (i == 0)
				{
					lXAll = lX;   hXAll = hX;
					lYAll = lY;   hYAll = hY;
				} else
				{
					if (lX < lXAll) lXAll = lX;
					if (hX > hXAll) hXAll = hX;
					if (lY < lYAll) lYAll = lY;
					if (hY > hYAll) hYAll = hY;
				}
			}
			bound = ERectangle.fromLambda(lXAll, lYAll, hXAll-lXAll, hYAll-lYAll);
		}

		public double getAreaWithOtherShiftedCluster(ProxyCluster other, EPoint delta)
		{
			double lXAll=bound.getMinX(), hXAll=bound.getMaxX(), lYAll=bound.getMinY(), hYAll=bound.getMaxY();
			for(int i=0; i<other.getNodesInCluster().size(); i++)
			{
				ProxyNode pn = other.getNodesInCluster().get(i);
				double lX = pn.getX() + delta.getX() - pn.getWidth()/2;
				double hX = lX + pn.getWidth();
				double lY = pn.getY() + delta.getY() - pn.getHeight()/2;
				double hY = lY + pn.getHeight();
				if (lX < lXAll) lXAll = lX;
				if (hX > hXAll) hXAll = hX;
				if (lY < lYAll) lYAll = lY;
				if (hY > hYAll) hYAll = hY;
			}
			double area = (hXAll - lXAll) * (hYAll - lYAll);
			return area;
		}

		public void rotate(Orientation spin)
		{
			if (spin == Orientation.IDENT) return;
			for(ProxyNode pn : nodesInCluster)
			{
				double ox = pn.getX() - bound.getCenterX();
				double oy = pn.getY() - bound.getCenterY();
				Point2D newOff = spin.transformPoint(new Point2D.Double(ox, oy));
				pn.setLocation(newOff.getX() + bound.getCenterX(), newOff.getY() + bound.getCenterY());
				pn.changeOrientation(spin);
			}
			computeBounds();
		}

		public void add(ProxyNode pNode)
		{
			nodesInCluster.add(pNode);
			ERectangle b = ERectangle.fromLambda(pNode.getX() - pNode.getWidth()/2,
				pNode.getY() - pNode.getHeight()/2, pNode.getWidth(), pNode.getHeight());
			if (nodesInCluster.size() == 1)
			{
				bound = b;
			} else
			{
				bound = (ERectangle)bound.createUnion(b);
			}
		}

		public String toString()
		{
			String str = "CLUSTER";
			String sep = " ";
			for(ProxyNode pNode : nodesInCluster)
			{
				str += sep + pNode.getNodeName();
				sep = "/";
			}
			return str;
		}
	}

	/*************************************** Proxy Node **********************************/

	/**
	 * This class is a proxy for the actual PlacementNode.
	 * It can be moved around and rotated without touching the actual PlacementNode.
	 */
	class ProxyNode implements RTBounds
	{
		// current state
		private double x, y;						// current location
		private Orientation orientation;			// orientation of the placement node
		private double width, height;				// height of the placement node

		// saved values for temporary adjustments
		private double xSave, ySave;				// current location
		private Orientation orientationSave;		// orientation of the placement node
		private double widthSave, heightSave;		// height of the placement node

		private ERectangle bounds;
		private PlacementNode original;

		/**
		 * Constructor to create a ProxyNode
		 * @param node the PlacementNode that should is being shadowed.
		 */
		public ProxyNode(PlacementNode node)
		{
			original = node;
			NodeInst ni = ((PlacementAdapter.PlacementNode)node).getOriginal();
			x = DBMath.round(ni.getTrueCenterX());
			y = DBMath.round(ni.getTrueCenterY());
			orientation = ni.getOrient();

			NodeProto np = ((PlacementAdapter.PlacementNode)node).getType();
			Rectangle2D spacing = null;
			if (np instanceof Cell)
				spacing = ((Cell)np).findEssentialBounds();

			if (spacing == null)
			{
				width = node.getWidth();
				height = node.getHeight();
			} else
			{
				width = spacing.getWidth();
				height = spacing.getHeight();
			}
			if (ni.getOrient().getAngle() == 900 || ni.getOrient().getAngle() == 2700)
			{
				double swap = width;   width = height;   height = swap;
			}
			computeBounds();
		}

		/**
		 * Constructor to duplicate a ProxyNode.
		 * @param pn the ProxyNode to copy.
		 */
		public ProxyNode(ProxyNode pn)
		{
			x = pn.x;
			y = pn.y;
			orientation = pn.orientation;
			width = pn.width;
			height = pn.height;
			original = pn.original;
			computeBounds();
		}

		public String toString()
		{
			NodeInst ni = ((PlacementAdapter.PlacementNode)original).getOriginal();
			return ni.describe(false);
		}

		public String getNodeName()
		{
			NodeInst ni = ((PlacementAdapter.PlacementNode)original).getOriginal();
			return ni.getName();
		}

		// ----------------------------- COORDINATES -----------------------------

		/**
		 * Method to get the X-coordinate of this ProxyNode.
		 * @return the X-coordinate of this ProxyNode.
		 */
		public double getX() { return x; }

		/**
		 * Method to get the Y-coordinate of this ProxyNode.
		 * @return the Y-coordinate of this ProxyNode.
		 */
		public double getY() { return y; }

		public void setLocation(double x, double y)
		{
			this.x = DBMath.round(x);
			this.y = DBMath.round(y);
			computeBounds();
		}

		/**
		 * Method to get the width of this ProxyNode.
		 * @return the width of this ProxyNode.
		 */
		public double getWidth() { return width; }

		/**
		 * Method to get the height of this ProxyNode.
		 * @return the height of this ProxyNode.
		 */
		public double getHeight() { return height; }

		/**
		 * Method to get the orientation of this ProxyNode.
		 * @return the orientation of this ProxyNode.
		 */
		public Orientation getOrientation() { return orientation; }

		public void changeOrientation(Orientation delta)
		{
			Orientation newOrientation = delta.concatenate(orientation);
			int deltaAng = Math.abs(orientation.getAngle() - newOrientation.getAngle());
			if (deltaAng == 900 || deltaAng == 2700)
			{
				double swap = width;   width = height;   height = swap;
				computeBounds();
			}
			orientation = newOrientation;
		}

		public ERectangle getBounds() { return bounds; }

		private void computeBounds()
		{
			bounds = ERectangle.fromLambda(x - width/2, y - height/2, width, height);
		}

		// ----------------------------- MISCELLANEOUS -----------------------------

		public void saveValues()
		{
			xSave = x;   ySave = y;
			orientationSave = orientation;
			widthSave = width;   heightSave = height;
		}

		public void restoreValues()
		{
			x = xSave;   y = ySave;
			orientation = orientationSave;
			width = widthSave;   height = heightSave;
			computeBounds();
		}
	}

	/* ********************************************* DEBUGGING ********************************************* */

//	static Map<Object,Integer> objMap = new HashMap<Object,Integer>();
//	static int objIndices = 0;
//
//	private String getObjName(Object obj)
//	{
//		Integer x = objMap.get(obj);
//		if (x == null) objMap.put(obj, x = Integer.valueOf(objIndices++));
//		return "OBJ"+x.intValue();
//	}

	private static final int INDENTPOLY = 5;
	private static int showIndex = 0;
	private void showProposal(ProposedPlacement pp)
	{
		showIndex++;
		Cell cell = Cell.newInstance(Library.getCurrent(), "DEBUG"+showIndex);
		Point2D fromPt = new Point2D.Double(0, 0);
		Point2D toPt = new Point2D.Double(0, 0);
		ProxyNode fromPN = pp.proxyMap.get(pp.movedPair.n1);
		ProxyNode toPN = pp.proxyMap.get(pp.movedPair.n2);
		for(ProxyNode pn : pp.nodesToPlace)
		{
			NodeInst ni = ((PlacementAdapter.PlacementNode)pn.original).getOriginal();
			double width = ni.getProto().getDefWidth();
			double height = ni.getProto().getDefHeight();
			double xPos = pn.getX();
			double yPos = pn.getY();
			if (ni.isCellInstance())
			{
				Cell placementCell = (Cell)ni.getProto();
				Rectangle2D bounds = placementCell.getBounds();
				Point2D centerOffset = new Point2D.Double(bounds.getCenterX(), bounds.getCenterY());
				pn.getOrientation().pureRotate().transform(centerOffset, centerOffset);
				xPos -= centerOffset.getX();
				yPos -= centerOffset.getY();
			}
			EPoint ctr = new EPoint(xPos, yPos);
			if (pn == fromPN) fromPt = ctr;
			if (pn == toPN) toPt = ctr;
			NodeInst newNI = NodeInst.makeInstance(ni.getProto(), ctr, width, height, cell, pn.getOrientation(), ni.getName());
			TextDescriptor td = TextDescriptor.getNodeTextDescriptor().withDisplay(true).withRelSize(20);
			newNI.setTextDescriptor(NodeInst.NODE_NAME, td);
		}

		// show all clusters
		for(ProxyCluster cluster : pp.allClusters)
		{
			if (cluster.getNodesInCluster().size() <= 1) continue;
			PolyMerge merge = new PolyMerge();
			for(ProxyNode pn : cluster.getNodesInCluster())
			{
				Poly poly = new Poly(pn.getBounds());
				merge.add(Artwork.tech().defaultLayer, poly);
			}
			List<PolyBase> polys = merge.getMergedPoints(Artwork.tech().defaultLayer, true);
			for(PolyBase pb : polys)
			{
				Point2D [] newPts = pb.getPoints();
				double cX = pb.getCenterX();
				double cY = pb.getCenterY();
				for(int i=0; i<newPts.length; i++)
				{
					double nX = newPts[i].getX();
					double nY = newPts[i].getY();
					if (nX < cX) nX += INDENTPOLY; else
						if (nX > cX) nX -= INDENTPOLY;
					if (nY < cY) nY += INDENTPOLY; else
						if (nY > cY) nY -= INDENTPOLY;
					newPts[i] = new EPoint(nX, nY);
				}
				NodeInst newNI = NodeInst.makeInstance(Artwork.tech().openedPolygonNode, new EPoint(cX, cY),
					pb.getBounds2D().getWidth()-INDENTPOLY*2, pb.getBounds2D().getHeight()-INDENTPOLY*2, cell);
				newNI.setTrace(newPts);
				newNI.newVar(Artwork.ART_COLOR, new Integer(EGraphics.RED));
			}
		}

		// show what was moved
		NodeInst arrowHead = NodeInst.makeInstance(Artwork.tech().pinNode, fromPt, 0, 0, cell);
		NodeInst arrowTail = NodeInst.makeInstance(Artwork.tech().pinNode, toPt, 0, 0, cell);
		ArcInst ai = ArcInst.makeInstance(Artwork.tech().thickerArc, arrowHead.getOnlyPortInst(), arrowTail.getOnlyPortInst());
		ai.newVar(Artwork.ART_COLOR, new Integer(EGraphics.GREEN));
		int angle = DBMath.figureAngle(toPt, fromPt);
		int ang1 = (angle + 450) % 3600;
		int ang2 = (angle + 3150) % 3600;
		double cX = (fromPt.getX() + toPt.getX()) / 2;
		double cY = (fromPt.getY() + toPt.getY()) / 2;
		double len = DBMath.distBetweenPoints(fromPt, toPt) / 5;
		double x1 = cX + DBMath.cos(ang1) * len;
		double y1 = cY + DBMath.sin(ang1) * len;
		double x2 = cX + DBMath.cos(ang2) * len;
		double y2 = cY + DBMath.sin(ang2) * len;
		NodeInst arrowCtr = NodeInst.makeInstance(Artwork.tech().pinNode, new EPoint(cX, cY), 0, 0, cell);
		NodeInst arrowEnd1 = NodeInst.makeInstance(Artwork.tech().pinNode, new EPoint(x1, y1), 0, 0, cell);
		NodeInst arrowEnd2 = NodeInst.makeInstance(Artwork.tech().pinNode, new EPoint(x2, y2), 0, 0, cell);
		ai = ArcInst.makeInstance(Artwork.tech().thickerArc, arrowCtr.getOnlyPortInst(), arrowEnd1.getOnlyPortInst());
		ai.newVar(Artwork.ART_COLOR, new Integer(EGraphics.GREEN));
		ai = ArcInst.makeInstance(Artwork.tech().thickerArc, arrowCtr.getOnlyPortInst(), arrowEnd2.getOnlyPortInst());
		ai.newVar(Artwork.ART_COLOR, new Integer(EGraphics.GREEN));

		// write text at top describing what happened
		double x = cell.getBounds().getCenterX();
		double y = cell.getBounds().getMaxY() + 10;
		PrimitiveNode np = Generic.tech().invisiblePinNode;
		for(int i=pp.furtherExplanation.size()-1; i >= 0; i--)
		{
			NodeInst ni = NodeInst.makeInstance(np, new EPoint(x, y), 0, 0, cell);
			TextDescriptor td = TextDescriptor.getAnnotationTextDescriptor().withDisplay(true).withRelSize(10);
			ni.newVar(Artwork.ART_MESSAGE, pp.furtherExplanation.get(i), td);
			y += 10;
		}
		NodeInst ni = NodeInst.makeInstance(np, new EPoint(x, y+5), 0, 0, cell);
		String msg = "Proposal "+pp.proposalNumber+" from parent proposal "+pp.parentProposalNumber;
		TextDescriptor td = TextDescriptor.getAnnotationTextDescriptor().withDisplay(true).withRelSize(15);
		ni.newVar(Artwork.ART_MESSAGE, msg, td);
	}

//	private static List<PlacementNode> debugPlacementNodes;
//	private static List<PlacementNetwork> debugPlacementNetworks;
//	private static List<PlacementConnection> debugAllConnections;
//	private static ProposedPlacement debugProposed;
//	private static Map<String, NodeInst> placementMap;
//	private static Map<ProxyNode,PlacementAdapter.PlacementNode> backMap;
//
//	private void initializeClusteringDebugging(List<PlacementNode> placementNodes, ProposedPlacement currentProposed,
//		List<PlacementNetwork> allNetworks)
//	{
//		debugPlacementNodes = placementNodes;
//		debugPlacementNetworks = allNetworks;
//		debugProposed = currentProposed;
//
//		debugAllConnections = new ArrayList<PlacementConnection>();
//		for(PlacementNetwork pNet : allNetworks)
//		{
//			List<PlacementConnection> cons = PlacementAdapter.getOptimalConnections(pNet);
//			for(PlacementConnection con : cons) debugAllConnections.add(con);
//		}
//
//		backMap = new HashMap<ProxyNode,PlacementAdapter.PlacementNode>();
//		for (PlacementNode pn : placementNodes)
//		{
//			PlacementAdapter.PlacementNode pnReal = (PlacementAdapter.PlacementNode)pn;
//			backMap.put(debugProposed.proxyMap.get(pn), pnReal);
//		}
//
//		SwingUtilities.invokeLater(new Runnable() {
//			public void run() { new PlacementProgress(); }
//		});
//	}
//
//	/**
//	 * Class to debug placement.
//	 */
//	public class PlacementProgress extends EDialog
//	{
//		public PlacementProgress()
//		{
//			super(TopLevel.getCurrentJFrame(), false);
//
//			GridBagConstraints gridBagConstraints;
//			getContentPane().setLayout(new GridBagLayout());
//			setTitle("Debug Placement");
//			setName("");
//			addWindowListener(new WindowAdapter()
//			{
//				public void windowClosing(WindowEvent evt) { closeDialog(evt); }
//			});
//
//			JButton theBut = new JButton("Next Step");
//			theBut.addActionListener(new ActionListener()
//			{
//				public void actionPerformed(ActionEvent evt) { planMove(); }
//			});
//			gridBagConstraints = new GridBagConstraints();
//			gridBagConstraints.gridx = 0;
//			gridBagConstraints.gridy = 0;
//			gridBagConstraints.insets = new Insets(4, 4, 4, 4);
//			getContentPane().add(theBut, gridBagConstraints);
//
//			pack();
//			finishInitialization();
//			setVisible(true);
//			
////			showNextPair();
//		}
//
//		protected void escapePressed() { closeDialog(null); }
//
//		private void planMove()
//		{
//			List<ProposedPlacement> proposals = currentProposal.clusteringPlacementStep(debugPlacementNetworks);
//			if (proposals != null)
//			{
//				placementMap = PlacementAdapter.getPlacementMap();
//				for (PlacementNode pn : debugPlacementNodes)
//				{
//					ProxyNode p = debugProposed.proxyMap.get(pn);
//					pn.setPlacement(p.getX(), p.getY());
//					pn.setOrientation(p.getOrientation());
//				}
//				new MakeIntermediateMove();
//			}
//		}
//
//		/** Closes the dialog */
//		private void closeDialog(WindowEvent evt)
//		{
//			setVisible(false);
//			dispose();
//		}
//	}
//
//	private static Orientation[] allOrientations = {
//		Orientation.IDENT, Orientation.R,   Orientation.RR,   Orientation.RRR,
//		Orientation.X,     Orientation.XR,  Orientation.XRR,  Orientation.XRRR,
//		Orientation.Y,     Orientation.YR,  Orientation.YRR,  Orientation.YRRR,
//		Orientation.XY,    Orientation.XYR, Orientation.XYRR, Orientation.XYRRR};
//	private static Map<Orientation,Map<Orientation,Orientation>> inverseMap =
//		new HashMap<Orientation,Map<Orientation,Orientation>>();
//
//	/**
//	 * Method to determine the Orientation to use to convert one Orientation to another.
//	 * @param a the original Orientation.
//	 * @param b the final Orientation.
//	 * @return the Orientation that converts the original to the final.
//	 */
//	private static Orientation subtractOrientation(Orientation a, Orientation b)
//	{
//		Map<Orientation,Orientation> cache = inverseMap.get(a);
//		if (cache == null) inverseMap.put(a, cache = new HashMap<Orientation,Orientation>());
//		Orientation resultO = cache.get(b);
//		if (resultO == null)
//		{		
//			resultO = a;
//			for(int i=0; i<allOrientations.length; i++)
//			{
//		        Orientation finalO = allOrientations[i].concatenate(a);
//				if (finalO == b) { resultO = allOrientations[i];   break; }
//			}
//			cache.put(b, resultO);
//		}
//		return resultO;
//	}
//
//	private static class MakeIntermediateMove extends Job
//	{
//		private MakeIntermediateMove()
//		{
//			super("Place cells", User.getUserTool(), Job.Type.CHANGE, null, null, Job.Priority.USER);
//			startJob();
//		}
//
//		public boolean doIt() throws JobException
//		{
//			Cell cell = null;
//			for(PlacementNode plNode : debugPlacementNodes)
//			{
//				PlacementAdapter.PlacementNode pan = (PlacementAdapter.PlacementNode)plNode;
//				NodeInst bmNI = pan.getOriginal();
//				if (bmNI == null) continue;
//				NodeInst newNI = placementMap.get(bmNI.getName());
//				if (newNI == null) continue;
//
//				double xPos = plNode.getPlacementX();
//				double yPos = plNode.getPlacementY();
//				Orientation orient = plNode.getPlacementOrientation();
//				Orientation deltaO = subtractOrientation(newNI.getOrient(), orient);
//				if (pan.getOriginal().isCellInstance())
//				{
//					Cell placementCell = (Cell)pan.getOriginal().getProto();
//					Rectangle2D bounds = placementCell.getBounds();
//					Point2D centerOffset = new Point2D.Double(bounds.getCenterX(), bounds.getCenterY());
//					orient.pureRotate().transform(centerOffset, centerOffset);
////System.out.println("FINALLY, NODE "+newNI.getName()+" MOVES TO ("+xPos+","+yPos+"), ROTATED "+orient.toString());
////System.out.println("  CONSIDERING CENTER OFFSET ("+centerOffset.getX()+","+centerOffset.getY()+") IT NOW IS ("+
////	(xPos-centerOffset.getX())+","+(yPos-centerOffset.getY())+")");
//					xPos -= centerOffset.getX();
//					yPos -= centerOffset.getY();
//				}
//				if (newNI.getAnchorCenterX() == xPos && newNI.getAnchorCenterY() == yPos)
//					continue;
//
//				double dX = xPos - newNI.getAnchorCenterX();
//				double dY = yPos - newNI.getAnchorCenterY();
//				newNI.modifyInstance(dX, dY, 0, 0, deltaO);
//				cell = newNI.getParent();
//			}
//
//			if (cell != null)
//			{
//				// rip out all arcs
//				List<ArcInst> allArcs = new ArrayList<ArcInst>();
//				for(Iterator<ArcInst> it = cell.getArcs(); it.hasNext(); ) allArcs.add(it.next());
//				for(ArcInst ai : allArcs) ai.kill();
//
//				// redo arcs in optimal way
//				ImmutableArcInst a = Generic.tech().unrouted_arc.getDefaultInst(cell.getEditingPreferences());
//				long gridExtend = a.getGridExtendOverMin();
//				for(PlacementConnection pc : debugAllConnections)
//				{
//					PlacementNode plNode1 = pc.getP1().getPlacementNode();
//					PlacementPort thisPp1 = pc.getP1();
//					PlacementAdapter.PlacementNode pan1 = (PlacementAdapter.PlacementNode)plNode1;
//					NodeInst bmNI1 = pan1.getOriginal();
//					NodeInst newNi1 = placementMap.get(bmNI1.getName());
//					PlacementAdapter.PlacementPort pp1 = (PlacementAdapter.PlacementPort)thisPp1;
//					PortInst thisPi1 = newNi1.findPortInstFromProto(pp1.getPortProto());
//					EPoint pt1 = thisPi1.getCenter();
//
//					PlacementNode plNode2 = pc.getP2().getPlacementNode();
//					PlacementPort thisPp2 = pc.getP2();
//					PlacementAdapter.PlacementNode pan2 = (PlacementAdapter.PlacementNode)plNode2;
//					NodeInst bmNI2 = pan2.getOriginal();
//					NodeInst newNi2 = placementMap.get(bmNI2.getName());
//					PlacementAdapter.PlacementPort pp2 = (PlacementAdapter.PlacementPort)thisPp2;
//					PortInst thisPi2 = newNi2.findPortInstFromProto(pp2.getPortProto());
//					EPoint pt2 = thisPi2.getCenter();
//
//					ArcInst.newInstanceNoCheck(cell, Generic.tech().unrouted_arc, null, null, thisPi1, thisPi2,
//						pt1, pt2, gridExtend, ArcInst.DEFAULTANGLE, a.flags);
//				}
//			}
//			return true;
//		}
//
//		public void terminateOK()
//		{
//			showNextPair();
//		}
//	}
//
//	private static void showNextPair()
//	{
//		EditWindow wnd = EditWindow.getCurrent();
//		Highlighter h = wnd.getHighlighter();
//		h.clear();
//		PNPair pnp = debugProposed.peekNextPNPair();
//		if (pnp != null)
//		{
//			ProxyNode pnp1 = debugProposed.proxyMap.get(pnp.n1);
//			ProxyNode pnp2 = debugProposed.proxyMap.get(pnp.n2);
//			ProxyCluster cluster1 = debugProposed.clusterMap.get(pnp1);
//			ProxyCluster cluster2 = debugProposed.clusterMap.get(pnp2);
//			ERectangle bound1 = ERectangle.fromLambda(pnp1.x-pnp1.width/2, pnp1.y-pnp1.height/2, pnp1.width, pnp1.height);
//			ERectangle bound2 = ERectangle.fromLambda(pnp2.x-pnp2.width/2, pnp2.y-pnp2.height/2, pnp2.width, pnp2.height);
//			h.addPoly(new Poly(bound1), wnd.getCell(), Color.GREEN);
//			h.addPoly(new Poly(bound2), wnd.getCell(), Color.RED);
//			if (cluster1.nodesInCluster.size() > 1)
//			{
//				ERectangle bound1c = cluster1.getBounds();
//				double stretch = Math.max(bound1c.getWidth(), bound1c.getHeight()) / 100;
//				ERectangle bound1Clus = ERectangle.fromLambda(bound1c.getMinX()-stretch,
//					bound1c.getMinY()-stretch, bound1c.getWidth()+stretch*2, bound1c.getHeight()+stretch*2);
//				h.addPoly(new Poly(bound1Clus), wnd.getCell(), Color.GREEN);
//			}
//			if (cluster2.nodesInCluster.size() > 1)
//			{
//				ERectangle bound2c = cluster2.getBounds();
//				double stretch = Math.max(bound2c.getWidth(), bound2c.getHeight()) / 100;
//				ERectangle bound2Clus = ERectangle.fromLambda(bound2c.getMinX()-stretch,
//					bound2c.getMinY()-stretch, bound2c.getWidth()+stretch*2, bound2c.getHeight()+stretch*2);
//				h.addPoly(new Poly(bound2Clus), wnd.getCell(), Color.RED);
//			}
//		}
//		h.finished();
//	}
}
